package it.unive.lisa.imp.expressions;

import it.unive.lisa.analysis.AbstractState;
import it.unive.lisa.analysis.AnalysisState;
import it.unive.lisa.analysis.SemanticException;
import it.unive.lisa.analysis.StatementStore;
import it.unive.lisa.analysis.lattices.ExpressionSet;
import it.unive.lisa.interprocedural.InterproceduralAnalysis;
import it.unive.lisa.program.SourceCodeLocation;
import it.unive.lisa.program.cfg.CFG;
import it.unive.lisa.program.cfg.statement.Expression;
import it.unive.lisa.program.cfg.statement.NaryExpression;
import it.unive.lisa.program.cfg.statement.Statement;
import it.unive.lisa.program.cfg.statement.VariableRef;
import it.unive.lisa.program.cfg.statement.call.Call.CallType;
import it.unive.lisa.program.cfg.statement.call.UnresolvedCall;
import it.unive.lisa.symbolic.SymbolicExpression;
import it.unive.lisa.symbolic.heap.HeapReference;
import it.unive.lisa.symbolic.heap.MemoryAllocation;
import it.unive.lisa.symbolic.value.Identifier;
import it.unive.lisa.type.ReferenceType;
import it.unive.lisa.type.Type;
import it.unive.lisa.type.UnitType;
import java.util.Objects;
import org.apache.commons.lang3.ArrayUtils;

/**
 * An expression modeling the object allocation and initialization operation
 * ({@code new className(...)}). The type of this expression is the
 * {@link UnitType} representing the created class. This expression corresponds
 * to a {@link MemoryAllocation} that is used as first parameter (i.e.,
 * {@code this}) for the {@link UnresolvedCall} targeting the invoked
 * constructor. All parameters of the constructor call are provided to the
 * {@link UnresolvedCall}.
 * 
 * @author <a href="mailto:luca.negrini@unive.it">Luca Negrini</a>
 */
public class IMPNewObj extends NaryExpression {

	private final boolean staticallyAllocated;

	/**
	 * Builds the object allocation and initialization.
	 * 
	 * @param cfg                 the {@link CFG} where this operation lies
	 * @param sourceFile          the source file name where this operation is
	 *                                defined
	 * @param line                the line number where this operation is
	 *                                defined
	 * @param col                 the column where this operation is defined
	 * @param type                the type of the object that is being created
	 * @param staticallyAllocated if this allocation is static or not
	 * @param parameters          the parameters of the constructor call
	 */
	public IMPNewObj(
			CFG cfg,
			String sourceFile,
			int line,
			int col,
			Type type,
			boolean staticallyAllocated,
			Expression... parameters) {
		super(cfg, new SourceCodeLocation(sourceFile, line, col), (staticallyAllocated ? "" : "new ") + type, type,
				parameters);
		this.staticallyAllocated = staticallyAllocated;
	}

	@Override
	protected int compareSameClassAndParams(
			Statement o) {
		return Boolean.compare(staticallyAllocated, ((IMPNewObj) o).staticallyAllocated);
	}

	@Override
	public <A extends AbstractState<A>> AnalysisState<A> forwardSemanticsAux(
			InterproceduralAnalysis<A> interprocedural,
			AnalysisState<A> state,
			ExpressionSet[] params,
			StatementStore<A> expressions)
			throws SemanticException {
		Type type = getStaticType();
		ReferenceType reftype = new ReferenceType(type);
		MemoryAllocation created = new MemoryAllocation(type, getLocation(), staticallyAllocated);
		HeapReference ref = new HeapReference(reftype, created, getLocation());

		// we need to add the receiver to the parameters
		VariableRef paramThis = new VariableRef(getCFG(), getLocation(), "$lisareceiver", reftype);
		Expression[] fullExpressions = ArrayUtils.insert(0, getSubExpressions(), paramThis);

		// we also have to add the receiver inside the state
		AnalysisState<A> callstate = paramThis.forwardSemantics(state, interprocedural, expressions);
		AnalysisState<A> tmp = state.bottom();
		for (SymbolicExpression v : callstate.getComputedExpressions())
			tmp = tmp.lub(callstate.assign(v, ref, paramThis));
		ExpressionSet[] fullParams = ArrayUtils.insert(0, params,
				callstate.getComputedExpressions());
		expressions.put(paramThis, tmp);

		UnresolvedCall call = new UnresolvedCall(getCFG(), getLocation(), CallType.INSTANCE, type.toString(),
				type.toString(), fullExpressions);
		AnalysisState<A> sem = call.forwardSemanticsAux(interprocedural, tmp, fullParams, expressions);

		// now remove the instrumented receiver
		expressions.forget(paramThis);
		for (SymbolicExpression v : callstate.getComputedExpressions())
			if (v instanceof Identifier)
				sem = sem.forgetIdentifier((Identifier) v);

		sem = sem.smallStepSemantics(created, this);

		AnalysisState<A> result = state.bottom();
		for (SymbolicExpression loc : sem.getComputedExpressions()) {
			ReferenceType staticType = new ReferenceType(loc.getStaticType());
			HeapReference locref = new HeapReference(staticType, loc, getLocation());
			result = result.lub(sem.smallStepSemantics(locref, call));
		}

		return result;
	}

	@Override
	public <A extends AbstractState<A>> AnalysisState<A> backwardSemanticsAux(
			InterproceduralAnalysis<A> interprocedural,
			AnalysisState<A> state,
			ExpressionSet[] params,
			StatementStore<A> expressions)
			throws SemanticException {
		// TODO implement this when backward analysis will be out of
		// beta
		throw new UnsupportedOperationException();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + Objects.hash(staticallyAllocated);
		return result;
	}

	@Override
	public boolean equals(
			Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		IMPNewObj other = (IMPNewObj) obj;
		return staticallyAllocated == other.staticallyAllocated;
	}
}
