import styled from 'styled-components'
import { StyledWrapper } from '../../SubnetsStyles'

import TypographyPrimary from 'components/Typography/Primary'
import {
  StyledCardsContainer,
  StyledColumn,
  StyledLogsContainer,
  StyledLogsHeader,
} from './LogsPanelStyles'
import LogsCard from './LogsCard'

const LogsPanel = () => {
  return (
    <StyledWrapper>
      <StyledLogsContainer>
        <StyledLogsHeader>
          <StyledColumn customWidth={294}>
            <TypographyPrimary size='small' value={'Service'} />
          </StyledColumn>

          <StyledColumn customWidth={400}>
            <TypographyPrimary size='small' value={'Path'} />
          </StyledColumn>
          <StyledColumn customWidth={150}>
            <TypographyPrimary size='small' value={'TimeStamp'} />
          </StyledColumn>
          <StyledColumn customWidth={100}>
            <TypographyPrimary size='small' value={'Status'} />
          </StyledColumn>
          <StyledColumn customWidth={120}>
            <TypographyPrimary size='small' value={'Consumer ID'} />
          </StyledColumn>
        </StyledLogsHeader>

        <StyledCardsContainer>
          {LOGS_DATA.map((data, index) => {
            return (
              <LogsCard
                key={index}
                service={{
                  name: data.service.name,
                  subName: data.service.subName,
                  logo: data.service.logo,
                }}
                path={{
                  method: data.path.method,
                  url: data.path.url,
                }}
                timeStamp={{
                  date: data.timeStamp.date,
                  hours: data.timeStamp.hours,
                }}
                requestStatus={data.status}
                consumerId={data.consumerId}
              />
            )
          })}
        </StyledCardsContainer>
      </StyledLogsContainer>
    </StyledWrapper>
  )
}

export default LogsPanel

const LOGS_DATA = [
  {
    service: {
      name: 'Bittensor Docs',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '200',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Google News Search',
      subName: 'Subnet 22',
      logo: 'data:image/jpeg;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '6/01/2024', hours: '3:15:30 PM' },
    status: '200',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Wikipedia Search',
      subName: 'Subnet 22',
      logo: 'data:image/jpeg;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '6/01/2024', hours: '3:15:30 PM' },
    status: '404',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Youtube Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '6/01/2024', hours: '3:15:30 PM' },
    status: '404',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Discord Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '6/02/2024', hours: '4:20:45 PM' },
    status: '200',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'ArXiv Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '200',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Youtube Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '6/04/2024', hours: '6:30:05 PM' },
    status: '404',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Writing',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '200',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Twitter Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '200',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Google Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '404',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Google Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '404',
    consumerId: 'test-id',
  },
  {
    service: {
      name: 'Reddit Search',
      subName: 'Subnet 22',
      logo: 'data:image/png;base64,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',
    },
    path: { method: 'GET', url: '/vault/consumers/test-consumer' },
    timeStamp: { date: '5/31/2024', hours: '2:10:01 PM' },
    status: '404',
    consumerId: 'test-id',
  },
]
