package synthetics

import (
	"fmt"
	"time"
)

// Time is a type used for unmarshaling timestamps generated by the Synthetics API.
// Its underlying type is time.Time.
type Time time.Time

var syntheticsTimeFormat = "2006-01-02T15:04:05.999999999-0700"

// MarshalJSON is responsible for marshaling the Time type.
func (t Time) MarshalJSON() ([]byte, error) {
	dt := time.Time(t).Format(syntheticsTimeFormat)
	return []byte(fmt.Sprintf(`"%s"`, dt)), nil
}

// UnmarshalJSON is responsible for unmarshaling the Time type.
func (t *Time) UnmarshalJSON(s []byte) (err error) {
	dt := string(s)[1 : len(s)-1]
	parsed, err := time.Parse(syntheticsTimeFormat, dt)

	if err != nil {
		return err
	}

	*(*time.Time)(t) = parsed
	return
}

// Equal provides a comparator for the Time type.
func (t Time) Equal(u Time) bool {
	return time.Time(t).Equal(time.Time(u))
}
