// SPDX-License-Identifier: MIT
// Modifier: kazuya kawaguchi (a.k.a. kazupon)
// Forked from `@vue/compiler-vapor`
// Author: Evan you (https://github.com/yyx990803) and Vapor team (https://github.com/orgs/vuejs/teams/vapor)
// Repository url: https://github.com/vuejs/core-vapor
// Code url: https://github.com/vuejs/core-vapor/blob/6608bb31973d35973428cae4fbd62026db068365/packages/compiler-vapor/src/transform.ts

import { isArray } from '@vue-vapor/shared'
import { IRNodeTypes } from './ir/index.ts'
import {
  TransformContext,
  newBlock,
  transformBind,
  transformChildren,
  transformComment,
  transformElement,
  transformFor,
  transformHtml,
  transformIf,
  transformModel,
  transformOn,
  transformSlot,
  transformSlotOutlet,
  transformTemplateRef,
  transformText
} from './transforms/index.ts'

import type { RootIRNode, RootNode } from './ir/index.ts'
import type { DirectiveTransform, NodeTransform, TransformOptions } from './transforms/index.ts'

// Svelte AST -> IR
export function transform(node: RootNode, options: TransformOptions = {}): RootIRNode {
  const ir: RootIRNode = {
    type: IRNodeTypes.ROOT,
    node,
    source: node.source,
    template: [],
    component: new Set(),
    directive: new Set(),
    block: newBlock(node)
  }

  const context = new TransformContext(ir, node, options)

  transformNode(context)

  return ir
}

export function transformNode(context: TransformContext): void {
  let { node } = context

  // apply transform plugins
  const { nodeTransforms } = context.options
  const exitFns: ReturnType<NodeTransform> = []
  for (const nodeTransform of nodeTransforms) {
    const onExit = nodeTransform(node, context)
    if (onExit) {
      if (isArray(onExit)) {
        exitFns.push(...onExit)
      } else {
        exitFns.push(onExit)
      }
    }
    if (context.node) {
      // node may have been replaced
      node = context.node
    } else {
      // node was removed
      return
    }
  }

  // exit transforms
  context.node = node
  let i = exitFns.length
  while (i--) {
    exitFns[i]()
  }

  if (context.node.type === IRNodeTypes.ROOT) {
    context.registerTemplate()
  }
}

export type TransformPreset = [NodeTransform[], Record<string, DirectiveTransform>]

export function getBaseTransformPreset(_prefixIdentifiers?: boolean): TransformPreset {
  return [
    [
      // transformOnce,
      transformIf,
      transformFor,
      transformSlotOutlet,
      transformTemplateRef,
      transformText,
      transformElement,
      transformSlot,
      transformComment,
      transformHtml,
      transformChildren
    ],
    {
      bind: transformBind,
      on: transformOn,
      // text: transformText,
      // show: transformShow,
      model: transformModel
    }
  ]
}
