﻿using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Diagnostics;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Text;
using TextToTalk.UI.SourceGeneration.Contexts;
using TextToTalk.UI.SourceGeneration.EqualityComparers;

namespace TextToTalk.UI.SourceGeneration;

/// <summary>
/// Generates ImGui components for configuration classes.
/// Based on https://github.com/Flash0ver/F0-Talks-SourceGenerators.
/// </summary>
[Generator(LanguageNames.CSharp)]
public class ConfigComponentsGenerator : IIncrementalGenerator
{
    private static readonly HashSet<string> SupportedTypes = new() { "bool" };

    /// <inheritdoc />
    public void Initialize(IncrementalGeneratorInitializationContext context)
    {
        // Parse the syntax tree to find the classes to generate code for
        var syntaxProvider = context.SyntaxProvider
            .CreateSyntaxProvider(IsTargetSyntax, SemanticTransform)
            .Where(static type => type.HasValue)
            .Select(static (type, ct) => TransformType(type!.Value, ct))
            .WithComparer(ConfigComponentsContextEqualityComparer.Default);

        context.RegisterSourceOutput(syntaxProvider, Execute);
    }

    /// <summary>
    /// Generates the partial class implementation for the target symbol.
    /// </summary>
    /// <param name="context">The code generation context.</param>
    /// <param name="data">The context for this generator implementation.</param>
    private static void Execute(SourceProductionContext context, ConfigComponentsContext data)
    {
        // Reuse the same modifiers; previous checks ensure this is partial and non-static
        var modifiers = string.Join(" ", data.Modifiers.Select(mod => mod.Text));

        // Note that the strings below must be LF, not CRLF. Git will auto-convert these,
        // which can cause unexpected test failures.
        // TODO: Autodetect this
        var methods = string.Join("\n\n    ", data.ConfigOptions.Select(option =>
        {
            // ReSharper disable once ConvertToLambdaExpression
            //lang=c#
            return
                $@"/// <summary>
    /// Creates a checkbox which toggles the provided configuration object's
    /// <see cref=""global::{data.ConfigNamespace}.{data.ConfigName}.{option.Name}""/> property.
    /// </summary>
    /// <param name=""label"">The label for the UI element.</param>
    /// <param name=""config"">The config object being modified.</param>
    public static void Toggle{option.Name}(string label, global::{data.ConfigNamespace}.{data.ConfigName} config)
    {{
        var value = config.{option.Name};
        if (global::ImGuiNET.ImGui.Checkbox(label, ref value))
        {{
            config.{option.Name} = value;
            config.Save();
        }}
    }}";
        }));

        //lang=c#
        var source = SourceText.From($@"// <auto-generated />
namespace {data.Namespace};

[global::System.CodeDom.Compiler.GeneratedCodeAttribute(""{ToolInfo.AssemblyName}"", ""{ToolInfo.AssemblyVersion}"")]
{modifiers} class {data.Name}
{{
    public global::System.Action OnOptionChanged {{ get; }}

    {methods}
}}
", Encoding.UTF8);

        context.AddSource($"{data.Name}.ConfigComponents.g.cs", source);
    }

    /// <summary>
    /// Returns true if the provided node matches the expected syntax for code generation.
    /// </summary>
    /// <param name="node">The input syntax node.</param>
    /// <param name="cancellationToken"></param>
    /// <returns></returns>
    private static bool IsTargetSyntax(SyntaxNode node, CancellationToken cancellationToken)
    {
        // Find classes with at least one attribute that are partial and non-static
        return node is ClassDeclarationSyntax { AttributeLists.Count: > 0 } classDecl &&
               classDecl.Modifiers.Any(SyntaxKind.PartialKeyword) &&
               !classDecl.Modifiers.Any(SyntaxKind.StaticKeyword);
    }

    /// <summary>
    /// Transforms the node referenced by the provided generator context into a more
    /// useful form, if it has a UseConfigComponentsAttribute attached.
    /// </summary>
    /// <param name="context">The code generation context.</param>
    /// <param name="cancellationToken"></param>
    /// <returns></returns>
    private static (SyntaxReference, INamedTypeSymbol, INamedTypeSymbol)? SemanticTransform(
        GeneratorSyntaxContext context,
        CancellationToken cancellationToken)
    {
        Debug.Assert(context.Node is ClassDeclarationSyntax);
        var classDecl = Unsafe.As<ClassDeclarationSyntax>(context.Node);

        ISymbol? symbol = context.SemanticModel.GetDeclaredSymbol(classDecl, cancellationToken);

        if (symbol is not INamedTypeSymbol type)
        {
            return null;
        }

        const string attrName = "TextToTalk.UI.Core.UseConfigComponentsAttribute";
        if (!TryGetAttribute(classDecl, attrName, context.SemanticModel, out var attr))
        {
            return null;
        }

        // Get the attribute target, e.g. the config class
        if (!TryGetAttributeTarget(attr, context.SemanticModel, cancellationToken, out var target))
        {
            return null;
        }

        // Ensure that the config class implements ISaveable
        const string interfaceName = "TextToTalk.UI.Core.ISaveable";
        if (!target.Interfaces.Any(static @interface =>
                @interface.ConstructedFrom.ToDisplayString().Equals(interfaceName, StringComparison.Ordinal)))
        {
            throw new InvalidOperationException("Configuration type must implement ISaveable.");
        }

        return (classDecl.GetReference(), type, target);
    }

    /// <summary>
    /// Transforms the extracted symbols into a specialized context for this generator
    /// implementation.
    /// </summary>
    /// <param name="type">The extracted symbols.</param>
    /// <param name="cancellationToken"></param>
    /// <returns></returns>
    private static ConfigComponentsContext TransformType(
        (SyntaxReference, INamedTypeSymbol, INamedTypeSymbol) type,
        CancellationToken cancellationToken)
    {
        var (targetReference, targetSymbol, configSymbol) = type;

        // Target object properties
        var targetNamespace = targetSymbol.ContainingNamespace?.IsGlobalNamespace == true
            ? null
            : targetSymbol.ContainingNamespace?.ToDisplayString();
        var targetName = targetSymbol.Name;

        var targetNode = targetReference.GetSyntax(cancellationToken);
        Debug.Assert(targetNode is ClassDeclarationSyntax);
        var targetClassDecl = Unsafe.As<ClassDeclarationSyntax>(targetNode);

        // Config object properties
        var configNamespace = configSymbol.ContainingNamespace?.IsGlobalNamespace == true
            ? null
            : configSymbol.ContainingNamespace?.ToDisplayString();
        var configName = configSymbol.Name;
        var configProperties = configSymbol.GetThisAndSubtypes()
            .Reverse()
            .SelectMany(static type => type.GetMembers())
            .Where(FilterProperty)
            .Select(static member => TransformProperty(member))
            .Where(prop => SupportedTypes.Contains(prop.TypeName))
            .Distinct()
            .ToImmutableArray();

        return new ConfigComponentsContext(
            targetNamespace,
            targetName,
            targetClassDecl.Modifiers,
            configNamespace,
            configName,
            configProperties);
    }

    /// <summary>
    /// A predicate for filtering symbols to accessible property types.
    /// </summary>
    /// <param name="member">The symbol to check.</param>
    /// <returns></returns>
    private static bool FilterProperty(ISymbol member)
    {
        if (member is { IsStatic: false, Kind: SymbolKind.Property })
        {
            Debug.Assert(member is IPropertySymbol);
            var property = Unsafe.As<IPropertySymbol>(member);

            return property.GetMethod is { DeclaredAccessibility: Accessibility.Public };
        }

        return false;
    }

    /// <summary>
    /// Converts the provided property into an option object.
    /// </summary>
    /// <param name="member">The symbol to convert.</param>
    /// <returns></returns>
    private static ConfigComponentsContext.Option TransformProperty(ISymbol member)
    {
        Debug.Assert(member is IPropertySymbol);
        var property = Unsafe.As<IPropertySymbol>(member);

        var type = property.Type.ToDisplayString();
        var name = property.Name;

        return new ConfigComponentsContext.Option(name, type);
    }

    /// <summary>
    /// Attempts to find the provided attribute on the member syntax instance.
    /// </summary>
    /// <param name="decl">The syntax declaration to check.</param>
    /// <param name="attrName">The target attribute's name.</param>
    /// <param name="model">The queryable syntax tree.</param>
    /// <param name="attr"></param>
    /// <returns></returns>
    private static bool TryGetAttribute(
        MemberDeclarationSyntax decl,
        string attrName,
        SemanticModel model,
        out AttributeSyntax? attr)
    {
        foreach (var attributeList in decl.AttributeLists)
        {
            foreach (var attribute in attributeList.Attributes)
            {
                var symbolInfo = model.GetSymbolInfo(attribute);
                var symbol = symbolInfo.Symbol;

                if (symbol is IMethodSymbol method
                    && method.ContainingType.ToDisplayString().Equals(attrName, StringComparison.Ordinal))
                {
                    attr = attribute;
                    return true;
                }
            }
        }

        attr = null;
        return false;
    }

    /// <summary>
    /// Attempts to retrieve the target type of the provided attribute.
    /// </summary>
    /// <param name="attribute">The attribute to search.</param>
    /// <param name="semanticModel">The query object for the syntax tree.</param>
    /// <param name="cancellationToken"></param>
    /// <param name="target"></param>
    /// <returns></returns>
    private static bool TryGetAttributeTarget(
        AttributeSyntax attribute,
        SemanticModel semanticModel,
        CancellationToken cancellationToken,
        out INamedTypeSymbol? target)
    {
        if (attribute.ArgumentList is
            {
                Arguments: [{ Expression: TypeOfExpressionSyntax typeOf }],
            })
        {
            var info = semanticModel.GetSymbolInfo(typeOf.Type, cancellationToken);
            var symbol = info.Symbol;

            if (symbol is INamedTypeSymbol type)
            {
                target = type;
                return true;
            }
        }

        target = null;
        return false;
    }
}