using System.Reflection;
using System.Text;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.Testing;
using Microsoft.CodeAnalysis.Testing.Verifiers;
using Microsoft.CodeAnalysis.Text;
using VerifyCS =
    TextToTalk.UI.SourceGeneration.Tests.CSharpSourceGeneratorVerifier<
        TextToTalk.UI.SourceGeneration.Tests.ConfigComponentsGeneratorTests,
        TextToTalk.UI.SourceGeneration.ConfigComponentsGenerator>;

namespace TextToTalk.UI.SourceGeneration.Tests;

public class ConfigComponentsGeneratorTests
{
    private static readonly ReferenceAssemblies Net80Windows = new("net8.0-windows",
        new PackageIdentity("Microsoft.NETCore.App.Ref", "8.0.0"),
        Path.Combine("ref", "net8.0-windows"));

    private static string GetTargetConfigSourceCode(string configInterfaces)
    {
        //lang=c#
        return $@"// <auto-generated/>

using TextToTalk.UI.Core;

namespace TextToTalk.Configuration
{{
    public class Test1Config {(!string.IsNullOrEmpty(configInterfaces) ? $": {configInterfaces}" : "")}
    {{
        public bool Option1 {{ get; set; }}

        private bool Option2 {{ get; set; }}

        public bool Option3 {{ get; set; }}

        public void Save()
        {{
            // no-op
        }}
    }}
}}
";
    }

    private static string GetTargetSourceCode(string modifiers)
    {
        //lang=c#
        return $@"// <auto-generated/>
using TextToTalk.Configuration;
using TextToTalk.UI.Core;

namespace TextToTalk.UI
{{
    [UseConfigComponents(typeof(Test1Config))]
    {modifiers} class Test1Target
    {{
    }}
}}
";
    }

    private static string GetExpectedGeneratedCode(string modifiers)
    {
        //lang=c#
        return $@"// <auto-generated />
namespace TextToTalk.UI;

[global::System.CodeDom.Compiler.GeneratedCodeAttribute(""TextToTalk.UI.SourceGeneration"", ""1.0.0.0"")]
{modifiers} class Test1Target
{{
    public global::System.Action OnOptionChanged {{ get; }}

    /// <summary>
    /// Creates a checkbox which toggles the provided configuration object's
    /// <see cref=""global::TextToTalk.Configuration.Test1Config.Option1""/> property.
    /// </summary>
    /// <param name=""label"">The label for the UI element.</param>
    /// <param name=""config"">The config object being modified.</param>
    public static void ToggleOption1(string label, global::TextToTalk.Configuration.Test1Config config)
    {{
        var value = config.Option1;
        if (global::ImGuiNET.ImGui.Checkbox(label, ref value))
        {{
            config.Option1 = value;
            config.Save();
        }}
    }}

    /// <summary>
    /// Creates a checkbox which toggles the provided configuration object's
    /// <see cref=""global::TextToTalk.Configuration.Test1Config.Option3""/> property.
    /// </summary>
    /// <param name=""label"">The label for the UI element.</param>
    /// <param name=""config"">The config object being modified.</param>
    public static void ToggleOption3(string label, global::TextToTalk.Configuration.Test1Config config)
    {{
        var value = config.Option3;
        if (global::ImGuiNET.ImGui.Checkbox(label, ref value))
        {{
            config.Option3 = value;
            config.Save();
        }}
    }}
}}
";
    }

    /// <summary>
    /// Gets the path to the ImGui.NET DLL.
    ///
    /// Importing ImGui.NET directly and using typeof(ImGui).Assembly.Location would
    /// be ideal for portability, but that requires that ImGui is present in the build
    /// output for the test runner to load that type.
    ///
    /// Copying ImGui.NET to the build output of this project would be fine for this
    /// since it's not part of the plugin's build output, but the D17 build system
    /// will flag that and fail the build.
    /// </summary>
    /// <returns></returns>
    private static string ImGuiNetPath()
    {
        var appData = Environment.GetEnvironmentVariable("AppData");
        return Path.Combine(appData!, "XIVLauncher", "addon", "Hooks", "dev", "ImGui.NET.dll");
    }

    private static string UICorePath()
    {
        var thisAssembly = Assembly.GetExecutingAssembly().Location;
        return Path.Combine(thisAssembly, "..", "..", "..", "..", "..", "TextToTalk.UI.Core", "bin", "Debug", "net8.0",
            "TextToTalk.UI.Core.dll");
    }

    [Fact]
    public async Task Generates_Expected_Code_When_Public()
    {
        await new VerifyCS.Test
        {
            TestState =
            {
                ReferenceAssemblies = Net80Windows,
                AdditionalReferences =
                {
                    MetadataReference.CreateFromFile(UICorePath()),
                    MetadataReference.CreateFromFile(ImGuiNetPath()),
                },
                Sources =
                {
                    ("Test1Config.cs", SourceText.From(GetTargetConfigSourceCode("ISaveable"), Encoding.UTF8)),
                    ("Test1Target.cs", SourceText.From(GetTargetSourceCode("public partial"), Encoding.UTF8)),
                },
                GeneratedSources =
                {
                    (@"TextToTalk.UI.SourceGeneration\TextToTalk.UI.SourceGeneration.ConfigComponentsGenerator\Test1Target.ConfigComponents.g.cs",
                        SourceText.From(GetExpectedGeneratedCode("public partial"), Encoding.UTF8)),
                },
            },
        }.AddGeneratedSources().RunAsync();
    }

    [Fact]
    public async Task Generates_Expected_Code_When_Internal()
    {
        await new VerifyCS.Test
        {
            TestState =
            {
                ReferenceAssemblies = Net80Windows,
                AdditionalReferences =
                {
                    MetadataReference.CreateFromFile(UICorePath()),
                    MetadataReference.CreateFromFile(ImGuiNetPath()),
                },
                Sources =
                {
                    ("Test1Config.cs", SourceText.From(GetTargetConfigSourceCode("ISaveable"), Encoding.UTF8)),
                    ("Test1Target.cs", SourceText.From(GetTargetSourceCode("internal partial"), Encoding.UTF8)),
                },
                GeneratedSources =
                {
                    (@"TextToTalk.UI.SourceGeneration\TextToTalk.UI.SourceGeneration.ConfigComponentsGenerator\Test1Target.ConfigComponents.g.cs",
                        SourceText.From(GetExpectedGeneratedCode("internal partial"), Encoding.UTF8)),
                },
            },
        }.AddGeneratedSources().RunAsync();
    }

    [Fact]
    public async Task Does_Not_Generate_Code_When_Not_Partial()
    {
        await Assert.ThrowsAsync<EqualWithMessageException>(() => new VerifyCS.Test
        {
            TestState =
            {
                ReferenceAssemblies = Net80Windows,
                AdditionalReferences =
                {
                    MetadataReference.CreateFromFile(UICorePath()),
                    MetadataReference.CreateFromFile(ImGuiNetPath()),
                },
                Sources =
                {
                    ("Test1Config.cs", SourceText.From(GetTargetConfigSourceCode("ISaveable"), Encoding.UTF8)),
                    ("Test1Target.cs", SourceText.From(GetTargetSourceCode("public"), Encoding.UTF8)),
                },
                GeneratedSources =
                {
                    (@"TextToTalk.UI.SourceGeneration\TextToTalk.UI.SourceGeneration.ConfigComponentsGenerator\Test1Target.ConfigComponents.g.cs",
                        SourceText.From(GetExpectedGeneratedCode("public"), Encoding.UTF8)),
                },
            },
        }.AddGeneratedSources().RunAsync());
    }

    [Fact]
    public async Task Does_Not_Generate_Code_When_Config_Not_Saveable()
    {
        await Assert.ThrowsAsync<EqualWithMessageException>(() => new VerifyCS.Test
        {
            TestState =
            {
                ReferenceAssemblies = Net80Windows,
                AdditionalReferences =
                {
                    MetadataReference.CreateFromFile(UICorePath()),
                    MetadataReference.CreateFromFile(ImGuiNetPath()),
                },
                Sources =
                {
                    ("Test1Config.cs", SourceText.From(GetTargetConfigSourceCode(""), Encoding.UTF8)),
                    ("Test1Target.cs", SourceText.From(GetTargetSourceCode("public"), Encoding.UTF8)),
                },
                GeneratedSources =
                {
                    (@"TextToTalk.UI.SourceGeneration\TextToTalk.UI.SourceGeneration.ConfigComponentsGenerator\Test1Target.ConfigComponents.g.cs",
                        SourceText.From(GetExpectedGeneratedCode("public"), Encoding.UTF8)),
                },
            },
        }.AddGeneratedSources().RunAsync());
    }
}