import defaults from '../defaults';
import SSMLSplit from '../index';

import {
  ssmlExampleLargeArticle,
  ssmlExampleLargeMultipleParagraphs,
  ssmlExampleLargeMultipleParagraphsWithSentences,
  ssmlExampleLargeSingleParagraph
} from './ssml-examples';

import { textExampleLarge, textExampleLargeSentence } from './text-examples';

const GOOGLE_SOFT_LIMIT = 4000;
const GOOGLE_HARD_LIMIT = 5000;

const AWS_SOFT_LIMIT = 2000;
const AWS_HARD_LIMIT = 3000;

/**
 * Remove SSML tags from the SSML string, so we can count like AWS counts.
 * To verify if the output SSML stays below AWS' hard limit.
 */
function removeSSMLTags(ssml: string): string {
  return ssml.replace(/(<([^>]+)>)/gi, '').trim();
}

describe('constructor', () => {
  it('Should set default options when no options object is given', () => {
    const ssmlSplit = new SSMLSplit();
    expect(ssmlSplit.options.synthesizer).toBe('aws');
    expect(ssmlSplit.options.extraSplitChars).toBe(defaults.EXTRA_SPLIT_CHARS);
    expect(ssmlSplit.options.hardLimit).toBe(defaults.HARD_LIMIT);
    expect(ssmlSplit.options.softLimit).toBe(defaults.SOFT_LIMIT);
    expect(ssmlSplit.options.breakParagraphsAboveHardLimit).toBe(defaults.BREAK_PARAGRAPHS_ABOVE_HARD_LIMIT);
    expect(ssmlSplit.options.defaultParagraphBreakSSMLTag).toBe(defaults.AWS_PARAGRAPH_BREAK_SSML_TAG);
  });

  it('Should set default options when synthesizer "google" is given.', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google'
    });
    expect(ssmlSplit.options.synthesizer).toBe('google');
    expect(ssmlSplit.options.defaultParagraphBreakSSMLTag).toBe(defaults.GOOGLE_PARAGRAPH_BREAK_SSML_TAG);
  });

  it('Should set default options when synthesizer "aws" is given.', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws'
    });
    expect(ssmlSplit.options.synthesizer).toBe('aws');
    expect(ssmlSplit.options.defaultParagraphBreakSSMLTag).toBe(defaults.AWS_PARAGRAPH_BREAK_SSML_TAG);
  });

  it('Should return an error when options is not an object', () => {
    try {
      // @ts-ignore
      const ssmlSplit = new SSMLSplit(1);
    } catch (err) {
      expect(err.message).toBe('Parameter `options` must be an object.');
    }
  });

  it('Should return an error when option breakParagraphsAboveHardLimit: true is set, but no synthesizer option is given', () => {
    try {
      // @ts-ignore
      const ssmlSplit = new SSMLSplit({
        breakParagraphsAboveHardLimit: true
      });
    } catch (err) {
      expect(err.message).toBe('Option `synthesizer` is required when using `breakParagraphsAboveHardLimit`.');
    }
  });

  it('Should return an error when option synthesizer uses an unknown value', () => {
    try {
      // @ts-ignore
      const ssmlSplit = new SSMLSplit({
        // @ts-ignore
        synthesizer: 'unknown'
      });
    } catch (err) {
      expect(err.message).toBe('Option `synthesizer` must be \"google\" or \"aws\".');
    }
  });

  it('Should return 5 when "softLimit: 5" is given as an option', () => {
    const ssmlSplit = new SSMLSplit({
      softLimit: 5
    });

    expect(ssmlSplit.options.softLimit).toBe(5);
  });

  it('Should return 10 when "hardLimit: 10" is given as an option', () => {
    const ssmlSplit = new SSMLSplit({
      hardLimit: 10
    });

    expect(ssmlSplit.options.hardLimit).toBe(10);
  });

  it('Should return google when synthesizer: "google" is given as an option', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google'
    });

    expect(ssmlSplit.options.synthesizer).toBe('google');
  });

  it('Should return "/" when "extraSplitChars: \'/\'" is given as an option', () => {
    const ssmlSplit = new SSMLSplit({
      extraSplitChars: '/'
    });

    expect(ssmlSplit.options.extraSplitChars).toBe('/');
  });

  it('Should return true when "breakParagraphsAboveHardLimit: true" is given as an option with synthesizer: "google"', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      breakParagraphsAboveHardLimit: true
    });

    expect(ssmlSplit.options.breakParagraphsAboveHardLimit).toBe(true);
  });

  it('Should return true when "breakParagraphsAboveHardLimit: true" is given as an option with synthesizer: "aws"', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      breakParagraphsAboveHardLimit: true
    });

    expect(ssmlSplit.options.breakParagraphsAboveHardLimit).toBe(true);
  });

  it('Should return true when "breakParagraphsAboveHardLimit: false" is given as an option with synthesizer: "google"', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      breakParagraphsAboveHardLimit: false
    });

    expect(ssmlSplit.options.breakParagraphsAboveHardLimit).toBe(false);
  });

  it('Should return true when "breakParagraphsAboveHardLimit: false" is given as an option with synthesizer: "aws"', () => {
    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      breakParagraphsAboveHardLimit: false
    });

    expect(ssmlSplit.options.breakParagraphsAboveHardLimit).toBe(false);
  });
});

describe('private methods', () => {
  it('characterCounter should return the correct character count', () => {
    const ssml = '<speak><p>some text inside a tag</p></speak>';

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws'
    });
    ssmlSplit.split(ssml);
    expect(ssmlSplit['characterCounter']).toBe(22);

    const ssmlSplit2 = new SSMLSplit({
      synthesizer: 'google'
    });
    ssmlSplit2.split(ssml);
    expect(ssmlSplit2['characterCounter']).toBe(44);
  });

  it('setDefault should correctly reset back to the default values', () => {
    const ssml = '<speak><p>some text</p></speak>';

    const ssmlSplit = new SSMLSplit();

    ssmlSplit.split(ssml);

    expect(ssmlSplit['root']).toMatchObject({
      children: [
        {
          children: [
            {
              parentNode: expect.anything(), // [Circular]
              type: 'TEXT',
              value: 'some text'
            }
          ],
          parentNode: {
            children: expect.anything(), // [Circular]
            parentNode: expect.anything(), // [Circular]
            type: 'speak',
            value: ''
          },
          type: 'p',
          value: ''
        }
      ],
      parentNode: null,
      type: 'root',
      value: ''
    });
    expect(ssmlSplit['batches']).toStrictEqual([
      '<speak><p>some text</p></speak>'
    ]);
    expect(ssmlSplit['accumulatedSSML']).toBe('<p>some text</p>');
    expect(ssmlSplit['textLength']).toBe(9);

    ssmlSplit['setDefaults']();

    expect(ssmlSplit['root']).toMatchObject({
      parentNode: null,
      type: 'root',
      children: [],
      value: ''
    });
    expect(ssmlSplit['batches']).toStrictEqual([]);
    expect(ssmlSplit['accumulatedSSML']).toBe('');
    expect(ssmlSplit['textLength']).toBe(0);
  });

  it('sanitize should correctly cleanup the string from excessive white spaces and empty tags', () => {
    const ssml = `
      <speak>
        <p>some text</p>

        
        <s></s>
        <p></p>
        <s> </s>
        <p> </p>
        <p>   </p>
        <p/>
        <s/>
      </speak>
    `;

    const ssmlSplit = new SSMLSplit();

    expect(ssmlSplit['sanitize'](ssml)).toBe('<speak><p>some text</p></speak>');
  });

  it('makeSpeakBatch should add a new SSML string to the array', () => {
    const ssmlSplit = new SSMLSplit();

    ssmlSplit['makeSpeakBatch']('Lorem ipsum dolor sit amet.');
    ssmlSplit['makeSpeakBatch']('Consectetuer adipiscing elit.');

    expect(ssmlSplit['batches']).toStrictEqual([
      '<speak>Lorem ipsum dolor sit amet.</speak>',
      '<speak>Consectetuer adipiscing elit.</speak>'
    ]);
  });

  it('buildTree should create an object tree using the SSML string', () => {
    const ssmlSplit = new SSMLSplit();

    ssmlSplit['buildTree'](
      '<speak><s>Lorem ipsum dolor sit amet, consectetuer adipiscing elit.</s><s>Some other sentence.</s></speak>'
    );

    expect(ssmlSplit['root']).toMatchObject({
      children: [
        {
          children: [
            {
              children: [
                {
                  parentNode: expect.anything(), // [Circular]
                  type: 'TEXT',
                  value:
                    'Lorem ipsum dolor sit amet, consectetuer adipiscing elit.'
                }
              ],
              parentNode: expect.anything(), // [Circular]
              type: 's',
              value: ''
            },
            {
              children: [
                {
                  parentNode: expect.anything(), // [Circular]
                  type: 'TEXT',
                  value: 'Some other sentence.'
                }
              ],
              parentNode: expect.anything(), // [Circular]
              type: 's',
              value: ''
            }
          ],
          parentNode: expect.anything(), // [Circular]
          type: 'speak',
          value: ''
        }
      ],
      parentNode: null,
      type: 'root',
      value: ''
    });
  });
});

describe('split', () => {
  it('Should return an empty array when no ssml is given and no options are given', () => {
    const ssml = '';
    const expected = [];

    const ssmlSplit = new SSMLSplit();

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an empty array when no ssml is given but options are given', () => {
    const ssml = '';
    const expected = [];

    const ssmlSplit = new SSMLSplit({
      softLimit: 5,
      hardLimit: 10
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an empty array when no ssml is given', () => {
    const ssml = '';
    const expected = [];

    const ssmlSplit = new SSMLSplit({
      softLimit: 5,
      hardLimit: 10
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an array with one ssml item when a single word is given', () => {
    const ssml = 'Hello';
    const expected = ['<speak>Hello</speak>'];

    const ssmlSplit = new SSMLSplit({
      softLimit: 5,
      hardLimit: 10
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an array with multiple ssml items when a text with a SSML tag is given', () => {
    const ssml = 'plain text <ssmltag>Hello</ssmltag> plain text';
    const expected = [
      '<speak>plain text </speak>',
      '<speak><ssmltag>Hello</ssmltag></speak>',
      '<speak> plain text</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      softLimit: 5,
      hardLimit: 15
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an array with multiple ssml items when a SSML string is given', () => {
    const ssml =
      '<speak>My favorite chemical element is <sub alias="aluminum">Al</sub>, but Al prefers <sub alias="magnesium">Mg</sub>.</speak>';
    const expected = [
      '<speak>My favorite </speak>',
      '<speak>chemical </speak>',
      '<speak>element is </speak>',
      '<speak><sub alias="aluminum">Al</sub>, but Al </speak>',
      '<speak>prefers </speak>',
      '<speak><sub alias="magnesium">Mg</sub>.</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      softLimit: 5,
      hardLimit: 15
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an array with multiple ssml items when a SSML string is given', () => {
    const ssml =
      '<speak>I want to tell you a secret. <amazon:effect name="whispered">I am not a real human.</amazon:effect> Can you believe it?</speak>';
    const expected = [
      '<speak>I want to tell you a secret. </speak>',
      '<speak><amazon:effect name="whispered">I am not a real human.</amazon:effect></speak>',
      '<speak> Can you believe it?</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      softLimit: 20,
      hardLimit: 30
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);
  });

  it('Should return an error when a SSML tag is too long to split', (done) => {
    const ssml =
      '<speak><s>Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.</s><s>smaller text</s></speak>';

    const ssmlSplit = new SSMLSplit({
      softLimit: 20,
      hardLimit: 30
    });

    try {
      ssmlSplit.split(ssml);
    } catch (err) {
      expect(err.message).toBe('SSML tag appeared to be too long.');
      done();
    }
  });

  it('Should return an error when last SSML tag is too long to split', (done) => {
    const ssml =
      '<speak><s>Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.</s></speak>';

    const ssmlSplit = new SSMLSplit({
      softLimit: 20,
      hardLimit: 30
    });

    try {
      ssmlSplit.split(ssml);
    } catch (err) {
      expect(err.message).toBe('Last SSML tag appeared to be too long.');
      done();
    }
  });

  it('Should return an error when SSML tag is invalid', (done) => {
    const ssml = '<speak><Lorem ipsum dolor sit amet></speak>';

    const ssmlSplit = new SSMLSplit({
      softLimit: 20,
      hardLimit: 30
    });

    try {
      ssmlSplit.split(ssml);
    } catch (err) {
      expect(err.message).toBe('Incorrect SSML: speak !== Lorem');
      done();
    }
  });

  it('Should return correct SSML when using empty tags like <break />', () => {
    const ssml =
      '<speak><break strength="x-strong" /><p>Lorem ipsum dolor sit amet</p></speak>';
    const expected = [
      '<speak><break strength="x-strong" /><p>Lorem ipsum dolor sit amet</p></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      softLimit: 30,
      hardLimit: 50
    });

    expect(ssmlSplit.split(ssml)).toStrictEqual(expected);
  });

  it('Should return correct SSML when using mixed SSML tags', () => {
    const ssml =
      '<speak>Here is a number spoken as a cardinal number: <say-as interpret-as="cardinal">12345</say-as>. Here is the same number with each digit spoken separately: <say-as interpret-as="digits">12345</say-as>. Here is a word spelled out: <say-as interpret-as="spell-out">hello</say-as></speak>';
    const expected = [
      '<speak>Here is a number spoken as a </speak>',
      '<speak>cardinal number: <say-as interpret-as="cardinal">12345</say-as></speak>',
      '<speak>. Here is the same number with </speak>',
      '<speak>each digit spoken separately: </speak>',
      '<speak><say-as interpret-as="digits">12345</say-as>. Here is a word spelled </speak>',
      '<speak>out: <say-as interpret-as="spell-out">hello</say-as></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      softLimit: 20,
      hardLimit: 30
    });

    expect(ssmlSplit.split(ssml)).toStrictEqual(expected);
  });
});

describe('Google Text to Speech API limitations', () => {
  it('Should correctly break up a paragraph and use <break /> tag as paragraph ending, when paragraph goes above the hardLimit with option "breakParagraphsAboveHardLimit: true".', () => {
    const ssml = ssmlExampleLargeSingleParagraph;

    const expected = [
      '<speak>Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum. Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem. Maecenas nec odio et ante tincidunt tempus. Donec vitae sapien ut libero venenatis faucibus. Nullam quis ante. Etiam sit amet orci eget eros faucibus tincidunt. Duis leo. Sed fringilla mauris sit amet nibh. Donec sodales sagittis magna. Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero. Fusce vulputate eleifend sapien. Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus. Nullam accumsan lorem in dui. Cras ultricies mi eu turpis hendrerit fringilla. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia. Nam pretium turpis et arcu. Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum. Sed aliquam ultrices mauris. Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris. Praesent adipiscing. Phasellus ullamcorper ipsum rutrum nunc. Nunc nonummy metus. Vestibulum volutpat pretium libero. Cras id dui. Aenean ut eros et nisl sagittis vestibulum. Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede. Sed lectus. Donec mollis hendrerit risus. Phasellus nec sem in justo pellentesque facilisis. Etiam imperdiet imperdiet orci. Nunc nec neque. Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi. Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo. Maecenas malesuada. Praesent congue erat at massa. Sed cursus turpis vitae tortor. Donec posuere vulputate arcu. Phasellus accumsan cursus velit. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci. Phasellus consectetuer vestibulum elit. Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc. Vestibulum fringilla pede sit amet augue. In turpis. Pellentesque posuere. Praesent turpis. Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus. Donec elit libero, sodales nec, volutpat a, suscipit non, turpis. Nullam sagittis. Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus. Ut varius tincidunt libero. Phasellus dolor. Maecenas vestibulum mollis diam. Pellentesque ut neque. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. In dui magna, posuere eget, vestibulum et, tempor auctor, justo. In ac felis quis tortor malesuada pretium. Pellentesque auctor neque nec urna. Proin sapien ipsum, porta a, auctor quis, euismod ut, mi. Aenean viverra rhoncus pede. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Ut non enim eleifend felis pretium feugiat. Vivamus quis mi. Phasellus a est. Phasellus magna. In hac habitasse platea dictumst. Curabitur at lacus ac velit ornare lobortis. Curabitur a felis in nunc fringilla tristique. Morbi mattis ullamcorper velit. Phasellus gravida semper nisi. Nullam vel sem. Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam. Sed hendrerit. Morbi ac felis. Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede. Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi. Nunc nulla. Fusce risus nisl, viverra et, tempor et, pretium in, sapien. Donec venenatis vulputate lorem. Morbi nec metus. Phasellus blandit leo ut odio. Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem. Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis. In auctor lobortis lacus. Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna.</speak>',
      '<speak> Vestibulum ullamcorper mauris at ligula. Fusce fermentum. Nullam cursus lacinia erat. Praesent blandit laoreet nibh. Fusce convallis metus id felis luctus adipiscing. Pellentesque egestas, neque sit amet convallis pulvinar, justo nulla eleifend augue, ac auctor orci leo non est. Quisque id mi. Ut tincidunt tincidunt erat. Etiam feugiat lorem non metus. Vestibulum dapibus nunc ac augue. Curabitur vestibulum aliquam leo. Praesent egestas neque eu enim. In hac habitasse platea dictumst. Fusce a quam. Etiam ut purus mattis mauris sodales aliquam. Curabitur nisi. Quisque malesuada placerat nisl. Nam ipsum risus, rutrum vitae, vestibulum eu, molestie vel, lacus. Sed augue ipsum, egestas nec, vestibulum et, malesuada adipiscing, dui. Vestibulum facilisis, purus nec pulvinar iaculis, ligula mi congue nunc, vitae euismod ligula urna in dolor. Mauris sollicitudin fermentum libero. Praesent nonummy mi in odio. Nunc interdum lacus sit amet orci. Vestibulum rutrum, mi nec elementum vehicula, eros quam gravida nisl, id fringilla neque a<break strength="x-weak" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });

  it('Should correctly break up a paragraph with sentences and use <break /> tag as paragraph ending, when paragraph goes above the hardLimit.', () => {
    const ssml = ssmlExampleLargeMultipleParagraphsWithSentences;

    const expected = [
      '<speak><s>Lorem ipsum dolor sit amet, consectetuer adipiscing elit.</s><s>Aenean commodo ligula eget dolor.</s><s>Aenean massa.</s><s>Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.</s><s>Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem.</s><s>Nulla consequat massa quis enim.</s><s>Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu.</s><s>In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo.</s><s>Nullam dictum felis eu pede mollis pretium.</s><s>Integer tincidunt.</s><s>Cras dapibus.</s><s>Vivamus elementum semper nisi.</s><s>Aenean vulputate eleifend tellus.</s><s>Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim.</s><s>Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus.</s><s>Phasellus viverra nulla ut metus varius laoreet.</s><s>Quisque rutrum.</s><s>Aenean imperdiet.</s><s>Etiam ultricies nisi vel augue.</s><s>Curabitur ullamcorper ultricies nisi.</s><s>Nam eget dui.</s><s>Etiam rhoncus.</s><s>Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum.</s><s>Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem.</s><s>Maecenas nec odio et ante tincidunt tempus.</s><s>Donec vitae sapien ut libero venenatis faucibus.</s><s>Nullam quis ante.</s><s>Etiam sit amet orci eget eros faucibus tincidunt.</s><s>Duis leo.</s><s>Sed fringilla mauris sit amet nibh.</s><s>Donec sodales sagittis magna.</s><s>Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero.</s><s>Fusce vulputate eleifend sapien.</s><s>Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus.</s><s>Nullam accumsan lorem in dui.</s><s>Cras ultricies mi eu turpis hendrerit fringilla.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia.</s><s>Nam pretium turpis et arcu.</s><s>Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum.</s><s>Sed aliquam ultrices mauris.</s><s>Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris.</s><s>Praesent adipiscing.</s><s>Phasellus ullamcorper ipsum rutrum nunc.</s><s>Nunc nonummy metus.</s><s>Vestibulum volutpat pretium libero.</s><s>Cras id dui.</s><s>Aenean ut eros et nisl sagittis vestibulum.</s><s>Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede.</s><s>Sed lectus.</s><s>Donec mollis hendrerit risus.</s><s>Phasellus nec sem in justo pellentesque facilisis.</s><s>Etiam imperdiet imperdiet orci.</s><s>Nunc nec neque.</s><s>Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi.</s><s>Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo.</s><s>Maecenas malesuada.</s><s>Praesent congue erat at massa.</s><s>Sed cursus turpis vitae tortor.</s><s>Donec posuere vulputate arcu.</s><s>Phasellus accumsan cursus velit.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci.</s><s>Phasellus consectetuer vestibulum elit.</s><s>Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc.</s><s>Vestibulum fringilla pede sit amet augue.</s><s>In turpis.</s><s>Pellentesque posuere.</s><s>Praesent turpis.</s><s>Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus.</s><s>Donec elit libero, sodales nec, volutpat a, suscipit non, turpis.</s><s>Nullam sagittis.</s><s>Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus.</s><s>Ut varius tincidunt libero.</s><s>Phasellus dolor.</s><s>Maecenas vestibulum mollis diam.</s><s>Pellentesque ut neque.</s><s>Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.</s></speak>',
      '<speak><s>In dui magna, posuere eget, vestibulum et, tempor auctor, justo.</s><s>In ac felis quis tortor malesuada pretium.</s><s>Pellentesque auctor neque nec urna.</s><s>Proin sapien ipsum, porta a, auctor quis, euismod ut, mi.</s><s>Aenean viverra rhoncus pede.</s><s>Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.</s><s>Ut non enim eleifend felis pretium feugiat.</s><s>Vivamus quis mi.</s><s>Phasellus a est.</s><s>Phasellus magna.</s><s>In hac habitasse platea dictumst.</s><s>Curabitur at lacus ac velit ornare lobortis.</s><s>Curabitur a felis in nunc fringilla tristique.</s><s>Morbi mattis ullamcorper velit.</s><s>Phasellus gravida semper nisi.</s><s>Nullam vel sem.</s><s>Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam.</s><s>Sed hendrerit.</s><s>Morbi ac felis.</s><s>Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede.</s><s>Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi.</s><s>Nunc nulla.</s><s>Fusce risus nisl, viverra et, tempor et, pretium in, sapien.</s><s>Donec venenatis vulputate lorem.</s><s>Morbi nec metus.</s><s>Phasellus blandit leo ut odio.</s><s>Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem.</s><s>Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis.</s><s>In auctor lobortis lacus.</s><s>Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna.</s><s>Vestibulum ullamcorper mauris at ligula.</s><s>Fusce fermentum.</s><s>Nullam cursus lacinia erat.</s><s>Praesent blandit laoreet nibh.</s><s>Fusce convallis metus id felis luctus adipiscing.</s><s>Pellentesque egestas, neque sit amet convallis pulvinar, justo nulla eleifend augue, ac auctor orci leo non est.</s><s>Quisque id mi.</s><s>Ut tincidunt tincidunt erat.</s><s>Etiam feugiat lorem non metus.</s><s>Vestibulum dapibus nunc ac augue.</s><s>Curabitur vestibulum aliquam leo.</s><s>Praesent egestas neque eu enim.</s><s>In hac habitasse platea dictumst.</s><s>Fusce a quam.</s><s>Etiam ut purus mattis mauris sodales aliquam.</s><s>Curabitur nisi.</s><s>Quisque malesuada placerat nisl.</s><s>Nam ipsum risus, rutrum vitae, vestibulum eu, molestie vel, lacus.</s><s>Sed augue ipsum, egestas nec, vestibulum et, malesuada adipiscing, dui.</s><s>Vestibulum facilisis, purus nec pulvinar iaculis, ligula mi congue nunc, vitae euismod ligula urna in dolor.</s><s>Mauris sollicitudin fermentum libero.</s><s>Praesent nonummy mi in odio.</s><s>Nunc interdum lacus sit amet orci.</s><s>Vestibulum rutrum, mi nec elementum vehicula, eros quam gravida nisl, id fringilla neque a.</s><break strength="x-weak" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input SSML (${ssmlExampleLargeMultipleParagraphs.length} characters) goes above the hardLimit`, () => {
    const ssml = ssmlExampleLargeMultipleParagraphs;

    const expected = [
      '<speak>Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. Vivamus elementum semper nisi.<break strength="x-weak" />Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum. Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem.<break strength="x-weak" />Maecenas nec odio et ante tincidunt tempus. Donec vitae sapien ut libero venenatis faucibus. Nullam quis ante. Etiam sit amet orci eget eros faucibus tincidunt. Duis leo. Sed fringilla mauris sit amet nibh. Donec sodales sagittis magna. Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero. Fusce vulputate eleifend sapien. Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus. Nullam accumsan lorem in dui. Cras ultricies mi eu turpis hendrerit fringilla.<break strength="x-weak" />Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia. Nam pretium turpis et arcu. Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum. Sed aliquam ultrices mauris. Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris. Praesent adipiscing. Phasellus ullamcorper ipsum rutrum nunc. Nunc nonummy metus. Vestibulum volutpat pretium libero. Cras id dui. Aenean ut eros et nisl sagittis vestibulum. Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede.<break strength="x-weak" />Sed lectus. Donec mollis hendrerit risus. Phasellus nec sem in justo pellentesque facilisis. Etiam imperdiet imperdiet orci. Nunc nec neque. Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi. Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo. Maecenas malesuada. Praesent congue erat at massa. Sed cursus turpis vitae tortor. Donec posuere vulputate arcu. Phasellus accumsan cursus velit.<break strength="x-weak" />Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci. Phasellus consectetuer vestibulum elit. Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc. Vestibulum fringilla pede sit amet augue. In turpis. Pellentesque posuere. Praesent turpis. Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus. Donec elit libero, sodales nec, volutpat a, suscipit non, turpis. Nullam sagittis.<break strength="x-weak" />Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus. Ut varius tincidunt libero. Phasellus dolor. Maecenas vestibulum mollis diam. Pellentesque ut neque. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. In dui magna, posuere eget, vestibulum et, tempor auctor, justo. In ac felis quis tortor malesuada pretium. Pellentesque auctor neque nec urna.<break strength="x-weak" />Proin sapien ipsum, porta a, auctor quis, euismod ut, mi. Aenean viverra rhoncus pede. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Ut non enim eleifend felis pretium feugiat. Vivamus quis mi. Phasellus a est. Phasellus magna. In hac habitasse platea dictumst. Curabitur at lacus ac velit ornare lobortis. Curabitur a felis in nunc fringilla tristique. Morbi mattis ullamcorper velit. Phasellus gravida semper nisi. Nullam vel sem. Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam. Sed hendrerit. Morbi ac felis.</speak>',
      '<speak><break strength="x-weak" />Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede. Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi. Nunc nulla. Fusce risus nisl, viverra et, tempor et, pretium in, sapien. Donec venenatis vulputate lorem. Morbi nec metus. Phasellus blandit leo ut odio. Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem. Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis. In auctor lobortis lacus.<break strength="x-weak" />Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna. Vestibulum ullamcorper mauris at ligula. Fusce fermentum. Nullam cursus lacinia erat. Praesent blandit laoreet nibh. Fusce convallis metus id felis luctus adipiscing. Pellentesque egestas, neque sit amet convallis pulvinar, justo nulla eleifend augue, ac auctor orci leo non est. Quisque id mi. Ut tincidunt tincidunt erat. Etiam feugiat lorem non metus. Vestibulum dapibus nunc ac augue. Curabitur vestibulum aliquam leo. Praesent egestas neque eu enim. In hac habitasse platea dictumst. Fusce a quam. Etiam ut purus mattis mauris sodales aliquam. Curabitur nisi. Quisque malesuada placerat nisl. Nam ipsum risus, rutrum vitae, vestibulum eu, molestie vel, lacus. Sed augue ipsum, egestas nec, vestibulum et, malesuada adipiscing, dui. Vestibulum facilisis, purus nec pulvinar iaculis, ligula mi congue nunc, vitae euismod ligula urna in dolor. Mauris sollicitudin fermentum libero. Praesent nonummy mi in odio. Nunc interdum lacus sit amet orci. Vestibulum rutrum, mi nec elementum vehicula, eros quam gravida nisl, id fringilla neque a.<break strength="x-weak" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input SSML (${ssmlExampleLargeMultipleParagraphsWithSentences.length} characters) goes above the hardLimit`, () => {
    const largeExample = ssmlExampleLargeMultipleParagraphsWithSentences;

    const expected = [
      '<speak><s>Lorem ipsum dolor sit amet, consectetuer adipiscing elit.</s><s>Aenean commodo ligula eget dolor.</s><s>Aenean massa.</s><s>Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.</s><s>Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem.</s><s>Nulla consequat massa quis enim.</s><s>Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu.</s><s>In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo.</s><s>Nullam dictum felis eu pede mollis pretium.</s><s>Integer tincidunt.</s><s>Cras dapibus.</s><s>Vivamus elementum semper nisi.</s><s>Aenean vulputate eleifend tellus.</s><s>Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim.</s><s>Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus.</s><s>Phasellus viverra nulla ut metus varius laoreet.</s><s>Quisque rutrum.</s><s>Aenean imperdiet.</s><s>Etiam ultricies nisi vel augue.</s><s>Curabitur ullamcorper ultricies nisi.</s><s>Nam eget dui.</s><s>Etiam rhoncus.</s><s>Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum.</s><s>Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem.</s><s>Maecenas nec odio et ante tincidunt tempus.</s><s>Donec vitae sapien ut libero venenatis faucibus.</s><s>Nullam quis ante.</s><s>Etiam sit amet orci eget eros faucibus tincidunt.</s><s>Duis leo.</s><s>Sed fringilla mauris sit amet nibh.</s><s>Donec sodales sagittis magna.</s><s>Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero.</s><s>Fusce vulputate eleifend sapien.</s><s>Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus.</s><s>Nullam accumsan lorem in dui.</s><s>Cras ultricies mi eu turpis hendrerit fringilla.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia.</s><s>Nam pretium turpis et arcu.</s><s>Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum.</s><s>Sed aliquam ultrices mauris.</s><s>Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris.</s><s>Praesent adipiscing.</s><s>Phasellus ullamcorper ipsum rutrum nunc.</s><s>Nunc nonummy metus.</s><s>Vestibulum volutpat pretium libero.</s><s>Cras id dui.</s><s>Aenean ut eros et nisl sagittis vestibulum.</s><s>Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede.</s><s>Sed lectus.</s><s>Donec mollis hendrerit risus.</s><s>Phasellus nec sem in justo pellentesque facilisis.</s><s>Etiam imperdiet imperdiet orci.</s><s>Nunc nec neque.</s><s>Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi.</s><s>Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo.</s><s>Maecenas malesuada.</s><s>Praesent congue erat at massa.</s><s>Sed cursus turpis vitae tortor.</s><s>Donec posuere vulputate arcu.</s><s>Phasellus accumsan cursus velit.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci.</s><s>Phasellus consectetuer vestibulum elit.</s><s>Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc.</s><s>Vestibulum fringilla pede sit amet augue.</s><s>In turpis.</s><s>Pellentesque posuere.</s><s>Praesent turpis.</s><s>Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus.</s><s>Donec elit libero, sodales nec, volutpat a, suscipit non, turpis.</s><s>Nullam sagittis.</s><s>Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus.</s><s>Ut varius tincidunt libero.</s><s>Phasellus dolor.</s><s>Maecenas vestibulum mollis diam.</s><s>Pellentesque ut neque.</s><s>Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.</s></speak>',
      '<speak><s>In dui magna, posuere eget, vestibulum et, tempor auctor, justo.</s><s>In ac felis quis tortor malesuada pretium.</s><s>Pellentesque auctor neque nec urna.</s><s>Proin sapien ipsum, porta a, auctor quis, euismod ut, mi.</s><s>Aenean viverra rhoncus pede.</s><s>Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.</s><s>Ut non enim eleifend felis pretium feugiat.</s><s>Vivamus quis mi.</s><s>Phasellus a est.</s><s>Phasellus magna.</s><s>In hac habitasse platea dictumst.</s><s>Curabitur at lacus ac velit ornare lobortis.</s><s>Curabitur a felis in nunc fringilla tristique.</s><s>Morbi mattis ullamcorper velit.</s><s>Phasellus gravida semper nisi.</s><s>Nullam vel sem.</s><s>Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam.</s><s>Sed hendrerit.</s><s>Morbi ac felis.</s><s>Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede.</s><s>Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi.</s><s>Nunc nulla.</s><s>Fusce risus nisl, viverra et, tempor et, pretium in, sapien.</s><s>Donec venenatis vulputate lorem.</s><s>Morbi nec metus.</s><s>Phasellus blandit leo ut odio.</s><s>Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem.</s><s>Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis.</s><s>In auctor lobortis lacus.</s><s>Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna.</s><s>Vestibulum ullamcorper mauris at ligula.</s><s>Fusce fermentum.</s><s>Nullam cursus lacinia erat.</s><s>Praesent blandit laoreet nibh.</s><s>Fusce convallis metus id felis luctus adipiscing.</s><s>Pellentesque egestas, neque sit amet convallis pulvinar, justo nulla eleifend augue, ac auctor orci leo non est.</s><s>Quisque id mi.</s><s>Ut tincidunt tincidunt erat.</s><s>Etiam feugiat lorem non metus.</s><s>Vestibulum dapibus nunc ac augue.</s><s>Curabitur vestibulum aliquam leo.</s><s>Praesent egestas neque eu enim.</s><s>In hac habitasse platea dictumst.</s><s>Fusce a quam.</s><s>Etiam ut purus mattis mauris sodales aliquam.</s><s>Curabitur nisi.</s><s>Quisque malesuada placerat nisl.</s><s>Nam ipsum risus, rutrum vitae, vestibulum eu, molestie vel, lacus.</s><s>Sed augue ipsum, egestas nec, vestibulum et, malesuada adipiscing, dui.</s><s>Vestibulum facilisis, purus nec pulvinar iaculis, ligula mi congue nunc, vitae euismod ligula urna in dolor.</s><s>Mauris sollicitudin fermentum libero.</s><s>Praesent nonummy mi in odio.</s><s>Nunc interdum lacus sit amet orci.</s><s>Vestibulum rutrum, mi nec elementum vehicula, eros quam gravida nisl, id fringilla neque a.</s><break strength="x-weak" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(largeExample);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input SSML (${ssmlExampleLargeArticle.length} characters) goes above the hardLimit`, () => {
    const largeExample = ssmlExampleLargeArticle;

    const expected = [
      '<speak><s>Als ich den später so berühmten und angebeteten Fra Celeste zum ersten Male predigen hörte, war er erst in den Ortschaften bekannt, die im Umkreise seines Klosters lagen.</s><s>Ich war in die Kirche eingetreten, weil ich so ungewöhnlich viele Leute hingehen sah, aber ich erwartete mir eigentlich nichts anderes, als die Nase zu rümpfen; denn ich war hochmütig wegen der Bildung, die ich auf den hohen Schulen gesammelt hatte, und glaubte durchaus nicht an diese Genialen, die es vermöge angeborener Kräfte mit den bestgeschulten Geistern aufnehmen können.</s><s>Von einem, der erst ein Bäckerjunge gewesen, dann ein Mönch geworden war, glaubte ich, daß er höchstens dem dummen Pöbel durch Lärm, Schlagwörter und leeren Bombast imponieren könne.</s><s>Als er nun aber, dieser wundervolle Mann, in Begleitung mehrerer Geistlicher, denen er um einige Schritte voranging, durch das geöffnete Portal eintrat, fühlte ich mich sogleich gesichert und erhoben, was ich schwerlich begründen könnte, denn bei der Dunkelheit, die in der tiefen alten Kirche herrschte, vermochte ich sein Äußeres nicht zu unterscheiden.</s><s>Auch als er auf der Kanzel stand, sah ich nur, daß er eine hochgewachsene kräftige Gestalt hatte, die sich mit Unbefangenheit, aber nicht häufig bewegte, und daß die Form seines Gesichtes schön, breit und stark war.</s><s>Obgleich er schnell gegangen war, fing er sofort, ohne daßman etwas von Atemnot bemerkt hätte, zu sprechen an mit dieser einzigen, unerschöpflichen Stimme, die aus dem Busen eines Gottes zu kommen schien.</s><s>Eine Weile sprach er schlicht, gleichförmig, mit mittlerer Stärke, bis er sowohl von den immer rascher aufquellenden Einfällen wie auch vom Klange seiner Stimme berauscht wurde.</s><s>Dann behandelte er sie wie ein Instrument, Geige oder Flöte, worauf er phantasierte; etwas vorgebogen, mit verzückten Augen schien er den Tönen nachzublicken, als wären es schimmernde Paradiesvögel, die die schwarzen Pfeiler umschwebten und das Gewölbe mit Musik erfüllten.</s><s>Das Merkwürdige war nun, daß die Predigt keineswegs nur ein wohlklingender Gesang war, sondern klar geordnet und voll eigenartiger Gedanken und Betrachtungen.</s><s>Sie handelte von der Sonntagsheiligung, und zwar zuerst von den Belustigungen, mit denen gewöhnlich der Feiertag verbracht wird, hernach von dem eigentlichen Sinn und Wert der gebotenen Ruhe.</s><s>Da war kein borniertes Eifern oder Scheinheiligkeit oder Tugendsalbung, nichts, worüber man hätte lachen oder was man hätte leicht widerlegen können, sondern ritterliche Verachtung gemeiner Lust.</s><s>»Gott hat uns zu Herren der Erde geschaffen,« sagte er unter anderem, »sollen wir uns von ihr knechten lassen für den vergänglichen Bettel, mit dem sie uns dingen will?</s><s>Wer wäre ich, wenn die Sklavenumarmung der Dirne Welt meine Sehnsucht stillte?</s><s>Regieren will ich die Erde von meinem Platz aus, nach meinen Kräften, als der König, der ich geschaffen bin, sechs Tage lang.</s><s>Aber am siebenten will ich den Herrn suchen, der über mir ist, und ihm dienen.</s><s>Denn auch wir möchten vergehen und anbeten.« Indem er dies letzte mit einemunbeschreiblichen Schmelz der Stimme sagte, sah ich zum ersten Male sein herzliches Lächeln, wobei die Zähne schimmerten wie Mondschein, der durch eine dunkle Wolke fällt.</s><s>Dann, nach einer kleinen Pause, fuhr er lauter fort, daß es klang, wie wenn ein Frühlingsdonner über die Himmelswölbung rollt: »Kniet mit mir und betet an!« worauf sich alle niederwarfen.</s><s>Die meisten blieben in dieser Stellung bis zum Ende der Predigt und schienen froh, ihrer leidenschaftlichen Inbrunst dadurch einen Ausdruck geben zu dürfen.</s><s>Durch die Reihen der Menschen, die sich an ihn drängten, als er die Kirche verließ, um sein Kleid oder seine Hände zu berühren, ging er mit aufrichtiger, ganz ungesuchter Gleichgültigkeit, obwohl es namentlich Frauen waren und seine Jugend eine gewisse Reizbarkeit gegenüber dem weiblichen Geschlecht entschuldigt hätte.</s></speak>',
      '<speak><s>Aber man sah wohl, daß es diesem Liebling Gottes nicht einmal einen Kampf kostete, derartige Versuchungen zu überwinden.</s><break strength="x-weak" /><s>Ich erinnere mich, daß inzwischen die Sonne untergegangen war und ein wonnig kühler, mehr zu empfindender als sichtbarer Flor sich über die graue Kirche, den Turm und alle Gegenstände herabsenkte.</s><s>Der Himmel schien mir höher zu sein, als ich ihn je gesehen hatte, und ich stand noch und schaute hinein, nachdem sich die Menge längst verlaufen hatte.</s><s>Allmählich war es mir, als höbe ich mich von der Erde und schwebte langsam nach oben, getragen von einem mächtigen, befreundeten Element, das, wie ich genau wußte, Fra Celeste regierte.</s><s>Seit diesem Abend brachte ich den Bruder nicht mehr aus meinem Sinn, und als ich kurz darauf erfuhr, daß er einen Sekretär suchte, der seine Korrespondenzen und weltlichen Angelegenheitenüberhaupt besorge, lief ich ohne Zögern zu ihm in der Meinung, die Vorsehung habe mich eigens für dieses Amt auserlesen, wie ehemals den guten Parsival für den Gral.</s><s>Ich wurde sogleich vorgelassen, der himmlische Mann betrachtete mich eine Weile aufmerksam, strich liebkosend über meine Wangen, erkundigte sich nach meinem Alter, verwunderte sich, daß ich schon dreiundzwanzig Jahre zählte, denn ich gliche, sagte er, mehr einem kleinen Fräulein von sechzehn Lenzen, und äußerte Zweifel, ob ich auch imstande sei, seine Geschäfte zu führen.</s><s>Er selbst nämlich, obschon er wie ein Erzengel mit Nachtigallen- und Adlerzungen redete, konnte weder fließend lesen noch schreiben, und da er schon damals zahlreiche Briefe empfing, die ihm gleichgültig oder gar lästig waren, suchte er jemand, der sie nach seinem Gutdünken schicklich beantwortete und ihn selbst nur in Ausnahmefällen damit behelligte.</s><s>Ich zählte ihm alle Studien auf, die ich gemacht hatte, was ihn sehr zu befriedigen schien, und er gab mir einige Briefe, die er kürzlich erhalten hatte, damit ich sie abfertige, wie es mir gut scheine, und ließ mich an seinem Schreibtisch allein.</s><break strength="x-weak" /><s>Der erste Brief, den ich öffnete, war von einer Frau, die einen starken Trieb fühlte, ins Kloster zu gehen, ihr Mann, schrieb sie, wollte sie aber nicht gehen lassen; ob es gottgefälliger sei, dem himmlischen Herrn zuliebe dem irdischen ungehorsam zu sein oder bei dem letzteren auszuharren.</s><s>Ich beantwortete den Brief folgendermaßen: »Frau, wenn Sie ins Kloster gehen, entwenden Sie Gott eine Seele, nämlich die Ihres Mannes, der mit Weltlust voll zu sein scheint und ohne Sie vielleicht zur Hölle fahren müßte</s><s>Bessern Sie ihn durch das Beispiel Ihrer Tugend, bis er selbst vor den irdischen Plagen sich ins Kloster zu flüchten sehnt, und gehen Sie dann in Gottes Namen miteinander.« Der zweite Brief war ebenfalls von einer Dame, welche, um im Glauben gestärkt und vom Verderben errettet zu werden, eine Unterredung mit dem heiligen Bruder wünschte und ihn auf die Dämmerstunde eines bestimmten Tages zu sich einlud.</s><s>Diesen Brief hielt ich für das beste, unbeantwortet zu zerreißen.</s><break strength="x-weak" /><s>Der dritte war von des Bruders Abte, der sich beklagte, daß er schon lange ohne die schuldige Berichterstattung bleibe, ferner, daß der Bruder, wie ihm zu Ohren gekommen sei, bei seinen Predigten oft ins Blaue fahre und nicht daran denke, der alleinseligmachenden katholischen Kirche die Ehre zu geben.</s><s>Nach einigem Bedenken antwortete ich so: »Hochehrwürden, über das, was ich predige, bin ich nicht Herr, denn es ist Eingebung, und zwar, wie ich glaube und hoffe, des guten, nicht des bösen Geistes.</s><s>Daß Hochehrwürden solange keine Nachricht von mir empfangen haben, ist die Schuld meines Sekretärs, den ich wegen dieser Nachlässigkeit scharf getadelt habe.« Eben als ich dies geschrieben hatte, trat Fra Celeste wieder ein und verlangte, meine Arbeit zu prüfen.</s><s>Nachdem er die beiden Briefe gelesen hatte, lobte er mich sehr und bat mich, bei ihm zu bleiben; er sei zufrieden mit mir.</s></speak>',
      '<speak><break strength="x-weak" /><s>Von diesem Augenblick an blieb ich der Gefährte dieses einzigen Mannes, hatte mein Zimmer unmittelbar neben dem seinigen und begleitete ihn auf seinen Reisen; denn nun kam die Zeit, wo alle Städte unseres Landes wetteiferten, ihn in ihren Mauern zu empfangen.</s><s>Die erste Reise, die wirzusammen unternahmen, ging nach seiner Vaterstadt, die er wiederzusehen wünschte.</s><s>Dort ereignete es sich, daß bei der ersten Predigt, die er hielt, eine vornehme, reich und edel gekleidete Dame, während er in einer Pause angab, worüber er das nächste Mal predigen würde, von einer Ohnmacht angewandelt wurde.</s><s>Sie erholte sich aber sogleich und drängte sich am Schlusse, obschon es ihr sichtlich ungewohnt und zuwider war, sich im Volksgewühl zu bewegen, dicht zu Fra Celeste hin, der sie mit einer Gebärde des Abscheues oder Zornes zurückstieß.</s><s>Als ich mit dem Bruder allein war, sagte ich hierauf bezüglich: »Nun glaube ich wirklich, daß uns der Teufel in Gestalt von Frauen versucht; denn wenn Sie nicht den verkappten Satan in jener lieblichen Dame durchschaut haben, begreife ich nicht, wie Sie die Kraft haben konnten, sie so hart zurückzustoßen.«</s><break strength="x-weak" /><s>Damals hatte ich den Bruder noch nicht in der fliegenden Wut gesehen, daher erschrak ich nicht wenig, als seine stillen, schattigen Augen plötzlich groß und fürchterlich wurden und er so mit gezückten Schwertblicken hart vor mich trat.</s><s>»Ist das deine Andacht,« rief er, »daß du nach den Frauen umherschaust?</s><s>O ekelhafte Schwäche und Verderbnis dieses Kotleibes!</s><s>Errötest du nicht über deine Zuchtlosigkeit, wenn du dein keusches Jugendgesicht im Spiegel betrachtest, das Gott dir gegeben hat?</s><s>Wenn du es besudeln solltest mit Gedanken oder Taten, verlasse dich auf mich, daß ich dich umbringe mit diesen meinen Händen.« Dabei war sein ganzer Körper gewaltsam gespannt, und seine Hände, die ich bisher nur schwermütig ruhen oder einen Nachdruck der Rede mit gleichgültig stolzerGebärde hatte begleiten sehen, bekamen etwas so Ehernes und Unerbittliches, daß ich, ohne es zu wollen, ein wenig vor ihm zurückwich.</s><s>Aber trotzdem es mir nicht recht geheuer war, entzückte mich der Anblick des feuerspeienden Mannes, wie man bei Ungewittern, Wolkenbrüchen und Stürmen zugleich vor Angst und Wonne schaudert.</s><s>Ich segnete den Umstand, daß Küsse kein rotes Mal oder sonst eine Spur zurücklassen, denn ich zweifelte nicht, der Bruder würde mich stehenden Fußes erschlagen, wenn er wüßte, daß ich die Lockungen meines unruhigen und naschhaften Herzens zuweilen erfolglos bekämpft hatte.</s><break strength="x-weak" /><s>Übrigens legte sich das Wetter, so schnell es gekommen war, vielleicht unter dem Eindruck meiner erschrockenen und liebenden Blicke; wenigstens behandelte mich Fra Celeste im Verlaufe des Tages mit so engelgleicher Zartheit, daß ich mich mitten im Himmel wähnte und mir schwur, künftig dem erhabenen Beispiel meines Herrn nachzueifern und aller Frauenliebe zu entsagen, um mich ihm ganz ohne Einschränkung zu ergeben.</s><break strength="x-weak" /><s>Am folgenden Morgen fiel mir unter den eingelaufenen Briefen sogleich einer durch die reizende, sichtlich einer Frau gehörige Handschrift auf.</s><s>Die Buchstaben waren kühn, schlank, behend und prächtig und tanzten in so anmutigen Neigungen über das Papier, als ob sie mit den Augen des Lesers kokettieren wollten.</s><s>Zu meinem wachsenden Erstaunen las ich nun folgendes: »Dolfin, ich habe Dich erkannt und Du hast mich von Dir gestoßen.</s><s>Als ich Dich sah, verwandelte sich mein Blut in Tränen und meine Tränen wurden Blut.</s><s>Liebst Du mich denn nicht mehr ?</s><s>Ich liebe Dich so sehr!</s><s>Ich will Heimat, Eltern und Mann verlassenund die Deine sein.</s><s>Laß mich, sag nicht nein zu meinem Herzen.</s><s>Wenn Du mich rufst, komm ich und bleibe bei Dir ewig.</s><s>Aglaia.«</s><break strength="x-weak" /><s>Dieser Schrei der Liebe, obschon er mir nicht galt, erschütterte mein Herz, und ich saß lange und starrte in den Brief hinein.</s></speak>',
      '<speak><s>Die stürmischen, geschmeidigen Buchstaben schienen mir wie arabische Rosse über eine Wüste zu jagen, ich hörte den Rhythmus ihrer aufschlagenden Hufe, versank darüber in Träumerei, und ehe ich mir&apos;s versah, hatte ich einen Reim erdacht als Antwort auf die geheimnisvolle Liebesklage.</s><s>Der Vers hieß so:</s><break strength="x-weak" /><s>Durch den Himmel hin wandr ich allein,</s><break strength="x-weak" /><s>Hast du Flügel, o komm und sei mein;</s><break strength="x-weak" /><s>Sei des einsamen Mondes Weib,</s><break strength="x-weak" /><s>Wie ein Stern blitzt dein silberner Leib –</s><break strength="x-weak" /><s>Ich hatte gerade die letzten Worte niedergeschrieben, als Fra Celeste bei mir eintrat und, wie er häufig zu tun pflegte, über meine Schulter sah.</s><s>Es gelang mir nicht mehr, das Geschriebene zu verdecken, und so kam es, daß er sowohl meinen Vers wie auch den Brief las, der ihn veranlaßt hatte.</s><s>Wenn ich auch gleich erraten hatte, daß es sich hier um nichts Geringes handle, überraschte mich doch, was ich nun wahrnahm.</s><s>Eine gewaltige Eiche, in die der Blitz fährt, die Feuer fängt und wie eine vom Sturme hin und her gewehte Fackel gegen den Himmel lodert, der möchte ich Fra Celeste vergleichen, als er den Inhalt des verhängnisvollen Schreibens ins Herz gefaßt hatte.</s><s>So hing aber alles zusammen.</s><break strength="x-weak" /><s>Fra Celeste, der mit seinem weltlichen Namen Dolfin hieß,war als armes, elternloses Kind zwischen fremden Leuten aufgewachsen und wurde, nachdem er kaum das Notdürftigste in der Schule erlernt hatte, einem Bäcker in die Lehre gegeben.</s><s>Als er etwa achtzehn Jahre alt war, kam er wie nach einem dumpfen Traume zu sich und verspürte einen heißhungrigen Trieb, seine Kräfte an etwas Ungeheuerem zu üben.</s><s>Auf ein bestimmtes Ziel richtete sich sein Ehrgeiz nicht, auch hätte er die Bildung dazu nicht besessen, aber alles, was er um sich her sah, schien ihm verächtlich und viel zu klein für den Riesenmut seines Herzens.</s><s>So war er beschaffen, als er sich in ein vornehmes und reiches Fräulein verliebte, in deren elterliches Haus sein Beruf ihn führte.</s><s>Wenn nicht ein überirdischer Geist in ihm gewesen wäre, möchte es ihm als einem armen Bäckerburschen niemals gelungen sein, eine Beziehung zu dem Fräulein herzustellen, besonders da er nicht einmal auffallend schön war, außer daß der später so wohlredende, damals noch blöde Mund, schön, stark und rot, die glänzendsten und vollkommensten Zähne besaß und mit seinem Lächeln, das freilich nur selten erschien, alle Herzen bezaubern konnte.</s><break strength="x-weak" /><s>Die Bekanntschaft hatte Fra Celeste so angebahnt, daß er einmal, anstatt das Haus zu verlassen, an der Gartentür stehenblieb und unentwegt in den Garten hineinblickte, wo das Fräulein allein auf einer Bank saß.</s><s>Etwas beunruhigt, fragte sie endlich, worauf er noch warte, und als er, ohne sich zu besinnen, sagte: auf eine von den Rosen, die an den hohen Stämmen längs eines Gebüsches hinter der Bank blühten, bog sie sich rückwärts, pflückte eine, ging zu ihm hin und gab sie ihm, der kurz dankte und fortging.</s><s>Erst dannwunderte sie sich über das, was sie, überrascht von der Seltsamkeit eines solchen Betragens, gleichsam unbewußt getan hatte, und lachte sich selbst aus; nichtsdestoweniger suchte sie eine Gelegenheit, den wunderlichen Bäcker wiederzusehen.</s><s>Ihre Phantasie wurde vollends angeregt, als er kleine Gespräche mit ihr zu führen begann; denn nun verspürte sie den Genius, der in seiner Brust wohnte.</s><s>Obgleich er wenig oder nichts gelernt hatte, wußte er seine Einfälle und Beobachtungen so zu äußern, daß das Geringste bedeutend erschien, ja, er konnte Albernheiten, wie daß der Himmel blau und das Wasser naß sei, so vortragen, daß man aufhorchte und es tagelang mit sich herumtrug.</s><s>Anfänglich gefiel es dem Fräulein, mit ihrem sonderbaren Verehrer von der Familie und den Bekannten geneckt zu werden, allmählich aber vermied sie es, von ihm zu sprechen und mit ihm gesehen zu werden, und wenn man sie dennoch auf ihn anredete, konnte sie spitzige Antworten geben, wie daß der arme Bäckerjunge gescheiter, stolzer und edler sei als ihre hochgeborene Umgebung.</s></speak>',
      '<speak><s>Nun zeigten ihre Eltern Mißfallen an dem Verkehr, was wiederum ihre Zuneigung verstärkte, und es kam dazu, wovon mir das einzelne nicht bekannt ist, daß Liebesworte und Küsse zwischen den beiden jungen Leuten gewechselt wurden.</s><s>Zugleich aber bemächtigte sich des Fräuleins heimliches Schuldgefühl, so daß sie ihn bald hochmütig vermied, bald mit leidenschaftlicher Hingebung suchte, während er dieser Verwirrung gegenüber sich immer sicherer und männlicher fühlte.</s><s>Des Fräuleins Eltern, deren Geschlechtsnamen ich nicht nennen will – sie selbst hieß Aglaia – erfuhren oder ahnten, wie gefährlich diese Spielerei mit der Zeit geworden war, undbeschlossen, damit nicht noch Ärgeres geschehe, ihr Kind so schnell wie möglich standesgemäß zu verheiraten.</s><s>Sie warfen ihr Auge auf den Grafen, den ich später selbst kennengelernt habe, einen großen, prächtigen Mann mit geräumigen Augen und geringelten schwarzen Haaren, die angenehm glänzten, so daß er wohl schön genannt zu werden verdiente.</s><s>Er war aber auf diese Vorzüge weit weniger eitel als auf seine Geistesgaben und Glücksgüter, in welchen letzteren er die ersten zeigen wollte, insofern nämlich, daß er von allem das Schönste und Seltenste besaß.</s><s>Was allgemein geschätzt und kostbar war, schaffte er an, und so bewarb er sich auch um Aglaia, deren Witz, Eigenart, liebliches Wesen und körperliche Vorzüge er von allen Seiten hatte rühmen hören.</s><s>Da nun das Fräulein von den älteren und verständigen Leuten Tag für Tag das Los einer armen Bäckersfrau in grellen Farben beschreiben hörte, verlor ihr Herz die kindische Zuversicht, und sie fing an, sich für eine überspannte, romanlesende Person zu halten, daß sie die gute, klare, gesunde Wirklichkeit so hatte übersehen können.</s><s>Sie sah sich von Zichorien- und Kartoffelnahrung aufgeschwemmt und unförmig geworden, von vielen Kindern umzetert, die gesäugt, gewaschen und geprügelt sein wollten.</s><s>Von Dolfin waren ihr hauptsächlich die weißen Arme und Hände gegenwärtig, die sie bald mit klebrigen Liebkosungen, bald mit Schlägen verfolgten; denn ohne das, hatte man ihr gesagt, würde es durchaus nicht abgehen.</s><s>Alle ihre Lieblichkeiten, die weiche Zartheit ihrer Hände, ihr träumerisches oder ausgelassenes Wesen, ihren Aberwillen gegen das Alltägliche und Gemeine würde man ihr dort zum Vorwurf machen und allmählich austreiben, bissie sich in die platte Form des Durchschnittspöbels hineingebückt hätte.</s><s>Wenn sie sich überlegte, daß sie von allen ihren Zweifeln auch nicht ein Wort zu Dolfin hätte sagen mögen, fühlte sie, wie fremd sie einander waren; sie gehörten zwei verschiedenen Welten an und hatten sich nur auf einem Inselchen des Grenzflusses ein Stelldichein gegeben.</s><s>Dies alles bedenkend, glitt sie dem behaglichen Schicksal, das man ihr zubereitet hatte, mit guter Miene in den Schoß, ja, um es besser bewerkstelligen zu können, verliebte sie sich sogar unwillkürlich ein wenig in ihren Bräutigam, wozu ihr der Ärger über sein pompöses Betragen und die Reue über die Unarten, die sie ihm zufügte, das beste Mittel waren.</s><s>Zwar hatte sie noch einige Begegnungen mit Dolfin, der, als er erfuhr, was im Werke war, den Jammer seines zärtlichen Herzens hinter grobem Zürnen verbergend, sie fürchterlich anherrschte und anfänglich dadurch das Feuer ihrer Liebe neu anfachte.</s><s>Aber als sich das rechtmäßige Gefühl für ihren künftigen Gatten mehr und mehr ausbildete, schämte sie sich der Worte und Küsse, die sie mit Dolfin gewechselt hatte, und wich ihm aus; und als er trotzdem ihren Weg kreuzte, wurde sie in ihrer Angst schnippisch und böse und nannte ihn einen Narren, daß er eine mutwillige Spielerei für Ernst genommen habe.</s><break strength="x-weak" /><s>Dies war der Anlaß, der ihn in das Kloster führte.</s><s>Der bloße Liebeskummer hätte ihm die Welt vielleicht nicht dermaßen verleidet, aber die Tatsache, daß sein Nebenbuhler das Glück davongetragen hatte, weil er reich war, daß man also auch Menschenseelen, das edelste und heiligste Besitztum, sich erkaufen kann, das füllte ihm die Brust bis andie Kehle mit Groll und Verachtung, dazu mit Stolz, weil er solcher Niedertracht und Schwäche nicht unterworfen war.</s></speak>',
      '<speak><s>Er hätte stundenlang auf geprägtem Golde gehen können, ohne sich einmal danach zu bücken, ja ohne nur hinzublicken.</s><s>Nun entwickelte sich die Krankheit, zu der er von jeher Anlage in sich gehabt hatte, mehr und mehr, die nämlich, daß er sich einsam fühlte, einsam wie eine göttliche Seele in einem unreinen Körper, einsam wie ein übriggebliebenes Tier in einem verschütteten, ausgestorbenen Dorfe.</s><s>Er litt unter dieser Einsamkeit in Gesellschaft von vielen ebensogut, wie wenn er sich allein mit Beten oder häuslicher Arbeit beschäftigte, wie auch wenn er mit mehreren ein Gespräch führte; es geschah ihm dann wohl, daß die Menschen und Gegenstände von ihm wegzurücken schienen in die leere Unendlichkeit, wo er sie nicht mehr erreichen konnte, obgleich er sie mit derselben Deutlichkeit wie vorher sah.</s><s>Denn auch die Mönche trieben es nicht viel anders als die Leute draußen in der Welt, ließen sich nichts abgehen, bekümmerten sich trotz allen Betens und Singens viel mehr um irdische als um himmlische Dinge, kurz, er konnte sich durchaus nicht vorstellen, daß Gott sein Reich mit diesen Maultieren bevölkern möchte.</s><s>Da er nun auch diese Menschen, deren Profession doch die überirdische war, so gemeinplätzig und überflüssig fand, wurde Zorn gegen alle Sterblichen in ihm herrschend, die das Ebenbild Gottes durch den Schmutz schleiften, und er wurde, so schweigsam er sonst auch war, stets beredt, wenn er sich darüber auslassen konnte.</s><s>Dies zeigte sich, als er an einem Bußtage einmal an der Reihe war, zu predigen, und der Sturm seiner Rede so stark daherrauschte, daßauch die schläfrigsten Mönche in ihrem dickhäutigen Herzen darunter erbebten.</s><s>Der Abt suchte das neuentdeckte Talent sogleich zu verwerten, indem er ihn öffentlich reden ließ, und die Wirkung war denn auch über alle Erwartung groß.</s><s>Die Melancholie seines jungen Gesichts und die Süßigkeit seiner Stimme, die seine Worte auf Schwanenfittichen über die Häupter der Menge dahintrug, verführten die Sinne; dazu kam aber die Macht seiner Seele über die Seelen, die er aus dem Schlamm der Erde in den Äther emporriß.</s><s>Er wurde nun häufig zum Predigen in die benachbarten Dörfer und Städte geschickt, was dem Kloster hübsche Summen eintrug, da er der Regel gemäß wie alle anderen Mönche persönliches Eigentum nicht haben durfte, die Gemeinden aber reichlich zusammensteuerten, um ihn auf der Kanzel zu sehen.</s><s>Er fing nun an, eine berühmte Persönlichkeit zu werden, und weltliche Freuden und Ehren taten sich ihm überall auf: wo er hinkam, suchten die angesehensten Männer und Frauen seine Gesellschaft.</s><s>Er, der als Kind und Jüngling in jeder Hinsicht gedarbt hatte, hätte nun nach Herzenslust genießen können; anstatt dessen steigerte sich seine Geringschätzung aller Lust der Welt, je mehr sie sich ihm darbot, weil er den Schmerz seiner Sehnsucht, wenn er innewurde, daß sie durch nichts gesättigt werden konnte, um so empfindlicher spürte.</s><break strength="x-weak" /><s>Dies war der Zeitpunkt, wo ich ihn kennenlernte und wo er Aglaia wieder begegnete.</s><s>Aglaia hatte unterdessen, an der Seite ihres Gemahls, der an Geist wie an Körper immer fetter und bequemer geworden war, ohne Sorgen zwar, aber auch ohne innere Genugtuung gelebt.</s><s>Da ihr einziges Kind früh gestorben war und ihr ein dringender Grundsich ernstlich zu beschäftigen, fehlte, blieb ihr Muße genug, über den Lauf der Welt nachzudenken und zu träumen, und sie malte sich oft aus, wie es geworden wäre, wenn sie, anstatt sich von der kümmerlichen Menschenvorsicht binden zu lassen, Dolfin gefolgt wäre, der jetzt in seinem Kloster nicht einsamer war als sie in ihrem Palaste.</s><s>Von seinem Ruf als Buß- und Sittenprediger hatte sie, die sich für das Kirchliche nicht interessierte, nichts vernommen, und als sie die Ankündigung an den Häusermauern las, daß der berühmte Fra Celeste eine Predigt über den Kirchenbesuch und das Gebet halten werde, trat sie in die betreffende Kirche, in deren Nähe sie eben war, ein, ohne zu ahnen, wen sie wiedersehen sollte.</s></speak>',
      '<speak><s>Eigentlich hatte sie sich nur unterhalten und ausruhen, allenfalls etwas Neues hören wollen, aber die Rede des Mönches schüttelte ihr Herz gewaltig, wenn sie eben auch nicht Schritt für Schritt dem Gedankengange folgte, den er entwickelte; sie vertraute sich dem starken Fluge seiner Worte, der sie weit wegtrug über Länder und Meere in eine fremde und doch heimische Herrlichkeit.</s><s>Ihre Seele kehrte von der Reise zurück, als die Musik der Stimme über ihr plötzlich schwieg; dann räusperte sich Fra Celeste und gab in seinem gewöhnlichen, halblauten Sprechtone an, welches der Text seiner nächsten Predigt sein würde.</s><break strength="x-weak" /><s>Gleich das allererste Wort schlug wie ein Hammer an Aglaias Herz, die Erinnerung stürzte daraus hervor und füllte aufschwellend ihre Brust an, was sie zuerst als eine Wollust, dann mit Angst empfand, bis es sie zu ersticken drohte.</s><s>Es begab sich nun alles, was ich bereits erzählt habe: sie schrieb jenen Brief an Dolfin, erwartete aber die Antwort nicht,sondern erforschte seine Wohnung, verschaffte sich Zutritt zu ihm und stand auf der Schwelle seines Zimmers, ehe er noch die Empörung seines Blutes, die durch ihr Schreiben entstanden war, bemeistert hatte.</s><break strength="x-weak" /><s>Wenn ich diese Liebliche aus ihrem Grabe oder vielmehr aus ihrer himmlischen Verklärung auf einen Augenblick zurückrufen könnte, möchte ich sie noch einmal so sehen wie jenes erste Mal: die Augen voll Tränen, die Lippen zitternd vor Sehnsucht und Hoffnung, wie eine zaghafte Seele vor dem halboffenen Tore des Paradieses, goldig umflossen von dem himmlischen Lichte, das daraus hervorströmt, aber demütig und furchtsam zurückgebogen, wenn vielleicht der Engel mit dem Schwerte die Sünderin verscheuchen wollte.</s><s>Einige Augenblicke mochte sie so in schwebendem Zögern dagestanden haben, dann stürzte Fra Celeste auf sie zu, nahm sie in die Arme, preßte sie gegen seine Brust, bedeckte ihr tränenüberströmtes, glückseliges Gesicht mit Küssen, faßte ihren Kopf zwischen seine Hände und drückte ihn an seinen Hals, das alles in einem Nu und abwechselnd viele Male hintereinander, so daß ich dachte, es müsse der feinen, schlanken Frau Hören und Sehen vergehen.</s><s>Was sich weiter zwischen ihnen begab, kann ich aus eigener Anschauung nicht erzählen, denn ich hielt es für angemessen, mich zu entfernen; jedenfalls fand eine völlige Aussöhnung statt, und der Beschluß wurde gefaßt, sich künftig nicht mehr voneinander zu trennen, vorausgesetzt natürlich alle die Beschränkungen, die die Verhältnisse ihnen auferlegten.</s><break strength="x-weak" /><s>An diesem Tage predigte Fra Celeste von der himmlischen und irdischen Liebe so über alle Maßen herrlich undweihevoll, daß man das allumfassende Herz Gottes selbst in seinen Worten klopfen zu hören glaubte.</s><s>Einige junge Wüstlinge und viele leichtfertige Frauen legten in seine Hand das Gelübde künftiger Keuschheit ab, und es war gewiß nicht einer unter den Zuhörern, der sich nicht dessen geschämt hätte, worauf er sich bisher in seiner Verblendung etwas zugute zu tun pflegte.</s><break strength="x-weak" /><s>Der Ruhm des wundervollen Mannes nahm jetzt schnell zu, alle Zeitungen sprachen von ihm, seine Reden wurden gedruckt und gesammelt.</s><s>Wenn er sie nun zufällig irgendwo fand und las, mißfielen sie ihm in solchem Grade, daß er sie geradezu für abgeschmacktes, leeres Gewäsch erklärte und beschloß, seinen Geist auszuarbeiten und seine Kenntnisse zu vermehren, um ihnen einen gediegenen Gehalt zu geben.</s><s>Zu diesem Zwecke studierte er viele wissenschaftliche Werke, theologische, historische, philosophische, ja auch naturwissenschaftliche, und zwar las er nach kurzer Übung ein solches Buch schneller, als man ein Glas leichten Weines trinkt, ohne daß ihm etwas Wesentliches entgangen wäre.</s><s>Nun kamen von nah und fern Gelehrte, Professoren der Universitäten und die ersten Staatsmänner und zahlten hohe Preise für einen Platz in der Kirche, wenn Fra Celeste predigte, während seine Anziehungskraft für das niedere Volk sich nicht verminderte.</s></speak>',
      '<speak><s>Unter der Last der Gelehrsamkeit wurde sein Genius nicht lahm und steif; niemals gab es einen Prediger, der die gepflegtesten wie die vernachlässigtsten Geister gleich mächtig beherrschte.</s><break strength="x-weak" /><s>Was Aglaia betrifft, so entschloß sie sich kurz und gut, ihren Mann ein für allemal zu verlassen und uns zu begleiten oder wenigstens mit uns, wo immer wir uns längere Zeitaufhielten, zusammenzutreffen.</s><s>Ohnehin war es dem Bruder unleidlich, auch nur das seelenloseste Stückchen ihrer Gegenwart auf Augenblicke einem anderen gönnen zu müssen, und seine Wut auf den Grafen machte es wünschenswert, daß Aglaia demselben entzogen und damit die Ursache seiner Eifersucht hinfällig würde.</s><s>Konnte sie auch seine Wohnung nicht teilen, so war sie ihm doch leicht erreichbar, und es verging selten ein Tag, ohne daß sie sich gesehen hätten.</s><s>Häufig begleitete ich ihn bei diesen Besuchen, besonders wenn wir die Abendmahlzeit zusammen einnahmen, welche Stunden mir immer wie ein Glanz und Leuchten in der Erinnerung bleiben werden.</s><s>Die allmächtige Zunge des herrlichen Mannes war dann meist schweigsam, und er ließ Aglaia mit mir scherzen und schwatzen.</s><s>Ich höre sie noch: wie ein kleines tanzendes Wasser, das sich von einem himmelhohen Felsen herabstürzt und schäumend von Klippe zu Klippe springt, hüpfte das übermütige Geplauder von ihren beweglichen Lippen.</s><s>Manchmal, wenn ich einschlafe, ist es mir, als neigte sie sich über mich und ließe ein feines, singendes, plätscherndes Wiegenlied in mein wehmütig horchendes Ohr rieseln.</s><s>Ach, ob die singende Seele, wenn sie die Erde wieder aufsuchte, bei mir verweilen würde?</s><s>Während sie mit mir sprach, ruhten ihre Augen mit dem stillen, feuchten und glänzenden Blick auf Dolfin, von dessen Antlitz an diesen Abenden die Traurigkeit ganz verschwunden war.</s><s>Er glich einem kleinen Jungen mit glücklichem Weihnachtsgesicht, in dem Augen und Zähne blinken und lachen.</s><break strength="x-weak" /><s>Überhaupt war er jetzt fast immer gehoben und heiter, nur dann und wann wurde er von einer unerhörtenMelancholie ergriffen, die ihn jählings wie eine hungrige Wölfin anpackte, auch wenn er in Aglaias Gesellschaft war.</s><s>Ich legte dies anfänglich als ein Symptom inneren Zwiespalts aus, hervorgerufen etwa durch das Unrechtmäßige seines Verhältnisses zu der geliebten Frau, was auch ihre eigene Meinung war.</s><s>Denn als ich sie einmal fragte, ob sie sich keine Gedanken darüber mache, daß sie wider göttliches und menschliches Recht ihren Gatten verlassen habe, sah sie mich, die Unvergleichliche, Süße, mit großen Augen ängstlich an und sagte, sie wisse wohl, daß sie unrecht gehandelt habe, könne es aber durchaus nicht bereuen, hingegen fürchte sie, Dolfin, der ja ein heiliger Mann sei, nehme Anstoß daran, und ich möchte ihr helfen, solche Bedenken, wenn sie ihm kämen, zu zerstreuen.</s><s>Allmählich fand ich aber heraus, daß diese Anwandlungen von Traurigkeit nichts als grimmige Rückfälle seiner alten Krankheit waren, daß er sich einsam fühlte zwischen den lebendigen Erscheinungen, sobald ihm irgendwie zu Bewußtsein kam, wie sie in beständiger Verwandlung begriffen in den Tod eilten.</s><s>Ein fallendes Blatt, ein körperliches Unbehagen, ein verlängerter Schatten, das Schwinden der Zeit konnte unversehens diesen grausamen Gedankengang in ihm wecken; ihn wegzuscheuchen, gelang aber fast immer bald einem innigen Wort oder Blick Aglaias.</s><s>Auch während des Predigens konnte die Melancholie ihn plötzlich ergreifen und aus der Stimmung des angefangenen Themas herauswerfen, was ihn aber keineswegs verwirrte, sondern ich mußte dann an einen Holzkloben denken, der langsam fortgeglimmt hat, nun aber recht Feuer fängt und mit lichterloher Flamme brennt.</s><s>Er sang das Lied von der Vergänglichkeit allesIrdischen bald mit den kreischenden Trompetentönen singender Schwäne, bald leise und traurig wie unter rinnenden Tränen; aber wie das Meer, dessen ewig wiederkehrender Wellenbrandung man Tag und Nacht nicht müde wird zu lauschen, langweilte er nie durch das eintönig tiefe Läuten und Rauschen seines Vortrages.</s></speak>',
      '<speak><break strength="x-weak" /><s>Während dieser Zeit war ich im Dienste Dolfins außerordentlich beschäftigt.</s><s>Der Graf nämlich hatte sich bei der Erklärung seiner Frau, sie werde nicht zu ihm zurückkehren, sondern sich der Einsamkeit ergeben, keineswegs beruhigt, vielmehr ihr nachgespürt, wohin sie ging und was sie tat, und war dabei auf die Vermutung geraten, daß an ihrer rätselhaften Flucht Fra Celeste schuld sei.</s><s>Da nun seine Briefe an Aglaia unbeantwortet blieben, wendete er sich an ihn, welcher aber nichts davon erfuhr, denn ich vollführte den ärgerlichen Briefwechsel, der sich daraus ergab, ohne ihn durch den weltlichen Plunder in seiner himmlischen Versunkenheit zu stören.</s><break strength="x-weak" /><s>Auf die erste warnende Anzeige des Grafen, daß ihm seine Frau, er wisse wohl mit wem, davongelaufen sei, antwortete ich im Namen Fra Celestes, daß ich ihn aufrichtig beklage, daß eben leider die Verderbnis auch im weiblichen Geschlechte immer zunehme, daß aber im Grunde an einer Frau nicht viel gelegen sei, er möge Trost bei Gott suchen, wo er ihn sicher mit der Zeit finden werde.</s><s>Die folgenden Briefe, welche nun schon bedrohlicher klangen, beantwortete ich, indem ich sagte, nichts schmerze mich mehr, als wenn Frauen in trauriger Kurzsichtigkeit und Verkennung statt meines Geistes, der von Gott erfüllt sei, meine arme, fleischliche Person liebten.</s><s>Wenn seine Frau dieseunglückliche Verwechslung begangen habe, wollte ich gerne für ihre Erleuchtung beten.</s><s>Ein andermal hielt ich ihn ernstlich dazu an, sich zu prüfen, ob er nicht durch etwelches Sündenwesen seiner Frau Ursache gegeben hätte, ihn zu meiden, worauf er sich eingehend mit Darlegung seines ganzen Lebenswandels verantwortete.</s><s>Ich machte ihn auf mancherlei Lasterhaftigkeit in demselben, den er für durchaus ehrbar und tadellos hielt und an dem auch vom landläufigen Standpunkt aus nicht viel auszusetzen war, aufmerksam; schließlich aber deutete ich ihm an, daß ich argwöhnte, er sei mit einer fixen Idee behaftet, er möge, wenn er glaube, daß es nützlich sei, meine Predigten lesen, schreiben könne ich ihm künftig nicht mehr, da es doch zu nichts führe und meine Zeit kostbar sei.</s><s>Ich erhielt darauf noch einen Brief des Grafen, worin er ankündigte, daß er sich nunmehr an den Abt von Fra Celestes Kloster wenden würde, um zu seinem Rechte zu kommen, und nach kurzer Frist hatte ich denn auch diesen Mann Gottes auf dem Halse, mit dem ich ohnehin in heftigem Streit begriffen war.</s><break strength="x-weak" /><s>Seitdem er für Aglaia zu sorgen hatte, welcher der Graf natürlich die Mittel, um fern von ihm zu leben, nicht gab und die auch von den erzürnten Eltern ununterstützt gelassen wurde, war Fra Celeste sehr auf Geld erpicht und wollte nur noch gegen hohe, ihm selbst ausgezahlte Vergütungen predigen, wie wenn er eine Privatperson gewesen wäre.</s><s>Hierauf wollte sich der Abt durchaus nicht einlassen und ergoß sich in langen Hirtenbriefen über die selige Armut, und daß der Bruder sich durch die Welt, mit der er jetzt in Berührung komme, nicht solle verderben lassen.</s><s>Ich schrieb, wie ich das Geld nur zu wohltätigenund ähnlichen Zwecken gebrauchte, übrigens aber so frugal lebe, wie ich es unter seiner Leitung im Kloster gelernt hätte.</s><break strength="x-weak" /><s>Der Abt, der ein höchst frommer und milder Mann war, gerade deswegen aber immer besorgte, man möchte sein Ansehen nicht respektieren und ihn in den Hintergrund drängen, gab sich damit nicht zufrieden und bestand trotz vieler erfinderischer und künstlicher Briefe, die ich verfaßte, auf dem Rechte des Klosters.</s><s>Als nun noch die Angelegenheit des Grafen dazu kam, fing er an, von der seligen Keuschheit zu sprechen, und daß er nun wisse, wozu der Bruder das viele Geld brauche, und wie es ihm weh tue, daß ein Schaf aus seinem Stalle sich so verirre.</s><s>Es sei wahr, antwortete ich, halb betrübt, halb entrüstet, daß unzählig viele Frauen zur Kirche wallten, um mich predigen zu hören, was aber nicht zu tadeln, sondern zu loben sei, da ich nur von erbaulichen Dingen spräche, die sie belehren und bessern könnten.</s></speak>',
      '<speak><s>Daß aber die neidischen und eifersüchtigen Männer diesen Umstand benützten, um mich zu verdächtigen, sei traurig, noch trauriger, daß er solchen Einflüsterungen Glauben schenke.</s><s>Der Abt schrieb zurück, er halte für das traurigste, daß sie auf Wahrheit beruhten, er sehe aber wohl ein, daß er mit seiner Sanftmut über mein verhärtetes Herz keine Macht habe, und werde deshalb die Sache dem Heiligen Vater selbst vorlegen, an welches höchste Haupt der Christenheit auch der Graf sich entschlossen habe, zu gelangen; dieser werde wohl noch Mittel in der Hand haben, mir beizukommen.</s><break strength="x-weak" /><s>Bald danach lief ein Schreiben aus Rom vom Kardinal San Fiori ein, einer überaus klugen, feinen und liebenPerson, was sich denn gleich in den ersten Zeilen verspüren ließ.</s><s>Nur der Schlechte, schrieb er, glaube nicht an die Tugend, er dürfe sich zwar nicht rühmen, gut zu sein, denn das sei bekanntlich nur der einige Gott und vielleicht einige Auserwählte, auf denen sein Geist ruhe, aber es sei sein Glück und sein Stolz, an das Gute zu glauben.</s><s>Er hoffe deshalb, daß es mir, das heißt dem Fra Celeste, gelingen werde, das schwarze Gewölk, das sich um die helle Sonne meines Tugendruhmes zusammenballe, zu zerstreuen, glaube aber, meiner Gegenwart würde das leichter werden als schriftlicher Verständigung.</s><s>Da außerdem der Heilige Vater selbst wie alle Kardinäle schon lange wünschten, die Stimme des berühmten Predigers zu vernehmen, lüde er mich ein, nach Rom zu kommen, wo ich meinem wahren Berufe, die Menschheit zu erleuchten, obliegen und nebenbei die beiden anhängigen Geschäfte erledigen könnte.</s><break strength="x-weak" /><s>Fra Celeste, dem ich nun in Kürze alles Vorgefallene mitteilte, erklärte sich sofort bereit, nach Rom zu gehen, doch fanden wir für gut, daß ich voranreiste, um die Verhandlungen einzuleiten.</s><s>In dem Kardinal San Fiori, dem ich mich sogleich vorstellte, fand ich einen Mann von äußerst höflichen und liebenswürdigen Sitten, dem man ansah, daß er ein Feinschmecker des Lebens war, der aber zugleich das Göttliche zu schätzen wußte, so daß es ihm fast ein Bedürfnis war, mit aszetischen oder wenigstens enthaltsamen Leuten zu verkehren.</s><s>Er wußte solche so zu behandeln, daß sie keinen Anstoß nahmen, den Gastmählern beizuwohnen, die er den Lebemännern unter der hohen Geistlichkeit veranstaltete, und das auserlesenste Gericht würde ihm nicht gemundet haben, wenn ein überirdischer Hauch, seies durch metaphysisches Gespräch oder Anwesenheit einer vergeistigten Persönlichkeit, es nicht eingesegnet hätte; ja, wenn er einen erprobten Koch und alle Bequemlichkeiten eines modernen Hauses hätte mit sich nehmen können, zweifle ich nicht, daß er sich unter den Anachoreten der Wüste am wohlsten befunden hätte.</s><s>Kurz, er stellte die Rolle eines edlen Kirchenfürsten mit Pietät und Talent dar, womit auch seine mittelgroße, volle Gestalt und sein regelmäßiges Gesicht mit der geraden Nase, das ein Paar geschwinde, kunstreiche Augen in verschiedener Weise beleuchteten, übereinstimmte.</s><s>Er empfing mich überaus liebenswürdig, bewirtete mich und unterhielt mich unterdessen eine Weile mit geistreichen Kleinigkeiten, bis er denn allgemach anfing, mich nach dem Tun und Treiben meines Herrn auszuhorchen.</s><break strength="x-weak" /><s>Ich erzählte ihm freiwillig, was ich irgend wußte, ausgenommen alles, was auf Aglaia Bezug hatte; verbreitete mich über seine himmlische Sanftmut, über seine erhabene Gleichgültigkeit gegen die Versuchungen der Welt, über seinen göttlichen Zorn gegen das Gemeine, was ich mit vielen Beispielen belegte, und sagte unter anderem, wenn man einen Engel an der Liebe erkennte, mit der alle Herzen, die sein Wesen verspürten, ihm dienen müßten, sei er ein Engel oder doch gewiß ein Kind Gottes zu nennen.</s><s>Der Kardinal war unermüdlich, mir zuzuhören, streichelte mir, während ich sprach, mehrmals Haar und Wangen und schenkte mir, ehe er mich entließ, einen Brillanten in einer Busennadel.</s></speak>',
      '<speak><s>Er ließ sodann in der zartesten Weise für alle meine Bedürfnisse sorgen und lud mich häufig zum Essen ein, wobei auch viele andere Kardinäle und geistlichePersonen anwesend waren.</s><s>Alle behandelten mich mit großem Wohlwollen, und der Kardinal San Fiori sprach von nichts anderem als von der Heiligkeit des Fra Celeste, was ihn selbst so rührte, daß ihm nicht selten Tränen aus den Augen flossen.</s><s>Von dem Grafen sagte er unverhohlen, er würde froh sein, wenn er ein Mittel wüßte, um diesen Plagegeist zum Schweigen zu bringen; viel schwieriger schien es mir, die Geldangelegenheit zu lösen, denn der Heilige Vater mochte wohl dem Abte nicht unrecht geben, sich andererseits aber auch ungern entschließen, den Aufwand des Bruders aus eigener Kasse zu bestreiten.</s><s>Auf wiederholte Nachfrage des Kardinals, wozu denn Fra Celeste so große Summen benötigte, brachte ich erstlich seine Familie vor, die ich zu diesem Zwecke, arm und zahlreich, erfunden hatte, ferner seine verschwenderische Wohltätigkeit, und da auch das noch nicht zu genügen schien, sagte ich, daß er Kapital sammle, um in Afrika einen Staat zu gründen, der in Gestalt eines großen Bundesklosters seine göttlichen Ideale verwirklichen sollte.</s><s>Dieser Plan interessierte den Kardinal so ungemein, daß meine Phantasie kaum Vorrat genug hatte, um seine Erkundigungen nach jeder Einzelheit des afrikanischen Gottesreichs zu genügen.</s><s>Trotzdem machte ich in dieser Sache keinen Fortschritt, sondern wie mir der Kardinal sagte, der Heilige Vater behielt sich vor, mit eigenem Munde dem Bruder seine Ansprüche auszureden.</s><break strength="x-weak" /><s>Aber die Reihe zu reden kam erst an Fra Celeste.</s><s>Der Papst hatte den sinnreichen Einfall gehabt, der Bruder möchte seine erste Predigt über die Heiligkeit der Ehe halten, was ich ihn sogleich wissen ließ, als er spät abends in Rom eintraf.</s><s>Er sagte, daß er etwas anderes im Sinn gehabt hätte,und schien unzufrieden über die Störung zu sein.</s><s>Am anderen Morgen erwachte er in bester Laune und bestand darauf, sofort zu predigen, ehe er sich noch dem Heiligen Vater vorgestellt hatte und ehe nur eine ordentliche Anzeige davon gemacht werden konnte.</s><s>Es verbreitete sich aber so schnell, wann und wo die Rede stattfinden würde, daß es nicht an Zuhörern fehlte, im Gegenteil drängte sich das Volk noch vor der Tür auf dem Markte, um vielleicht von dort aus einige Worte aufzufangen.</s><s>Freilich glich auch diese Predigt einem alten öligen Weine, von dem ein einziger Tropfen eines greisen Mannes Blut in Feuer verjüngen kann.</s><break strength="x-weak" /><s>Man stelle gewöhnlich, fing er an, die Liebe der Eltern zu den Kindern als die uneigennützigste am höchsten, aber das sei irrig, denn in den Kindern liebten die Eltern nur sich selbst, während Mann und Frau oft um so heftiger zueinander strebten, von desto verschiedener Art sie wären.</s><s>In solchem Verhältnisse könne man lernen, was die Bestimmung des Menschen sei: das eigene Selbst einem fremden aufzutun und hinzugeben.</s><s>Dazu sei es nicht notwendig, daß das geliebte Wesen vollkommen sei, doch dafür gehalten solle es werden.</s><s>»Ich bin kein Tier oder Sklave,« so etwa sagte er, »daß ich an meine Sinne gebunden wäre: wenn ich will, daß sie so herrlich und engelgleich ist, wie ich sie im ersten Traume meiner Liebe sah, so ist sie es.</s><s>Und daß der Zwang meiner Liebe sie so mache, deshalb ist ihre Hand unlöslich in meine versiegelt.« Er sprach dann über Untreue, von leichtfertigen Ausschweifungen und allen Üppigkeiten der Genußsucht mit so viel Ekel und ungeduldigem Hohne, daß sich die Getroffenen beinah sichtlichdarunter duckten.</s><s>Aber nachdem er noch viel Wundervolles über die eheliche Liebe und Treue gesagt hatte, was wie ein hohes Freudengeläute aus der stillen, kühlen Kirche in die goldene Mittagsluft des Marktes hinausflutete, hielt er plötzlich inne, weil irgend etwas, sei es der Anblick einer weinenden Witwe oder eines steinernen Grabmals an der Mauer oder ein Einfall der eigenen Brust die schwarzen Gedanken in ihm aufgeweckt hatten.</s></speak>',
      '<speak><s>Es folgte eine Pause, die allen willkommen war, damit sie das Vernommene in ihrem Herzen konnten ausklingen lassen, dann sagte er plötzlich mit verändertem Tone langsam: »Nun will ich euch das Geheimnis sagen.</s><s>Gott hat die Ehe gemacht für die Menschen, aber der Selige ist der, welcher der Liebe entbehren kann.«</s><break strength="x-weak" /><s>Mochte es nun der dunkle, zitternde Rhythmus seiner Worte sein, es war uns Hörern, als ob von oben her ein kalter Schatten auf unsere lichte Rosenpracht gefallen sei aus einer schwarzen, langsam steigenden und wachsenden, unentrinnbaren Wolke.</s><s>»Kann ich denn überhaupt«, sagte er, »mein Selbst einem anderen Menschen geben?</s><s>Denkt euch, ein Vater habe seinen Sohn im Gefängnis.</s><s>Er besucht ihn, aber er darf nicht zu ihm hinein, nur schauen darf er ihn durch das vergitterte Fenster.</s><s>Auch das ist Glück, aber er möchte doch dichter bei dem geliebten Kinde sein und das teure Gesicht küssen.</s><s>Indessen das ist unmöglich, weil der Sohn festgebunden ist hinten an die Mauer des Kerkers, und der Vater drückt sein Gesicht fest an das Gitter, wie wenn er es zerbrechen wollte, damit er hinein könnte zu seinem Kinde.</s><s>So ist die Liebe, es gibt keinen Weg, der die Seele zur Seele führt.</s><s>Die Seele kann die Seele nichtberühren, nicht küssen, nicht vernehmen; eingemauert in verkittete Steine spüren sie ihre Nähe nur an einem halberstickten Schrei oder an einer Träne, die durch das lückenhafte Gefüge rinnt.</s><s>Wie kann die Verborgene, die Eingekerkerte mein sein?</s><s>Aber nun kommt das Ärgste: wenn ich das Gefängnis zerschlagen und die geliebte Seele befreien könnte, wer weiß, ob mein Herz sie erkennte?</s><s>Ich habe sie ja nie gesehen, nie gefühlt, nie gehört, nur geträumt von ihr und nach ihr mich gesehnt.</s><s>Hätte sie das zärtliche Auge, das so warm und kosend über meinen Körper blickte, spräche sie mit der lachenden Stimme, die mich mit Rosenblättern überschüttete?</s><s>Während ich mich mühte, die Seele zu gewinnen, die ich nicht fand, gewöhnte ich mich an das zerbrechliche Bild, das sie verhüllte.</s><s>Vergänglichkeit ist aber die Spur von Gottes Finger, als er mit einem Fluch anrührte alles, was wir nicht begehren sollen, oder wir sind des Todes.«</s><break strength="x-weak" /><s>Mit solchen und ähnlichen Worten schlug er die Liebe, die er zuvor verherrlicht hatte, ans Kreuz, und seine Augen schienen mit göttlichem Jammer mitleidvoll auf die Gemarterte zu blicken.</s><s>Am meisten zu bewundern fand ich aber dies, daß Fra Celeste seine Zuhörer mit dieser trostlosen Ansicht nicht entließ.</s><s>Von einem künstlerischen oder frommen Impulse getrieben, schloß er mit der erhebenden Wendung, daß diese Schmerzen und Enttäuschungen nur in der irdischen Liebe stattfänden, die besitzen wolle.</s><s>Wer diese überwände, die Menschen nur suchte, um ihnen wohlzutun, sonst aber mit sich und Gott genug hätte, würde eine Liebe ohne Ende und ohne Bitterkeit erlernen.</s><s>Und das wahre, heilige Sakrament der Ehe sei, daß jedem in dereigenen Brust ein Gefährte anvertraut sei, der ihm treu bleibe bis zum Tode und über den Tod hinaus; ein göttlicher Keim oder Schutzengel, der jedem mitgegeben sei, den man hegen und pflegen und lieben solle, damit man, wie es in der Ehe vorgeschrieben sei, ein Fleisch und eine Seele mit ihm werde.</s><break strength="x-weak" /><s>Diese Predigt hatte Fra Celeste den Leuten wie eine Art Liebeszauber zu kosten gegeben, wenigstens war die Begeisterung für ihn so stark und einmütig, wie sonst die ansteckenden Seuchen auftreten, an denen man sterben kann, nur weil man glaubt, sie zu haben.</s><s>Die Geldfrage wurde ohne weiteres so erledigt, daß der Papst dem Bruder monatlich fünftausend Franken auszahlte, wofür derselbe sich verpflichtete, mindestens eine Predigt in der Woche zu halten, an welchem Orte innerhalb der christlichen Welt der Papst für gut finden werde.</s><s>Da wir an demselben Tage Aglaia in Rom erwarteten, kaufte er so viel Blumen und Früchte, daß man einen Wagen damit hätte anfüllen können, und ließ sie nach ihrer Wohnung schaffen, die sich außerhalb der Stadt ganz im Freien befand.</s></speak>',
      '<speak><s>Dort feierten sie ein solches Wiedersehensfest, daß einer, der es von außen betrachtete, hätte meinen können, Nero, der schönste und wildeste Kaiser der Welt, wäre für eine Sommernacht ins Leben zurückgekehrt, um an dem Busen einer liebestrunkenen Frau sich wieder in den Tod zu schwelgen.</s><break strength="x-weak" /><s>Aber Fra Celeste war am folgenden Tage voller Kraft und Lust und hielt eine Rede über die Wonne der Armut, die, wenn das möglich war, die vorige an eindringlicher Wärme übertraf.</s><s>Er ging davon aus, daß das Beste am Menschen die Sehnsucht sei, daß aber die Sehnsucht aus Mangelentspringe, daher der Elendeste und Erbärmlichste eigentlich der sei, welcher genug habe.</s><s>Er sei gleichsam in einem Kerker eingesperrt und müsse, weil er seine Lebensorgane zu wenig üben könne, langsam erstarren und erlahmen und aus seiner dumpfen Sattigkeit in den Tod hinüberdämmern.</s><break strength="x-weak" /><s>Wozu soll ich wiederholen, was Fra Celeste von der Sklaverei des Geldgierigen und von der Freiheit des Armen sagte?</s><s>Man mußte ihn auf der Kanzel stehen sehen im einfachen Mönchsgewande wie einen Imperator auf dem Triumphwagen, wenn er mit seinen Königsaugen über die große und wahrhaft prächtige Versammlung hinsah, ohne sie auch nur mit einem Blicke zu streifen!</s><s>Nicht eine Schiffsladung von Gold und Edelsteinen hätte der reichste von allen ihm anzubieten gewagt, weil er sich des schäbigen Geschenkes geschämt hätte.</s><break strength="x-weak" /><s>Auf das Ersuchen des Papstes, der den erstaunlichen Einfluß Fra Celestes auf das Volk zum Guten zu verwerten wünschte, predigte er ferner über die großen Sünden, wie Diebstahl, Raub und Mord, weil das Banditenwesen namentlich auf dem Lande und im Gebirge blühte.</s><s>Obgleich nun solche Fehler der Liebe des erhabenen Mannes ganz fremd waren, wußte er doch darüber zu reden, als ob er jahrelang das Geschäft eines Räuberhauptmannes betrieben und das Blut zahlloser Christen auf dem Gewissen hätte; denn er besaß die Gabe, jedem Menschen durchs Antlitz in die Brust zu schauen und alles, was dort wallte und wogte, ins eigene Herz zu fassen.</s><s>Den Leuten schien er deshalb wie ein Bruder, der alles mit ihnen durchgemacht hätte, dann aber vorausgeeilt wäre und ihnen freundlich die kräftige Hand reichte, damit sie ihm nachkämen.</s><break strength="x-weak" /><s>Unsere Wohnung war nun beständig umlagert von Bittstellern, die Rats einholen oder Beichte ablegen wollten, vornehmen und geringen, Männern und Frauen.</s><s>Viele, die er, weil er zu überhäuft war, wegschicken mußte, kamen zu mir, die ich auch in seinem Namen so gut wie möglich zufriedenstellte; besonders mußte ich mir ein großes Lager von Reliquien zusammenstellen, Haarlocken, Stückchen von abgelegten Gewändern und dergleichen mehr, die ich ohne Wissen meines Herrn unter die Gläubigen verteilte.</s><break strength="x-weak" /><s>Bei diesem Ansehen, das Fra Celeste anfing im ganzen christlichen Europa zu genießen, war es dem Kardinal San Fiori höchst zuwider, daß der Graf, der, um seine Sache besser zu betreiben, selbst nach Rom gekommen war, ihn unablässig bedrängte, den großen Mönch zur Rechenschaft zu ziehen, widrigenfalls er alles bekanntmachen und den Feinden unserer Kirche eine Handhabe geben würde, sich auf unsere Kosten lustig zu machen.</s><s>Ich hatte das Glück, auf ein unschädliches Mittel zu verfallen, wie man den Grafen loswerden könnte, das den Beifall des Kardinals fand und das wir unter viel Spaß und heimlichem Gelächter ins Werk setzten.</s><s>Der Kardinal lud nämlich den Grafen zu einem Essen ein, währenddessen er die Bemerkung machte, die Anhänglichkeit des Grafen an seine pflichtvergessene Frau sei um so rührender, als diese, wenn sie einmal Reize besessen, dieselben jedenfalls längst verloren und guten Grund hätte, sich in die Einsamkeit zurückzuziehen.</s><s>Um meine Meinung befragt, äußerte ich mich bescheiden, ich hätte zwar die Gräfin in der Kirche gesehen, pflegte aber nicht auf das Äußere der Frauen zu achten, sie scheineaufrichtig fromm zu sein, und das genüge mir.</s></speak>',
      '<speak><s>Der Kardinal von Grossetto, welcher neben mir saß, streichelte mich und lobte meine Genügsamkeit, während er den übrigen zublinzelte, man müsse mich in meiner Einfalt nicht stören; dann sagte er schmunzelnd, leider gebe es viele Frauen in Rom, die schöner als fromm wären, und zählte mehrere von diesen auf.</s><s>Als die schönste priesen alle einmütig eine gewisse Dame, die wir vorher für unser Spiel auserlesen und von allem unterrichtet hatten.</s><s>Sie war vor nicht langer Zeit eine intime Freundin des Kardinals San Fiori gewesen, ebenso schlau wie pompös anzusehen und sehr willig, die Rolle, die wir ihr zugedacht hatten, zu übernehmen.</s><s>Die Herren wurden nicht müde, die unerhörte Schönheit dieser Dame zu preisen, als deren einzigen Fehler sie bezeichneten, daß sie unnahbar sei und jeden, der sich ihr in liebevoller Absicht nähere, unbarmherzig heimschicke.</s><s>Sie führten mehrere reiche und edle Herren an, deren Herzen sie durch ihre Sprödigkeit gebrochen hatte, bis der Graf vor Begierde brannte, diese Kostbarkeit in seinen Besitz zu bringen.</s><s>Er sagte in hochfahrendem Tone, daß keine Frau unüberwindlich sei und daß wohl nur der rechte Mann noch nicht erschienen sei, wobei er durchblicken ließ, daß er das Zeug dazu in sich spüre.</s><s>Der Kardinal San Fiori erbot sich zwar, die Bekanntschaft zu vermitteln, warnte aber den Grafen freundschaftlich, er möge sich lieber bittere Qualen und Demütigungen ersparen.</s><s>Die Dame, die der Graf nun kennenlernte, behandelte ihn anfangs mit gleichgültiger Kälte, zwischenhinein zeichnete sie ihn durch liebenswürdige Koketterie vor den anderen aus, was sie dann aber durch vermehrte Strenge wieder ungültig zu machen schien, durchwelches Betragen sie seine Verliebtheit aufs äußerste reizte.</s><s>Bald war er dermaßen in ihrer Gewalt, daß sie es wagen konnte, ihm Hoffnung auf ihre Liebe zu machen, wenn er sie zu seiner rechtmäßigen Gemahlin machen würde.</s><s>Der Graf schwur, nie etwas anderes im Sinne gehabt zu haben, und reichte augenblicklich dem Heiligen Vater die Bitte ein, derselbe möchte seine Ehe mit Aglaia lösen.</s><s>Kardinal San Fiori, welcher die Führung der ganzen Angelegenheit in der Hand hatte, bezeugte über diese Wendung Erstaunen und Entrüstung, sagte, daß keinerlei Gründe zu einer Scheidung vorlägen, da Aglaia, wie jedermann wisse, sich nicht eigentlich gegen ihn vergangen hätte, vielmehr im Grunde aus lobenswerter Frömmigkeit die weltliche Luft seines Hauses meide, und daß die unheilige Raserei eines Mannes, der kein schönes Weib ansehen könne, ohne sie besitzen zu wollen, nimmermehr durch die Kirche gerechtfertigt werden dürfe.</s><s>Nachdem er den Grafen in dieser Weise eine Zeitlang auf der Folter gelassen hatte, wurde die Scheidung ausgesprochen, und er führte die Angebetete heim, wodurch wir in alle Zukunft vor seinen Verfolgungen sicher waren.</s><break strength="x-weak" /><s>Fra Celeste hätte nun sein Leben mit größerem Behagen genießen können, wenn sich nicht sogleich neue Widerwärtigkeiten erhoben hätten, welche diesmal von Aglaias Eltern ausgingen.</s><s>Sie verlangten nämlich, ihre Tochter solle, anstatt in der Welt umherzuvagabondieren, in ihr Haus zurückkehren, und kamen selbst nach Rom, um die Widerspenstige abzuholen.</s><s>Der Heilige Vater sowie die Kardinäle sprachen sich dahin aus, wenn Aglaia nicht bei ihren Eltern leben wolle, möchte sie wenigstens in einKloster eintreten, wo sie ja der Einsamkeit nach Herzenslust frönen könne; denn die Verehrung für Fra Celeste, so berechtigt sie auch sei, könne ungebildete oder bösartige Menschen doch zu Mißdeutungen veranlassen, und da sie auf die Freuden der Gattin und Tochter des Hauses ohnehin verzichtet hätte, sei nicht einzusehen, was sie noch in der Welt zurückhielte.</s><s>Ich sagte dem Kardinal, daß Fra Celeste die Dame längst in diesem Sinne zu beeinflussen versucht hätte, daß es bislang aber vergeblich gewesen sei, da sie die Eintönigkeit des Klosterlebens fürchtete, und fügte hinzu, vielleicht werde seiner Beredsamkeit und Menschenkenntnis ein besserer Erfolg zuteil.</s></speak>',
      '<speak><s>Dieser Aufforderung kam der Kardinal unverweilt nach; er besuchte die Gräfin, sprach mit ihr von den Annehmlichkeiten des Klosterlebens, dem er sich herzlich gerne hingeben möchte, wenn ihn nicht Geschäfte zum Nutzen der Kirche noch in der Welt zurückhielten, und verliebte sich mittlerweile über die Maßen in Aglaia, was sowohl ihr wie mir sehr zur Belustigung diente.</s><s>Eines Tages kam aber Fra Celeste dazu, dem ich diese Vorgänge aus Rücksicht verschwiegen hatte, und geriet über den unerwarteten Anblick des Kardinals in jene Wut, deren Symptome ich früher beschrieben habe.</s><s>Ohne abzuwarten, daß einer von uns die Anwesenheit des Kardinals bei Aglaia erklärte, fuhr er heraus, man habe ihm den Verkehr mit der Gräfin zum Vorwurf gemacht, jetzt sähe er die Ursache dieses Eifers, man wolle selbst bei ihr ein und aus gehen, dabei sei er im Wege gewesen.</s><s>Im Feuer des Sprechens erhitzte er sich mehr und mehr, bis er von oben bis unten in eine prasselnde Flamme verwandelt schien mit seinen knirschenden Zähnen und lechzenden Augen;wir schauten ihn an wie einen gläsernen Zylinder, der glüht und kracht, so daß man sein Auseinanderspringen jeden Augenblick erwartet und sich bereit hält wegzulaufen, um von den heißen Scherben nicht getroffen zu werden.</s><s>Während ich Stellung nahm, um ihm rechtzeitig in den Arm zu fallen, wenn er etwa auf den Kardinal los wollte, ließ die göttliche Vorsehung diesen entschlüpfen, den der aufgeregte Mann, von der Eifersucht wie von einer Windsbraut gefaßt und mitgerissen, sonst, wie er selbst glaubte, vielleicht erdrosselt hätte.</s><break strength="x-weak" /><s>Da zeigte es sich wieder, wie Gott diesen wunderbaren Menschen liebte und auch die scheinbaren Verirrungen zu seinem Wohle ausschlagen ließ; denn der Kardinal, zu dem ich mich, nicht ohne eine gewisse Befangenheit überwinden zu müssen, am folgenden Tage begab, empfing mich zwar betrübt, aber herzlich, indem er mir von dem Ruhebette, wo er lag, beide Hände entgegenstreckte.</s><s>Er erzählte mir, daß ihn einerseits die Eitelkeit zu Aglaia geführt hätte, nämlich die Aussicht, größere Macht über ihren Geist zu gewinnen als der berühmte Prediger, daneben lüsterne Neugier, die Frau kennenzulernen, von der man sich so bedeutende Gerüchte zuflüsterte.</s><s>Die Holdseligkeit ihrer Erscheinung und ihres Wesens hätte es ihm gleich beim ersten Besuche angetan, und anstatt sich nun zurückzuhalten, wie es seine Pflicht gewesen sei, habe er sich dem verbotenen Spiele hingegeben, noch dazu unter geflissentlicher Täuschung des eigenen Gewissens, dem er vorgelogen habe, es geschehe zu einem heiligen Zwecke.</s><s>Fra Celeste aber habe mit dem Blitzen seiner Augen und seiner Zunge das Truggewebe entzündet und zu Asche gebrannt, so daßer sich selbst angeschaut habe, wie er wirklich sei, schwach und voll Sünden.</s><s>Jetzt stehe es ihm fest, sagte er, daß der Bruder ein Seher sei, auf dem der Geist Gottes unmittelbar ruhe, und trug mir auf, Aglaia, die seine Torheit ermutigt und dadurch gleichfalls gesündigt hätte, Unterwerfung unter die Ratschläge des prophetischen Mannes zu empfehlen.</s><s>Ich sagte, Fra Celeste hätte im Sinne, einige große Predigten über das Klosterleben zu halten, wodurch er hauptsächlich Aglaias Herz zu treffen gedächte, was zwar nur ein Einfall war, der mir im Augenblicke kam, von meinem Herrn aber, dem ich davon Bericht erstattete, sofort aufgegriffen wurde.</s><s>Sein Genius bemächtigte sich der hingeworfenen Idee so vollkommen, daß er bald darauf, an drei aufeinanderfolgenden Tagen, drei Predigten über diesen Gegenstand hielt, die eine Horde von Kannibalen in Säulenheilige hätte umwandeln können.</s><s>Die Folge davon war, daß unter vielen anderen Leuten auch Aglaias Eltern sich voll Abscheu von der Welt abkehrten und Kutte und Schleier nahmen.</s><s>Die Arme war so gerührt über das Betragen der beiden Alten, die, ohne sie noch ferner anzuklagen, noch irgend etwas von ihr zu fordern, mit großer Würde, Demut und vielen Zeichen beginnender Heiligkeit sich zum Eintritte ins Kloster vorbereiteten, daß sie stundenlang Tränen vergoß, sich mit Selbstvorwürfen peinigte und sich ihnen vielleicht sogar angeschlossen hätte, wenn nicht Fra Celestes Krankheit dazwischengekommen wäre.</s></speak>',
      '<speak><break strength="x-weak" /><s>Er wurde nämlich mitten in einer Predigt durch einen Hustenanfall unterbrochen, konnte sie nur mit größter Anstrengung zu Ende bringen und warf bei der nächsten Wiederkehr des bösartigen Hustens sogar Blut aus.</s><s>Die Ärzteerklärten einmütig, daß Fra Celeste das Predigen für unabsehbare Zeit ganz aufgeben müsse, widrigenfalls das Schlimmste zu befürchten stehe.</s><s>Dies Verbot war für ihn, dem Reden so notwendig war wie Atmen, hart genug; nun aber kam hinzu, daß der Heilige Vater wünschte, der Bruder möge sich während der Ruhezeit in sein Kloster begeben; denn weil sein Tod als nach aller Wahrscheinlichkeit nahe bevorstehend erklärt wurde, war man darauf bedacht, daß der heilige Mann sein ruhmwürdiges Leben durch ein entsprechendes Sterben kröne.</s><s>Im Kloster, glaubte man, werde sich diese weihevolle Szene am schönsten abspielen, während sehr zu befürchten war, daß in einem Kurorte am Mittelmeer, wohin die Ärzte ihn gewiesen hatten und wohin Aglaia ihn als Pflegerin begleiten würde, sich allerlei Weltliches, sei es auch nur in Form von Gerüchten und Verdächtigungen, profanierend einmische.</s><s>Um den teuren Kranken nicht durch diese Widerwärtigkeiten aufzuregen, griff ich, ohne es ihm zu sagen, zu einem freilich etwas gewagten Mittel; ich erzählte dem Kardinal San Fiori, nachdem ich mir Geheimhaltung meiner Mitteilung hatte beteuern lassen, daß ich Ursache hätte zu fürchten, mein Herr werde zum protestantischen Glauben übertreten.</s><s>Er sei in Verkehr mit einem protestantischen Geistlichen von hoher Stellung geraten, dem augenscheinlich daran gelegen sei, den berühmten Prediger für seine Kirche zu gewinnen; er entblöde sich nicht, darauf anzuspielen, was für weltliche Vorteile dem Bruder dadurch zufließen würden, daß er Reichtümer sammeln und sogar eine Frau nehmen könnte.</s><s>Obgleich ich nun gewiß wüßte, sagte ich, daß mein Herr weder durch Geld noch durch Frauenzu bestechen sei, trüge ich mich doch mit ängstlichen Befürchtungen, denn in sein Kloster wolle er wegen des Streites, den er in früherer Zeit mit dem Abte gehabt hätte, keinesfalls zurückkehren, ohne Mittel könne er in der Welt nicht leben, also treibe ihn vielleicht die Not in das Lager der Feinde.</s><break strength="x-weak" /><s>Bei dieser Gelegenheit habe ich den ersten von den drei Küssen bekommen, die mir Aglaia gegeben hat.</s><s>Ich erzählte ihr, um sie zu erheitern, meine eben angeführte Unterhaltung mit dem Kardinal, und indem ich meine Phantasie laufen ließ, wohin sie wollte, fügte ich ausschmückenderweise hinzu, daß er mich insgeheim gedungen hätte, alle Schritte des Bruders zu überwachen, ihn beständig von allem zu unterrichten und seinen Übertritt zum protestantischen Glauben, wenn es soweit komme, nötigenfalls mit den äußersten Mitteln zu hintertreiben.</s><s>Aglaia fragte erschreckt, was unter diesen Mitteln zu verstehen sei, worauf ich wieder fragte, ob sie niemals von einem farblosen Pulver gehört hätte, das man mit unfehlbarer Wirkung in den Wein seines Todfeindes mischen könne, und ob sie nicht wisse, daß fanatische Gläubige sogar ihre Freunde mit solchen Pulvern bewirteten, wenn das zum Heile der Kirche dienlich sei.</s><s>Dann sprach ich noch von den glänzenden Anerbietungen, die der Kardinal mir gemacht hätte, und daß ich ihn zwar in dem Glauben gelassen hätte, als sei ich nicht abgeneigt, ihm zu dienen, daß ich aber, wie es sich von selbst verstehe, meinen Herrn niemals verraten würde.</s><s>Die Liebliche sagte lächelnd, da ich des Sündenlohnes verlustig gehe, wolle sie mich schadlos halten, was für Lohn ich mir wünsche?</s><s>worauf ich vor ihr niederkniete und umeinen Kuß von ihrem Munde bat.</s><s>Sie nahm meinen Kopf zwischen ihre kühlen Hände, daß es mir wonnig über den Leib rieselte, und küßte mich langsam auf beide Augen, indem sie sagte: »Deine Augen sind fromm wie die Taube und klug wie die Schlange.«</s><break strength="x-weak" /><s>Übrigens trug der Kardinal mir wirklich auf, ihn von allen auffallenden Schritten, die der Bruder unternähme, in Kenntnis zu setzen; denn seit der Androhung des Glaubenswechsels erschien er ihnen vollends wie ein schönes, aber furchtbares Tier, von dem man der heillosesten Dinge gewärtig sein müsse.</s></speak>',
      '<speak><s>Sie behandelten ihn denn auch mit großer Sorgfalt wie die Ägypter ihre heiligen Krokodile und Ochsen, überschütteten ihn mit Ehren, versprachen ihm soviel Geld er irgend wollte und entließen ihn mit Segenswünschen nach dem kleinen Ort am Meere, der ihm empfohlen war.</s><s>Der vielgeliebte Mann aber war so krank, daß er sich um nichts mehr bekümmerte, nicht einmal um Aglaia, die uns möglichst schnell nachgereist kam, um ihn zu pflegen.</s><s>Wie von Mohammed erzählt wird, daß Gott ihn bei seinen Lebzeiten durch die sieben Himmel geführt habe, während er auf Erden körperlich anwesend war, so, hätte man glauben können, sei es mit Dolfin geschehen.</s><s>Er lag häufig so unempfänglich da, als hätte die Seele den Körper verlassen, der doch lebte.</s><break strength="x-weak" /><s>Auch wenn der Geist gegenwärtig war, achtete Fra Celeste nicht auf seine Umgebung, und wenn er Aglaia und mich auch um sich duldete, ließ er doch deutlich merken, daß ihm unsere Gegenwart weder lieb noch leid war.</s><s>Sprach er zu uns, war es zwar mit der Geduld und Sanftmut eines abgeschiedenen Geistes, aber auch wie aus weiter Ferne undso unverständlich, daß man nicht wissen konnte, ob er im Fieber phantasierte oder ob er schon in andere Sphären hinüberschwebte.</s><s>Was Aglaia am wehesten tat, war, daß er von ihrer gemeinsamen Vergangenheit sprach wie von einem dumpfen Traume oder von der törichten Kinderzeit, ja, wenn sie ihm leidenschaftlich Gesicht und Hände küßte, um seine Erinnerung zu erwärmen und zu beleben, betrachtete er sie mit mitleidiger Verwunderung, etwa wie ein Christ einen Wilden, der sich noch darin gefällt, seinem Götzen geschlachtete Feinde als Opfer darzubringen.</s><break strength="x-weak" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(largeExample);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input text (${textExampleLarge.length} characters) goes above the hardLimit`, () => {
    const text = textExampleLarge

    const expected = [
      '<speak>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Morbi imperdiet dui in mattis fringilla. Vivamus scelerisque id erat eu imperdiet. Mauris eu neque sit amet lectus facilisis tristique. Vestibulum finibus quam vel tempus tempor. Donec in mollis augue. Duis hendrerit aliquam tortor at volutpat. Vivamus ac fringilla arcu, eget bibendum enim.Cras consectetur, dui vel aliquet tincidunt, velit nisl lacinia sem, non accumsan risus est at mauris. Vestibulum dignissim metus nec nisl dignissim, at volutpat libero dignissim. Sed sollicitudin lacus felis, quis vehicula nisi fermentum eget. Nullam venenatis, orci ac bibendum pharetra, justo turpis consequat orci, vitae lobortis mauris mauris a mi. In ut consequat lectus. Curabitur purus metus, vestibulum a mauris in, gravida egestas enim. Nunc diam erat, interdum in hendrerit vel, consectetur in ex. Mauris est dui, varius sed nunc sit amet, aliquam pellentesque justo. Sed placerat quis dui in blandit. Nullam porta leo ipsum, sit amet maximus mi mollis et. Donec ac lectus dolor. Mauris dictum libero dignissim tincidunt aliquam. Aliquam quis urna ac ligula lobortis vulputate quis porta ligula. Etiam semper ante non risus ultricies rutrum.Aliquam eu facilisis sem, eu maximus felis. Aenean quis ultricies augue, a tincidunt lacus. Nunc vitae massa vestibulum, malesuada sem eget, iaculis ante. Aliquam lorem velit, iaculis at ipsum non, facilisis vulputate lorem. Aenean fermentum imperdiet eleifend. Pellentesque magna ipsum, varius vel tortor in, dignissim faucibus est. Nullam velit nisl, accumsan in tellus id, suscipit varius massaAliquam eget mauris at felis bibendum semper. Ut ultrices dui eu orci ultricies molestie. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros, sed tincidunt est. Mauris lobortis tempor nulla id elementum. Etiam gravida arcu et nibh vulputate dapibus. Aliquam eget tempus diamNullam quis ullamcorper erat. Vestibulum ultricies erat quis leo efficitur consequat. Ut id risus facilisis, ornare sem at, rutrum tellus. Donec a urna odio. Ut et nisl quis enim dignissim sodales. Vestibulum risus augue, posuere id tortor vitae, efficitur fermentum massa. Etiam at eros rhoncus, aliquet tellus interdum, vestibulum nibh. Phasellus consequat velit quis mi lobortis varius. Phasellus semper purus nec tellus fringilla condimentum. Morbi cursus nec ante eget porttitor. Vestibulum vestibulum arcu sit amet elit congue tristique. Nulla eleifend purus non ornare interdum. Aliquam quis nulla aliquet, mattis dolor nec, auctor massa. Vestibulum placerat placerat ante sit amet laciniaLorem ipsum dolor sit amet, consectetur adipiscing elit. Morbi imperdiet dui in mattis fringilla. Vivamus scelerisque id erat eu imperdiet. Mauris eu neque sit amet lectus facilisis tristique. Vestibulum finibus quam vel tempus tempor. Donec in mollis augue. Duis hendrerit aliquam tortor at volutpat. Vivamus ac fringilla arcu, eget bibendum enimCras consectetur, dui vel aliquet tincidunt, velit nisl lacinia sem, non accumsan risus est at mauris. Vestibulum dignissim metus nec nisl dignissim, at volutpat libero dignissim. Sed sollicitudin lacus felis, quis vehicula nisi fermentum eget. Nullam venenatis, orci ac bibendum pharetra, justo turpis consequat orci, vitae lobortis mauris mauris a mi. In ut consequat lectus. Curabitur purus metus, vestibulum a mauris in, gravida egestas enim. Nunc diam erat, interdum in hendrerit vel, consectetur in ex. Mauris est dui, varius sed nunc sit amet, aliquam pellentesque justo. Sed placerat quis dui in blandit. Nullam porta leo ipsum, sit amet maximus mi mollis et. Donec ac lectus dolor. Mauris dictum libero dignissim tincidunt aliquam. Aliquam quis urna ac ligula lobortis vulputate quis porta ligula. Etiam semper ante non risus ultricies rutrumAliquam eu facilisis sem, eu maximus felis. Aenean quis ultricies augue, a tincidunt lacus. Nunc vitae massa vestibulum, malesuada sem eget, iaculis ante. Aliquam lorem velit, iaculis at ipsum non, facilisis vulputate lorem. Aenean fermentum imperdiet eleifend. Pellentesque magna ipsum, varius vel tortor in, dignissim faucibus est. Nullam velit nisl, accumsan in tellus id, suscipit varius massaAliquam eget mauris at felis bibendum semper. Ut ultrices dui eu orci ultricies molestie. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros, sed tincidunt est. Mauris lobortis tempor nulla id elementum. Etiam gravida arcu et nibh vulputate dapibus. Aliquam eget tempus diam Nullam quis ullamcorper erat. Vestibulum ultricies erat quis leo efficitur consequat. Ut id risus facilisis, ornare sem at, rutrum tellus. Donec a urna odio. Ut et nisl quis enim dignissim sodales. Vestibulum risus augue, posuere id tortor vitae, efficitur fermentum massa. Etiam at eros rhoncus, aliquet tellus interdum, vestibulum nibh. Phasellus consequat velit quis mi lobortis varius.</speak>',
      '<speak> Phasellus semper purus nec tellus fringilla condimentum. Morbi cursus nec ante eget porttitor. Vestibulum vestibulum arcu sit amet elit congue tristique. Nulla eleifend purus non ornare interdum. Aliquam quis nulla aliquet, mattis dolor nec, auctor massa. Vestibulum placerat placerat ante sit amet lacinia.</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(text);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input text (${textExampleLargeSentence.length} characters) without any punctuation goes above the hardLimit`, () => {
    const text = textExampleLargeSentence

    const expected = [
      '<speak>Lorem ipsum dolor sit amet consectetur adipiscing elit Morbi imperdiet dui in mattis fringilla Vivamus scelerisque id erat eu imperdiet Mauris eu neque sit amet lectus facilisis tristique Vestibulum finibus quam vel tempus tempor Donec in mollis augue Duis hendrerit aliquam tortor at volutpat Vivamus ac fringilla arcu eget bibendum enim.Cras consectetur dui vel aliquet tincidunt velit nisl lacinia sem non accumsan risus est at mauris Vestibulum dignissim metus nec nisl dignissim at volutpat libero dignissim Sed sollicitudin lacus felis quis vehicula nisi fermentum eget Nullam venenatis orci ac bibendum pharetra justo turpis consequat orci vitae lobortis mauris mauris a mi In ut consequat lectus Curabitur purus metus vestibulum a mauris in gravida egestas enim Nunc diam erat interdum in hendrerit vel consectetur in ex Mauris est dui varius sed nunc sit amet aliquam pellentesque justo Sed placerat quis dui in blandit Nullam porta leo ipsum sit amet maximus mi mollis et Donec ac lectus dolor Mauris dictum libero dignissim tincidunt aliquam Aliquam quis urna ac ligula lobortis vulputate quis porta ligula Etiam semper ante non risus ultricies rutrum.Aliquam eu facilisis sem eu maximus felis Aenean quis ultricies augue a tincidunt lacus Nunc vitae massa vestibulum malesuada sem eget iaculis ante Aliquam lorem velit iaculis at ipsum non facilisis vulputate lorem Aenean fermentum imperdiet eleifend Pellentesque magna ipsum varius vel tortor in dignissim faucibus est Nullam velit nisl accumsan in tellus id suscipit varius massaAliquam eget mauris at felis bibendum semper Ut ultrices dui eu orci ultricies molestie Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros sed tincidunt est Mauris lobortis tempor nulla id elementum Etiam gravida arcu et nibh vulputate dapibus Aliquam eget tempus diamNullam quis ullamcorper erat Vestibulum ultricies erat quis leo efficitur consequat Ut id risus facilisis ornare sem at rutrum tellus Donec a urna odio Ut et nisl quis enim dignissim sodales Vestibulum risus augue posuere id tortor vitae efficitur fermentum massa Etiam at eros rhoncus aliquet tellus interdum vestibulum nibh Phasellus consequat velit quis mi lobortis varius Phasellus semper purus nec tellus fringilla condimentum Morbi cursus nec ante eget porttitor Vestibulum vestibulum arcu sit amet elit congue tristique Nulla eleifend purus non ornare interdum Aliquam quis nulla aliquet mattis dolor nec auctor massa Vestibulum placerat placerat ante sit amet laciniaLorem ipsum dolor sit amet consectetur adipiscing elit Morbi imperdiet dui in mattis fringilla Vivamus scelerisque id erat eu imperdiet Mauris eu neque sit amet lectus facilisis tristique Vestibulum finibus quam vel tempus tempor Donec in mollis augue Duis hendrerit aliquam tortor at volutpat Vivamus ac fringilla arcu eget bibendum enimCras consectetur dui vel aliquet tincidunt velit nisl lacinia sem non accumsan risus est at mauris Vestibulum dignissim metus nec nisl dignissim at volutpat libero dignissim Sed sollicitudin lacus felis quis vehicula nisi fermentum eget Nullam venenatis orci ac bibendum pharetra justo turpis consequat orci vitae lobortis mauris mauris a mi In ut consequat lectus Curabitur purus metus vestibulum a mauris in gravida egestas enim Nunc diam erat interdum in hendrerit vel consectetur in ex Mauris est dui varius sed nunc sit amet aliquam pellentesque justo Sed placerat quis dui in blandit Nullam porta leo ipsum sit amet maximus mi mollis et Donec ac lectus dolor Mauris dictum libero dignissim tincidunt aliquam Aliquam quis urna ac ligula lobortis vulputate quis porta ligula Etiam semper ante non risus ultricies rutrumAliquam eu facilisis sem eu maximus felis Aenean quis ultricies augue a tincidunt lacus Nunc vitae massa vestibulum malesuada sem eget iaculis ante Aliquam lorem velit iaculis at ipsum non facilisis vulputate lorem Aenean fermentum imperdiet eleifend Pellentesque magna ipsum varius vel tortor in dignissim faucibus est Nullam velit nisl accumsan in tellus id suscipit varius massaAliquam eget mauris at felis bibendum semper Ut ultrices dui eu orci ultricies molestie Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae;</speak>',
      '<speak> Nam sit amet posuere eros sed tincidunt est Mauris lobortis tempor nulla id elementum Etiam gravida arcu et nibh vulputate dapibus Aliquam eget tempus diam Nullam quis ullamcorper erat Vestibulum ultricies erat quis leo efficitur consequat Ut id risus facilisis ornare sem at rutrum tellus Donec a urna odio Ut et nisl quis enim dignissim sodales Vestibulum risus augue posuere id tortor vitae efficitur fermentum massa Etiam at eros rhoncus aliquet tellus interdum vestibulum nibh Phasellus consequat velit quis mi lobortis varius Phasellus semper purus nec tellus fringilla condimentum Morbi cursus nec ante eget porttitor Vestibulum vestibulum arcu sit amet elit congue tristique Nulla eleifend purus non ornare interdum Aliquam quis nulla aliquet mattis dolor nec auctor massa Vestibulum placerat placerat ante sit amet lacinia</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'google',
      softLimit: GOOGLE_SOFT_LIMIT,
      hardLimit: GOOGLE_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(text);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      expect(item.length).toBeLessThanOrEqual(GOOGLE_HARD_LIMIT);
    });
  });
});

describe('AWS Polly limitations', () => {
  it(`Should return the correct SSML chunks when the input SSML (${ssmlExampleLargeMultipleParagraphs.length} characters) goes above the hardLimit`, () => {
    const ssml = ssmlExampleLargeMultipleParagraphs;

    const expected = [
      '<speak>Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. Vivamus elementum semper nisi.<break strength="x-strong" />Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum. Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem.<break strength="x-strong" />Maecenas nec odio et ante tincidunt tempus. Donec vitae sapien ut libero venenatis faucibus. Nullam quis ante. Etiam sit amet orci eget eros faucibus tincidunt. Duis leo. Sed fringilla mauris sit amet nibh. Donec sodales sagittis magna. Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero. Fusce vulputate eleifend sapien. Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus. Nullam accumsan lorem in dui. Cras ultricies mi eu turpis hendrerit fringilla.<break strength="x-strong" />Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia. Nam pretium turpis et arcu. Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum. Sed aliquam ultrices mauris. Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris. Praesent adipiscing. Phasellus ullamcorper ipsum rutrum nunc. Nunc nonummy metus. Vestibulum volutpat pretium libero. Cras id dui. Aenean ut eros et nisl sagittis vestibulum. Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede.</speak>',
      '<speak><break strength="x-strong" />Sed lectus. Donec mollis hendrerit risus. Phasellus nec sem in justo pellentesque facilisis. Etiam imperdiet imperdiet orci. Nunc nec neque. Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi. Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo. Maecenas malesuada. Praesent congue erat at massa. Sed cursus turpis vitae tortor. Donec posuere vulputate arcu. Phasellus accumsan cursus velit.<break strength="x-strong" />Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci. Phasellus consectetuer vestibulum elit. Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc. Vestibulum fringilla pede sit amet augue. In turpis. Pellentesque posuere. Praesent turpis. Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus. Donec elit libero, sodales nec, volutpat a, suscipit non, turpis. Nullam sagittis.<break strength="x-strong" />Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus. Ut varius tincidunt libero. Phasellus dolor. Maecenas vestibulum mollis diam. Pellentesque ut neque. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. In dui magna, posuere eget, vestibulum et, tempor auctor, justo. In ac felis quis tortor malesuada pretium. Pellentesque auctor neque nec urna.<break strength="x-strong" />Proin sapien ipsum, porta a, auctor quis, euismod ut, mi. Aenean viverra rhoncus pede. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Ut non enim eleifend felis pretium feugiat. Vivamus quis mi. Phasellus a est. Phasellus magna. In hac habitasse platea dictumst. Curabitur at lacus ac velit ornare lobortis. Curabitur a felis in nunc fringilla tristique. Morbi mattis ullamcorper velit. Phasellus gravida semper nisi. Nullam vel sem. Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam. Sed hendrerit. Morbi ac felis.</speak>',
      '<speak><break strength="x-strong" />Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede. Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi. Nunc nulla. Fusce risus nisl, viverra et, tempor et, pretium in, sapien. Donec venenatis vulputate lorem. Morbi nec metus. Phasellus blandit leo ut odio. Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem. Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis. In auctor lobortis lacus.<break strength="x-strong" />Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna. Vestibulum ullamcorper mauris at ligula. Fusce fermentum. Nullam cursus lacinia erat. Praesent blandit laoreet nibh. Fusce convallis metus id felis luctus adipiscing. Pellentesque egestas, neque sit amet convallis pulvinar, justo nulla eleifend augue, ac auctor orci leo non est. Quisque id mi. Ut tincidunt tincidunt erat. Etiam feugiat lorem non metus. Vestibulum dapibus nunc ac augue. Curabitur vestibulum aliquam leo. Praesent egestas neque eu enim. In hac habitasse platea dictumst. Fusce a quam. Etiam ut purus mattis mauris sodales aliquam. Curabitur nisi. Quisque malesuada placerat nisl. Nam ipsum risus, rutrum vitae, vestibulum eu, molestie vel, lacus. Sed augue ipsum, egestas nec, vestibulum et, malesuada adipiscing, dui. Vestibulum facilisis, purus nec pulvinar iaculis, ligula mi congue nunc, vitae euismod ligula urna in dolor. Mauris sollicitudin fermentum libero. Praesent nonummy mi in odio. Nunc interdum lacus sit amet orci. Vestibulum rutrum, mi nec elementum vehicula, eros quam gravida nisl, id fringilla neque a.<break strength="x-strong" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      softLimit: AWS_SOFT_LIMIT,
      hardLimit: AWS_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(ssml);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      const characterCountHowAWSCountsIt = removeSSMLTags(item).length;
      expect(characterCountHowAWSCountsIt).toBeLessThanOrEqual(AWS_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input SSML (${ssmlExampleLargeMultipleParagraphsWithSentences.length} characters) goes above the hardLimit`, () => {
    const largeExample = ssmlExampleLargeMultipleParagraphsWithSentences;

    const expected = [
      '<speak><s>Lorem ipsum dolor sit amet, consectetuer adipiscing elit.</s><s>Aenean commodo ligula eget dolor.</s><s>Aenean massa.</s><s>Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.</s><s>Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem.</s><s>Nulla consequat massa quis enim.</s><s>Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu.</s><s>In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo.</s><s>Nullam dictum felis eu pede mollis pretium.</s><s>Integer tincidunt.</s><s>Cras dapibus.</s><s>Vivamus elementum semper nisi.</s><s>Aenean vulputate eleifend tellus.</s><s>Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim.</s><s>Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus.</s><s>Phasellus viverra nulla ut metus varius laoreet.</s><s>Quisque rutrum.</s><s>Aenean imperdiet.</s><s>Etiam ultricies nisi vel augue.</s><s>Curabitur ullamcorper ultricies nisi.</s><s>Nam eget dui.</s><s>Etiam rhoncus.</s><s>Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum.</s><s>Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem.</s><s>Maecenas nec odio et ante tincidunt tempus.</s><s>Donec vitae sapien ut libero venenatis faucibus.</s><s>Nullam quis ante.</s><s>Etiam sit amet orci eget eros faucibus tincidunt.</s><s>Duis leo.</s><s>Sed fringilla mauris sit amet nibh.</s><s>Donec sodales sagittis magna.</s><s>Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero.</s><s>Fusce vulputate eleifend sapien.</s><s>Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus.</s><s>Nullam accumsan lorem in dui.</s><s>Cras ultricies mi eu turpis hendrerit fringilla.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; In ac dui quis mi consectetuer lacinia.</s><s>Nam pretium turpis et arcu.</s><s>Duis arcu tortor, suscipit eget, imperdiet nec, imperdiet iaculis, ipsum.</s><s>Sed aliquam ultrices mauris.</s><s>Integer ante arcu, accumsan a, consectetuer eget, posuere ut, mauris.</s><s>Praesent adipiscing.</s><s>Phasellus ullamcorper ipsum rutrum nunc.</s><s>Nunc nonummy metus.</s><s>Vestibulum volutpat pretium libero.</s><s>Cras id dui.</s><s>Aenean ut eros et nisl sagittis vestibulum.</s></speak>',
      '<speak><s>Nullam nulla eros, ultricies sit amet, nonummy id, imperdiet feugiat, pede.</s><s>Sed lectus.</s><s>Donec mollis hendrerit risus.</s><s>Phasellus nec sem in justo pellentesque facilisis.</s><s>Etiam imperdiet imperdiet orci.</s><s>Nunc nec neque.</s><s>Phasellus leo dolor, tempus non, auctor et, hendrerit quis, nisi.</s><s>Curabitur ligula sapien, tincidunt non, euismod vitae, posuere imperdiet, leo.</s><s>Maecenas malesuada.</s><s>Praesent congue erat at massa.</s><s>Sed cursus turpis vitae tortor.</s><s>Donec posuere vulputate arcu.</s><s>Phasellus accumsan cursus velit.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Sed aliquam, nisi quis porttitor congue, elit erat euismod orci, ac placerat dolor lectus quis orci.</s><s>Phasellus consectetuer vestibulum elit.</s><s>Aenean tellus metus, bibendum sed, posuere ac, mattis non, nunc.</s><s>Vestibulum fringilla pede sit amet augue.</s><s>In turpis.</s><s>Pellentesque posuere.</s><s>Praesent turpis.</s><s>Aenean posuere, tortor sed cursus feugiat, nunc augue blandit nunc, eu sollicitudin urna dolor sagittis lacus.</s><s>Donec elit libero, sodales nec, volutpat a, suscipit non, turpis.</s><s>Nullam sagittis.</s><s>Suspendisse pulvinar, augue ac venenatis condimentum, sem libero volutpat nibh, nec pellentesque velit pede quis nunc.</s><s>Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce id purus.</s><s>Ut varius tincidunt libero.</s><s>Phasellus dolor.</s><s>Maecenas vestibulum mollis diam.</s><s>Pellentesque ut neque.</s><s>Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.</s><s>In dui magna, posuere eget, vestibulum et, tempor auctor, justo.</s><s>In ac felis quis tortor malesuada pretium.</s><s>Pellentesque auctor neque nec urna.</s><s>Proin sapien ipsum, porta a, auctor quis, euismod ut, mi.</s><s>Aenean viverra rhoncus pede.</s><s>Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.</s><s>Ut non enim eleifend felis pretium feugiat.</s><s>Vivamus quis mi.</s><s>Phasellus a est.</s><s>Phasellus magna.</s><s>In hac habitasse platea dictumst.</s><s>Curabitur at lacus ac velit ornare lobortis.</s><s>Curabitur a felis in nunc fringilla tristique.</s></speak>',
      '<speak><s>Morbi mattis ullamcorper velit.</s><s>Phasellus gravida semper nisi.</s><s>Nullam vel sem.</s><s>Pellentesque libero tortor, tincidunt et, tincidunt eget, semper nec, quam.</s><s>Sed hendrerit.</s><s>Morbi ac felis.</s><s>Nunc egestas, augue at pellentesque laoreet, felis eros vehicula leo, at malesuada velit leo quis pede.</s><s>Donec interdum, metus et hendrerit aliquet, dolor diam sagittis ligula, eget egestas libero turpis vel mi.</s><s>Nunc nulla.</s><s>Fusce risus nisl, viverra et, tempor et, pretium in, sapien.</s><s>Donec venenatis vulputate lorem.</s><s>Morbi nec metus.</s><s>Phasellus blandit leo ut odio.</s><s>Maecenas ullamcorper, dui et placerat feugiat, eros pede varius nisi, condimentum viverra felis nunc et lorem.</s><s>Sed magna purus, fermentum eu, tincidunt eu, varius ut, felis.</s><s>In auctor lobortis lacus.</s><s>Quisque libero metus, condimentum nec, tempor a, commodo mollis, magna.</s><s>Vestibulum ullamcorper mauris at ligula.</s><s>Fusce fermentum.</s><s>Nullam cursus lacinia erat.</s><s>Praesent blandit laoreet nibh.</s><s>Fusce convallis metus id felis luctus adipiscing.</s><s>Pellentesque egestas, neque sit amet convallis pulvinar, justo nulla eleifend augue, ac auctor orci leo non est.</s><s>Quisque id mi.</s><s>Ut tincidunt tincidunt erat.</s><s>Etiam feugiat lorem non metus.</s><s>Vestibulum dapibus nunc ac augue.</s><s>Curabitur vestibulum aliquam leo.</s><s>Praesent egestas neque eu enim.</s><s>In hac habitasse platea dictumst.</s><s>Fusce a quam.</s><s>Etiam ut purus mattis mauris sodales aliquam.</s><s>Curabitur nisi.</s><s>Quisque malesuada placerat nisl.</s><s>Nam ipsum risus, rutrum vitae, vestibulum eu, molestie vel, lacus.</s><s>Sed augue ipsum, egestas nec, vestibulum et, malesuada adipiscing, dui.</s><s>Vestibulum facilisis, purus nec pulvinar iaculis, ligula mi congue nunc, vitae euismod ligula urna in dolor.</s><s>Mauris sollicitudin fermentum libero.</s><s>Praesent nonummy mi in odio.</s><s>Nunc interdum lacus sit amet orci.</s><s>Vestibulum rutrum, mi nec elementum vehicula, eros quam gravida nisl, id fringilla neque a.</s><break strength="x-strong" /></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      softLimit: AWS_SOFT_LIMIT,
      hardLimit: AWS_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(largeExample);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      const characterCountHowAWSCountsIt = removeSSMLTags(item).length;
      expect(characterCountHowAWSCountsIt).toBeLessThanOrEqual(AWS_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input SSML (${ssmlExampleLargeArticle.length} characters) goes above the hardLimit`, () => {
    const largeExample = ssmlExampleLargeArticle;

    const expected = [
      '<speak><s>Als ich den später so berühmten und angebeteten Fra Celeste zum ersten Male predigen hörte, war er erst in den Ortschaften bekannt, die im Umkreise seines Klosters lagen.</s><s>Ich war in die Kirche eingetreten, weil ich so ungewöhnlich viele Leute hingehen sah, aber ich erwartete mir eigentlich nichts anderes, als die Nase zu rümpfen; denn ich war hochmütig wegen der Bildung, die ich auf den hohen Schulen gesammelt hatte, und glaubte durchaus nicht an diese Genialen, die es vermöge angeborener Kräfte mit den bestgeschulten Geistern aufnehmen können.</s><s>Von einem, der erst ein Bäckerjunge gewesen, dann ein Mönch geworden war, glaubte ich, daß er höchstens dem dummen Pöbel durch Lärm, Schlagwörter und leeren Bombast imponieren könne.</s><s>Als er nun aber, dieser wundervolle Mann, in Begleitung mehrerer Geistlicher, denen er um einige Schritte voranging, durch das geöffnete Portal eintrat, fühlte ich mich sogleich gesichert und erhoben, was ich schwerlich begründen könnte, denn bei der Dunkelheit, die in der tiefen alten Kirche herrschte, vermochte ich sein Äußeres nicht zu unterscheiden.</s><s>Auch als er auf der Kanzel stand, sah ich nur, daß er eine hochgewachsene kräftige Gestalt hatte, die sich mit Unbefangenheit, aber nicht häufig bewegte, und daß die Form seines Gesichtes schön, breit und stark war.</s><s>Obgleich er schnell gegangen war, fing er sofort, ohne daßman etwas von Atemnot bemerkt hätte, zu sprechen an mit dieser einzigen, unerschöpflichen Stimme, die aus dem Busen eines Gottes zu kommen schien.</s><s>Eine Weile sprach er schlicht, gleichförmig, mit mittlerer Stärke, bis er sowohl von den immer rascher aufquellenden Einfällen wie auch vom Klange seiner Stimme berauscht wurde.</s><s>Dann behandelte er sie wie ein Instrument, Geige oder Flöte, worauf er phantasierte; etwas vorgebogen, mit verzückten Augen schien er den Tönen nachzublicken, als wären es schimmernde Paradiesvögel, die die schwarzen Pfeiler umschwebten und das Gewölbe mit Musik erfüllten.</s><s>Das Merkwürdige war nun, daß die Predigt keineswegs nur ein wohlklingender Gesang war, sondern klar geordnet und voll eigenartiger Gedanken und Betrachtungen.</s></speak>',
      '<speak><s>Sie handelte von der Sonntagsheiligung, und zwar zuerst von den Belustigungen, mit denen gewöhnlich der Feiertag verbracht wird, hernach von dem eigentlichen Sinn und Wert der gebotenen Ruhe.</s><s>Da war kein borniertes Eifern oder Scheinheiligkeit oder Tugendsalbung, nichts, worüber man hätte lachen oder was man hätte leicht widerlegen können, sondern ritterliche Verachtung gemeiner Lust.</s><s>»Gott hat uns zu Herren der Erde geschaffen,« sagte er unter anderem, »sollen wir uns von ihr knechten lassen für den vergänglichen Bettel, mit dem sie uns dingen will?</s><s>Wer wäre ich, wenn die Sklavenumarmung der Dirne Welt meine Sehnsucht stillte?</s><s>Regieren will ich die Erde von meinem Platz aus, nach meinen Kräften, als der König, der ich geschaffen bin, sechs Tage lang.</s><s>Aber am siebenten will ich den Herrn suchen, der über mir ist, und ihm dienen.</s><s>Denn auch wir möchten vergehen und anbeten.« Indem er dies letzte mit einemunbeschreiblichen Schmelz der Stimme sagte, sah ich zum ersten Male sein herzliches Lächeln, wobei die Zähne schimmerten wie Mondschein, der durch eine dunkle Wolke fällt.</s><s>Dann, nach einer kleinen Pause, fuhr er lauter fort, daß es klang, wie wenn ein Frühlingsdonner über die Himmelswölbung rollt: »Kniet mit mir und betet an!« worauf sich alle niederwarfen.</s><s>Die meisten blieben in dieser Stellung bis zum Ende der Predigt und schienen froh, ihrer leidenschaftlichen Inbrunst dadurch einen Ausdruck geben zu dürfen.</s><s>Durch die Reihen der Menschen, die sich an ihn drängten, als er die Kirche verließ, um sein Kleid oder seine Hände zu berühren, ging er mit aufrichtiger, ganz ungesuchter Gleichgültigkeit, obwohl es namentlich Frauen waren und seine Jugend eine gewisse Reizbarkeit gegenüber dem weiblichen Geschlecht entschuldigt hätte.</s><s>Aber man sah wohl, daß es diesem Liebling Gottes nicht einmal einen Kampf kostete, derartige Versuchungen zu überwinden.</s><break strength="x-strong" /><s>Ich erinnere mich, daß inzwischen die Sonne untergegangen war und ein wonnig kühler, mehr zu empfindender als sichtbarer Flor sich über die graue Kirche, den Turm und alle Gegenstände herabsenkte.</s></speak>',
      '<speak><s>Der Himmel schien mir höher zu sein, als ich ihn je gesehen hatte, und ich stand noch und schaute hinein, nachdem sich die Menge längst verlaufen hatte.</s><s>Allmählich war es mir, als höbe ich mich von der Erde und schwebte langsam nach oben, getragen von einem mächtigen, befreundeten Element, das, wie ich genau wußte, Fra Celeste regierte.</s><s>Seit diesem Abend brachte ich den Bruder nicht mehr aus meinem Sinn, und als ich kurz darauf erfuhr, daß er einen Sekretär suchte, der seine Korrespondenzen und weltlichen Angelegenheitenüberhaupt besorge, lief ich ohne Zögern zu ihm in der Meinung, die Vorsehung habe mich eigens für dieses Amt auserlesen, wie ehemals den guten Parsival für den Gral.</s><s>Ich wurde sogleich vorgelassen, der himmlische Mann betrachtete mich eine Weile aufmerksam, strich liebkosend über meine Wangen, erkundigte sich nach meinem Alter, verwunderte sich, daß ich schon dreiundzwanzig Jahre zählte, denn ich gliche, sagte er, mehr einem kleinen Fräulein von sechzehn Lenzen, und äußerte Zweifel, ob ich auch imstande sei, seine Geschäfte zu führen.</s><s>Er selbst nämlich, obschon er wie ein Erzengel mit Nachtigallen- und Adlerzungen redete, konnte weder fließend lesen noch schreiben, und da er schon damals zahlreiche Briefe empfing, die ihm gleichgültig oder gar lästig waren, suchte er jemand, der sie nach seinem Gutdünken schicklich beantwortete und ihn selbst nur in Ausnahmefällen damit behelligte.</s><s>Ich zählte ihm alle Studien auf, die ich gemacht hatte, was ihn sehr zu befriedigen schien, und er gab mir einige Briefe, die er kürzlich erhalten hatte, damit ich sie abfertige, wie es mir gut scheine, und ließ mich an seinem Schreibtisch allein.</s><break strength="x-strong" /><s>Der erste Brief, den ich öffnete, war von einer Frau, die einen starken Trieb fühlte, ins Kloster zu gehen, ihr Mann, schrieb sie, wollte sie aber nicht gehen lassen; ob es gottgefälliger sei, dem himmlischen Herrn zuliebe dem irdischen ungehorsam zu sein oder bei dem letzteren auszuharren.</s><s>Ich beantwortete den Brief folgendermaßen: »Frau, wenn Sie ins Kloster gehen, entwenden Sie Gott eine Seele, nämlich die Ihres Mannes, der mit Weltlust voll zu sein scheint und ohne Sie vielleicht zur Hölle fahren müßte</s></speak>',
      '<speak><s>Bessern Sie ihn durch das Beispiel Ihrer Tugend, bis er selbst vor den irdischen Plagen sich ins Kloster zu flüchten sehnt, und gehen Sie dann in Gottes Namen miteinander.« Der zweite Brief war ebenfalls von einer Dame, welche, um im Glauben gestärkt und vom Verderben errettet zu werden, eine Unterredung mit dem heiligen Bruder wünschte und ihn auf die Dämmerstunde eines bestimmten Tages zu sich einlud.</s><s>Diesen Brief hielt ich für das beste, unbeantwortet zu zerreißen.</s><break strength="x-strong" /><s>Der dritte war von des Bruders Abte, der sich beklagte, daß er schon lange ohne die schuldige Berichterstattung bleibe, ferner, daß der Bruder, wie ihm zu Ohren gekommen sei, bei seinen Predigten oft ins Blaue fahre und nicht daran denke, der alleinseligmachenden katholischen Kirche die Ehre zu geben.</s><s>Nach einigem Bedenken antwortete ich so: »Hochehrwürden, über das, was ich predige, bin ich nicht Herr, denn es ist Eingebung, und zwar, wie ich glaube und hoffe, des guten, nicht des bösen Geistes.</s><s>Daß Hochehrwürden solange keine Nachricht von mir empfangen haben, ist die Schuld meines Sekretärs, den ich wegen dieser Nachlässigkeit scharf getadelt habe.« Eben als ich dies geschrieben hatte, trat Fra Celeste wieder ein und verlangte, meine Arbeit zu prüfen.</s><s>Nachdem er die beiden Briefe gelesen hatte, lobte er mich sehr und bat mich, bei ihm zu bleiben; er sei zufrieden mit mir.</s><break strength="x-strong" /><s>Von diesem Augenblick an blieb ich der Gefährte dieses einzigen Mannes, hatte mein Zimmer unmittelbar neben dem seinigen und begleitete ihn auf seinen Reisen; denn nun kam die Zeit, wo alle Städte unseres Landes wetteiferten, ihn in ihren Mauern zu empfangen.</s><s>Die erste Reise, die wirzusammen unternahmen, ging nach seiner Vaterstadt, die er wiederzusehen wünschte.</s><s>Dort ereignete es sich, daß bei der ersten Predigt, die er hielt, eine vornehme, reich und edel gekleidete Dame, während er in einer Pause angab, worüber er das nächste Mal predigen würde, von einer Ohnmacht angewandelt wurde.</s><s>Sie erholte sich aber sogleich und drängte sich am Schlusse, obschon es ihr sichtlich ungewohnt und zuwider war, sich im Volksgewühl zu bewegen, dicht zu Fra Celeste hin, der sie mit einer Gebärde des Abscheues oder Zornes zurückstieß.</s></speak>',
      '<speak><s>Als ich mit dem Bruder allein war, sagte ich hierauf bezüglich: »Nun glaube ich wirklich, daß uns der Teufel in Gestalt von Frauen versucht; denn wenn Sie nicht den verkappten Satan in jener lieblichen Dame durchschaut haben, begreife ich nicht, wie Sie die Kraft haben konnten, sie so hart zurückzustoßen.«</s><break strength="x-strong" /><s>Damals hatte ich den Bruder noch nicht in der fliegenden Wut gesehen, daher erschrak ich nicht wenig, als seine stillen, schattigen Augen plötzlich groß und fürchterlich wurden und er so mit gezückten Schwertblicken hart vor mich trat.</s><s>»Ist das deine Andacht,« rief er, »daß du nach den Frauen umherschaust?</s><s>O ekelhafte Schwäche und Verderbnis dieses Kotleibes!</s><s>Errötest du nicht über deine Zuchtlosigkeit, wenn du dein keusches Jugendgesicht im Spiegel betrachtest, das Gott dir gegeben hat?</s><s>Wenn du es besudeln solltest mit Gedanken oder Taten, verlasse dich auf mich, daß ich dich umbringe mit diesen meinen Händen.« Dabei war sein ganzer Körper gewaltsam gespannt, und seine Hände, die ich bisher nur schwermütig ruhen oder einen Nachdruck der Rede mit gleichgültig stolzerGebärde hatte begleiten sehen, bekamen etwas so Ehernes und Unerbittliches, daß ich, ohne es zu wollen, ein wenig vor ihm zurückwich.</s><s>Aber trotzdem es mir nicht recht geheuer war, entzückte mich der Anblick des feuerspeienden Mannes, wie man bei Ungewittern, Wolkenbrüchen und Stürmen zugleich vor Angst und Wonne schaudert.</s><s>Ich segnete den Umstand, daß Küsse kein rotes Mal oder sonst eine Spur zurücklassen, denn ich zweifelte nicht, der Bruder würde mich stehenden Fußes erschlagen, wenn er wüßte, daß ich die Lockungen meines unruhigen und naschhaften Herzens zuweilen erfolglos bekämpft hatte.</s><break strength="x-strong" /><s>Übrigens legte sich das Wetter, so schnell es gekommen war, vielleicht unter dem Eindruck meiner erschrockenen und liebenden Blicke; wenigstens behandelte mich Fra Celeste im Verlaufe des Tages mit so engelgleicher Zartheit, daß ich mich mitten im Himmel wähnte und mir schwur, künftig dem erhabenen Beispiel meines Herrn nachzueifern und aller Frauenliebe zu entsagen, um mich ihm ganz ohne Einschränkung zu ergeben.</s></speak>',
      '<speak><break strength="x-strong" /><s>Am folgenden Morgen fiel mir unter den eingelaufenen Briefen sogleich einer durch die reizende, sichtlich einer Frau gehörige Handschrift auf.</s><s>Die Buchstaben waren kühn, schlank, behend und prächtig und tanzten in so anmutigen Neigungen über das Papier, als ob sie mit den Augen des Lesers kokettieren wollten.</s><s>Zu meinem wachsenden Erstaunen las ich nun folgendes: »Dolfin, ich habe Dich erkannt und Du hast mich von Dir gestoßen.</s><s>Als ich Dich sah, verwandelte sich mein Blut in Tränen und meine Tränen wurden Blut.</s><s>Liebst Du mich denn nicht mehr ?</s><s>Ich liebe Dich so sehr!</s><s>Ich will Heimat, Eltern und Mann verlassenund die Deine sein.</s><s>Laß mich, sag nicht nein zu meinem Herzen.</s><s>Wenn Du mich rufst, komm ich und bleibe bei Dir ewig.</s><s>Aglaia.«</s><break strength="x-strong" /><s>Dieser Schrei der Liebe, obschon er mir nicht galt, erschütterte mein Herz, und ich saß lange und starrte in den Brief hinein.</s><s>Die stürmischen, geschmeidigen Buchstaben schienen mir wie arabische Rosse über eine Wüste zu jagen, ich hörte den Rhythmus ihrer aufschlagenden Hufe, versank darüber in Träumerei, und ehe ich mir&apos;s versah, hatte ich einen Reim erdacht als Antwort auf die geheimnisvolle Liebesklage.</s><s>Der Vers hieß so:</s><break strength="x-strong" /><s>Durch den Himmel hin wandr ich allein,</s><break strength="x-strong" /><s>Hast du Flügel, o komm und sei mein;</s><break strength="x-strong" /><s>Sei des einsamen Mondes Weib,</s><break strength="x-strong" /><s>Wie ein Stern blitzt dein silberner Leib –</s><break strength="x-strong" /><s>Ich hatte gerade die letzten Worte niedergeschrieben, als Fra Celeste bei mir eintrat und, wie er häufig zu tun pflegte, über meine Schulter sah.</s><s>Es gelang mir nicht mehr, das Geschriebene zu verdecken, und so kam es, daß er sowohl meinen Vers wie auch den Brief las, der ihn veranlaßt hatte.</s><s>Wenn ich auch gleich erraten hatte, daß es sich hier um nichts Geringes handle, überraschte mich doch, was ich nun wahrnahm.</s><s>Eine gewaltige Eiche, in die der Blitz fährt, die Feuer fängt und wie eine vom Sturme hin und her gewehte Fackel gegen den Himmel lodert, der möchte ich Fra Celeste vergleichen, als er den Inhalt des verhängnisvollen Schreibens ins Herz gefaßt hatte.</s><s>So hing aber alles zusammen.</s></speak>',
      '<speak><break strength="x-strong" /><s>Fra Celeste, der mit seinem weltlichen Namen Dolfin hieß,war als armes, elternloses Kind zwischen fremden Leuten aufgewachsen und wurde, nachdem er kaum das Notdürftigste in der Schule erlernt hatte, einem Bäcker in die Lehre gegeben.</s><s>Als er etwa achtzehn Jahre alt war, kam er wie nach einem dumpfen Traume zu sich und verspürte einen heißhungrigen Trieb, seine Kräfte an etwas Ungeheuerem zu üben.</s><s>Auf ein bestimmtes Ziel richtete sich sein Ehrgeiz nicht, auch hätte er die Bildung dazu nicht besessen, aber alles, was er um sich her sah, schien ihm verächtlich und viel zu klein für den Riesenmut seines Herzens.</s><s>So war er beschaffen, als er sich in ein vornehmes und reiches Fräulein verliebte, in deren elterliches Haus sein Beruf ihn führte.</s><s>Wenn nicht ein überirdischer Geist in ihm gewesen wäre, möchte es ihm als einem armen Bäckerburschen niemals gelungen sein, eine Beziehung zu dem Fräulein herzustellen, besonders da er nicht einmal auffallend schön war, außer daß der später so wohlredende, damals noch blöde Mund, schön, stark und rot, die glänzendsten und vollkommensten Zähne besaß und mit seinem Lächeln, das freilich nur selten erschien, alle Herzen bezaubern konnte.</s><break strength="x-strong" /><s>Die Bekanntschaft hatte Fra Celeste so angebahnt, daß er einmal, anstatt das Haus zu verlassen, an der Gartentür stehenblieb und unentwegt in den Garten hineinblickte, wo das Fräulein allein auf einer Bank saß.</s><s>Etwas beunruhigt, fragte sie endlich, worauf er noch warte, und als er, ohne sich zu besinnen, sagte: auf eine von den Rosen, die an den hohen Stämmen längs eines Gebüsches hinter der Bank blühten, bog sie sich rückwärts, pflückte eine, ging zu ihm hin und gab sie ihm, der kurz dankte und fortging.</s><s>Erst dannwunderte sie sich über das, was sie, überrascht von der Seltsamkeit eines solchen Betragens, gleichsam unbewußt getan hatte, und lachte sich selbst aus; nichtsdestoweniger suchte sie eine Gelegenheit, den wunderlichen Bäcker wiederzusehen.</s><s>Ihre Phantasie wurde vollends angeregt, als er kleine Gespräche mit ihr zu führen begann; denn nun verspürte sie den Genius, der in seiner Brust wohnte.</s></speak>',
      '<speak><s>Obgleich er wenig oder nichts gelernt hatte, wußte er seine Einfälle und Beobachtungen so zu äußern, daß das Geringste bedeutend erschien, ja, er konnte Albernheiten, wie daß der Himmel blau und das Wasser naß sei, so vortragen, daß man aufhorchte und es tagelang mit sich herumtrug.</s><s>Anfänglich gefiel es dem Fräulein, mit ihrem sonderbaren Verehrer von der Familie und den Bekannten geneckt zu werden, allmählich aber vermied sie es, von ihm zu sprechen und mit ihm gesehen zu werden, und wenn man sie dennoch auf ihn anredete, konnte sie spitzige Antworten geben, wie daß der arme Bäckerjunge gescheiter, stolzer und edler sei als ihre hochgeborene Umgebung.</s><s>Nun zeigten ihre Eltern Mißfallen an dem Verkehr, was wiederum ihre Zuneigung verstärkte, und es kam dazu, wovon mir das einzelne nicht bekannt ist, daß Liebesworte und Küsse zwischen den beiden jungen Leuten gewechselt wurden.</s><s>Zugleich aber bemächtigte sich des Fräuleins heimliches Schuldgefühl, so daß sie ihn bald hochmütig vermied, bald mit leidenschaftlicher Hingebung suchte, während er dieser Verwirrung gegenüber sich immer sicherer und männlicher fühlte.</s><s>Des Fräuleins Eltern, deren Geschlechtsnamen ich nicht nennen will – sie selbst hieß Aglaia – erfuhren oder ahnten, wie gefährlich diese Spielerei mit der Zeit geworden war, undbeschlossen, damit nicht noch Ärgeres geschehe, ihr Kind so schnell wie möglich standesgemäß zu verheiraten.</s><s>Sie warfen ihr Auge auf den Grafen, den ich später selbst kennengelernt habe, einen großen, prächtigen Mann mit geräumigen Augen und geringelten schwarzen Haaren, die angenehm glänzten, so daß er wohl schön genannt zu werden verdiente.</s><s>Er war aber auf diese Vorzüge weit weniger eitel als auf seine Geistesgaben und Glücksgüter, in welchen letzteren er die ersten zeigen wollte, insofern nämlich, daß er von allem das Schönste und Seltenste besaß.</s><s>Was allgemein geschätzt und kostbar war, schaffte er an, und so bewarb er sich auch um Aglaia, deren Witz, Eigenart, liebliches Wesen und körperliche Vorzüge er von allen Seiten hatte rühmen hören.</s></speak>',
      '<speak><s>Da nun das Fräulein von den älteren und verständigen Leuten Tag für Tag das Los einer armen Bäckersfrau in grellen Farben beschreiben hörte, verlor ihr Herz die kindische Zuversicht, und sie fing an, sich für eine überspannte, romanlesende Person zu halten, daß sie die gute, klare, gesunde Wirklichkeit so hatte übersehen können.</s><s>Sie sah sich von Zichorien- und Kartoffelnahrung aufgeschwemmt und unförmig geworden, von vielen Kindern umzetert, die gesäugt, gewaschen und geprügelt sein wollten.</s><s>Von Dolfin waren ihr hauptsächlich die weißen Arme und Hände gegenwärtig, die sie bald mit klebrigen Liebkosungen, bald mit Schlägen verfolgten; denn ohne das, hatte man ihr gesagt, würde es durchaus nicht abgehen.</s><s>Alle ihre Lieblichkeiten, die weiche Zartheit ihrer Hände, ihr träumerisches oder ausgelassenes Wesen, ihren Aberwillen gegen das Alltägliche und Gemeine würde man ihr dort zum Vorwurf machen und allmählich austreiben, bissie sich in die platte Form des Durchschnittspöbels hineingebückt hätte.</s><s>Wenn sie sich überlegte, daß sie von allen ihren Zweifeln auch nicht ein Wort zu Dolfin hätte sagen mögen, fühlte sie, wie fremd sie einander waren; sie gehörten zwei verschiedenen Welten an und hatten sich nur auf einem Inselchen des Grenzflusses ein Stelldichein gegeben.</s><s>Dies alles bedenkend, glitt sie dem behaglichen Schicksal, das man ihr zubereitet hatte, mit guter Miene in den Schoß, ja, um es besser bewerkstelligen zu können, verliebte sie sich sogar unwillkürlich ein wenig in ihren Bräutigam, wozu ihr der Ärger über sein pompöses Betragen und die Reue über die Unarten, die sie ihm zufügte, das beste Mittel waren.</s><s>Zwar hatte sie noch einige Begegnungen mit Dolfin, der, als er erfuhr, was im Werke war, den Jammer seines zärtlichen Herzens hinter grobem Zürnen verbergend, sie fürchterlich anherrschte und anfänglich dadurch das Feuer ihrer Liebe neu anfachte.</s><s>Aber als sich das rechtmäßige Gefühl für ihren künftigen Gatten mehr und mehr ausbildete, schämte sie sich der Worte und Küsse, die sie mit Dolfin gewechselt hatte, und wich ihm aus; und als er trotzdem ihren Weg kreuzte, wurde sie in ihrer Angst schnippisch und böse und nannte ihn einen Narren, daß er eine mutwillige Spielerei für Ernst genommen habe.</s></speak>',
      '<speak><break strength="x-strong" /><s>Dies war der Anlaß, der ihn in das Kloster führte.</s><s>Der bloße Liebeskummer hätte ihm die Welt vielleicht nicht dermaßen verleidet, aber die Tatsache, daß sein Nebenbuhler das Glück davongetragen hatte, weil er reich war, daß man also auch Menschenseelen, das edelste und heiligste Besitztum, sich erkaufen kann, das füllte ihm die Brust bis andie Kehle mit Groll und Verachtung, dazu mit Stolz, weil er solcher Niedertracht und Schwäche nicht unterworfen war.</s><s>Er hätte stundenlang auf geprägtem Golde gehen können, ohne sich einmal danach zu bücken, ja ohne nur hinzublicken.</s><s>Nun entwickelte sich die Krankheit, zu der er von jeher Anlage in sich gehabt hatte, mehr und mehr, die nämlich, daß er sich einsam fühlte, einsam wie eine göttliche Seele in einem unreinen Körper, einsam wie ein übriggebliebenes Tier in einem verschütteten, ausgestorbenen Dorfe.</s><s>Er litt unter dieser Einsamkeit in Gesellschaft von vielen ebensogut, wie wenn er sich allein mit Beten oder häuslicher Arbeit beschäftigte, wie auch wenn er mit mehreren ein Gespräch führte; es geschah ihm dann wohl, daß die Menschen und Gegenstände von ihm wegzurücken schienen in die leere Unendlichkeit, wo er sie nicht mehr erreichen konnte, obgleich er sie mit derselben Deutlichkeit wie vorher sah.</s><s>Denn auch die Mönche trieben es nicht viel anders als die Leute draußen in der Welt, ließen sich nichts abgehen, bekümmerten sich trotz allen Betens und Singens viel mehr um irdische als um himmlische Dinge, kurz, er konnte sich durchaus nicht vorstellen, daß Gott sein Reich mit diesen Maultieren bevölkern möchte.</s><s>Da er nun auch diese Menschen, deren Profession doch die überirdische war, so gemeinplätzig und überflüssig fand, wurde Zorn gegen alle Sterblichen in ihm herrschend, die das Ebenbild Gottes durch den Schmutz schleiften, und er wurde, so schweigsam er sonst auch war, stets beredt, wenn er sich darüber auslassen konnte.</s><s>Dies zeigte sich, als er an einem Bußtage einmal an der Reihe war, zu predigen, und der Sturm seiner Rede so stark daherrauschte, daßauch die schläfrigsten Mönche in ihrem dickhäutigen Herzen darunter erbebten.</s></speak>',
      '<speak><s>Der Abt suchte das neuentdeckte Talent sogleich zu verwerten, indem er ihn öffentlich reden ließ, und die Wirkung war denn auch über alle Erwartung groß.</s><s>Die Melancholie seines jungen Gesichts und die Süßigkeit seiner Stimme, die seine Worte auf Schwanenfittichen über die Häupter der Menge dahintrug, verführten die Sinne; dazu kam aber die Macht seiner Seele über die Seelen, die er aus dem Schlamm der Erde in den Äther emporriß.</s><s>Er wurde nun häufig zum Predigen in die benachbarten Dörfer und Städte geschickt, was dem Kloster hübsche Summen eintrug, da er der Regel gemäß wie alle anderen Mönche persönliches Eigentum nicht haben durfte, die Gemeinden aber reichlich zusammensteuerten, um ihn auf der Kanzel zu sehen.</s><s>Er fing nun an, eine berühmte Persönlichkeit zu werden, und weltliche Freuden und Ehren taten sich ihm überall auf: wo er hinkam, suchten die angesehensten Männer und Frauen seine Gesellschaft.</s><s>Er, der als Kind und Jüngling in jeder Hinsicht gedarbt hatte, hätte nun nach Herzenslust genießen können; anstatt dessen steigerte sich seine Geringschätzung aller Lust der Welt, je mehr sie sich ihm darbot, weil er den Schmerz seiner Sehnsucht, wenn er innewurde, daß sie durch nichts gesättigt werden konnte, um so empfindlicher spürte.</s><break strength="x-strong" /><s>Dies war der Zeitpunkt, wo ich ihn kennenlernte und wo er Aglaia wieder begegnete.</s><s>Aglaia hatte unterdessen, an der Seite ihres Gemahls, der an Geist wie an Körper immer fetter und bequemer geworden war, ohne Sorgen zwar, aber auch ohne innere Genugtuung gelebt.</s><s>Da ihr einziges Kind früh gestorben war und ihr ein dringender Grundsich ernstlich zu beschäftigen, fehlte, blieb ihr Muße genug, über den Lauf der Welt nachzudenken und zu träumen, und sie malte sich oft aus, wie es geworden wäre, wenn sie, anstatt sich von der kümmerlichen Menschenvorsicht binden zu lassen, Dolfin gefolgt wäre, der jetzt in seinem Kloster nicht einsamer war als sie in ihrem Palaste.</s><s>Von seinem Ruf als Buß- und Sittenprediger hatte sie, die sich für das Kirchliche nicht interessierte, nichts vernommen, und als sie die Ankündigung an den Häusermauern las, daß der berühmte Fra Celeste eine Predigt über den Kirchenbesuch und das Gebet halten werde, trat sie in die betreffende Kirche, in deren Nähe sie eben war, ein, ohne zu ahnen, wen sie wiedersehen sollte.</s></speak>',
      '<speak><s>Eigentlich hatte sie sich nur unterhalten und ausruhen, allenfalls etwas Neues hören wollen, aber die Rede des Mönches schüttelte ihr Herz gewaltig, wenn sie eben auch nicht Schritt für Schritt dem Gedankengange folgte, den er entwickelte; sie vertraute sich dem starken Fluge seiner Worte, der sie weit wegtrug über Länder und Meere in eine fremde und doch heimische Herrlichkeit.</s><s>Ihre Seele kehrte von der Reise zurück, als die Musik der Stimme über ihr plötzlich schwieg; dann räusperte sich Fra Celeste und gab in seinem gewöhnlichen, halblauten Sprechtone an, welches der Text seiner nächsten Predigt sein würde.</s><break strength="x-strong" /><s>Gleich das allererste Wort schlug wie ein Hammer an Aglaias Herz, die Erinnerung stürzte daraus hervor und füllte aufschwellend ihre Brust an, was sie zuerst als eine Wollust, dann mit Angst empfand, bis es sie zu ersticken drohte.</s><s>Es begab sich nun alles, was ich bereits erzählt habe: sie schrieb jenen Brief an Dolfin, erwartete aber die Antwort nicht,sondern erforschte seine Wohnung, verschaffte sich Zutritt zu ihm und stand auf der Schwelle seines Zimmers, ehe er noch die Empörung seines Blutes, die durch ihr Schreiben entstanden war, bemeistert hatte.</s><break strength="x-strong" /><s>Wenn ich diese Liebliche aus ihrem Grabe oder vielmehr aus ihrer himmlischen Verklärung auf einen Augenblick zurückrufen könnte, möchte ich sie noch einmal so sehen wie jenes erste Mal: die Augen voll Tränen, die Lippen zitternd vor Sehnsucht und Hoffnung, wie eine zaghafte Seele vor dem halboffenen Tore des Paradieses, goldig umflossen von dem himmlischen Lichte, das daraus hervorströmt, aber demütig und furchtsam zurückgebogen, wenn vielleicht der Engel mit dem Schwerte die Sünderin verscheuchen wollte.</s><s>Einige Augenblicke mochte sie so in schwebendem Zögern dagestanden haben, dann stürzte Fra Celeste auf sie zu, nahm sie in die Arme, preßte sie gegen seine Brust, bedeckte ihr tränenüberströmtes, glückseliges Gesicht mit Küssen, faßte ihren Kopf zwischen seine Hände und drückte ihn an seinen Hals, das alles in einem Nu und abwechselnd viele Male hintereinander, so daß ich dachte, es müsse der feinen, schlanken Frau Hören und Sehen vergehen.</s></speak>',
      '<speak><s>Was sich weiter zwischen ihnen begab, kann ich aus eigener Anschauung nicht erzählen, denn ich hielt es für angemessen, mich zu entfernen; jedenfalls fand eine völlige Aussöhnung statt, und der Beschluß wurde gefaßt, sich künftig nicht mehr voneinander zu trennen, vorausgesetzt natürlich alle die Beschränkungen, die die Verhältnisse ihnen auferlegten.</s><break strength="x-strong" /><s>An diesem Tage predigte Fra Celeste von der himmlischen und irdischen Liebe so über alle Maßen herrlich undweihevoll, daß man das allumfassende Herz Gottes selbst in seinen Worten klopfen zu hören glaubte.</s><s>Einige junge Wüstlinge und viele leichtfertige Frauen legten in seine Hand das Gelübde künftiger Keuschheit ab, und es war gewiß nicht einer unter den Zuhörern, der sich nicht dessen geschämt hätte, worauf er sich bisher in seiner Verblendung etwas zugute zu tun pflegte.</s><break strength="x-strong" /><s>Der Ruhm des wundervollen Mannes nahm jetzt schnell zu, alle Zeitungen sprachen von ihm, seine Reden wurden gedruckt und gesammelt.</s><s>Wenn er sie nun zufällig irgendwo fand und las, mißfielen sie ihm in solchem Grade, daß er sie geradezu für abgeschmacktes, leeres Gewäsch erklärte und beschloß, seinen Geist auszuarbeiten und seine Kenntnisse zu vermehren, um ihnen einen gediegenen Gehalt zu geben.</s><s>Zu diesem Zwecke studierte er viele wissenschaftliche Werke, theologische, historische, philosophische, ja auch naturwissenschaftliche, und zwar las er nach kurzer Übung ein solches Buch schneller, als man ein Glas leichten Weines trinkt, ohne daß ihm etwas Wesentliches entgangen wäre.</s><s>Nun kamen von nah und fern Gelehrte, Professoren der Universitäten und die ersten Staatsmänner und zahlten hohe Preise für einen Platz in der Kirche, wenn Fra Celeste predigte, während seine Anziehungskraft für das niedere Volk sich nicht verminderte.</s><s>Unter der Last der Gelehrsamkeit wurde sein Genius nicht lahm und steif; niemals gab es einen Prediger, der die gepflegtesten wie die vernachlässigtsten Geister gleich mächtig beherrschte.</s><break strength="x-strong" /><s>Was Aglaia betrifft, so entschloß sie sich kurz und gut, ihren Mann ein für allemal zu verlassen und uns zu begleiten oder wenigstens mit uns, wo immer wir uns längere Zeitaufhielten, zusammenzutreffen.</s></speak>',
      '<speak><s>Ohnehin war es dem Bruder unleidlich, auch nur das seelenloseste Stückchen ihrer Gegenwart auf Augenblicke einem anderen gönnen zu müssen, und seine Wut auf den Grafen machte es wünschenswert, daß Aglaia demselben entzogen und damit die Ursache seiner Eifersucht hinfällig würde.</s><s>Konnte sie auch seine Wohnung nicht teilen, so war sie ihm doch leicht erreichbar, und es verging selten ein Tag, ohne daß sie sich gesehen hätten.</s><s>Häufig begleitete ich ihn bei diesen Besuchen, besonders wenn wir die Abendmahlzeit zusammen einnahmen, welche Stunden mir immer wie ein Glanz und Leuchten in der Erinnerung bleiben werden.</s><s>Die allmächtige Zunge des herrlichen Mannes war dann meist schweigsam, und er ließ Aglaia mit mir scherzen und schwatzen.</s><s>Ich höre sie noch: wie ein kleines tanzendes Wasser, das sich von einem himmelhohen Felsen herabstürzt und schäumend von Klippe zu Klippe springt, hüpfte das übermütige Geplauder von ihren beweglichen Lippen.</s><s>Manchmal, wenn ich einschlafe, ist es mir, als neigte sie sich über mich und ließe ein feines, singendes, plätscherndes Wiegenlied in mein wehmütig horchendes Ohr rieseln.</s><s>Ach, ob die singende Seele, wenn sie die Erde wieder aufsuchte, bei mir verweilen würde?</s><s>Während sie mit mir sprach, ruhten ihre Augen mit dem stillen, feuchten und glänzenden Blick auf Dolfin, von dessen Antlitz an diesen Abenden die Traurigkeit ganz verschwunden war.</s><s>Er glich einem kleinen Jungen mit glücklichem Weihnachtsgesicht, in dem Augen und Zähne blinken und lachen.</s><break strength="x-strong" /><s>Überhaupt war er jetzt fast immer gehoben und heiter, nur dann und wann wurde er von einer unerhörtenMelancholie ergriffen, die ihn jählings wie eine hungrige Wölfin anpackte, auch wenn er in Aglaias Gesellschaft war.</s><s>Ich legte dies anfänglich als ein Symptom inneren Zwiespalts aus, hervorgerufen etwa durch das Unrechtmäßige seines Verhältnisses zu der geliebten Frau, was auch ihre eigene Meinung war.</s><s>Denn als ich sie einmal fragte, ob sie sich keine Gedanken darüber mache, daß sie wider göttliches und menschliches Recht ihren Gatten verlassen habe, sah sie mich, die Unvergleichliche, Süße, mit großen Augen ängstlich an und sagte, sie wisse wohl, daß sie unrecht gehandelt habe, könne es aber durchaus nicht bereuen, hingegen fürchte sie, Dolfin, der ja ein heiliger Mann sei, nehme Anstoß daran, und ich möchte ihr helfen, solche Bedenken, wenn sie ihm kämen, zu zerstreuen.</s></speak>',
      '<speak><s>Allmählich fand ich aber heraus, daß diese Anwandlungen von Traurigkeit nichts als grimmige Rückfälle seiner alten Krankheit waren, daß er sich einsam fühlte zwischen den lebendigen Erscheinungen, sobald ihm irgendwie zu Bewußtsein kam, wie sie in beständiger Verwandlung begriffen in den Tod eilten.</s><s>Ein fallendes Blatt, ein körperliches Unbehagen, ein verlängerter Schatten, das Schwinden der Zeit konnte unversehens diesen grausamen Gedankengang in ihm wecken; ihn wegzuscheuchen, gelang aber fast immer bald einem innigen Wort oder Blick Aglaias.</s><s>Auch während des Predigens konnte die Melancholie ihn plötzlich ergreifen und aus der Stimmung des angefangenen Themas herauswerfen, was ihn aber keineswegs verwirrte, sondern ich mußte dann an einen Holzkloben denken, der langsam fortgeglimmt hat, nun aber recht Feuer fängt und mit lichterloher Flamme brennt.</s><s>Er sang das Lied von der Vergänglichkeit allesIrdischen bald mit den kreischenden Trompetentönen singender Schwäne, bald leise und traurig wie unter rinnenden Tränen; aber wie das Meer, dessen ewig wiederkehrender Wellenbrandung man Tag und Nacht nicht müde wird zu lauschen, langweilte er nie durch das eintönig tiefe Läuten und Rauschen seines Vortrages.</s><break strength="x-strong" /><s>Während dieser Zeit war ich im Dienste Dolfins außerordentlich beschäftigt.</s><s>Der Graf nämlich hatte sich bei der Erklärung seiner Frau, sie werde nicht zu ihm zurückkehren, sondern sich der Einsamkeit ergeben, keineswegs beruhigt, vielmehr ihr nachgespürt, wohin sie ging und was sie tat, und war dabei auf die Vermutung geraten, daß an ihrer rätselhaften Flucht Fra Celeste schuld sei.</s><s>Da nun seine Briefe an Aglaia unbeantwortet blieben, wendete er sich an ihn, welcher aber nichts davon erfuhr, denn ich vollführte den ärgerlichen Briefwechsel, der sich daraus ergab, ohne ihn durch den weltlichen Plunder in seiner himmlischen Versunkenheit zu stören.</s><break strength="x-strong" /><s>Auf die erste warnende Anzeige des Grafen, daß ihm seine Frau, er wisse wohl mit wem, davongelaufen sei, antwortete ich im Namen Fra Celestes, daß ich ihn aufrichtig beklage, daß eben leider die Verderbnis auch im weiblichen Geschlechte immer zunehme, daß aber im Grunde an einer Frau nicht viel gelegen sei, er möge Trost bei Gott suchen, wo er ihn sicher mit der Zeit finden werde.</s></speak>',
      '<speak><s>Die folgenden Briefe, welche nun schon bedrohlicher klangen, beantwortete ich, indem ich sagte, nichts schmerze mich mehr, als wenn Frauen in trauriger Kurzsichtigkeit und Verkennung statt meines Geistes, der von Gott erfüllt sei, meine arme, fleischliche Person liebten.</s><s>Wenn seine Frau dieseunglückliche Verwechslung begangen habe, wollte ich gerne für ihre Erleuchtung beten.</s><s>Ein andermal hielt ich ihn ernstlich dazu an, sich zu prüfen, ob er nicht durch etwelches Sündenwesen seiner Frau Ursache gegeben hätte, ihn zu meiden, worauf er sich eingehend mit Darlegung seines ganzen Lebenswandels verantwortete.</s><s>Ich machte ihn auf mancherlei Lasterhaftigkeit in demselben, den er für durchaus ehrbar und tadellos hielt und an dem auch vom landläufigen Standpunkt aus nicht viel auszusetzen war, aufmerksam; schließlich aber deutete ich ihm an, daß ich argwöhnte, er sei mit einer fixen Idee behaftet, er möge, wenn er glaube, daß es nützlich sei, meine Predigten lesen, schreiben könne ich ihm künftig nicht mehr, da es doch zu nichts führe und meine Zeit kostbar sei.</s><s>Ich erhielt darauf noch einen Brief des Grafen, worin er ankündigte, daß er sich nunmehr an den Abt von Fra Celestes Kloster wenden würde, um zu seinem Rechte zu kommen, und nach kurzer Frist hatte ich denn auch diesen Mann Gottes auf dem Halse, mit dem ich ohnehin in heftigem Streit begriffen war.</s><break strength="x-strong" /><s>Seitdem er für Aglaia zu sorgen hatte, welcher der Graf natürlich die Mittel, um fern von ihm zu leben, nicht gab und die auch von den erzürnten Eltern ununterstützt gelassen wurde, war Fra Celeste sehr auf Geld erpicht und wollte nur noch gegen hohe, ihm selbst ausgezahlte Vergütungen predigen, wie wenn er eine Privatperson gewesen wäre.</s><s>Hierauf wollte sich der Abt durchaus nicht einlassen und ergoß sich in langen Hirtenbriefen über die selige Armut, und daß der Bruder sich durch die Welt, mit der er jetzt in Berührung komme, nicht solle verderben lassen.</s><s>Ich schrieb, wie ich das Geld nur zu wohltätigenund ähnlichen Zwecken gebrauchte, übrigens aber so frugal lebe, wie ich es unter seiner Leitung im Kloster gelernt hätte.</s></speak>',
      '<speak><break strength="x-strong" /><s>Der Abt, der ein höchst frommer und milder Mann war, gerade deswegen aber immer besorgte, man möchte sein Ansehen nicht respektieren und ihn in den Hintergrund drängen, gab sich damit nicht zufrieden und bestand trotz vieler erfinderischer und künstlicher Briefe, die ich verfaßte, auf dem Rechte des Klosters.</s><s>Als nun noch die Angelegenheit des Grafen dazu kam, fing er an, von der seligen Keuschheit zu sprechen, und daß er nun wisse, wozu der Bruder das viele Geld brauche, und wie es ihm weh tue, daß ein Schaf aus seinem Stalle sich so verirre.</s><s>Es sei wahr, antwortete ich, halb betrübt, halb entrüstet, daß unzählig viele Frauen zur Kirche wallten, um mich predigen zu hören, was aber nicht zu tadeln, sondern zu loben sei, da ich nur von erbaulichen Dingen spräche, die sie belehren und bessern könnten.</s><s>Daß aber die neidischen und eifersüchtigen Männer diesen Umstand benützten, um mich zu verdächtigen, sei traurig, noch trauriger, daß er solchen Einflüsterungen Glauben schenke.</s><s>Der Abt schrieb zurück, er halte für das traurigste, daß sie auf Wahrheit beruhten, er sehe aber wohl ein, daß er mit seiner Sanftmut über mein verhärtetes Herz keine Macht habe, und werde deshalb die Sache dem Heiligen Vater selbst vorlegen, an welches höchste Haupt der Christenheit auch der Graf sich entschlossen habe, zu gelangen; dieser werde wohl noch Mittel in der Hand haben, mir beizukommen.</s><break strength="x-strong" /><s>Bald danach lief ein Schreiben aus Rom vom Kardinal San Fiori ein, einer überaus klugen, feinen und liebenPerson, was sich denn gleich in den ersten Zeilen verspüren ließ.</s><s>Nur der Schlechte, schrieb er, glaube nicht an die Tugend, er dürfe sich zwar nicht rühmen, gut zu sein, denn das sei bekanntlich nur der einige Gott und vielleicht einige Auserwählte, auf denen sein Geist ruhe, aber es sei sein Glück und sein Stolz, an das Gute zu glauben.</s><s>Er hoffe deshalb, daß es mir, das heißt dem Fra Celeste, gelingen werde, das schwarze Gewölk, das sich um die helle Sonne meines Tugendruhmes zusammenballe, zu zerstreuen, glaube aber, meiner Gegenwart würde das leichter werden als schriftlicher Verständigung.</s></speak>',
      '<speak><s>Da außerdem der Heilige Vater selbst wie alle Kardinäle schon lange wünschten, die Stimme des berühmten Predigers zu vernehmen, lüde er mich ein, nach Rom zu kommen, wo ich meinem wahren Berufe, die Menschheit zu erleuchten, obliegen und nebenbei die beiden anhängigen Geschäfte erledigen könnte.</s><break strength="x-strong" /><s>Fra Celeste, dem ich nun in Kürze alles Vorgefallene mitteilte, erklärte sich sofort bereit, nach Rom zu gehen, doch fanden wir für gut, daß ich voranreiste, um die Verhandlungen einzuleiten.</s><s>In dem Kardinal San Fiori, dem ich mich sogleich vorstellte, fand ich einen Mann von äußerst höflichen und liebenswürdigen Sitten, dem man ansah, daß er ein Feinschmecker des Lebens war, der aber zugleich das Göttliche zu schätzen wußte, so daß es ihm fast ein Bedürfnis war, mit aszetischen oder wenigstens enthaltsamen Leuten zu verkehren.</s><s>Er wußte solche so zu behandeln, daß sie keinen Anstoß nahmen, den Gastmählern beizuwohnen, die er den Lebemännern unter der hohen Geistlichkeit veranstaltete, und das auserlesenste Gericht würde ihm nicht gemundet haben, wenn ein überirdischer Hauch, seies durch metaphysisches Gespräch oder Anwesenheit einer vergeistigten Persönlichkeit, es nicht eingesegnet hätte; ja, wenn er einen erprobten Koch und alle Bequemlichkeiten eines modernen Hauses hätte mit sich nehmen können, zweifle ich nicht, daß er sich unter den Anachoreten der Wüste am wohlsten befunden hätte.</s><s>Kurz, er stellte die Rolle eines edlen Kirchenfürsten mit Pietät und Talent dar, womit auch seine mittelgroße, volle Gestalt und sein regelmäßiges Gesicht mit der geraden Nase, das ein Paar geschwinde, kunstreiche Augen in verschiedener Weise beleuchteten, übereinstimmte.</s><s>Er empfing mich überaus liebenswürdig, bewirtete mich und unterhielt mich unterdessen eine Weile mit geistreichen Kleinigkeiten, bis er denn allgemach anfing, mich nach dem Tun und Treiben meines Herrn auszuhorchen.</s><break strength="x-strong" /><s>Ich erzählte ihm freiwillig, was ich irgend wußte, ausgenommen alles, was auf Aglaia Bezug hatte; verbreitete mich über seine himmlische Sanftmut, über seine erhabene Gleichgültigkeit gegen die Versuchungen der Welt, über seinen göttlichen Zorn gegen das Gemeine, was ich mit vielen Beispielen belegte, und sagte unter anderem, wenn man einen Engel an der Liebe erkennte, mit der alle Herzen, die sein Wesen verspürten, ihm dienen müßten, sei er ein Engel oder doch gewiß ein Kind Gottes zu nennen.</s></speak>',
      '<speak><s>Der Kardinal war unermüdlich, mir zuzuhören, streichelte mir, während ich sprach, mehrmals Haar und Wangen und schenkte mir, ehe er mich entließ, einen Brillanten in einer Busennadel.</s><s>Er ließ sodann in der zartesten Weise für alle meine Bedürfnisse sorgen und lud mich häufig zum Essen ein, wobei auch viele andere Kardinäle und geistlichePersonen anwesend waren.</s><s>Alle behandelten mich mit großem Wohlwollen, und der Kardinal San Fiori sprach von nichts anderem als von der Heiligkeit des Fra Celeste, was ihn selbst so rührte, daß ihm nicht selten Tränen aus den Augen flossen.</s><s>Von dem Grafen sagte er unverhohlen, er würde froh sein, wenn er ein Mittel wüßte, um diesen Plagegeist zum Schweigen zu bringen; viel schwieriger schien es mir, die Geldangelegenheit zu lösen, denn der Heilige Vater mochte wohl dem Abte nicht unrecht geben, sich andererseits aber auch ungern entschließen, den Aufwand des Bruders aus eigener Kasse zu bestreiten.</s><s>Auf wiederholte Nachfrage des Kardinals, wozu denn Fra Celeste so große Summen benötigte, brachte ich erstlich seine Familie vor, die ich zu diesem Zwecke, arm und zahlreich, erfunden hatte, ferner seine verschwenderische Wohltätigkeit, und da auch das noch nicht zu genügen schien, sagte ich, daß er Kapital sammle, um in Afrika einen Staat zu gründen, der in Gestalt eines großen Bundesklosters seine göttlichen Ideale verwirklichen sollte.</s><s>Dieser Plan interessierte den Kardinal so ungemein, daß meine Phantasie kaum Vorrat genug hatte, um seine Erkundigungen nach jeder Einzelheit des afrikanischen Gottesreichs zu genügen.</s><s>Trotzdem machte ich in dieser Sache keinen Fortschritt, sondern wie mir der Kardinal sagte, der Heilige Vater behielt sich vor, mit eigenem Munde dem Bruder seine Ansprüche auszureden.</s><break strength="x-strong" /><s>Aber die Reihe zu reden kam erst an Fra Celeste.</s><s>Der Papst hatte den sinnreichen Einfall gehabt, der Bruder möchte seine erste Predigt über die Heiligkeit der Ehe halten, was ich ihn sogleich wissen ließ, als er spät abends in Rom eintraf.</s><s>Er sagte, daß er etwas anderes im Sinn gehabt hätte,und schien unzufrieden über die Störung zu sein.</s></speak>',
      '<speak><s>Am anderen Morgen erwachte er in bester Laune und bestand darauf, sofort zu predigen, ehe er sich noch dem Heiligen Vater vorgestellt hatte und ehe nur eine ordentliche Anzeige davon gemacht werden konnte.</s><s>Es verbreitete sich aber so schnell, wann und wo die Rede stattfinden würde, daß es nicht an Zuhörern fehlte, im Gegenteil drängte sich das Volk noch vor der Tür auf dem Markte, um vielleicht von dort aus einige Worte aufzufangen.</s><s>Freilich glich auch diese Predigt einem alten öligen Weine, von dem ein einziger Tropfen eines greisen Mannes Blut in Feuer verjüngen kann.</s><break strength="x-strong" /><s>Man stelle gewöhnlich, fing er an, die Liebe der Eltern zu den Kindern als die uneigennützigste am höchsten, aber das sei irrig, denn in den Kindern liebten die Eltern nur sich selbst, während Mann und Frau oft um so heftiger zueinander strebten, von desto verschiedener Art sie wären.</s><s>In solchem Verhältnisse könne man lernen, was die Bestimmung des Menschen sei: das eigene Selbst einem fremden aufzutun und hinzugeben.</s><s>Dazu sei es nicht notwendig, daß das geliebte Wesen vollkommen sei, doch dafür gehalten solle es werden.</s><s>»Ich bin kein Tier oder Sklave,« so etwa sagte er, »daß ich an meine Sinne gebunden wäre: wenn ich will, daß sie so herrlich und engelgleich ist, wie ich sie im ersten Traume meiner Liebe sah, so ist sie es.</s><s>Und daß der Zwang meiner Liebe sie so mache, deshalb ist ihre Hand unlöslich in meine versiegelt.« Er sprach dann über Untreue, von leichtfertigen Ausschweifungen und allen Üppigkeiten der Genußsucht mit so viel Ekel und ungeduldigem Hohne, daß sich die Getroffenen beinah sichtlichdarunter duckten.</s><s>Aber nachdem er noch viel Wundervolles über die eheliche Liebe und Treue gesagt hatte, was wie ein hohes Freudengeläute aus der stillen, kühlen Kirche in die goldene Mittagsluft des Marktes hinausflutete, hielt er plötzlich inne, weil irgend etwas, sei es der Anblick einer weinenden Witwe oder eines steinernen Grabmals an der Mauer oder ein Einfall der eigenen Brust die schwarzen Gedanken in ihm aufgeweckt hatten.</s></speak>',
      '<speak><s>Es folgte eine Pause, die allen willkommen war, damit sie das Vernommene in ihrem Herzen konnten ausklingen lassen, dann sagte er plötzlich mit verändertem Tone langsam: »Nun will ich euch das Geheimnis sagen.</s><s>Gott hat die Ehe gemacht für die Menschen, aber der Selige ist der, welcher der Liebe entbehren kann.«</s><break strength="x-strong" /><s>Mochte es nun der dunkle, zitternde Rhythmus seiner Worte sein, es war uns Hörern, als ob von oben her ein kalter Schatten auf unsere lichte Rosenpracht gefallen sei aus einer schwarzen, langsam steigenden und wachsenden, unentrinnbaren Wolke.</s><s>»Kann ich denn überhaupt«, sagte er, »mein Selbst einem anderen Menschen geben?</s><s>Denkt euch, ein Vater habe seinen Sohn im Gefängnis.</s><s>Er besucht ihn, aber er darf nicht zu ihm hinein, nur schauen darf er ihn durch das vergitterte Fenster.</s><s>Auch das ist Glück, aber er möchte doch dichter bei dem geliebten Kinde sein und das teure Gesicht küssen.</s><s>Indessen das ist unmöglich, weil der Sohn festgebunden ist hinten an die Mauer des Kerkers, und der Vater drückt sein Gesicht fest an das Gitter, wie wenn er es zerbrechen wollte, damit er hinein könnte zu seinem Kinde.</s><s>So ist die Liebe, es gibt keinen Weg, der die Seele zur Seele führt.</s><s>Die Seele kann die Seele nichtberühren, nicht küssen, nicht vernehmen; eingemauert in verkittete Steine spüren sie ihre Nähe nur an einem halberstickten Schrei oder an einer Träne, die durch das lückenhafte Gefüge rinnt.</s><s>Wie kann die Verborgene, die Eingekerkerte mein sein?</s><s>Aber nun kommt das Ärgste: wenn ich das Gefängnis zerschlagen und die geliebte Seele befreien könnte, wer weiß, ob mein Herz sie erkennte?</s><s>Ich habe sie ja nie gesehen, nie gefühlt, nie gehört, nur geträumt von ihr und nach ihr mich gesehnt.</s><s>Hätte sie das zärtliche Auge, das so warm und kosend über meinen Körper blickte, spräche sie mit der lachenden Stimme, die mich mit Rosenblättern überschüttete?</s><s>Während ich mich mühte, die Seele zu gewinnen, die ich nicht fand, gewöhnte ich mich an das zerbrechliche Bild, das sie verhüllte.</s><s>Vergänglichkeit ist aber die Spur von Gottes Finger, als er mit einem Fluch anrührte alles, was wir nicht begehren sollen, oder wir sind des Todes.«</s></speak>',
      '<speak><break strength="x-strong" /><s>Mit solchen und ähnlichen Worten schlug er die Liebe, die er zuvor verherrlicht hatte, ans Kreuz, und seine Augen schienen mit göttlichem Jammer mitleidvoll auf die Gemarterte zu blicken.</s><s>Am meisten zu bewundern fand ich aber dies, daß Fra Celeste seine Zuhörer mit dieser trostlosen Ansicht nicht entließ.</s><s>Von einem künstlerischen oder frommen Impulse getrieben, schloß er mit der erhebenden Wendung, daß diese Schmerzen und Enttäuschungen nur in der irdischen Liebe stattfänden, die besitzen wolle.</s><s>Wer diese überwände, die Menschen nur suchte, um ihnen wohlzutun, sonst aber mit sich und Gott genug hätte, würde eine Liebe ohne Ende und ohne Bitterkeit erlernen.</s><s>Und das wahre, heilige Sakrament der Ehe sei, daß jedem in dereigenen Brust ein Gefährte anvertraut sei, der ihm treu bleibe bis zum Tode und über den Tod hinaus; ein göttlicher Keim oder Schutzengel, der jedem mitgegeben sei, den man hegen und pflegen und lieben solle, damit man, wie es in der Ehe vorgeschrieben sei, ein Fleisch und eine Seele mit ihm werde.</s><break strength="x-strong" /><s>Diese Predigt hatte Fra Celeste den Leuten wie eine Art Liebeszauber zu kosten gegeben, wenigstens war die Begeisterung für ihn so stark und einmütig, wie sonst die ansteckenden Seuchen auftreten, an denen man sterben kann, nur weil man glaubt, sie zu haben.</s><s>Die Geldfrage wurde ohne weiteres so erledigt, daß der Papst dem Bruder monatlich fünftausend Franken auszahlte, wofür derselbe sich verpflichtete, mindestens eine Predigt in der Woche zu halten, an welchem Orte innerhalb der christlichen Welt der Papst für gut finden werde.</s><s>Da wir an demselben Tage Aglaia in Rom erwarteten, kaufte er so viel Blumen und Früchte, daß man einen Wagen damit hätte anfüllen können, und ließ sie nach ihrer Wohnung schaffen, die sich außerhalb der Stadt ganz im Freien befand.</s><s>Dort feierten sie ein solches Wiedersehensfest, daß einer, der es von außen betrachtete, hätte meinen können, Nero, der schönste und wildeste Kaiser der Welt, wäre für eine Sommernacht ins Leben zurückgekehrt, um an dem Busen einer liebestrunkenen Frau sich wieder in den Tod zu schwelgen.</s></speak>',
      '<speak><break strength="x-strong" /><s>Aber Fra Celeste war am folgenden Tage voller Kraft und Lust und hielt eine Rede über die Wonne der Armut, die, wenn das möglich war, die vorige an eindringlicher Wärme übertraf.</s><s>Er ging davon aus, daß das Beste am Menschen die Sehnsucht sei, daß aber die Sehnsucht aus Mangelentspringe, daher der Elendeste und Erbärmlichste eigentlich der sei, welcher genug habe.</s><s>Er sei gleichsam in einem Kerker eingesperrt und müsse, weil er seine Lebensorgane zu wenig üben könne, langsam erstarren und erlahmen und aus seiner dumpfen Sattigkeit in den Tod hinüberdämmern.</s><break strength="x-strong" /><s>Wozu soll ich wiederholen, was Fra Celeste von der Sklaverei des Geldgierigen und von der Freiheit des Armen sagte?</s><s>Man mußte ihn auf der Kanzel stehen sehen im einfachen Mönchsgewande wie einen Imperator auf dem Triumphwagen, wenn er mit seinen Königsaugen über die große und wahrhaft prächtige Versammlung hinsah, ohne sie auch nur mit einem Blicke zu streifen!</s><s>Nicht eine Schiffsladung von Gold und Edelsteinen hätte der reichste von allen ihm anzubieten gewagt, weil er sich des schäbigen Geschenkes geschämt hätte.</s><break strength="x-strong" /><s>Auf das Ersuchen des Papstes, der den erstaunlichen Einfluß Fra Celestes auf das Volk zum Guten zu verwerten wünschte, predigte er ferner über die großen Sünden, wie Diebstahl, Raub und Mord, weil das Banditenwesen namentlich auf dem Lande und im Gebirge blühte.</s><s>Obgleich nun solche Fehler der Liebe des erhabenen Mannes ganz fremd waren, wußte er doch darüber zu reden, als ob er jahrelang das Geschäft eines Räuberhauptmannes betrieben und das Blut zahlloser Christen auf dem Gewissen hätte; denn er besaß die Gabe, jedem Menschen durchs Antlitz in die Brust zu schauen und alles, was dort wallte und wogte, ins eigene Herz zu fassen.</s><s>Den Leuten schien er deshalb wie ein Bruder, der alles mit ihnen durchgemacht hätte, dann aber vorausgeeilt wäre und ihnen freundlich die kräftige Hand reichte, damit sie ihm nachkämen.</s><break strength="x-strong" /><s>Unsere Wohnung war nun beständig umlagert von Bittstellern, die Rats einholen oder Beichte ablegen wollten, vornehmen und geringen, Männern und Frauen.</s></speak>',
      '<speak><s>Viele, die er, weil er zu überhäuft war, wegschicken mußte, kamen zu mir, die ich auch in seinem Namen so gut wie möglich zufriedenstellte; besonders mußte ich mir ein großes Lager von Reliquien zusammenstellen, Haarlocken, Stückchen von abgelegten Gewändern und dergleichen mehr, die ich ohne Wissen meines Herrn unter die Gläubigen verteilte.</s><break strength="x-strong" /><s>Bei diesem Ansehen, das Fra Celeste anfing im ganzen christlichen Europa zu genießen, war es dem Kardinal San Fiori höchst zuwider, daß der Graf, der, um seine Sache besser zu betreiben, selbst nach Rom gekommen war, ihn unablässig bedrängte, den großen Mönch zur Rechenschaft zu ziehen, widrigenfalls er alles bekanntmachen und den Feinden unserer Kirche eine Handhabe geben würde, sich auf unsere Kosten lustig zu machen.</s><s>Ich hatte das Glück, auf ein unschädliches Mittel zu verfallen, wie man den Grafen loswerden könnte, das den Beifall des Kardinals fand und das wir unter viel Spaß und heimlichem Gelächter ins Werk setzten.</s><s>Der Kardinal lud nämlich den Grafen zu einem Essen ein, währenddessen er die Bemerkung machte, die Anhänglichkeit des Grafen an seine pflichtvergessene Frau sei um so rührender, als diese, wenn sie einmal Reize besessen, dieselben jedenfalls längst verloren und guten Grund hätte, sich in die Einsamkeit zurückzuziehen.</s><s>Um meine Meinung befragt, äußerte ich mich bescheiden, ich hätte zwar die Gräfin in der Kirche gesehen, pflegte aber nicht auf das Äußere der Frauen zu achten, sie scheineaufrichtig fromm zu sein, und das genüge mir.</s><s>Der Kardinal von Grossetto, welcher neben mir saß, streichelte mich und lobte meine Genügsamkeit, während er den übrigen zublinzelte, man müsse mich in meiner Einfalt nicht stören; dann sagte er schmunzelnd, leider gebe es viele Frauen in Rom, die schöner als fromm wären, und zählte mehrere von diesen auf.</s><s>Als die schönste priesen alle einmütig eine gewisse Dame, die wir vorher für unser Spiel auserlesen und von allem unterrichtet hatten.</s><s>Sie war vor nicht langer Zeit eine intime Freundin des Kardinals San Fiori gewesen, ebenso schlau wie pompös anzusehen und sehr willig, die Rolle, die wir ihr zugedacht hatten, zu übernehmen.</s></speak>',
      '<speak><s>Die Herren wurden nicht müde, die unerhörte Schönheit dieser Dame zu preisen, als deren einzigen Fehler sie bezeichneten, daß sie unnahbar sei und jeden, der sich ihr in liebevoller Absicht nähere, unbarmherzig heimschicke.</s><s>Sie führten mehrere reiche und edle Herren an, deren Herzen sie durch ihre Sprödigkeit gebrochen hatte, bis der Graf vor Begierde brannte, diese Kostbarkeit in seinen Besitz zu bringen.</s><s>Er sagte in hochfahrendem Tone, daß keine Frau unüberwindlich sei und daß wohl nur der rechte Mann noch nicht erschienen sei, wobei er durchblicken ließ, daß er das Zeug dazu in sich spüre.</s><s>Der Kardinal San Fiori erbot sich zwar, die Bekanntschaft zu vermitteln, warnte aber den Grafen freundschaftlich, er möge sich lieber bittere Qualen und Demütigungen ersparen.</s><s>Die Dame, die der Graf nun kennenlernte, behandelte ihn anfangs mit gleichgültiger Kälte, zwischenhinein zeichnete sie ihn durch liebenswürdige Koketterie vor den anderen aus, was sie dann aber durch vermehrte Strenge wieder ungültig zu machen schien, durchwelches Betragen sie seine Verliebtheit aufs äußerste reizte.</s><s>Bald war er dermaßen in ihrer Gewalt, daß sie es wagen konnte, ihm Hoffnung auf ihre Liebe zu machen, wenn er sie zu seiner rechtmäßigen Gemahlin machen würde.</s><s>Der Graf schwur, nie etwas anderes im Sinne gehabt zu haben, und reichte augenblicklich dem Heiligen Vater die Bitte ein, derselbe möchte seine Ehe mit Aglaia lösen.</s><s>Kardinal San Fiori, welcher die Führung der ganzen Angelegenheit in der Hand hatte, bezeugte über diese Wendung Erstaunen und Entrüstung, sagte, daß keinerlei Gründe zu einer Scheidung vorlägen, da Aglaia, wie jedermann wisse, sich nicht eigentlich gegen ihn vergangen hätte, vielmehr im Grunde aus lobenswerter Frömmigkeit die weltliche Luft seines Hauses meide, und daß die unheilige Raserei eines Mannes, der kein schönes Weib ansehen könne, ohne sie besitzen zu wollen, nimmermehr durch die Kirche gerechtfertigt werden dürfe.</s><s>Nachdem er den Grafen in dieser Weise eine Zeitlang auf der Folter gelassen hatte, wurde die Scheidung ausgesprochen, und er führte die Angebetete heim, wodurch wir in alle Zukunft vor seinen Verfolgungen sicher waren.</s></speak>',
      '<speak><break strength="x-strong" /><s>Fra Celeste hätte nun sein Leben mit größerem Behagen genießen können, wenn sich nicht sogleich neue Widerwärtigkeiten erhoben hätten, welche diesmal von Aglaias Eltern ausgingen.</s><s>Sie verlangten nämlich, ihre Tochter solle, anstatt in der Welt umherzuvagabondieren, in ihr Haus zurückkehren, und kamen selbst nach Rom, um die Widerspenstige abzuholen.</s><s>Der Heilige Vater sowie die Kardinäle sprachen sich dahin aus, wenn Aglaia nicht bei ihren Eltern leben wolle, möchte sie wenigstens in einKloster eintreten, wo sie ja der Einsamkeit nach Herzenslust frönen könne; denn die Verehrung für Fra Celeste, so berechtigt sie auch sei, könne ungebildete oder bösartige Menschen doch zu Mißdeutungen veranlassen, und da sie auf die Freuden der Gattin und Tochter des Hauses ohnehin verzichtet hätte, sei nicht einzusehen, was sie noch in der Welt zurückhielte.</s><s>Ich sagte dem Kardinal, daß Fra Celeste die Dame längst in diesem Sinne zu beeinflussen versucht hätte, daß es bislang aber vergeblich gewesen sei, da sie die Eintönigkeit des Klosterlebens fürchtete, und fügte hinzu, vielleicht werde seiner Beredsamkeit und Menschenkenntnis ein besserer Erfolg zuteil.</s><s>Dieser Aufforderung kam der Kardinal unverweilt nach; er besuchte die Gräfin, sprach mit ihr von den Annehmlichkeiten des Klosterlebens, dem er sich herzlich gerne hingeben möchte, wenn ihn nicht Geschäfte zum Nutzen der Kirche noch in der Welt zurückhielten, und verliebte sich mittlerweile über die Maßen in Aglaia, was sowohl ihr wie mir sehr zur Belustigung diente.</s><s>Eines Tages kam aber Fra Celeste dazu, dem ich diese Vorgänge aus Rücksicht verschwiegen hatte, und geriet über den unerwarteten Anblick des Kardinals in jene Wut, deren Symptome ich früher beschrieben habe.</s><s>Ohne abzuwarten, daß einer von uns die Anwesenheit des Kardinals bei Aglaia erklärte, fuhr er heraus, man habe ihm den Verkehr mit der Gräfin zum Vorwurf gemacht, jetzt sähe er die Ursache dieses Eifers, man wolle selbst bei ihr ein und aus gehen, dabei sei er im Wege gewesen.</s></speak>',
      '<speak><s>Im Feuer des Sprechens erhitzte er sich mehr und mehr, bis er von oben bis unten in eine prasselnde Flamme verwandelt schien mit seinen knirschenden Zähnen und lechzenden Augen;wir schauten ihn an wie einen gläsernen Zylinder, der glüht und kracht, so daß man sein Auseinanderspringen jeden Augenblick erwartet und sich bereit hält wegzulaufen, um von den heißen Scherben nicht getroffen zu werden.</s><s>Während ich Stellung nahm, um ihm rechtzeitig in den Arm zu fallen, wenn er etwa auf den Kardinal los wollte, ließ die göttliche Vorsehung diesen entschlüpfen, den der aufgeregte Mann, von der Eifersucht wie von einer Windsbraut gefaßt und mitgerissen, sonst, wie er selbst glaubte, vielleicht erdrosselt hätte.</s><break strength="x-strong" /><s>Da zeigte es sich wieder, wie Gott diesen wunderbaren Menschen liebte und auch die scheinbaren Verirrungen zu seinem Wohle ausschlagen ließ; denn der Kardinal, zu dem ich mich, nicht ohne eine gewisse Befangenheit überwinden zu müssen, am folgenden Tage begab, empfing mich zwar betrübt, aber herzlich, indem er mir von dem Ruhebette, wo er lag, beide Hände entgegenstreckte.</s><s>Er erzählte mir, daß ihn einerseits die Eitelkeit zu Aglaia geführt hätte, nämlich die Aussicht, größere Macht über ihren Geist zu gewinnen als der berühmte Prediger, daneben lüsterne Neugier, die Frau kennenzulernen, von der man sich so bedeutende Gerüchte zuflüsterte.</s><s>Die Holdseligkeit ihrer Erscheinung und ihres Wesens hätte es ihm gleich beim ersten Besuche angetan, und anstatt sich nun zurückzuhalten, wie es seine Pflicht gewesen sei, habe er sich dem verbotenen Spiele hingegeben, noch dazu unter geflissentlicher Täuschung des eigenen Gewissens, dem er vorgelogen habe, es geschehe zu einem heiligen Zwecke.</s><s>Fra Celeste aber habe mit dem Blitzen seiner Augen und seiner Zunge das Truggewebe entzündet und zu Asche gebrannt, so daßer sich selbst angeschaut habe, wie er wirklich sei, schwach und voll Sünden.</s><s>Jetzt stehe es ihm fest, sagte er, daß der Bruder ein Seher sei, auf dem der Geist Gottes unmittelbar ruhe, und trug mir auf, Aglaia, die seine Torheit ermutigt und dadurch gleichfalls gesündigt hätte, Unterwerfung unter die Ratschläge des prophetischen Mannes zu empfehlen.</s></speak>',
      '<speak><s>Ich sagte, Fra Celeste hätte im Sinne, einige große Predigten über das Klosterleben zu halten, wodurch er hauptsächlich Aglaias Herz zu treffen gedächte, was zwar nur ein Einfall war, der mir im Augenblicke kam, von meinem Herrn aber, dem ich davon Bericht erstattete, sofort aufgegriffen wurde.</s><s>Sein Genius bemächtigte sich der hingeworfenen Idee so vollkommen, daß er bald darauf, an drei aufeinanderfolgenden Tagen, drei Predigten über diesen Gegenstand hielt, die eine Horde von Kannibalen in Säulenheilige hätte umwandeln können.</s><s>Die Folge davon war, daß unter vielen anderen Leuten auch Aglaias Eltern sich voll Abscheu von der Welt abkehrten und Kutte und Schleier nahmen.</s><s>Die Arme war so gerührt über das Betragen der beiden Alten, die, ohne sie noch ferner anzuklagen, noch irgend etwas von ihr zu fordern, mit großer Würde, Demut und vielen Zeichen beginnender Heiligkeit sich zum Eintritte ins Kloster vorbereiteten, daß sie stundenlang Tränen vergoß, sich mit Selbstvorwürfen peinigte und sich ihnen vielleicht sogar angeschlossen hätte, wenn nicht Fra Celestes Krankheit dazwischengekommen wäre.</s><break strength="x-strong" /><s>Er wurde nämlich mitten in einer Predigt durch einen Hustenanfall unterbrochen, konnte sie nur mit größter Anstrengung zu Ende bringen und warf bei der nächsten Wiederkehr des bösartigen Hustens sogar Blut aus.</s><s>Die Ärzteerklärten einmütig, daß Fra Celeste das Predigen für unabsehbare Zeit ganz aufgeben müsse, widrigenfalls das Schlimmste zu befürchten stehe.</s><s>Dies Verbot war für ihn, dem Reden so notwendig war wie Atmen, hart genug; nun aber kam hinzu, daß der Heilige Vater wünschte, der Bruder möge sich während der Ruhezeit in sein Kloster begeben; denn weil sein Tod als nach aller Wahrscheinlichkeit nahe bevorstehend erklärt wurde, war man darauf bedacht, daß der heilige Mann sein ruhmwürdiges Leben durch ein entsprechendes Sterben kröne.</s><s>Im Kloster, glaubte man, werde sich diese weihevolle Szene am schönsten abspielen, während sehr zu befürchten war, daß in einem Kurorte am Mittelmeer, wohin die Ärzte ihn gewiesen hatten und wohin Aglaia ihn als Pflegerin begleiten würde, sich allerlei Weltliches, sei es auch nur in Form von Gerüchten und Verdächtigungen, profanierend einmische.</s></speak>',
      '<speak><s>Um den teuren Kranken nicht durch diese Widerwärtigkeiten aufzuregen, griff ich, ohne es ihm zu sagen, zu einem freilich etwas gewagten Mittel; ich erzählte dem Kardinal San Fiori, nachdem ich mir Geheimhaltung meiner Mitteilung hatte beteuern lassen, daß ich Ursache hätte zu fürchten, mein Herr werde zum protestantischen Glauben übertreten.</s><s>Er sei in Verkehr mit einem protestantischen Geistlichen von hoher Stellung geraten, dem augenscheinlich daran gelegen sei, den berühmten Prediger für seine Kirche zu gewinnen; er entblöde sich nicht, darauf anzuspielen, was für weltliche Vorteile dem Bruder dadurch zufließen würden, daß er Reichtümer sammeln und sogar eine Frau nehmen könnte.</s><s>Obgleich ich nun gewiß wüßte, sagte ich, daß mein Herr weder durch Geld noch durch Frauenzu bestechen sei, trüge ich mich doch mit ängstlichen Befürchtungen, denn in sein Kloster wolle er wegen des Streites, den er in früherer Zeit mit dem Abte gehabt hätte, keinesfalls zurückkehren, ohne Mittel könne er in der Welt nicht leben, also treibe ihn vielleicht die Not in das Lager der Feinde.</s><break strength="x-strong" /><s>Bei dieser Gelegenheit habe ich den ersten von den drei Küssen bekommen, die mir Aglaia gegeben hat.</s><s>Ich erzählte ihr, um sie zu erheitern, meine eben angeführte Unterhaltung mit dem Kardinal, und indem ich meine Phantasie laufen ließ, wohin sie wollte, fügte ich ausschmückenderweise hinzu, daß er mich insgeheim gedungen hätte, alle Schritte des Bruders zu überwachen, ihn beständig von allem zu unterrichten und seinen Übertritt zum protestantischen Glauben, wenn es soweit komme, nötigenfalls mit den äußersten Mitteln zu hintertreiben.</s><s>Aglaia fragte erschreckt, was unter diesen Mitteln zu verstehen sei, worauf ich wieder fragte, ob sie niemals von einem farblosen Pulver gehört hätte, das man mit unfehlbarer Wirkung in den Wein seines Todfeindes mischen könne, und ob sie nicht wisse, daß fanatische Gläubige sogar ihre Freunde mit solchen Pulvern bewirteten, wenn das zum Heile der Kirche dienlich sei.</s><s>Dann sprach ich noch von den glänzenden Anerbietungen, die der Kardinal mir gemacht hätte, und daß ich ihn zwar in dem Glauben gelassen hätte, als sei ich nicht abgeneigt, ihm zu dienen, daß ich aber, wie es sich von selbst verstehe, meinen Herrn niemals verraten würde.</s></speak>',
      '<speak><s>Die Liebliche sagte lächelnd, da ich des Sündenlohnes verlustig gehe, wolle sie mich schadlos halten, was für Lohn ich mir wünsche?</s><s>worauf ich vor ihr niederkniete und umeinen Kuß von ihrem Munde bat.</s><s>Sie nahm meinen Kopf zwischen ihre kühlen Hände, daß es mir wonnig über den Leib rieselte, und küßte mich langsam auf beide Augen, indem sie sagte: »Deine Augen sind fromm wie die Taube und klug wie die Schlange.«</s><break strength="x-strong" /><s>Übrigens trug der Kardinal mir wirklich auf, ihn von allen auffallenden Schritten, die der Bruder unternähme, in Kenntnis zu setzen; denn seit der Androhung des Glaubenswechsels erschien er ihnen vollends wie ein schönes, aber furchtbares Tier, von dem man der heillosesten Dinge gewärtig sein müsse.</s><s>Sie behandelten ihn denn auch mit großer Sorgfalt wie die Ägypter ihre heiligen Krokodile und Ochsen, überschütteten ihn mit Ehren, versprachen ihm soviel Geld er irgend wollte und entließen ihn mit Segenswünschen nach dem kleinen Ort am Meere, der ihm empfohlen war.</s><s>Der vielgeliebte Mann aber war so krank, daß er sich um nichts mehr bekümmerte, nicht einmal um Aglaia, die uns möglichst schnell nachgereist kam, um ihn zu pflegen.</s><s>Wie von Mohammed erzählt wird, daß Gott ihn bei seinen Lebzeiten durch die sieben Himmel geführt habe, während er auf Erden körperlich anwesend war, so, hätte man glauben können, sei es mit Dolfin geschehen.</s><s>Er lag häufig so unempfänglich da, als hätte die Seele den Körper verlassen, der doch lebte.</s><break strength="x-strong" /><s>Auch wenn der Geist gegenwärtig war, achtete Fra Celeste nicht auf seine Umgebung, und wenn er Aglaia und mich auch um sich duldete, ließ er doch deutlich merken, daß ihm unsere Gegenwart weder lieb noch leid war.</s><s>Sprach er zu uns, war es zwar mit der Geduld und Sanftmut eines abgeschiedenen Geistes, aber auch wie aus weiter Ferne undso unverständlich, daß man nicht wissen konnte, ob er im Fieber phantasierte oder ob er schon in andere Sphären hinüberschwebte.</s><s>Was Aglaia am wehesten tat, war, daß er von ihrer gemeinsamen Vergangenheit sprach wie von einem dumpfen Traume oder von der törichten Kinderzeit, ja, wenn sie ihm leidenschaftlich Gesicht und Hände küßte, um seine Erinnerung zu erwärmen und zu beleben, betrachtete er sie mit mitleidiger Verwunderung, etwa wie ein Christ einen Wilden, der sich noch darin gefällt, seinem Götzen geschlachtete Feinde als Opfer darzubringen.</s></speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      softLimit: AWS_SOFT_LIMIT,
      hardLimit: AWS_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(largeExample);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      const characterCountHowAWSCountsIt = removeSSMLTags(item).length;
      expect(characterCountHowAWSCountsIt).toBeLessThanOrEqual(AWS_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input text (${textExampleLarge.length} characters) goes above the hardLimit`, () => {
    const text = textExampleLarge

    const expected = [
      '<speak>Lorem ipsum dolor sit amet, consectetur adipiscing elit. Morbi imperdiet dui in mattis fringilla. Vivamus scelerisque id erat eu imperdiet. Mauris eu neque sit amet lectus facilisis tristique. Vestibulum finibus quam vel tempus tempor. Donec in mollis augue. Duis hendrerit aliquam tortor at volutpat. Vivamus ac fringilla arcu, eget bibendum enim.Cras consectetur, dui vel aliquet tincidunt, velit nisl lacinia sem, non accumsan risus est at mauris. Vestibulum dignissim metus nec nisl dignissim, at volutpat libero dignissim. Sed sollicitudin lacus felis, quis vehicula nisi fermentum eget. Nullam venenatis, orci ac bibendum pharetra, justo turpis consequat orci, vitae lobortis mauris mauris a mi. In ut consequat lectus. Curabitur purus metus, vestibulum a mauris in, gravida egestas enim. Nunc diam erat, interdum in hendrerit vel, consectetur in ex. Mauris est dui, varius sed nunc sit amet, aliquam pellentesque justo. Sed placerat quis dui in blandit. Nullam porta leo ipsum, sit amet maximus mi mollis et. Donec ac lectus dolor. Mauris dictum libero dignissim tincidunt aliquam. Aliquam quis urna ac ligula lobortis vulputate quis porta ligula. Etiam semper ante non risus ultricies rutrum.Aliquam eu facilisis sem, eu maximus felis. Aenean quis ultricies augue, a tincidunt lacus. Nunc vitae massa vestibulum, malesuada sem eget, iaculis ante. Aliquam lorem velit, iaculis at ipsum non, facilisis vulputate lorem. Aenean fermentum imperdiet eleifend. Pellentesque magna ipsum, varius vel tortor in, dignissim faucibus est. Nullam velit nisl, accumsan in tellus id, suscipit varius massaAliquam eget mauris at felis bibendum semper. Ut ultrices dui eu orci ultricies molestie. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros, sed tincidunt est. Mauris lobortis tempor nulla id elementum. Etiam gravida arcu et nibh vulputate dapibus. Aliquam eget tempus diamNullam quis ullamcorper erat. Vestibulum ultricies erat quis leo efficitur consequat. Ut id risus facilisis, ornare sem at, rutrum tellus. Donec a urna odio. Ut et nisl quis enim dignissim sodales. Vestibulum risus augue, posuere id tortor vitae, efficitur fermentum massa. Etiam at eros rhoncus, aliquet tellus interdum, vestibulum nibh. Phasellus consequat velit quis mi lobortis varius. Phasellus semper purus nec tellus fringilla condimentum. Morbi cursus nec ante eget porttitor. Vestibulum vestibulum arcu sit amet elit congue tristique. Nulla eleifend purus non ornare interdum. Aliquam quis nulla aliquet, mattis dolor nec, auctor massa. Vestibulum placerat placerat ante sit amet laciniaLorem ipsum dolor sit amet, consectetur adipiscing elit. Morbi imperdiet dui in mattis fringilla. Vivamus scelerisque id erat eu imperdiet. Mauris eu neque sit amet lectus facilisis tristique. Vestibulum finibus quam vel tempus tempor. Donec in mollis augue. Duis hendrerit aliquam tortor at volutpat.</speak>',
      '<speak> Vivamus ac fringilla arcu, eget bibendum enimCras consectetur, dui vel aliquet tincidunt, velit nisl lacinia sem, non accumsan risus est at mauris. Vestibulum dignissim metus nec nisl dignissim, at volutpat libero dignissim. Sed sollicitudin lacus felis, quis vehicula nisi fermentum eget. Nullam venenatis, orci ac bibendum pharetra, justo turpis consequat orci, vitae lobortis mauris mauris a mi. In ut consequat lectus. Curabitur purus metus, vestibulum a mauris in, gravida egestas enim. Nunc diam erat, interdum in hendrerit vel, consectetur in ex. Mauris est dui, varius sed nunc sit amet, aliquam pellentesque justo. Sed placerat quis dui in blandit. Nullam porta leo ipsum, sit amet maximus mi mollis et. Donec ac lectus dolor. Mauris dictum libero dignissim tincidunt aliquam. Aliquam quis urna ac ligula lobortis vulputate quis porta ligula. Etiam semper ante non risus ultricies rutrumAliquam eu facilisis sem, eu maximus felis. Aenean quis ultricies augue, a tincidunt lacus. Nunc vitae massa vestibulum, malesuada sem eget, iaculis ante. Aliquam lorem velit, iaculis at ipsum non, facilisis vulputate lorem. Aenean fermentum imperdiet eleifend. Pellentesque magna ipsum, varius vel tortor in, dignissim faucibus est. Nullam velit nisl, accumsan in tellus id, suscipit varius massaAliquam eget mauris at felis bibendum semper. Ut ultrices dui eu orci ultricies molestie. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros, sed tincidunt est. Mauris lobortis tempor nulla id elementum. Etiam gravida arcu et nibh vulputate dapibus. Aliquam eget tempus diam Nullam quis ullamcorper erat. Vestibulum ultricies erat quis leo efficitur consequat. Ut id risus facilisis, ornare sem at, rutrum tellus. Donec a urna odio. Ut et nisl quis enim dignissim sodales. Vestibulum risus augue, posuere id tortor vitae, efficitur fermentum massa. Etiam at eros rhoncus, aliquet tellus interdum, vestibulum nibh. Phasellus consequat velit quis mi lobortis varius. Phasellus semper purus nec tellus fringilla condimentum. Morbi cursus nec ante eget porttitor. Vestibulum vestibulum arcu sit amet elit congue tristique. Nulla eleifend purus non ornare interdum. Aliquam quis nulla aliquet, mattis dolor nec, auctor massa. Vestibulum placerat placerat ante sit amet lacinia.</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      softLimit: AWS_SOFT_LIMIT,
      hardLimit: AWS_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(text);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      const characterCountHowAWSCountsIt = removeSSMLTags(item).length;
      expect(characterCountHowAWSCountsIt).toBeLessThanOrEqual(AWS_HARD_LIMIT);
    });
  });

  it(`Should return the correct SSML chunks when the input text (${textExampleLargeSentence.length} characters) without any punctuation goes above the hardLimit`, () => {
    const text = textExampleLargeSentence

    const expected = [
      '<speak>Lorem ipsum dolor sit amet consectetur adipiscing elit Morbi imperdiet dui in mattis fringilla Vivamus scelerisque id erat eu imperdiet Mauris eu neque sit amet lectus facilisis tristique Vestibulum finibus quam vel tempus tempor Donec in mollis augue Duis hendrerit aliquam tortor at volutpat Vivamus ac fringilla arcu eget bibendum enim.Cras consectetur dui vel aliquet tincidunt velit nisl lacinia sem non accumsan risus est at mauris Vestibulum dignissim metus nec nisl dignissim at volutpat libero dignissim Sed sollicitudin lacus felis quis vehicula nisi fermentum eget Nullam venenatis orci ac bibendum pharetra justo turpis consequat orci vitae lobortis mauris mauris a mi In ut consequat lectus Curabitur purus metus vestibulum a mauris in gravida egestas enim Nunc diam erat interdum in hendrerit vel consectetur in ex Mauris est dui varius sed nunc sit amet aliquam pellentesque justo Sed placerat quis dui in blandit Nullam porta leo ipsum sit amet maximus mi mollis et Donec ac lectus dolor Mauris dictum libero dignissim tincidunt aliquam Aliquam quis urna ac ligula lobortis vulputate quis porta ligula Etiam semper ante non risus ultricies rutrum.Aliquam eu facilisis sem eu maximus felis Aenean quis ultricies augue a tincidunt lacus Nunc vitae massa vestibulum malesuada sem eget iaculis ante Aliquam lorem velit iaculis at ipsum non facilisis vulputate lorem Aenean fermentum imperdiet eleifend Pellentesque magna ipsum varius vel tortor in dignissim faucibus est Nullam velit nisl accumsan in tellus id suscipit varius massaAliquam eget mauris at felis bibendum semper Ut ultrices dui eu orci ultricies molestie Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros sed tincidunt est Mauris lobortis tempor nulla id elementum Etiam gravida arcu et nibh vulputate dapibus Aliquam eget tempus diamNullam quis ullamcorper erat Vestibulum ultricies erat quis leo efficitur consequat Ut id risus facilisis ornare sem at rutrum tellus Donec a urna odio Ut et nisl quis enim dignissim sodales Vestibulum risus augue posuere id tortor vitae efficitur fermentum massa Etiam at eros rhoncus aliquet tellus interdum vestibulum nibh Phasellus consequat velit quis mi lobortis varius Phasellus semper purus nec tellus fringilla condimentum Morbi cursus nec ante eget porttitor Vestibulum vestibulum arcu sit amet elit congue tristique Nulla eleifend purus non ornare interdum Aliquam quis nulla aliquet mattis dolor nec auctor massa Vestibulum placerat placerat ante sit amet laciniaLorem ipsum dolor sit amet consectetur adipiscing elit Morbi imperdiet dui in mattis fringilla Vivamus scelerisque id erat eu imperdiet Mauris eu neque sit amet lectus facilisis tristique Vestibulum finibus quam vel tempus tempor Donec in mollis augue Duis hendrerit aliquam tortor at volutpat Vivamus ac fringilla arcu eget bibendum enimCras consectetur dui vel aliquet tincidunt velit nisl lacinia sem non accumsan risus est at mauris Vestibulum </speak>',
      '<speak>dignissim metus nec nisl dignissim at volutpat libero dignissim Sed sollicitudin lacus felis quis vehicula nisi fermentum eget Nullam venenatis orci ac bibendum pharetra justo turpis consequat orci vitae lobortis mauris mauris a mi In ut consequat lectus Curabitur purus metus vestibulum a mauris in gravida egestas enim Nunc diam erat interdum in hendrerit vel consectetur in ex Mauris est dui varius sed nunc sit amet aliquam pellentesque justo Sed placerat quis dui in blandit Nullam porta leo ipsum sit amet maximus mi mollis et Donec ac lectus dolor Mauris dictum libero dignissim tincidunt aliquam Aliquam quis urna ac ligula lobortis vulputate quis porta ligula Etiam semper ante non risus ultricies rutrumAliquam eu facilisis sem eu maximus felis Aenean quis ultricies augue a tincidunt lacus Nunc vitae massa vestibulum malesuada sem eget iaculis ante Aliquam lorem velit iaculis at ipsum non facilisis vulputate lorem Aenean fermentum imperdiet eleifend Pellentesque magna ipsum varius vel tortor in dignissim faucibus est Nullam velit nisl accumsan in tellus id suscipit varius massaAliquam eget mauris at felis bibendum semper Ut ultrices dui eu orci ultricies molestie Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nam sit amet posuere eros sed tincidunt est Mauris lobortis tempor nulla id elementum Etiam gravida arcu et nibh vulputate dapibus Aliquam eget tempus diam Nullam quis ullamcorper erat Vestibulum ultricies erat quis leo efficitur consequat Ut id risus facilisis ornare sem at rutrum tellus Donec a urna odio Ut et nisl quis enim dignissim sodales Vestibulum risus augue posuere id tortor vitae efficitur fermentum massa Etiam at eros rhoncus aliquet tellus interdum vestibulum nibh Phasellus consequat velit quis mi lobortis varius Phasellus semper purus nec tellus fringilla condimentum Morbi cursus nec ante eget porttitor Vestibulum vestibulum arcu sit amet elit congue tristique Nulla eleifend purus non ornare interdum Aliquam quis nulla aliquet mattis dolor nec auctor massa Vestibulum placerat placerat ante sit amet lacinia</speak>'
    ];

    const ssmlSplit = new SSMLSplit({
      synthesizer: 'aws',
      softLimit: AWS_SOFT_LIMIT,
      hardLimit: AWS_HARD_LIMIT,
      breakParagraphsAboveHardLimit: true
    });

    const result = ssmlSplit.split(text);

    expect(result).toStrictEqual(expected);

    result.forEach(item => {
      const characterCountHowAWSCountsIt = removeSSMLTags(item).length;
      expect(characterCountHowAWSCountsIt).toBeLessThanOrEqual(AWS_HARD_LIMIT);
    });
  });
});
