package sift.core.api

import sift.core.api.AccessFlags.Scope.*

/** Enum representing Java modifiers. */
@Suppress("EnumEntryName")
enum class AccessFlags(val flag: Int, internal vararg val scopes: Scope) {

    /** Accessible from any class. */
    acc_public(0x0001, Class, Field, Method),

    /** Accessible only from within the same class. */
    acc_private(0x0002, Class, Field, Method),

    /** Accessible within the same package or subclass. */
    acc_protected(0x0004, Class, Field, Method),

    /** Belongs to the class, not an instance of the class. */
    acc_static(0x0008, Field, Method),

    /** Cannot be modified after initialization. */
    acc_final(0x0010, Class, Field, Method, Parameter),

    /** Exists for backward compatibility for code compiled by Sun's older compilers. */
    acc_super(0x0020, Class),

    /** Accessible only by one thread at a time. */
    acc_synchronized(0x0020, Method),

    /** Open module which can be used by any module. */
    acc_open(0x0020, Module),

    /** Indicates a module dependency. */
    acc_transitive(0x0020, Module),

    /** Can be changed unexpectedly by other threads. */
    acc_volatile(0x0040, Field),

    /** Used in method signature during type erasure. */
    acc_bridge(0x0040, Method),

    /** Specifies that a module is read in `--limit-modules` mode. */
    static_acc_phase(0x0040, Module),

    /** Variable-length argument list for a method. */
    acc_varargs(0x0080, Method),

    /** Field is not serialized. */
    acc_transient(0x0080, Field),

    /** Implemented in platform-specific code. */
    acc_native(0x0100, Method),

    /** Interface definition. */
    acc_interface(0x0200, Class),

    /** Cannot be instantiated directly, must be subclassed. */
    acc_abstract(0x0400, Class, Method),

    /** Strict floating-point arithmetic. */
    acc_strict(0x0800, Method),

    /** Not part of the source code, generated by compiler. */
    acc_synthetic(0x1000, Class, Field, Method, Parameter, Module),

    /** Annotation definition. */
    acc_annotation(0x2000, Class),

    /** Enum definition. */
    acc_enum(0x4000, Class, Field),

    /** Indicates mandated API elements. */
    acc_mandated(0x8000, Class, Field, Method, Parameter, Module),

    /** Module definition. */
    acc_module(0x8000, Class);

    override fun toString(): String = name.lowercase()

    fun check(bitmask: Int): Boolean = bitmask and flag != 0

    companion object {
        internal fun bitmaskOf(
            accessFlags: Iterable<AccessFlags>
        ): Int = accessFlags.fold(0) { acc, m -> acc or m.flag }

        internal fun bitmaskOf(
            vararg accessFlags: AccessFlags
        ): Int = accessFlags.fold(0) { acc, m -> acc or m.flag }
    }

    enum class Scope {
        Class,
        Field,
        Method,
        Module,
        Parameter,
        TypeErased // disables checks
    }
}