/**
 * @file host_link.h
 * @brief contains functions and data types for host link module
 *
 * The host link initializes an interface with the host on startup through the host_link_init function.
 * Whether the interface is USB or UART, host_link_init will initialize the "link" struct with four function
 * pointers that provide access to the interface. When this is complete, the host link can parse commands from
 * the host to operate the YeeNet router. Operation of the YeeNet router through the host link occurs in
 * discrete transactions. A transaction with the host_link always begins with a frame from the host. The frame
 * is always of the following format: [0] - a byte that indicates which command to run [1:end] - data to be
 * passed to the command's function The interface must buffer this frame and wait for host_link to request it
 * through iface_get_command The driver on the host's end must not send any additional frames and should wait
 * for the host link's response. When host_link_parse is executed, the host_link will run iface_get_command
 * and start reading the packet. If the 0th byte is within range of the jump table, the first byte of the
 * response will be 0x00. If the 0th byte is not within range of the jump table, the first byte of the
 * response will be 0x02. If the length check passes, the host link will use the 0th byte to look up the
 * command in a jump table called "commands" defined in host_link.c. The command will run and be passed the
 * [1:end] data along with the number of bytes in [1:end]. The command will use link.iface_write_bytes and
 * link.iface_write_byte write its response. When the command returns and no further bytes need to be written
 * to the response frame, the host link will run iface_release and host_link_parse will return. So, a response
 * frame may look like: [0] - 0x02
 *
 * or
 *
 * [0] - 0x00
 *
 * or
 *
 * [0] - 0x00
 * [0:end] - bytes written by command
 *
 * When the response frame is received in full, the host may initiate another transaction.
 * Notes:
 * A response of 0x01 in the 0th byte is reserved for reporting a COBS error on the UART interface an is not
 * generated by the host link. The host link assumes it has exclusive access to the interface.
 */

#pragma once
#include "uart.h"
#include <stdbool.h>
#include <sys/types.h>

/**
 * @struct host_interface
 *
 * Represents an interface the host link may use to communicate with a host.
 * Any interface must implement the following functions for functions in the host_link jump table to work.
 */
struct host_interface {
    /**
     * The function should accept two parameters:
     * A pointer to an array and the length of the array.
     * The function should write the bytes in the array to the interface and return how many bytes were
     * written.
     */
    ssize_t (*iface_write_bytes)(uint8_t *, size_t);
    /**
     * the function should accept one byte as a parameter.
     * The function should write that byte to the interface and return how many bytes were written.
     */
    ssize_t (*iface_write_byte)(uint8_t);
    /**
     * The function should accept a pointer-pointer as an input.
     * The function should populate the pointer that the pointer-pointer points to with a pointer to an array
     * of bytes that is the incoming command. The function should also return the length of that command as a
     * uint16_t. If a command is not present, the function should return 0.
     */
    uint16_t (*iface_get_command)(uint8_t **);
    /**
     * This function will be run by the host link when it is done writing its response.
     */
    void (*iface_release)(void);
};

/**
 * Provides all files with access to the interface with the host.
 */
extern struct host_interface link;

/**
 * Enum for the different interfaces the host link can use.
 */
enum iface_setting { USART, USB };

/**
 * Enum for the first byte of a response to a command.
 * Indicates outcome of first steps of packet parsing.
 */
enum iface_status_byte { GOOD = 0, COBS_ERROR = 1, PARSE_ERROR = 2 };

/**
 * @brief initializes the host link
 * This function should be run at startup and will configure the chose interface.
 *
 * @param my_interface the interface for the host link to use
 *
 */
void host_link_init(enum iface_setting my_interface);

/**
 * @brief checks if a command is available on the interface an parses it
 *
 * If this function is not run regularly, the host link will not be able to respond to commands.
 */
void host_link_parse(void);
