import { module, test } from 'qunit';
import { setupRenderingTest } from 'ember-qunit';
import { render } from '@ember/test-helpers';
import hbs from 'htmlbars-inline-precompile';
import { registerCustomStyles } from '@frontile/theme';
import { tv } from 'tailwind-variants';

registerCustomStyles({
  progressBar: tv({
    slots: {
      base: ['pb-base'],
      label: ['pb-label'],
      progress: ['pb-progress']
    },
    variants: {
      isIndeterminate: {
        true: {
          progress: ['pb-is-indeterminate']
        }
      },
      intent: {
        default: 'intent-default',
        primary: 'intent-primary',
        success: 'intent-success',
        warning: 'intent-warning',
        danger: 'intent-danger'
      },
      size: {
        xs: 'size-xs',
        sm: 'size-sm',
        md: 'size-md',
        lg: 'size-lg'
      },
      radius: {
        none: 'radius-none',
        sm: 'radius-sm',
        lg: 'radius-lg',
        full: 'radius-full'
      }
    },
    defaultVariants: {
      size: 'md',
      intent: 'default',
      radius: 'sm'
    }
  }) as never
});

module(
  'Integration | Component | ProgressBar | @frontile/progress-bar',
  function (hooks) {
    setupRenderingTest(hooks);

    test('it renders', async function (assert) {
      await render(hbs`<ProgressBar data-test-id="progress-bar" />`);
      assert.dom('[data-test-id="progress-bar"]').exists();
      assert.dom('[data-test-id="progress-bar"] [role="progressbar"]').exists();
    });

    module('style classes', () => {
      module('intent', () => {
        test('it adds class for the intent', async function (assert) {
          await render(
            hbs`<ProgressBar @intent="primary" data-test-id="progress-bar" />`
          );

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base')
            .hasClass('intent-primary');
        });

        test('it adds class for the default intent', async function (assert) {
          await render(hbs`<ProgressBar data-test-id="progress-bar" />`);

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base')
            .hasClass('intent-default');
        });
      });

      module('sizes', () => {
        test('it adds class for default size"', async function (assert) {
          await render(hbs`<ProgressBar data-test-id="progress-bar" />`);

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base')
            .hasClass('size-md');
        });

        test('it adds class size xs"', async function (assert) {
          await render(
            hbs`<ProgressBar @size="xs" data-test-id="progress-bar" />`
          );

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base')
            .hasClass('size-xs');
        });

        test('it adds class size lg', async function (assert) {
          await render(
            hbs`<ProgressBar @size="lg" data-test-id="progress-bar" />`
          );

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base ')
            .hasClass('size-lg');
        });
      });

      module('radius', () => {
        test('it adds class default radius size sm"', async function (assert) {
          await render(hbs`<ProgressBar data-test-id="progress-bar" />`);

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base')
            .hasClass('radius-sm');
        });

        test('it adds class radius size lg"', async function (assert) {
          await render(
            hbs`<ProgressBar @radius="lg" data-test-id="progress-bar" />`
          );

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base')
            .hasClass('radius-lg');
        });

        test('it adds class radius full', async function (assert) {
          await render(
            hbs`<ProgressBar @radius="full" data-test-id="progress-bar" />`
          );

          assert
            .dom('[data-test-id="progress-bar"] > div.pb-base ')
            .hasClass('radius-full');
        });
      });

      module('is-indeterminate', () => {
        test('it adds class for indeterminate state "', async function (assert) {
          await render(
            hbs`<ProgressBar @isIndeterminate={{true}} data-test-id="progress-bar" />`
          );

          assert
            .dom(
              '[data-test-id="progress-bar"] > div.pb-base > div.pb-progress'
            )
            .hasClass('pb-is-indeterminate');
        });
      });
    });

    module('progress value', () => {
      test('it renders value and arias', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar" @progress={{50}} />`
        );

        const selector = '[data-test-id="progress-bar"] div.pb-progress';
        const width = document.querySelector(selector)?.getAttribute('style');

        assert.equal(width, 'width: 50%');
        assert.dom(selector).hasAttribute('role');
        assert.dom(selector).hasAria('valuenow', '50');
        assert.dom(selector).hasAria('valuemin', '0');
        assert.dom(selector).hasAria('valuemax', '100');
        assert.dom(selector).hasNoAria('labeledBy');
      });

      test('it interpolates values', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar" @progress={{20}} @minValue={{10}} @maxValue={{30}} />`
        );

        const selector = '[data-test-id="progress-bar"] div.pb-progress';
        const width = document.querySelector(selector)?.getAttribute('style');

        assert.equal(width, 'width: 50%');
        assert.dom(selector).hasAttribute('role');
        assert.dom(selector).hasAria('valuenow', '20');
        assert.dom(selector).hasAria('valuemin', '10');
        assert.dom(selector).hasAria('valuemax', '30');
      });
    });

    module('progress label', () => {
      test('it renders label and arias', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar" @progress={{50}} @label="Progress" />`
        );

        assert
          .dom('[data-test-id="progress-bar"] div.pb-progress')
          .hasAttribute('aria-labelledby');

        assert
          .dom('[data-test-id="progress-bar"] div.pb-label > label')
          .containsText('Progress');
        assert
          .dom('[data-test-id="progress-bar"] div.pb-label > div')
          .containsText('50%');
      });

      test('it renders value label', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar" @progress={{50}} @label="Progress" @valueLabel="value 2/4" />`
        );
        assert
          .dom('[data-test-id="progress-bar"] div.pb-label > div')
          .containsText('value 2/4');
      });

      test('it formats value label', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar"
                @progress={{50}} @label="Progress"
                @formatOptions={{(hash style='currency' currency='USD')}}
              />`
        );
        assert
          .dom('[data-test-id="progress-bar"] div.pb-label > div')
          .containsText('$50.00');
      });

      test('it formats value label interpolated', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar"
                @progress={{20}}
                @label="Progress"
                @minValue={{10}}
                @maxValue={{30}}
                @formatOptions={{(hash style='currency' currency='USD')}}
              />`
        );
        assert
          .dom('[data-test-id="progress-bar"] div.pb-label > div')
          .containsText('$20.00');
      });

      test('it hides value label if showValueLabel false', async function (assert) {
        await render(
          hbs`<ProgressBar data-test-id="progress-bar"
                @progress={{20}}
                @label="Progress"
                @showValueLabel={{false}}
              />`
        );
        assert
          .dom('[data-test-id="progress-bar"] div.pb-label > div')
          .doesNotExist();
      });
    });
  }
);
