﻿// SPDX-License-Identifier: MIT
using Microsoft.Extensions.Options;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Reflection;
using System.Runtime.Serialization;
using System.Text;
using System.Text.Json.Serialization;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace Whetstone.ChatGPT.Models
{
    public class ChatGPTChatCompletionMessage : IChatCompletionMessage
    {
        public ChatGPTChatCompletionMessage()
        {
        }

        public ChatGPTChatCompletionMessage(string role, string content)
        {
            if (string.IsNullOrWhiteSpace(role))
                throw new ArgumentException("Cannot be null, empty, or whitespace", nameof(role));

            this.Role = role;
            this.Content = content;
        }
        
        /// <summary>
        /// The role of the messages author, supported values include `assistant`, `system`, `user`, `tool`. `function` is deprecated.
        /// </summary>
        /// <remarks>Defaults to `system`.</remarks>
        [JsonPropertyName("role")]
        public string Role { get; set; } = "system";

        /// <summary>
        /// The contents of the message.
        /// </summary>
        [JsonPropertyName("content")]
        public string? Content { get; set; }

        /// <summary>
        /// An optional name for the participant. Provides the model information to differentiate between participants of the same role.
        /// </summary>
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        [JsonPropertyName("name")]
        public string? Name { get; set; }


        /// <summary>
        /// The tool calls generated by the model, such as function calls.
        /// </summary>
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        [JsonPropertyName("tool_calls")]
        public List<ToolCall>? ToolCalls { get; set; }
    }

    public class ChatGPTAssistantCompletionMessage : IChatCompletionMessage
    { 
        /// <summary>
        /// The role of the messages author is `assistant`.
        /// </summary>
        [JsonPropertyName("role")]
        public string Role { get; } = "assistant";

        /// <summary>
        /// The contents of the message.
        /// </summary>
        [JsonPropertyName("content")]
        public string? Content { get; set; }

        /// <summary>
        /// An optional name for the participant. Provides the model information to differentiate between participants of the same role.
        /// </summary>
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        [JsonPropertyName("name")]
        public string? Name { get; set; }

        /// <summary>
        /// The tool calls generated by the model, such as function calls.
        /// </summary>
        [JsonPropertyName("tool_calls")]
        public List<ToolCall>? ToolCalls { get; set; }
    }

    public class ToolCall
    {

        /// <summary>
        /// The ID of the tool call.
        /// </summary>
        [JsonPropertyName("id")]
        public string Id { get; set; } = string.Empty;

        /// <summary>
        /// The type of the tool.Currently, only `function` is supported.
        /// </summary>
        [JsonPropertyName("type")]
        public string Type { get; set; } = "function";

        /// <summary>
        /// The function that the model called.
        /// </summary>
        [JsonPropertyName("function")]
        public ToolFunction? Function { get; set; }
    }
    
    public class ToolFunction
    {
        /// <summary>
        /// The name of the function to call.
        /// </summary>
        [JsonPropertyName("name")]
        public string Name { get; set; } = String.Empty;

        /// <summary>
        /// The arguments to call the function with, as generated by the model in JSON format. 
        /// Note that the model does not always generate valid JSON, and may hallucinate parameters not defined by your function schema.
        /// Validate the arguments in your code before calling your function.
        /// </summary>
        [JsonPropertyName("arguments")]
        public string Arguments { get; set; } = String.Empty;
    }

    public class ChatGPTUserCompletionMessage : IChatCompletionMessage
    {
        /// <summary>
        /// The role of the messages author is `user`.
        /// </summary>
        [JsonPropertyName("role")]
        public string Role { get; } = "user";

        /// <summary>
        /// The contents of the message.
        /// </summary>
        [JsonPropertyName("content")]
        public string? Content { get; set; }

        /// <summary>
        /// An optional name for the participant. Provides the model information to differentiate between participants of the same role.
        /// </summary>
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        [JsonPropertyName("name")]
        public string? Name { get; set; }
    }

    public class ChatGPTSystemCompletionMessage : IChatCompletionMessage
    {
        /// <summary>
        /// The role of the messages author is `system`.
        /// </summary>
        [JsonPropertyName("role")]
        public string Role { get; } = "system";

        /// <summary>
        /// The contents of the message.
        /// </summary>
        [JsonPropertyName("content")]
        public string? Content { get; set; }

        /// <summary>
        /// An optional name for the participant. Provides the model information to differentiate between participants of the same role.
        /// </summary>
        [JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
        [JsonPropertyName("name")]
        public string? Name { get; set; }
    }

    public class ChatGPTToolCompletionMessage : IChatCompletionMessage
    {
        /// <summary>
        /// The role of the messages author is `tool`.
        /// </summary>
        [JsonPropertyName("role")]
        public string Role { get; } = "tool";

        /// <summary>
        /// The contents of the message.
        /// </summary>
        [JsonPropertyName("content")]
        public string? Content { get; set; }

        /// <summary>
        /// Tool call that this message is responding to.
        /// </summary>
        [JsonPropertyName("tool_call_id")]
        public string ToolCallId { get; set; } = string.Empty;
    }

    // tool_call_id

    public interface IChatCompletionMessage
    {
        public string Role { get; }

        public string? Content { get; set; }
    }
}
