/*
 *  Copyright (C) 2010-2024 JPEXS, All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3.0 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.
 */
package com.jpexs.helpers;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * Future that happens now.
 *
 * @param <V> Value type
 * @author JPEXS
 */
public class ImmediateFuture<V> implements Future<V> {

    private final V value;

    private final Throwable throwable;

    private final boolean cancelled;

    public ImmediateFuture(V value) {
        this.value = value;
        throwable = null;
        cancelled = false;
    }

    public ImmediateFuture(V value, Throwable throwable, boolean cancelled) {
        this.value = value;
        this.throwable = throwable;
        this.cancelled = cancelled;
    }

    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        return false;
    }

    @Override
    public boolean isCancelled() {
        return cancelled;
    }

    @Override
    public boolean isDone() {
        return true;
    }

    @Override
    public V get() throws InterruptedException, ExecutionException {
        if (cancelled) {
            throw new CancellationException();
        }

        if (throwable != null) {
            throw new ExecutionException(throwable);
        }

        return value;
    }

    @Override
    public V get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return get();
    }
}
