﻿/**
 *  Grasscutter Tools
 *  Copyright (C) 2022 jie65535
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published
 *  by the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 **/

using System;
using System.Windows.Forms;

namespace GrasscutterTools
{
    internal static class MultiLanguage
    {
        /// <summary>
        /// 语言名称列表
        /// </summary>
        public static readonly string[] LanguageNames = new string[] { "简体中文", "繁體中文", "English", "Русский" };

        /// <summary>
        /// 语言代码列表
        /// </summary>
        public static readonly string[] Languages = new string[] { "zh-CN", "zh-TW", "en-US", "ru-RU" };

        public static void SetDefaultLanguage(string lang)
        {
            System.Threading.Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo(lang);
            Properties.Settings.Default.DefaultLanguage = lang;
        }

        /// <summary>
        /// 加载语言
        /// </summary>
        /// <param name="form">加载语言的窗口</param>
        /// <param name="formType">窗口的类型</param>
        public static void LoadLanguage(Control form, Type formType)
        {
            if (form != null)
            {
                System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(formType);
                resources.ApplyResources(form, "$this");
                Loading(form, resources);
            }
        }

        /// <summary>
        /// 加载语言
        /// </summary>
        /// <param name="control">控件</param>
        /// <param name="resources">语言资源</param>
        private static void Loading(Control control, System.ComponentModel.ComponentResourceManager resources)
        {
            if (control is MenuStrip strip)
            {
                //将资源与控件对应
                resources.ApplyResources(control, control.Name);
                if (strip.Items.Count > 0)
                {
                    foreach (ToolStripMenuItem c in strip.Items)
                    {
                        //遍历菜单
                        Loading(c, resources);
                    }
                }
            }

            foreach (Control c in control.Controls)
            {
                if (c is UserControl uc)
                {
                    LoadLanguage(uc, uc.GetType());
                }
                else
                {
                    resources.ApplyResources(c, c.Name);
                    Loading(c, resources);
                }
            }
        }

        /// <summary>
        /// 遍历菜单
        /// </summary>
        /// <param name="item">菜单项</param>
        /// <param name="resources">语言资源</param>
        private static void Loading(ToolStripMenuItem item, System.ComponentModel.ComponentResourceManager resources)
        {
            if (item is ToolStripMenuItem tsmi)
            {
                resources.ApplyResources(item, item.Name);
                if (tsmi.DropDownItems.Count > 0)
                {
                    foreach (ToolStripMenuItem c in tsmi.DropDownItems)
                    {
                        Loading(c, resources);
                    }
                }
            }
        }
    }
}