/*
 * Copyright 2015 JAXIO http://www.jaxio.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jaxio.celerio.maven.plugin.celerio;

import com.jaxio.celerio.output.FileTracker;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import java.io.File;
import java.util.List;

import static org.apache.commons.io.FilenameUtils.normalize;

/**
 * This goal deletes the files that were previously generated by Celerio.
 * Generated files that have been manually modified (file size has changed or date of last modification has
 * changed) or files that are excluded by configuration (using the excludedFiles parameter) won't be deleted.
 *
 * @since 1.0.0
 */
@Mojo(name = "cleanGenerated", defaultPhase = LifecyclePhase.CLEAN,requiresProject = false)
public class CleanGeneratedMojo extends AbstractMojo {
    /**
     * Maven project, this is by default the current maven project.
     */
    @Parameter(property = "project",required = true)
    protected MavenProject project;
    /**
     * The current folder.
     */
    @Parameter(property = "basedir")
    protected String baseDir;
    /**
     * Should the clean goal be skipped ?
     * <p>
     * This is a common pattern in Maven, where you can skip plugins using profiles to fully adapt your build.
     */
    @Parameter(property = "celerio-maven-plugin.cleanGenerated.skip",defaultValue = "false")
    protected boolean skip;

    /**
     * Generated files' relative path that MUST NOT be deleted.
     * The path separator must be '/'.
     * <p>
     * You may use the following wildcards:
     * <ul>
     * <li>? matches one character</li>
     * <li>* matches zero or more characters</li>
     * <li>** matches zero or more directories in a path</li>
     * </ul>
     *
     */
    @Parameter
    protected List<String> excludedFiles;
    private ApplicationContext context;

    public void execute() throws MojoExecutionException {
        try {
            process();
        } catch (Exception e) {
            getLog().error(e.getMessage());
            throw new MojoExecutionException(e.getLocalizedMessage(), e);
        }
    }

    private void process() throws Exception {
        if (skip) {
            getLog().info("skipping celerio-maven-plugin cleanGenerated");
            return;
        }
        context = new ClassPathXmlApplicationContext("classpath:applicationContext-celerio.xml");

        FileTracker fileTracker = context.getBean(FileTracker.class);
        fileTracker.deleteGeneratedFileIfIdentical(new File(getProjectBaseDir()), excludedFiles);
    }

    private String getProjectBaseDir() {
        return normalize(project.getBasedir().getAbsoluteFile().getAbsolutePath() + File.separatorChar);
    }
}