/*
 * Copyright 2015 JAXIO http://www.jaxio.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jaxio.celerio.model.support.h2;

import com.jaxio.celerio.model.Attribute;
import com.jaxio.celerio.spi.AttributeSpi;
import com.jaxio.celerio.util.MappedType;

import java.sql.Types;

public class H2Attribute implements AttributeSpi {
    private Attribute attribute;

    @Override
    public void init(Attribute attribute) {
        this.attribute = attribute;
    }

    /**
     * @return "h2"
     */
    @Override
    public String velocityVar() {
        return "h2";
    }

    @Override
    public Object getTarget() {
        return this;
    }

    public String getSqlType() {
        MappedType type = attribute.getMappedType();
        if (type.isBlob()) {
            return "blob";
        } else if (type.isClob()) {
            return "clob";
        } else if (type.isNumeric()) {
            if (type.isInteger()) {
                return "integer";
            } else if (type.isLong()) {
                return "int8";
            } else {
                return "decimal(" + attribute.getColumnConfig().getSize() + "," + attribute.getColumnConfig().getDecimalDigits() + ")";
            }
        } else if (type.isBoolean()) {
            return "boolean";
        } else if (type.isDate()) {
            switch (attribute.getJdbcType().getJdbcType()) {
                case Types.TIME:
                    return "time";
                case Types.DATE:
                    return "date";
                case Types.TIMESTAMP:
                    return "timestamp";
                default:
                    return "unsupported type";
            }
        } else if (type.isString()) {
            return "varchar(" + attribute.getColumnConfig().getSize() + ")";
        } else if (type.isChar()) {
            return "char(1)";
        } else {
            return "unsupported type";
        }
    }

    public String getSqlDefinition() {
        StringBuffer ret = new StringBuffer();
        appendTechnicalName(ret);
        appendSqlType(ret);
        appendIndentity(ret);
        appendDefaultValue(ret);
        appendNullable(ret);
        return ret.toString();
    }

    private void appendTechnicalName(StringBuffer ret) {
        ret.append(attribute.getColumnName());
    }

    private void appendSqlType(StringBuffer ret) {
        ret.append(" ").append(getSqlType());
    }

    private void appendNullable(StringBuffer ret) {
        if (!attribute.isNullable()) {
            ret.append(" not null");
        }
    }

    private void appendDefaultValue(StringBuffer ret) {
        if (attribute.isSimple() && attribute.hasDefaultValue()) {
            String quote = "";
            if (attribute.isString()) {
                quote = "'";
            }
            ret.append(" default ").append(quote).append(quoteSafe(attribute.getColumnConfig().getDefaultValue().trim())).append(quote);
        }
    }

    private void appendIndentity(StringBuffer ret) {
        if (attribute.isSimplePk()) {
            ret.append(" generated by default as identity");
        }
    }

    private String quoteSafe(String s) {
        if (s == null) {
            return s;
        }
        return s.replace("'", "''");
    }
}
