//packer version
///@FILE:../src/intro.js
///@INFO: BASE
//LiteScene by javi.agenjo@gmail.com 2013 @tamats
// github.com/jagenjo/litescene
// dependencies: litegl.js glMatrix.js (and litegraph.js)
//Here goes the licence and some info
//************************************************
//and the commonJS header...
'use strict';

(function(global){

///@FILE:../src/utils/wbin.js
/* WBin: Javi Agenjo javi.agenjo@gmail.com  Febrary 2014

WBin allows to pack binary information easily
Works similar to WAD file format from ID Software. You have binary lumps with a given name (and a special type code).
First we store a file header, then info about every lump, then a big binary chunk where all the lumps data is located.
The lump headers contain info to the position of the data in the lump binary chunk (positions are relative to the binary chung starting position)

Header: (64 bytes total)
	* FOURCC: 4 bytes with "WBIN"
	* Version: 4 bytes for Float32, represents WBin version used to store
	* Flags: 2 bytes to store flags (first byte reserved, second is free to use)
	* Num. lumps: 2 bytes number with the total amount of lumps in this wbin
	* ClassName: 32 bytes to store a classname, used to know info about the object stored
	* extra space for future improvements

Lump header: (64 bytes total)
	* start: 4 bytes (Uint32), where the lump start in the binary area
	* length: 4 bytes (Uint32), size of the lump
	* code: 2 bytes to represent data type using code table (Uint8Array, Float32Array, ...)
	* name: 54 bytes name for the lump

Lump binary: all the binary data...

*/

/**
* WBin allows to create binary files easily (similar to WAD format). You can pack lots of resources in one file or extract them.
* @class WBin
*/

(function(global){

function WBin()
{
}

WBin.classes = {};//if the WBin contains a class it will be seaerch here first (otherwise it will search in the global scope)

WBin.HEADER_SIZE = 64; //num bytes per header, some are free to future improvements
WBin.FOUR_CC = "WBIN";
WBin.VERSION = 0.3; //use numbers, never strings, fixed size in binary
WBin.CLASSNAME_SIZE = 32; //32 bytes: stores a type for the object stored inside this binary

WBin.LUMPNAME_SIZE = 54; //max size of a lump name, it is big because sometimes some names have urls
WBin.LUMPHEADER_SIZE = 4+4+2+WBin.LUMPNAME_SIZE; //32 bytes: 4 start, 4 length, 2 code, 54 name

WBin.CODES = {
	"ArrayBuffer":"AB", "Int8Array":"I1", "Uint8Array":"i1", "Int16Array":"I2", "Uint16Array":"i2", "Int32Array":"I4", "Uint32Array":"i4",
	"Float32Array":"F4", "Float64Array": "F8", "Object":"OB","WideObject":"WO","String":"ST","WideString":"WS","Number":"NU", "null":"00"
};

WBin.REVERSE_CODES = {};
for(var i in WBin.CODES)
	WBin.REVERSE_CODES[ WBin.CODES[i] ] = i;

WBin.FULL_BINARY = 1; //means this binary should be passed as binary, not as object of chunks

/**
* Allows to check if one Uint8Array contains a WBin file
* @method WBin.isWBin
* @param {UInt8Array} data
* @return {boolean}
*/
WBin.isWBin = function(data)
{
	var fourcc = data.subarray(0,4);
	for(var i = 0; i < fourcc.length; i++)
		if(fourcc[i] != 0 && fourcc[i] != WBin.FOUR_CC.charCodeAt(i))
			return false;
	return true;
}

/**
* Builds a WBin data stream from an object (every property of the object will be considered a lump with data)
* It supports Numbers, Strings and TypedArrays or ArrayBuffer
* @method WBin.create
* @param {Object} origin object containing all the lumps, the key will be used as lump name
* @param {String} origin_class_name [Optional] allows to add a classname to the WBin, this is used to detect which class to instance when extracting it
* @return {Uint8Array} all the bytes
*/
WBin.create = function( origin, origin_class_name )
{
	if(!origin)
		throw("WBin null origin passed");

	var flags = new Uint8Array([0,0]);
	var version = new Uint8Array( new Float32Array( [WBin.VERSION] ).buffer );
	origin_class_name = origin_class_name || "";

	//use class binary creator
	if(origin.toBinary)
	{
		var content = origin.toBinary();
		if(!content)
			return null;

		if(content.constructor == ArrayBuffer)
		{
			flags[0] |= WBin.FULL_BINARY;

			var classname = WBin.getObjectClassName( origin );
			//alloc memory
			var data = new Uint8Array(WBin.HEADER_SIZE + content.length);
			//set fourcc
			data.set(WBin.stringToUint8Array( WBin.FOUR_CC ));
			//set version
			data.set(version, 4);
			//Set flags
			data.set(flags, 8);
			//set classname
			data.set(WBin.stringToUint8Array(classname,WBin.CLASSNAME_SIZE), 14);
			//set data
			data.set(content, WBin.HEADER_SIZE);
			return data;
		}
		else
			origin = content;
	}

	//create 
	var total_size = WBin.HEADER_SIZE;
	var lumps = [];
	var lump_offset = 0;

	//gather lumps
	for(var i in origin)
	{
		var data = origin[i];
		if(data == null)
			continue;

		if(data.constructor === Blob || data.constructor === File)
			throw("Wbin does not allow Blobs or Files as data to store, conver to ArrayBuffer");

		var classname = WBin.getObjectClassName(data);

		var code = WBin.CODES[ classname ];
		if(!code) 
			code = "OB"; //generic

		//class specific actions
		if (code == "NU")
			data = new Float64Array([data]);  //data.toString(); //numbers are stored as strings
		else if(code == "OB")
			data = JSON.stringify(data); //serialize the data

		var data_length = 0;

		//convert any string related data to typed arrays
		if( data && data.constructor == String )
		{
			data = WBin.stringToTypedArray( data );
			if(data.constructor === Uint16Array) //careful when wide characters found (international characters)
				code = (code == "OB") ? "WO" : "WS";
		}

		//typed array
		if(data.buffer && data.buffer.constructor == ArrayBuffer)
		{
			//clone the data, to avoid problems with shared arrays
			data = new Uint8Array( new Uint8Array( data.buffer, data.byteOffset, data.byteLength ) ); 
			data_length = data.byteLength;
		}
		else if(data.constructor == ArrayBuffer) //plain buffer
			data_length = data.byteLength;
		else
			throw("WBin: cannot be anything different to ArrayBuffer");

		var lumpname = i.substring(0,WBin.LUMPNAME_SIZE);
		if(lumpname.length < i.length)
			console.error("Lump name is too long (max is "+WBin.LUMPNAME_SIZE+"), it has been cut down, this could lead to an error in the future");
		lumps.push({code: code, name: lumpname, data: data, start: lump_offset, size: data_length});
		lump_offset += data_length;
		total_size += WBin.LUMPHEADER_SIZE + data_length;
	}

	//construct the final file
	var data = new Uint8Array(total_size);
	//set fourcc
	data.set(WBin.stringToUint8Array( WBin.FOUR_CC ));
	//set version
	data.set(version, 4);
	//set flags
	data.set(flags, 8);	
	//set num lumps
	data.set( new Uint8Array( new Uint16Array([lumps.length]).buffer ), 10);	
	//set origin_class_name
	if(origin_class_name)
		data.set( WBin.stringToUint8Array( origin_class_name, WBin.CLASSNAME_SIZE ), 12);

	var lump_data_start = WBin.HEADER_SIZE + lumps.length * WBin.LUMPHEADER_SIZE;

	//copy lumps to final file
	var nextPos = WBin.HEADER_SIZE;
	for(var j in lumps)
	{
		var lump = lumps[j];
		var buffer = lump.data;

		//create lump header
		var lump_header = new Uint8Array( WBin.LUMPHEADER_SIZE );
		lump_header.set( new Uint8Array( (new Uint32Array([lump.start])).buffer ), 0);
		lump_header.set( new Uint8Array( (new Uint32Array([lump.size])).buffer ), 4);
		lump_header.set( WBin.stringToUint8Array( lump.code, 2), 8);
		lump_header.set( WBin.stringToUint8Array( lump.name, WBin.LUMPNAME_SIZE), 10);

		//copy lump header
		data.set(lump_header,nextPos); 
		nextPos += WBin.LUMPHEADER_SIZE;

		//copy lump data
		var view = new Uint8Array( lump.data );
		data.set(view, lump_data_start + lump.start);
	}

	return data;
}


/**
* Extract the info from a Uint8Array containing WBin info and returns the object with all the lumps.
* If the data contains info about the class to instantiate, the WBin instantiates the class and passes the data to it
* @method WBin.load
* @param {UInt8Array} data_array 
* @param {bool} skip_classname avoid getting the instance of the class specified in classname, and get only the lumps
* @param {String} filename [optional] the filename where this wbin came from (important to mark resources)
* @return {*} Could be an Object with all the lumps or an instance to the class specified in the WBin data
*/
WBin.load = function( data_array, skip_classname, filename )
{
	if(!data_array || ( data_array.constructor !== Uint8Array && data_array.constructor !== ArrayBuffer ) )
		throw("WBin data must be ArrayBuffer or Uint8Array");

	//clone to avoid possible memory aligment problems
	data_array = new Uint8Array(data_array);

	var header = WBin.getHeaderInfo(data_array);
	if(!header)
	{
		console.error("Wrong WBin Header");
		return null;
	}

	if(header.version > (new Float32Array([WBin.VERSION])[0]) ) //all this because sometimes there are precission problems
		console.log("ALERT: WBin version is higher that code version");

	var object = null;

	//lump unpacking
	for(var i in header.lumps)
	{
		if(!object) //we do not create the object unless there is a lump
			object = {};

		var lump = header.lumps[i];
		var lump_data = header.lump_data.subarray( lump.start, lump.start + lump.size );

		if(lump.size != lump_data.length )
			throw("WBin: incorrect wbin lump size");

		var lump_final = null;

		var data_class_name = WBin.REVERSE_CODES[ lump.code ];
		if(!data_class_name)
			throw("WBin: Incorrect data code");

		switch(data_class_name)
		{
			case "null": break;
			case "WideString": 
							lump_data = new Uint16Array( (new Uint8Array( lump_data )).buffer ); //no break
			case "String":	lump_final = WBin.TypedArrayToString( lump_data ); break;
			case "Number": 
					if(header.version < 0.3) //LEGACY: remove
						lump_final = parseFloat( WBin.TypedArrayToString( lump_data ) );
					else
						lump_final = (new Float64Array( lump_data.buffer ))[0];
					break;
			case "WideObject": 
							lump_data = new Uint16Array( (new Uint8Array( lump_data )).buffer ); //no break
			case "Object":	
				var str = WBin.TypedArrayToString( lump_data );
				if(str)
					lump_final = JSON.parse( str ); 
				else
					console.warn("WBIN: lump \""+ lump.name +"\" string is empty, skipping.");
				break;
			case "ArrayBuffer": lump_final = new Uint8Array(lump_data).buffer; break; //clone
			default:
				lump_data = new Uint8Array(lump_data); //clone to avoid problems with bytes alignment
				var ctor = WBin.classes[ data_class_name ] || window[ data_class_name ];
				if(!ctor)
					throw("WBin referenced class not found: " + data_class_name );
				if( (lump_data.length / ctor.BYTES_PER_ELEMENT)%1 != 0)
					throw("WBin: size do not match type");
				lump_final = new ctor(lump_data.buffer);
		}
		object[ lump.name ] = lump_final;
	}

	//check if className exists, if it does use internal class parser
	if(!skip_classname && header.classname)
	{
		var ctor = WBin.classes[ header.classname ] || window[ header.classname ];
		if(ctor && ctor.fromBinary)
			return ctor.fromBinary( object, filename );
		else if(ctor && ctor.prototype.fromBinary)
		{
			var inst = new ctor();
			inst.fromBinary( object, filename );
			return inst;
		}
		else
		{
			object["@classname"] = header.classname;
		}
	}	

	return object;
}


/**
* Extract the header info from an ArrayBuffer (it contains version, and lumps info)
* @method WBin.getHeaderInfo
* @param {UInt8Array} data_array 
* @return {Object} Header
*/
WBin.getHeaderInfo = function(data_array)
{
	//check FOURCC
	var fourcc = data_array.subarray(0,4);
	var good_header = true;
	for(var i = 0; i < fourcc.length; i++)
		if(fourcc[i] != 0 && fourcc[i] != WBin.FOUR_CC.charCodeAt(i))
			return null; //wrong fourcc

	var version = WBin.readFloat32( data_array, 4);
	var flags = new Uint8Array( data_array.subarray(8,10) );
	var numlumps = WBin.readUint16(data_array, 10);
	var classname = WBin.TypedArrayToString( data_array.subarray(12,12 + WBin.CLASSNAME_SIZE) );

	var lumps = [];
	for(var i = 0; i < numlumps; ++i)
	{
		var start = WBin.HEADER_SIZE + i * WBin.LUMPHEADER_SIZE;
		var lumpheader = data_array.subarray( start, start + WBin.LUMPHEADER_SIZE );
		var lump = {};
		lump.start = WBin.readUint32(lumpheader,0);
		lump.size  = WBin.readUint32(lumpheader,4);
		lump.code  = WBin.TypedArrayToString(lumpheader.subarray(8,10));
		lump.name  = WBin.TypedArrayToString(lumpheader.subarray(10));
		lumps.push(lump);
	}

	var lump_data = data_array.subarray( WBin.HEADER_SIZE + numlumps * WBin.LUMPHEADER_SIZE );

	return {
		version: version,
		flags: flags,
		classname: classname,
		numlumps: numlumps,
		lumps: lumps,
		lump_data: lump_data
	};
}

WBin.getObjectClassName = function(obj) {
    if (obj && obj.constructor && obj.constructor.toString) {
        var arr = obj.constructor.toString().match(
            /function\s*(\w+)/);
        if (arr && arr.length == 2) {
            return arr[1];
        }
    }
    return undefined;
}

WBin.stringToUint8Array = function(str, fixed_length)
{
	var r = new Uint8Array( fixed_length ? fixed_length : str.length);
	var warning = false;
	for(var i = 0; i < str.length; i++)
	{
		var c = str.charCodeAt(i);
		if(c > 255)
			warning = true;
		r[i] = c;
	}

	if(warning)
		console.warn("WBin: there are characters in the string that cannot be encoded in 1 byte.");
	return r;
}

WBin.TypedArrayToString = function(typed_array, same_size)
{
	var r = "";
	for(var i = 0; i < typed_array.length; i++)
		if (typed_array[i] == 0 && !same_size)
			break;
		else
			r += String.fromCharCode( typed_array[i] );
	//return String.fromCharCode.apply(null,typed_array)
	return r;
}

WBin.stringToTypedArray = function(str, fixed_length)
{
	var r = new Uint8Array( fixed_length ? fixed_length : str.length);
	var warning = false;
	for(var i = 0; i < str.length; i++)
	{
		var c = str.charCodeAt(i);
		if(c > 255)
			warning = true;
		r[i] = c;
	}

	if(!warning)
		return r;

	//convert to 16bits per character
	var r = new Uint16Array( fixed_length ? fixed_length : str.length);
	for(var i = 0; i < str.length; i++)
	{
		var c = str.charCodeAt(i);
		r[i] = c;
	}

	return r;
}

WBin.readUint16 = function( buffer, pos )
{
	var dv = new DataView( buffer.buffer, buffer.byteOffset );
	return dv.getUint16( pos, true );
}

WBin.readUint32 = function(buffer, pos)
{
	var dv = new DataView( buffer.buffer, buffer.byteOffset);
	return dv.getUint32( pos, true );
}

WBin.readFloat32 = function(buffer, pos)
{
	var dv = new DataView( buffer.buffer, buffer.byteOffset );
	return dv.getFloat32( pos, true );
}

global.WBin = WBin;

})( typeof(global) != "undefined" ? global : this );
///@FILE:../src/utils/lscript.js
///@INFO: SCRIPTS
// ******* LScript  **************************

/**
* LScript allows to compile code during execution time having a clean context
* It adds some syntactic features and controls errors in a safe way
* @class LScript
* @constructor
*/

function LScript()
{
	this.code = "function update(dt) {\n\n}";
	this.exported_callbacks = []; //detects if there is a function with this name and exports it as a property
	this.extracode = "";
	this.extra_methods = null; //add object with methods here to attach methods
}


LScript.onerror = null; //global used to catch errors in scripts

LScript.eval = function(argv_names,code) { return eval("(function("+argv_names+"){\n"+code+"\n})"); }; //not used

LScript.catch_exceptions = false;
LScript.show_errors_in_console = true;

//compiles the string, tryes to keep the current state
LScript.prototype.compile = function( arg_vars, save_context_vars )
{
	var argv_names = [];
	var argv_values = [];
	if(arg_vars)
	{
		for(var i in arg_vars)
		{
			argv_names.push(i);
			argv_values.push( arg_vars[i]);
		}
	}
	argv_names = argv_names.join(",");

	var code = this.code;
	code = LScript.expandCode( code );

	var extra_code = "";
	for(var i in this.exported_callbacks)
	{
		var callback_name = this.exported_callbacks[i];
		extra_code += "	if(typeof("+callback_name+") != 'undefined' && "+callback_name+" != window[\""+callback_name+"\"] ) this."+callback_name + " = "+callback_name+";\n";
	}
	code += extra_code;
	this._last_executed_code = code;

	var old_context = this._context;

	if(!LScript.catch_exceptions)
	{
		this._class = new Function(argv_names, code);//<-- PARSING POINT HERE ***************************************
		var context_function = LScript.applyToConstructor( this._class, argv_values, this.extra_methods ); //bind globals and methods to context
		this._context = new context_function(); //<-- EXECUTION POINT HERE ***************************************
	}
	else
	{
		try
		{
			//LScript.eval(argv_names,code);
			this._class = new Function(argv_names, code);
			var context_function = LScript.applyToConstructor( this._class, argv_values, this.extra_methods ); //bind globals and methods to context
			this._context = new context_function(); //<-- EXECUTION POINT HERE ***************************************
		}
		catch (err)
		{
			if(!this._class)
			{
				console.error("Parsing error in script\n" + err);
			}

			this._class = null;
			this._context = null;
			if(LScript.show_errors_in_console)
			{
				var error_line = LScript.computeLineFromError(err);
				console.error("Error in script\n" + err);
				if( console.groupCollapsed )
				{
					console.groupCollapsed("Error line: " + error_line + " Watch code");
					LScript.showCodeInConsole( this._last_executed_code, error_line );
					console.groupEnd();
				}
				else
					console.error("Error line: " + error_line);
			}
			if(this.onerror)
				this.onerror(err, this._last_executed_code);
			if(LScript.onerror)
				LScript.onerror(err, this._last_executed_code, this);
			return false;
		}
	}

	if(save_context_vars && old_context)
	{
		for(var i in old_context)
			if( this._context[i] !== undefined && old_context[i] && old_context[i].constructor !== Function && (!this._context[i] || this._context[i].constructor !== Function) )
				this._context[i] = old_context[i];
	}

	return true;
}

//does this script contains this method?
LScript.prototype.hasMethod = function(name)
{
	if(!this._context || !this._context[name] || typeof(this._context[name]) != "function") 
		return false;
	return true;
}

//argv must be an array with parameters, unless skip_expand is true
LScript.prototype.callMethod = function( name, argv, expand_parameters, parent_object )
{
	if(!this._context || !this._context[name]) 
		return;

	if(!LScript.catch_exceptions)
	{
		//call expanding parameters
		if(argv && argv.constructor === Array && expand_parameters)
			return this._context[name].apply(this._context, argv);
		//call without expanding parameters
		return this._context[name].call(this._context, argv);
	}

	try
	{
		//call expanding parameters
		if(argv && argv.constructor === Array && expand_parameters)
			return this._context[name].apply(this._context, argv);
		//call without expanding parameters
		return this._context[name].call(this._context, argv);
	}
	catch(err)
	{
		//catch error in script, detect line and show console info
		var error_line = LScript.computeLineFromError(err);
		var parent_info = ""; 
		if (parent_object && parent_object.toInfoString )
			parent_info = " from " + parent_object.toInfoString();
		console.error("Error from function " + name + parent_info + ": ", err.toString());
		if( console.groupCollapsed )
		{
			console.groupCollapsed("Error line: " + error_line + " Watch code");
			LScript.showCodeInConsole( this._last_executed_code, error_line );
			console.groupEnd();
		}
		else
			console.error("Error line: " + error_line);
		if(this.onerror)
			this.onerror({ error: err, msg: err.toString(), line: error_line, lscript: this, code: this._last_executed_code, method_name: name });
	}
}

//Given a constructor, it attaches several global arguments and methods (from kybernetikos in stackoverflow)
LScript.applyToConstructor = function(constructor, argArray, methods) {
    var args = [null].concat(argArray);
	if(methods)
		for(var i in methods)
			Object.defineProperty( constructor.prototype, i, { value: methods[i], enumerable: true });
    var factoryFunction = constructor.bind.apply(constructor, args);
    return factoryFunction;
}

//dumps all the code to the console
LScript.showCodeInConsole = function( code, error_line)
{
	if(!code)
		return;
	var lines = code.split("\n");
	var gutter_style = "display: inline-block; width: 40px; background-color:#999; color: white;";
	for(var i = 0; i < lines.length; i++ )
		if(i == error_line)
			console.log("%c "+i+". " + lines[i], "background-color: #A33; color: #FAA;" );
		else
			console.log("%c "+i+". ", gutter_style, lines[i] );
}

//remove comments and trims empty lines
LScript.cleanCode = function(code)
{
	if(!code)
		return "";
	/*  this should be removed 
		I write this to test this func using LScript.cleanCode( LScript.cleanCode.toString() );
	*/
	//var rx = /(\/\*([^*]|[\r\n]|(\*+([^*\/]|[\r\n])))*\*+\/)|(\/\/.*)/g;
	var rx = /\*([^*]|[\r\n]|(\*+([^*/]|[\r\n])))*\*+/g;
	var code = code.replace( rx ,"");
	var lines = code.split("\n");
	var result = [];
	for(var i = 0; i < lines.length; ++i)
	{
		var line = lines[i]; 
		var pos = line.indexOf("//");
		if(pos != -1 && line.substr(0,pos).indexOf("\"") == -1) //avoid removing lines with comments inside strings
			line = line.substr(0,pos);
		line = line.trim();
		if(line.length)
			result.push(line);
	}
	return result.join("\n");
}

// Adds some extra features to JS like:
// - support for multiline strings (this feature was introduced in ES6 but in case is not supported)
// - the use of private or public in variables.
LScript.expandCode = function(code)
{
	if(!code)
		return "";

	//allow support to multiline strings
	if( code.indexOf("'''") != -1 )
	{
		var lines = code.split("'''");
		code = "";
		for(var i = 0; i < lines.length; i++)
		{
			if(i % 2 == 0)
			{
				code += lines[i];
				continue;
			}

			code += '"' + lines[i].split("\n").join("\\n\\\n") + '"';
		}
	}

	//allow to use public var foo = 10;
	var lines = code.split("\n");
	var update = false;
	for(var i = 0; i < lines.length; ++i)
	{
		var line = lines[i].trim();
		var pos = line.indexOf(" ");
		var first_word = line.substr(0,pos);

		//all this horrendous code to parse "public var name : type = value;" and all the possible combinations
		if( first_word == "public" || first_word == "private" || first_word == "hidden" )
		{
			var index = line.indexOf("//");
			if(index != -1)
				line = line.substr(0,index); //remove one-line comments
			var index = line.lastIndexOf(";");
			if(index != -1)
				line = line.substr(0,index); //remove semicolon
			var t = line.split(" ");
			if( t[1] != 'var')
				continue;
			var text = line;
			var type = null;
			var value = "undefined";
			var equal_index = text.indexOf("=");
			if( equal_index != -1 )
			{
				value = text.substr( equal_index + 1 ).trim();
				text = text.substr( 0, equal_index ).trim();
			}

			var colon_index = text.indexOf(":");
			if(colon_index != -1)
			{
				type = text.substr( colon_index + 1 ).trim();
				text = text.substr( 0, colon_index ).trim();
			}
			var keywords = text.split(" ");

			var varname = keywords[2];
			if(!varname)
				continue;
			var type_options = {};
			if(type)
			{
				var array_index = type.indexOf('[]');
				if( array_index != -1 )
				{
					type_options.type = LS.TYPES.ARRAY;
					type_options.data_type = type.substr( 0, array_index );
					if(!value || value === "undefined")
						value = "[]";
				}
				else
				{
					type_options.type = type;
				}

				if( LS.Components[ type ] ) //for components
				{
					type_options.component_class = type;
					type_options.type = LS.TYPES.COMPONENT;
					if(!value)
						value = "null";
				}
				else if( LS.ResourceClasses[ type ] ) //for resources
				{
					type_options.resource_classname = type;
					type_options.type = LS.TYPES.RESOURCE;
					if(!value)
						value = "null";
				}
				else if( type == "int" || type == "integer")
				{
					type_options.step = 1;
					type_options.type = LS.TYPES.NUMBER;
					if(!value)
						value = 0;
				}
			}
			if( keywords[0] == "private" || keywords[0] == "hidden" )
				type_options.widget = "null";
			lines[i] = "this.createProperty('" + varname + "'," + value + ", "+JSON.stringify( type_options )+" );";
			update = true;
		}
	}
	if(update)
		code = lines.join("\n");

	return code;
}

// In case of error inside the scripts, tries to determine the error line (not as easy as it seems)
// Doesnt work in all cases
LScript.computeLineFromError = function( err )
{
	if(err.lineNumber !== undefined)
	{
		return err.lineNumber;
	}
	else if(err.stack)
	{
		var lines = err.stack.split("\n");
		var line = lines[1].trim();
		if(line.indexOf("(native)") != -1)
			return -1;
		var tokens = line.split(" ");
		var pos = line.lastIndexOf(":");
		var pos2 = line.lastIndexOf(":",pos-1);
		var num = parseInt( line.substr(pos2+1,pos-pos2-1) );
		var ch = parseInt( line.substr(pos+1, line.length - 2 - pos) );
		if(tokens[1] == "Object.CodingModule.eval")
			return -1;
		if (line.indexOf("LScript") != -1 || line.indexOf("<anonymous>") != -1 )
			num -= 3; //ignore the header lines of the LScript class
		return num;
	}
	return -1;
}


global.LScript = LScript;


///@FILE:../src/core.js
//Global Scope
//better array conversion to string for serializing
if( !Uint8Array.prototype.toJSON )
{
	var typed_arrays = [ Uint8Array, Int8Array, Uint16Array, Int16Array, Uint32Array, Int32Array, Float32Array, Float64Array ];
	typed_arrays.forEach( function(v) { 
		v.prototype.toJSON = function(){ return Array.prototype.slice.call(this); }
	});
}

if( typeof(GL) === "undefined" )
	console.error("LiteScene requires to include LiteGL first. More info: https://github.com/jagenjo/litegl.js");


/**
* LS is the global scope for the global functions and containers of LiteScene
*
///@INFO: BASE
* @class  LS
* @module LS
*/

var LS = {

	Version: 0.5,

	//systems: defined in their own files
	ResourcesManager: null,
	Picking: null,
	Player: null,
	GUI: null,
	Network: null,
	Input: null,
	Renderer: null,
	Physics: null,
	ShadersManager: null,
	Formats: null,
	Tween: null,

	//containers
	Classes: {}, //maps classes name like "Prefab" or "Animation" to its namespace "LS.Prefab". Used in Formats and ResourceManager when reading classnames from JSONs or WBin.
	ResourceClasses: {}, //classes that can contain a resource of the system
	ResourceClasses_by_extension: {}, //used to associate JSONs to resources, only used by GRAPHs
	Globals: {}, //global scope to share info among scripts

	/**
	* Contains all the registered components
	* 
	* @property Components
	* @type {Object}
	* @default {}
	*/
	Components: {},

	/**
	* Contains all the registered material classes
	* 
	* @property MaterialClasses
	* @type {Object}
	* @default {}
	*/
	MaterialClasses: {},

	//vars used for uuid genereration
	_last_uid: 1,
	_uid_prefix: "@", //WARNING: must be one character long
	debug: false, //enable to see verbose output
	allow_static: true, //used to disable static instances in the editor
	allow_scripts: true, //if true, then Script components and Graphs can contain code

	//for HTML GUI
	_gui_element: null,
	_gui_style: null,

	/**
	* Generates a UUID based in the user-agent, time, random and sequencial number. Used for Nodes and Components.
	* @method generateUId
	* @return {string} uuid
	*/
	generateUId: function ( prefix, suffix ) {
		prefix = prefix || "";
		suffix = suffix || "";
		var str = this._uid_prefix + prefix + (window.navigator.userAgent.hashCode() % 0x1000000).toString(16) + "-"; //user agent
		str += (GL.getTime()|0 % 0x1000000).toString(16) + "-"; //date
		str += Math.floor((1 + Math.random()) * 0x1000000).toString(16) + "-"; //rand
		str += (this._last_uid++).toString(16); //sequence
		str += suffix;
		return str; 
	},

	/**
	* validates name string to ensure there is no forbidden characters
	* valid characters are letters, numbers, spaces, dash, underscore and dot
	* @method validateName
	* @param {string} name
	* @return {boolean} 
	*/
	validateName: function(v)
	{
		var exp = /^[a-z\s0-9-_.]+$/i; //letters digits and dashes
		return v.match(exp);
	},

	valid_property_types: ["String","Number","Boolean","color","vec2","vec3","vec4","quat","mat3","mat4","Resource","Animation","Texture","Prefab","Mesh","ShaderCode","node","component"],
	
	//used when creating a property to a component, to see if the type is valid
	validatePropertyType: function(v)
	{
		if(	this.valid_property_types.indexOf(v) == -1 )
		{
			console.error( v + " is not a valid property value type." );
			return false;
		}
		return true;
	},

	_catch_exceptions: false, //used to try/catch all possible callbacks (used mostly during development inside an editor) It is linked to LScript too

	/**
	* Register a component (or several) so it is listed when searching for new components to attach
	*
	* @method registerComponent
	* @param {Component} component component class to register
	* @param {String} old_classname [optional] the name of the component that this class replaces (in case you are renaming it)
	*/
	registerComponent: function( component, old_classname ) { 

		//to know from what file does it come from
		//console.log( document.currentScript.src );

		//allows to register several at the same time
		var name = LS.getClassName( component );

		if(old_classname && old_classname.constructor !== String)
			throw("old_classname must be null or a String");

		//save the previous class in case we are replacing it
		var old_class = this.Components[ old_classname || name ];

		//register
		this.Components[ name ] = component; 
		component.is_component = true;	
		component.resource_type = "Component";

		//Helper: checks for errors
		if( !!component.prototype.onAddedToNode != !!component.prototype.onRemovedFromNode ||
			!!component.prototype.onAddedToScene != !!component.prototype.onRemovedFromScene )
			console.warn("%c Component "+name+" could have a bug, check events: " + name , "font-size: 2em");
		if( component.prototype.getResources && !component.prototype.onResourceRenamed )
			console.warn("%c Component "+name+" could have a bug, it uses resources but doesnt implement onResourceRenamed, this could lead to problems when resources are renamed.", "font-size: 1.2em");
		//if( !component.prototype.serialize || !component.prototype.configure )
		//	console.warn("%c Component "+name+" could have a bug, it doesnt have a serialize or configure method. No state will be saved.", "font-size: 1.2em");

		//add stuff to the class
		if(!component.actions)
			component.actions = {};

		//add default methods
		LS.extendClass( component, LS.BaseComponent );
		BaseComponent.addExtraMethods( component );

		if( LS.debug )
		{
			var c = new component();
			var r = c.serialize();
			if(!r.object_class)
				console.warn("%c Component "+name+" could have a bug, serialize() method has object_class missing.", "font-size: 1.2em");
		}

		//event
		LEvent.trigger(LS, "component_registered", component ); 

		if(LS.GlobalScene) //because main components are create before the global scene is created
		{
			this.replaceComponentClass( LS.GlobalScene, old_classname || name, name );
			if( old_classname != name )
				this.unregisterComponent( old_classname );
		}

	},

	/**
	* Unregisters a component from the system (although existing instances are kept in the scene)
	*
	* @method unregisterComponent
	* @param {String} name the name of the component to unregister
	*/
	unregisterComponent: function( name ) { 
		//not found
		if(!this.Components[name])
			return;
		//delete from the list of component (existing components will still exists)
		delete this.Components[name];
	},


	/**
	* Tells you if one class is a registered component class
	*
	* @method isClassComponent
	* @param {ComponentClass} comp component class to evaluate
	* @return {boolean} true if the component class is registered
	*/
	isClassComponent: function( comp_class )
	{
		var name = this.getClassName( comp_class );
		return !!this.Components[name];
	},

	/**
	* Replaces all components of one class in the scene with components of another class
	*
	* @method replaceComponentClass
	* @param {Scene} scene where to apply the replace
	* @param {String} old_class_name name of the class to be replaced
	* @param {String} new_class_name name of the class that will be used instead
	* @return {Number} the number of components replaced
	*/
	replaceComponentClass: function( scene, old_class_name, new_class_name )
	{
		var proposed_class = new_class_name.constructor === String ? LS.Components[ new_class_name ] : new_class_name;
		if(!proposed_class)
			return 0;

		//this may be a problem if the old class has ben unregistered...
		var old_class = null;
		
		if(	old_class_name.constructor === String )
		{
			old_class = LS.Components[ old_class_name ];
			if( old_class )
				old_class_name = LS.getClassName( old_class );
		}

		var num = 0;

		for(var i = 0; i < scene._nodes.length; ++i)
		{
			var node = scene._nodes[i];

			//search in current components
			for(var j = 0; j < node._components.length; ++j)
			{
				var comp = node._components[j];
				var comp_name = comp.constructor === LS.MissingComponent ? comp._comp_class : LS.getObjectClassName( comp );

				//it it is the exact same class then skip it
				if( comp.constructor === proposed_class )
					continue;

				//if this component is neither the old comp nor the new one
				if( comp_name != old_class_name && comp_name != new_class_name ) 
					continue;

				var info = comp.serialize();
				node.removeComponent( comp );

				var new_comp = null;
				try
				{
					new_comp = new proposed_class();
				}
				catch (err)
				{
					console.error("Error in replacement component constructor");
					console.error(err);
					continue;
				}
				node.addComponent( new_comp, j < node._components.length ? j : undefined );
				new_comp.configure( info );
				num++;
			}
		}

		return num;
	},

	/**
	* Register a resource class so we know which classes could be use as resources
	*
	* @method registerResourceClass
	* @param {ComponentClass} c component class to register
	*/
	registerResourceClass: function( resourceClass )
	{
		var class_name = LS.getClassName( resourceClass );
		this.ResourceClasses[ class_name ] = resourceClass;
		this.Classes[ class_name ] = resourceClass;
		resourceClass.is_resource = true;

		if( !resourceClass.FORMAT )
			resourceClass.FORMAT = {};

		resourceClass.FORMAT.resource = class_name;
		resourceClass.FORMAT.resourceClass = resourceClass;

		if( resourceClass.EXTENSION )
			resourceClass.FORMAT.extension = resourceClass.EXTENSION.toLowerCase();

		var extension = resourceClass.FORMAT.extension;
		if(!extension && resourceClass != LS.Resource )
			console.warn("Resource without extension info? " + class_name );
		else
		{
			this.ResourceClasses_by_extension[ extension ] = resourceClass;
			resourceClass.EXTENSION = extension;
		}

		if( LS.Formats && extension )
		{
			var format_info = LS.Formats.supported[ extension ];
			if( !format_info )
				LS.Formats.supported[ extension ] = format_info = resourceClass.FORMAT;
			else
			{
				if(!format_info.resourceClass)
					format_info.resourceClass = resourceClass;
				//else if(format_info.resourceClass != resourceClass) //animations and prefab use the same file extension
				//	console.warn("format has resourceClass that do not match this resource: ", LS.getClassName(format_info.resourceClass), LS.getClassName(resourceClass) );
			}
		}

		//some validation here? maybe...
	},

	//Coroutines that allow to work with async functions
	coroutines: {},

	addWaitingCoroutine: function( resolve, event )
	{
		event = event || "render";
		var coroutines = this.coroutines[ event ];
		if(!coroutines)
			coroutines = this.coroutines[ event ] = [];
		coroutines.push( resolve );
	},

	triggerCoroutines: function( event, data )
	{
		event = event || "render";
		var coroutines = this.coroutines[ event ];
		if(!coroutines)
			return;
		for(var i = 0; i < coroutines.length; ++i)
			LS.safeCall( coroutines[i], data ); //call resolves
		coroutines.length = 0;
	},

	createCoroutine: function( event )
	{
		return new Promise(function(resolve){
			LS.addWaitingCoroutine( resolve, event );
		});
	},

	/**
	* Returns a Promise that will be fulfilled once the time has passed
	* @method sleep
	* @param {Number} ms time in milliseconds
	* @return {Promise} 
	*/
	sleep: function(ms) {
	  return new Promise( function(resolve){ setTimeout(resolve, ms); });
	},

	/**
	* Returns a Promise that will be fulfilled when the next frame is rendered
	* @method nextFrame
	* @return {Promise} 
	*/
	nextFrame: function( skip_request )
	{
		if(!skip_request)
			LS.GlobalScene.requestFrame();
		return new Promise(function(resolve){
			LS.addWaitingCoroutine( resolve, "render" );
		});
	},

	/**
	* Is a wrapper for callbacks that throws an LS "exception" in case something goes wrong (needed to catch the error from the system and editor)
	* @method safeCall
	* @param {function} callback
	* @param {array} params
	* @param {object} instance
	*/
	safeCall: function(callback, params, instance)
	{
		if(!LS.catch_exceptions)
		{
			if(instance)
				return callback.apply( instance, params );
			return callback( params );
		}

		try
		{
			return callback.apply( instance, params );
		}
		catch (err)
		{
			LEvent.trigger(LS,"exception",err);
			//test this
			//throw new Error( err.stack );
			console.error( err.stack );
		}
	},

	/**
	* Is a wrapper for setTimeout that throws an LS "code_error" in case something goes wrong (needed to catch the error from the system)
	* @method setTimeout
	* @param {function} callback
	* @param {number} time in ms
	* @param {number} timer_id
	*/
	setTimeout: function(callback, time)
	{
		if(!LS.catch_exceptions)
			return setTimeout( callback,time );

		try
		{
			return setTimeout( callback,time );
		}
		catch (err)
		{
			LEvent.trigger(LS,"exception",err);
		}
	},

	/**
	* Is a wrapper for setInterval that throws an LS "code_error" in case something goes wrong (needed to catch the error from the system)
	* @method setInterval
	* @param {function} callback
	* @param {number} time in ms
	* @param {number} timer_id
	*/
	setInterval: function(callback, time)
	{
		if(!LS.catch_exceptions)
			return setInterval( callback,time );

		try
		{
			return setInterval( callback,time );
		}
		catch (err)
		{
			LEvent.trigger(LS,"exception",err);
		}
	},

	/**
	* copy the properties (methods and properties) of origin class into target class
	* @method extendClass
	* @param {Class} target
	* @param {Class} origin
	*/
	extendClass: function( target, origin ) {
		for(var i in origin) //copy class properties
		{
			if(target.hasOwnProperty(i))
				continue;
			target[i] = origin[i];
		}

		if(origin.prototype) //copy prototype properties
			for(var i in origin.prototype) //only enumerables
			{
				if(!origin.prototype.hasOwnProperty(i)) 
					continue;

				if(target.prototype.hasOwnProperty(i)) //avoid overwritting existing ones
					continue;

				//copy getters 
				if(origin.prototype.__lookupGetter__(i))
					target.prototype.__defineGetter__(i, origin.prototype.__lookupGetter__(i));
				else 
					target.prototype[i] = origin.prototype[i];

				//and setters
				if(origin.prototype.__lookupSetter__(i))
					target.prototype.__defineSetter__(i, origin.prototype.__lookupSetter__(i));
			}
	},

	/**
	* Clones an object (no matter where the object came from)
	* - It skip attributes starting with "_" or "jQuery" or functions
	* - it tryes to see which is the best copy to perform
	* - to the rest it applies JSON.parse( JSON.stringify ( obj ) )
	* - use it carefully
	* @method cloneObject
	* @param {Object} object the object to clone
	* @param {Object} target=null optional, the destination object
	* @param {bool} recursive=false optional, if you want to encode objects recursively
	* @param {bool} only_existing=false optional, only assign to methods existing in the target object
	* @param {bool} encode_objets=false optional, if a special object is found, encode it as ["@ENC",node,object]
	* @return {Object} returns the cloned object (target if it is specified)
	*/
	cloneObject: function( object, target, recursive, only_existing, encode_objects )
	{
		if(object === undefined)
			return undefined;
		if(object === null)
			return null;

		//base type
		switch( object.constructor )
		{
			case String:
			case Number:
			case Boolean:
				return object;
		}

		//typed array
		if( object.constructor.BYTES_PER_ELEMENT )
		{
			if(!target)
				return new object.constructor( object );
			if(target.set)
				target.set(object);
			else if(target.construtor === Array)
			{
				for(var i = 0; i < object.length; ++i)
					target[i] = object[i];
			}
			else
				throw("cloneObject: target has no set method");
			return target;
		}

		var o = target;
		if(o === undefined || o === null)
		{
			if(object.constructor === Array)
				o = [];
			else
				o = {};
		}

		//copy every property of this object
		for(var i in object)
		{
			if(i[0] == "@" || i[0] == "_" || i.substr(0,6) == "jQuery") //skip vars with _ (they are private) or '@' (they are definitions)
				continue;

			if(only_existing && !target.hasOwnProperty(i) && !target.__proto__.hasOwnProperty(i) ) //target[i] === undefined)
				continue;

			//if(o.constructor === Array) //not necessary
			//	i = parseInt(i);

			var v = object[i];
			if(v == null)
				o[i] = null;			
			else if ( isFunction(v) ) //&& Object.getOwnPropertyDescriptor(object, i) && Object.getOwnPropertyDescriptor(object, i).get )
				continue;//o[i] = v;
			else if (v.constructor === File ) 
				o[i] = null;
			else if (v.constructor === Number || v.constructor === String || v.constructor === Boolean ) //elemental types
				o[i] = v;
			else if( v.buffer && v.byteLength && v.buffer.constructor === ArrayBuffer ) //typed arrays are ugly when serialized
			{
				if(o[i] && v && only_existing) 
				{
					if(o[i].length == v.length) //typed arrays force to fit in the same container
						o[i].set( v );
				}
				else
					o[i] = new v.constructor(v); //clone typed array
			}
			else if ( v.constructor === Array ) //clone regular array (container and content!)
			{
				//not safe to use concat or slice(0) because it doesnt clone content, only container
				if( o[i] && o[i].set && o[i].length >= v.length ) //reuse old container
				{
					o[i].set(v);
					continue;
				}

				if( encode_objects && target && v[0] == "@ENC" ) //encoded object (component, node...)
				{
					var decoded_obj = LS.decodeObject(v);
					o[i] = decoded_obj;
					if(!decoded_obj) //object not found
					{
						if( LS._pending_encoded_objects )
							LS._pending_encoded_objects.push([o,i,v]);
						else
							console.warn( "Object UID referencing object not found in the scene:", v[2] );
					}
				}
				else
					o[i] = LS.cloneObject( v ); 
			}
			else //Objects: 
			{
				if( v.constructor.is_resource )
				{
					console.error("Resources cannot be saved as a property of a component nor script, they must be saved individually as files in the file system. If assigning them to a component/script use private variables (name start with underscore) to avoid being serialized.");
					continue;
				}

				if( encode_objects && !target )
				{
					o[i] = LS.encodeObject(v);
					continue;
				}

				if( v.constructor !== Object && !target && !v.toJSON )
				{
					console.warn("Cannot clone internal classes:", LS.getObjectClassName( v )," When serializing an object I found a var with a class that doesnt support serialization. If this var shouldnt be serialized start the name with underscore.'");
					continue;
				}

				if( v.toJSON )
					o[i] = v.toJSON();
				else if( recursive )
					o[i] = LS.cloneObject( v, null, true );
				else {
					if(v.constructor !== Object && LS.Classes[ LS.getObjectClassName(v) ])
						console.warn("Cannot clone internal classes:", LS.getObjectClassName(v)," When serializing an object I found a var with a class that doesnt support serialization. If this var shouldnt be serialized start the name with underscore.'" );

					if(LS.catch_exceptions)
					{
						try
						{
							//prevent circular recursions //slow but safe
							o[i] = JSON.parse( JSON.stringify(v) );
						}
						catch (err)
						{
							console.error(err);
						}
					}
					else //slow but safe
					{
						o[i] = JSON.parse( JSON.stringify(v) );
					}
				}
			}
		}
		return o;
	},

	encodeObject: function( obj )
	{
		if( !obj || obj.constructor === Number || obj.constructor === String || obj.constructor === Boolean || obj.constructor === Object ) //regular objects
			return obj;
		if( obj.constructor.is_component && obj._root) //in case the value of this property is an actual component in the scene
			return [ "@ENC", LS.TYPES.COMPONENT, obj.getLocator(), LS.getObjectClassName( obj ) ];
		if( obj.constructor == LS.SceneNode && obj._in_tree) //in case the value of this property is an actual node in the scene
			return [ "@ENC", LS.TYPES.SCENENODE, obj.uid ];
		if( obj.constructor == LS.Scene)
			return [ "@ENC", LS.TYPES.SCENE, obj.fullpath ]; //weird case
		if( obj.serialize || obj.toJSON )
		{
			//return obj.serialize ? obj.serialize() : obj.toJSON(); //why not this?
			return [ "@ENC", LS.TYPES.OBJECT, obj.serialize ? obj.serialize() : obj.toJSON(), LS.getObjectClassName( obj ) ];
		}
		console.warn("Cannot clone internal classes:", LS.getObjectClassName( obj )," When serializing an object I found a property with a class that doesnt support serialization. If this property shouldn't be serialized start the name with underscore.'");
		return null;
	},

	decodeObject: function( data )
	{
		if(!data || data.constructor !== Array || data[0] != "@ENC" )
			return null;

		switch( data[1] )
		{
			case LS.TYPES.COMPONENT: 
			case "node":  //legacy
			case LS.TYPES.SCENENODE: 
				var obj = LSQ.get( data[2] );
				if( obj )
					return obj;
				return null;
				break;
				//return  break;
			case LS.TYPES.SCENE: return null; break; //weird case
			case LS.TYPES.OBJECT: 
			default:
				if( !data[2] || !data[2].object_class )
					return null;
				var ctor = LS.Classes[ data[2].object_class ];
				if(!ctor)
					return null;
				var v = new ctor();
				v.configure(data[2]);
				return v;
			break;
		}
		return null;
	},

	//used during scene configure because when configuring objects they may have nodes encoded in the properties
	resolvePendingEncodedObjects: function()
	{
		if(!LS._pending_encoded_objects)
		{
			console.warn("no pending enconded objects");
			return;
		}
		for(var i = 0; i < LS._pending_encoded_objects.length; ++i)
		{
			var pending = LS._pending_encoded_objects[i];
			var decoded_object = LS.decodeObject(pending[2]);
			if(decoded_object)
				pending[0][ pending[1] ] = decoded_object;
			else
				console.warn("Decoded object not found when configuring from JSON");
		}
		LS._pending_encoded_objects = null;
	},

	switchGlobalScene: function( scene )
	{
		if(scene === LS.GlobalScene)
			return;
		if( scene === null || scene.constructor !== LS.Scene )
			throw("Not an scene");
		var old_scene = LS.GlobalScene;
		LEvent.trigger( LS, "global_scene_changed", scene );
		LS.GlobalScene = scene;
	},

	/**
	* Clears all the uids inside this object and children (it also works with serialized object)
	* @method clearUIds
	* @param {Object} root could be a node or an object from a node serialization
	*/
	clearUIds: function( root, uids_removed )
	{
		uids_removed = uids_removed || {};

		if(root.uid)
		{
			uids_removed[ root.uid ] = root;
			delete root.uid;
		}

		//remove for embeded materials
		if(root.material && root.material.uid)
		{
			uids_removed[ root.material.uid ] = root.material;
			delete root.material.uid;
		}

		var components = root.components;
		if(!components && root.getComponents)
			components = root.getComponents();

		if(!components)
			return;

		if(components)
		{
			for(var i in components)
			{
				var comp = components[i];
				if(comp[1].uid)
				{
					uids_removed[ comp[1].uid ] = comp[1];
					delete comp[1].uid;
				}
				if(comp[1]._uid)
				{
					uids_removed[ comp[1]._uid ] = comp[1];
					delete comp[1]._uid;
				}
			}
		}

		var children = root.children;
		if(!children && root.getChildren)
			children = root.getChildren();

		if(!children)
			return;

		for(var i in children)
			LS.clearUIds( children[i], uids_removed );

		return uids_removed;
	},


	/**
	* Returns an object class name (uses the constructor toString)
	* @method getObjectClassName
	* @param {Object} the object to see the class name
	* @return {String} returns the string with the name
	*/
	getObjectClassName: function(obj)
	{
		if (!obj)
			return;

		if(obj.constructor.fullname) //this is to overwrite the common name "Prefab" for a global name "LS.Prefab"
			return obj.constructor.fullname;

		if(obj.constructor.name)
			return obj.constructor.name;

		var arr = obj.constructor.toString().match(
			/function\s*(\w+)/);

		if (arr && arr.length == 2) {
			return arr[1];
		}
	},

	/**
	* Returns an string with the class name
	* @method getClassName
	* @param {Object} class object
	* @return {String} returns the string with the name
	*/
	getClassName: function(obj)
	{
		if (!obj)
			return;

		//from function info, but not standard
		if(obj.name)
			return obj.name;

		//from sourcecode
		if(obj.toString) {
			var arr = obj.toString().match(
				/function\s*(\w+)/);
			if (arr && arr.length == 2) {
				return arr[1];
			}
		}
	},

	/**
	* Returns the public properties of one object and the type (not the values)
	* @method getObjectProperties
	* @param {Object} object
	* @return {Object} returns object with attribute name and its type
	*/
	//TODO: merge this with the locator stuff
	getObjectProperties: function( object )
	{
		if(object.getPropertiesInfo)
			return object.getPropertiesInfo();
		var class_object = object.constructor;
		if(class_object.properties)
			return class_object.properties;

		var o = {};
		for(var i in object)
		{
			//ignore some
			if(i[0] == "_" || i[0] == "@" || i.substr(0,6) == "jQuery") //skip vars with _ (they are private)
				continue;

			if(class_object != Object)
			{
				var hint = class_object["@"+i];
				if(hint && hint.type)
				{
					o[i] = hint.type;
					continue;
				}
			}

			var v = object[i];
			if(v == null)
				o[i] = null;
			else if ( isFunction(v) )//&& Object.getOwnPropertyDescriptor(object, i) && Object.getOwnPropertyDescriptor(object, i).get )
				continue; //o[i] = v;
			else if (  v.constructor === Boolean )
				o[i] = LS.TYPES.BOOLEAN;
			else if (  v.constructor === Number )
				o[i] = LS.TYPES.NUMBER;
			else if ( v.constructor === String )
				o[i] = LS.TYPES.STRING;
			else if ( v.buffer && v.buffer.constructor === ArrayBuffer ) //typed array
			{
				if(v.length == 2)
					o[i] = LS.TYPES.VEC2;
				else if(v.length == 3)
					o[i] = LS.TYPES.VEC3;
				else if(v.length == 4)
					o[i] = LS.TYPES.VEC4;
				else if(v.length == 9)
					o[i] = LS.TYPES.MAT3;
				else if(v.length == 16)
					o[i] = LS.TYPES.MAT4;
				else
					o[i] = 0;
			}
			else
				o[i] = 0;
		}
		return o;
	},

	//TODO: merge this with the locator stuff
	setObjectProperty: function( obj, name, value )
	{
		if(obj.setProperty)
			return obj.setProperty(name, value);
		obj[ name ] = value; //clone?
		if(obj.onPropertyChanged)
			obj.onPropertyChanged( name, value );
	},

	/**
	* Register a Material class so it is listed when searching for new materials to attach
	*
	* @method registerMaterialClass
	* @param {ComponentClass} comp component class to register
	*/
	registerMaterialClass: function( material_class )
	{ 
		var class_name = LS.getClassName( material_class );

		//register
		this.MaterialClasses[ class_name ] = material_class;
		this.Classes[ class_name ] = material_class;

		//add extra material methods
		LS.extendClass( material_class, Material );

		//event
		LEvent.trigger( LS, "materialclass_registered", material_class );
		material_class.resource_type = "Material";
		material_class.is_material = true;
	},

	/**
	* Returns an script context using the script name (not the node name), usefull to pass data between scripts.
	*
	* @method getScript
	* @param {String} name the name of the script according to the Script component.
	* @return {Object} the context of the script.
	*/
	getScript: function( name )
	{
		var script = LS.Script.active_scripts[name];
		if(script)
			return script.context;
		return null;
	},

	getDebugRender: function()
	{
		if(!LS.debug_render)
			LS.debug_render = new LS.DebugRender();
		return LS.debug_render;
	},

	//we do it in a function to make it more standard and traceable
	//used by the editor to show the error
	dispatchCodeError: function( err, line, resource, extra )
	{
		var error_info = { error: err, line: line, resource: resource, extra: extra };
		console.error(error_info);
		LEvent.trigger( this, "code_error", error_info );
	},

	//used to tell the editor it must remove the error marker
	dispatchNoErrors: function( resource, extra )
	{
		var info = { resource: resource, extra: extra };
		LEvent.trigger( this, "code_no_errors", info );
	},

	convertToString: function( data )
	{
		if(!data)
			return "";
		if(data.constructor === String)
			return data;
		if(data.constructor === Object)
			return JSON.stringify( object.serialize ? object.serialize() : object );
		if(data.constructor === ArrayBuffer)
			data = new Uint8Array(data);
		return String.fromCharCode.apply(null,data);
	},

	
	/**
	* Checks if this locator belongs to a property inside a prefab, which could be tricky in some situations
	* @method checkLocatorBelongsToPrefab
	**/
	checkLocatorBelongsToPrefab: function( locator, root )
	{
		root = root || LS.GlobalScene.root;
		var property_path = locator.split("/");

		if( !property_path.length )
			return null;

		var node = LSQ.get( property_path[0], root );
		if(!node)
			return null;

		return node.insidePrefab();
	},

	/**
	* Used to convert locators so instead of using UIDs for properties it uses Names
	* This is used when you cannot rely on the UIDs because they belong to prefabs that could change them
	* @method convertLocatorFromUIDsToName
	* @param {String} locator string with info about a property (p.e. "my_node/Transform/y")
	* @param {boolean} use_basename if you want to just use the node name, othewise it uses the fullname (name with path)
	* @param {LS.SceneNode} root
	* @return {String} the result name without UIDs
	*/
	convertLocatorFromUIDsToName: function( locator, use_basename, root )
	{
		root = root || LS.GlobalScene.root;
		var property_path = locator.split("/");

		if( !property_path.length )
			return null;

		if( property_path[0][0] !== LS._uid_prefix && ( property_path.length == 1 || property_path[1][0] !== LS._uid_prefix))
			return null; //is already using names

		var node = LSQ.get( property_path[0], root );
		if(!node)
		{
			console.warn("getIDasName: node not found in LS.GlobalScene: " + property_path[0] );
			return false;
		}

		if(!node.name)
		{
			console.warn("getIDasName: node without name?");
			return false;
		}

		//what about the component?
		if( property_path.length > 1 && property_path[1][0] == LS._uid_prefix )
		{
			var comp = LS.GlobalScene.findComponentByUId( property_path[1] );
			if(comp)
			{
				var comp_name = comp.constructor.name;
				if(comp_name == "Script" || comp_name == "ScriptFromFile")
				{
					comp_name = comp.name;
					if( comp_name == "unnamed" )
					{
						console.error("converting component UIDs to component name, but property belongs to a Script without name. You must name the script to avoid errors.");
						comp_name = comp.constructor.name;
					}
				}
				property_path[1] = comp_name;	
			}
		}

		var result = property_path.concat();
		if(use_basename)
			result[0] = node.name;
		else
			result[0] = node.fullname;
		return result.join("/");
	},

	/**
	* clears the global scene and the resources manager
	*
	* @method reset
	*/
	reset: function()
	{
		LS.GlobalScene.clear();
		LS.ResourcesManager.reset();
		LEvent.trigger( LS, "reset" );
	},

	log: function()
	{
		console.log.apply( console, arguments );
	},

	stringToValue: function( v )
	{
		var value = v;
		try
		{
			value = JSON.parse(v);
		}
		catch (err)
		{
			console.error( "Not a valid value: " + v );
		}
		return value;
	},

	isValueOfType: function( value, type )
	{
		if(value === null || value === undefined)
		{
			switch (type)
			{
				case "float": 
				case "sampler2D": 
				case "samplerCube":
				case LS.TYPES.NUMBER: 
				case LS.TYPES.VEC2: 
				case LS.TYPES.VEC3:
				case LS.TYPES.VEC4:
				case LS.TYPES.COLOR:
				case LS.TYPES.COLOR4:
				case "mat3": 
				case "mat4":
					return false;
			}
			return true;
		}

		switch (type)
		{
			//used to validate shaders
			case "float": 
			case "sampler2D": 
			case "samplerCube":
			case LS.TYPES.NUMBER: return isNumber(value);
			case LS.TYPES.VEC2: return value.length === 2;
			case LS.TYPES.VEC3: return value.length === 3;
			case LS.TYPES.VEC4: return value.length === 4;
			case LS.TYPES.COLOR: return value.length === 3;
			case LS.TYPES.COLOR4: return value.length === 4;
			case "mat3": return value.length === 9;
			case "mat4": return value.length === 16;
		}
		return true;
	},

	//solution from http://stackoverflow.com/questions/979975/how-to-get-the-value-from-the-url-parameter
	queryString: function () {
	  // This function is anonymous, is executed immediately and 
	  // the return value is assigned to QueryString!
	  var query_string = {};
	  var query = window.location.search.substring(1);
	  var vars = query.split("&");
	  for (var i=0;i<vars.length;i++) {
		var pair = vars[i].split("=");
			// If first entry with this name
		if (typeof query_string[pair[0]] === "undefined") {
		  query_string[pair[0]] = decodeURIComponent(pair[1]);
			// If second entry with this name
		} else if (typeof query_string[pair[0]] === "string") {
		  var arr = [ query_string[pair[0]],decodeURIComponent(pair[1]) ];
		  query_string[pair[0]] = arr;
			// If third or later entry with this name
		} else {
		  query_string[pair[0]].push(decodeURIComponent(pair[1]));
		}
	  } 
		return query_string;
	}(),

	downloadFile: function( filename, data, dataType )
	{
		if(!data)
		{
			console.warn("No file provided to download");
			return;
		}

		if(!dataType)
		{
			if(data.constructor === String )
				dataType = 'text/plain';
			else
				dataType = 'application/octet-stream';
		}

		var file = null;
		if(data.constructor !== File && data.constructor !== Blob)
			file = new Blob( [ data ], {type : dataType});
		else
			file = data;

		var url = URL.createObjectURL( file );
		var element = document.createElement("a");
		element.setAttribute('href', url);
		element.setAttribute('download', filename );
		element.style.display = 'none';
		document.body.appendChild(element);
		element.click();
		document.body.removeChild(element);
		setTimeout( function(){ URL.revokeObjectURL( url ); }, 1000*60 ); //wait one minute to revoke url
	}
}

//ensures no exception is catched by the system (useful for developers)
Object.defineProperty( LS, "catch_exceptions", { 
	set: function(v){ 
		this._catch_exceptions = v; 
		LScript.catch_exceptions = v; 
		LScript.catch_important_exceptions = v;
	},
	get: function() { return this._catch_exceptions; },
	enumerable: true
});

//ensures no exception is catched by the system (useful for developers)
Object.defineProperty( LS, "block_scripts", { 
	set: function(v){ 
		LS._block_scripts = v; 
		LScript.block_execution = v; 
	},
	get: function() { 
		return !!LS._block_scripts;
	},
	enumerable: true
});


//Add some classes
LS.Classes.WBin = LS.WBin = global.WBin = WBin;

/**
* LSQ allows to set or get values easily from the global scene, using short strings as identifiers
* similar to jQuery and the DOM:  LSQ("nod_name").material = ...
*
* @class  LSQ
*/
function LSQ(v)
{
	return LSQ.get(v);
}

/**
* Assigns a value to a property of one node in the scene, just by using a string identifier
* Example:  LSQ.set("mynode|a_child/MeshRenderer/enabled",false);
*
* @method set
* @param {String} locator the locator string identifying the property
* @param {*} value value to assign to property
*/
LSQ.set = function( locator, value, root, scene )
{
	scene = scene || LS.GlobalScene;
	if(!root)
		scene.setPropertyValue( locator, value );
	else
	{
		if(root.constructor === LS.SceneNode)
		{
			var path = locator.split("/");
			var node = root.findNodeByUId( path[0] );
			if(!node)
				return null;
			return node.setPropertyValueFromPath( path.slice(1), value );
		}
	}

	scene.requestFrame();
}

/**
* Retrieves the value of a property of one node in the scene, just by using a string identifier
* Example: var value = LSQ.get("mynode|a_child/MeshRenderer/enabled");
*
* @method get
* @param {String} locator the locator string identifying the property
* @return {*} value of the property
*/
LSQ.get = function( locator, root, scene )
{
	if(!locator) //sometimes we have a var with a locator that is null
		return null;
	scene = scene || LS.GlobalScene;
	var info;
	if(!root)
		info = scene.getPropertyInfo( locator );
	else
	{
		if(root.constructor === LS.SceneNode)
		{
			var path = locator.split("/");
			var node = root.findNodeByUId( path[0] );
			if(!node)
				return null;
			info = node.getPropertyInfoFromPath( path.slice(1) );
		}
	}
	if(info)
		return info.value;
	return null;
}

/**
* Shortens a locator that uses unique identifiers to a simpler one, but be careful, because it uses names instead of UIDs it could point to the wrong property
* Example: "@NODE--a40661-1e8a33-1f05e42-56/@COMP--a40661-1e8a34-1209e28-57/size" -> "node|child/Collider/size"
*
* @method shortify
* @param {String} locator the locator string to shortify
* @return {String} the locator using names instead of UIDs
*/
LSQ.shortify = function( locator, scene )
{
	if(!locator)
		return;

	var t = locator.split("/");
	var node = null;

	//already short
	if( t[0][0] != LS._uid_prefix )
		return locator;

	scene = scene || LS.GlobalScene;

	node = scene._nodes_by_uid[ t[0] ];
	if(!node) //node not found
		return locator;

	t[0] = node.getPathName();
	if(t[1])
	{
		if( t[1][0] == LS._uid_prefix )
		{
			var compo = node.getComponentByUId(t[1]);
			if(compo)
				t[1] = LS.getObjectClassName( compo );
		}
	}
	return t.join("/");
}

/**
* Assigns a value using the getLocatorInfo object instead of searching it again
* This is faster but if the locator points to a different object it wont work.
*
* @method setFromInfo
* @param {Object} info information of a location (obtain using scene.getLocatorInfo
* @param {*} value to assign
*/
LSQ.setFromInfo = function( info, value )
{
	if(!info || !info.target)
		return;
	var target = info.target;
	if( target.setPropertyValue  )
		if( target.setPropertyValue( info.name, value ) === true )
			return target;
	if( target[ info.name ] === undefined && info.value === undefined )
		return;
	target[ info.name ] = value;	
}

LSQ.getFromInfo = function( info )
{
	if(!info || !info.target)
		return;
	var target = info.target;
	var varname = info.name;
	var v = undefined;
	if( target.getPropertyValue )
		v = target.getPropertyValue( varname );
	if( v === undefined && target[ varname ] === undefined )
		return null;
	return v !== undefined ? v : target[ varname ];
}

//register resource classes
if(global.GL)
{
	GL.Mesh.EXTENSION = "wbin";
	GL.Texture.EXTENSION = "png";

	LS.registerResourceClass( GL.Mesh );
	LS.registerResourceClass( GL.Texture );

	LS.Mesh = GL.Mesh;
	LS.Texture = GL.Texture;
	LS.Buffer = GL.Buffer;
	//LS.Shader = GL.Shader; //this could be confussing since there is also ShaderBlocks etc in LiteScene
}


global.LSQ = LSQ;


///@FILE:../src/defines.js
///@INFO: BASE
if(typeof(GL) == "undefined")
	throw("LiteSCENE requires to have litegl.js included before litescene.js");

//blending mode
var Blend = {
	AUTOMATIC: 1,
	NORMAL: 2,
	ALPHA: 3,
	ADD: 4,
	MULTIPLY: 5,
	SCREEN: 6,
	CUSTOM: 7
}

LS.Blend = Blend;

LS.BlendFunctions = {};

LS.BlendFunctions[ Blend.AUTOMATIC ] = [GL.ONE, GL.ZERO];
LS.BlendFunctions[ Blend.NORMAL ] = [GL.ONE, GL.ZERO];
LS.BlendFunctions[ Blend.ALPHA ] = [GL.SRC_ALPHA, GL.ONE_MINUS_SRC_ALPHA];
LS.BlendFunctions[ Blend.ADD ] = [GL.SRC_ALPHA, GL.ONE];
LS.BlendFunctions[ Blend.MULTIPLY ] = [GL.DST_COLOR, GL.ONE_MINUS_SRC_ALPHA];
LS.BlendFunctions[ Blend.SCREEN ] =	[GL.SRC_ALPHA, GL.ONE];
LS.BlendFunctions[ Blend.CUSTOM ] =	[GL.SRC_ALPHA, GL.ONE_MINUS_SRC_ALPHA];

//Used for interpolation and splines
LS.NONE = 0;
LS.LINEAR = 1;
LS.TRIGONOMETRIC = 2;
LS.CUBIC = 3;
LS.SPLINE = 4;
LS.BEZIER = 5;
LS.HERMITE = 6;

//used to know the state of the application
LS.STOPPED = 0;
LS.PLAYING = 1; 
LS.PAUSED = 2;
LS.LOADING = 3;

LS.RUNNING = 1; //LEGACY

//helpful consts
LS.ZEROS = vec3.create();
LS.ZEROS4 = vec4.create();
LS.ONES = vec3.fromValues(1,1,1);
LS.ONES4 = vec4.fromValues(1,1,1,1);
LS.TOP = vec3.fromValues(0,1,0);
LS.BOTTOM = vec3.fromValues(0,-1,0);
LS.RIGHT = vec3.fromValues(1,0,0);
LS.LEFT = vec3.fromValues(-1,0,0);
LS.FRONT = vec3.fromValues(0,0,-1);
LS.BACK = vec3.fromValues(0,0,1);
LS.IDENTITY = mat4.create();
LS.QUAT_IDENTITY = quat.create();
LS.WHITE = LS.ONES;
LS.BLACK = LS.ZEROS;

LS.POSX = 1;
LS.POSY = 2;
LS.POSZ = 3;
LS.NEGX = 4;
LS.NEGY = 5;
LS.NEGZ = 6;

//types
LS.TYPES = {
	BOOLEAN: "boolean",
	NUMBER : "number",
	STRING : "string",
	VEC2 : "vec2",
	VEC3 : "vec3",
	VEC4 : "vec4",
	COLOR : "color",
	COLOR4 : "color4",
	EVENT : "event",
	RESOURCE: "resource",
	TEXTURE : "texture",
	MESH: "mesh",
	OBJECT: "object",
	SCENE: "scene",
	NODE: "node",
	SCENENODE: "node",
	SCENENODE_ID: "node_id",
	COMPONENT: "component",
	COMPONENT_ID: "component_id",
	MATERIAL: "material",
	ANIMATION: "animation",
	ARRAY: "array",
	QUAT : "quat",
	TRANS10 : "trans10",
	POSITION : "position"
};

LS.TYPES_INDEX = {};
var index = 0;
for(var i in LS.TYPES)
{
	LS.TYPES_INDEX[ LS.TYPES[i] ] = index;
	LS.TYPES_INDEX[ LS.TYPES[i].toUpperCase() ] = index;
	index++
}

LS.RESOURCE_TYPES = {};
LS.RESOURCE_TYPES[ LS.TYPES.RESOURCE ] = true;
LS.RESOURCE_TYPES[ LS.TYPES.TEXTURE ] = true;
LS.RESOURCE_TYPES[ LS.TYPES.MESH ] = true;
LS.RESOURCE_TYPES[ LS.TYPES.ANIMATION ] = true;
//audio and video?


//Events
var EVENT = LS.EVENT = {};
//events are defined in the file that triggers them:
//- renderer.js: render related events (RENDER_INSTANCES,etc)
//- scene.js: scene related (INIT,START,UPDATE)
//- input.js: player related (MOUSEDOWN,KEYDOWN)


/**
* A Ray that contains an origin and a direction (it uses the Ray class from litegl, so to check documentation go to litegl doc
* @class Ray
* @namespace LS
* @constructor
* @param {vec3} origin
* @param {vec3} direction
*/
LS.Ray = GL.Ray;

///@FILE:../src/network.js
///@INFO: BASE
var Network = {

	default_dataType: "arraybuffer",
	protocol: null,

	withCredentials: false, //for CORS urls: not sure which one is the best for every case so I leave it configurable

	/**
	* A front-end for XMLHttpRequest so it is simpler and more cross-platform
	*
	* @method request
	* @param {Object} request object with the fields for the request: 
    *			dataType: result type {text,xml,json,binary,arraybuffer,image},
				data: object with form fields,
				method: "POST","GET","DELETE","PUT", if omited if will use post or get depending on the parameters,
				mimeType: to overwrite,
				use_proxy: if true it will use LiteScene proxy if available
				callbacks supported: {success, error, progress}
	* @return {XMLHttpRequest} the XMLHttpRequest of the petition
	*/
	request: function(request)
	{
		if(typeof(request) === "string")
			throw("LS.Network.request expects object, not string. Use LS.Network.requestText or LS.Network.requestJSON");

		//change protocol when working over https
		var url = request.url;

		//apply proxy
		if(request.use_proxy)
			url = LS.ResourcesManager.getFullURL(url);

		if( this.protocol === null )
			this.protocol = LS.ResourcesManager.getProtocol( location.href );
		var protocol = LS.ResourcesManager.getProtocol( url );
		if( this.protocol == "https" && protocol && protocol != "https" )
			url = "https" + url.substr( url.indexOf(":") );

		//update dataType
		var dataType = request.dataType || this.default_dataType;
		if(dataType == "json") //parse it locally
			dataType = "text";
		else if(dataType == "xml") //parse it locally
			dataType = "text";
		else if (dataType == "binary")
		{
			//request.mimeType = "text/plain; charset=x-user-defined";
			dataType = "arraybuffer";
			request.mimeType = "application/octet-stream";
		}	
		else if(dataType == "image") //special case: images are loaded using regular images request
		{
			var img = new Image();
			img.onload = function() {
				if(request.success)
					request.success.call(this);
			};
			img.onerror = request.error;
			img.src = url;
			return img;
		}

		//regular case, use AJAX call
        var xhr = new XMLHttpRequest();
        xhr.open( request.method || (request.data ? 'POST' : 'GET'), url, true);
		xhr.withCredentials = this.withCredentials; //if true doesnt work
		if(request.withCredentials !== undefined)
			xhr.withCredentials = request.withCredentials;
        if(dataType)
            xhr.responseType = dataType;
        if (request.mimeType)
            xhr.overrideMimeType( request.mimeType );
		if(request.nocache)
			xhr.setRequestHeader('Cache-Control', 'no-cache');
        xhr.onload = function(load)
		{
			var response = this.response;
			if(this.status && this.status != 200) //status 0 is when working with local files
			{
				var err = "Error " + this.status;
				if(request.error)
					request.error(err);
				return;
			}
	
			//parse input
			if(request.dataType == "json") //chrome doesnt support json format
			{
				try
				{
					response = JSON.parse(response);
				}
				catch (err)
				{
					if(request.error)
						request.error(err);
				}
			}
			else if(request.dataType == "xml")
			{
				try
				{
					var xmlparser = new DOMParser();
					response = xmlparser.parseFromString(response,"text/xml");
				}
				catch (err)
				{
					if(request.error)
						request.error(err);
				}
			}
			else if(request.dataType == "blob")
			{
				response.name = LS.ResourcesManager.getFilename(url);
			}

			//call callback
			if(LS.catch_errors)
			{
				try
				{
					if(request.success)
						request.success.call(this, response, request.url );
					LEvent.trigger(xhr,"done",response);
				}
				catch (err)
				{
					LEvent.trigger(LS,"code_error",err);
				}
			}
			else
			{
				if(request.success)
					request.success.call(this, response, request.url );
				LEvent.trigger(xhr,"done",response);
			}
		};
        xhr.onerror = function(err) {
			if(request.error)
				request.error(err);
			LEvent.trigger(this,"fail", err);
		}

		if( request.uploadProgress )
		{
			xhr.upload.addEventListener("progress", function(e){
				var progress = 0;
				if (e.lengthComputable)
					progress = e.loaded / e.total;
				request.uploadProgress( e, progress );
			}, false);
		}

		if( request.progress )
			xhr.addEventListener( "progress", function(e){
				var progress = 0;
				if (e.lengthComputable)
					progress = e.loaded / e.total;
				request.progress( e, progress );
			});

        xhr.send(request.data);

		return xhr;
	},

	/**
	* retrieve a text file from url (you can bind LEvents to done and fail)
	* @method requestText
	* @param {string} url
	* @param {object} params form params
	* @param {function} callback( data )
	*/
	requestText: function( url, data, callback, callback_error )
	{
		if(typeof(data) == "function")
		{
			callback_error = callback;
			callback = data;
			data = null;
		}
		return LS.Network.request({url:url, dataType:"text", success: callback, error: callback_error});
	},

	/**
	* retrieve a JSON file from url (you can bind LEvents to done and fail)
	* @method requestJSON
	* @param {string} url
	* @param {object} params form params
	* @param {function} callback( json )
	*/
	requestJSON: function( url, data, callback, callback_error )
	{
		if(typeof(data) == "function")
		{
			callback_error = callback;
			callback = data;
			data = null;
		}
		return LS.Network.request({url:url, data:data, dataType:"json", success: callback, error: callback_error });
	},

	/**
	* retrieve a file from url (you can bind LEvents to done and fail) as a ArrayBuffer or Blob
	* @method requestFile
	* @param {string} url
	* @param {object} params form params
	* @param {function} callback( file )
	*/
	requestFile: function( url, form_data, callback, callback_error, as_blob )
	{
		if(typeof(form_data) == "function")
		{
			callback_error = callback;
			callback = form_data;
			form_data = null;
		}
		return LS.Network.request({ url:url, dataType: as_blob ? "blob" : "binary", data: form_data, success: callback, error: callback_error });
	},

	/**
	* Request script and inserts it in the DOM
	* @method requestScript
	* @param {String} url could be an array with urls to load in order
	* @param {Function} on_complete
	* @param {Function} on_error
	* @param {Function} on_progress (if several files are required, on_progress is called after every file is added to the DOM)
	**/
	requestScript: function( url, on_complete, on_error, on_progress )
	{
		if( !url )
			throw("No url");

		if( LS._block_scripts )
		{
			console.error("Safety: LS.block_scripts enabled, cannot request script");
			return;
		}

		if( url.constructor === String )
			url = [url];

		var total = url.length;
		var size = total;
		for( var i in url )
		{
			var script = document.createElement('script');
			script.num = i;
			script.type = 'text/javascript';
			var full_url = url[i].trim();
			if(full_url.substr(0,5) != "blob:")
				full_url += "?" + LS.RM.getNoCache(true);
			script.src = full_url;
			script.async = false;
			//if( script.src.substr(0,5) == "blob:") //local scripts could contain utf-8
				script.charset = "UTF-8";
			script.onload = inner_script_loaded;
			if(on_error)
				script.onerror = function(err) { 
					on_error(err, this.src, this.num );
				}
			document.getElementsByTagName('head')[0].appendChild( script );
		}

		function inner_script_loaded(e) { 
			total--;
			if(total)
			{
				if(on_progress)
					on_progress(this.src, this.num);
			}
			else 
				inner_check_pending();
		};

		function inner_check_pending()
		{
			if( LS.Network.pending_scripts.length ) //scripts included from scripts
				setTimeout(inner_check_pending,1000); //wait one second
			else if(on_complete)
				on_complete();
		}
	},

	//used to import scripts from scripts
	pending_scripts: [],

	importScript: function( url, on_complete, on_error )
	{
		var script = document.createElement('script');
		script.type = 'text/javascript';
		script.src = url;
		script.async = false;
		script.charset = "UTF-8";
		script.onload = function(e) { 
			var index = LS.Network.pending_scripts.indexOf( this );
			if(index != -1)
				LS.Network.pending_scripts.splice(index);
			if(on_complete)
				on_complete();
		};
		script.onerror = function(err) { 
			var index = LS.Network.pending_scripts.indexOf( this );
			if(index != -1)
				LS.Network.pending_scripts.splice(index);
			if(on_error)
				on_error();
		}
		this.pending_scripts.push( script );
		document.getElementsByTagName('head')[0].appendChild( script );
		return script;
	},

	requestFont: function( name, url )
	{
		if(!name || !url)
			throw("LS.Network.requestFont: Wrong font name or url");

		var fonts = this._loaded_fonts;
		if(!fonts)
			fonts = this._loaded_fonts = {};

		if(fonts[name] == url)
			return;
		fonts[name] = url;

		var style = document.getElementById("ls_fonts");
		if(!style)
		{
			style = document.createElement("style");
			style.id = "ls_fonts";
			style.setAttribute("type","text/css");
			document.head.appendChild(style);
		}
		var str = "";
		for(var i in fonts)
		{
			var url = fonts[i];
			str += "@font-face {\n" +
					"\tfont-family: \""+i+"\";\n" + 
					"\tsrc: url('"+url+"');\n" + 
			"}\n";
		}
		style.innerHTML = str;
	}

	//NOT TESTED: to load script asyncronously, not finished. similar to require.js
	/*
	requireScript: function(files, on_complete)
	{
		if(typeof(files) == "string")
			files = [files];

		//store for the callback
		var last = files[ files.length - 1];
		if(on_complete)
		{
			if(!ResourcesManager._waiting_callbacks[ last ])
				ResourcesManager._waiting_callbacks[ last ] = [on_complete];
			else
				ResourcesManager._waiting_callbacks[ last ].push(on_complete);
		}
		require_file(files);

		function require_file(files)
		{
			//avoid require twice a file
			var url = files.shift(1); 
			while( ResourcesManager._required_files[url] && url )
				url = files.shift(1);

			ResourcesManager._required_files[url] = true;

			LS.Network.request({
				url: url,
				success: function(response)
				{
					eval(response);
					if( ResourcesManager._waiting_callbacks[ url ] )
						for(var i in ResourcesManager._waiting_callbacks[ url ])
							ResourcesManager._waiting_callbacks[ url ][i]();
					require_file(files);
				}
			});
		}
	},
	_required_files: {},
	_waiting_callbacks: {}
	*/
};

LS.Network = Network;
///@FILE:../src/input.js
///@INFO: BASE
/**
* Input is a static class used to read the input state (keyboard, mouse, gamepad, etc)
*
* @class Input
* @namespace LS
* @constructor
*/

//help info:
//mouse.mousey 0 is top
//mouse.canvasy 0 is bottom
//mouse.y is mousey

EVENT.MOUSEDOWN = "mousedown";
EVENT.MOUSEMOVE = "mousemove";
EVENT.MOUSEUP = "mouseup";
EVENT.MOUSEWHEEL = "mousewheel";
EVENT.TOUCHSTART = "touchstart";
EVENT.TOUCHMOVE = "touchmove";
EVENT.TOUCHEND = "touchend";
EVENT.KEYDOWN = "keydown";
EVENT.KEYUP = "keyup";

var Input = {
	mapping: {

		//xbox
		A_BUTTON: 0,
		B_BUTTON: 1,
		X_BUTTON: 2,
		Y_BUTTON: 3,
		LB_BUTTON: 4,
		RB_BUTTON: 5,
		BACK_BUTTON: 6,
		START_BUTTON: 7,
		LS_BUTTON: 8,
		RS_BUTTON: 9,

		LX: 0,
		LY: 1,
		RX: 2,
		RY: 3,
		TRIGGERS: 4,
		LEFT_TRIGGER: 4,
		RIGHT_TRIGGER: 5,

		//generic
		JUMP:0,
		FIRE:1,

		//mouse
		LEFT:0,
		MIDDLE:1,
		RIGHT:2
	},

	//according to the specification, read from which
	LEFT_MOUSE_BUTTON: 0,
	MIDDLE_MOUSE_BUTTON: 1,
	RIGHT_MOUSE_BUTTON: 2,

	//MouseEvent.buttons use a different identifier than e.which...
	BUTTONS_LEFT: 1,
	//BUTTONS_RIGHT: 2,
	//BUTTONS_MIDDLE: 3,

	Keyboard: null, //gl.keys
	Keyboard_previous: {},

	Mouse: {},
	Gamepads: [],

	//used for GUI elements
	last_mouse: null,
	last_click: null,
	current_click: null,
	current_key: null,
	keys_buffer: [], //array of keys that have been pressed from the last frame

	//_mouse_event_offset: [0,0],
	_last_frame: -1, //internal

	init: function()
	{
		this.Keyboard = gl.keys;
		this.Mouse = gl.mouse;
		this.Gamepads = gl.getGamepads();
	},

	reset: function()
	{
		this.Gamepads = gl.gamepads = []; //force reset so they send new events 
	},

	update: function()
	{
		//copy prev keys state
		for(var i in this.Keyboard_previous) //first reset
			this.Keyboard_previous[i] = false;
		for(var i in this.Keyboard) //then copy
			this.Keyboard_previous[i] = this.Keyboard[i];

		//copy prev mouse state (this is only necessary if the update is not called from litegl main loop)
		this.Mouse.last_buttons = this.Mouse.buttons;

		//capture gamepads snapshot
		this.Gamepads = gl.getGamepads();
	},

	/**
	* returns true is the key is pressed now
	*
	* @method isKeyPressed
	* @param {Number} key_code
	* @return {boolean}
	*/
	isKeyPressed: function(key_code)
	{
		return !!this.Keyboard[ key_code ];
	},

	/**
	* returns true is the key was pressed between previous frame and now
	*
	* @method wasKeyPressed
	* @param {Number} key_code as in https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/keyCode#Browser_compatibility
	* @return {boolean}
	*/
	wasKeyPressed: function(key_code)
	{
		return this.Keyboard[ key_code ] && !this.Keyboard_previous[ key_code ];
	},

	/**
	* returns true is the mouse button is pressed now
	*
	* @method isMouseButtonPressed
	* @param {Number} button could be "left","middle","right" or GL.LEFT_MOUSE_BUTTON, GL.MIDDLE_MOUSE_BUTTON, GL.RIGHT_MOUSE_BUTTON
	* @return {boolean}
	*/
	isMouseButtonPressed: function(button)
	{
		var num = 0;
		if(button && button.constructor === String)
			num = this.mapping[button];
		else
			num = button;
		if(button === undefined)
			return false;

		return this.Mouse.isButtonPressed(num);
	},

	/**
	* returns true is the mouse button was pressed between previous frame and now
	*
	* @method wasMouseButtonPressed
	* @param {Number} button could be "left","middle","right" or GL.LEFT_MOUSE_BUTTON, GL.MIDDLE_MOUSE_BUTTON, GL.RIGHT_MOUSE_BUTTON
	* @return {boolean}
	*/
	wasMouseButtonPressed: function(button)
	{
		var num = 0;
		if(button && button.constructor === String)
			num = this.mapping[button];
		else
			num = button;
		if(button === undefined)
			return false;

		return this.Mouse.wasButtonPressed(num);
	},

	/**
	* locks the mouse (to use with first person view cameras) so when the mouse moves, the cameras moves
	*
	* @method lockMouse
	* @param {Boolean} v if true, the camera is locked, otherwise unlocked
	* @return {boolean}
	*/
	lockMouse: function(v)
	{
		if(v)
		{
			gl.canvas.requestPointerLock();

		}
		else
			document.exitPointerLock();
	},

	/**
	* returns true is mouse is in pointer lock mode
	*
	* @method isMouseLocked
	* @return {boolean}
	*/
	isMouseLocked: function()
	{
		return !!document.pointerLockElement;
	},

	//called from LS.Player when onmouse (or from RenderModule in editor)
	//returns true if the event was blocked
	onMouse: function(e)
	{
		this.last_mouse = e;

		if( this.isMouseLocked() )
		{
			e.canvasx = e.mousex = (gl.canvas.width * 0.5)|0;
			e.canvasy = e.mousey = (gl.canvas.height * 0.5)|0;
		}

		//mousey is from top
		this.Mouse.x = this.Mouse.mousex = e.mousex;
		this.Mouse.y = this.Mouse.mousey = e.mousey;
		//canvasy is from bottom
		this.Mouse.canvasx = e.canvasx;
		this.Mouse.canvasy = e.canvasy;

		//save it in case we need to know where was the last click
		if(e.type == "mousedown")
		{
			if(e.button == 0)
				this.last_click = e;
			this.current_click = e;
			LS.triggerCoroutines( "click", e );
		}
		else if(e.type == "mouseup")
		{
			//if(e.button == 0) //last click is the last position clicked
			//	this.last_click = null;
			this.current_click = null;
		}

		//we test if this event should be sent to the components or it was blocked by the GUI
		return LS.GUI.testEventInBlockedArea(e);
	},

	//called from LS.Player when onkey
	onKey: function(e)
	{
		//Info: this.Keyboard is not updated here, litegl already does it and stores it in gl.keys which is the same object as LS.Input.Keyboard

		if(e.type == "keydown")
		{
			this.current_key = e;
			if( LS.Renderer._frame != this._last_frame )
			{
				this.keys_buffer.length = 0;
				LS.Renderer._frame = this._last_frame;
			}
			if( this.keys_buffer.length < 10 ) //safety first!
				this.keys_buffer.push(e);
		}
		else
		{
			this.current_key = null;
		}
	},

	/**
	* returns if the mouse is inside the rect defined by x,y, width,height
	*
	* @method isMouseInRect
	* @param {Number} x x coordinate of the mouse in canvas coordinates 
	* @param {Number} y y coordinate of the mouse in canvas coordinates (0 is bottom)
	* @param {Number} width rectangle width in pixels
	* @param {Number} height rectangle height in pixels
	* @param {boolean} flip [optional] if you want to flip the y coordinate
	* @return {boolean}
	*/
	isMouseInRect: function( x, y, width, height, flip_y )
	{
		return this.Mouse.isInsideRect(x,y,width,height,flip_y);
	},

	//uses {x,y}, instead of mousex,mousey
	isEventInRect: function( mouse, area, offset )
	{
		var x = mouse.mousex != null ? mouse.mousex : mouse.x;
		var y = mouse.mousey != null ? mouse.mousey : mouse.y;
		if(offset)
		{
			x -= offset[0];
			y -= offset[1];
		}
		return ( x >= area[0] && x < (area[0] + area[2]) && y >= area[1] && y < (area[1] + area[3]) );
	},

	/**
	* Returns the axis based on the gamepad or the keyboard cursors. Useful when you do now know if the player will use keyboard of gamepad
	*
	* @method getAxis
	* @param {String} "vertical" or "horizontal"
	* @return {Number} the value of the axis
	*/
	getAxis: function( axis )
	{
		if( axis == "vertical" )
		{
			if( this.isKeyPressed( 38 )	|| this.isKeyPressed( "W" )) //up
				return 1;
			if( this.isKeyPressed( 40 )	|| this.isKeyPressed( "S" )) //down
				return -1;
		}
		else if( axis == "horizontal" )
		{
			if( this.isKeyPressed( 37 )	|| this.isKeyPressed( "A" )) //left
				return -1;
			else if( this.isKeyPressed( 39 ) || this.isKeyPressed( "D" )) //right
				return 1;
		}

		var gamepad = this.Gamepads[0];
		if(gamepad)
		{
			if(axis == "horizontal")
				return gamepad.axes[0];
			else if(axis == "vertical")
				return gamepad.axes[1];
		}

		return 0;
	},

	/**
	* Returns a gamepad snapshot if it is connected
	*
	* @method getGamepad
	* @param {Number} index the index of the gamepad
	* @return {Object} gamepad snapshot with all the info
	*/
	getGamepad: function(index)
	{
		index = index || 0;
		return this.Gamepads[index];
	},

	/**
	* Returns a gamepad snapshot if it is connected
	*
	* @method getGamepadAxis
	* @param {Number} index the index of the gamepad
	* @param {String} name the name of the axis (also you could specify the number)
	* @param {boolean} raw [optional] if you want the data unfiltered
	* @return {Number} axis value from -1 to 1
	*/
	getGamepadAxis: function(index, name, raw)
	{
		var gamepad = this.Gamepads[index];
		if(!gamepad)
			return 0;

		var num = 0;
		if(name && name.constructor === String)
			num = this.mapping[name];
		else
			num = name;
		if(num === undefined)
			return 0;
		var v = gamepad.axes[num];
		if(!raw && v > -0.1 && v < 0.1 ) //filter
			return 0;
		return v;
	},

	/**
	* Returns if the given button of the specified gamepad is pressed
	*
	* @method isGamepadButtonPressed
	* @param {Number} index the index of the gamepad
	* @param {String} name the name of the button "A","B","X","Y","LB","RB","BACK","START","LS","RS" (also you could specify the number)
	* @return {Boolean} if the button is pressed
	*/
	isGamepadButtonPressed: function(input, name)
	{
		var gamepad = this.Gamepads[input];
		if(!gamepad)
			return null;

		var num = 0;
		if(name && name.constructor === String)
			num = this.mapping[name];
		else
			num = name;
		if(num === undefined)
			return 0;
		var button = gamepad.buttons[num];
		return button && button.pressed;
	},

	/**
	* Returns a Promise that will be fulfilled when the user clicks the screen
	* @method mouseClick
	* @return {Promise} 
	*/
	mouseClick: function()
	{
		return new Promise(function(resolve){
			LS.addWaitingCoroutine( resolve, "click" );
		});
	}
};


Object.defineProperty( MouseEvent.prototype, "getRay", { value: function(){
		//get camera under position
		var camera = LS.Renderer.getCameraAtPosition( this.mousex, this.mousey, LS.Renderer._visible_cameras );
		if(!camera)
			return null;
		//get ray
		return camera.getRay( this.mousex, this.mousey );
	},
	enumerable: false 
});

LS.Input = Input;
///@FILE:../src/gui.js

/**
* GUI is a static class used to create two kinds of GUIs: HTML GUIs on top of the 3D Canvas (in a safe way) or Immediate GUI using a Canvas2D (fast gui)
* For HTML GUIs check the getHTMLRoot function.
* For Immediate GUIs check the Box,Button,Toggle,Textfield,HorizontalSlider,VerticalSlider and Toolbar.
* To change colors of the immediate GUI check the LS.GUI.GUIStyle
*
* @class GUI
* @namespace LS
* @constructor
*/
var GUI = {

	_root: null, //root DOM element containing the GUI
	_allow_change_cursor: true,
	_is_on_top_of_immediate_widget: false,

	defaultGUIStyle: {
		font: "Arial",
		color: "#FFF",
		colorTextOver: "#FFF",
		backgroundColor: "#333",
		backgroundColorOver: "#AAA",
		selected: "#AAF",
		unselected: "#AAA",
		outline: "#000",
		margin: 0.2
	},

	GUIStyle: null,
	_style_stack: [],

	_offset: [0,0],

	_gui_areas: {
		data: new Float32Array(1024),
		offset: 0
	},

	_ctx: null, //

	pressed_enter: false,

	/**
	* Returns the DOM element responsible for the HTML GUI of the app. This is helpful because this GUI will be automatically removed if the app finishes.
	* Any HTML must be attached to this element, otherwise it may have problems with the editor.
	*
	* @method getHTMLRoot
	* @return {HTMLElement} 
	*/
	getHTMLRoot: function()
	{
		if( this._root )
		{
			if(!this._root.parentNode && gl.canvas.parentNode)
				gl.canvas.parentNode.appendChild( gui );
			return this._root;
		}

		if(LS.GlobalScene._state != LS.PLAYING)
			console.warn("GUI element created before the scene is playing will be deleted once the app starts. Only create the GUI elements from onStart or after, otherwise the GUI elements will be lost.");

		var gui = document.createElement("div");
		gui.className = "litescene-gui";
		gui.style.position = "absolute";
		gui.style.top = "0";
		gui.style.left = "0";

		//normalize
		gui.style.color = "#999";
		gui.style.font = "20px Arial";

		//make it fullsize
		gui.style.width = "100%";
		gui.style.height = "100%";
		gui.style.overflow = "hidden";
		gui.style.pointerEvents = "none";

		if(!this._style)
		{
			var style = this._style = document.createElement("style");
			style.appendChild(document.createTextNode(""));
			document.head.appendChild(style);
			style.sheet.insertRule(".litescene-gui button, .litescene-gui input { pointer-events: auto; }",0);
		}

		//append on top of the canvas
		gl.canvas.parentNode.appendChild( gui );
		
		this._root = gui;
		return gui;
	},

	/**
	* Creates a HTMLElement of the tag_type and adds it to the DOM on top of the canvas
	*
	* @method createElement
	* @param {String} tag_type the tag type "div"
	* @param {String} anchor "top-left", "top-right", "bottom-left", "bottom-right" or "none"
	* @return {HTMLElement} 
	*/
	createElement: function( tag_type, anchor )
	{
		tag_type = tag_type || "div";

		var element = document.createElement(tag_type);
		element.style.pointerEvents = "auto";
		return this.attach( element, anchor );
	},

	/**
	* attach HTMLElement to HTML GUI Root in the anchor position specified
	*
	* @method attach
	* @param {HTMLElement} element
	* @param {String} anchor "top-left", "top-right", "bottom-left", "bottom-right" or "none"
	*/
	attach: function( element, anchor )
	{
		if(!element)
		{
			console.error("attachToGUI: element cannot be null");
			return;
		}

		element.style.position = "absolute";

		anchor = anchor || "none"; //"top-left";

		switch(anchor)
		{
			case "bottom":
			case "bottom-left":
				element.style.bottom = "0";
				element.style.left = "0";
				break;
			case "bottom-right":
				element.style.bottom = "0";
				element.style.right = "0";
				break;
			case "bottom-middle":
				element.style.bottom = "0";
				element.style.width = "50%";
				element.style.margin = "0 auto";
				break;
			case "right":
			case "top-right":
				element.style.top = "0";
				element.style.right = "0";
				break;
			case "top-middle":
				element.style.top = "0";
				element.style.width = "50%";
				element.style.margin = "0 auto";
				break;
			case "left":
			case "top":
			case "top-left":
				element.style.top = "0";
				element.style.left = "0";
				break;
			case "none": break;
			default:
				console.warn("invalid GUI anchor position: ",anchor);
		}

		var gui_root = this.getHTMLRoot();
		gui_root.appendChild( element );
		return element;
	},

	/**
	* Removes an element from the GUI (same as  element.parentNode.removeChild( element ); )
	*
	* @method detach
	* @param {HTMLElement} element HTML element to detach from the GUI
	*/
	detach: function( element )
	{
		if(element && element.parentNode )
			element.parentNode.removeChild( element );
	},

	/**
	* Removes all the GUI elements from the DOM
	*
	* @method reset
	*/
	reset: function()
	{
		if( !this._root )
			return;

		if(this._root.parentNode)
			this._root.parentNode.removeChild( this._root );
		this._root = null;

		if(this._style)
		{
			this._style.parentNode.removeChild( this._style );
			this._style = null;		
		}
		return;
	},

	/**
	* shows the HTML GUI 
	*
	* @method showHTML
	*/
	showHTML: function()
	{
		if(!this._root)
			return;
		this._root.style.display = "";

	},

	/**
	* hides the HTML GUI (but it is still existing) 
	*
	* @method hideHTML
	*/
	hideHTML: function()
	{
		if(!this._root)
			return;
		this._root.style.display = "none";
	},

	/**
	* Loads resource containing the HTML code for the GUI and attachs it inside a div to the hud
	*
	* @method loadHTML
	* @param {String} url the url of the resource containing all the HTML code
	* @param {Function} on_complete callback that will be called once the HTML has been loaded and attached to the doom, it receives the HTMLElement containing all the HTML
	*/
	loadHTML: function( url, on_complete )
	{
		LS.ResourcesManager.load( url, function(res){
			var gui_root = LS.GUI.getHTMLRoot();
			var html = res.getAsHTML();
			if(!html)
			{
				console.error("html resource is not a string");
				return;
			}
			html.style.pointerEvents = "none";
			html.style.width = "100%";
			html.style.height = "100%";
			gui_root.appendChild( html );

			LS.GUI.replaceHTMLSources( gui_root );

			if(on_complete)
				on_complete( html, res );
		});
	},

	//WIP: allows to use resources 
	replaceHTMLSources: function(root)
	{
		//fetch all the tags with a src attribute
		var elements = root.querySelectorAll("*[src]");
		for(var i = 0; i < elements.length; ++i)
		{
			var element = elements[i];
			var src = element.getAttribute("src");

			//check if the src contains a @
			if(!src || src[0] != "@" )
				continue;

			src = src.substr(1);
			//replace that with a local URL to that resource in case is loaded
			var resource = LS.ResourcesManager.getResource( src );
			if( resource && resource._local_url )
				src = resource._local_url;
			else
				src = LS.ResourcesManager.getFullURL( src );
			element.setAttribute("src", src );
		}

	},

	//IMMEDIATE GUI STUFF

	/**
	* Called by the LS.Renderer to clear inmediate stuff
	*
	* @method ResetImmediateGUI
	*/
	ResetImmediateGUI: function( skip_redraw )
	{
		this._is_on_top_of_immediate_widget = false;
		this.setCursor(null);
		this.pressed_enter = false;
		this._offset[0] = 0;
		this._offset[1] = 0;
		this._gui_areas.offset = 0;
		this._ctx = gl;
		this.GUIStyle = this.defaultGUIStyle;
		this._style_stack.length = 0;
		if(!skip_redraw)
			LS.GlobalScene.requestFrame(); //force redraws
	},

	//this is done so when clicking in the area where there is an immediate GUI widget the events are not send to the app
	blockEventArea: function( area )
	{
		var data = this._gui_areas.data;
		var offset = this._gui_areas.offset;

		if(offset > data.length)
			return; //too many guis?

		data[ offset ] = area[0] + this._offset[0];
		data[ offset + 1] = area[1] + this._offset[1];
		data[ offset + 2] = area[2];
		data[ offset + 3] = area[3];
		this._gui_areas.offset += 4;

		//double the size (weird situation)
		if( this._gui_areas.offset >= data.length && data.length < 1024*24 )
		{
			this._gui_areas.data = new Float32Array( data.length * 2 );
			this._gui_areas.data.set(data);
		}
	},

	testEventInBlockedArea: function( e )
	{
		if(e.type != "mousedown")
			return false;

		var data = this._gui_areas.data;

		for(var i = 0; i < this._gui_areas.offset; i+=4)
		{
			if( e.mousex >= data[i] && 
				e.mousex < (data[i] + data[i+2]) &&
				e.mousey >= data[i+1] && 
				e.mousey < (data[i+1] + data[i+3]))
				return true;
		}
		return false;
	},

	/**
	* Renders an immediate gui BOX, used as background for panels
	* It blocks mouse events
	* @method Box
	* @param {Array} area [x,y,width,height]
	* @param {String} color a color in string format "#AFAFAF"
	* @param {Number} border_radius [optional] 
	* @param {Number} bottom_border_radius [optional] 
	*/
	Box: function( area, color, border_radius, bottom_border_radius )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );

		border_radius = border_radius || 0;
		bottom_border_radius = bottom_border_radius || border_radius;

		var ctx = gl;
		ctx.fillStyle = color || "#333";
		if( border_radius )
		{
			ctx.beginPath();
			ctx.roundRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3], border_radius, bottom_border_radius );
			ctx.fill();
		}
		else
			ctx.fillRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
	},

	/**
	* Renders a text (or a texture)
	*
	* @method Label
	* @param {Array} area [x,y,width,height]
	* @param {String|GL.Texture} content could be a string or a GL.Texture
	*/
	Label: function( area, content )
	{
		if(!area)
			throw("No area");
		if(content == null)
			return;

		var ctx = this._ctx;

		if(content.constructor === GL.Texture)
		{
			if(ctx.constructor === CanvasRenderingContext2D) //canvas 2D cannot render images
				content = content.data && (content.data.constructor === HTMLImageElement || content.data.constructor === Image) ? content.data : null;
			if(content)
				ctx.drawImage( content, area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
		}
		else if(content.constructor === HTMLImageElement || content.constructor === Image)
		{
			if(ctx.constructor === CanvasRenderingContext2D)
				ctx.drawImage( content, area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
		}
		else 
		{
			if(content.constructor === Number)
				content = content.toFixed(3);
			else if (content.constructor !== String)
				content = String(content);
			ctx.fillStyle = this.GUIStyle.color;
			ctx.font = (area[3]*0.75).toFixed(0) + "px " + this.GUIStyle.font;
			ctx.textAlign = "left";
			ctx.fillText( content, area[0] + area[3] * 0.2 + this._offset[0], area[1] + area[3] * 0.75  + this._offset[1]);
		}
	},

	/**
	* Just defines an area that could be clicked
	*
	* @method ClickArea
	* @param {Array} area [x,y,width,height]
	* @return {Boolean} true if the button was pressed inside the area
	*/
	ClickArea: function( area )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
				LS.Input.current_click = false; //consume event
		}

		return clicked;
	},

	/**
	* Renders a Button and returns if the button was pressed
	*
	* @method Button
	* @param {Array} area [x,y,width,height]
	* @param {String|GL.Texture} content could be a string or a GL.Texture (if null the button will be invisible)
	* @param {String|GL.Texture} content_over same as before but in case the mouse is over
	* @return {Boolean} true if the button was pressed 
	*/
	Button: function( area, content, content_over )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
				LS.Input.current_click = false; //consume event
		}

		if(content == null) //allows to create invisible buttons
			return clicked;

		if( content.constructor === String )
		{
			ctx.fillStyle = clicked ? "#FFF" : (is_over ? this.GUIStyle.backgroundColorOver : this.GUIStyle.backgroundColor );
			ctx.fillRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
		}

		if(content.constructor === GL.Texture)
		{
			var texture = content;
			if( is_over && content_over && content_over.constructor === GL.Texture)
				texture = content_over;
			ctx.drawImage( texture, area[0] + this._offset[0], area[1] + this._offset[0], area[2], area[3] );
		}
		else if(content.constructor === String)
		{
			ctx.fillStyle = is_over ? this.GUIStyle.colorTextOver : this.GUIStyle.color;
			ctx.font = (area[3]*0.75).toFixed(0) + "px " + this.GUIStyle.font;
			ctx.textAlign = "center";
			ctx.fillText( content, area[0] + area[2] * 0.5 + this._offset[0], area[1] + area[3] * 0.75 + this._offset[1]);
			ctx.textAlign = "left";
		}

		return clicked;
	},

	/**
	* Renders a Toolbar (list of buttons) and returns the active one
	*
	* @method Toolbar
	* @param {Array} area [x,y,width,height]
	* @param {Number} selected the index of the selected option
	* @param {Array[String|GL.Texture]} options an array containing either strings or GL.Texture
	* @return {Number} the selected index
	*/
	Toolbar: function( area, selected, options )
	{
		if( !area )
			throw("No area");
		if( !options || options.constructor !== Array )
			throw("No options");
		this.blockEventArea( area );

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var num = options.length;
		var x = area[0];
		var w = area[2];
		area[2] = w/num;

		for(var i = 0; i < num; ++i)
		{
			var content = options[i];
			var is_selected = selected == i;
			var clicked = false;
			area[0] = x + area[2] * i;

			if( mouse )
			{
				clicked = LS.Input.isEventInRect( mouse, area, this._offset );
				if(clicked)
				{
					selected = i;
					is_selected = true;
					LS.Input.current_click = false; //consume event
				}
			}

			if( !content || content.constructor === String )
			{
				ctx.fillStyle = is_selected ? this.GUIStyle.backgroundColorOver : this.GUIStyle.backgroundColor;
				ctx.fillRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
			}

			if(content)
			{
				if(content.constructor === GL.Texture)
				{
					var texture = content;
					if(!is_selected)
						ctx.globalAlpha = 0.5;
					ctx.drawImage( texture, area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
					ctx.globalAlpha = 1;
				}
				else if(content.constructor === String)
				{
					ctx.fillStyle = this.GUIStyle.color;
					ctx.font = (area[3]*0.75).toFixed(0) + "px " + this.GUIStyle.font;
					ctx.textAlign = "center";
					ctx.fillText( content, area[0] + area[2] * 0.5 + this._offset[0], area[1] + area[3] * 0.75 + this._offset[1] );
					ctx.textAlign = "left";
				}
			}
		}

		area[0] = x;
		area[2] = w;

		return selected;
	},

	/**
	* Renders a checkbox widget, and returns the current state
	* Remember: you must pass as value the same value returned by this function in order to work propertly
	*
	* @method Toggle
	* @param {Array} area [x,y,width,height]
	* @param {Boolean} value if the checkbox is on or off
	* @param {String|GL.Texture} content an string or image in case the checkbox is on
	* @param {String|GL.Texture} content_off an string or image in case the checkbox is off 
	* @param {Boolean} circle if true the checkboxes are circles instead of squares
	* @return {Boolean} the current state of the checkbox (will be different from value if it was pressed)
	*/
	Toggle: function( area, value, content, content_off, circle )
	{
		if(!area)
			throw("No area");
		value = !!value;
		this.blockEventArea( area );

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
			{
				LS.Input.current_click = false; //consume event
			}
		}

		var margin = (area[3]*0.2)

		if(content)
		{
			if(content.constructor === GL.Texture)
			{
				var texture = content;
				if( !value && content_off && content_off.constructor === GL.Texture)
					texture = content_off;
				ctx.drawImage( texture, area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
			}
			else if(content.constructor === String)
			{
				ctx.fillStyle = this.GUIStyle.color;
				ctx.font = (area[3]*0.75).toFixed(0) + "px " + this.GUIStyle.font;
				ctx.textAlign = "left";
				ctx.fillText( content, area[0] + margin + this._offset[0], area[1] + area[3] * 0.75 + this._offset[1]);

				var w = area[3] * 0.6;
				ctx.fillStyle = this.GUIStyle.backgroundColor;
				var x = area[0] + area[2] - margin*1.5 - w + this._offset[0];
				var y = area[1] + margin*0.5 + this._offset[1];

				if(circle)
					ctx.fillCircle( x, y, area[3] - margin );
				else
					ctx.fillRect(x, y, w+margin, area[3] - margin );
				ctx.fillStyle = value ? this.GUIStyle.selected : "#000";
				if(circle)
					ctx.fillCircle( area[0] + area[2] - margin - w + this._offset[0], area[1] + margin + this._offset[1], area[3] - margin*2 );
				else
					ctx.fillRect( area[0] + area[2] - margin - w + this._offset[0], area[1] + margin + this._offset[1], w, area[3] - margin*2 );
			}
		}

		return clicked ? !value : value;
	},


	/**
	* Renders a textfield widget and returns the current text value
	* Remember: you must pass as text the same text returned by this function in order to work propertly
	*
	* @method TextField
	* @param {Array} area [x,y,width,height]
	* @param {String} text the text to show in the textfield
	* @param {Number} max_length to limit the text, otherwise leave blank
	* @param {Boolean} is_password set to true to show as password
	* @param {Function} on_intro callback executed when clicked intro/return key
	* @param {Boolean} keep_focus_on_intro retains focus after intro
	* @return {Boolean} the current state of the checkbox (will be different from value if it was pressed)
	*/
	TextField: function( area, text, max_length, is_password, on_intro, keep_focus_on_intro )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );

		text = text === undefined ? "" : String(text);
		max_length = max_length || 1024;

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
			{
				LS.Input.current_click = null; //consume event
				LS.Input.last_click = mouse;
			}
		}
		var is_selected = false;
		if( LS.Input.last_click && LS.Input.isEventInRect( LS.Input.last_click, area, this._offset ) )
		{
			is_selected = true;
		}

		this.pressed_enter = false;
		if(is_selected)
		{
			var keys = LS.Input.keys_buffer;
			for( var i = 0; i < keys.length; ++i )
			{
				var key = keys[i];
				switch(key.keyCode)
				{
					case 8: text = text.substr(0, text.length - 1 ); break; //backspace
					case 13: 
						this.pressed_enter = true;
						if(!keep_focus_on_intro)
							LS.Input.last_click = null;
						if(on_intro)
						{
							var r = on_intro(text);
							if(r != null)
								text = r;
						}
						break; //return
					case 32: if(text.length < max_length) text += " "; break;
					default:
						if(text.length < max_length && key.key && key.key.length == 1) //length because control keys send a string like "Shift"
							text += key.key;
						/*
						if( key.keyCode >= 65 && key.keyCode <= 122 ) //letters
							text += key.shiftKey ? key.character.toUpperCase() : key.character.toLowerCase();
						*/
				}
				//console.log(key.charCode, key.keyCode, key.character, key.which, key );
			}
			keys.length = 0; //consume them
			LS.Input.current_key = null;
		}

		var line = (area[3]*0.02);
		var margin = (area[3]*0.2);

		//contour
		ctx.fillStyle = this.GUIStyle.backgroundColor;
		ctx.fillRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
		ctx.fillStyle = "#000";
		ctx.fillRect( area[0] + line + this._offset[0], area[1] + line + this._offset[1], area[2] - line*2, area[3] - line*2 );

		ctx.fillStyle = this.GUIStyle.color;
		ctx.font = (area[3]*0.75).toFixed(0) + "px " + this.GUIStyle.font;
		ctx.textAlign = "left";

		var cursor = "";
		if( is_selected && (((getTime() * 0.002)|0) % 2) == 0 )
			cursor = "|";

		var final_text = text;
		if(is_password)
		{
			final_text = "";
			for(var i = 0; i < text.length; ++i)
				final_text += "*";
		}

		ctx.fillText( final_text + cursor, area[0] + margin*2 + this._offset[0], area[1] + area[3] * 0.75 + this._offset[1] );

		return text;
	},

	isTextFieldSelected: function( area )
	{
		return LS.Input.last_click && LS.Input.isEventInRect( LS.Input.last_click, area, this._offset );
	},

	/**
	* Renders an horizontal slider widget, returns the current value
	* Remember: you must pass as value the same value returned by this function in order to work propertly
	*
	* @method HorizontalSlider
	* @param {Array} area [x,y,width,height]
	* @param {Number} value the value to show in the slider
	* @param {Number} left_value the minimum value for the slider
	* @param {Number} right_value the maximum value for the slider
	* @param {Boolean} show_value if you want to see a caption in text format with the value
	* @return {Number} the current value of the slider (will be different from value if it was clicked)
	*/
	HorizontalSlider: function( area, value, left_value, right_value, show_value )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );

		if(left_value === undefined)
			left_value = 0;
		if(right_value === undefined)
			right_value = 1;
		value = Number(value);
		left_value = Number(left_value);
		right_value = Number(right_value);

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		var range = right_value - left_value;
		var norm_value = (value - left_value) / range;
		if(norm_value < 0) norm_value = 0;
		if(norm_value > 1) norm_value = 1;

		var margin = (area[3]*this.GUIStyle.margin);

		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
			{
				norm_value = ( (LS.Input.Mouse.x - this._offset[0]) - (area[0] + margin)) / (area[2] - margin*2);
				if(norm_value < 0) norm_value = 0;
				if(norm_value > 1) norm_value = 1;
				value = norm_value * range + left_value;
			}
		}

		//bg
		ctx.fillStyle = this.GUIStyle.backgroundColor;
		ctx.fillRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
		//slider
		ctx.fillStyle = is_over ? this.GUIStyle.selected : this.GUIStyle.unselected;
		ctx.fillRect( area[0] + margin + this._offset[0], area[1] + margin + this._offset[1], Math.max(2, (area[2] - margin*2) * norm_value ), area[3] - margin*2 );

		if(show_value)
		{
			ctx.textAlign = "center";
			ctx.fillStyle = this.GUIStyle.color;
			ctx.font = (area[3]*0.5).toFixed(0) + "px " + this.GUIStyle.font;
			ctx.fillText( value.toFixed(2), area[0] + area[2] * 0.5 + this._offset[0], area[1] + area[3] * 0.7 + this._offset[1] );
			ctx.textAlign = "left";
		}

		return value;
	},

	/**
	* Renders an vertical slider widget, returns the current value
	* Remember: you must pass as value the same value returned by this function in order to work propertly
	*
	* @method VerticalSlider
	* @param {Array} area [x,y,width,height]
	* @param {Number} value the value to show in the slider
	* @param {Number} bottom_value the minimum value for the slider
	* @param {Number} top_value the maximum value for the slider
	* @return {Number} the current value of the slider (will be different from value if it was clicked)
	*/
	VerticalSlider: function( area, value, bottom_value, top_value )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );

		value = Number(value);
		if(bottom_value === undefined)
			bottom_value = 0;
		if(top_value === undefined)
			top_value = 1;
		bottom_value = Number(bottom_value);
		top_value = Number(top_value);

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		var range = top_value - bottom_value;
		var norm_value = (value - bottom_value) / range;
		if(norm_value < 0) norm_value = 0;
		if(norm_value > 1) norm_value = 1;

		var margin = (area[2]*this.GUIStyle.margin)

		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
			{
				norm_value = ( (LS.Input.Mouse.y - this._offset[1]) - (area[1] + margin)) / (area[3] - margin*2);
				if(norm_value < 0) norm_value = 0;
				if(norm_value > 1) norm_value = 1;
				norm_value = 1 - norm_value; //reverse slider
				value = norm_value * range + bottom_value;
			}
		}
	
		//bg
		ctx.fillStyle = this.GUIStyle.backgroundColor;
		ctx.fillRect( area[0] + this._offset[0], area[1] + this._offset[1], area[2], area[3] );
		//slider
		ctx.fillStyle = is_over ? this.GUIStyle.selected : this.GUIStyle.unselected;
		var slider_height = Math.max(2, (area[3] - margin*2) * norm_value);
		ctx.fillRect( area[0] + margin + this._offset[0], area[1] + area[3] - slider_height - margin + this._offset[1], area[2] - margin*2, slider_height );

		return value;
	},

	/**
	* Renders an knob slider widget, returns the current value
	* Remember: you must pass as value the same value returned by this function in order to work propertly
	*
	* @method Knob
	* @param {Array} area [x,y,width,height]
	* @param {Number} value the value to show in the slider
	* @param {Number} bottom_value the minimum value for the slider
	* @param {Number} top_value the maximum value for the slider
	* @param {Number} steps [optional] the numeber of steps (if 0 then infinite)
	* @param {Image|GL.Texture} content [optional] a texture or image to use as the knob
	* @return {Number} the current value of the slider (will be different from value if it was clicked)
	*/
	Knob: function( area, value, bottom_value, top_value, steps, content )
	{
		if(!area)
			throw("No area");
		this.blockEventArea( area );

		value = Number(value);
		if(bottom_value === undefined)
			bottom_value = 0;
		if(top_value === undefined)
			top_value = 1;
		steps = steps || 0;
		bottom_value = Number(bottom_value);
		top_value = Number(top_value);

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		var range = top_value - bottom_value;
		var norm_value = (value - bottom_value) / range;
		if(norm_value < 0) norm_value = 0;
		if(norm_value > 1) norm_value = 1;

		var margin = (area[2]*this.GUIStyle.margin)
		var start_angle = -Math.PI*0.75;
		var total_angle = 1.5*Math.PI;

		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
			{
				var dx = LS.Input.Mouse.x - (area[0] + area[2] * 0.5) - this._offset[0];
				var dy = LS.Input.Mouse.y - (area[1] + area[3] * 0.5) - this._offset[1];
				//var angle = Math.atan2( dx, -dy ) / Math.PI;
				var angle = ( Math.atan2( dx, -dy ) - start_angle ) / total_angle;
				norm_value = angle;
				//norm_value = ( (LS.Input.Mouse.y - this._offset[1]) - (area[1] + margin)) / (area[3] - margin*2);
				//norm_value = 1 - norm_value; //reverse slider
				if(norm_value < 0) norm_value = 0;
				if(norm_value > 1) norm_value = 1;
				value = norm_value * range + bottom_value;
			}
		}

		if(steps)
			norm_value = Math.round(norm_value * steps) / steps;

		if( content !== undefined ) //texture
		{
			if( content !== null ) // in case we are loading the texture
			{
				var texture = null;
				if(content.constructor === GL.Texture)
				{
					if(ctx.constructor === CanvasRenderingContext2D) //canvas 2D cannot render images
						content = content.data && (content.data.constructor === HTMLImageElement || content.data.constructor === Image) ? content.data : null;
					if(content)
						texture = content;
				}
				else if(content.constructor === HTMLImageElement || content.constructor === Image)
				{
					if(ctx.constructor === CanvasRenderingContext2D)
						texture = content;
				}
				if(texture)
				{
					ctx.save();
					ctx.translate( area[0] + area[2] * 0.5 + this._offset[0], area[1] + area[3] * 0.5 + this._offset[1] );
					ctx.rotate( norm_value * total_angle + start_angle );
					ctx.scale( area[3] / texture.height , area[3] / texture.height );
					ctx.drawImage( texture, -texture.width * 0.5, -texture.height * 0.5 );
					ctx.restore();
				}
			}
		}
		else
		{
			//bg
			ctx.strokeStyle = this.GUIStyle.outline;
			ctx.fillStyle = this.GUIStyle.backgroundColor;
			ctx.beginPath();
			ctx.arc( area[0] + area[2] * 0.5 + this._offset[0], area[1] + area[3] * 0.5 + this._offset[1], area[3] * 0.45, 0, 2 * Math.PI, false );
			ctx.fill();
			ctx.stroke();

			//slider
			ctx.lineWidth = area[3]*0.1;
			ctx.strokeStyle = is_over ? this.GUIStyle.selected : this.GUIStyle.unselected;
			ctx.beginPath();

			start_angle = -Math.PI*1.25;
			ctx.arc( area[0] + area[2] * 0.5 + this._offset[0], area[1] + area[3] * 0.5 + this._offset[1], area[3] * 0.35, start_angle, start_angle + Math.max(DEG2RAD,total_angle * norm_value), false );
			ctx.stroke();
			ctx.lineWidth = 1;
		}

		return value;
	},

	Pad: function( area, value, background )
	{
		this.DragArea( area, value );
		var ctx = this._ctx;

		if( background && ( background.constructor === GL.Texture || background.constructor === HTMLImageElement || background.constructor === HTMLCanvasElement) )
		{
			ctx.drawImage( background, area[0], area[1], area[2], area[3] );
		}
		else
		{
			ctx.globalAlpha = 1;
			ctx.fillStyle = this.GUIStyle.backgroundColor;
			ctx.fillRect( area[0]-2, area[1]-2, area[2]+4, area[3]+4 );
			ctx.fillStyle = "black";
			ctx.fillRect( area[0], area[1], area[2], area[3] );
		}

		//ball
		ctx.fillStyle = "white";
		ctx.globalAlpha = 1;
		var x = area[0] + value[0] * area[2];
		var y = area[1] + value[1] * area[3];
		//ctx.fillRect( x-5, y-5, 10,10 );
		ctx.beginPath();
		ctx.arc( x, y, 5, 0, 2 * Math.PI, false );
		ctx.fill();
	},

	//*
	DragArea: function( area, value, only_delta )
	{
		if(!area)
			throw("No area");
		if(!value)
			throw("No value");
		this.blockEventArea( area );

		var ctx = this._ctx;
		var is_over = LS.Input.isEventInRect( LS.Input.Mouse, area, this._offset );
		if(is_over)
		{
			this._is_on_top_of_immediate_widget = true;
			this.setCursor("pointer");
		}
		var mouse = LS.Input.current_click;
		var clicked = false;
		if( mouse )
		{
			clicked = LS.Input.isEventInRect( mouse, area, this._offset );
			if(clicked)
			{
				LS.Input.current_click = null; //consume event
				LS.Input.last_click = mouse;
			}
		}
		var is_selected = false;
		if( LS.Input.last_click && LS.Input.isEventInRect( LS.Input.last_click, area, this._offset ) )
		{
			is_selected = true;
			if( LS.Input.Mouse.dragging )
			{
				if(only_delta)
				{
					value[0] += LS.Input.Mouse.deltax || 0;
					value[1] += LS.Input.Mouse.deltay || 0;
				}
				else
				{
					var x = (LS.Input.Mouse.x - area[0]) / area[2];
					var y = (LS.Input.Mouse.y - area[1]) / area[3];
					value[0] = Math.clamp(x,0,1);
					value[1] = Math.clamp(y,0,1);
				}
			}
		}

		return value;
	},

	//*/

	pushStyle: function()
	{
		var new_style = LS.cloneObject( this.GUIStyle );
		this._style_stack.push(this.GUIStyle);
		this.GUIStyle = new_style;
	},

	popStyle: function()
	{
		if(this._style_stack.length)
			this.GUIStyle = this._style_stack.pop();
	},

	setCursor: function(type)
	{
		if(!this._allow_change_cursor)
			return;
		gl.canvas.style.cursor = type || "";
	}
};

Object.defineProperty( GUI, "GUIOffset", {
	set: function(v){
		if(!v.length || v.length < 2)
			return;
		this._offset[0] = v[0];
		this._offset[1] = v[1];
	},
	get: function()
	{
		return this._offset;
	},
	enumerable: true
});

//LEGACY API
GUI.show = GUI.showHTML;
GUI.hide = GUI.hideHTML;
GUI.load = GUI.loadHTML;

GUI.getRoot = function()
{
	console.warn("LS.GUI.getRoot() deprecated, use LS.GUI.getHTMLRoot() instead.");
	return LS.GUI.getHTMLRoot();
}

LS.GUI = GUI;
///@FILE:../src/resourcesManager.js
///@INFO: BASE
/**
* Static class that contains all the resources loaded, parsed and ready to use.
* It also contains the parsers and methods in charge of processing them
*
* @class ResourcesManager
* @constructor
*/

// **** RESOURCES MANANGER *********************************************
// Resources should follow the text structure:
// + id: number, if stored in remote server
// + resource_type: string ("Mesh","Texture",...) or if omitted the classname will be used
// + filename: string (this string will be used to get the filetype)
// + fullpath: the full path to reach the file on the server (folder + filename)
// + preview: img url
// + toBinary: generates a binary version to store on the server
// + serialize: generates an stringifible object to store on the server

// + _original_data: ArrayBuffer with the bytes form the original file
// + _original_file: File with the original file where this res came from

var ResourcesManager = {

	path: "", //url to retrieve resources relative to the index.html
	proxy: "", //url to retrieve resources outside of this host
	ignore_cache: false, //change to true to ignore client cache
	free_data: false, //free all data once it has been uploaded to the VRAM
	keep_files: false, //keep the original files inside the resource (used mostly in the editor)
	keep_urls: false, //keep the local URLs of loaded files
	allow_base_files: false, //allow to load files that are not in a subfolder

	scene_external_repository: null, //this is used by some scenes to specify where are the resources located

	//some containers
	resources: {}, //filename associated to a resource (texture,meshes,audio,script...)
	meshes: {}, //loadead meshes
	textures: {}, //loadead textures
	materials: {}, //shared materials (indexed by name)
	materials_by_uid: {}, //shared materials (indexed by uid)

	resources_not_found: {}, //resources that will be skipped because they werent found
	resources_being_loaded: {}, //resources waiting to be loaded
	resources_being_processed: {}, //used to avoid loading stuff that is being processes
	resources_renamed_recently: {}, //used to find resources with old names
	num_resources_being_loaded: 0,
	MAX_TEXTURE_SIZE: 4096,

	resource_pre_callbacks: {}, //used to extract resource info from a file ->  "obj":callback
	resource_post_callbacks: {}, //used to post process a resource type -> "Mesh":callback
	resource_once_callbacks: {}, //callback called once

	virtual_file_systems: {}, //protocols associated to urls  "VFS":"../"
	skip_proxy_extensions: ["mp3","wav","ogg","mp4","webm"], //this file formats should not be passed through the proxy
	force_nocache_extensions: ["js","glsl","json"], //this file formats should be reloaded without using the cache
	nocache_files: {}, //this is used by the editor to avoid using cached version of recently loaded files

	valid_resource_name_reg: /^[A-Za-z\d\s\/\_\-\.]+$/,

	/**
	* Returns a string to append to any url that should use the browser cache (when updating server info)
	*
	* @method getNoCache
	* @param {Boolean} force force to return a nocache string ignoring the default configuration
	* @return {String} a string to attach to a url so the file wont be cached
	*/

	getNoCache: function(force) { return (!this.ignore_cache && !force) ? "" : "nocache=" + getTime() + Math.floor(Math.random() * 1000); },

	/**
	* Resets all the resources cached, so it frees the memory
	*
	* @method reset
	*/
	reset: function()
	{
		this.resources = {};
		this.meshes = {};
		this.textures = {};
		this.materials = {};
		this.materials_by_uid = {};

		this.scene_external_repository = null;
	},

	/**
	* Resources need to be parsed once the data has been received, some formats could be parsed using native functions (like images) others 
	* require to pass the data through a series of functions (extract raw content, parse it, upload it to the GPU...
	* Registering a resource preprocessor the data will be converted once it is in memory 
	*
	* @method registerResourcePreProcessor
	* @param {String} fileformats the extension of the formats that this function will parse
	* @param {Function} callback the function to call once the data must be processed, if the process is async it must return true
	* @param {string} data_type 
	* @param {string} resource_type 
	*/
	registerResourcePreProcessor: function( fileformats, callback, data_type, resource_type )
	{
		if(!fileformats)
			return;

		var ext = fileformats.split(",");
		for(var i in ext)
		{
			var extension = ext[i].toLowerCase();
			this.resource_pre_callbacks[ extension ] = callback;
		}
	},

	/**
	* Some resources require to be post-processed right after being parsed to validate, extend, register (meshes need to have the AABB computed...)
	* This job could be done inside the parser but it is better to do it separatedly so it can be reused among different parsers.
	*
	* @method registerResourcePostProcessor
	* @param {String} resource_type the name of the class of the resource
	* @param {Function} callback the function to call once the data has been processed
	*/
	registerResourcePostProcessor: function(resource_type, callback)
	{
		this.resource_post_callbacks[ resource_type ] = callback;
	},

	/**
	* Returns the filename extension from an url
	*
	* @method getExtension
	* @param {String} fullpath url or filename
	* @param {boolean} complex_extension [optional] returns the extension from the first dot, otherwise only the part from last dot
	* @return {String} filename extension
	*/
	getExtension: function( fullpath, complex_extension )
	{
		if(!fullpath)
			return "";
		var question = fullpath.indexOf("?");
		if(question != -1)
			fullpath = fullpath.substr(0,question);

		var point = complex_extension ? fullpath.indexOf(".") : fullpath.lastIndexOf(".");
		if(point == -1)
			return "";
		return fullpath.substr(point+1).toLowerCase().trim();
	},

	/**
	* Returns the url without the extension
	*
	* @method removeExtension
	* @param {String} fullpath url or filename
	* @param {boolean} complex_extension [optional] removes the extension from the first dot, otherwise only the part from last dot
	* @return {String} url without extension
	*/
	removeExtension: function( fullpath, complex_extension )
	{
		if(!fullpath)
			return "";
		var question = fullpath.indexOf("?");
		if(question != -1)
			fullpath = fullpath.substr(0,question);
		var point = complex_extension ? fullpath.indexOf(".") : fullpath.lastIndexOf(".");
		if(point == -1)
			return fullpath;
		return fullpath.substr(0,point);
	},

	/**
	* Replaces the extension of a filename
	*
	* @method replaceExtension
	* @param {String} fullpath url or filename
	* @param {String} extension
	* @return {String} url with the new extension
	*/
	replaceExtension: function( fullpath, extension )
	{
		if(!fullpath)
			return "";
		extension = extension || "";
		var folder = this.getFolder( fullpath );
		var filename = this.getFilename( fullpath );
		return this.cleanFullpath( (folder ? ( folder + "/" ) : "") + filename + "." + extension );
	},

	/**
	* Returns the filename from a full path
	*
	* @method getFilename
	* @param {String} fullpath
	* @return {String} filename extension
	*/
	getFilename: function( fullpath )
	{
		if(!fullpath)
			return "";
		var pos = fullpath.lastIndexOf("/");
		var question = fullpath.lastIndexOf("?"); //to avoid problems with URLs line scene.json?nocache=...
		question = (question == -1 ? fullpath.length : (question - 1) ) - pos;
		return fullpath.substr(pos+1,question);
	},	

	/**
	* Returns the folder from a fullpath
	*
	* @method getFolder
	* @param {String} fullpath
	* @return {String} folder name
	*/
	getFolder: function(fullpath)
	{
		if(!fullpath)
			return "";
		var pos = fullpath.lastIndexOf("/");
		return fullpath.substr(0,pos);
	},	

	/**
	* Returns the filename without the folder or the extension
	*
	* @method getBasename
	* @param {String} fullpath
	* @return {String} filename extension
	*/
	getBasename: function( fullpath )
	{
		if(!fullpath)
			return "";

		var name = this.getFilename(fullpath);
		var pos = name.indexOf(".");
		if(pos == -1)
			return name;
		return name.substr(0,pos);
	},

	/**
	* Returns the url protocol (http, https) or empty string if no protocol was found
	*
	* @method getProtocol
	* @param {String} url
	* @return {String} protocol
	*/
	getProtocol: function( url )
	{
		if(!url)
			return "";

		var pos = url.substr(0,10).indexOf(":");
		var protocol = "";
		if(pos != -1)
			protocol = url.substr(0,pos);
		return protocol;
	},

	/**
	* Cleans resource name (removing double slashes to avoid problems) 
	* It is slow, so use it only in changes, not in getters
	*
	* @method cleanFullpath
	* @param {String} fullpath
	* @return {String} fullpath cleaned
	*/
	cleanFullpath: function(fullpath)
	{
		if(!fullpath)
			return "";

		if( fullpath.indexOf("//") == -1 )
		{
			if(fullpath.charCodeAt(0) == 47) // the '/' char
				return fullpath.substr(1);
			return fullpath;
		}

		//clean up the filename (to avoid problems with //)
		if(fullpath.indexOf("://") == -1)
			return fullpath.split("/").filter(function(v){ return !!v; }).join("/");
		return fullpath;
	},

	/**
	* Loads all the resources in the Object (it uses an object to store not only the filename but also the type)
	*
	* @method loadResources
	* @param {Object|Array} resources contains all the resources, associated with its type
	* @param {Object}[options={}] options to apply to the loaded resources
	* @return {number} the actual amount of resources being loaded (this differs fromt he resources passed because some could be already in memory)
	*/
	loadResources: function( resources, options )
	{
		if(!resources)
			return;

		if(resources.constructor === Array)
		{
			for( var i = 0; i < resources.length; ++i )
			{
				var name = resources[i];
				if( !name || name[0] == ":" || name[0] == "_" )
					continue;
				this.load( name, options );
			}
		}
		else //object
			for(var i in resources)
			{
				if( !i || i[0] == ":" || i[0] == "_" )
					continue;
				this.load( i, options );
			}

		this._total_resources_to_load = this.num_resources_being_loaded;
		LEvent.trigger( this, "start_loading_resources", this._total_resources_to_load );
		if(!this._total_resources_to_load) //all resources were already in memory
			LEvent.trigger( this, "end_loading_resources" );
		return this._total_resources_to_load;
	},	

	/**
	* Set the base path where all the resources will be fetched (unless they have absolute URL)
	* By default it will use the website home address
	*
	* @method setPath
	* @param {String} url
	*/
	setPath: function( url )
	{
		this.path = url;
	},

	/**
	* Set a proxy url where all non-local resources will be requested, allows to fetch assets to other servers.
	* request will be in this form: proxy_url + "/" + url_with_protocol: ->   http://myproxy.com/google.com/images/...
	*
	* @method setProxy
	* @param {String} proxy_url
	*/
	setProxy: function( proxy_url )
	{
		if(!proxy_url)
		{
			this.proxy = null;
			return;
		}

		if( proxy_url.indexOf("@") != -1 )
			this.proxy = location.protocol + "//" + proxy_url.replace("@", window.location.host );
		else
			this.proxy = proxy_url;

		if(	typeof(LiteGraph) !== "undefined" )
			LiteGraph.proxy = this.proxy;

	},

	/**
	* transform a url to a full url taking into account proxy, virtual file systems and external_repository
	* used only when requesting a resource to be loaded
	*
	* @method getFullURL
	* @param {String} url
	* @param {Object} options
	* @return {String} full url
	*/
	getFullURL: function( url, options )
	{
		if(!url)
			return null;

		var pos = url.substr(0,10).indexOf(":");
		var protocol = "";
		if(pos != -1)
			protocol = url.substr(0,pos);

		var resources_path = this.path;

		//from scene.external_repository
		if(this.scene_external_repository) 
			resources_path = this.scene_external_repository;

		if(options && options.force_local_url)
			resources_path = ".";

		//used special repository
		if(options && options.external_repository)
			resources_path = options.external_repository;

		if(protocol)
		{
			switch(protocol)
			{
				//external urls
				case 'http':
				case 'https':
					var full_url = url;
					var extension = this.getExtension( url ).toLowerCase();
					var host = url.substr(0,url.indexOf("/",protocol.length + 3));
					host = host.substr(protocol.length+3);
					if(this.proxy && host != location.host && this.skip_proxy_extensions.indexOf( extension ) == -1 && (!options || (options && !options.ignore_proxy)) ) //proxy external files
						return this.proxy + url; //this.proxy + url.substr(pos+3); //"://"
					return full_url;
					break;
				case 'blob':
					return url; //special case for local urls like URL.createObjectURL
				case '': //strange case
					return url;
					break;
				default:
					if(url[0] == ":" || url[0] == "_") //local resource
						return url;
					//test for virtual file system address
					var root_path = this.virtual_file_systems[ protocol ] || resources_path;
					return root_path + "/" + url.substr(pos+1);
			}
		}
		else
			return resources_path + "/" + url;
	},

	/**
	* Allows to associate a resource path like "vfs:myfile.png" to an url according to the value before the ":".
	* This way we can have alias for different folders where the assets are stored.
	* P.e:   "e","http://domain.com"  -> will transform "e:myfile.png" in "http://domain.com/myfile.png"
	*
	* @method registerFileSystem
	* @param {String} name the filesystem name (the string before the colons in the path)
	* @param {String} url the url to attach before 
	*/
	registerFileSystem: function(name, url)
	{
		this.virtual_file_systems[ name ] = url;
	},

	/**
	* Returns the resource if it has been loaded, if you want to force to load it, use load
	*
	* @method getResource
	* @param {String} url where the resource is located (if its a relative url it depends on the path attribute)
	* @param {Function} constructor [optional] allows to specify the class expected for this resource, if the resource doesnt match, it returns null
	* @return {*} the resource
	*/
	getResource: function( url, constructor )
	{
		if(!url)
			return null;
		url = this.cleanFullpath( url );
		if(!constructor)
			return this.resources[ url ];
		
		var res = this.resources[ url ];
		if(res && res.constructor === constructor )
			return res;
		return null;
	},

	/**
	* Returns the resource type ("Mesh","Texture","Material","SceneNode",...) of a given resource
	*
	* @method getResourceType
	* @param {*} resource
	* @return {String} the type in string format
	*/
	getResourceType: function( resource )
	{
		if(!resource)
			return null;
		if(resource.object_class)
			return resource.object_class;
		if(resource.constructor.resource_type)
			return resource.constructor.resource_type;
		return LS.getObjectClassName( resource );
	},

	/**
	* Returns an object containig all the resources and its data (used to export resources)
	*
	* @method getResourcesData
	* @param {Array} resource_names an array containing the resources names
	* @param {bool} allow_files [optional] used to allow to retrieve the data in File or Blob, otherwise only String and ArrayBuffer is supported
	* @return {Object} object with name:data
	*/
	getResourcesData: function( resource_names, allow_files )
	{
		if( resource_names.constructor !== Array )
		{
			console.error("getResourcesData expects Array");
			return null;
		}

		var result = {};

		for(var i = 0; i < resource_names.length; ++i)
		{
			var res_name = resource_names[i];
			var resource = LS.ResourcesManager.resources[ res_name ];
			if(!resource)
				continue;

			var data = null;
			if(resource._original_data) //must be string or bytes
				data = resource._original_data;
			else
			{
				var data_info = LS.Resource.getDataToStore( resource );
				data = data_info.data;
			}

			if(!data)
			{
				console.warn("Wrong data in resource");
				continue;
			}

			if(data.constructor === Blob || data.constructor === File)
			{
				if( !allow_files && (!data.data || data.data.constructor !== ArrayBuffer) )
				{
					console.warn("Not support to store File or Blob, please, use ArrayBuffer");
					continue;
				}
				data = data.data; //because files have an arraybuffer with the data if it was read
			}

			result[ res_name ] = data;
		}

		return result;
	},

	createResource: function( filename, data, must_register )
	{
		var resource = null;

		var extension = this.getExtension( filename );
		//get all the info about this file format
		var format_info = null;
		if(extension)
			format_info = LS.Formats.supported[ extension ];

		//has this resource an special class specified?
		if(format_info && format_info.resourceClass)
			resource = new format_info.resourceClass();
		else //otherwise create a generic LS.Resource (they store data or scripts)
		{
			//if we already have a LS.Resource, reuse it (this is to avoid garbage and solve a problem with the editor
			var old_res = this.resources[ filename ];
			if( old_res && old_res.constructor === LS.Resource )
			{
				resource = old_res;
				delete resource._original_data;
				delete resource._original_file;
				resource._modified = false;
			}
			else
				resource = new LS.Resource();
		}

		if(data)
		{
			if(resource.setData)
				resource.setData( data, true );
			else if(resource.fromData)
				resource.fromData( data );
			else
				throw("Resource without setData, cannot assign");
		}

		if(must_register)
			LS.ResourcesManager.registerResource( filename, resource );

		return resource;
	},

	/**
	* Marks the resource as modified, used in editor to know when a resource data should be updated
	*
	* @method resourceModified
	* @param {Object} resource
	*/
	resourceModified: function( resource )
	{
		if(!resource)
			return;

		if(resource.constructor === String)
		{
			console.warn("resourceModified parameter must be a resource, not a string");
			return;
		}

		//if the file has been modified we cannot keep using the original data
		delete resource._original_data;
		delete resource._original_file;

		resource._version = (resource._version || 0) + 1;

		if( resource.remotepath )
			resource._modified = true;

		LEvent.trigger(this, "resource_modified", resource );

		//TODO: from_prefab and from_pack should be the sabe property
		if(resource.from_pack)
		{
			if (resource.from_pack.constructor === String)
			{
				var pack = LS.ResourcesManager.getResource( resource.from_pack );
				if(pack)
					this.resourceModified(pack);
			}
		}
		if(resource.from_prefab)
		{
			if (resource.from_prefab.constructor === String)
			{
				var prefab = LS.ResourcesManager.getResource( resource.from_prefab );
				if(prefab)
					this.resourceModified(prefab);
			}
		}


	},

	/**
	* Unmarks the resource as modified
	*
	* @method resourceSaved
	* @param {Object} resource
	*/
	resourceSaved: function(resource)
	{
		if(!resource)
			return;
		delete resource._modified;
		resource.remotepath = resource.fullpath;
		LEvent.trigger(this, "resource_saved", resource );
	},

	/**
	* Loads a generic resource, the type will be infered from the extension, if it is json or wbin it will be processed
	* Do not use to load regular files (txts, csv, etc), instead use the LS.Network methods
	*
	* @method load
	* @param {String} url where the resource is located (if its a relative url it depends on the path attribute)
	* @param {Object}[options={}] options to apply to the loaded resource when processing it { force: to force a reload }
	* @param {Function} [on_complete=null] callback when the resource is loaded and cached, params: callback( resource, url  ) //( url, resource, options )
	* @param {Boolean} [force_load=false] if true it will load the resource, even if it already exists
	* @param {Function} [on_error=null] callback in case the file wasnt found
	*/
	load: function( url, options, on_complete, force_load, on_error )
	{
		if(!url)
			return console.error("LS.ResourcesManager.load requires url");

		//parameter swap...
		if(options && options.constructor === Function && !on_complete )
		{
			on_complete = options;
			options = null;
		}

		//if we already have it, then nothing to do
		var resource = this.resources[url];
		if( resource != null && !resource.is_preview && (!options || !options.force) && !force_load )
		{
			if(on_complete)
				on_complete(resource,url);
			return true;
		}

		options = options || {};

		//extract the filename extension
		var extension = this.getExtension( url );
		if(!extension && !this.resources[url] ) //unknown file type and didnt came from a pack or prefab
		{
			console.warn("Cannot load a file without extension: " + url );
			return false;
		}

		//we already tryed to load it and we couldnt find it, better not try again
		if( this.resources_not_found[url] )
			return;

		//if it is already being loaded, then add the callback and wait
		if(this.resources_being_loaded[url])
		{
			if(on_complete)
				this.resources_being_loaded[url].push( { options: options, callback: on_complete } );
			return;
		}

		if(this.resources_being_processed[url])
			return; //nothing to load, just waiting for the callback to process it

		if(!this.allow_base_files && url.indexOf("/") == -1)
		{
			if(!this._parsing_local_file) //to avoid showing this warning when parsing scenes with local resources
				console.warn("Cannot load resource, filename has no folder and LS.ResourcesManager.allow_base_files is set to false: ", url );
			return; //this is not a valid file to load
		}

		//otherwise we have to load it
		//set the callback
		this.resources_being_loaded[url] = [{options: options, callback: on_complete}];

		LEvent.trigger( LS.ResourcesManager, "resource_loading", url );
		//send an event if we are starting to load (used for loading icons)
		//if(this.num_resources_being_loaded == 0)
		//	LEvent.trigger( LS.ResourcesManager,"start_loading_resources", url );
		this.num_resources_being_loaded++;
		var full_url = this.getFullURL(url);

		//which type?
		var format_info = LS.Formats.getFileFormatInfo( extension );
		if(format_info && format_info.has_preview && !options.is_preview )
			LEvent.trigger( this, "load_resource_preview", url );

		//create the ajax request
		var settings = {
			url: full_url,
			success: function(response){
				LS.ResourcesManager.processResource( url, response, options, LS.ResourcesManager._resourceLoadedEnd, true );
			},
			error: function(err) { 	
				LS.ResourcesManager._resourceLoadedError(url,err);
				if(on_error)
					on_error(url);
			},
			progress: function(e) { 
				var partial_load = 0;
				if(e.total) //sometimes we dont have the total so we dont know the amount
					partial_load = e.loaded / e.total;
				if( LEvent.hasBind(  LS.ResourcesManager, "resource_loading_progress" ) ) //used to avoid creating objects during loading
					LEvent.trigger( LS.ResourcesManager, "resource_loading_progress", { url: url, event: e, progress: partial_load } );
				if( LEvent.hasBind(  LS.ResourcesManager, "loading_resources_progress" ) ) //used to avoid creating objects during loading
					LEvent.trigger( LS.ResourcesManager, "loading_resources_progress", 1.0 - (LS.ResourcesManager.num_resources_being_loaded - partial_load) / LS.ResourcesManager._total_resources_to_load );
			}
		};

		//force no cache by request
		settings.nocache = this.ignore_cache || (this.force_nocache_extensions.indexOf[ extension ] != -1) || this.nocache_files[ url ];

		//in case we need to force a response format 
		var format_info = LS.Formats.supported[ extension ];
		if( format_info )
		{
			if( format_info.dataType ) //force dataType, otherwise it will be set by http server
				settings.dataType = format_info.dataType;
			if( format_info.mimeType ) //force mimeType
				settings.mimeType = format_info.mimeType;
			if( format_info["native"] )
				settings.dataType = null;
		}

		//send the REQUEST
		LS.Network.request( settings ); //ajax call
		return false;
	},

	/**
	* Takes some resource data and transforms it to a resource (and Object ready to be used by the engine) and REGISTERs it in the ResourcesManager.
	* In most cases the process involves parsing and uploading to the GPU
	* It is called for every single resource that comes from an external source (URL) right after being loaded
	*
	* @method processResource
	* @param {String} url where the resource is located (if its a relative url it depends on the path attribute)
	* @param {*} data the data of the resource (could be string, arraybuffer, image... )
	* @param {Object}[options={}] options to apply to the loaded resource
	* @param {Function} on_complete once the resource is ready
	*/
	processResource: function( url, data, options, on_complete, was_loaded )
	{
		options = options || {};

		if(options && options.constructor !== Object)
			throw("processResource options must be object");
		if( data === null || data === undefined )
			throw("No data found when processing resource: " + url);

		var resource = null;
		var extension = this.getExtension( url );
		//get all the info about this file format
		var format_info = null;
		
		if(extension)
			format_info = LS.Formats.supported[ extension ];

		//callback to embed a parameter, ugly but I dont see a work around to create this
		var process_final = function( url, resource, options ){
			if(!resource)
			{
				LS.ResourcesManager._resourceLoadedEnd( url, null ); //to remove it from loading 
				return;
			}

			//do it again to avoid reusing old
			var extension = LS.ResourcesManager.getExtension( url );
			if(extension)
				format_info = LS.Formats.supported[ extension ];

			//convert format
			if( format_info && format_info.convert_to && extension != format_info.convert_to )
			{
				url += "." + format_info.convert_to;
				resource.filename += "." + format_info.convert_to;
				if( resource.fullpath )
					resource.fullpath += "." + format_info.convert_to;
				if(options.filename)
					options.filename += "." + format_info.convert_to;
			}

			//apply last changes: add to containers, remove from pending_loads, add special properties like fullpath, load associated resources...
			LS.ResourcesManager.processFinalResource( url, resource, options, on_complete, was_loaded );

			//Keep original file inside the resource in case we want to save it
			if(LS.ResourcesManager.keep_files && (data.constructor == ArrayBuffer || data.constructor == String) && (!resource._original_data && !resource._original_file) )
			{
				if( extension == LS.ResourcesManager.getExtension( resource.filename ) )
					resource._original_data = data;
			}
		}

		//this.resources_being_loaded[url] = [];
		this.resources_being_processed[url] = true;

		//no extension, then or it is a JSON, or an object with object_class or a WBin
		if(!extension)
			return this.processDataResource( url, data, options, process_final );


		// PRE-PROCESSING Stage (transform raw data in a resource) 
		// *******************************************************

		//special preprocessor
		var preprocessor_callback = this.resource_pre_callbacks[ extension.toLowerCase() ];
		if( preprocessor_callback )
		{
			//this callback should return the resource or true if it is processing it
			var resource = preprocessor_callback( url, data, options, process_final );
			if(resource === true)
				return;
			if( resource )
				process_final( url, resource, options );
			else //resource is null
			{
				console.warn("resource preprocessor_callback returned null");
				this._resourceLoadedError( url, "Resource couldn't be processed" );
				return;
			}
		}
		else if( format_info && (format_info.type || format_info.parse) ) //or you can rely on the format info parser
		{
			var resource = null;
			switch( format_info.type )
			{
				case "scene":
					resource = LS.ResourcesManager.processScene( url, data, options, process_final );
					break;
				case "mesh":
					resource = LS.ResourcesManager.processTextMesh( url, data, options, process_final );
					break;
				case "texture":
				case "image":
					resource = LS.ResourcesManager.processImage( url, data, options, process_final );
					break;
				case "data":
				default:
					if( format_info.parse )
					{
						//console.warn("Fallback to default parser");
						var resource = format_info.parse( data );
						if(resource)
							process_final( url, resource, options );
					}
					else
						console.warn("Format Info without parse function");
			}

			//we have a resource
			if( resource && resource !== true )
				process_final( url, resource, options );
		}
		else if( format_info && format_info.resourceClass) //this format has a class associated
		{
			var resource = new format_info.resourceClass();
			if(resource.fromData)
				resource.fromData( data );
			else if(resource.configure)
				resource.configure( JSON.parse(data) );
			else
				console.error("Resource Class doesnt have a function to process data after loading: ", format_info.resourceClass.name );

			//we have a resource
			if( resource && resource !== true )
				process_final( url, resource, options );
		}
		else //or just store the resource as a plain data buffer
		{
			var resource = LS.ResourcesManager.createResource( url, data );
			if(resource)
			{
				resource.filename = resource.fullpath = url;
				process_final( url, resource, options );
			}
		}
	},

	/**
	* Takes a resource instance, and adds some extra properties and register it
	*
	* @method processFinalResource
	* @param {String} url where the resource is located (if its a relative url it depends on the path attribute)
	* @param {*} the resource class
	* @param {Object}[options={}] options to apply to the loaded resource
	* @param {Function} on_complete once the resource is ready
	*/
	processFinalResource: function( fullpath, resource, options, on_complete, was_loaded )
	{
		if(!resource || resource.constructor === String)
			return LS.ResourcesManager._resourceLoadedError( fullpath, "error processing the resource" );

		//EXTEND add properties as basic resource ********************************
		resource.filename = fullpath;
		if(options.filename) //used to overwrite
			resource.filename = options.filename;
		if(!options.is_local)
			resource.fullpath = fullpath;
		else
			fullpath = resource.fullpath = resource.filename;
		if(options.from_prefab)
			resource.from_prefab = options.from_prefab;
		if(options.from_pack)
			resource.from_pack = options.from_pack;
		if(was_loaded)
			resource.remotepath = fullpath; //it was url but is the same as fullpath?
		if(options.is_preview)
			resource.is_preview = true;

		//Remove from temporal containers
		if( LS.ResourcesManager.resources_being_processed[ fullpath ] )
			delete LS.ResourcesManager.resources_being_processed[ fullpath ];

		//Load associated resources (some resources like LS.Prefab or LS.Scene have other resources associated that must be loaded too)
		if( resource.getResources )
			LS.ResourcesManager.loadResources( resource.getResources({}) );

		//REGISTER adds to containers *******************************************
		LS.ResourcesManager.registerResource( fullpath, resource );
		if(options.preview_of)
			LS.ResourcesManager.registerResource( options.preview_of, resource );

		//POST-PROCESS is done from inside registerResource, this way we ensure that every registered resource
		//has been post-processed, not only the loaded ones.

		//READY ***************************************
		if(on_complete)
			on_complete( fullpath, resource, options );
	},

	/**
	* Stores the resource inside the manager containers. This way it will be retrieveble by anybody who needs it.
	*
	* @method registerResource
	* @param {String} filename fullpath 
	* @param {Object} resource 
	*/
	registerResource: function( filename, resource )
	{
		if(!filename || !resource)
			throw("registerResource missing filename or resource");

		//test filename is valid (alphanumeric with spaces, dot or underscore and dash and slash
		if( filename[0] != ":" && this.valid_resource_name_reg.test( filename ) == false )
		{
			if( filename.substr(0,4) != "http" )
				console.warn( "invalid filename for resource: ", filename );
		}

		//clean up the filename (to avoid problems with //)
		filename = this.cleanFullpath( filename );

		if( this.resources[ filename ] === resource )
			return; //already registered

		if(resource.is_preview && this.resources[ filename ] )
			return; //previews cannot overwrite resources

		resource.filename = filename; //filename is a given name
		//resource.fullpath = filename; //fullpath only if they are in the server

		//Compute resource type
		if(!resource.object_class)
			resource.object_class = LS.getObjectClassName( resource );
		var type = resource.object_class;
		if(resource.constructor.resource_type)
			type = resource.constructor.resource_type;

		//Add to global container
		this.resources[ filename ] = resource;

		//POST-PROCESS resources extra final action (done here to ensure any registered resource is post-processed)
		var post_callback = this.resource_post_callbacks[ type ];
		if(post_callback)
			post_callback( filename, resource );

		//send message to inform new resource is available
		if(!resource.is_preview)
			LEvent.trigger(this,"resource_registered", resource);

		LS.GlobalScene.requestFrame(); //render scene
	},	

	/**
	* removes the resources from all the containers
	*
	* @method unregisterResource
	* @param {String} filename 
	* @return {boolean} true is removed, false if not found
	*/
	unregisterResource: function(filename)
	{
		var resource = this.resources[filename];

		if(!resource)
			return false; //not found

		delete this.resources[filename];

		//ugly: too hardcoded, maybe implement unregister_callbacks
		if( this.meshes[filename] )
			delete this.meshes[ filename ];
		if( this.textures[filename] )
			delete this.textures[ filename ];
		if( this.materials[filename] )
			delete this.materials[ filename ];

		if(resource.constructor === LS.Pack || resource.constructor === LS.Prefab)
			resource.setResourcesLink(null);

		LEvent.trigger(this,"resource_unregistered", resource);
		LS.GlobalScene.requestFrame(); //render scene
		return true;
	},

	/**
	* Used to load files and get them as File (or Blob)
	* @method getURLasFile
	* @param {String} filename 
	* @return {File} the file
	*/
	getURLasFile: function( url, on_complete )
	{
		var oReq = new XMLHttpRequest();
		oReq.open("GET", this.getFullURL(url), true);
		oReq.responseType = "blob";
		oReq.onload = function(oEvent) {
		  var blob = oReq.response;
		  if(on_complete)
			  on_complete(blob, url);
		};
		oReq.send();
	},

	/**
	* Changes the name of a resource and sends an event to all components to change it accordingly
	* @method renameResource
	* @param {String} old_name 
	* @param {String} new_name
	* @param {Boolean} [skip_event=false] ignore sending an event to all components to rename the resource
	* @return {boolean} if the file was found
	*/
	renameResource: function( old_name, new_name, skip_event )	
	{
		var res = this.resources[ old_name ];
		if(!res)
			return false;

		if(this.resources[ new_name ])
			console.warn("There is a resource already with this name, overwritting it: " + new_name );

		res.filename = new_name;
		if(res.fullpath)
			res.fullpath = new_name;

		this.resources[new_name] = res;
		delete this.resources[ old_name ];

		//inform everybody in the scene
		if(!skip_event)
			LS.GlobalScene.sendResourceRenamedEvent( old_name, new_name, res );

		//inform prefabs and packs...
		for(var i in this.resources)
		{
			var alert_res = this.resources[i];
			if( alert_res != res && alert_res.onResourceRenamed )
				if( alert_res.onResourceRenamed( old_name, new_name, res ) )
					this.resourceModified(alert_res);
		}

		//ugly: too hardcoded
		if( this.meshes[old_name] ) {
			delete this.meshes[ old_name ];
			this.meshes[ new_name ] = res;
		}
		if( this.textures[old_name] ) {
			delete this.textures[ old_name ];
			this.textures[ new_name ] = res;
		}
		if( this.materials[old_name] ) {
			delete this.materials[ old_name ];
			this.materials[ new_name ] = res;
		}

		//in case somebody needs to know where a resource has gone
		this.resources_renamed_recently[ old_name ] = new_name;

		if(!skip_event)
			LEvent.trigger( LS.ResourcesManager, "resource_renamed", [ old_name, new_name, res ] );
		return true;
	},

	/**
	* Tells if it is loading resources (or an specific resource)
	*
	* @method isLoading
	* @return {Boolean}
	*/
	isLoading: function( fullpath )
	{
		if(!fullpath)
			return this.num_resources_being_loaded > 0;
		if(this.resources_being_loaded[ fullpath ] || this.resources_being_processed[ fullpath ])
			return true;
		return false;
	},	

	/**
	* forces to try to reload again resources not found
	*
	* @method isLoading
	* @return {Boolean}
	*/
	clearNotFoundResources: function()
	{
		this.resources_not_found = {};
	},

	computeImageMetadata: function(texture)
	{
		var metadata = { width: texture.width, height: texture.height };
		return metadata;
	},

	/**
	* returns a mesh resource if it is loaded
	*
	* @method getMesh
	* @param {String} filename 
	* @return {Mesh}
	*/
	getMesh: function(name) {
		if(!name)
			return null;
		if(name.constructor === String)
			return this.meshes[name];
		if(name.constructor === GL.Mesh)
			return name;
		return null;
	},

	/**
	* returns a texture resource if it is loaded
	*
	* @method getTexture
	* @param {String} filename could be a texture itself in which case returns the same texture
	* @return {Texture} 
	*/
	getTexture: function(name) {
		if(!name)
			return null;
		if(name.constructor === String)
			return this.textures[name];
		if(name.constructor === GL.Texture)
			return name;
		return null;
	},

	getMaterial: function( name_or_id )
	{
		if(!name_or_id)
			return;
		if(name_or_id[0] == "@")
			return this.materials_by_uid[ name_or_id ];
		return this.materials[ name_or_id ];
	},

	convertFilenameToLocator: function( filename )
	{
		return "@RES-" + filename.replace(/\//gi,"\\");
	},

	convertLocatorToFilename: function( locator )
	{
		return locator.substr(5).replace(/\\/gi,"/");
	},

	/**
	* Binds a callback for when a resource is loaded (in case you need to do something special)
	*
	* @method onceLoaded
	* @param {String} fullpath of the resource you want to get the notification once is loaded
	* @param {Function} callback the function to call, it will be called as callback( fullpath, resource )
	* @return (number) index of the position in the array, use this index to cancel the event 
	*/
	onceLoaded: function( fullpath, callback )
	{
		var array = this.resource_once_callbacks[ fullpath ];
		if(!array)
		{
			this.resource_once_callbacks[ fullpath ] = [ callback ];
			return;
		}

		//avoid repeating
		if( array.indexOf( callback ) != -1 )
			return;

		array.push( callback );
		return array.length - 1;
	},

	/**
	* Cancels the binding of a onceLoaded
	*
	* @method cancelOnceLoaded
	* @param {String} fullpath fullpath of the resource you want to cancel the binding
	* @param {number} index the index of the callback to cancel (as it was returned by onceLoaded)
	*/
	cancelOnceLoaded: function( fullpath, index )
	{
		var array = this.resource_once_callbacks[ fullpath ];
		if(!array)
			return;
		array[ index ] = null;
	},

	//*************************************

	//Called after a resource has been loaded and processed
	_resourceLoadedEnd: function(url,res)
	{
		if( LS.ResourcesManager.debug )
			console.log("RES: " + url + " ---> " + LS.ResourcesManager.num_resources_being_loaded);

		if(res)
		{
			//trigger all associated load callbacks
			var callbacks_array = LS.ResourcesManager.resources_being_loaded[url];
			if(callbacks_array)
				for(var i = 0; i < callbacks_array.length; ++i )
				{
					if( callbacks_array[i].callback != null )
						callbacks_array[i].callback( res, url );
				}

			//triggers 'once' callbacks
			var callbacks_array = LS.ResourcesManager.resource_once_callbacks[url];
			if(callbacks_array)
			{
				for(var i = 0; i < callbacks_array.length; ++i)
					if(callbacks_array[i]) //could be null if it has been canceled
						callbacks_array[i](url, res);
				delete LS.ResourcesManager.resource_once_callbacks[url];
			}
		}

		//two pases, one for launching, one for removing
		if( LS.ResourcesManager.resources_being_loaded[url] )
		{
			delete LS.ResourcesManager.resources_being_loaded[url];
			LS.ResourcesManager.num_resources_being_loaded--;
			if(res)
				LEvent.trigger( LS.ResourcesManager, "resource_loaded", url );
			else
				LEvent.trigger( LS.ResourcesManager, "resource_problem_loading", url );
			LEvent.trigger( LS.ResourcesManager, "loading_resources_progress", 1.0 - LS.ResourcesManager.num_resources_being_loaded / LS.ResourcesManager._total_resources_to_load );
			if( LS.ResourcesManager.num_resources_being_loaded == 0)
			{
				LEvent.trigger( LS.ResourcesManager, "end_loading_resources", true);
				LS.ResourcesManager._total_resources_to_load = 0;
			}
		}

		//request frame
		LS.GlobalScene.requestFrame(); 
	},

	_resourceLoadedError: function( url, error )
	{
		console.log("Error loading " + url);
		delete LS.ResourcesManager.resources_being_loaded[url];
		delete LS.ResourcesManager.resource_once_callbacks[url];
		LS.ResourcesManager.resources_not_found[url] = true;
		LEvent.trigger( LS.ResourcesManager, "resource_not_found", url);
		LS.ResourcesManager.num_resources_being_loaded--;
		if( LS.ResourcesManager.num_resources_being_loaded == 0 )
			LEvent.trigger( LS.ResourcesManager, "end_loading_resources", false);
			//$(ResourcesManager).trigger("end_loading_resources");
	}
};

LS.RM = LS.ResourcesManager = ResourcesManager;

LS.getTexture = function( name_or_texture ) {
	return LS.ResourcesManager.getTexture( name_or_texture );
}	


// Resources readers and processors *********************************************
// When loading resources there are two stages:
// * Pre-process: extract from a container, parse raw data and transform it in a LS resource class (Texture,Mesh,SceneNode,Resource, ...)
// * Post-processed: validate, add extra metadata, and register
// This actions depend on the resource type, and format, and it is open so future formats are easy to implement.

//global formats: take a file and extract info
LS.ResourcesManager.registerResourcePreProcessor("wbin", function( filename, data, options) {
	//this object has already been expanded, it happens with objects created from parsers that encode the wbin extension
	if(data.constructor === Object && data.object_class )
	{
		if( LS.Classes[ data.object_class ] )
		{
			var ctor = LS.Classes[ data.object_class ] || window[ data.object_class ];
			if( ctor && ctor.fromBinary )
				return ctor.fromBinary( data, filename );
			else if(ctor && ctor.prototype.fromBinary)
			{
				var inst = new ctor();
				inst.fromBinary( object, filename );
				return inst;
			}
		}
		return data; 
	}
	//WBin will detect if there is a class name inside the data and do the conversion to the specified class (p.e. a Prefab or a Mesh)
	var final_data = WBin.load( data, false, filename );
	return final_data;
},"binary");

LS.ResourcesManager.registerResourcePreProcessor("json", function(filename, data, options) {
	var resource = data;
	if( data.constructor === String )
	{
		try
		{
			data = JSON.parse( data );
		}
		catch (err)
		{
			console.error("invalid JSON");
			return null;
		}
	}

	var class_name = data.object_class || data.object_type; //object_type for LEGACY
	if(!class_name && data.material_class)
		class_name = data.material_class; //HACK to fix one error

	if(!class_name)
	{
		var complex = LS.ResourcesManager.getExtension( filename, true );
		var ctor = LS.ResourceClasses_by_extension[ complex ];
		if(ctor)
			class_name = LS.getClassName( ctor );
	}

	if( class_name && !data.is_data )
	{
		var ctor = LS.Classes[ class_name ] || window[ class_name ];
		if(ctor)
		{
			if(ctor.prototype.configure)
			{
				resource = new ctor();
				resource.configure( data );
			}
			else
				resource = new ctor( data );
		}
		else
		{
			console.error( "JSON object_class class not found: " + class_name );
			return null;
		}
	}
	else
	{
		//unknown JSON, create a resource
		resource = new LS.Resource();
		resource.filename = filename;
		resource._data = data;
		resource.type = "json";
		resource.category = "json";
	}
	return resource;
});

//global formats: take a file and extract info
LS.ResourcesManager.registerResourcePreProcessor("zip", function( filename, data, options ) {
	
	if(!global.JSZip)
		throw("JSZip not found. To use ZIPs you must have the JSZip.js library included in the website.");

	var zip = new JSZip();
	zip.loadAsync( data ).then(function(zip){
		zip.forEach(function (relativePath, file){
			if(file.dir)
				return; //ignore folders
			var ext = LS.ResourcesManager.getExtension( relativePath );
			var format = LS.Formats.supported[ ext ];
			file.async( format && format.dataType == "text" ? "string" : "arraybuffer").then( function(filedata){
				if( relativePath == "scene.json" && (!options || !options.to_memory) )
					LS.GlobalScene.configure( JSON.parse( filedata ) );
				else
					LS.ResourcesManager.processResource( relativePath, filedata );
			});
		});
	});

	return true;

},"binary");

//For resources without file extension (JSONs and WBINs)
LS.ResourcesManager.processDataResource = function( url, data, options, callback )
{
	//JSON?
	if( data.constructor === String )
		data = JSON.parse(data);

	//WBIN?
	if(data.constructor == ArrayBuffer)
	{
		if(!data.byteLength) //empty file?
		{
			console.warn("Empty WBin?");
			return null;
		}

		resource = WBin.load(data);
		if(callback)
			callback(url, resource, options);
		return resource;
	}

	//JS OBJECT?
	var class_name = data.object_class;
	if(class_name && LS.Classes[class_name] )
	{
		var ctor = LS.Classes[class_name];
		var resource = null;
		if(ctor.prototype.configure)
		{
			resource = new LS.Classes[class_name]();
			resource.configure( data );
		}
		else
			resource = new LS.Classes[class_name]( data );
		if(callback)
			callback(url, resource, options);
		return resource;
	}

	console.warn("Resource Class name unknown: " + class_name );
	return false;
}

//Images ********

//Called after the http request for an image
//Takes image data in some raw format and transforms it in regular image data, then converts it to GL.Texture
LS.ResourcesManager.processImage = function( filename, data, options, callback ) {

	var extension = LS.ResourcesManager.getExtension(filename);
	var mimetype = "application/octet-stream";
	if(extension == "jpg" || extension == "jpeg")
		mimetype = "image/jpg";
	else if(extension == "webp")
		mimetype = "image/webp";
	else if(extension == "gif")
		mimetype = "image/gif";
	else if(extension == "png")
		mimetype = "image/png";
	else {
		var format = LS.Formats.supported[ extension ];
		if(format.mimetype)
			mimetype = format.mimetype;
		else
		{
			var texture = this.processImageNonNative( filename, data, options );
			inner_on_texture( texture );
			return;
		}
	}

	//blob and load
	var blob = new Blob([data],{type: mimetype});
	var objectURL = URL.createObjectURL( blob );

	//regular image
	var image = new Image();
	image.src = objectURL;
	image.real_filename = filename; //hard to get the original name from the image
	image.onload = function()
	{
		var filename = this.real_filename;
		var texture = LS.ResourcesManager.processTexture( filename, this, options );
		inner_on_texture( texture );
	}
	image.onerror = function(err){
		URL.revokeObjectURL(objectURL); //free memory
		if(callback)
			callback( filename, null, options );
		console.error("Error while loading image, file is not native image format: " + filename); //error if image is not an image I guess
	}

	function inner_on_texture( texture )
	{
		if(texture)
		{
			//LS.ResourcesManager.registerResource( filename, texture ); //this is done already by processResource
			if(LS.ResourcesManager.keep_files)
				texture._original_data = data;
		}

		if( objectURL )
		{
			if( !LS.ResourcesManager.keep_urls )
				URL.revokeObjectURL( objectURL ); //free memory
			else
				texture._local_url = objectURL; //used in strange situations
		}

		if(callback)
			callback(filename,texture,options);
	}

	return true;
}

//Similar to processImage but for non native file formats
LS.ResourcesManager.processImageNonNative = function( filename, data, options ) {

	//clone because DDS changes the original data
	var cloned_data = new Uint8Array(data).buffer;
	var texture_data = LS.Formats.parse( filename, cloned_data, options );

	if(!texture_data)
	{
		console.error("Cannot parse image format");
		return null;
	}

	if(texture_data.constructor == GL.Texture)
	{
		var texture = texture_data;
		texture.filename = filename;
		texture._original_data = cloned_data;
		return texture;
	}

	//texture in object format
	var texture = LS.ResourcesManager.processTexture( filename, texture_data );
	return texture;
}

//Takes one image (or canvas or object with width,height,pixels) as input and creates a GL.Texture
LS.ResourcesManager.processTexture = function(filename, img, options)
{
	if(img.width == (img.height / 6) || filename.indexOf("CUBECROSS") != -1) //cubemap
	{
		var cubemap_options = { wrapS: gl.MIRROR, wrapT: gl.MIRROR, magFilter: gl.LINEAR, minFilter: gl.LINEAR_MIPMAP_LINEAR };
		if( filename.indexOf("CUBECROSSL") != -1 )
			cubemap_options.is_cross = 1;
		var texture = GL.Texture.cubemapFromImage( img, cubemap_options );
		if(!texture) //happens if the image is not a cubemap
			return;
		texture.img = img;
	}
	else //regular texture
	{
		var default_mag_filter = gl.LINEAR;
		var default_wrap = gl.REPEAT;
		//var default_min_filter = img.width == img.height ? gl.LINEAR_MIPMAP_LINEAR : gl.LINEAR;
		var default_min_filter = gl.LINEAR_MIPMAP_LINEAR;
		if( !isPowerOfTwo(img.width) || !isPowerOfTwo(img.height) )
		{
			default_min_filter = gl.LINEAR;
			default_wrap = gl.CLAMP_TO_EDGE; 
		}
		var texture = null;

		//from TGAs...
		if(img.pixels) //not a real image, just an object with width,height and a buffer with all the pixels
			texture = GL.Texture.fromMemory(img.width, img.height, img.pixels, { format: (img.bpp == 24 ? gl.RGB : gl.RGBA), no_flip: img.flipY, wrapS: default_wrap, wrapT: default_wrap, magFilter: default_mag_filter, minFilter: default_min_filter });
		else //default format is RGBA (because particles have alpha)
			texture = GL.Texture.fromImage(img, { format: gl.RGBA,  wrapS: default_wrap, wrapT: default_wrap, magFilter: default_mag_filter, minFilter: default_min_filter });
		if(!texture)
			return;
		texture.img = img;
	}

	texture.filename = filename;
	texture.generateMetadata(); //useful
	return texture;
}

//Transform text mesh data in a regular GL.Mesh
LS.ResourcesManager.processTextMesh = function( filename, data, options ) {

	var mesh_data = LS.Formats.parse( filename, data, options );

	if(mesh_data == null)
	{
		console.error("Error parsing mesh: " + filename);
		return null;
	}

	var mesh = GL.Mesh.load( mesh_data );
	return mesh;
}

//this is called when loading a scene from a format that is not the regular serialize of our engine (like from ASE, G3DJ, BVH,...)
//converts scene data in a SceneNode
LS.ResourcesManager.processScene = function( filename, data, options ) {
	//options = options || {};

	var scene_data = LS.Formats.parse( filename, data, options );

	if(scene_data == null)
	{
		console.error( "Error parsing scene: " + filename );
		return null;
	}

	if( scene_data && scene_data.constructor === LS.Scene )
		throw("processScene must receive object, no Scene");

	if(!scene_data.root)
		throw("this is not an scene, root property missing");

	LS.ResourcesManager._parsing_local_file = true;

	//resources (meshes, textures...)
	for(var i in scene_data.meshes)
	{
		var mesh = scene_data.meshes[i];
		LS.ResourcesManager.processResource( i, mesh );
	}

	//used for anims mostly
	for(var i in scene_data.resources)
	{
		var res = scene_data.resources[i];
		LS.ResourcesManager.processResource(i,res);
	}

	for(var i in scene_data.materials)
	{
		var material = scene_data.materials[i];
		LS.ResourcesManager.processResource(i,material);
	}

	var node = new LS.SceneNode();
	node.configure( scene_data.root );

	//make it a pack or prefab
	if(options && options.filename)
	{
		var ext = LS.RM.getExtension( options.filename );
		if(ext != "json")
			options.filename += ".json";
	}

	LS.ResourcesManager._parsing_local_file = false;

	return node;
}

LS.ResourcesManager.loadTextureAtlas = function( atlas_info, on_complete, force )
{
	var image = new Image();
	image.src = this.getFullURL( atlas_info.filename );
	image.onload = inner_process;

	function inner_process()
	{
		var size = atlas_info.thumbnail_size;
		var canvas = document.createElement("canvas");
		canvas.width = size;
		canvas.height = size;
		var ctx = canvas.getContext("2d");

		for(var i in atlas_info.textures )
		{
			var info = atlas_info.textures[i];
			ctx.clearRect(0,0,canvas.width,canvas.height);
			ctx.drawImage( this, -info.pos[0], -info.pos[1] );
			var texture = GL.Texture.fromImage( canvas, { format: GL.RGBA, magFilter: gl.LINEAR, minFilter: gl.LINEAR_MIPMAP_LINEAR, wrap: gl.REPEAT } );
			if(!force)
				texture.is_preview = true;
			LS.ResourcesManager.registerResource( info.name, texture );
		}

		LS.GlobalScene.requestFrame();
		if(on_complete)
			on_complete();
	}
}

// Post processors **********************************************************************************
// Take a resource already processed and does some final actions (like validate, register or compute metadata)

LS.ResourcesManager.registerResourcePostProcessor("Mesh", function(filename, mesh ) {

	mesh.object_class = "Mesh"; //useful
	if(mesh.metadata)
	{
		mesh.metadata = {};
		mesh.generateMetadata(); //useful
	}
	//force to regenerate boundings
	if(!mesh.bounding || mesh.bounding.length != BBox.data_length || (mesh.info && mesh.info.groups && mesh.info.groups.length && !mesh.info.groups[0].bounding) )
	{
		mesh.bounding = null; //remove bad one (just in case)
		mesh.updateBoundingBox();
	}
	if(!mesh.getBuffer("normals"))
		mesh.computeNormals();

	if(LS.ResourcesManager.free_data) //free buffers to reduce memory usage
		mesh.freeData();

	LS.ResourcesManager.meshes[filename] = mesh;
});

LS.ResourcesManager.registerResourcePostProcessor("Texture", function( filename, texture ) {
	//store in appropiate container
	LS.ResourcesManager.textures[filename] = texture;
});

LS.ResourcesManager.registerResourcePostProcessor("Material", function( filename, material ) {
	//store in appropiate containers
	LS.ResourcesManager.materials[filename] = material;
	LS.ResourcesManager.materials_by_uid[ material.uid ] = material;
	if(material.prepare)
		material.prepare( LS.GlobalScene );
});

LS.ResourcesManager.registerResourcePostProcessor("Pack", function( filename, pack ) {
	//flag contents to specify where do they come from
	pack.flagResources();
});

LS.ResourcesManager.registerResourcePostProcessor("Prefab", function( filename, prefab ) {
	//apply to nodes in the scene that use this prefab
	prefab.applyToNodes();
});

LS.ResourcesManager.registerResourcePostProcessor("ShaderCode", function( filename, shader_code ) {
	//apply to materials that are using this ShaderCode
	shader_code.applyToMaterials();
});


//load priority
GL.Mesh.load_priority = -10;

///@FILE:../src/shaders.js
///@INFO: BASE
/* Basic shader manager 
	- Allows to load all shaders from XML
	- Allows to use a global shader
*/

//************************************
/**
* Shaders is the static class in charge of loading, compiling and storing shaders for reuse.
*
* @class Shaders
* @namespace LS
* @constructor
*/

var Shaders = {

	snippets: {},//to save source snippets
	shader_blocks_by_id: new Map(),//to save shader block
	shader_blocks: [],
	num_shaderblocks: 0, //used to know the index

	global_extra_shader_code: "",
	dump_compile_errors: true, //dump errors in console
	on_compile_error: null, //callback 


	/**
	* Initializes the shader manager
	*
	* @method init
	* @param {string} url a url to a shaders.xml can be specified to load the shaders
	*/
	init: function(url, ignore_cache)
	{
		//this.shader_blocks = {};//do not initialize, or we will loose all

		//base intro code for shaders
		var supported_features = []; //[name, webgl1_extension_name, enabling code]
		supported_features.push( ["STANDARD_DERIVATIVES", "OES_standard_derivatives", "#extension GL_OES_standard_derivatives : enable"] );
		supported_features.push( ["DRAW_BUFFERS","WEBGL_draw_buffers"] ); //#extension GL_EXT_draw_buffers : require

		this.global_extra_shader_code = String.fromCharCode(10) + "#define WEBGL_VERSION "+gl.webgl_version+"\n";

		for(var i in supported_features)
		{
			var feature = supported_features[i];
			if( gl.webgl_version == 2 || gl.extensions[ feature[1] ] )
			{
				this.global_extra_shader_code += "#define " + feature[0] + "\n";
				if(gl.webgl_version == 1 && feature[2]) 
					this.global_extra_shader_code += feature[2] + "\n";
			}
		}
	},

	/**
	* Reloads the XML file with the shaders, useful when editing the file
	*
	* @method reloadShaders
	* @param {function} on_complete call when the shaders have been reloaded
	*/
	reloadShaders: function(on_complete)
	{
		//TODO: crawl all materials and clear shaders
	},

	/**
	* Compiles a shader, the vertex and fragment shader are cached indepently to speed up compilations but a unique name must be provided
	*
	* @method compileShader
	* @param {string} vs_code the final source code for the vertex shader
	* @param {string} fs_code the final source code for the fragment shader
	* @param {string} name an unique name that should be associated with this shader
	* @return {GL.Shader} shader
	*/
	compile: function( vs_code, fs_code, name )
	{
		if(!name)
			throw("compileShader must have a name specified");

		if(!gl)
			return null;
		var shader = null;
		try
		{
			vs_code = this.global_extra_shader_code + vs_code;
			fs_code = this.global_extra_shader_code + fs_code;

			//speed up compilations by caching shaders compiled
			var vs_shader = this.compiled_shaders[name + ":VS"];
			if(!vs_shader)
				vs_shader = this.compiled_shaders[name + ":VS"] = GL.Shader.compileSource(gl.VERTEX_SHADER, vs_code);
			var fs_shader = this.compiled_shaders[name + ":FS"];
			if(!fs_shader)
				fs_shader = this.compiled_shaders[name + ":FS"] = GL.Shader.compileSource(gl.FRAGMENT_SHADER, fs_code);

			var old = getTime();
			shader = new GL.Shader( vs_shader, fs_shader );
			if(this.debug)
				console.log("Shader compile time: ", (getTime() - old).toFixed(3), "ms");
			shader.name = name;
			//console.log("Shader compiled: " + name);
		}
		catch (err)
		{
			if(this.dump_compile_errors)
			{
				this.dumpShaderError(name, err, vs_code, fs_code );
				this.dump_compile_errors = false; //disable so the console dont get overflowed
			}

			if(this.on_compile_error)
				this.on_compile_error(err);

			return null;
		}
		return shader;
	},

	clearShaderCodeCache: function()
	{
		var scs = [];

		//get all shadercodes...
		var shadercodes = LS.StandardMaterial.shader_codes;
		for(var i in shadercodes)
			scs.push( shadercodes[i] );

		var res = LS.ResourcesManager.resources;
		for(var i in res)
			if( res[i].constructor === LS.ShaderCode )
				scs.push( res[i] );

		//clear caches
		for(var i in scs)
		{
			var sb = scs[i];
			sb.clearCache();
		}
	},

	dumpShaderError: function( name, err, vs_code, fs_code )
	{
		console.error("Error compiling shader: " + name);
		console.log(err);
		console.groupCollapsed("Vertex Shader Code");
		//console.log("VS CODE\n************");
		var lines = vs_code.split("\n");
		for(var i in lines)
			console.log(i + ": " + lines[i]);
		console.groupEnd();

		console.groupCollapsed("Fragment Shader Code");
		//console.log("FS CODE\n************");
		lines = fs_code.split("\n");
		for(var i in lines)
			console.log(i + ": " + lines[i]);
		console.groupEnd();
	},

	/**
	* Register a code snippet ready to be used by the #import clause in the shader
	*
	* @method registerSnippet
	* @param {string} id
	* @param {string} code
	*/
	registerSnippet: function(id, code)
	{
		this.snippets[ id ] = { id: id, code: code };
	},

	/**
	* Returns the code of a snipper
	*
	* @method getSnippet
	* @param {string} id
	* @return {string} code
	*/
	getSnippet: function(id)
	{
		return this.snippets[ id ];
	},

	/**
	* register a shaderblock in the global container so it can be used by shadermaterials
	*
	* @method registerShaderBlock
	* @param {string} id
	* @param {LS.ShaderBlock} shader_block
	*/
	registerShaderBlock: function( id, shader_block, ignore_warning )
	{
		var block_id = -1;

		if( !ignore_warning && this.shader_blocks_by_id.get( id ) )
		{
			console.warn("There is already a ShaderBlock with that name, replacing it: ", id);
			block_id = this.shader_blocks_by_id.get(id).flag_id;
			this.clearShaderCodeCache();
		}
		else
			block_id = this.num_shaderblocks++;
		if(block_id >= 64)
			console.warn("Too many shaderblocks registered, not enought bits in a 64bits variable");

		shader_block.flag_id = block_id;
		shader_block.flag_mask = 1<<block_id;
		this.shader_blocks_by_id.set( id, shader_block );
		this.shader_blocks[ block_id ] = shader_block;
	},

	/**
	* register a shaderblock with the given id
	*
	* @method getShaderBlock
	* @param {string|Number} id
	* @return {LS.ShaderBlock} shader_block
	*/
	getShaderBlock: function( id )
	{
		if(id.constructor === String)
			return this.shader_blocks_by_id.get( id );
		return this.shader_blocks[id];
	},

	//this is global code for default shaders
	common_vscode: "\n\
		precision mediump float;\n\
		attribute vec3 a_vertex;\n\
		attribute vec3 a_normal;\n\
		attribute vec2 a_coord;\n\
		uniform mat4 u_model;\n\
		uniform mat4 u_viewprojection;\n\
	",
	common_fscode: "\n\
		precision mediump float;\n\
	"

};

LS.Shaders = Shaders;

/**
* A ShaderBlock represents a block of GLSL code that could be requested by a shader in order to obtain a functionality.
* SBs are registered and given a number, then if a shader wants that functionality it could use #pragma shaderblock "sb_name"
* it will be inserted in the material in the line of the pragma
*
* @class ShaderBlock
* @namespace LS
* @constructor
*/
function ShaderBlock( name )
{
	this.dependency_blocks = []; //blocks referenced by this block
	this.flag_id = -1;
	this.flag_mask = 0;
	this.events = null; //{};
	if(!name)
		throw("ShaderBlock must have a name");
	if(name.indexOf(" ") != -1)
		throw("ShaderBlock name cannot have spaces: " + name);
	this.name = name;
	this.code_map = new Map();
	this.context_macros = null;
}

ShaderBlock.prototype.defineContextMacros = function( macros )
{
	this.context_macros = macros;
}

/**
* register a shaderblock with the given id
* shader_type: vertex or fragment shader
*
* @method addCode
* @param {enum} shader_type could be  GL.VERTEX_SHADER or  GL.FRAGMENT_SHADER
* @param {string} enabled_code the code to insert if the shaderblock is enabled
* @param {string} disabled_code the code to insert if the shaderblock is disabled
* @param {Object} macros [optional] a set of macros to use when compiling this shader codes
*/
ShaderBlock.prototype.addCode = function( shader_type, enabled_code, disabled_code, macros )
{
	enabled_code  = enabled_code || "";
	disabled_code  = disabled_code || "";

	//this.checkDependencies( enabled_code );
	//this.checkDependencies( disabled_code );

	var info = { 
		enabled: new LS.GLSLCode( enabled_code ),
		disabled: new LS.GLSLCode( disabled_code ),
		macros: macros
	};
	this.code_map.set( shader_type, info );
}

ShaderBlock.prototype.bindEvent = function( event, code )  //priority?
{
	if(!this.events)
		this.events = {};
	this.events[ event ] = code;
}

/**
* Returns the full code of a shaderblock resolving all includes, shaderblocks, etc
* shadertype: GL.VERTEX_SHADER = 35633, GL.FRAGMENT_SHADER = 35632
*
* @method getFinalCode
* @param {enum} shader_type could be GL.VERTEX_SHADER or  GL.FRAGMENT_SHADER
* @param {number} block_flags a number containing the mask (every bit is a flag for a shaderblock) with all the enabled shader blocks
* @param {string} context an object with variable that could be fetched by the shaderblocks
* @return {String} the final code ready to be compiled
*/
ShaderBlock.prototype.getFinalCode = function( shader_type, block_flags, context )
{
	block_flags = block_flags || 0;
	var code = this.code_map.get( shader_type );
	if(!code)
		return null;
	var glslcode = (block_flags & this.flag_mask) ? code.enabled : code.disabled;
	var finalcode = glslcode.getFinalCode( shader_type, block_flags, context );

	if( code.macros )
	{
		var macros_code = "";
		for(var i in code.macros)
			macros_code += "#define " + i + code.macros[i] + "\n";
		finalcode = macros_code + finalcode;
	}
	return finalcode;
}

/**
* Registers this shaderblock in the global LS.Shaders container
*
* @method register
**/
ShaderBlock.prototype.register = function( overwrite )
{
	LS.Shaders.registerShaderBlock(this.name, this, overwrite );
}

ShaderBlock.prototype.checkDependencies = function( code )
{
//TODO
}


LS.ShaderBlock = ShaderBlock;



/**
* Used for parsing GLSL code and precompute info (mostly preprocessor macros)
* @class GLSLCode
* @constructor
* @param {String} code
*/
function GLSLCode( code )
{
	this.code = code;

	this.blocks = [];
	this.pragmas = {};
	this.uniforms = {};
	this.attributes = {};
	this.includes = {};
	this.snippets = {};
	this.shader_blocks = {}; //warning: this not always contain which shaderblocks are in use, because they could be dynamic using pragma define
	this.is_dynamic = false; //means this shader has no variations using pragmas or macros
	if(code)
		this.parse();
}

LS.GLSLCode = GLSLCode;

GLSLCode.pragma_methods = {};

GLSLCode.types_conversor = {
	"number":"float",
	"texture":"sampler2D",
	"textureCube":"samplerCube"
};


//block types
GLSLCode.CODE = 1;
GLSLCode.PRAGMA = 2;

//pargma types
GLSLCode.INCLUDE = 1;
GLSLCode.SHADERBLOCK = 2;
GLSLCode.SNIPPET = 3;
GLSLCode.EVENT = 4;

//given a code with some pragmas, it separates them
GLSLCode.prototype.parse = function()
{
	//remove comments
	var code = this.code.replace(/(\/\*([\s\S]*?)\*\/)|(\/\/(.*)$)/gm, '');

	this.fragments = [];
	this.pragmas = {};
	this.uniforms = {};
	this.streams = {};
	this.includes = {};
	this.snippets = {};
	this.shader_blocks = {};
	this.is_dynamic = false; //means this shader has no variations using pragmas or macros

	var current_fragment = [];
	var lines = code.split("\n");

	//parse
	for(var i = 0; i < lines.length; i++)
	{
		var line = lines[i].trim();
		if(!line.length)
			continue;//empty line

		if(line[0] != "#")
		{
			var words = line.split(" ");
			if( words[0] == "uniform" ) //store which uniforms we found in the code (not used yet)
			{
				var uniform_name = words[2].split(";");
				this.uniforms[ uniform_name[0] ] = words[1];
			}
			else if( words[0] == "attribute" ) //store which streams we found in the code (not used yet)
			{
				var uniform_name = words[2].split(";");
				this.attributes[ uniform_name[0] ] = words[1];
			}
			current_fragment.push(line);
			continue;
		}

		var t = line.split(" ");
		if(t[0] == "#pragma")
		{
			//merge lines and add previous fragment
			var current_fragment_code = current_fragment.join("\n");
			if(current_fragment_code.trim()) //in case is empty this code fragment
				this.fragments.push( { type: GLSLCode.CODE, code: current_fragment_code } ); 

			this.is_dynamic = true;
			this.pragmas[ t[2] ] = true;
			var action = t[1];
			current_fragment.length = 0;
			var pragma_info = { type: GLSLCode.PRAGMA, line: line, action: action, param: t[2] };

			var method = LS.GLSLCode.pragma_methods[ action ];
			if( !method || !method.parse )
			{
				console.warn("#pragma action unknown: ", action );
				continue;
			}
			if( method.parse.call( this, pragma_info, t ) === false )
			{
				//current_fragment.push("\n"); //add line to current fragment lines
				continue;
			}
			this.fragments.push( pragma_info ); //add pragma fragment
		}
		else
			current_fragment.push( line ); //add line to current fragment lines
	}

	if(current_fragment.length)
	{
		var current_fragment_code = current_fragment.join("\n");
		if(current_fragment_code.trim()) //in case is empty this code fragment
			this.fragments.push( { type: GLSLCode.CODE, code: current_fragment_code } ); //merge lines and add as fragment
	}

	//done
	return true;
}

GLSLCode.prototype.getFinalCode = function( shader_type, block_flags, context )
{
	if( !this.is_dynamic )
		return this.code;

	var code = "";
	context = context || {};
	var fragments = this.fragments;

	for(var i = 0; i < fragments.length; ++i)
	{
		var fragment = fragments[i];
		if( fragment.type === GLSLCode.CODE ) //regular code
		{
			code += fragment.code;
			continue;
		}

		var pragma_method = GLSLCode.pragma_methods[ fragment.action ];
		if(!pragma_method || !pragma_method.getCode )
		{
			code += "\n";
			continue;
		}

		var r = pragma_method.getCode.call( this, shader_type, fragment, block_flags, context );
		if( r )
			code += r;
	}

	return code;
}

// PRAGMA METHODS ****************************

GLSLCode.pragma_methods["include"] = {
	parse: function( pragma_info, t )
	{
		if(!t[2])
		{
			console.error("shader include without path");
			return false;
		}

		pragma_info.action_type = GLSLCode.INCLUDE;
		//resolve include
		var include = t[2].substr(1, t[2].length - 2); //safer than JSON.parse
		var fullname = include.split(":");
		var filename = fullname[0];
		var subfile = fullname[1];
		pragma_info.include = filename;
		pragma_info.include_subfile = subfile;
		this.includes[ pragma_info.include ] = true;
	},

	getCode: function( shader_type, fragment, block_flags, context )
	{
		var extra_code = "";

		var filename = fragment.include;
		var ext = LS.ResourcesManager.getExtension( filename );
		if(ext)
		{
			var extra_shadercode = LS.ResourcesManager.getResource( filename, LS.ShaderCode );
			if(!extra_shadercode)
			{
				LS.ResourcesManager.load( filename ); //force load
				return null;
			}
			if(!fragment.include_subfile)
				extra_code = "\n" + extra_shadercode._subfiles[""] + "\n";
			else
			{
				var extra = extra_shadercode._subfiles[ fragment.include_subfile ];
				if(extra === undefined)
					return null;
				extra_code = "\n" + extra + "\n";
			}
		}
		else
		{
			var snippet_code = LS.Shaders.getSnippet( filename );
			if( !snippet_code )
				return null; //snippet not found
			extra_code = "\n" + snippet_code.code + "\n";
		}

		return extra_code;
	}
};

GLSLCode.pragma_methods["define"] = {
	parse: function( pragma_info, t )
	{
		var param1 = t[2];
		var param2 = t[3];
		if(!param1 || !param2)
		{
			console.error("#pragma define missing parameters");
			return false;
		}
		pragma_info.define = [ param1, param2.substr(1, param2.length - 2) ];
	},
	getCode: function( shader_type, fragment, block_flags, context )
	{
		context[ fragment.define[0] ] = fragment.define[1];
	}
}

GLSLCode.pragma_methods["shaderblock"] = {
	parse: function( pragma_info, t )
	{
		if(!t[2])
		{
			console.error("#pragma shaderblock without name");
			return false;
		}
		pragma_info.action_type = GLSLCode.SHADERBLOCK;

		var param = t[2];
		if(param[0] == '"') //one means "shaderblock_name", two means shaderblock_var
		{
			pragma_info.shader_block = [1, param.substr(1, param.length - 2)]; //safer than JSON.parse
			this.shader_blocks[ pragma_info.shader_block[1] ] = true;
		}
		else
		{
			pragma_info.shader_block = [2, param];
			if(t[3]) //thirth parameter for default
			{
				pragma_info.shader_block.push( t[3].substr(1, t[3].length - 2) );
				this.shader_blocks[ pragma_info.shader_block[2] ] = true;
			}
		}
	},
	getCode: function( shader_type, fragment, block_flags, context )
	{
		var shader_block_name = fragment.shader_block[1];
		if( fragment.shader_block[0] == 2 ) //is dynamic shaderblock name
		{
			//dynamic shaderblock name
			if( context[ shader_block_name ] ) //search for the name in the context
				shader_block_name = context[ shader_block_name ];
			else 
				shader_block_name = fragment.shader_block[2]; //if not found use the default

			if(!shader_block_name)
			{
				console.error("ShaderBlock: no context var found: " + shader_block_name );
				return null;
			}
		}
		
		var shader_block = LS.Shaders.getShaderBlock( shader_block_name );
		if(!shader_block)
		{
			//console.error("ShaderCode uses unknown ShaderBlock: ", fragment.shader_block);
			return null;
		}

		var code = "\n";

		//add the define BLOCK_name only if enabled
		if( shader_block.flag_mask & block_flags )
			code = "\n#define BLOCK_" + ( shader_block.name.toUpperCase() ) +"\n";

		var block_code = shader_block.getFinalCode( shader_type, block_flags, context );
		if( block_code )
			code += block_code + "\n";

		return code;
	}
};

GLSLCode.pragma_methods["snippet"] = { 
	parse: function( pragma_info, t )
	{
		if(!t[2])
		{
			console.error("#pragma snippet without name");
			return false;
		}
		pragma_info.action_type = GLSLCode.SNIPPET;
		var snippet_name = t[2].substr(1, t[2].length - 2); //safer than JSON.parse
		pragma_info.snippet = snippet_name;
		this.snippets[ snippet_name ] = true;
	},
	getCode: function( shader_type, fragment, block_flags, context )
	{
		var snippet = LS.Shaders.getSnippet( fragment.snippet );
		if(!snippet)
		{
			console.error("ShaderCode uses unknown Snippet: ", fragment.snippet);
			return null;
		}

		return "\n" + snippet.code + "\n";
	}
};

GLSLCode.pragma_methods["event"] = { 
	parse: function( pragma_info, t )
	{
		if(!t[2])
		{
			console.error("#pragma event without name");
			return false;
		}
		pragma_info.action_type = GLSLCode.EVENT;
		var name = t[2].substr(1, t[2].length - 2); //safer than JSON.parse
		pragma_info.event = name;
	},
	getCode: function( shader_type, fragment, block_flags, context )
	{
		//dispatch event
		var code = "\n";
		var mask = 1;
		for(var i = 0, l = LS.Shaders.shader_blocks.length; i < l; ++i)
		{
			var block = LS.Shaders.shader_blocks[i];
			if(block_flags & block.flag_mask)
			{
				if(!block.events)
					continue;
				var block_code = block.events[ fragment.event ];
				if(!block_code)
					continue;
				code += block_code + "\n";
			}
		}
		//catch results
		return code;
	}
};

//not used
GLSLCode.breakLines = function(lines)
{
	//clean (this helps in case a line contains two instructions, like "uniform float a; uniform float b;"
	var clean_lines = [];
	for(var i = 0; i < lines.length; i++)
	{
		var line = lines[i].trim();
		if(!line)
			continue;
		var pos = line.lastIndexOf(";");
		if(pos == -1 || pos == lines.length - 1)
			clean_lines.push(line);
		else
		{
			var sublines = line.split(";");
			for(var j = 0; j < sublines.length; ++j)
			{
				if(sublines[j])
					clean_lines.push( sublines[j] + ";" );
			}
		}
	}
	return clean_lines;
}


// shaders

LS.Shaders.registerSnippet("input","\n\
			#ifndef SNIPPET_INPUT\n\
			#define SNIPPET_INPUT\n\
			//used to store topology input information\n\
			struct Input {\n\
				vec4 color;\n\
				vec3 vertex;\n\
				vec3 normal;\n\
				vec2 uv;\n\
				vec2 uv1;\n\
				\n\
				vec3 camPos;\n\
				float camDist;\n\
				vec3 viewDir;\n\
				vec3 worldPos;\n\
				vec3 worldNormal;\n\
				vec4 screenPos;\n\
			};\n\
			\n\
			Input getInput()\n\
			{\n\
				Input IN;\n\
				IN.color = vec4(1.0);\n\
				IN.vertex = v_pos;\n\
				IN.normal = v_normal;\n\
				IN.uv = v_uvs;\n\
				IN.uv1 = IN.uv;\n\
				\n\
				IN.camPos = u_camera_eye;\n\
				IN.viewDir = u_camera_eye - v_pos;\n\
				IN.camDist = length(IN.viewDir);\n\
				IN.viewDir /= IN.camDist;\n\
				IN.worldPos = v_pos;\n\
				IN.worldNormal = normalize(v_normal);\n\
				//IN.screenPos = vec4( (v_screenpos.xy / v_screenpos.w) * 0.5 + vec2(0.5), v_screenpos.zw );  //sometimes we need also z and w, thats why we pass all\n\
				IN.screenPos = vec4( (gl_FragCoord.xy / gl_FragCoord.w) * 0.5 + vec2(0.5), gl_FragCoord.zw );  //sometimes we need also z and w, thats why we pass all\n\
				return IN;\n\
			}\n\
			#endif\n\
	");

LS.Shaders.registerSnippet("structs","\n\
			//used to store topology input information\n\
			struct Input {\n\
				vec4 color;\n\
				vec3 vertex;\n\
				vec3 normal;\n\
				vec2 uv;\n\
				vec2 uv1;\n\
				\n\
				vec3 camPos;\n\
				vec3 viewDir;\n\
				float camDist;\n\
				vec3 worldPos;\n\
				vec3 worldNormal;\n\
				vec4 screenPos;\n\
			};\n\
			\n\
			//used to store surface shading properties\n\
			struct SurfaceOutput {\n\
				vec3 Albedo;\n\
				vec3 Normal; //separated in case there is a normal map\n\
				vec3 Emission;\n\
				vec3 Ambient;\n\
				float Specular;\n\
				float Gloss;\n\
				float Alpha;\n\
				float Reflectivity;\n\
				vec4 Extra; //for special purposes\n\
			};\n\
			\n\
			//used to store light contribution\n\
			//CAREFUL: this one is different than \n\
			struct FinalLight {\n\
				vec3 Color;\n\
				vec3 Ambient;\n\
				float Diffuse; //NdotL\n\
				float Specular; //RdotL\n\
				vec3 Emission;\n\
				vec3 Reflection;\n\
				float Attenuation;\n\
				float Shadow; //1.0 means fully lit\n\
			};\n\
	");

LS.Shaders.registerSnippet("spotFalloff","\n\
			float spotFalloff(vec3 spotDir, vec3 lightDir, float angle_phi, float angle_theta)\n\
			{\n\
				float sqlen = dot(lightDir,lightDir);\n\
				float atten = 1.0;\n\
				\n\
				vec4 spotParams = vec4( angle_phi, angle_theta, 1.0, 0.0 );\n\
				spotParams.w = 1.0 / (spotParams.x-spotParams.y);\n\
				\n\
				vec3 dirUnit = lightDir * sqrt(sqlen); //we asume they are normalized\n\
				float spotDot = dot(spotDir, -dirUnit);\n\
				if (spotDot <= spotParams.y)// spotDot <= cos phi/2\n\
					return 0.0;\n\
				else if (spotDot > spotParams.x) // spotDot > cos theta/2\n\
					return 1.0;\n\
				\n\
				// vertex lies somewhere beyond the two regions\n\
				float ifallof = pow( (spotDot-spotParams.y)*spotParams.w,spotParams.z );\n\
				return ifallof;\n\
			}\n\
	");

LS.Shaders.registerSnippet("getFlatNormal","\n\
			#ifdef STANDARD_DERIVATIVES\n\
				vec3 getFlatNormal(vec3 pos)\n\
				{\n\
					vec3 A = dFdx( pos );\n\
					vec3 B = dFdy( pos );\n\
					return normalize( cross(A,B) );\n\
				}\n\
			#else\n\
				vec3 getFlatNormal(vec3 pos)\n\
				{\n\
					return vec3(0.0);\n\
				}\n\
			#endif\n\
	");


LS.Shaders.registerSnippet("PackDepth32","\n\
			\n\
			float linearDepth(float z, float near, float far)\n\
			{\n\
				return (z - near) / (far - near);\n\
			}\n\
			float linearDepthNormalized(float z, float near, float far)\n\
			{\n\
				float z_n = 2.0 * z - 1.0;\n\
				return 2.0 * near * far / (far + near - z_n * (far - near));\n\
			}\n\
			\n\
			//packs depth normalized \n\
			vec4 PackDepth32(float depth)\n\
			{\n\
			  const vec4 bitShift = vec4( 256.0 * 256.0 * 256.0, 256.0 * 256.0, 256.0, 1.0 );\n\
			  const vec4 bitMask = vec4( 0.0, 1.0 / 256.0, 1.0 / 256.0, 1.0 / 256.0 );\n\
			  vec4 comp = fract(depth * bitShift);\n\
			  comp -= comp.xxyz * bitMask;\n\
			  return comp;\n\
			}\n\
");


LS.Shaders.registerSnippet("perturbNormal","\n\
				mat3 cotangent_frame(vec3 N, vec3 p, vec2 uv)\n\
				{\n\
					// get edge vectors of the pixel triangle\n\
					#ifdef STANDARD_DERIVATIVES\n\
					\n\
					vec3 dp1 = dFdx( p );\n\
					vec3 dp2 = dFdy( p );\n\
					vec2 duv1 = dFdx( uv );\n\
					vec2 duv2 = dFdy( uv );\n\
					\n\
					// solve the linear system\n\
					vec3 dp2perp = cross( dp2, N );\n\
					vec3 dp1perp = cross( N, dp1 );\n\
					vec3 T = dp2perp * duv1.x + dp1perp * duv2.x;\n\
					vec3 B = dp2perp * duv1.y + dp1perp * duv2.y;\n\
					#else\n\
					vec3 T = vec3(1.0,0.0,0.0); //this is wrong but its a fake solution\n\
					vec3 B = cross(N,T);\n\
					T = cross(B,N);\n\
					#endif\n\
					 \n\
					// construct a scale-invariant frame \n\
					float invmax = inversesqrt( max( dot(T,T), dot(B,B) ) );\n\
					return mat3( T * invmax, B * invmax, N );\n\
				}\n\
				\n\
				vec3 perturbNormal( vec3 N, vec3 V, vec2 texcoord, vec3 normal_pixel )\n\
				{\n\
					#ifdef USE_POINTS\n\
						return N;\n\
					#endif\n\
					\n\
					// assume N, the interpolated vertex normal and \n\
					// V, the view vector (vertex to eye)\n\
					//vec3 normal_pixel = texture2D(normalmap, texcoord ).xyz;\n\
					normal_pixel = normal_pixel * 255./127. - 128./127.;\n\
					mat3 TBN = cotangent_frame(N, V, texcoord);\n\
					return normalize(TBN * normal_pixel);\n\
				}\n\
		");

LS.Shaders.registerSnippet("bumpNormal","\n\
				\n\
				// Calculate the surface normal using screen-space partial derivatives of the height field\n\
				vec3 bumpNormal(vec3 position, vec3 normal, sampler2D texture, vec2 uvs, float factor)\n\
				{\n\
				#ifdef STANDARD_DERIVATIVES\n\
			        vec3 dpdx = dFdx(position);\n\
			        vec3 dpdy = dFdy(position);\n\
					vec3 r1 = cross(dpdy, normal);\n\
					vec3 r2 = cross(normal, dpdx);\n\
					\n\
					vec2 dtdx = dFdx(uvs) * factor;\n\
					vec2 dtdy = dFdy(uvs) * factor;\n\
					\n\
			        float h = texture2D( texture,  uvs ).r;\n\
			        float hdx = texture2D( texture,  uvs + dtdx ).r;\n\
			        float hdy = texture2D( texture,  uvs + dtdy ).r;\n\
					\n\
					return normalize(normal + (r1 * (hdx - h) - r2 * (hdy - h)) / dot(dpdx, r1));\n\
				#else\n\
					return normal;\n\
				#endif\n\
				}\n\
		");

LS.Shaders.registerSnippet("testClippingPlane","\n\
			float testClippingPlane(vec4 plane, vec3 p)\n\
			{\n\
				if(plane.x == 0.0 && plane.y == 0.0 && plane.z == 0.0)\n\
					return 0.0;\n\
				return (dot(plane.xyz, p) - plane.w) / dot(plane.xyz,plane.xyz);\n\
			}\n\
	");

LS.Shaders.registerSnippet("computePointSize","\n\
			float computePointSize(float radius, float w)\n\
			{\n\
				if(radius < 0.0)\n\
					return -radius;\n\
				return u_viewport.w * u_camera_perspective.z * radius / w;\n\
			}\n\
	");

LS.Shaders.registerSnippet("vec3ToCubemap2D","\n\
	vec2 vec3ToCubemap2D( vec3 v )\n\
	{\n\
		vec3 abs_ = abs(v);\n\
		float max_ = max(max(abs_.x, abs_.y), abs_.z); // Get the largest component\n\
		vec3 weights = step(max_, abs_); // 1.0 for the largest component, 0.0 for the others\n\
		float sign_ = dot(weights, sign(v)) * 0.5 + 0.5; // 0 or 1\n\
		float sc = dot(weights, mix(vec3(v.z, v.x, -v.x), vec3(-v.z, v.x, v.x), sign_));\n\
	    float tc = dot(weights, mix(vec3(-v.y, -v.z, -v.y), vec3(-v.y, v.z, -v.y), sign_));\n\
	    vec2 uv = (vec2(sc, tc) / max_) * 0.5 + 0.5;\n\
		// Offset into the right region of the texture\n\
		float offsetY = dot(weights, vec3(1.0, 3.0, 5.0)) - sign_;\n\
		uv.y = (uv.y + offsetY) / 6.0;\n\
		return uv;\n\
	}\n\
");

//base blocks that behave more like booleans 

//used to have the BLOCK_FIRSTPASS macro
var firstpass_block = LS.Shaders.firstpass_block = new LS.ShaderBlock("firstPass");
firstpass_block.addCode( GL.FRAGMENT_SHADER, "", "" );
firstpass_block.register();

//used to have the BLOCK_LASTPASS macro
var lastpass_block = LS.Shaders.lastpass_block = new LS.ShaderBlock("lastPass");
lastpass_block.addCode( GL.FRAGMENT_SHADER, "", "" );
lastpass_block.register();

//used when a mesh contains color info by vertex
var vertex_color_block = LS.Shaders.vertex_color_block = new LS.ShaderBlock("vertex_color");
vertex_color_block.register();

//used when a mesh contains extra uv set
var coord1_block = LS.Shaders.coord1_block = new LS.ShaderBlock("coord1");
coord1_block.register();

//used when a mesh contains extra buffers
var extra2_block = LS.Shaders.extra2_block = new LS.ShaderBlock("extra2");
extra2_block.bindEvent("vs_attributes", "attribute vec2 a_extra2;\n");
extra2_block.register();

//used when a mesh contains extra buffers
var extra3_block = LS.Shaders.extra3_block = new LS.ShaderBlock("extra3");
extra3_block.bindEvent("vs_attributes", "attribute vec3 a_extra3;\n");
extra3_block.register();

//used when a mesh contains extra buffers
var extra4_block = LS.Shaders.extra4_block = new LS.ShaderBlock("extra4");
extra4_block.bindEvent("vs_attributes", "attribute vec4 a_extra4;\n");
extra4_block.register();

//used to render normalinfo to buffer
var normalbuffer_block = LS.Shaders.normalbuffer_block = new LS.ShaderBlock("normalBuffer");
normalbuffer_block.addCode( GL.FRAGMENT_SHADER, "", "" );
normalbuffer_block.register();

//used when a mesh contains extra buffers
var instancing_block = LS.Shaders.instancing_block = new LS.ShaderBlock("instancing");
instancing_block.register();

//@point size with perspective
var pointparticles_block = new LS.ShaderBlock("pointparticles");
pointparticles_block.bindEvent("vs_final", "\n\
	gl_PointSize = u_point_size * u_viewport.w * u_camera_perspective.z / gl_Position.w;\n\
	#ifdef EXTRA2_BLOCK\n\
		gl_PointSize *= a_extra2.x;\n\
	#endif\n\
");
pointparticles_block.register();


LS.Shaders.registerSnippet("snoise","\n\
//	Simplex 3D Noise \n\
//	by Ian McEwan, Ashima Arts\n\
vec4 permute(vec4 x){return mod(((x*34.0)+1.0)*x, 289.0);}\n\
vec4 taylorInvSqrt(vec4 r){return 1.79284291400159 - 0.85373472095314 * r;}\n\
float snoise(vec3 v){ \n\
  const vec2  C = vec2(1.0/6.0, 1.0/3.0) ;\n\
  const vec4  D = vec4(0.0, 0.5, 1.0, 2.0);\n\
// First corner\n\
  vec3 i  = floor(v + dot(v, C.yyy) );\n\
  vec3 x0 =   v - i + dot(i, C.xxx) ;\n\
// Other corners\n\
  vec3 g = step(x0.yzx, x0.xyz);\n\
  vec3 l = 1.0 - g;\n\
  vec3 i1 = min( g.xyz, l.zxy );\n\
  vec3 i2 = max( g.xyz, l.zxy );\n\
  //  x0 = x0 - 0. + 0.0 * C \n\
  vec3 x1 = x0 - i1 + 1.0 * C.xxx;\n\
  vec3 x2 = x0 - i2 + 2.0 * C.xxx;\n\
  vec3 x3 = x0 - 1. + 3.0 * C.xxx;\n\
// Permutations\n\
  i = mod(i, 289.0 ); \n\
  vec4 p = permute( permute( permute( \n\
             i.z + vec4(0.0, i1.z, i2.z, 1.0 ))\n\
           + i.y + vec4(0.0, i1.y, i2.y, 1.0 )) \n\
           + i.x + vec4(0.0, i1.x, i2.x, 1.0 ));\n\
// Gradients\n\
// ( N*N points uniformly over a square, mapped onto an octahedron.)\n\
  float n_ = 1.0/7.0; // N=7\n\
  vec3  ns = n_ * D.wyz - D.xzx;\n\
  vec4 j = p - 49.0 * floor(p * ns.z *ns.z);  //  mod(p,N*N)\n\
  vec4 x_ = floor(j * ns.z);\n\
  vec4 y_ = floor(j - 7.0 * x_ );    // mod(j,N)\n\
  vec4 x = x_ *ns.x + ns.yyyy;\n\
  vec4 y = y_ *ns.x + ns.yyyy;\n\
  vec4 h = 1.0 - abs(x) - abs(y);\n\
  vec4 b0 = vec4( x.xy, y.xy );\n\
  vec4 b1 = vec4( x.zw, y.zw );\n\
  vec4 s0 = floor(b0)*2.0 + 1.0;\n\
  vec4 s1 = floor(b1)*2.0 + 1.0;\n\
  vec4 sh = -step(h, vec4(0.0));\n\
  vec4 a0 = b0.xzyw + s0.xzyw*sh.xxyy ;\n\
  vec4 a1 = b1.xzyw + s1.xzyw*sh.zzww ;\n\
  vec3 p0 = vec3(a0.xy,h.x);\n\
  vec3 p1 = vec3(a0.zw,h.y);\n\
  vec3 p2 = vec3(a1.xy,h.z);\n\
  vec3 p3 = vec3(a1.zw,h.w);\n\
//Normalise gradients\n\
  vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2, p2), dot(p3,p3)));\n\
  p0 *= norm.x;\n\
  p1 *= norm.y;\n\
  p2 *= norm.z;\n\
  p3 *= norm.w;\n\
// Mix final noise value\n\
  vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);\n\
  m = m * m;\n\
  return 42.0 * dot( m*m, vec4( dot(p0,x0), dot(p1,x1), \n\
                                dot(p2,x2), dot(p3,x3) ) );\n\
}");
///@FILE:../src/formats.js
///@INFO: BASE
/**
* Formats is the class where all the info about what is every format, how to parse it, etc, is located
*
* @class LS.Formats
* @param{String} id the id (otherwise a random one is computed)
* @constructor
*/
LS.Formats = {

	//all the supported file formats and their parsers
	supported: {},

	safe_parsing: false, //catch exceptions during parsing
	merge_smoothgroups: false,

	/**
	* Tells the system info about this file format
	* Info should contain fields like type:"image", resource: "Mesh|Texture", format: "text|binary", parse: function, native: true|false
	* 
	* @method addFormat
	*/
	addSupportedFormat: function( extensions, info )
	{
		if( extensions.constructor === String )
			extensions = extensions.split(",");

		for(var i = 0; i < extensions.length; ++i)
		{
			var extension = extensions[i].toLowerCase();
			if( this.supported[ extension ] )
				console.warn("There is already another parser associated to this extension: " + extension);
			this.supported[ extension ] = info;
		}
	},

	/**
	* Parse some data and returns the resulting resource
	* 
	* @method parse
	* @param {string} filename
	* @param {*} data could be a string, binary, arraybuffer, xml...
	* @param {Object} options how the file should be parsed
	* @return {*} the final resource, could be a Texture, a Mesh, or an object
	*/
	parse: function( filename, data, options)
	{
		options = options || {};
		var info = this.getFileFormatInfo( filename );
		if(!info) //unsupported extension
			return null;

		if(options.extension)
			info.extension = options.extension; //force a format
		else
			info.extension = LS.ResourcesManager.getExtension( filename );

		var format = this.supported[ info.extension ];
		if(!format || !format.parse)
		{
			console.error("Parser Error: No parser found for " + info.extension + " format");
			return null;
		}

		var result = null;
		if(!this.safe_parsing)
			result = format.parse( data, options, filename );
		else
			try
			{
				result = format.parse( data, options, filename );
			}
			catch (err)
			{
				console.error("Error parsing content", err );
				return null;
			}
		if(result)
			result.name = filename;
		return result;
	},

	//Returns info about a resource according to its filename
	TEXT_FORMAT: "text",
	JSON_FORMAT: "json",
	XML_FORMAT: "xml",
	BINARY_FORMAT: "binary",

	MESH_DATA: "MESH",
	IMAGE_DATA: "IMAGE",
	NONATIVE_IMAGE_DATA: "NONATIVE_IMAGE",
	SCENE_DATA: "SCENE",
	GENERIC_DATA: "GENERIC",
	
	getFileFormatInfo: function( filename )
	{
		var extension = filename.substr( filename.lastIndexOf(".") + 1).toLowerCase();
		return this.supported[ extension ];
	},

	guessType: function( filename )
	{
		if(!filename)
			return null;

		var ext = LS.RM.getExtension( filename ).toLowerCase();
		var info = this.supported[ ext ];
		if(!info)
			return null;
		return info.resource;
	},

	//Helpers ******************************

	//gets raw image information {width,height,pixels:ArrayBuffer} and create a dataurl to use in images
	convertToDataURL: function( img_data )
	{
		var canvas = document.createElement("canvas");
		canvas.width = img_data.width;
		canvas.height = img_data.height;
		//document.body.appendChild(canvas);
		var ctx = canvas.getContext("2d");
		var pixelsData = ctx.createImageData(img_data.width, img_data.height);
		var num_pixels = canvas.width * canvas.height;

		//flip and copy the pixels
		if(img_data.bytesPerPixel == 3)
		{
			for(var i = 0; i < canvas.width; ++i)
				for(var j = 0; j < canvas.height; ++j)
				{
					var pos = j*canvas.width*4 + i*4;
					var pos2 = (canvas.height - j - 1)*canvas.width*3 + i*3;
					pixelsData.data[pos+2] = img_data.pixels[pos2];
					pixelsData.data[pos+1] = img_data.pixels[pos2+1];
					pixelsData.data[pos+0] = img_data.pixels[pos2+2];
					pixelsData.data[pos+3] = 255;
				}
		}
		else {
			for(var i = 0; i < canvas.width; ++i)
				for(var j = 0; j < canvas.height; ++j)
				{
					var pos = j*canvas.width*4 + i*4;
					var pos2 = (canvas.height - j - 1)*canvas.width*4 + i*4;
					pixelsData.data[pos+0] = img_data.pixels[pos2+2];
					pixelsData.data[pos+1] = img_data.pixels[pos2+1];
					pixelsData.data[pos+2] = img_data.pixels[pos2+0];
					pixelsData.data[pos+3] = img_data.pixels[pos2+3];
				}
		}

		ctx.putImageData(pixelsData,0,0);
		img_data.dataurl = canvas.toDataURL("image/png");
		return img_data.dataurl;
	},

	/* extract important Mesh info from vertices (center, radius, bouding box) */
	computeMeshBounding: function(vertices)
	{
		//compute AABB and useful info
		var min = [vertices[0],vertices[1],vertices[2]];
		var max = [vertices[0],vertices[1],vertices[2]];
		for(var i = 0; i < vertices.length; i += 3)
		{
			var v = [vertices[i],vertices[i+1],vertices[i+2]];
			if (v[0] < min[0]) min[0] = v[0];
			else if (v[0] > max[0]) max[0] = v[0];
			if (v[1] < min[1]) min[1] = v[1];
			else if (v[1] > max[1]) max[1] = v[1];
			if (v[2] < min[2]) min[2] = v[2];
			else if (v[2] > max[2]) max[2] = v[2];
		}

		var center = [(min[0] + max[0]) * 0.5,(min[1] + max[1]) * 0.5, (min[2] + max[2]) * 0.5];
		var halfsize = [ min[0] - center[0], min[1] - center[1], min[2] - center[2]];
		return BBox.setCenterHalfsize( BBox.create(), center, halfsize );
	}
};

//native formats do not need parser
LS.Formats.addSupportedFormat( "png,jpg,jpeg,webp,bmp,gif", { "native": true, dataType: "arraybuffer", resource: "Texture", "resourceClass": GL.Texture, has_preview: true, type: "image" } );
LS.Formats.addSupportedFormat( "wbin", { dataType: "arraybuffer" } );
LS.Formats.addSupportedFormat( "json", { dataType: "text" } );
LS.Formats.addSupportedFormat( "js", { dataType: "text" } );
LS.Formats.addSupportedFormat( "txt,html,css,csv", { dataType: "text" } );
LS.Formats.addSupportedFormat( "glsl", { dataType: "text", resource: "ShaderCode", "resourceClass": LS.ShaderCode  } );
LS.Formats.addSupportedFormat( "zip", { dataType: "arraybuffer" } );
WBin.classes = LS.Classes; //WBin need to know which classes are accesible to be instantiated right from the WBin data info, in case the class is not a global class


var parserMESH = {
	extension: 'mesh',
	type: 'mesh',
	resource: 'Mesh',
	format: 'text',
	dataType:'text',

	parse: function(text, options)
	{
		options = options || {};
		var support_uint = true;

		var parser = GL.Mesh.parsers["mesh"];
		var mesh = parser(text, options);
		if( mesh.bounding.radius == 0 || isNaN(mesh.bounding.radius))
			console.log("no radius found in mesh");
		//console.log(mesh);
		return mesh;
	}
}

LS.Formats.addSupportedFormat( "mesh", parserMESH );
///@FILE:../src/utils/litegl-extra.js
///@INFO: BASE
//when working with animations sometimes you want the bones to be referenced by node name and no node uid, because otherwise you cannot reuse
//the same animation with different characters in the same scene.
GL.Mesh.prototype.convertBoneNames = function( root_node, use_uids )
{
	if(!this.bones || !this.bones.length)
		return 0;

	root_node = root_node || LS.GlobalScene;
	if( root_node.constructor == LS.Scene  )
		root_node = root_node.root;
	if(!root_node.findNode)
	{
		console.error("convertBoneNames first parameter must be node or scene");
		return 0;
	}

	var modified = false;

	//Rename the id to a relative name
	for(var i = 0; i < this.bones.length; ++i)
	{
		var bone = this.bones[i];
		var bone_name = bone[0];

		if( !use_uids )
		{
			if( bone_name[0] != LS._uid_prefix)
				continue; //already using a name, not a uid
			var node = root_node.findNode( bone_name );
			if(!node)
			{
				console.warn("Bone node not found: " + bone_name );
				continue;
			}
			bone[0] = node.name;
			modified = true;
		}
		else
		{
			if( bone_name[0] == LS._uid_prefix)
				continue; //already using a uid
			var node = root_node.findNode( bone_name );
			if(!node)
			{
				console.warn("Bone node not found: " + bone_name );
				continue;
			}
			bone[0] = node.uid;
			modified = true;
		}
	}

	//flag it
	if(modified)
		LS.RM.resourceModified( this );
}
///@FILE:../src/materials/material.js
///@INFO: BASE

//Material class **************************
/**
* A Material is a class in charge of defining how to render an object, there are several classes for Materials
* but this class is more like a template for other material classes.
* The rendering of a material is handled by the material itself, if not provided then uses the Renderer default one
* @namespace LS
* @class Material
* @constructor
* @param {String} object to configure from
*/

function Material( o )
{
	this.uid = LS.generateUId("MAT-");
	this._must_update = true;

	//used locally during rendering
	this._index = -1;
	this._local_id = Material.last_index++;
	this._last_frame_update = -1;

	/**
	* materials have at least a basic color property and opacity
	* @property color
	* @type {[[r,g,b]]}
	* @default [1,1,1]
	*/
	this._color = new Float32Array([1.0,1.0,1.0,1.0]);

	/**
	* render queue: which order should this be rendered
	* @property queue
	* @type {Number}
	* @default LS.RenderQueue.AUTO
	*/
	this._queue = LS.RenderQueue.AUTO;

	/**
	* render state: which flags should be used (in StandardMaterial this is overwritten due to the multipass lighting)
	* TODO: render states should be moved to render passes defined by the shadercode in the future to allow multipasses like cellshading outline render
	* @property render_state
	* @type {LS.RenderState}
	*/
	this._render_state = new LS.RenderState();


	this._light_mode = LS.Material.NO_LIGHTS;

	/**
	* matrix used to define texture tiling in the shader (passed as u_texture_matrix)
	* @property uvs_matrix
	* @type {mat3}
	* @default [1,0,0, 0,1,0, 0,0,1]
	*/
	this.uvs_matrix = new Float32Array([1,0,0, 0,1,0, 0,0,1]);

	/**
	* texture channels
	* contains info about the samplers for every texture channel
	* @property textures
	* @type {Object}
	*/
	this.textures = {};

	/**
	* used internally by LS.StandardMaterial
	* This will be gone in the future in order to use the new ShaderMaterial rendering system
	* @property query
	* @type {LS.ShaderQuery}
	*/
	//this._query = new LS.ShaderQuery();

	/**
	* flags to control cast_shadows, receive_shadows or ignore_frustum
	* @property flags
	* @type {Object}
	* @default { cast_shadows: true, receive_shadows: true, ignore_frutum: false }
	*/
	this.flags = {
		cast_shadows: true,
		receive_shadows: true,
		ignore_frustum: false
	};

	//properties with special storage (multiple vars shared among single properties)

	Object.defineProperty( this, 'color', {
		get: function() { return this._color.subarray(0,3); },
		set: function(v) { vec3.copy( this._color, v ); },
		enumerable: true
	});

	/**
	* The alpha component to control opacity
	* @property opacity
	* @default 1
	**/
	Object.defineProperty( this, 'opacity', {
		get: function() { return this._color[3]; },
		set: function(v) { this._color[3] = v; },
		enumerable: true
	});

	/**
	* the render queue id where this instance belongs
	* @property queue
	* @default LS.RenderQueue.DEFAULT;
	**/
	Object.defineProperty( this, 'queue', {
		get: function() { return this._queue; },
		set: function(v) { 
			if( isNaN(v) || !isNumber(v) )
				return;
			this._queue = v;
		},
		enumerable: true
	});

	/**
	* the render state flags to control how the GPU behaves
	* @property render_state
	**/
	Object.defineProperty( this, 'render_state', {
		get: function() { return this._render_state; },
		set: function(v) { 
			if(!v)
				return;
			for(var i in v) //copy from JSON object
				this._render_state[i] = v[i];
		},
		enumerable: true
	});


	if(o) 
		this.configure(o);
}

Material["@color"] = { type:"color" };

Material.icon = "mini-icon-material.png";
Material.last_index = 0;

Material.NO_LIGHTS = 0;
Material.ONE_LIGHT = 1;
Material.SEVERAL_LIGHTS = 2;

Material.EXTENSION = "json";

//material info attributes, use this to avoid errors when settings the attributes of a material

/**
* Surface color
* @property color
* @type {vec3}
* @default [1,1,1]
*/
Material.COLOR = "color";
/**
* Opacity. It must be < 1 to enable alpha sorting. If it is <= 0 wont be visible.
* @property opacity
* @type {number}
* @default 1
*/
Material.OPACITY = "opacity";

Material.SPECULAR_FACTOR = "specular_factor";
/**
* Specular glossiness: the glossines (exponent) of specular light
* @property specular_gloss
* @type {number}
* @default 10
*/
Material.SPECULAR_GLOSS = "specular_gloss";

Material.OPACITY_TEXTURE = "opacity";	//used for baked GI
Material.COLOR_TEXTURE = "color";	//material color
Material.AMBIENT_TEXTURE = "ambient";
Material.SPECULAR_TEXTURE = "specular"; //defines specular factor and glossiness per pixel
Material.EMISSIVE_TEXTURE = "emissive";
Material.ENVIRONMENT_TEXTURE = "environment";
Material.IRRADIANCE_TEXTURE = "irradiance";

Material.COORDS_UV0 = 0;
Material.COORDS_UV1 = 1;
Material.COORDS_UV0_TRANSFORMED = 2;
Material.COORDS_UV1_TRANSFORMED = 3;
Material.COORDS_UV_POINTCOORD = 4;

Material.TEXTURE_COORDINATES = { "uv0":Material.COORDS_UV0, "uv1":Material.COORDS_UV1, "uv0_transformed":Material.COORDS_UV0_TRANSFORMED, "uv1_transformed":Material.COORDS_UV1_TRANSFORMED, "uv_pointcoord": Material.COORDS_UV_POINTCOORD  };

Material.available_shaders = ["default","global","lowglobal","phong_texture","flat","normal","phong","flat_texture","cell_outline"];

Material.prototype.fillUniforms = function( scene, options )
{
	var uniforms = {};
	var samplers = [];

	uniforms.u_material_color = this._color;
	uniforms.u_ambient_color = scene.info ? scene.info.ambient_color : LS.ONES;
	uniforms.u_texture_matrix = this.uvs_matrix;

	uniforms.u_specular = vec2.create([1,50]);
	uniforms.u_reflection = 0.0;

	//iterate through textures in the material
	var last_texture_slot = 0;
	for(var i in this.textures) 
	{
		var sampler = this.getTextureSampler(i);
		if(!sampler)
			continue;

		var texture = Material.getTextureFromSampler( sampler );
		if(!texture) //loading or non-existant
			continue;

		samplers[ last_texture_slot ] = sampler;
		var uniform_name = i + (texture.texture_type == gl.TEXTURE_2D ? "_texture" : "_cubemap");
		uniforms[ uniform_name ] = last_texture_slot;
		last_texture_slot++;
	}

	//add extra uniforms
	for(var i in this.extra_uniforms)
		uniforms[i] = this.extra_uniforms[i];

	this._uniforms = uniforms;
	this._samplers = samplers; //samplers without fixed slot
}

/**
* Configure the material getting the info from the object
* @method configure
* @param {Object} object to configure from
*/
Material.prototype.configure = function(o)
{
	for(var i in o)
	{
		if(typeof (o[i]) === "function")
			continue;
		if(!this.setProperty( i, o[i] ) && LS.debug)
			console.warn("Material property not assigned: " + i );
	}
}

/**
* Serialize this material 
* @method serialize
* @return {Object} object with the serialization info
*/
Material.prototype.serialize = function( simplified )
{
	//remove hardcoded data from containers before serializing
	for(var i in this.textures)
		if (this.textures[i] && this.textures[i].constructor === GL.Texture)
			this.textures[i] = null;

	var o = LS.cloneObject(this);
	delete o.filename;
	delete o.fullpath;
	delete o.remotepath;
	o.material_class = LS.getObjectClassName(this);

	if( simplified )
	{
		delete o.render_state;
		delete o.flags;
		if( o.uvs_matrix && o.uvs_matrix.equal([1,0,0, 0,1,0, 0,0,1]) )
			delete o.uvs_matrix;
	}

	return o;
}


/**
* Clone this material (keeping the class)
* @method clone
* @return {Material} Material instance
*/
Material.prototype.clone = function()
{
	var data = this.serialize();
	if(data.uid)
		delete data.uid;
	return new this.constructor( JSON.parse( JSON.stringify( data )) );
}

/**
* Loads and assigns a texture to a channel
* @method loadAndSetTexture
* @param {Texture || url} texture_or_filename
* @param {String} channel
*/
Material.prototype.loadAndSetTexture = function( channel, texture_or_filename, options )
{
	options = options || {};
	var that = this;

	if( texture_or_filename && texture_or_filename.constructor === String ) //it could be the url or the internal texture name 
	{
		if(texture_or_filename[0] != ":")//load if it is not an internal texture
			LS.ResourcesManager.load(texture_or_filename,options, function(texture) {
				that.setTexture(channel, texture);
				if(options.on_complete)
					options.on_complete();
			});
		else
			this.setTexture(channel, texture_or_filename);
	}
	else //otherwise just assign whatever
	{
		this.setTexture( channel, texture_or_filename );
		if(options.on_complete)
			options.on_complete();
	}
}

/**
* gets all the properties and its types
* @method getPropertiesInfo
* @return {Object} object with name:type
*/
Material.prototype.getPropertiesInfo = function()
{
	var o = {
		color:"vec3",
		opacity:"number",
		uvs_matrix:"mat3"
	};

	var textures = this.getTextureChannels();
	for(var i in textures)
		o["tex_" + textures[i]] = "Texture"; //changed from Sampler
	return o;
}

/**
* gets all the properties and its types
* @method getProperty
* @return {Object} object with name:type
*/
Material.prototype.getProperty = function(name)
{
	if(name.substr(0,4) == "tex_")
		return this.textures[ name.substr(4) ];
	return this[name];
}


/**
* gets all the properties and its types
* @method getProperty
* @return {Object} object with name:type
*/
Material.prototype.setProperty = function( name, value )
{
	if( value === undefined )
		return;

	if( name.substr(0,4) == "tex_" )
	{
		if( (value && (value.constructor === String || value.constructor === GL.Texture)) || !value)
			this.setTexture( name.substr(4), value );
		return true;
	}

	switch( name )
	{
		//numbers
		case "queue": 
			if(value === 0)
				value = RenderQueue.AUTO; //legacy
			//nobreak
		case "opacity": 
			if(value !== null && value.constructor === Number)
				this[name] = value; 
			break;
		//bools
		//strings
		case "uid":
			this[name] = value; 
			break;
		//vectors
		case "uvs_matrix":
		case "color": 
			if(this[name].length == value.length)
				this[name].set( value );
			break;
		case "textures":
			for(var i in value)
			{
				var tex = value[i]; //sampler
				if(tex == null)
				{
					delete this.textures[i];
					continue;
				}
				if( tex.constructor === String )
					tex = { texture: tex, uvs: 0, wrap: 0, minFilter: 0, magFilter: 0 };
				if( tex.constructor !== GL.Texture && tex.constructor != Object )
				{
					console.warn("invalid value for texture:",tex);
					break;
				}
				tex._must_update = true;
				this.textures[i] = tex;
				if( tex.uvs != null && tex.uvs.constructor === String )
					tex.uvs = 0;
				//this is to ensure there are no wrong characters in the texture name
				if( this.textures[i] && this.textures[i].texture )
					this.textures[i].texture = LS.ResourcesManager.cleanFullpath( this.textures[i].texture );
			}
			//this.textures = cloneObject(value);
			break;
		case "flags":
			for(var i in value)
				this.flags[i] = value[i];
			break;
		case "transparency": //special cases
			this.opacity = 1 - value;
			break;
		case "render_state":
			this._render_state.configure( value );
			break;
		//ignore
		case "material_class":
		case "object_type":
			return true;
		default:
			return false;
	}
	return true;
}

Material.prototype.setPropertyValueFromPath = function( path, value, offset )
{
	offset = offset || 0;

	if( path.length < (offset+1) )
		return;

	//maybe check if path is texture?
	//TODO

	//assign
	this.setProperty( path[ offset ], value );
}

Material.prototype.getPropertyInfoFromPath = function( path )
{
	if( path.length < 1)
		return;

	var varname = path[0];
	var type = null;

	switch(varname)
	{
		case "queue": 
		case "opacity": 
		case "transparency":
			type = "number"; break;
		//vectors
		case "uvs_matrix":
			type = "mat3"; break;
		case "color": 
			type = "vec3"; break;
		case "textures":
			if( path.length > 1 )
			{
				return {
					node: this._root,
					target: this.textures,
					name: path[1],
					value: this.textures[path[1]] || null,
					type: "Texture"
				}
			}
			type = "Texture"; 
			break;
		default:
			return null;
	}

	return {
		node: this._root,
		target: this,
		name: varname,
		value: this[varname],
		type: type
	};
}

/**
* gets all the texture channels supported by this material
* @method getTextureChannels
* @return {Array} array with the name of every channel supported by this material
*/
Material.prototype.getTextureChannels = function()
{
	//console.warn("this function should never be called, it should be overwritten");
	return [];
}

/**
* Assigns a texture to a channel and its sampling parameters
* @method setTexture
* @param {String} channel for a list of supported channels by this material call getTextureChannels()
* @param {Texture} texture
* @param {Object} sampler_options
*/
Material.prototype.setTexture = function( channel, texture, sampler_options ) {

	if(!channel)
		throw("Material.prototype.setTexture channel must be specified");

	if(!texture)
	{
		delete this.textures[ channel ];
		return;
	}

	//clean to avoid names with double slashes
	if( texture.constructor === String )
		texture = LS.ResourcesManager.cleanFullpath( texture );

	//get current info
	var sampler = this.textures[ channel ];
	if(!sampler)
		this.textures[channel] = sampler = { 
			texture: texture, 
			uvs: 0, 
			wrap: 0, 
			minFilter: 0, 
			magFilter: 0,
			missing: "white"
		};
	else if(sampler.texture == texture && !sampler_options)
		return sampler;
	else
		sampler.texture = texture;

	if(sampler_options)
		for(var i in sampler_options)
			sampler[i] = sampler_options[i];
	sampler._must_update = true;

	if(texture.constructor === String && texture[0] != ":")
		LS.ResourcesManager.load( texture );

	return sampler;
}

/**
* Set a property of the sampling (wrap, uvs, filter)
* @method setTextureProperty
* @param {String} channel for a list of supported channels by this material call getTextureChannels()
* @param {String} property could be "uvs", "filter", "wrap"
* @param {*} value the value, for uvs check Material.TEXTURE_COORDINATES, filter is gl.NEAREST or gl.LINEAR and wrap gl.CLAMP_TO_EDGE, gl.MIRROR or gl.REPEAT
*/
Material.prototype.setTextureProperty = function( channel, property, value )
{
	var sampler = this.textures[channel];

	if(!sampler)
	{
		if(property == "texture")
			this.textures[channel] = sampler = { texture: value, uvs: 0, wrap: 0, minFilter: 0, magFilter: 0 };
		return;
	}

	sampler[ property ] = value;
}

/**
* Returns the texture in a channel
* @method getTexture
* @param {String} channel default is COLOR
* @return {Texture}
*/
Material.prototype.getTexture = function( channel ) {
	channel = channel || Material.COLOR_TEXTURE;

	var v = this.textures[channel];
	if(!v) 
		return null;

	if(v.constructor === String)
		return LS.ResourcesManager.textures[v];

	var tex = v.texture;
	if(!tex)
		return null;
	if(tex.constructor === String)
		return LS.ResourcesManager.textures[tex];
	else if(tex.constructor == Texture)
		return tex;
	return null;
}

/**
* Returns the texture sampler info of one texture channel (filter, wrap, uvs)
* @method getTextureSampler
* @param {String} channel get available channels using getTextureChannels
* @return {Texture}
*/
Material.prototype.getTextureSampler = function(channel) {
	return this.textures[ channel ];
}

Material.getTextureFromSampler = function(sampler)
{
	var texture = sampler.constructor === String ? sampler : sampler.texture;
	if(!texture) //weird case
		return null;

	//fetch
	if(texture.constructor === String)
		texture = LS.ResourcesManager.textures[ texture ];
	
	if (!texture || texture.constructor != GL.Texture)
		return null;
	return texture;
}

/**
* Assigns a texture sampler to one texture channel (filter, wrap, uvs)
* @method setTextureInfo
* @param {String} channel default is COLOR
* @param {Object} sampler { texture, uvs, wrap, filter }
*/
Material.prototype.setTextureSampler = function(channel, sampler) {
	if(!channel)
		throw("Cannot call Material setTextureSampler without channel");
	if(!sampler)
		delete this.textures[ channel ];
	else
		this.textures[ channel ] = sampler;
}

/**
* Collects all the resources needed by this material (textures)
* @method getResources
* @param {Object} resources object where all the resources are stored
* @return {Texture}
*/
Material.prototype.getResources = function (res)
{
	for(var i in this.textures)
	{
		var sampler = this.textures[i];
		if(!sampler) 
			continue;
		if(typeof(sampler.texture) == "string")
			res[ sampler.texture ] = GL.Texture;
	}
	return res;
}

/**
* Event used to inform if one resource has changed its name
* @method onResourceRenamed
* @param {Object} resources object where all the resources are stored
* @return {Boolean} true if something was modified
*/
Material.prototype.onResourceRenamed = function (old_name, new_name, resource)
{
	var v = false;
	for(var i in this.textures)
	{
		var sampler = this.textures[i];
		if(!sampler)
			continue;
		if(sampler.texture == old_name)
		{
			sampler.texture = new_name;
			v = true;

		}
	}
	return v;
}

/**
* Loads all the textures inside this material, by sending the through the ResourcesManager
* @method loadTextures
*/

Material.prototype.loadTextures = function ()
{
	var res = this.getResources({});
	for(var i in res)
		LS.ResourcesManager.load( i );
}

/**
* Register this material in a materials pool to be shared with other nodes
* @method registerMaterial
* @param {String} name name given to this material, it must be unique
*/
Material.prototype.registerMaterial = function(name)
{
	this.name = name;
	LS.ResourcesManager.registerResource(name, this);
	this.material = name;
}

Material.prototype.getCategory = function()
{
	return this.category || "Material";
}

Material.prototype.getLocator = function()
{
	if(this.filename)
		return LS.ResourcesManager.convertFilenameToLocator(this.fullpath || this.filename);

	if(this._root)
		return this._root.uid + "/material";
	return this.uid;
}

Material.prototype.assignToNode = function(node)
{
	if(!node)
		return false;
	var filename = this.fullpath || this.filename;
	node.material = filename ? filename : this;
	return true;
}

/**
* Creates a new property in this material class. Helps with some special cases
* like when we have a Float32Array property and we dont want it to be replaced by another array, but setted
* @method createProperty
* @param {String} name the property name as it should be accessed ( p.e.  "color" -> material.color )
* @param {*} value
* @param {String} type a valid value type ("Number","Boolean","Texture",...)
*/
Material.prototype.createProperty = function( name, value, type, options )
{
	if(type)
	{
		LS.validatePropertyType(type);
		this.constructor[ "@" + name ] = { type: type };
	}

	if(options)
	{
		if(!this.constructor[ "@" + name ])
			this.constructor[ "@" + name ] = {};
		LS.cloneObject( options, this.constructor[ "@" + name ] );
	}

	if(value == null)
		return;

	//basic type
	if(value.constructor === Number || value.constructor === String || value.constructor === Boolean)
	{
		this[ name ] = value;
		return;
	}

	//for vector type
	if(value.constructor === Float32Array )
	{
		var private_name = "_" + name;
		value = new Float32Array( value ); //clone
		this[ private_name ] = value; //this could be removed...

		Object.defineProperty( this, name, {
			get: function() { return value; },
			set: function(v) { value.set( v ); },
			enumerable: true,
			configurable: true
		});
	}
}

Material.prototype.prepare = function( scene )
{
	if(!this._uniforms)
	{
		this._uniforms = {};
		this._samplers = [];
	}

	if(this.onPrepare)
		this.onPrepare(scene);

	//this.fillShaderQuery( scene ); //update shader macros on this material
	this.fillUniforms( scene ); //update uniforms
}

Material.prototype.getShader = function( pass_name )
{
	var shader = Material._shader_color;
	if(!shader)
		shader = Material._shader_color = new GL.Shader( LS.Shaders.common_vscode + "void main(){ vec4 vertex = u_model * a_vertex;\ngl_Position = u_viewprojection * vertex;\n }", LS.Shaders.common_vscode + "uniform vec4 u_color;\n\void main(){ gl_FragColor = u_color;\n }");
	return shader;
}

//main function called to render an object
Material.prototype.renderInstance = function( instance, render_settings, pass )
{
	//some globals
	var renderer = LS.Renderer;
	var camera = LS.Renderer._current_camera;
	var scene = LS.Renderer._current_scene;
	var model = instance.matrix;

	//node matrix info
	var instance_final_query = instance._final_query;
	var instance_final_samplers = instance._final_samplers;
	var render_uniforms = LS.Renderer._render_uniforms;

	//maybe this two should be somewhere else
	render_uniforms.u_model = model; 
	render_uniforms.u_normal_model = instance.normal_matrix; 

	//global stuff
	this._render_state.enable();
	LS.Renderer.bindSamplers( this._samplers );
	var global_flags = 0;

	if(this.onRenderInstance)
		this.onRenderInstance( instance );

	//extract shader compiled
	var shader = shader_code.getShader( pass.name );
	if(!shader)
		return false;

	//assign
	shader.uniformsArray( [ scene._uniforms, camera._uniforms, render_uniforms, this._uniforms, instance.uniforms ] ); 

	//render
	instance.render( shader, this._primitive != -1 ? this._primitive : undefined );
	renderer._rendercalls += 1;

	return true;
}


//LS.registerMaterialClass( Material );
LS.registerResourceClass( Material );
LS.Material = Material;
///@FILE:../src/materials/shaderMaterial.js

/**
* ShaderMaterial allows to use your own shader from scratch
* @namespace LS
* @class ShaderMaterial
* @constructor
* @param {Object} object [optional] to configure from
*/
function ShaderMaterial( o )
{
	Material.call( this, null );

	this._shader = ""; //resource filename to a GL.ShaderCode
	this._shader_version = -1; //if the shader gets modified, the material should be modified too
	this._shader_flags = 0; //not used
	this._shader_code = null; //here the final code is stored (for debug)

	this._uniforms = {};	//uniforms to send to the shader
	this._samplers = [];	//textures to send to the shader
	this._properties = [];	//public properties to manipulate this material 
	this._properties_by_name = {};

	this._passes = {};		//the same ShaderCode is  used for different render passes (like color, shadowmap, picking), so here we cache the final GL.Shader for every type of pass
	this._light_mode = 0;	//info if this material should be rendered using lights: Material.NO_LIGHTS, Material.SEVERAL_LIGHTS 
	this._primitive = -1;	//which primitive to use when rendering this material
	this._allows_instancing = false;	//not supported yet

	this._version = -1;	

	this._last_valid_properties = null; //used to recover from a shader error

	if(o) 
		this.configure(o);
}

ShaderMaterial.description = "This material allows full control of the shader being used to render it.\nIt forces to code not only the surface properties but also the light equation.\nIt may be a little bit complex but it comes with examples.";

//assign a shader from a filename to a shadercode and reprocesses the code
Object.defineProperty( ShaderMaterial.prototype, "shader", {
	enumerable: true,
	get: function() {
		return this._shader;
	},
	set: function(v) {
		if(v)
			v = LS.ResourcesManager.cleanFullpath(v);
		if(this._shader == v)
			return;
		this._shader_code = null;
		this._shader = v;
		this.processShaderCode();
	}
});

//allows to assign a shader code that doesnt come from a resource (used from StandardMaterial)
Object.defineProperty( ShaderMaterial.prototype, "shader_code", {
	enumerable: false,
	get: function() {
		return this._shader_code;
	},
	set: function(v) {
		this._shader = null;
		this._shader_code = v;
		this.processShaderCode();
	}
});

Object.defineProperty( ShaderMaterial.prototype, "properties", {
	enumerable: true,
	get: function() {
		return this._properties;
	},
	set: function(v) {
		if(!v)
			return;
		this._properties = v;
		this._properties_by_name = {};
		for(var i in this._properties)
		{
			var p = this._properties[i];
			this._properties_by_name[ p.name ] = p;
		}
	}
});

Object.defineProperty( ShaderMaterial.prototype, "enableLights", {
	enumerable: true,
	get: function() {
		return this._light_mode != 0;
	},
	set: function(v) {
		this._light_mode = v ? 1 : 0;
	}
});

Object.defineProperty( ShaderMaterial.prototype, "version", {
	enumerable: false,
	get: function() {
		return this._version;
	},
	set: function(v) {
		console.error("version cannot be set manually");
	}
});

ShaderMaterial.prototype.addPass = function( name, vertex_shader, fragment_shader, macros )
{
	this._passes[ name ] = {
		vertex: vertex_shader,
		fragment: fragment_shader,
		macros: macros
	};
}

//called when preparing materials before rendering the scene
ShaderMaterial.prototype.prepare = function( scene )
{
	this.fillUniforms();

	if( this.onPrepare )
		this.onPrepare( scene );
}

//called when filling uniforms from this.prepare
ShaderMaterial.prototype.fillUniforms = function()
{
	//gather uniforms & samplers
	var samplers = this._samplers;
	samplers.length = 0;

	this._uniforms.u_material_color = this._color;

	for(var i = 0; i < this._properties.length; ++i)
	{
		var p = this._properties[i];
		if(p.internal) //internal is a property that is not for the shader (is for internal computations)
			continue;

		if(p.is_texture)
		{
			this._uniforms[ p.uniform ] = samplers.length;
			if(p.value)
				samplers.push( p.value );
			else
				samplers.push( " " ); //force missing texture
		}
		else
			this._uniforms[ p.uniform ] = p.value;
	}
}

//assigns a value to a property
ShaderMaterial.prototype.setProperty = function(name, value)
{
	//redirect to base material
	if( Material.prototype.setProperty.call(this,name,value) )
		return true;

	if(name == "shader")
		this.shader = value;
	else if(name == "properties")
	{
		this.properties.length = 0;
		this._properties_by_name = {};
		for(var i = 0; i < value.length; ++i)
		{
			var prop = value[i];
			if(prop.is_texture && prop.value && prop.value.constructor === String)
				prop.value = { texture: prop.value };
			this.properties[i] = prop;
			this._properties_by_name[ prop.name ] = prop;
			//if(prop.is_texture)
			//	this._samplers.push( prop.value );
		}
	}
	else if( this._properties_by_name[ name ] )
	{
		var prop = this._properties_by_name[ name ];
		if( !prop.value || prop.value.constructor === String || !prop.value.length )
			prop.value = value;
		else
			prop.value.set( value );
	}
	else
		return false;
	return true;
}

//check the ShaderCode associated and applies it to this material (keeping the state of the properties)
ShaderMaterial.prototype.processShaderCode = function()
{
	if(!this._shader_code && !this._shader)
	{
		this._properties.length = 0;
		this._properties_by_name = {};
		this._passes = {};
		this._samplers.length = 0;
		return false;
	}

	//get shader code
	var shader_code = this._shader_code;
	
	if( !shader_code && this._shader )
		shader_code = LS.ResourcesManager.getResource( this.shader );

	if( !shader_code || shader_code.constructor !== LS.ShaderCode )
		return false;

	var old_properties = this._properties_by_name;
	var old_state = this._render_state.serialize();
	if( shader_code._has_error ) //save them
		this._last_valid_properties = old_properties; 
	else if( this._last_valid_properties )
	{
		old_properties = this._last_valid_properties;
		this._last_valid_properties = null;
	}

	this._properties.length = 0;
	this._properties_by_name = {};
	this._passes = {};
	this._samplers.length = 0;
	this._light_mode = 0;
	this._primitive = -1;

	//reset material properties
	this._queue = LS.RenderQueue.GEOMETRY;
	this._render_state.init();

	//clear old functions
	for(var i in this)
	{
		if(!this.hasOwnProperty(i))
			continue;
		if( this[i] && this[i].constructor === Function )
			delete this[i];
	}

	this._render_state.configure(old_state);

	//apply init 
	if( shader_code._functions.init )
	{
		if(!LS.catch_exceptions)
			shader_code._functions.init.call( this );
		else
		{
			try
			{
				shader_code._functions.init.call( this );
			}
			catch (err)
			{
				LS.dispatchCodeError(err);
			}
		}
	}

	for(var i in shader_code._global_uniforms)
	{
		var global = shader_code._global_uniforms[i];
		if( global.disabled ) //in case this var is not found in the shader
			continue;
		this.createUniform( global.name, global.uniform, global.type, global.value, global.options );
	}

	//set version before asssignOldProperties
	this._shader_version = shader_code._version;
	this._version++;

	//restore old values
	this.assignOldProperties( old_properties );
}

//used after changing the code of the ShaderCode and wanting to reload the material keeping the old properties
ShaderMaterial.prototype.assignOldProperties = function( old_properties )
{
	//get shader code
	var shader = null;
	var shader_code = this.getShaderCode(); //no parameters because we just want the render_state and init stuff
	if( shader_code )
		shader = shader_code.getShader();

	for(var i = 0; i < this._properties.length; ++i)
	{
		var new_prop = this._properties[i];

		if(!old_properties[ new_prop.name ])
			continue;
		var old = old_properties[ new_prop.name ];
		if(old.value === undefined)
			continue;

		//validate (avoids error if we change the type of a uniform and try to reassign a value)
		if( !old.internal && shader && !new_prop.is_texture ) //textures are not validated (because they are samplers, not values)
		{
			var uniform_info = shader.uniformInfo[ new_prop.uniform ];
			if(!uniform_info)
				continue;
			if(new_prop.value !== undefined)
			{
				if( !GL.Shader.validateValue( new_prop.value, uniform_info ) )
				{
					new_prop.value = undefined;
					continue;
				}
			}
		}

		//this is to keep current values when coding the shader from the editor
		if( new_prop.value && new_prop.value.set ) //special case for typed arrays avoiding generating GC
		{
			//this is to be careful when an array changes sizes
			if( old.value && old.value.length && new_prop.value.length && old.value.length <= new_prop.value.length)
				new_prop.value.set( old.value );
			else
				new_prop.value = old.value;
		}
		else
			new_prop.value = old.value;
	}
}

ShaderMaterial.nolights_vec4 = new Float32Array([0,0,0,1]);
ShaderMaterial.missing_color = new Float32Array([1,0,1,1]);

//called from LS.Renderer when rendering an instance
ShaderMaterial.prototype.renderInstance = function( instance, render_settings, pass )
{
	//get shader code
	var shader_code = this.getShaderCode( instance, render_settings, pass );
	if(!shader_code || shader_code.constructor !== LS.ShaderCode )
	{
		//return true; //skip rendering
		shader_code = LS.ShaderCode.getDefaultCode( instance, render_settings, pass  ); //use default shader
		if( pass.id == COLOR_PASS.id) //to assign some random color
			this._uniforms.u_material_color = ShaderMaterial.missing_color;
	}

	//this is in case the shader has been modified in the editor (reapplies the shadercode to the material)
	if( shader_code._version !== this._shader_version && this.processShaderCode )
		this.processShaderCode();

	//some globals
	var renderer = LS.Renderer;
	var camera = LS.Renderer._current_camera;
	var scene = LS.Renderer._current_scene;
	var model = instance.matrix;
	var renderer_uniforms = LS.Renderer._uniforms;

	//maybe this two should be somewhere else
	renderer_uniforms.u_model = model; 
	renderer_uniforms.u_normal_model = instance.normal_matrix; 

	//compute flags: checks the ShaderBlocks attached to this instance and resolves the flags
	var block_flags = instance.computeShaderBlockFlags();
	var global_flags = LS.Renderer._global_shader_blocks_flags;

	//find environment texture
	if( pass == COLOR_PASS ) //allow reflections only in color pass
	{
		global_flags |= LS.ShaderMaterial.reflection_block.flag_mask;

		var environment_sampler = this.textures["environment"];
		var environment_texture = environment_sampler && environment_sampler.texture ? environment_sampler.texture : null;

		if( !environment_texture ) //use global
		{
			if( LS.Renderer._global_textures.environment )
				environment_texture = LS.Renderer._global_textures.environment;
			if( instance._nearest_reflection_probe )
			{
				if( instance._nearest_reflection_probe._texture )
					environment_texture = instance._nearest_reflection_probe._tex_id;
			}
		}

		if( environment_texture )
		{
			var tex = environment_texture.constructor === String ? LS.ResourcesManager.textures[ environment_texture ] : environment_texture;
			if( tex && tex.texture_type == GL.TEXTURE_2D )
			{
				if( tex._is_planar )
					global_flags |= environment_planar_block.flag_mask;
				else
					global_flags |= environment_2d_block.flag_mask;
			}
			else
				global_flags |= environment_cubemap_block.flag_mask;
		}

		this._samplers[ LS.Renderer.ENVIRONMENT_TEXTURE_SLOT ] = environment_texture;
	}
	else
	{
		this._samplers[ LS.Renderer.ENVIRONMENT_TEXTURE_SLOT ] = null;
	}

	//global stuff
	this._render_state.enable( render_settings );
	LS.Renderer.bindSamplers( this._samplers ); //material samplers
	LS.Renderer.bindSamplers( instance.samplers ); //RI samplers (like morph targets encoded in textures)

	//blocks for extra streams and instancing
	if( instance.vertex_buffers["colors"] )
		block_flags |= LS.Shaders.vertex_color_block.flag_mask;
	if( instance.vertex_buffers["coords1"] )
		block_flags |= LS.Shaders.coord1_block.flag_mask;
	if( instance.instanced_models && instance.instanced_models.length && gl.extensions.ANGLE_instanced_arrays ) //use instancing if supported
		block_flags |= LS.Shaders.instancing_block.flag_mask;

	//for those cases
	if(this.onRenderInstance)
		this.onRenderInstance( instance, pass );

	if( pass == SHADOW_PASS )
	{
		//global flags (like environment maps, irradiance, etc)
		block_flags |= LS.Shaders.firstpass_block.flag_mask;
		block_flags |= LS.Shaders.lastpass_block.flag_mask;
		//extract shader compiled
		var shader = shader_code.getShader( pass.name, block_flags ); //pass.name
		if(!shader)
			return false;

		//assign
		shader.uniformsArray( [ scene._uniforms, camera._uniforms, renderer_uniforms, this._uniforms, instance.uniforms ] ); //removed, why this was in?? light ? light._uniforms : null, 

		//render
		gl.disable( gl.BLEND );
		instance.render( shader, this._primitive != -1 ? this._primitive : undefined );
		renderer._rendercalls += 1;
	
		return true;
	}

	//add flags related to lights
	var lights = null;

	//ignore lights renders the object with flat illumination
	var ignore_lights = pass != COLOR_PASS || render_settings.lights_disabled || this._light_mode === Material.NO_LIGHTS;

	if( !ignore_lights )
		lights = LS.Renderer.getNearLights( instance );

	if(LS.Renderer._use_normalbuffer)
		block_flags |= LS.Shaders.normalbuffer_block.flag_mask;

	//if no lights are set or the render mode is flat
	if( !lights || lights.length == 0 || ignore_lights )
	{
		//global flags (like environment maps, irradiance, etc)
		if( !ignore_lights )
			block_flags |= global_flags;
		block_flags |= LS.Shaders.firstpass_block.flag_mask;
		block_flags |= LS.Shaders.lastpass_block.flag_mask;

		//extract shader compiled
		var shader = shader_code.getShader( null, block_flags ); //pass.name
		if(!shader)
		{
			//var shader = shader_code.getShader( "surface", block_flags );
			return false;
		}

		//assign
		shader.uniformsArray( [ scene._uniforms, camera._uniforms, renderer_uniforms, this._uniforms, instance.uniforms ] ); //removed, why this was in?? light ? light._uniforms : null, 

		shader.setUniform( "u_light_info", ShaderMaterial.nolights_vec4 );
		if( ignore_lights )
			shader.setUniform( "u_ambient_light", LS.ONES );

		//render
		instance.render( shader, this._primitive != -1 ? this._primitive : undefined );
		renderer._rendercalls += 1;
	
		return true;
	}

	var base_block_flags = block_flags;

	var uniforms_array = [ scene._uniforms, camera._uniforms, renderer_uniforms, null, this._uniforms, instance.uniforms ];

	//render multipass with several lights
	var prev_shader = null;
	for(var i = 0, l = lights.length; i < l; ++i)
	{
		var light = lights[i];
		block_flags = light.applyShaderBlockFlags( base_block_flags, pass, render_settings );

		//global
		block_flags |= global_flags;

		//shaders require to know in which pass they are (ambient is applied in the first, reflections in the last)
		if(i == 0)
			block_flags |= LS.Shaders.firstpass_block.flag_mask;
		if(i == l - 1)
			block_flags |= LS.Shaders.lastpass_block.flag_mask;

		//extract shader compiled
		var shader = shader_code.getShader( null, block_flags );
		if(!shader)
		{
			console.warn("material without pass: " + pass.name );
			continue;
		}

		//light texture like shadowmap and cookie
		LS.Renderer.bindSamplers( light._samplers );

		//light parameters (like index of pass or num passes)
		light._uniforms.u_light_info[2] = i; //num pass
		light._uniforms.u_light_info[3] = lights.length; //total passes
		uniforms_array[3] = light._uniforms;

		//assign
		if(prev_shader != shader)
			shader.uniformsArray( uniforms_array );
		else
			shader.uniforms( light._uniforms );
		prev_shader = shader;

		if(i == 1)
		{
			gl.depthMask( false );
			gl.depthFunc( gl.EQUAL );
			gl.enable( gl.BLEND );
			gl.blendFunc( gl.SRC_ALPHA, gl.ONE );
		}

		//render
		instance.render( shader, this._primitive != -1 ? this._primitive : undefined );
		renderer._rendercalls += 1;
	}

	//optimize this
	gl.disable( gl.BLEND );
	gl.depthMask( true );
	gl.depthFunc( gl.LESS );

	return true;
}

ShaderMaterial.prototype.renderPickingInstance = function( instance, render_settings, pass )
{
	//get shader code
	var shader_code = this.getShaderCode( instance, render_settings, pass );
	if(!shader_code || shader_code.constructor !== LS.ShaderCode )
		shader_code = LS.ShaderCode.getDefaultCode( instance, render_settings, pass  ); //use default shader


	//some globals
	var renderer = LS.Renderer;
	var camera = LS.Renderer._current_camera;
	var scene = LS.Renderer._current_scene;
	var model = instance.matrix;
	var node = instance.node;
	var renderer_uniforms = LS.Renderer._uniforms;

	//maybe this two should be somewhere else
	renderer_uniforms.u_model = model; 
	renderer_uniforms.u_normal_model = instance.normal_matrix; 

	//compute flags
	var block_flags = instance.computeShaderBlockFlags();

	//global stuff
	this._render_state.enable( render_settings );
	gl.disable( gl.BLEND ); //picking shouldnt use blending or colors will be wrong
	LS.Renderer.bindSamplers( this._samplers );
	LS.Renderer.bindSamplers( instance.samplers );

	//extract shader compiled
	var shader = shader_code.getShader( pass.name, block_flags );
	if(!shader)
	{
		shader_code = LS.ShaderMaterial.getDefaultPickingShaderCode();
		shader = shader_code.getShader( pass.name, block_flags );
		if(!shader)
			return false; //??!
	}

	//assign uniforms
	shader.uniformsArray( [ camera._uniforms, renderer_uniforms, this._uniforms, instance.uniforms ] );

	//set color
	var pick_color = LS.Picking.getNextPickingColor( instance.picking_node || node );
	shader.setUniform("u_material_color", pick_color );

	//render
	instance.render( shader, this._primitive != -1 ? this._primitive : undefined );
	renderer._rendercalls += 1;

	//optimize this
	gl.disable( gl.BLEND );
	gl.depthMask( true );
	gl.depthFunc( gl.LESS );

	return true;
}

//used by the editor to know which possible texture channels are available
ShaderMaterial.prototype.getTextureChannels = function()
{
	var channels = [];

	for(var i in this._properties)
	{
		var p = this._properties[i];
		if(p.is_texture)
			channels.push( p.name );
	}

	return channels;
}

/**
* Collects all the resources needed by this material (textures)
* @method getResources
* @param {Object} resources object where all the resources are stored
* @return {Texture}
*/
ShaderMaterial.prototype.getResources = function ( res )
{
	if(this.shader)
		res[ this.shader ] = LS.ShaderCode;

	var shadercode = LS.ResourcesManager.getResource( this._shader );
	if(shadercode)
		shadercode.getResources( res );

	for(var i in this._properties)
	{
		var p = this._properties[i];
		if(p.value && p.is_texture)
		{
			if(!p.value)
				continue;
			var name = null;
			if(p.value.texture)
				name = 	p.value.texture;
			else
				name = res[ p.value ];
			if(name && name.constructor === String)
				res[name] = GL.Texture;
		}
	}
	return res;
}


ShaderMaterial.prototype.getPropertyInfoFromPath = function( path )
{
	if( path.length < 1)
		return;

	var info = Material.prototype.getPropertyInfoFromPath.call(this,path);
	if(info)
		return info;

	var varname = path[0];

	var prop = this._properties_by_name[ varname ];
	if(!prop)
		return null;

	var type = prop.type;
	if(type == "float" || type == "int")
		type = "number";

	return {
		node: this._root,
		target: this,
		name: prop.name,
		value: prop.value,
		type: type
	};

	/*
	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.name != varname)
			continue;

		var type = prop.type;
		if(type == "float" || type == "int")
			type = "number";

		return {
			node: this._root,
			target: this,
			name: prop.name,
			value: prop.value,
			type: type
		};
	}
	return;
	*/
}

ShaderMaterial.prototype.setPropertyValue = function( name, value )
{
	//redirect to base material
	if( Material.prototype.setProperty.call(this,name,value) )
		return;

	var prop = this._properties_by_name[ name ];
	if(!prop)
		return null;

	if(prop.value && prop.value.set)
		prop.value.set( value );
	else
		prop.value = value;
}

//get shader code
ShaderMaterial.prototype.getShaderCode = function( instance, render_settings, pass )
{
	var shader_code = this._shader_code || LS.ResourcesManager.getResource( this._shader );
	if(!shader_code || shader_code.constructor !== LS.ShaderCode )
		return null;

	//this is in case the shader has been modified in the editor (reapplies the shadercode to the material)
	if( shader_code._version !== this._shader_version && this.processShaderCode )
	{
		shader_code._version = this._shader_version;
		this.processShaderCode();
	}

	return shader_code;
}

/**
* Takes an input texture and applies the ShaderMaterial, the result is shown on the viewport or stored in the output_texture
* The ShaderCode must contain a "fx" method.
* Similar to the method BlitTexture in Unity
* @method applyToTexture
* @param {Texture} input_texture
* @param {Texture} output_texture [optional] where to store the result, if omitted it will be shown in the viewport
*/
ShaderMaterial.prototype.applyToTexture = function( input_texture, output_texture )
{
	if( !this.shader || !input_texture )
		return false;

	//get shader code
	var shader_code = this.getShaderCode(); //special use
	if(!shader_code)
		return false;

	//extract shader compiled
	var shader = shader_code.getShader("fx");
	if(!shader)
		return false;

	//global vars
	this.fillUniforms();
	this._uniforms.u_time = LS.GlobalScene._time;
	this._uniforms.u_viewport = gl.viewport_data;

	//bind samplers
	LS.Renderer.bindSamplers( this._samplers );

	gl.disable( gl.DEPTH_TEST );
	gl.disable( gl.CULL_FACE );

	//render
	if(!output_texture)
		input_texture.toViewport( shader, this._uniforms );
	else
		output_texture.drawTo( function(){
			input_texture.toViewport( shader, this._uniforms );
		});
}

/**
* Makes one shader variable (uniform) public so it can be assigned from the engine (or edited from the editor)
* @method createUniform
* @param {String} name the property name as it should be shown
* @param {String} uniform the uniform name in the shader
* @param {String} type the var type in case we want to edit it (use LS.TYPES)
* @param {*} value
* @param {Object} options an object containing all the possible options (used mostly for widgets)
*/
ShaderMaterial.prototype.createUniform = function( name, uniform, type, value, options )
{
	if(!name || !uniform)
		throw("parameter missing in createUniform");

	//
	type = type || "Number";
	if( type.constructor !== String )
		throw("type must be string");

	//cast to typed-array
	value = value || 0;
	if(value && value.length)
		value = new Float32Array( value );//cast them always
	else
	{
		//create a value, otherwise is null
		switch (type)
		{
			case "vec2": value = vec2.create(); break;
			case "color":
			case "vec3": value = vec3.create(); break;
			case "color4":
			case "vec4": value = vec4.create(); break;
			case "mat3": value = mat3.create(); break;
			case "mat4": value = mat4.create(); break;
			default:
		}
	}

	//define info
	var prop = { name: name, uniform: uniform, value: value, type: type, is_texture: 0 };

	//mark as texture (because this need to go to the textures container so they are binded)
	if(type.toLowerCase() == "texture" || type == "sampler2D" || type == "samplerCube" || type == "sampler")
		prop.is_texture = (type == "samplerCube") ? 2 : 1;

	if(prop.is_texture)
	{
		prop.sampler = {};
		prop.type = "sampler";
		prop.sampler_slot = this._samplers.length;
		this._samplers.push( prop.sampler );
	}

	if(options)
		for(var i in options)
			prop[i] = options[i];

	this._properties.push( prop );
	this._properties_by_name[ name ] = prop;
}

/**
* Similar to createUniform but for textures, it helps specifying sampler options
* @method createSampler
* @param {String} name the property name as it should be shown
* @param {String} uniform the uniform name in the shader
* @param {Object} options an object containing all the possible options (used mostly for widgets)
* @param {String} value default value (texture name)
*/
ShaderMaterial.prototype.createSampler = function( name, uniform, sampler_options, value  )
{
	if(!name || !uniform)
		throw("parameter missing in createSampler");

	var type = "sampler";
	if( sampler_options && sampler_options.type )
		type = sampler_options.type;

	var sampler = null;

	//do not overwrite
	if( this._properties_by_name[ name ] )
	{
		var current_prop = this._properties_by_name[ name ];
		if( current_prop.type == type && current_prop.value )
			sampler = current_prop.value;
	}

	if(!sampler)
		sampler = {
			texture: value
		};

	var prop = { name: name, uniform: uniform, value: sampler, type: type, is_texture: 1, sampler_slot: -1 };

	if(sampler_options)
	{
		if(sampler_options.filter)
		{
			sampler.magFilter = sampler_options.filter;
			sampler.minFilter = sampler_options.filter;
			delete sampler_options.filter;
		}

		if(sampler_options.wrap)
		{
			sampler.wrapS = sampler_options.wrap;
			sampler.wrapT = sampler_options.wrap;
			delete sampler_options.wrap;
		}

		for(var i in sampler_options)
			sampler[i] = sampler_options[i];
	}
	prop.sampler_slot = this._samplers.length;
	this._properties.push( prop );
	this._properties_by_name[ name ] = prop;
	this._samplers.push( prop.value );
}

/**
* Creates a property for this material, this property wont be passed to the shader but can be used from source code.
* You must used this function if you want the data to be stored when serializing or changing the ShaderCode
* @method createProperty
* @param {String} name the property name as it should be shown
* @param {*} value the default value
* @param {String} type the data type (use LS.TYPES)
* @param {Object} options an object containing all the possible options (used mostly for widgets)
*/
ShaderMaterial.prototype.createProperty = function( name, value, type, options )
{
	var prop = this._properties_by_name[ name ];
	if(prop && prop.type == type) //already exist with the same type
		return;

	prop = { name: name, type: type, internal: true, value: value };
	if(options)
		for(var i in options)
			prop[i] = options[i];

	this._properties.push( prop );
	this._properties_by_name[ name ] = prop;

	Object.defineProperty( this, name, {
		get: function() { 
			var prop = this._properties_by_name[ name ]; //fetch it because could have been overwritten
			if(prop)
				return prop.value;
		},
		set: function(v) { 
			var prop = this._properties_by_name[ name ]; //fetch it because could have been overwritten
			if(!prop)
				return;
			if(prop.value && prop.value.set) //for typed arrays
				prop.value.set( v );
			else
				prop.value = v;
		},
		enumerable: false, //must not be serialized
		configurable: true //allows to overwrite this property
	});
}

/**
* returns the value of a property taking into account dynamic properties defined in the material
* @method getProperty
* @param {String} name the property name as it should be shown
* @param {*} value of the property
*/
ShaderMaterial.prototype.getProperty = function(name)
{
	var r = Material.prototype.getProperty.call( this, name );
	if(r != null)
		return;
	var p = this._properties_by_name[ name ];
	if (p)
		return p.value;
	return null;
}

/**
* Event used to inform if one resource has changed its name
* @method onResourceRenamed
* @param {Object} resources object where all the resources are stored
* @return {Boolean} true if something was modified
*/
ShaderMaterial.prototype.onResourceRenamed = function (old_name, new_name, resource)
{
	var v = Material.prototype.onResourceRenamed.call(this, old_name, new_name, resource );
	if( this.shader == old_name)
	{
		this.shader = new_name;
		v = true;
	}

	//change texture also in shader values... (this should be automatic but it is not)
	for(var i = 0; i < this._properties.length; ++i)
	{
		var p = this._properties[i];
		if(p.internal) //internal is a property that is not for the shader (is for internal computations)
			continue;

		if( !p.is_texture || !p.value )
			continue;
		if( p.value.texture != old_name )
			continue;
		p.value.texture = new_name;
		v = true;
	}

	return v;
}


ShaderMaterial.getDefaultPickingShaderCode = function()
{
	if( ShaderMaterial.default_picking_shader_code )
		return ShaderMaterial.default_picking_shader_code;
	var sc = new LS.ShaderCode();
	sc.code = LS.ShaderCode.flat_code;
	ShaderMaterial.default_picking_shader_code = sc;
	return sc;
}

//creates a material with flat color, used for debug stuff, shadowmaps, picking, etc
ShaderMaterial.createFlatMaterial = function()
{
	var material = new LS.ShaderMaterial();
	material.shader_code = LS.ShaderCode.getDefaultCode();
	return material;
}

LS.registerMaterialClass( ShaderMaterial );
LS.ShaderMaterial = ShaderMaterial;

///@FILE:../src/materials/standardMaterial.js
//modes
//- per texture
//- texture coordinates
//- vertex color and extras
//- alpha test

//StandardMaterial class **************************
/* Warning: a material is not a component, because it can be shared by multiple nodes */

/**
* StandardMaterial class improves the material class
* @namespace LS
* @class StandardMaterial
* @constructor
* @param {Object} object [optional] to configure from
*/

function StandardMaterial(o)
{
	ShaderMaterial.call(this,null); //do not pass the data object, it is called later

	this.blend_mode = LS.Blend.NORMAL;

	this.createProperty( "diffuse", new Float32Array([1.0,1.0,1.0]), "color" );
	this.createProperty( "ambient", new Float32Array([1.0,1.0,1.0]), "color" );
	this.createProperty( "emissive", new Float32Array([0,0,0,0]), "color" ); //fourth component to control if emissive is affected by albedo

	this._specular_data = vec4.fromValues( 0.1, 10.0, 0.0, 0.0 ); //specular factor, glossiness, specular_on_top
	this.specular_on_top = false;
	this.specular_on_alpha = false;

	this.backlight_factor = 0;
	this.translucency = 0;

	this.reflection_factor = 0.0;
	this.reflection_fresnel = 1.0;
	this.reflection_specular = false;

	this.createProperty( "velvet", new Float32Array([0.5,0.5,0.5]), "color" );
	this.velvet_exp = 0.0;
	this.velvet_additive = false;
	this._velvet_info = vec4.create();

	this._detail = new Float32Array([0.0, 10, 10]);

	this.normalmap_factor = 1.0;
	this.normalmap_tangent = true;
	this.bumpmap_factor = 1.0;

	this.displacementmap_factor = 0.1;
	this._texture_settings = new Uint8Array(11);

	this.use_scene_ambient = true;
	this.point_size = 1.0;

	this.createProperty( "extra", new Float32Array([0,0,0,1]), "color" ); //used in special situations

	//used to change the render state
	this.flags = {
		alpha_test: false,
		alpha_test_shadows: false,
		two_sided: false,
		flip_normals: false,
		depth_test: true,
		depth_write: true,
		ignore_lights: false,
		cast_shadows: true,
		receive_shadows: true,
		ignore_frustum: false
	};

	//used for special fx 
	this._uniforms = {
		u_material_color: this._color,
		u_ambient_color: this._ambient,
		u_emissive_color: this._emissive,
		u_specular: this._specular_data,
		u_reflection_info: vec2.create(), //factor and fresnel
		u_velvet_info: vec4.create(),
		u_normal_info: vec2.create(),
		u_detail_info: this._detail,
		u_texture_matrix: this.uvs_matrix,
		u_extra_color: this._extra,
		u_texture_settings: this._texture_settings
	};

	this._samplers = [];

	this._allows_instancing = true;
	this.needsUpdate = true;

	if(o) 
		this.configure(o);
}


Object.defineProperty( StandardMaterial.prototype, 'detail_factor', {
	get: function() { return this._detail[0]; },
	set: function(v) { this._detail[0] = v; },
	enumerable: true
});

Object.defineProperty( StandardMaterial.prototype, 'detail_scale', {
	get: function() { return this._detail.subarray(1,3); },
	set: function(v) { this._detail[1] = v[0]; this._detail[2] = v[1]; },
	enumerable: true
});

Object.defineProperty( StandardMaterial.prototype, 'emissive_extra', {
	get: function() { return this._emissive[3]; },
	set: function(v) { this._emissive[3] = v; },
	enumerable: true
});

Object.defineProperty( StandardMaterial.prototype, 'specular_factor', {
	get: function() { return this._specular_data[0]; },
	set: function(v) { 
		if( v != null && v.constructor === Number)
			this._specular_data[0] = v;
	},
	enumerable: true
});

Object.defineProperty( StandardMaterial.prototype, 'specular_gloss', {
	get: function() { return this._specular_data[1]; },
	set: function(v) { this._specular_data[1] = v; },
	enumerable: true
});

StandardMaterial.description = "This material is a general use material that allows to control the most common properties.";

StandardMaterial["@blend_mode"] = { type: "enum", values: LS.Blend };
StandardMaterial.actions = {};

StandardMaterial.DETAIL_TEXTURE = "detail";
StandardMaterial.NORMAL_TEXTURE = "normal";
StandardMaterial.DISPLACEMENT_TEXTURE = "displacement";
StandardMaterial.BUMP_TEXTURE = "bump";
StandardMaterial.REFLECTIVITY_TEXTURE = "reflectivity";
StandardMaterial.EXTRA_TEXTURE = "extra";
StandardMaterial.IRRADIANCE_TEXTURE = "irradiance";

StandardMaterial.TEXTURES_INDEX = { "color":0, "opacity":1, "ambient":2, "specular":3, "emissive":4, "detail":5, "normal":6, "displacement":7, "bump":8, "reflectivity":9, "extra":10, "environment":11 };

StandardMaterial.prototype.renderInstance = ShaderMaterial.prototype.renderInstance;
StandardMaterial.prototype.renderShadowInstance = ShaderMaterial.prototype.renderShadowInstance;
StandardMaterial.prototype.renderPickingInstance = ShaderMaterial.prototype.renderPickingInstance;

//called from LS.Renderer.processVisibleData
StandardMaterial.prototype.prepare = function( scene )
{
	var flags = this.flags;

	var render_state = this._render_state;

	if(!this._texture_settings) //HACK to fix BUG
		this._texture_settings = this._uniforms.u_texture_settings = new Uint8Array(9);

	//set flags in render state
	render_state.cull_face = !flags.two_sided;
	render_state.front_face = flags.flip_normals ? GL.CW : GL.CCW;
	render_state.depth_test = flags.depth_test;
	render_state.depth_mask = flags.depth_write;

	render_state.blend = this.blend_mode != LS.Blend.NORMAL;
	if( this.blend_mode != LS.Blend.NORMAL )
	{
		var func = LS.BlendFunctions[ this.blend_mode ];
		if(func)
		{
			render_state.blendFunc0 = func[0];
			render_state.blendFunc1 = func[1];
		}
	}

	for(var i in this.textures)
	{
		var tex = this.textures[i];
		if(!tex)
			continue;
		if(tex.index == null)
			tex.index = StandardMaterial.TEXTURES_INDEX[i];
		this._texture_settings[ tex.index ] = tex.uvs;
	}

	this._light_mode = this.flags.ignore_lights ? Material.NO_LIGHTS : 1;

	this.fillUniforms( scene ); //update uniforms
}

//options vec4: channel, degamma, transform, contrast

StandardMaterial.FLAGS = {
	COLOR_TEXTURE: 1<<1,
	OPACITY_TEXTURE: 1<<2,
	SPECULAR_TEXTURE: 1<<3,
	REFLECTIVITY_TEXTURE: 1<<4,
	AMBIENT_TEXTURE: 1<<5,
	EMISSIVE_TEXTURE: 1<<6,
	DETAIL_TEXTURE: 1<<7,
	NORMAL_TEXTURE: 1<<8,
	DISPLACEMENT_TEXTURE: 1<<9,
	EXTRA_TEXTURE: 1<<10,
	ENVIRONMENT_TEXTURE: 1<<11,
	ENVIRONMENT_CUBEMAP: 1<<12,
	IRRADIANCE_CUBEMAP: 1<<13,

	DEGAMMA_COLOR: 1<<26,
	SPEC_ON_ALPHA: 1<<27,
	SPEC_ON_TOP: 1<<28,
	ALPHA_TEST: 1<<29
}; //max is 32	



StandardMaterial.shader_codes = {};

//returns the LS.ShaderCode required to render
//here we cannot filter by light pass because this is done before applying shaderblocks
//in the StandardMaterial we cache versions of the ShaderCode according to the settings
StandardMaterial.prototype.getShaderCode = function( instance, render_settings, pass )
{
	var FLAGS = StandardMaterial.FLAGS;

	//lets check which code flags are active according to the configuration of the shader
	var code_flags = 0;
	var scene = LS.Renderer._current_scene;

	//TEXTURES
	if( this.textures.color )
	{
		code_flags |= FLAGS.COLOR_TEXTURE;
		if( this.textures.color.degamma )
			code_flags |= FLAGS.DEGAMMA_COLOR;
	}
	if( this.textures.opacity )
		code_flags |= FLAGS.OPACITY_TEXTURE;
	if( this.textures.displacement )
		code_flags |= FLAGS.DISPLACEMENT_TEXTURE;
	if( this.textures.normal )
		code_flags |= FLAGS.NORMAL_TEXTURE;
	if( this.textures.specular )
		code_flags |= FLAGS.SPECULAR_TEXTURE;
	if( this.reflection_factor > 0 )
	{
		//code_flags |= FLAGS.REFLECTION;
		if( this.textures.reflectivity )
			code_flags |= FLAGS.REFLECTIVITY_TEXTURE;
	}
	if( this.textures.emissive )
		code_flags |= FLAGS.EMISSIVE_TEXTURE;
	if( this.textures.ambient )
		code_flags |= FLAGS.AMBIENT_TEXTURE;
	if( this.textures.detail )
		code_flags |= FLAGS.DETAIL_TEXTURE;
	if( this.textures.extra )
		code_flags |= FLAGS.EXTRA_TEXTURE;
	if( this.specular_on_alpha )
		code_flags |= FLAGS.SPEC_ON_ALPHA;
	if( this.specular_on_top )
		code_flags |= FLAGS.SPEC_ON_TOP;

	//flags
	if( this.flags.alpha_test )
		code_flags |= FLAGS.ALPHA_TEST;

	//check if we already have this ShaderCode created
	var shader_code = LS.StandardMaterial.shader_codes[ code_flags ];

	//reuse shader codes when possible **************************************
	if(shader_code)
		return shader_code;

	//generate code
	var code = {
		vs_local: "",
		vs_global: "",
		fs: "",
		fs_shadows: ""
	};

	if( code_flags & FLAGS.DISPLACEMENT_TEXTURE )
		code.vs_local += "	vertex4.xyz += v_normal * texture2D( displacement_texture, v_uvs ).x * u_displacementmap_factor;\n";	

	//uvs
	var uvs_common = "\n\
	uvs[0] = IN.uv;\n\
	uvs[1] = IN.uv1;\n\
	uvs[2] = (u_texture_matrix * vec3(uvs[0],1.0)).xy;\n\
	#ifdef COORD1_BLOCK\n\
		uvs[3] = (vec3(uvs[1],1.0) * u_texture_matrix).xy;\n\
	#else\n\
		uvs[3] = uvs[2];\n\
	#endif\n\
	uvs[4] = gl_PointCoord;\n\
	";

	code.fs += uvs_common;
	code.fs_shadows += uvs_common;

	if( code_flags & FLAGS.NORMAL_TEXTURE )
	{
		code.fs += "vec2 normal_uv = getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["normal"]+"]);\n\
		vec3 normal_pixel = texture2D( normal_texture, normal_uv ).xyz;\n\
		if( u_normal_info.y > 0.0 )\n\
		{\n\
			normal_pixel.xy = vec2(1.0) - normal_pixel.xy;\n\
			normal_pixel = normalize( perturbNormal( IN.worldNormal, IN.viewDir, normal_uv, normal_pixel ));\n\
		}\n\
		else\n\
			normal_pixel = normal_pixel * 2.0 - vec3(1.0);\n\
		o.Normal = normalize( mix( o.Normal, normal_pixel, u_normal_info.x ) );\n";
	}

	if( code_flags & FLAGS.COLOR_TEXTURE )
	{
		var str = "	vec4 tex_color = texture2D( color_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["color"]+"] ) );\n";
		code.fs += str;
		code.fs_shadows += str;

		if( code_flags & FLAGS.DEGAMMA_COLOR )
			code.fs += "	tex_color.xyz = pow( tex_color.xyz, vec3(2.0) );\n";
		str = "	o.Albedo *= tex_color.xyz;\n\
	o.Alpha *= tex_color.w;\n";
		code.fs += str;
		code.fs_shadows += str;
	}
	if( code_flags & FLAGS.OPACITY_TEXTURE )
	{
		var str =  "	o.Alpha *= texture2D( opacity_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["opacity"]+"]) ).x;\n";
		code.fs += str;
		code.fs_shadows += str;
	}
	if( code_flags & FLAGS.SPECULAR_TEXTURE )
	{
		code.fs += "	vec4 spec_info = texture2D( specular_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["specular"]+"]) );\n\
	o.Specular *= spec_info.x;\n\
	o.Gloss *= spec_info.y;\n";
	}
	if( code_flags & FLAGS.REFLECTIVITY_TEXTURE )
		code.fs += "	o.Reflectivity *= texture2D( reflectivity_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["reflectivity"]+"]) ).x;\n";
	if( code_flags & FLAGS.EMISSIVE_TEXTURE )
		code.fs += "	o.Emission *= texture2D( emissive_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["emissive"]+"]) ).xyz;\n";
	if( code_flags & FLAGS.AMBIENT_TEXTURE )
		code.fs += "	o.Ambient *= texture2D( ambient_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["ambient"]+"]) ).xyz;\n";
	if( code_flags & FLAGS.DETAIL_TEXTURE )
		code.fs += "	o.Albedo += (texture2D( detail_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["detail"]+"]) * u_detail_info.yz).xyz - vec3(0.5)) * u_detail_info.x;\n";
	if( code_flags & FLAGS.EXTRA_TEXTURE )
		code.fs += "	if(u_light_info.z == 0.0) o.Extra = u_extra_color * texture2D( extra_texture, getUVs( u_texture_settings["+StandardMaterial.TEXTURES_INDEX["extra"]+"] ) );\n";

	//flags
	if( code_flags & FLAGS.ALPHA_TEST )
	{
		var str = "	if(o.Alpha < 0.01) discard;\n";
		code.fs += str;
		code.fs_shadows += str;
	}

	if( code_flags & FLAGS.SPEC_ON_TOP )
		code.fs += "	#define SPEC_ON_TOP\n";

	if( code_flags & FLAGS.SPEC_ON_ALPHA )
		code.fs += "	#define SPEC_ON_ALPHA\n";

	//if( code_flags & FLAGS.FLAT_NORMALS )
	//	flat_normals += "";

	//compile shader and cache
	shader_code = new LS.ShaderCode();
	var final_code = StandardMaterial.code_template;

	if( StandardMaterial.onShaderCode )
		StandardMaterial.onShaderCode( code, this, code_flags );

	shader_code.code = ShaderCode.replaceCode( final_code, code );
	/*
	shader_code.code = final_code.replace(/\{\{[a-zA-Z0-9_]*\}\}/g, function(v){
		v = v.replace( /[\{\}]/g, "" );
		return code[v] || "";
	});
	*/

	LS.StandardMaterial.shader_codes[ code_flags ] = shader_code;
	return shader_code;
}

StandardMaterial.prototype.fillUniforms = function( scene, options )
{
	var uniforms = this._uniforms;

	uniforms.u_reflection_info[0] = this.reflection_factor;
	uniforms.u_reflection_info[1] = this.reflection_fresnel;
	uniforms.u_backlight_factor = this.backlight_factor;
	uniforms.u_translucency = this.translucency;
	uniforms.u_normal_info[0] = this.normalmap_factor;
	uniforms.u_normal_info[1] = this.normalmap_tangent ? 1 : 0;
	uniforms.u_displacementmap_factor = this.displacementmap_factor;
	uniforms.u_velvet_info.set( this._velvet );
	uniforms.u_velvet_info[3] = this.velvet_additive ? this.velvet_exp : -this.velvet_exp;
	uniforms.u_point_size = this.point_size;

	//iterate through textures in the material
	var last_texture_slot = 0;
	var samplers = this._samplers;
	samplers.length = 0; //clear
	for(var i in this.textures) 
	{
		var sampler = this.getTextureSampler(i);
		if(!sampler)
			continue;

		var texture = null;
		
		//hardcoded textures
		if(sampler.constructor === GL.Texture)
			texture = sampler;
		else
			texture = sampler.texture;

		if(!texture)
			continue;

		if(texture.constructor === String) //name of texture
			texture = LS.ResourcesManager.textures[texture];
		else if (texture.constructor != Texture)
			continue;		
		
		if(!texture)  //loading or non-existant
			sampler = { texture: ":missing" };

		var slot = last_texture_slot;
		if( i == "environment" )
			slot = LS.Renderer.ENVIRONMENT_TEXTURE_SLOT;
		else if( i == "irradiance" )
			slot = LS.Renderer.IRRADIANCE_TEXTURE_SLOT;
		else
			last_texture_slot++;

		samplers[ slot ] = sampler;
		//var uniform_name = i + ( (!texture || texture.texture_type == gl.TEXTURE_2D) ? "_texture" : "_cubemap");
		uniforms[ i + "_texture" ] = slot;
	}
}

StandardMaterial.prototype.getTextureChannels = function()
{
	return [ Material.COLOR_TEXTURE, Material.OPACITY_TEXTURE, Material.AMBIENT_TEXTURE, Material.SPECULAR_TEXTURE, Material.EMISSIVE_TEXTURE, StandardMaterial.DETAIL_TEXTURE, StandardMaterial.NORMAL_TEXTURE, StandardMaterial.DISPLACEMENT_TEXTURE, StandardMaterial.BUMP_TEXTURE, StandardMaterial.REFLECTIVITY_TEXTURE, StandardMaterial.EXTRA_TEXTURE, Material.ENVIRONMENT_TEXTURE, StandardMaterial.IRRADIANCE_TEXTURE ];
}

/**
* assign a value to a property in a safe way
* @method setProperty
* @param {Object} object to configure from
*/
StandardMaterial.prototype.setProperty = function(name, value)
{
	//redirect to base material
	if( Material.prototype.setProperty.call(this,name,value) )
		return true;

	//regular
	switch(name)
	{
		//objects
		case "render_state":
		//numbers
		case "specular_factor":
		case "specular_gloss":
		case "backlight_factor":
		case "translucency":
		case "reflection_factor":
		case "reflection_fresnel":
		case "velvet_exp":
		case "velvet_additive":
		case "normalmap_tangent":
		case "normalmap_factor":
		case "bumpmap_factor":
		case "displacementmap_factor":
		case "detail_factor":
		case "emissive_extra":
		case "point_size":
		//strings
		case "shader_name":
		//bools
		case "specular_on_top":
		case "specular_on_alpha":
		case "normalmap_tangent":
		case "reflection_specular":
		case "use_scene_ambient":
		case "blend_mode":
			if(value !== null)
				this[name] = value; 
			break;
		case "flags":
			if(value)
			{
				for(var i in value)
					this.flags[i] = value[i];
			}
			break;
		//vectors
		case "ambient":	
		case "emissive": 
		case "velvet":
		case "extra":
		case "detail_scale":
			if(this[name].length >= value.length)
				this[name].set(value);
			break;
		default:
			return false;
	}
	return true;
}

/**
* gets all the properties and its types
* @method getPropertiesInfo
* @return {Object} object with name:type
*/
StandardMaterial.prototype.getPropertiesInfo = function()
{
	//get from the regular material
	var o = Material.prototype.getPropertiesInfo.call(this);

	//add some more
	o.merge({
		shader_name:  LS.TYPES.STRING,

		blend_mode: LS.TYPES.NUMBER,
		specular_factor: LS.TYPES.NUMBER,
		specular_gloss: LS.TYPES.NUMBER,
		backlight_factor: LS.TYPES.NUMBER,
		translucency: LS.TYPES.NUMBER,
		reflection_factor: LS.TYPES.NUMBER,
		reflection_fresnel: LS.TYPES.NUMBER,
		velvet_exp: LS.TYPES.NUMBER,
		point_size: LS.TYPES.NUMBER,

		normalmap_factor: LS.TYPES.NUMBER,
		bumpmap_factor: LS.TYPES.NUMBER,
		displacementmap_factor: LS.TYPES.NUMBER,
		emissive_extra: LS.TYPES.NUMBER,

		ambient: LS.TYPES.VEC3,
		emissive: LS.TYPES.VEC3,
		velvet: LS.TYPES.VEC3,
		extra: LS.TYPES.VEC4,
		detail_factor: LS.TYPES.NUMBER,
		detail_scale: LS.TYPES.VEC2,

		specular_on_top: LS.TYPES.BOOLEAN,
		normalmap_tangent: LS.TYPES.BOOLEAN,
		reflection_specular: LS.TYPES.BOOLEAN,
		use_scene_ambient: LS.TYPES.BOOLEAN,
		velvet_additive: LS.TYPES.BOOLEAN
	});

	return o;
}

StandardMaterial.prototype.getPropertyInfoFromPath = function( path )
{
	if( path.length < 1)
		return;

	var info = Material.prototype.getPropertyInfoFromPath.call(this,path);
	if(info)
		return info;

	var varname = path[0];
	var type;

	switch(varname)
	{
		case "blend_mode":
		case "backlight_factor":
		case "translucency":
		case "reflection_factor":
		case "reflection_fresnel":
		case "velvet_exp":
		case "normalmap_factor":
		case "bumpmap_factor":
		case "displacementmap_factor":
		case "emissive_extra":
		case "detail_factor":
		case "point_size":
			type = LS.TYPES.NUMBER; break;
		case "extra":
			type = LS.TYPES.VEC4; break;
		case "ambient":
		case "emissive":
		case "velvet":
			type = LS.TYPES.VEC3; break;
		case "detail_scale":
			type = LS.TYPES.VEC2; break;
		case "specular_on_top":
		case "specular_on_alpha":
		case "normalmap_tangent":
		case "reflection_specular":
		case "use_scene_ambient":
		case "velvet_additive":
			type = LS.TYPES.BOOLEAN; break;
		default:
			return null;
	}

	return {
		node: this._root,
		target: this,
		name: varname,
		value: this[varname],
		type: type
	};
}

StandardMaterial.clearShadersCache = function()
{
	LS.log("StandardMaterial ShaderCode cache cleared");
	StandardMaterial.shader_codes = {};
}

LS.registerMaterialClass( StandardMaterial );
LS.StandardMaterial = StandardMaterial;

//legacy
LS.Classes["newStandardMaterial"] = StandardMaterial;
//LS.newStandardMaterial = StandardMaterial;
//LS.MaterialClasses.newStandardMaterial = StandardMaterial;

//**********************************************
var UVS_CODE = "\n\
uniform int u_texture_settings[11];\n\
\n\
vec2 uvs[5];\n\
vec2 getUVs(int index)\n\
{\n\
	if(index == 0)\n\
		return uvs[0];\n\
	if(index == 1)\n\
		return uvs[1];\n\
	if(index == 2)\n\
		return uvs[2];\n\
	if(index == 3)\n\
		return uvs[3];\n\
	if(index == 4)\n\
		return uvs[4];\n\
	return uvs[0];\n\
}\n\
";

StandardMaterial.code_template = "\n\
\n\
\n\
\\default.vs\n\
\n\
precision mediump float;\n\
//global defines from blocks\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
#pragma shaderblock \"instancing\"\n\
\n\
attribute vec3 a_vertex;\n\
attribute vec3 a_normal;\n\
attribute vec2 a_coord;\n\
#ifdef BLOCK_COORD1\n\
	attribute vec2 a_coord1;\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	attribute vec4 a_color;\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
#pragma event \"vs_attributes\"\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
varying vec4 v_screenpos;\n\
\n\
//matrices\n\
#ifdef BLOCK_INSTANCING\n\
	attribute mat4 u_model;\n\
#else\n\
	uniform mat4 u_model;\n\
#endif\n\
uniform mat4 u_normal_model;\n\
uniform mat4 u_view;\n\
uniform mat4 u_viewprojection;\n\
//material\n\
uniform float u_displacementmap_factor;\n\
uniform sampler2D displacement_texture;\n\
\n\
//globals\n\
uniform float u_time;\n\
uniform vec4 u_viewport;\n\
uniform float u_point_size;\n\
\n\
#pragma shaderblock \"light\"\n\
#pragma shaderblock \"morphing\"\n\
#pragma shaderblock \"skinning\"\n\
\n\
//camera\n\
uniform vec3 u_camera_eye;\n\
uniform vec2 u_camera_planes;\n\
uniform vec3 u_camera_perspective;\n\
\n\
#pragma event \"vs_functions\"\n\
\n\
//special cases\n\
{{vs_out}}\n\
\n\
void main() {\n\
	\n\
	vec4 vertex4 = vec4(a_vertex,1.0);\n\
	v_local_pos = a_vertex;\n\
	v_local_normal = a_normal;\n\
	v_normal = a_normal;\n\
	v_uvs = a_coord;\n\
	#ifdef BLOCK_COORD1\n\
		v_uvs1 = a_coord1;\n\
	#endif\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		v_vertex_color = a_color;\n\
	#endif\n\
	\n\
	//local deforms\n\
	{{vs_local}}\n\
	applyMorphing( vertex4, v_normal );\n\
	applySkinning( vertex4, v_normal );\n\
	\n\
	//vertex\n\
	v_pos = (u_model * vertex4).xyz;\n\
	\n\
	applyLight(v_pos);\n\
	\n\
	//normal\n\
	#ifdef BLOCK_INSTANCING\n\
		v_normal = (u_model * vec4(v_normal,0.0)).xyz;\n\
	#else\n\
		v_normal = (u_normal_model * vec4(v_normal,0.0)).xyz;\n\
	#endif\n\
	//world deform\n\
	{{vs_global}}\n\
	\n\
	#pragma event \"vs_final_pass\"\n\
	\n\
	gl_Position = u_viewprojection * vec4(v_pos,1.0);\n\
	gl_PointSize = u_point_size;\n\
	v_screenpos = gl_Position;\n\
	#pragma event \"vs_final\"\n\
}\n\
\n\
\\color.fs\n\
\n\
#ifdef DRAW_BUFFERS\n\
	#extension GL_EXT_draw_buffers : require \n\
#endif\n\
\n\
precision mediump float;\n\
\n\
//global defines from blocks\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
//#pragma shaderblock \"firstPass\"\n\
//#pragma shaderblock \"lastPass\"\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
#ifdef BLOCK_COORD1\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
varying vec4 v_screenpos;\n\
\n\
//globals\n\
uniform vec4 u_viewport;\n\
uniform mat4 u_view;\n\
uniform vec3 u_camera_eye;\n\
uniform vec4 u_clipping_plane;\n\
uniform vec4 u_background_color;\n\
uniform vec4 u_material_color;\n\
\n\
uniform vec3 u_ambient_color;\n\
uniform vec4 u_emissive_color;\n\
uniform vec4 u_specular;\n\
uniform vec2 u_reflection_info;\n\
uniform vec4 u_velvet_info;\n\
uniform vec2 u_normal_info;\n\
uniform vec3 u_detail_info;\n\
uniform mat3 u_texture_matrix;\n\
uniform vec4 u_extra_color;\n\
uniform float u_backlight_factor;\n\
uniform float u_translucency;\n\
\n\
uniform sampler2D color_texture;\n\
uniform sampler2D opacity_texture;\n\
uniform sampler2D specular_texture;\n\
uniform sampler2D ambient_texture;\n\
uniform sampler2D emissive_texture;\n\
uniform sampler2D reflectivity_texture;\n\
uniform sampler2D detail_texture;\n\
uniform sampler2D normal_texture;\n\
uniform sampler2D extra_texture;\n\
\n\
\n\
#pragma snippet \"input\"\n\
#pragma shaderblock \"light\"\n\
#pragma shaderblock \"light_texture\"\n\
#pragma shaderblock \"applyReflection\"\n\
#pragma shaderblock \"normalBuffer\"\n\
\n\
#pragma snippet \"perturbNormal\"\n\
\n\
#pragma shaderblock \"extraBuffers\"\n\
\n\
"+ UVS_CODE +"\n\
\n\
void surf(in Input IN, out SurfaceOutput o)\n\
{\n\
	o.Albedo = u_material_color.xyz;\n\
	o.Alpha = u_material_color.a;\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
	o.Albedo *= IN.color.xyz;\n\
	o.Alpha *= IN.color.a;\n\
	#endif\n\
	o.Normal = normalize( v_normal );\n\
	o.Specular = u_specular.x;\n\
	o.Gloss = u_specular.y;\n\
	o.Ambient = u_ambient_color;\n\
	o.Emission = u_emissive_color.xyz;\n\
	o.Reflectivity = u_reflection_info.x;\n\
	o.Extra = u_extra_color;\n\
	\n\
	{{fs}}\n\
	\n\
	if(u_velvet_info.w > 0.0)\n\
		o.Albedo += u_velvet_info.xyz * ( 1.0 - pow( max(0.0, dot( IN.viewDir, o.Normal )), u_velvet_info.w ));\n\
	else if(u_velvet_info.w < 0.0)\n\
		o.Albedo = mix( o.Albedo, u_velvet_info.xyz, 1.0 - pow( max(0.0, dot( IN.viewDir, o.Normal )), abs(u_velvet_info.w) ) );\n\
	if(u_emissive_color.w > 0.0)\n\
		o.Emission *= o.Albedo;\n\
	o.Reflectivity *= max(0.0, pow( 1.0 - clamp(0.0, dot(IN.viewDir,o.Normal),1.0), u_reflection_info.y ));\n\
}\n\
\n\
#pragma event \"fs_functions\"\n\
#pragma snippet \"testClippingPlane\"\n\
\n\
{{fs_out}}\n\
\n\
void main() {\n\
	Input IN = getInput();\n\
	if(testClippingPlane(u_clipping_plane,IN.worldPos) < 0.0)\n\
		discard;\n\
	\n\
	IN.vertex = v_local_pos;\n\
	IN.normal = v_local_normal;\n\
	SurfaceOutput o = getSurfaceOutput();\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		IN.color = v_vertex_color;\n\
	#endif\n\
	#ifdef BLOCK_COORD1\n\
		IN.uv1 = v_uvs1;\n\
	#endif\n\
	surf(IN,o);\n\
	Light LIGHT = getLight();\n\
	applyLightTexture( IN, LIGHT );\n\
	if( !gl_FrontFacing )\n\
		o.Normal *= -1.0;\n\
	FinalLight FINALLIGHT = computeLight( o, IN, LIGHT );\n\
	FINALLIGHT.Diffuse += u_backlight_factor * max(0.0, dot(FINALLIGHT.Vector, -o.Normal));\n\
	FINALLIGHT.Diffuse = mix(FINALLIGHT.Diffuse,1.0, u_translucency);\n\
	vec4 final_color = vec4( 0.0,0.0,0.0, o.Alpha );\n\
	#ifdef SPEC_ON_ALPHA\n\
		final_color.a += FINALLIGHT.Specular;\n\
	#endif\n\
	#ifdef SPEC_ON_TOP\n\
		float specular = FINALLIGHT.Specular;\n\
		FINALLIGHT.Specular = 0.0;\n\
	#endif\n\
	final_color.xyz = applyLight( o, FINALLIGHT );\n\
	#ifdef SPEC_ON_TOP\n\
		final_color.xyz += specular * LIGHT.Color * FINALLIGHT.Shadow;\n\
	#endif\n\
	final_color = applyReflection( IN, o, final_color );\n\
	#pragma event \"fs_final_pass\"\n\
	{{fs_encode}}\n\
	#ifdef DRAW_BUFFERS\n\
	  gl_FragData[0] = final_color;\n\
	  #ifdef BLOCK_FIRSTPASS\n\
		  #ifdef BLOCK_NORMALBUFFER\n\
			  gl_FragData[1] = vec4( o.Normal * 0.5 + vec3(0.5), 1.0 );\n\
		  #else\n\
			  gl_FragData[1] = o.Extra;\n\
		  #endif\n\
	  #else\n\
		  gl_FragData[1] = vec4(0.0);\n\
	 #endif\n\
	#else\n\
	  gl_FragColor = final_color;\n\
	#endif\n\
	#pragma event \"fs_final\"\n\
}\n\
\n\
\\shadow.fs\n\
\n\
precision mediump float;\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec4 v_screenpos;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
\n\
//globals\n\
uniform vec3 u_camera_eye;\n\
uniform vec2 u_camera_planes;\n\
uniform vec4 u_clipping_plane;\n\
uniform vec4 u_material_color;\n\
\n\
uniform mat3 u_texture_matrix;\n\
\n\
"+ UVS_CODE +"\n\
\n\
\n\
uniform sampler2D color_texture;\n\
uniform sampler2D opacity_texture;\n\
\n\
#pragma snippet \"input\"\n\
#pragma snippet \"surface\"\n\
#pragma snippet \"PackDepth32\"\n\
\n\
void surf(in Input IN, out SurfaceOutput o)\n\
{\n\
	o.Albedo = u_material_color.xyz;\n\
	o.Alpha = u_material_color.a;\n\
	\n\
	{{fs_shadows}}\n\
}\n\
\n\
{{fs_shadow_out}}\n\
\n\
void main() {\n\
	Input IN = getInput();\n\
	IN.vertex = v_local_pos;\n\
	IN.normal = v_local_normal;\n\
	SurfaceOutput o = getSurfaceOutput();\n\
	surf(IN,o);\n\
	//float depth = length( IN.worldPos - u_camera_eye );\n\
	//depth = linearDepth( depth, u_camera_planes.x, u_camera_planes.y );\n\
	float depth = (v_screenpos.z / v_screenpos.w) * 0.5 + 0.5;\n\
	//depth = linearDepthNormalized( depth, u_camera_planes.x, u_camera_planes.y );\n\
	vec4 final_color;\n\
	final_color = PackDepth32(depth);\n\
	{{fs_shadow_encode}}\n\
	gl_FragColor = final_color;\n\
}\n\
\\picking.fs\n\
	precision mediump float;\n\
	varying vec4 v_screenpos;\n\
	uniform vec2 u_camera_planes;\n\
	uniform vec4 u_material_color;\n\
	void main() {\n\
		float n = u_camera_planes.x;\n\
		float f = u_camera_planes.y;\n\
		float z = v_screenpos.z / v_screenpos.w * 0.5 + 0.5;\n\
		//float linear = n * (z + 1.0) / (f + n - z * (f - n));\n\
		gl_FragColor = vec4( u_material_color.xyz, gl_FragCoord.z );\n\
	}\n\
";


/* example to inject code in the standardMaterial without having to edit it
//hooks are vs_out (out of main), vs_local (vertex4 to deform vertices localy), vs_global (v_pos to deform final position), fs_out (out of main), fs_encode (final_color before being written)
this.onStart = function()
{
  LS.StandardMaterial.onShaderCode = function(code,mat)
  {
  	code.fs_encode = "final_color.x = final_color.y;";
  }
	LS.StandardMaterial.clearShadersCache();
}
*/
///@FILE:../src/materials/surfaceMaterial.js
function SurfaceMaterial( o )
{
	Material.call( this, null );

	this.shader_name = "surface";

	this.blend_mode = LS.Blend.NORMAL;
	this._light_mode = 1;

	this.flags = {
		alpha_test: false,
		alpha_test_shadows: false,
		two_sided: false,
		flip_normals: false,
		depth_test: true,
		depth_write: true,
		ignore_lights: false,
		cast_shadows: true,
		receive_shadows: true,
		ignore_frustum: false
	};

	this._code = "void surf(in Input IN, inout SurfaceOutput o) {\n\
	o.Albedo = vec3(1.0) * IN.color.xyz;\n\
	o.Normal = IN.worldNormal;\n\
	o.Emission = vec3(0.0);\n\
	o.Specular = 1.0;\n\
	o.Gloss = 40.0;\n\
	o.Reflectivity = max(0.0, 0.5 - dot(IN.viewDir,o.Normal));\n\
	o.Alpha = IN.color.a;\n}\n";

	this._uniforms = {};
	this._samplers = [];

	this._mustUpdate = false;

	this.properties = []; //array of configurable properties
	if(o) 
		this.configure(o);

	this.computeCode();
}

SurfaceMaterial.description = "This material allows to control the surface properties by coding your own shader in GLSL.\nYou dont have to worry about the complexities of the render engine and light equation, just the surface properties for every pixel.";
SurfaceMaterial.prototype.prepare = StandardMaterial.prototype.prepare;

SurfaceMaterial.icon = "mini-icon-material.png";

SurfaceMaterial.prototype.onCodeChange = function()
{
	this._mustUpdate = true;
	//this.computeCode();
}

Object.defineProperty( SurfaceMaterial.prototype, "code", {
	enumerable: true,
	get: function() {
		return this._code;
	},
	set: function(v) {
		this._code = String(v);
		this._mustUpdate = true;
	}
});

SurfaceMaterial.prototype.getCode = function()
{
	return this._code;
}

SurfaceMaterial.prototype.computeCode = function()
{
	var uniforms_code = "";
	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var code = "uniform ";
		var prop = this.properties[i];
		switch(prop.type)
		{
			case 'number': code += "float "; break;
			case 'vec2': code += "vec2 "; break;
			case 'color':
			case 'vec3': code += "vec3 "; break;
			case 'color4':
			case 'vec4': code += "vec4 "; break;
			case 'sampler':
			case 'texture': code += "sampler2D "; break;
			case 'cubemap': code += "samplerCube "; break;
			default: 
				continue;
		}
		code += prop.name + ";\n";
		uniforms_code += code;
	}

	/*
	var lines = this._code.split("\n");
	for(var i = 0, l = lines.length; i < l; ++i )
		lines[i] = lines[i].split("//")[0]; //remove comments
	*/

	this.surf_code = uniforms_code + "\n" + this._code;

	var context = {
		fs_out: this.surf_code
	};

	var final_code = LS.ShaderCode.replaceCode( LS.SurfaceMaterial.code_template, context );
	//var final_code = LS.SurfaceMaterial.code_template.replace( /{{}}/gi, this.surf_code );

	if(!this._shadercode)
		this._shadercode = new LS.ShaderCode();
	this._shadercode.code = final_code;
	this._mustUpdate = false;
}

SurfaceMaterial.prototype.renderInstance = ShaderMaterial.prototype.renderInstance;

SurfaceMaterial.prototype.getShaderCode = function( instance, render_settings, pass )
{
	if(!this._shadercode || this._mustUpdate )
		this.computeCode();
	return this._shadercode;
}

SurfaceMaterial.prototype.fillUniforms = function( scene, options )
{
	var samplers = this._samplers;
	samplers.length = 0;

	var last_texture_slot = 0;
	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.type == "texture" || prop.type == "cubemap" || prop.type == "sampler")
		{
			var texture = prop.value;
			samplers[ last_texture_slot ] = texture;
			this._uniforms[ prop.name ] = last_texture_slot;
			last_texture_slot++;
		}
		else
			this._uniforms[ prop.name ] = prop.value;
	}

	this._uniforms.u_material_color = this._color;
}

SurfaceMaterial.prototype.configure = function(o) { 
	if(o.flags !== undefined && o.flags.constructor === Number)
		delete o["flags"]; //LEGACY
	Material.prototype.configure.call( this, o ); //it will call setProperty
	//LS.cloneObject( o, this );
	if(o.properties)
		this.properties = LS.cloneObject( o.properties );
	this.computeCode();
}

/**
* gets all the properties and its types
* @method getPropertiesInfo
* @return {Object} object with name:type
*/
SurfaceMaterial.prototype.getPropertiesInfo = function()
{
	var o = {
		color: LS.TYPES.VEC3,
		opacity: LS.TYPES.NUMBER,
		shader_name: LS.TYPES.STRING,
		blend_mode: LS.TYPES.NUMBER,
		code: LS.TYPES.STRING
	};

	//from this material
	for(var i in this.properties)
	{
		var prop = this.properties[i];
		o[prop.name] = prop.type;
	}	

	return o;
}

/**
* Event used to inform if one resource has changed its name
* @method onResourceRenamed
* @param {Object} resources object where all the resources are stored
* @return {Texture}
*/
SurfaceMaterial.prototype.onResourceRenamed = function (old_name, new_name, resource)
{
	//global
	Material.prototype.onResourceRenamed.call( this, old_name, new_name, resource );

	//specific
	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if( prop.value == old_name)
			prop.value = new_name;
	}
}


/**
* gets all the properties and its types
* @method getProperty
* @return {Object} object with name:type
*/
SurfaceMaterial.prototype.getProperty = function( name )
{
	if(this[name])
		return this[name];

	if( name.substr(0,4) == "tex_")
	{
		var tex = this.textures[ name.substr(4) ];
		if(!tex) return null;
		return tex.texture;
	}

	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.name == name)
			return prop.value;
	}	

	return null;
}

/**
* assign a value to a property in a safe way
* @method setProperty
* @param {Object} object to configure from
*/
SurfaceMaterial.prototype.setProperty = function(name, value)
{
	//redirect to base material
	if( Material.prototype.setProperty.call(this,name,value) )
		return true;

	if(name == "shader_name")
		this.shader_name = value;

	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.name != name)
			continue;
		prop.value = value;
		return true;
	}

	if( this[name] !== undefined)
		this[name] = value;
	else
		return false;
	return true;
}

/*
SurfaceMaterial.prototype.setPropertyValueFromPath = function( path, value, offset )
{
	offset = offset || 0;
	if( path.length < (offset+1) )
		return;
	return this.setProperty( path[offset], value );
}
*/

SurfaceMaterial.prototype.getPropertyInfoFromPath = function( path )
{
	if( path.length < 1)
		return;

	var info = Material.prototype.getPropertyInfoFromPath.call(this,path);
	if(info)
		return info;

	var varname = path[0];

	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.name != varname)
			continue;

		return {
			node: this._root,
			target: this,
			name: prop.name,
			value: prop.value,
			type: prop.type
		};
	}

	return;
}


SurfaceMaterial.prototype.getTextureChannels = function()
{
	var channels = [];

	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.type != "texture" && prop.type != "cubemap" && prop.type != "sampler" )
			continue;
		channels.push( prop.name );
	}

	return channels;
}

/**
* Assigns a texture to a channel
* @method setTexture
* @param {String} channel 
* @param {Texture} texture
*/
SurfaceMaterial.prototype.setTexture = function( channel, texture, sampler_options ) {
	if(!channel)
		throw("SurfaceMaterial.prototype.setTexture channel must be specified");

	var sampler = null;


	//special case
	if(channel == "environment")
		return Material.prototype.setTexture.call(this, channel, texture, sampler_options );

	for(var i = 0; i < this.properties.length; ++i)
	{
		var prop = this.properties[i];
		if(prop.type != "texture" && prop.type != "cubemap" && prop.type != "sampler")
			continue;

		if(channel && prop.name != channel) //assign to the channel or if there is no channel just to the first one
			continue;

		//assign sampler
		sampler = this.textures[ channel ];
		if(!sampler)
			sampler = this.textures[channel] = { texture: texture, uvs: "0", wrap: 0, minFilter: 0, magFilter: 0 }; //sampler

		if(sampler_options)
			for(var i in sampler_options)
				sampler[i] = sampler_options[i];

		prop.value = prop.type == "sampler" ? sampler : texture;
		break;
	}

	//preload texture
	if(texture && texture.constructor == String && texture[0] != ":")
		LS.ResourcesManager.load( texture );

	return sampler;
}

/**
* Collects all the resources needed by this material (textures)
* @method getResources
* @param {Object} resources object where all the resources are stored
* @return {Texture}
*/
SurfaceMaterial.prototype.getResources = function (res)
{
	for(var i = 0, l = this.properties.length; i < l; ++i )
	{
		var prop = this.properties[i];
		if(prop.type != "texture" && prop.type != "cubemap" && prop.type != "sampler")
			continue;
		if(!prop.value)
			continue;

		var texture = prop.type == "sampler" ? prop.value.texture : prop.value;
		if( typeof( texture ) == "string" )
			res[ texture ] = GL.Texture;
	}

	return res;
}

LS.registerMaterialClass( SurfaceMaterial );
LS.SurfaceMaterial = SurfaceMaterial;

SurfaceMaterial.code_template = "\n\
\n\
\n\
\\color.vs\n\
\n\
precision mediump float;\n\
attribute vec3 a_vertex;\n\
attribute vec3 a_normal;\n\
attribute vec2 a_coord;\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
#ifdef BLOCK_COORD1\n\
	attribute vec2 a_coord1;\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	attribute vec4 a_color;\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
\n\
//matrices\n\
uniform mat4 u_model;\n\
uniform mat4 u_normal_model;\n\
uniform mat4 u_view;\n\
uniform mat4 u_viewprojection;\n\
\n\
//globals\n\
uniform float u_time;\n\
uniform vec4 u_viewport;\n\
uniform float u_point_size;\n\
\n\
#pragma snippet \"input\"\n\
#pragma shaderblock \"light\"\n\
#pragma shaderblock \"morphing\"\n\
#pragma shaderblock \"skinning\"\n\
\n\
//camera\n\
uniform vec3 u_camera_eye;\n\
{{vs_out}}\n\
void main() {\n\
	\n\
	vec4 vertex4 = vec4(a_vertex,1.0);\n\
	v_local_pos = a_vertex;\n\
	v_local_normal = a_normal;\n\
	v_normal = a_normal;\n\
	v_uvs = a_coord;\n\
	#ifdef BLOCK_COORD1\n\
		v_uvs1 = a_coord1;\n\
	#endif\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		v_vertex_color = a_color;\n\
	#endif\n\
  \n\
  //deforms\n\
  {{vs_local}}\n\
  applyMorphing( vertex4, v_normal );\n\
  applySkinning( vertex4, v_normal );\n\
	\n\
	//vertex\n\
	v_pos = (u_model * vertex4).xyz;\n\
  \n\
  applyLight(v_pos);\n\
  \n\
	//normal\n\
	v_normal = (u_normal_model * vec4(v_normal,0.0)).xyz;\n\
    {{vs_global}}\n\
	gl_Position = u_viewprojection * vec4(v_pos,1.0);\n\
}\n\
\n\
\\color.fs\n\
\n\
precision mediump float;\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
varying vec2 v_uvs;\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
#ifdef BLOCK_COORD1\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
\n\
//globals\n\
uniform vec3 u_camera_eye;\n\
uniform vec4 u_clipping_plane;\n\
uniform float u_time;\n\
uniform vec4 u_background_color;\n\
uniform vec4 u_material_color;\n\
\n\
#pragma snippet \"input\"\n\
#pragma shaderblock \"light\"\n\
#pragma shaderblock \"applyReflection\"\n\
\n\
#pragma snippet \"perturbNormal\"\n\
#pragma snippet \"testClippingPlane\"\n\
\n\
{{fs_out}}\n\
\n\
void main() {\n\
	Input IN = getInput();\n\
	if(testClippingPlane(u_clipping_plane,IN.worldPos) < 0.0)\n\
		discard;\n\
	\n\
	IN.vertex = v_local_pos;\n\
	IN.normal = v_local_normal;\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		IN.color = v_vertex_color;\n\
	#endif\n\
	#ifdef BLOCK_COORD1\n\
		IN.uv1 = v_uvs1;\n\
	#endif\n\
	SurfaceOutput o = getSurfaceOutput();\n\
	surf(IN,o);\n\
	vec4 final_color = vec4(0.0);\n\
	Light LIGHT = getLight();\n\
	FinalLight final_light = computeLight( o, IN, LIGHT );\n\
	final_color.xyz = applyLight( o, final_light );\n\
	final_color.a = o.Alpha;\n\
	if( o.Reflectivity > 0.0 )\n\
		final_color = applyReflection( IN, o, final_color );\n\
	\n\
	gl_FragColor = final_color;\n\
}\n\
\n\
\\shadow.vs\n\
\n\
precision mediump float;\n\
attribute vec3 a_vertex;\n\
attribute vec3 a_normal;\n\
attribute vec2 a_coord;\n\
#ifdef USE_COLORS\n\
attribute vec4 a_color;\n\
#endif\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
varying vec2 v_uvs;\n\
\n\
//matrices\n\
uniform mat4 u_model;\n\
uniform mat4 u_normal_model;\n\
uniform mat4 u_view;\n\
uniform mat4 u_viewprojection;\n\
\n\
//globals\n\
uniform float u_time;\n\
uniform vec4 u_viewport;\n\
uniform float u_point_size;\n\
\n\
#pragma snippet \"input\"\n\
#pragma shaderblock \"light\"\n\
#pragma shaderblock \"morphing\"\n\
#pragma shaderblock \"skinning\"\n\
\n\
//camera\n\
uniform vec3 u_camera_eye;\n\
{{vs_out}}\n\
void main() {\n\
	\n\
	vec4 vertex4 = vec4(a_vertex,1.0);\n\
	v_local_pos = a_vertex;\n\
	v_local_normal = a_normal;\n\
	v_normal = a_normal;\n\
	v_uvs = a_coord;\n\
  \n\
  //deforms\n\
  {{vs_local}}\n\
  applyMorphing( vertex4, v_normal );\n\
  applySkinning( vertex4, v_normal );\n\
	\n\
	//vertex\n\
	v_pos = (u_model * vertex4).xyz;\n\
  \n\
  applyLight(v_pos);\n\
  \n\
	//normal\n\
	v_normal = (u_normal_model * vec4(v_normal,0.0)).xyz;\n\
    {{vs_global}}\n\
	gl_Position = u_viewprojection * vec4(v_pos,1.0);\n\
}\n\
\\shadow.fs\n\
\n\
precision mediump float;\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
\n\
//globals\n\
uniform vec3 u_camera_eye;\n\
uniform vec4 u_clipping_plane;\n\
uniform vec4 u_material_color;\n\
\n\
uniform mat3 u_texture_matrix;\n\
\n\
#pragma snippet \"input\"\n\
#pragma snippet \"surface\"\n\
#pragma snippet \"perturbNormal\"\n\
#define SHADOWMAP\n\
\n\
{{fs_out}}\n\
\n\
void main() {\n\
	Input IN = getInput();\n\
	IN.vertex = v_local_pos;\n\
	IN.normal = v_local_normal;\n\
	SurfaceOutput o = getSurfaceOutput();\n\
	surf(IN,o);\n\
	gl_FragColor = vec4(o.Albedo,o.Alpha);\n\
}\n\
";
///@FILE:../src/materials/graphMaterial.js
//allows to create shaders using the graph editor
function GraphMaterial(o)
{
	ShaderMaterial.call(this,null); //do not pass the data object, it is called later

	this.blend_mode = LS.Blend.NORMAL;

	this._filename = "";

	this._shader = "";
	this._shader_version = -1;
	this._shader_flags = 0; //?

	this._graphcode = null; //resource that contains the graph json

	this._uniforms = {};
	this._samplers = [];
	this._properties = [];
	this._properties_by_name = {};

	this._passes = {};
	this._light_mode = Material.ONE_LIGHT;
	this._primitive = -1;
	this._allows_instancing = false;

	this._version = -1;
	this._shader_version = -1;

	this._loading = false;

	if(o)
		this.configure(o);
}

GraphMaterial.icon = "mini-icon-graph.png";

GraphMaterial["@filename"] = { type:"resource", data_type: "graph" };

GraphMaterial.prototype.renderInstance = ShaderMaterial.prototype.renderInstance;
GraphMaterial.prototype.renderShadowInstance = ShaderMaterial.prototype.renderShadowInstance;
GraphMaterial.prototype.renderPickingInstance = ShaderMaterial.prototype.renderPickingInstance;

GraphMaterial.valid_properties = ["float","vec2","vec3","vec4","color","texture"];

GraphMaterial.description = "This material allows to design the shader using the build-in visual graph designer, this helps prototyping materials very fast.";

Object.defineProperty( GraphMaterial.prototype, "filename", {
	enumerable: false,
	get: function() {
		return this._filename;
	},
	set: function(v) {
		if( this._filename == v )
		{
			if( (v && this._graphcode) || (!v && !this._graphcode) )
			return;
		}

		if(v) //to avoid double slashes
			v = LS.ResourcesManager.cleanFullpath( v );
		this._filename = v;
		this._loading = false;
		this.processGraph();
	}
});

Object.defineProperty( GraphMaterial.prototype, "graphcode", {
	enumerable: false,
	get: function() {
		return this._graphcode;
	},
	set: function(v) {
		//if(this._graphcode == v) return; //disabled because sometimes we want to force reload
		this._loading = false;
		this._graphcode = v;
		if( this._graphcode )
			this._filename = this._graphcode.fullpath || this._graphcode.filename;
		else 
			this._filename = null;
		//this._graph_properties = this.serializeProperties();
		this.processGraph();
	}
});

Object.defineProperty( GraphMaterial.prototype, "graph", {
	enumerable: false,
	get: function() {
		return this._graphcode ? this._graphcode.graph : null;
	},
	set: function(v) {
		throw("graph cannot be set to a material, you must assign a graphcode instead");
	}
});

GraphMaterial.shader_codes = {};

//returns the LS.ShaderCode required to render
//here we cannot filter by light pass because this is done before applying shaderblocks
//in the StandardMaterial we cache versions of the ShaderCode according to the settings
GraphMaterial.prototype.getShaderCode = function( instance, render_settings, pass )
{
	if(!this._graphcode || !this._graphcode.getShaderCode)
		return null;
	return this._graphcode.getShaderCode(null, GraphMaterial.code_template );
}

/**
* Collects all the resources needed by this material (textures)
* @method getResources
* @param {Object} resources object where all the resources are stored
* @return {Texture}
*/
GraphMaterial.prototype.getResources = function (res)
{
	for(var i = 0; i < this._properties.length; ++i)
	{
		var p = this._properties[i];
		if(p.type == "texture" && p.value )
			res[ p.value ] = GL.Texture;
	}

	if(!this._graphcode)
		return res;

	res[ this.filename ] = true;
	if(this._graphcode)
		this._graphcode.graph.sendEventToAllNodes("getResources",res);
	
	return res;
}

GraphMaterial.prototype.serialize = function() { 
	//var o = LS.Material.prototype.serialize.apply(this);
	return {
		uid: this.uid,
		material_class: LS.getObjectClassName(this),
		filename: this.filename,
		properties: LS.cloneObject( this._properties )
	}
	return o;
}


GraphMaterial.prototype.configure = function(o) { 
	LS.cloneObject(o, this);
	if(o.properties)
	{
		this._properties = LS.cloneObject( o.properties );
		for(var i = 0; i < this._properties.length; ++i)
		{
			var p = this._properties[i];
			this._properties_by_name[ p.name ] = p;
		}
	}
	this.processGraph();
}

GraphMaterial.prototype.processGraph = function( skip_events, on_complete )
{
	if(!this._filename)
	{
		this._graphcode = null;
		return;
	}

	var that = this;
	this._graphcode = LS.ResourcesManager.getResource( this._filename );
	if(!this._graphcode && !this._loading) //must be loaded
	{
		this._loading = true;
		LS.ResourcesManager.load( this._filename, null, function( res, url ){
			this._loading = false;
			if( url != that.filename )
				return;
			if( res && res.type == GraphCode.SHADER_GRAPH )
				that._graphcode = res;
			else
				console.error("Shader Graph not found or not a Shader Graph");
			if(on_complete)
				on_complete(that);
		});
		return;
	}
}

GraphMaterial.prototype.updatePropertiesFromGraph = function()
{
	var new_properties = [];
	var new_properties_by_name = {};

	var graphcode = this._graphcode;
	if(!graphcode)
	{
		this._properties = new_properties;
		this._properties_by_name = new_properties_by_name;
		return;
	}

	//search for uniforms
	for(var i = 0; i < graphcode.properties.length; ++i)
	{
		var prop = graphcode.properties[i];

		var old_p = this._properties_by_name[ prop.name ];
		var value = old_p && old_p.type == prop.type ? old_p.value : LS.cloneObject( prop.value );

		var p = { name: prop.name, type: prop.type, widget: prop.widget || null, value: value };
		new_properties.push( p );
		new_properties_by_name[ prop.name ] = p;
	}

	this._properties = new_properties;
	this._properties_by_name = new_properties_by_name;
}

GraphMaterial.prototype.fillUniforms = function()
{
	var samp_index = 0;
	for(var i in this._properties )
	{
		var p = this._properties[i];

		if(p.type == "texture")
		{
			var index = samp_index++;
			this._samplers[ index ] = p.value || ":white";
			this._uniforms[ "u_" + p.name ] = index;
		}
		else
			this._uniforms[ "u_" + p.name ] = p.value;

	}
}

/**
* gets all the properties and its types
* @method getProperties
* @return {Object} object with name:type
*/
GraphMaterial.prototype.getProperties = function()
{
	var graph = this.graph;
	if(!graph)
		return null;

	var o = {};
	for(var i = 0; i < this._properties.length; ++i)
	{
		var p = this._properties[i];
		o[ p.name ] = p.type;
	}

	return o;
}

/**
* Event used to inform if one resource has changed its name
* @method onResourceRenamed
* @param {Object} resources object where all the resources are stored
* @return {Texture}
*/
GraphMaterial.prototype.onResourceRenamed = function (old_name, new_name, resource)
{
	//global
	Material.prototype.onResourceRenamed.call( this, old_name, new_name, resource );

	//specific
	for(var i in this._properties)
	{
		var prop = this._properties[i];
		if( prop.value == old_name)
			prop.value = new_name;
	}
}


/**
* gets all the properties and its types
* @method getProperty
* @return {Object} object with name:type
*/
GraphMaterial.prototype.getProperty = function(name)
{
	if(this[name])
		return this[name];

	if( name.substr(0,4) == "tex_")
		return this.textures[ name.substr(4) ];

	for(var i in this._properties)
	{
		var prop = this._properties[i];
		if(prop.name == name)
			return prop.value;
	}	

	return null;
}

/**
* assign a value to a property in a safe way
* @method setProperty
* @param {Object} object to configure from
*/
GraphMaterial.prototype.setProperty = function(name, value)
{
	//redirect to base material
	if( Material.prototype.setProperty.call(this,name,value) )
		return true;

	for(var i in this._properties)
	{
		var prop = this._properties[i];
		if(prop.name != name)
			continue;
		prop.value = value;
		return true;
	}

	return false;
}


GraphMaterial.prototype.getTextureChannels = function()
{
	var channels = [];

	for(var i in this._properties)
	{
		var prop = this._properties[i];
		if(prop.type != "texture" && prop.type != "cubemap")
			continue;
		channels.push(prop.name);
	}

	return channels;
}

/**
* Assigns a texture to a channel
* @method setTexture
* @param {Texture} texture
* @param {String} channel default is COLOR
*/
GraphMaterial.prototype.setTexture = function(texture, channel, uvs) {

	for(var i in this._properties)
	{
		var prop = this._properties[i];
		if(prop.type != "texture" && prop.type != "cubemap")
			continue;
		if(channel && prop.name != channel) //assign to the channel or if there is no channel just to the first one
			continue;

		prop.value = texture;
		if(this.textures)
			this.textures[channel] = texture;
		if(!channel)
			break;
	}

	if(!texture) return;
	if(texture.constructor == String && texture[0] != ":")
		ResourcesManager.load(texture);
}

LS.registerMaterialClass( GraphMaterial );
LS.GraphMaterial = GraphMaterial;

GraphMaterial.default_graph = {"last_node_id":2,"last_link_id":1,"nodes":[{"id":2,"type":"shader/phong","pos":[328,242],"size":[140,186],"flags":{},"order":0,"mode":0,"inputs":[{"name":"albedo","type":"vec3","link":null},{"name":"ambient","type":"vec3","link":null},{"name":"emission","type":"vec3","link":null},{"name":"normal","type":"vec3","link":null},{"name":"specular","type":"float","link":null},{"name":"gloss","type":"float","link":null},{"name":"reflectivity","type":"float","link":null},{"name":"alpha","type":"float","link":null},{"name":"extra","type":"vec4","link":null}],"outputs":[{"name":"out","type":"vec4","links":[1]}],"properties":{}},{"id":1,"type":"shader/fs_output","pos":[651,241],"size":[140,66],"flags":{},"order":1,"mode":0,"inputs":[{"name":"","type":"T,float,vec2,vec3,vec4","link":1}],"properties":{}}],"links":[[1,2,0,1,0,"T,float,vec2,vec3,vec4"]],"groups":[],"config":{},"version":0.4}

GraphMaterial.code_template = "\n\
\n\
\\default.vs\n\
\n\
precision mediump float;\n\
attribute vec3 a_vertex;\n\
attribute vec3 a_normal;\n\
attribute vec2 a_coord;\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
#ifdef BLOCK_COORD1\n\
	attribute vec2 a_coord1;\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	attribute vec4 a_color;\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
#pragma shaderblock \"instancing\"\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
\n\
//matrices\n\
#ifdef BLOCK_INSTANCING\n\
	attribute mat4 u_model;\n\
	varying mat4 v_model;\n\
	//varying float v_instance_id;\n\
#else\n\
	uniform mat4 u_model;\n\
#endif\n\
uniform mat4 u_normal_model;\n\
uniform mat4 u_view;\n\
uniform mat4 u_viewprojection;\n\
\n\
//globals\n\
uniform float u_time;\n\
uniform vec4 u_viewport;\n\
uniform float u_point_size;\n\
\n\
#pragma shaderblock \"morphing\"\n\
#pragma shaderblock \"skinning\"\n\
\n\
//camera\n\
uniform vec3 u_camera_eye;\n\
{{vs_out}}\n\
void main() {\n\
	\n\
	vec4 vertex4 = vec4(a_vertex,1.0);\n\
	v_local_pos = a_vertex;\n\
	v_local_normal = a_normal;\n\
	v_normal = a_normal;\n\
	v_uvs = a_coord;\n\
	#ifdef BLOCK_COORD1\n\
		v_uvs1 = a_coord1;\n\
	#endif\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		v_vertex_color = a_color;\n\
	#endif\n\
  \n\
  //deforms\n\
  {{vs_local}}\n\
  applyMorphing( vertex4, v_normal );\n\
  applySkinning( vertex4, v_normal );\n\
	\n\
	//vertex\n\
	v_pos = (u_model * vertex4).xyz;\n\
  \n\
  \n\
	//normal\n\
	#ifdef BLOCK_INSTANCING\n\
		v_normal = (u_model * vec4(v_normal,0.0)).xyz;\n\
		//v_instance_id = gl_InstanceID;\n\
	#else\n\
		v_normal = (u_normal_model * vec4(v_normal,0.0)).xyz;\n\
	#endif\n\
	{{vs_global}}\n\
	gl_Position = u_viewprojection * vec4(v_pos,1.0);\n\
}\n\
\n\
\\color.fs\n\
\n\
#ifdef DRAW_BUFFERS\n\
	#extension GL_EXT_draw_buffers : require \n\
#endif\n\
precision mediump float;\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
varying vec2 v_uvs;\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
#ifdef BLOCK_COORD1\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
#pragma shaderblock \"instancing\"\n\
\n\
//globals\n\
uniform vec3 u_camera_eye;\n\
uniform vec4 u_clipping_plane;\n\
uniform float u_time;\n\
uniform vec4 u_background_color;\n\
uniform vec4 u_material_color;\n\
#ifdef BLOCK_INSTANCING\n\
	mat4 u_model;\n\
	varying mat4 v_model;\n\
	//varying v_instance_id;\n\
#else\n\
	uniform mat4 u_model;\n\
	//float v_instance_id;\n\
#endif\n\
uniform mat4 u_normal_model;\n\
uniform mat4 u_view;\n\
uniform mat4 u_viewprojection;\n\
#pragma snippet \"input\"\n\
\n\
#pragma snippet \"testClippingPlane\"\n\
\n\
{{fs_out}}\n\
\n\
void main() {\n\
	Input IN = getInput();\n\
	if(testClippingPlane(u_clipping_plane,IN.worldPos) < 0.0)\n\
		discard;\n\
	\n\
	#ifdef BLOCK_INSTANCING\n\
		u_model = v_model;\n\
	#else\n\
		//v_instance_id = 0.0;\n\
	#endif\n\
	IN.vertex = v_local_pos;\n\
	IN.normal = v_local_normal;\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		IN.color = v_vertex_color;\n\
	#endif\n\
	#ifdef BLOCK_COORD1\n\
		IN.uv1 = v_uvs1;\n\
	#endif\n\
	vec4 _final_color = vec4(1.0);\n\
	vec4 _final_color1 = vec4(0.0);\n\
{{fs_code}}\n\
	\n\
	#ifdef DRAW_BUFFERS\n\
	  gl_FragData[0] = _final_color;\n\
	  #ifdef BLOCK_FIRSTPASS\n\
		  #ifdef BLOCK_NORMALBUFFER\n\
			  gl_FragData[1] = vec4( o.Normal * 0.5 + vec3(0.5), 1.0 );\n\
		  #else\n\
			  gl_FragData[1] = _final_color1;\n\
		  #endif\n\
	  #else\n\
		  gl_FragData[1] = vec4(0.0);\n\
	 #endif\n\
	#else\n\
	  gl_FragColor = _final_color;\n\
	#endif\n\
}\n\
\n\
\\shadow.fs\n\
\n\
precision mediump float;\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
\n\
//globals\n\
uniform vec3 u_camera_eye;\n\
uniform vec4 u_clipping_plane;\n\
uniform vec4 u_material_color;\n\
\n\
uniform mat3 u_texture_matrix;\n\
\n\
#pragma snippet \"input\"\n\
#pragma snippet \"surface\"\n\
#pragma snippet \"perturbNormal\"\n\
#define SHADOWMAP\n\
\n\
{{fs_out}}\n\
\n\
void main() {\n\
	Input IN = getInput();\n\
	IN.vertex = v_local_pos;\n\
	IN.normal = v_local_normal;\n\
	gl_FragColor = vec4(u_material_color,1.0);\n\
}\n\
";
///@FILE:../src/componentContainer.js
///@INFO: BASE
/*
	A component container is someone who could have components attached to it.
	Mostly used for SceneNodes but it could be used for other classes (like Scene or Project).
*/

/**
* ComponentContainer class allows to add component based properties to any other class
* @class ComponentContainer
* @constructor
*/
function ComponentContainer()
{
	//this function never will be called (because only the methods are attached to other classes)
	//unless you instantiate this class directly, something that would be weird
	this._components = [];
	//this._components_by_uid = {}; //TODO
}

/*
Object.defineProperty( ComponentContainer.prototype, "components", {
	enumerable: false,
	get: function() {
		return this._components;
	},
	set: function(v) {
		throw("Components cannot be set, you must use addComponent");
	}
});
*/

/**
* Adds a component to this node.
* @method configureComponents
* @param {Object} info object containing all the info from a previous serialization
*/
ComponentContainer.prototype.configureComponents = function( info )
{
	if(!info.components)
		return;

	var to_configure = [];

	//attach first, configure later
	for(var i = 0, l = info.components.length; i < l; ++i)
	{
		var comp_info = info.components[i];
		var comp_class = comp_info[0];
		var comp = null;

		//special case: this is the only component that comes by default
		if(comp_class == "Transform" && i == 0 && this.transform) 
		{
			comp = this.transform;
		}
		else
		{
			//search for the class
			var classObject = LS.Components[ comp_class ];
			if(!classObject){
				console.error("Unknown component found: " + comp_class);
				classObject = LS.MissingComponent;
			}
			//create component
			comp = new classObject();
			//attach to node
			this.addComponent( comp );

			if( comp.constructor === LS.MissingComponent )
				comp._comp_class = comp_class;
		}

		//what about configure the comp after adding it? 
		//comp.configure( comp_info[1] );
		to_configure.push( comp, comp_info[1] );

		//HACK very special case: due to requireScript
		if( comp.constructor === LS.Components.ScriptFromFile )
			comp._filename = comp_info[1].filename;

		//editor stuff
		if( comp_info[1].editor )
			comp._editor = comp_info[1].editor;

		//ensure the component uid is stored, some components may forgot about it
		if( comp_info[1].uid && comp_info[1].uid !== comp.uid )
			comp.uid = comp_info[1].uid;
	}

	//configure components now that all of them are created
	//this is to avoid problems with components that check if the node has other components and if not they create it
	for(var i = 0, l = to_configure.length; i < l; i+=2)
	{
		var comp = to_configure[i];
		var data = to_configure[i+1];
		if(LS.catch_exceptions)
		{
			try
			{
				comp.configure( data );
			}
			catch (err)
			{
				console.error("Error found when configuring node of type ", LS.getObjectClassName(comp),", skipping. All data for this component is lost.");
				console.error(err);
			}
		}
		else
			comp.configure( data );
	}
}



/**
* Adds a component to this node.
* @method serializeComponents
* @param {Object} o container where the components will be stored
*/
ComponentContainer.prototype.serializeComponents = function( o, simplified )
{
	if(!this._components)
		return;

	o.components = [];
	for(var i = 0, l = this._components.length; i < l; ++i)
	{
		var comp = this._components[i];
		if( !comp.serialize || comp.skip_serialize )
			continue;
		var obj = comp.serialize( simplified );

		//check for bad stuff inside the component
		/*
		for(var j in obj)
		{
			var v = obj[j];
			if( !v || v.constructor === Number || v.constructor === String || v.constructor === Boolean || v.constructor === Object || v.constructor === Array ) //regular data
				continue;
			obj[j] = LS.encodeObject(v);
		}
		*/

		if(comp._editor && !simplified )
			obj.editor = comp._editor;

		//enforce uid storage
		if(comp.hasOwnProperty("_uid") && !obj.uid)
			obj.uid = comp.uid;

		var object_class = null;
		if( comp.constructor === LS.MissingComponent )
			object_class = comp._comp_class;
		else
			object_class = LS.getObjectClassName( comp );

		if( LS.debug && object_class != obj.object_class )
			console.warn("Component serialize without object_class: ", object_class );
		if(!obj.object_class)
			obj.object_class = object_class; //enforce
		
		o.components.push([ object_class, obj ]);
	}
}

/**
* returns an array with all the components
* @method getComponents
* @return {Array} all the components
*/
ComponentContainer.prototype.getComponents = function( class_type )
{
	if(class_type)
	{
		var result = [];
		if(class_type.constructor === String)
			class_type = LS.Components[class_type];
		for(var i = 0, l = this._components.length; i < l; ++i)
		{
			var compo = this._components[i];
			if( compo.constructor === class_type )
				result.push( compo );
		}
		return result;
	}

	return this._components;
}

/**
* Adds a component to this node. (maybe attach would been a better name)
* @method addComponent
* @param {Object} component
* @return {Object} component added
*/
ComponentContainer.prototype.addComponent = function( component, index )
{
	if(!component)
		throw("addComponent cannot receive null");

	//you may pass a component class instead of an instance
	if(component.constructor === String)
	{
		component = LS.Components[ component ];
		if(!component)
			throw("component class not found: " + arguments[0] );
	}
	if(component.is_component)
		component = new component();
	
	//link component with container
	component._root = this;

	//must have uid
	if( !component.uid )
		component.uid = LS.generateUId("COMP-");

	//not very clean, ComponetContainer shouldnt know about LS.SceneNode, but this is more simple
	if( component.onAddedToNode)
		component.onAddedToNode(this);

	if( this._in_tree )
	{
		if( component.uid )
			this._in_tree._components_by_uid[ component.uid ] = component;
		else
			console.warn("component without uid?", component);
		if(	component.onAddedToScene )
			component.onAddedToScene( this.constructor == LS.Scene ? this : this._in_tree );
	}

	//link node with component
	if(!this._components) 
		Object.defineProperty( this, "_components", { value: [], enumerable: false });
	if(this._components.indexOf(component) != -1)
		throw("inserting the same component twice");

	if(index !== undefined && index <= this._components.length )
		this._components.splice(index,0,component);
	else
		this._components.push( component );

	LEvent.trigger( this, "componentAdded", component );

	return component;
}

/**
* Removes a component from this node.
* @method removeComponent
* @param {Object} component
*/
ComponentContainer.prototype.removeComponent = function(component)
{
	if(!component)
		throw("removeComponent cannot receive null");

	//unlink component with container
	component._root = null;

	//not very clean, ComponetContainer shouldnt know about LS.SceneNode, but this is more simple
	if( component.onRemovedFromNode )
		component.onRemovedFromNode(this);

	if( this._in_tree )
	{
		delete this._in_tree._components_by_uid[ component.uid ];
		if(component.onRemovedFromScene)
			component.onRemovedFromScene( this._in_tree );
	}

	//remove all events
	LEvent.unbindAll(this,component);

	//remove from components list
	var pos = this._components.indexOf(component);
	if(pos != -1)
		this._components.splice(pos,1);
	else
		console.warn("removeComponent: Component not found in node");

	LEvent.trigger( this, "componentRemoved", component );
}

/**
* Removes all components from this node.
* @method removeAllComponents
* @param {Object} component
*/
ComponentContainer.prototype.removeAllComponents = function()
{
	while(this._components.length)
		this.removeComponent( this._components[0] );
}


/**
* Returns if the container has a component of this class
* @method hasComponent
* @param {String|Class} component_class the component to search for, could be a string or the class itself
*/
ComponentContainer.prototype.hasComponent = function( component_class )
{
	if(!this._components)
		return false;

	//string
	if( component_class.constructor === String )
	{
		component_class = LS.Components[ component_class ];
		if(!component_class)
			return false;
	}

	//search in components
	for(var i = 0, l = this._components.length; i < l; ++i)
		if( this._components[i].constructor === component_class )
			return true;
	
	return false;
}


/**
* Returns the first component of this container that is of the same class
* @method getComponent
* @param {Object|String} component_class the class to search a component from (could be the class or the name)
* @param {Number} index [optional] if you want the Nth component of this class
*/
ComponentContainer.prototype.getComponent = function( component_class, index )
{
	if(!this._components || !component_class)
		return null;

	//convert string to class
	if( component_class.constructor === String )
	{
		//special case, locator by name (the locator starts with an underscore if it is meant to be a name)
		if( component_class[0] == "_" ) 
		{
			component_class = component_class.substr(1); //remove underscore
			for(var i = 0, l = this._components.length; i < l; ++i)
			{
				if( this._components[i].name == component_class )
				{
					if(index !== undefined && index > 0)
					{
						index--;
						continue;
					}
					return this._components[i];
				}
			}
			return false;
		}

		//otherwise the string represents the class name
		component_class = LS.Components[ component_class ];
		if(!component_class)
			return;
	}

	//search components
	for(var i = 0, l = this._components.length; i < l; ++i)
	{
		if( this._components[i].constructor === component_class )
		{
			if(index !== undefined && index > 0)
			{
				index--;
				continue;
			}
			return this._components[i];
		}
	}

	return null;
}

/**
* Returns the component with the given uid
* @method getComponentByUId
* @param {string} uid the uid to search 
*/
ComponentContainer.prototype.getComponentByUId = function(uid)
{
	if(!this._components)
		return null;
	for(var i = 0, l = this._components.length; i < l; ++i)
		if( this._components[i].uid == uid )
			return this._components[i];
	return null;
}

/**
* Returns the position in the components array of this component
* @method getIndexOfComponent
* @param {Number} position in the array, -1 if not found
*/
ComponentContainer.prototype.getIndexOfComponent = function(component)
{
	if(!this._components)
		return -1;
	return this._components.indexOf( component );
}

/**
* Returns the component at index position
* @method getComponentByIndex
* @param {Object} component
*/
ComponentContainer.prototype.getComponentByIndex = function(index)
{
	if(!this._components)
		return null;
	return this._components[index];
}

/**
* Returns a list of components matching the search, it search in the node and child nodes
* @method findComponent
* @param {Class|String} component the component class or the class name
* @return {Array} an array with all the components of the same class
*/
ComponentContainer.prototype.findComponents = function( comp_name, out )
{
	out = out || [];
	if(!comp_name)
		return out;
	if( comp_name.constructor === String )
		comp_name = LS.Components[ comp_name ];
	if(!comp_name)
		return out;

	for(var i = 0; i < this._components.length; ++i )
	{
		var comp = this._components[i];
		if( comp && comp.constructor === comp_name )
			out.push( comp );
	}

	if(this._children)
		for(var i = 0; i < this._children.length; ++i )
			this._children[i].findComponents( comp_name, out );
	return out;
}

/**
* Changes the order of a component
* @method setComponentIndex
* @param {Object} component
*/
ComponentContainer.prototype.setComponentIndex = function( component, index )
{
	if(!this._components)
		return null;
	if(index < 0)
		index = 0;
	var old_index = this._components.indexOf( component );
	if (old_index == -1)
		return;

	this._components.splice( old_index, 1 );

	/*
	if(index >= old_index)
		index--; 
	*/
	if(index >= this._components.length)
		this._components.push( component );
	else
		this._components.splice( index, 0, component );

}


/**
* Ensures this node has a component of the specified class, if not it creates one and attaches it
* @method requireComponent
* @param {Object|String} component_class the class to search a component from (could be the class or the name)
* @param {Object} data [optional] the object to configure the component from
* @return {Component} the component found or created
*/
ComponentContainer.prototype.requireComponent = function( component_class, data )
{
	if(!component_class)
		throw("no component class specified");

	//convert string to class
	if( component_class.constructor === String )
	{
		component_class = LS.Components[ component_class ];
		if(!component_class)
		{
			console.error("component class not found:", arguments[0] );
			return null;
		}
	}

	//search component
	var l = this._components.length;
	for(var i = 0; i < l; ++i)
	{
		if( this._components[i].constructor === component_class )
			return this._components[i];
	}

	var compo = new component_class();
	this.addComponent(compo, l ); //insert before the latest scripts, to avoid situations where when partially parsed the components the component is attached but not parsed yet
	if(data)
		compo.configure(data);
	return compo;
}

/**
* Ensures this node has a ScriptFromFile component of the specified script url, if not it creates one and attaches it
* @method requireScript
* @param {String} url the url to the script
* @return {Component} the ScriptFromFile component found or created
*/
ComponentContainer.prototype.requireScript = function( url )
{
	if(!url)
		throw("no url specified");

	var component_class = LS.Components.ScriptFromFile;
	url = LS.ResourcesManager.cleanFullpath( url ); //remove double slashes or spaces

	//search component
	var l = this._components.length;
	for(var i = 0; i < l; ++i)
	{
		var comp = this._components[i];
		if( comp.constructor === component_class && comp._filename == url )
			return comp;
	}

	var compo = new component_class();
	compo.filename = url;
	this.addComponent( compo, l );
	return compo;
}

/**
* executes the method with a given name in all the components
* @method processActionInComponents
* @param {String} method_name the name of the function to execute in all components (in string format)
* @param {Array} params array with every parameter that the function may need
* @param {Boolean} skip_scripts [optional] skip scripts
*/
ComponentContainer.prototype.processActionInComponents = function( method_name, params, skip_scripts )
{
	if(this._components && this._components.length)
	{
		for(var i = 0, l = this._components.length; i < l; ++i)
		{
			var comp = this._components[i];
			if( comp[method_name] && comp[method_name].constructor === Function )
			{
				if(!params || params.constructor !== Array)
					comp[method_name].call(comp, params);
				else
					comp[method_name].apply(comp, params);
				continue;
			}

			if(skip_scripts)
				continue;

			if(comp.callMethod)
				comp.callMethod( method_name, params, true );
			else if(comp._script)
				comp._script.callMethod( method_name, params, true );
		}
	}
}

/**
* executes the method with a given name in all the components and its children
* @method broadcastMessage
* @param {String} method_name the name of the function to execute in all components (in string format)
* @param {Array} params array with every parameter that the function may need
*/
ComponentContainer.prototype.broadcastMessage = function( method_name, params )
{
	this.processActionInComponents( method_name, params );

	if(this._children && this._children.length )
		for(var i = 0, l = this._children.length; i < l; ++i)
			this._children[i].broadcastMessage( method_name, params );
}


///@FILE:../src/compositePattern.js
///@INFO: BASE
/**
* CompositePattern implements the Composite Pattern, which allows to one class to contain instances of its own class
* creating a tree-like structure.
* @class CompositePattern
* @constructor
*/
function CompositePattern()
{
	//WARNING! do not add anything here, it will never be called
}

CompositePattern.prototype.compositeCtor = function()
{
	//this method is not called by SceneNode 
	this._parentNode = null;
	this._children = null;
	this._in_tree = null;
}

/* use .scene instead
CompositePattern.prototype.getScene = function()
{
	this._in_tree;
}
*/

/**
* Adds one child to this instance
* @method addChild
* @param {*} child
* @param {number} index [optional]  in which position you want to insert it, if not specified it goes to the last position
* @param {*} options [optional] data to be passed when adding it, used for special cases when moving nodes around
**/

CompositePattern.prototype.addChild = function(node, index, options)
{
	if( !node )
		throw("cannot addChild of null");

	if( node.constructor !== this.constructor )
		throw("added child must be of the same type");

	//be careful with weird recursions...
	var aux = this;
	while( aux._parentNode )
	{
		if(aux == node)
		{
			console.error("addChild: Cannot insert a node as his own child");
			return false;
		}
		aux = aux._parentNode;
	}

	//siblings
	if( node._parentNode && node._parentNode == this._parentNode && index !== undefined)
	{
		var prev_index = this._parentNode._children.indexOf( node );
		if(prev_index < index) //in case it was in the position before
			index--;
	}

	//has a parent
	if(node._parentNode)
		node._parentNode.removeChild(node, options);

	/*
	var moved = false;
	if(node._parentNode)
	{
		moved = true;
		node._onChangeParent(this, options);
		//remove from parent children
		var pos = node._parentNode._children.indexOf(node);
		if(pos != -1)
			node._parentNode._children.splice(pos,1);
	}
	*/

	//attach to this
	node._parentNode = this;
	if( !this._children )
		this._children = [node];
	else if(index == undefined)
		this._children.push(node);
	else
	{
		this._children.splice(index,0,node);
	}

	//the same as scene but we called tree to make it more generic
	var tree = this._in_tree;

	//this would never fire but just in case
	if(tree && node._in_tree && node._in_tree != tree)
		throw("Cannot add a node that belongs to another scene tree");

	//Same tree
	node._in_tree = tree;

	//overwritten from SceneNode
	if(this._onChildAdded)
		this._onChildAdded(node, options);

	LEvent.trigger(this,"childAdded", node);
	if(tree)
	{
		//added to scene tree
		LEvent.trigger(tree, "treeItemAdded", node);
		inner_recursive(node);
	}

	//recursive action
	function inner_recursive(item)
	{
		if(!item._children) return;
		for(var i in item._children)
		{
			var child = item._children[i];
			if(!child._in_tree)
			{
				//added to scene tree
				LEvent.trigger( tree, "treeItemAdded", child );
				child._in_tree = tree;
			}
			inner_recursive( child );
		}
	}
}

/**
* Removes the node from its parent (and from the scene tree)
*
* @method removeChild
* @param {Node} node this child to remove
* @param {*} param1 data passed to onChildRemoved
* @param {*} param2 data passed to onChildRemoved as second parameter
* @return {Boolean} returns true if it was found and removed
*/
CompositePattern.prototype.removeChild = function(node, param1, param2)
{
	if(!node)
		throw("cannot removeChild of null");

	if(!this._children || node._parentNode != this)
		return false;
	if( node._parentNode != this)
		return false; //not his son
	var pos = this._children.indexOf(node);
	if(pos == -1)
		return false; //not his son ?
	this._children.splice(pos,1);

	if(this._onChildRemoved)
		this._onChildRemoved(node, param1, param2);

	LEvent.trigger(this,"childRemoved", node);

	if(node._in_tree)
	{
		LEvent.trigger(node._in_tree, "treeItemRemoved", node);
		//propagate to childs
		inner_recursive(node);
	}
	node._in_tree = null;

	//recursive action to remove tree
	function inner_recursive(item)
	{
		if(!item._children)
			return;
		for(var i = 0, l = item._children.length; i < l; ++i)
		{
			var child = item._children[i];
			if(child._in_tree)
			{
				LEvent.trigger( child._in_tree, "treeItemRemoved", child );
				child._in_tree = null;
			}
			inner_recursive( child );
		}
	}

	return true;
}


/**
* Remove every child node
*
* @method removeAllChildren
*/
CompositePattern.prototype.removeAllChildren = function( param1, param2 )
{
	if(this._children)
		while( this._children.length )
			this.removeChild( this._children[0], param1, param2 );
}

/**
* Serialize the data from all the children
*
* @method serializeChildren
* @return {Array} array containing all serialized data from every children
*/
CompositePattern.prototype.serializeChildren = function( simplified )
{
	var r = [];
	if(this._children)
		for(var i in this._children)
			r.push( this._children[i].serialize( false, simplified ) ); //serialize calls serializeChildren
	return r;
}

/**
* Configure every children with the data
*
* @method configureChildren
* @return {Array} o array containing all serialized data 
*/
CompositePattern.prototype.configureChildren = function( o, components_aside )
{
	if(!o.children)
		return;

	for(var i = 0; i < o.children.length; ++i)
	{
		var c = o.children[i];

		//create instance
		var node = new this.constructor(c.id); //id is hardcoded...
		//we do this before because otherwise the event fired by addChild wont have all the info which is crucial in some cases in the editor
		if(c.uid) 
			node.uid = c.uid;
		if(c.editor) 
			node._editor = c.editor;
		//add before configure, so every child has a scene tree
		this.addChild(node);
		//we configure afterwards otherwise children wouldnt have a scene tree to bind anything
		node.configure(c, components_aside);
	}
}

/**
* Returns parent node
*
* @method getParent
* @return {SceneNode} parent node
*/
CompositePattern.prototype.getParent = function()
{
	return this._parentNode;
}

/**
* returns a list with all direct children (if you want below that use getDescendants)
* @method getChildren
* @param {Array} Original array containing the children
**/
CompositePattern.prototype.getChildren = function()
{
	return this._children || [];
}

/**
* returns the index of a child in the children array
* @method getChildIndex
* @param {SceneNode} child the child to search for
* @return {number} the index of this child in the array, if it is not inside returns -1
**/
CompositePattern.prototype.getChildIndex = function( child )
{
	return this._children ? this._children.indexOf( child ) : -1;
}

/**
* Returns the child in the index position
* @method getChildByIndex
* @param {number} index the index in the array 
* @return {SceneNode} the child in that position
**/
CompositePattern.prototype.getChildByIndex = function( index )
{
	return this._children && this._children.length > index ? this._children[ index ] : null;
}

/**
* Returns the child that matches that name
* @method getChildByName
* @param {String} name
* @return {SceneNode} the child with that name otherwise returns null;
**/
CompositePattern.prototype.getChildByName = function( name )
{
	if(!this._children)
		return null;

	for(var i = 0; i < this._children.length; ++i)
		if(this._children[i].name == name )
			return this._children[i];

	return null;
}

/**
* Returns the path name of the node (a path name is a concatenation of the name of the nodes an its ancestors: "root|parent|child"
* @method getPathName
* @return {String} the pathname
**/
CompositePattern.prototype.getPathName = function()
{
	if(!this._in_tree)
		return null;

	if(this === this._in_tree.root )
		return "";

	var path = this.name;
	var parent = this._parentNode;
	while(parent)
	{
		if(parent === this._in_tree.root )
			return path;
		path = parent.name + "|" + path;
		parent = parent._parentNode;
	}
	return null;
}

//DOM style
Object.defineProperty( CompositePattern.prototype, "childNodes", {
	enumerable: true,
	get: function() {
		return this._children || [];
	},
	set: function(v) {
		//TODO
	}
});

Object.defineProperty( CompositePattern.prototype, "parentNode", {
	enumerable: true,
	get: function() {
		return this._parentNode;
	},
	set: function(v) {
		throw("parentNode cannot be assigned, use parent.addChild(node) instead.");
	}
});

Object.defineProperty( CompositePattern.prototype, "scene", {
	enumerable: true,
	get: function() {
		return this._in_tree;
	},
	set: function(v) {
		throw("Scene cannot be set, you must use addChild in parent");
	}
});


/**
* get all nodes above this in his hierarchy (parent, parent of parent, ...)
*
* @method getAncestors
* @param {Boolean} include_itself if it must include first itself
* @return {Array} array containing all descendants
*/
CompositePattern.prototype.getAncestors = function( include_itself )
{
	var r = [];
	var aux = this._parentNode;
	if(include_itself)
		r.push(this);
	while( aux )
	{
		r.push(aux);
		aux = aux._parentNode;
	}
	return r;
}

/**
* get all nodes below this in the hierarchy (children and children of children)
*
* @method getDescendants
* @return {Array} array containing all descendants
*/
CompositePattern.prototype.getDescendants = function()
{
	if(!this._children || this._children.length == 0)
		return [];
	var r = this._children.concat();
	for(var i = 0;  i < this._children.length; ++i)
		r = r.concat( this._children[i].getDescendants() );
	return r;
}

/**
* Swaps the index in the children array so it is before 
* @method moveBefore
* @param {SceneNode} sibling [optional] allows to put before given node, otherwise it will be moved one position before of current position
* @return {number} new index
**/
CompositePattern.prototype.moveBefore = function( sibling )
{
	if(!this._parentNode || (sibling && this._parentNode !== sibling._parentNode) )
		return -1;

	var parent_children = this._parentNode._children;
	var index = parent_children.indexOf( this );
	if(index == -1)
		throw("moveBefore node not found in parent, this is impossible");

	var new_index = index - 1;
	if(sibling)
	{
		new_index = parent_children.indexOf( sibling );
		if(new_index == -1)
			return -1;
		new_index = new_index - 1; //before
	}

	if(index == new_index || new_index < 0)
		return new_index; //nothing to do

	parent_children.splice( index, 1 ); //remove
	if(new_index > index) //sibling is after
		new_index -= 1;
	parent_children.splice( new_index, 0, this); //insert
	LEvent.trigger(this._in_tree,"node_rearranged", this );
	return new_index;
}

/**
* Swaps the index in the children array so it is before 
* @method moveAfter
* @param {SceneNode} sibling [optional] allows to put after given node, otherwise it will be moved one position after current position
* @return {number} new index
**/
CompositePattern.prototype.moveAfter = function( sibling )
{
	if(!this._parentNode || (sibling && this._parentNode !== sibling._parentNode) )
		return -1;

	var parent_children = this._parentNode._children;
	var index = parent_children.indexOf( this );
	if(index == -1)
		throw("moveBefore node not found in parent, this is impossible");

	var new_index = index + 1;
	if(sibling)
	{
		new_index = parent_children.indexOf( sibling );
		if(new_index == -1)
			return -1;
		new_index = new_index + 1; //before
	}

	if( index == new_index || new_index >= parent_children.length )
		return new_index; //nothing to do

	parent_children.splice( index, 1 ); //remove
	if(new_index > index) //sibling is after
		new_index -= 1;
	parent_children.splice( new_index, 0, this); //insert
	LEvent.trigger(this._in_tree,"node_rearranged", this );
	return new_index;
}


/**
* Search for a node using a string that could be a name, a fullname or a uid
* @method findNode
* @param {String} name_or_uid
* @return {SceneNode} the node or null
**/
CompositePattern.prototype.findNode = function( name_or_uid )
{
	if(name_or_uid == "")
		return this;
	if(!name_or_uid)
		return null;
	if(name_or_uid.charAt(0) != LS._uid_prefix)
		return this.findNodeByName( name_or_uid );
	return this.findNodeByUId( name_or_uid );
}

/**
* search a node by its name
* this function gets called a lot when using animations
* @method findNodeByName
* @param {String} name
* @return {SceneNode} the node or null
**/
CompositePattern.prototype.findNodeByName = function( name )
{
	if(!name)
		return null;

	if(this.name == name)
		return this;

	var children = this._children;

	if(children)
	{
		for(var i = 0, l = children.length; i < l; ++i)
		{
			var node = children[i];
			if( node.name == name )
				return node;
			if(node._children)
			{
				var r = node.findNodeByName( name );
				if(r)
					return r;
			}
		}
	}
	return null;
}

/**
* search a node by its uid
* @method findNodeByUId
* @param {String} id
* @return {SceneNode} the node or null
**/
CompositePattern.prototype.findNodeByUId = function( uid )
{
	if(!uid)
		return null;

	if(this.uid == uid)
		return this;

	var children = this._children;

	if(children)
		for(var i = 0; i < children.length; ++i)
		{
			var node = children[i];
			if( node.uid == uid )
				return node;
			if(node._children)
			{
				var r = node.findNodeByUId(uid);
				if(r)
					return r;
			}
		}
	return null;
}

/**
* returns how many levels deep is the node in the hierarchy
* @method getHierarchyLevel
* @return {Number} the level, 0 if it is the root
**/
CompositePattern.prototype.getHierarchyLevel = function()
{
	if(!this._parentNode)
		return 0;
	return this._parentNode.getHierarchyLevel() + 1;
}

///@FILE:../src/baseComponent.js
///@INFO: BASE
/*
*  Components are elements that attach to Nodes or other objects to add functionality
*  Some important components are Transform, Light or Camera
*
*	*  ctor: must accept an optional parameter with the serialized data
*	*  onAddedToNode: triggered when added to node
*	*  onRemovedFromNode: triggered when removed from node
*	*  onAddedToScene: triggered when the node is added to the scene
*	*  onRemovedFromScene: triggered when the node is removed from the scene
*	*  serialize: returns a serialized version packed in an object
*	*  configure: recieves an object to unserialize and configure this instance
*	*  getResources: adds to the object the resources to load
*	*  _root contains the node where the component is added
*
*	*  use the LEvent system to hook events to the node or the scene
*	*  never share the same component instance between two nodes
*
*/

/**
* This is an example class for a component, should never be instantiated by itself, 
* instead components get all the methods from this class attached when the component is registered.
* Components can overwrite this methods if they want.
*
* @class  BaseComponent
* @namespace  LS
*/
function BaseComponent(o)
{
	if(o)
		this.configure(o);
}

/**
* Returns the node where this components is attached
* @method getRootNode
**/
BaseComponent.prototype.getRootNode = function()
{
	return this._root;
}

/**
* Configures the components based on an object that contains the serialized info
* @method configure
* @param {Object} o object with the serialized info
**/
BaseComponent.prototype.configure = function(o)
{ 
	if( !o )
		return;
	if( o.uid ) 
		this.uid = o.uid;
	LS.cloneObject( o, this, false, true, true ); 

	if( this.onConfigure )
		this.onConfigure( o );
}

/**
* Returns an object with all the info about this component in an object form
* @method serialize
* @return {Object} object with the serialized info
**/
BaseComponent.prototype.serialize = function()
{
	var o = LS.cloneObject(this,null,false,false,true);
	if(this.uid) //special case, not enumerable
		o.uid = this.uid;
	if(!o.object_class)
		o.object_class = LS.getObjectClassName( this );

	if( this.onSerialize )
		this.onSerialize( o );

	return o;
}

/**
* Create a clone of this node (the UID is removed to avoid collisions)
* @method clone
* @return {*} component clone
**/
BaseComponent.prototype.clone = function()
{
	var data = this.serialize();
	data.uid = null; //remove id when cloning
	var new_component = new this.constructor( data );
	return new_component;
}

/**
* To create a new property for this component adding some extra useful info to help the editor
* @method createProperty
* @param {String} name the name of the property as it will be accessed
* @param {*} value the value to assign by default to this property
* @param {String|Object} type [optional] an string identifying the type of the variable, could be "number","string","Texture","vec3","mat4", or an object with all the info
* @param {Function} setter [optional] setter function, otherwise one will be created
* @param {Function} getter [optional] getter function, otherwise one will be created
**/
BaseComponent.prototype.createProperty = function( name, value, type, setter, getter )
{
	if(this[name] !== undefined)
		return; //console.warn("createProperty: this component already has a property called " + name );

	//if we have type info, we must store it in the constructor, useful for GUIs
	if(type)
	{
		//control errors
		if(type == "String" || type == "Number" || type == "Boolean")
		{
			console.warn("createProperty: Basic types must be in lowercase -> " + type );
			type = type.toLowerCase();
		}

		if( typeof(type) == "object" )
			this.constructor[ "@" + name ] = type;
		else
			this.constructor[ "@" + name ] = { type: type };

		//is a component
		if( type == LS.TYPES.COMPONENT || LS.Components[ type ] || type.constructor.is_component || type.type == LS.TYPES.COMPONENT )
		{
			var property_root = this; //with proto is problematic, because the getters cannot do this.set (this is the proto, not the component)
			var private_name = "_" + name;
			Object.defineProperty( property_root, name, {
				get: function() { 
					if( !this[ private_name ] )
						return null;
					var scene = this._root && this._root.scene ? this._root._in_tree : LS.GlobalScene;
					return LSQ.get( this[ private_name ], null, scene );
				},
				set: function(v) { 
					if(!v)
						this[ private_name ] = v;
					else
						this[ private_name ] = v.constructor === String ? v : v.uid;
				},
				enumerable: true
				//writable: false //cannot be set to true if setter/getter
			});

			if( LS.Components[ type ] || type.constructor.is_component ) //passing component class name or component class constructor
				type = { type: LS.TYPES.COMPONENT, component_class: type.constructor === String ? type : LS.getClassName( type ) };

			if( typeof(type) == "object" )
				this.constructor[ "@" + name ] = type;
			else
				this.constructor[ "@" + name ] = { type: type };
			return;
		}
	}

	//basic type
	if(  (value === null || value === undefined || value.constructor === Number || value.constructor === String || value.constructor === Boolean) && !setter && !getter )
	{
		this[ name ] = value;
		return;
	}

	var private_name = "_" + name;

	if( Object.hasOwnProperty( this, private_name ) )
		return;


	var property_root = this; //with proto is problematic, because the getters cannot do this.set (this is the proto, not the component)

	//vector type has special type with setters and getters to avoid replacing the container during assignations
	if(value && value.constructor === Float32Array)
	{
		value = new Float32Array( value ); //clone
		this[ private_name ] = value; //this could be removed...

		//create setter
		Object.defineProperty( property_root, name, {
			get: getter || function() { return value; },
			set: setter || function(v) { value.set( v ); },
			enumerable: true
			//writable: false //cannot be set to true if setter/getter
		});
	}
	else //this is for vars that has their own setter/getter
	{
		//define private (writable because it can be overwriten with different values)
		Object.defineProperty( property_root, private_name, { 
			value: value, 
			enumerable: false,
			writable: true 
		});

		var that = this;

		//define public
		Object.defineProperty( property_root, name, {
			get: getter || function() { 
				return this[ private_name ];
			},
			set: setter || function(v) { 
				this[ private_name ] = v;
			},
			enumerable: true
			//writable: false //cannot be set to true if setter/getter
		});
	}
}

//not finished
BaseComponent.prototype.createAction = function( name, callback, options )
{
	if(!callback)
		console.error("action '" + name + "' with no callback associated. Remember to create the action after the callback is defined.");
	var safe_name = name.replace(/ /gi,"_"); //replace spaces
	this[ safe_name ] = callback;
	this.constructor["@" + safe_name ] = options || { type: "function", button_text: name, widget:"button", callback: callback };
}


/**
* Returns the locator string of this component
* @method getLocator
* @param {string} property_name [optional] you can pass the name of a property in this component
* @return {String} the locator string of this component
**/
BaseComponent.prototype.getLocator = function( property_name )
{
	if(!this._root)
		return "";
	if(property_name)
	{
		if(this[ property_name ] === undefined )
			console.warn("No property found in this component with that name:",property_name);
		return this._root.uid + "/" + this.uid + "/" + property_name;
	}
	return this._root.uid + "/" + this.uid;
}

BaseComponent.prototype.getPropertyInfoFromPath = function( path )
{
	if( !path.length )
		return null;

	var v;
	var varname = path[0];

	//to know the value of a property of the given target
	if( this.getPropertyValue )
		v = this.getPropertyValue( varname );

	//special case when the component doesnt specify any locator info but the property referenced does
	//used in TextureFX
	if (v === undefined && path.length > 1 && this[ varname ] && this[ varname ].getPropertyInfoFromPath )
	{
		var r = this[ varname ].getPropertyInfoFromPath( path.slice(1) );
		if(r)
		{
			r.node = this.root;
			return r;
		}
	}

	if( v === undefined && Object.hasOwnProperty( this, varname ) )//this[ varname ] === undefined )
		return null;

	//if we dont have a value yet then take it directly from the object
	var value = v !== undefined ? v : this[ varname ];

	var extra_info = this.constructor[ "@" + varname ];
	var type = "";
	if(extra_info)
		type = extra_info.type;
	if(!type && value !== null && value !== undefined)
	{
		if(value.constructor === String)
			type = "string";
		else if(value.constructor === Boolean)
			type = "boolean";
		else if(value.length)
			type = "vec" + value.length;
		else if(value.constructor === Number)
			type = "number";
	}

	return {
		node: this.root,
		target: this,
		name: varname,
		value: value,
		type: type
	};	
}

/**
* calls a method in all components in this node and all the children nodes
* @method broadcastMessage
* @param {String} method_name 
* @param {*} data
**/
BaseComponent.prototype.broadcastMessage = function( method_name, data )
{
	var node = this._root;
	if(!node)
		return;
	node.broadcastMessage( method_name, data );
}

/**
* returns the first component of type class_name of the SceneNode where this component belongs
* @method getComponent
* @param {String|Component} class_name the name of the class in string format or the component class itself
* @return {*} Component or null
**/
BaseComponent.prototype.getComponent = function( class_name )
{
	if(!this._root)
		return null;
	return this._root.getComponent( class_name );
}

/**
* Bind one object event to a method in this component
* @method bind
* @param {*} object the dispatcher of the event you want to react to
* @param {String} event the name of the event to bind to
* @param {Function} callback the callback to call
* @param {String|Object} type [optional] an string identifying the type of the variable, could be "number","string","Texture","vec3","mat4", or an object with all the info
* @param {Function} setter [optional] setter function, otherwise one will be created
* @param {Function} getter [optional] getter function, otherwise one will be created
**/
BaseComponent.prototype.bind = function( object, method, callback )
{
	var instance = this;
	if(arguments.length > 3 )
	{
		console.error("Component.bind cannot use a fourth parameter, all callbacks will be binded to the component");
		return;
	}

	if(!object)
	{
		console.error("Cannot bind to null.");
		return;
	}

	if(!callback)
	{
		console.error("You cannot bind a method before defining it.");
		return;
	}

	/*
	var found = false;
	for(var i in this)
	{
		if(this[i] == callback)
		{
			found = true;
			break;
		}
	}
	if(!found)
		console.warn("Callback function not found in this object, this is dangerous, remember to unbind it manually or use LEvent instead.");
	*/

	//store info about which objects have events pointing to this instance
	if(!this.__targeted_instances)
		Object.defineProperty( this,"__targeted_instances", { value: [], enumerable: false, writable: true });
	var index = this.__targeted_instances.indexOf( object );
	if(index == -1)
		this.__targeted_instances.push( object );

	return LEvent.bind( object, method, callback, instance );
}

BaseComponent.prototype.unbind = function( object, method, callback )
{
	var instance = this;

	var r = LEvent.unbind( object, method, callback, instance );

	//erase from targeted instances
	if( this.__targeted_instances )
	{
		if( !LEvent.hasBindTo( object, this ) )
			return r;

		var index = this.__targeted_instances.indexOf( object );
		if(index == -1)
			this.__targeted_instances.splice( index, 1 );
		if(this.__targeted_instances.length == 0)
			delete this.__targeted_instances;
	}

	return r;
}

BaseComponent.prototype.unbindAll = function()
{
	if( !this.__targeted_instances )
		return;

	for( var i = 0; i < this.__targeted_instances.length; ++i )
		LEvent.unbindAll( this.__targeted_instances[i], this );
	this.__targeted_instances = null; //delete dont work??
}

//called by register component to add setters and getters to registered Component Classes
BaseComponent.addExtraMethods = function( component )
{
	//add uid property
	Object.defineProperty( component.prototype, 'uid', {
		set: function( uid )
		{
			if(!uid)
				return;

			if(uid[0] != LS._uid_prefix)
			{
				console.warn("Invalid UID, renaming it to: " + uid );
				uid = LS._uid_prefix + uid;
			}

			if(uid == this._uid)
				return;
			//if( this._root && this._root._components_by_uid[ this.uid ] )
			//	delete this._root && this._root._components_by_uid[ this.uid ];
			this._uid = uid;
			//if( this._root )
			//	this._root && this._root._components_by_uid[ this.uid ] = this;
		},
		get: function(){
			return this._uid;
		},
		enumerable: false //uid better not be enumerable (so it doesnt show in the editor)
	});

	Object.defineProperty( component.prototype, 'root', {
		set: function(v)
		{
			throw("root cannot be set, call addComponent to the root");
		},
		get: function(){
			return this._root;
		},
		enumerable: false //uid better not be enumerable (so it doesnt show in the editor)
	});

	//same as root...
	Object.defineProperty( component.prototype, 'parentNode', {
		set: function()
		{
			throw("parentNode cannot be set, call addComponent to the parentNode");
		},
		get: function(){
			return this._root;
		},
		enumerable: false //uid better not be enumerable (so it doesnt show in the editor)
	});

};

LS.BaseComponent = BaseComponent;

//Used when a component is missing
function MissingComponent()
{
	this._last_serialization = null;
	this._comp_class = "";
}

MissingComponent.prototype.configure = function(o)
{
	this.uid = o.uid;
	this._last_serialization = o;
}

MissingComponent.prototype.serialize = function()
{
	return this._last_serialization;
}

LS.MissingComponent = MissingComponent;

///@FILE:../src/scene.js
///@INFO: BASE
/**
* The Scene contains all the info about the Scene and nodes
*
* @class Scene
* @constructor
*/

//event definitions for scene
EVENT.INIT = "init";
EVENT.CLEAR = "clear";
EVENT.PRECONFIGURE = "preConfigure";
EVENT.CONFIGURE = "configure";
EVENT.CHANGE = "change";
EVENT.BEFORE_LOAD = "beforeLoad";
EVENT.LOAD = "load";
EVENT.LOAD_COMPLETED = "load_completed";
EVENT.BEFORE_RELOAD = "beforeReload";
EVENT.RELOAD = "reload";
EVENT.AWAKE = "awake";
EVENT.START = "start";
EVENT.PAUSE = "pause";
EVENT.UNPAUSE = "unpause";
EVENT.FINISH = "finish";
EVENT.BEFORE_UPDATE = "before_update";
EVENT.UPDATE = "update";
EVENT.FIXED_UPDATE = "fixedUpdate";
EVENT.AFTER_UPDATE = "afterUpdate";
EVENT.COLLECT_RENDER_INSTANCES = "collectRenderInstances";
EVENT.COLLECT_PHYSIC_INSTANCES = "collectPhysicInstances";
EVENT.COLLECT_LIGHTS = "collectLights";
EVENT.COLLECT_CAMERAS = "collectCameras";
EVENT.COLLECT_DATA = "collectData";
EVENT.SERIALIZE = "serialize";
EVENT.NODE_ADDED = "nodeAdded";
EVENT.NODE_REMOVED = "nodeRemoved";
EVENT.REQUEST_FRAME = "requestFrame";



function Scene()
{
	this.uid = LS.generateUId("TREE-");

	this._state = LS.STOPPED;

	this._root = new LS.SceneNode("root");
	this._root.removeAllComponents();
	this._root._is_root  = true;
	this._root._in_tree = this;
	this._nodes = [ this._root ];
	this._nodes_by_name = { "root" : this._root };
	this._nodes_by_uid = {};
	this._nodes_by_uid[ this._root.uid ] = this._root;
	this._components_by_uid = {};

	//used to stored info when collecting from nodes
	this._uniforms = {};
	this._instances = [];
	this._lights = [];
	this._cameras = [];
	this._colliders = [];
	this._reflection_probes = [];

	//MOST OF THE PARAMETERS ARE CREATED IN init() METHOD

	//in case the resources base path are located somewhere else, if null the default is used
	this.external_repository = null;

	//work in progress, not finished yet. This will contain all the objects in cells
	this._spatial_container = new LS.SpatialContainer();

	this.external_scripts = []; //external scripts that must be loaded before initializing the scene (mostly libraries used by this scene) they do not have access to the data in the scene
	this.global_scripts = []; //scripts that are located in the resources folder and must be loaded before launching the app. they have access to the scene data
	this.preloaded_resources = {}; //resources that must be loaded, appart from the ones in the components

	//track with global animations of the scene
	this.animation = null;

	//FEATURES NOT YET FULLY IMPLEMENTED
	this._local_resources = {}; //used to store resources that go with the scene
	this.texture_atlas = null;

	this.layer_names = ["main","secondary"];

	LEvent.bind( this, "treeItemAdded", this.onNodeAdded, this );
	LEvent.bind( this, "treeItemRemoved", this.onNodeRemoved, this );

	this._shaderblock_info = null;

	this.init();
}

//LS.extendClass( Scene, ComponentContainer ); //scene could also have components

Object.defineProperty( Scene.prototype, "root", {
	enumerable: true,
	get: function() {
		return this._root;
	},
	set: function(v) {
		throw("Root node cannot be replaced");
	}
});

Object.defineProperty( Scene.prototype, "time", {
	enumerable: true,
	get: function() {
		return this._time;
	},
	set: function(v) {
		throw("Cannot set time directly");
	}
});

Object.defineProperty( Scene.prototype, "state", {
	enumerable: true,
	get: function() {
		return this._state;
	},
	set: function(v) {
		throw("Cannot set state directly, use start, finish, pause, unpause");
	}
});

Object.defineProperty( Scene.prototype, "globalTime", {
	enumerable: true,
	get: function() {
		return this._global_time;
	},
	set: function(v) {
		throw("Cannot set global_time directly");
	}
});

Object.defineProperty( Scene.prototype, "frame", {
	enumerable: true,
	get: function() {
		return this._frame;
	},
	set: function(v) {
		throw("Cannot set frame directly");
	}
});

//Some useful events
Scene.supported_events = [LS.EVENT.START,LS.EVENT.UPDATE,LS.EVENT.FINISH,LS.EVENT.CLEAR,LS.EVENT.BEFORE_RELOAD,LS.EVENT.CHANGE,EVENT.AFTER_RENDER,LS.EVENT.CONFIGURE,EVENT.NODE_ADDED,"nodeChangeParent","nodeComponentRemoved",LS.EVENT.RELOAD,"renderPicking","scene_loaded",LS.EVENT.SERIALIZE];

//methods

/**
* This initializes the content of the scene.
* Call it to clear the scene content
*
* @method init
* @return {Boolean} Returns true on success
*/
Scene.prototype.init = function()
{
	this.id = "";
	//this.materials = {}; //shared materials cache: moved to LS.RM.resources
	this.external_repository = null;

	this.global_scripts = [];
	this.external_scripts = [];
	this.preloaded_resources = {};
	this.texture_atlas = null;

	this._root.removeAllComponents();
	this._root.uid = LS.generateUId("NODE-");

	this._nodes = [ this._root ];
	this._nodes_by_name = { "root": this._root };
	this._nodes_by_uid = {};
	this._nodes_by_uid[ this._root.uid ] = this._root;
	this._components_by_uid = {};

	//WIP
	this._spatial_container.clear();

	//default components
	this.info = new LS.Components.GlobalInfo();
	this._root.addComponent( this.info );
	this._root.addComponent( new LS.Camera({ eye:[0,100,100], center:[0,0,0]} ) );
	this._root.addComponent( new LS.Light({ position: vec3.fromValues(100,100,100), target: vec3.fromValues(0,0,0) }) );

	this._frame = 0;
	this._last_collect_frame = -1; //force collect
	this._state = LS.STOPPED;

	this._time = 0;
	this._global_time = 0; //in seconds
	this._start_time = 0; //in seconds
	this._last_dt = 1/60; //in seconds
	this._must_redraw = true;
	this._fixed_update_timestep = 1/60;
	this._remaining_fixed_update_time = 0;

	if(this.selected_node) 
		delete this.selected_node;

	this.layer_names = ["main","secondary"];
	this.animation = null;
	this._local_resources = {}; //not used yet
	this.extra = {};
}

/**
* Clears the scene using the init function
* and trigger a "clear" LEvent
*
* @method clear
*/
Scene.prototype.clear = function()
{
	//remove all nodes to ensure no lose callbacks are left
	while(this._root._children && this._root._children.length)
		this._root.removeChild(this._root._children[0], false, true ); //recompute_transform, remove_components

	//remove scene components
	this._root.processActionInComponents("onRemovedFromNode",this); //send to components
	this._root.processActionInComponents("onRemovedFromScene",this); //send to components

	this._instances.length = 0;
	this._lights.length = 0;
	this._cameras.length = 0;
	this._colliders.length = 0;
	this._reflection_probes.length = 0;
	this._local_resources = {};

	this.init();
	/**
	 * Fired when the whole scene is cleared
	 *
	 * @event clear
	 */
	LEvent.trigger(this, EVENT.CLEAR );
	LEvent.trigger(this, EVENT.CHANGE );
}

/**
* Configure the Scene using an object (the object can be obtained from the function serialize)
* Inserts the nodes, configure them, and change the parameters
* ATTENTION: Destroys all previously existing info
*
* @method configure
* @param {Object} scene_info the object containing all the info about the nodes and config of the scene
*/
Scene.prototype.configure = function( scene_info )
{
	if(!scene_info || scene_info.constructor === String)
		throw("Scene configure requires object");

	LEvent.trigger(this, EVENT.PRECONFIGURE, scene_info);

	this._root.removeAllComponents(); //remove light, camera, skybox

	//this._components = [];
	//this.camera = this.light = null; //legacy

	if(scene_info.uid)
		this.uid = scene_info.uid;

	if((scene_info.object_class || scene_info.object_type) != "Scene") //legacy
		console.warn("Warning: object set to scene doesnt look like a propper one.", scene_info);

	if(scene_info.external_repository)
		this.external_repository = scene_info.external_repository;

	//extra info that the user wanted to save (comments, etc)
	if(scene_info.extra)
		this.extra = scene_info.extra;

	//this clears all the nodes
	if(scene_info.root)
	{
		this._spatial_container.clear(); // is this necessary? never used
		//two passes configure, first nodes, then components, in case a component requires a node
		var pending_components = []; 
		//components info could store data about other nodes/components, better catch it in case they are created later during the process
		LS._pending_encoded_objects = [];
		this._root.configure( scene_info.root, pending_components );
		for(var i = 0; i < pending_components.length; i+=2)
			pending_components[i].configureComponents( pending_components[i+1] );
		LS.resolvePendingEncodedObjects();
	}

	if( scene_info.global_scripts )
		this.global_scripts = scene_info.global_scripts.concat();

	if( scene_info.external_scripts )
		this.external_scripts = scene_info.external_scripts.concat();

	if( scene_info.preloaded_resources )
		this.preloaded_resources = LS.cloneObject( scene_info.preloaded_resources );

	if( scene_info.local_resources )
		this._local_resources = scene_info.local_resources;

	if( scene_info.layer_names )
		this.layer_names = scene_info.layer_names.concat();

	if( scene_info.animation )
		this.animation = new LS.Animation( scene_info.animation );

	//if(scene_info.components)
	//	this.configureComponents( scene_info );

	if( scene_info.editor )
		this._editor = scene_info.editor;

	if( scene_info.texture_atlas )
		this.texture_atlas = scene_info.texture_atlas;

	/**
	 * Fired after the scene has been configured
	 * @event configure
	 * @param {Object} scene_info contains all the info to do the configuration
	 */
	LEvent.trigger(this, EVENT.CONFIGURE,scene_info);
	LEvent.trigger(this, EVENT.AWAKE );
	/**
	 * Fired when something changes in the scene
	 * @event change
	 * @param {Object} scene_info contains all the info to do the configuration
	 */
	LEvent.trigger(this, EVENT.CHANGE );
}

/**
* Creates and object containing all the info about the scene and nodes.
* The oposite of configure.
* It calls the serialize method in every node
*
* @method serialize
* @return {Object} return a JS Object with all the scene info
*/

Scene.prototype.serialize = function( simplified  )
{
	var o = {};

	o.version = LS.Version;

	o.uid = this.uid;
	o.object_class = LS.getObjectClassName(this);

	o.external_repository = this.external_repository;

	//o.nodes = [];
	o.extra = this.extra || {};

	//add nodes
	o.root = this.root.serialize( false, simplified );

	if(this.animation)
		o.animation = this.animation.serialize();

	o.layer_names = this.layer_names.concat();
	o.global_scripts = this.global_scripts.concat();
	o.external_scripts = this.external_scripts.concat();
	o.preloaded_resources = LS.cloneObject( this.preloaded_resources );
	o.texture_atlas = LS.cloneObject( this.texture_atlas );
	o.local_resources = LS.cloneObject( this._local_resources );

	if( this._editor )
		o.editor = this._editor;

	//this.serializeComponents( o );

	/**
	 * Fired after the scene has been serialized to an object
	 * @event serialize
	 * @param {Object} object to store the persistent info
	 */
	LEvent.trigger(this,EVENT.SERIALIZE,o);

	return o;
}


/**
* Assigns a scene from a JSON description (or WBIN,ZIP)
*
* @method setFromJSON
* @param {String} data JSON object containing the scene
* @param {Function}[on_complete=null] the callback to call when the scene is ready
* @param {Function}[on_error=null] the callback to call if there is a  loading error
* @param {Function}[on_progress=null] it is called while loading the scene info (not the associated resources)
* @param {Function}[on_resources_loaded=null] it is called when all the resources had been loaded
* @param {Function}[on_scripts_loaded=null] the callback to call when the loading is complete but before assigning the scene
*/

Scene.prototype.setFromJSON = function( data, on_complete, on_error, on_progress, on_resources_loaded, on_scripts_loaded )
{
	if(!data)
		return;

	var that = this;

	if(data.constructor === String)
	{
		try
		{
			data = JSON.parse( data );
		}
		catch (err)
		{
			console.log("Error: " + err );
			return;
		}
	}

	var scripts = LS.Scene.getScriptsList( data, true );
	this.external_scripts = data.external_scripts; //must be copyed before, because it is used inside loadScripts to check origin 

	//check JSON for special scripts
	if ( scripts.length )
		this.loadScripts( scripts, function(){ inner_success( data ); }, inner_error );
	else
		inner_success( data );

	function inner_success( response )
	{
		if(on_scripts_loaded)
			on_scripts_loaded(that,response);

		that.init();
		that.configure(response);

		if( that.texture_atlas )
			LS.RM.loadTextureAtlas( that.texture_atlas, inner_preloaded_all );
		else
			inner_preloaded_all();
	}

	function inner_preloaded_all()
	{
		that.loadResources( inner_all_loaded );
		/**
		 * Fired when the scene has been loaded but before the resources
		 * @event load
		 */
		LEvent.trigger(that, EVENT.LOAD );

		if(!LS.ResourcesManager.isLoading())
			inner_all_loaded();

		if(on_complete)
			on_complete(that);
	}

	function inner_all_loaded()
	{
		if(on_resources_loaded)
			on_resources_loaded(that);
		/**
		 * Fired after all resources have been loaded
		 * @event loadCompleted
		 */
		LEvent.trigger( that, EVENT.LOAD_COMPLETED );
	}

	function inner_error(err,script_url)
	{
		console.error("Error loading script: " + script_url);
		if(on_error)
			on_error(err);
	}
}


/**
* Loads a scene from a relative url pointing to a JSON description (or WBIN,ZIP)
* Warning: this url is not passed through the LS.ResourcesManager so the url is absolute
*
* @method load
* @param {String} url where the JSON object containing the scene is stored
* @param {Function}[on_complete=null] the callback to call when the loading is complete
* @param {Function}[on_error=null] the callback to call if there is a  loading error
* @param {Function}[on_progress=null] it is called while loading the scene info (not the associated resources)
* @param {Function}[on_resources_loaded=null] it is called when all the resources had been loaded
*/

Scene.prototype.load = function( url, on_complete, on_error, on_progress, on_resources_loaded, on_loaded )
{
	if(!url)
		return;

	var that = this;

	var extension = LS.ResourcesManager.getExtension( url );
	var format_info = LS.Formats.getFileFormatInfo( extension );
	if(!format_info) //hack, to avoid errors
		format_info = { dataType: "json" };

	//request scene file using our own library
	LS.Network.request({
		url: url,
		nocache: true,
		dataType: extension == "json" ? "json" : (format_info.dataType || "text"), //datatype of json is text...
		success: extension == "json" ? inner_json_loaded : inner_data_loaded,
		progress: on_progress,
		error: inner_error
	});

	this._state = LS.LOADING;

	/**
	 * Fired before loading scene
	 * @event beforeLoad
	 */
	LEvent.trigger(this,EVENT.BEFORE_LOAD);

	function inner_data_loaded( response )
	{
		//process whatever we loaded (in case it is a pack)
		LS.ResourcesManager.processResource( url, response, null, inner_data_processed );
	}

	function inner_data_processed( pack_url, pack )
	{
		if(!pack)
			return;

		//for DAEs
		if( pack.object_class == "Scene")
		{
			inner_json_loaded( pack );
			return;
		}
		else if( pack.object_class == "SceneNode") 
		{
			var root = pack.serialize();
			inner_json_loaded( { object_class: "Scene", root: root } );
			return;
		}

		//for packs
		if( !pack._data || !pack._data["scene.json"] )
		{
			console.error("Error loading PACK, doesnt look like it has a valid scene inside");
			return;
		}
		var scene = JSON.parse( pack._data["scene.json"] );

		inner_json_loaded( scene );
	}

	function inner_json_loaded( response )
	{
		if( response.constructor !== Object )
			throw("response must be object");

		var scripts = LS.Scene.getScriptsList( response, true );

		//check JSON for special scripts
		if ( scripts.length )
			that.loadScripts( scripts, function(){ inner_success(response); }, on_error );
		else
			inner_success( response );
	}

	function inner_success( response )
	{
		if(on_loaded)
			on_loaded(that, url);

		that.init();
		that.configure(response);

		if(on_complete)
			on_complete(that, url);

		that.loadResources( inner_all_loaded );
		LEvent.trigger(that, EVENT.LOAD );

		if(!LS.ResourcesManager.isLoading())
			inner_all_loaded();
	}

	function inner_all_loaded()
	{
		if(on_resources_loaded)
			on_resources_loaded(that, url);
		LEvent.trigger(that, EVENT.LOAD_COMPLETED );
	}

	function inner_error(e)
	{
		var err_code = (e && e.target) ? e.target.status : 0;
		console.warn("Error loading scene: " + url + " -> " + err_code);
		if(on_error)
			on_error(url, err_code, e);
	}
}


/**
* Loads a scene from a relative url pointing to a JSON description (or WBIN,ZIP)
* It uses the resources folder as the root folder (in comparison with the regular load function)
*
* @method loadFromResources
* @param {String} url where the JSON object containing the scene is stored
* @param {Function}[on_complete=null] the callback to call when the loading is complete
* @param {Function}[on_error=null] the callback to call if there is a  loading error
* @param {Function}[on_progress=null] it is called while loading the scene info (not the associated resources)
* @param {Function}[on_resources_loaded=null] it is called when all the resources had been loaded
*/
Scene.prototype.loadFromResources = function( url, on_complete, on_error, on_progress, on_resources_loaded )
{
	url = LS.ResourcesManager.getFullURL( url );
	this.load( url, on_complete, on_error, on_progress, on_resources_loaded );
}



/**
* Static method, returns a list of all the scripts that must be loaded, in order and with the full path
*
* @method Scene.getScriptsList
* @param {Scene|Object} scene the object containing info about the scripts (could be a scene or a JSON object)
* @param {Boolean} allow_local if we allow local resources
* @param {Boolean} full_paths if true it will return the full path to every resource
*/
Scene.getScriptsList = function( scene, allow_local, full_paths )
{
	if(!scene)
		throw("Scene.getScriptsList: scene cannot be null");

	var scripts = [];
	if ( scene.external_scripts && scene.external_scripts.length )
		scripts = scripts.concat( scene.external_scripts );
	if ( scene.global_scripts && scene.global_scripts.length )
	{
		for(var i in scene.global_scripts)
		{
			var script_url = scene.global_scripts[i];
			if(!script_url || LS.ResourcesManager.getExtension( script_url ) != "js" )
				continue;

			var res = LS.ResourcesManager.getResource( script_url );
			if(res)
			{
				if( allow_local )
					script_url = LS.ResourcesManager.cleanFullpath( script_url );
			}

			if(full_paths)
				script_url = LS.ResourcesManager.getFullURL( script_url );

			scripts.push( script_url );
		}
	}

	scripts = scripts.map(function(a){ return a.trim(); }); //careful with spaces

	return scripts;
}

//reloads external and global scripts taking into account if they come from wbins
Scene.prototype.loadScripts = function( scripts, on_complete, on_error, force_reload )
{
	if(!LS.allow_scripts)
	{
		console.error("LiteScene.allow_scripts is set to false, so scripts imported into this scene are ignored.");
		if(on_complete)
			on_complete();
		return;
	}

	//get a list of scripts (they cannot be fullpaths)
	scripts = scripts || LS.Scene.getScriptsList( this, true );

	if(!scripts.length)
	{
		if(on_complete)
			on_complete();
		return;
	}

	if( LS._block_scripts )
	{
		console.error("Safety: LS.block_scripts enabled, cannot request script");
		return;
	}

	//All this is to allow the use of scripts that are in memory (they came packed inside a WBin with the scene)
	var final_scripts = [];
	var revokable = [];

	for(var i in scripts)
	{
		var script_url = scripts[i];
		var is_external = this.external_scripts.indexOf( script_url ) != -1;
		if( !is_external ) //comes from scene.external_scripts
		{
			var res = LS.ResourcesManager.getResource( script_url );
			if(!res || force_reload)
			{
				var final_url = LS.ResourcesManager.getFullURL( script_url );
				final_scripts.push( final_url );
				continue;
			}

			//we use blobs because we have the data locally but we need to load it from an url (the script loader works like that)
			var blob = new Blob([res.data],{encoding:"UTF-8", type: 'text/plain;charset=UTF-8'});
			var objectURL = URL.createObjectURL( blob );
			final_scripts.push( objectURL );
			revokable.push( objectURL );
		}
		else
			final_scripts.push( script_url );
	}

	LS.Network.requestScript( final_scripts, inner_complete, on_error );

	function inner_complete()
	{
		//revoke urls created
		for(var i in revokable)
			URL.revokeObjectURL( revokable[i] );

		if(on_complete)
			on_complete();
	}
}

//used to ensure that components use the right class when the class comes from a global script
Scene.prototype.checkComponentsCodeModification = function()
{
	for(var i = 0; i < this._nodes.length; ++i )
	{
		//current components
		var node = this._nodes[i];
		for(var j = 0; j < node._components.length; ++j)
		{
			var compo = node._components[j];
			var class_name = LS.getObjectClassName( compo );
			if( compo.constructor == LS.MissingComponent )
				class_name = compo._comp_class;

			var current_class = LS.Components[ class_name ];
			if( !current_class || current_class == compo.constructor ) //already uses the right class
				continue;

			//replace class instance in-place
			var data = compo.serialize();
			var new_compo = new current_class( data );
			var index = node.getIndexOfComponent( compo );
			node.removeComponent( compo );
			node.addComponent( new_compo, index );
			console.log("Class replaced: " + class_name );
		}
	}
}

Scene.prototype.appendScene = function(scene)
{
	//clone: because addNode removes it from scene.nodes array
	var nodes = scene.root.childNodes;

	/*
	//bring materials
	for(var i in scene.materials)
		this.materials[i] = scene.materials[i];
	*/
	
	//add every node one by one
	for(var i in nodes)
	{
		var node = nodes[i];
		var new_node = new LS.SceneNode( node.id );
		this.root.addChild( new_node );
		new_node.configure( node.constructor == LS.SceneNode ? node.serialize() : node  );
	}
}

Scene.prototype.getCamera = function()
{
	var camera = this._root.camera;
	if(camera) 
		return camera;

	if(this._cameras && this._cameras.length)
		return this._cameras[0];

	this.collectData(); //slow
	return this._cameras[0];
}

/**
* Returns an array with all the cameras enabled in the scene
*
* @method getActiveCameras
* @param {boolean} force [optional] if you want to collect the cameras again, otherwise it returns the last ones collected
* @return {Array} cameras
*/
Scene.prototype.getActiveCameras = function( force )
{
	if(force)
		LEvent.trigger(this, EVENT.COLLECT_CAMERAS, this._cameras );
	return this._cameras;
}

/**
* Returns an array with all the cameras in the scene (even if they are disabled)
*
* @method getAllCameras
* @return {Array} cameras
*/
Scene.prototype.getAllCameras = function()
{
	var cameras = [];
	for(var i = 0; i < this._nodes.length; ++i)
	{
		var node = this._nodes[i];
		var node_cameras = node.getComponents( LS.Components.Camera );
		if(node_cameras && node_cameras.length)
			cameras = cameras.concat( node_cameras );
	}
	return cameras;
}

Scene.prototype.getLight = function()
{
	return this._root.light;
}

/**
* Returns an array with all the lights enabled in the scene
*
* @method getActiveLights
* @param {boolean} force [optional] if you want to collect the lights again, otherwise it returns the last ones collected
* @return {Array} lights
*/
Scene.prototype.getActiveLights = function( force )
{
	if(force)
		LEvent.trigger(this, EVENT.COLLECT_LIGHTS, this._lights );
	return this._lights;
}

Scene.prototype.onNodeAdded = function(e,node)
{
	//remove from old scene
	if(node._in_tree && node._in_tree != this)
		throw("Cannot add a node from other scene, clone it");

	if( node._name && !this._nodes_by_name[ node._name ] )
		this._nodes_by_name[ node._name ] = node;

	/*
	//generate unique id
	if(node.id && node.id != -1)
	{
		if(this._nodes_by_id[node.id] != null)
			node.id = node.id + "_" + (Math.random() * 1000).toFixed(0);
		this._nodes_by_id[node.id] = node;
	}
	*/

	//store by uid
	if(!node.uid || this._nodes_by_uid[ node.uid ])
		node._uid = LS.generateUId("NODE-");
	//if( this._nodes_by_uid[ node.uid ] )
	//	console.warn("There are more than one node with the same UID: ", node.uid );
	this._nodes_by_uid[ node.uid ] = node;

	//store nodes linearly
	this._nodes.push(node);

	node.processActionInComponents("onAddedToScene",this); //send to components
	for(var i = 0; i < node._components.length; ++i)
		if(node._components[i].uid)
			this._components_by_uid[ node._components[i].uid ] = node._components[i];
		else
			console.warn("component without uid?", node._components[i].uid );

	/**
	 * Fired when a new node is added to this scene
	 *
	 * @event nodeAdded
	 * @param {LS.SceneNode} node
	 */
	LEvent.trigger(this, EVENT.NODE_ADDED, node);
	LEvent.trigger(this, EVENT.CHANGE );
}

Scene.prototype.onNodeRemoved = function(e,node)
{
	var pos = this._nodes.indexOf(node);
	if(pos == -1) 
		return;

	this._nodes.splice(pos,1);
	if(node._name && this._nodes_by_name[ node._name ] == node )
		delete this._nodes_by_name[ node._name ];
	if(node.uid)
		delete this._nodes_by_uid[ node.uid ];

	//node.processActionInComponents("onRemovedFromNode",node);
	node.processActionInComponents("onRemovedFromScene",this); //send to components
	for(var i = 0; i < node._components.length; ++i)
		delete this._components_by_uid[ node._components[i].uid ];

	/**
	 * Fired after a node has been removed
	 *
	 * @event nodeRemoved
	 * @param {LS.SceneNode} node
	 */
	LEvent.trigger(this, EVENT.NODE_REMOVED, node);
	LEvent.trigger(this, EVENT.CHANGE );
	return true;
}

/**
* all nodes are stored in an array, this function recomputes the array so they are in the right order in case one has changed order
*
* @method recomputeNodesArray
*/
Scene.prototype.recomputeNodesArray = function()
{
	var nodes = this._nodes;
	var pos = 0;
	inner( this._root );

	function inner(node)
	{
		nodes[pos] = node;
		pos+=1;
		if(!node._children || !node._children.length)
			return;
		for(var i = 0; i < node._children.length; ++i)
			inner( node._children[i] );
	}
}

//WIP
Scene.prototype.attachSceneElement = function( element )
{
	this._spatial_container.add( element );
}

Scene.prototype.detachSceneElement = function( element )
{
	this._spatial_container.remove( element );
}


/**
* Returns the array containing all the nodes in the scene
*
* @method getNodes
* @param {bool} recompute [optional] in case you want to rearrange the nodes
* @return {Array} array containing every SceneNode in the scene
*/
Scene.prototype.getNodes = function( recompute )
{
	if(recompute)
		this.recomputeNodesArray();
	return this._nodes;
}

/**
* retrieves a Node based on the name, path ( name|childname|etc ) or uid
*
* @method getNode
* @param {String} name node name to search
* @return {Object} the node or null if it didnt find it
*/
Scene.prototype.getNode = function( name )
{
	if(name == "")
		return this.root;
	if(!name || name.constructor !== String)
		return null;
	if(name.charAt(0) == LS._uid_prefix)
		return this._nodes_by_uid[ name ];

	// the | char is used to specify a node child of another node
	if( name.indexOf("|") != -1)
	{
		var tokens = name.split("|");
		var node = this.root; //another option could be to start in this._nodes_by_name[ tokens[0] ]
		for(var i = 0; i < tokens.length && node; ++i)
			node = node.getChildByName( tokens[i] );
		return node;
	}

	return this._nodes_by_name[ name ];
}

/**
* retrieves a Node that matches that name. It is fast because they are stored in an object.
* If more than one object has the same name, the first one added to the tree is returned
*
* @method getNodeByName
* @param {String} name name of the node
* @return {Object} the node or null if it didnt find it
*/
Scene.prototype.getNodeByName = function( name )
{
	return this._nodes_by_name[ name ];
}

/**
* retrieves a Node based on a given uid. It is fast because they are stored in an object
*
* @method getNodeByUId
* @param {String} uid uid of the node
* @return {Object} the node or null if it didnt find it
*/
Scene.prototype.getNodeByUId = function( uid )
{
	return this._nodes_by_uid[ uid ];
}

/**
* retrieves a Node by its index
*
* @method getNodeByIndex
* @param {Number} node index
* @return {Object} returns the node at the 'index' position in the nodes array
*/
Scene.prototype.getNodeByIndex = function(index)
{
	return this._nodes[ index ];
}

//for those who are more traditional
Scene.prototype.getElementById = Scene.prototype.getNode;

/**
* retrieves a node array filtered by the filter function
*
* @method filterNodes
* @param {function} filter a callback function that receives every node and must return true or false
* @return {Array} array containing the nodes that passes the filter
*/
Scene.prototype.filterNodes = function( filter )
{
	var r = [];
	for(var i = 0; i < this._nodes.length; ++i)
		if( filter(this._nodes[i]) )
			r.push(this._nodes[i]);
	return r;
}

/**
* searches the component with this uid, it iterates through all the nodes and components (slow)
*
* @method findComponentByUId
* @param {String} uid uid of the node
* @return {Object} component or null
*/
Scene.prototype.findComponentByUId = function(uid)
{
	for(var i = 0; i < this._nodes.length; ++i)
	{
		var compo = this._nodes[i].getComponentByUId( uid );
		if(compo)
			return compo;
	}
	return null;
}

/**
* searches the material with this uid, it iterates through all the nodes (slow)
*
* @method findMaterialByUId
* @param {String} uid uid of the material
* @return {Object} Material or null
*/
Scene.prototype.findMaterialByUId = function(uid)
{
	if(LS.RM.materials[uid])
		return LS.RM.materials[uid];

	for(var i = 0; i < this._nodes.length; ++i)
	{
		var material = this._nodes[i].getMaterial();
		if(material && material.uid == uid)
			return material;
	}

	return null;
}


/**
* Returns information of a node component property based on the locator of that property
* Locators are in the form of "{NODE_UID}/{COMPONENT_UID}/{property_name}"
*
* @method getPropertyInfo
* @param {String} locator locator of the property
* @return {Object} object with node, component, name, and value
*/
Scene.prototype.getPropertyInfo = function( property_uid )
{
	var path = property_uid.split("/");

	var start = path[0].substr(0,5);

	//for resources
	if( start == "@RES-")
	{
		var filename = LS.ResourcesManager.convertLocatorToFilename(path[0]);
		var resource = LS.ResourcesManager.getResource(filename);
		if(path.length == 1)
			return resource;
		if(resource && resource.getPropertyInfoFromPath)
			return resource.getPropertyInfoFromPath( path.slice(1) );
		return null;
	}

	//for global materials
	if( start == "@MAT-")
	{
		var material = LS.RM.materials_by_uid[ path[0] ];
		if(!material)
			return null;
		return material.getPropertyInfoFromPath( path.slice(1) );
	}

	//for components
	if( start == "@COMP")
	{
		var comp = this.findComponentByUId( path[0] );
		if(!comp)
			return null;
		if(path.length == 1)
			return {
				node: comp.root,
				target: comp,
				name: comp ? LS.getObjectClassName( comp ) : "",
				type: "component",
				value: comp
			};
		return comp.getPropertyInfoFromPath( path.slice(1) );
	}

	//for regular locators
	var node = this.getNode( path[0] );
	if(!node)
		return null;

	return node.getPropertyInfoFromPath( path.slice(1) );
}

/**
* Returns information of a node component property based on the locator of that property
* Locators are in the form of "{NODE_UID}/{COMPONENT_UID}/{property_name}"
*
* @method getPropertyInfoFromPath
* @param {Array} path
* @return {Object} object with node, component, name, and value
*/
Scene.prototype.getPropertyInfoFromPath = function( path )
{
	var start = path[0].substr(0,5);
	//for resources
	if( start == "@RES-")
	{
		var filename = LS.ResourcesManager.convertLocatorToFilename(path[0]);
		var resource = LS.ResourcesManager.getResource(filename);
		if(path.length == 1)
			return resource;
		if(resource && resource.getPropertyInfoFromPath)
			return resource.getPropertyInfoFromPath( path.slice(1) );
		return null;
	}

	if(start == "@MAT-")
	{
		var material = LS.RM.materials_by_uid[ path[0] ];
		if(!material)
			return null;
		return material.getPropertyInfoFromPath( path.slice(1) );
	}

	var node = this.getNode( path[0] );
	if(!node)
		return null;
	return node.getPropertyInfoFromPath( path.slice(1) );
}


/**
* returns the value of a property given its locator
*
* @method getPropertyValue
* @param {String} locator locator of the property
* @param {*} value the value to assign
* @param {SceneNode} root [Optional] if you want to limit the locator to search inside a node
* @return {Component} the target where the action was performed
*/
Scene.prototype.getPropertyValue = function( locator, root_node )
{
	var path = locator.split("/");
	if(root_node)
		return root_node.getPropertyValueFromPath( path );
	return this.getPropertyValueFromPath( path );
}

Scene.prototype.getPropertyValueFromPath = function( path )
{
	var start = path[0].substr(0,5);

	if( start == "@RES-")
	{
		var filename = LS.ResourcesManager.convertLocatorToFilename(path[0]);
		var resource = LS.ResourcesManager.getResource(filename);
		if(path.length == 1)
			return resource;
		if(resource && resource.getPropertyInfoFromPath)
			return resource.getPropertyInfoFromPath( path.slice(1) );
		return null;
	}

	if(start == "@MAT-")
	{
		var material = LS.RM.materials_by_uid[ path[0] ];
		if(!material)
			return null;
		return material.getPropertyValueFromPath( path.slice(1) );
	}
	var node = this.getNode( path[0] );
	if(!node)
		return null;
	return node.getPropertyValueFromPath( path.slice(1) );
}

/**
* Assigns a value to the property of a component in a node based on the locator of that property
* Locators are in the form of "{NODE_UID}/{COMPONENT_UID}/{property_name}"
*
* @method setPropertyValue
* @param {String} locator locator of the property
* @param {*} value the value to assign
* @param {SceneNode} root [Optional] if you want to limit the locator to search inside a node
* @return {Component} the target where the action was performed
*/
Scene.prototype.setPropertyValue = function( locator, value, root_node )
{
	var path = locator.split("/");
	this.setPropertyValueFromPath( path, value, root_node, 0 );
}

/**
* Assigns a value to the property of a component in a node based on the locator that property
* Locators are in the form of "{NODE_UID}/{COMPONENT_UID}/{property_name}"
*
* @method setPropertyValueFromPath
* @param {Array} path a property locator split by "/"
* @param {*} value the value to assign
* @param {SceneNode} root_node [optional] the root node where you want to search the locator (this is to limit the locator to a branch of the scene tree)
* @param {Number} offset [optional] used to avoir generating garbage, instead of slicing the array every time, we pass the array index
* @return {Component} the target where the action was performed
*/
Scene.prototype.setPropertyValueFromPath = function( path, value, root_node, offset )
{
	offset = offset || 0;
	if(path.length < (offset+1))
		return;

	var start = path[offset].substr(0,5);

	if( start == "@RES-")
	{
		var filename = LS.ResourcesManager.convertLocatorToFilename(path[0]);
		var resource = LS.ResourcesManager.getResource(filename);
		if(path.length == 1)
		{
			console.warn("assigning a value to a locator with only the name of a resource doesn't make any sense");
			return null; 
		}
		if( resource && resource.setPropertyValueFromPath )
			return resource.setPropertyValueFromPath( path, value, offset + 1 );
		return null;
	}

	if(start == "@MAT-")
	{
		var material = LS.RM.materials_by_uid[ path[offset] ];
		if(!material)
			return null;
		return material.setPropertyValueFromPath( path, value, offset + 1 );
	}

	//get node
	var node = null;
	if( root_node )
	{
		var name = path[offset];
		if( name.indexOf("|") != -1)
		{
			var tokens = name.split("|");
			var node = root_node;
			for(var i = 0; i < tokens.length && node; ++i)
				node = node.getChildByName( tokens[i] );
		}
		else
			node = root_node.findNode( path[offset] );
	}
	else
		node = this.getNode( path[offset] );

	if(!node)
		return null;

	return node.setPropertyValueFromPath( path, value, offset + 1 );
}


/**
* Returns the resources used by the scene
* includes the nodes, components, preloads and global_scripts
* doesn't include external_scripts
*
* @method getResources
* @param {Object} resources [optional] object with resources
* @param {Boolean} as_array [optional] returns data in array format instead of object format
* @param {Boolean} skip_in_pack [optional] skips resources that come from a pack
* @param {Boolean} skip_local [optional] skips resources whose name starts with ":" (considered local resources)
* @return {Object|Array} the resources in object format (or if as_array is true, then an array)
*/
Scene.prototype.getResources = function( resources, as_array, skip_in_pack, skip_local )
{
	resources = resources || {};

	//to get the resources as array
	var array = null;
	if(resources.constructor === Array)
	{
		array = resources;
		resources = {};
		as_array = true;
	}

	//first the preload
	//resources that must be preloaded (because they will be used in the future)
	if(this.preloaded_resources)
		for(var i in this.preloaded_resources)
			resources[ i ] = true;

	if(this.texture_atlas)
		resources[ this.texture_atlas.filename ] = true;

	//global scripts
	for(var i = 0; i < this.global_scripts.length; ++i)
		if( this.global_scripts[i] )
			resources[ this.global_scripts[i] ] = true;

	//resources from nodes
	for(var i = 0; i < this._nodes.length; ++i)
		this._nodes[i].getResources( resources );

	//remove the resources that belong to packs or prefabs
	if(skip_in_pack)
		for(var i in resources)
		{
			var resource = LS.ResourcesManager.resources[i];
			if(!resource)
				continue;
			if(resource && (resource.from_prefab || resource.from_pack))
				delete resources[i];
		}

	//remove the resources that are local (generated by the system)
	if(skip_local)
		for(var i in resources)
		{
			if(i[0] == ":")
				delete resources[i];
		}

	//check if any resource requires another resource (a material that requires textures)
	for(var i in resources)
	{
		var resource = LS.ResourcesManager.resources[i];
		if(!resource)
			continue;
		if(resource.getResources)
			resource.getResources(resources);
	}

	//Hack: sometimes some component add this shit
	delete resources[""];
	delete resources["null"];

	//return as object
	if(!as_array)
		return resources;

	//return as array
	var r = array || [];
	for(var i in resources)
		r.push(i);
	return r;
}

/**
* Loads all the resources of all the nodes in this scene
* it sends a signal to every node to get all the resources info
* and load them in bulk using the ResourceManager
*
* @method loadResources
* @param {Function} on_complete called when the load of all the resources is complete
*/
Scene.prototype.loadResources = function( on_complete )
{
	//resources is an object format
	var resources = this.getResources([]);

	//used for scenes with special repository folders
	var options = {};
	if( this.external_repository )
		options.external_repository = this.external_repository;

	//count resources
	var num_resources = 0;
	for(var i in resources)
		++num_resources;

	//load them
	if(num_resources == 0)
	{
		if(on_complete)
			on_complete();
		return;
	}

	LEvent.bind( LS.ResourcesManager, "end_loading_resources", on_loaded );
	LS.ResourcesManager.loadResources( resources );

	function on_loaded()
	{
		LEvent.unbind( LS.ResourcesManager, "end_loading_resources", on_loaded );
		if(on_complete)
			on_complete();
	}
}


/**
* Adds a resource that must be loaded when the scene is loaded
*
* @method addPreloadResource
* @param {String} fullpath the name of the resource
*/
Scene.prototype.addPreloadResource = function( fullpath )
{
	this.preloaded_resources[ fullpath ] = true;
}

/**
* Remove a resource from the list of resources to preload
*
* @method removePreloadResource
* @param {String} fullpath the name of the resource
*/
Scene.prototype.removePreloadResource = function( fullpath )
{
	delete this.preloaded_resources[ fullpath ];
}


/**
* start the scene (triggers an "start" event)
*
* @method start
* @param {Number} dt delta time
*/
Scene.prototype.start = function()
{
	if(this._state == LS.PLAYING)
		return;

	this._state = LS.PLAYING;
	this._start_time = getTime() * 0.001;
	/**
	 * Fired when the nodes need to be initialized
	 *
	 * @event init
	 * @param {LS.Scene} scene
	 */
	LEvent.trigger(this, EVENT.INIT, this);
	this.triggerInNodes( EVENT.INIT );
	/**
	 * Fired when the scene is starting to play
	 *
	 * @event start
	 * @param {LS.Scene} scene
	 */
	LEvent.trigger(this, EVENT.START ,this);
	this.triggerInNodes( EVENT.START );
}

/**
* pauses the scene (triggers an "pause" event)
*
* @method pause
*/
Scene.prototype.pause = function()
{
	if( this._state != LS.PLAYING )
		return;

	this._state = LS.PAUSED;
	/**
	 * Fired when the scene pauses (mostly in the editor)
	 *
	 * @event pause
	 * @param {LS.Scene} scene
	 */
	LEvent.trigger(this, EVENT.PAUSE,this);
	this.triggerInNodes( EVENT.PAUSE );
	this.purgeResidualEvents();
}

/**
* unpauses the scene (triggers an "unpause" event)
*
* @method unpause
*/
Scene.prototype.unpause = function()
{
	if(this._state != LS.PAUSED)
		return;

	this._state = LS.PLAYING;
	/**
	 * Fired when the scene unpauses (mostly in the editor)
	 *
	 * @event unpause
	 * @param {LS.Scene} scene
	 */
	LEvent.trigger(this, EVENT.UNPAUSE,this);
	this.triggerInNodes( EVENT.UNPAUSE );
	this.purgeResidualEvents();
}


/**
* stop the scene (triggers an "finish" event)
*
* @method finish
* @param {Number} dt delta time
*/
Scene.prototype.finish = function()
{
	if(this._state == LS.STOPPED)
		return;

	this._state = LS.STOPPED;
	/**
	 * Fired when the scene stops playing
	 *
	 * @event finish
	 * @param {LS.Scene} scene
	 */
	LEvent.trigger(this, EVENT.FINISH,this);
	this.triggerInNodes( EVENT.FINISH );
	this.purgeResidualEvents();
}

/**
* This methods crawls the whole tree and collects all the useful info (cameras, lights, render instances, colliders, etc)
* Mostly rendering stuff but also some collision info.
* TO DO: refactor this so it doesnt redo the same task in every frame, only if changes are made
* @param {Array} cameras [optional] an array of cameras in case we want to force some viewpoint
* @method collectData
*/
Scene.prototype.collectData = function( cameras )
{
	var instances = this._instances;
	var lights = this._lights;
	var colliders = this._colliders;

	//empty containers
	instances.length = 0;
	lights.length = 0;
	colliders.length = 0;

	//first collect cameras (in case we want to filter nodes by proximity to camera
	if(!cameras || cameras.length == 0)
	{
		cameras = this._cameras;
		cameras.length = 0;
		LEvent.trigger( this, EVENT.COLLECT_CAMERAS, cameras );
	}

	//get nodes: TODO find nodes close to the active cameras
	var nodes = this.getNodes();

	//collect render instances and lights
	for(var i = 0, l = nodes.length; i < l; ++i)
	{
		var node = nodes[i];

		//skip stuff inside invisible nodes
		if(node.flags.visible == false) 
			continue;

		//compute global matrix: shouldnt it be already computed?
		if(node.transform)
			node.transform.updateGlobalMatrix();

		//clear instances per node: TODO: if static maybe just leave it as it is
		node._instances.length = 0;

		//get render instances: remember, triggers only support one parameter
		LEvent.trigger( node, EVENT.COLLECT_RENDER_INSTANCES, node._instances );
		LEvent.trigger( node, EVENT.COLLECT_PHYSIC_INSTANCES, colliders );

		//concatenate all instances in a single array
		instances.push.apply(instances, node._instances);
	}

	//we also collect from the scene itself (used for lights, skybox, etc)
	LEvent.trigger( this, EVENT.COLLECT_RENDER_INSTANCES, instances );
	LEvent.trigger( this, EVENT.COLLECT_PHYSIC_INSTANCES, colliders );
	LEvent.trigger( this, EVENT.COLLECT_LIGHTS, lights );

	//before processing (in case somebody wants to add some data to the containers)
	LEvent.trigger( this, EVENT.COLLECT_DATA );

	//for each render instance collected
	for(var i = 0, l = instances.length; i < l; ++i)
	{
		var instance = instances[i];

		//compute the axis aligned bounding box
		if(instance.use_bounding)
			instance.updateAABB();
	}

	//for each physics instance collected
	for(var i = 0, l = colliders.length; i < l; ++i)
	{
		var collider = colliders[i];
		collider.updateAABB();
	}

	//remember when was last time I collected to avoid repeating it
	this._last_collect_frame = this._frame;
}

/**
* updates the scene (it handles variable update and fixedUpdate)
*
* @method update
* @param {Number} dt delta time in seconds
*/
Scene.prototype.update = function(dt)
{
	/**
	 * Fired before doing an update
	 *
	 * @event beforeUpdate
	 * @param {LS.Scene} scene
	 */
	LEvent.trigger(this,LS.EVENT.BEFORE_UPDATE, this);

	this._global_time = getTime() * 0.001;
	//this._time = this._global_time - this._start_time;
	this._time += dt;
	this._last_dt = dt;

	/**
	 * Fired while updating
	 *
	 * @event update
	 * @param {number} dt
	 */
	LEvent.trigger(this, LS.EVENT.UPDATE, dt);

	/**
	 * Fired while updating but using a fixed timestep (1/60)
	 *
	 * @event fixedUpdate
	 * @param {number} dt
	 */
	if(this._fixed_update_timestep > 0)
	{
		this._remaining_fixed_update_time += dt;
		if(LEvent.hasBind(this, LS.EVENT.FIXED_UPDATE))
			while( this._remaining_fixed_update_time > this._fixed_update_timestep )
			{
				LEvent.trigger(this, LS.EVENT.FIXED_UPDATE, this._fixed_update_timestep );
				this._remaining_fixed_update_time -= this._fixed_update_timestep;
			}
		else
			this._remaining_fixed_update_time = this._remaining_fixed_update_time % this._fixed_update_timestep;
	}

	/**
	 * Fired after updating the scene
	 *
	 * @event afterUpdate
	 */
	LEvent.trigger(this, LS.EVENT.AFTER_UPDATE, this);
}

/**
* triggers an event to all nodes in the scene
* this is slow if the scene has too many nodes, thats why we use bindings
*
* @method triggerInNodes
* @param {String} event_type event type name
* @param {Object} data data to send associated to the event
*/

Scene.prototype.triggerInNodes = function(event_type, data)
{
	LEvent.triggerArray( this._nodes, event_type, data);
}

/**
* generate a unique node name given a prefix
*
* @method generateUniqueNodeName
* @param {String} prefix the prefix, if not given then "node" is used
* @return {String} a node name that it is not in the scene
*/
Scene.prototype.generateUniqueNodeName = function(prefix)
{
	prefix = prefix || "node";
	var i = 1;

	var pos = prefix.lastIndexOf("_");
	if(pos)
	{
		var n = prefix.substr(pos+1);
		if( parseInt(n) )
		{
			i = parseInt(n);
			prefix = prefix.substr(0,pos);
		}
	}

	var node_name = prefix + "_" + i;
	while( this.getNode(node_name) != null )
		node_name = prefix + "_" + (i++);
	return node_name;
}

/**
* Marks that this scene must be rendered again
*
* @method requestFrame
*/
Scene.prototype.requestFrame = function()
{
	this._must_redraw = true;
	LEvent.trigger( this, LS.EVENT.REQUEST_FRAME );
}

Scene.prototype.refresh = Scene.prototype.requestFrame; //DEPRECATED

/**
* returns current scene time (remember that scene time remains freezed if the scene is not playing)
*
* @method getTime
* @return {Number} scene time in seconds
*/
Scene.prototype.getTime = function()
{
	return this._time;
}

//This is ugly but sometimes if scripts fail there is a change the could get hooked to the scene forever
//so this way we remove any event that belongs to a component thats doesnt belong to this scene tree
Scene.prototype.purgeResidualEvents = function()
{
	if(!this.__events)
		return;

	//crawl all 
	for(var i in this.__events)
	{
		var event = this.__events[i];
		if(!event)
			continue;
		var to_keep = [];
		for(var j = 0; j < event.length; ++j)
		{
			var inst = event[j][1];
			if(inst && LS.isClassComponent( inst.constructor ) )
			{
				//no attached node or node not attached to any scene
				if(!inst._root || inst._root.scene !== this )
				{
					console.warn("Event attached to the Scene belongs to a removed node, purged. Event:",i,"Class:", LS.getObjectClassName( inst ) );
					continue; //skip keeping it, so it will no longer exist
				}
			}
			to_keep.push(event[j]);
		}
		this.__events[i] = to_keep;
	}
}

/**
* returns an array with the name of all the layers given a layers mask
*
* @method getLayerNames
* @param {Number} layers a number with the enabled layers in bit mask format, if ommited all layers are returned
* @return {Array} array of strings with the layer names
*/
Scene.prototype.getLayerNames = function(layers)
{
	var r = [];

	for(var i = 0; i < 32; ++i)
	{
		if( layers === undefined || layers & (1<<i) )
			r.push( this.layer_names[i] || ("layer"+i) );
	}
	return r;
}

/**
* returns an array with all the components in the scene and scenenodes that matches this class
*
* @method findNodeComponents
* @param {String||Component} type the type of the components to search (could be a string with the name or the class itself)
* @return {Array} array with the components found
*/
Scene.prototype.findNodeComponents = function( type )
{
	if(!type)
		return;

	var find_component = null;
	if(type.constructor === String)
		find_component = LS.Components[ type ];
	else
		find_component = type;
	if(!find_component)
		return;

	var result = [];
	var nodes = this._nodes;
	for(var i = 0; i < nodes.length; ++i)
	{
		var node = nodes[i];
		var components = node._components;
		for(var j = 0; j < components.length; ++j)
			if( components[j].constructor === find_component )
				result.push( components[j] );
	}
	return result;
}

/**
* Allows to instantiate a prefab from the fullpath of the resource
*
* @method instantiate
* @param {String} prefab_url the filename to the resource containing the prefab
* @param {vec3} position where to instantiate
* @param {quat} rotation the orientation
* @param {SceneNode} parent [optional] if no parent then scene.root will be used
* @return {SceneNode} the resulting prefab node
*/
Scene.prototype.instantiate = function( prefab_url, position, rotation, parent )
{
	if(!prefab_url || prefab_url.constructor !== String)
		throw("prefab must be the url to the prefab");

	var node = new LS.SceneNode();
	if(position && position.length === 3)
		node.transform.position = position;
	if(rotation && rotation.length === 4)
		node.transform.rotation = rotation;

	parent = parent || this.root;
	parent.addChild( node );

	node.prefab = prefab_url;

	return node;
}

/**
* returns a pack containing all the scene and resources, used to save a scene to harddrive
*
* @method toPack
* @param {String} fullpath a given fullpath name, it will be assigned to the scene with the appropiate extension
* @param {Array} resources [optional] array with all the resources to add, if no array is given it will get the active resources in this scene
* @return {LS.Pack} the pack
*/
Scene.prototype.toPack = function( fullpath, resources )
{
	fullpath = fullpath || "unnamed_scene";

	//change name to valid name
	var basename = LS.RM.removeExtension( fullpath, true );
	var final_fullpath = basename + ".SCENE.wbin";

	//extract json info
	var scene_json = JSON.stringify( this.serialize() );

	//get all resources
	if(!resources)
		resources = this.getResources( null, true, true, true );

	//create pack
	var pack = LS.Pack.createPack( LS.RM.getFilename( final_fullpath ), resources, { "scene.json": scene_json } );
	pack.fullpath = final_fullpath;
	pack.category = "Scene";

	return pack;
}

//WIP: this is in case we have static nodes in the scene
Scene.prototype.updateStaticObjects = function()
{
	var old = LS.allow_static;
	LS.allow_static = false;
	this.collectData();
	LS.allow_static = old;
}

/**
* search for the nearest reflection probe to the point
*
* @method findNearestReflectionProbe
* @param {vec3} position
* @return {LS.ReflectionProbe} the reflection probe
*/
Scene.prototype.findNearestReflectionProbe = function( position )
{
	if(!this._reflection_probes.length)
		return null;

	if( this._reflection_probes.length == 1 )
		return this._reflection_probes[0];

	var probes = this._reflection_probes;
	var min_dist = 1000000;
	var nearest_probe = null;
	for(var i = 0; i < probes.length; ++i)
	{
		var probe = probes[i];
		var dist = vec3.squaredDistance( position, probe._position );
		if( dist > min_dist )
			continue;
		min_dist = dist;
		nearest_probe = probe;
	}
	return nearest_probe;
}


//tells to all the components, nodes, materials, etc, that one resource has changed its name so they can update it inside
Scene.prototype.sendResourceRenamedEvent = function( old_name, new_name, resource )
{
	//scene globals that use resources
	for(var i = 0; i < this.external_scripts.length; i++)
	{
		if(this.external_scripts[i] == old_name)
			this.external_scripts[i] = new_name;
	}

	for(var i = 0; i < this.global_scripts.length; i++)
	{
		if(this.global_scripts[i] == old_name)
			this.global_scripts[i] = new_name;
	}

	for(var i in this.preloaded_resources)
	{
		if(i == old_name)
		{
			delete this.preloaded_resources[old_name];
			this.preloaded_resources[ new_name ] = true;
		}
	}

	if( this.texture_atlas && this.texture_atlas.filename == old_name )
		this.texture_atlas.filename = new_name;

	//to nodes
	var nodes = this._nodes.concat();

	//for every node
	for(var i = 0; i < nodes.length; i++)
	{
		//nodes
		var node = nodes[i];

		//prefabs
		if( node.prefab && node.prefab === old_name )
			node.prefab = new_name; //does this launch a reload prefab? dont know

		//components
		for(var j = 0; j < node._components.length; j++)
		{
			var component = node._components[j];
			if(component.onResourceRenamed)
				component.onResourceRenamed( old_name, new_name, resource );
			else //automatic
			{
				for(var k in component)
				{
					if(component[k] != old_name )
						continue;
					var propinfo = component.constructor["@" + k];
					if(!propinfo)
						continue;
					var type = propinfo.type || propinfo.widget;
					if(type && (type == LS.TYPES.RESOURCE || LS.ResourceClasses[ type ])) //is a resource
						component[k] = new_name;
				}
			}
		}

		//materials
		if( node.material )
		{
			if( node.material == old_name )
				node.material = new_name;
			else
			{
				var material = node.getMaterial();
				if( material && material.onResourceRenamed )
				{
					var modified = material.onResourceRenamed( old_name, new_name, resource );
					if(modified) //we need this to remove material._original_data or anything that could interfiere
						LS.RM.resourceModified( material );
				}
				else
					console.warn("sendResourceRenamedEvent: Material not found or it didnt have a onResourceRenamed");
			}
		}
	}
}

//used to search a resource according to the data path of this scene
Scene.prototype.getDataPath = function( path )
{
	path = path || "";
	var folder = this.extra.data_folder || this.extra.folder;
	return LS.RM.cleanFullpath( folder + "/" + path );
}


/**
* Creates and returns an scene animation track
*
* @method createAnimation
* @return {LS.Animation} the animation track
*/
Scene.prototype.createAnimation = function()
{
	if(this.animation)
		return this.animation;
	this.animation = new LS.Animation();
	this.animation.name = LS.Animation.DEFAULT_SCENE_NAME;
	this.animation.createTake( "default", LS.Animation.DEFAULT_DURATION );
	return this.animation;
}

LS.Scene = Scene;
LS.Classes.Scene = Scene;
LS.Classes.SceneTree = Scene; //LEGACY


///@FILE:../src/sceneNode.js
///@INFO: BASE
//****************************************************************************

/**
* The SceneNode class represents and object in the scene
* Is the base class for all objects in the scene as meshes, lights, cameras, and so
*
* @class SceneNode
* @param {String} name the name for this node (otherwise a random one is computed)
* @constructor
*/

function SceneNode( name )
{
	if(name && name.constructor !== String)
	{
		name = null;
		console.warn("SceneNode constructor first parameter must be a String with the name");
	}

	//Generic identifying info
	this._name = name || ("node_" + (Math.random() * 10000).toFixed(0)); //generate random number
	this._uid = LS.generateUId("NODE-");
	this._classList = {}; //to store classes
	this.layers = 3|0; //32 bits for layers (force to int)
	this.node_type = null; //used to store a string defining the node info

	//more generic info
	this._prefab = null;
	this._material = null;

	//from Componentcontainer
	this._components = []; //used for logic actions

	//from CompositePattern
	this._parentNode = null;
	this._children = null;
	this._in_tree = null;
	this._instances = []; //render instances

	//bounding box in world space
	//this._aabb = BBox.create();

	//flags
	this.flags = {
		visible: true,
		is_static: false,
		selectable: true,
		locked: false
	};

	this.init(false,true);

	/** Fired here (from Transform) when the node transform changes
	 * @event transformChanged
	 */
}

SceneNode.prototype.init = function( keep_components, keep_info )
{
	if(!keep_info)
	{
		this.layers = 3|0; //32 bits for layers (force to int)
		this._name = name || ("node_" + (Math.random() * 10000).toFixed(0)); //generate random number
		this._uid = LS.generateUId("NODE-");
		this._classList = {};

		//material
		this._material = null;
		this.extra = {}; //for extra info
		this.node_type = null;

		//flags
		this.flags = {
			visible: true,
			is_static: false,
			selectable: true
		};
	}

	//Basic components
	if(!keep_components)
	{
		if( this._components && this._components.length )
			console.warn("SceneNode.init() should not be called if it contains components, call clear instead");
		this._components = []; //used for logic actions
		this.addComponent( new LS.Transform() );
	}
}

//get methods from other classes
LS.extendClass( SceneNode, ComponentContainer ); //container methods
LS.extendClass( SceneNode, CompositePattern ); //container methods

/**
* changes the node name
* @method setName
* @param {String} new_name the new name
* @return {Object} returns true if the name changed
*/

Object.defineProperty( SceneNode.prototype, 'name', {
	set: function(name)
	{
		this.setName( name );
	},
	get: function(){
		return this._name;
	},
	enumerable: true
});

Object.defineProperty( SceneNode.prototype, 'fullname', {
	set: function(name)
	{
		throw("You cannot set fullname, it depends on the parent nodes");
	},
	get: function(){
		return this.getPathName();
	},
	enumerable: false
});

//Changing the UID  has lots of effects (because nodes are indexed by UID in the scene)
//If you want to catch the event of the uid_change, remember, the previous uid is stored in LS.SceneNode._last_uid_changed (it is not passed in the event)
Object.defineProperty( SceneNode.prototype, 'uid', {
	set: function(uid)
	{
		if(!uid)
			return;

		//valid uid?
		if(uid[0] != LS._uid_prefix)
		{
			console.warn("Invalid UID, renaming it to: " + uid );
			uid = LS._uid_prefix + uid;
		}

		//no changes?
		if(uid == this._uid)
			return;

		SceneNode._last_uid_changed = this._uid; //hack, in case we want the previous uid of a node 

		//update scene tree indexing
		if( this._in_tree && this._in_tree._nodes_by_uid[ this.uid ] )
			delete this._in_tree._nodes_by_uid[ this.uid ];
		this._uid = uid;
		if( this._in_tree )
			this._in_tree._nodes_by_uid[ this.uid ] = this;
		//events
		LEvent.trigger( this, "uid_changed", uid );
		if(this._in_tree)
			LEvent.trigger( this._in_tree, "node_uid_changed", this );
	},
	get: function(){
		return this._uid;
	},
	enumerable: true
});


Object.defineProperty( SceneNode.prototype, 'visible', {
	set: function(v)
	{
		this.flags.visible = v;
		if( this._children )
		for(var i = 0; i < this._children.length; ++i )
			this._children[i].visible = v;
	},
	get: function(){
		return this.flags.visible;
	},
	enumerable: true
});

Object.defineProperty( SceneNode.prototype, 'is_static', {
	set: function(v)
	{
		this.flags.is_static = v;
		if( v && this._children )
		for(var i = 0; i < this._children.length; ++i )
			this._children[i].is_static = v;
	},
	get: function(){
		return this.flags.is_static;
	},
	enumerable: true
});

Object.defineProperty( SceneNode.prototype, 'material', {
	set: function(v)
	{
		if( this._material == v )
			return;

		this._material = v;
		if(v)
		{
			if(v.constructor === String)
				return;
			if(v._root && v._root != this) //has root and its not me
				console.warn( "Cannot assign a material of one SceneNode to another, you must clone it or register it" )
			else
				v._root = this; //link
		}
		LEvent.trigger( this, "materialChanged" );
	},
	get: function(){
		return this._material;
	},
	enumerable: true
});

Object.defineProperty( SceneNode.prototype, 'prefab', {
	set: function(name)
	{
		this._prefab = name;
		if(!this._prefab)
			return;
		var prefab = LS.RM.getResource(name);
		var that = this;
		if(prefab)
			this.reloadFromPrefab();
		else 
			LS.ResourcesManager.load( name, function(){
				that.reloadFromPrefab();
			});
	},
	get: function(){
		return this._prefab;
	},
	enumerable: true
});

SceneNode.prototype.clear = function()
{
	this.removeAllComponents();
	this.removeAllChildren();
	this.init();
}

SceneNode.prototype.setName = function(new_name)
{
	if(this._name == new_name) 
		return true; //no changes

	//check that the name is valid (doesnt have invalid characters)
	if(!LS.validateName(new_name))
	{
		console.warn("invalid name for node: " + new_name );
		//new_name = new_name.replace(/[^a-z0-9\.\-]/gi,"_");
		return false;
	}

	var scene = this._in_tree;
	if(!scene)
	{
		this._name = new_name;
		return true;
	}

	//remove old link
	if( this._name )
		delete scene._nodes_by_name[ this._name ];

	//assign name
	this._name = new_name;

	//we already have another node with this name
	if( new_name && !scene._nodes_by_name[ new_name ] )
		scene._nodes_by_name[ this._name ] = this;

	/**
	 * Node changed name
	 *
	 * @event name_changed
	 * @param {String} new_name
	 */
	LEvent.trigger( this, "name_changed", new_name );
	if(scene)
		LEvent.trigger( scene, "node_name_changed", this );
	return true;
}

Object.defineProperty( SceneNode.prototype, 'classList', {
	get: function() { return this._classList },
	set: function(v) {},
	enumerable: false
});

/**
* @property className {String}
*/
Object.defineProperty( SceneNode.prototype, 'className', {
	get: function() {
			var keys = null;
			if(Object.keys)
				keys = Object.keys(this._classList); 
			else
			{
				keys = [];
				for(var k in this._classList)
					keys.push(k);
			}
			return keys.join(" ");
		},
	set: function(v) { 
		this._classList = {};
		if(!v)
			return;
		var t = v.split(" ");
		for(var i in t)
			this._classList[ t[i] ] = true;
	},
	enumerable: true
});

/**
* Destroys this node
* @method destroy
* @param {number} time [optional] time in seconds to wait till destroying the node
**/
SceneNode.prototype.destroy = function( time )
{
	if(time && time.constructor === Number && time > 0)
	{
		setTimeout( this.destroy.bind(this,0), time * 0.001 );
		return;
	}

	LEvent.trigger( this, "destroy" );
	this.removeAllComponents();
	if(this.children)
		while(this.children.length)
			this.children[0].destroy();
	if(this._parentNode)
		this._parentNode.removeChild( this );
}

/**
* Returns the locator string of this node
* @method getLocator
* @param {string} property_name [optional] you can pass the name of a property in this node to get the locator of that one
* @return {String} the locator string of this node
**/
SceneNode.prototype.getLocator = function( property_name )
{
	if(!property_name)
		return this.uid;
	return this.uid + "/" + property_name;
}

/**
* Returns and object with info about a property given a locator
* @method getPropertyInfo
* @param {string} locator
* @return {Object} object with { node, target, name, value and type }
**/
SceneNode.prototype.getPropertyInfo = function( locator )
{
	var path = locator.split("/");
	return this.getPropertyInfoFromPath(path);
}

/**
* Returns and object with info about a property given a locator in path format
* @method getPropertyInfoFromPath
* @param {Array} path a locator in path format (split by /)
* @return {Object} object with { node, target, name, value and type }
**/
SceneNode.prototype.getPropertyInfoFromPath = function( path )
{
	var target = this;
	var varname = path[0];
	var no_slice = false;

	if(path.length == 0)
	{
		return {
			node: this,
			target: null,
			name: "",
			value: this,
			type: "node" //node because thats the global type for nodes
		};
	}
    else if(path.length == 1) //compo or //var
	{
		if(path[0][0] == "@") //compo uid
		{
			target = this.getComponentByUId( path[0] );
			return {
				node: this,
				target: target,
				name: target ? LS.getObjectClassName( target ) : "",
				type: "component",
				value: target
			};
		}
		else if (path[0] == "material")
		{
			target = this.getMaterial();
			return {
				node: this,
				target: target,
				name: target ? LS.getObjectClassName( target ) : "",
				type: "material",
				value: target
			};
		}
		else if (path[0] == "visible")
		{
			return {
				node: this,
				target: this,
				name: "visible",
				type: "boolean",
				value: this.visible
			};
		}

		var target = this.getComponent( path[0] );
		if(target)
		{
			return {
				node: this,
				target: target,
				name: target ? LS.getObjectClassName( target ) : "",
				type: "component",
				value: target
			};
		}

		//special cases for a node
		switch(path[0])
		{
			case "matrix":
			case "x":
			case "y": 
			case "z": 
			case "position":
			case "rotX":
			case "rotY":
			case "rotZ":
				target = this.transform;
				varname = path[0];
				no_slice = true;
				break;
			default: 
				target = this;
				varname = path[0];
			break;
		}
	}
    else if(path.length > 1) //compo/var
	{
		if(path[0][0] == "@")
		{
			varname = path[1];
			target = this.getComponentByUId( path[0] );
		}
		else if (path[0] == "material")
		{
			target = this.getMaterial();
			varname = path[1];
		}
		else if (path[0] == "flags")
		{
			target = this.flags;
			varname = path[1];
		}
		else
		{
			target = this.getComponent( path[0] );
			varname = path[1];
		}

		if(!target)
			return null;
	}
	else //�?
	{
	}

	if(!target) //unknown target
		return null;

	//this was moved to Component.prototype.getPropertyInfoFromPath  (if any errors check cases)
	if( target != this && target.getPropertyInfoFromPath ) //avoid weird recursion
		return target.getPropertyInfoFromPath( no_slice ? path : path.slice(1) );

	return null;
}

/**
* Returns the value of a property given a locator in string format
* @method getPropertyValue
* @param {String} locaator
* @return {*} the value of that property
**/
SceneNode.prototype.getPropertyValue = function( locator )
{
	var path = locator.split("/");
	return this.getPropertyValueFromPath(path);
}

/**
* Returns the value of a property given a locator in path format
* @method getPropertyValueFromPath
* @param {Array} locator in path format (array)
* @return {*} the value of that property
**/
SceneNode.prototype.getPropertyValueFromPath = function( path )
{
	var target = this;
	var varname = path[0];
	var no_slice = false;

	if(path.length == 0)
		return null
    else if(path.length == 1) //compo or //var
	{
		if(path[0][0] == "@")
			return this.getComponentByUId( path[0] );
		else if (path[0] == "material")
			return this.getMaterial();
		var target = this.getComponent( path[0] );
		if(target)
			return target;

		switch(path[0])
		{
			case "matrix":
			case "x":
			case "y": 
			case "z": 
			case "position":
			case "rotX":
			case "rotY":
			case "rotZ":
				target = this.transform;
				varname = path[0];
				no_slice = true;
				break;
			default: 
				target = this;
				varname = path[0];
			break;
		}
	}
    else if(path.length > 1) //compo/var
	{
		if(path[0][0] == "@")
		{
			varname = path[1];
			target = this.getComponentByUId( path[0] );
		}
		else if (path[0] == "material")
		{
			target = this.getMaterial();
			varname = path[1];
		}
		else if (path[0] == "flags")
		{
			target = this.flags;
			varname = path[1];
		}
		else if (path[0] == "visible")
		{
			target = this;
			varname = path[0];
		}
		else
		{
			target = this.getComponent( path[0] );
			varname = path[1];
		}

		if(!target)
			return null;
	}
	else //�?
	{
	}

	var v = undefined;

	if( target.getPropertyValueFromPath && target != this )
	{
		var r = target.getPropertyValueFromPath( no_slice ? path : path.slice(1) );
		if(r)
			return r;
	}

	//to know the value of a property of the given target
	if( target.getPropertyValue && target != this )
		v = target.getPropertyValue( varname );

	//special case when the component doesnt specify any locator info but the property referenced does
	//used in TextureFX
	if (v === undefined && path.length > 2 && target[ varname ] && target[ varname ].getPropertyValueFromPath )
	{
		var r = target[ varname ].getPropertyValueFromPath( no_slice ? path.slice(1) : path.slice(2) );
		if(r)
		{
			r.node = this;
			return r;
		}
	}

	if(v === undefined && target[ varname ] === undefined )
		return null;
	return v !== undefined ? v : target[ varname ];
}

/**
* assigns a value to a property given the locator for that property
* @method setPropertyValue
* @param {String} locator
* @param {*} value
**/
SceneNode.prototype.setPropertyValue = function( locator, value )
{
	var path = locator.split("/");
	return this.setPropertyValueFromPath(path, value, 0);
}

/**
* given a locator in path mode (array) and a value, it searches for the corresponding value and applies it
* @method setPropertyValueFromPath
* @param {Array} path
* @param {*} value
* @param {Number} [optional] offset used to skip the firsst positions in the array
**/
SceneNode.prototype.setPropertyValueFromPath = function( path, value, offset )
{
	offset = offset || 0;

	if(this.flags && this.flags.locked)
		return; //lock ignores changes from animations or graphs

	var target = null;
	var varname = path[offset];

	if(path.length > (offset+1))
	{
		if(path[offset][0] == "@")
		{
			varname = path[offset+1];
			target = this.getComponentByUId( path[offset] );
		}
		else if( path[offset] == "material" )
		{
			target = this.getMaterial();
			varname = path[offset+1];
		}
		else if( path[offset] == "flags" )
		{
			target = this.flags;
			varname = path[offset+1];
		}
		else if( path[offset] == "visible" )
		{
			target = this;
			varname = path[offset];
		}
		else 
		{
			target = this.getComponent( path[offset] );
			varname = path[offset+1];
		}

		if(!target)
			return null;
	}
	else { //special cases 
		switch ( path[offset] )
		{
			case "matrix": target = this.transform; break;
			case "position":
			case "rotation":
			case "x":
			case "y":
			case "z":
			case "xrotation": 
			case "yrotation": 
			case "zrotation": 
				target = this.transform; 
				varname = path[offset];
				break;
			case "translate.X": target = this.transform; varname = "x"; break;
			case "translate.Y": target = this.transform; varname = "y"; break;
			case "translate.Z": target = this.transform; varname = "z"; break;
			case "rotateX.ANGLE": target = this.transform; varname = "pitch"; break;
			case "rotateY.ANGLE": target = this.transform; varname = "yaw"; break;
			case "rotateZ.ANGLE": target = this.transform; varname = "roll"; break;
			default: target = this; //null
		}
	}

	if(!target)
		return null;

	if(target.setPropertyValueFromPath && target != this)
		if( target.setPropertyValueFromPath( path, value, offset+1 ) === true )
			return target;
	
	if(target.setPropertyValue  && target != this)
		if( target.setPropertyValue( varname, value ) === true )
			return target;

	if( target[ varname ] === undefined )
		return;

	//special case when the component doesnt specify any locator info but the property referenced does
	//used in TextureFX
	if ( path.length > 2 && target[ varname ] && target[ varname ].setPropertyValueFromPath )
		return target[ varname ].setPropertyValueFromPath( path, value, offset+2 );

	//disabled because if the vars has a setter it wont be called using the array.set
	//if( target[ varname ] !== null && target[ varname ].set )
	//	target[ varname ].set( value );
	//else
		target[ varname ] = value;

	return target;
}

/**
* Returns all the resources used by this node and its components (you can include the resources from the children too)
* @method getResources
* @param {Object} res object where to store the resources used (in "res_name":LS.TYPE format)
* @param {Boolean} include_children if you want to add also the resources used by the children nodes
* @return {Object} the same object passed is returned 
**/
SceneNode.prototype.getResources = function( res, include_children )
{
	//resources in components
	for(var i in this._components)
	{
		var comp = this._components[i];
		if( comp.getResources )
			comp.getResources( res );
		else
		{
			//automatic
			for(var j in comp)
			{
				if(!comp[j] || comp[j].constructor === Function || j[0] == "_")
					continue;
				var propinfo = comp.constructor["@" + j];
				if(!propinfo)
					continue;
				var type = propinfo.type || propinfo.widget;
				if(type && (type == LS.TYPES.RESOURCE || LS.ResourceClasses[ type ]) ) //is a resource
					res[ comp[j] ] = LS.ResourceClasses[ type ];
			}
		}
	}

	//res in material
	if(this.material)
	{
		if( this.material.constructor === String )
		{
			if(this.material[0] != ":") //not a local material, then its a reference
			{
				res[this.material] = LS.Material;
			}
		}

		var mat = this.getMaterial();
		if(mat)
			mat.getResources( res );
	}

	//prefab
	if(this.prefab)
		res[ this.prefab ] = LS.Prefab;

	//propagate
	if(include_children)
		for(var i in this._children)
			this._children[i].getResources(res, true);

	return res;
}

SceneNode.prototype.getTransform = function() {
	return this.transform;
}

//Helpers

SceneNode.prototype.getMesh = function( use_lod_mesh ) {
	var mesh = this.mesh;
	var mesh_renderer = this.getComponent( LS.Components.MeshRenderer );
	if(!mesh && mesh_renderer)
	{
		if(use_lod_mesh)
			mesh = mesh_renderer.lod_mesh;
		if(!mesh)
			mesh = mesh_renderer.mesh;
	}
	if(!mesh)
		return null;
	if(mesh.constructor === String)
		return LS.ResourcesManager.meshes[mesh];
	return mesh;
}

//Light component
SceneNode.prototype.getLight = function() {
	return this.light;
}

//Camera component
SceneNode.prototype.getCamera = function() {
	return this.camera;
}

/**
* Allows to load some kind of resource and associate it to this node.
* It can be for prefabs, meshes, scenes from daes, etc
* @method load
* @param {string} url
* @param {Function} on_complete
**/
SceneNode.prototype.load = function( url, on_complete )
{
	var that = this;
	LS.ResourcesManager.load( url, inner );
	function inner( resource )
	{
		if(!resource)
			return;
		that.assign( resource );
		if(on_complete)
			on_complete();
	}
}

/**
* Assign a resource/element inteligently to a node: if it is a mesh it creates a MeshRenderer, if it is a Material it assigns it, if it is an animation creates a PlayAnimation, if it is a prefab assigns the prefab. etc
* @method assign
* @param {*} resource the resource to assign (it also accepts a resource filename that has been previously loaded).
* @param {Function} on_complete
**/
SceneNode.prototype.assign = function( item, extra )
{
	if(!item)
	{
		console.error("assignResource cannot have null as resource");
		return;
	}

	//assume is the filename of a resource
	if(item.constructor === String)
		item = LS.ResourcesManager.getResource( item );

	if(!item)
		return;

	switch( item.constructor )
	{
		case LS.SceneNode: 
			this.addChild( item );
			break;
		case LS.Scene:
			var node = this;
			item.loadScripts( null, function(){
				item.loadResources( function(){ 
					node.addChild( item.root.clone() );
				});
			});
			break;
		case LS.Prefab: 
			this.prefab = item.fullpath || item.filename; 
			break;
		case GL.Mesh: 
			var component = this.getComponent( LS.Components.MeshRenderer );
			if(component)
				component.configure({ mesh: item.fullpath || item.filename });
			else
				this.addComponent( new LS.MeshRenderer({ mesh: mesh_name, submesh_id: submesh_id }) );
			break;
		case LS.Animation: 
			var comp = this.getComponent( LS.Components.PlayAnimation );
			if(!comp)
				comp = this.addComponent( new LS.Components.PlayAnimation() );
			comp.animation = item.fullpath || item.filename;
			break;
		case LS.Resource: //generic resource
			var ext = LS.ResourcesManager.getExtension( item.filename );
			if(ext == "js") //scripts
			{
				var comp = this.getComponent( LS.Components.ScriptFromFile );
				if(!comp)
					comp = this.addComponent( new LS.Components.ScriptFromFile() );
				comp.src = item.fullpath || item.filename;
			}
			break;
		default:
			console.error("feature not supported loading this type of resource" , item );
	}
}

/**
* Simple way to assign a mesh to a node, it created a MeshRenderer component or reuses and existing one and assigns the mesh
* @method setMesh
* @param {string} mesh_name the name of the mesh (path to the file)
* @param {Number} submesh_id if you want to assign a submesh
**/
SceneNode.prototype.setMesh = function(mesh_name, submesh_id)
{
	var component = this.getComponent( LS.Components.MeshRenderer );
	if(component)
		component.configure({ mesh: mesh_name, submesh_id: submesh_id });
	else
		this.addComponent( new LS.MeshRenderer({ mesh: mesh_name, submesh_id: submesh_id }) );
}

SceneNode.prototype.getMaterial = function()
{
	if (!this.material)
		return null;
	if(this.material.constructor === String)
	{
		if( !this._in_tree )
			return null;
		if( this.material[0] == "@" )//uid
			return LS.ResourcesManager.materials_by_uid[ this.material ];
		return LS.ResourcesManager.materials[ this.material ];
	}
	return this.material;
}

/**
* Apply prefab info (skipping the root components) to node, so all children will be removed and components lost and overwritten
* It is called from prefab.applyToNodes when a prefab is loaded in memory
* @method reloadFromPrefab
**/
SceneNode.prototype.reloadFromPrefab = function()
{
	if(!this.prefab)
		return;

	var prefab = LS.ResourcesManager.resources[ this.prefab ];
	if(!prefab)
		return;

	if( prefab.constructor !== LS.Prefab )
		throw("prefab must be a LS.Prefab class");

	//apply info
	this.removeAllChildren();
	this.init( true, true ); //keep components, keep_info
	var prefab_data = prefab.prefab_data;
	
	//remove all but children info (prefabs overwrite only children info)
	prefab_data = { children: prefab.prefab_data.children };

	//uid data is already removed from the prefab
	this.configure( prefab_data );

	//load secondary resources 
	var resources = this.getResources( {}, true );
	LS.ResourcesManager.loadResources( resources );

	LEvent.trigger( this, "prefabReady", prefab );
}


/**
* Assigns this node to one layer
* @method setLayer
* @param {number|String} the index of the layer or the name (according to scene.layer_names)
* @param {boolean} value 
*/
SceneNode.prototype.setLayer = function( num_or_name, value )
{
	if( num_or_name == null )
		throw("setLayer expects layer");

	var num;

	if(num_or_name.constructor === String)
	{
		var scene = this.scene || LS.GlobalScene;
		var layer_num = scene.layer_names.indexOf( num_or_name );
		if(layer_num == -1)
		{
			console.error("Layer with name:",num_or_name,"not found in scene");
			return;
		}
		num = layer_num;
	}
	else
		num = num_or_name;

	var f = 1<<num;
	this.layers = (this.layers & (~f));
	if(value)
		this.layers |= f;
}

/**
* checks if this node is in the given layer
* @method isInLayer
* @param {number|String} index of layer or name according to scene.layer_names
* @return {boolean} true if belongs to this layer
*/
SceneNode.prototype.isInLayer = function( num_or_name )
{
	if( num_or_name == null )
		throw("setLayer expects layer");

	var num;

	if(num_or_name.constructor === String)
	{
		var scene = this.scene || LS.GlobalScene;
		var layer_num = scene.layer_names.indexOf( num_or_name );
		if(layer_num == -1)
		{
			console.error("Layer with name:",num_or_name,"not found in scene");
			return;
		}
		num = layer_num;
	}
	else
		num = num_or_name;

	return (this.layers & (1<<num)) !== 0;
}

SceneNode.prototype.getLayers = function()
{
	var r = [];
	if(!this.scene)
		return r;

	for(var i = 0; i < 32; ++i)
	{
		if( this.layers & (1<<i) )
			r.push( this.scene.layer_names[i] || ("layer"+i) );
	}
	return r;
}

/**
* Returns the root node of the prefab incase it is inside a prefab, otherwise null
* @method insidePrefab
* @return {Object} returns the node where the prefab starts
*/
SceneNode.prototype.insidePrefab = function()
{
	var aux = this;
	while( aux )
	{
		if(aux.prefab)
			return aux;
		aux = aux._parentNode;
	}
	return null;
}

/**
* remember clones this node and returns the new copy (you need to add it to the scene to see it)
* @method clone
* @return {Object} returns a cloned version of this node
*/
SceneNode.prototype.clone = function()
{
	var scene = this._in_tree;

	var new_name = scene ? scene.generateUniqueNodeName( this._name ) : this._name ;
	var newnode = new LS.SceneNode( new_name );
	var info = this.serialize();

	//remove all uids from nodes and components
	LS.clearUIds( info );

	info.uid = LS.generateUId("NODE-");
	newnode.configure( info );

	return newnode;
}

/**
* Configure this node from an object containing the info
* @method configure
* @param {Object} info the object with all the info (comes from the serialize method)
* @param {Array} components_aside array to store the data about components so they are configured after creating the scene has been created
*/
SceneNode.prototype.configure = function(info, components_aside)
{
	//identifiers parsing
	if (info.name)
		this.setName(info.name);
	else if (info.id)
		this.setName(info.id);
	if(info.layers !== undefined)
		this.layers = info.layers;

	if (info.uid)
		this.uid = info.uid;

	if (info.className && info.className.constructor == String)	
		this.className = info.className;

	if(info.node_type)
	{
		this.node_type = info.node_type;
		if(info.node_type == "JOINT") //used in editor
			this._is_bone = true;
	}

	//some helpers (mostly for when loading from js object that come from importers or code)
	if(info.camera)
		this.addComponent( new LS.Camera( info.camera ) );

	if(info.light)
		this.addComponent( new LS.Light( info.light ) );

	//in case more than one mesh in on e node
	if(info.meshes)
	{
		for(var i = 0; i < info.meshes.length; ++i)
			this.addMeshComponents( info.meshes[i], info );
	}
	else if(info.mesh)
		this.addMeshComponents( info.mesh, info );

	//transform in matrix format could come from importers so we leave it
	if((info.position || info.model || info.transform) && !this.transform)
		this.addComponent( new LS.Transform() );
	if(info.position) 
		this.transform.position = info.position;
	if(info.model) 
		this.transform.fromMatrix( info.model ); 
	if(info.matrix) 
		this.transform.fromMatrix( info.matrix ); 
	if(info.transform) 
		this.transform.configure( info.transform ); 

	//first the no components
	if(info.material)
	{
		var mat_classname = info.material.material_class;
		if(!mat_classname || mat_classname == "newStandardMaterial") //legacy
			mat_classname = "StandardMaterial";
		var constructor = LS.MaterialClasses[mat_classname];
		if(constructor)
			this.material = typeof(info.material) == "string" ? info.material : new constructor( info.material );
		else
			console.warn("Material not found: " + mat_classname );
	}

	if(info.flags) //merge
		for(var i in info.flags)
			this.flags[i] = info.flags[i];
	
	//add animation tracks player
	if(info.animation)
	{
		this.animation = info.animation;
		this.addComponent( new LS.Components.PlayAnimation({ animation: this.animation }) );
	}

	//extra user info
	if(info.extra)
		this.extra = info.extra;

	if(info.editor)
		this._editor = info.editor;


	if(info.comments)
		this.comments = info.comments;

	//configure components
	if(info.components)
	{
		if(components_aside)
			components_aside.push( this, info );
		else
			this.configureComponents( info );
	}

	if(info.prefab && !this._is_root)  //is_root because in some weird situations the prefab was set to the root node
		this.prefab = info.prefab; //assign and calls this.reloadFromPrefab();
	else //configure children if it is not a prefab
		this.configureChildren(info, components_aside);

	LEvent.trigger(this,"configure",info);
}

//adds components according to a mesh
//used mostly to addapt a node to a collada mesh info
SceneNode.prototype.addMeshComponents = function( mesh_id, extra_info )
{
	extra_info = extra_info || {};

	if(!mesh_id)
		return;

	if( mesh_id.constructor !== String )
	{
		extra_info = mesh_id;
		mesh_id = extra_info.mesh;
		if(!mesh_id)
		{
			console.warn("Mesh info without mesh id");
			return null;
		}
	}

	var mesh = LS.ResourcesManager.meshes[ mesh_id ];

	if(!mesh)
	{
		console.warn( "SceneNode mesh not found: " + mesh_id );
		return;
	}

	var mesh_render_config = { mesh: mesh_id };

	if(extra_info.submesh_id !== undefined)
		mesh_render_config.submesh_id = extra_info.submesh_id;
	if(extra_info.morph_targets !== undefined)
		mesh_render_config.morph_targets = extra_info.morph_targets;
	if(extra_info.material !== undefined)
		mesh_render_config.material = extra_info.material;

	var compo = new LS.Components.MeshRenderer( mesh_render_config );

	//parsed meshes have info about primitive
	if( mesh.primitive )
    {
        switch(mesh.primitive)
        {
    		case 'points': compo.primitive = GL.POINTS; break;
    		case 'lines': compo.primitive = GL.LINES; break;
    		case 'line_strip': compo.primitive = GL.LINE_STRIP; break;
        }
		delete mesh.primitive;
    }

	//add MeshRenderer
	this.addComponent( compo );

	//skinning
	if(mesh && mesh.bones)
	{
		compo = new LS.Components.SkinDeformer({ search_bones_in_parent: false }); //search_bones_in_parent is false because usually DAEs come that way
		this.addComponent( compo );
	}

	//morph targets
	if( mesh && mesh.morph_targets )
	{
		var compo = new LS.Components.MorphDeformer( { morph_targets: mesh.morph_targets } );
		this.addComponent( compo );
	}

}

/**
* Serializes this node by creating an object with all the info
* it contains info about the components too
* @method serialize
* @param {bool} ignore_prefab serializing wont returns children if it is a prefab, if you set this to ignore_prefab it will return all the info
* @return {Object} returns the object with the info
*/
SceneNode.prototype.serialize = function( ignore_prefab, simplified )
{
	var o = {
		object_class: "SceneNode"
	};

	if(this._name) 
		o.name = this._name;
	if(this.uid) 
		o.uid = this.uid;
	if(this.className) 
		o.className = this.className;
	o.layers = this.layers;

	//work in progress
	if(this.node_type)
		o.node_type = this.node_type;

	//modules
	if(this.mesh && typeof(this.mesh) == "string") 
		o.mesh = this.mesh; //do not save procedural meshes
	if(this.submesh_id != null) 
		o.submesh_id = this.submesh_id;
	if(this.material) 
		o.material = typeof(this.material) == "string" ? this.material : this.material.serialize( simplified );
	if(this.prefab && !ignore_prefab && !this._is_root ) 
		o.prefab = this.prefab;

	if(this.flags) 
		o.flags = LS.cloneObject(this.flags);

	//extra user info
	if(this.extra) 
		o.extra = this.extra;
	if(this.comments) 
		o.comments = this.comments;

	if(this._children && (!this.prefab || ignore_prefab) )
		o.children = this.serializeChildren( simplified );

	if(this._editor)
		o.editor = this._editor;

	//save components
	this.serializeComponents( o, simplified );

	//extra serializing info
	LEvent.trigger(this,"serialize",o);

	return o;
}

//used to recompute matrix so when parenting one node it doesnt lose its global transformation
SceneNode.prototype._onChildAdded = function( child_node, recompute_transform )
{
	if(recompute_transform && this.transform)
	{
		var M = child_node.transform.getGlobalMatrix(); //get son transform
		var M_parent = this.transform.getGlobalMatrix(); //parent transform
		mat4.invert(M_parent,M_parent);
		child_node.transform.fromMatrix( mat4.multiply(M_parent,M_parent,M) );
		child_node.transform.getGlobalMatrix(); //refresh
	}
	//link transform
	if(this.transform)
	{
		if(!child_node.transform)
			child_node.transform.addComponent( new LS.Transform() );
		child_node.transform._parent = this.transform;
	}
}

SceneNode.prototype._onChangeParent = function( future_parent, recompute_transform )
{
	if(recompute_transform && future_parent.transform)
	{
		var M = this.transform.getGlobalMatrix(); //get son transform
		var M_parent = future_parent.transform.getGlobalMatrix(); //parent transform
		mat4.invert(M_parent,M_parent);
		this.transform.fromMatrix( mat4.multiply(M_parent,M_parent,M) );
	}
	//link transform
	if(future_parent.transform)
		this.transform._parent = future_parent.transform;
}

SceneNode.prototype._onChildRemoved = function( node, recompute_transform, remove_components )
{
	if(this.transform)
	{
		//unlink transform
		if(node.transform)
		{
			if(recompute_transform)
			{
				var m = node.transform.getGlobalMatrix();
				node.transform._parent = null;
				node.transform.fromMatrix(m);
			}
			else
				node.transform._parent = null;
		}
	}

	if( remove_components )
		node.removeAllComponents();
}

//Computes the bounding box from the render instance of this node
//doesnt take into account children
SceneNode.prototype.getBoundingBox = function( bbox, only_instances )
{
	bbox = bbox || BBox.create();
	var render_instances = this._instances;
	if(render_instances)
		for(var i = 0; i < render_instances.length; ++i)
		{
			if(i == 0)
				bbox.set( render_instances[i].aabb );
			else
				BBox.merge( bbox, bbox, render_instances[i].aabb );
		}

	if(only_instances)
		return bbox;

	if( (!render_instances || render_instances.length == 0) && this.transform )
		return BBox.fromPoint( this.transform.getGlobalPosition() );

	return bbox;
}

LS.Scene.Node = SceneNode;
LS.SceneNode = SceneNode;
LS.Classes.SceneNode = SceneNode;

///@FILE:../src/resources/resource.js
///@INFO: BASE
/**
* This class contains all the info about a resource and it works as a template for any resource class
* Keep in mind that there are many resource classes like Meshes or Textures that DONT INHERIT FROM THIS CLASS.
* This class is used mainly to generic file resources like text files (scripts, csvs, etc)
*
* @class Resource
* @constructor
*/

function Resource()
{
	this.filename = null; //name of file without folder or path
	this.fullpath = null; //contains the unique name as is to be used to fetch it by the resources manager
	this.remotepath = null; //the string to fetch this resource in internet (local resources do not have this name)
	this._data = null;
	//this.type = 0;
}

//Resource.DATA = 1;
//Resource.SCRIPT = 2;

Object.defineProperty( Resource.prototype, "data", {
	set: function(v){ 
		this._data = v;
		this._modified = true;
	},
	get: function() { 
		return this._data;
	},
	enumerable: true
});

/** makes this resource available by registering in the resources manager
* @method rename
*/
Resource.prototype.register = function()
{
	LS.ResourcesManager.registerResource( this.fullpath || this.filename, this );
}

/** Renames the resource and ensures the resources manager is updated accordingly
* @method rename
* @param {String} new_filename the new filename
*/
Resource.prototype.rename = function( new_filename )
{
	LS.ResourcesManager.renameResource( this.fullpath || this.filename, new_filename );
}

Object.defineProperty( Resource.prototype, "uid", { 
	get: function(){ return this.fullpath || this.filename },
	set: function(v){},
	enumerable: true
});

/**
* Static method: Returns an object with a representation of the resource internal data
* The order to obtain that object is:
* 0. checks if getDataToStore function in resource
* 1. test for _original_file (File or Blob)
* 2. test for _original_data (ArrayBuffer)
* 3. toBinary() (ArrayBuffer)
* 4. toBlob() (Blob)
* 5. toBase64() (String)
* 6. serialize() (Object in JSON format)
* 7. data property 
* 8. JSON.stringify(...)
*
* @method Resource.getDataToStore
* @param {Object} resource 
* @param {Boolean} allow_blob [optional] 
* @return {Object} it has two fields: data and encoding
*/
Resource.getDataToStore = function( resource, allow_blob )
{
	var data = null;
	var encoding = "text";
	var extension = "";

	//get the data
	if (resource.getDataToStore) //function
	{
		data = resource.getDataToStore();
		if(data && data.constructor == ArrayBuffer)
			encoding = "binary";
	}
	else if (resource._original_file) //file
	{
		data = resource._original_file;
		if(data && data.constructor !== File && data.constructor !== Blob)
			console.warn("Resource._original_file is not File or Blob");
		encoding = "file";
	}
	else if( resource._original_data ) //file in ArrayBuffer format
	{
		data = resource._original_data;
		if( data && data.constructor === ArrayBuffer )
			encoding = "binary";
	}
	else if(resource.toBinary) //a function to compute the ArrayBuffer format
	{
		if( resource.constructor === GL.Texture ) //HACK: textures require that extra parameter...
			data = resource.toBinary(true);
		else
			data = resource.toBinary();
		encoding = "binary";
		if(resource.constructor.binary_extension) //special case, textures are in PNG to keep alpha
			extension = resource.constructor.binary_extension;
		else
			extension = "wbin";
	}
	else if(resource.toBlob && allow_blob) //a blob (Canvas should have this)
	{
		data = resource.toBlob();
		encoding = "file";
	}
	else if(resource.toBase64) //a base64 string
	{
		data = resource.toBase64();
		encoding = "base64";
	}
	else if(resource.serialize) //a json object
	{
		var obj = resource.serialize();
		//remove inner stuff from the editor
		delete obj.filename;
		delete obj.fullpath;
		delete obj.remotepath;
		delete obj.preview_url; //just in case is an old resource
		//convert to string
		data = JSON.stringify( obj );
	}
	else if(resource.data) //regular string data
		data = resource.data;
	else
		data = JSON.stringify( resource );

	if(data.buffer && data.buffer.constructor == ArrayBuffer)
		data = data.buffer; //store the data in the arraybuffer

	return { data:data, encoding: encoding, extension: extension };
}

//used in the coding pad to assign content to generic text files
Resource.prototype.getData = function()
{
	return this._data;
}

Resource.prototype.setData = function( v, skip_modified_flag )
{
	//remove old file
	if( this._original_data )
		this._original_data = null;
	this._data = v;
	if(!skip_modified_flag)
		this._modified = true;
}

Resource.prototype.getDataToStore = function()
{
	var data = this.data || "";
	if(data.constructor === Object )
		data = JSON.stringify( data );
	return data;
}

/** Clone the resource
* @method clone
* @return {LS.Resource} the clone of the resource
*/
Resource.prototype.clone = function()
{
	var r = new LS.Resource();
	r._data = this._data;
	return r;
}

/** Returns a string representing to which category this resource belongs
* @method getCategory
*/
Resource.prototype.getCategory = function()
{
	var filename = this.fullpath || this.filename;
	var ext = LS.ResourcesManager.getExtension( filename );
	if(ext == "js")
		return "Script";
	return "Data";
}

/** When dropping this resource into a SceneNode
* @method assignToNode
*/
Resource.prototype.assignToNode = function(node)
{
	if(!node) 
		return false;

	var filename = this.fullpath || this.filename;
	var category = this.getCategory();

	if( category == "Script" )
	{
		var script_component = new LS.Components.ScriptFromFile({ filename: filename });
		node.addComponent( script_component );
	}

	return true;
}

/** Parses the resource data as subfiles (subfiles are fragments of the code identified by a slash followed by name string), used by ShaderCode
* @method getAsSubfiles
* @return {Object} the object that contains every subfile
*/
Resource.prototype.getAsSubfiles = function()
{
	if(!this._data)
		return null;
	return GL.processFileAtlas( this._data );
}

/** Parses the resource as HTML code and returns a HTMLElement containing the html code
* @method getAsHTML
* @return {HTMLElement} the root HTMLElement that contains the code
*/
Resource.prototype.getAsHTML = function()
{
	if(!this._data || this._data.constructor !== String)
		return null;

	var container = document.createElement("div");
	container.innerHTML = this._data;
	return container;
}

/** Used by the editor to know if it can be edited in the text editor
* @method hasEditableText
*/
Resource.prototype.hasEditableText = function()
{
	return this._data && this._data.constructor === String;
}

Resource.hasPreview = false; //should this resource use a preview image?

LS.Resource = Resource;
LS.registerResourceClass( Resource );
///@FILE:../src/resources/animation.js
///@INFO: ANIMATION

/**
* An Animation is a resource that contains samples of properties over time, similar to animation curves
* Values could be associated to an specific node.
* Data is contained in tracks
*
* @class Animation
* @namespace LS
* @constructor
*/

function Animation(o)
{
	this.name = "";
	this.takes = {}; //packs of tracks
	if(o)
		this.configure(o);
}

LS.Classes["Animation"] = LS.Animation = Animation;

//Animation.KEYFRAME_ANIMATION = 0;
//Animation.CLIP_ANIMATION = 1;

Animation.EXTENSION = "wbin";
Animation.DEFAULT_SCENE_NAME = "@scene";
Animation.DEFAULT_DURATION = 10;

/**
* Create a new take inside this animation (a take contains all the tracks)
* @method createTake
* @param {String} name the name
* @param {Number} duration
* @return {LS.Animation.Take} the take
*/
Animation.prototype.createTake = function( name, duration )
{
	if(!name)
		throw("Animation Take name missing");

	var take = new Animation.Take();
	take.name = name;
	take.duration = duration;
	if(duration === undefined)
		take.duration = Animation.DEFAULT_DURATION;
	this.addTake( take );
	return take;
}

/**
* adds an existing take
* @method addTake
* @param {LS.Animation.Take} name the name
*/
Animation.prototype.addTake = function(take)
{
	this.takes[ take.name ] = take;
	return take;
}

/**
* returns the take with a given name
* @method getTake
* @param {String} name
* @return {LS.Animation.Take} the take
*/
Animation.prototype.getTake = function( name )
{
	return this.takes[ name ];
}

/**
* renames a take name
* @method renameTake
* @param {String} old_name
* @param {String} new_name
*/
Animation.prototype.renameTake = function( old_name, new_name )
{
	var take = this.takes[ old_name ];
	if(!take)
		return;
	delete this.takes[ old_name ];
	take.name = new_name;
	this.takes[ new_name ] = take;
	LEvent.trigger( this, "take_renamed", [old_name, new_name] );
}

/**
* removes a take
* @method removeTake
* @param {String} name
*/
Animation.prototype.removeTake = function( name )
{
	var take = this.takes[ name ];
	if(!take)
		return;
	delete this.takes[ name ];
	LEvent.trigger( this, "take_removed", name );
}

/**
* returns the number of takes
* @method getNumTakes
* @return {Number} the number of takes
*/
Animation.prototype.getNumTakes = function()
{
	var num = 0;
	for(var i in this.takes)
		num++;
	return num;
}

Animation.prototype.addTrackToTake = function(takename, track)
{
	var take = this.takes[ takename ];
	if(!take)
		take = this.createTake( takename );
	take.addTrack( track );
}


Animation.prototype.configure = function(data)
{
	if(data.name)
		this.name = data.name;

	if(data.takes)
	{
		var num_takes = 0;
		this.takes = {};
		for(var i in data.takes)
		{
			var take = new LS.Animation.Take( data.takes[i] );
			if(!take.name)
				console.warn("Take without name");
			else
			{
				this.addTake( take );
				take.loadResources(); //load associated resources
			}
			num_takes++;
		}
		if(!num_takes)
			this.createTake("default", LS.Animation.DEFAULT_DURATION );
	}
}

Animation.prototype.serialize = function()
{
	return LS.cloneObject(this, null, true);
}

Animation.fromBinary = function( data )
{
	if(data.constructor == ArrayBuffer)
		data = WBin.load(data, true);

	var o = data["@json"];
	if(!o) //sometimes the data already comes extractedin the object itself
		o = data;

	for(var i in o.takes)
	{
		var take = o.takes[i];
		for(var j in take.tracks)
		{
			var track = take.tracks[j];
			var name = "@take_" + i + "_track_" + j;
			if( data[name] )
				track.data = data[name];
		}
	}

	return new LS.Animation( o );
}

Animation.prototype.toBinary = function()
{
	var o = {};
	var tracks_data = [];

	//we need to remove the bin data to generate the JSON
	for(var i in this.takes)
	{
		var take = this.takes[i];
		for(var j in take.tracks)
		{
			var track = take.tracks[j];
			track.packData(); //reduce storage space and speeds up loading

			if(track.packed_data)
			{
				var bindata = track.data;
				var name = "@take_" + i + "_track_" + j;
				o[name] = bindata;
				track.data = null;
				tracks_data.push( bindata );
			}
		}
	}

	//create the binary
	o["@json"] = LS.cloneObject(this, null, true);
	var bin = WBin.create(o, "Animation");

	//restore the bin data state in this instance
	for(var i in this.takes)
	{
		var take = this.takes[i];
		for(var j in take.tracks)
		{
			var track = take.tracks[j];
			var name = "@take_" + i + "_track_" + j;
			if(o[name])
				track.data = o[name];
		}
	}

	return bin;
}

//Used when the animation tracks use names instead of node ids
//to convert the track locator to node names, so they affect to only one node
Animation.prototype.convertNamesToIDs = function( use_basename, root )
{
	var num = 0;
	for(var i in this.takes)
	{
		var take = this.takes[i];
		num += take.convertNamesToIDs( use_basename, root );
	}
	return num;
}

//Used when the animation tracks use UIDs instead of node names
//to convert the track locator to node names, so they can be reused between nodes in the same scene
Animation.prototype.convertIDsToNames = function( use_basename, root )
{
	var num = 0;
	for(var i in this.takes)
	{
		var take = this.takes[i];
		num += take.convertIDsToNames( use_basename, root );
	}
	return num;
}

/**
* changes the packing mode of the tracks inside all takes
* @method setTracksPacking
* @param {boolean} pack if true the tracks will be packed (used a typed array)
* @return {Number} te number of modifyed tracks
*/
Animation.prototype.setTracksPacking = function(v)
{
	var num = 0;
	for(var i in this.takes)
	{
		var take = this.takes[i];
		num += take.setTracksPacking(v);
	}
	return num;
}

/**
* optimize all the tracks in all the takes, so they take less space and are faster to compute (when possible)
* @method optimizeTracks
* @return {Number} the number of takes
*/
Animation.prototype.optimizeTracks = function()
{
	var num = 0;
	for(var i in this.takes)
	{
		var take = this.takes[i];
		num += take.optimizeTracks();
	}
	return num;
}

/**
* It creates a PlayAnimation component to the node (or reuse and old existing one). Used when a resource is assigned to a node
* @method assignToNode
* @param {LS.SceneNode} node node where to assign this animation
*/
Animation.prototype.assignToNode = function(node)
{
	var component = node.getComponent( LS.Components.PlayAnimation );
	if(!component)
		component = node.addComponent( LS.Components.PlayAnimation );
	component.animation = this.fullpath || this.filename;
}



/**  
* Represents a set of animations
*
* @class Take
* @namespace LS.Animation
* @constructor
*/
function Take(o)
{
	/** 
	* @property name {String}
	**/
	this.name = null;
	/** 
	* @property tracks {Array}
	**/
	this.tracks = [];
	/** 
	* @property duration {Number} in seconds
	**/
	this.duration = LS.Animation.DEFAULT_DURATION;
	
	if(o)
		this.configure(o);

}

Take.prototype.clear = function()
{
	this.tracks = [];
}

Take.prototype.configure = function( o )
{
	if( o.name )
		this.name = o.name;
	if( o.tracks ) 
	{
		this.tracks = []; //clear
		for(var i in o.tracks)
		{
			var track = new LS.Animation.Track( o.tracks[i] );
			this.addTrack( track );
		}
	}
	if( o.duration )
		this.duration = o.duration;
}

Take.prototype.serialize = Take.prototype.toJSON = function()
{
	return LS.cloneObject(this, null, true);
}

/**
* creates a new track from a given data
* @method createTrack
* @param {Object} data in serialized format
* @return {LS.Animation.Track} the track
*/
Take.prototype.createTrack = function( data )
{
	if(!data)
		throw("Data missing when creating track");

	var track = this.getTrack( data.property );
	if( track )
		return track;

	var track = new LS.Animation.Track( data );
	this.addTrack( track );
	return track;
}

/**
* For every track, gets the interpolated value between keyframes and applies the value to the property associated with the track locator
* Locators are in the form of "{NODE_UID}/{COMPONENT_UID}/{property_name}"
*
* @method applyTracks
* @param {number} current_time the time of the anim to sample
* @param {number} last_time this is used for events, we need to know where you were before
* @param {boolean} ignore_interpolation in case you want to sample the nearest one
* @param {SceneNode} weight [Optional] allows to blend animations with current value (default is 1)
* @param {Number} root [Optional] if you want to limit the locator to search inside a node
* @param {Function} on_pre_apply [Optional] a callback called per track to see if this track should be applyed, if it returns false it is skipped. callback receives (track, current_time, root_node, weight)
* @param {Function} on_apply_sample [Optional] a callback called before applying a keyframe, if the callback returns false the keyframe will be skipped. callback parameters ( track, sample, root_node, weight )
* @return {Component} the target where the action was performed
*/
Take.prototype.applyTracks = function( current_time, last_time, ignore_interpolation, root_node, scene, weight, on_pre_apply, on_apply_sample )
{
	scene = scene || LS.GlobalScene;
	if(weight === 0)
		return;

	weight = weight || 1;

	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		if( track.enabled === false || !track.data )
			continue;

		if(on_pre_apply && on_pre_apply( track, current_time, root_node, weight ) === false)
			continue;

		//events are an special kind of tracks, they execute actions
		if( track._type_index == Track.EVENT )
		{
			var keyframe = track.getKeyframeByTime( current_time );
			if( !keyframe || keyframe[0] < last_time || keyframe[0] > current_time )
				return;

			//need info to search for node
			var info = scene.getPropertyInfoFromPath( track._property_path );
			if(!info)
				return;

			var value = keyframe[1];

			if(value[2] == 1) //on function call events the thirth value [2] is 1
			{
				//functions
				if(info.node && info.target && info.target[ value[0] ] )
					info.target[ value[0] ].call( info.target, value[1] );
			}
			else
			{
				//events
				if(info.target) //components
					LEvent.trigger( info.target, keyframe[1], keyframe[1][1] );
				else if(info.node) //nodes
					LEvent.trigger( info.node, keyframe[1][0], keyframe[1][1] );
			}
		}
		else //regular tracks
		{
			//read from the animation track the value
			var sample = track.getSample( current_time, !ignore_interpolation );

			//to blend between animations...
			if(weight !== 1)
			{
				var current_value = scene.getPropertyValueFromPath( track._property_path, sample, root_node, 0 );
				sample = LS.Animation.interpolateLinear( sample, current_value, weight, null, track._type, track.value_size, track );
			}

			//apply the value to the property specified by the locator
			if( sample !== undefined ) 
			{
				if( on_apply_sample && on_apply_sample( track, sample, root_node, weight ) === false)
					continue; //skip
				track._target = scene.setPropertyValueFromPath( track._property_path, sample, root_node, 0 );
			}
		}
	}
}



Take.prototype.addTrack = function( track )
{
	this.tracks.push( track );
}

/**
* returns a track given its index or the property string
* @method getTrack
* @param {Number|String} property could be index or property
* @return {LS.Animation.Track} the track
*/
Take.prototype.getTrack = function( property )
{
	if(property == null)
		return null;
	if(property.constructor === Number)
		return this.tracks[property];
	if(property.constructor === String)
	for(var i = 0; i < this.tracks.length; ++i)
		if(this.tracks[i].property == property)
			return this.tracks[i];
	return null;
}

Take.prototype.removeTrack = function( track )
{
	for(var i = 0; i < this.tracks.length; ++i)
		if(this.tracks[i] == track)
		{
			this.tracks.splice( i, 1 );
			return;
		}
}


Take.prototype.getPropertiesSample = function( time, result )
{
	result = result || [];
	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		var value = track.getSample( time );
		result.push([track.property, value]);
	}
	return result;
}

Take.prototype.actionPerSample = function(time, callback, options)
{
	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		var value = track.getSample(time, true);
		if( options.disabled_tracks && options.disabled_tracks[ track.property ] )
			continue;
		callback(track.property, value, options);
	}
}

//Ensures all the resources associated to keyframes are loaded in memory
Take.prototype.loadResources = function()
{
	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		if(track._type == "texture")
		{
			var l = track.getNumberOfKeyframes();
			for(var j = 0; j < l; ++j)
			{
				var keyframe = track.getKeyframe(j);
				if(keyframe && keyframe[1] && keyframe[1][0] != ":")
					LS.ResourcesManager.load( keyframe[1] );
			}
		}
	}
}

//convert track locators from using UIDs to use node names (this way the same animation can be used in several parts of the scene)
Take.prototype.convertNamesToIDs = function( use_basename, root )
{
	var num = 0;
	for(var j = 0; j < this.tracks.length; ++j)
	{
		var track = this.tracks[j];
		num += track.convertNameToID( use_basename, root )
	}
	return num;
}

//convert track locators from using UIDs to use node names (this way the same animation can be used in several parts of the scene)
Take.prototype.convertIDsToNames = function( use_basename, root )
{
	var num = 0;
	for(var j = 0; j < this.tracks.length; ++j)
	{
		var track = this.tracks[j];
		num += track.convertIDtoName( use_basename, root )
	}
	return num;
}

Take.prototype.setTracksPacking = function(v)
{
	var num = 0;
	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		if( track.packed_data == v)
			continue;
		if(v)
			track.packData();
		else
			track.unpackData();
		num += 1;
	}
	return num;
}

/**
* Optimizes the tracks by changing the Matrix tracks to Trans10 tracks which are way faster and use less space
* @method optimizeTracks
*/
Take.prototype.optimizeTracks = function()
{
	var num = 0;
	var temp = new Float32Array(10);

	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		if( track.convertToTrans10() )
			num += 1;
	}
	return num;
}

Take.prototype.setInterpolationToAllTracks = function( interpolation )
{
	var num = 0;
	for(var i = 0; i < this.tracks.length; ++i)
	{
		var track = this.tracks[i];
		if(track.interpolation == interpolation)
			continue;
		track.interpolation = interpolation;
		num += 1;
	}

	return num;
}




Animation.Take = Take;


/**
* Represents one track with data over time about one property
* Data could be stored in two forms, or an array containing arrays of [time,data] (unpacked data) or in a single typed array (packed data), depends on the attribute typed_mode
*
* @class Animation.Track
* @namespace LS
* @constructor
*/

//KeyframesTrack: 0
//ClipsTrack: 1

function Track(o)
{
	/** 
	* @property enabled {Boolean} if it must be applied
	**/
	this.enabled = true;
	/** 
	* @property name {String} title to show in the editor
	**/
	this.name = ""; //title
	/** 
	* @property type {String} if the data is number, vec2, color, etc
	**/
	this._type = null; //type of data (number, vec2, color, texture, etc)
	this._type_index = null; //type in number format (to optimize)
	/** 
	* @property interpolation {Number} type of interpolation LS.NONE, LS.LINEAR, LS.TRIGONOMETRIC, LS.CUBIC, LS.SPLICE
	**/
	this.interpolation = LS.NONE;
	/** 
	* @property looped {Boolean} if the last and the first keyframe should be connected
	**/
	this.looped = false; //interpolate last keyframe with first

	//data
	this.packed_data = false; //this means the data is stored in one continuous datatype, faster to load but not editable
	this.value_size = 0; //how many numbers contains every sample of this property, 0 means basic type (string, boolean)
	/** 
	* @property data {*} contains all the keyframes, could be an array or a typed array
	**/
	this.data = null; //array or typed array where you have the time value followed by this.value_size bytes of data
	this.data_table = null; //used to index data when storing it

	//to speed up sampling
	Object.defineProperty( this, '_property', {
		value: "",
		enumerable: false,
		writable: true
	});

	Object.defineProperty( this, '_property_path', {
		value: [],
		enumerable: false,
		writable: true
	});

	if(o)
		this.configure(o);
}

Animation.Track = Track;

Track.FRAMERATE = 30;

//for optimization
Track.QUAT = LS.TYPES_INDEX["quat"];
Track.TRANS10 = LS.TYPES_INDEX["trans10"];
Track.EVENT = LS.TYPES_INDEX["event"];

/** 
* @property property {String} the locator to the property this track should modify ( "node/component_uid/property" )
**/
Object.defineProperty( Track.prototype, 'property', {
	set: function( property )
	{
		this._property = property.trim();
		this._property_path = this._property.split("/");
	},
	get: function(){
		return this._property;
	},
	enumerable: true
});

Object.defineProperty( Track.prototype, 'type', {
	set: function( t )
	{
		this._type = t;
		this._type_index = LS.TYPES_INDEX[t];
	},
	get: function(){
		return this._type;
	},
	enumerable: true
});

Track.prototype.configure = function( o )
{
	if(!o.property)
		console.warn("Track with property name");

	if(o.enabled !== undefined) this.enabled = o.enabled;
	if(o.name) this.name = o.name;
	if(o.property) this.property = o.property;
	if(o.type) this.type = o.type;
	if(o.looped) this.looped = o.looped;
	if(o.interpolation !== undefined)
		this.interpolation = o.interpolation;
	else
		this.interpolation = LS.LINEAR;

	if(o.data_table) this.data_table = o.data_table;

	if(o.value_size) this.value_size = o.value_size;

	//data
	if(o.data)
	{
		this.data = o.data;

		//this is ugly but makes it easy to work with the collada importer
		if(o.packed_data === undefined && this.data.constructor !== Array)
			this.packed_data = true;
		else
			this.packed_data = !!o.packed_data;

		if( o.data.constructor == Array )
		{
			if( this.packed_data )
				this.data = new Float32Array( o.data );
			else
				this.data = o.data.concat();
		}
		//else
		//	this.unpackData();
	}

	if(o.interpolation && !this.value_size)
		o.interpolation = LS.NONE;
}

Track.prototype.serialize = function()
{
	var o = {
		enabled: this.enabled,
		name: this.name,
		property: this.property, 
		type: this._type,
		interpolation: this.interpolation,
		looped: this.looped,
		value_size: this.value_size,
		packed_data: this.packed_data,
		data_table: this.data_table
	}

	if(this.data)
	{
		if(this.value_size <= 1)
		{
			if(this.data.type == "event")
			{
				//weird bug where the track contains components
				for(var i = 0; i < data.length; ++i)
				{
					var k = data[i];
					if(k[1] && k[1].constructor.is_component)
						k[1] = null;
				}
			}

			if(this.data.concat)
				o.data = this.data.concat(); //regular array, clone it
			else
				o.data = new this.data.constructor( this.data ); //clone for typed arrays (weird, this should never happen but it does)
		}
		else //pack data
		{
			this.packData();
			o.data = new Float32Array( this.data ); //clone it
			o.packed_data = this.packed_data;
		}
	}

	return o;
}

Track.prototype.toJSON = Track.prototype.serialize;

Track.prototype.clear = function()
{
	this.data = [];
	this.packed_data = false;
}

/**
* used to change every track so instead of using UIDs for properties it uses names
* this is used when you want to apply the same animation to different nodes in the scene
* @method getIDasName
* @param {boolean} use_basename if you want to just use the node name, othewise it uses the fullname (name with path)
* @param {LS.SceneNode} root
* @return {String} the result name
*/
Track.prototype.getIDasName = function( use_basename, root )
{
	if( !this._property_path || !this._property_path.length )
		return null;

	return LS.convertLocatorFromUIDsToName( this._property,  use_basename, root );
}

//used to change every track so instead of using node names for properties it uses node uids
//this is used when you want to apply an animation to an specific node
Track.prototype.convertNameToID = function( root )
{
	if(this._property_path[0][0] === LS._uid_prefix)
		return false; //is already using UIDs

	var node = LSQ.get( this._property_path[0], root );
	if(!node)
		return false;

	//convert node uid
	this._property_path[0] = node.uid;

	//convert component uid
	if( this._property_path.length > 1)
	{
		var comp = node.getComponent( this._property_path[1] );
		if(comp)
			this._property_path[1] = comp.uid;
	}

	this._property = this._property_path.join("/");
	return true;
}

//used to change every track so instead of using UIDs for properties it uses node names
//this is used when you want to apply the same animation to different nodes in the scene
Track.prototype.convertIDtoName = function( use_basename, root )
{
	var name = this.getIDasName( use_basename, root );
	if(!name)
		return false;
	this._property = name;
	this._property_path = this._property.split("/");
	return true;
}

/**
* If the track used matrices, it transform them to position,quaternion and scale (10 floats, also known as trans10)
* this makes working with animations faster
* @method convertToTrans10
*/
Track.prototype.convertToTrans10 = function()
{
	if( this.value_size != 16 )
		return false;

	//convert samples
	if(!this.packed_data)
		this.packData();

	//convert locator
	var path = this.property.split("/");
	if( path[ path.length - 1 ] != "matrix") //from "nodename/matrix" to "nodename/transform/data"
		return false;

	path[ path.length - 1 ] = "Transform/data";
	this.property = path.join("/");
	this.type = "trans10";
	this.value_size = 10;
	var temp = new Float32Array(10);

	var data = this.data;
	var num_samples = data.length / 17;
	for(var k = 0; k < num_samples; ++k)
	{
		var sample = data.subarray(k*17+1,(k*17)+17);
		LS.Transform.fromMatrix4ToTransformData( sample, temp );
		data[k*11] = data[k*17]; //timestamp
		data.set(temp,k*11+1); //overwrite inplace (because the output is less big that the input)
	}
	this.data = new Float32Array( data.subarray(0,num_samples*11) );

	return true;
}

/**
* Adds a new keyframe from the current value of that property
* @method addKeyframeFromCurrent
* @param {Number} time time stamp in seconds
* @param {LS.SceneTree} scene 
*/
Track.prototype.addKeyframeFromCurrent = function( time, scene )
{
	scene = scene || LS.GlobalScene;
	var info = scene.getPropertyInfoFromPath( this._property_path );
	if(!info)
		return null;
	return this.addKeyframe( time, info.value );
}

/**
* Adds a new keyframe to this track given a value
* @method addKeyframe
* @param {Number} time time stamp in seconds
* @param {*} value anything you want to store, if omited then the current value is used
* @param {Boolean} skip_replace if you want to replace existing keyframes at same time stamp or add it next to that
* @return {Number} index of keyframe
*/
Track.prototype.addKeyframe = function( time, value, skip_replace )
{
	if(this.value_size > 1)
		value = new Float32Array( value ); //clone

	if(this.packed_data)
		this.unpackData();

	if(!this.data)
		this.data = [];

	for(var i = 0; i < this.data.length; ++i)
	{
		if(this.data[i][0] < time )
			continue;
		if(this.data[i][0] == time && !skip_replace )
			this.data[i][1] = value;
		else
			this.data.splice(i,0, [time,value]);
		return i;
	}

	this.data.push( [time,value] );
	return this.data.length - 1;
}

/**
* returns a keyframe given an index
* @method getKeyframe
* @param {Number} index
* @return {Array} the keyframe in [time,data] format
*/
Track.prototype.getKeyframe = function( index )
{
	if(index < 0 || index >= this.data.length)
	{
		console.warn("keyframe index out of bounds");
		return null;
	}

	if(this.packed_data)
	{
		var pos = index * (1 + this.value_size );
		if(pos > (this.data.length - this.value_size) )
			return null;
		return [ this.data[pos], this.data.subarray(pos+1, pos+this.value_size+1) ];
		//return this.data.subarray(pos, pos+this.value_size+1) ];
	}

	return this.data[ index ];
}

/**
* returns the first keyframe that matches this time
* @method getKeyframeByTime
* @param {Number} time
* @return {Array} keyframe in [time,value]
*/
Track.prototype.getKeyframeByTime = function( time )
{
	var index = this.findTimeIndex( time );
	if(index == -1)
		return;
	return this.getKeyframe( index );
}

/**
* changes a keyframe time and rearranges it
* @method moveKeyframe
* @param {Number} index
* @param {Number} new_time
* @return {Number} new index
*/
Track.prototype.moveKeyframe = function(index, new_time)
{
	if(this.packed_data)
	{
		//TODO
		console.warn("Cannot move keyframes if packed");
		return -1;
	}

	if(index < 0 || index >= this.data.length)
	{
		console.warn("keyframe index out of bounds");
		return -1;
	}

	var new_index = this.findTimeIndex( new_time );
	var keyframe = this.data[ index ];
	var old_time = keyframe[0];
	if(old_time == new_time)
		return index;
	keyframe[0] = new_time; //set time
	if(old_time > new_time)
		new_index += 1;
	if(index == new_index)
	{
		//console.warn("same index");
		return index;
	}

	//extract
	this.data.splice(index, 1);
	//reinsert
	index = this.addKeyframe( keyframe[0], keyframe[1], true );

	this.sortKeyframes();
	return index;
}

/**
* Sometimes when moving keyframes they could end up not sorted by timestamp, which will cause problems when sampling, to avoid it, we can force to sort all keyframes
* @method sortKeyframes
*/
Track.prototype.sortKeyframes = function()
{
	if(this.packed_data)
	{
		this.unpackData();
		this.sortKeyframes();
		this.packData();
	}
	this.data.sort( function(a,b){ return a[0] - b[0];  });
}

/**
* removes one keyframe
* @method removeKeyframe
* @param {Number} index
*/
Track.prototype.removeKeyframe = function(index)
{
	if(this.packed_data)
		this.unpackData();

	if(index < 0 || index >= this.data.length)
	{
		console.warn("keyframe index out of bounds");
		return;
	}

	this.data.splice(index, 1);
}

/**
* returns the number of keyframes
* @method getNumberOfKeyframes
*/

Track.prototype.getNumberOfKeyframes = function()
{
	if(!this.data || this.data.length == 0)
		return 0;

	if(this.packed_data)
		return this.data.length / (1 + this.value_size );
	return this.data.length;
}

//check for the last sample time
Track.prototype.computeDuration = function()
{
	if(!this.data || this.data.length == 0)
		return 0;

	if(this.packed_data)
	{
		var time = this.data[ this.data.length - 2 - this.value_size ];
		this.duration = time;
		return time;
	}

	//not typed
	var last = this.data[ this.data.length - 1 ];
	if(last)
		return last[0];
	return 0;
}

Track.prototype.isInterpolable = function()
{
	if( this.value_size > 0 || LS.Interpolators[ this._type ] )
		return true;
	return false;
}

/**
* takes all the keyframes and stores them inside a typed-array so they are faster to store in server or work with
* @method packData
*/
Track.prototype.packData = function()
{
	if(!this.data || this.data.length == 0)
		return 0;

	if(this.packed_data)
		return;

	if(this.value_size == 0)
		return; //cannot be packed (bools and strings cannot be packed)

	var offset = this.value_size + 1;
	var data = this.data;
	var typed_data = new Float32Array( data.length * offset );

	for(var i = 0; i < data.length; ++i)
	{
		typed_data[i*offset] = data[i][0];
		if( this.value_size == 1 )
			typed_data[i*offset+1] = data[i][1];
		else
			typed_data.set( data[i][1], i*offset+1 );
	}

	this.data = typed_data;
	this.packed_data = true;
}

/**
* takes all the keyframes and unpacks them so they are in a simple array, easier to work with
* @method unpackData
*/
Track.prototype.unpackData = function()
{
	if(!this.data || this.data.length == 0)
		return 0;

	if(!this.packed_data)
		return;

	var offset = this.value_size + 1;
	var typed_data = this.data;
	var data = Array( typed_data.length / offset );

	for(var i = 0; i < typed_data.length; i += offset )
		data[i/offset] = [ typed_data[i], typed_data.subarray( i+1, i+offset ) ];

	this.data = data;
	this.packed_data = false;
}

/**
* Returns nearest index of keyframe with time equal or less to specified time (Dichotimic search)
* @method findTimeIndex
* @param {number} time
* @return {number} the nearest index (lower-bound)
*/
Track.prototype.findTimeIndex = function(time)
{
	var data = this.data;
	if(!data || data.length == 0)
		return -1;

	if(this.packed_data)
	{
		var offset = this.value_size + 1; //data size plus timestamp
		var l = data.length;
		var n = l / offset; //num samples
		var imin = 0;
		var imid = 0;
		var imax = n;

		if(n == 0)
			return -1;
		if(n == 1)
			return 0;

		//time out of duration
		if( data[ (imax - 1) * offset ] < time )
			return (imax - 1);

		//dichotimic search
		// continue searching while [imin,imax] are continuous
		while (imax >= imin)
		{
			// calculate the midpoint for roughly equal partition
			imid = ((imax + imin)*0.5)|0;
			var t = data[ imid * offset ]; //get time
			if( t == time )
				return imid; 
			//when there are no more elements to search
			if( imin == (imax - 1) )
				return imin;
			// determine which subarray to search
			if (t < time)
				// change min index to search upper subarray
				imin = imid;
			else         
				// change max index to search lower subarray
				imax = imid;
		}
		return imid;
	}

	//unpacked data
	var n = data.length; //num samples
	var imin = 0;
	var imid = 0;
	var imax = n;

	if(n == 0)
		return -1;
	if(n == 1)
		return 0;

	//time out of duration
	if( data[ (imax - 1) ][0] < time )
		return (imax - 1);

	while (imax >= imin)
	{
		// calculate the midpoint for roughly equal partition
		imid = ((imax + imin)*0.5)|0;
		var t = data[ imid ][0]; //get time
		if( t == time )
			return imid; 
		//when there are no more elements to search
		if( imin == (imax - 1) )
			return imin;
		// determine which subarray to search
		if (t < time)
			// change min index to search upper subarray
			imin = imid;
		else         
			// change max index to search lower subarray
			imax = imid;
	}

	return imid;
}

/**
* Samples the data in one time, taking into account interpolation.
* Warning: if no result container is provided the same container is reused between samples to avoid garbage, be careful.
* @method getSample
* @param {number} time
* @param {number} interpolation [optional] the interpolation method could be LS.NONE, LS.LINEAR, LS.CUBIC
* @param {*} result [optional] the container where to store the data (in case is an array). IF NOT CONTAINER IS PROVIDED THE SAME ONE IS RETURNED EVERY TIME!
* @return {*} data
*/
Track.prototype.getSample = function( time, interpolate, result )
{
	if(!this.data || this.data.length === 0)
		return undefined;

	if(this.packed_data)
		return this.getSamplePacked( time, interpolate, result );
	return this.getSampleUnpacked( time, interpolate, result );
}

//used when sampling from a unpacked track (where data is an array of arrays)
Track.prototype.getSampleUnpacked = function( time, interpolate, result )
{
	time = Math.clamp( time, 0, this.duration );

	var index = this.findTimeIndex( time );
	if(index === -1)
		index = 0;

	var index_a = index;
	var index_b = index + 1;
	var data = this.data;
	var value_size = this.value_size;

	interpolate = interpolate && this.interpolation && (this.value_size > 0 || LS.Interpolators[ this._type ] );

	if(!interpolate || (data.length == 1) || index_b == data.length || (index_a == 0 && this.data[0][0] > time)) //(index_b == this.data.length && !this.looped)
		return this.data[ index ][1];

	var a = data[ index_a ];
	var b = data[ index_b ];

	var t = (b[0] - time) / (b[0] - a[0]);

	//multiple data
	if( value_size > 1 )
	{
		result = result || this._result;
		if( !result || result.length != value_size )
			result = this._result = new Float32Array( value_size );
	}

	if(this.interpolation === LS.LINEAR)
	{
		if( value_size == 1 )
			return a[1] * t + b[1] * (1-t);

		return LS.Animation.interpolateLinear( a[1], b[1], t, result, this._type, value_size, this );
	}
	else if(this.interpolation === LS.CUBIC)
	{
		//cubic not implemented for interpolators
		if(value_size === 0 && LS.Interpolators[ this._type ] )
		{
			var func = LS.Interpolators[ this._type ];
			var r = func( a[1], b[1], t, this._last_value );
			this._last_value = r;
			return r;
		}

		var pre_a = index > 0 ? data[ index - 1 ] : a;
		var post_b = index < data.length - 2 ? data[ index + 2 ] : b;

		if(value_size === 1)
			return Animation.EvaluateHermiteSpline(a[1],b[1],pre_a[1],post_b[1], 1 - t );

		result = Animation.EvaluateHermiteSplineVector( a[1], b[1], pre_a[1], post_b[1], 1 - t, result );

		if(this._type_index == Track.QUAT)
		{
			quat.slerp( result, b[1], a[1], t ); //force quats without CUBIC interpolation
			quat.normalize( result, result );
		}
		else if(this._type_index == Track.TRANS10)
		{
			var rotR = result.subarray(3,7);
			var rotA = a[1].subarray(3,7);
			var rotB = b[1].subarray(3,7);
			quat.slerp( rotR, rotB, rotA, t );
			quat.normalize( rotR, rotR );
		}

		return result;
	}

	return null;
}

//used when sampling from a packed track (where data is a typed-array)
Track.prototype.getSamplePacked = function( time, interpolate, result )
{
	time = Math.clamp( time, 0, this.duration );

	var index = this.findTimeIndex( time );
	if(index == -1)
		index = 0;

	var value_size = this.value_size;
	var offset = (value_size+1);
	var index_a = index;
	var index_b = index + 1;
	var data = this.data;
	var num_keyframes = data.length / offset;

	interpolate = interpolate && this.interpolation && (value_size > 0 || LS.Interpolators[ this._type ] );

	if( !interpolate || num_keyframes == 1 || index_b == num_keyframes || (index_a == 0 && this.data[0] > time)) //(index_b == this.data.length && !this.looped)
		return this.getKeyframe( index )[1];

	//multiple data
	if( value_size > 1 )
	{
		result = result || this._result;
		if( !result || result.length != value_size )
			result = this._result = new Float32Array( value_size );
	}

	var a = data.subarray( index_a * offset, (index_a + 1) * offset );
	var b = data.subarray( index_b * offset, (index_b + 1) * offset );

	var t = (b[0] - time) / (b[0] - a[0]);

	if(this.interpolation === LS.LINEAR)
	{
		if( value_size == 1 ) //simple case
			return a[1] * t + b[1] * (1-t);

		var a_data = a.subarray(1, value_size + 1 );
		var b_data = b.subarray(1, value_size + 1 );
		return LS.Animation.interpolateLinear( a_data, b_data, t, result, this._type, value_size, this );
	}
	else if(this.interpolation === LS.CUBIC)
	{
		if( value_size === 0 ) //CUBIC not supported in interpolators
			return a[1];

		var pre_a = index > 0 ? data.subarray( (index-1) * offset, (index) * offset ) : a;
		var post_b = index_b < (num_keyframes - 1) ? data.subarray( (index_b+1) * offset, (index_b+2) * offset ) : b;

		if( value_size === 1 )
			return Animation.EvaluateHermiteSpline( a[1], b[1], pre_a[1], post_b[1], 1 - t );

		var a_value = a.subarray(1,offset);
		var b_value = b.subarray(1,offset);

		result = Animation.EvaluateHermiteSplineVector( a_value, b_value, pre_a.subarray(1,offset), post_b.subarray(1,offset), 1 - t, result );

		if(this._type_index == Track.QUAT )
		{
			quat.slerp( result, b_value, a_value, t );
			quat.normalize( result, result ); //is necesary?
		}
		else if(this._type_index == Track.TRANS10 )
		{
			var rotR = result.subarray(3,7);
			var rotA = a_value.subarray(3,7);
			var rotB = b_value.subarray(3,7);
			quat.slerp( rotR, rotB, rotA, t );
			quat.normalize( rotR, rotR ); //is necesary?
		}

		return result;
	}

	return null;
}

/**
* returns information about the object being affected by this track based on its locator
* the object contains a reference to the object, the property name, the type of the data
* @method getPropertyInfo
* @param {LS.Scene} scene [optional]
* @return {Object} an object with the info { target, name, type, value }
*/
Track.prototype.getPropertyInfo = function( scene )
{
	scene = scene || LS.GlobalScene;

	return scene.getPropertyInfo( this.property );
}

/**
* returns the node to which this track is affecting (in case it is a node, if it is something else it returns null)
* @method getPropertyNode
* @param {LS.Scene} scene [optional]
* @return {LS.SceneNode} the node being affected by the track
*/
Track.prototype.getPropertyNode = function( scene )
{
	return (scene || LS.GlobalScene).getNode( this.property.split("/")[0] );
}


/**
* returns an array containing N samples for this property over time using the interpolation of the track
* @method getSampledData
* @param {Number} start_time when to start sampling
* @param {Number} end_time when to finish sampling
* @param {Number} num_samples the number of samples
* @return {Array} an array containing all the samples
*/
Track.prototype.getSampledData = function( start_time, end_time, num_samples )
{
	var delta = (end_time - start_time) / num_samples;
	if(delta <= 0)
		return null;

	var samples = [];
	for(var i = 0; i < num_samples; ++i)
	{
		var t = i * delta + start_time;
		var sample = this.getSample( t, true );
		if(this.value_size > 1)
			sample = new sample.constructor( sample );
		samples.push(sample);
	}

	return samples;
}

Animation.interpolateLinear = function( a, b, t, result, type, value_size, track )
{
	if(value_size == 1)
		return a * t + b * (1-t);

	if( LS.Interpolators[ type ] )
	{
		var func = LS.Interpolators[ type ];
		var r = func( a, b, t, track._last_v );
		track._last_v = r;
		return r;
	}

	result = result || track._result;

	if(!result || result.length != value_size)
		result = track._result = new Float32Array( value_size );

	var type_index = LS.TYPES_INDEX[ type ];

	switch( type_index )
	{
		case Track.QUAT:
			quat.slerp( result, b, a, t );
			quat.normalize( result, result );
			break;
		case Track.TRANS10: 
			for(var i = 0; i < 3; i++) //this.value_size should be 10
				result[i] = a[i] * t + b[i] * (1-t);
			for(var i = 7; i < 10; i++) //this.value_size should be 10
				result[i] = a[i] * t + b[i] * (1-t);
			var rotA = a.subarray(3,7);
			var rotB = b.subarray(3,7);
			var rotR = result.subarray(3,7);
			quat.slerp( rotR, rotB, rotA, t );
			quat.normalize( rotR, rotR );
			break;
		default:
			for(var i = 0; i < value_size; i++)
				result[i] = a[i] * t + b[i] * (1-t);
	}
	return result;
}

Animation.EvaluateHermiteSpline = function( p0, p1, pre_p0, post_p1, s )
{
	var s2 = s * s;
	var s3 = s2 * s;
	var h1 =  2*s3 - 3*s2 + 1;          // calculate basis function 1
	var h2 = -2*s3 + 3*s2;              // calculate basis function 2
	var h3 =   s3 - 2*s2 + s;         // calculate basis function 3
	var h4 =   s3 -  s2;              // calculate basis function 4
	var t0 = p1 - pre_p0;
	var t1 = post_p1 - p0;

	return h1 * p0 + h2 * p1 + h3 * t0 + h4 * t1;
}

Animation.EvaluateHermiteSplineVector = function( p0, p1, pre_p0, post_p1, s, result )
{
	result = result || new Float32Array( result.length );

	var s2 = s * s;
	var s3 = s2 * s;
	var h1 =  2*s3 - 3*s2 + 1;          // calculate basis function 1
	var h2 = -2*s3 + 3*s2;              // calculate basis function 2
	var h3 =   s3 - 2*s2 + s;         // calculate basis function 3
	var h4 =   s3 -  s2;              // calculate basis function 4

	for(var i = 0, l = result.length; i < l; ++i)
	{
		var t0 = p1[i] - pre_p0[i];
		var t1 = post_p1[i] - p0[i];
		result[i] = h1 * p0[i] + h2 * p1[i] + h3 * t0 + h4 * t1;
	}

	return result;
}

LS.registerResourceClass( Animation );

//extra interpolators ***********************************
LS.Interpolators = {};

LS.Interpolators["texture"] = function( a, b, t, last )
{
	var texture_a = a ? LS.getTexture( a ) : null;
	var texture_b = b ? LS.getTexture( b ) : null;

	if(a && !texture_a && a[0] != ":" )
		LS.ResourcesManager.load(a);
	if(b && !texture_b && b[0] != ":" )
		LS.ResourcesManager.load(b);

	var texture = texture_a || texture_b;

	var black = gl.textures[":black"];
	if(!black)
		black = gl.textures[":black"] = new GL.Texture(1,1, { format: gl.RGB, pixel_data: [0,0,0], filter: gl.NEAREST });

	if(!texture)
		return black;

	var w = texture ? texture.width : 256;
	var h = texture ? texture.height : 256;

	if(!texture_a)
		texture_a = black;
	if(!texture_b)
		texture_b = black;

	if(!last || last.width != w || last.height != h || last.format != texture.format )
		last = new GL.Texture( w, h, { format: texture.format, type: texture.type, filter: gl.LINEAR } );

	var shader = gl.shaders[":interpolate_texture"];
	if(!shader)
		shader = gl.shaders[":interpolate_texture"] = GL.Shader.createFX("color = mix( texture2D( u_texture_b, uv ), color , u_factor );", "uniform sampler2D u_texture_b; uniform float u_factor;" );

	gl.disable( gl.DEPTH_TEST );
	last.drawTo( function() {
		gl.clearColor(0,0,0,0);
		gl.clear( gl.COLOR_BUFFER_BIT );
		texture_b.bind(1);
		texture_a.toViewport( shader, { u_texture_b: 1, u_factor: t } );
	});

	return last;
}

///@FILE:../src/resources/skeletalAnimation.js
// Standalone class for skeletal animations
// By Javi Agenjo (@tamat)
// ***************************************
// It uses a filetype called SKANIM, the format is similar to BVH but much more easy to parser
// ASCII Format description:
// HEADER: {duration}, {samples_per_second}, {num_keyframes}, {num_bones}
// FOR EVERY BONE (ordered by hierarchy): B{bone index}, {bone_name}, {bind matrix of bone in mat44}
// KEYFRAMES HEADER: @{num_animated_bones},{index to bone referenced by the first matrix}, {index to bone referenced by the second matrix}, ...
// KEYFRAME: K{time},{mat4},{mat4},{mat4},....

function lerp(a,b,f) { return a*(1.0-f)+b*f; }

if(!Math.lerp)
	Math.lerp = lerp;

function Skeleton()
{
	this.bones = []; //array of bones
	this.global_bone_matrices = []; //internal array of mat4
	this.bones_by_name = new Map(); //map of nodenames and index in the bones array
}

//Skeleton.EXTENSION = "skanim";


function Bone()
{
	this.name = "";
	this.model = mat4.create();
	this.parent = -1;
	this.layer = 0;
	this.num_children = 0;
	this.children = new Int8Array(16);
}

Bone.prototype.serialize = function()
{
	return {
		name: this.name,
		model: typedArrayToArray( this.model ),
		parent: this.parent,
		layer: this.layer,
		children: this.num_children ? typedArrayToArray( this.children.subarray(0,this.num_children) ) : null
	};
}

Bone.prototype.configure = function(o)
{
	this.name = o.name;
	this.model.set( o.model );
	this.parent = o.parent;
	this.layer = o.layer;
	this.num_children = 0;
	if(o.children)
	{
		this.children.set(o.children);
		if(o.children.constructor === Array)
			this.num_children = o.children.length;
		else
			this.num_children = o.num_children;
	}
}

Bone.prototype.copyFrom = Bone.prototype.configure;

Skeleton.Bone = Bone;

//given a bone name and matrix, it multiplies the matrix to the bone
Skeleton.prototype.applyTransformToBones = function(root, transform)
{
	var bone = this.getBone(root);
	if (!bone)
		return;
	mat4.multiply( bone.model, bone.model, transform );
};

Skeleton.prototype.getBone = function(name)
{
	return this.bones[ this.bones_by_name.get(name) ];
}

Skeleton.identity = mat4.create();

Skeleton.prototype.getBoneMatrix = function( name, local )
{
	if(local === undefined)
		local = true;
	var index = this.bones_by_name.get(name);
	if( index === undefined )
		return Skeleton.identity;
	if(local)
		return this.bones[ index ].model;
	return this.global_bone_matrices[ index ];
}

Skeleton.temp_mat4 = mat4.create();
Skeleton.temp_mat43 = Skeleton.temp_mat4.subarray(0,12);

//fills the array with the bones ready for the shader
//simplify allows to store as mat4x3 instead of mat4x4 (because the last column is always 0,0,0,1)
Skeleton.prototype.computeFinalBoneMatrices = function( bone_matrices, mesh, simplify )
{
	if(!this.bones.length || !mesh || !mesh.bones)
		return bone_matrices || [];

	this.updateGlobalMatrices();

	var size = simplify ? mesh.bones.length * 12 : mesh.bones.length * 16;

	if(!bone_matrices || bone_matrices.length != size )
		bone_matrices = new Float32Array( size );

	if(simplify) //convert to mat4x3
	{
		var m = Skeleton.temp_mat4;
		var m43 = Skeleton.temp_mat43;
		for (var i = 0; i < mesh.bones.length; ++i)
		{
			var bone_info = mesh.bones[i];
			mat4.multiply( temp_mat4, this.getBoneMatrix( bone_info[0], false ), bone_info[1] ); //use globals
			mat4.transpose( temp_mat4, temp_mat4 );
			bone_matrices.set(m43,i*12);
		}
	}
	else
		for (var i = 0; i < mesh.bones.length; ++i)
		{
			var bone_info = mesh.bones[i];
			var m = bone_matrices.subarray(i*16,i*16+16);
			mat4.multiply( m, this.getBoneMatrix( bone_info[0], false ), bone_info[1] ); //use globals
		}

	return bone_matrices;
}

//returns an array with the final global bone matrix in the order specified by the mesh, global_model is optional
Skeleton.prototype.computeFinalBoneMatricesAsArray = function( bone_matrices, mesh, global_model )
{
	if(!this.bones.length || !mesh || !mesh.bones)
		return bone_matrices || [];

	this.updateGlobalMatrices();

	bone_matrices = bone_matrices || [];
	bone_matrices.length = mesh.bones.length;

	for (var i = 0; i < mesh.bones.length; ++i)
	{
		var bone_info = mesh.bones[i];
		if(!bone_matrices[i])
			bone_matrices[i] = mat4.create();
		var m = bone_matrices[i];
		mat4.multiply( m, this.getBoneMatrix( bone_info[0], false ), bone_info[1] ); //use globals
		if(mesh.bind_matrix)
			mat4.multiply( m, m, mesh.bind_matrix );
		if(global_model)
			mat4.multiply( m, global_model, m );
	}

	return bone_matrices;
}

//updates the list of global matrices according to the local matrices
Skeleton.prototype.updateGlobalMatrices = function()
{
	var bones = this.bones;
	if(!bones.length)
		return;

	var num_bones = this.bones.length;

	//compute global matrices
	this.global_bone_matrices[0].set( bones[0].model );
	//order dependant
	for (var i = 1; i < num_bones; ++i)
	{
		var bone = bones[i];
		mat4.multiply( this.global_bone_matrices[i], this.global_bone_matrices[ bone.parent ], bone.model );
	}
}

//assigns a layer to a node and all its children
Skeleton.prototype.assignLayer = function(bone, layer)
{
	//TODO
}

//for rendering the skeleton
Skeleton.prototype.getVertices = function()
{
	if(!this.bones.length)
		return null;
	var size = (this.bones.length - 1) * 3 * 2;
	if(!this._vertices || this._vertices.length != size)
		this._vertices = new Float32Array( size );
	var vertices = this._vertices;
	for (var i = 0; i < this.bones.length - 1; ++i)
	{
		var bone = this.bones[i+1];
		var parent_global_matrix = this.global_bone_matrices[ bone.parent ];
		var global_matrix = this.global_bone_matrices[i+1];
		var v1 = vertices.subarray(i*6,i*6+3);
		var v2 = vertices.subarray(i*6+3,i*6+6);
		mat4.getTranslation( v1, global_matrix );
		mat4.getTranslation( v2, parent_global_matrix );
	}
	return vertices;
}

Skeleton.prototype.resizeBones = function(num)
{
	if(this.bones.length == num)
		return;
	if(this.bones.length > num)
	{
		this.bones.length = num;
		this.global_bone_matrices.length = num;
		return;
	}

	var old_num = this.bones.length;
	this.bones.length = num;
	for(var i = old_num; i < num; ++i)
	{
		this.bones[i] = new Bone();
		this.global_bone_matrices[i] = mat4.create();
	}
}

Skeleton.prototype.copyFrom = function( skeleton )
{
	this.resizeBones( skeleton.bones.length );
	for(var i = 0; i < skeleton.bones.length; ++i)
	{
		this.bones[i].copyFrom( skeleton.bones[i] );
		this.global_bone_matrices[i].set( skeleton.global_bone_matrices[i] );
	}
	this.bones_by_name = new Map( skeleton.bones_by_name );
}

Skeleton.prototype.serialize = function()
{
	var o = {
		bones: [],
		bone_names: {}
	};

	for(var i = 0; i < this.bones.length; ++i)
		o.bones.push(this.bones[i].serialize());
	return o;
}

Skeleton.prototype.configure = function(o)
{
	this.resizeBones( o.bones.length );
	if(o.bones_by_name)
		this.bones_by_name = new Map( o.bones_by_name );
	else
		this.bones_by_name.clear();
	for(var i = 0; i < o.bones.length; ++i)
	{
		this.bones[i].copyFrom( o.bones[i] );
		if(o.global_bone_matrices) //is an skeleton
			this.global_bone_matrices[i].set( o.global_bone_matrices[i] );
		else //is an object
			this.bones_by_name[this.bones[i].name] = i;
	}
}

var temp_axis = vec3.create();

//blends between two skeletons
Skeleton.blend = function(a, b, w, result, layer, skip_normalize )
{
	if(a.bones.length != b.bones.length)
	{
		console.error("skeleton must contain the same number of bones");
		return;
	}

	w = Math.clamp(w, 0.0, 1.0);//safety

	if (layer == 0xFF)
	{
		if (w == 0.0)
		{
			if(result == a) //nothing to do
				return;
			result.copyFrom(a); //copy A in Result
			return;
		}
		if (w == 1.0) //copy B in result
		{
			result.copyFrom(b);
			return;
		}
	}

	if (result != a) //copy bone names
	{
		result.bones.length = a.bones.length;
		for (var i = 0; i < result.bones.length; ++i)
		{
			var b = result.bones[i];
			if(!b)
				b = new Skeleton.Bone();
			b.copyFrom(a.bones[i]);
		}
		result.bones_by_name = new Map(a.bones_by_name); //TODO: IMPROVE!
	}

	//blend bones locally
	for (var i = 0; i < result.bones.length; ++i)
	{
		var bone = result.bones[i];
		var boneA = a.bones[i];
		var boneB = b.bones[i];
		//if ( layer != 0xFF && !(bone.layer & layer) ) //not in the same layer
		//	continue;
		for (var j = 0; j < 16; ++j)
			bone.model[j] = Math.lerp( boneA.model[j], boneB.model[j], w);

		if(!skip_normalize)
		{
			var m = bone.model;
			//not sure which one is the right one, row major or column major
			//vec3.normalize(m.subarray(0,3),	m.subarray(0,3) );
			//vec3.normalize(m.subarray(4,7),	m.subarray(4,7) );
			//vec3.normalize(m.subarray(8,11), m.subarray(8,11) );
			//*
			for(var j = 0; j < 3; ++j)
			{
				temp_axis[0] = m[0+j]; temp_axis[1] = m[4+j]; temp_axis[2] = m[8+j];
				vec3.normalize(temp_axis,temp_axis);
				m[0+j] = temp_axis[0]; m[4+j] = temp_axis[1]; m[8+j] = temp_axis[2];
			}
			//*/
		}
	}
}

Skeleton.shader_code = '\n\
	attribute vec4 a_bone_indices;\n\
	attribute vec4 a_weights;\n\
	uniform mat4 u_bones[64];\n\
	void computeSkinning(inout vec3 vertex, inout vec3 normal)\n\
	{\n\
		vec4 v = vec4(vertex,1.0);\n\
		vertex = (u_bones[int(a_bone_indices.x)] * a_weights.x * v + \n\
				u_bones[int(a_bone_indices.y)] * a_weights.y * v + \n\
				u_bones[int(a_bone_indices.z)] * a_weights.z * v + \n\
				u_bones[int(a_bone_indices.w)] * a_weights.w * v).xyz;\n\
		vec4 N = vec4(normal,0.0);\n\
		normal =	(u_bones[int(a_bone_indices.x)] * a_weights.x * N + \n\
				u_bones[int(a_bone_indices.y)] * a_weights.y * N + \n\
				u_bones[int(a_bone_indices.z)] * a_weights.z * N + \n\
				u_bones[int(a_bone_indices.w)] * a_weights.w * N).xyz;\n\
		normal = normalize(normal);\n\
	}\n\
';

//*******************************************************

function SkeletalAnimation()
{
	this.skeleton = new Skeleton();

	this.duration = 0;
	this.samples_per_second = 0;
	this.num_animated_bones = 0;
	this.num_keyframes = 0;
	this.bones_map = new Uint8Array(64); //maps from keyframe data index to bone

	this.keyframes = null; //mat4 array
}

SkeletalAnimation.FORMAT = { extension: "skanim", dataType: "text" };

//change the skeleton to the given pose according to time
SkeletalAnimation.prototype.assignTime = function(time, loop, interpolate, layers )
{
	if(!this.duration)
		return;

	if (loop || loop === undefined)
	{
		time = time % this.duration;
		if (time < 0)
			time = this.duration + time;
	}
	else
		time = Math.clamp( time, 0.0, this.duration - (1.0/this.samples_per_second) );

	if(interpolate === undefined)
		interpolate = true;

	var v = this.samples_per_second * time;
	var index = Math.floor(v);
	var index2 = (index + 1) % this.num_keyframes;
	index = index % this.num_keyframes;
	var f = v - Math.floor(v);
	var num_animated_bones = this.num_animated_bones;

	var offset = 16 * num_animated_bones;
	var k = index * offset;
	var k2 = index2 * offset;
	var skeleton = this.skeleton;
	var keyframes = this.keyframes;
	var bones_map = this.bones_map;

	//compute local bones
	for (var i = 0; i < num_animated_bones; ++i)
	{
		var bone_index = bones_map[i];
		var bone = skeleton.bones[bone_index];
		var offset = i*16;
		//if (layers != 0xFF && !(bone.layer & layers))
		//	continue;
		if(!interpolate)
			bone.model.set( keyframes.subarray( k + offset, k + offset + 16) );
		else
			for (var j = 0; j < 16; ++j)
			{
				//lerp matrix
				bone.model[j] = lerp( keyframes[ k + offset + j ], keyframes[ k2 + offset + j ], f );
			}
	}
}

SkeletalAnimation.prototype.resize = function( num_keyframes, num_animated_bones )
{
	this.num_keyframes = num_keyframes;
	this.num_animated_bones = num_animated_bones;
	this.keyframes = new Float32Array( num_keyframes * num_animated_bones * 16);
}

SkeletalAnimation.prototype.fromData = function(txt)
{
	var lines = txt.split("\n");
	var header = lines[0].split(",");
	this.duration = Number(header[0]);
	this.samples_per_second = Number(header[1]);
	this.num_keyframes = Number(header[2]);
	
	this.skeleton.resizeBones( Number(header[3]) );
	var current_keyframe = 0;
	for(var i = 1; i < lines.length; ++i)
	{
		var line = lines[i];
		var type = line[0];
		var t = line.substr(1).split(",");
		if( type == 'B')
		{
			var index = Number(t[0]);
			var bone = this.skeleton.bones[index];
			bone.name = t[1];
			bone.parent = Number(t[2]);
			for(var j = 0; j < 16; ++j)
				bone.model[j] = Number(t[3+j]);
			if (bone.parent != -1)
			{
				var parent_bone = this.skeleton.bones[ bone.parent ];
				if(parent_bone.num_children >= 16)
					console.warn("too many child bones, max is 16");
				else
					parent_bone.children[ parent_bone.num_children++ ] = index;
			}
			this.skeleton.bones_by_name.set(bone.name,index);
		}
		else if( type == '@')
		{
			this.num_animated_bones = Number(t[0]);
			for(var j = 0; j < this.num_animated_bones; ++j)
				this.bones_map[j] = Number(t[j+1]);
			this.resize( this.num_keyframes, this.num_animated_bones );
		}
		else if( type == 'K')
		{
			var pos = current_keyframe * this.num_animated_bones * 16;
			for(var j = 0, l = this.num_animated_bones * 16; j < l; ++j)
				this.keyframes[ pos + j ] = Number( t[j+1] );
			current_keyframe++;
		}
		else 
			break;
	}

	this.assignTime(0,false,false);
}

SkeletalAnimation.prototype.toData = function()
{
	var str = "";
	//this is currently done from WebGLStudio in the AnimationModule exportTakeInSKANIM
	console.error("no toData in Skeletal Animation");
	return str;
}

//so it can be used from LiteScene or Rendeer
if(typeof(LS) != "undefined")
{
	LS.Skeleton = Skeleton;
	LS.Classes["SkeletalAnimation"] = LS.SkeletalAnimation = SkeletalAnimation;
	LS.registerResourceClass( SkeletalAnimation );
}






///@FILE:../src/resources/pack.js
///@INFO: PACK
//defined before Prefab

/**
* Pack is an object that contain several resources, helpful when you want to carry a whole scene in one single file
* 
* @class Pack
* @constructor
*/

function Pack(o)
{
	this.resource_names = []; 
	this.metadata = null;
	this._data = {}; //the original chunks from the WBin, including the @JSON and @resource_names
	this._resources_data = {}; //every resource in arraybuffer format
	if(o)
		this.configure(o);
}

Pack.version = "0.2"; //used to know where the file comes from 
Pack.EXTENSION = "wbin";

/**
* configure the pack from an unpacked WBin
* @method configure
* @param {Object} data an unpacked WBIN (object with every chunk)
**/
Pack.prototype.configure = function( data )
{
	this._data = LS.cloneObject( data );

	//extract resource names
	this.resource_names = data["@resource_names"];
	this._resources_data = {};
	if(this.resource_names)
	{
		delete this._data["@resource_names"];
		for(var i in this.resource_names)
		{
			this._resources_data[ this.resource_names[i] ] = data[ "@RES_" + i ];
			delete this._data[ "@RES_" + i ];
		}
	}

	//store resources in LS.ResourcesManager
	this.processResources();
}

Object.defineProperty( Pack.prototype, 'bindata', {
	set: function(name)
	{
		throw("Pack bindata cannot be assigned");
	},
	get: function(){
		if(!this._original_data)
			this._original_data = LS.Pack.packResources( this.resource_names, this._data );
		return this._original_data;
	},
	enumerable: true
});


Pack.fromBinary = function(data)
{
	if(data.constructor == ArrayBuffer)
		data = WBin.load(data, true);
	return new LS.Pack(data);
}

//given a list of resources that come from the Pack (usually a wbin) it extracts, process and register them 
Pack.prototype.processResources = function()
{
	if(!this.resource_names)
		return;

	var pack_filename = this.fullpath || this.filename;

	//block this resources of being loaded, this is to avoid chain reactions when a resource uses 
	//another one contained in this pack
	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var resname = this.resource_names[i];
		if( LS.ResourcesManager.resources[ resname ] )
			continue; //already loaded
		LS.ResourcesManager.resources_being_processed[ resname ] = true;
	}

	//process and store in LS.ResourcesManager
	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var resname = this.resource_names[i];
		if( LS.ResourcesManager.resources[resname] )
			continue; //already loaded

		var resdata = this._resources_data[ resname ];
		if(!resdata)
		{
			console.warn("resource data in Pack is undefined, skipping it:" + resname);
			continue;
		}
		var resource = LS.ResourcesManager.processResource( resname, resdata, { is_local: true, from_pack: pack_filename } );
	}
}

Pack.prototype.setResources = function( resource_names, mark_them )
{
	this.resource_names = [];
	this._resources_data = {};

	var pack_filename = this.fullpath || this.filename;

	//get resources
	for(var i = 0; i < resource_names.length; ++i)
	{
		var res_name = resource_names[i];
		if(this.resource_names.indexOf(res_name) != -1)
			continue;
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;
		if(mark_them)
			resource.from_pack = pack_filename;
		this.resource_names.push( res_name );
	}

	//repack the pack info
	this._original_data = LS.Pack.packResources( resource_names, this.getBaseData() );
	this._modified = true;
}

Pack.prototype.getBaseData = function()
{
	return { "@metadata": this.metadata, "@version": LS.Pack.version };
}

//adds to every resource in this pack info about where it came from (the pack)
Pack.prototype.setResourcesLink = function( value )
{
	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var res_name = this.resource_names[i];
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;
		if(value)
			resource.from_pack = value;
		else
			delete resource.from_pack;
	}
}

//adds a new resource (or array of resources) to this pack
Pack.prototype.addResources = function( resource_names, mark_them )
{
	if(!resource_names)
		return;
	if(resource_names.constructor !== Array)
		resource_names = [ resource_names ];
	this.setResources( this.resource_names.concat( resource_names ), mark_them );
}

/**
* Adds a resource to the prefab
* @method addResource
* @param {String} filename filename of the resource
**/
Pack.prototype.addResource = function( filename )
{
	filename = LS.ResourcesManager.cleanFullpath( filename );
	var index = this.resource_names.indexOf(filename);
	if(index == -1)
		this.resource_names.push( filename );
}

/**
* Remove a resource to the prefab
* @method removeResource
* @param {String} filename filename of the resource
**/
Pack.prototype.removeResource = function(filename)
{
	filename = LS.ResourcesManager.cleanFullpath( filename );
	var index = this.resource_names.indexOf(filename);
	if(index != -1)
		this.resource_names.splice( index, 1 );
}

/**
* to create a WBin containing all the resource and metadata
* @method Pack.createWBin
* @param {String} fullpath for the pack
* @param {Array} resource_names array with the names of all the resources to store
* @param {Object} metadata [optional] extra data to store
* @param {boolean} mark_them [optional] marks all the resources as if they come from a pack
* @return object containing the pack data ready to be converted to WBin
**/
Pack.createPack = function( filename, resource_names, extra_data, mark_them )
{
	if(!filename)
		return;

	if(!resource_names || resource_names.constructor !== Array)
		throw("Pack.createPack resources must be array with names");
	if(extra_data && extra_data.constructor !== Object)
		throw("Pack.createPack extra_data must be an object with the chunks to store");

	filename = filename.replace(/ /gi,"_");

	var pack = new LS.Pack();
	filename += ".wbin";
	pack.filename = filename;
	if(extra_data)
		pack._data = extra_data;

	pack.resource_names = resource_names;
	for(var i = 0; i < resource_names.length; ++i)
	{
		var res_name = resource_names[i];
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;
		if(mark_them)
			resource.from_pack = pack.filename;
	}

	//create the WBIN in case this pack gets stored
	this.metadata = extra_data;
	var bindata = LS.Pack.packResources( resource_names, pack.getBaseData() );
	pack._original_data = bindata;

	return pack;
}

//Given a bunch of resource names it creates a WBin with all inside
Pack.packResources = function( resource_names, base_object )
{
	var to_binary = base_object || {};
	var final_resource_names = [];

	for(var i = 0; i < resource_names.length; ++i)
	{
		var res_name = resource_names[i];
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;

		var data = null;
		if(resource._original_data) //must be string or bytes
			data = resource._original_data;
		else
		{
			var data_info = LS.Resource.getDataToStore( resource );
			data = data_info.data;
		}

		if(!data)
		{
			console.warn("Wrong data in resource");
			continue;
		}

		if(data.constructor === Blob || data.constructor === File)
		{
			if(!data.data || data.data.constructor !== ArrayBuffer )
			{
				console.warn("WBin does not support to store File or Blob, please, use ArrayBuffer");
				continue;
			}
			data = data.data; //because files have an arraybuffer with the data if it was read
		}

		to_binary["@RES_" + final_resource_names.length ] = data;
		final_resource_names.push( res_name );
		//to_binary[res_name] = data;
	}

	to_binary["@resource_names"] = final_resource_names;
	return WBin.create( to_binary, "Pack" );
}

//just tells the resources where they come from, we cannot do that before because we didnt have the name of the pack
Pack.prototype.flagResources = function()
{
	if(!this.resource_names)
		return;

	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var res_name = this.resource_names[i];
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;

		resource.from_pack = this.fullpath || this.filename;
	}
}

Pack.prototype.getDataToStore = function()
{
	return LS.Pack.packResources( this.resource_names, this.getBaseData() );
}

Pack.prototype.checkResourceNames = function()
{
	if(!this.resource_names)
		return 0;

	var changed = 0;

	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var res_name = this.resource_names[i];
		var old_name = res_name;
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;

		//avoid problematic symbols
		if( LS.ResourcesManager.valid_resource_name_reg.test( res_name ) == false )
		{
			console.warn("Invalid filename in pack/prefab: ", res_name  );
			res_name = res_name.replace( /[^a-zA-Z0-9-_\.\/]/g, '_' );
		}

		//ensure extensions
		var extension = LS.ResourcesManager.getExtension( res_name );
		if(!extension)
		{
			extension = resource.constructor.EXTENSION;
			if(!extension)
				console.warn("Resource without extension and not known default extension: ", res_name , resource.constructor.name );
			else
				res_name = res_name + "." + extension;
		}

		if(old_name == res_name)
			continue;

		this.resource_names[i] = res_name;
		LS.ResourcesManager.renameResource( old_name, res_name ); //force change
		changed++;
	}

	if(changed)
		LS.ResourcesManager.resourceModified( this );

	return changed;
}

Pack.prototype.onResourceRenamed = function( old_name, new_name, resource )
{
	if(!this.resource_names)
		return;
	var index = this.resource_names[ old_name ];
	if( index == -1 )
		return;
	this.resource_names[ index ] = new_name;
	LS.ResourcesManager.resourceModified( this );
}

Pack.prototype.containsResources = function()
{
	return this.resource_names && this.resource_names.length > 0 ? true : false;
}

Pack.prototype.getSizeInBytes = function()
{
	if(this._original_data)
		return this._original_data.byteLength;
	return 0;
}

LS.Pack = Pack;
LS.registerResourceClass( Pack );


///@FILE:../src/resources/prefab.js
///@INFO: PACK
/**
* Prefab work in two ways: 
* - It can contain a node structure and all the associated resources (textures, meshes, animations, etc)
* - When a node in the scene was created from a Prefab, the prefab is loaded so the associated resources are recovered, but the node structure is not modified.
* 
* @class Prefab
* @constructor
*/

function Prefab( o, filename )
{
	this.filename = filename || null; //base file
	this.fullpath = filename || null; //full path 
	this.resource_names = []; 
	this.prefab_json = null;
	this.prefab_data = null; //json object
	this._resources_data = {}; //data as it was loaded from the WBin

	if(o)
		this.configure(o);
}

Prefab.version = "0.2"; //used to know where the file comes from 
Prefab.EXTENSION = "wbin";

/**
* assign the json object
* @method setData
* @param {object|SceneNode} data
**/
Prefab.prototype.setData = function(data)
{
	if( data && data.constructor === LS.SceneNode )
		data = data.serialize();
	data.object_class = "SceneNode";
	this.prefab_data = data;
	this.prefab_json = JSON.stringify( data );
}

/**
* configure the prefab
* @method configure
* @param {*} data
**/

Prefab.prototype.configure = function(data)
{
	if(!data)
		throw("No prefab data found");

	if(data.hasOwnProperty("prefab_data"))
	{
		this.prefab_data = data.prefab_data;
		this.prefab_json = data.prefab_json || JSON.stringify( this.prefab_data );
	}
	else
	{
		//read from WBin info
		var prefab_json = data["@json"];
		if(!prefab_json)
		{
			console.warn("No JSON found in prefab");
			return;
		}

		var version = data["@version"]; //not used yet
		this.prefab_json = prefab_json;
		this.prefab_data = JSON.parse( prefab_json );
	}

	this.resource_names = data["@resources_name"] || data.resource_names || [];

	//extract resource names
	if(this.resource_names)
	{
		var resources = {};
		for(var i in this.resource_names)
		{
			if(!version) //legacy
				resources[ this.resource_names[i] ] = data[ this.resource_names[i] ];
			else
				resources[ this.resource_names[i] ] = data[ "@RES_" + i ];
		}
		this._resources_data = resources;
	}

	//store resources in ResourcesManager
	this.processResources();
}

Prefab.fromBinary = function( data, filename )
{
	if(data.constructor == ArrayBuffer)
		data = WBin.load(data, true);

	return new LS.Prefab( data, filename );
}

//given a list of resources that come from a Prefab (usually a wbin) it extracts, process and register them 
Prefab.prototype.processResources = function()
{
	if(!this._resources_data)
		return;

	var pack_filename = this.fullpath || this.filename;

	var resources = this._resources_data;

	//block this resources of being loaded, this is to avoid chain reactions when a resource uses 
	//another one contained in this Prefab
	for(var resname in resources)
	{
		if( LS.ResourcesManager.resources[ resname ] )
			continue; //already loaded
		LS.ResourcesManager.resources_being_processed[ resname ] = true;
	}

	//process and store in ResourcesManager
	for(var resname in resources)
	{
		if( LS.ResourcesManager.resources[resname] )
			continue; //already loaded

		var resdata = resources[resname];
		if(!resdata)
		{
			console.warn( "resource data in prefab is undefined, skipping it:" + resname );
			continue;
		}
		var resource = LS.ResourcesManager.processResource( resname, resdata, { is_local: true, from_prefab: pack_filename } );
	}
}

/**
* Creates an instance of the object inside the prefab
* @method createObject
* @return object contained 
**/

Prefab.prototype.createObject = function()
{
	if(!this.prefab_json)
		throw("No prefab_json data found");

	var conf_data = JSON.parse(this.prefab_json);

	if(!conf_data)
	{
		console.error("Prefab data is null");
		return null;
	}

	var node = new LS.SceneNode();
	node.configure( conf_data );
	LS.ResourcesManager.loadResources( node.getResources({},true) );

	if(this.fullpath)
		node.prefab = this.fullpath;

	return node;
}

/**
* Adds a resource to the prefab
* @method addResource
* @param {String} filename filename of the resource
**/
Prefab.prototype.addResource = function( filename )
{
	filename = LS.ResourcesManager.cleanFullpath( filename );
	var index = this.resource_names.indexOf(filename);
	if(index == -1)
		this.resource_names.push( filename );
}

/**
* Remove a resource to the prefab
* @method removeResource
* @param {String} filename filename of the resource
**/
Prefab.prototype.removeResource = function(filename)
{
	filename = LS.ResourcesManager.cleanFullpath( filename );
	var index = this.resource_names.indexOf(filename);
	if(index != -1)
		this.resource_names.splice( index, 1 );
}


/**
* to create a new prefab, it packs all the data an instantiates the resource
* @method Prefab.createPrefab
* @param {String} filename a name for this prefab (if wbin is not appended, it will)
* @param {Object} node_data an object containing all the node data to store
* @param {Array} resource_names_list an array with the name of the resources to store
* @return object containing the prefab data ready to be converted to WBin (it also stores _original_data with the WBin)
**/
Prefab.createPrefab = function( filename, node_data, resource_names_list )
{
	if(!filename)
		return;

	if(!node_data)
		throw("No node_data in prefab");

	filename = filename.replace(/ /gi,"_");
	resource_names_list = resource_names_list || [];

	var prefab = new LS.Prefab();
	var ext = LS.ResourcesManager.getExtension(filename);
	if( ext != "wbin" )
		filename += ".wbin";

	//checkfilenames and rename them to short names
	prefab.filename = filename;
	prefab.resource_names = resource_names_list;

	//assign data
	prefab.setData( node_data );

	//get all the resources and store them in a WBin
	var bindata = LS.Prefab.packResources( resource_names_list, { "@json": prefab.prefab_json, "@version": Prefab.version }, this );
	prefab._original_data = bindata;

	return prefab;
}

//Given a list of resources and some base data, it creates a WBin with all the data
Prefab.packResources = function( resource_names_list, base_data, from_prefab )
{
	var to_binary = base_data || {};
	var resource_names = [];

	if(resource_names_list && resource_names_list.length)
	{
		for(var i = 0; i < resource_names_list.length; ++i)
		{
			var res_name = resource_names_list[i];
			var resource = LS.ResourcesManager.resources[ res_name ];
			if(!resource)
				continue;

			var data = null;
			if(resource._original_data) //must be string or bytes
				data = resource._original_data;
			else
			{
				var data_info = LS.Resource.getDataToStore( resource );
				if(!data_info)
				{
					console.warn("Data to store from resource is null, skipping: ", res_name );
					continue;
				}
				//HACK: resource could be renamed to extract the binary info (this happens in jpg textures that are converted to png) or meshes that add wbin
				if(data_info.extension && data_info.extension != LS.ResourcesManager.getExtension( res_name ))
				{
					console.warn("The resource extension has changed while saving, this could lead to problems: ", res_name, data_info.extension );
					//after this change all the references will be wrong
					var old_name = res_name;
					res_name = res_name + "." + data_info.extension;
					resource_names_list[i] = res_name;
					LS.GlobalScene.sendResourceRenamedEvent( old_name, res_name, resource ); //force change
				}
				data = data_info.data;
			}

			if(!data)
			{
				console.warn("Wrong data in resource");
				continue;
			}

			if(data.constructor === Blob || data.constructor === File)
			{
				console.warn("WBin does not support to store File or Blob, please convert to ArrayBuffer using FileReader");
				continue;
			}

			to_binary["@RES_" + resource_names.length ] = data;
			resource_names.push( res_name );
		}
	}

	to_binary["@resources_name"] = resource_names;
	return WBin.create( to_binary, "Prefab" );
}

Prefab.prototype.updateFromNode = function( node, clear_uids )
{
	var data = node.serialize(true);
	if(clear_uids)
		LS.clearUIds(data); //remove UIDs
	this.prefab_data = data;
	this.prefab_json = JSON.stringify( data );
}

Prefab.prototype.flagResources = function()
{
	if(!this.resource_names)
		return;

	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var res_name = this.resource_names[i];
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;

		resource.from_prefab = this.fullpath || this.filename;
	}
}

Prefab.prototype.setResourcesLink = function( value )
{
	if(!this.resource_names)
		return;

	for(var i = 0; i < this.resource_names.length; ++i)
	{
		var res_name = this.resource_names[i];
		var resource = LS.ResourcesManager.resources[ res_name ];
		if(!resource)
			continue;
		if(value)
			resource.from_prefab = value;
		else
			delete resource.from_prefab;
	}
}

//search for nodes using this prefab and creates the nodes
Prefab.prototype.applyToNodes = function( scene )
{
	scene = scene || LS.GlobalScene;	
	var name = this.fullpath || this.filename;

	for(var i = 0; i < scene._nodes.length; ++i)
	{
		var node = scene._nodes[i];
		if(node.prefab != name)
			continue;
		node.reloadFromPrefab();
	}
}

Prefab.prototype.getDataToStore = function()
{
	this.prefab_json = JSON.stringify( this.prefab_data );
	var filename = this.fullpath || this.filename;

	//prefab in json format
	if( !(this.resource_names && this.resource_names.length) && filename && LS.RM.getExtension(filename) == "json" )
		return JSON.stringify( { object_class: LS.getObjectClassName( this ), "@json": this.prefab_json } );

	//return the binary data of the wbin
	return LS.Prefab.packResources( this.resource_names, this.getBaseData(), this );
}

Prefab.prototype.getBaseData = function()
{
	return { "@json": this.prefab_json, "@version": LS.Prefab.version };
}

Prefab.prototype.recomputeData = function()
{
	var data = this.getDataToStore();
	if(data)
		this._original_data = data.buffer;
}

//inheritet methods from Pack
Prefab.prototype.containsResources = Pack.prototype.containsResources;
Prefab.prototype.onResourceRenamed = Pack.prototype.onResourceRenamed;
Prefab.prototype.checkResourceNames = Pack.prototype.checkResourceNames;
Prefab.prototype.setResources = Pack.prototype.setResources;
Prefab.prototype.getSizeInBytes = Pack.prototype.getSizeInBytes;

LS.Prefab = Prefab;
LS.registerResourceClass( Prefab );

///@FILE:../src/resources/shaderCode.js
/**
* ShaderCode is a resource containing all the code associated to a shader
* It is used to define special ways to render scene objects, having full control of the rendering algorithm
* Having a special class helps to parse the data in advance and share it between different materials
* 
* @class ShaderCode
* @constructor
*/

function ShaderCode( code )
{
	this._code = null;

	this._functions = {};
	this._global_uniforms = {};
	this._code_parts = {};
	this._subfiles = {};
	this._compiled_shaders = {}; //all shaders compiled using this ShaderCode

	this._shaderblock_flags_num = 0; //used to assign flags to dependencies
	this._shaderblock_flags = {}; //used to store which shaderblock represent to every flag bit

	this._version = 0;

	if(code)
		this.code = code;
}

ShaderCode.help_url = "https://github.com/jagenjo/litescene.js/blob/master/guides/shaders.md";
//ShaderCode.examples //defined from webglstudio coding.js

//block types
ShaderCode.CODE = 1;
ShaderCode.PRAGMA = 2;

//pargma types
ShaderCode.INCLUDE = 1;
ShaderCode.SHADERBLOCK = 2;
ShaderCode.SNIPPET = 3;

ShaderCode.EXTENSION = "glsl";

Object.defineProperty( ShaderCode.prototype, "code", {
	enumerable: true,
	get: function() {
		return this._code;
	},
	set: function(v) {
		if(this._code == v)
			return;
		this._code = v;
		this.processCode();
	}
});

Object.defineProperty( ShaderCode.prototype, "version", {
	enumerable: false,
	get: function() {
		return this._version;
	},
	set: function(v) {
		console.error("version cannot be set manually");
	}
});

ShaderCode.prototype.getResources = function( res )
{
	for(var i in this._code_parts)
	{
		var part = this._code_parts[i];

		for(var j in part)
		{
			var m = part[j];
			for(var k in m.includes)
			{
				res[ k ] = true;
			}
		}
	}
}

//parse the code
//store in a easy to use way
ShaderCode.prototype.processCode = function()
{
	var code = this._code;
	this._global_uniforms = {};
	this._code_parts = {};
	this._compiled_shaders = {};
	this._functions = {};
	this._shaderblock_flags_num = 0;
	this._shaderblock_flags = {};
	this._shaderblock_vars = null;
	this._has_error = false;

	var subfiles = GL.processFileAtlas( this._code );
	this._subfiles = subfiles;

	var num_subfiles = 0;
	var init_code = null; 

	//add default codes
	if(!subfiles["default.vs"])
		subfiles["default.vs"] = ShaderCode.default_vs;
	if(!subfiles["default.fs"])
		subfiles["default.fs"] = ShaderCode.default_fs;

	for(var i in subfiles)
	{
		var subfile_name = i;
		var subfile_data = subfiles[i];
		num_subfiles++;

		if(!subfile_name)
			continue;

		if(subfile_name == "js")
		{
			init_code = subfile_data;
			continue;
		}

		//used to declare uniforms without using javascript
		if(subfile_name == "uniforms")
		{
			var lines = subfile_data.split("/n");
			for(var j = 0; j < lines.length; ++j)
			{
				var line = lines[j].trim();
				var words = line.split(" ");
				var varname = words[0];
				var uniform_name = words[1];
				var property_type = words[2];
				var value = words[3];
				if( value !== undefined )
					value = LS.stringToValue(value);
				var options = null;
				var options_index = line.indexOf("{");
				if(options_index != -1)
					options = LS.stringToValue( line.substr(options_index) );
				this._global_uniforms[ varname ] = { name: varname, uniform: uniform_name, type: property_type, value: value, options: options };
			}
			continue;
		}

		var name = LS.ResourcesManager.removeExtension( subfile_name );
		var extension = LS.ResourcesManager.getExtension( subfile_name );

		if(extension == "vs" || extension == "fs")
		{
			var code_part = this._code_parts[name];
			if(!code_part)
				code_part = this._code_parts[name] = {};

			//parse data (extract pragmas and stuff)
			var glslcode = new GLSLCode( subfile_data );
			for(var j in glslcode.blocks)
			{
				var pragma_info = glslcode.blocks[j];
				if(!pragma_info || pragma_info.type != ShaderCode.PRAGMA)
					continue;
				//assign a flag position in case this block is enabled
				pragma_info.shader_block_flag = this._shaderblock_flags_num; 
				this._shaderblock_flags[ pragma_info.shader_block ] = pragma_info.shader_block_flag;
				this._shaderblock_flags_num += 1;
			}

			code_part[ extension ] = glslcode;
		}
	}

	//compile the shader before using it to ensure there is no errors
	var shader = this.getShader();
	if(!shader)
		return;

	//process init code
	if(init_code)
	{
		//clean code
		init_code = LS.ShaderCode.removeComments( init_code );

		if(init_code) //still some code? (we test it because if there is a single line of code the behaviour changes)
		{
			if(LS.catch_exceptions)
			{
				try
				{
					this._functions.init = new Function( init_code );
				}
				catch (err)
				{
					LS.dispatchCodeError( err, LScript.computeLineFromError(err), this );
				}
			}
			else
				this._functions.init = new Function( init_code );
		}
	}

	//check that all uniforms are correct
	this.validatePublicUniforms( shader );


	//to alert all the materials out there using this shader that they must update themselves.
	LEvent.trigger( LS.ShaderCode, "modified", this );
	this._version += 1;
}

//used when storing/retrieving the resource
ShaderCode.prototype.setData = function(v, skip_modified_flag)
{
	this.code = v;
	if(!skip_modified_flag)
		this._modified = true;
}

ShaderCode.prototype.getData = function()
{
	return this._code;
}

ShaderCode.prototype.fromData = ShaderCode.prototype.setData;
ShaderCode.prototype.toData = ShaderCode.prototype.getData;

ShaderCode.prototype.getDataToStore = function()
{
	return this._code;
}

//compile the shader, cache and return
ShaderCode.prototype.getShader = function( render_mode, block_flags )
{
	if( this._has_error )
		return null;

	render_mode = render_mode || "color";
	block_flags = block_flags || 0;

	//search for a compiled version of the shader (by render_mode and block_flags)
	var shaders_map = this._compiled_shaders[ render_mode ];
	if(shaders_map)
	{
		var shader = shaders_map.get( block_flags );
		if(shader)
			return shader;
	}

	//search for the code 'color', or 'shadow'
	var code = this._code_parts[ render_mode ];
	var default_code = this._code_parts.default;
	if(!code && !default_code)
		return null;

	var context = {}; //used to store metaprogramming defined vars in the shader

	//compute context defines
	for(var i = 0, l = LS.Shaders.shader_blocks.length; i < l; ++i)
	{
		if( !(block_flags & 1<<i) ) //is flag enabled
			continue;
		var shader_block = LS.Shaders.shader_blocks[i];
		if(!shader_block)
			continue; //???
		if(shader_block.context_macros)
		{
			for(var j in shader_block.context_macros)
				context[ j ] = shader_block.context_macros[j];
		}
	}

	//vertex shader code
	var vs_code = null;
	if(render_mode == "fx")
		vs_code = GL.Shader.SCREEN_VERTEX_SHADER;
	else if( code && code.vs )
		vs_code = code.vs.getFinalCode( GL.VERTEX_SHADER, block_flags, context );
	else if( default_code && default_code.vs )
		vs_code = default_code.vs.getFinalCode( GL.VERTEX_SHADER, block_flags, context );
	else 
		return null;

	//fragment shader code
	var fs_code = null;
	if( code && code.fs )
		fs_code = code.fs.getFinalCode( GL.FRAGMENT_SHADER, block_flags, context );
	else if( default_code && default_code.fs )
		fs_code = default_code.fs.getFinalCode( GL.FRAGMENT_SHADER, block_flags, context );
	else 
		return null;

	//no code or code includes something missing
	if(!vs_code || !fs_code) 
	{
		this._has_error = true;
		return null;
	}

	//add globals
	vs_code = LS.Shaders.global_extra_shader_code + vs_code;
	fs_code = LS.Shaders.global_extra_shader_code + fs_code;

	//compile the shader and return it
	var shader = this.compileShader( vs_code, fs_code );
	if(!shader)
		return null;

	//check if this shader will support rendering to draw buffers
	var clean_fs_code = LS.ShaderCode.removeComments( fs_code );
	shader.supports_drawbuffers = clean_fs_code.indexOf("gl_FragData") != -1;

	//DEBUG
	if(LS.debug)
	{
		var blocks = [];
		for(var i = 0; i < LS.Shaders.num_shaderblocks; ++i)
		{
			if( !(block_flags & 1<<i) ) //is flag enabled
				continue;
			var shader_block = LS.Shaders.shader_blocks[i];
			if(!shader_block)
				continue; //???
			blocks.push( shader_block );
		}
		shader._shadercode_info = {
			vs: vs_code,
			fs: fs_code,
			context: context,
			blocks: blocks,
			flags: block_flags
		}
	}

	//cache as render_mode,flags
	if( !this._compiled_shaders[ render_mode ] )
		this._compiled_shaders[ render_mode ] = new Map();
	this._compiled_shaders[ render_mode ].set( block_flags, shader );

	return shader;
}

ShaderCode.prototype.compileShader = function( vs_code, fs_code )
{
	if( this._has_error )
		return null;

	if( LS.Debug ) //debug shaders
	{
		console.log("Shader Compiled: ", this.fullpath || this.filename )
		console.groupCollapsed("VS shader");
		console.log(vs_code);
		console.groupEnd();
		console.groupCollapsed("FS shader");
		console.log(fs_code);
		console.groupEnd();
	}

	var shader = null;

	if(!LS.catch_exceptions)
	{
		shader = new GL.Shader( vs_code, fs_code );
	}
	else
	{
		try
		{
			shader = new GL.Shader( vs_code, fs_code );
		}
		catch(err)
		{
			this._has_error = true;
			LS.Shaders.dumpShaderError( this.filename, err, vs_code, fs_code );
			var error_info = GL.Shader.parseError( err, vs_code, fs_code );
			var line = error_info.line_number;
			var lines = this._code.split("\n");
			var code_line = -1;
			if(error_info.line_code)
			{
				var error_line_code = error_info.line_code.trim();
				for(var i = 0; i < lines.length; ++i)
					lines[i] = lines[i].trim();
				code_line = lines.indexOf( error_line_code ); //bug: what if this line is twice in the code?...
			}
			LS.dispatchCodeError( err, code_line, this, "shader" );
		}
	}

	if(shader)
	{
		if( LS.debug )
			console.log(" + shader compiled: ", this.fullpath || this.filename );
		LS.dispatchNoErrors( this, "shader" );
	}
	return shader;
}

ShaderCode.prototype.clearCache =  function()
{
	this._compiled_shaders = {};
}

ShaderCode.prototype.validatePublicUniforms = function( shader )
{
	if(!shader)
		throw("ShaderCode: Shader cannot be null");

	for( var i in this._global_uniforms )
	{
		var property_info = this._global_uniforms[i];
		var uniform_info = shader.uniformInfo[ property_info.uniform ];
		if(!uniform_info)
		{
			info.disabled = true;
			continue;
		}
	}
}


//makes this resource available 
ShaderCode.prototype.register = function()
{
	LS.ResourcesManager.registerResource( this.fullpath || this.filename, this );
}

//searches for materials using this ShaderCode and forces them to be updated (update the properties)
ShaderCode.prototype.applyToMaterials = function( scene )
{
	scene = scene || LS.GlobalScene;
	var filename = this.fullpath || this.filename;

	//materials in the resources
	for(var i in LS.ResourcesManager.resources)
	{
		var res = LS.ResourcesManager.resources[i];
		if( res.constructor !== LS.ShaderMaterial || res._shader != filename )
			continue;

		res.processShaderCode();
	}

	//embeded materials
	var nodes = scene.getNodes();
	for(var i = 0; i < nodes.length; ++i)
	{
		var node = nodes[i];
		if(node.material && node.material.constructor === LS.ShaderMaterial && node.material._shader == filename )
			node.material.processShaderCode();
	}
}

//used in editor
ShaderCode.prototype.hasEditableText = function() { return true; }

ShaderCode.removeComments = function( code )
{
	// /^\s*[\r\n]/gm
	return code.replace(/(\/\*([\s\S]*?)\*\/)|(\/\/(.*)$)/gm, '');
}

ShaderCode.replaceCode = function( code, context )
{
	return GL.Shader.replaceCodeUsingContext( code, context );
}

//WIP: parses ShaderLab (unity) syntax
ShaderCode.parseShaderLab = function( code )
{
	var root = {};
	var current = root;
	var current_token = [];
	var stack = [];
	var mode = 0;
	var current_code = "";

	var lines = ShaderCode.removeComments( code ).split("\n");
	for(var i = 0; i < lines.length; ++i)
	{
		var line = lines[i].trim();
		var words = line.match(/[^\s"]+|"([^"]*)"/gi);
		if(!words)
			continue;

		if(mode != 0)
		{
			var w = words[0].trim();
			if(w == "ENDGLSL" || w == "ENDCG" )
			{
				mode = 0;
				current.codetype = mode;
				current.code = current_code;
				current_code = "";
			}
			else
			{
				current_code += line + "\n";
			}
			continue;
		}

		for(var j = 0; j < words.length; ++j)
		{
			var w = words[j];

			if(w == "{")
			{
				var node = {
					name: current_token[0], 
					params: current_token.slice(1).join(" "),
					content: {}
				};
				current[ node.name ] = node;
				current_token = [];
				stack.push( current );
				current = node.content;
			}
			else if(w == "}")
			{
				if(stack.length == 0)
				{
					console.error("error parsing ShaderLab code, the number of { do not matches the }");
					return null;
				}
				if(current_token.length)
				{
					current[ current_token[0] ] = current_token.join(" ");
					current_token = [];
				}
				current = stack.pop();
			}
			else if(w == "{}")
			{
				var node = {
					name: current_token[0], 
					params: current_token.slice(1).join(" "),
					content: {}
				};
				current[ node.name ] = node;
				current_token = [];
			}
			else if(w == "GLSLPROGRAM" || w == "CGPROGRAM" )
			{
				if( w == "GLSLPROGRAM" )
					mode = 1;
				else
					mode = 2;
				current_code = "";
			}
			else 
				current_token.push(w);
		}
	}

	return root;
}

ShaderCode.getDefaultCode = function( instance,  render_settings, pass )
{
	if( ShaderCode.default_code_instance )
		return ShaderCode.default_code_instance;

	var shader_code = ShaderCode.default_code_instance = new LS.ShaderCode();
	shader_code.code = ShaderCode.flat_code;
	return shader_code;
}

//default vertex shader code
ShaderCode.default_vs = "\n\
precision mediump float;\n\
attribute vec3 a_vertex;\n\
attribute vec3 a_normal;\n\
attribute vec2 a_coord;\n\
#pragma shaderblock \"vertex_color\"\n\
#pragma shaderblock \"coord1\"\n\
#ifdef BLOCK_COORD1\n\
	attribute vec2 a_coord1;\n\
	varying vec2 v_uvs1;\n\
#endif\n\
#ifdef BLOCK_VERTEX_COLOR\n\
	attribute vec4 a_color;\n\
	varying vec4 v_vertex_color;\n\
#endif\n\
\n\
//varyings\n\
varying vec3 v_pos;\n\
varying vec3 v_normal;\n\
varying vec2 v_uvs;\n\
varying vec3 v_local_pos;\n\
varying vec3 v_local_normal;\n\
varying vec4 v_screenpos;\n\
\n\
//matrices\n\
uniform mat4 u_model;\n\
uniform mat4 u_normal_model;\n\
uniform mat4 u_view;\n\
uniform mat4 u_viewprojection;\n\
\n\
//globals\n\
uniform float u_time;\n\
uniform vec4 u_viewport;\n\
uniform float u_point_size;\n\
\n\
#pragma shaderblock \"morphing\"\n\
#pragma shaderblock \"skinning\"\n\
\n\
//camera\n\
uniform vec3 u_camera_eye;\n\
void main() {\n\
	\n\
	vec4 vertex4 = vec4(a_vertex,1.0);\n\
	v_local_pos = a_vertex;\n\
	v_local_normal = a_normal;\n\
	v_normal = a_normal;\n\
	v_uvs = a_coord;\n\
	#ifdef BLOCK_COORD1\n\
		v_uvs1 = a_coord1;\n\
	#endif\n\
	#ifdef BLOCK_VERTEX_COLOR\n\
		v_vertex_color = a_color;\n\
	#endif\n\
  \n\
  //deforms\n\
  applyMorphing( vertex4, v_normal );\n\
  applySkinning( vertex4, v_normal );\n\
	\n\
	//vertex\n\
	v_pos = (u_model * vertex4).xyz;\n\
  \n\
  \n\
	//normal\n\
	v_normal = (u_normal_model * vec4(v_normal,0.0)).xyz;\n\
	gl_Position = u_viewprojection * vec4(v_pos,1.0);\n\
	v_screenpos = gl_Position;\n\
}\n\
"

//default fragment shader code
ShaderCode.default_fs = "\n\
	#ifdef DRAW_BUFFERS\n\
		#extension GL_EXT_draw_buffers : require \n\
	#endif\n\
	precision mediump float;\n\
	uniform vec4 u_material_color;\n\
	void main() {\n\
		#ifdef DRAW_BUFFERS\n\
			gl_FragData[0] = u_material_color;\n\
		#else\n\
			gl_FragColor = u_material_color;\n\
		#endif\n\
	}\n\
";

ShaderCode.flat_code = "\n\
\\color.fs\n\
"+ ShaderCode.default_fs +"\n\
";


LS.ShaderCode = ShaderCode;
LS.registerResourceClass( ShaderCode );

///@FILE:../src/resources/graphCode.js
/**
* This is a class to contain the code from a graph, it doesnt execute the graph (this is done in GraphComponent or GraphMaterial)
* but it stores an instance of the graph that is never executed, this is used in the GraphMaterial
* It is here so different GraphComponent can share the same Graph structure and it can be stored in a JSON
* 
* @class GraphCode
* @constructor
*/

function GraphCode( data )
{
	this._data = { "object_class":"GraphCode" };
	this._modified = false;

	//graph
	this._graph = new LiteGraph.LGraph();
	this._graph._graphcode = this; //double link

	//type
	this.type = LS.GraphCode.LOGIC_GRAPH;

	//properties
	this.properties = [];

	//extra
	this.extra = {};

	//versioning
	this._version = 0;

	//this._shader_code is created in case is a shader_code

	if(data)
		this.setData( data, true );
}

GraphCode.LOGIC_GRAPH = 1;
GraphCode.SHADER_GRAPH = 2;

GraphCode.EXTENSION = "GRAPH.json";
GraphCode.hasPreview = false; //should this resource use a preview image?

Object.defineProperty( GraphCode.prototype, "graph", {
	enumerable: false,
	get: function() {
		return this._graph;
	},
	set: function(v) {
		console.error("graph cannot be set manually");
	}
});

Object.defineProperty( GraphCode.prototype, "data", {
	enumerable: false,
	get: function() {
		return this.getData();
	},
	set: function(v) {
		this.setData( v );
	}
});

Object.defineProperty( GraphCode.prototype, "version", {
	enumerable: false,
	get: function() {
		return this._version;
	},
	set: function(v) {
		console.error("version cannot be set manually");
	}
});

//used when storing/retrieving the resource
GraphCode.prototype.setData = function( data, skip_modified_flag )
{
	if(!data)
	{
		this._data = null;
		this.properties = [];
		this.type = LS.GraphCode.LOGIC_GRAPH;
		this.extra = {};
		return;
	}

	if(LS.catch_exceptions)
		try
		{
			if(data.constructor === String)
				this._data = JSON.parse( data );
			else
				this._data = JSON.parse( JSON.stringify( data ) ); //clone...
		}
		catch (err)
		{
			console.error("error in graph data");
			return;
		}
	else
	{
		if(data.constructor === String)
			this._data = JSON.parse( data );
		else
			this._data = JSON.parse( JSON.stringify( data ) ); //clone...
	}

	this._graph.configure( this._data );
	
	this.type = this._data.type || LS.GraphCode.LOGIC_GRAPH;
	this.extra = this._data.extra || {};
	this.properties = this._data.properties || [];

	if(!skip_modified_flag)
	{
		this._version++;
		this._modified = true;
	}
}

GraphCode.prototype.getData = function() {
	var data = this.graph.serialize();
	data.object_class = "GraphCode";
	data.type = this.type;
	data.properties = this.properties;
	data.extra = this.extra;
	return data;
}

GraphCode.prototype.getDataToStore = function() {
	return JSON.stringify( this.getData() );
}

GraphCode.prototype.getCategory = function() {
	return "Graph";
}

GraphCode.prototype.getProperty = function(name)
{
	for(var i = 0; i < this.properties.length; ++i)
		if( this.properties[i].name == name )
			return this.properties[i];
	return null;
}

//sends changes in this graphcode to all nodes using this graph
GraphCode.prototype.propagate = function()
{
	var filename = this.fullpath || this.filename;

	if( this.type == GraphCode.LOGIC_GRAPH )
	{
		var components = LS.GlobalScene.findNodeComponents( LS.Components.GraphComponent );
		for(var i = 0; i < components.length; ++i)
		{
			var comp = components[i];
			if(comp.filename != this.filename )
				continue;
			comp.graphcode = this;
		}
	}
}

//used in graph materials
//as_string for debug
GraphCode.prototype.getShaderCode = function( as_string, template )
{
	if( this._shader_code && this._code_version == this._graph._version )
	{
		if( as_string )
			return this._shader_code._code;
		return this._shader_code;
	}

	if(!this._shader_code)
		this._shader_code = new LS.ShaderCode();

	var output_node = this._graph.findNodesByClass("shader/output")[0];
	if(!output_node)
		return null;

	//get code
	var code = output_node.getShaderCode( template );
	if( as_string )
		return code;
	this._shader_code.code = code;
	this._code_version = this._graph._version;
	return this._shader_code;

	/*
	var context = {
		vs_out: "",
		vs_local: "",
		vs_global: "",
		fs_snippets: {}, //to request once snippets from LS.Shaders.snippets
		fs_functions: {}, //to add once functions code
		fs_out: "",
		fs_code: ""
	};

	//place uniforms
	for(var i = 0; i < this.properties.length; ++i)
	{
		var prop = this.properties[i];
		var type = LS.GLSLCode.types_conversor[ prop.type.toLowerCase() ] || prop.type;
		context.fs_out += "	uniform " + type + " u_" + prop.name + ";\n";
	}

	var nodes = this._graph._nodes_in_order;
	this._graph.runStep(1);
	if(nodes)
		for(var i = 0; i < nodes.length; ++i)
		{
			var node = nodes[i];
			if( node.onGetCode )
				node.onGetCode( "glsl", context );
		}

	if( as_string )
		return context.fs_code;

	//expand requested functions
	var fs_snippets_code = "";
	for(var i in context.fs_snippets)
		fs_snippets_code += "#pragma snippet \"" + i + "\"\n";
	if(fs_snippets_code)
		context.fs_out = fs_snippets_code + "\n" + context.fs_out;

	var fs_functions_code = "";
	for(var i in context.fs_functions)
		fs_functions_code += context.fs_functions[i] + "\n";
	if(fs_functions_code)
		context.fs_out = fs_functions_code + "\n" + context.fs_out;


	this._shader_code.code = LS.ShaderCode.replaceCode( template, context );
	this._code_version = this._graph._version;
	return this._shader_code;
	*/
}

LS.GraphCode = GraphCode;
LS.registerResourceClass( GraphCode );

///@FILE:../src/graph/core.js
///@INFO: GRAPHS
if(typeof(LiteGraph) != "undefined")
{
	function LGraphTween() {
		this.addInput("tween", LiteGraph.ACTION);
		this.addInput("value", "");
		this.addInput("prop", "");
		this.addOutput("start", LiteGraph.EVENT);
		this.addOutput("finish", LiteGraph.EVENT);

		this.properties = {
			duration: 1,
			locator: ""
		};
	}

	LGraphTween.title = "Tween";
	LGraphTween.desc = "tween between two values";

	LGraphTween.prototype.onAction = function( action, param )
	{
		var in_node = this.getInputNode(2);
		var scene = this.graph._scene || LS.GlobalScene; //subgraphs do not have an scene assigned
		var info = null;

		if(in_node)
		{	
			if(!in_node.getLocatorInfo )
				return;
			info = in_node.getLocatorInfo();
		}
		else if( this.properties.locator )
		{
			if(!this._locator_split)
				this._locator_split = this.properties.locator.split("/");
			info = scene.getPropertyInfoFromPath( this._locator_split );
		}

		if(!info)
			return;

		var target_value = this.getInputData(1);
		if( target_value == null )
			return;

		var that = this;

		this.triggerSlot(0);
		LS.Tween.easeProperty( info.target, info.name, target_value, this.properties.duration, null, inner_complete );

		function inner_complete()
		{
			that.triggerSlot(1);			
		}
	}

	LGraphTween.prototype.onPropertyChanged = function(name,value)
	{
		if(name == "locator")
		{
			if( value )
				this._locator_split = value.split("/");
			else
				this._locator_split = null;
		}
	}

	LGraphTween.prototype.onDropItem = function( event )
	{
		var locator = event.dataTransfer.getData("uid");
		if(!locator)
			return;
		this.properties.locator = locator;
		return true;
	}

	LiteGraph.registerNodeType( "core/tween", LGraphTween );	

	//gets a resource
	function LGraphResource() {
		this.addOutput("out", "");
		this.properties = {
			filename: "",
			type: ""
		};
	}

	LGraphResource.title = "resource";
	LGraphResource.desc = "gets a resource";

	LGraphResource.widgets_info = {
		filename: { widget: "resource" }
	};

	LGraphResource.prototype.onExecute = function() {
		var res = null;
		if(this.properties.filename)
			res = LS.ResourcesManager.resources[this.properties.filename];

		//wrong type
		if( res && this.properties.type && res.constructor.name.toLowerCase() !== this.properties.type.toLowerCase() )
			res = null;

		this.setOutputData(0,res);
	}

	LGraphResource.prototype.getResources = function(o)
	{
		if(this.properties.filename)
			o[this.properties.filename] = true;
	}

	LiteGraph.registerNodeType( "core/resource", LGraphResource );	


	function LGraphGetMesh() {
		this.addOutput("out", "mesh");
		this.properties = {
			name: ""
		};
	}

	LGraphGetMesh.title = "mesh";
	LGraphGetMesh.desc = "gets mesh";

	LGraphGetMesh.widgets_info = {
		name: { widget: "mesh" }
	};

	LGraphGetMesh.prototype.onExecute = function() {
		var mesh = null;
		if(this.properties.name)
			mesh = LS.ResourcesManager.meshes[this.properties.name];
		if(mesh && (mesh.constructor !== GL.Mesh || mesh.ready === false) )
			mesh = null;
		this.setOutputData(0,mesh);
	}

	LGraphGetMesh.prototype.getResources = function(o)
	{
		if(this.properties.name)
			o[this.properties.name] = true;
	}

	LiteGraph.registerNodeType( "geometry/mesh", LGraphGetMesh );	


	//****************************************

	function LGraphRenderMeshInScene() {
		this.addInput("mesh", "mesh");
		this.addInput("material", "material");
		this.addInput("mat4", "mat4");
		this.addInput("instances", "[mat4]");

		this.properties = {
			enabled: true,
			primitive: GL.TRIANGLES,
			use_node_transform: true,
			use_node_material: true
		};
	}

	LGraphRenderMeshInScene.title = "Render";
	LGraphRenderMeshInScene.desc = "renders a mesh with a material inside the scene";

	LGraphRenderMeshInScene.PRIMITIVE_VALUES = { "points":GL.POINTS, "lines":GL.LINES, "line_loop":GL.LINE_LOOP,"line_strip":GL.LINE_STRIP, "triangles":GL.TRIANGLES, "triangle_fan":GL.TRIANGLE_FAN, "triangle_strip":GL.TRIANGLE_STRIP };

	LGraphRenderMeshInScene.widgets_info = {
		primitive: { widget: "combo", values: LGraphRenderMeshInScene.PRIMITIVE_VALUES }
	};

	LGraphRenderMeshInScene.prototype.onExecute = function() {

		if(!this.properties.enabled)
			return;

		//as renderinstance
		if(!this._RI)
			this._RI = new LS.RenderInstance();

		//root node
		var node = this.graph._scenenode;
		if(!node)
			return;
		this._RI.fromNode( node, true );

		var mesh = this.getInputData(0);
		if(!mesh)
			return;
		this._RI.setMesh( mesh );

		//material
		var material = this.getInputData(1);
		if(!material && this.properties.use_node_material)
			material = node.getMaterial();
		if(!material)
			material = LS.Renderer.default_material;
		this._RI.setMaterial( material );

		this._RI.primitive = this.properties.primitive;

		//transform
		var model = this.getInputData(2);
		if(!model)
		{
			if(this.properties.use_node_transform && node && node.transform )
				model = node.transform.getGlobalMatrixRef();
			else
				model = LS.IDENTITY;
		}
			
		this._RI.setMatrix( model );

		//instancing
		var instances = this.getInputData(3);
		if(instances)
		{
			//if(model == LS.IDENTITY) //if not multiply all by model?
				this._RI.instanced_models = instances;
		}
		else
			this._RI.instanced_models = null;
		this._RI.use_bounding = !instances;

		LS.Renderer.addImmediateRenderInstance( this._RI );
	}

	LiteGraph.registerNodeType( "geometry/render_mesh_in_scene", LGraphRenderMeshInScene );
	
}
///@FILE:../src/graph/scene.js
///@INFO: GRAPHS
if(typeof(LiteGraph) != "undefined")
{
	//scene/component is in another file, components.js
	
	/* Scene LNodes ***********************/

	global.LGraphScene = function()
	{
		this.addOutput("Time","number");
		this._scene = null;
	}

	LGraphScene.title = "Scene";
	LGraphScene.desc = "Scene";

	LGraphScene.getComponents = function(node, result)
	{
		result = result || [];
		var compos = node.getComponents();
		if(!compos)
			return result;

		for(var i = 0; i < compos.length; ++i)
		{
			var name = LS.getClassName( compos[i].constructor );
			result.push( [name, name] );
		}

		return result;
	}

	LGraphScene.prototype.onAdded = function( graph )
	{
		this.bindEvents( this.graph.getScene ? this.graph.getScene() : LS.GlobalScene );
	}

	LGraphScene.prototype.onRemoved = function()
	{
		this.bindEvents( null );
	}

	LGraphScene.prototype.onConnectionsChange = function()
	{
		this.bindEvents( this.graph.getScene ? this.graph.getScene() : LS.GlobalScene );
	}

	//bind events attached to this component
	LGraphScene.prototype.bindEvents = function( scene )
	{
		if(this._scene)
			LEvent.unbindAll( this._scene, this );

		this._scene = scene;
		if( !this._scene )
			return;
		
		//iterate outputs
		if(this.outputs && this.outputs.length)
			for(var i = 0; i < this.outputs.length; ++i )
			{
				var output = this.outputs[i];
				if( output.type !== LiteGraph.EVENT )
					continue;
				var event_name = output.name.substr(3);
				LEvent.bind( this._scene, event_name, this.onEvent, this );
			}
	}

	LGraphScene.prototype.onEvent = function( event_name, params )
	{
		this.trigger( "on_" + event_name, params );
	}

	LGraphScene.prototype.onExecute = function()
	{
		var scene = this.graph.getScene ? this.graph.getScene() : LS.GlobalScene;

		//read inputs
		if(this.inputs)
		for(var i = 0; i < this.inputs.length; ++i)
		{
			var input = this.inputs[i]; //??
			var v = this.getInputData(i);
			if(v === undefined)
				continue;
		}

		//write outputs
		if(this.outputs)
		for(var i = 0; i < this.outputs.length; ++i)
		{
			var output = this.outputs[i];
			if(!output.links || !output.links.length || output.type == LiteGraph.EVENT )
				continue;
			var result = null;
			switch( output.name )
			{
				case "Time": result = scene.getTime(); break;
				case "Elapsed": result = (scene._last_dt != null ? scene._last_dt : 0); break;
				case "Frame": result = (scene._frame != null ? scene._frame : 0); break;
				default:
					result = scene.root.getComponent(output.name);
			}
			this.setOutputData(i,result);
		}
	}

	LGraphScene.prototype.onGetOutputs = function()
	{
		var r = [["Elapsed","number"],["Frame","number"],["on_start",LiteGraph.EVENT],["on_finish",LiteGraph.EVENT]];
		return LGraphScene.getComponents( this.graph.getScene().root, r);
	}

	LiteGraph.registerNodeType("scene/scene", LGraphScene );

	//********************************************************

	global.LGraphSceneNode = function()
	{
		this.properties = { node_id: "" };
		this.size = [140,30];
		this._node = null;
	}

	LGraphSceneNode.title = "SceneNode";
	LGraphSceneNode.desc = "Node on the scene";
	LGraphSceneNode.highlight_color = "#CCC";

	LGraphSceneNode.prototype.onRemoved = function()
	{
		if(this._node)
			this.bindNodeEvents(null);
	}

	LGraphSceneNode.prototype.onConnectionsChange = function()
	{
		this.bindNodeEvents( this._component );
	}

	LGraphSceneNode.prototype.getTitle = function()
	{
		var node = this._node || this.getNode();
		if(node)
			return node.name;
		return this.title;
	}

	LGraphSceneNode.prototype.getNode = function()
	{
		var node_id = null;

		if(!this.graph)
			return null;

		//first check input
		if(this.inputs && this.inputs[0])
			node_id = this.getInputData(0);

		//hardcoded node
		if( node_id && node_id.constructor === LS.SceneNode )
		{
			if(this._node != node_id)
				this.bindNodeEvents(node_id);
			return node_id;
		}

		if(node_id && node_id.constructor === String)
			this.properties.node_id = node_id;

		//then check properties
		if(	!node_id && this.properties.node_id )
			node_id = this.properties.node_id;

		if( node_id == "@" || !node_id )
		{
			if( this.graph._scenenode )
				return this.graph._scenenode;
			return null;
		}

		//get node from scene
		var scene = this.graph && this.graph.getScene ? this.graph.getScene() : LS.GlobalScene;
		if(!scene)
			return;

		var node = null;
		if(node_id)
			node = scene.getNode( node_id );

		//hook events
		if(this._node != node)
			this.bindNodeEvents(node);

		return node;
	}

	LGraphSceneNode.prototype.onExecute = function()
	{
		var node = this.getNode();
		if(!node)
			return;

		//read inputs
		if(this.inputs) //there must be inputs always but just in case
		{
			for(var i = 0; i < this.inputs.length; ++i)
			{
				var input = this.inputs[i];
				if( input.type === LiteGraph.ACTION )
					continue;
				var v = this.getInputData(i);
				if(v === undefined)
					continue;
				switch( input.name )
				{
					case "UID": this.properties.node_id = v; break;
					case "SceneNode": this.properties.node_id = v ? v.uid : null; if(v) node = v; break;
					case "Transform": node.transform.copyFrom(v); break;
					case "Material": node.material = v;	break;
					case "Visible": node.flags.visible = v; break;
					default:
						break;
				}
			}
		}

		//write outputs
		if(this.outputs)
		for(var i = 0; i < this.outputs.length; ++i)
		{
			var output = this.outputs[i];
			if(!output.links || !output.links.length || output.type == LiteGraph.EVENT )
				continue;
			switch( output.name )
			{
				case "SceneNode": this.setOutputData( i, node ); break;
				case "Material": this.setOutputData( i, node.getMaterial() ); break;
				case "Mesh": this.setOutputData( i, node.getMesh() ); break;
				case "Transform": this.setOutputData( i, node.transform ); break;
				case "Global Model": this.setOutputData( i, node.transform ? node.transform._global_matrix : LS.IDENTITY ); break;
				case "Name": this.setOutputData( i, node.name ); break;
				case "Children": this.setOutputData( i, node.children ); break;
				case "UID": this.setOutputData( i, node.uid ); break;
				case "Mesh": this.setOutputData(i, node.getMesh()); break;
				case "Visible": this.setOutputData(i, node.flags.visible ); break;
				default:
					//this must be refactored
					var compo = node.getComponentByUId( output.name );
					if(!compo)
					{
						//SPECIAL CASE: maybe the node id changed so the output.name contains the uid of another node, in that case replace it
						var old_compo = node.scene.findComponentByUId( output.name );
						if(old_compo)
						{
							var class_name = LS.getObjectClassName( old_compo );
							compo = node.getComponent( class_name );
							if( compo )
								output.name = compo.uid; //replace the uid
						}
					}

					this.setOutputData(i, compo );
					break;
			}
		}
	}

	LGraphSceneNode.prototype.onDrawBackground = function(ctx)
	{
		var node = this.getNode();
		if(!node)
		{
			this.boxcolor = "red";
			return;
		}

		var highlight = node._is_selected;

		if(highlight)
		{
			this.boxcolor = LGraphSceneNode.highlight_color;
			if(!this.flags.collapsed)
			{
				ctx.fillStyle = LGraphSceneNode.highlight_color;
				ctx.fillRect(0,0,this.size[0],2);
			}
		}
		else
			this.boxcolor = null;
	}

	LGraphSceneNode.prototype.getComponents = function(result)
	{
		result = result || [];
		var node = this.getNode();
		if(!node)
			return result;
		var compos = node.getComponents();
		if(!compos)
			return result;

		for(var i = 0; i < compos.length; ++i)
		{
			var name = LS.getClassName( compos[i].constructor );
			result.push( [ compos[i].uid, name, { label: name } ] );
		}

		return result;
	}

	LGraphSceneNode.prototype.onDropItem = function( event )
	{
		var node_id = event.dataTransfer.getData("node_uid");
		if(!node_id)
			return;
		this.properties.node_id = node_id;
		this.onExecute();
		return true;
	}

	LGraphSceneNode.prototype.onPropertyChanged = function(name,value)
	{
		if(name == "node_id")
			this.getNode(); //updates this._node and binds events
	}

	//bind events attached to this component
	LGraphSceneNode.prototype.bindNodeEvents = function( node )
	{
		if(this._node)
			LEvent.unbindAll( this._node, this );

		this._node = node;
		if( !this._node )
			return;
		
		//iterate outputs
		if(this.outputs && this.outputs.length)
			for(var i = 0; i < this.outputs.length; ++i )
			{
				var output = this.outputs[i];
				if( output.type !== LiteGraph.EVENT )
					continue;
				var event_name = output.name.substr(3);
				LEvent.bind( this._node, event_name, this.onNodeEvent, this );
			}
	}

	LGraphSceneNode.prototype.onNodeEvent = function( e, params )
	{
		//trigger event
		this.trigger( "on_" + e, params );
	}

	LGraphSceneNode.prototype.onGetInputs = function()
	{
		var result = [["Visible","boolean"],["UID","string"],["SceneNode","SceneNode"],["Material","Material"]];
		return this.getComponents(result);
	}

	LGraphSceneNode.prototype.onGetOutputs = function()
	{
		var result = [["SceneNode","SceneNode"],["Visible","boolean"],["Material","Material"],["Mesh","Mesh"],["Name","string"],["UID","string"],["Global Model","mat4"],["Children","scenenode[]"],["on_clicked",LiteGraph.EVENT]];
		return this.getComponents(result);
	}

	LGraphSceneNode.prototype.onInspect = function( inspector )
	{
		var that = this;
		inspector.addButton(null, "Inspect node", function(){
			var node = that.getNode();
			if(node)
				EditorModule.inspect( node );
		});
	}


	LiteGraph.registerNodeType("scene/node", LGraphSceneNode );

	//********************************************************

	global.LGraphMaterial = function()
	{
		this.properties = { material_id: "", node_id: "" };
		this.addInput("","Material");
		this.addOutput("","Material");
		this.material = null;

		this.addWidget("button","Create", "", this.onCreateMaterial.bind(this) );
	}

	LGraphMaterial.title = "Material";
	LGraphMaterial.desc = "Material of a node";

	LGraphMaterial.prototype.onCreateMaterial = function(w, graphcanvas, node, pos, event)
	{
		var types = Object.keys( LS.MaterialClasses );
		types.push(null,"clear");
		var menu = new LiteGraph.ContextMenu(types, { event: event, callback: inner });
		var that = this;
		function inner(v)
		{
			if(v == "clear")
			{
				that.material = null;
				return;
			}

			if(!LS.MaterialClasses[ v ])
				return;
			var mat = new LS.MaterialClasses[ v ];
			that.material = mat;
			EditorModule.inspect( that );
		}
	}

	LGraphMaterial.prototype.getResources = function(res)
	{
		if(this.material && this.material.getResources)
			this.material.getResources(res);
	}

	LGraphTexture.prototype.onResourceRenamed = function( old_name, new_name ) {
		if(this.material && this.material.onResourceRenamed)
			this.material.onResourceRenamed( old_name, new_name );
	};

	LGraphMaterial.prototype.onExecute = function()
	{
		var mat = this.getMaterial();
		if(!mat)
			return;

		//read inputs
		for(var i = 0; i < this.inputs.length; ++i)
		{
			var input = this.inputs[i];
			var v = this.getInputData(i);
			if(v === undefined)
				continue;

			if(input.name == "Material")
				continue;
			mat.setProperty( input.name, v );
		}

		//write outputs
		if(this.outputs)
		{
			this.setOutputData( 0, mat );
			for(var i = 1; i < this.outputs.length; ++i)
			{
				var output = this.outputs[i];
				if(!output.links || !output.links.length)
					continue;
				var v = mat.getProperty( output.name );
				this.setOutputData( i, v );
			}
		}
	}

	LGraphMaterial.prototype.getMaterial = function()
	{
		if(this.material)
			return this.material;

		//if it has an input material, use that one
		var slot = this.findInputSlot("Material");
		if( slot != -1)
			return this.getInputData( slot );

		if(	this.properties.material_id && LS.RM.materials[ this.properties.material_id ] )
			return LS.RM.materials[ this.properties.material_id ];

		if(	this.properties.node_id )
		{
			var scene = this.graph.getScene();
			var node = scene.getNode( this.properties.node_id );
			if(node)
				return node.getMaterial();
		}

		return null;
	}

	LGraphMaterial.prototype.onGetInputs = function()
	{
		var mat = this.getMaterial();
		if(!mat)
			return;
		var o = mat.getPropertiesInfo();
		var results = [["Material","Material"]];
		for(var i in o)
			results.push([i,o[i]]);
		return results;

		/*
		var results = [["Material","Material"],["Alpha","number"],["Specular f.","number"],["Diffuse","color"],["Ambient","color"],["Emissive","color"],["UVs trans.","texmatrix"]];
		for(var i in Material.TEXTURE_CHANNELS)
			results.push(["Tex." + Material.TEXTURE_CHANNELS[i],"Texture"]);
		return results;
		*/
	}

	LGraphMaterial.prototype.onGetOutputs = function()
	{
		var mat = this.getMaterial();
		if(!mat)
			return;
		var o = mat.getPropertiesInfo();
		var results = [["Material","Material"]];
		for(var i in o)
			results.push([i,o[i]]);
		return results;

		/*
		var results = [["Material","Material"],["Alpha","number"],["Specular f.","number"],["Diffuse","color"],["Ambient","color"],["Emissive","color"],["UVs trans.","texmatrix"]];
		for(var i in Material.TEXTURE_CHANNELS)
			results.push(["Tex." + Material.TEXTURE_CHANNELS[i],"Texture"]);
		return results;
		*/
	}

	LGraphMaterial.prototype.onSerialize = function(o)
	{
		if( this.material && this.material.serialize )
		{
			o.material = this.material.serialize();
			o.material.className = LS.getObjectClassName( this.material );
		}
	}

	LGraphMaterial.prototype.onConfigure = function(o)
	{
		if(o.material)
		{
			var ctor = LS.MaterialClasses[ o.material.className ];
			if(ctor)
			{
				this.material = new ctor();
				this.material.configure( o.material );
			}
		}
	}

	LGraphMaterial.prototype.onInspect = function( inspector )
	{
		var that = this;
		var mat = this.getMaterial();
		if(mat)
		{
			inspector.addTitle("Material (" + LS.getObjectClassName(mat) + ")" );
			EditorModule.showMaterialProperties( mat, inspector );
		}
		/*
		inspector.addButton(null, "Inspect material", function(){
			var mat = that.getMaterial();
			if(mat)
				EditorModule.inspect( mat );
		});
		*/
	}

	LiteGraph.registerNodeType("scene/material", LGraphMaterial );
	global.LGraphMaterial = LGraphMaterial;

	//************************************

	global.LGraphGlobal = function LGraphGlobal()
	{
		this.addOutput("Value");
		this.properties = {name:"myvar", value: 0, type: "number", widget: "default", min:0, max:1 };
	}

	LGraphGlobal.title = "Global";
	LGraphGlobal.desc = "Global var for the graph";
	LGraphGlobal["@type"] = { type:"enum", values:["number","boolean","string","node","vec2","vec3","vec4","color","texture"]};
	LGraphGlobal["@widget"] = { type:"enum", values:[ "default", "slider", "pad" ]};

	LGraphGlobal.prototype.onExecute = function()
	{
		if(!this.properties.name)
			return;

		this.setOutputData(0, this.properties.value);
	}

	LGraphGlobal.prototype.onDrawBackground = function()
	{
		var name = this.properties.name;
		this.outputs[0].label = name;
	}

	LiteGraph.registerNodeType("scene/global", LGraphGlobal );

	global.LGraphSceneTime = function()
	{
		this.addOutput("Time","number");
		this._scene = null;
	}


	LGraphSceneTime.title = "Time";
	LGraphSceneTime.desc = "Time";

	LGraphSceneTime.prototype.onExecute = function()
	{
		var scene = this.graph.getScene();
		if(!scene)
			return;

		//read inputs
		if(this.inputs)
		for(var i = 0; i < this.inputs.length; ++i)
		{
			var input = this.inputs[i]; //??
			var v = this.getInputData(i);
			if(v === undefined)
				continue;
		}

		//write outputs
		if(this.outputs)
		for(var i = 0; i < this.outputs.length; ++i)
		{
			var output = this.outputs[i];
			if(!output.links || !output.links.length || output.type == LiteGraph.EVENT )
				continue;
			var result = null;
			switch( output.name )
			{
				case "Time": result = scene.getTime(); break;
				case "Elapsed": result = (scene._last_dt != null ? scene._last_dt : 0); break;
				case "Frame": result = (scene._frame != null ? scene._frame : 0); break;
				default:
					continue;
			}
			this.setOutputData(i,result);
		}
	}

	LGraphSceneTime.prototype.onGetOutputs = function()
	{
		return [["Elapsed","number"],["Time","number"]];
	}

	LiteGraph.registerNodeType("scene/time", LGraphSceneTime );

	//************************************

	global.LGraphLocatorProperty = function LGraphLocatorProperty()
	{
		this.addInput("in");
		this.addOutput("out");
		this.properties = { locator: "", cache_object: true };
		this._locator_split = null;
		this._locator_info = null;
	}

	LGraphLocatorProperty.title = "Property";
	LGraphLocatorProperty.desc = "A property of a node or component of the scene specified by its locator string";
	LGraphLocatorProperty.highlight_color = "#CCC";

	LGraphLocatorProperty.prototype.getLocatorInfo = function( force )
	{
		if(!this._locator_split)
			return null;
		if( !force && this.properties.cache_object && this._locator_info )
			return this._locator_info;
		if( !this.graph )
			return null;
		var scene = this.graph._scene || LS.GlobalScene; //subgraphs do not have an scene assigned
		this._locator_info = scene.getPropertyInfoFromPath( this._locator_split );
		if(this._locator_info && this.inputs && this.inputs.length)
			this.inputs[0].type = this._locator_info.type;
		return this._locator_info;
	}

	LGraphLocatorProperty.prototype.onDropItem = function( event )
	{
		var item_type = event.dataTransfer.getData("type");
		//if(item_type != "property")
		//	return;
		var locator = event.dataTransfer.getData("uid");
		if(!locator)
			return;
		this.properties.locator = locator;
		this.onExecute();
		return true;
	}

	LGraphLocatorProperty.prototype.onPropertyChanged = function(name,value)
	{
		if(name == "locator")
		{
			if( value )
				this._locator_split = value.split("/");
			else
				this._locator_split = null;
		}
	}

	LGraphLocatorProperty.prototype.onAction = function( action, param )
	{
		//toggle
		var info = this.getLocatorInfo();
		LSQ.setFromInfo( info, !LSQ.getFromInfo( info ) );
	}

	LGraphLocatorProperty.prototype.getTitle = function()
	{
		if( (!this.title || this.title == LGraphLocatorProperty.title) && this._locator_info)
			return this._locator_info.name;
		return this.title || LGraphLocatorProperty.title;
	}

	LGraphLocatorProperty.prototype.onDrawBackground = function(ctx)
	{
		var info = this.getLocatorInfo();
		if(!info)
		{
			this.boxcolor = "red";
			return;
		}

		var highlight = info.node && info.node._is_selected;

		if(highlight)
		{
			this.boxcolor = LGraphLocatorProperty.highlight_color;
			if(!this.flags.collapsed) //render line
			{
				ctx.fillStyle = LGraphLocatorProperty.highlight_color;
				ctx.fillRect(0,0,this.size[0],2);
			}
		}
		else
			this.boxcolor = null;
	}

	LGraphLocatorProperty.prototype.onExecute = function()
	{
		var info = this.getLocatorInfo();
		this._last_info = info;

		if(info && info.target)
		{
			if( this.inputs.length && this.inputs[0].link !== null )
			{
				var v = this.getInputData(0);
				if(v !== undefined)
					LSQ.setFromInfo( info, v );
			}
			if( this.outputs.length && this.outputs[0].links && this.outputs[0].links.length )
				this.setOutputData( 0, LSQ.getFromInfo( info ));
		}

		this.setOutputData( 1, this.properties.locator );
	}

	LGraphLocatorProperty.prototype.onInspect = function( inspector )
	{
		var info = this.getLocatorInfo(true);
		if(!info)
			return;
		inspector.addSeparator();
		var type = info.type;
		var var_info = null;
		if( info.target && info.target.constructor["@" + info.name] )
		{
			var_info = info.target.constructor["@" + info.name];
			if( var_info.widget )
				type = var_info.widget;
			else if( var_info.type )
				type = var_info.type;
		}
		inspector.add( type, info.name, info.value, { callback: function(v){
			LS.setObjectProperty( info.target, info.name, v );
		}});
	}

	LGraphLocatorProperty.prototype.onGetOutputs = function()
	{
		return [["locator","string"]];
	}

	LiteGraph.registerNodeType("scene/property", LGraphLocatorProperty );

	//***********************************

	//*
	global.LGraphToggleValue = function()
	{
		this.addInput("target","Component");
		this.addInput("toggle",LiteGraph.ACTION);
		this.properties = { property_name: "enabled" };
	}

	LGraphToggleValue.title = "Toggle";
	LGraphToggleValue.desc = "Toggle a property value";

	LGraphToggleValue.prototype.getTitle = function()
	{
		return "Toggle: " + this.properties.property_name;
	}

	LGraphToggleValue.prototype.onAction = function( action_name, params ) { 

		var target = this.getInputData(0,true);
		if(!target)
			return;
		var prop_name = this.properties.property_name || "enabled";
		if( target[ prop_name ] !== undefined )
			target[ prop_name ] = !target[ prop_name ];
	}

	LiteGraph.registerNodeType("scene/toggle", LGraphToggleValue );
	//*/

	//************************************

	global.LGraphFrame = function LGraphFrame()
	{
		this.addOutput("Color","Texture");
		this.addOutput("Depth","Texture");
		this.addOutput("Extra","Texture");
		this.addOutput("Camera","Camera");
		this.properties = {};
	}

	LGraphFrame.title = "Frame";
	LGraphFrame.desc = "One frame rendered from the scene renderer";

	LGraphFrame.prototype.onExecute = function()
	{
		this.setOutputData(0, LGraphTexture.getTexture( this._color_texture ) );
		this.setOutputData(1, LGraphTexture.getTexture( this._depth_texture ) );
		this.setOutputData(2, LGraphTexture.getTexture( this._extra_texture ) );
		this.setOutputData(3, this._camera );
	}

	LGraphFrame.prototype.onDrawBackground = function( ctx )
	{
		if( this.flags.collapsed || this.size[1] <= 20 )
			return;

		if(!this._color_texture)
			return;

		if( !ctx.webgl )
			return; //is not working well

		//Different texture? then get it from the GPU
		if(this._last_preview_tex != this._last_tex || !this._last_preview_tex)
		{
			if( ctx.webgl && this._canvas && this._canvas.constructor === GL.Texture )
			{
				this._canvas = this._last_tex;
			}
			else
			{
				var texture = LGraphTexture.getTexture( this._color_texture );
				if(!texture)
					return;

				var tex_canvas = LGraphTexture.generateLowResTexturePreview( texture );
				if(!tex_canvas) 
					return;
				this._last_preview_tex = this._last_tex;
				this._canvas = cloneCanvas( tex_canvas );
			}
		}

		if(!this._canvas)
			return;

		//render to graph canvas
		ctx.save();
		if(!ctx.webgl) //reverse image
		{
			if( this._canvas.constructor === GL.Texture )
			{
				this._canvas = null;
				return;
			}

			ctx.translate( 0,this.size[1] );
			ctx.scale(1,-1);
		}
		ctx.drawImage( this._canvas, 0, 0, this.size[0], this.size[1] );
		ctx.restore();
	}

	LGraphFrame.prototype.onInspect = function( inspector )
	{
		var that = this;
		if(this.graph.component)
		{
			var render_context = this.graph.component.frame;
			inspector.showObjectFields( render_context );
			inspector.addSeparator();
			inspector.addCheckbox("Antialiasing", this.graph.component.use_antialiasing, function(v){ that.graph.component.use_antialiasing = v; });
		}
	}

	LiteGraph.registerNodeType("scene/frame", LGraphFrame );
};


///@FILE:../src/graph/components.js
///@INFO: GRAPHS
if(typeof(LiteGraph) != "undefined")
{
	//generic for all components that do not define their own node type
	global.LGraphComponent = function()
	{
		this.properties = {
			node_id: "",
			component_id: ""
		};

		//this.addInput("Component", undefined, { locked: true });

		this._component = null;
	}

	LGraphComponent.title = "Component";
	LGraphComponent.desc = "A component from a node";
	LGraphComponent.highlight_color = "#CCC";

	LGraphComponent.prototype.onRemoved = function()
	{
		this.bindComponentEvents(null); //remove binding		
	}

	LGraphComponent.prototype.onConnectionsChange = function( side )
	{
		this.bindComponentEvents( this._component );
	}

	LGraphComponent.prototype.onInit = function()
	{
		var compo = this.getComponent();
		if(!compo)
			return;
		this.processOutputs( compo );
	}

	LGraphComponent.prototype.processOutputs = function( compo )
	{
		if(!this.outputs || !this.outputs.length  )
			return;

		//write outputs
		for(var i = 0; i < this.outputs.length; i++)
		{
			var output = this.outputs[i];
			if( !output.links || !output.links.length || output.type == LiteGraph.EVENT )
				continue;
			if(output.name == "Component")
				this.setOutputData(i, compo );
			else
			{
				if(compo.getProperty)
				{
					var v = compo.getProperty(output.name);
					if(v !== undefined)
						this.setOutputData(i, v );
					else
						this.setOutputData(i, compo[ output.name ] );
				}
				else
					this.setOutputData(i, compo[ output.name ] );
			}
		}
	}


	LGraphComponent.prototype.onExecute = function()
	{
		var compo = this.getComponent();
		if(!compo)
			return;

		//read inputs (skip 1, is the component)
		if(this.inputs)
		for(var i = 0; i < this.inputs.length; i++)
		{
			var input = this.inputs[i];
			if( input.type === LiteGraph.ACTION )
				continue;
			var v = this.getInputData(i);
			if(v === undefined)
				continue;
			LS.setObjectProperty( compo, input.name, v );
		}

		//write outputs (done in a function so it can be reused by other methods)
		this.processOutputs( compo );
	}

	LGraphComponent.updateOutputData = function( slot )
	{
		if(!this.outputs || slot >= this.outputs.length  )
			return;

		var output = this.outputs[i];
		if( !output.links || !output.links.length || output.type == LiteGraph.EVENT )
			return;

		var compo = this.getComponent();
		if(!compo)
			return;

		if(output.name == "Component")
			this.setOutputData( slot, compo );
		else
			this.setOutputData( slot, compo[ output.name ] );
	}

	LGraphComponent.prototype.onDrawBackground = function(ctx)
	{
		var compo = this.getComponent();
		if(compo && compo._root)
		{
			this.boxcolor = null;
			var color = null;
			if(compo._root._is_selected)
				color = LGraphComponent.highlight_color;
			if(compo._is_selected)
				color = "#39F";

			if(color)
			{
				this.boxcolor = color;
				if(!this.flags.collapsed)
				{
					ctx.fillStyle = color;
					ctx.fillRect(0,0,this.size[0],2);
				}
			}

			this.title = LS.getClassName( compo.constructor );
		}
		else
			this.boxcolor = "red";
	}

	LGraphComponent.prototype.onConnectionsChange = function( type, slot, created, link_info, slot_info )
	{
		if(type == LiteGraph.INPUT && slot_info && slot_info.name == "Component" )
		{
			var node = this.getInputNode(slot);
			if(node && node.onExecute)
			{
				node.onExecute();
				this.setDirtyCanvas(true,true);
			}
		}
	}

	LGraphComponent.prototype.getComponent = function()
	{
		var scene = this.graph._scene || LS.GlobalScene;
		//TODO: if no graph found, then crawl up in the graph hierarchy because probalby it is a subgraph

		var node_id = this.properties.node_id;
		if(!node_id)
		{
			if( this.inputs && this.inputs.length )
			{
				var slot = this.findInputSlot("Component");
				if(slot != -1)
				{
					var component = this.getInputData(slot);
					return component ? component : null;
				}
			}

			return null;
		}

		//find node
		var node = scene.getNode( node_id );
		if(!node)
			return null;

		//find compo
		var compo_id = this.properties.component_id;
		var compo = null;
		if(compo_id.charAt(0) == "@")
			compo = node.getComponentByUId( compo_id );
		else if( LS.Components[ compo_id ] )
			compo = node.getComponent( LS.Components[ compo_id ] );
		else
			return null;

		if(compo && !compo.constructor.is_component)
			return null;

		if(this._component != compo)
			this.bindComponentEvents( compo );

		this._component = compo;
		return compo;
	}

	//bind events attached to this component
	LGraphComponent.prototype.bindComponentEvents = function( component )
	{
		if(this._component)
			LEvent.unbindAll( this._component, this );

		this._component = component;
		if( !this._component )
			return;
		
		//iterate outputs
		if(this.outputs && this.outputs.length)
			for(var i = 0; i < this.outputs.length; ++i )
			{
				var output = this.outputs[i];
				if( output.type !== LiteGraph.EVENT )
					continue;
				var event_name = output.name.substr(3);
				LEvent.bind( this._component, event_name, this.onComponentEvent, this );
			}
	}

	LGraphComponent.prototype.onComponentEvent = function ( e, params )
	{
		this.trigger( "on_" + e, params );
	}

	LGraphComponent.prototype.getComponentProperties = function( get_inputs, result )
	{
		var compo = this.getComponent();
		if(!compo)
			return null;

		var attrs = null;
		if(compo.getPropertiesInfo)
			attrs = compo.getPropertiesInfo( get_inputs );
		else
			attrs = LS.getObjectProperties( compo );

		result = result || [];
		for(var i in attrs)
			result.push( [i, attrs[i]] );

		if(compo.constructor.getExtraProperties)
			compo.constructor.getExtraProperties( result );

		if(compo.getExtraProperties)
			compo.getExtraProperties( result );

		return result;
	}

	LGraphComponent.prototype.onAction = function( action_name, params ) { 
		if(!action_name)
			return;
		var compo = this.getComponent();
		if(!compo)
			return;
		if(compo.onAction)
			compo.onAction( action_name, params );
		else if( compo[ action_name ] )
			compo[ action_name ](); //params will be mostly MouseEvent, so for now I wont pass it
	}

	//used by the LGraphSetValue node
	LGraphComponent.prototype.onSetValue = function( property_name, value ) { 
		var compo = this.getComponent();
		if(!compo)
			return;

		var current = compo[ property_name ];
		var final_value;

		if( current == null)
		{
			if(value && value.constructor === String)
				final_value = value;
		}
		else
		{
			switch( current.constructor )
			{
				case Number: final_value = Number( value ); break;
				case Boolean: final_value = (value == "true" || value == "1"); break;
				case String: final_value = String( value ); break;
				case Array:
				case Float32Array: 
					if( value != null )
					{
						if( value.constructor === String )
							final_value = JSON.parse("["+value+"]");
						else if( value.constructor === Number )
							final_value = [value];
						else
							final_value = value;
					}
					else
						final_value = value;
					break;
			}
		}

		if(final_value === undefined)
			return;

		if(compo.setPropertyValue)
			compo.setPropertyValue( property_name, final_value );
		else
			compo[ property_name ] = final_value;
	}

	LGraphComponent.prototype.onGetInputs = function()
	{ 
		var inputs = [["Node",0],["Component",0],null];

		this.getComponentProperties("input", inputs);

		var compo = this.getComponent();
		if(compo && compo.getActions)
		{
			var actions = compo.getActions({});
			if(actions)
			{
				if(actions.constructor === Array)
					for(var i = 0; i < actions.length; ++i)
						inputs.push( [ actions[i], LiteGraph.ACTION ] );
				else
					for(var i in actions)
						inputs.push( [i, LiteGraph.ACTION ] );
			}
		}

		return inputs;
	}

	LGraphComponent.prototype.onGetOutputs = function()
	{ 
		var outputs = [];
		outputs.push( ["Component", "Component" ], null ); //compo + separator

		this.getComponentProperties( "output", outputs);

		var compo = this.getComponent();
		if(compo && compo.getEvents)
		{
			var events = compo.getEvents();
			if(events)
			{
				if(events.constructor === Array)
					for(var i = 0; i < events.length; ++i)
						outputs.push( ["on_" + events[i], LiteGraph.EVENT ] );
				else
					for(var i in events)
						outputs.push( ["on_" + i, LiteGraph.EVENT ] );
			}
		}
		return outputs;
	}

	LGraphComponent.prototype.onInspect = function( inspector )
	{ 
		var that = this;

		var component = this.getComponent();
		if(component && component.onInspectNode )
			component.onInspectNode( inspector, this );

		inspector.addButton(null, "Inspect Component", function(){
			var compo = that.getComponent();
			if(!compo)
				return;
			EditorModule.inspect( compo );
		});
	}


	LiteGraph.registerNodeType("scene/component", LGraphComponent );
	
	//********************************************************

	/* LGraphNode representing an object in the Scene */

	global.LGraphTransform = function()
	{
		this.properties = {node_id:""};
		if(LGraphSceneNode._current_node_id)
			this.properties.node_id = LGraphSceneNode._current_node_id;
		this.addInput("Transform", "Transform", { locked: true });
		this.addOutput("Position","vec3");
	}

	LGraphTransform.title = "Transform";
	LGraphTransform.desc = "Transform info of a node";

	LGraphTransform.prototype.onExecute = function()
	{
		var transform = null;

		if(this.inputs && this.inputs[0])
			transform = this.getInputData(0);

		if(!transform)
		{
			var scene = this.graph.getScene ? this.graph.getScene() : LS.GlobalScene;

			var node = this._node;
			if(	this.properties.node_id )
			{
				node = scene.getNode( this.properties.node_id );
				if(!node)
					return;
			}

			if(!node)
				node = this.graph._scenenode;

			transform = node.transform;
		}

		if(!transform)
			return;

		//read inputs
		if(this.inputs)
		for(var i = 1; i < this.inputs.length; ++i)
		{
			var input = this.inputs[i];
			var v = this.getInputData(i);
			if(v === undefined)
				continue;
			switch( input.name )
			{
				case "x": transform.x = v; break;
				case "y": transform.y = v; break;
				case "z": transform.z = v; break;
				case "Position": transform.setPosition(v); break;
				case "Rotation": transform.setRotation(v); break;
				case "Scale": transform.setScale(v); break;
				case "Matrix": transform.fromMatrix(v); break;
				case "Mult.Matrix": transform.applyTransformMatrix(v); break;
				case "Translate": transform.translate(v); break;
				case "Translate Global": transform.translateGlobal(v); break;
				case "Rotate": quat.multiply( transform._rotation, transform._rotation, v ); transform._must_update = true; break;
				case "RotateX": transform.rotateX(v); break;
				case "RotateY": transform.rotateY(v); break;
				case "RotateZ": transform.rotateZ(v); break;
			}
		}

		//write outputs
		if(this.outputs)
		for(var i = 0; i < this.outputs.length; ++i)
		{
			var output = this.outputs[i];
			if(!output.links || !output.links.length)
				continue;

			var value = undefined;
			switch( output.name )
			{
				case "x": value = transform.x; break;
				case "y": value = transform.y; break;
				case "z": value = transform.z; break;
				case "Position": value = transform.position; break;
				case "Global Position": value = transform.getGlobalPosition(); break;
				case "Rotation": value = transform.rotation; break;
				case "Global Rotation": value = transform.getGlobalRotation(); break;
				case "Scale": value = transform.scaling; break;
				case "Matrix": value = transform.getMatrix(); break;
				default:
					break;
			}

			if(value !== undefined)
				this.setOutputData( i, value );
		}
	}

	LGraphTransform.prototype.onGetInputs = function()
	{
		return [["Position","vec3"],["Rotation","quat"],["Scale","number"],["x","number"],["y","number"],["z","number"],["Global Position","vec3"],["Global Rotation","quat"],["Matrix","mat4"],["Mult.Matrix","mat4"],["Translate","vec3"],["Translate Global","vec3"],["Rotate","quat"],["RotateX","number"],["RotateY","number"],["RotateZ","number"]];
	}

	LGraphTransform.prototype.onGetOutputs = function()
	{
		return [["Position","vec3"],["Rotation","quat"],["Scale","number"],["x","number"],["y","number"],["z","number"],["Global Position","vec3"],["Global Rotation","quat"],["Matrix","mat4"]];
	}

	LiteGraph.registerNodeType("scene/transform", LGraphTransform );

};

///@FILE:../src/graph/logic.js
///@INFO: GRAPHS
if(typeof(LiteGraph) != "undefined")
{
	//special kind of node
	global.LGraphSetValue = function LGraphSetValue()
	{
		this.properties = { property_name: "", value: "", type: "String" };
		this.addInput("on_set", LiteGraph.ACTION );
		this.addInput("value", "" );
		this.addOutput("on", LiteGraph.EVENT ); //to chain
		this.addOutput("node", 0 );
		this.mode = LiteGraph.ON_TRIGGER;
	}

	LGraphSetValue.prototype.onAction = function( action_name, params ) { 
		//is on_set

		if(!this.properties.property_name)
			return;

		//get the connected node
		var nodes = this.getOutputNodes(1);
		if(!nodes)
			return;

		var value = this.getInputOrProperty("value");

		//check for a setValue method, otherwise assign to property if exists one with that name
		for(var i = 0; i < nodes.length; ++i)
		{
			var node = nodes[i];
			//call it
			if(node.onSetValue)
				node.onSetValue( this.properties.property_name, value );
			else if(node.properties && node.properties.value !== undefined)
			{
				node.properties[ this.properties.property_name ] = value;
				if(node.onPropertyChanged)
					node.onPropertyChanged( this.properties.property_name, value );
			}
		}

		this.trigger("on");
	}

	LGraphSetValue.title = "SetValue";
	LGraphSetValue.desc = "sets a value to a node (could be a property)";

	LiteGraph.registerNodeType("logic/setValue", LGraphSetValue );
}




///@FILE:../src/graph/fx.js
///@INFO: GRAPHS
if(typeof(LiteGraph) != "undefined")
{

var litegraph_texture_found = false;
if(typeof(LGraphTexture) == "undefined")
	console.error("LiteGraph found but no LGraphTexture, this means LiteGL wasnt not included BEFORE litegraph. Be sure to include LiteGL before LiteGraph to ensure all functionalities.");
else
	litegraph_texture_found = true;



// Stack of FX *****************************************
function LGraphFXStack()
{
	this.addInput("Color","Texture");
	this.addInput("Depth","Texture");
	this.addInput("Intensity","number");
	this.addOutput("Final","Texture");
	this.properties = { intensity: 1, preserve_aspect: true };

	this._fx_stack = new LS.FXStack();
	this._fx_options = {};
}

LGraphFXStack.title = "FXStack";
LGraphFXStack.desc = "Apply FXs to Texture";

LGraphFXStack.prototype.onExecute = function()
{
	var tex = this.getInputData(0);
	if(!tex)
		return;

	if(!this.isOutputConnected(0))
		return; //saves work

	var temp = this._final_texture;

	if(!temp || temp.width != tex.width || temp.height != tex.height || temp.type != tex.type )
	{
		//we need two textures to do the blurring
		this._final_texture = new GL.Texture( tex.width, tex.height, { type: tex.type, format: gl.RGBA, filter: gl.LINEAR });
	}

	if( this.isInputConnected(1) )
		this._fx_options.depth_texture = this.getInputData(1);

	var intensity = this.properties.intensity;
	if( this.isInputConnected(2) )
	{
		intensity = this.getInputData(2);
		this.properties.intensity = intensity;
	}

	//blur sometimes needs an aspect correction
	var aspect = LiteGraph.camera_aspect;
	if(!aspect && window.gl !== undefined)
		aspect = gl.canvas.height / gl.canvas.width;
	if(!aspect)
		aspect = 1;
	aspect = this.properties.preserve_aspect ? aspect : 1;

	this._fx_stack.applyFX( tex, this._final_texture, this._fx_options );

	this.setOutputData(0, this._final_texture);
}

LGraphFXStack.prototype.onInspect = function( inspector )
{
	return this._fx_stack.inspect( inspector );
}

LGraphFXStack.prototype.getResources = function( resources )
{
	return this._fx_stack.getResources( resources );
}

LGraphFXStack.prototype.onSerialize = function( o )
{
	o.stack = this._fx_stack.serialize();
}

LGraphFXStack.prototype.onConfigure = function( o )
{
	if(o.stack)
		this._fx_stack.configure( o.stack );
}

LiteGraph.registerNodeType("texture/fxstack", LGraphFXStack );


function LGraphHistogram()
{
	this.addInput("in","Texture");
	this.addInput("enabled","Boolean");
	this.addOutput("out","Texture");
	this.properties = { enabled: true, scale: 0.5, position: [0,0], size: [1,1] };
}

LGraphHistogram.title = "Histogram";
LGraphHistogram.desc = "Overlaps an histogram";
LGraphHistogram.priority = 2; //render at the end

LGraphHistogram.prototype.onExecute = function()
{
	var tex = this.getInputData(0);

	if( !tex )
		return; //saves work

	var enabled = this.getInputData(1);
	if(enabled != null)
		this.properties.enabled = Boolean( enabled );

	if(this.properties.precision === LGraphTexture.PASS_THROUGH || this.properties.enabled === false )
	{
		this.setOutputData(0, tex);
		return;
	}

	if(!this._points_mesh)
	{
		var w = 512;
		var h = 256;
		var vertices = new Float32Array(w*h*3);
		for(var y = 0; y < h; ++y)
			for(var x = 0; x < w; ++x)
				vertices.set([x/w,y/h,0], y*w*3 + x*3);
		this._points_mesh = GL.Mesh.load({ vertices: vertices });
	}

	var histogram_bins = 256;

	if(!this._texture)
		this._texture = new GL.Texture(histogram_bins,1,{ type: gl.FLOAT, magFilter: gl.LINEAR, format: gl.RGB});

	if(!LGraphHistogram._shader)
		LGraphHistogram._shader = new GL.Shader( LGraphHistogram.vertex_shader, LGraphHistogram.pixel_shader );

	var mesh = this._points_mesh;
	var shader = LGraphHistogram._shader;
	var scale = this.properties.scale;
	shader.setUniform("u_texture",0);
	shader.setUniform("u_factor",1/512);
	tex.bind(0);

	gl.disable(gl.DEPTH_TEST);
	gl.enable(gl.BLEND);
	gl.blendFunc(gl.ONE,gl.ONE);

	//compute
	this._texture.drawTo( function(){
		gl.clearColor(0,0,0,1);
		gl.clear( gl.COLOR_BUFFER_BIT );
		for(var i = 0; i < 3; ++i)
		{
			gl.colorMask( i == 0, i == 1, i == 2, true );
			shader.setUniform("u_mask",LGraphHistogram.masks[i]);
			shader.draw( mesh, gl.POINTS );
		}
		gl.colorMask( true,true,true, true );
	});

	if(!this._line_mesh)
	{
		var vertices = new Float32Array(histogram_bins*3);
		for(var x = 0; x < histogram_bins; ++x)
			vertices.set([x/histogram_bins,0,0], x*3);
		this._line_mesh = GL.Mesh.load({ vertices: vertices });
	}

	if(!LGraphHistogram._line_shader)
		LGraphHistogram._line_shader = new GL.Shader( LGraphHistogram.line_vertex_shader, LGraphHistogram.line_pixel_shader );

	var mesh = this._line_mesh;
	var shader = LGraphHistogram._line_shader;
	shader.setUniform("u_texture",0);
	shader.setUniform("u_scale",scale);
	this._texture.bind(0);
	gl.disable(gl.BLEND);

	gl.viewport( this.properties.position[0] * gl.canvas.width, this.properties.position[1] * gl.canvas.height, 
				this.properties.size[0] * gl.canvas.width, this.properties.size[1] * gl.canvas.height );

	for(var i = 0; i < 3; ++i)
	{
		shader.setUniform("u_mask",LGraphHistogram.masks[i]);
		shader.draw( mesh, gl.LINE_STRIP );
	}

	this.setOutputData(0, this._texture );

	gl.viewport( 0,0, gl.canvas.width, gl.canvas.height );
}

LGraphHistogram.masks = [vec3.fromValues(1,0,0),vec3.fromValues(0,1,0),vec3.fromValues(0,0,1)];

LGraphHistogram.vertex_shader = "precision highp float;\n\
	\n\
	attribute vec3 a_vertex;\n\
	uniform sampler2D u_texture;\n\
	uniform vec3 u_mask;\n\
	\n\
	void main() {\n\
		vec3 color = texture2D( u_texture, a_vertex.xy ).xyz * u_mask;\n\
		float pos = min(1.0,length(color));\n\
		gl_Position = vec4(pos*2.0-1.0,0.5,0.0,1.0);\n\
		gl_PointSize = 1.0;\n\
	}\n\
";

LGraphHistogram.pixel_shader = "precision highp float;\n\
	uniform float u_factor;\n\
	void main() {\n\
		gl_FragColor = vec4(u_factor);\n\
	}\n\
";

LGraphHistogram.line_vertex_shader = "precision highp float;\n\
	\n\
	attribute vec3 a_vertex;\n\
	uniform sampler2D u_texture;\n\
	uniform vec3 u_mask;\n\
	uniform float u_scale;\n\
	\n\
	void main() {\n\
		vec3 color = texture2D( u_texture, a_vertex.xy ).xyz * u_mask;\n\
		float pos = length(color);\n\
		gl_Position = vec4(a_vertex.x*2.0-1.0, u_scale*pos*2.0-1.0,0.0,1.0);\n\
	}\n\
";

LGraphHistogram.line_pixel_shader = "precision highp float;\n\
	uniform vec3 u_mask;\n\
	void main() {\n\
		gl_FragColor = vec4(u_mask,1.0);\n\
	}\n\
";

LiteGraph.registerNodeType("texture/histogram", LGraphHistogram );


function LGraphCameraMotionBlur()
{
	this.addInput("color","Texture");
	this.addInput("depth","Texture");
	this.addInput("camera","Camera");
	this.addOutput("out","Texture");
	this.properties = { enabled: true, intensity: 1, ghosting_mitigation: true, ghosting_threshold: 0.4, freeze_camera: false, low_quality: false, precision: LGraphTexture.DEFAULT };

	this._inv_matrix = mat4.create();
	this._previous_viewprojection_matrix = mat4.create();

	this._uniforms = { 
		u_color_texture:0,
		u_depth_texture:1,
		u_inv_vp: this._inv_matrix,
		u_intensity: 1,
		u_camera_planes: null,
		u_viewprojection_matrix: null,
		u_previous_viewprojection_matrix: this._previous_viewprojection_matrix
	};
}

LGraphCameraMotionBlur.widgets_info = {
	"precision": { widget:"combo", values: litegraph_texture_found ? LGraphTexture.MODE_VALUES : [] }
};

LGraphCameraMotionBlur.title = "Camera Motion Blur";
LGraphCameraMotionBlur.desc = "A motion blur but only for camera movement";

LGraphCameraMotionBlur.prototype.onExecute = function()
{
	var tex = this.getInputData(0);
	var depth = this.getInputData(1);
	var camera = this.getInputData(2);

	if( !this.isOutputConnected(0) || !tex || !depth || !camera)
		return; //saves work

	var enabled = this.getInputData(3);
	if(enabled != null)
		this.properties.enabled = Boolean( enabled );

	if(this.properties.precision === LGraphTexture.PASS_THROUGH || this.properties.enabled === false )
	{
		this.setOutputData(0, tex);
		return;
	}

	var width = tex.width;
	var height = tex.height;
	var type = this.precision === LGraphTexture.LOW ? gl.UNSIGNED_BYTE : gl.HIGH_PRECISION_FORMAT;
	if (this.precision === LGraphTexture.DEFAULT)
		type = tex.type;
	if(!this._tex || this._tex.width != width || this._tex.height != height || this._tex.type != type )
		this._tex = new GL.Texture( width, height, { type: type, format: gl.RGBA, filter: gl.LINEAR });

	if( this.properties.low_quality )
	{
		if(!LGraphCameraMotionBlur._shader_low)
		{
			LGraphCameraMotionBlur._shader_low = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphCameraMotionBlur.pixel_shader, { SAMPLES:"4" } );
			LGraphCameraMotionBlur._shader_no_ghosting_low = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphCameraMotionBlur.pixel_shader, { GHOST_CORRECTION: "", SAMPLES:"4" } );
		}
	}
	else
	{
		if(!LGraphCameraMotionBlur._shader)
		{
			LGraphCameraMotionBlur._shader = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphCameraMotionBlur.pixel_shader );
			LGraphCameraMotionBlur._shader_no_ghosting = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphCameraMotionBlur.pixel_shader, { GHOST_CORRECTION: "" } );
		}
	}

	var shader = null;
	
	if( this.properties.low_quality )
		shader = this.properties.ghosting_mitigation ? LGraphCameraMotionBlur._shader_no_ghosting_low : LGraphCameraMotionBlur._shader_low;
	else
		shader = this.properties.ghosting_mitigation ? LGraphCameraMotionBlur._shader_no_ghosting : LGraphCameraMotionBlur._shader;

	var inv = this._inv_matrix;
	var vp = camera._viewprojection_matrix;
	var prev = this._previous_viewprojection_matrix;
	var optimize = false; //skip algorithm when camera hasnt moved
	var intensity = this.properties.intensity;

	var uniforms = this._uniforms;
	uniforms.u_intensity = intensity;
	uniforms.u_viewprojection_matrix =  camera._viewprojection_matrix;
	uniforms.u_camera_planes =  camera._uniforms.u_camera_planes;
	uniforms.u_ghosting_threshold = this.properties.ghosting_threshold || 0.4;

	var diff = 0;
	for(var i = 0; i < prev.length; ++i)
		diff += Math.abs( vp[i] - prev[i] );
	if(diff < 0.0001 && optimize) //no camera movement, skip process
	{
		tex.copyTo( this._tex );
	}
	else
	{
		mat4.invert( inv, camera._viewprojection_matrix );
		this._tex.drawTo(function() {
			gl.disable( gl.DEPTH_TEST );
			gl.disable( gl.CULL_FACE );
			gl.disable( gl.BLEND );
			tex.bind(0);
			depth.bind(1);
			var mesh = GL.Mesh.getScreenQuad();
			shader.uniforms( uniforms ).draw( mesh );
		});
	}

	if(!this.properties.freeze_camera)
		prev.set( camera._viewprojection_matrix );

	this.setOutputData( 0, this._tex );
}

LGraphCameraMotionBlur.prototype.onGetInputs = function()
{
	return [["enabled","boolean"]];
}

LGraphCameraMotionBlur.pixel_shader = "precision highp float;\n\
		\n\
		uniform sampler2D u_color_texture;\n\
		uniform sampler2D u_depth_texture;\n\
		varying vec2 v_coord;\n\
		uniform mat4 u_inv_vp;\n\
		uniform mat4 u_viewprojection_matrix;\n\
		uniform mat4 u_previous_viewprojection_matrix;\n\
		uniform vec2 u_camera_planes;\n\
		uniform float u_intensity;\n\
		uniform float u_ghosting_threshold;\n\
		#ifndef SAMPLES\n\
			#define SAMPLES 16\n\
		#endif\n\
		\n\
		void main() {\n\
			vec2 uv = v_coord;\n\
			float depth = texture2D(u_depth_texture, uv).x;\n\
			float zNear = u_camera_planes.x;\n\
			float zFar = u_camera_planes.y;\n\
			//float z = (2.0 * zNear) / (zFar + zNear - depth * (zFar - zNear));\n\
			depth = depth * 2.0 - 1.0;\n\
			float z = zNear * (depth + 1.0) / (zFar + zNear - depth * (zFar - zNear));\n\
			vec4 screenpos = vec4( uv * 2.0 - vec2(1.0), depth, 1.0 );\n\
			vec4 pos = u_inv_vp * screenpos;\n\
			pos /= pos.w;\n\
			vec4 old_screenpos = u_previous_viewprojection_matrix * pos;\n\
			old_screenpos /= old_screenpos.w;\n\
			vec2 uv_final = old_screenpos.xy * 0.5 + vec2(0.5);\n\
			vec2 uv_delta = (uv_final - uv);\n\
			uv -= uv_delta * 0.5;\n\
			//uv_delta *= 1.0 - z;\n\
			uv_delta *= u_intensity / float(SAMPLES);\n\
			vec4 color = vec4(0.0);\n\
			float total = 0.0;\n\
			float amount = 1.0;\n\
			for(int i = 0; i < SAMPLES; ++i)\n\
			{\n\
				#ifdef GHOST_CORRECTION\n\
					float old_depth = texture2D(u_depth_texture, uv).x;\n\
					old_depth = old_depth * 2.0 - 1.0;\n\
					float old_z = zNear * (old_depth + 1.0) / (zFar + zNear - old_depth * (zFar - zNear));\n\
					if( abs(old_z - z) > u_ghosting_threshold )\n\
					{\n\
						uv += uv_delta;\n\
						continue;\n\
					}\n\
				#endif\n\
				color += texture2D( u_color_texture, uv );\n\
				uv += uv_delta;\n\
				total += amount;\n\
			}\n\
			gl_FragColor = color / total;\n\
			//gl_FragColor = vec4( abs( old_screenpos.xy ), 1.0, 1.0 );\n\
			//gl_FragColor = texture2D( u_color_texture, uv_final );\n\
			//gl_FragColor = vec4( abs( pos.xyz * 0.001 ), 1.0 );\n\
			//gl_FragColor = vec4( vec3( abs(old_z - z) ), 1.0);\n\
		}\n\
		";

LiteGraph.registerNodeType("texture/motionBlur", LGraphCameraMotionBlur );


function LGraphVolumetricLight()
{
	this.addInput("color","Texture");
	this.addInput("depth","Texture");
	this.addInput("camera","Camera");
	this.addInput("light","Light,Component");
	this.addOutput("out","Texture");
	this.properties = { enabled: true, intensity: 1, attenuation: 2.0, precision: LGraphTexture.DEFAULT };

	this._inv_matrix = mat4.create();

	this._uniforms = { 
		u_color_texture:0,
		u_depth_texture:1,
		u_shadow_texture:2,
		u_noise_texture:3,
		u_intensity: 1,
		u_camera_planes: null,
		u_inv_vp: this._inv_matrix,
		u_rand: vec2.create()
	};
}

LGraphVolumetricLight.widgets_info = {
	"precision": { widget:"combo", values: LGraphTexture.MODE_VALUES }
};

LGraphVolumetricLight.title = "Volumetric Light";
LGraphVolumetricLight.desc = "Adds fog with volumetric light";

LGraphVolumetricLight.prototype.onExecute = function()
{
	var tex = this.getInputData(0);
	var depth = this.getInputData(1);
	var camera = this.getInputData(2);
	var light = this.getInputData(3);

	if( !this.isOutputConnected(0))
		return; //saves work

	var enabled = this.getInputData(4);
	if(enabled != null)
		this.properties.enabled = Boolean( enabled );

	var shadowmap = null;
	if(light && light._shadowmap)
		shadowmap = light._shadowmap._texture;

	if(this.properties.precision === LGraphTexture.PASS_THROUGH || this.properties.enabled === false || !depth || !camera || !light || !shadowmap )
	{
		this.setOutputData(0, tex);
		return;
	}

	var width = depth.width;
	var height = depth.height;
	var type = this.precision === LGraphTexture.LOW ? gl.UNSIGNED_BYTE : gl.HIGH_PRECISION_FORMAT;
	if (this.precision === LGraphTexture.DEFAULT)
		type = tex ? tex.type : gl.UNSIGNED_BYTE;
	if(!this._tex || this._tex.width != width || this._tex.height != height || this._tex.type != type )
		this._tex = new GL.Texture( width, height, { type: type, format: gl.RGBA, filter: gl.LINEAR });

	if(!LGraphVolumetricLight._shader_spot)
	{
		LGraphVolumetricLight._shader_spot = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphVolumetricLight.pixel_shader, { USE_SPOT:"" } );
		LGraphVolumetricLight._shader_directional = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphVolumetricLight.pixel_shader, { USE_DIRECTIONAL:"" } );
		//LGraphVolumetricLight._shader_omni = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphVolumetricLight.pixel_shader, { USE_OMNI:"" } );
	}

	var shader = null;

	switch( light.type )
	{
		case LS.Light.SPOT: shader = LGraphVolumetricLight._shader_spot; break;
		case LS.Light.DIRECTIONAL: shader = LGraphVolumetricLight._shader_directional; break;
		case LS.Light.OMNI: //shader = LGraphVolumetricLight._shader_omni;
			//not supported yet
			console.warn("volumetric light not supported for omni lights");
			this.properties.enabled = false;
			return;
			break;
		default:
			return;
	}

	var vp = camera._viewprojection_matrix;
	var intensity = this.properties.intensity;
	var inv = this._inv_matrix;
	mat4.invert( inv, camera._viewprojection_matrix );

	var uniforms = this._uniforms;
	uniforms.u_intensity = intensity;
	uniforms.u_camera_planes = camera._uniforms.u_camera_planes;
	uniforms.u_shadow_params = light._uniforms.u_shadow_params;
	uniforms.u_attenuation = this.properties.attenuation;
	uniforms.u_rand[1] = uniforms.u_rand[0] = 0;

	var light_uniforms = light._uniforms;

	if(!tex)
		tex = LS.Renderer._black_texture;
	var noise_texture = LGraphTexture.getNoiseTexture();

	this._tex.drawTo(function() {
		gl.disable( gl.DEPTH_TEST );
		gl.disable( gl.CULL_FACE );
		gl.disable( gl.BLEND );

		if(!light.enabled)
			tex.toViewport();
		else
		{
			tex.bind(0);
			depth.bind(1);
			shadowmap.bind(2);
			noise_texture.bind(3);
			var mesh = GL.Mesh.getScreenQuad();
			shader.uniforms( light_uniforms );
			shader.uniforms( uniforms ).draw( mesh );
		}
	});

	this.setOutputData( 0, this._tex );
}

LGraphVolumetricLight.prototype.onGetInputs = function()
{
	return [["enabled","boolean"]];
}

/* from http://www.alexandre-pestana.com/volumetric-lights/
// Mie scaterring approximated with Henyey-Greenstein phase function.

accumFog += ComputeScattering(dot(rayDirection, sunDirection)).xxx * g_SunColor;
*/

LGraphVolumetricLight.pixel_shader = "precision highp float;\n\
		\n\
		uniform sampler2D u_color_texture;\n\
		uniform sampler2D u_depth_texture;\n\
		uniform sampler2D u_shadow_texture;\n\
		uniform sampler2D u_noise_texture;\n\
		varying vec2 v_coord;\n\
		uniform mat4 u_inv_vp;\n\
		uniform mat4 u_light_viewprojection_matrix;\n\
		uniform vec2 u_camera_planes;\n\
		uniform vec4 u_shadow_params;\n\
		uniform vec4 u_light_info;\n\
		uniform vec3 u_light_front;\n\
		uniform vec3 u_light_color;\n\
		uniform mat4 u_light_matrix;\n\
		uniform float u_intensity;\n\
		uniform float u_attenuation;\n\
		uniform vec2 u_rand;\n\
		const float G_SCATTERING = 0.5;\n\
		const float PI = 3.14159265358979323846;\n\
		float ComputeScattering(float lightDotView)\n\
		{\n\
			float result = 1.0 - G_SCATTERING * G_SCATTERING;\n\
			result /= (4.0 * PI * pow(1.0 + G_SCATTERING * G_SCATTERING - (2.0 * G_SCATTERING) * lightDotView, 1.5));\n\
			return result;\n\
		}\n\
		#define SAMPLES 64\n\
		\n\
		void main() {\n\
			vec2 uv = v_coord;\n\
			vec4 color = texture2D(u_color_texture, uv);\n\
			float depth = texture2D(u_depth_texture, uv).x;\n\
			float zNear = u_camera_planes.x;\n\
			float zFar = u_camera_planes.y;\n\
			depth = depth * 2.0 - 1.0;\n\
			float z = zNear * (depth + 1.0) / (zFar + zNear - depth * (zFar - zNear));\n\
			vec4 screenpos = vec4( uv * 2.0 - vec2(1.0), depth, 1.0 );\n\
			vec4 farpos = u_inv_vp * screenpos;\n\
			farpos /= farpos.w;\n\
			screenpos.z = 0.0;\n\
			vec4 nearpos = u_inv_vp * screenpos;\n\
			nearpos.xyz /= nearpos.w;\n\
			vec3 rayDir = (farpos.xyz - nearpos.xyz) / float(SAMPLES);\n\
			float weight = length(rayDir);\n\
			vec4 current_pos = vec4( nearpos.xyz, 1.0 );\n\
			current_pos.xyz += rayDir * texture2D(u_noise_texture, uv * 2.0 + u_rand ).x;\n\
			float brightness = 0.0;\n\
			float bias = u_shadow_params.y;\n\
			float accum = ComputeScattering( dot( normalize(rayDir), -u_light_front));\n\
			for(int i = 0; i < SAMPLES; ++i)\n\
			{\n\
				vec4 light_uv = u_light_matrix * current_pos;\n\
				light_uv.xy /= light_uv.w;\n\
				light_uv.xy = light_uv.xy * 0.5 + vec2(0.5);\n\
				float shadow_depth = texture2D( u_shadow_texture, light_uv.xy ).x;\n\
				if (((light_uv.z - bias) / light_uv.w * 0.5 + 0.5) < shadow_depth )\n\
					brightness += accum * weight;\n\
				current_pos.xyz += rayDir;\n\
			}\n\
			//color.xyz += u_intensity * u_light_color * brightness / float(SAMPLES);\n\
			color.xyz = mix(color.xyz, u_light_color, clamp( pow(u_intensity * brightness / float(SAMPLES), u_attenuation),0.0,1.0));\n\
			gl_FragColor = color;\n\
		}\n\
		";

LiteGraph.registerNodeType("texture/volumetric_light", LGraphVolumetricLight );

if( LiteGraph.Nodes.LGraphTextureCanvas2D )
{

	function LGraphTextureCanvas2DFromScript() {
        this.addInput("v");
		this.addOutput("out", "Texture");
		this.properties = {
			filename: "",
			width: 512,
			height: 512,
			clear: true,
			precision: LGraphTexture.DEFAULT,
			use_html_canvas: false
		};
		this._func = null;
		this._temp_texture = null;
		this.size = [180,30];
	}

	LGraphTextureCanvas2DFromScript.title = "Canvas2DFromScript";
	LGraphTextureCanvas2DFromScript.desc = "Executes Canvas2D script file inside a texture or the viewport.";
	LGraphTextureCanvas2DFromScript.help = "Set width and height to 0 to match viewport size.";

	LGraphTextureCanvas2DFromScript.widgets_info = {
		precision: { widget: "combo", values: LGraphTexture.MODE_VALUES },
		filename: { type: "script" },
		width: { type: "Number", precision: 0, step: 1 },
		height: { type: "Number", precision: 0, step: 1 }
	};

	LGraphTextureCanvas2DFromScript.prototype.onPropertyChanged = function(	name, value ) {
		var that = this;
		if (name == "filename" && LiteGraph.allow_scripts) {
			this._func = null;
			if(!value)
				return;
			LS.ResourcesManager.load( value, function(script_resource){
				that.compileCode(script_resource.data);
				that._code_version = script_resource._version || 0;
			});
		}
	}

	LGraphTextureCanvas2DFromScript.prototype.onExecute = function() {

		if (!this.isOutputConnected(0))
			return;

		var script_resource = LS.ResourcesManager.getResource( this.properties.filename );
		if( script_resource && script_resource._version != this._code_version )
		{
			this.compileCode( script_resource.data );
			this._code_version = script_resource._version || 0;
		}

		var func = this._func;
		if (!func)
			return;
		this.executeDraw( func );
	}

	LGraphTextureCanvas2DFromScript.prototype.getResources = function(res)
	{
		if(this.properties.filename)
			res[this.properties.filename] = true;
	}


	LGraphTextureCanvas2DFromScript.prototype.compileCode = LiteGraph.Nodes.LGraphTextureCanvas2D.prototype.compileCode;

	LGraphTextureCanvas2DFromScript.prototype.compileCode = LiteGraph.Nodes.LGraphTextureCanvas2D.prototype.compileCode;
	LGraphTextureCanvas2DFromScript.prototype.executeDraw = LiteGraph.Nodes.LGraphTextureCanvas2D.prototype.executeDraw;

	LiteGraph.registerNodeType("texture/canvas2DfromScript", LGraphTextureCanvas2DFromScript);
}

function LGraphReprojectDepth()
{
	this.addInput("color","Texture");
	this.addInput("depth","Texture");
	this.addInput("camera","Camera,Component");
	this.properties = { enabled: true, pointSize: 1, offset: 0, factor: 1, triangles: false, depth_is_linear: false, skip_center: false, layers:0xFF };

	this._view_matrix = mat4.create();
	this._projection_matrix = mat4.create();
	this._viewprojection_matrix = mat4.create();

	this._uniforms = { 
		u_depth_ivp: mat4.create(),
		u_point_size: 1,
		u_depth_offset: 0,
		u_ires: vec2.create(),
		u_near_far: vec2.create(),
		u_color_texture:0,
		u_depth_texture:1,
		u_factor: this.properties.factor,
		u_vp: this._viewprojection_matrix
	};
}

LGraphReprojectDepth.widgets_info = {
	"layers": {widget:"layers"}
};

LGraphReprojectDepth.title = "Reproj.Depth";
LGraphReprojectDepth.desc = "Reproject Depth";

LGraphReprojectDepth.prototype.onGetInputs = function()
{
	return [["enabled","boolean"],["factor","number"]];
}


LGraphReprojectDepth.prototype.onExecute = function()
{
	var color = this.getInputData(0);
	var depth = this.getInputData(1);
	var camera = this.getInputData(2);

	if( !depth || !camera || camera.constructor !== LS.Camera )
		return;

	var enabled = this.getInputOrProperty("enabled");
	if(!enabled)
		return;

	var no_color = false;
	if(!color)
	{
		color = GL.Texture.getWhiteTexture();
		no_color = true;
	}

	if(!LiteGraph.LGraphRender.onRequestCameraMatrices)
	{
		console.warn("cannot render geometry, LiteGraph.onRequestCameraMatrices is null, remember to fill this with a callback(view_matrix, projection_matrix,viewprojection_matrix) to use 3D rendering from the graph");
		return;
	}

	LiteGraph.LGraphRender.onRequestCameraMatrices( this._view_matrix, this._projection_matrix,this._viewprojection_matrix );
	//var current_camera = LS.Renderer.getCurrentCamera();
	if(!(LS.Renderer._current_layers_filter & this.properties.layers ))
		return;

	var mesh = this._mesh;
	if(!mesh)
		mesh = this._mesh = GL.Mesh.plane({detailX: depth.width, detailY: depth.height});
	var shader = null;
	if(!LGraphReprojectDepth._shader)
		LGraphReprojectDepth._shader = new GL.Shader( LGraphReprojectDepth.vertex_shader, LGraphReprojectDepth.pixel_shader );
	shader = LGraphReprojectDepth._shader;
	var uniforms = this._uniforms;
	uniforms.u_point_size = this.properties.pointSize;
	uniforms.u_depth_offset = this.properties.offset;
	uniforms.u_is_linear = this.properties.depth_is_linear ? 1 : 0;
	uniforms.u_use_depth_as_color = no_color ? 1 : 0;
	uniforms.u_near_far.set(depth.near_far_planes ? depth.near_far_planes : [0,1]);
	uniforms.u_factor = this.getInputOrProperty("factor");

	if(this.properties.skip_center)
		uniforms.u_ires.set([0,0]);
	else
		uniforms.u_ires.set([1/depth.width,1/depth.height]);
	mat4.invert( uniforms.u_depth_ivp, camera._viewprojection_matrix );
	gl.enable( gl.DEPTH_TEST );
	gl.disable( gl.BLEND );
	color.bind(0);
	gl.texParameteri( color.texture_type, gl.TEXTURE_MAG_FILTER, this.properties.skip_center ? gl.LINEAR : gl.NEAREST );
	depth.bind(1);
	gl.texParameteri( depth.texture_type, gl.TEXTURE_MAG_FILTER, this.properties.skip_center ? gl.LINEAR : gl.NEAREST );
	shader.uniforms( uniforms ).draw( mesh, this.properties.triangles ? gl.TRIANGLES : gl.POINTS );
}


LGraphReprojectDepth.vertex_shader = "\n\
	\n\
	precision highp float;\n\
	attribute vec3 a_vertex;\n\
	attribute vec2 a_coord;\n\
	uniform sampler2D u_color_texture;\n\
	uniform sampler2D u_depth_texture;\n\
	uniform mat4 u_depth_ivp;\n\
	uniform mat4 u_vp;\n\
	uniform vec2 u_ires;\n\
	uniform float u_depth_offset;\n\
	uniform float u_point_size;\n\
	uniform float u_factor;\n\
	uniform vec2 u_near_far;\n\
	uniform int u_is_linear;\n\
	uniform int u_use_depth_as_color;\n\
	varying vec4 color;\n\
	\n\
	void main() {\n\
		vec2 uv = a_coord + u_ires * 0.5;\n\
		float depth = texture2D( u_depth_texture, uv ).x * u_factor;\n\
		if(u_is_linear == 1)\n\
		{\n\
			//must delinearize, doesnt work...\n\
			//lz = u_near_far.x * (depth + 1.0) / (u_near_far.y + u_near_far.x - depth * (u_near_far.y - u_near_far.x));\n\
			//depth = depth * 0.5 + 0.5;\n\
			//depth = depth * 2.0 - 1.0;\n\
			depth = ((u_near_far.x + u_near_far.y) * depth + u_near_far.x) / (1.0 + u_near_far.y - u_near_far.x);\n\
			//depth = depth * 0.5 + 0.5;\n\
			//depth = depth * 2.0 - 1.0;\n\
		}\n\
		else\n\
			depth = depth * 2.0 - 1.0;\n\
		if(u_use_depth_as_color == 1)\n\
			color = vec4( depth * 0.5 + 0.5);\n\
		else\n\
			color = texture2D( u_color_texture, uv );\n\
		vec4 pos2d = vec4( uv*2.0-vec2(1.0),depth + u_depth_offset,1.0);\n\
		vec4 pos3d = u_depth_ivp * pos2d;\n\
		//if(u_must_linearize == 0)\n\
		//	pos3d /= pos3d.w;\n\
		gl_Position = u_vp * pos3d;\n\
		gl_PointSize = u_point_size;\n\
	}\n\
";

LGraphReprojectDepth.pixel_shader = "\n\
precision mediump float;\n\
varying vec4 color;\n\
void main()\n\
{\n\
	gl_FragColor = color;\n\
}\n\
";

LiteGraph.registerNodeType("texture/reproj.depth", LGraphReprojectDepth );


//*********************

function LGraphSSAO()
{
	this.addInput("depth","Texture");
	this.addInput("camera","Camera");
	this.addOutput("out","Texture");
	this.properties = { enabled: true, intensity: 1, radius: 5, precision: LGraphTexture.DEFAULT };

	this._uniforms = { 
		tDepth:0,
		samples: 64,
		u_resolution: vec2.create(),
		radius: 5,
		zNear: 0.1,
		zFar: 1000
	};
}

LGraphSSAO.widgets_info = {
	"precision": { widget:"combo", values: litegraph_texture_found ? LGraphTexture.MODE_VALUES : [] }
};

LGraphSSAO.title = "SSAO";
LGraphSSAO.desc = "Screen Space Ambient Occlusion";

LGraphSSAO.prototype.onExecute = function()
{
	var depth = this.getInputData(0);
	var camera = this.getInputData(1);

	if( !this.isOutputConnected(0) || !depth || !camera)
		return; //saves work

	var enabled = this.getInputData(2);
	if(enabled != null)
		this.properties.enabled = Boolean( enabled );

	if(this.properties.precision === LGraphTexture.PASS_THROUGH || this.properties.enabled === false )
	{
		this.setOutputData(0, GL.Texture.getWhiteTexture() );
		return;
	}

	var width = depth.width;
	var height = depth.height;
	var type = this.precision === LGraphTexture.LOW ? gl.UNSIGNED_BYTE : gl.HIGH_PRECISION_FORMAT;
	if (this.precision === LGraphTexture.DEFAULT)
		type = gl.UNSIGNED_BYTE;
	if(!this._tex || this._tex.width != width || this._tex.height != height || this._tex.type != type )
		this._tex = new GL.Texture( width, height, { type: type, format: gl.RGBA, filter: gl.LINEAR });

	var shader = null;
	if(!LGraphSSAO._shader)
		LGraphSSAO._shader = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphSSAO.pixel_shader );
	shader = LGraphSSAO._shader;

	var intensity = this.properties.intensity;

	var uniforms = this._uniforms;
	uniforms.u_resolution.set([width,height]);
	uniforms.zNear = camera.near;
	uniforms.zFar = camera.far;
	uniforms.radius = this.properties.radius;
	uniforms.strength = intensity;

	this._tex.drawTo(function() {
		gl.disable( gl.DEPTH_TEST );
		gl.disable( gl.CULL_FACE );
		gl.disable( gl.BLEND );
		depth.bind(0);
		var mesh = GL.Mesh.getScreenQuad();
		shader.uniforms( uniforms ).draw( mesh );
	});

	this.setOutputData( 0, this._tex );
}

//from https://github.com/spite/Wagner/blob/master/fragment-shaders/ssao-simple-fs.glsl
LGraphSSAO.pixel_shader = "\n\
precision mediump float;\n\
varying vec2 v_coord;\n\
uniform sampler2D tDepth;\n\
uniform vec2 u_resolution;\n\
uniform float zNear;\n\
uniform float zFar;\n\
uniform float strength;\n\
#define PI    3.14159265\n\
\n\
float width; //texture width\n\
float height; //texture height\n\
\n\
vec2 texCoord;\n\
\n\
//general stuff\n\
\n\
//user variables\n\
uniform int samples; //ao sample count //64.0\n\
uniform float radius; //ao radius //5.0\n\
\n\
float aoclamp = 0.125; //depth clamp - reduces haloing at screen edges\n\
bool noise = true; //use noise instead of pattern for sample dithering\n\
float noiseamount = 0.0002; //dithering amount\n\
\n\
float diffarea = 0.3; //self-shadowing reduction\n\
float gdisplace = 0.4; //gauss bell center //0.4\n\
\n\
bool mist = false; //use mist?\n\
float miststart = 0.0; //mist start\n\
float mistend = zFar; //mist end\n\
bool onlyAO = false; //use only ambient occlusion pass?\n\
float lumInfluence = 0.7; //how much luminance affects occlusion\n\
\n\
vec2 rand(vec2 coord) //generating noise/pattern texture for dithering\n\
{\n\
  float noiseX = ((fract(1.0-coord.s*(width/2.0))*0.25)+(fract(coord.t*(height/2.0))*0.75))*2.0-1.0;\n\
  float noiseY = ((fract(1.0-coord.s*(width/2.0))*0.75)+(fract(coord.t*(height/2.0))*0.25))*2.0-1.0;\n\
  if (noise)\n\
  {\n\
    noiseX = clamp(fract(sin(dot(coord ,vec2(12.9898,78.233))) * 43758.5453),0.0,1.0)*2.0-1.0;\n\
    noiseY = clamp(fract(sin(dot(coord ,vec2(12.9898,78.233)*2.0)) * 43758.5453),0.0,1.0)*2.0-1.0;\n\
  }\n\
  return vec2(noiseX,noiseY)*noiseamount;\n\
}\n\
\n\
float doMist()\n\
{\n\
  float zdepth = texture2D(tDepth,texCoord.xy).x;\n\
  float depth = -zFar * zNear / (zdepth * (zFar - zNear) - zFar);\n\
  return clamp((depth-miststart)/mistend,0.0,1.0);\n\
}\n\
\n\
float readDepth(vec2 coord)\n\
{\n\
  if (v_coord.x<0.0||v_coord.y<0.0) return 1.0;\n\
  else {\n\
    float z_b = texture2D(tDepth, coord ).x;\n\
    float z_n = 2.0 * z_b - 1.0;\n\
    return (2.0 * zNear) / (zFar + zNear - z_n * (zFar-zNear));\n\
  }\n\
}\n\
\n\
int compareDepthsFar(float depth1, float depth2) {\n\
  float garea = 2.0; //gauss bell width\n\
  float diff = (depth1 - depth2)*100.0; //depth difference (0-100)\n\
  //reduce left bell width to avoid self-shadowing\n\
  return diff<gdisplace ? 0 : 1;\n\
}\n\
\n\
float compareDepths(float depth1, float depth2)\n\
{\n\
  float garea = 2.0; //gauss bell width\n\
  float diff = (depth1 - depth2)*100.0; //depth difference (0-100)\n\
  //reduce left bell width to avoid self-shadowing\n\
  if (diff<gdisplace)\n\
  {\n\
    garea = diffarea;\n\
  }\n\
	\n\
  float gauss = pow(2.7182,-2.0*(diff-gdisplace)*(diff-gdisplace)/(garea*garea));\n\
  return gauss;\n\
}\n\
\n\
float calAO(float depth,float dw, float dh)\n\
{\n\
  float dd = (1.0-depth)*radius;\n\
\n\
  float temp = 0.0;\n\
  float temp2 = 0.0;\n\
  float coordw = v_coord.x + dw*dd;\n\
  float coordh = v_coord.y + dh*dd;\n\
  float coordw2 = v_coord.x - dw*dd;\n\
  float coordh2 = v_coord.y - dh*dd;\n\
\n\
  vec2 coord = vec2(coordw , coordh);\n\
  vec2 coord2 = vec2(coordw2, coordh2);\n\
\n\
  float cd = readDepth(coord);\n\
  int far = compareDepthsFar(depth, cd);\n\
  temp = compareDepths(depth, cd);\n\
  //DEPTH EXTRAPOLATION:\n\
  if (far > 0)\n\
  {\n\
    temp2 = compareDepths(readDepth(coord2),depth);\n\
    temp += (1.0-temp)*temp2;\n\
  }\n\
\n\
  return temp;\n\
}\n\
\n\
void main(void)\n\
{\n\
	width = u_resolution.x; //texture width\n\
	height = u_resolution.y; //texture height\n\
	texCoord = v_coord;\n\
	vec2 noise = rand(texCoord);\n\
	float depth = readDepth(texCoord);\n\
	\n\
	float w = (1.0 / width)/clamp(depth,aoclamp,1.0)+(noise.x*(1.0-noise.x));\n\
	float h = (1.0 / height)/clamp(depth,aoclamp,1.0)+(noise.y*(1.0-noise.y));\n\
	\n\
	float pw = 0.0;\n\
	float ph = 0.0;\n\
	\n\
	float ao = 0.0;\n\
	\n\
	float dl = PI * (3.0 - sqrt(5.0));\n\
	float dz = 1.0 / float(samples);\n\
	float l = 0.0;\n\
	float z = 1.0 - dz/2.0;\n\
	\n\
	for (int i = 0; i < 64; i++)\n\
	{\n\
		if (i > samples) break;\n\
		float r = sqrt(1.0 - z);\n\
		\n\
		pw = cos(l) * r;\n\
		ph = sin(l) * r;\n\
		ao += calAO(depth,pw*w,ph*h);\n\
		z = z - dz;\n\
		l = l + dl;\n\
	}\n\
\n\
\n\
  ao /= float(samples);\n\
  ao *= strength;\n\
  ao = 1.0-ao;\n\
\n\
  if (mist)\n\
  {\n\
    ao = mix(ao, 1.0, doMist());\n\
  }\n\
\n\
	vec3 final = vec3(ao); //ambient occlusion only\n\
  gl_FragColor = vec4(final,1.0);\n\
}\n\
";

LiteGraph.registerNodeType("texture/SSAO", LGraphSSAO );


/*
function LGraphDepthAwareUpscale()
{
	this.addInput("tex","Texture");
	this.addInput("lowdepth","Texture");
	this.addInput("highdepth","Texture");
	this.addInput("camera","Camera");
	this.addOutput("out","Texture");
	this.properties = { enabled: true, precision: LGraphTexture.DEFAULT };

	this._uniforms = { 
		u_color_texture:0,
		u_lowdepth_texture:1,
		u_highdepth_texture:2,
		u_lowsize: vec4.create(),
		u_highsize: vec4.create(),
		u_viewprojection: null
	};
}

LGraphDepthAwareUpscale.widgets_info = {
	"precision": { widget:"combo", values: LGraphTexture.MODE_VALUES }
};

LGraphDepthAwareUpscale.title = "DepthAware Upscale";
LGraphDepthAwareUpscale.desc = "Upscales a texture";

LGraphDepthAwareUpscale.prototype.onExecute = function()
{
	var tex = this.getInputData(0);
	var lowdepth = this.getInputData(1);
	var highdepth = this.getInputData(2);
	var camera = this.getInputData(3);

	if( !this.isOutputConnected(0) || !tex )
		return; //saves work

	var enabled = this.getInputData(4);
	if(enabled != null)
		this.properties.enabled = Boolean( enabled );

	if(this.properties.precision === LGraphTexture.PASS_THROUGH || this.properties.enabled === false || !lowdepth || !highdepth || !camera )
	{
		this.setOutputData(0, tex);
		return;
	}

	var width = tex.width;
	var height = tex.height;
	var type = this.precision === LGraphTexture.LOW ? gl.UNSIGNED_BYTE : gl.HIGH_PRECISION_FORMAT;
	if (this.precision === LGraphTexture.DEFAULT)
		type = tex ? tex.type : gl.UNSIGNED_BYTE;
	if(!this._tex || this._tex.width != width || this._tex.height != height || this._tex.type != type || this._tex.format != tex.format )
		this._tex = new GL.Texture( width, height, { type: type, format: tex.format, filter: gl.LINEAR });

	if(!LGraphDepthAwareUpscale._shader)
		LGraphDepthAwareUpscale._shader = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, LGraphDepthAwareUpscale.pixel_shader );

	var shader = null;

	var uniforms = this._uniforms;
	uniforms.u_lowsize[0] = lowdepth.width; uniforms.u_lowsize[1] = lowdepth.height; 	uniforms.u_lowsize[2] = 1/lowdepth.width; uniforms.u_lowsize[3] = 1/lowdepth.height;
	uniforms.u_highsize[0] = highdepth.width; uniforms.u_highsize[1] = highdepth.height; uniforms.u_highsize[2] = 1/highdepth.width; uniforms.u_highsize[3] = 1/highdepth.height;
	uniforms.u_viewprojection = camera._viewprojection_matrix;

	this._tex.drawTo(function() {
		gl.disable( gl.DEPTH_TEST );
		gl.disable( gl.CULL_FACE );
		gl.disable( gl.BLEND );
		tex.bind(0);
		lowdepth.bind(1);
		highdepth.bind(2);
		var mesh = GL.Mesh.getScreenQuad();
		shader.uniforms( uniforms ).draw( mesh );
	});

	this.setOutputData( 0, this._tex );
}

LGraphDepthAwareUpscale.pixel_shader = "precision highp float;\n\
		\n\
		uniform sampler2D u_color_texture;\n\
		uniform sampler2D u_lowdepth_texture;\n\
		uniform sampler2D u_highdepth_texture;\n\
		varying vec2 v_coord;\n\
		uniform vec4 u_lowsize;\n\
		uniform vec4 u_highsize;\n\
		uniform mat4 u_viewprojection;\n\
		\n\
		void main() {\n\
			vec2 uv = v_coord;\n\
			vec2 low_ltuv = floor(v_coord * u_lowsize.xy) * u_lowsize.zw;\n\
			vec2 low_rbuv = ceil(v_coord * u_lowsize.xy) * u_lowsize.zw;\n\
			vec2 high_ltuv = floor(v_coord * u_highsize.xy) * u_highsize.zw;\n\
			vec2 high_rbuv = ceil(v_coord * u_highsize.xy) * u_highsize.zw;\n\
			vec4 color = texture2D(u_color_texture, uv);\n\
			float lowdepth = texture2D(u_lowdepth_texture, uv).x;\n\
			float highdepth = texture2D(u_highdepth_texture, uv).x;\n\
			color.xyz = mix(color.xyz, u_light_color, clamp( pow(u_intensity * brightness / float(SAMPLES), u_attenuation),0.0,1.0));\n\
			gl_FragColor = color;\n\
		}\n\
		";

LiteGraph.registerNodeType("texture/depthaware_upscale", LGraphDepthAwareUpscale );
*/


}
///@FILE:../src/graph/actions.js
//not used yet
//meant for triggers
///@FILE:../src/graph/helpers.js
///@INFO: GRAPHS
if(typeof(LiteGraph) != "undefined")
{
	LiteGraph.CORNER_TOP_LEFT = 0;
	LiteGraph.CORNER_TOP_RIGHT = 1;
	LiteGraph.CORNER_BOTTOM_LEFT = 2;
	LiteGraph.CORNER_BOTTOM_RIGHT = 3;
	LiteGraph.CORNER_TOP_CENTER = 4;
	LiteGraph.CORNER_BOTTOM_CENTER = 5;

	var corner_options = { type:"enum", values:{ 
		"top-left": LiteGraph.CORNER_TOP_LEFT, 
		"top-right": LiteGraph.CORNER_TOP_RIGHT,
		"bottom-left": LiteGraph.CORNER_BOTTOM_LEFT,
		"bottom-right": LiteGraph.CORNER_BOTTOM_RIGHT,
		"top-center": LiteGraph.CORNER_TOP_CENTER,
		"bottom-center": LiteGraph.CORNER_BOTTOM_CENTER
	}};

	function positionToArea( position, corner, area )
	{
		var x = position[0];
		var y = position[1];

		switch( corner )
		{
			case LiteGraph.CORNER_TOP_RIGHT: x = gl.canvas.width - x; break;
			case LiteGraph.CORNER_BOTTOM_LEFT: y = gl.canvas.height - y; break;
			case LiteGraph.CORNER_BOTTOM_RIGHT: x = gl.canvas.width - x; y = gl.canvas.height - y; break;
			case LiteGraph.CORNER_TOP_CENTER: x = gl.canvas.width * 0.5; break;
			case LiteGraph.CORNER_BOTTOM_CENTER: x = gl.canvas.width * 0.5; y = gl.canvas.height - y; break;
			case LiteGraph.CORNER_TOP_LEFT:
			default:
		}

		area[0] = x;
		area[1] = y;
	}


	function LGraphInputMouse()
	{
		this.addOutput("pos","vec2");
		this.addOutput("left_button","boolean");
		this.addOutput("right_button","boolean");
		this.properties = {};
	}

	LGraphInputMouse.title = "Mouse";
	LGraphInputMouse.desc = "Mouse state info";

	LGraphInputMouse.prototype.onExecute = function()
	{
		this.setOutputData(0, LS.Input.Mouse.position );
		this.setOutputData(1, LS.Input.Mouse.buttons & LS.Input.BUTTONS_LEFT );
		this.setOutputData(2, LS.Input.Mouse.buttons & LS.Input.BUTTONS_RIGHT );
	}

	LiteGraph.registerNodeType("input/mouse", LGraphInputMouse );

	//special kind of node
	function LGraphGUIPanel()
	{
		this.addOutput("pos","vec2");
		this.addOutput("enabled","boolean");
		this.properties = { enabled: true, draggable: false, title: "", color: [0.1,0.1,0.1], opacity: 0.7, titlecolor: [0,0,0], position: [10,10], size: [300,200], rounding: 8, corner: LiteGraph.CORNER_TOP_LEFT };
		this._area = vec4.create();
		this._color = vec4.create();
		this._titlecolor = vec4.create();
		this._offset = [0,0];
	}

	LGraphGUIPanel.title = "GUIPanel";
	LGraphGUIPanel.desc = "renders a rectangle on webgl canvas";
	LGraphGUIPanel.priority = -1; //render first

	LGraphGUIPanel["@corner"] = corner_options;
	LGraphGUIPanel["@color"] = { type:"color" };
	LGraphGUIPanel["@titlecolor"] = { type:"color" };
	LGraphGUIPanel["@opacity"] = { widget:"slider", min:0,max:1 };

	LGraphGUIPanel.prototype.onExecute = function()
	{
		this.setOutputData(0, this._area );
		this.setOutputData(1, this.properties.enabled );
	}

	LGraphGUIPanel.prototype.onRenderGUI = function()
	{ 
		this.properties.enabled = this.getInputOrProperty("enabled");
		if(this.properties.enabled === false)
			return;

		var ctx = window.gl;
		if(!ctx)
			return;

		this._color.set( this.properties.color || [0.1,0.1,0.1] );
		this._color[3] = this.properties.opacity;
		ctx.fillColor = this._color;
		positionToArea( this.properties.position, this.properties.corner, this._area );
		this._area[0] += this._offset[0];
		this._area[1] += this._offset[1];
		this._area[2] = this.properties.size[0];
		this._area[3] = this.properties.size[1];

		//var mouse = LS.Input.current_click;
		//var clicked = LS.Input.isEventInRect( mouse, this._area, LS.GUI._offset );
		//if(clicked)
		//	LS.Input.current_click = false; //consume event

		gl.disable( gl.DEPTH_TEST );
		gl.enable( gl.BLEND );
		gl.blendFunc( gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA );
		var rounding = Math.min(15,this.properties.rounding);
		if(rounding > 0)
		{
			ctx.beginPath();
			ctx.roundRect( this._area[0], this._area[1], this.properties.size[0], this.properties.size[1], rounding, rounding );
			ctx.fill();
		}
		else
			ctx.fillRect( this._area[0], this._area[1], this.properties.size[0], this.properties.size[1] );

		if(this.properties.title)
		{
			this._titlecolor.set( this.properties.titlecolor || [0.3,0.3,0.3] );
			this._titlecolor[3] = this.properties.opacity;
			ctx.fillColor = this._titlecolor;
			if(rounding > 0)
			{
				ctx.beginPath();
				ctx.roundRect( this._area[0], this._area[1], this.properties.size[0], 30, rounding, 0 );
				ctx.fill();
			}
			else
				ctx.fillRect( this._area[0], this._area[1], this.properties.size[0], 30 );
			ctx.fillColor = [0.8,0.8,0.8,this.properties.opacity];
			ctx.font = "20px Arial";
			ctx.fillText( this.properties.title, 10 + this._area[0],24 + this._area[1]);
		}
	}

	LGraphGUIPanel.prototype.onMouse = function(e,v)
	{
		if(!this.properties.enabled || !this.properties.draggable )
			return;

		var area = this._area;
		var x = e.mousex;
		var y = e.mousey;
		if( e.type == "mousedown" )
		{
			//check if inside
			if( x >= area[0] && x < (area[0] + area[2]) && 
				y >= area[1] && y < (area[1] + area[3]) )
			{
				this._dragging = true;
				return true;
			}
		}
		else if( e.type == "mousemove" )
		{
			if( this._dragging )
			{
				this._offset[0] += e.deltax;
				this._offset[1] += e.deltay;
				return true;
			}
		}
		else //mouse up
			this._dragging = false;
	}

	LGraphGUIPanel.prototype.onGetInputs = function(){
		return [["enabled","boolean"]];
	}

	LiteGraph.registerNodeType("gui/panel", LGraphGUIPanel );

	function LGraphGUIText()
	{
		this.addInput("text");
		this.properties = { enabled: true, text: "", font: "", color: [1,1,1,1], precision: 3, position: [20,20], corner: LiteGraph.CORNER_TOP_LEFT };
		this._pos = vec2.create();
		this._text = "";
	}

	LGraphGUIText.title = "GUIText";
	LGraphGUIText.desc = "renders text on webgl canvas";

	LGraphGUIText["@corner"] = corner_options;
	LGraphGUIText["@color"] = { type:"color" };

	LGraphGUIText.prototype.onGetInputs = function(){
		return [["enabled","boolean"]];
	}

	LGraphGUIText.prototype.onExecute = function()
	{
		var v = this.getInputData(0);
		if(v != null)
		{
			if( v.constructor === Number )
				this._text = v.toFixed( this.properties.precision );
			else
				this._text = String(v);
		}
	}

	LGraphGUIText.prototype.onRenderGUI = function()
	{ 
		var ctx = window.gl;
		if(!ctx)
			return;

		var enabled = this.getInputOrProperty("enabled");
		if(enabled === false)
			return;

		var text = (this.properties.text || "") + this._text;
		if(text == "")
			return;

		ctx.font = this.properties.font || "20px Arial";
		ctx.textAlign = "left";
		ctx.fillColor = this.properties.color || [1,1,1,1];

		positionToArea( this.properties.position, this.properties.corner, this._pos );

		gl.disable( gl.DEPTH_TEST );
		gl.enable( gl.BLEND );
		gl.blendFunc( gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA );
		ctx.fillText( text, this._pos[0], this._pos[1] );
	}

	LiteGraph.registerNodeType("gui/text", LGraphGUIText );

	function LGraphGUIImage()
	{
		this.addInput("","image,canvas,texture");
		this.properties = { enabled: true, opacity: 1, keep_aspect: true, flipX: false, flipY: false, force_update: false, position: [20,20], size: [300,200], corner: LiteGraph.CORNER_TOP_LEFT };
		this._pos = vec2.create();
	}

	LGraphGUIImage.title = "GUIImage";
	LGraphGUIImage.desc = "renders an image on webgl canvas";

	LGraphGUIImage["@corner"] = corner_options;
	LGraphGUIImage["@opacity"] = { widget:"slider", min:0,max:1 };

	LGraphGUIImage.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"]];
	}

	LGraphGUIImage.prototype.onRenderGUI = function()
	{ 
		var ctx = window.gl;
		if(!ctx)
			return;

		var img = this.getInputData(0);
		var enabled = this.getInputOrProperty("enabled");
		if(enabled === false || !img)
			return;

		if(this.properties.force_update)
			img.mustUpdate = true;

		positionToArea( this.properties.position, this.properties.corner, this._pos );

		gl.disable( gl.DEPTH_TEST );
		gl.enable( gl.BLEND );
		gl.blendFunc( gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA );
		var tmp = ctx.globalAlpha;
		ctx.globalAlpha *= this.properties.opacity;
		var x = this._pos[0];
		var y = this._pos[1];
		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			x += parent_pos[0];
			y += parent_pos[1];
		}
		var w = this.properties.size[0];
		var h = this.properties.size[1];
		if(this.properties.keep_aspect)
			h = (this.properties.size[0] / img.width) * img.height;
		if(this.properties.flipX)
		{
			x += w;
			w *= -1;
		}
		if(this.properties.flipY)
		{
			y += h;
			h *= -1;
		}
		ctx.drawImage( img, x, y, w, h );
		ctx.globalAlpha = tmp;
	}

	LiteGraph.registerNodeType("gui/image", LGraphGUIImage );


	//special kind of node
	function LGraphGUISlider()
	{
		this.addOutput("v");
		this.properties = { enabled: true, text: "", min: 0, max: 1, value: 0, position: [20,20], size: [200,40], corner: LiteGraph.CORNER_TOP_LEFT };
		this._area = vec4.create();
	}

	LGraphGUISlider.title = "GUISlider";
	LGraphGUISlider.desc = "Renders a slider on the main canvas";
	LGraphGUISlider["@corner"] = corner_options;

	LGraphGUISlider.prototype.onRenderGUI = function()
	{
		if(!this.getInputOrProperty("enabled"))
			return;
		positionToArea( this.properties.position, this.properties.corner, this._area );
		this._area[2] = this.properties.size[0];
		this._area[3] = this.properties.size[1];

		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			this._area[0] += parent_pos[0];
			this._area[1] += parent_pos[1];
		}

		this.properties.value = LS.GUI.HorizontalSlider( this._area, Number(this.properties.value), Number(this.properties.min), Number(this.properties.max), true );
		if(this.properties.text)
		{
			gl.textAlign = "right";
			gl.fillStyle = "#AAA";
			gl.fillText( this.properties.text, this._area[0] - 20, this._area[1] + this._area[3] * 0.75);
			gl.textAlign = "left";
		}
	}

	LGraphGUISlider.prototype.onExecute = function()
	{
		if(this.inputs && this.inputs.length)
			this.properties.enabled = this.getInputOrProperty("enabled");
		this.setOutputData(0, this.properties.value );
	}

	LGraphGUISlider.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"]];
	}

	LiteGraph.registerNodeType("gui/slider", LGraphGUISlider );


	function LGraphGUIToggle()
	{
		this.addOutput("v");
		this.properties = { enabled: true, value: true, text:"toggle", position: [20,20], size: [140,40], corner: LiteGraph.CORNER_TOP_LEFT };
		this._area = vec4.create();
	}

	LGraphGUIToggle.title = "GUIToggle";
	LGraphGUIToggle.desc = "Renders a toggle widget on the main canvas";
	LGraphGUIToggle["@corner"] = corner_options;

	LGraphGUIToggle.prototype.onRenderGUI = function()
	{
		if(!this.getInputOrProperty("enabled"))
			return;

		positionToArea( this.properties.position, this.properties.corner, this._area );
		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			this._area[0] += parent_pos[0];
			this._area[1] += parent_pos[1];
		}
		this._area[2] = this.properties.size[0];
		this._area[3] = this.properties.size[1];
		this.properties.value = LS.GUI.Toggle( this._area, this.properties.value, this.properties.text );
	}

	LGraphGUIToggle.prototype.onExecute = function()
	{
		this.setOutputData(0, this.properties.value );
	}

	LGraphGUIToggle.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"]];
	}

	LiteGraph.registerNodeType("gui/toggle", LGraphGUIToggle );

	function LGraphGUIButton()
	{
		this.addOutput("",LiteGraph.EVENT);
		this.addOutput("was_pressed");
		this.properties = { enabled: true, text:"clickme", position: [20,20], size: [140,40], corner: LiteGraph.CORNER_TOP_LEFT };
		this.widgets_start_y = 2;
		this.addWidget("text","text","clickme","text");
		this._area = vec4.create();
		this._was_pressed = false;
	}

	LGraphGUIButton.title = "GUIButton";
	LGraphGUIButton.desc = "Renders a toggle widget on the main canvas";
	LGraphGUIButton["@corner"] = corner_options;

	LGraphGUIButton.prototype.onRenderGUI = function()
	{
		if(!this.getInputOrProperty("enabled"))
			return;
		positionToArea( this.properties.position, this.properties.corner, this._area );
		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			this._area[0] += parent_pos[0];
			this._area[1] += parent_pos[1];
		}
		this._area[2] = this.properties.size[0];
		this._area[3] = this.properties.size[1];
		this._was_pressed = LS.GUI.Button( this._area, this.properties.text );
	}

	LGraphGUIButton.prototype.onExecute = function()
	{
		var enabled = this.getInputDataByName("enabled");
		if(enabled === false || enabled === true)
			this.properties.enabled = enabled;
		if(this._was_pressed)
			this.triggerSlot(0);
		this.setOutputData(1, this._was_pressed );
		this._was_pressed = false;
	}

	LGraphGUIButton.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"]];
	}

	LiteGraph.registerNodeType("gui/button", LGraphGUIButton );


	function LGraphGUITextField()
	{
		this.addInput("","string");
		this.addInput("clear",LiteGraph.ACTION);
		this.addOutput("change",LiteGraph.EVENT);
		this.addOutput("str");
		this.properties = { enabled: true, caption:"", value: "", clear_on_intro: false, keep_focus_on_intro: false, position: [20,20], size: [200,40], corner: LiteGraph.CORNER_TOP_LEFT };
		this.widgets_start_y = 2;
		this.addWidget("text","Caption","","caption");
		this._area = vec4.create();
		this._changed = false;
		this._prev_value = "";
	}

	LGraphGUITextField.title = "GUITextField";
	LGraphGUITextField.desc = "Renders a input text widget on the main canvas";
	LGraphGUITextField["@corner"] = corner_options;

	LGraphGUITextField.prototype.onRenderGUI = function()
	{
		if(!this.getInputOrProperty("enabled"))
			return;
		positionToArea( this.properties.position, this.properties.corner, this._area );
		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			this._area[0] += parent_pos[0];
			this._area[1] += parent_pos[1];
		}
		this._area[2] = this.properties.size[0];
		this._area[3] = this.properties.size[1];
		if(this.properties.caption != null)
		{
			this._area[2] *= 0.5;
			LS.GUI.Label( this._area, String( this.properties.caption ) );
			this._area[0] += this._area[2] * 0.5;
		}
		var that = this;
		this.properties.value = LS.GUI.TextField( this._area, this.properties.value, null, false, function(){
			that._changed = true;
			that._value = that.properties.value;
			if( that.properties.clear_on_intro)
				return "";
		}, this.properties.keep_focus_on_intro );
	}

	LGraphGUITextField.prototype.onAction = function(name)
	{
		//clear
		this.properties.value = "";
	}

	LGraphGUITextField.prototype.onExecute = function()
	{
		var enabled = this.getInputDataByName("enabled");
		if(enabled === false || enabled === true)
			this.properties.enabled = enabled;
		var str = this.getInputData(0);
		if(str != null)
			this.setProperty("caption",str);
		if(this._value != this._last_value && this._changed)
		{
			this._changed = false;
			this._last_value = this._value;
			this.triggerSlot(0);
		}
		this.setOutputData(1, this.properties.value );
	}

	LGraphGUITextField.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"]];
	}

	LiteGraph.registerNodeType("gui/textfield", LGraphGUITextField );

	function LGraphGUIMultipleChoice()
	{
		this.addOutput("v");
		this.addOutput("i");
		this.properties = { enabled: true, selected: 0, values:"option1;option2;option3", one_line: false, position: [20,20], size: [180,100], corner: LiteGraph.CORNER_TOP_LEFT };
		this._area = vec4.create();
		this._values = this.properties.values.split(";");
		var that = this;
		this.widget = this.addWidget("text","Options",this.properties.values,"values");
		this.size = [240,70];
	}

	LGraphGUIMultipleChoice.title = "GUIMultipleChoice";
	LGraphGUIMultipleChoice.desc = "Renders a multiple choice widget on the main canvas";
	LGraphGUIMultipleChoice["@corner"] = corner_options;

	LGraphGUIMultipleChoice.prototype.onPropertyChanged = function(name,value)
	{
		if(name == "values")
			this._values = value.split(";");
	}

	LGraphGUIMultipleChoice.prototype.onAction = function(name, param)
	{
		if(name == "prev")
			this.properties.selected -= 1;
		else if(name == "next")
			this.properties.selected += 1;
		this.properties.selected = this.properties.selected % this._values.length;
		if(this.properties.selected < 0)
			this.properties.selected += this._values.length;
	}

	LGraphGUIMultipleChoice.prototype.onRenderGUI = function()
	{
		var enabled = this.getInputOrProperty("enabled");

		if(!this._values.length || !enabled )
			return;

		var selected = this.properties.selected = Math.floor( this.properties.selected );
		positionToArea( this.properties.position, this.properties.corner, this._area );
		var ctx = gl;
		
		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			this._area[0] += parent_pos[0];
			this._area[1] += parent_pos[1];
		}

		if(this.properties.one_line)
		{
			var pos = this.properties.position;
			var size = this.properties.size;
			var w = size[1]; //use height as width
			this._area[2] = w * 2;
			this._area[3] = size[1];
			if( LS.GUI.ClickArea( this._area ) )
				selected -= 1;
			this._area[0] += size[0] - w*2;
			if( LS.GUI.ClickArea( this._area ) )
				selected += 1;
			selected = selected % this._values.length;
			if(selected < 0)
				selected += this._values.length;
			ctx.fillStyle = "black";
			ctx.strokeStyle = "#AAA";
			ctx.lineWidth = 2;
			ctx.beginPath();
			ctx.roundRect( pos[0], pos[1], size[0], size[1], w * 0.5 );
			ctx.fill();
			ctx.stroke();
			ctx.fillStyle = "white";
			ctx.beginPath();
			var m = w * 0.25;
			ctx.moveTo( pos[0] + m, pos[1] + w * 0.5 );
			ctx.lineTo( pos[0] + w*0.5 + m*2, pos[1] + m );
			ctx.lineTo( pos[0] + w*0.5 + m*2, pos[1] + w - m);
			ctx.fill();
			ctx.beginPath();
			ctx.moveTo( pos[0] + size[0] - m, pos[1] + w * 0.5 );
			ctx.lineTo( pos[0] + size[0] - w*0.5 - m*2, pos[1] + m);
			ctx.lineTo( pos[0] + size[0] - w*0.5 - m*2, pos[1] + w - m);
			ctx.fill();
			ctx.fillStyle = "#AAA";
			ctx.textAlign = "center";
			ctx.font = (w*0.75).toFixed(0) + "px " + LS.GUI.GUIStyle.font;
			ctx.fillText( String(this._values[selected]), pos[0] + size[0] * 0.5, pos[1] + size[1] * 0.75 );
		}
		else
		{
			this._area[2] = this.properties.size[0];
			this._area[3] = this.properties.size[1] / this._values.length;
			var y = this._area[1];
			for(var i = 0; i < this._values.length; ++i)
			{
				this._area[1] = y + i * this._area[3];
				if( LS.GUI.Toggle( this._area, i == selected, this._values[i], null, true ) )
					selected = i;
			}
		}

		this.properties.selected = selected;

		var mouse = LS.Input.current_click;
		if(mouse)
		{
			var clicked = LS.Input.isEventInRect( mouse, this._area, LS.GUI._offset );
			if(clicked)
				LS.Input.current_click = false; //consume event
		}
	}

	LGraphGUIMultipleChoice.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"],["options","array"],["next",LiteGraph.ACTION],["prev",LiteGraph.ACTION]];
	}

	LGraphGUIMultipleChoice.prototype.onExecute = function()
	{
		if(this.inputs)
		{
			for(var i = 0; i < this.inputs.length; ++i)
			{
				var input_info = this.inputs[i];
				var v = this.getInputData(i);
				if( input_info.name == "enabled" )
					this.properties.enabled = Boolean(v);
				else if( input_info.name == "options" && v)
				{
					this._values = v;
					this.properties.values = v.join(";");
					this.widget.value = this.properties.values;
				}
			}
		}
		this.setOutputData( 0, this._values[ this.properties.selected ] );
		this.setOutputData( 1, this.properties.selected );
	}

	LiteGraph.registerNodeType("gui/multiple_choice", LGraphGUIMultipleChoice );


	//special kind of node
	function LGraphGUIPad()
	{
		this.addInput("bg","image,texture");
		this.addOutput("x","number");
		this.addOutput("y","number");
		this.addOutput("v","vec2");
		this.properties = { enabled: true, value: [0,0], position: [20,20], min:[0,0], max:[1,1], size: [200,200], corner: LiteGraph.CORNER_TOP_LEFT };
		this._area = vec4.create();
		this._value_norm = vec2.create();
	}

	LGraphGUIPad.title = "GUIPad";
	LGraphGUIPad.desc = "Renders a 2D pad on the main canvas";
	LGraphGUIPad["@corner"] = corner_options;

	LGraphGUIPad.prototype.onRenderGUI = function()
	{
		if(!this.getInputOrProperty("enabled"))
			return;
		positionToArea( this.properties.position, this.properties.corner, this._area );
		this._area[2] = this.properties.size[0];
		this._area[3] = this.properties.size[1];
		var parent_pos = this.getInputOrProperty("parent_pos");
		if(parent_pos)
		{
			this._area[0] += parent_pos[0];
			this._area[1] += parent_pos[1];
		}

		var bg = this.getInputData(0);

		var rangex = (this.properties.max[0] - this.properties.min[0]);
		var rangey = (this.properties.max[1] - this.properties.min[1]);
		this._value_norm[0] = (this.properties.value[0] - this.properties.min[0]) / rangex;
		this._value_norm[1] = (this.properties.value[1] - this.properties.min[1]) / rangey;

		LS.GUI.Pad( this._area, this._value_norm, bg );

		this.properties.value[0] = this.properties.min[0] + this._value_norm[0] * rangex;
		this.properties.value[1] = this.properties.min[1] + this._value_norm[1] * rangey;
	}

	LGraphGUIPad.prototype.onExecute = function()
	{
		if(this.inputs && this.inputs.length)
			this.properties.enabled = this.getInputOrProperty("enabled");
		this.setOutputData(0, this.properties.value[0] );
		this.setOutputData(1, this.properties.value[1] );
		this.setOutputData(2, this.properties.value );
	}

	LGraphGUIPad.prototype.onGetInputs = function(){
		return [["enabled","boolean"],["parent_pos","vec2"]];
	}

	LiteGraph.registerNodeType("gui/pad", LGraphGUIPad );

	//*****************************************************

	//text in 3D
	function LGraphDebugText()
	{
		this.addInput("text");
		this.properties = { enabled: true, text: "", font: "", position: [0,0,0], scale: 0.1, color: [1,1,1,1], precision: 3 };
		this._pos = vec2.create();
		this._text = "";
	}

	LGraphDebugText.title = "DebugText";
	LGraphDebugText.desc = "renders text on world space";

	LGraphDebugText["@color"] = { type:"color" };

	LGraphDebugText.prototype.onGetInputs = function(){
		return [["enabled","boolean"]];
	}

	LGraphDebugText.prototype.onExecute = function()
	{
		var v = this.getInputData(0);
		if(v != null)
		{
			if( v.constructor === Number )
				this._text = v.toFixed( this.properties.precision );
			else
				this._text = String(v);
		}
		else
			this._text = this.properties.text;

		if(this._text.length == 0)
			return;

		var node = this.graph._scenenode;
		LS.Draw.setColor( this.properties.color );
		gl.disable( gl.CULL_FACE );
		if(node && node.transform )
		{
			LS.Draw.push();
			LS.Draw.setMatrix( node.transform.getGlobalMatrixRef() );
			LS.Draw.renderText( this._text, this.properties.position, this.properties.scale );
			LS.Draw.pop();
		}
		else
			LS.Draw.renderText( this._text, this.properties.position );
		gl.enable( gl.CULL_FACE );
	}

	LiteGraph.registerNodeType("debug/text", LGraphDebugText );



	//based in the NNI distance 
	function LGraphMap2D()
	{
		this.addInput("x","number");
		this.addInput("y","number");
		this.addOutput("[]","array");
		this.addOutput("obj","object");
		this.addOutput("img","image");
		this.addProperty("circular",false);
		this.points = [];
		this.weights = [];
		this.weights_obj = {};
		this.current_pos = new Float32Array([0.5,0.5]);
		this.size = [200,200];
		this.dragging = false;
		this.show_names = true;
		this.circle_center = [0,0];
		this.circle_radius = 1;
		this.margin = 20;
		this._values_changed = true;
		this._visualize_weights = false;
		this._version = 0;
		this._selected_point = null;
	}

	LiteGraph.LGraphMap2D = LGraphMap2D;
	LGraphMap2D.title = "Map2D";
	LGraphMap2D.colors = [[255,0,0],[0,255,0],[0,0,255],[0,128,128,0],[128,0,128],[128,128,0],[255,128,0],[255,0,128],[0,128,255],[128,0,255]];
	LGraphMap2D.grid_size = 64;

	LGraphMap2D.prototype.onExecute = function()
	{
		var pos = this.current_pos;
		pos[0] = this.getInputData(0) || pos[0];
		pos[1] = this.getInputData(1) || pos[1];
		this.computeWeights(pos);
		this.setOutputData(0, this.weights );
		this.setOutputData(1, this.weights_obj );

		if(this.isOutputConnected(2))
		{
			if(!this.temp_canvas)
				this.temp_canvas = document.createElement("canvas");
			this.temp_canvas.width = this.size[0];
			this.temp_canvas.height = this.size[1];
			var temp_ctx = this.temp_canvas.getContext("2d");
			this.renderToCanvas( temp_ctx, this.temp_canvas );
			this.setOutputData(2, this.temp_canvas );
		}
	}
	
	//now to compute the final weight we iterate for every cell to see if our point is nearer to the cell than the nearest point of the cell,
	//if that is the case we increase the weight of the nearest point. At the end we normalize the weights of the points by the number of near points
	//and that give us the weight for every point
	LGraphMap2D.prototype.computeWeights = function(pos)
	{
		if(!this.points.length)
			return;
		var values = this._precomputed_weights;
		if(!values || this._values_changed )
			values = this.precomputeWeights();
		var pos2 = vec2.create();
		var circular = this.properties.circular;
		var weights = this.weights;
		weights.length = this.points.length;
		var gridsize = LGraphMap2D.grid_size;
		for(var i = 0; i < weights.length; ++i)
			weights[i] = 0;
		var total_inside = 0;
		for(var y = 0; y < gridsize; ++y)
			for(var x = 0; x < gridsize; ++x)
			{
				pos2[0] = x / gridsize;
				pos2[1] = y / gridsize;
				if(circular)
				{
					pos2[0] = pos2[0] * 2 - 1;
					pos2[1] = pos2[1] * 2 - 1;
				}
				var data_pos = x*2 + y * gridsize*2;
				var point_index = values[ data_pos ];
				var is_inside = vec2.distance( pos2, pos ) < (values[ data_pos + 1] + 0.001); //epsilon
				if(is_inside)
				{
					weights[ point_index ] += 1;
					total_inside++;
				}
			}
		for(var i = 0; i < weights.length; ++i)
		{
			weights[i] /= total_inside;
			this.weights_obj[ this.points[i].name ] = weights[i];
		}
		return weights;
	}

	LGraphMap2D.prototype.onMouseDown = function(e,pos)
	{
		if(this.flags.collapsed || pos[1] < 0 || pos[0] < this.margin || pos[0] > (this.size[0] - this.margin) || pos[1] < this.margin )
			return false;
		if( pos[1] > (this.size[1] - this.margin))
		{
			this._visualize_weights = !this._visualize_weights;
			return true;
		}

		this.dragging = true;
		return true;
	}

	LGraphMap2D.prototype.onMouseUp = function(e,pos)
	{
		this.dragging = false;
	}

	LGraphMap2D.prototype.onMouseMove = function(e,pos)
	{
		if( !this.dragging || this.flags.collapsed )
			return;
		var margin = this.margin;
		var center = [0,0];
		var radius = this.size[1] * 0.5 - margin;
		var cpos = this.current_pos;
		cpos[0] = Math.clamp( (pos[0] - margin) / (this.size[0] - margin*2), 0,1 );
		cpos[1] = Math.clamp( (pos[1] - margin) / (this.size[1] - margin*2), 0,1 );
		if( this.properties.circular )
		{
			cpos[0] = cpos[0] * 2 - 1;
			cpos[1] = cpos[1] * 2 - 1;
			var dist = vec2.distance( cpos, center );
			if(dist > 1)
				vec2.normalize(cpos,cpos);
		}
		return true;
	}

	LGraphMap2D.prototype.onDrawBackground = function( ctx )
	{
		if(this.flags.collapsed)
			return;

		this.renderToCanvas( ctx );

		//weights
		var margin = this.margin;
		var w = this.size[0];
		var h = this.size[1];

		ctx.save();
		ctx.fillStyle = "black";
		ctx.fillRect( margin, h - margin + 2, w - margin * 2, margin - 4);
		ctx.strokeStyle = "white";
		ctx.strokeRect( margin, h - margin + 2, w - margin * 2, margin - 4);
		ctx.textAlign = "center";
		ctx.fillStyle = "white";
		ctx.fillText( "Visualize Weights", w * 0.5, h - margin * 0.3 );
		ctx.textAlign = "left";

		if(this.weights.length && this._visualize_weights)
		{
			var x = w + 5;
			var y = 16; //h - this.weights.length * 5 - 10;
			for(var i = 0; i < this.weights.length; ++i)
			{
				var c = LGraphMap2D.colors[i % LGraphMap2D.colors.length];
				ctx.fillStyle = "black";
				ctx.fillRect(x, y + i*5, 100,4 );
				ctx.fillStyle = "rgb(" + ((c[0]*255)|0) + "," + ((c[1]*255)|0) + "," + ((c[2]*255)|0) + ")";
				ctx.fillRect(x, y + i*5, this.weights[i]*100,4 );
			}
		}
		ctx.restore();
	}

	LGraphMap2D.prototype.renderToCanvas = function( ctx, canvas )
	{
		var pos = this.current_pos;
		var circular = this.properties.circular;
		var show_names = this.show_names;
		var margin = this.margin;
		var w = canvas ? canvas.width : this.size[0];
		var h = canvas ? canvas.height : this.size[1];

		ctx.fillStyle = "black";
		ctx.strokeStyle = "#BBB";
		if(circular)
		{
			this.circle_center[0] = w * 0.5;
			this.circle_center[1] = h * 0.5;
			this.circle_radius = h * 0.5 - margin;
			ctx.lineWidth = 2;
			ctx.beginPath();
			ctx.arc( this.circle_center[0], this.circle_center[1], this.circle_radius, 0, Math.PI * 2 );
			ctx.fill();
			ctx.stroke();
			ctx.lineWidth = 1;
			ctx.beginPath();
			ctx.moveTo( this.circle_center[0] + 0.5, this.circle_center[1] - this.circle_radius );
			ctx.lineTo( this.circle_center[0] + 0.5, this.circle_center[1] + this.circle_radius );
			ctx.moveTo( this.circle_center[0] - this.circle_radius, this.circle_center[1]);
			ctx.lineTo( this.circle_center[0] + this.circle_radius, this.circle_center[1]);
			ctx.stroke();
		}
		else
		{
			ctx.fillRect(margin,margin,w-margin*2, h-margin*2);
			ctx.strokeRect(margin,margin,w-margin*2, h-margin*2);
		}

		var image = this.precomputeWeightsToImage( pos );
		if(image)
		{
			ctx.globalAlpha = 0.5;
			ctx.imageSmoothingEnabled = false;
			if(circular)
			{
				ctx.save();
				ctx.beginPath();
				ctx.arc( this.circle_center[0], this.circle_center[1], this.circle_radius, 0, Math.PI * 2 );
				ctx.clip();
				ctx.drawImage( image, this.circle_center[0] - this.circle_radius, this.circle_center[1] - this.circle_radius, this.circle_radius*2, this.circle_radius*2 );
				ctx.restore();
			}
			else
				ctx.drawImage( image, margin, margin,w-margin*2, h-margin*2 );
			ctx.imageSmoothingEnabled = true;
			ctx.globalAlpha = 1;
		}

		for(var i = 0; i < this.points.length; ++i)
		{
			var point = this.points[i];
			var x = point.pos[0];
			var y = point.pos[1];
			if(circular)
			{
				x = x*0.5 + 0.5;
				y = y*0.5 + 0.5;
			}
			x = x * (w-margin*2) + margin;
			y = y * (h-margin*2) + margin;
			x = Math.clamp( x, margin, w-margin);
			y = Math.clamp( y, margin, h-margin);
			ctx.fillStyle = point == this._selected_point ? "#9DF" : "#789";
			ctx.beginPath();
			ctx.arc(x,y,3,0,Math.PI*2);
			ctx.fill();
			if( show_names )
				ctx.fillText( point.name, x + 5, y + 5);
		}

		ctx.fillStyle = "white";
		ctx.beginPath();
		var x = pos[0];
		var y = pos[1];
		if(circular)
		{
			x = x*0.5 + 0.5;
			y = y*0.5 + 0.5;
		}
		x = x * (w-margin*2) + margin;
		y = y * (h-margin*2) + margin;
		x = Math.clamp( x, margin, w-margin);
		y = Math.clamp( y, margin, h-margin);
		ctx.arc(x,y,5,0,Math.PI*2);
		ctx.fill();

		if(canvas)
			canvas.mustUpdate = true;
	}

	LGraphMap2D.prototype.addPoint = function( name, pos )
	{
		if( this.findPoint(name) )
		{
			console.warn("there is already a point with that name" );
			return;
		}
		if(!pos)
			pos = [this.current_pos[0], this.current_pos[1]];
		pos[0] = Math.clamp( pos[0], -1,1 );
		pos[1] = Math.clamp( pos[1], -1,1 );
		var point = { name: name, pos: pos };
		this.points.push(point);
		this._values_changed = true;
		this.setDirtyCanvas(true);
		return point;
	}

	LGraphMap2D.prototype.removePoint = function(name)
	{
		for(var i = 0; i < this.points.length; ++i)
			if( this.points[i].name == name )
			{
				this.points.splice(i,1);
				this._values_changed = true;
				return;
			}
	}

	LGraphMap2D.prototype.findPoint = function( name )
	{
		for(var i = 0; i < this.points.length; ++i)
			if( this.points[i].name == name )
				return this.points[i];
		return null;
	}

	//here we precompute for every cell, which is the closest point of the points set and how far it is from the center of the cell
	//we store point index and distance in this._precomputed_weights
	//this is done only when the points set change
	LGraphMap2D.prototype.precomputeWeights = function()
	{
		var points = this.points;
		var num_points = points.length;
		var pos = vec2.create();
		var circular = this.properties.circular;
		this._values_changed = false;
		this._version++;
		var gridsize = LGraphMap2D.grid_size;
		var total_nums = 2 * gridsize * gridsize;
		if(!this._precomputed_weights || this._precomputed_weights.length != total_nums )
			this._precomputed_weights = new Float32Array( total_nums );
		var values = this._precomputed_weights;
		this._precomputed_weights_gridsize = gridsize;

		for(var y = 0; y < gridsize; ++y)
			for(var x = 0; x < gridsize; ++x)
			{
				var nearest = -1;
				var min_dist = 100000;
				for(var i = 0; i < num_points; ++i)
				{
					pos[0] = x / gridsize;
					pos[1] = y / gridsize;
					if(circular)
					{
						pos[0] = pos[0] * 2 - 1;
						pos[1] = pos[1] * 2 - 1;
					}

					var dist = vec2.distance( pos, points[i].pos );
					if( dist > min_dist )
						continue;
					nearest = i;
					min_dist = dist;
				}

				values[ x*2 + y*2*gridsize ] = nearest;
				values[ x*2 + y*2*gridsize + 1] = min_dist;
			}

		return values;
	}

	LGraphMap2D.prototype.precomputeWeightsToImage = function(pos)
	{
		if(!this.points.length)
			return null;
		var values = this._precomputed_weights;
		if(!values || this._values_changed || this._precomputed_weights_gridsize != LGraphMap2D.grid_size)
			values = this.precomputeWeights();
		var canvas = this._canvas;
		var gridsize = LGraphMap2D.grid_size;
		if(!canvas)
			canvas = this._canvas = document.createElement("canvas");
		canvas.width = canvas.height = gridsize;
		var ctx = canvas.getContext("2d");
		var white = [255,255,255];
		var pos2 = vec2.create();
		var circular = this.properties.circular;
		var weights = this.weights;
		weights.length = this.points.length;
		for(var i = 0; i < weights.length; ++i)
			weights[i] = 0;
		var total_inside = 0;
		var pixels = ctx.getImageData(0,0,gridsize,gridsize);
		for(var y = 0; y < gridsize; ++y)
			for(var x = 0; x < gridsize; ++x)
			{
				pos2[0] = x / gridsize;
				pos2[1] = y / gridsize;
				if(circular)
				{
					pos2[0] = pos2[0] * 2 - 1;
					pos2[1] = pos2[1] * 2 - 1;
				}

				var pixel_pos = x*4 + y*gridsize*4;
				var data_pos = x*2 + y * gridsize*2;
				var point_index = values[ data_pos ];
				var c = LGraphMap2D.colors[ point_index % LGraphMap2D.colors.length ];
				var is_inside = vec2.distance( pos2, pos ) < (values[ data_pos + 1] + 0.001);
				if(is_inside)
				{
					weights[ point_index ] += 1;
					total_inside++;
				}
				pixels.data[pixel_pos] = c[0] + (is_inside ? 128 : 0);
				pixels.data[pixel_pos+1] = c[1] + (is_inside ? 128 : 0);
				pixels.data[pixel_pos+2] = c[2] + (is_inside ? 128 : 0);
				pixels.data[pixel_pos+3] = 255;
			}
		for(var i = 0; i < weights.length; ++i)
			weights[i] /= total_inside;
		ctx.putImageData(pixels,0,0);
		return canvas;
	}

	LGraphMap2D.prototype.clear = function()
	{
		this.points.length = 0;
		this._precomputed_weights = null;
		this._canvas = null;
		this._selected_point = null;
		this.setDirtyCanvas(true);
	}

	LGraphMap2D.prototype.getExtraMenuOptions = function()
	{
		return[{content:"Clear Points", callback: this.clear.bind(this) }];
	}

	LGraphMap2D.prototype.onInspect = function( inspector )
	{
		var node = this;
		if(!this._selected_point && this.points.length)
			this._selected_point = this.points[0];
		inspector.addTitle("Points");

		inspector.widgets_per_row = 4;

		for(var i = 0; i < this.points.length; ++i)
		{
			var point = this.points[i];
			inspector.addString( null, point.name, { point: point, width: "40%", callback: function(v){
				this.options.point.name = v;
				node.setDirtyCanvas(true);
			}});
			var posX_widget = inspector.addNumber(null, point.pos[0], { point: point, width: "20%", min:-1, max:1, step: 0.01, callback: function(v){
				this.options.point.pos[0] = Math.clamp( v, -1, 1 );
				node._values_changed = true;
			}});
			var posY_widget = inspector.addNumber(null,point.pos[1], { point: point, width: "20%", min:-1, max:1, step: 0.01, callback: function(v){
				this.options.point.pos[1] = Math.clamp( v, -1, 1 );
				node._values_changed = true;
			}});
			inspector.addButton(null,"o", { point: point, width: "10%", callback: function(){
				this.options.point.pos[0] = node.current_pos[0];
				this.options.point.pos[1] = node.current_pos[1];
				node._values_changed = true;
			}});
			inspector.addButton(null,"X", { point: point, width: "10%", callback: function(){
				LiteGUI.confirm("Are you sure? Removing one point could mess up the whole weights order", (function(v){
					if(!v)
						return;
					node.removePoint( this.point.name );	
					inspector.refresh();
				}).bind(this.options));
			}});
		}
		inspector.widgets_per_row = 1;

		var new_point_name = "";
		inspector.widgets_per_row = 2;
		inspector.addString("New Point","",{ width:"75%", callback: function(v){
			new_point_name = v;
		}});
		inspector.addButton(null,"Create",{ width:"25%",callback: function(v){
			if(new_point_name)
				node.addPoint( new_point_name );
			inspector.refresh();
		}});
		inspector.widgets_per_row = 1;

		inspector.addSeparator();
	}

	LGraphMap2D.prototype.onSerialize = function(o)
	{
		o.current_pos = this.current_pos;
		for(var i = 0; i < this.points.length; ++i)
			delete this.points[i]._dist;
		o.points = this.points;
	}

	LGraphMap2D.prototype.onConfigure = function(o)
	{
		if(o.current_pos)
			this.current_pos = o.current_pos;
		if(o.points)
			this.points = o.points;
	}


	LiteGraph.registerNodeType("math/map2D", LGraphMap2D );


	function LGraphRemapWeights()
	{
		this.addInput("in","array"); //array because they are already in order
		this.addOutput("out","object");
		this.points = [];	//2D points, name of point ("happy","sad") and weights ("mouth_left":0.4, "mouth_right":0.3)
		this.current_weights = {}; //object that tells the current state of weights, like "mouth_left":0.3, ...
		this.properties = { enabled: true, dimensions: 1 };
		var node = this;
		this.combo = this.addWidget("combo","Point", "", function(v){
			node._selected_point = node.findPoint(v);
		}, { values:[] } );
		this.import_button = this.addWidget("button", "import weights", "", function(){
			node.importWeights(true,true);
		});
		this.size = [170,80];
		this._selected_point = null;
		this._dims = 1;
	}

	LGraphRemapWeights.title = "Remap Weights";

	LGraphRemapWeights["@dimensions"] = { widget:"number", min:1,step:1,precision:0 };

	LGraphRemapWeights.prototype.onPropertyChanged = function(name,value)
	{
		if(name != "dimensions" || this._dims == value)
			return;

		//this.properties.dimensions = value || 1; //already changed

		//adjust dimensions
		var dim = value;
		this._dims = value;

		for(var i = 0; i < this.points.length; ++i)
		{
			var p = this.points[i];
			for(var j in p.weights )
			{
				p.weights[j] = dim == 1 ? 0 : new Array(dim).fill(0);
			}
		}

		for(var i in this.current_weights)
		{
			if( this.current_weights[i] == null )
			{
				this.current_weights[i] = dim == 1 ? 0 : new Array(dim).fill(0);
				continue;
			}

			if( this.current_weights[i].constructor === Number && dim == 1)
				continue;

			if( this.current_weights[i].length != dim )
				this.current_weights[i] = new Array(dim).fill(0);
		}
	}

	LGraphRemapWeights.prototype.onExecute = function()
	{
		var enabled = this.getInputOrProperty("enabled");
		if(!enabled)
			return;

		var point_weights = this.getInputData(0); //array

		if(this.inputs)
		for(var i = 1; i < this.inputs.length; ++i)
		{
			var input_info = this.inputs[i];
			if(input_info.name == "selected")
			{
				var selected = this.getInputData(i); 
				if(selected)
				{
					this._selected_point = this.findPoint(selected);
					this.combo.value = selected;
				}
			}
		}

		//clear
		var dimensions = this.properties.dimensions || 1;
		for(var i in this.current_weights)
		{
			if( dimensions == 1 )
				this.current_weights[i] = 0;
			else if( !this.current_weights[i] || this.current_weights[i].length != dimensions )
				this.current_weights[i] = new Array(dimensions).fill(0);
			else
				this.current_weights[i].fill(0);
		}

		var points_has_changed = false;
		if( point_weights )
		for(var i = 0; i < point_weights.length; ++i)
		{
			var point = this.points[i];
			if(!point)
			{
				points_has_changed = true;
				continue;
			}
			var w = point_weights[i]; //input
			for(var j in point.weights)
			{
				if(dimensions == 1)
				{
					var v = (point.weights[j] || 0) * w;
					this.current_weights[j] += v;
				}
				else
					for(var k = 0; k < dimensions; ++k)
					{
						var v = point.weights[j][k] * w;
						this.current_weights[j][k] += v;
					}
			}
		}

		//output weights
		this.setOutputData(0, this.current_weights );

		if(this.outputs)
		for(var i = 1; i < this.outputs.length; ++i)
		{
			var output_info = this.outputs[i];
			if(!output_info)
				continue;
			if(output_info.name == "selected")
				this.setOutputData(i, this._selected_point ? this._selected_point.name : "" );
			else
				this.setOutputData(i, this.current_weights[output_info.name] );
		}

		if(points_has_changed)
			this.importPoints();
	}

	LGraphRemapWeights.prototype.onAction = function(name, params)
	{
		if(name == "import")
			this.importWeights(true); //do not force or recursion ahead
	}

	//adds a 2D point with the weights associated to it (not used?)
	LGraphRemapWeights.prototype.addPoint = function( name, weights )
	{
		if(!weights)
		{
			console.warn("no weights passed to add point");
			return;
		}
		var w = {};
		for(var i in weights)
			w[i] = weights[i];
		this.points.push({name: name, weights: w});	
	}

	//import 2D points from input node (usually LGraphMap2D), just the names
	LGraphRemapWeights.prototype.importPoints = function()
	{
		var input_node = this.getInputNode(0);
		if(!input_node || !input_node.points || !input_node.points.length )
			return;
		this.points.length = input_node.points.length;
		for(var i = 0; i < this.points.length; ++i)
			this.points[i] = { name: input_node.points[i].name, weights: {} };
		this._selected_point = this.points[0];
		if( this._selected_point )
		{
			this.combo.value = this._selected_point.name;
			this.combo.options.values = this.points.map(function(a){return a.name;});
		}
		else
		{
			this.combo.value = "";
			this.combo.options.values = [""];
		}
	}

	//when called it reads the output nodes to get which morph targets it is using and read their weights
	//then sets the current 2D point to this weights
	LGraphRemapWeights.prototype.importWeights = function( assign, run_graph )
	{
		//force data to flow from inputs to here
		if(this.graph && run_graph)
			this.graph.runStep(1,false, this.order );

		var name_weights = this.getInputDataByName("name_weights");
		var dimensions = this.properties.dimensions || 1;
		
		if(name_weights)
		{
			for(var j in name_weights)
				this.current_weights[j] = dimensions == 1 ? name_weights[j] : name_weights[j].concat();
		}
		else //get from output
		{
			var output_nodes = this.getOutputNodes(0);
			if(!output_nodes || output_nodes.length == 0)
				return;

			for(var i = 0; i < output_nodes.length; ++i)
			{
				var output_node = output_nodes[i];
				if( !output_node.getComponent )
					continue;

				var component = output_node.getComponent();
				if(!component)
					continue;

				var compo_weights = component.name_weights;
				var compo_weights = component.name_weights;
				for(var j in compo_weights)
					this.current_weights[j] = dimensions == 1 ? compo_weights[j] : name_weights[j].concat();
			}
		}

		this.setDirtyCanvas(true);

		if( !assign || !this._selected_point)
			return;
		this._selected_point.weights = {};
		for(var i in this.current_weights)
			this._selected_point.weights[i] = dimensions == 1 ? this.current_weights[i] : this.current_weights[i].concat();
	}

	LGraphRemapWeights.prototype.findPoint = function( name )
	{
		for(var i = 0; i < this.points.length; ++i)
			if( this.points[i].name == name )
				return this.points[i];
		return null;
	}

	LGraphRemapWeights.prototype.assignCurrentWeightsToPoint = function( point )
	{
		var dimensions = this.properties.dimensions || 1;
		for(var i in this.current_weights)
			if( dimensions == 1 )
				point.weights[i] = dimensions == 1 ? this.current_weights[i] : this.current_weights[i].concat();
	}

	LGraphRemapWeights.prototype.onSerialize = function(o)
	{
		o.current_weights = this.current_weights;
		o.points = this.points;
		o.enabled = this.enabled;
	}

	LGraphRemapWeights.prototype.removeWeight = function( name )
	{
		for(var i = 0; i < this.points.length; ++i)
			delete this.points[i].weights[name];
		delete this.current_weights[name];
	}

	LGraphRemapWeights.prototype.onConfigure = function(o)
	{
		if(o.enabled !== undefined)
			this.properties.enabled = o.enabled;
		if( o.current_weights )
			this.current_weights = o.current_weights;
		this.properties.dimensions = o.properties.dimensions || 1;
		this._dims = this.properties.dimensions;

		if(o.points)
		{
			this.points = o.points;
			var dimensions = o.properties.dimensions;

			//error checking from legacy
			for(var i = 0;i < this.points.length; ++i)
			{
				var p = this.points[i];
				if(p.weights && p.weights.constructor !== Object)
					p.weights = {};
				for(var j in p.weights)
				{
					if( p.weights[j].constructor == Number && dimensions != 1)
						p.weights[j] = 0;
					else if (p.weights[j].constructor !== Array && dimensions > 1)
						p.weights[j] = new Array(dimensions).fill(0);
				}
			}

			//widget
			this.combo.options.values = this.points.map(function(a){ return a.name; });
			this.combo.value = this.combo.options.values[0] || "";
		}
	}

	LGraphRemapWeights.prototype.onInspect = function( inspector )
	{
		var node = this;

		inspector.addButton(null,"Import points from input", { callback: function(){
			node.importPoints();
			inspector.refresh();
		}});
		inspector.addButton(null,"Import weights from output", { callback: function(){
			node.importWeights(null,true);
			inspector.refresh();
		}});

		inspector.addSeparator();

		inspector.addTitle("Points");

		var point_names = [];
		for(var i = 0; i < this.points.length; ++i)
		{
			var point = this.points[i];
			point_names.push( point.name );
		}

		if(!this._selected_point && this.points.length)
			this._selected_point = this.points[0];

		inspector.addCombo("Points",this._selected_point ? this._selected_point.name : "", { values: point_names, callback: function(v){
			node._selected_point = node.findPoint(v);
			node.combo.value = v;
			if(node._selected_point)
				for(var i in node._selected_point.weights)
					node.current_weights[i] = node._selected_point.weights[i];
			node.setDirtyCanvas(true);
			inspector.refresh();
		}});

		inspector.addButton(null,"current weights to point", { callback: function(){
			if(!node._selected_point)
				return;
			node.assignCurrentWeightsToPoint(node._selected_point);
			inspector.refresh();
		}});

		inspector.addStringButton("New Point","", { button: "+", callback_button: function(v){
			if(!v)
			{
				LiteGUI.alert("You must specify a name");
				return;
			}
			var weights = JSON.parse( JSON.stringify( node.current_weights) );
			node._selected_point = node.addPoint(v,weights);
			inspector.refresh();
		}});

		inspector.addSeparator();
		inspector.addTitle("Weights");

		var dimensions = this.properties.dimensions || 1;
		var type = null;
		switch(dimensions)
		{
			case 2: type = "vec2"; break;
			case 3: type = "vec3"; break;
			case 4: type = "vec4"; break;
			default: type = "number"; break;
		}

		inspector.widgets_per_row = 2;
		if(type)
		for(var i in this.current_weights)
		{
			var name = i;
			inspector.add( type, name || "", this.current_weights[ name ], { width: "calc(100% - 40px)", name_width: 160, index: name, callback: function(v){
				node.current_weights[ this.options.index ] = v;
			}});
			inspector.addButton( null, InterfaceModule.icons.trash, { width: 30, index: name, callback: function(v){
				var name = this.options.index;
				LiteGUI.confirm("Are you sure? It will be removed from all points", function(v){
					if(v)
						node.removeWeight( name );
					inspector.refresh();
				});
			}});
		}
		inspector.widgets_per_row = 1;

		inspector.addStringButton("Add Weight","", { button: "+", callback_button: function(v){
			if(!v)
			{
				LiteGUI.alert("You must specify a name");
				return;
			}
			node.current_weights[v] = dimensions == 1 ? 1 : new Array(dimensions).fill(0);
			inspector.refresh();
		}});

		inspector.addSeparator();
	}

    LGraphRemapWeights.prototype.onGetOutputs = function() {
        var r = [["selected","string"]];
		var type = null;
		switch(this.properties.dimensions)
		{
			case 2: type = "vec2"; break;
			case 3: type = "vec3"; break;
			case 4: type = "vec4"; break;
			default: type = "number"; break;
		}
		for(var i in this.current_weights)
			r.push([i,type]);
		return r;
    };

	LGraphRemapWeights.prototype.onGetInputs = function()
	{
		return [["enabled","boolean"],["import",LiteGraph.ACTION],["selected","string"],["name_weights","object"]];
	}

	LiteGraph.registerNodeType("math/remap_weights", LGraphRemapWeights );

	//******************************************

	function LGraphCameraRay()
	{
		this.addInput("camera","component,camera");
		this.addInput("pos2D","vec2");
		this.addOutput("ray","ray");
		this.properties = {
			reverse_y: false
		};
		this._ray = new LS.Ray();
	}

	LGraphCameraRay.title = "Camera Ray";

	LGraphCameraRay.prototype.onExecute = function()
	{
		var camera = this.getInputData(0) || LS.Renderer.getCurrentCamera();
		var pos = this.getInputData(1);
		if(!camera || camera.constructor != LS.Camera || !pos)
			return;
		var viewport = null;
		var y = pos[1];
		if( this.properties.reverse_y )
			y = gl.canvas.height - pos[1];
		camera.getRay( pos[0], y, viewport, false, this._ray );
		this.setOutputData(0, this._ray);
	}

	LiteGraph.registerNodeType("math3d/camera_ray", LGraphCameraRay );

	function LGraphCameraProject()
	{
		this.addInput("camera","component,camera");
		this.addInput("pos3D","vec3");
		this.addOutput("screen_pos","vec4");
		this.properties = {
			clamp_to_viewport: false,
			reverse_y: true
		};

		this._screen_pos = vec4.create();
		this.size = [160,50];
	}

	LGraphCameraProject.title = "Camera Project";

	LGraphCameraProject.prototype.onExecute = function()
	{
		var camera = this.getInputData(0);
		var pos = this.getInputData(1);
		if(!camera || camera.constructor != LS.Camera || !pos)
			return;

		camera.project( pos, null, this._screen_pos, this.properties.reverse_y );
		var dist = vec3.distance( camera.eye, pos );
		this._screen_pos[3] = (Math.sin(camera.fov * DEG2RAD) / dist) * 100.0;

		if( this.properties.clamp_to_viewport )
		{
			this._screen_pos[0] = Math.clamp( this._screen_pos[0], 0, gl.canvas.width);
			this._screen_pos[1] = Math.clamp( this._screen_pos[1], 0, gl.canvas.height);
		}
		this.setOutputData(0, this._screen_pos);
	}

	LiteGraph.registerNodeType("math3d/camera_project", LGraphCameraProject );

	//*****************************

	function LGraphRayPlaneTest()
	{
		this.addInput("ray","ray");
		this.addInput("P","vec3");
		this.addInput("N","vec3");
		this.addOutput("pos","vec3");
		this.properties = {};
	}

	LGraphRayPlaneTest.title = "Ray-Plane test";

	LGraphRayPlaneTest.prototype.onExecute = function()
	{
		var ray = this.getInputData(0);
		var p = this.getInputData(1);
		var n = this.getInputData(2);
		if(!ray || ray.constructor != LS.Ray )
			return;
		if(!p)
			p = LS.ZEROS;
		if(!n)
			n = LS.TOP;
		var r = ray.testPlane(p,n);
		this.setOutputData( 0, ray.collision_point );
	}

	LiteGraph.registerNodeType("math3d/rayplane-test", LGraphRayPlaneTest );


	function LGraphRayCollidersTest()
	{
		this.addInput("enabled","boolean");
		this.addInput("ray","ray");
		this.addOutput("collides","boolean");
		this.addOutput("node","scenenode");
		this.addOutput("pos","vec3");
		this.addOutput("normal","vec3");
		this.properties = { max_dist: 1000, layers: 0xFF, mode: 0 };
		this.options = {};
	}

	LGraphRayCollidersTest.COLLIDERS = 0;
	LGraphRayCollidersTest.RENDERINSTANCES_BOUNDING = 1;
	LGraphRayCollidersTest.RENDERINSTANCES_MESH = 2;

	LGraphRayCollidersTest.title = "Ray-Colliders test";
	LGraphRayCollidersTest["@layers"] = { widget:"layers" };
	LGraphRayCollidersTest["@mode"] = { type:"enum", values: { "colliders": LGraphRayCollidersTest.COLLIDERS, "renderInstance_bounding": LGraphRayCollidersTest.RENDERINSTANCES_BOUNDING, "renderInstance_mesh": LGraphRayCollidersTest.RENDERINSTANCES_MESH } };

	LGraphRayCollidersTest.prototype.onExecute = function()
	{
		var enabled = this.getInputData(0);
		var ray = this.getInputData(1);
		if(enabled === false || enabled === 0 || enabled === null || !ray || ray.constructor != LS.Ray )
			return;
		var options = this.options;
		options.max_dist = this.properties.max_dist;
		options.normal = this.isInputConnected(3);
		options.layers = this.properties.layers;
		options.triangle_collision = this.properties.mode == LGraphRayCollidersTest.RENDERINSTANCES_MESH;
			
		var collisions = null;
		if(this.properties.mode == LGraphRayCollidersTest.COLLIDERS)
			collisions = LS.Physics.raycast( ray.origin, ray.direction, options );
		else 
			collisions = LS.Physics.raycastRenderInstances( ray.origin, ray.direction, options );

		if( collisions && collisions.length )
		{
			var coll = collisions[0];
			this.setOutputData( 0, true );
			this.setOutputData( 1, coll.node );
			this.setOutputData( 2, coll.position );
			this.setOutputData( 3, coll.normal );
		}
		else
			this.setOutputData( 0, false );
	}

	LiteGraph.registerNodeType("math3d/raycolliders-test", LGraphRayCollidersTest );

	//*********************************************

	function LGraphInputKey()
	{
		this.addOutput("","boolean");
		this.addOutput("",LiteGraph.EVENT);
		this.properties = {
			key: "SPACE"
		};
		var that = this;
		this.widgets_up = true;
		this.addWidget("text","Key",this.properties.key,function(v){
			if(v)
				that.properties.key = v;
		});
	}

	LGraphInputKey.title = "Key";

    LGraphInputKey.prototype.getTitle = function() {
        if (this.flags.collapsed) {
            return "Key: " + this.properties.key;
        }
        return this.title;
    };

	LGraphInputKey.prototype.onExecute = function()
	{
		var v = LS.Input.wasKeyPressed(this.properties.key);
		this.boxcolor = v ? "#fff" : "#000";
		this.setOutputData(0,v);
		if(v)
			this.triggerSlot(1,this.properties.key);
	}

	LiteGraph.registerNodeType("input/key", LGraphInputKey );
}

///@FILE:../src/helpers/path.js
///@INFO: UNCOMMON
/** Path
* Used to store splines
* types defined in defines.js: LINEAR, HERMITE, BEZIER
* @class Path
*/
function Path()
{
	this.points = [];
	this.closed = false;
	this.type = LS.LINEAR;
}

Path.prototype.clear = function()
{
	this.points.length = 0;
}

//points stored are cloned
Path.prototype.addPoint = function(p)
{
	var pos = vec3.create();
	pos[0] = p[0];
	pos[1] = p[1];
	if(p.length > 2)
		pos[2] = p[2];
	this.points.push( pos );
}

Path.prototype.getSegments = function()
{
	var l = this.points.length;

	switch(this.type)
	{
		case LS.LINEAR: 
			if(l < 2) 
				return 0;
			return l - 1 + (this.closed ? 1 : 0); 
			break;
		case LS.HERMITE:
			if(l < 2) 
				return 0;
			return l - 1 + (this.closed ? 1 : 0); 
		case LS.BEZIER:
			if(l < 3) 
				return 0;
			return (((l-1)/3)|0) + (this.closed ? 1 : 0);
			break;
	}
	return 0;
}

Path.prototype.movePoint = function( index, pos, preserve_tangents )
{
	if(index < 0 && index >= this.points.length)
		return;

	var p = this.points[ index ];
	var total_diff = vec3.sub( vec3.create(), pos, p );
	vec3.copy(p, pos);

	if( !preserve_tangents || this.type != LS.BEZIER )
		return;

	if(index % 3 == 2 && this.points.length > index + 2 )
	{
		var middle_pos = this.points[index + 1];
		var next_pos = this.points[index + 2];
		var diff = vec3.sub( vec3.create(), middle_pos, p );
		vec3.add( next_pos, middle_pos, diff );
	}
	else if(index % 3 == 1 && index > 3 )
	{
		var middle_pos = this.points[index - 1];
		var prev_pos = this.points[index - 2];
		var diff = vec3.sub( vec3.create(), middle_pos, p );
		vec3.add( prev_pos, middle_pos, diff );
	}
	else if( index % 3 == 0 )
	{
		if( index > 1 )
		{
			var prev_pos = this.points[index - 1];
			vec3.add( prev_pos, prev_pos, total_diff );
		}
		if( index < this.points.length - 1 )
		{
			var next_pos = this.points[index + 1];
			vec3.add( next_pos, next_pos, total_diff );
		}
	}
}

Path.prototype.computePoint = function(f, out)
{
	switch(this.type)
	{
		case LS.HERMITE: return this.getHermitePoint(f,out); break;
		case LS.BEZIER: return this.getBezierPoint(f,out); break;
		case LS.LINEAR: 
		default:
			return this.getLinearPoint(f,out);
			break;
	}
	//throw("Impossible path type");
}


Path.prototype.getLinearPoint = function(f, out)
{
	out = out || vec3.create();
	var num = this.points.length;
	var l = num;
	if(l < 2)
		return out;

	if(f <= 0)
		return vec3.copy(out, this.points[0]);
	if(f >= 1)
	{
		if(this.closed)
			return vec3.copy(out, this.points[0]);
		return vec3.copy(out, this.points[l-1]);
	}

	if( this.closed )
		l += 1;

	var v = ((l-1) * f);
	var i = v|0;
	var fract = v-i;
	var p = this.points[ i % num ];
	var p2 = this.points[ (i+1) % num ];
	return vec3.lerp(out, p, p2, fract);
}

Path.temp_vec3a = vec3.create();
Path.temp_vec3b = vec3.create();
Path.temp_vec3c = vec3.create();

Path.prototype.getBezierPoint = function(f, out)
{
	out = out || vec3.create();
	var l = this.points.length;
	if(l < 4)
		return out;
	l = (((l-1)/3)|0) * 3 + 1; //take only useful points

	if(f <= 0)
		return vec3.copy(out, this.points[0]);
	if(f >= 1)
		return vec3.copy(out, this.points[ this.closed ? 0 : l-1 ]);

	var num = (l-1)/3 + (this.closed ? 1 : 0); //num segment
	var v = num*f; //id.weight
	var i = (v|0); //id
	var t = v-i;//weight

	var i1 = (i*3);
	var i2 = (i*3+1);
	var i3 = (i*3+2);
	var i4 = (i*3+3);

	var p,p1,p2,p3;

	if( this.closed && i == num-1 )
	{
		p = this.points[l-1];
		p3 = this.points[0];
		var diff = vec3.sub( Path.temp_vec3c, p, this.points[l-2] );
		p1 = vec3.add( Path.temp_vec3a, p, diff );
		diff = vec3.sub( Path.temp_vec3c, p3, this.points[1] );
		p2 = vec3.add( Path.temp_vec3b, p3, diff );
	}
	else
	{
		p = this.points[ i1 ];
		p1 = this.points[ i2 ];
		p2 = this.points[ i3 ];
		p3 = this.points[ i4 ];
	}

	var b1 = (1-t)*(1-t)*(1-t);
	var b2 = 3*t*(1-t)*(1-t);
	var b3 = 3*t*t*(1-t);
	var b4 = t*t*t;

	out[0] = p[0] * b1 + p1[0] * b2 + p2[0] * b3 + p3[0] * b4;
	out[1] = p[1] * b1 + p1[1] * b2 + p2[1] * b3 + p3[1] * b4;
	out[2] = p[2] * b1 + p1[2] * b2 + p2[2] * b3 + p3[2] * b4;
	return out;
}

Path.prototype.getHermitePoint = function(f, out)
{
	out = out || vec3.create();
	var l = this.points.length;
	if(l < 2)
		return out;
	if(f <= 0)
		return vec3.copy(out, this.points[0]);
	if(f >= 1)
		return vec3.copy(out, this.points[ this.closed ? 0 : l-1]);

	var num = (l-1) + (this.closed ? 1 : 0); //num segments
	var v = num*f; //id.weight
	var i = (v|0); //id
	var t = v-i;//weight

	var pre_p0 = this.points[i - 1];
	var p0 = this.points[ i ];
	var p1 = this.points[ i+1 ];
	var post_p1 = this.points[ i+2 ];

	if(!pre_p0)
		pre_p0 = this.closed ? this.points[l - 1] : p0;
	if(!p1)
		p1 = this.points[ (i+1) % l ];
	if(!post_p1)
		post_p1 = this.closed ? this.points[ (i+2) % l ] : p1;

	Animation.EvaluateHermiteSplineVector( p0, p1, pre_p0, post_p1, t, out );
	return out;
}


/*
Path.prototype.getCatmullPoint = function(f, out)
{
	out = out || vec3.create();
	var l = this.points.length;
	if(l < 4)
		return out;
	l = (((l-1)/3)|0) * 3 + 1; //take only useful points
	if(f <= 0)
		return vec3.copy(out, this.points[0]);
	if(f >= 1)
		return vec3.copy(out, this.points[l-1]);

	var v = ((l-1)/3*f); 
	var i = v|0;//spline number
	var fract = v-i;//weight
	var p = this.points[ i ];
	var p1 = this.points[ i+1 ];
	var p2 = this.points[ i+2 ];
	var p3 = this.points[ i+3 ];
	var w = fract;
	var w2 = w*w;
	var w3 = w2*w;
	out[0] = Path.interpolate( p[0], p1[0], p2[0], p3[0], w,w2,w3 );
	out[1] = Path.interpolate( p[1], p1[1], p2[1], p3[1], w,w2,w3 );
	out[2] = Path.interpolate( p[2], p1[2], p2[2], p3[2], w,w2,w3 );
	return out;
}

//catmull-rom
Path.interpolate = function ( p0, p1, p2, p3, t, t2, t3 ) {
	var v0 = ( p2 - p0 ) * 0.5;
	var v1 = ( p3 - p1 ) * 0.5;
	return ( 2 * ( p1 - p2 ) + v0 + v1 ) * t3 + ( - 3 * ( p1 - p2 ) - 2 * v0 - v1 ) * t2 + v0 * t + p1;
};
*/

Path.prototype.samplePoints = function( n, out )
{
	if(n <= 0)
	{
		var segments = this.getSegments();
		if(this.type == LS.LINEAR)
			n = segments + 1;
		else
			n = segments * 20;
	}

	out = out || Array(n);
	out.length = n;

	for(var i = 0; i < n; i++)
		out[i] = this.computePoint(i/(n-1));
	return out;
}

Path.prototype.samplePointsTyped = function( n, out )
{
	if(out && out.length < (n * 3))
		n = Math.floor(out.length / 3);

	if(n <= 0)
	{
		var segments = this.getSegments();
		if(this.type == LS.LINEAR)
			n = segments + 1;
		else
			n = segments * 20;
	}

	out = out || new Float32Array( n * 3 );
	for(var i = 0; i < n; i++)
		this.computePoint(i/(n-1),out.subarray(i*3,i*3+3));
	return out;
}


Path.prototype.serialize = function()
{
	var o = {};
	var points = Array( this.points.length * 3 );
	for(var i = 0; i < this.points.length; i++)
	{
		var p = this.points[i];
		points[i*3] = p[0];
		points[i*3+1] = p[1];
		points[i*3+2] = p[2];
	}

	o.points = points;
	o.type = this.type;
	o.closed = this.closed;
	return o;
}

Path.prototype.configure = function(o)
{
	this.type = o.type;
	this.closed = o.closed;

	if(o.points)
	{
		this.points.length = o.points.length / 3;
		var points = o.points;
		for(var i = 0; i < this.points.length; i++)
			this.points[i] = vec3.fromValues( points[i*3], points[i*3+1], points[i*3+2] );
	}
}


LS.Path = Path;
///@FILE:../src/helpers/FXstack.js
///@INFO: UNCOMMON
/** FXStack
* Helps apply a stack of FXs to a texture with as fewer render calls as possible with low memory footprint
* Used by CameraFX and FrameFX but also available for any other use
* You can add new FX to the FX pool if you want.
* @class FXStack
*/
function FXStack( o )
{
	this.apply_fxaa = false;
	this.filter = true;
	this.fx = [];

	this._uniforms = { u_aspect: 1, u_viewport: vec2.create(), u_iviewport: vec2.create(), u_texture: 0, u_depth_texture: 1, u_random: vec2.create() };

	this._passes = null;
	this._must_update_passes = true;

	if(o)
		this.configure(o);
}

FXStack.available_fx = {
	"brightness_contrast": {
		name: "Brightness & Contrast",
		uniforms: {
			brightness: { name: "u_brightness", type: "float", value: 1, step: 0.01 },
			contrast: { name: "u_contrast", type: "float", value: 1, step: 0.01 }
		},
		code:"color.xyz = (color.xyz * u_brightness@ - vec3(0.5)) * u_contrast@ + vec3(0.5);"
	},
	"hue_saturation": {
		name: "Hue & Saturation",
		functions: ["HSV"],
		uniforms: {
			hue: { name: "u_hue", type: "float", value: 0, step: 0.01 },
			saturation: { name: "u_saturation", type: "float", value: 1, step: 0.01 },
			brightness: { name: "u_brightness", type: "float", value: 0, step: 0.01 }
		},
		code:"color.xyz = rgb2hsv(color.xyz); color.xz += vec2(u_hue@,u_brightness@); color.y *= u_saturation@; color.xyz = hsv2rgb(color.xyz);"
	},
	"invert": {
		name: "Invert color",
		code:"color.xyz = vec3(1.0) - color.xyz;"
	},
	"threshold": {
		name: "Threshold",
		uniforms: {
			threshold: { name: "u_threshold", type: "float", value: 0.5, min: 0, max: 2, step: 0.01 },
			threshold_width: { name: "u_threshold_width", type: "float", value: 0.01, min: 0, max: 1, step: 0.001 }
		},
		code:"color.xyz = vec3( smoothstep( u_threshold@ - u_threshold_width@ * 0.5, u_threshold@ + u_threshold_width@ * 0.5,  length(color.xyz) ));"
	},
	"colorize": {
		name: "Colorize",
		uniforms: {
			colorize: { name: "u_colorize", type: "color3", value: [1,1,1] },
			vibrance: { name: "u_vibrance", type: "float", value: 0.0, min: 0, max: 2, step: 0.01 }
		},
		code:"color.xyz = color.xyz * (u_colorize@ + vec3(u_vibrance@ * 0.1)) * (1.0 + u_vibrance@);"
	},
	"color_add": {
		name: "Color add",
		uniforms: {
			color_add: { name: "u_coloradd", type: "color3", value: [0.1,0.1,0.1] }
		},
		code:"color.xyz = color.xyz + u_coloradd@;"
	},
	"fog":{
		name:"fog",
		uniforms: {
			fog_color: { name: "u_fog_color", type: "color3", value: [0.1,0.1,0.1] },
			fog_start: { name: "u_fog_start", type: "float", value: 10 },
			fog_density: { name: "u_fog_density", type: "float", precision: 0.00001, value: 0.001, step: 0.00001 }
		},
		code:"float z_n@ = 2.0 * texture2D( u_depth_texture, v_coord).x - 1.0;" +
			"float cam_dist@ = 2.0 * u_depth_range.x * u_depth_range.y / (u_depth_range.y + u_depth_range.x - z_n@ * (u_depth_range.y - u_depth_range.x));" +
			"float fog_factor@ = 1. - 1.0 / exp(max(0.0,cam_dist@ - u_fog_start@) * u_fog_density@);" +
			"color.xyz = mix( color.xyz, u_fog_color@, fog_factor@ );"
	},
	"vigneting": {
		name: "Vigneting",
		uniforms: {
			radius: { name: "u_radius", type: "float", value: 1 },
			intensity: { name: "u_vigneting", type: "float", value: 1, min: 0, max: 2, step: 0.01 }
		},
		code:"color.xyz = mix( color.xyz * max( 1.0 - (dist_to_center * u_radius@ / 0.7071), 0.0), color.xyz, u_vigneting@);"
	},
	"aberration": {
		name: "Chromatic Aberration",
		break_pass: true,
		uniforms: {
			difraction: { name: "u_difraction", type: "float", value: 1 }
		},
		code: "color.x = texture2D(u_texture, uv - to_center * 0.001 * u_difraction@ ).x;" + 
			"color.z = texture2D(u_texture, uv + to_center * 0.001 * u_difraction@ ).z;"
	},
	"halftone": {
		name: "Halftone",
		uniforms: {
			"Halftone angle": { name: "u_halftone_angle", type: "float", value: 0, step: 0.01 },
			"Halftone size": { name: "u_halftone_size", type: "float", value: 1, step: 0.01 }
		},
		functions: ["pattern"],
		code:"color.x = ( (color.x * 10.0 - 5.0) + pattern( u_halftone_angle@, u_halftone_size@ ) );" + 
			"color.y = ( (color.y * 10.0 - 5.0) + pattern( u_halftone_angle@ + 0.167, u_halftone_size@ ) );" + 
			"color.z = ( (color.z * 10.0 - 5.0) + pattern( u_halftone_angle@ + 0.333, u_halftone_size@ ) );"
	},
	"halftoneBN": {
		name: "Halftone B/N",
		uniforms: {
			"Halftone angle": { name: "u_halftone_angle", type: "float", value: 0, step: 0.01 },
			"Halftone size": { name: "u_halftone_size", type: "float", value: 1, step: 0.01 }
		},
		functions: ["pattern"],
		code:"color.xyz = vec3( (length(color.xyz) * 10.0 - 5.0) + pattern( u_halftone_angle@, u_halftone_size@ ) );"
	},
	"lens": {
		name: "Lens Distortion",
		break_pass: true,
		uniforms: {
			lens_k: { name: "u_lens_k", type: "float", value: -0.15 },
			lens_kcube: { name: "u_lens_kcube", type: "float", value: 0.8 },
			lens_scale: { name: "u_lens_scale", type: "float", value: 1 }
		},
		uv_code:"float r2 = u_aspect * u_aspect * (uv.x-0.5) * (uv.x-0.5) + (uv.y-0.5) * (uv.y-0.5); float distort@ = 1. + r2 * (u_lens_k@ + u_lens_kcube@ * sqrt(r2)); uv = vec2( u_lens_scale@ * distort@ * (uv.x-0.5) + 0.5, u_lens_scale@  * distort@ * (uv.y-0.5) + 0.5 );"
	},
	"image": {
		name: "Image",
		uniforms: {
			image_texture: { name: "u_image_texture", type: "sampler2D", widget: "Texture", value: "" },
			image_alpha: { name: "u_image_alpha", type: "float", value: 1, step: 0.001 },
			image_scale: { name: "u_image_scale", type: "vec2", value: [1,1], step: 0.001 }
		},
		code:"vec4 image@ = texture2D( u_image_texture@, (uv - vec2(0.5)) * u_image_scale@ + vec2(0.5)); color.xyz = mix(color.xyz, image@.xyz, image@.a * u_image_alpha@ );"
	},
	"warp": {
		name: "Warp",
		break_pass: true,
		uniforms: {
			warp_amp: { name: "u_warp_amp", type: "float", value: 0.01, step: 0.001 },
			warp_offset: { name: "u_warp_offset", type: "vec2", value: [0,0], step: 0.001 },
			warp_scale: { name: "u_warp_scale", type: "vec2", value: [1,1], step: 0.001 },
			warp_texture: { name: "u_warp_texture", type: "sampler2D", widget: "Texture", value: "" }
		},
		uv_code:"uv = uv + u_warp_amp@ * (texture2D( u_warp_texture@, uv * u_warp_scale@ + u_warp_offset@ ).xy - vec2(0.5));"
	},
	"LUT": {
		name: "LUT",
		functions: ["LUT"],
		uniforms: {
			lut_intensity: { name: "u_lut_intensity", type: "float", value: 1, step: 0.01 },
			lut_texture: { name: "u_lut_texture", type: "sampler2D", filter: "nearest", wrap: "clamp", widget: "Texture", value: "" }
		},
		code:"color.xyz = mix(color.xyz, LUT( color.xyz, u_lut_texture@ ), u_lut_intensity@);"
	},
	"pixelate": {
		name: "Pixelate",
		uniforms: {
			width: { name: "u_width", type: "float", value: 256, step: 1, min: 1 },
			height: { name: "u_height", type: "float", value: 256, step: 1, min: 1 }
		},
		uv_code:"uv = vec2( floor(uv.x * u_width@) / u_width@, floor(uv.y * u_height@) / u_height@ );"
	},
	"quantize": {
		name: "Quantize",
		functions: ["dither"],
		uniforms: {
			levels: { name: "u_levels", type: "float", value: 8, step: 1, min: 1 },
			dither: { name: "u_dither", type: "float", value: 0.1, max: 1 }
		},
		code:"\n\
		if( u_dither@ > 0.0 )\n\
		{\n\
			vec3 qcolor@ = floor(color.xyz * u_levels@) / u_levels@;\n\
			vec3 diff@ = (color.xyz - qcolor@) * u_levels@ * u_dither@;\n\
			color.xyz = qcolor@ + vec3(dither(diff@.x),dither(diff@.y),dither(diff@.z)) / u_levels@;\n\
		}\n\
		else\n\
			color.xyz = floor(color.xyz * u_levels@) / u_levels@;\n"
	},
	"edges": {
		name: "Edges",
		break_pass: true,
		uniforms: {
			"Edges factor": { name: "u_edges_factor", type: "float", value: 1 }
		},
		code:"vec4 color@ = texture2D(u_texture, uv );\n\
				vec4 color_up@ = texture2D(u_texture, uv + vec2(0., u_iviewport.y));\n\
				vec4 color_right@ = texture2D(u_texture, uv + vec2(u_iviewport.x,0.));\n\
				vec4 color_down@ = texture2D(u_texture, uv + vec2(0., -u_iviewport.y));\n\
				vec4 color_left@ = texture2D(u_texture, uv + vec2(-u_iviewport.x,0.));\n\
				color = u_edges_factor@ * (abs(color@ - color_up@) + abs(color@ - color_down@) + abs(color@ - color_left@) + abs(color@ - color_right@));"
	},
	"depth": {
		name: "Depth",
		uniforms: {
			"near": { name: "u_near", type: "float", value: 0.01, step: 0.1 },
			"far": { name: "u_far", type: "float", value: 1000, step: 1 }
		},
		code:"color.xyz = vec3( (2.0 * u_near@) / (u_far@ + u_near@ - texture2D( u_depth_texture, uv ).x * (u_far@ - u_near@)) );"
	},
	"logarithmic": {
		name: "Logarithmic",
		uniforms: {
			"Log. A Factor": { name: "u_logfactor_a", type: "float", value: 2, step: 0.01 },
			"Log. B Factor": { name: "u_logfactor_b", type: "float", value: 2, step: 0.01 }
		},
		code:"color.xyz = log( color.xyz * u_logfactor_a@ ) * u_logfactor_b@;"
	},
	"ditherBN": {
		name: "dither B/N",
		functions: ["dither"],
		code:"color.xyz = vec3( dither( color.x ) );"
	},
	"dither": {
		name: "Dither",
		functions: ["dither"],
		code:"color.xyz = vec3( dither( color.x ), dither( color.y ), dither( color.z ) );"
	},
	"gamma": {
		name: "Gamma",
		uniforms: {
			"Gamma": { name: "u_gamma", type: "float", value: 2.2, step: 0.01 }
		},
		code:"color.xyz = pow( color.xyz, vec3( 1.0 / u_gamma@) );"
	},
	"noiseBN": {
		name: "Noise B&N",
		functions: ["noise"],
		uniforms: {
			"noise": { name: "u_noise", type: "float", value: 0.1, step: 0.01 }
		},
		code:"color.xyz += u_noise@ * vec3( noise( (u_random + v_coord) * u_viewport) );"
	}
	/*
	"blur": {
			name: "Blur",
			break_pass: true,
			uniforms: {
				"blur_intensity": { name: "u_blur_intensity", type: "float", value: 0.1, step: 0.01 }
			},
			local_callback: FXStack.applyBlur
		}
	}
	*/
	//median: https://github.com/patriciogonzalezvivo/flatLand/blob/master/bin/data/median.frag
};

//functions that could be used
FXStack.available_functions = {
	pattern: "float pattern(float angle, float size) {\n\
				float s = sin(angle * 3.1415), c = cos(angle * 3.1415);\n\
				vec2 tex = v_coord * u_viewport.xy;\n\
				vec2 point = vec2( c * tex.x - s * tex.y , s * tex.x + c * tex.y ) * size;\n\
				return (sin(point.x) * sin(point.y)) * 4.0;\n\
			}\n\
		",
	dither: "float dither(float v) {\n\
				vec2 pixel = v_coord * u_viewport;\n\
				int i = int(floor(clamp(v,0.0,1.0) * 16.0 + 0.5));\n\
				if(i < 1)\n\
					return 0.0;\n\
				if(i >= 15)\n\
					return 1.0;\n\
				float x = floor(pixel.x);\n\
				float y = floor(pixel.y);\n\
				bool xmod4 = mod(x, 4.0) == 0.0;\n\
				bool ymod4 = mod(y, 4.0) == 0.0;\n\
				bool xmod2 = mod(x, 2.0) == 0.0;\n\
				bool ymod2 = mod(y, 2.0) == 0.0;\n\
				bool xmod4_2 = mod(x + 2.0, 4.0) == 0.0;\n\
				bool ymod4_2 = mod(y + 2.0, 4.0) == 0.0;\n\
				bool xmod2_1 = mod(x + 1.0, 2.0) == 0.0;\n\
				bool ymod2_1 = mod(y + 1.0, 2.0) == 0.0;\n\
				bool xmod4_1 = mod(x + 1.0, 4.0) == 0.0;\n\
				bool ymod4_1 = mod(y + 1.0, 4.0) == 0.0;\n\
				bool xmod4_3 = mod(x + 3.0, 4.0) == 0.0;\n\
				bool ymod4_3 = mod(y + 3.0, 4.0) == 0.0;\n\
				\n\
				if(i < 9)\n\
				{\n\
					if(i >= 1 && xmod4 && ymod4 )\n\
						return 1.0;\n\
					if(i >= 2 && xmod4_2 && ymod4_2)\n\
						return 1.0;\n\
					if(i >= 3 && xmod4_2 && ymod2 )\n\
						return 1.0;\n\
					if(i >= 4 && xmod2 && ymod2 )\n\
						return 1.0;\n\
					if(i >= 5 && xmod4_1 && ymod4_1 )\n\
						return 1.0;\n\
					if(i >= 6 && xmod4_3 && ymod4_3 )\n\
						return 1.0;\n\
					if(i >= 7 && xmod4_1 && ymod4_3 )\n\
						return 1.0;\n\
					if(i >= 8 && xmod4_3 && ymod4_1 )\n\
						return 1.0;\n\
					return 0.0;\n\
				}\n\
				else\n\
				{\n\
					if(i < 15 && xmod4_1 && ymod4 )\n\
						return 0.0;\n\
					if(i < 14 && xmod4_3 && ymod4_2)\n\
						return 0.0;\n\
					if(i < 13 && xmod4_3 && ymod2 )\n\
						return 0.0;\n\
					if(i < 12 && xmod2_1 && ymod2 )\n\
						return 0.0;\n\
					if(i < 11 && xmod4_2 && ymod4_1 )\n\
						return 0.0;\n\
					if(i < 10 && xmod4 && ymod4_3 )\n\
						return 0.0;\n\
					return 1.0;\n\
				}\n\
			}\n\
		",
	//ugly but effective: https://github.com/hughsk/glsl-dither/blob/master/8x8.glsl
	dither8x8: "\n\
		float dither8x8(vec2 position, float brightness) {\n\
		  int x = int(mod(position.x, 8.0));\n\
		  int y = int(mod(position.y, 8.0));\n\
		  int index = x + y * 8;\n\
		  float limit = 0.0;\n\
		  if (x < 8) {\n\
			if (index == 0) limit = 0.015625;\n\
			else if (index == 1) limit = 0.515625;\n\
			else if (index == 2) limit = 0.140625;\n\
			else if (index == 3) limit = 0.640625;\n\
			else if (index == 4) limit = 0.046875;\n\
			else if (index == 5) limit = 0.546875;\n\
			else if (index == 6) limit = 0.171875;\n\
			else if (index == 7) limit = 0.671875;\n\
			else if (index == 8) limit = 0.765625;\n\
			else if (index == 9) limit = 0.265625;\n\
			else if (index == 10) limit = 0.890625;\n\
			else if (index == 11) limit = 0.390625;\n\
			else if (index == 12) limit = 0.796875;\n\
			else if (index == 13) limit = 0.296875;\n\
			else if (index == 14) limit = 0.921875;\n\
			else if (index == 15) limit = 0.421875;\n\
			else if (index == 16) limit = 0.203125;\n\
			else if (index == 17) limit = 0.703125;\n\
			else if (index == 18) limit = 0.078125;\n\
			else if (index == 19) limit = 0.578125;\n\
			else if (index == 20) limit = 0.234375;\n\
			else if (index == 21) limit = 0.734375;\n\
			else if (index == 22) limit = 0.109375;\n\
			else if (index == 23) limit = 0.609375;\n\
			else if (index == 24) limit = 0.953125;\n\
			else if (index == 25) limit = 0.453125;\n\
			else if (index == 26) limit = 0.828125;\n\
			else if (index == 27) limit = 0.328125;\n\
			else if (index == 28) limit = 0.984375;\n\
			else if (index == 29) limit = 0.484375;\n\
			else if (index == 30) limit = 0.859375;\n\
			else if (index == 31) limit = 0.359375;\n\
			else if (index == 32) limit = 0.0625;\n\
			else if (index == 33) limit = 0.5625;\n\
			else if (index == 34) limit = 0.1875;\n\
			else if (index == 35) limit = 0.6875;\n\
			else if (index == 36) limit = 0.03125;\n\
			else if (index == 37) limit = 0.53125;\n\
			else if (index == 38) limit = 0.15625;\n\
			else if (index == 39) limit = 0.65625;\n\
			else if (index == 40) limit = 0.8125;\n\
			else if (index == 41) limit = 0.3125;\n\
			else if (index == 42) limit = 0.9375;\n\
			else if (index == 43) limit = 0.4375;\n\
			else if (index == 44) limit = 0.78125;\n\
			else if (index == 45) limit = 0.28125;\n\
			else if (index == 46) limit = 0.90625;\n\
			else if (index == 47) limit = 0.40625;\n\
			else if (index == 48) limit = 0.25;\n\
			else if (index == 49) limit = 0.75;\n\
			else if (index == 50) limit = 0.125;\n\
			else if (index == 51) limit = 0.625;\n\
			else if (index == 52) limit = 0.21875;\n\
			else if (index == 53) limit = 0.71875;\n\
			else if (index == 54) limit = 0.09375;\n\
			else if (index == 55) limit = 0.59375;\n\
			else if (index == 56) limit = 1.0;\n\
			else if (index == 57) limit = 0.5;\n\
			else if (index == 58) limit = 0.875;\n\
			else if (index == 59) limit = 0.375;\n\
			else if (index == 60) limit = 0.96875;\n\
			else if (index == 61) limit = 0.46875;\n\
			else if (index == 62) limit = 0.84375;\n\
			else if (index == 63) limit = 0.34375;\n\
		  }\n\
		  return brightness < limit ? 0.0 : 1.0;\n\
		}\n",

	LUT:  "vec3 LUT(in vec3 color, in sampler2D textureB) {\n\
		 lowp vec3 textureColor = clamp( color, vec3(0.0), vec3(1.0) );\n\
		 mediump float blueColor = textureColor.b * 63.0;\n\
		 mediump vec2 quad1;\n\
		 quad1.y = floor(floor(blueColor) / 8.0);\n\
		 quad1.x = floor(blueColor) - (quad1.y * 8.0);\n\
		 mediump vec2 quad2;\n\
		 quad2.y = floor(ceil(blueColor) / 8.0);\n\
		 quad2.x = ceil(blueColor) - (quad2.y * 8.0);\n\
		 highp vec2 texPos1;\n\
		 texPos1.x = (quad1.x * 0.125) + 0.5/512.0 + ((0.125 - 1.0/512.0) * textureColor.r);\n\
		 texPos1.y = 1.0 - ((quad1.y * 0.125) + 0.5/512.0 + ((0.125 - 1.0/512.0) * textureColor.g));\n\
		 highp vec2 texPos2;\n\
		 texPos2.x = (quad2.x * 0.125) + 0.5/512.0 + ((0.125 - 1.0/512.0) * textureColor.r);\n\
		 texPos2.y = 1.0 - ((quad2.y * 0.125) + 0.5/512.0 + ((0.125 - 1.0/512.0) * textureColor.g));\n\
		 lowp vec3 newColor1 = texture2D(textureB, texPos1).xyz;\n\
		 lowp vec3 newColor2 = texture2D(textureB, texPos2).xyz;\n\
		 lowp vec3 newColor = mix(newColor1, newColor2, fract(blueColor));\n\
		 return newColor.rgb;\n\
	 }",
	noise:  "\n\
		float hash(float n) { return fract(sin(n) * 1e4); }\n\
		float hash(vec2 p) { return fract(1e4 * sin(17.0 * p.x + p.y * 0.1) * (0.1 + abs(sin(p.y * 13.0 + p.x)))); }\n\
		float noise(float x) {\n\
			float i = floor(x);\n\
			float f = fract(x);\n\
			float u = f * f * (3.0 - 2.0 * f);\n\
			return mix(hash(i), hash(i + 1.0), u);\n\
		}\n\
		float noise(vec2 x) {\n\
			vec2 i = floor(x);\n\
			vec2 f = fract(x);\n\
			float a = hash(i);\n\
			float b = hash(i + vec2(1.0, 0.0));\n\
			float c = hash(i + vec2(0.0, 1.0));\n\
			float d = hash(i + vec2(1.0, 1.0));\n\
			vec2 u = f * f * (3.0 - 2.0 * f);\n\
			return mix(a, b, u.x) + (c - a) * u.y * (1.0 - u.x) + (d - b) * u.x * u.y;\n\
		}\n\
	",
	HSV: "vec3 rgb2hsv(vec3 c)\n\
		{\n\
			vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);\n\
			vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));\n\
			vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));\n\
			\n\
			float d = q.x - min(q.w, q.y);\n\
			float e = 1.0e-10;\n\
			return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);\n\
		}\n\
		\n\
		vec3 hsv2rgb(vec3 c)\n\
		{\n\
			vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);\n\
			vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);\n\
			return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);\n\
		}"
}

/**
* Returns the first component of this container that is of the same class
* @method configure
* @param {Object} o object with the configuration info from a previous serialization
*/
FXStack.prototype.configure = function(o)
{
	this.apply_fxaa = !!o.apply_fxaa;
	if(o.fx)
		this.fx = o.fx.concat();
	this._must_update_passes = true;
}

FXStack.prototype.serialize = FXStack.prototype.toJSON = function()
{
	return { 
		apply_fxaa: this.apply_fxaa,
		fx: this.fx.concat()
	};
}

FXStack.prototype.getResources = function(res)
{
	var fxs = this.fx;
	for(var i = 0; i < fxs.length; i++)
	{
		var fx = fxs[i];
		var fx_info = FXStack.available_fx[ fx.name ];
		if(!fx_info)
			continue;
		if(!fx_info.uniforms)
			continue;
		for(var j in fx_info.uniforms)
		{
			var uniform = fx_info.uniforms[j];
			if(uniform.type == "sampler2D" && fx[j])
				res[ fx[j] ] = GL.Texture;
		}
	}
	return res;
}

FXStack.prototype.onResourceRenamed = function(old_name, new_name, resource)
{
	var fxs = this.fx;
	for(var i = 0; i < fxs.length; i++)
	{
		var fx = fxs[i];
		var fx_info = FXStack.available_fx[ fx.name ];
		if(!fx_info)
			continue;
		if(!fx_info.uniforms)
			continue;
		for(var j in fx_info.uniforms)
		{
			var uniform = fx_info.uniforms[j];
			if(uniform.type == "sampler2D" && fx[j] == old_name )
				fx[j] = new_name;
		}
	}
}


//attach a new FX to the FX Stack
FXStack.prototype.addFX = function( name )
{
	if(!name)
		return;
	if( !FXStack.available_fx[ name ] )
	{
		console.warn( "FXStack not found: " + name );
		return;
	}
	this.fx.push({ name: name });
	this._must_update_passes = true;
}

//returns the Nth FX in the FX Stack
FXStack.prototype.getFX = function(index)
{
	return this.fx[ index ];
}

//rearranges an FX
FXStack.prototype.moveFX = function( fx, offset )
{
	offset = offset || -1;

	var index = this.fx.indexOf(fx);
	if( index == -1 )
		return;

	this.fx.splice(index,1);
	index += offset;


	if(index >= 0 && index < this.fx.length)
		this.fx.splice(index,0,fx);
	else
		this.fx.push(fx);
	this._must_update_passes = true;
}

//removes an FX from the FX stack
FXStack.prototype.removeFX = function( fx )
{
	for(var i = 0; i < this.fx.length; i++)
	{
		if(this.fx[i] !== fx)
			continue;

		this.fx.splice(i,1);
		this._must_update_passes = true;
		return;
	}
}

//extract the number of passes to do according to the fx enabled
FXStack.prototype.buildPasses = function()
{
	var fxs = this.fx;

	var passes = [];
	var current_pass = {
		fxs:[],
		uniforms:{},
		shader:null,
		first_fx_id: 0
	};

	var uv_code = "";
	var color_code = "";
	var uniforms_code = "";
	var included_functions = {};

	var is_first = true;

	var fx_id = 0;
	for(var i = 0; i < fxs.length; i++)
	{
		//the FX settings
		var fx = fxs[i];
		fx_id = i;

		//the FX definition
		var fx_info = FXStack.available_fx[ fx.name ];
		if(!fx_info)
			continue;

		//break this pass
		if( fx_info.break_pass && !is_first)
		{
			current_pass.uv_code = uv_code;
			current_pass.color_code = color_code;
			current_pass.uniforms_code = uniforms_code;
			current_pass.included_functions = included_functions;
			passes.push(current_pass);
			this.buildPassShader( current_pass );

			uv_code = "";
			color_code = "";
			uniforms_code = "";
			included_functions = {};

			current_pass = {
				fxs:[],
				uniforms:{},
				first_fx_id: fx_id
			};
			is_first = true;
		}
		else
			is_first = false;

		if(fx_info.functions)
			for(var z in fx_info.functions)
				included_functions[ fx_info.functions[z] ] = true;
		if( fx_info.code )
			color_code += fx_info.code.split("@").join( fx_id ) + ";\n";
		if( fx_info.uv_code )
			uv_code += fx_info.uv_code.split("@").join( fx_id ) + ";\n";

		if(fx_info.uniforms)
			for(var j in fx_info.uniforms)
			{
				var uniform = fx_info.uniforms[j];
				var varname = uniform.name + fx_id;
				uniforms_code += "uniform " + uniform.type + " " + varname + ";\n";
			}

		current_pass.fxs.push( fx );
	}

	if(!is_first)
	{
		current_pass.uv_code = uv_code;
		current_pass.color_code = color_code;
		current_pass.included_functions = included_functions;
		passes.push( current_pass );
		this.buildPassShader( current_pass );
	}

	this._passes = passes;
}

FXStack.prototype.buildPassShader = function( pass )
{
	var functions_code = "";
	for(var i in pass.included_functions)
	{
		var func = FXStack.available_functions[ i ];
		if(!func)
		{
			console.error("FXStack: Function not found: " + i);
			continue;
		}
		functions_code += func + "\n";
	}

	var fullcode = "\n\
		#extension GL_OES_standard_derivatives : enable\n\
		precision highp float;\n\
		#define color3 vec3\n\
		#define color4 vec4\n\
		uniform sampler2D u_texture;\n\
		uniform sampler2D u_depth_texture;\n\
		varying vec2 v_coord;\n\
		uniform vec2 u_viewport;\n\
		uniform vec2 u_iviewport;\n\
		uniform float u_aspect;\n\
		uniform vec2 u_depth_range;\n\
		uniform vec2 u_random;\n\
		vec2 uv;\n\
		" + pass.uniforms_code + "\n\
		" + functions_code + "\n\
		void main() {\n\
			uv = v_coord;\n\
			vec2 to_center = vec2(0.5) - uv;\n\
			float dist_to_center = length(to_center);\n\
			" + pass.uv_code + "\n\
			vec4 color = texture2D(u_texture, uv);\n\
			float temp = 0.0;\n\
			" + pass.color_code + "\n\
			gl_FragColor = color;\n\
		}\n\
		";

	this._must_update_passes = false;
	pass.shader = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, fullcode );
	return pass.shader;
}


FXStack.prototype.applyFX = function( input_texture, output_texture, options )
{
	var color_texture = input_texture;
	var depth_texture = options.depth_texture;

	var global_uniforms = this._uniforms;
	global_uniforms.u_viewport[0] = color_texture.width;
	global_uniforms.u_viewport[1] = color_texture.height;
	global_uniforms.u_iviewport[0] = 1 / color_texture.width;
	global_uniforms.u_iviewport[1] = 1 / color_texture.height;
	global_uniforms.u_aspect = color_texture.width / color_texture.height;
	global_uniforms.u_random[0] = Math.random();
	global_uniforms.u_random[1] = Math.random();

	if(!this._passes || this._must_update_passes )
		this.buildPasses();

	if(!this._passes.length)
	{
		if(output_texture)
			input_texture.copyTo( output_texture );
		else
		{
			var fxaa_shader = GL.Shader.getFXAAShader();
			fxaa_shader.setup();
			input_texture.toViewport( this.apply_fxaa ? fxaa_shader : null );
		}
		return;
	}

	var w = output_texture ? output_texture.width : input_texture.width;
	var h = output_texture ? output_texture.height : input_texture.height;

	var origin_texture = GL.Texture.getTemporary( w, h, { type: input_texture.type, format: input_texture.format } );
	var target_texture = GL.Texture.getTemporary( w, h, { type: input_texture.type, format: input_texture.format } );

	input_texture.copyTo( origin_texture );

	var fx_id = 0;
	for(var i = 0; i < this._passes.length; i++)
	{
		var pass = this._passes[i];
		var texture_slot = 2;
		var uniforms = pass.uniforms;

		//gather uniform values
		for(var j = 0; j < pass.fxs.length; ++j)
		{
			var fx = pass.fxs[j];
			fx_id = pass.first_fx_id + j;

			//the FX definition
			var fx_info = FXStack.available_fx[ fx.name ];
			if(!fx_info)
				continue;

			if(!fx_info.uniforms)
				continue;

			for(var k in fx_info.uniforms)
			{
				var uniform = fx_info.uniforms[k];
				var varname = uniform.name + fx_id;
				if(uniform.type == "sampler2D")
				{
					uniforms[ varname ] = texture_slot;
					var tex = this.getTexture( fx[k] );
					if(tex)
					{
						tex.bind( texture_slot );
						if(uniform.filter == "nearest")
						{
							gl.texParameteri( tex.texture_type, gl.TEXTURE_MAG_FILTER, gl.NEAREST );
							gl.texParameteri( tex.texture_type, gl.TEXTURE_MIN_FILTER, gl.NEAREST );
						}
						if(uniform.wrap == "clamp")
						{
							gl.texParameteri( tex.texture_type, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE );
							gl.texParameteri( tex.texture_type, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE );
						}
					}
					else
					{
						//bind something to avoid problems
						tex = LS.Renderer._missing_texture;
						if(tex)
							tex.bind( texture_slot );
					}
					texture_slot++;
				}
				else
					uniforms[ varname ] = fx[j] !== undefined ? fx[j] : uniform.value;
			}
		}

		//apply pass
		var shader = pass.shader;
		//error compiling shader
		if(!shader)
		{
			input_texture.toViewport(); //what about output_texture?
			break;
		}

		//set the depth texture for some FXs like fog or depth
		if(depth_texture && shader.hasUniform("u_depth_texture"))
		{
			depth_texture.bind(1);
			if(depth_texture.near_far_planes)
				uniforms.u_depth_range = depth_texture.near_far_planes;
		}

		//apply FX and accumulate in secondary texture ***************
		shader.uniforms( global_uniforms );
		origin_texture.copyTo( target_texture, shader, uniforms );

		//swap
		var tmp = origin_texture;
		origin_texture = target_texture;
		target_texture = tmp;
	}

	//to the screen or the output_texture
	var final_texture = target_texture;
	final_texture.setParameter( gl.TEXTURE_MAG_FILTER, this.filter ? gl.LINEAR : gl.NEAREST );
	final_texture.setParameter( gl.TEXTURE_MIN_FILTER, gl.LINEAR );

	gl.disable( gl.DEPTH_TEST );
	gl.disable( gl.BLEND );
	gl.disable( gl.CULL_FACE );

	//to screen
	if( this.apply_fxaa )
	{
		var fx_aa_shader = GL.Shader.getFXAAShader();
		fx_aa_shader.setup();
		if(!output_texture)
			final_texture.toViewport( fx_aa_shader );
		else
			final_texture.copyTo( output_texture, fx_aa_shader );
	}
	else
	{
		if(!output_texture)
			final_texture.toViewport();
		else
		{
			shader.uniforms( uniforms );
			final_texture.copyTo( output_texture, shader );
		}
	}

	//release textures back to the pool
	GL.Texture.releaseTemporary( origin_texture );
	GL.Texture.releaseTemporary( target_texture );
}


//executes the FX stack in the input texture and outputs the result in the output texture (or the screen)
FXStack.prototype.applyFX = function( input_texture, output_texture, options )
{
	var color_texture = input_texture;
	var depth_texture = options.depth_texture;

	var fxs = this.fx;

	var update_shader = this._must_update_passes;
	this._must_update_passes = false;

	var uniforms = this._uniforms;
	uniforms.u_viewport[0] = color_texture.width;
	uniforms.u_viewport[1] = color_texture.height;
	uniforms.u_iviewport[0] = 1 / color_texture.width;
	uniforms.u_iviewport[1] = 1 / color_texture.height;
	uniforms.u_aspect = color_texture.width / color_texture.height;
	uniforms.u_random[0] = Math.random();
	uniforms.u_random[1] = Math.random();

	var uv_code = "";
	var color_code = "";
	var included_functions = {};
	var uniforms_code = "";
	var texture_slot = 2;

	var fx_id = 0;
	for(var i = 0; i < fxs.length; i++)
	{
		//the FX settings
		var fx = fxs[i];
		fx_id = i;

		//the FX definition
		var fx_info = FXStack.available_fx[ fx.name ];
		if(!fx_info)
			continue;

		if(update_shader)
		{
			if(fx_info.functions)
				for(var z in fx_info.functions)
					included_functions[ fx_info.functions[z] ] = true;
			if( fx_info.code )
				color_code += fx_info.code.split("@").join( fx_id ) + ";\n";
			if( fx_info.uv_code )
				uv_code += fx_info.uv_code.split("@").join( fx_id ) + ";\n";
		}

		if(fx_info.uniforms)
			for(var j in fx_info.uniforms)
			{
				var uniform = fx_info.uniforms[j];
				var varname = uniform.name + fx_id;
				if(update_shader)
				{
					uniforms_code += "uniform " + uniform.type + " " + varname + ";\n";
				}

				if(uniform.type == "sampler2D")
				{
					uniforms[ varname ] = texture_slot;
					var tex = this.getTexture( fx[j] );
					if(tex)
					{
						tex.bind( texture_slot );
						if(uniform.filter == "nearest")
						{
							gl.texParameteri( tex.texture_type, gl.TEXTURE_MAG_FILTER, gl.NEAREST );
							gl.texParameteri( tex.texture_type, gl.TEXTURE_MIN_FILTER, gl.NEAREST );
						}
						if(uniform.wrap == "clamp")
						{
							gl.texParameteri( tex.texture_type, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE );
							gl.texParameteri( tex.texture_type, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE );
						}
					}
					else
					{
						//bind something to avoid problems
						tex = LS.Renderer._missing_texture;
						if(tex)
							tex.bind( texture_slot );
					}

					texture_slot++;
				}
				else
					uniforms[ varname ] = fx[j] !== undefined ? fx[j] : uniform.value;
			}
	}


	var shader = null;
	if(update_shader)
	{
		var functions_code = "";
		for(var i in included_functions)
		{
			var func = FXStack.available_functions[ i ];
			if(!func)
			{
				console.error("FXStack: Function not found: " + i);
				continue;
			}
			functions_code += func + "\n";
		}

		var fullcode = "\n\
			#extension GL_OES_standard_derivatives : enable\n\
			precision highp float;\n\
			#define color3 vec3\n\
			#define color4 vec4\n\
			uniform sampler2D u_texture;\n\
			uniform sampler2D u_depth_texture;\n\
			varying vec2 v_coord;\n\
			uniform vec2 u_viewport;\n\
			uniform vec2 u_iviewport;\n\
			uniform float u_aspect;\n\
			uniform vec2 u_depth_range;\n\
			uniform vec2 u_random;\n\
			vec2 uv;\n\
			" + uniforms_code + "\n\
			" + functions_code + "\n\
			void main() {\n\
				uv = v_coord;\n\
				vec2 to_center = vec2(0.5) - uv;\n\
				float dist_to_center = length(to_center);\n\
				" + uv_code + "\n\
				vec4 color = texture2D(u_texture, uv);\n\
				float temp = 0.0;\n\
				" + color_code + "\n\
				gl_FragColor = color;\n\
			}\n\
			";

		this._last_shader = new GL.Shader( GL.Shader.SCREEN_VERTEX_SHADER, fullcode );
	}

	shader = this._last_shader;

	gl.disable( gl.DEPTH_TEST );
	gl.disable( gl.BLEND );
	gl.disable( gl.CULL_FACE );

	//error compiling shader
	if(!shader)
	{
		input_texture.toViewport();
		return;
	}

	//set the depth texture for some FXs like fog or depth
	if(shader.hasUniform("u_depth_texture") && depth_texture )
	{
		depth_texture.bind(1);
		if(depth_texture.near_far_planes)
			uniforms.u_depth_range = depth_texture.near_far_planes;
	}

	color_texture.setParameter( gl.TEXTURE_MAG_FILTER, this.filter ? gl.LINEAR : gl.NEAREST );
	color_texture.setParameter( gl.TEXTURE_MIN_FILTER, gl.LINEAR );

	if( this.apply_fxaa )
	{
		if(!this.temp_tex || this.temp_tex.width != gl.viewport_data[2] || this.temp_tex.height != gl.viewport_data[3])
			this.temp_tex = new GL.Texture(gl.viewport_data[2],gl.viewport_data[3]);
		this.temp_tex.drawTo(function(){
			color_texture.toViewport( shader, uniforms );
		});
		var fx_aa_shader = GL.Shader.getFXAAShader();
		fx_aa_shader.setup();

		if(!output_texture)
			this.temp_tex.toViewport( fx_aa_shader );
		else
			this.temp_tex.copyTo( output_texture, fx_aa_shader );
	}
	else
	{
		this.temp_tex = null;
		if(!output_texture)
			color_texture.toViewport( shader, uniforms );
		else
		{
			shader.uniforms( uniforms );
			color_texture.copyTo( output_texture, shader );
		}
	}
}

FXStack.prototype.getTexture = function( name )
{
	return LS.ResourcesManager.getTexture( name );
}

FXStack.prototype.getPropertyInfoFromPath = function( path )
{
	if(path.length < 2)
		return null;

	var fx_num = parseInt( path[0] );

	//fx not active
	if(fx_num >= this.fx.length)
		return null;
	var fx = this.fx[ fx_num ];

	var fx_info = FXStack.available_fx[ fx.name ];
	if(!fx_info)
		return null;

	var varname = path[1];
	if(varname == "name")
		return null;

	var uniform = fx_info.uniforms[ varname ];
	if(!uniform)
		return null;

	var type = uniform.type;

	if(type == "float")
		type = "number";
	else if(type == "sampler2D")
		type = "texture";

	return {
		target: fx,
		name: varname,
		value: fx[ varname ],
		type: uniform.type || "number"
	};
}

FXStack.prototype.setPropertyValueFromPath = function( path, value, offset )
{
	offset = offset || 0;

	if( path.length < (offset+1) )
		return null;

	var fx_num = parseInt( path[offset] );
	if(fx_num >= this.fx.length)
		return null;
	var fx = this.fx[ fx_num ];
	if(!fx)
		return null;
	
	var varname = path[offset+1];
	if (fx[ varname ] === undefined )
		return null;

	//to avoid incompatible types
	if( fx[ varname ] !== undefined && value !== undefined && fx[ varname ].constructor === value.constructor )
		fx[ varname ] = value;
}

//static method to register new FX in the system
FXStack.registerFX = function( name, fx_info )
{
	if( !fx_info.name )
		fx_info.name = name;
	if( fx_info.code === undefined )
		throw("FXStack must have a code");
	if( fx_info.uniforms && Object.keys( fx_info.uniforms ) && fx_info.code && fx_info.code.indexOf("@") == -1 )
		console.warn("FXStack using uniforms must use the character '@' at the end of every use to avoid collisions with other variables with the same name.");

	FXStack.available_fx[ name ] = fx_info;
}

//for common functions shared among different FXs...
FXStack.registerFunction = function( name, code )
{
	FXStack.available_functions[name] = code;
}

LS.FXStack = FXStack;
LS.TextureFX = FXStack; //LEGACY
///@FILE:../src/helpers/tween.js
///@INFO: UNCOMMON
/**
* Allows to launch tweening 
*
* @class Tween
* @namespace LS
* @constructor
*/

LS.Tween = {
	MAX_EASINGS: 256, //to avoid problems

	EASE_IN_QUAD: 1,
	EASE_OUT_QUAD: 2,
	EASE_IN_OUT_QUAD: 3,
	QUAD: 3,

	EASE_IN_CUBIC: 4,
	EASE_OUT_CUBIC: 5,
	EASE_IN_OUT_CUBIC: 6,
	CUBIC: 6,

	EASE_IN_QUART: 7,
	EASE_OUT_QUART: 8,
	EASE_IN_OUT_QUART: 9,
	QUART: 9,

	EASE_IN_SINE: 10,
	EASE_OUT_SINE: 11,
	EASE_IN_OUT_SINE: 12,
	SINE: 12,

	EASE_IN_EXPO: 13,
	EASE_OUT_EXPO: 14,
	EASE_IN_OUT_EXPO: 15,
	EXPO: 15,

	EASE_IN_BACK: 16,
	EASE_OUT_BACK: 17,
	EASE_IN_OUT_BACK: 18,
	BACK: 18,

	current_easings: [],
	_alife: [], //temporal array
	_temp: [], //another temporal

	reset: function()
	{
		this.current_easings = [];
		this._alife = [];
	},

	easeProperty: function( object, property, target, time, easing_function, on_complete, on_progress )
	{
		if( !object )
			throw("ease object cannot be null");
		if( target === undefined )
			throw("target value must be defined");
		if(object[property] === undefined)
			throw("property not found in object, must be initialized to a value");

		//cancel previous in case we already have one for this property
		if(this.current_easings.length)
		{
			for(var i = 0; i < this.current_easings.length; ++i)
			{
				var easing = this.current_easings[i];
				if( easing.object !== object || easing.property != property )
					continue;
				this.current_easings.splice(i,1); //remove old one
				break;		
			}
		}

		easing_function = easing_function || this.EASE_IN_OUT_QUAD;

		//clone to avoid problems
		var origin = null;
		
		if(property)
			origin = LS.cloneObject( object[ property ] );
		else
			origin = LS.cloneObject( object );
		target = LS.cloneObject( target );

		//precompute target value size
		var size = 0;
		if(target.constructor === Number)
			size = -1;
		else if(target && target.length !== undefined)
			size = target.length;

		var type = null;
		var type_info = object.constructor["@" + property];
		if( type_info )
			type = type_info.type;

		var data = { 
			object: object, 
			property: property, 
			origin: origin, 
			target: target, 
			current: 0, 
			time: time, 
			easing: easing_function, 
			on_complete: on_complete, 
			on_progress: on_progress, 
			size: size, 
			type: type,
			running: true
		};

		for(var i = 0; i < this.current_easings.length; ++i)
		{
			if( this.current_easings[i].object == object && this.current_easings[i].property == property )
			{
				this.current_easings[i] = data; //replace old easing
				break;
			}
		}

		if(this.current_easings.length >= this.MAX_EASINGS)
		{
			var easing = this.current_easings.shift();
			//TODO: this could be improved applyting the target value right now
		}

		this.current_easings.push( data );
		return data;
	},

	easeObject: function( object, target, time, easing_function, on_complete, on_progress )
	{
		if( !object || !target )
			throw("ease object cannot be null");

		easing_function = easing_function || this.EASE_IN_OUT_QUAD;

		//clone to avoid problems
		var origin = LS.cloneObject( object );
		target = LS.cloneObject( target );

		//precompute size
		var size = 0;
		if(target.length !== undefined)
			size = target.length;

		var data = { object: object, origin: origin, target: target, current: 0, time: time, easing: easing_function, on_complete: on_complete, on_progress: on_progress, size: size };

		for(var i = 0; i < this.current_easings.length; ++i)
		{
			if( this.current_easings[i].object == object )
			{
				this.current_easings[i] = data; //replace old easing
				break;
			}
		}

		if(this.current_easings.length >= this.MAX_EASINGS)
		{
			this.current_easings.shift();
		}

		this.current_easings.push( data );
		return data;
	},

	cancelEaseObject: function( object, property )
	{
		if( !this.current_easings.length )
			return;
		
		var easings = this.current_easings;
		for(var i = 0, l = easings.length; i < l; ++i)
		{
			var item = easings[i];
			if( item.object != object)
				continue;
			if( property && item.property != property)
				continue;
			item.cancel = true;
		}
	},

	//updates all the active tweens
	update: function( dt )
	{
		if( !this.current_easings.length )
			return;

		var easings = this.current_easings;
		this.current_easings = this._temp; //empty it to control incomming tweens during this update
		this.current_easings.length = 0;
		var alive = this._alife;
		alive.length = easings.length;
		var pos = 0;

		//for every pending easing method
		for(var i = 0, l = easings.length; i < l; ++i)
		{
			var item = easings[i];
			item.current += dt;
			var t = 1;

			if(item.cancel) //wont be added to the alive list
				continue;

			if(item.current < item.time)
			{
				t = item.current / item.time;
				alive[ pos ] = item;
				pos += 1;
			}

			var f = this.getEaseFactor( t, item.easing );

			var result = null;

			if(item.size)
			{
				if(item.size == -1) //number
					item.object[ item.property ] = item.target * f + item.origin * ( 1.0 - f );
				else //array
				{
					var property = item.object[ item.property ];

					if(item.type && item.type == "quat")
						quat.slerp( property, item.origin, item.target, f );
					else
					{
						//regular linear interpolation
						for(var j = 0; j < item.size; ++j)
							property[j] = item.target[j] * f + item.origin[j] * ( 1.0 - f );
					}
				}
				if(item.object.mustUpdate !== undefined)
					item.object.mustUpdate = true;
			}

			if(item.on_progress)
				item.on_progress( item );

			if(t >= 1)
			{
				if(item.on_complete)
					item.on_complete( item );
				item.running = false;
			}
		}

		alive.length = pos; //trim

		//add incomming tweens
		for(var i = 0; i < this.current_easings.length; ++i)
			alive.push( this.current_easings[i] );

		this.current_easings = alive;
		this._alife = easings;
	},

	getEaseFactor: function(t,type)
	{
		if(t>1) 
			t = 1;
		else if(t < 0)
			t = 0;
		var s = 1.70158;
		type = type || this.QUAD;
		switch(type)
		{
			case this.EASE_IN_QUAD: return (t*t);
			case this.EASE_OUT_QUAD: return 1-(t*t);
			case this.EASE_IN_OUT_QUAD: { 
				t *= 2;
				if( t < 1 ) return 0.5 * t * t;
				t -= 1;
				return -0.5 * ((t)*(t-2) - 1);
			};

			case this.EASE_IN_CUBIC: return t*t*t;
			case this.EASE_OUT_CUBIC: {
				t -= 1;
				return t*t*t + 1;
			};
			case this.EASE_IN_OUT_CUBIC: {
				t *= 2;
				if( t < 1 )
					return 0.5 * t*t*t;
				t -= 2;
				return 0.5*(t*t*t + 2);
			};

			case this.EASE_IN_QUART: return t*t*t*t;
			case this.EASE_OUT_QUART: {
				t -= 1;
				return -(t*t*t*t - 1);
			}
			case this.EASE_IN_OUT_QUART: {
				t *= 2;
				if( t < 1 ) return 0.5*t*t*t*t;
				else {
					t -= 2;
					return -0.5 * (t*t*t*t - 2);
				}
			}

			case this.EASE_IN_SINE:	return 1-Math.cos( t * Math.PI / 2 );
			case this.EASE_OUT_SINE:	return Math.sin( t * Math.PI / 2 );
			case this.EASE_IN_OUT_SINE: return -0.5 * ( Math.cos( Math.PI * t ) - 1 );

			case this.EASE_IN_EXPO: return t == 0 ? 0 : Math.pow( 2, 10 * (t - 1) );
			case this.EASE_OUT_EXPO: return t == 1 ? 1 : 1 - Math.pow( 2, -10 * t );
			case this.EASE_IN_OUT_EXPO: {
				if( t == 0 ) return 0;
				if( t == 1 ) return 1;
				t *= 2;
				if( t < 1 ) return 0.5 * Math.pow( 2, 10 * (t - 1) );
				return 0.5 * ( -Math.pow( 2, -10 * (t - 1)) + 2);
			}

			case this.EASE_IN_BACK: return t * t * ((s+1)*t - s);
			case this.EASE_OUT_BACK: return (t*t*((s+1)*t + s) + 1);
			case this.EASE_IN_OUT_BACK: {
				t *= 2;
				if( t < 1 ) {
					s *= 1.525;
					return 0.5*(t*t*((s+1)*t - s));
				}
				else {
					t -= 2;
					s *= 1.525;
					return 0.5*(t*t*((s+1)*t+ s) + 2);
				}
			};
		}
		return t;
	}
};

///@FILE:../src/helpers/animationBlender.js
///@INFO: UNCOMMON,ANIMATION

function AnimationBlender()
{
	this.active_entries = [];
}

AnimationBlender.prototype.addEntry = function( animation, take, time )
{
	if(animation.constructor === LS.AnimationBlender.Entry)
	{
		this.active_entries.push(animation);
		return;
	}

	var entry = new LS.AnimationBlender.Entry();
	entry.animation_name = animation;
	entry.take_name = take;
	entry.time = time;
	this.active_entries.push(entry);
	return entry;
}

AnimationBlender.prototype.removeEntry = function( entry )
{
	var index = this.active_entries.indexOf( entry );
	if(index != -1)
		this.active_entries.splice( index, 1 );
}

AnimationBlender.prototype.execute = function( root_node, scene )
{
	var tracks = {};

	//compute total weight (sum of all weights)
	var total_weight = 0;
	for(var i = 0; i < this.active_entries.length; ++i)
	{
		var entry = this.active_entries[i];
		if(!entry._animation_take)
			continue;
		total_weight += entry.weight;
		var take = entry._animation_take;
		for(var j = 0; j < take.tracks.length; ++j)
		{
			var track = take.tracks[j];
			var samples = tracks[ track.property ];
			if(!samples)
				samples = tracks[ track.property ] = [];
			var sample = track.getSample();
			samples.push( sample );
		}
	}

	//reverse weight system (hard to explain here...)
	for(var i = 0; i < this.active_entries.length; ++i)
	{
		var entry = this.active_entries[i];
		total_weight += entry.weight;
	}

	//
	for(var i = 0; i < this.active_entries.length; ++i)
	{
		var entry = this.active_entries[i];
		entry.execute( remaining / total_weight, false, root_node, scene );
		remaining -= entry.weight;
	}
}

LS.AnimationBlender = AnimationBlender;



function Entry()
{
	this.time = 0;
	this.weight = 1;

	this._animation_take = null; //pointer to the take

	this._animation_name = null;
	this._take_name = null;

	this._last_time = 0;
	this._must_update = false;
}

Object.defineProperty( Entry.prototype, "take_name", {
	set: function(v)
	{
		if( this._take_name == v )
			return;
		this._take_name = v;
		this._must_update = true;
	},
	get: function()
	{
		return this._take_name;
	},
	enumerable: false
});

Object.defineProperty( Entry.prototype, "animation_name", {
	set: function(v)
	{
		if( this._animation_name == v )
			return;
		this._animation_name = v;
		this._must_update = true;
	},
	get: function()
	{
		return this._animation_name;
	},
	enumerable: false
});

Object.defineProperty( Entry.prototype, "duration", {
	set: function(v)
	{
		throw("duration cannot be set. It depends in the animation duration");
	},
	get: function()
	{
		if(!this._animation_take)
			return -1;
		return this._animation_take.duration;
	},
	enumerable: false
});

Object.defineProperty( Entry.prototype, "loaded", {
	set: function(v)
	{
		throw("duration cannot be set. It depends in the animation duration");
	},
	get: function()
	{
		return !!this._animation_take; //to bool
	},
	enumerable: false
});


Entry.prototype.execute = function( final_weight, ignore_interpolation, root_node, scene )
{
	if( !this._animation_name || !this._take_name )
		return false;

	if( this._must_update )
	{
		var animation = LS.ResourcesManager.get( this._animation_name );
		if( !animation )
			return false;

		this._animation_take = animation.takes[ this._take_name ];
	}

	if(!this._animation_take)
		return;

	this._animation_take.applyTracks( this.time, this._last_time, ignore_interpolation, root_node, scene, final_weight );

	this._last_time = this.time;
	return true;
}


AnimationBlender.Entry = Entry;


///@FILE:../src/helpers/spatialContainer.js
///@INFO: BASE
//WORK IN PROGRESS

/** SpatialContainer
* This class allows to store data spatially so it can be retrieved by proximity or by camera frustum view.
* It can store objects associated with a bounding box.
* It is used by the renderer to store all the RenderInstances, Lights and Colliders
* IT IS A WORK IN PROGRESS SO FOR NOW IT DO NOT STORES THE INFO SPATIALLY, IT JUST EXPOSES AN INTERFACE
* @class SpatialContainer
*/

function SpatialContainer()
{
	this.size = 100000;
	this.root = [];
	this.objects_cell_by_id = new WeakMap();
}

//adds a new object to the container
SpatialContainer.prototype.add = function( object, bounding )
{
	var cell = this.root;

	cell.push( object );
	this.objects_cell_by_id.set( object, cell ); //in which container is
	return object.uid;
}

SpatialContainer.prototype.updateBounding = function( object, new_bounding )
{
	//...
}


//adds a new object to the container
SpatialContainer.prototype.remove = function( object, bounding )
{
	var cell = 	this.objects_cell_by_id.get( object );
	if(!cell)
		return;
	var index = cell.indexOf( object );
	if(index !== -1)
		cell.splice( index, 1 );
	this.objects_cell_by_id["delete"]( object );
}

//retrieves a list of objects overlaping this area
SpatialContainer.prototype.retrieve = function( bounding )
{
	//TODO: search cells that lay inside the bounding
	return this.root;
}

//retrieves a list of objects overlaping this area
SpatialContainer.prototype.retrieveFromCamera = function( camera )
{
	//TODO: search cells that lay inside the frustum
	return this.root;
}


SpatialContainer.prototype.clear = function()
{
	this.root.length = 0;
	this.objects_cell_by_id = new WeakMap();
}

LS.SpatialContainer = SpatialContainer;
///@FILE:../src/render/renderSettings.js
///@INFO: BASE
/** RenderSettings contains how the scene should be renderer 
* There could be different renderSettings for different scene quality.
* @class RenderSettings
* @constructor
**/

function RenderSettings( o )
{
	this.renderer_name = null; //null means default

	//global render settings
	this.default_shadowmap_resolution = LS.RenderSettings.default_shadowmap_resolution; //let the system decide best shadowmap resolution according to quality settings
	this.ignore_viewports = false;	//render to full viewport, ignoring the viewport in the cameras
	this.ignore_clear = false;	//skip global clear, used in case you want to mix LiteScene with another renderer
	this.keep_viewport = false; //do not force a full canvas viewport at render start (use the current one in WebGL as the full)
	//this.linear_depth = false; //forces depth to be stored lineally

	this.shadows_enabled = true; //allow shadowmaps
	this.update_shadowmaps = true; //automatically update shadowmaps in every frame (enable if there are dynamic objects)
	this.update_all_shadowmaps = false; //update shadowmaps even if they are not visible

	this.force_wireframe = false; //render everything in wireframe
	this.lights_disabled = false; //flat lighting
	this.quality = RenderSettings.AUTO_QUALITY;

	this.render_all_cameras = true; //render secundary cameras too
	this.render_fx = true; //postprocessing fx
	this.render_gui = true; //render gui
	this.render_helpers = true; //render helpers (for the editor)

	this.layers = 0xFFFF; //this is masked with the camera layers when rendering

	this.z_pass = false; //enable when the shaders are too complex (normalmaps, etc) to reduce work of the GPU (still some features missing)
	this.frustum_culling = true; //test bounding box by frustum to determine visibility

	//info
	this.in_player = true; //is in the player (not in the editor)

	if(o)
		this.configure(o);
}

RenderSettings.AUTO_QUALITY = 0;
RenderSettings.HIGH_QUALITY = 1;
RenderSettings.MEDIUM_QUALITY = 2;
RenderSettings.LOW_QUALITY = 3;

RenderSettings.default_shadowmap_resolution = 1024;

RenderSettings["@default_shadowmap_resolution"] = { widget: "combo", values: [0,256,512,1024,2048,4096] };
RenderSettings["@layers"] = { type: "layers" };

RenderSettings.prototype.serialize = function()
{
	return LS.cloneObject(this);
}

RenderSettings.prototype.configure = function(o)
{
	if(o)
		for(var i in o)
			this[i] = o[i];

	//legacy
	if(this.layers === null)
		this.layers = 0xFF;
}

RenderSettings.prototype.toJSON = RenderSettings.prototype.serialize;

LS.RenderSettings = RenderSettings;
///@FILE:../src/render/renderState.js
///@INFO: BASE
/**
* RenderState sets the flags for the GPU associated with a rendering action (blending, masking, depth test, etc)
* It is stored in the material (although defined usually from ShaderCode) so the material can use it.
*
* @class RenderState
* @namespace LS
* @constructor
*/

/* gpu flags

0: front_face: GL.CCW
1: cull_face: 1
2: cull_face_mode: GL.BACK

//depth buffer
4: depth_test: 1
5: depth_mask: 1 //write in depth buffer
6: depth_func: GL.LESS
7: depth_range0: 0
8: depth_range1: 1

//blend function
9: blend: 0;
10: blendFunc0: GL.SRC_ALPHA
11: blendFunc1: GL.ONE_MINUS_SRC_ALPHA

//color mask
12:	colorMask0: 1
13:	colorMask1: 1
14:	colorMask2: 1
15:	colorMask3: 1

//stencil buffer
16: stencil_test: 0
17:	stencil_mask: 0xFF,
18:	stencil_func_func: GL.ALWAYS,
19:	stencil_func_ref: 0,
20:	stencil_func_mask: 0xFF,
21:	stencil_op_sfail: GL.KEEP,
22:	stencil_op_dpfail: GL.KEEP,
23:	stencil_op_dppass: GL.KEEP

24: flags
*/

function RenderState( o )
{
	this._data = new Uint32Array(25);
	this.init();

	if(o)
		this.configure(o);
}

Object.defineProperty( RenderState.prototype, "front_face", {
	set: function(v) { this._data[0] = v; },
	get: function() { return this._data[0];	},
	enumerable: true
});

RenderState.SKIP_BLEND = 1;
RenderState.SKIP_DEPTH = 2;
RenderState.SKIP_STENCIL = 4;

RenderState["@front_face"] = { widget: "combo", values: { CW: GL.CW, CCW: GL.CCW } };

Object.defineProperty( RenderState.prototype, "cull_face", {
	set: function(v) { this._data[1] = v ? 1 : 0; },
	get: function() { return this._data[1] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "cull_face_mode", {
	set: function(v) { this._data[2] = v; },
	get: function() { return this._data[2];	},
	enumerable: true
});

RenderState["@cull_face_mode"] = { widget: "combo", values: { FRONT: GL.FRONT, BACK: GL.BACK, FRONT_AND_BACK: GL.FRONT_AND_BACK } };

Object.defineProperty( RenderState.prototype, "depth_test", {
	set: function(v) { this._data[4] = v ? 1 : 0; },
	get: function() { return this._data[4] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "depth_mask", {
	set: function(v) { this._data[5] = v ? 1 : 0; },
	get: function() { return this._data[5] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "depth_func", {
	set: function(v) { this._data[6] = v; },
	get: function() { return this._data[6];	},
	enumerable: true
});

RenderState["@depth_func"] = { widget: "combo", values: { LESS: GL.LESS, LEQUAL: GL.LEQUAL, EQUAL: GL.EQUAL, NOTEQUAL: GL.NOTEQUAL, GREATER: GL.GREATER, GEQUAL: GL.GEQUAL, ALWAYS: GL.ALWAYS, NEVER: GL.NEVER } };

Object.defineProperty( RenderState.prototype, "depth_range", {
	set: function(v) { 
		if(!v || v.length != 2)
			return;
		this._data[7] = v[0];
		this._data[8] = v[1];
	},
	get: function() { return this._data.subarray(7,9);	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "blend", {
	set: function(v) { this._data[9] = v ? 1 : 0; },
	get: function() { return this._data[9] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "blendFunc0", {
	set: function(v) { this._data[10] = v; },
	get: function() { return this._data[10];	},
	enumerable: true
});

RenderState["@blendFunc0"] = { widget: "combo", values: { ZERO: GL.ZERO, ONE: GL.ONE, SRC_COLOR: GL.SRC_COLOR, ONE_MINUS_SRC_COLOR: GL.ONE_MINUS_SRC_COLOR, DST_COLOR: GL.DST_COLOR, ONE_MINUS_DST_COLOR: GL.ONE_MINUS_DST_COLOR, SRC_ALPHA: GL.SRC_ALPHA, ONE_MINUS_SRC_ALPHA: GL.ONE_MINUS_SRC_ALPHA, DST_ALPHA: GL.DST_ALPHA, ONE_MINUS_DST_ALPHA: GL.ONE_MINUS_DST_ALPHA, CONSTANT_COLOR: GL.CONSTANT_COLOR, ONE_MINUS_CONSTANT_COLOR: GL.ONE_MINUS_CONSTANT_COLOR, CONSTANT_ALPHA: GL.CONSTANT_ALPHA, ONE_MINUS_CONSTANT_ALPHA: GL.ONE_MINUS_CONSTANT_ALPHA, SRC_ALPHA_SATURATE: GL.SRC_ALPHA_SATURATE } };

Object.defineProperty( RenderState.prototype, "blendFunc1", {
	set: function(v) { this._data[11] = v; },
	get: function() { return this._data[11];	},
	enumerable: true
});

RenderState["@blendFunc1"] = { widget: "combo", values: { ZERO: GL.ZERO, ONE: GL.ONE, SRC_COLOR: GL.SRC_COLOR, ONE_MINUS_SRC_COLOR: GL.ONE_MINUS_SRC_COLOR, DST_COLOR: GL.DST_COLOR, ONE_MINUS_DST_COLOR: GL.ONE_MINUS_DST_COLOR, SRC_ALPHA: GL.SRC_ALPHA, ONE_MINUS_SRC_ALPHA: GL.ONE_MINUS_SRC_ALPHA, DST_ALPHA: GL.DST_ALPHA, ONE_MINUS_DST_ALPHA: GL.ONE_MINUS_DST_ALPHA, CONSTANT_COLOR: GL.CONSTANT_COLOR, ONE_MINUS_CONSTANT_COLOR: GL.ONE_MINUS_CONSTANT_COLOR, CONSTANT_ALPHA: GL.CONSTANT_ALPHA, ONE_MINUS_CONSTANT_ALPHA: GL.ONE_MINUS_CONSTANT_ALPHA, SRC_ALPHA_SATURATE: GL.SRC_ALPHA_SATURATE } };

Object.defineProperty( RenderState.prototype, "blendFunc", {
	set: function(v)
	{
		if(!v || v.length != 2)
			return;
		this._data[10] = v[0];
		this._data[11] = v[1];
	},
	get: function()
	{
		return this._data.subarray(10,12);
	},
	enumerable: false
});

Object.defineProperty( RenderState.prototype, "colorMask0", {
	set: function(v) { this._data[12] = v ? 1 : 0; },
	get: function() { return this._data[12] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "colorMask1", {
	set: function(v) { this._data[13] = v ? 1 : 0; },
	get: function() { return this._data[13] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "colorMask2", {
	set: function(v) { this._data[14] = v ? 1 : 0; },
	get: function() { return this._data[14] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "colorMask3", {
	set: function(v) { this._data[15] = v ? 1 : 0; },
	get: function() { return this._data[15] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "colorMask", {
	set: function(v)
	{
		if(!v || v.length != 4)
			return;
		this._data[12] = v[0];
		this._data[13] = v[1];
		this._data[14] = v[2];
		this._data[15] = v[3];
	},
	get: function()
	{
		return this._data.subarray(12,16);
	},
	enumerable: false
});

/*
16: stencil_test: 0
17:	stencil_mask: 0xFF,
18:	stencil_func_func: GL.ALWAYS,
19:	stencil_func_ref: 0,
20:	stencil_func_mask: 0xFF,
21:	stencil_op_sfail: GL.KEEP,
22:	stencil_op_dpfail: GL.KEEP,
23:	stencil_op_dppass: GL.KEEP
*/

Object.defineProperty( RenderState.prototype, "stencil_test", {
	set: function(v) { this._data[16] = v ? 1 : 0; },
	get: function() { return this._data[16] !== 0;	},
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "stencil_mask", {
	set: function(v) { this._data[17] = v; },
	get: function() { return this._data[17]; },
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "skip_blend", {
	set: function(v) { this._data[25] = v ? (this._data[25] | RenderState.SKIP_BLEND) : (this._data[25] & ~(RenderState.SKIP_BLEND)); },
	get: function() { return Boolean(this._data[25] & RenderState.SKIP_BLEND); },
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "skip_depth", {
	set: function(v) { this._data[25] = v ? (this._data[25] | RenderState.SKIP_DEPTH) : (this._data[25] & ~(RenderState.SKIP_DEPTH)); },
	get: function() { return Boolean(this._data[25] & RenderState.SKIP_DEPTH); },
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "skip_stencil", {
	set: function(v) { this._data[25] = v ? (this._data[25] | RenderState.SKIP_STENCIL) : (this._data[25] & ~(RenderState.SKIP_STENCIL)); },
	get: function() { return Boolean(this._data[25] & RenderState.SKIP_STENCIL); },
	enumerable: true
});


RenderState["@stencil_mask"] = { widget: "number", min: 0, max: 256, step: 1, precision: 0 };

Object.defineProperty( RenderState.prototype, "stencil_func", {
	set: function(v) {
		if(!v || v.length != 3)
			return;
		this._data[18] = v[0];
		this._data[19] = v[1];
		this._data[20] = v[2];
	},
	get: function() { return this._data.subarray(18,21); },
	enumerable: false
});

Object.defineProperty( RenderState.prototype, "stencil_func_func", {
	set: function(v) { this._data[18] = v; },
	get: function() { return this._data[18]; },
	enumerable: true
});

RenderState["@stencil_func_func"] = { widget: "combo", values: { LESS: GL.LESS, LEQUAL: GL.LEQUAL, EQUAL: GL.EQUAL, NOTEQUAL: GL.NOTEQUAL, GREATER: GL.GREATER, GEQUAL: GL.GEQUAL, ALWAYS: GL.ALWAYS, NEVER: GL.NEVER } };

Object.defineProperty( RenderState.prototype, "stencil_func_ref", {
	set: function(v) { this._data[19] = v; },
	get: function() { return this._data[19]; },
	enumerable: true
});

RenderState["@stencil_func_ref"] = { widget: "number", min: 0, max: 256, step: 1, precision: 0 };

Object.defineProperty( RenderState.prototype, "stencil_func_mask", {
	set: function(v) { this._data[20] = v; },
	get: function() { return this._data[20]; },
	enumerable: true
});

RenderState["@stencil_func_mask"] = { widget: "number", min: 0, max: 256, step: 1, precision: 0 };

Object.defineProperty( RenderState.prototype, "stencil_op", {
	set: function(v) {
		if(!v || v.length != 3)
			return;
		this._data[21] = v[0];
		this._data[22] = v[1];
		this._data[23] = v[2];
	},
	get: function() { return this._data.subarray(21,24); },
	enumerable: false
});

Object.defineProperty( RenderState.prototype, "stencil_op_sfail", {
	set: function(v) { this._data[21] = v; },
	get: function() { return this._data[21]; },
	enumerable: true
});

RenderState["@stencil_op_sfail"] = { widget: "combo", values: { KEEP: GL.KEEP, ZERO: GL.ZERO, REPLACE: GL.REPLACE, INCR: GL.INCR, INCR_WRAP: GL.INCR_WRAP, DECR: GL.DECR_WRAP, INVERT: GL.INVERT } };

Object.defineProperty( RenderState.prototype, "stencil_op_dpfail", {
	set: function(v) { this._data[22] = v; },
	get: function() { return this._data[22]; },
	enumerable: true
});

RenderState["@stencil_op_dpfail"] = { widget: "combo", values: { KEEP: GL.KEEP, ZERO: GL.ZERO, REPLACE: GL.REPLACE, INCR: GL.INCR, INCR_WRAP: GL.INCR_WRAP, DECR: GL.DECR_WRAP, INVERT: GL.INVERT } };

Object.defineProperty( RenderState.prototype, "stencil_op_dppass", {
	set: function(v) { this._data[23] = v; },
	get: function() { return this._data[23]; },
	enumerable: true
});

Object.defineProperty( RenderState.prototype, "flags", {
	set: function(v) { this._data[24] = v; },
	get: function() { return this._data[24]; },
	enumerable: true
});

RenderState["@stencil_op_dppass"] = { widget: "combo", values: { KEEP: GL.KEEP, ZERO: GL.ZERO, REPLACE: GL.REPLACE, INCR: GL.INCR, INCR_WRAP: GL.INCR_WRAP, DECR: GL.DECR_WRAP, INVERT: GL.INVERT } };

RenderState.default_state = {
	front_face: GL.CCW,
	cull_face: true,
	depth_test: true,
	depth_func: GL.LESS,
	depth_mask: true,
	blend: false,
	blendFunc0: GL.SRC_ALPHA,
	blendFunc1: GL.ONE_MINUS_SRC_ALPHA,
	colorMask0: true,
	colorMask1: true,
	colorMask2: true,
	colorMask3: true,
	stencil_test: false,
	stencil_mask: 0xFF,
	stencil_func_func: GL.ALWAYS,
	stencil_func_ref: 0,
	stencil_func_mask: 0xFF,
	stencil_op_sfail: GL.KEEP,
	stencil_op_dpfail: GL.KEEP,
	stencil_op_dppass: GL.KEEP
};

RenderState.last_state = null;

RenderState.prototype.init = function()
{
	//gpu flags
	this.front_face = GL.CCW;
	this.cull_face = true;
	//this.cull_face_mode = GL.BACK;

	//depth buffer
	this.depth_test = true;
	this.depth_mask = true; //write in depth buffer
	this.depth_func = GL.LESS;
	//depth range: never used

	//blend function
	this.blend = false;
	this.blendFunc0 = GL.SRC_ALPHA;
	this.blendFunc1 = GL.ONE_MINUS_SRC_ALPHA;
	//blend equation

	//color mask
	this.colorMask0 = true;
	this.colorMask1 = true;
	this.colorMask2 = true;
	this.colorMask3 = true;

	//stencil buffer
	this.stencil_test = false;
	this.stencil_mask = 0xFF;
	this.stencil_func_func = GL.ALWAYS;
	this.stencil_func_ref = 0;
	this.stencil_func_mask = 0xFF;
	this.stencil_op_sfail = GL.KEEP;
	this.stencil_op_dpfail = GL.KEEP;
	this.stencil_op_dppass = GL.KEEP;

	this.flags = 0;
}

//helper, allows to set the blend mode from a string
RenderState.prototype.setBlendMode = function( mode )
{
	var functions = LS.BlendFunctions[ mode ];
	if(!mode || mode == LS.Blend.NORMAL )
	{
		this.blend = false;
		return;
	}

	this.blend = true;
	this.blendFunc0 = mode[0];
	this.blendFunc1 = mode[1];
}

RenderState.prototype.enable = function( render_settings )
{
	RenderState.enable( this, null, render_settings );
}

RenderState.enable = function( state, prev, render_settings )
{
	var flags = state.flags;

	var force_two_sided = render_settings && render_settings.force_two_sided ? true : false;

	if(!prev)
	{
		//faces
		if(LS.Renderer._reverse_faces)
			gl.frontFace( state.front_face == GL.CCW ? GL.CW : GL.CCW );
		else
			gl.frontFace( state.front_face );
		if(state.cull_face && !force_two_sided )
			gl.enable( gl.CULL_FACE );
		else
			gl.disable( gl.CULL_FACE );

		//depth
		if( !(flags & RenderState.SKIP_DEPTH) )
		{
			if(state.depth_test)
				gl.enable( gl.DEPTH_TEST );
			else
				gl.disable( gl.DEPTH_TEST );
			gl.depthMask( state.depth_mask );
			gl.depthFunc( state.depth_func );
		}

		//blend
		if( !(flags & RenderState.SKIP_BLEND) )
		{
			if(state.blend)
				gl.enable( gl.BLEND );
			else
				gl.disable( gl.BLEND );
			gl.blendFunc( state.blendFunc0, state.blendFunc1 );
		}

		//color
		gl.colorMask( state.colorMask0, state.colorMask1, state.colorMask2, state.colorMask3 );

		//stencil
		if( !(flags & RenderState.SKIP_STENCIL) )
		{
			if(state.stencil_test)
			{
				gl.enable( gl.STENCIL_TEST );
				gl.stencilFunc( state.stencil_func_func, state.stencil_func_ref, state.stencil_func_mask );
				gl.stencilOp( state.stencil_op_sfail, state.stencil_op_dpfail, state.stencil_op_dppass );
				gl.stencilMask( state.stencil_mask );
			}
			else
				gl.disable( gl.STENCIL_TEST );
		}

		this.last_state = state;
		return;
	}

	//***********************************************

	//faces
	if(LS.Renderer._reverse_faces)
		gl.frontFace( state.front_face == GL.CCW ? GL.CW : GL.CCW );
	else
	{
		if( prev.front_face !== state.front_face )
			gl.frontFace( state.front_face );
	}
	if( prev.cull_face !== state.cull_face || force_two_sided )
	{
		if( state.cull_face && !force_two_sided )
			gl.enable( gl.CULL_FACE );
		else
			gl.disable( gl.CULL_FACE );
	}

	//depth
	if( !(flags & RenderState.SKIP_DEPTH) )
	{
		if(prev.depth_test !== state.depth_test)
		{
			if(state.depth_test)
				gl.enable( gl.DEPTH_TEST );
			else
				gl.disable( gl.DEPTH_TEST );
		}
		if(prev.depth_mask !== state.depth_mask)
			gl.depthMask( state.depth_mask );
		if(prev.depth_func !== state.depth_func)
			gl.depthFunc( state.depth_func );
	}

	//blend
	if( !(flags & RenderState.SKIP_BLEND) )
	{
		if(prev.blend !== state.blend)
		{
			if(state.blend)
				gl.enable( gl.BLEND );
			else
				gl.disable( gl.BLEND );
		}
		if(prev.blendFunc0 !== state.blendFunc0 || prev.blendFunc1 !== state.blendFunc1)
			gl.blendFunc( state.blendFunc0, state.blendFunc1 );
	}

	//color
	if(prev.colorMask0 !== state.colorMask0 || prev.colorMask1 !== state.colorMask1 || prev.colorMask2 !== state.colorMask2 || prev.colorMask3 !== state.colorMask3 )
		gl.colorMask( state.colorMask0, state.colorMask1, state.colorMask2, state.colorMask3 );

	//stencil
	if( !(flags & RenderState.SKIP_STENCIL) )
	{
		if(prev.stencil_test != state.stencil_test )
		{
			if(state.stencil_test)
				gl.enable( gl.STENCIL_TEST);
			else
				gl.disable( gl.STENCIL_TEST );
		}

		if(state.stencil_test)
		{
			if( state.stencil_func_func !== prev.stencil_func_func || state.stencil_func_ref !== prev.stencil_func_ref || state.stencil_func_mask !== prev.stencil_func_mask )
				gl.stencilFunc( state.stencil_func_func, state.stencil_func_ref, state.stencil_func_mask );

			if(state.stencil_op_sfail !== prev.stencil_op_sfail || state.stencil_op_dpfail !== stencil_op_dpfail || state.stencil_op_dppass !== stencil_op_dppass )
				gl.stencilOp( state.stencil_op_sfail, state.stencil_op_dpfail, state.stencil_op_dppass );

			if(state.stencil_mask !== prev.stencil_mask)
				gl.stencilMask( prev.stencil_mask );
		}
	}

	//save state
	this.last_state = state;
}

RenderState.reset = function()
{
	this.enable( this.default_state );
}

RenderState.prototype.serialize = function()
{
	return LS.cloneObject(this);
}

RenderState.prototype.toJSON = RenderState.prototype.serialize;

RenderState.prototype.configure = function(o)
{
	LS.cloneObject(o,this);
}

RenderState.prototype.copyFrom = function( rs )
{
	this._data.set( rs._data );
}


LS.RenderState = RenderState;
///@FILE:../src/render/renderCall.js
///@INFO: UNCOMMON
//WIP: this is the lowest GPU rendering object, which encapsulates all about a render call
//by encapsulating every render action into an object we can have materials that produce several render passes in different moments
//of the rendering process
//the only problem is that uniform containrs could change between render calls which will lead to errors 

function RenderCall()
{
	this.shader = null;
	this.uniforms_containers = [];
	this.vertex_buffers = null;
	this.index_buffer = null;
	this.offset_start = -1;
	this.offset_range = -1;
	this.primitive = -1;

	this.renderState = null;
}

RenderCall.prototype.draw = function()
{
	this.renderState.enable();

	this.shader.uniforms( this.uniforms ).drawBuffers( this.vertex_buffers,
	  this.index_buffer,
	  this.primitive, this.offset_start, this.offset_range );
}

//Pool
RenderCall.pool = [];

RenderCall.get = function()
{
	if( RenderCall.pool.length > 0 )
		return RenderCall.pool.pop();
	return new RenderCall();
}

RenderCall.prototype.release = function()
{
	RenderCall.pool.push(this);
}

///@FILE:../src/render/renderInstance.js
///@INFO: BASE
/**
* RenderInstance contains info of one object to be rendered on the scene.
* It shouldnt contain ids to resources (strings), instead if must contain the direct reference (to mesh, material)
*
* @class RenderInstance
* @namespace LS
* @constructor
*/

function RenderInstance( node, component )
{
	this.uid = LS.generateUId("RINS"); //unique identifier for this RI
	this.layers = 3; //in layer 1 and 2 by default
	this.index = -1; //used to know the rendering order
	this.version = -1; //not used yet

	//info about the mesh
	this.vertex_buffers = {};
	this.index_buffer = null;
	this.wireframe_index_buffer = null;
	this.range = new Int32Array([0,-1]); //start, offset
	this.primitive = GL.TRIANGLES;

	this.transform = null; //parented transform: not finished

	this.mesh = null; //shouldnt be used (buffers are added manually), but just in case
	this.collision_mesh = null; //in case of raycast

	//where does it come from
	this.node = node;
	this.component = component;
	this.priority = 10; //only used if the RenderQueue is in PRIORITY MODE, instances are rendered from higher to lower priority
	this.sort_mode = RenderInstance.NO_SORT;

	//transformation
	this.matrix = mat4.create();
	this.normal_matrix = mat4.create();
	this.position = vec3.create(); //the origin of the node

	//for visibility computation
	this.oobb = BBox.create(); //object space bounding box
	this.aabb = BBox.create(); //axis aligned bounding box
	this.center = BBox.getCenter( this.aabb ); //the center of the AABB

	//info about the material
	this.material = null; //the material, cannot be a string
	this.use_bounding = true; //in case it has vertex shader deformers the bounding box is not usable

	//for extra data for the shader
	this.uniforms = {};
	this.samplers = [];

	//array of all the model matrix for all the instanced
	this.instanced_models = null;

	this.shader_block_flags = 0;
	this.shader_blocks = [];

	this.picking_node = null; //in case of picking, which node to reference

	//this.deformers = []; //TODO

	//TO DO: instancing
	//this.uniforms_instancing = {};

	//for internal use
	this._camera_visibility = 0; //tells in which camera was visible this instance during the last rendering (using bit operations)
	this._is_visible = false; //used during the rendering to mark if it was seen
	this._dist = 0; //computed during rendering, tells the distance to the current camera
	this._nearest_reflection_probe = null;
}

RenderInstance.NO_SORT = 0;
RenderInstance.SORT_NEAR_FIRST = 1;
RenderInstance.SORT_FAR_FIRST = 2;

RenderInstance.fast_normalmatrix = false; //avoid doint the inverse transpose for normal matrix, and just copies the model

RenderInstance.prototype.fromNode = function(node, skip_matrices)
{
	if(!node)
		throw("no node");
	this.node = node;
	this.layers = node.layers;

	if(!skip_matrices)
	{
		if(node.transform)
			this.setMatrix( node.transform._global_matrix );
		else
			this.setMatrix( LS.IDENTITY );
		mat4.multiplyVec3( this.position, this.matrix, LS.ZEROS );
	}
}

//set the matrix 
RenderInstance.prototype.setMatrix = function(matrix, normal_matrix)
{
	this.matrix.set( matrix );

	if( normal_matrix )
		this.normal_matrix.set( normal_matrix )
	else
		this.computeNormalMatrix();
}

/**
* Updates the normal matrix using the matrix
*
* @method computeNormalMatrix
*/
RenderInstance.prototype.computeNormalMatrix = function()
{
	if(RenderInstance.fast_normalmatrix)
	{
		this.normal_matrix.set( this.matrix );
		mat4.setTranslation( this.normal_matrix, LS.ZEROS );
		return;
	}

	var m = mat4.invert(this.normal_matrix, this.matrix);
	if(m)
		mat4.transpose(this.normal_matrix, m);
}

/**
* applies a transformation to the current matrix
*
* @method applyTransform
* @param {mat4} matrix
* @param {mat4} normal_matrix [optional]
*/
RenderInstance.prototype.applyTransform = function( matrix, normal_matrix )
{
	mat4.mul( this.matrix, matrix, this.matrix );
	if( normal_matrix )
		mat4.mul( this.normal_matrix, normal_matrix, this.normal_matrix );
	else
		this.computeNormalMatrix();
}

//set the material and apply material flags to render instance
RenderInstance.prototype.setMaterial = function(material)
{
	if(material && !material.constructor.is_material)
	{
		//console.error("Material in RenderInstance is not a material class:",material);
		return;
	}
	this.material = material;
	if(material && material.applyToRenderInstance)
		material.applyToRenderInstance(this);
}

//sets the buffers to render, the primitive, and the bounding
RenderInstance.prototype.setMesh = function( mesh, primitive )
{
	if( primitive == -1 || primitive === undefined )
		primitive = gl.TRIANGLES;
	this.primitive = primitive;

	if(mesh != this.mesh)
	{
		this.mesh = mesh;
		this.vertex_buffers = {};
	}

	if(!this.mesh)
		return;

	//this.vertex_buffers = mesh.vertexBuffers;
	for(var i in mesh.vertexBuffers)
		this.vertex_buffers[i] = mesh.vertexBuffers[i];

	switch(primitive)
	{
		case gl.TRIANGLES: 
			this.index_buffer = mesh.indexBuffers["triangles"]; //works for indexed and non-indexed
			break;
		case gl.LINES: 
			/*
			if(!mesh.indexBuffers["lines"])
				mesh.computeWireframe();
			*/
			this.index_buffer = mesh.indexBuffers["lines"];
			break;
		case 10:  //wireframe
			this.primitive = gl.LINES;
			if(!mesh.indexBuffers["wireframe"])
				mesh.computeWireframe();
			this.index_buffer = mesh.indexBuffers["wireframe"];
			break;

		case gl.POINTS: 
		default:
			this.index_buffer = null;
			break;
	}

	if(mesh.bounding)
	{
		this.oobb.set( mesh.bounding ); //copy
		this.use_bounding = true;
	}
	else
		this.use_bounding = false;
}

/**
* Sets the object oriented bounding box using the BBox format (usually is the mesh bounding but in some cases could be different like with skinning or submeshes)
*
* @method setBoundinbBox
* @param {BBox} bbox bounding in bbox format
*/
RenderInstance.prototype.setBoundingBox = function(bbox)
{
	this.oobb.set( bbox );
}

/**
* specifies the rendering range for the mesh (from where and how many primitives), if -1 then ignored
*
* @method setRange
* @param {Number} start
* @param {Number} length
*/
RenderInstance.prototype.setRange = function( start, length )
{
	this.range[0] = start;
	this.range[1] = length;
}

/**
* Enable flag in the flag bit field
*
* @method enableFlag
* @param {number} flag id
*/
RenderInstance.prototype.enableFlag = function(flag)
{
	this.flags |= flag;
}

/**
* Disable flag in the flag bit field
*
* @method enableFlag
* @param {number} flag id
*/
RenderInstance.prototype.disableFlag = function(flag)
{
	this.flags &= ~flag;
}

/**
* Tells if a flag is enabled
*
* @method enableFlag
* @param {number} flag id
* @return {boolean} flag value
*/
RenderInstance.prototype.isFlag = function(flag)
{
	return (this.flags & flag);
}

/**
* Computes the instance bounding box in world space from the one in local space
*
* @method updateAABB
*/
RenderInstance.prototype.updateAABB = function()
{
	BBox.transformMat4( this.aabb, this.oobb, this.matrix );
}

/**
* Used to update the RI info without having to go through the collectData process, it is faster but some changes may take a while
*
* @method update
*/
RenderInstance.prototype.update = function()
{
	if(!this.node || !this.node.transform)
		return;
	this.setMatrix( this.node.transform._global_matrix );
}

/**
* Calls render taking into account primitive and range
*
* @method render
* @param {Shader} shader
*/
RenderInstance.prototype.render = function(shader, primitive)
{
	//in case no normals found but they are required
	if(shader.attributes["a_normal"] && !this.vertex_buffers["normals"])
	{
		this.mesh.computeNormals();		
		this.vertex_buffers["normals"] = this.mesh.vertexBuffers["normals"];
	}

	//in case no coords found but they are required
	/*
	if(shader.attributes["a_coord"] && !this.vertex_buffers["coords"])
	{
		//this.mesh.computeTextureCoordinates();		
		//this.vertex_buffers["coords"] = this.mesh.vertexBuffers["coords"];
	}
	*/

	//in case no tangents found but they are required
	if(shader.attributes["a_tangent"] && !this.vertex_buffers["tangents"])
	{
		this.mesh.computeTangents();		
		this.vertex_buffers["tangents"] = this.mesh.vertexBuffers["tangents"];
	}

	//in case no secondary coords found but they are required
	if(shader.attributes["a_coord1"] && !this.vertex_buffers["coords1"])
	{
		this.mesh.createVertexBuffer("coords1",2, vertex_buffers["coords"].data );
		this.vertex_buffers["coords1"] = this.mesh.vertexBuffers["coords1"];
	}

	if(shader.attributes["a_normal"] && !this.vertex_buffers["normals"])
	{
		this.mesh.computeNormals();		
		this.vertex_buffers["normals"] = this.mesh.vertexBuffers["normals"];
	}

	//in case no secondary coords found but they are required
	if(shader.attributes["a_extra"] && !this.vertex_buffers["extra"])
	{
		this.mesh.createVertexBuffer("extra", "a_extra", 1 );
		this.vertex_buffers["extra"] = this.mesh.vertexBuffers["extra"];
	}

	if(shader.attributes["a_extra2"] && !this.vertex_buffers["extra2"])
	{
		this.mesh.createVertexBuffer("extra2","a_extra2", 2 );
		this.vertex_buffers["extra2"] = this.mesh.vertexBuffers["extra2"];
	}

	if(shader.attributes["a_extra3"] && !this.vertex_buffers["extra3"])
	{
		this.mesh.createVertexBuffer("extra3","a_extra3", 3 );
		this.vertex_buffers["extra3"] = this.mesh.vertexBuffers["extra3"];
	}

	//in case no secondary coords found but they are required
	if(shader.attributes["a_color"] && !this.vertex_buffers["colors"])
	{
		this.mesh.createVertexBuffer( "colors", "a_color", 4 );
		this.vertex_buffers["colors"] = this.mesh.vertexBuffers["colors"];
	}


	if(primitive === undefined)
		primitive = this.primitive;

	var changed_draw_buffers = false;
	if(!shader.supports_drawbuffers && GL.FBO.current && GL.FBO.current.color_textures.length > 1)
	{
		changed_draw_buffers = true;
		GL.FBO.current.toSingle();
	}

	//instancing
	if(this.instanced_models && this.instanced_models.length)
	{
		if( shader.attributes["u_model"] ) //if extension enabled
		{
			if(!this._instanced_uniforms)
				this._instanced_uniforms = {};
			this._instanced_uniforms.u_model = this.instanced_models;
			shader.drawInstanced( this.mesh, primitive,
			  this.index_buffer, this._instanced_uniforms,
			  this.range[0], this.range[1], this.instanced_models.length );
		}
		else //not supported the extension
		{
			for(var i = 0; i < this.instanced_models.length; ++i)
			{
				shader.setUniform("u_model", this.instanced_models[i] );
				shader.drawBuffers( this.vertex_buffers, this.index_buffer, primitive, this.range[0], this.range[1] );
			}
		}
	}
	else //no instancing
	{
		shader.drawBuffers( this.vertex_buffers, this.index_buffer, primitive, this.range[0], this.range[1] );
	}

	if(changed_draw_buffers)
		GL.FBO.current.toMulti();
}

RenderInstance.prototype.addShaderBlock = function( block, uniforms )
{
	if( block.flag_mask & this.shader_block_flags && uniforms === undefined )
		return;

	for(var i = 0; i < this.shader_blocks.length; ++i)
	{
		if(!this.shader_blocks[i])
			continue;
		if( this.shader_blocks[i].block == block )
		{
			if(uniforms !== undefined)
				this.shader_blocks[i].uniforms = uniforms;
			return i;
		}
	}
	this.shader_blocks.push( { block: block, uniforms: uniforms } );
	this.shader_block_flags |= block.flag_mask;
	return this.shader_blocks.length - 1;
}

RenderInstance.prototype.disableShaderBlock = function( block )
{
	if( ! (block.flag_mask & this.shader_block_flags) )
		return;

	for(var i = 0; i < this.shader_blocks.length; ++i)
	{
		if(!this.shader_blocks[i])
			continue;
		if( this.shader_blocks[i].block !== block )
			continue;
		this.shader_block_flags &= ~block.flag_mask;
		break;
	}
}


RenderInstance.prototype.removeShaderBlock = function( block )
{
	if( ! (block.flag_mask & this.shader_block_flags) )
		return;

	for(var i = 0; i < this.shader_blocks.length; ++i)
	{
		if(!this.shader_blocks[i])
			continue;
		if( this.shader_blocks[i].block !== block )
			continue;

		this.shader_blocks.splice(i,1);
		this.shader_block_flags &= ~block.flag_mask;
		break;
	}
}

//checks the ShaderBlocks attached to this instance and resolves the flags
RenderInstance.prototype.computeShaderBlockFlags = function()
{
	return this.shader_block_flags;

	/*
	var r = 0;
	for(var i = 0; i < this.shader_blocks.length; ++i)
	{
		var shader_block = this.shader_blocks[i];
		if(!shader_block)
			continue;
		var block = this.shader_blocks[i].block;
		r |= block.flag_mask;
	}
	return r;
	*/
}

/*
RenderInstance.prototype.renderInstancing = function( shader )
{
	var instances_info = this.instances_info;

	var matrices = new Float32Array( instances_info.length * 16 );
	for(var j = 0; j < instances_info.length; ++j)
	{
		var matrix = instances_info[j].matrix;
		matrices.set( matrix, j*16 );
	}

	gl.bindBuffer(gl.ARRAY_BUFFER, matricesBuffer );
	gl.bufferData(gl.ARRAY_BUFFER, matrices, gl.STREAM_DRAW);

	// Bind the instance matrices data (mat4 behaves as 4 attributes)
	for(var k = 0; k < 4; ++k)
	{
		gl.enableVertexAttribArray( location+k );
		gl.vertexAttribPointer( location+k, 4, gl.FLOAT , false, 16*4, k*4*4 );
		ext.vertexAttribDivisorANGLE( location+k, 1 ); // This makes it instanced!
	}

	//gl.drawElements( gl.TRIANGLES, length, indexBuffer.buffer.gl_type, 0 ); //gl.UNSIGNED_SHORT
	ext.drawElementsInstancedANGLE( gl.TRIANGLES, length, indexBuffer.buffer.gl_type, 0, batch.length );
	GFX.stats.calls += 1;
	for(var k = 0; k < 4; ++k)
	{
		ext.vertexAttribDivisorANGLE( location+k, 0 );
		gl.disableVertexAttribArray( location+k );
	}
}
*/

RenderInstance.prototype.overlapsSphere = function( center, radius )
{
	//we dont know if the bbox of the instance is valid
	if( !this.use_bounding )
		return true;
	return geo.testSphereBBox( center, radius, this.aabb );
}

/**
* Checks if this object was visible by a camera during the last frame
*
* @method wasVisibleByCamera
* @param {LS.Camera} camera [optional] if a camera is supplied it checks if it was visible by that camera, otherwise tells you if it was visible by any camera
* @return {Boolean} true if it was visible by the camera (or any camera if no camera supplied), false otherwise
*/
RenderInstance.prototype.wasVisibleByCamera = function( camera )
{
	if(!camera)
		return this._camera_visibility != 0;
	return (this._camera_visibility | (1<<(camera._rendering_index))) ? true : false;
}

LS.RenderInstance = RenderInstance;
///@FILE:../src/render/renderFrameContext.js
///@INFO: BASE
/**	RenderFrameContext
*	This class is used when you want to render the scene not to the screen but to some texture for postprocessing
*	It helps to create the textures and bind them easily, add extra buffers or show it on the screen.
*	Check the FrameFX and CameraFX components to see it in action.
*   Dependencies: LS.Renderer (writes there only)
*
* @class RenderFrameContext
* @namespace LS
* @constructor
*/
function RenderFrameContext( o )
{
	this.width = 0; //0 means the same size as the viewport, negative numbers mean reducing the texture in half N times
	this.height = 0; //0 means the same size as the viewport
	this.precision = RenderFrameContext.DEFAULT_PRECISION; //LOW_PRECISION uses a byte, MEDIUM uses a half_float, HIGH uses a float, or directly the texture type (p.e gl.UNSIGNED_SHORT_4_4_4_4 )
	this.filter_texture = true; //magFilter: in case the texture is shown, do you want to see it pixelated?
	this.format = GL.RGB; //how many color channels, or directly the texture internalformat 
	this.use_depth_texture = true; //store the depth in a texture
	this.use_stencil_buffer = false; //add an stencil buffer (cannot be read as a texture in webgl)
	this.num_extra_textures = 0; //number of extra textures in case we want to render to several buffers
	this.name = null; //if a name is provided all the textures will be stored in the LS.ResourcesManager

	this.generate_mipmaps = false; //try to generate mipmaps if possible (only when the texture is power of two)
	this.adjust_aspect = false; //when the size doesnt match the canvas size it could look distorted, settings this to true will fix the problem
	this.clone_after_unbind = false; //clones the textures after unbinding it. Used when the texture will be in the 3D scene

	this._fbo = null;
	this._color_texture = null;
	this._depth_texture = null;
	this._textures = []; //all color textures (the first will be _color_texture)
	this._cloned_textures = null; //in case we set the clone_after_unbind to true
	this._cloned_depth_texture = null;

	this._version = 1; //to detect changes
	this._minFilter = gl.NEAREST;

	if(o)
		this.configure(o);
}


RenderFrameContext.current = null;
RenderFrameContext.stack = [];

RenderFrameContext.DEFAULT_PRECISION = 0; //selected by the renderer
RenderFrameContext.LOW_PRECISION = 1; //byte
RenderFrameContext.MEDIUM_PRECISION = 2; //half_float or float
RenderFrameContext.HIGH_PRECISION = 3; //float

RenderFrameContext.DEFAULT_PRECISION_WEBGL_TYPE = GL.UNSIGNED_BYTE;

RenderFrameContext["@width"] = { type: "number", step: 1, precision: 0 };
RenderFrameContext["@height"] = { type: "number", step: 1, precision: 0 };

//definitions for the GUI
RenderFrameContext["@precision"] = { widget: "combo", values: { 
	"default": RenderFrameContext.DEFAULT_PRECISION, 
	"low": RenderFrameContext.LOW_PRECISION,
	"medium": RenderFrameContext.MEDIUM_PRECISION,
	"high": RenderFrameContext.HIGH_PRECISION
	}
};

RenderFrameContext["@format"] = { widget: "combo", values: { 
		"RGB": GL.RGB,
		"RGBA": GL.RGBA
//		"R8": GL.LUMINANCE,
//		"LUMINANCE_ALPHA": GL.LUMINANCE_ALPHA,
//		"ALPHA": GL.ALPHA
	}
};

RenderFrameContext["@num_extra_textures"] = { type: "number", step: 1, min: 0, max: 4, precision: 0 };
RenderFrameContext["@name"] = { type: "string" };

RenderFrameContext.prototype.clear = function()
{
	if(this.name)
	{
		for(var i = 0; i < this._textures.length; ++i)
			delete LS.ResourcesManager.textures[ this.name + (i > 1 ? i : "") ];
		if(this._depth_texture)
			delete LS.ResourcesManager.textures[ this.name + "_depth"];
	}

	this._fbo = null;
	this._textures = [];
	this._color_texture = null;
	this._depth_textures = null;
}

RenderFrameContext.prototype.configure = function(o)
{
	this.width = o.width || 0;
	this.height = o.height || 0;
	this.format = o.format || GL.RGBA;
	this.precision = o.precision || 0;
	this.filter_texture = !!o.filter_texture;
	this.adjust_aspect = !!o.adjust_aspect;
	this.use_depth_texture = !!o.use_depth_texture;
	this.use_stencil_buffer = !!o.use_stencil_buffer;
	this.num_extra_textures = o.num_extra_textures || 0;
	this.name = o.name;
	this.clone_after_unbind = !!o.clone_after_unbind;
}

RenderFrameContext.prototype.serialize = function()
{
	return {
		width: this.width,
		height:  this.height,
		filter_texture: this.filter_texture,
		precision:  this.precision,
		format: this.format,
		adjust_aspect: this.adjust_aspect,
		use_depth_texture:  this.use_depth_texture,
		use_stencil_buffer: this.use_stencil_buffer,
		num_extra_textures:  this.num_extra_textures,
		clone_after_unbind: this.clone_after_unbind,
		name: this.name
	};
}

RenderFrameContext.prototype.prepare = function( viewport_width, viewport_height )
{
	//compute the right size for the textures
	var final_width = this.width;
	var final_height = this.height;
	if(final_width == 0)
		final_width = viewport_width;
	else if(final_width < 0)
		final_width = viewport_width >> Math.abs( this.width ); //subsampling
	if(final_height == 0)
		final_height = viewport_height;
	else if(final_height < 0)
		final_height = viewport_height >> Math.abs( this.height ); //subsampling

	var format = this.format;
	var magFilter = this.filter_texture ? gl.LINEAR : gl.NEAREST ;
	var type = 0;

	var minFilter = gl.LINEAR;
	if(this.generate_mipmaps && GL.isPowerOfTwo(final_width) && GL.isPowerOfTwo(final_height) )
		minFilter = gl.LINEAR_MIPMAP_LINEAR;
	this._minFilter = minFilter;

	switch( this.precision )
	{
		case RenderFrameContext.LOW_PRECISION:
			type = gl.UNSIGNED_BYTE; break;
		case RenderFrameContext.MEDIUM_PRECISION:
			type = gl.HIGH_PRECISION_FORMAT; break; //gl.HIGH_PRECISION_FORMAT is HALF_FLOAT_OES, if not supported then is FLOAT, otherwise is UNSIGNED_BYTE
		case RenderFrameContext.HIGH_PRECISION:
			type = gl.FLOAT; break;
		case RenderFrameContext.DEFAULT_PRECISION:
			type = RenderFrameContext.DEFAULT_PRECISION_WEBGL_TYPE; break;
		default:
			type = this.precision; break; //used for custom formats
	}

	//check support due to weirdeness of webgl 1.0
	if( type == GL.HALF_FLOAT_OES && !GL.FBO.testSupport( type, format ) )
		format = gl.RGBA;
	if( type == GL.HALF_FLOAT_OES && !GL.FBO.testSupport( type, format ) )
		type = gl.FLOAT;

	var textures = this._textures;

	//for the color: check that the texture size matches
	if( !this._color_texture || 
		this._color_texture.width != final_width || this._color_texture.height != final_height || 
		this._color_texture.type != type || this._color_texture.format != format || this._color_texture.minFilter != minFilter )
		this._color_texture = new GL.Texture( final_width, final_height, { minFilter: minFilter, magFilter: magFilter, format: format, type: type });
	else
		this._color_texture.setParameter( gl.TEXTURE_MAG_FILTER, magFilter );
	textures[0] = this._color_texture;

	//extra color texture (multibuffer rendering)
	var total_extra = Math.min( this.num_extra_textures, 4 );
	
	//extra buffers not supported in this webgl context
	if(gl.webgl_version == 1 && !gl.extensions["WEBGL_draw_buffers"])
		total_extra = 0;

	for(var i = 0; i < total_extra; ++i) //MAX is 4
	{
		var extra_texture = textures[1 + i];
		if( (!extra_texture || extra_texture.width != final_width || extra_texture.height != final_height || extra_texture.type != type || extra_texture.format != format || extra_texture.minFilter != minFilter) )
			extra_texture = new GL.Texture( final_width, final_height, { minFilter: minFilter, magFilter: magFilter, format: format, type: type });
		else
			extra_texture.setParameter( gl.TEXTURE_MAG_FILTER, magFilter );
		textures[1 + i] = extra_texture;
	}

	//for the depth
	var depth_format = gl.DEPTH_COMPONENT;
	var depth_type = gl.UNSIGNED_INT;

	if(this.use_stencil_buffer && gl.extensions.WEBGL_depth_texture)
	{
		depth_format = gl.DEPTH_STENCIL;
		depth_type = gl.extensions.WEBGL_depth_texture.UNSIGNED_INT_24_8_WEBGL;
	}

	if( this.use_depth_texture && 
		(!this._depth_texture || this._depth_texture.width != final_width || this._depth_texture.height != final_height || this._depth_texture.format != depth_format || this._depth_texture.type != depth_type ) && 
		gl.extensions["WEBGL_depth_texture"] )
		this._depth_texture = new GL.Texture( final_width, final_height, { filter: gl.NEAREST, format: depth_format, type: depth_type });
	else if( !this.use_depth_texture )
		this._depth_texture = null;

	//we will store some extra info in the depth texture for the near and far plane distances
	if(this._depth_texture)
	{
		if(!this._depth_texture.near_far_planes)
			this._depth_texture.near_far_planes = vec2.create();
	}

	//create FBO
	if( !this._fbo )
		this._fbo = new GL.FBO();

	//cut extra
	textures.length = 1 + total_extra;

	//assign textures (this will enable the FBO but it will restore the old one after finishing)
	this._fbo.stencil = this.use_stencil_buffer;
	this._fbo.setTextures( textures, this._depth_texture );
	this._version += 1;
}

/**
* Called to bind the rendering to this context, from now on all the render will be stored in the textures inside
*
* @method enable
*/
RenderFrameContext.prototype.enable = function( render_settings, viewport, camera )
{
	viewport = viewport || gl.viewport_data;

	//create FBO and textures (pass width and height of current viewport)
	this.prepare( viewport[2], viewport[3] );

	if(!this._fbo)
		throw("No FBO created in RenderFrameContext");

	//enable FBO
	RenderFrameContext.enableFBO( this._fbo, this.adjust_aspect );

	if(LS.RenderFrameContext.current)
		RenderFrameContext.stack.push( LS.RenderFrameContext.current );
	LS.RenderFrameContext.current = this;

	//set depth info inside the texture
	camera = camera || LS.Renderer._current_camera;
	if(this._depth_texture && camera)
	{
		this._depth_texture.near_far_planes[0] = camera.near;
		this._depth_texture.near_far_planes[1] = camera.far;
	}
}

//we cannot read and write in the same buffer, so we need to clone the textures
//done from... ?
RenderFrameContext.prototype.cloneBuffers = function()
{
	//we do not call this._fbo.unbind because it will set the previous FBO
	gl.bindFramebuffer( gl.FRAMEBUFFER, null );

	///for every color texture
	if( this._textures.length )
	{
		if(!this._cloned_textures)
			this._cloned_textures = [];
		var textures = this._textures;
		this._cloned_textures.length = textures.length;
		for(var i = 0; i < textures.length; ++i)
		{
			var texture = textures[i];
			var cloned_texture = this._cloned_textures[i];
			if( !cloned_texture || !cloned_texture.hasSameSize( texture ) || !cloned_texture.hasSameProperties( texture ) )
				cloned_texture = this._cloned_textures[i] = new GL.Texture( texture.width, texture.height, texture.getProperties() );
			texture.copyTo( cloned_texture );
			if(i == 0)
				LS.ResourcesManager.textures[":color_buffer" ] = cloned_texture;
		}
	}

	//for depth
	if( this._depth_texture )
	{
		var depth = this._depth_texture;
		if(!this._cloned_depth_texture || this._cloned_depth_texture.width != depth.width || this._cloned_depth_texture.height != depth.height || !this._cloned_depth_texture.hasSameProperties( depth ) )
			this._cloned_depth_texture = new GL.Texture( depth.width, depth.height, depth.getProperties() );

		depth.copyTo( this._cloned_depth_texture );
		if(!this._cloned_depth_texture.near_far_planes)
			this._cloned_depth_texture.near_far_planes = vec2.create();
		this._cloned_depth_texture.near_far_planes.set( depth.near_far_planes );

		LS.ResourcesManager.textures[":depth_buffer" ] = this._cloned_depth_texture;
	}

	//rebind FBO
	gl.bindFramebuffer( gl.FRAMEBUFFER, this._fbo.handler );
}

/**
* Called to stop rendering to this context
*
* @method disable
*/
RenderFrameContext.prototype.disable = function()
{
	//sets some global parameters for aspect and current RFC
	RenderFrameContext.disableFBO( this._fbo );

	//if we need to store the textures in the ResourcesManager
	if(this.name)
	{
		var textures = this._textures;
		for(var i = 0; i < textures.length; ++i)
		{
			var name = this.name + (i > 0 ? i : "");
			textures[i].filename = name;
			var final_texture = textures[i];

			//only clone main color if requested
			if( this.clone_after_unbind && i === 0 )
			{
				if( !this._cloned_texture || 
					this._cloned_texture.width !== final_texture.width || 
					this._cloned_texture.height !== final_texture.height ||
					this._cloned_texture.type !== final_texture.type )
					this._cloned_texture = final_texture.clone();
				else
					final_texture.copyTo( this._cloned_texture );
				final_texture = this._cloned_texture;
			}

			if( this._minFilter == gl.LINEAR_MIPMAP_LINEAR )
			{
				final_texture.bind(0);
				gl.generateMipmap(gl.TEXTURE_2D);
				final_texture.has_mipmaps = true;
			}

			LS.ResourcesManager.textures[ name ] = final_texture;
		}

		if(this._depth_texture)
		{
			var name = this.name + "_depth";
			var depth_texture = this._depth_texture;
			if( this.clone_after_unbind )
			{
				if( !this._cloned_depth_texture || 
					this._cloned_depth_texture.width !== depth_texture.width || 
					this._cloned_depth_texture.height !== depth_texture.height ||
					this._cloned_depth_texture.type !== depth_texture.type )
					this._cloned_depth_texture = depth_texture.clone();
				else
					depth_texture.copyTo( this._cloned_depth_texture );
				if(!this._cloned_depth_texture.near_far_planes)
					this._cloned_depth_texture.near_far_planes = vec2.create();
				this._cloned_depth_texture.near_far_planes.set( depth_texture.near_far_planes );
				depth_texture = this._cloned_depth_texture;
			}

			depth_texture.filename = name;
			LS.ResourcesManager.textures[ name ] = depth_texture;
		}
	}

	if( RenderFrameContext.stack.length )
		LS.RenderFrameContext.current = RenderFrameContext.stack.pop();
	else
		LS.RenderFrameContext.current = null;
}

/**
* returns the texture containing the data rendered in this context
*
* @method getColorTexture
* @param {number} index the number of the texture (in case there is more than one)
* @return {GL.Texture} the texture
*/
RenderFrameContext.prototype.getColorTexture = function(num)
{
	return this._textures[ num || 0 ] || null;
}

/**
* returns the depth texture containing the depth data rendered in this context (in case the use_depth_texture is set to true)
*
* @method getDepthTexture
* @return {GL.Texture} the depth texture
*/
RenderFrameContext.prototype.getDepthTexture = function()
{
	return this._depth_texture || null;
}

/**
* Fills the textures with a flat color
* @method clearTextures
*/
RenderFrameContext.prototype.clearTextures = function()
{
	for(var i = 0; i < this._textures.length; ++i)
	{
		var texture = this._textures[i];
		if(!texture)
			continue;
		texture.fill([0,0,0,0]);
	}
}

//enables the FBO and sets every texture with a flag so it cannot be used during the rendering process
RenderFrameContext.enableFBO = function( fbo, adjust_aspect )
{
	fbo.bind( true ); //changes viewport to full FBO size (saves old)

	LS.Renderer._full_viewport.set( gl.viewport_data );
	if( adjust_aspect )
	{
		fbo._old_aspect = LS.Renderer.global_aspect;
		LS.Renderer.global_aspect = (gl.canvas.width / gl.canvas.height) / (fbo.color_textures[0].width / fbo.color_textures[0].height);
	}
	else
		delete fbo._old_aspect;
}

RenderFrameContext.disableFBO = function( fbo )
{
	fbo.unbind(); //restores viewport to old saved one
	LS.Renderer._full_viewport.set( fbo._old_viewport );
	if( fbo._old_aspect )
		LS.Renderer.global_aspect = fbo._old_aspect;
}


/**
* Render the context of the context to the viewport (allows to apply FXAA)
*
* @method show
* @param {boolean} use_antialiasing in case you want to render with FXAA antialiasing
*/
RenderFrameContext.prototype.show = function( use_antialiasing )
{
	var texture = this._color_texture;
	if(!use_antialiasing)
	{
		texture.toViewport();
		return;
	}

	var viewport = gl.getViewport();
	var shader = GL.Shader.getFXAAShader();
	var mesh = GL.Mesh.getScreenQuad();
	texture.bind(0);
	shader.uniforms( { u_texture:0, uViewportSize: viewport.subarray(2,4), u_iViewportSize: [1 / texture.width, 1 / texture.height]} ).draw( mesh );
}

//Resets the current WebGL fbo so it renders to the screen
RenderFrameContext.reset = function()
{
	gl.bindFramebuffer( gl.FRAMEBUFFER, null );
	LS.RenderFrameContext.current = null;
	LS.RenderFrameContext.stack.length = 0;
}


LS.RenderFrameContext = RenderFrameContext;

///@FILE:../src/render/renderQueue.js
///@INFO: BASE
//RenderQueue is in charge of storing the RenderInstances that must be rendered
//There could be several RenderQueue (for opaque, transparent, overlays, etc)
//It works similar to the one in Unity
function RenderQueue( value, sort_mode, options )
{
	this.enabled = true; //if disabled it will be skipped

	//container for all instances that belong to this render queue
	this.instances = [];

	this.value = value || 0;
	this.sort_mode = sort_mode || LS.RenderQueue.NO_SORT;
	this.must_clone_buffers = false; //used for readback rendering like refracion
	//this.visible_in_pass = null;

	//callbacks
	this.onStart = null;
	this.onFinish = null;

	//configure
	if(options)
		for(var i in options)
			this[i] = options[i];
}

RenderQueue.readback_allowed = true;

RenderQueue.prototype.sort = function()
{
	if(!this.instances.length)
		return;

	var func = null;
	switch(this.sort_mode)
	{
		case 1: func = LS.RenderQueue.sort_near_to_far_func; break;
		case 2: func = LS.RenderQueue.sort_far_to_near_func; break;
		case 3: func = LS.RenderQueue.sort_by_priority_func; break;
	}

	if(func)
		this.instances.sort( func );
}

RenderQueue.prototype.add = function( ri )
{
	this.instances.push( ri );
}

RenderQueue.prototype.clear = function()
{
	this.instances.length = 0;
}

RenderQueue.prototype.start = function( pass, render_settings )
{
	if(this.onStart)
	{
		var r = this.onStart( pass, render_settings); //cancels rendering
		if (r === false)
			return false;
	}

	if(this.instances.length && this.must_clone_buffers && RenderQueue.readback_allowed && pass === LS.COLOR_PASS )
	{
		if( LS.RenderFrameContext.current )
			LS.RenderFrameContext.current.cloneBuffers();
		//cubemaps are not cloned... too much work
	}
}

//not used...
RenderQueue.prototype.finish = function( pass )
{
	if(this.onFinish)
		this.onFinish( pass, render_settings );
}

//we use 5 so from 0 to 9 is one queue, from 10 to 19 another one, etc
RenderQueue.AUTO =			-1;
RenderQueue.BACKGROUND =	5; //0..9
RenderQueue.GEOMETRY =		35; //30..39
RenderQueue.TRANSPARENT =	75; //70..79
RenderQueue.READBACK_COLOR = 95;//90..99
RenderQueue.OVERLAY =		115; //100..119

RenderQueue.NO_SORT = 0;
RenderQueue.SORT_NEAR_TO_FAR = 1;
RenderQueue.SORT_FAR_TO_NEAR = 2;
RenderQueue.SORT_BY_PRIORITY = 3;

RenderQueue.sort_far_to_near_func = function(a,b) { return b._dist - a._dist; },
RenderQueue.sort_near_to_far_func = function(a,b) { return a._dist - b._dist; },
RenderQueue.sort_by_priority_func = function(a,b) { return b.priority - a.priority; },
RenderQueue.sort_by_priority_and_near_to_far_func = function(a,b) { var r = b.priority - a.priority; return r ? r : (a._dist - b._dist) },
RenderQueue.sort_by_priority_and_far_to_near_func = function(a,b) { var r = b.priority - a.priority; return r ? r : (b._dist - a._dist) },

LS.RenderQueue = RenderQueue;
///@FILE:../src/render/renderer.js
///@INFO: BASE

//************************************
/**
* The Renderer is in charge of generating one frame of the scene. Contains all the passes and intermediate functions to create the frame.
*
* @class Renderer
* @namespace LS
* @constructor
*/

//passes
var COLOR_PASS = LS.COLOR_PASS = { name: "color", id: 1 };
var SHADOW_PASS = LS.SHADOW_PASS = { name: "shadow", id: 2 };
var PICKING_PASS = LS.PICKING_PASS = { name: "picking", id: 3 };

//render events
EVENT.BEFORE_RENDER = "beforeRender";
EVENT.READY_TO_RENDER = "readyToRender";
EVENT.RENDER_SHADOWS = "renderShadows";
EVENT.AFTER_VISIBILITY = "afterVisibility";
EVENT.RENDER_REFLECTIONS = "renderReflections";
EVENT.BEFORE_RENDER_MAIN_PASS = "beforeRenderMainPass";
EVENT.ENABLE_FRAME_CONTEXT = "enableFrameContext";
EVENT.SHOW_FRAME_CONTEXT = "showFrameContext";
EVENT.AFTER_RENDER = "afterRender";
EVENT.BEFORE_RENDER_FRAME = "beforeRenderFrame";
EVENT.BEFORE_RENDER_SCENE = "beforeRenderScene";
EVENT.COMPUTE_VISIBILITY = "computeVisibility";
EVENT.AFTER_RENDER_FRAME = "afterRenderFrame";
EVENT.AFTER_RENDER_SCENE = "afterRenderScene";
EVENT.RENDER_HELPERS = "renderHelpers";
EVENT.RENDER_PICKING = "renderPicking";
EVENT.BEFORE_SHOW_FRAME_CONTEXT = "beforeShowFrameContext";
EVENT.BEFORE_CAMERA_ENABLED = "beforeCameraEnabled";
EVENT.AFTER_CAMERA_ENABLED = "afterCameraEnabled";
EVENT.BEFORE_RENDER_INSTANCES = "beforeRenderInstances";
EVENT.RENDER_INSTANCES = "renderInstances";
EVENT.RENDER_SCREEN_SPACE = "renderScreenSpace";
EVENT.AFTER_RENDER_INSTANCES = "afterRenderInstances";
EVENT.RENDER_GUI = "renderGUI";
EVENT.FILL_SCENE_UNIFORMS = "fillSceneUniforms";
EVENT.AFTER_COLLECT_DATA = "afterCollectData";
EVENT.PREPARE_MATERIALS = "prepareMaterials";

var Renderer = {

	default_render_settings: new LS.RenderSettings(), //overwritten by the global info or the editor one
	default_material: new LS.StandardMaterial(), //used for objects without material

	global_aspect: 1, //used when rendering to a texture that doesnt have the same aspect as the screen
	default_point_size: 1, //point size in pixels (could be overwritte by render instances)

	render_profiler: false,

	_global_viewport: vec4.create(), //the viewport we have available to render the full frame (including subviewports), usually is the 0,0,gl.canvas.width,gl.canvas.height
	_full_viewport: vec4.create(), //contains info about the full viewport available to render (current texture size or canvas size)

	//temporal info during rendering
	_current_scene: null,
	_current_render_settings: null,
	_current_camera: null,
	_current_target: null, //texture where the image is being rendered
	_current_pass: COLOR_PASS, //object containing info about the pass
	_current_layers_filter: 0xFFFF,// do a & with this to know if it must be rendered
	_global_textures: {}, //used to speed up fetching global textures
	_global_shader_blocks: [], //used to add extra shaderblocks to all objects in the scene (it gets reseted every frame)
	_global_shader_blocks_flags: 0, 
	_reverse_faces: false,
	_in_player: true, //true if rendering in the player

	_queues: [], //render queues in order

	_main_camera: null,

	_visible_cameras: null,
	_active_lights: null, //array of lights that are active in the scene
	_visible_instances: null,
	_visible_materials: [],
	_near_lights: [],
	_active_samples: [],

	//stats
	_frame_time: 0,
	_frame_cpu_time: 0,
	_rendercalls: 0, //calls to instance.render
	_rendered_instances: 0, //instances processed
	_rendered_passes: 0,
	_frame: 0,
	_last_time: 0,

	//using timer queries
	gpu_times: {
		total: 0,
		shadows: 0,
		reflections: 0,
		main: 0,
		postpo: 0,
		gui: 0
	},

	//to measure performance
	timer_queries_enabled: true,
	_timer_queries: {},
	_waiting_queries: false,

	//settings
	_collect_frequency: 1, //used to reuse info (WIP)

	//reusable locals
	_view_matrix: mat4.create(),
	_projection_matrix: mat4.create(),
	_viewprojection_matrix: mat4.create(),
	_2Dviewprojection_matrix: mat4.create(),

	_temp_matrix: mat4.create(),
	_temp_cameye: vec3.create(),
	_identity_matrix: mat4.create(),
	_uniforms: {},
	_samplers: [],
	_instancing_data: [],

	//safety
	_is_rendering_frame: false,
	_ignore_reflection_probes: false,

	//debug
	allow_textures: true,
	_sphere_mesh: null,
	_debug_instance: null,

	//fixed texture slots for global textures
	SHADOWMAP_TEXTURE_SLOT: 7,
	ENVIRONMENT_TEXTURE_SLOT: 6,
	IRRADIANCE_TEXTURE_SLOT: 5,
	LIGHTPROJECTOR_TEXTURE_SLOT: 4,
	LIGHTEXTRA_TEXTURE_SLOT: 3,

	//used in special cases
	BONES_TEXTURE_SLOT: 3,
	MORPHS_TEXTURE_SLOT: 2,
	MORPHS_TEXTURE2_SLOT: 1,

	//called from...
	init: function()
	{
		//create some useful textures: this is used in case a texture is missing
		this._black_texture = new GL.Texture(1,1, { pixel_data: [0,0,0,255] });
		this._gray_texture = new GL.Texture(1,1, { pixel_data: [128,128,128,255] });
		this._white_texture = new GL.Texture(1,1, { pixel_data: [255,255,255,255] });
		this._normal_texture = new GL.Texture(1,1, { pixel_data: [128,128,255,255] });
		this._white_cubemap_texture = new GL.Texture(1,1, { texture_type: gl.TEXTURE_CUBE_MAP, pixel_data: (new Uint8Array(6*4)).fill(255) });
		this._missing_texture = this._gray_texture;
		var internal_textures = [ this._black_texture, this._gray_texture, this._white_texture, this._normal_texture, this._missing_texture ];
		internal_textures.forEach(function(t){ t._is_internal = true; });
		LS.ResourcesManager.textures[":black"] = this._black_texture;
		LS.ResourcesManager.textures[":gray"] = this._gray_texture;
		LS.ResourcesManager.textures[":white"] = this._white_texture;
		LS.ResourcesManager.textures[":flatnormal"] = this._normal_texture;

		//some global meshes could be helpful: used for irradiance probes
		this._sphere_mesh = GL.Mesh.sphere({ size:1, detail:32 });

		//draw helps rendering debug stuff
		if(LS.Draw)
		{
			LS.Draw.init();
			LS.Draw.onRequestFrame = function() { LS.GlobalScene.requestFrame(); }
		}

		//enable webglCanvas lib so it is easy to render in 2D
		if(global.enableWebGLCanvas && !gl.canvas.canvas2DtoWebGL_enabled)
			global.enableWebGLCanvas( gl.canvas );

		// we use fixed slots to avoid changing texture slots all the time
		// from more common to less (to avoid overlappings with material textures)
		// the last slot is reserved for litegl binding stuff
		var max_texture_units = this._max_texture_units = gl.getParameter( gl.MAX_TEXTURE_IMAGE_UNITS );
		this.SHADOWMAP_TEXTURE_SLOT = max_texture_units - 2;
		this.ENVIRONMENT_TEXTURE_SLOT = max_texture_units - 3;
		this.IRRADIANCE_TEXTURE_SLOT = max_texture_units - 4;

		this.BONES_TEXTURE_SLOT = max_texture_units - 5;
		this.MORPHS_TEXTURE_SLOT = max_texture_units - 6;
		this.MORPHS_TEXTURE2_SLOT = max_texture_units - 7;

		this.LIGHTPROJECTOR_TEXTURE_SLOT = max_texture_units - 8;
		this.LIGHTEXTRA_TEXTURE_SLOT = max_texture_units - 9;

		this._active_samples.length = max_texture_units;

		this.createRenderQueues();

		this._full_viewport.set([0,0,gl.drawingBufferWidth,gl.drawingBufferHeight]);

		this._uniforms.u_viewport = gl.viewport_data;
		this._uniforms.environment_texture = this.ENVIRONMENT_TEXTURE_SLOT;
		this._uniforms.u_clipping_plane = vec4.create();
	},

	reset: function()
	{
	},

	//used to clear the state
	resetState: function()
	{
		this._is_rendering_frame = false;
		this._reverse_faces = false;
	},

	//used to store which is the current full viewport available (could be different from the canvas in case is a FBO or the camera has a partial viewport)
	setFullViewport: function(x,y,w,h)
	{
		if(arguments.length == 0) //restore
		{
			this._full_viewport[0] = this._full_viewport[1] = 0;
			this._full_viewport[2] = gl.drawingBufferWidth;
			this._full_viewport[3] = gl.drawingBufferHeight;
		}
		else if(x.constructor === Number)
		{
			this._full_viewport[0] = x; this._full_viewport[1] = y; this._full_viewport[2] = w; this._full_viewport[3] = h;
		}
		else if(x.length)
			this._full_viewport.set(x);
	},

	/**
	* Renders the current scene to the screen
	* Many steps are involved, from gathering info from the scene tree, generating shadowmaps, setup FBOs, render every camera
	* If you want to change the rendering pipeline, do not overwrite this function, try to understand it first, otherwise you will miss lots of features
	*
	* @method render
	* @param {Scene} scene
	* @param {RenderSettings} render_settings
	* @param {Array} [cameras=null] if no cameras are specified the cameras are taken from the scene
	*/
	render: function( scene, render_settings, cameras )
	{
		scene = scene || LS.GlobalScene;

		if( this._is_rendering_frame )
		{
			console.error("Last frame didn't finish and a new one was issued. Remember that you cannot call LS.Renderer.render from an event dispatched during the render, this would cause a recursive loop. Call LS.Renderer.reset() to clear from an error.");
			//this._is_rendering_frame = false; //for safety, we setting to false 
			return;
		}

		//init frame
		this._is_rendering_frame = true;
		render_settings = render_settings || this.default_render_settings;
		this._current_render_settings = render_settings;
		this._current_scene = scene;
		this._main_camera = cameras ? cameras[0] : null;
		scene._frame += 1; //done at the beginning just in case it crashes
		this._frame += 1;
		scene._must_redraw = false;

		var start_time = getTime();
		this._frame_time = start_time - this._last_time;
		this._last_time = start_time;
		this._rendercalls = 0;
		this._rendered_instances = 0;
		this._rendered_passes = 0;
		this._global_shader_blocks.length = 0;
		this._global_shader_blocks_flags = 0;
		for(var i in this._global_textures)
			this._global_textures[i] = null;
		if(!this._current_pass)
			this._current_pass = COLOR_PASS;
		this._reverse_faces = false;

		//extract info about previous frame
		this.resolveQueries();

		//to restore from a possible exception (not fully tested, remove if problem)
		if(!render_settings.ignore_reset)
			LS.RenderFrameContext.reset();

		if(gl.canvas.canvas2DtoWebGL_enabled)
			gl.resetTransform(); //reset 

		LS.GUI.ResetImmediateGUI(true);//just to let the GUI ready

		//force fullscreen viewport
		if( !render_settings.keep_viewport )
		{
			gl.viewport(0, 0, gl.drawingBufferWidth, gl.drawingBufferHeight );
			this.setFullViewport(0, 0, gl.drawingBufferWidth, gl.drawingBufferHeight); //assign this as the full available viewport
		}
		else
			this.setFullViewport( gl.viewport_data );
		this._global_viewport.set( gl.viewport_data );

		//Event: beforeRender used in actions that could affect which info is collected for the rendering
		this.startGPUQuery( "beforeRender" );
		LEvent.trigger( scene, EVENT.BEFORE_RENDER, render_settings );
		this.endGPUQuery();

		//get render instances, cameras, lights, materials and all rendering info ready (computeVisibility)
		this.processVisibleData( scene, render_settings, cameras );

		//Define the main camera, the camera that should be the most important (used for LOD info, or shadowmaps)
		cameras = cameras && cameras.length ? cameras : scene._cameras;//this._visible_cameras;
		if(cameras.length == 0)
			throw("no cameras");
		this._visible_cameras = cameras; //the cameras being rendered
		this._main_camera = cameras[0];

		//Event: readyToRender when we have all the info to render
		LEvent.trigger( scene, EVENT.READY_TO_RENDER, render_settings );

		//remove the lights that do not lay in front of any camera (this way we avoid creating shadowmaps)
		//TODO

		//Event: renderShadowmaps helps to generate shadowMaps that need some camera info (which could be not accessible during processVisibleData)
		this.startGPUQuery("shadows");
		LEvent.trigger(scene, EVENT.RENDER_SHADOWS, render_settings );
		this.endGPUQuery();

		//Event: afterVisibility allows to cull objects according to the main camera
		LEvent.trigger(scene, EVENT.AFTER_VISIBILITY, render_settings );

		//Event: renderReflections in case some realtime reflections are needed, this is the moment to render them inside textures
		this.startGPUQuery("reflections");
		LEvent.trigger(scene, EVENT.RENDER_REFLECTIONS, render_settings );
		this.endGPUQuery();

		//Event: beforeRenderMainPass in case a last step is missing
		LEvent.trigger(scene, EVENT.BEFORE_RENDER_MAIN_PASS, render_settings );

		//enable global FX context
		if(render_settings.render_fx)
			LEvent.trigger( scene, EVENT.ENABLE_FRAME_CONTEXT, render_settings );

		//render what every camera can see
		if(this.onCustomRenderFrameCameras)
			this.onCustomRenderFrameCameras( cameras, render_settings );
		else
			this.renderFrameCameras( cameras, render_settings );

		//keep original viewport
		if( render_settings.keep_viewport )
			gl.setViewport( this._global_viewport );

		//disable and show final FX context
		if(render_settings.render_fx)
		{
			this.startGPUQuery("postpo");
			LEvent.trigger( scene, EVENT.SHOW_FRAME_CONTEXT, render_settings );
			this.endGPUQuery();
		}

		//renderGUI
		this.startGPUQuery("gui");
		this.renderGUI( render_settings );
		this.endGPUQuery();

		//profiling must go here
		this._frame_cpu_time = getTime() - start_time;
		if( LS.Draw ) //developers may decide not to include LS.Draw
			this._rendercalls += LS.Draw._rendercalls; LS.Draw._rendercalls = 0; //stats are not centralized

		//Event: afterRender to give closure to some actions
		LEvent.trigger( scene, EVENT.AFTER_RENDER, render_settings ); 
		this._is_rendering_frame = false;

		//coroutines
		LS.triggerCoroutines("render");

		if(this.render_profiler)
			this.renderProfiler();
	},

	/**
	* Calls renderFrame of every camera in the cameras list (triggering the appropiate events)
	*
	* @method renderFrameCameras
	* @param {Array} cameras
	* @param {RenderSettings} render_settings
	*/
	renderFrameCameras: function( cameras, render_settings )
	{
		var scene = this._current_scene;

		//for each camera
		for(var i = 0; i < cameras.length; ++i)
		{
			var current_camera = cameras[i];

			LEvent.trigger(scene, EVENT.BEFORE_RENDER_FRAME, render_settings );
			LEvent.trigger(current_camera, EVENT.BEFORE_RENDER_FRAME, render_settings );
			LEvent.trigger(current_camera, EVENT.ENABLE_FRAME_CONTEXT, render_settings );

			//main render
			this.startGPUQuery("main");
			if(this.onCustomRenderFrame)
				this.onCustomRenderFrame( current_camera, render_settings ); 
			else
				this.renderFrame( current_camera, render_settings ); 
			this.endGPUQuery();

			//show buffer on the screen
			this.startGPUQuery("postpo");
			LEvent.trigger(current_camera, EVENT.SHOW_FRAME_CONTEXT, render_settings );
			LEvent.trigger(current_camera, EVENT.AFTER_RENDER_FRAME, render_settings );
			LEvent.trigger(scene, EVENT.AFTER_RENDER_FRAME, render_settings );
			this.endGPUQuery();
		}
	},

	/**
	* renders the view from one camera to the current viewport (could be the screen or a texture)
	*
	* @method renderFrame
	* @param {Camera} camera 
	* @param {Object} render_settings [optional]
	* @param {Scene} scene [optional] this can be passed when we are rendering a different scene from LS.GlobalScene (used in renderMaterialPreview)
	*/
	renderFrame: function ( camera, render_settings, scene )
	{
		render_settings = render_settings || this.default_render_settings;

		//get all the data
		if(scene) //in case we use another scene than the default one
		{
			scene._frame++;
			this.processVisibleData( scene, render_settings );
		}
		this._current_scene = scene = scene || this._current_scene; //ugly, I know

		//set as active camera and set viewport
		this.enableCamera( camera, render_settings, render_settings.skip_viewport, scene ); 

		//clear buffer
		this.clearBuffer( camera, render_settings );

		//send before events
		LEvent.trigger(scene, EVENT.BEFORE_RENDER_SCENE, camera );
		LEvent.trigger(this, EVENT.BEFORE_RENDER_SCENE, camera );

		//in case the user wants to filter instances
		LEvent.trigger(this, EVENT.COMPUTE_VISIBILITY, this._visible_instances );

		//here we render all the instances
		if(this.onCustomRenderInstances)
			this.onCustomRenderInstances( render_settings, this._visible_instances );
		else
			this.renderInstances( render_settings, this._visible_instances );

		//send after events
		LEvent.trigger( scene, EVENT.AFTER_RENDER_SCENE, camera );
		LEvent.trigger( this, EVENT.AFTER_RENDER_SCENE, camera );
		if(this.onRenderScene)
			this.onRenderScene( camera, render_settings, scene);

		//render helpers (guizmos)
		if(render_settings.render_helpers)
		{
			if(GL.FBO.current) //rendering to multibuffer gives warnings if the shader outputs to a single fragColor
				GL.FBO.current.toSingle(); //so we disable multidraw for debug rendering (which uses a single render shader)
			LEvent.trigger(this, EVENT.RENDER_HELPERS, camera );
			LEvent.trigger(scene, EVENT.RENDER_HELPERS, camera );
			if(GL.FBO.current)
				GL.FBO.current.toMulti();
		}
	},

	//shows a RenderFrameContext to the viewport (warning, some components may do it bypassing this function)
	showRenderFrameContext: function( render_frame_context, camera )
	{
		//if( !this._current_render_settings.onPlayer)
		//	return;
		LEvent.trigger(this, EVENT.BEFORE_SHOW_FRAME_CONTEXT, render_frame_context );
		render_frame_context.show();
	},

	/**
	* Sets camera as the current camera, sets the viewport according to camera info, updates matrices, and prepares LS.Draw
	*
	* @method enableCamera
	* @param {Camera} camera
	* @param {RenderSettings} render_settings
	*/
	enableCamera: function(camera, render_settings, skip_viewport, scene )
	{
		scene = scene || this._current_scene || LS.GlobalScene;

		LEvent.trigger( camera, EVENT.BEFORE_CAMERA_ENABLED, render_settings );
		LEvent.trigger( scene, EVENT.BEFORE_CAMERA_ENABLED, camera );

		//assign viewport manually (shouldnt use camera.getLocalViewport to unify?)
		var startx = this._full_viewport[0];
		var starty = this._full_viewport[1];
		var width = this._full_viewport[2];
		var height = this._full_viewport[3];
		if(width == 0 && height == 0)
		{
			console.warn("enableCamera: full viewport was 0, assigning to full viewport");
			width = gl.viewport_data[2];
			height = gl.viewport_data[3];
		}

		var final_x = Math.floor(width * camera._viewport[0] + startx);
		var final_y = Math.floor(height * camera._viewport[1] + starty);
		var final_width = Math.ceil(width * camera._viewport[2]);
		var final_height = Math.ceil(height * camera._viewport[3]);

		if(!skip_viewport)
		{
			//force fullscreen viewport?
			if(render_settings && render_settings.ignore_viewports )
			{
				camera.final_aspect = this.global_aspect * camera._aspect * (width / height);
				gl.viewport( this._full_viewport[0], this._full_viewport[1], this._full_viewport[2], this._full_viewport[3] );
			}
			else
			{
				camera.final_aspect = this.global_aspect * camera._aspect * (final_width / final_height); //what if we want to change the aspect?
				gl.viewport( final_x, final_y, final_width, final_height );
			}
		}
		camera._last_viewport_in_pixels.set( gl.viewport_data );

		//recompute the matrices (view,proj and viewproj)
		camera.updateMatrices();

		//store matrices locally
		mat4.copy( this._view_matrix, camera._view_matrix );
		mat4.copy( this._projection_matrix, camera._projection_matrix );
		mat4.copy( this._viewprojection_matrix, camera._viewprojection_matrix );

		//safety in case something went wrong in the camera
		for(var i = 0; i < 16; ++i)
			if( isNaN( this._viewprojection_matrix[i] ) )
				console.warn("warning: viewprojection matrix contains NaN when enableCamera is used");

		//2D Camera: TODO: MOVE THIS SOMEWHERE ELSE
		mat4.ortho( this._2Dviewprojection_matrix, -1, 1, -1, 1, 1, -1 );

		//set as the current camera
		this._current_camera = camera;
		LS.Camera.current = camera;
		this._current_layers_filter = render_settings ? camera.layers & render_settings.layers : camera.layers;

		//Draw allows to render debug info easily
		if(LS.Draw)
		{
			LS.Draw.reset(); //clear 
			LS.Draw.setCamera( camera );
		}

		LEvent.trigger( camera, EVENT.AFTER_CAMERA_ENABLED, render_settings );
		LEvent.trigger( scene, EVENT.AFTER_CAMERA_ENABLED, camera ); //used to change stuff according to the current camera (reflection textures)
	},

	/**
	* Returns the camera active
	*
	* @method getCurrentCamera
	* @return {Camera} camera
	*/
	getCurrentCamera: function()
	{
		return this._current_camera;
	},

	/**
	* clear color using camera info ( background color, viewport scissors, clear depth, etc )
	*
	* @method clearBuffer
	* @param {Camera} camera
	* @param {LS.RenderSettings} render_settings
	*/
	clearBuffer: function( camera, render_settings )
	{
		if( render_settings.ignore_clear || (!camera.clear_color && !camera.clear_depth) )
			return;

		//scissors test for the gl.clear, otherwise the clear affects the full viewport
		gl.scissor( gl.viewport_data[0], gl.viewport_data[1], gl.viewport_data[2], gl.viewport_data[3] );
		gl.enable(gl.SCISSOR_TEST);

		//clear color buffer 
		gl.colorMask( true, true, true, true );
		gl.clearColor( camera.background_color[0], camera.background_color[1], camera.background_color[2], camera.background_color[3] );

		//clear depth buffer
		gl.depthMask( true );

		//to clear the stencil
		gl.enable( gl.STENCIL_TEST );
		gl.clearStencil( 0x0 );

		//do the clearing
		if(GL.FBO.current)
			GL.FBO.current.toSingle();
		gl.clear( ( camera.clear_color ? gl.COLOR_BUFFER_BIT : 0) | (camera.clear_depth ? gl.DEPTH_BUFFER_BIT : 0) | gl.STENCIL_BUFFER_BIT );
		if(GL.FBO.current)
			GL.FBO.current.toMulti();

		//in case of multibuffer we want to clear with black the secondary buffers with black
		if( GL.FBO.current )
			GL.FBO.current.clearSecondary( LS.ZEROS4 );
		/*
		if( fbo && fbo.color_textures.length > 1 && gl.extensions.WEBGL_draw_buffers )
		{
			var ext = gl.extensions.WEBGL_draw_buffers;
			var new_order = [gl.NONE];
			for(var i = 1; i < fbo.order.length; ++i)
				new_order.push(fbo.order[i]);
			ext.drawBuffersWEBGL( new_order );
			gl.clearColor( 0,0,0,0 );
			gl.clear( gl.COLOR_BUFFER_BIT );
			GL.FBO.current.toMulti();
		}
		*/

		gl.disable( gl.SCISSOR_TEST );
		gl.disable( gl.STENCIL_TEST );
	},

	//creates the separate render queues for every block of instances
	createRenderQueues: function()
	{
		this._queues.length = 0;

		this._renderqueue_background = this.addRenderQueue( new LS.RenderQueue( LS.RenderQueue.BACKGROUND, LS.RenderQueue.NO_SORT, { name: "BACKGROUND" } ));
		this._renderqueue_geometry = this.addRenderQueue( new LS.RenderQueue( LS.RenderQueue.GEOMETRY, LS.RenderQueue.SORT_NEAR_TO_FAR, { name: "GEOMETRY" } ));
		this._renderqueue_transparent = this.addRenderQueue( new LS.RenderQueue( LS.RenderQueue.TRANSPARENT, LS.RenderQueue.SORT_FAR_TO_NEAR, { name: "TRANSPARENT" } ));
		this._renderqueue_readback = this.addRenderQueue( new LS.RenderQueue( LS.RenderQueue.READBACK_COLOR, LS.RenderQueue.SORT_FAR_TO_NEAR , { must_clone_buffers: true, name: "READBACK" }));
		this._renderqueue_overlay = this.addRenderQueue( new LS.RenderQueue( LS.RenderQueue.OVERLAY, LS.RenderQueue.SORT_BY_PRIORITY, { name: "OVERLAY" }));
	},

	addRenderQueue: function( queue )
	{
		var index = Math.floor(queue.value * 0.1);
		if( this._queues[ index ] )
			console.warn("Overwritting render queue:", queue.name );
		this._queues[ index ] = queue;
		return queue;
	},

	//clears render queues and inserts objects according to their settings
	updateRenderQueues: function( camera, instances )
	{
		//compute distance to camera
		var camera_eye = camera.getEye( this._temp_cameye );
		for(var i = 0, l = instances.length; i < l; ++i)
		{
			var instance = instances[i];
			if(instance)
				instance._dist = vec3.dist( instance.center, camera_eye );
		}

		var queues = this._queues;

		//clear render queues
		for(var i = 0; i < queues.length; ++i)
			if(queues[i])
				queues[i].clear();

		//add to their queues
		for(var i = 0, l = instances.length; i < l; ++i)
		{
			var instance = instances[i];
			if( !instance || !instance.material || !instance._is_visible )
				continue;
			this.addInstanceToQueue( instance );
		}

		//sort queues
		for(var i = 0, l = queues.length; i < l; ++i)
		{
			var queue = queues[i];
			if(!queue || !queue.sort_mode || !queue.instances.length)
				continue;
			queue.sort();
		}
	},

	addInstanceToQueue: function(instance)
	{
		var queues = this._queues;
		var queue = null;
		var queue_index = -1;

		if( instance.material.queue == RenderQueue.AUTO || instance.material.queue == null ) 
		{
			if( instance.material._render_state.blend )
				queue = this._renderqueue_transparent;
			else
				queue = this._renderqueue_geometry;
		}
		else
		{
			//queue index use the tens digit
			queue_index = Math.floor( instance.material.queue * 0.1 );
			queue = queues[ queue_index ];
		}

		if( !queue ) //create new queue
		{
			queue = new LS.RenderQueue( queue_index * 10 + 5, LS.RenderQueue.NO_SORT );
			queues[ queue_index ] = queue;
		}

		if(queue)
			queue.add( instance );
		return queue;
	},

	/**
	* To set gl state to a known and constant state in every render pass
	*
	* @method resetGLState
	* @param {RenderSettings} render_settings
	*/
	resetGLState: function( render_settings )
	{
		render_settings = render_settings || this._current_render_settings;

		//maybe we should use this function instead
		//LS.RenderState.reset(); 

		gl.enable( gl.CULL_FACE );
		gl.frontFace(gl.CCW);

		gl.colorMask(true,true,true,true);

		gl.enable( gl.DEPTH_TEST );
		gl.depthFunc( gl.LESS );
		gl.depthMask(true);

		gl.disable( gl.BLEND );
		gl.blendFunc( gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA );

		gl.disable( gl.STENCIL_TEST );
		gl.stencilMask( 0xFF );
		gl.stencilOp( gl.KEEP, gl.KEEP, gl.KEEP );
		gl.stencilFunc( gl.ALWAYS, 1, 0xFF );
	},

	/**
	* Calls the render method for every RenderInstance (it also takes into account events and frustrum culling)
	*
	* @method renderInstances
	* @param {RenderSettings} render_settings
	* @param {Array} instances array of RIs, if not specified the last visible_instances are rendered
	*/
	renderInstances: function( render_settings, instances, scene )
	{
		scene = scene || this._current_scene;
		if(!scene)
		{
			console.warn("LS.Renderer.renderInstances: no scene found in LS.Renderer._current_scene");
			return 0;
		}

		this._rendered_passes += 1;

		var pass = this._current_pass;
		var camera = this._current_camera;
		var camera_index_flag = camera._rendering_index != -1 ? (1<<(camera._rendering_index)) : 0;
		var apply_frustum_culling = render_settings.frustum_culling;
		var frustum_planes = camera.updateFrustumPlanes();
		var layers_filter = this._current_layers_filter = camera.layers & render_settings.layers;

		LEvent.trigger( scene, EVENT.BEFORE_RENDER_INSTANCES, render_settings );
		//scene.triggerInNodes( EVENT.BEFORE_RENDER_INSTANCES, render_settings );

		//reset state of everything!
		this.resetGLState( render_settings );

		LEvent.trigger( scene, EVENT.RENDER_INSTANCES, render_settings );
		LEvent.trigger( this, EVENT.RENDER_INSTANCES, render_settings );

		//reset again!
		this.resetGLState( render_settings );

		/*
		var render_instance_func = pass.render_instance;
		if(!render_instance_func)
			return 0;
		*/

		var render_instances = instances || this._visible_instances;

		//global samplers
		this.bindSamplers( this._samplers );

		var instancing_data = this._instancing_data;

		//compute visibility pass: checks which RIs are visible from this camera according to its flags, layers and AABB
		for(var i = 0, l = render_instances.length; i < l; ++i)
		{
			//render instance
			var instance = render_instances[i];
			var node_flags = instance.node.flags;
			instance._is_visible = false;

			//hidden nodes
			if( pass == SHADOW_PASS && !(instance.material.flags.cast_shadows) )
				continue;
			if( pass == PICKING_PASS && node_flags.selectable === false )
				continue;
			if( (layers_filter & instance.layers) === 0 )
				continue;

			//done here because sometimes some nodes are moved in this action
			if(instance.onPreRender)
				if( instance.onPreRender( render_settings ) === false)
					continue;

			if(!instance.material) //in case something went wrong...
				continue;

			var material = camera._overwrite_material || instance.material;

			if(material.opacity <= 0) //TODO: remove this, do it somewhere else
				continue;

			//test visibility against camera frustum
			if( apply_frustum_culling && instance.use_bounding && !material.flags.ignore_frustum )
			{
				if(geo.frustumTestBox( frustum_planes, instance.aabb ) == CLIP_OUTSIDE )
					continue;
			}

			//save visibility info
			instance._is_visible = true;
			if(camera_index_flag) //shadowmap cameras dont have an index
				instance._camera_visibility |= camera_index_flag;
		}

		//separate in render queues, and sort them according to distance or priority
		this.updateRenderQueues( camera, render_instances, render_settings );

		var start = this._rendered_instances;
		var debug_instance = this._debug_instance;

		//process render queues
		for(var j = 0; j < this._queues.length; ++j)
		{
			var queue = this._queues[j];
			if(!queue || !queue.instances.length || !queue.enabled) //empty
				continue;

			//used to change RenderFrameContext stuff (cloning textures for refraction, etc)
			if(queue.start( pass, render_settings ) == false)
				continue;

			var render_instances = queue.instances;

			//for each render instance
			for(var i = 0, l = render_instances.length; i < l; ++i)
			{
				//render instance
				var instance = render_instances[i];

				//used to debug
				if(instance == debug_instance)
				{
					console.log(debug_instance);
					debugger; 
				}

				if( !instance._is_visible || !instance.mesh )
					continue;

				this._rendered_instances += 1;

				var material = camera._overwrite_material || instance.material;

				if( pass == PICKING_PASS && material.renderPickingInstance )
					material.renderPickingInstance( instance, render_settings, pass );
				else if( material.renderInstance )
					material.renderInstance( instance, render_settings, pass );
				else
					continue;

				//some instances do a post render action (DEPRECATED)
				if(instance.onPostRender)
					instance.onPostRender( render_settings );
			}

			queue.finish( pass, render_settings );
		}

		this.resetGLState( render_settings );

		LEvent.trigger( scene, EVENT.RENDER_SCREEN_SPACE, render_settings);

		//restore state
		this.resetGLState( render_settings );

		LEvent.trigger( scene, EVENT.AFTER_RENDER_INSTANCES, render_settings );
		LEvent.trigger( this, EVENT.AFTER_RENDER_INSTANCES, render_settings );

		//and finally again
		this.resetGLState( render_settings );

		return this._rendered_instances - start;
	},

	/*
	groupingInstances: function(instances)
	{
		//TODO: if material supports instancing WIP
		var instancing_supported = gl.webgl_version > 1 || gl.extensions["ANGLE_instanced_arrays"];
		if( instancing_supported && material._allows_instancing && !instance._shader_blocks.length )
		{
			var instancing_ri_info = null;
			if(!instancing_data[ material._index ] )
				instancing_data[ material._index ] = instancing_ri_info = [];
			instancing_ri_info.push( instance );
		}
	},
	*/

	renderGUI: function( render_settings )
	{
		//renders GUI items using mostly the Canvas2DtoWebGL library
		gl.viewport( this._full_viewport[0], this._full_viewport[1], this._full_viewport[2], this._full_viewport[3] ); //assign full viewport always?
		if(gl.start2D) //in case we have Canvas2DtoWebGL installed (it is optional)
			gl.start2D();
		if( render_settings.render_gui )
		{
			if( LEvent.hasBind( this._current_scene, EVENT.RENDER_GUI ) ) //to avoid forcing a redraw if no gui is set
			{
				if(LS.GUI)
					LS.GUI.ResetImmediateGUI(); //mostly to change the cursor (warning, true to avoid forcing redraw)
				LEvent.trigger( this._current_scene, EVENT.RENDER_GUI, gl );
			}
		}
		if( this.on_render_gui ) //used by the editor (here to ignore render_gui flag)
			this.on_render_gui( render_settings );
		if( gl.finish2D )
			gl.finish2D();
	},

	/**
	* returns a list of all the lights overlapping this instance (it uses sperical bounding so it could returns lights that are not really overlapping)
	* It is used by the multipass lighting to iterate 
	*
	* @method getNearLights
	* @param {RenderInstance} instance the render instance
	* @param {Array} result [optional] the output array
	* @return {Array} array containing a list of LS.Light affecting this RenderInstance
	*/
	getNearLights: function( instance, result )
	{
		result = result || [];

		result.length = 0; //clear old lights

		//it uses the lights gathered by prepareVisibleData
		var lights = this._active_lights;
		if(!lights || !lights.length)
			return result;

		//Compute lights affecting this RI (by proximity, only takes into account spherical bounding)
		result.length = 0;
		var numLights = lights.length;
		for(var j = 0; j < numLights; j++)
		{
			var light = lights[j];
			//same layer?
			if( (light.illuminated_layers & instance.layers) == 0 || (light.illuminated_layers & this._current_camera.layers) == 0)
				continue;
			var light_intensity = light.computeLightIntensity();
			//light intensity too low?
			if(light_intensity < 0.0001)
				continue;
			var light_radius = light.computeLightRadius();
			var light_pos = light.position;
			//overlapping?
			if( light_radius == -1 || instance.overlapsSphere( light_pos, light_radius ) )
				result.push( light );
		}

		return result;
	},

	regenerateShadowmaps: function( scene, render_settings )
	{
		scene = scene || this._current_scene;
		render_settings = render_settings || this.default_render_settings;
		LEvent.trigger( scene, EVENT.RENDER_SHADOWS, render_settings );
		for(var i = 0; i < this._active_lights.length; ++i)
		{
			var light = this._active_lights[i];
			light.prepare( render_settings );
			light.onGenerateShadowmap();
		}
	},

	mergeSamplers: function( samplers, result )
	{
		result = result || [];
		result.length = this._max_texture_units;

		for(var i = 0; i < result.length; ++i)
		{
			for(var j = samplers.length - 1; j >= 0; --j)
			{
				if(	samplers[j][i] )
				{
					result[i] = samplers[j][i];
					break;
				}
			}
		}

		return result;
	},

	//to be sure we dont have anything binded
	clearSamplers: function()
	{
		for(var i = 0; i < this._max_texture_units; ++i)
		{
			gl.activeTexture(gl.TEXTURE0 + i);
			gl.bindTexture( gl.TEXTURE_2D, null );
			gl.bindTexture( gl.TEXTURE_CUBE_MAP, null );
			this._active_samples[i] = null;
		}
	},

	bindSamplers: function( samplers )
	{
		if(!samplers)
			return;

		var allow_textures = this.allow_textures; //used for debug

		for(var slot = 0; slot < samplers.length; ++slot)
		{
			var sampler = samplers[slot];
			if(!sampler) 
				continue;

			//REFACTOR THIS
			var tex = null;
			if(sampler.constructor === String || sampler.constructor === GL.Texture) //old way
			{
				tex = sampler;
				sampler = null;
			}
			else if(sampler.texture)
				tex = sampler.texture;
			else //dont know what this var type is?
			{
				//continue; //if we continue the sampler slot will remain empty which could lead to problems
			}

			if( tex && tex.constructor === String)
				tex = LS.ResourcesManager.textures[ tex ];
			if(!allow_textures)
				tex = null;

			if(!tex)
			{
				if(sampler)
				{
					switch( sampler.missing )
					{
						case "black": tex = this._black_texture; break;
						case "white": tex = this._white_texture; break;
						case "gray": tex = this._gray_texture; break;
						case "normal": tex = this._normal_texture; break;
						case "cubemap": tex = this._white_cubemap_texture; break;
						default: 
							if(sampler.is_cubemap) //must be manually specified
								tex = this._white_cubemap_texture;
							else
								tex = this._missing_texture;
					}
				}
				else
					tex = this._missing_texture;
			}

			//avoid to read from the same texture we are rendering to (generates warnings)
			if(tex._in_current_fbo) 
				tex = this._missing_texture;

			tex.bind( slot );
			this._active_samples[slot] = tex;

			//texture properties
			if(sampler)// && sampler._must_update ) //disabled because samplers ALWAYS must set to the value, in case the same texture is used in several places in the scene
			{
				if(sampler.minFilter)
				{
					if( sampler.minFilter !== gl.LINEAR_MIPMAP_LINEAR || (GL.isPowerOfTwo( tex.width ) && GL.isPowerOfTwo( tex.height )) )
						gl.texParameteri(tex.texture_type, gl.TEXTURE_MIN_FILTER, sampler.minFilter);
				}
				if(sampler.magFilter)
					gl.texParameteri(tex.texture_type, gl.TEXTURE_MAG_FILTER, sampler.magFilter);
				if(sampler.wrap)
				{
					gl.texParameteri(tex.texture_type, gl.TEXTURE_WRAP_S, sampler.wrap);
					gl.texParameteri(tex.texture_type, gl.TEXTURE_WRAP_T, sampler.wrap);
				}
				if(sampler.anisotropic != null && gl.extensions.EXT_texture_filter_anisotropic )
					gl.texParameteri(tex.texture_type, gl.extensions.EXT_texture_filter_anisotropic.TEXTURE_MAX_ANISOTROPY_EXT, sampler.anisotropic );

				//sRGB textures must specified ON CREATION, so no
				//if(sampler.anisotropic != null && gl.extensions.EXT_sRGB )
				//sampler._must_update = false;
			}
		}
	},

	//Called at the beginning of processVisibleData 
	fillSceneUniforms: function( scene, render_settings )
	{
		//global uniforms
		var uniforms = scene._uniforms;
		uniforms.u_time = scene._time || getTime() * 0.001;
		uniforms.u_ambient_light = scene.info ? scene.info.ambient_color : vec3.create();

		this._samplers.length = 0;

		//clear globals
		this._global_textures.environment = null;

		//fetch global textures
		if(scene.info)
		for(var i in scene.info.textures)
		{
			var texture = LS.getTexture( scene.info.textures[i] );
			if(!texture)
				continue;

			var slot = 0;
			if( i == "environment" )
				slot = LS.Renderer.ENVIRONMENT_TEXTURE_SLOT;
			else
				continue; 

			var type = (texture.texture_type == gl.TEXTURE_2D ? "_texture" : "_cubemap");
			if(texture.texture_type == gl.TEXTURE_2D)
			{
				texture.bind(0);
				texture.setParameter( gl.TEXTURE_MIN_FILTER, gl.LINEAR ); //avoid artifact
			}
			this._samplers[ slot ] = texture;
			scene._uniforms[ i + "_texture" ] = slot; 
			scene._uniforms[ i + type ] = slot; //LEGACY

			if( i == "environment" )
				this._global_textures.environment = texture;
		}

		LEvent.trigger( scene, EVENT.FILL_SCENE_UNIFORMS, scene._uniforms );
	},	

	/**
	* Collects and process the rendering instances, cameras and lights that are visible
	* Its a prepass shared among all rendering passes
	* Called ONCE per frame from LS.Renderer.render before iterating cameras
	* Warning: rendering order is computed here, so it is shared among all the cameras (TO DO, move somewhere else)
	*
	* @method processVisibleData
	* @param {Scene} scene
	* @param {RenderSettings} render_settings
	* @param {Array} cameras in case you dont want to use the scene cameras
	*/
	processVisibleData: function( scene, render_settings, cameras, instances, skip_collect_data )
	{
		//options = options || {};
		//options.scene = scene;
		var frame = scene._frame;
		instances = instances || scene._instances;

		this._current_scene = scene;
		//compute global scene info
		this.fillSceneUniforms( scene, render_settings );

		//update info about scene (collecting it all or reusing the one collected in the frame before)
		if(!skip_collect_data)
		{
			if( this._frame % this._collect_frequency == 0)
				scene.collectData( cameras );
			LEvent.trigger( scene, EVENT.AFTER_COLLECT_DATA, scene );
		}

		//set cameras: use the parameters ones or the ones found in the scene
		cameras = (cameras && cameras.length) ? cameras : scene._cameras;
		if( cameras.length == 0 )
		{
			console.error("no cameras found");
			return;
		}
				
		//find which materials are going to be seen
		var materials = this._visible_materials; 
		materials.length = 0;

		//prepare cameras: TODO: sort by priority
		for(var i = 0, l = cameras.length; i < l; ++i)
		{
			var camera = cameras[i];
			camera._rendering_index = i;
			camera.prepare();
			if(camera.overwrite_material)
			{
				var material = camera.overwrite_material.constructor === String ? LS.ResourcesManager.resources[ camera.overwrite_material ] : camera.overwrite_material;
				if(material)
				{
					camera._overwrite_material = material;
					materials.push( material );
				}
			}
			else
				camera._overwrite_material = null;
		}

		//define the main camera (the camera used for some algorithms)
		if(!this._main_camera)
		{
			if( cameras.length )
				this._main_camera = cameras[0];
			else
				this._main_camera = new LS.Camera(); // ??
		}

		//nearest reflection probe to camera
		var nearest_reflection_probe = scene.findNearestReflectionProbe( this._main_camera.getEye() );

		//process instances
		this.processRenderInstances( instances, materials, scene, render_settings );

		//store all the info
		this._visible_instances = scene._instances;
		this._active_lights = scene._lights;
		this._visible_cameras = cameras; 
		//this._visible_materials = materials;

		//prepare lights (collect data and generate shadowmaps)
		for(var i = 0, l = this._active_lights.length; i < l; ++i)
			this._active_lights[i].prepare( render_settings );

		LEvent.trigger( scene, EVENT.AFTER_COLLECT_DATA, scene );
	},

	//this processes the instances 
	processRenderInstances: function( instances, materials, scene, render_settings )
	{
		materials = materials || this._visible_materials;
		var frame = scene._frame;
		render_settings = render_settings || this._current_render_settings;

		//process render instances (add stuff if needed, gather materials)
		for(var i = 0, l = instances.length; i < l; ++i)
		{
			var instance = instances[i];
			if(!instance)
				continue;

			var node_flags = instance.node.flags;

			if(!instance.mesh)
			{
				console.warn("RenderInstance must always have mesh");
				continue;
			}

			//materials
			if(!instance.material)
				instance.material = this.default_material;

			if( instance.material._last_frame_update != frame )
			{
				instance.material._last_frame_update = frame;
				materials.push( instance.material );
			}

			//add extra info: distance to main camera (used for sorting)
			instance._dist = 0;

			//find nearest reflection probe
			if( scene._reflection_probes.length && !this._ignore_reflection_probes )
				instance._nearest_reflection_probe = scene.findNearestReflectionProbe( instance.center ); //nearest_reflection_probe;
			else
				instance._nearest_reflection_probe = null;

			//change conditionaly
			if(render_settings.force_wireframe && instance.primitive != gl.LINES ) 
			{
				instance.primitive = gl.LINES;
				if(instance.mesh)
				{
					if(!instance.mesh.indexBuffers["wireframe"])
						instance.mesh.computeWireframe();
					instance.index_buffer = instance.mesh.indexBuffers["wireframe"];
				}
			}

			//clear camera visibility mask (every flag represents a camera index)
			instance._camera_visibility = 0|0;
			instance.index = i;
		}

		//prepare materials 
		for(var i = 0; i < materials.length; ++i)
		{
			var material = materials[i];
			material._index = i;
			if( material.prepare )
				material.prepare( scene );
		}

		LEvent.trigger( scene, EVENT.PREPARE_MATERIALS );
	},

	/**
	* Renders a frame into a texture (could be a cubemap, in which case does the six passes)
	*
	* @method renderInstancesToRT
	* @param {Camera} cam
	* @param {Texture} texture
	* @param {RenderSettings} render_settings
	*/
	renderInstancesToRT: function( cam, texture, render_settings, instances )
	{
		render_settings = render_settings || this.default_render_settings;
		this._current_target = texture;
		var scene = LS.Renderer._current_scene;
		texture._in_current_fbo = true;

		if(texture.texture_type == gl.TEXTURE_2D)
		{
			this.enableCamera(cam, render_settings);
			texture.drawTo( inner_draw_2d );
		}
		else if( texture.texture_type == gl.TEXTURE_CUBE_MAP)
			this.renderToCubemap( cam.getEye(), texture.width, texture, render_settings, cam.near, cam.far );
		this._current_target = null;
		texture._in_current_fbo = false;

		function inner_draw_2d()
		{
			LS.Renderer.clearBuffer( cam, render_settings );
			/*
			gl.clearColor(cam.background_color[0], cam.background_color[1], cam.background_color[2], cam.background_color[3] );
			if(render_settings.ignore_clear != true)
				gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
			*/
			//render scene
			LS.Renderer.renderInstances( render_settings, instances );
		}
	},

	/**
	* Renders the current scene to a cubemap centered in the given position
	*
	* @method renderToCubemap
	* @param {vec3} position center of the camera where to render the cubemap
	* @param {number} size texture size
	* @param {Texture} texture to reuse the same texture
	* @param {RenderSettings} render_settings
	* @param {number} near
	* @param {number} far
	* @return {Texture} the resulting texture
	*/
	renderToCubemap: function( position, size, texture, render_settings, near, far, background_color, instances )
	{
		size = size || 256;
		near = near || 1;
		far = far || 1000;

		if(render_settings && render_settings.constructor !== LS.RenderSettings)
			throw("render_settings parameter must be LS.RenderSettings.");

		var eye = position;
		if( !texture || texture.constructor != GL.Texture)
			texture = null;

		var scene = this._current_scene;
		if(!scene)
			scene = this._current_scene = LS.GlobalScene;

		var camera = this._cubemap_camera;
		if(!camera)
			camera = this._cubemap_camera = new LS.Camera();
		camera.configure({ fov: 90, aspect: 1.0, near: near, far: far });

		texture = texture || new GL.Texture(size,size,{texture_type: gl.TEXTURE_CUBE_MAP, minFilter: gl.NEAREST});
		this._current_target = texture;
		texture._in_current_fbo = true; //block binding this texture during rendering of the reflection

		texture.drawTo( function(texture, side) {

			var info = LS.Camera.cubemap_camera_parameters[side];
			if(texture._is_shadowmap || !background_color )
				gl.clearColor(0,0,0,0);
			else
				gl.clearColor( background_color[0], background_color[1], background_color[2], background_color[3] );
			gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
			camera.configure({ eye: eye, center: [ eye[0] + info.dir[0], eye[1] + info.dir[1], eye[2] + info.dir[2]], up: info.up });

			LS.Renderer.enableCamera( camera, render_settings, true );
			LS.Renderer.renderInstances( render_settings, instances, scene );
		});

		this._current_target = null;
		texture._in_current_fbo = false;
		return texture;
	},

	/**
	* Returns the last camera that falls into a given screen position
	*
	* @method getCameraAtPosition
	* @param {number} x in canvas coordinates (0,0 is bottom-left)
	* @param {number} y in canvas coordinates (0,0 is bottom-left)
	* @param {Scene} scene if not specified last rendered scene will be used
	* @return {Camera} the camera
	*/
	getCameraAtPosition: function(x,y, cameras)
	{
		cameras = cameras || this._visible_cameras;
		if(!cameras || !cameras.length)
			return null;

		for(var i = cameras.length - 1; i >= 0; --i)
		{
			var camera = cameras[i];
			if(!camera.enabled || camera.render_to_texture)
				continue;

			if( camera.isPoint2DInCameraViewport(x,y) )
				return camera;
		}
		return null;
	},

	setRenderPass: function( pass )
	{
		if(!pass)
			pass = COLOR_PASS;
		this._current_pass = pass;
	},

	addImmediateRenderInstance: function( instance )
	{
		if(!instance.material)
			return;

		//this is done in collect so...
		instance.updateAABB(); 

		//add material to the list of visible materials
		if( instance.material._last_frame_update != this._frame )
		{
			instance.material._last_frame_update = this._frame;
			this._visible_materials.push( instance.material );
			if( instance.material.prepare )
				instance.material.prepare( this._current_scene );
		}

		this.addInstanceToQueue( instance );

		this._visible_instances.push( instance );
	},
	
	/**
	* Enables a ShaderBlock ONLY DURING THIS FRAME
	* must be called during frame rendering (event like fillSceneUniforms)
	*
	* @method enableFrameShaderBlock
	* @param {String} shader_block_name
	*/
	enableFrameShaderBlock: function( shader_block_name, uniforms, samplers )
	{
		var shader_block = shader_block_name.constructor === LS.ShaderBlock ? shader_block_name : LS.Shaders.getShaderBlock( shader_block_name );

		if( !shader_block || this._global_shader_blocks_flags & shader_block.flag_mask )
			return; //already added

		this._global_shader_blocks.push( shader_block );
		this._global_shader_blocks_flags |= shader_block.flag_mask;

		//add uniforms to renderer uniforms?
		if(uniforms)
			for(var i in uniforms)
				this._uniforms[i] = uniforms[i];

		if(samplers)
			for(var i = 0; i < samplers.length; ++i)
				if( samplers[i] )
					this._samplers[i] = samplers[i];
	},

	/**
	* Disables a ShaderBlock ONLY DURING THIS FRAME
	* must be called during frame rendering (event like fillSceneUniforms)
	*
	* @method disableFrameShaderBlock
	* @param {String} shader_block_name
	*/
	disableFrameShaderBlock:  function( shader_block_name, uniforms, samplers )
	{
		var shader_block = shader_block_name.constructor === LS.ShaderBlock ? shader_block_name : LS.Shaders.getShaderBlock( shader_block_name );
		if( !shader_block || !(this._global_shader_blocks_flags & shader_block.flag_mask) )
			return; //not active

		var index = this._global_shader_blocks.indexOf( shader_block );
		if(index != -1)
			this._global_shader_blocks.splice( index, 1 );
		this._global_shader_blocks_flags &= ~( shader_block.flag_mask ); //disable bit
	},

	//time queries for profiling
	_current_query: null,

	startGPUQuery: function( name )
	{
		if(!gl.extensions["disjoint_timer_query"] || !this.timer_queries_enabled) //if not supported
			return;
		if(this._waiting_queries)
			return;
		var ext = gl.extensions["disjoint_timer_query"];
		var query = this._timer_queries[ name ];
		if(!query)
			query = this._timer_queries[ name ] = ext.createQueryEXT();
		ext.beginQueryEXT( ext.TIME_ELAPSED_EXT, query );
		this._current_query = query;
	},

	endGPUQuery: function()
	{
		if(!gl.extensions["disjoint_timer_query"] || !this.timer_queries_enabled) //if not supported
			return;
		if(this._waiting_queries)
			return;
		var ext = gl.extensions["disjoint_timer_query"];
		ext.endQueryEXT( ext.TIME_ELAPSED_EXT );
		this._current_query = null;
	},

	resolveQueries: function()
	{
		if(!gl.extensions["disjoint_timer_query"] || !this.timer_queries_enabled) //if not supported
			return;

		//var err = gl.getError();
		//if(err != gl.NO_ERROR)
		//	console.log("GL_ERROR: " + err );

		var ext = gl.extensions["disjoint_timer_query"];

		var last_query = this._timer_queries["gui"];
		if(!last_query)
			return;

		var available = ext.getQueryObjectEXT( last_query, ext.QUERY_RESULT_AVAILABLE_EXT );
		if(!available)
		{
			this._waiting_queries = true;
			return;
		}
	
		var disjoint = gl.getParameter( ext.GPU_DISJOINT_EXT );
		if(!disjoint)
		{
			var total = 0;
			for(var i in this._timer_queries)
			{
				var query = this._timer_queries[i];
				// See how much time the rendering of the object took in nanoseconds.
				var timeElapsed = ext.getQueryObjectEXT( query, ext.QUERY_RESULT_EXT ) * 10e-6; //to milliseconds;
				total += timeElapsed;
				this.gpu_times[ i ] = timeElapsed;
				//ext.deleteQueryEXT(query);
				//this._timer_queries[i] = null;
			}
			this.gpu_times.total = total;
		}

		this._waiting_queries = false;
	},

	profiler_text: [],

	renderProfiler: function()
	{
		if(!gl.canvas.canvas2DtoWebGL_enabled)
			return;

		var text = this.profiler_text;
		var ext = gl.extensions["disjoint_timer_query"];

		if(this._frame % 5 == 0)
		{
			text.length = 0;
			var fps = 1000 / this._frame_time;
			text.push( fps.toFixed(2) + " FPS" );
			text.push( "CPU: " + this._frame_cpu_time.toFixed(2) + " ms" );
			text.push( " - Passes: " + this._rendered_passes );
			text.push( " - RIs: " + this._rendered_instances );
			text.push( " - Draws: " + this._rendercalls );

			if( ext )
			{
				text.push( "GPU: " + this.gpu_times.total.toFixed(2) );
				text.push( " - PreRender: " + this.gpu_times.beforeRender.toFixed(2) );
				text.push( " - Shadows: " + this.gpu_times.shadows.toFixed(2) );
				text.push( " - Reflections: " + this.gpu_times.reflections.toFixed(2) );
				text.push( " - Scene: " + this.gpu_times.main.toFixed(2) );
				text.push( " - Postpo: " + this.gpu_times.postpo.toFixed(2) );
				text.push( " - GUI: " + this.gpu_times.gui.toFixed(2) );
			}
			else
				text.push( "GPU: ???");
		}

		var ctx = gl;
		ctx.save();
		ctx.translate( gl.canvas.width - 200, gl.canvas.height - 280 );
		ctx.globalAlpha = 0.7;
		ctx.font = "14px Tahoma";
		ctx.fillStyle = "black";
		ctx.fillRect(0,0,200,280);
		ctx.fillStyle = "white";
		ctx.fillText( "Profiler", 20, 20 );
		ctx.fillStyle = "#AFA";
		for(var i = 0; i < text.length; ++i)
			ctx.fillText( text[i], 20,50 + 20 * i );
		ctx.restore();
	},

	/**
	* Renders one texture into another texture, it allows to apply a shader
	*
	* @method blit
	* @param {GL.Texture} source
	* @param {GL.Texture} destination
	* @param {GL.Shader} shader [optional] shader to apply, it must use the GL.Shader.QUAD_VERTEX_SHADER as vertex shader
	* @param {Object} uniforms [optional] uniforms for the shader
	*/
	blit: function( source, destination, shader, uniforms )
	{
		if(!source || !destination)
			throw("data missing in blit");

		if(source != destination)
		{
			destination.drawTo( function(){
				source.toViewport( shader, uniforms );
			});
			return;
		}

		if(!shader)
			throw("blitting texture to the same texture doesnt makes sense unless a shader is specified");

		var temp = GL.Texture.getTemporary( source.width, source.height, source );
		source.copyTo( temp );
		temp.copyTo( source, shader, uniforms );
		GL.Texture.releaseTemporary( temp );
	}
};

//Add to global Scope
LS.Renderer = Renderer;



///@FILE:../src/render/debug.js
///@INFO: UNCOMMON
/**	DebugRender
* Used to render debug information like skeletons, a grid, etc
* I moved it from WebGLStudio to LS so it could help when working with scenes coded without the editor
*
* @class DebugRender
* @namespace LS
* @constructor
*/
function DebugRender()
{
	this.debug_points = []; //used for debugging, allows to draw points easily

	//current frame data to render (we store it so we can render with less drawcalls)
	this._points = []; //linear array with x,y,z, x,y,z, ...
	this._points_color = [];
	this._points_nodepth = []; //linear array with x,y,z, x,y,z, ...
	this._points_color_nodepth = [];
	this._lines = []; //vec3,vec3 array
	this._lines_color = []; //
	this._names = []; //array of [vec3, string]

	this.grid_texture_url = "imgs/grid.png";

	//this camera is used to render names
	this.camera2D = new LS.Camera({eye:[0,0,0],center:[0,0,-1]});
	this.createMeshes();

	this.colors = {
		selected: vec4.fromValues(1,1,1,1),
		node: vec4.fromValues(1,0.5,0,1),
		bone: vec4.fromValues(1,0,0.5,1)
	};

	this.settings = {
		render_grid: true,
		grid_scale: 1.0,
		grid_alpha: 0.5,
		grid_plane: "xz",
		render_names: false,
		render_skeletons: true,
		render_tree: false,
		render_components: true,
		render_null_nodes: true,
		render_axis: false,
		render_colliders: true,
		render_paths: true,
		render_origin: true,
		render_colliders_aabb: false
	};

	this._in_scene = false;
}

DebugRender.prototype.enable = function( scene )
{
	if(this._in_scene)
		return;
	scene = scene || LS.GlobalScene;
	LEvent.bind( scene, "afterRenderInstances", this.onRender, this );
	this._in_scene = scene;
}

DebugRender.prototype.disable = function( scene )
{
	if(!this._in_scene)
		return;
	LEvent.unbind( this._in_scene, "afterRenderInstances", this.onRender, this );
	this._in_scene = null;
}

DebugRender.prototype.onRender = function( e, render_settings )
{
	this.render( LS.Renderer._current_camera );
}

//we pass a callback to check if something is selected
DebugRender.prototype.render = function( camera, is_selected_callback, scene )
{
	var settings = this.settings;

	scene = scene || LS.GlobalScene;

	gl.blendFunc(gl.SRC_ALPHA,gl.ONE_MINUS_SRC_ALPHA);
	gl.enable( gl.DEPTH_TEST );
	gl.disable(gl.BLEND);
	gl.disable( gl.CULL_FACE );
	gl.depthFunc( gl.LEQUAL );
	//gl.depthMask( false );
	var selected_node = null;

	if( settings.render_grid && settings.grid_alpha > 0 )
		this.renderGrid();

	if( settings.render_origin )
	{
		LS.Draw.setColor([0.3,0.3,0.3,1.0]);
		LS.Draw.push();
		LS.Draw.scale(0.01,0.01,0.01);
		LS.Draw.rotate(-90,[1,0,0]);
		gl.blendFunc(gl.SRC_ALPHA,gl.ONE);
		LS.Draw.renderText("Origin");
		gl.blendFunc(gl.SRC_ALPHA,gl.ONE_MINUS_SRC_ALPHA);
		LS.Draw.pop();
	}

	if( settings.render_components )
	{
		//Node components
		for(var i = 0, l = scene._nodes.length; i < l; ++i)
		{
			var node = scene._nodes[i];
			var is_node_selected = node._is_selected;
			selected_node = node;
			if(node.renderEditor)
				node.renderEditor( is_node_selected );
			for(var j = 0, l2 = node._components.length; j < l2; ++j)
			{
				var component = node._components[j];
				var is_component_selected = false;
				if(is_selected_callback)
					is_component_selected = is_selected_callback( component );
				if(component.renderEditor)
					component.renderEditor( is_node_selected, is_component_selected );
			}
		}
	}

	//render local things		
	var zero = vec3.create();
	for(var i = 0, l = scene._nodes.length; i < l; ++i)
	{
		var node = scene._nodes[i];
		if(node._is_root || !node.flags.visible ) 
			continue;

		var global = node.transform ? node.transform.getGlobalMatrixRef() : mat4.create();
		var pos = mat4.multiplyVec3( vec3.create(), global, zero ); //create a new one to store them

		if( settings.render_null_nodes)
		{
			if( node._is_selected )
				this.renderPoint( pos, true, this.colors.selected );
			else if( node._is_bone )
				this.renderPoint( pos, true, this.colors.bone );
			else
				this.renderPoint( pos, false, this.colors.node );
		}

		if(settings.render_names)
			this.renderText(pos, node.name, node._is_selected ? [0.94, 0.8, 0.4,1] : [0.8,0.8,0.8,0.9] );

		if (node._parentNode && node._parentNode.transform && (settings.render_tree || (settings.render_skeletons && node._is_bone && node._parentNode._is_bone)) )
		{
			this.renderLine( pos , node._parentNode.transform.getGlobalPosition(), this.colors.bone );
			//this.renderPoint( pos, true, this.colors.bone );
		}

		if( settings.render_axis )
		{
			LS.Draw.push();
			LS.Draw.multMatrix(global);
			LS.Draw.setColor([1,1,1,1]);
			LS.Draw.renderMesh( this.axis_mesh, gl.LINES );
			LS.Draw.pop();
		}
	}

	if( settings.render_colliders )
		this.renderColliders( scene );
	if( settings.render_paths )
		this.renderPaths( scene );

	//Render primitives (points, lines, text) ***********************

	if( this._points.length )
	{
		LS.Draw.setPointSize(4);
		LS.Draw.setColor([1,1,1,1]);
		LS.Draw.renderPoints( this._points, this._points_color );
		this._points.length = 0;
		this._points_color.length = 0;
	}

	if( this._points_nodepth.length )
	{
		LS.Draw.setPointSize(4);
		LS.Draw.setColor([1,1,1,1]);
		gl.disable( gl.DEPTH_TEST );
		LS.Draw.renderPoints( this._points_nodepth, this._points_color_nodepth );
		gl.enable( gl.DEPTH_TEST );
		this._points_nodepth.length = 0;
		this._points_color_nodepth.length = 0;
	}

	if( this._lines.length )
	{
		gl.disable( gl.DEPTH_TEST );
		LS.Draw.setColor([1,1,1,1]);
		LS.Draw.renderLines( this._lines, this._lines_color );
		gl.enable( gl.DEPTH_TEST );
		this._lines.length = 0;
		this._lines_color.length = 0;
	}

	if( this.debug_points.length )
	{
		LS.Draw.setPointSize(5);
		LS.Draw.setColor([1,0,1,1]);
		LS.Draw.renderPoints( this.debug_points );
	}

	//this require Canvas2DtoWebGL library
	if( settings.render_names && gl.start2D )
	{
		gl.disable( gl.DEPTH_TEST );
		var camera2D = this.camera2D;
		var viewport = gl.getViewport();
		camera2D.setOrthographic(0,viewport[2], 0,viewport[3], -1,1);
		camera2D.updateMatrices();
		gl.start2D();
		//gl.disable( gl.BLEND );
		gl.font = "14px Arial";
		var black_color = vec4.fromValues(0,0,0,0.5);

		for(var i = 0; i < this._names.length; ++i)
		{
			var pos2D = camera.project( this._names[i][1] );
			if(pos2D[2] < 0)
				continue;
			pos2D[2] = 0;

			var text_size = gl.measureText( this._names[i][0] );
			gl.fillColor = black_color;
			gl.fillRect( Math.floor(pos2D[0] + 10), viewport[3] - (Math.floor(pos2D[1] + 8)), text_size.width, text_size.height );
			gl.fillColor = this._names[i][2];
			gl.fillText( this._names[i][0], Math.floor(pos2D[0] + 10), viewport[3] - (Math.floor(pos2D[1] - 4) ) );
		}
		gl.finish2D();
		this._names.length = 0;
	}

	//DEBUG
	if(settings.render_axis && selected_node && selected_node.transform ) //render axis for all nodes
	{
		LS.Draw.push();
		var Q = selected_node.transform.getGlobalRotation();
		var R = mat4.fromQuat( mat4.create(), Q );
		LS.Draw.setMatrix( R );
		LS.Draw.setColor([1,1,1,1]);
		LS.Draw.scale(10,10,10);
		LS.Draw.renderMesh( this.axis_mesh, gl.LINES );
		LS.Draw.pop();
	}

	gl.depthFunc( gl.LESS );
}

//this primitives are rendered after all the components editors are rendered
DebugRender.prototype.renderPoint = function( p, ignore_depth, c )
{
	c = c || [1,1,1,1];
	if(ignore_depth)
	{
		this._points_nodepth.push( p[0], p[1], p[2] );
		this._points_color_nodepth.push( c[0], c[1], c[2], c[3] );
	}
	else
	{
		this._points.push( p[0], p[1], p[2] );
		this._points_color.push( c[0], c[1], c[2], c[3] );
	}
}

DebugRender.prototype.renderLine = function( start, end, color )
{
	color = color || [1,1,1,1];
	this._lines.push( start, end );
	this._lines_color.push( color, color );
}

DebugRender.prototype.renderText = function( position, text, color )
{
	color = color || [1,1,1,1];
	this._names.push([text,position, color]);
}

DebugRender.prototype.renderGrid = function()
{
	var settings = this.settings;

	//textured grid
	if(!this.grid_shader)
	{
		//this.grid_shader = LS.Draw.createSurfaceShader("float PI2 = 6.283185307179586; return vec4( vec3( max(0.0, cos(pos.x * PI2 * 0.1) - 0.95) * 10.0 + max(0.0, cos(pos.z * PI2 * 0.1) - 0.95) * 10.0 ),1.0);");
		this.grid_shader = LS.Draw.createSurfaceShader("vec2 f = vec2(1.0/64.0,-1.0/64.0); float brightness = texture2D(u_texture, pos.xz + f).x * 0.6 + texture2D(u_texture, pos.xz * 0.1 + f ).x * 0.3 + texture2D(u_texture, pos.xz * 0.01 + f ).x * 0.2; brightness /= max(1.0,0.001 * length(u_camera_position.xz - pos.xz));vec4 color = u_color * vec4(vec3(1.0),brightness); if( abs(pos.x) < 0.1 ) color = mix(vec4(0.4,0.4,1.0,0.5),color,abs(pos.x/0.1)); if( abs(pos.z) < 0.1 ) color = mix(vec4(1.0,0.4,0.4,0.5),color,abs(pos.z/0.1)); return color;");
		//this.grid_shader = LS.Draw.createSurfaceShader("vec2 f = vec2(1.0/64.0,-1.0/64.0); float brightness = texture2D(u_texture, pos.xz + f).x * 0.6 + texture2D(u_texture, pos.xz * 0.1 + f ).x * 0.3 + texture2D(u_texture, pos.xz * 0.01 + f ).x * 0.2; brightness /= max(1.0,0.001 * length(u_camera_position.xz - pos.xz));vec4 color = u_color * vec4(vec3(1.0),brightness); return color;");
		this.grid_shader_xy = LS.Draw.createSurfaceShader("vec2 f = vec2(1.0/64.0,-1.0/64.0); float brightness = texture2D(u_texture, pos.xy + f).x * 0.6 + texture2D(u_texture, pos.xy * 0.1 + f ).x * 0.3 + texture2D(u_texture, pos.xy * 0.01 + f ).x * 0.2; brightness /= max(1.0,0.001 * length(u_camera_position.xy - pos.xy));vec4 color = u_color * vec4(vec3(1.0),brightness);  if( abs(pos.x) < 0.025 ) color *= vec4(0.4,1.0,0.4,1.0); if( abs(pos.y) < 0.025 ) color *= vec4(1.0,0.4,0.4,1.0); return color;");
		//this.grid_shader_xy = LS.Draw.createSurfaceShader("vec2 f = vec2(1.0/64.0,-1.0/64.0); float brightness = texture2D(u_texture, pos.xy + f).x * 0.6 + texture2D(u_texture, pos.xy * 0.1 + f ).x * 0.3 + texture2D(u_texture, pos.xy * 0.01 + f ).x * 0.2; brightness /= max(1.0,0.001 * length(u_camera_position.xy - pos.xy));return u_color * vec4(vec3(1.0),brightness);");
		this.grid_shader_yz = LS.Draw.createSurfaceShader("vec2 f = vec2(1.0/64.0,-1.0/64.0); float brightness = texture2D(u_texture, pos.yz + f).x * 0.6 + texture2D(u_texture, pos.yz * 0.1 + f ).x * 0.3 + texture2D(u_texture, pos.yz * 0.01 + f ).x * 0.2; brightness /= max(1.0,0.001 * length(u_camera_position.yz - pos.yz)); vec4 color = u_color * vec4(vec3(1.0),brightness);  if( abs(pos.y) < 0.025 ) color *= vec4(0.4, 0.4, 1.0, 1.0); if( abs(pos.z) < 0.025 ) color *= vec4(0.4,1.0,0.4,1.0); return color;");
		//this.grid_shader_yz = LS.Draw.createSurfaceShader("vec2 f = vec2(1.0/64.0,-1.0/64.0); float brightness = texture2D(u_texture, pos.yz + f).x * 0.6 + texture2D(u_texture, pos.yz * 0.1 + f ).x * 0.3 + texture2D(u_texture, pos.yz * 0.01 + f ).x * 0.2; brightness /= max(1.0,0.001 * length(u_camera_position.yz - pos.yz));return u_color * vec4(vec3(1.0),brightness);");
		this.grid_shader.uniforms({u_texture:0});

		if( this.grid_img && this.grid_img.loaded )
			this.grid_texture = GL.Texture.fromImage( this.grid_img, {format: gl.RGB, wrap: gl.REPEAT, anisotropic: 4, minFilter: gl.LINEAR_MIPMAP_LINEAR } );
		else
			this.grid_texture = GL.Texture.fromURL( this.grid_texture_url, {format: gl.RGB, wrap: gl.REPEAT, anisotropic: 4, minFilter: gl.LINEAR_MIPMAP_LINEAR } );
	}

	LS.Draw.push();

	if(settings.grid_plane == "xy")
		LS.Draw.rotate(90,1,0,0);
	else if(settings.grid_plane == "yz")
		LS.Draw.rotate(90,0,0,1);


	if(!this.grid_texture || this.grid_texture.ready === false)
	{
		var grid_scale = 1;			
		var grid_alpha = 1;
		//lines grid
		LS.Draw.setColor([0.2,0.2,0.2, grid_alpha * 0.75]);
		LS.Draw.scale( grid_scale , grid_scale , grid_scale );
		LS.Draw.renderMesh( this.grid_mesh, gl.LINES );
		LS.Draw.scale(10,10,10);
		LS.Draw.renderMesh( this.grid_mesh, gl.LINES );
	}
	else
	{
		//texture grid
		gl.enable( gl.POLYGON_OFFSET_FILL );
		gl.depthFunc( gl.LEQUAL );
		gl.polygonOffset(1,-100.0);
		gl.enable(gl.BLEND);
		this.grid_texture.bind(0);
		gl.depthMask( false );
		LS.Draw.setColor([1,1,1, this.settings.grid_alpha ]);
		LS.Draw.translate( LS.Draw.camera_position[0], 0, LS.Draw.camera_position[2] ); //follow camera
		LS.Draw.scale( 10000, 10000, 10000 );
		LS.Draw.renderMesh( this.plane_mesh, gl.TRIANGLES, settings.grid_plane == "xy" ? this.grid_shader_xy : (settings.grid_plane == "yz" ? this.grid_shader_yz : this.grid_shader) );
		gl.depthMask( true );
		gl.depthFunc( gl.LESS );
		gl.disable( gl.POLYGON_OFFSET_FILL );
		gl.polygonOffset(0,0);
	}

	LS.Draw.pop();
}

DebugRender.prototype.renderColliders = function( scene )
{
	scene = scene || LS.GlobalScene;
	if(!scene._colliders)
		return;

	LS.Draw.setColor([0.33,0.71,0.71,0.5]);

	for(var i = 0; i < scene._colliders.length; ++i)
	{
		var instance = scene._colliders[i];
		var oobb = instance.oobb;

		if(this.settings.render_colliders_aabb) //render AABB
		{
			var aabb = instance.aabb;
			LS.Draw.push();
			var center = BBox.getCenter(aabb);
			var halfsize = BBox.getHalfsize(aabb);
			LS.Draw.translate(center);
			//LS.Draw.setColor([0.33,0.71,0.71,0.5]);
			LS.Draw.renderWireBox(halfsize[0]*2,halfsize[1]*2,halfsize[2]*2);
			LS.Draw.pop();
		}

		LS.Draw.push();
		LS.Draw.multMatrix( instance.matrix );
		var halfsize = BBox.getHalfsize(oobb);

		if(instance.type == LS.PhysicsInstance.BOX)
		{
			LS.Draw.translate( BBox.getCenter(oobb) );
			LS.Draw.renderWireBox( halfsize[0]*2, halfsize[1]*2, halfsize[2]*2 );
		}
		else if(instance.type == LS.PhysicsInstance.PLANE)
		{
			LS.Draw.translate( BBox.getCenter(oobb) );
			LS.Draw.renderWireBox( halfsize[0]*2, 0.0001, halfsize[2]*2 );
		}
		else if(instance.type == LS.PhysicsInstance.SPHERE)
		{
			//Draw.scale(,halfsize[0],halfsize[0]);
			LS.Draw.translate( BBox.getCenter(oobb) );
			LS.Draw.renderWireSphere( halfsize[0], 20 );
		}
		else if(instance.type == LS.PhysicsInstance.MESH)
		{
			var mesh = instance.mesh;
			if(mesh)
			{
				if(!mesh.indexBuffers["wireframe"])
					mesh.computeWireframe();
				LS.Draw.renderMesh(mesh, gl.LINES, null, "wireframe" );
			}
		}

		LS.Draw.pop();
	}
}

DebugRender.prototype.renderPaths = function( scene )
{
	scene = scene || LS.GlobalScene;

	if(!scene._paths)
		return;

	LS.Draw.setColor([0.7,0.6,0.3,0.5]);

	for(var i = 0; i < scene._paths.length; ++i)
	{
		var path = scene._paths[i];
		var points = path.samplePoints(0);
		LS.Draw.renderLines( points, null, true );
	}
}

DebugRender.prototype.createMeshes = function()
{
	//plane
	this.plane_mesh = GL.Mesh.plane({xz:true, detail: 10});

	//grid
	var dist = 10;
	var num = 10;
	var vertices = [];
	for(var i = -num; i <= num; i++)
	{
		vertices.push([i*dist,0,dist*num]);
		vertices.push([i*dist,0,-dist*num]);
		vertices.push([dist*num,0,i*dist]);
		vertices.push([-dist*num,0,i*dist]);
	}
	this.grid_mesh = GL.Mesh.load({vertices:vertices});

	//box
	vertices = new Float32Array([-1,1,1 , -1,1,-1, 1,1,-1, 1,1,1, -1,-1,1, -1,-1,-1, 1,-1,-1, 1,-1,1]);
	var triangles = new Uint16Array([0,1, 0,4, 0,3, 1,2, 1,5, 2,3, 2,6, 3,7, 4,5, 4,7, 6,7, 5,6 ]);
	this.box_mesh = GL.Mesh.load({vertices: vertices, lines:triangles });

	//circle
	this.circle_mesh = GL.Mesh.circle({size:1,slices:50});
	this.circle_empty_mesh = GL.Mesh.circle({size:1,slices:50,empty:1});
	this.sphere_mesh = GL.Mesh.icosahedron({size:1, subdivisions: 3});

	//dummy
	vertices = [];
	vertices.push([-dist*0.5,0,0],[+dist*0.5,0,0]);
	vertices.push([0,-dist*0.5,0],[0,+dist*0.5,0]);
	vertices.push([0,0,-dist*0.5],[0,0,+dist*0.5]);
	this.dummy_mesh = GL.Mesh.load({vertices:vertices});

	//box
	vertices = [];
	vertices.push([-1.0,1.0,1.0],[1.0,1.0,1.0],[-1.0,1.0,-1.0], [1.0,1.0,-1.0],[-1.0,-1.0,1.0], [1.0,-1.0,1.0],[-1.0,-1.0,-1.0], [1.0,-1.0,-1.0]);
	vertices.push([1.0,-1.0,1.0],[1.0,1.0,1.0],[1.0,-1.0,-1.0],[1.0,1.0,-1.0],[-1.0,-1.0,1.0],[-1.0,1.0,1.0],[-1.0,-1.0,-1.0],[-1.0,1.0,-1.0]);
	vertices.push([1.0,1.0,1.0],[1.0,1.0,-1.0],[1.0,-1.0,1.0],[1.0,-1.0,-1.0],[-1.0,1.0,1.0],[-1.0,1.0,-1.0],[-1.0,-1.0,1.0],[-1.0,-1.0,-1.0]);
	this.cube_mesh = GL.Mesh.load({vertices:vertices});

	for(var i = 1; i >= 0.0; i -= 0.02)
	{
		var f = ( 1 - 0.001/(i) )*2-1;
		vertices.push([-1.0,1.0,f],[1.0,1.0,f],[-1.0,-1.0,f], [1.0,-1.0,f]);
		vertices.push([1.0,-1.0,f],[1.0,1.0,f],[-1.0,-1.0,f],[-1.0,1.0,f]);
	}

	this.frustum_mesh = GL.Mesh.load({vertices:vertices});

	//cylinder
	this.cylinder_mesh = GL.Mesh.cylinder({radius:10,height:2});

	//axis
	vertices = [];
	var colors = [];
	dist = 2;
	vertices.push([0,0,0],[+dist*0.5,0,0]);
	colors.push([1,0,0,1],[1,0,0,1]);
	vertices.push([0,0,0],[0,+dist*0.5,0]);
	colors.push([0,1,0,1],[0,1,0,1]);
	vertices.push([0,0,0],[0,0,+dist*0.5]);
	colors.push([0,0,1,1],[0,0,1,1]);
	this.axis_mesh = GL.Mesh.load({vertices:vertices, colors: colors});

	//top
	vertices = [];
	vertices.push([0,0,0],[0,+dist*0.5,0]);
	vertices.push([0,+dist*0.5,0],[0.1*dist,+dist*0.4,0]);
	vertices.push([0,+dist*0.5,0],[-0.1*dist,+dist*0.4,0]);
	this.top_line_mesh = GL.Mesh.load({vertices:vertices});

	//front
	vertices = [];
	vertices.push([0,0,0],[0,0,+dist*0.5]);
	vertices.push([0,0,+dist*0.5],[0,0.1*dist,+dist*0.4]);
	vertices.push([0,0,+dist*0.5],[0,-0.1*dist,+dist*0.4]);
	this.front_line_mesh = GL.Mesh.load({vertices:vertices});
}

LS.DebugRender = DebugRender;
///@FILE:../src/render/draw.js
//this module is in charge of rendering basic objects like lines, points, and primitives
//it works over litegl (no need of scene)
//carefull, it is very slow

/**
* LS.Draw allows to render basic primitives, similar to the OpenGL Fixed pipeline.
* It reuses local meshes when possible to avoid fragmenting the VRAM.
* @class Draw
* @constructor
*/

var Draw = {
	ready: false,
	images: {},
	image_last_id: 1,

	onRequestFrame: null,
	reset_stack_on_reset: true,
	_rendercalls: 0,

	/**
	* Sets up everything (prepare meshes, shaders, and so)
	* @method init
	*/
	init: function()
	{
		if(this.ready)
			return;
		if(!gl)
			return;

		this.color = new Float32Array(4);
		this.color[3] = 1;
		this.mvp_matrix = mat4.create();
		this.temp_matrix = mat4.create();
		this.point_size = 2;
		this.line_width = 1;

		this.stack = new Float32Array(16 * 32); //stack max size
		this.model_matrix = new Float32Array(this.stack.buffer,0,16);
		mat4.identity( this.model_matrix );

		//matrices
		this.camera = null;
		this.camera_position = vec3.create();
		this.view_matrix = mat4.create();
		this.projection_matrix = mat4.create();
		this.viewprojection_matrix = mat4.create();

		this.camera_stack = []; //not used yet

		this.uniforms = {
				u_model: this.model_matrix,
				u_viewprojection: this.viewprojection_matrix,
				u_mvp: this.mvp_matrix,
				u_color: this.color,
				u_camera_position: this.camera_position,
				u_point_size: this.point_size,
				u_point_perspective: 0,
				u_perspective: 1, //viewport.w * this._projection_matrix[5]
				u_texture: 0
		};

		//temp containers
		this._temp = vec3.create();

		//Meshes
		var vertices = [[-1,1,0],[1,1,0],[1,-1,0],[-1,-1,0]];
		var coords = [[0,1],[1,1],[1,0],[0,0]];
		this.quad_mesh = GL.Mesh.load({vertices:vertices, coords: coords});

		var vertex_shader = Draw.vertex_shader_code;
		var pixel_shader = Draw.fragment_shader_code;

		//create shaders
		this.shader = new Shader( vertex_shader, pixel_shader );
		this.shader_instancing = new Shader(vertex_shader,pixel_shader,{"USE_INSTANCING":""});
		this.shader_color = new Shader(vertex_shader,pixel_shader,{"USE_COLOR":""});
		this.shader_color_instancing = new Shader(vertex_shader,pixel_shader,{"USE_COLOR":"","USE_INSTANCING":""});
		this.shader_texture = new Shader(vertex_shader,pixel_shader,{"USE_TEXTURE":""});
		this.shader_texture_instancing = new Shader(vertex_shader,pixel_shader,{"USE_TEXTURE":"","USE_INSTANCING":""});
		this.shader_points = new Shader(vertex_shader,pixel_shader,{"USE_POINTS":""});
		this.shader_points_color = new Shader(vertex_shader,pixel_shader,{"USE_COLOR":"","USE_POINTS":""});
		this.shader_points_color_size = new Shader(vertex_shader,pixel_shader,{"USE_COLOR":"","USE_SIZE":"","USE_POINTS":""});


		this.shader_image = new Shader('\
			precision mediump float;\n\
			attribute vec3 a_vertex;\n\
			uniform mat4 u_mvp;\n\
			uniform float u_point_size;\n\
			void main() {\n\
				gl_PointSize = u_point_size;\n\
				gl_Position = u_mvp * vec4(a_vertex,1.0);\n\
			}\
			','\
			precision mediump float;\n\
			uniform vec4 u_color;\n\
			uniform sampler2D u_texture;\n\
			void main() {\n\
			  vec4 tex = texture2D(u_texture, vec2(gl_PointCoord.x,1.0 - gl_PointCoord.y) );\n\
			  if(tex.a < 0.01)\n\
				discard;\n\
			  gl_FragColor = u_color * tex;\n\
			}\
		');

		this.shader_points_color_texture_size = new Shader('\
			precision mediump float;\n\
			attribute vec3 a_vertex;\n\
			attribute vec4 a_color;\n\
			attribute float a_extra;\n\
			uniform mat4 u_mvp;\n\
			uniform float u_point_size;\n\
			varying vec4 v_color;\n\
			void main() {\n\
				v_color = a_color;\n\
				gl_PointSize = u_point_size * a_extra;\n\
				gl_Position = u_mvp * vec4(a_vertex,1.0);\n\
			}\
			','\
			precision mediump float;\n\
			uniform vec4 u_color;\n\
			varying vec4 v_color;\n\
			uniform sampler2D u_texture;\n\
			void main() {\n\
			  vec4 tex = texture2D(u_texture, vec2(gl_PointCoord.x,1.0 - gl_PointCoord.y) );\n\
			  if(tex.a < 0.1)\n\
				discard;\n\
			  vec4 color = u_color * v_color * tex;\n\
			  gl_FragColor = color;\n\
			}\
		');

		this.shader_text2D = new Shader('\
			precision mediump float;\n\
			attribute vec3 a_vertex;\n\
			attribute vec4 a_extra4;\n\
			uniform mat4 u_mvp;\n\
			uniform float u_point_size;\n\
			void main() {\n\
				gl_PointSize = u_point_size;\n\
				gl_Position = u_mvp * vec4(a_vertex,1.0);\n\
			}\
			','\
			precision mediump float;\n\
			uniform vec4 u_color;\n\
			uniform sampler2D u_texture;\n\
			void main() {\n\
			  vec4 tex = texture2D(u_texture, vec2(gl_PointCoord.x,1.0 - gl_PointCoord.y) );\n\
			  if(tex.a < 0.1)\n\
				discard;\n\
			  vec4 color = u_color * tex;\n\
			  gl_FragColor = color;\n\
			}\
		');

		//create shaders
		var phong_vertex_code = "\
			precision mediump float;\n\
			attribute vec3 a_vertex;\n\
			attribute vec3 a_normal;\n\
			varying vec3 v_pos;\n\
			varying vec3 v_normal;\n\
			#ifdef USE_INSTANCING\n\
				attribute mat4 u_model;\n\
			#else\n\
				uniform mat4 u_model;\n\
			#endif\n\
			uniform mat4 u_viewprojection;\n\
			void main() {\n\
				v_pos = ( u_model * vec4( a_vertex, 1.0 )).xyz;\n\
				v_normal = (u_model * vec4(a_normal,0.0)).xyz;\n\
				gl_Position = u_viewprojection * vec4( v_pos, 1.0 );\n\
			}\n";
		
		var phong_pixel_shader = "\n\
			precision mediump float;\n\
			uniform vec3 u_ambient_color;\n\
			uniform vec3 u_light_color;\n\
			uniform vec3 u_light_dir;\n\
			uniform vec4 u_color;\n\
			varying vec3 v_pos;\n\
			varying vec3 v_normal;\n\
			void main() {\n\
				vec3 N = normalize(v_normal);\n\
				float NdotL = max(0.0, dot(N,u_light_dir));\n\
				gl_FragColor = u_color * vec4(u_ambient_color + u_light_color * NdotL, 1.0);\n\
			}\n";

		this.shader_phong = new Shader( phong_vertex_code, phong_pixel_shader);
		this.shader_phong_instanced = new Shader( phong_vertex_code, phong_pixel_shader, { "USE_INSTANCING":"" } );
		var phong_uniforms = {u_ambient_color:[0.1,0.1,0.1], u_light_color:[0.8,0.8,0.8], u_light_dir: [0,1,0] };
		this.shader_phong.uniforms( phong_uniforms );
		this.shader_phong_instanced.uniforms( phong_uniforms );

		//create shaders
		this.shader_depth = new Shader('\
			precision mediump float;\n\
			attribute vec3 a_vertex;\n\
			varying vec4 v_pos;\n\
			uniform mat4 u_model;\n\
			uniform mat4 u_mvp;\n\
			void main() {\n\
				v_pos = u_model * vec4(a_vertex,1.0);\n\
				gl_Position = u_mvp * vec4(a_vertex,1.0);\n\
			}\
			','\
			precision mediump float;\n\
			varying vec4 v_pos;\n\
			\n\
			vec4 PackDepth32(float depth)\n\
			{\n\
				const vec4 bitSh  = vec4(   256*256*256, 256*256,   256,         1);\n\
				const vec4 bitMsk = vec4(   0,      1.0/256.0,    1.0/256.0,    1.0/256.0);\n\
				vec4 comp;\n\
				comp	= depth * bitSh;\n\
				comp	= fract(comp);\n\
				comp	-= comp.xxyz * bitMsk;\n\
				return comp;\n\
			}\n\
			void main() {\n\
				float depth = (v_pos.z / v_pos.w) * 0.5 + 0.5;\n\
				gl_FragColor = PackDepth32(depth);\n\
			}\
		');

		this.ready = true;
	},

	/**
	* A helper to create shaders when you only want to specify some basic shading
	* @method createSurfaceShader
	* @param {string} surface_function GLSL code like: "vec4 surface_function( vec3 pos, vec3 normal, vec2 coord ) { return vec4(1.0); } ";
	* @param {object} macros [optional] object containing the macros and value
	* @param {object} uniforms [optional] object with name and type
	* @return {GL.Shader} the resulting shader
	*/
	createSurfaceShader: function( surface_function, uniforms, macros )
	{
		//"vec4 surface_function( vec3 pos, vec3 normal, vec2 coord ) { return vec4(1.0); } ";

		if( surface_function.indexOf("surface_function") == -1 )
			surface_function = "vec4 surface_function( vec3 pos, vec3 normal, vec2 coord ) { " + surface_function + "\n } ";

		if(uniforms)
		{
			if (uniforms.constructor === String)
				surface_function = uniforms + ";\n" + surface_function;
			else
				for(var i in uniforms)
					surface_function += "uniform " + uniforms[i] + " " + i + ";\n";
		}

		var vertex_shader = "\
			precision mediump float;\n\
			attribute vec3 a_vertex;\n\
			attribute vec3 a_normal;\n\
			attribute vec2 a_coord;\n\
			varying vec2 v_coord;\n\
			varying vec3 v_pos;\n\
			varying vec3 v_normal;\n\
			uniform mat4 u_mvp;\n\
			uniform mat4 u_model;\n\
			void main() {\n\
				v_coord = a_coord;\n\
				v_pos = (u_model * vec4(a_vertex,1.0)).xyz;\n\
				v_normal = (u_model * vec4(a_normal,0.0)).xyz;\n\
				gl_Position = u_mvp * vec4(a_vertex,1.0);\n\
			}\
			";

		var pixel_shader = "\
			precision mediump float;\n\
			varying vec2 v_coord;\n\
			varying vec3 v_pos;\n\
			varying vec3 v_normal;\n\
			uniform vec4 u_color;\n\
			uniform vec3 u_camera_position;\n\
			uniform sampler2D u_texture;\n\
			"+ surface_function +"\n\
			void main() {\n\
				gl_FragColor = surface_function(v_pos,v_normal,v_coord);\n\
			}\
		";	

		return new GL.Shader( vertex_shader, pixel_shader, macros );
	},

	/**
	* clears the stack
	* @method reset
	*/
	reset: function( reset_memory )
	{
		if(!this.ready)
			this.init();
		else
		{
			this.color.set([1,1,1,1]);
			this.point_size = 2;
			this.line_width = 1;
		}

		if( reset_memory )
			this.images = {}; //clear images

		if(this.reset_stack_on_reset)
		{
			this.model_matrix = new Float32Array(this.stack.buffer,0,16);
			this.uniforms.u_model = this.model_matrix;
			mat4.identity( this.model_matrix );
		}
	},

	/**
	* Sets the color used to paint primitives
	* @method setColor
	* @param {vec3|vec4} color
	*/
	setColor: function(color)
	{
		if( arguments.length >= 3 )
		{
			this.color[0] = arguments[0];
			this.color[1] = arguments[1];
			this.color[2] = arguments[2];
			if( arguments.length == 4 )
				this.color[3] = arguments[3];
		}
		else
		for(var i = 0; i < color.length; i++)
			this.color[i] = color[i];
	},

	/**
	* Sets the alpha used to paint primitives
	* @method setAlpha
	* @param {number} alpha
	*/
	setAlpha: function(alpha)
	{
		this.color[3] = alpha;
	},

	/**
	* Sets the point size
	* @method setPointSize
	* @param {number} v size of points
	* @param {number} perspective [optional] if set to true, the points will be affected by perspective
	*/
	setPointSize: function(v, perspective)
	{
		this.point_size = v;
		this.uniforms.u_point_size = v;
		this.uniforms.u_point_perspective = perspective ? 1 : 0;
	},

	/**
	* Sets the line width
	* @method setLineWidth
	* @param {number} v width in pixels
	*/
	setLineWidth: function(v)
	{
		if(gl.setLineWidth)
			gl.setLineWidth(v);
		else
			gl.lineWidth(v);
		this.line_width = v;
	},

	/**
	* Sets the camera to use during the rendering, this is already done by LS.Renderer
	* @method setCamera
	* @param {LS.Camera} camera
	*/
	setCamera: function( camera )
	{
		this.camera = camera;
		camera.updateMatrices();
		vec3.copy( this.camera_position, camera.getEye() );	
		this.view_matrix.set( camera._view_matrix );
		this.projection_matrix.set( camera._projection_matrix );
		this.viewprojection_matrix.set( camera._viewprojection_matrix );
		this.uniforms.u_perspective = gl.viewport_data[3] * this.projection_matrix[5];
	},

	/**
	* Specifies the camera position (used to compute point size)
	* @method setCameraPosition
	* @param {vec3} center
	*/
	setCameraPosition: function(center)
	{
		vec3.copy( this.camera_position, center);
	},

	pushCamera: function()
	{
		this.camera_stack.push( mat4.create( this.viewprojection_matrix ) );
	},

	popCamera: function()
	{
		if(this.camera_stack.length == 0)
			throw("too many pops");
		this.viewprojection_matrix.set( this.camera_stack.pop() );
	},

	/**
	* Specifies the camera view and projection matrices
	* @method setViewProjectionMatrix
	* @param {mat4} view
	* @param {mat4} projection
	* @param {mat4} vp viewprojection matrix [optional]
	*/
	setViewProjectionMatrix: function(view, projection, vp)
	{
		mat4.copy( this.view_matrix, view);
		mat4.copy( this.projection_matrix, projection);
		if(vp)
			mat4.copy( this.viewprojection_matrix, vp);
		else
			mat4.multiply( this.viewprojection_matrix, view, vp);
	},

	/**
	* Specifies the transformation matrix to apply to the mesh
	* @method setMatrix
	* @param {mat4} matrix
	*/
	setMatrix: function(matrix)
	{
		mat4.copy(this.model_matrix, matrix);
	},

	/**
	* Multiplies the current matrix by a given one
	* @method multMatrix
	* @param {mat4} matrix
	*/
	multMatrix: function(matrix)
	{
		mat4.multiply(this.model_matrix, matrix, this.model_matrix);
	},

	/**
	* Render lines given a set of points
	* @method renderLines
	* @param {Float32Array|Array} points
	* @param {Float32Array|Array} colors [optional]
	* @param {bool} strip [optional] if the lines are a line strip (one consecutive line)
	* @param {bool} loop [optional] if strip, close loop
	*/
	renderLines: function(lines, colors, strip, loop)
	{
		if(!lines || !lines.length) return;
		var vertices = null;

		vertices = lines.constructor == Float32Array ? lines : this.linearize(lines);
		if(colors)
			colors = colors.constructor == Float32Array ? colors : this.linearize(colors);
		if(colors && (colors.length/4) != (vertices.length/3))
			colors = null;

		var type = gl.LINES;
		if(loop)
			type = gl.LINE_LOOP;
		else if(strip)
			type = gl.LINE_STRIP;

		var mesh = this.toGlobalMesh({vertices: vertices, colors: colors});
		return this.renderMesh( mesh, type, colors ? this.shader_color : this.shader, undefined, 0, vertices.length / 3 );
	},

	/**
	* Render points given a set of positions (and colors)
	* @method renderPoints
	* @param {Float32Array|Array} points
	* @param {Float32Array|Array} colors [optional]
	* @param {GL.Shader} shader [optional]
	*/
	renderPoints: function(points, colors, shader)
	{
		if(!points || !points.length)
			return;

		var vertices = null;

		if(points.constructor == Float32Array)
			vertices = points;
		else if(points[0].length) //array of arrays
			vertices = this.linearize(points);
		else
			vertices = new Float32Array(points);

		if(colors && colors.constructor != Float32Array)
		{
			if(colors.constructor === Array && colors[0].constructor === Number)
				colors = new Float32Array( colors );
			else
				colors = this.linearize(colors);
		}

		var mesh = this.toGlobalMesh({vertices: vertices, colors: colors});
		if(!shader)
			shader = colors ? this.shader_color : this.shader;

		return this.renderMesh(mesh, gl.POINTS, shader, undefined, 0, vertices.length / 3 );
	},

	/**
	* Render round points given a set of positions (and colors)
	* @method renderRoundPoints
	* @param {Float32Array|Array} points
	* @param {Float32Array|Array} colors [optional]
	* @param {GL.Shader} shader [optional]
	*/
	renderRoundPoints: function(points, colors, shader)
	{
		if(!points || !points.length)
			return;

		var vertices = null;

		if(points.constructor == Float32Array)
			vertices = points;
		else if(points[0].length) //array of arrays
			vertices = this.linearize(points);
		else
			vertices = new Float32Array(points);

		if(colors)
			colors = colors.constructor == Float32Array ? colors : this.linearize(colors);

		var mesh = this.toGlobalMesh({vertices: vertices, colors: colors});
		if(!shader)
			shader = colors ? this.shader_points_color : this.shader_points;
		return this.renderMesh( mesh, gl.POINTS, shader, undefined, 0, vertices.length / 3 );
	},

	/**
	* Render points with color, size, and texture binded in 0
	* @method renderPointsWithSize
	* @param {Float32Array|Array} points
	* @param {Float32Array|Array} colors [optional]
	* @param {Float32Array|Array} sizes [optional]
	* @param {GL.Texture} texture [optional]
	* @param {GL.Shader} shader [optional]
	*/
	renderPointsWithSize: function(points, colors, sizes, texture, shader)
	{
		if(!points || !points.length) return;
		var vertices = null;

		if(points.constructor == Float32Array)
			vertices = points;
		else if(points[0].length) //array of arrays
			vertices = this.linearize(points);
		else
			vertices = new Float32Array(points);

		if(!colors)
			throw("colors required in Draw.renderPointsWithSize");
		colors = colors.constructor == Float32Array ? colors : this.linearize(colors);
		if(!sizes)
			throw("sizes required in Draw.renderPointsWithSize");
		sizes = sizes.constructor == Float32Array ? sizes : this.linearize(sizes);

		var mesh = this.toGlobalMesh({vertices: vertices, colors: colors, extra: sizes});
		shader = shader || (texture ? this.shader_points_color_texture_size : this.shader_points_color_size);
		
		return this.renderMesh(mesh, gl.POINTS, shader, undefined, 0, vertices.length / 3 );
	},

	createRectangleMesh: function( width, height, in_z, use_global )
	{
		var vertices = new Float32Array(4 * 3);
		if(in_z)
			vertices.set([-width*0.5,0,height*0.5, width*0.5,0,height*0.5, width*0.5,0,-height*0.5, -width*0.5,0,-height*0.5]);
		else
			vertices.set([-width*0.5,height*0.5,0, width*0.5,height*0.5,0, width*0.5,-height*0.5,0, -width*0.5,-height*0.5,0]);

		if(use_global)
			return this.toGlobalMesh( {vertices: vertices} );

		return GL.Mesh.load({vertices: vertices});
	},

	/**
	* Render a wireframe rectangle of width x height 
	* @method renderRectangle
	* @param {number} width
	* @param {number} height
	* @param {boolean} in_z [optional] if the plane is aligned with the z plane
	*/
	renderRectangle: function(width, height, in_z, fill)
	{
		var mesh = this.createRectangleMesh( width, height, in_z, true );
		return this.renderMesh( mesh, fill ? gl.TRIANGLE_FAN : gl.LINE_LOOP, undefined, undefined, 0, this._global_mesh_last_size );
	},

	createCircleMesh: function(radius, segments, in_z, use_global)
	{
		segments = segments || 32;
		var axis = [0,1,0];
		var num_segments = segments || 100;
		var R = quat.create();
		var temp = this._temp;
		var vertices = new Float32Array(num_segments * 3);

		var offset =  2 * Math.PI / num_segments;

		for(var i = 0; i < num_segments; i++)
		{
			temp[0] = Math.sin(offset * i) * radius;
			if(in_z)
			{
				temp[1] = 0;
				temp[2] = Math.cos(offset * i) * radius;
			}
			else
			{
				temp[2] = 0;
				temp[1] = Math.cos(offset * i) * radius;
			}

			vertices.set(temp, i*3);
		}

		if(use_global)
			return this.toGlobalMesh({vertices: vertices});

		return GL.Mesh.load({vertices: vertices});
	},

	/**
	* Renders a circle 
	* @method renderCircle
	* @param {number} radius
	* @param {number} segments
	* @param {boolean} in_z [optional] if the circle is aligned with the z plane
	* @param {boolean} filled [optional] renders the interior
	*/
	renderCircle: function(radius, segments, in_z, filled)
	{
		var mesh = this.createCircleMesh(radius, segments, in_z, true);
		return this.renderMesh(mesh, filled ? gl.TRIANGLE_FAN : gl.LINE_LOOP, undefined, undefined, 0, this._global_mesh_last_size );
	},

	/**
	* Render a filled circle
	* @method renderSolidCircle
	* @param {number} radius
	* @param {number} segments
	* @param {boolean} in_z [optional] if the circle is aligned with the z plane
	*/
	renderSolidCircle: function(radius, segments, in_z)
	{
		return this.renderCircle(radius, segments, in_z, true);
	},

	createWireSphereMesh: function(radius, segments, use_global )
	{
		var axis = [0,1,0];
		segments = segments || 100;
		var R = quat.create();
		var temp = this._temp;
		var vertices = new Float32Array( segments * 2 * 3 * 3); 

		var delta = 1.0 / segments * Math.PI * 2;

		for(var i = 0; i < segments; i++)
		{
			temp.set([ Math.sin( i * delta) * radius, Math.cos( i * delta) * radius, 0]);
			vertices.set(temp, i*18);
			temp.set([Math.sin( (i+1) * delta) * radius, Math.cos( (i+1) * delta) * radius, 0]);
			vertices.set(temp, i*18 + 3);

			temp.set([ Math.sin( i * delta) * radius, 0, Math.cos( i * delta) * radius ]);
			vertices.set(temp, i*18 + 6);
			temp.set([Math.sin( (i+1) * delta) * radius, 0, Math.cos( (i+1) * delta) * radius ]);
			vertices.set(temp, i*18 + 9);

			temp.set([ 0, Math.sin( i * delta) * radius, Math.cos( i * delta) * radius ]);
			vertices.set(temp, i*18 + 12);
			temp.set([ 0, Math.sin( (i+1) * delta) * radius, Math.cos( (i+1) * delta) * radius ]);
			vertices.set(temp, i*18 + 15);
		}

		if(use_global)
			return this.toGlobalMesh({vertices: vertices});
		
		return GL.Mesh.load({vertices: vertices});
	},

	/**
	* Renders three circles to form a simple spherical shape
	* @method renderWireSphere
	* @param {number} radius
	* @param {number} segments
	*/
	renderWireSphere: function(radius, segments)
	{
		var mesh = this.createWireSphereMesh( radius, segments, true );
		return this.renderMesh( mesh, gl.LINES, undefined, undefined, 0, this._global_mesh_last_size );
	},

	/**
	* Renders an sphere
	* @method renderSolidSphere
	* @param {number} radius
	*/
	renderSolidSphere: function(radius)
	{
		var mesh = this._sphere_mesh;
		if(!this._sphere_mesh)
			mesh = this._sphere_mesh = GL.Mesh.sphere({ size: 1 });
		this.push();
		this.scale( radius,radius,radius );
		this.renderMesh( mesh, gl.TRIANGLES );
		this.pop();
	},


	createWireBoxMesh: function( sizex, sizey, sizez, use_global )
	{
		sizex = sizex*0.5;
		sizey = sizey*0.5;
		sizez = sizez*0.5;
		var vertices = new Float32Array([-sizex,sizey,sizez , -sizex,sizey,-sizez, sizex,sizey,-sizez, sizex,sizey,sizez,
						-sizex,-sizey,sizez, -sizex,-sizey,-sizez, sizex,-sizey,-sizez, sizex,-sizey,sizez]);
		var triangles = new Uint16Array([0,1, 0,4, 0,3, 1,2, 1,5, 2,3, 2,6, 3,7, 4,5, 4,7, 6,7, 5,6   ]);

		if(use_global)
			return this.toGlobalMesh( {vertices: vertices}, triangles );

		return GL.Mesh.load({vertices: vertices, lines:triangles });
	},

	/**
	* Renders a wire box (box made of lines, not filled)
	* @method renderWireBox
	* @param {number} sizex
	* @param {number} sizey
	* @param {number} sizez
	*/
	renderWireBox: function(sizex,sizey,sizez)
	{
		var mesh = this.createWireBoxMesh(sizex,sizey,sizez, true);
		return this.renderMesh( mesh, gl.LINES, undefined, "indices", 0, this._global_mesh_last_size );
	},

	createSolidBoxMesh: function( sizex,sizey,sizez, use_global)
	{
		sizex = sizex*0.5;
		sizey = sizey*0.5;
		sizez = sizez*0.5;
		//var vertices = [[-sizex,sizey,-sizez],[-sizex,-sizey,+sizez],[-sizex,sizey,sizez],[-sizex,sizey,-sizez],[-sizex,-sizey,-sizez],[-sizex,-sizey,+sizez],[sizex,sizey,-sizez],[sizex,sizey,sizez],[sizex,-sizey,+sizez],[sizex,sizey,-sizez],[sizex,-sizey,+sizez],[sizex,-sizey,-sizez],[-sizex,sizey,sizez],[sizex,-sizey,sizez],[sizex,sizey,sizez],[-sizex,sizey,sizez],[-sizex,-sizey,sizez],[sizex,-sizey,sizez],[-sizex,sizey,-sizez],[sizex,sizey,-sizez],[sizex,-sizey,-sizez],[-sizex,sizey,-sizez],[sizex,-sizey,-sizez],[-sizex,-sizey,-sizez],[-sizex,sizey,-sizez],[sizex,sizey,sizez],[sizex,sizey,-sizez],[-sizex,sizey,-sizez],[-sizex,sizey,sizez],[sizex,sizey,sizez],[-sizex,-sizey,-sizez],[sizex,-sizey,-sizez],[sizex,-sizey,sizez],[-sizex,-sizey,-sizez],[sizex,-sizey,sizez],[-sizex,-sizey,sizez]];
		var vertices = [-sizex,sizey,-sizez,-sizex,-sizey,+sizez,-sizex,sizey,sizez,-sizex,sizey,-sizez,-sizex,-sizey,-sizez,-sizex,-sizey,+sizez,sizex,sizey,-sizez,sizex,sizey,sizez,sizex,-sizey,+sizez,sizex,sizey,-sizez,sizex,-sizey,+sizez,sizex,-sizey,-sizez,-sizex,sizey,sizez,sizex,-sizey,sizez,sizex,sizey,sizez,-sizex,sizey,sizez,-sizex,-sizey,sizez,sizex,-sizey,sizez,-sizex,sizey,-sizez,sizex,sizey,-sizez,sizex,-sizey,-sizez,-sizex,sizey,-sizez,sizex,-sizey,-sizez,-sizex,-sizey,-sizez,-sizex,sizey,-sizez,sizex,sizey,sizez,sizex,sizey,-sizez,-sizex,sizey,-sizez,-sizex,sizey,sizez,sizex,sizey,sizez,-sizex,-sizey,-sizez,sizex,-sizey,-sizez,sizex,-sizey,sizez,-sizex,-sizey,-sizez,sizex,-sizey,sizez,-sizex,-sizey,sizez];
		if(use_global)
			return this.toGlobalMesh( {vertices: vertices} );

		return GL.Mesh.load({vertices: vertices });
	},

	/**
	* Renders a solid box 
	* @method renderSolidBox
	* @param {number} sizex
	* @param {number} sizey
	* @param {number} sizez
	*/
	renderSolidBox: function(sizex,sizey,sizez)
	{
		var mesh = this.createSolidBoxMesh(sizex,sizey,sizez, true);
		return this.renderMesh( mesh, gl.TRIANGLES, undefined, undefined, 0, this._global_mesh_last_size );
	},

	/**
	* Renders a wire cube of size size
	* @method renderWireCube
	* @param {number} size
	*/
	renderWireCube: function(size)
	{
		return this.renderWireBox(size,size,size);
	},

	/**
	* Renders a solid cube of size size
	* @method renderSolidCube
	* @param {number} size
	*/
	renderSolidCube: function(size)
	{
		return this.renderSolidBox(size,size,size);
	},

	/**
	* Renders a solid plane (could be textured or even with an specific shader)
	* @method renderPlane
	* @param {vec3} position
	* @param {vec2} size
	* @param {GL.Texture} texture
	* @param {GL.Shader} shader
	*/
	renderPlane: function( position, size, texture, shader)
	{
		if(!position || !size)
			throw("LS.Draw.renderPlane param missing");

		this.push();
		this.translate(position);
		this.scale( size[0], size[1], 1 );
		if(texture)
			texture.bind(0);

		if(!shader && texture)
			shader = this.shader_texture;

		this.renderMesh(this.quad_mesh, gl.TRIANGLE_FAN, shader );

		if(texture)
			texture.unbind(0);
		
		this.pop();
	},	

	createGridMesh: function(dist,num)
	{
		dist = dist || 20;
		num = num || 10;
		var vertices = new Float32Array( (num*2+1) * 4 * 3);
		var pos = 0;
		for(var i = -num; i <= num; i++)
		{
			vertices.set( [i*dist,0,dist*num], pos);
			vertices.set( [i*dist,0,-dist*num],pos+3);
			vertices.set( [dist*num,0,i*dist], pos+6);
			vertices.set( [-dist*num,0,i*dist],pos+9);
			pos += 3*4;
		}
		return GL.Mesh.load({vertices: vertices});
	},

	/**
	* Renders a grid of lines
	* @method renderGrid
	* @param {number} dist distance between lines
	* @param {number} num number of lines
	*/
	renderGrid: function(dist,num)
	{
		var mesh = this.createGridMesh(dist,num);
		return this.renderMesh(mesh, gl.LINES);
	},

	createConeMesh: function(radius, height, segments, in_z, use_global )
	{
		var axis = [0,1,0];
		segments = segments || 100;
		var R = quat.create();
		var temp = this._temp;
		var vertices = new Float32Array( (segments+2) * 3);
		vertices.set(in_z ? [0,0,height] : [0,height,0], 0);

		for(var i = 0; i <= segments; i++)
		{
			quat.setAxisAngle(R,axis, 2 * Math.PI * (i/segments) );
			vec3.transformQuat(temp, [0,0,radius], R );
			if(in_z)
				vec3.set(temp, temp[0],temp[2],temp[1] );
			vertices.set(temp, i*3+3);
		}

		if(use_global)
			return this.toGlobalMesh( {vertices: vertices} );

		return GL.Mesh.load({vertices: vertices});
	},

	/**
	* Renders a cone 
	* @method renderCone
	* @param {number} radius
	* @param {number} height
	* @param {number} segments
	* @param {boolean} in_z aligned with z axis
	*/
	renderCone: function(radius, height, segments, in_z)
	{
		var mesh = this.createConeMesh(radius, height, segments, in_z, true);
		return this.renderMesh(mesh, gl.TRIANGLE_FAN, undefined, undefined, 0, this._global_mesh_last_size );
	},

	createCylinderMesh: function( radius, height, segments, in_z, use_global )
	{
		var axis = [0,1,0];
		segments = segments || 100;
		var R = quat.create();
		var temp = this._temp;
		var vertices = new Float32Array( (segments+1) * 3 * 2);

		for(var i = 0; i <= segments; i++)
		{
			quat.setAxisAngle(R, axis, 2 * Math.PI * (i/segments) );
			vec3.transformQuat(temp, [0,0,radius], R );
			vertices.set(temp, i*3*2+3);
			temp[1] = height;
			vertices.set(temp, i*3*2);
		}

		if(use_global)
			return this.toGlobalMesh( {vertices: vertices} );

		return GL.Mesh.load({vertices: vertices});
	},

	/**
	* Renders a cylinder
	* @method renderCylinder
	* @param {number} radius
	* @param {number} height
	* @param {number} segments
	* @param {boolean} in_z aligned with z axis
	*/
	renderCylinder: function( radius, height, segments, in_z )
	{
		var mesh = this.createCylinderMesh(radius, height, segments, in_z, true);
		return this.renderMesh( mesh, gl.TRIANGLE_STRIP, undefined, undefined, 0, this._global_mesh_last_size );
	},

	/**
	* Renders an image in 2D (billboarded)
	* @method renderImage
	* @param {vec3} position that will be projected
	* @param {Image|Texture|String} image from an URL, or a texture
	* @param {number} size [optional=10]
	* @param {boolean} fixed_size [optional=false] (camera distance do not affect size)
	*/
	renderImage: function( position, image, size, fixed_size )
	{
		if(!position || !image)
			throw("LS.Draw.renderImage param missing");
		size = size || 10;
		var texture = null;

		if(typeof(image) == "string")
		{
			if(window.LS)
				texture = LS.ResourcesManager.textures[image];
			if(!texture)
				texture = this.images[image];
			if(texture == null)
			{
				Draw.images[image] = 1; //loading
				var img = new Image();
				img.src = image;
				img.onload = function()
				{
					var texture = GL.Texture.fromImage(this);
					Draw.images[image] = texture;
					if(Draw.onRequestFrame)
						Draw.onRequestFrame();
					return;
				}	
				return;
			}
			else if(texture == 1)
				return; //loading
		}
		else if(image.constructor == Image)
		{
			if(!image.texture)
				image.texture = GL.Texture.fromImage( this );
			texture = image.texture;
		}
		else if(image.constructor == Texture)
			texture = image;

		if(!texture)
			return;

		if(fixed_size)
		{
			this.setPointSize( size );
			texture.bind(0);
			this.renderPoints( position, null, this.shader_image );
		}
		else
		{
			this.push();
			//this.lookAt(position, this.camera_position,[0,1,0]);
			this.billboard(position);
			this.scale(size,size,size);
			texture.bind(0);
			this.renderMesh(this.quad_mesh, gl.TRIANGLE_FAN, this.shader_texture );
			this.pop();
		}
	},

	/**
	* Renders a given mesh applyting the stack transformations
	* @method renderMesh
	* @param {GL.Mesh} mesh
	* @param {enum} primitive [optional=gl.TRIANGLES] GL.TRIANGLES, gl.LINES, gl.POINTS, ...
	* @param {string} indices [optional="triangles"] the name of the buffer in the mesh with the indices
	* @param {number} range_start [optional] in case of rendering a range, the start primitive
	* @param {number} range_length [optional] in case of rendering a range, the number of primitives
	*/
	renderMesh: function( mesh, primitive, shader, indices, range_start, range_length )
	{
		if(!this.ready)
			throw ("Draw.js not initialized, call Draw.init()");
		if(!mesh)
			throw ("LS.Draw.renderMesh mesh cannot be null");

		if(!shader)
		{
			if(mesh === this._global_mesh && this._global_mesh_ignore_colors )
				shader = this.shader;
			else
				shader = mesh.vertexBuffers["colors"] ? this.shader_color : this.shader;
		}

		mat4.multiply(this.mvp_matrix, this.viewprojection_matrix, this.model_matrix );

		shader.uniforms( this.uniforms );
				
		if( range_start === undefined )
			shader.draw(mesh, primitive === undefined ? gl.TRIANGLES : primitive, indices );
		else
			shader.drawRange(mesh, primitive === undefined ? gl.TRIANGLES : primitive, range_start, range_length, indices );

		//used for repeating render 
		this._last_mesh = mesh;
		this._last_primitive = primitive;
		this._last_shader = shader;
		this._last_indices = indices;
		this._last_range_start = range_start;
		this._last_range_length = range_length;
		this._rendercalls += 1;

		this.last_mesh = mesh;
		return mesh;
	},

	/**
	* Renders several meshes in one draw call, keep in mind the shader and the browser should support instancing
	* @method renderMeshesInstanced
	* @param {GL.Mesh} mesh
	* @param {Array} matrices an array containing all the matrices
	* @param {enum} primitive [optional=gl.TRIANGLES] GL.TRIANGLES, gl.LINES, gl.POINTS, ...
	* @param {string} indices [optional="triangles"] the name of the buffer in the mesh with the indices
	*/
	renderMeshesInstanced: (function(){ 
		
		var tmp = { u_model: null };
		var tmp_matrix = mat4.create();

		return function( mesh, matrices, primitive, shader, indices )
		{
			if(!this.ready)
				throw ("Draw.js not initialized, call Draw.init()");
			if(!mesh)
				throw ("LS.Draw.renderMeshesInstanced mesh cannot be null");

			if( gl.webgl_version == 1 && !gl.extensions.ANGLE_instanced_arrays )
				return null; //instancing not supported

			if(!shader)
				shader = mesh.vertexBuffers["colors"] ? this.shader_color_instancing : this.shader_instancing;

			if( !shader.attributes.u_model )
				throw("Shader does not support instancing, it must have a attribute u_model");

			tmp.u_model = matrices;
			//this hack is done so we dont have to multiply the global model for every matrix, the VP is in reality a MVP
			tmp_matrix.set( this.viewprojection_matrix );
			mat4.multiply( this.viewprojection_matrix, this.viewprojection_matrix, this.model_matrix );

			shader.uniforms( this.uniforms );
			shader.drawInstanced( mesh, primitive === undefined ? gl.TRIANGLES : primitive, indices, tmp );

			this.viewprojection_matrix.set( tmp_matrix );
			this._rendercalls += 1;
			return mesh;
		};
	})(),

	//used in some special cases
	repeatLastRender: function()
	{
		this.renderMesh( this._last_mesh, this._last_primitive, this._last_shader, this._last_indices, this._last_range_start, this._last_range_length );
	},

	/**
	* Renders a text in 3D, in the XY plane, using the current matrix position
	* @method renderText
	* @param {string} text
	* @param {vec3} position [optional] 3D coordinate in relation to matrix
	* @param {number} scale [optional] scale modifier, default 1
	*/
	renderText: function( text, position, scale )
	{
		position = position || LS.ZEROS;
		scale = scale || 1;
		var l = text.length;
		if(l==0 || scale == 0)
			return;

		if(!Draw.font_atlas)
			this.createFontAtlas();
		var atlas = this.font_atlas;
		var char_size = atlas.atlas.char_size * scale;
		var i_char_size = 1 / atlas.atlas.char_size;
		var spacing = atlas.atlas.spacing * scale;

		var num_valid_chars = 0;
		for(var i = 0; i < l; ++i)
			if(atlas.atlas[ text.charCodeAt(i) ] != null)
				num_valid_chars++;

		var vertices = new Float32Array( num_valid_chars * 6 * 3);
		var coords = new Float32Array( num_valid_chars * 6 * 2);

		var pos = 0;
		var x = 0, y = 0;
		for(var i = 0; i < l; ++i)
		{
			var c = atlas.atlas[ text.charCodeAt(i) ];
			if(!c)
			{
				if(text.charCodeAt(i) == 10)
				{
					x = 0;
					y -= char_size;
				}
				else
					x += char_size;
				continue;
			}

			vertices.set( [x + position[0], y + position[1], position[2]], pos*6*3);
			vertices.set( [x + position[0], y + position[1] + char_size, position[2]], pos*6*3+3);
			vertices.set( [x + position[0] + char_size, y + position[1] + char_size, position[2]], pos*6*3+6);
			vertices.set( [x + position[0] + char_size, y + position[1], position[2]], pos*6*3+9);
			vertices.set( [x + position[0], y + position[1], position[2]], pos*6*3+12);
			vertices.set( [x + position[0] + char_size, y + position[1] + char_size, position[2]], pos*6*3+15);

			coords.set( [c[0], c[1]], pos*6*2);
			coords.set( [c[0], c[3]], pos*6*2+2);
			coords.set( [c[2], c[3]], pos*6*2+4);
			coords.set( [c[2], c[1]], pos*6*2+6);
			coords.set( [c[0], c[1]], pos*6*2+8);
			coords.set( [c[2], c[3]], pos*6*2+10);

			x+= spacing;
			++pos;
		}
		var mesh = this.toGlobalMesh({vertices: vertices, coords: coords});
		atlas.bind(0);
		return this.renderMesh( mesh, gl.TRIANGLES, this.shader_texture, undefined, 0, vertices.length / 3 );
	},

	/*
	renderText2D: function( text, position )
	{
		position = position || LS.ZEROS;
		if(!Draw.font_atlas)
			this.createFontAtlas();
		var atlas = this.font_atlas;
		var l = text.length;
		var char_size = atlas.atlas.char_size;
		var i_char_size = 1 / atlas.atlas.char_size;
		var spacing = atlas.atlas.spacing;

		var num_valid_chars = 0;
		for(var i = 0; i < l; ++i)
			if(atlas.atlas[ text.charCodeAt(i) ] != null)
				num_valid_chars++;

		var vertices = new Float32Array( num_valid_chars * 3 );
		var extra4 = new Float32Array( num_valid_chars * 4 );

		var pos = 0;
		var x = 0, y = 0;
		for(var i = 0; i < l; ++i)
		{
			var c = atlas.atlas[ text.charCodeAt(i) ];
			if(!c)
			{
				if(text.charCodeAt(i) == 10) //breakline
				{
					x = 0;
					y += char_size;
				}
				else
					x += char_size;
				continue;
			}

			vertices.set( position, i*3 );
			extra4.set([ c[0], c[1], x,y );

			x+= spacing;
			++pos;
		}
		var mesh = this.toGlobalMesh({ vertices: vertices, extra4: extra4 });
		this.setPointSize(20);
		atlas.bind(0);
		this.shader_text2D.uniforms({ u_char_offset: atlas.offset });
		return this.renderMesh( mesh, gl.POINTS, this.shader_text2D, undefined, 0, vertices.length / 3 );
	},
	*/

	createFontAtlas: function()
	{
		var canvas = createCanvas(512,512);
		var fontsize = (canvas.width * 0.09)|0;
		var char_size = (canvas.width * 0.1)|0;

		//$("body").append(canvas);
		var ctx = canvas.getContext("2d");
		//ctx.fillRect(0,0,canvas.width,canvas.height);
		ctx.fillStyle = "white";
		ctx.font = fontsize + "px Courier New";
		ctx.textAlign = "center";
		var x = 0;
		var y = 0;
		var xoffset = 0.5, yoffset = fontsize * -0.3;
		var atlas = { char_size: char_size, offset: char_size / canvas.width , spacing: char_size * 0.6 };

		for(var i = 6; i < 100; i++)//valid characters
		{
			var character = String.fromCharCode(i+27);
			atlas[i+27] = [x/canvas.width, 1-(y+char_size)/canvas.height, (x+char_size)/canvas.width, 1-(y)/canvas.height];
			ctx.fillText( character,Math.floor(x+char_size*xoffset),Math.floor(y+char_size+yoffset),char_size);
			x += char_size;
			if((x + char_size) > canvas.width)
			{
				x = 0;
				y += char_size;
			}
		}

		this.font_atlas = GL.Texture.fromImage(canvas, {magFilter: gl.LINEAR, minFilter: gl.LINEAR_MIPMAP_LINEAR} );
		gl.colorMask(true,true,true,false);
		this.font_atlas.fill([1,1,1,0]);
		gl.colorMask(true,true,true,true);
		this.font_atlas.atlas = atlas;
	},

	linearize: function(array) //fairly optimized
	{
		if(!array.length)
			return [];
		if(array[0].constructor === Number) //array of numbers
			return array.constructor === Float32Array ? array : new Float32Array(array);
		//linearize
		var n = array[0].length; //assuming all values have the same size!
		var result = new Float32Array(array.length * n);
		var l = array.length;
		for(var i = 0; i < l; ++i)
			result.set(array[i], i*n);
		return result;
	},

	/**
	* pushes the transform matrix into the stack to save the state
	* @method push
	*/
	push: function()
	{
		if(this.model_matrix.byteOffset >= (this.stack.byteLength - 16*4))
			throw("matrices stack overflow");

		var old = this.model_matrix;
		this.model_matrix = new Float32Array(this.stack.buffer,this.model_matrix.byteOffset + 16*4,16);
		this.uniforms.u_model = this.model_matrix;
		mat4.copy(this.model_matrix, old);
	},

	/**
	* takes the matrix from the top position of the stack to restore the last saved state
	* @method push
	*/
	pop: function()
	{
		if(this.model_matrix.byteOffset == 0)
			throw("too many pops");
		this.model_matrix = new Float32Array(this.stack.buffer,this.model_matrix.byteOffset - 16*4,16);
		this.uniforms.u_model = this.model_matrix;
	},

	/**
	* clears the transform matrix setting it to an identity
	* @method identity
	*/
	identity: function()
	{
		mat4.identity(this.model_matrix);
	},

	/**
	* changes the scale of the transform matrix. The parameters could be a vec3, a single number (then the scale is uniform in all axis) or three numbers
	* @method scale
	* @param {vec3|array|number} x could be an array of 3, one value (if no other values are specified then it is an uniform scaling)
	* @param {number} y
	* @param {number} z
	*/
	scale: function(x,y,z)
	{
		if(arguments.length == 3)
		{
			var temp = this._temp;
			temp[0] = x; temp[1] = y; temp[2] = z;
			mat4.scale(this.model_matrix,this.model_matrix,temp);
		}
		else if(x.length)//one argument: x is vec3
			mat4.scale(this.model_matrix,this.model_matrix,x);
		else //is number
		{
			var temp = this._temp;
			temp[0] = temp[1] = temp[2] = x;
			mat4.scale(this.model_matrix,this.model_matrix,temp);
		}
	},

	/**
	* applies a translation to the transform matrix
	* @method translate
	* @param {vec3|number} x could be an array of 3 or the x transform
	* @param {number} y
	* @param {number} z
	*/
	translate: function(x,y,z)
	{
		if(arguments.length == 3)
		{
			var temp = this._temp;
			temp[0] = x; temp[1] = y; temp[2] = z;
			mat4.translate(this.model_matrix,this.model_matrix,temp);
		}
		else  //one argument: x -> vec3
			mat4.translate(this.model_matrix,this.model_matrix,x);
	},

	/**
	* applies a translation to the transform matrix
	* @method rotate
	* @param {number} angle in degrees
	* @param {number|vec3} x could be the x component or the full axis
	* @param {number} y
	* @param {number} z
	*/
	rotate: function(angle, x,y,z)
	{
		if(arguments.length == 4)
		{
			var temp = this._temp;
			temp[0] = x; temp[1] = y; temp[2] = z;
			mat4.rotate(this.model_matrix, this.model_matrix, angle * DEG2RAD, [x,y,z]);
		}
		else //two arguments: x -> vec3
			mat4.rotate(this.model_matrix, this.model_matrix, angle * DEG2RAD, x);
	},

	/**
	* moves an object to a given position and forces it to look to another direction
	* Warning: it doesnt changes the camera in any way, only the transform matrix
	* @method lookAt
	* @param {vec3} position
	* @param {vec3} target
	* @param {vec3} up
	*/
	lookAt: function(position, target, up)
	{
		mat4.lookAt( this.model_matrix, position, target, up );
		mat4.invert( this.model_matrix, this.model_matrix );
	},

	billboard: function(position)
	{
		mat4.invert(this.model_matrix, this.view_matrix);
		mat4.setTranslation(this.model_matrix, position);
	},

	fromTranslationFrontTop: function(position, front, top)
	{
		mat4.fromTranslationFrontTop(this.model_matrix, position, front, top);
	},

	/**
	* projects a point from 3D space to 2D space (multiply by MVP)
	* @method project
	* @param {vec3} position
	* @param {vec3} dest [optional]
	* @return {vec3} the point in screen space (in normalized coordinates)
	*/
	project: function( position, dest )
	{
		dest = dest || vec3.create();
		return mat4.multiplyVec3(dest, this.mvp_matrix, position);
	},

	getPhongShader: function( ambient_color, light_color, light_dir, instanced )
	{
		var shader = instanced ? this.shader_phong_instanced : this.shader_phong;
		vec3.normalize( light_dir, light_dir );
		shader.uniforms({ u_ambient_color: ambient_color, u_light_color: light_color, u_light_dir: light_dir });
		return shader;
	},

	getDepthShader: function()
	{
		return this.shader_depth;
	},

	//reuses a global mesh to avoid fragmenting the VRAM 
	toGlobalMesh: function( buffers, indices )
	{
		if(!this._global_mesh)
		{
			//global mesh: to reuse memory and save fragmentation
			this._global_mesh_max_vertices = 1024;
			this._global_mesh = new GL.Mesh({
				vertices: new Float32Array(this._global_mesh_max_vertices * 3),
				normals: new Float32Array(this._global_mesh_max_vertices * 3),
				coords: new Float32Array(this._global_mesh_max_vertices * 2),
				colors: new Float32Array(this._global_mesh_max_vertices * 4),
				extra: new Float32Array(this._global_mesh_max_vertices * 1)
			},{
				indices: new Uint16Array(this._global_mesh_max_vertices * 3)
			}, { stream_type: gl.DYNAMIC_STREAM });
		}

		//take every stream and store it inside the mesh buffers
		for(var i in buffers)
		{
			var mesh_buffer = this._global_mesh.getBuffer( i );
			if(!mesh_buffer)
			{
				console.warn("Draw: global mesh lacks one buffer: " + i );
				continue;
			}

			var buffer_data = buffers[i];
			if(!buffer_data)
				continue;
			if(!buffer_data.buffer)
				buffer_data = new Float32Array( buffer_data ); //force typed arrays

			//some data would be lost here
			if(buffer_data.length > mesh_buffer.data.length)
			{
				console.warn("Draw: data is too big, resizing" );
				this.resizeGlobalMesh();
				mesh_buffer = this._global_mesh.getBuffer( i );
				buffer_data = buffer_data.subarray(0,mesh_buffer.data.length);
			}

			mesh_buffer.setData( buffer_data ); //set and upload
		}

		this._global_mesh_ignore_colors = !(buffers.colors);

		if(indices)
		{
			var mesh_buffer = this._global_mesh.getIndexBuffer("indices");			
			mesh_buffer.setData( indices );
			this._global_mesh_last_size = indices.length;
		}
		else
			this._global_mesh_last_size = buffers["vertices"].length / 3;
		return this._global_mesh;
	},

	resizeGlobalMesh: function()
	{
		if(!this._global_mesh)
			throw("No global mesh to resize");

		//global mesh: to reuse memory and save fragmentation
		this._global_mesh_max_vertices = this._global_mesh_max_vertices * 2;
		this._global_mesh.deleteBuffers();

		this._global_mesh = new GL.Mesh({
			vertices: new Float32Array(this._global_mesh_max_vertices * 3),
			normals: new Float32Array(this._global_mesh_max_vertices * 3),
			coords: new Float32Array(this._global_mesh_max_vertices * 2),
			colors: new Float32Array(this._global_mesh_max_vertices * 4),
			extra: new Float32Array(this._global_mesh_max_vertices * 1)
		},{
			indices: new Uint16Array(this._global_mesh_max_vertices * 3)
		}, { stream_type: gl.DYNAMIC_STREAM });
	}
};


Draw.vertex_shader_code = '\
	precision mediump float;\n\
	attribute vec3 a_vertex;\n\
	varying vec3 v_vertex;\n\
	attribute vec3 a_normal;\n\
	varying vec3 v_normal;\n\
	#ifdef USE_COLOR\n\
		attribute vec4 a_color;\n\
		varying vec4 v_color;\n\
	#endif\n\
	#ifdef USE_TEXTURE\n\
		attribute vec2 a_coord;\n\
		varying vec2 v_coord;\n\
	#endif\n\
	#ifdef USE_SIZE\n\
		attribute float a_extra;\n\
	#endif\n\
	#ifdef USE_INSTANCING\n\
		attribute mat4 u_model;\n\
	#else\n\
		uniform mat4 u_model;\n\
	#endif\n\
	uniform mat4 u_viewprojection;\n\
	uniform float u_point_size;\n\
	uniform float u_perspective;\n\
	uniform float u_point_perspective;\n\
	float computePointSize(float radius, float w)\n\
	{\n\
		if(radius < 0.0)\n\
			return -radius;\n\
		return u_perspective * radius / w;\n\
	}\n\
	void main() {\n\
		#ifdef USE_TEXTURE\n\
			v_coord = a_coord;\n\
		#endif\n\
		#ifdef USE_COLOR\n\
			v_color = a_color;\n\
		#endif\n\
		v_vertex = ( u_model * vec4( a_vertex, 1.0 )).xyz;\n\
		v_normal = ( u_model * vec4( a_normal, 0.0 )).xyz;\n\
		gl_Position = u_viewprojection * vec4(v_vertex,1.0);\n\
		gl_PointSize = u_point_size;\n\
		#ifdef USE_SIZE\n\
			gl_PointSize = a_extra;\n\
		#endif\n\
		if(u_point_perspective != 0.0)\n\
			gl_PointSize = computePointSize( gl_PointSize, gl_Position.w );\n\
	}\
';

Draw.fragment_shader_code = '\
	precision mediump float;\n\
	uniform vec4 u_color;\n\
	#ifdef USE_COLOR\n\
		varying vec4 v_color;\n\
	#endif\n\
	#ifdef USE_TEXTURE\n\
		varying vec2 v_coord;\n\
		uniform sampler2D u_texture;\n\
	#endif\n\
	void main() {\n\
		vec4 color = u_color;\n\
		#ifdef USE_TEXTURE\n\
		  color *= texture2D(u_texture, v_coord);\n\
		  if(color.a < 0.1)\n\
			discard;\n\
		#endif\n\
		#ifdef USE_POINTS\n\
			float dist = length( gl_PointCoord.xy - vec2(0.5) );\n\
			if( dist > 0.45 )\n\
				discard;\n\
		#endif\n\
		#ifdef USE_COLOR\n\
			color *= v_color;\n\
		#endif\n\
		gl_FragColor = color;\n\
	}\
';


if(typeof(LS) != "undefined")
	LS.Draw = Draw;
///@FILE:../src/render/deformer.js
///@INFO: UNCOMMON
//WORK IN PROGRESS
/*
//Is the class in charge of applying deformations to meshes (skinning and morph targets)
function Deformer()
{
	this.shader_block = null;
	this.macros = {}
	this.uniforms = {}
}

Deformer.prototype.applyToRenderInstance = function( RI )
{
}


Deformer.prototype.applyByCPU = function( vertex_data, normals_data )
{
	//overwrite
}

LS.Deformer = Deformer;
*/
///@FILE:../src/render/shadows.js
///@INFO: COMMON
// Shadows are complex because there are too many combinations: SPOT/DIRECT,OMNI or DEPTH_COMPONENT,RGBA or HARD,SOFT,VARIANCE
// This class encapsulates the shadowmap generation, and also how how it is read from the shader (using a ShaderBlock)

/**
* Shadowmap contains all the info necessary to generate the shadowmap
* @class Shadowmap
* @constructor
* @param {Object} object to configure from
*/
function Shadowmap( light )
{
	//maybe useful
	//this.enabled = true;

	/**
	* Shadowmap resolution, if let to 0 it will use the system default
	* @property resolution
	* @type {Number}
	* @default 0
	*/
	this.resolution = 0;

	/**
	* The offset applied to every depth before comparing it to avoid shadow acne
	* @property bias
	* @type {Number}
	* @default 0.0
	*/
	this.bias = 0;

	/**
	* Which format to use to store the shadowmaps
	* @property format
	* @type {Number}
	* @default GL.DEPTH_COMPONENT
	*/
	this.format = GL.DEPTH_COMPONENT;

	/**
	* Layers mask, this layers define which objects affect the shadow map (cast shadows)
	* @property layers
	* @type {Number}
	* @default true
	*/
	this.layers = 0xFF; //visible layers

	/**
	* If true objects inside the shadowmap will be rendered with the back faces only
	* @property reverse_faces
	* @type {Boolean}
	* @default false
	*/
	this.reverse_faces = true; //improves quality in some cases


	this.shadow_mode = 1; //0:hard, 1:bilinear, ...

	this.linear_filter = true;

	/**
	* The shadowmap texture, could be stored as color or depth depending on the settings
	* @property texture
	* @type {GL.Texture}
	* @default true
	*/
	this._texture = null;
	this._light = light;
	this._fbo = null;
	this._shadow_params = vec4.create(); //1.0 / this._texture.width, this.shadow_bias, this.near, closest_far
	this._shadow_extra_params = vec4.create(); //custom params in case the user wants to tweak the shadowmap with a cusstom shader
}

LS.Shadowmap = Shadowmap;

Shadowmap.use_shadowmap_depth_texture = true;

Shadowmap.prototype.getLocator = function()
{
	return this._light.getLocator() + "/" + "shadowmap";
}

Shadowmap.prototype.configure = function(v)
{
	for(var i in v)
		this[i] = v[i];
}

//enable block
Shadowmap.prototype.getReadShaderBlock = function()
{
	if( this._texture.format != GL.DEPTH_COMPONENT )
		return Shadowmap.shader_block.flag_mask | Shadowmap.depth_in_color_block.flag_mask;
	return Shadowmap.shader_block.flag_mask;
}

Shadowmap.prototype.getWriteShaderBlock = function()
{
	return 0;
}

Shadowmap.prototype.precomputeStaticShadowmap = function()
{

}

Shadowmap.prototype.generate = function( instances, render_settings, precompute_static )
{
	var light = this._light;

	var light_intensity = light.computeLightIntensity();
	if( light_intensity < 0.0001 )
		return;

	//create the texture
	var shadowmap_resolution = this.resolution;
	if(shadowmap_resolution == 0)
		shadowmap_resolution = render_settings.default_shadowmap_resolution;

	//shadowmap size
	var shadowmap_width = shadowmap_resolution;
	var shadowmap_height = shadowmap_resolution;
	if( light.type == LS.Light.OMNI)
		shadowmap_height *= 6; //for every face
	var magFilter = this.linear_filter ? gl.LINEAR : gl.NEAREST;

	//var tex_type = this.type == Light.OMNI ? gl.TEXTURE_CUBE_MAP : gl.TEXTURE_2D;
	var tex_type = gl.TEXTURE_2D;
	if(this._texture == null || this._texture.width != shadowmap_width || this._texture.height != shadowmap_height ||  this._texture.texture_type != tex_type || this._texture.magFilter != magFilter )
	{
		var type = gl.UNSIGNED_BYTE;
		var format = gl.RGBA;

		//not all webgl implementations support depth textures
		if( Shadowmap.use_shadowmap_depth_texture && gl.extensions.WEBGL_depth_texture )
		{
			format = gl.DEPTH_COMPONENT;
			type = gl.UNSIGNED_INT;
		}

		//create texture to store the shadowmap
		this._texture = new GL.Texture( shadowmap_width, shadowmap_height, { type: type, texture_type: tex_type, format: format, magFilter: magFilter, minFilter: gl.NEAREST });

		//if( this.precompute_static_shadowmap && (format != gl.DEPTH_COMPONENT || gl.extensions.EXT_frag_depth) )
		//	this._static_shadowmap = new GL.Texture( shadowmap_width, shadowmap_height, { type: type, texture_type: tex_type, format: format, magFilter: gl.NEAREST, minFilter: gl.NEAREST });

		//index, for debug
		this._texture.filename = ":shadowmap_" + light.uid;
		LS.ResourcesManager.textures[ this._texture.filename ] = this._texture; 

		if( this._texture.texture_type == gl.TEXTURE_2D )
		{
			if(format == gl.RGBA)
				this._fbo = new GL.FBO( [this._texture] );
			else
				this._fbo = new GL.FBO( null, this._texture );
		}
	}

	var prev_pass = LS.Renderer._current_pass;

	LS.Renderer.setRenderPass( SHADOW_PASS );
	LS.Renderer._current_light = light;
	var tmp_layer = render_settings.layers;
	render_settings.layers = this.layers;

	//render the scene inside the texture
	// Render the object viewed from the light using a shader that returns the fragment depth.
	this._texture.unbind(); 

	LS.Renderer._current_target = this._texture;
	this._fbo.bind();

	var sides = 1;
	var viewport_width = this._texture.width;
	var viewport_height = this._texture.height;
	if( light.type == LS.Light.OMNI )
	{
		sides = 6;
		viewport_height /= 6;
	}

	gl.clearColor(1, 1, 1, 1);
	if( this._texture.type == gl.DEPTH_COMPONENT )
		gl.colorMask(false,false,false,false);
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

	for(var i = 0; i < sides; ++i) //in case of omni
	{
		var shadow_camera = light.getLightCamera(i);
		if(!this._texture.near_far_planes)
			this._texture.near_far_planes = vec2.create();
		this._shadow_params[2] = this._texture.near_far_planes[0] = shadow_camera.near;
		this._shadow_params[3] = this._texture.near_far_planes[1] = shadow_camera.far;
		LS.Renderer.enableCamera( shadow_camera, render_settings, true );

		var viewport_y = 0;
		if( light.type == LS.Light.OMNI )
			viewport_y = i * viewport_height;
		gl.viewport(0,viewport_y,viewport_width,viewport_height);

		if(this.reverse_faces) //used to avoid leaking in some situations
			LS.Renderer._reverse_faces = true;

		//RENDER INSTANCES in the shadowmap
		LS.Renderer.renderInstances( render_settings, instances );

		LS.Renderer._reverse_faces = false;
	}

	this._fbo.unbind();
	LS.Renderer._current_target = null;
	gl.colorMask(true,true,true,true);

	render_settings.layers = tmp_layer;
	LS.Renderer.setRenderPass( prev_pass );
	LS.Renderer._current_light = null;
	
	if(this.onPostProcessShadowMap)
		this.onPostProcessShadowMap( this._texture );
}

Shadowmap.prototype.prepare = function( uniforms, samplers )
{
	if(!this._texture)
	{
		console.warn("shadowmap without texture?");
		return;
	}

	var light = this._light;
	var closest_far = light.computeFar();
	uniforms.u_shadow_params = this._shadow_params;
	this._shadow_params[0] = 1.0 / this._texture.width;
	this._shadow_params[1] = this.bias;
	this._shadow_extra_params[0] = this.shadow_mode;
	uniforms.u_shadow_extra = this._shadow_extra_params;
	//2 and 3 are set when rendering the shadowmap

	uniforms.shadowmap = LS.Renderer.SHADOWMAP_TEXTURE_SLOT;

	samplers[ LS.Renderer.SHADOWMAP_TEXTURE_SLOT ] = this._texture;
}

//called when we no longer need this shadowmap
Shadowmap.prototype.release = function()
{
	this._texture = null;
}

Shadowmap.prototype.toViewport = function()
{
	if(!this._texture)
		return;
	this._texture.toViewport(); //TODO: create shader to visualize correctly
}

//*******************************

Shadowmap._enabled_vertex_code ="\n\
	#pragma snippet \"light_structs\"\n\
	varying vec4 v_light_coord;\n\
	void applyLight( vec3 pos ) { \n\
		if( u_light_info.x == 1.0 ) //Omni\n\
			v_light_coord.xyz = pos - u_light_position;\n\
		else\n\
			v_light_coord = u_light_matrix * vec4(pos,1.0);\n\
	}\n\
";

Shadowmap._disabled_vertex_code ="\n\
	void applyLight(vec3 pos) {}\n\
";

Shadowmap._enabled_fragment_code = "\n\
	#ifndef TESTSHADOW\n\
		#define TESTSHADOW\n\
	#endif\n\
	#pragma shaderblock \"depth_in_color\"\n\
	#pragma snippet \"PackDepth32\"\n\
	\n\
	uniform sampler2D shadowmap;\n\
	varying vec4 v_light_coord;\n\
	uniform vec4 u_shadow_params; //[ 1.0/(texture_size), bias, near, far ]\n\
	uniform vec4 u_shadow_extra; //[hard, ...]\n\
	\n\
	float UnpackDepth(vec4 depth)\n\
	{\n\
		#ifdef BLOCK_DEPTH_IN_COLOR\n\
			const vec4 bitShift = vec4( 1.0 / (256.0 * 256.0 * 256.0), 1.0 / (256.0 * 256.0), 1.0 / 256.0, 1.0 );\n\
			return dot(depth, bitShift);\n\
		#else\n\
			return depth.x;\n\
		#endif\n\
	}\n\
	float VectorToDepthValue(vec3 Vec)\n\
	{\n\
		vec3 AbsVec = abs(Vec);\n\
		float LocalZcomp = max(AbsVec.x, max(AbsVec.y, AbsVec.z));\n\
		float n = u_shadow_params.z;\n\
		float f = u_shadow_params.w;\n\
		float NormZComp = (f+n) / (f-n) - (2.0*f*n)/(f-n)/LocalZcomp;\n\
		return (NormZComp + 1.0) * 0.5;\n\
	}\n\
	\n\
	float texsize = 1.0 / u_shadow_params.x;\n\
	float real_depth = 0.0;\n\
	\n\
	float pixelShadow( vec2 uv )\n\
	{\n\
		float sampleDepth = UnpackDepth( texture2D(shadowmap, uv) );\n\
		float depth = (sampleDepth == 1.0) ? 1.0e9 : sampleDepth; //on empty data send it to far away\n\
		if (depth > 0.0) \n\
			return real_depth > depth ? 0.0 : 1.0;\n\
		return 0.0;\n\
	}\n\
	float expFunc(float f)\n\
	{\n\
		return f*f*f*(f*(f*6.0-15.0)+10.0);\n\
	}\n\
	\n\
	#pragma snippet \"vec3ToCubemap2D\"\n\
	\n\
	float testShadow( Light LIGHT )\n\
	{\n\
		vec3 offset = vec3(0.0);\n\
		float depth = 0.0;\n\
		float bias = u_shadow_params.y;\n\
		\n\
		vec2 sample;\n\
		if( LIGHT.Info.x == 1.0 ) //Omni\n\
		{\n\
			vec3 l_vector = (v_pos - u_light_position);\n\
			float dist = length(l_vector);\n\
			float pixel_z = VectorToDepthValue( l_vector );\n\
			if(pixel_z >= 0.998)\n\
				return 1.0; //fixes a little bit the far edge bug\n\
			//vec4 depth_color = textureCube( shadowmap, l_vector + offset * dist );\n\
			sample = vec3ToCubemap2D( l_vector/dist );\n\
			vec4 depth_color = texture2D( shadowmap, sample );\n\
			float ShadowVec = UnpackDepth( depth_color );\n\
			if ( ShadowVec > pixel_z - bias )\n\
				return 1.0; //no shadow\n\
			return 0.0; //full shadow\n\
		}\n\
		sample = (v_light_coord.xy / v_light_coord.w) * vec2(0.5) + vec2(0.5) + offset.xy;\n\
		//is inside light frustum\n\
		if (clamp(sample, 0.0, 1.0) != sample) \n\
			return LIGHT.Info.x == 3.0 ? 1.0 : 0.0; //directional: outside of shadowmap, no shadow\n\
		real_depth = (v_light_coord.z - bias) / v_light_coord.w * 0.5 + 0.5;\n\
		#ifdef BLOCK_DEPTH_IN_COLOR\n\
			//real_depth = linearDepthNormalized( real_depth, u_shadow_params.z, u_shadow_params.w );\n\
		#endif\n\
		vec2 topleft_uv = sample * texsize;\n\
		if(u_shadow_extra.x == 0.0) //hard \n\
			return pixelShadow( sample );\n\
		vec2 offset_uv = fract( topleft_uv );\n\
		offset_uv.x = expFunc(offset_uv.x);\n\
		offset_uv.y = expFunc(offset_uv.y);\n\
		topleft_uv = floor(topleft_uv) * u_shadow_params.x;\n\
		float topleft = pixelShadow( topleft_uv );\n\
		float topright = pixelShadow( topleft_uv + vec2(u_shadow_params.x,0.0) );\n\
		float bottomleft = pixelShadow( topleft_uv + vec2(0.0, u_shadow_params.x) );\n\
		float bottomright = pixelShadow( topleft_uv + vec2(u_shadow_params.x, u_shadow_params.x) );\n\
		float top = mix( topleft, topright, offset_uv.x );\n\
		float bottom = mix( bottomleft, bottomright, offset_uv.x );\n\
		return mix( top, bottom, offset_uv.y );\n\
	}\n\
";

Shadowmap._disabled_fragment_code = "\nfloat testShadow( Light LIGHT ) { return 1.0; }\n";

var shadowmapping_depth_in_color_block = new LS.ShaderBlock("depth_in_color");
shadowmapping_depth_in_color_block.register();
Shadowmap.depth_in_color_block = shadowmapping_depth_in_color_block;

var shadowmapping_block = new LS.ShaderBlock("testShadow");
shadowmapping_block.addCode( GL.VERTEX_SHADER, Shadowmap._enabled_vertex_code, Shadowmap._disabled_vertex_code);
shadowmapping_block.addCode( GL.FRAGMENT_SHADER, Shadowmap._enabled_fragment_code, Shadowmap._disabled_fragment_code );
//shadowmapping_block.defineContextMacros({"SHADOWBLOCK":"testShadow"});
shadowmapping_block.register();
Shadowmap.shader_block = shadowmapping_block;


///@FILE:../src/picking.js
///@INFO: UNCOMMON
/**
* Picking is used to detect which element is below one pixel (using the GPU) or using raycast
*
* @class Picking
* @namespace LS
* @constructor
*/
var Picking = {

	picking_color_offset: 10, //color difference between picking objects
	_picking_points: [], //used during picking fetching
	_picking_nodes: null, //created before picking

	//picking
	_pickingMap: null,
	_picking_color: new Uint8Array(4),
	_picking_depth: 0,
	_picking_next_color_id: 0,
	_picking_render_settings: new RenderSettings(),
	_picking_position: vec3.create(), //last picking position in world coordinates
	_use_scissor_test: true,

	/**
	* Renders the pixel and retrieves the color to detect which object it was, slow but accurate
	* @method getNodeAtCanvasPosition
	* @param {number} x in canvas coordinates
	* @param {number} y in canvas coordinates
	* @param {Camera} camera default is all cameras
	* @param {number} layers default is 0xFFFF which is all
	* @param {Scene} scene default is GlobalScene
	*/
	getNodeAtCanvasPosition: function( x, y, camera, layers, scene )
	{
		var instance = this.getInstanceAtCanvasPosition( x, y, camera, layers, scene );
		if(!instance)
			return null;

		if(instance.constructor == LS.SceneNode)
			return instance;

		if(instance._root && instance._root.constructor == LS.SceneNode)
			return instance._root;

		if(instance.node)
			return instance.node;

		return null;
	},

	/**
	* Returns the instance under a screen position
	* @method getInstanceAtCanvasPosition
	* @param {number} x in canvas coordinates (0,0 is bottom-left)
	* @param {number} y in canvas coordinates
	* @param {Camera} camera
	* @param {number} layers default is 0xFFFF which is all
	* @param {Scene} scene
	* @return {Object} the info supplied by the picker (usually a SceneNode)
	*/
	getInstanceAtCanvasPosition: function( x, y, camera, layers, scene )
	{
		scene = scene || LS.GlobalScene;

		if(!camera)
			camera = LS.Renderer.getCameraAtPosition( x, y, scene._cameras );

		if(!camera)
			return null;

		this._picking_nodes = {};

		//render all Render Instances
		this.getPickingColorFromBuffer( scene, camera, x, y, layers );
		this._picking_color[3] = 0; //remove alpha, because alpha is always 255
		var id = new Uint32Array(this._picking_color.buffer)[0]; //get only element

		var instance_info = this._picking_nodes[id];
		this._picking_nodes = {};
		return instance_info;
	},	

	/**
	* Returns a color you should use to paint this node during picking rendering
	* you tell what info you want to retrieve associated with this object if it is clicked
	* @method getNextPickingColor
	* @param {*} info
	* @return {vec3} array containing all the RenderInstances that collided with the ray
	*/
	getNextPickingColor: function( info )
	{
		this._picking_next_color_id += this.picking_color_offset;
		var pick_color = new Uint32Array(1); //store four bytes number
		pick_color[0] = this._picking_next_color_id; //with the picking color for this object
		var byte_pick_color = new Uint8Array( pick_color.buffer ); //read is as bytes
		//byte_pick_color[3] = 255; //Set the alpha to 1

		if(!this._picking_nodes) //not necessary but used for debug
			this._picking_nodes = {};
		this._picking_nodes[ this._picking_next_color_id ] = info;
		return vec4.fromValues( byte_pick_color[0] / 255, byte_pick_color[1] / 255, byte_pick_color[2] / 255, 1 );
	},

	//x,y must be in canvas coordinates (0,0 is bottom-left)
	getPickingColorFromBuffer: function( scene, camera, x, y, layers )
	{
		//create texture
		if(this._pickingMap == null || this._pickingMap.width != gl.canvas.width || this._pickingMap.height != gl.canvas.height )
		{
			this._pickingMap = new GL.Texture( gl.canvas.width, gl.canvas.height, { format: gl.RGBA, filter: gl.NEAREST });
			this._pickingFBO = new GL.FBO([this._pickingMap]);
			//LS.ResourcesManager.textures[":picking"] = this._pickingMap; //debug the texture
		}

		var small_area = this._use_scissor_test;
		LS.Renderer._current_target = this._pickingMap;

		this._pickingFBO.bind();

			//var viewport = camera.getLocalViewport();
			//camera._real_aspect = viewport[2] / viewport[3];
			//gl.viewport( viewport[0], viewport[1], viewport[2], viewport[3] );

			if(small_area)
			{
				gl.scissor(x-1,y-1,2,2);
				gl.enable(gl.SCISSOR_TEST);
			}

			this.renderPickingBuffer( scene, camera, layers, [x,y] );

			gl.readPixels(x,y,1,1,gl.RGBA,gl.UNSIGNED_BYTE, this._picking_color );

			var depth = (this._picking_color[3] / 255);
			var linear_depth = camera.near * (depth + 1.0) / (camera.far + camera.near - depth * (camera.far - camera.near));
			this._last_depth = linear_depth * (camera.far - camera.near) + camera.near;
			this._picking_position = camera.unproject([x,y,depth],null,this._picking_position);
			//console.log(this._picking_color,this._last_depth);

			if(small_area)
				gl.disable(gl.SCISSOR_TEST);

		this._pickingFBO.unbind();

		LS.Renderer._current_target = null; //??? deprecated

		//if(!this._picking_color) this._picking_color = new Uint8Array(4); //debug
		return this._picking_color;
	},

	//pos must be in canvas coordinates (0,0 is bottom-left)
	renderPickingBuffer: function( scene, camera, layers, pos )
	{
		if(layers === undefined)
			layers = 0xFFFF;
		var picking_render_settings = this._picking_render_settings;

		LS.Renderer.enableCamera( camera, this._picking_render_settings );

		gl.clearColor(0,0,0,0);
		gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

		this._picking_next_color_id = 0;
		LS.Renderer.setRenderPass( PICKING_PASS );
		picking_render_settings.layers = layers;

		//check instances colliding with cursor using a ray against AABBs
		var instances = null;
		if( pos ) //not tested yet
		{
			var ray = camera.getRay( pos[0], pos[1] );
			var instances_collisions = LS.Physics.raycastRenderInstances( ray.origin, ray.direction, { add_instances_without_aabb: true } );
			if( instances_collisions )
			{
				instances = Array( instances_collisions.length );
				for(var i = 0; i < instances_collisions.length; ++i)
					instances[i] = instances_collisions[i].instance;
			}
			//console.log("Instances ray collided:", instances_collisions.length);
		}
		else
			instances = scene._instances;

		LS.Renderer.renderInstances( picking_render_settings, instances );

		//Nodes
		/* done in EditorView
		var ray = null;
		if(mouse_pos)
		{
			ray = camera.getRayInPixel( pos[0], pos[1] );
			ray.end = vec3.add( vec3.create(), ray.origin, vec3.scale(vec3.create(), ray.direction, 10000 ) );
		}

		for(var i = 0, l = scene._nodes.length; i < l; ++i)
		{
			var node = scene._nodes[i];
			if(!node.visible)
				continue;

			//nodes with special pickings?
			if(node.renderPicking)
				node.renderPicking(ray);

			if( node.transform )
			{
				var pos = vec3.create();
				mat4.multiplyVec3(pos, node.transform.getGlobalMatrixRef(), pos); //create a new one to store them
			}

			for(var j in node._components)
			{
				var component = node._components[j];
				if(component.renderPicking)
					component.renderPicking(ray);
			}
		}
		*/

		LEvent.trigger( scene, "renderPicking", pos );
		LEvent.trigger( LS.Renderer, "renderPicking", pos );

		LS.Renderer.setRenderPass( COLOR_PASS );
	},

	addPickingPoint: function( position, size, info )
	{
		size = size || 5.0;
		var color = LS.Picking.getNextPickingColor( info );
		this._picking_points.push([ position,color,size]);
	},

	renderPickingPoints: function()
	{
		//render all the picking points 
		if(this._picking_points.length)
		{
			var points = new Float32Array( this._picking_points.length * 3 );
			var colors = new Float32Array( this._picking_points.length * 4 );
			var sizes = new Float32Array( this._picking_points.length );
			for(var i = 0; i < this._picking_points.length; i++)
			{
				points.set( this._picking_points[i][0], i*3 );
				colors.set( this._picking_points[i][1], i*4 );
				sizes[i] = this._picking_points[i][2];
			}
			LS.Draw.setPointSize(1);
			LS.Draw.setColor([1,1,1,1]);
			gl.disable( gl.DEPTH_TEST ); //because nodes are show over meshes
			LS.Draw.renderPointsWithSize( points, colors, sizes );
			gl.enable( gl.DEPTH_TEST );
			this._picking_points.length = 0;
		}
	},

	visualize: function(v)
	{
		//to visualize picking buffer
		LS.Renderer.setRenderPass( v ? LS.PICKING_PASS : LS.COLOR_PASS );
		LS.GlobalScene.requestFrame();
	}
};

//Extra info
// renderPicking is not called from LiteScene, only from WebGLStudio EditorView

LS.Picking = Picking;



///@FILE:../src/physics.js
///@INFO: UNCOMMON
/* This is in charge of basic physics actions like ray tracing against the colliders */

/**
* Contains information about the collision of a ray and the scene
* - position: vec3
* - node: SceneNode
* - instance: could be a RenderInstance or a PhysicsInstance
* - distance: number
* @class Collision
* @namespace LS
* @constructor
* @param {SceneNode} node
* @param {PhysicsInstance|RenderInstance} instance
* @param {vec3} position collision position
* @param {number} distance
*/
function Collision( node, instance, position, distance, normal, hit )
{
	this.position = vec3.create();
	if(position)
		this.position.set(position);
	this.node = node || null; //the node belonging to this colliding object
	this.instance = instance || null; //could be a RenderInstance or a PhysicsInstance
	this.distance = distance || 0; //distance from the ray start
	this.normal = normal;
	this.hit = hit; //contains info about the collision in local space
}

Collision.isCloser = function(a,b) { return a.distance - b.distance; }

LS.Collision = Collision;





/**
* PhysicsInstance contains info of a colliding object. Used to test collisions with the scene
*
* @class PhysicsInstance
* @namespace LS
* @constructor
*/
function PhysicsInstance( node, component )
{
	this.uid = LS.generateUId("PHSX"); //unique identifier for this RI
	this.layers = 3|0;

	this.type = PhysicsInstance.BOX; //SPHERE, MESH
	this.mesh = null; 

	//where does it come from
	this.node = node;
	this.component = component;

	//transformation
	this.matrix = mat4.create();
	this.center = vec3.create(); //in world space

	//for visibility computation
	this.oobb = BBox.create(); //local object oriented bounding box
	this.aabb = BBox.create(); //world axis aligned bounding box
}

PhysicsInstance.BOX = 1;
PhysicsInstance.SPHERE = 2;
PhysicsInstance.PLANE = 3;
PhysicsInstance.CAPSULE = 4;
PhysicsInstance.MESH = 5;
PhysicsInstance.FUNCTION = 6; //used to test against a internal function

/**
* Computes the instance bounding box in world space from the one in local space
*
* @method updateAABB
*/
PhysicsInstance.prototype.updateAABB = function()
{
	BBox.transformMat4( this.aabb, this.oobb, this.matrix );
}

PhysicsInstance.prototype.setMesh = function(mesh)
{
	this.mesh = mesh;
	this.type = PhysicsInstance.MESH;	
	BBox.setCenterHalfsize( this.oobb, BBox.getCenter( mesh.bounding ), BBox.getHalfsize( mesh.bounding ) );
}

LS.PhysicsInstance = PhysicsInstance;



/**
* Physics is in charge of all physics testing methods
* Right now is mostly used for testing collisions with rays agains the colliders in the scene
*
* @class Physics
* @namespace LS
* @constructor
*/
var Physics = {

	/**
	* Cast a ray that traverses the scene checking for collisions with Colliders
	* @method raycast
	* @param {vec3} origin in world space
	* @param {vec3} direction in world space
	* @param {Object} options ( max_dist maxium distance, layers which layers to check, scene, first_collision )
	* @return {Array} Array of Collision objects containing all the nodes that collided with the ray or null in the form of a LS.Collision
	*/
	raycast: function( origin, direction, options )
	{
		options = options || {};

		if(!origin || !direction)
			throw("Physics.raycast: origin or direction missing.");

		var layers = options.layers;
		if(layers === undefined)
			layers = 0xFFFF;
		var max_distance = options.max_distance || Number.MAX_VALUE;
		var scene = options.scene || LS.GlobalScene;
		var first_collision = options.first_collision;

		var colliders = options.colliders || scene._colliders;
		var collisions = [];

		var compute_normal = !!options.normal;

		if(!colliders)
			return null;

		var local_origin = vec3.create();
		var local_direction = vec3.create();

		//for every instance
		for(var i = 0; i < colliders.length; ++i)
		{
			var instance = colliders[i]; //of LS.Collider

			if( (layers & instance.layers) === 0 )
				continue;

			//test against AABB
			var collision_point = vec3.create();
			var collision_normal = null;
			if( !geo.testRayBBox(origin, direction, instance.aabb, null, collision_point, max_distance) )
				continue;

			var model = instance.matrix;
			var hit = null;

			//spheres are tested in world space, is cheaper (if no non-uniform scales...)
			if( instance.type == PhysicsInstance.SPHERE )
			{
				if(!geo.testRaySphere( origin, direction, instance.center, instance.oobb[3], collision_point, max_distance))
					continue;
				if(compute_normal)
					collision_normal = vec3.sub( vec3.create(), collision_point, instance.center );
			}
			else if( instance.type == PhysicsInstance.PLANE )
			{
				var N = vec3.fromValues(0,1,0);
				mat4.rotateVec3( N, model, N );
				if(!geo.testRayPlane( origin, direction, instance.center, N, collision_point, max_distance))
					continue;
				if(compute_normal)
					collision_normal = N;
			}
			else //the rest test first with the local BBox
			{
				//ray to local instance coordinates
				var inv = mat4.invert( mat4.create(), model );
				mat4.multiplyVec3( local_origin, inv, origin);
				mat4.rotateVec3( local_direction, inv, direction);

				//test against OOBB (a little bit more expensive)
				if( !geo.testRayBBox( local_origin, local_direction, instance.oobb, null, collision_point, max_distance) )
					continue;

				//if mesh use Octree
				if( instance.type == PhysicsInstance.MESH )
				{
					var octree = instance.mesh.octree;
					if(!octree)
						octree = instance.mesh.octree = new GL.Octree( instance.mesh );
					hit = octree.testRay( local_origin, local_direction, 0.0, max_distance );
					if(!hit)
						continue;

					mat4.multiplyVec3( collision_point, model, hit.pos );
					if(compute_normal)
						collision_normal = mat4.rotateVec3( vec3.create(), model, hit.normal );

				}
				else //if just a BBox collision
				{
					vec3.transformMat4( collision_point, collision_point, model );
				}
			}

			var distance = vec3.distance( origin, collision_point );
			collisions.push( new LS.Collision( instance.node, instance, collision_point, distance, collision_normal, hit ));

			if(first_collision)
				return collisions;
		}

		//sort collisions by distance
		collisions.sort( Collision.isCloser );
		return collisions;
	},

	/**
	* Test if a sphere collides with any of the colliders in the scene
	* @method testSphere
	* @param {vec3} origin in world space
	* @param {radius} radius
	* @param {Object} options layers, colliders, scene
	* @return {PhysicsInstance} the first PhysicsObject that collided with, otherwise null
	*/
	testSphere: function( origin, radius, options )
	{
		options = options || {};
		var layers = options.layers;
		if(layers === undefined)
			layers = 0xFFFF;
		var scene = options.scene || LS.GlobalScene;

		var colliders = options.colliders || scene._colliders;
		var collisions = [];

		var local_origin = vec3.create();

		if(!colliders)
			return null;

		//for every instance
		for(var i = 0; i < colliders.length; ++i)
		{
			var instance = colliders[i];

			if( (layers & instance.layers) === 0 )
				continue;

			//test against AABB
			if( !geo.testSphereBBox( origin, radius, instance.aabb ) )
				continue;

			var model = instance.matrix;

			//ray to local
			var inv = mat4.invert( mat4.create(), model );
			mat4.multiplyVec3( local_origin, inv, origin);

			//test in world space, is cheaper
			if( instance.type == LS.PhysicsInstance.SPHERE)
			{
				if( vec3.distance( origin, local_origin ) > (radius + BBox.getRadius(instance.oobb)) )
					continue;
			}
			else //the rest test first with the local BBox
			{
				//test against OOBB (a little bit more expensive)
				if( !geo.testSphereBBox( local_origin, radius, instance.oobb) )
					continue;

				if( instance.type == LS.PhysicsInstance.MESH )
				{
					var octree = instance.mesh.octree;
					if(!octree)
						octree = instance.mesh.octree = new GL.Octree( instance.mesh );
					if( !octree.testSphere( local_origin, radius ) )
						continue;
				}
			}

			return instance;
		}

		return null;
	},

	//test collision between two PhysicsInstance 
	testCollision: function( A, B )
	{
		//test AABBs
		if( !geo.testBBoxBBox( A.aabb, B.aabb ) )
			return false;

		return true; //TODO

		//conver A to B local Space

		//test box box

		//test box sphere

		//test box mesh

		//test sphere box

		//test sphere sphere

		//mesh mesh not supported

		//return true;
	},

	testAllCollisions: function( on_collision, layers, scene )
	{
		if(layers === undefined)
			layers = 0xFFFF;
		scene = scene || LS.GlobalScene;

		var colliders = scene._colliders;
		var l = colliders.length;

		var collisions = false;

		for(var i = 0; i < l; ++i)
		{
			var instance_A = colliders[i];

			if( (layers & instance_A.layers) === 0 )
				continue;

			for(var j = i+1; j < l; ++j)
			{
				var instance_B = colliders[j];

				if( (layers & instance_B.layers) === 0 )
					continue;

				if( this.testCollision( instance_A, instance_B ) )
				{
					if(on_collision)
						on_collision( instance_A, instance_B );
					collisions = true;
				}
			}
		}

		return collisions;
	},

	/**
	* Cast a ray that traverses the scene checking for collisions with RenderInstances instead of colliders
	* Similar to Physics.raycast but using the RenderInstances (if options.triangle_collision it builds Octrees for the RIs whose OOBB collides with the ray)
	* @method raycastRenderInstances
	* @param {vec3} origin in world space
	* @param {vec3} direction in world space
	* @param {Object} options { instances: array of instances, if not the scene will be used, triangle_collision: true if you want to test against triangles, max_distance: maxium ray distance, layers, scene, max_distance, first_collision : returns the first collision (which could be not the closest one) }
	* @return {Array} array containing all the RenderInstances that collided with the ray in the form [SceneNode, RenderInstance, collision point, distance]
	*/
	raycastRenderInstances: function( origin, direction, options )
	{
		options = options || {};
		var layers = options.layers;
		if(layers === undefined)
			layers = 0xFFFF;
		var max_distance = options.max_distance || Number.MAX_VALUE;
		var scene = options.scene || LS.GlobalScene;

		var triangle_collision = !!options.triangle_collision;
		var first_collision = !!options.first_collision;
		var compute_normal = !!options.normal;
		var ignore_transparent = !!options.ignore_transparent;

		var instances = options.instances || scene._instances;
		if(!instances || !instances.length)
			return null;

		var collisions = [];

		var local_origin = vec3.create();
		var local_direction = vec3.create();

		//for every instance
		for(var i = 0, l = instances.length; i < l; ++i)
		{
			var instance = instances[i];

			if((layers & instance.layers) === 0 ) //|| !(instance.flags & RI_RAYCAST_ENABLED) 
				continue;

			if(instance.material && instance.material.render_state.blend && ignore_transparent)
				continue; //avoid semitransparent

			if( !instance.use_bounding && options.add_instances_without_aabb)
			{
				collisions.push( new LS.Collision( instance.node, instance, vec3.clone(origin), 0, vec3.clone(direction), null ) );
				continue;
			}

			//test against AABB
			var collision_point = vec3.create();
			if( !geo.testRayBBox( origin, direction, instance.aabb, null, collision_point, max_distance ) )
				continue;

			var model = instance.matrix;
			var hit = null;

			//ray to local
			var inv = mat4.invert( mat4.create(), model );
			mat4.multiplyVec3( local_origin, inv, origin );
			mat4.rotateVec3( local_direction, inv, direction );

			//test against OOBB (a little bit more expensive)
			if( !geo.testRayBBox( local_origin, local_direction, instance.oobb, null, collision_point, max_distance) )
				continue;

			//check which mesh to use
			var collision_normal = null;
			
			//test against mesh
			if( triangle_collision )
			{
				var collision_mesh = instance.lod_mesh || instance.mesh;
				var octree = collision_mesh.octree;
				if(!octree)
					octree = collision_mesh.octree = new GL.Octree( collision_mesh );
				hit = octree.testRay( local_origin, local_direction, 0.0, max_distance );
				if(!hit)
					continue;
				mat4.multiplyVec3( collision_point, model, hit.pos );
				if(compute_normal)
					collision_normal = mat4.rotateVec3( vec3.create(), model, hit.normal );
			}
			else
				vec3.transformMat4( collision_point, collision_point, model );

			//compute distance
			var distance = vec3.distance( origin, collision_point );
			if(distance < max_distance)
				collisions.push( new LS.Collision( instance.node, instance, collision_point, distance, collision_normal, hit ) );

			if(first_collision)
				return collisions;
		}

		collisions.sort( LS.Collision.isCloser );
		return collisions;
	},

	/**
	* Cast a ray that traverses the scene checking for collisions with RenderInstances instead of colliders
	* Similar to Physics.raycast but using the RenderInstances (if options.triangle_collision it builds Octrees for the RIs whose OOBB collides with the ray)
	* @method raycastRenderInstances
	* @param {vec3} origin in world space
	* @param {vec3} direction in world space
	* @param {LS.SceneNode} node 
	* @param {Object} options ( triangle_collision: true if you want to test against triangles, max_distance: maxium ray distance, layers, scene, max_distance, first_collision : returns the first collision (which could be not the closest one) )
	* @return {Array} array containing all the RenderInstances that collided with the ray in the form [SceneNode, RenderInstance, collision point, distance]
	*/
	raycastNode: function( origin, direction, node, options )
	{
		options = options || {};
		options.instances = node._instances;
		return this.raycastRenderInstances( origin, direction, options );
	}
}


LS.Physics = Physics;
///@FILE:../src/components/transform.js
///@INFO: BASE
/**
* Transform that contains the position (vec3), rotation (quat) and scale (vec3) 
* It uses lazy update to recompute the matrices.
* @class Transform
* @namespace LS.Components
* @constructor
* @param {Object} object to configure from
*/

function Transform( o )
{
	//packed data (helpful for animation stuff)
	this._data = new Float32Array( 3 + 4 + 3 ); //pos, rot, scale, also known as trans10

	//TSR
	this._position = this._data.subarray(0,3);
	this._rotation = this._data.subarray(3,7);
	quat.identity(this._rotation);
	this._scaling = this._data.subarray(7,10);
	this._scaling[0] = this._scaling[1] = this._scaling[2] = 1;

	//matrices
	this._local_matrix = mat4.create();
	this._global_matrix = mat4.create();

	this._uid = null;
	this._root = null;
	this._parent = null;

	this._must_update = false; //local matrix must be redone
	this._version = 0;

	/* JS feature deprecated
	if(Object.observe)
	{
		var inner_transform_change = (function(c) { 
			this._must_update = true;
		}).bind(this);
		Object.observe( this._position, inner_transform_change );
		Object.observe( this._rotation, inner_transform_change );
		Object.observe( this._scaling, inner_transform_change );
		Object.observe( this._data, inner_transform_change );
	}
	*/

	if(o)
		this.configure(o);
}

Transform.temp_matrix = mat4.create();
Transform.icon = "mini-icon-gizmo.png";
Transform.ZERO = vec3.create();
Transform.UP = vec3.fromValues(0,1,0);
Transform.RIGHT = vec3.fromValues(1,0,0);
Transform.FRONT = vec3.fromValues(0,0,-1);

Transform.TRANS10_IDENTITY = new Float32Array([0,0,0,0,0,0,1,1,1,1]);

Transform["@rotation"] = { type: "quat"};
Transform["@data"] = { type: "trans10" };

//what is this used for??
Transform.properties = {
	position:"vec3",
	scaling:"vec3",
	rotation:"quat"
};

Transform.prototype.onAddedToNode = function(node)
{
	if(!node.transform)
		node.transform = this;
}

Transform.prototype.onRemovedFromNode = function(node)
{
	if(node.transform == this)
		delete node["transform"];
}

/**
* The position relative to its parent in vec3 format
* @property position {vec3}
*/
Object.defineProperty( Transform.prototype, 'position', {
	get: function() { return this._position; },
	set: function(v) { 
		if(!v || !v.length)
			return;
		this._position.set(v); 
		this._must_update = true; 
	},
	enumerable: true
});

Object.defineProperty( Transform.prototype, 'x', {
	get: function() { return this._position[0]; },
	set: function(v) { 
		this._position[0] = v; 
		this._must_update = true; 
	},
	enumerable: false
});

Object.defineProperty( Transform.prototype, 'y', {
	get: function() { return this._position[1]; },
	set: function(v) { 
		this._position[1] = v; 
		this._must_update = true; 
	},
	enumerable: false
});

Object.defineProperty( Transform.prototype, 'z', {
	get: function() { return this._position[2]; },
	set: function(v) { 
		this._position[2] = v; 
		this._must_update = true; 
	},
	enumerable: false
});

/*
Object.defineProperty( Transform.prototype, 'pitch', {
	get: function() { return 0; },
	set: function(v) { 
		this.rotateX(v);
		this._must_update = true; 
	},
	enumerable: false
});
*/

/**
* The orientation relative to its parent in quaternion format
* @property rotation {quat}
*/
Object.defineProperty( Transform.prototype, 'rotation', {
	get: function() { return this._rotation; },
	set: function(v) { 
		this._rotation.set(v);
		this._must_update = true;
	},
	enumerable: true //avoid problems
});

/**
* The scaling relative to its parent in vec3 format (default is [1,1,1])
* @property scaling {vec3}
*/
Object.defineProperty( Transform.prototype, 'scaling', {
	get: function() { return this._scaling; },
	set: function(v) { 
		if(v.constructor === Number)
			this._scaling[0] = this._scaling[1] = this._scaling[2] = v;
		else
			this._scaling.set(v);
		this._must_update = true;
	},
	enumerable: true
});

/**
* The scaling relative to its parent in vec3 format (default is [1,1,1])
* @property scaling {vec3}
*/
Object.defineProperty( Transform.prototype, 'uniform_scaling', {
	get: function() { return this._scaling[0]; },
	set: function(v) { 
		this._scaling[0] = this._scaling[1] = this._scaling[2] = v;
		this._must_update = true;
	},
	enumerable: true
});

/**
* The local matrix transform relative to its parent in mat4 format
* @property matrix {mat4}
*/
Object.defineProperty( Transform.prototype, 'matrix', {
	get: function() { 
		if(this._must_update)
			this.updateMatrix();
		return this._local_matrix;
	},
	set: function(v) { 
		this.fromMatrix(v);	
	},
	enumerable: true
});

//this is used to speed up copying between transforms and for animation (better to animate one track than several)
Object.defineProperty( Transform.prototype, 'data', {
	get: function() { 
		return this._data;
	},
	set: function(v) { 
		this._data.set(v);	
		this._must_update = true;
	},
	enumerable: false
});

//in degrees
Object.defineProperty( Transform.prototype, 'xrotation', {
	get: function() { return 0; },
	set: function(v) { 
		this.rotateX(v * DEG2RAD);
	},
	enumerable: false
});

//in degrees
Object.defineProperty( Transform.prototype, 'yrotation', {
	get: function() { return 0; },
	set: function(v) { 
		this.rotateY(v * DEG2RAD);
	},
	enumerable: false
});

//in degrees
Object.defineProperty( Transform.prototype, 'zrotation', {
	get: function() { return 0; },
	set: function(v) { 
		this.rotateZ(v * DEG2RAD);
	},
	enumerable: false
});



/**
* The position relative to its parent in vec3 format
* @property position {vec3}
*/
Object.defineProperty( Transform.prototype, 'globalPosition', {
	get: function() { return this.getGlobalPosition(); },
	set: function(v) { 
	},
	enumerable: true
});

/**
* The matrix transform relative to world coordinates
* @property globalMatrix {mat4}
*/
Object.defineProperty( Transform.prototype, 'globalMatrix', {
	get: function() { 
		this.updateGlobalMatrix();
		return this._global_matrix;
	},
	set: function(v) { 
		throw("globalMatrix cannot be set, use fromMatrix(m,true)");
	},
	enumerable: true
});

/**
* The forward vector in global coordinates
* @property forward {mat4}
*/
Object.defineProperty( Transform.prototype, 'forward', {
	get: function() { 
		this.updateGlobalMatrix();
		return mat4.rotateVec3( vec3.create(), this._global_matrix, LS.FRONT );
	},
	set: function(v) { 
		throw("forward cannot be set");
	},
	enumerable: false //dont worry, it uses its own serialize
});

/**
* Force object to update matrices in case they were modified
* @property mustUpdate {boolean}
*/
Object.defineProperty( Transform.prototype, 'mustUpdate', {
	get: function() { 
		return this._must_update;
	},
	set: function(v) { 
		this._must_update = true;
	},
	enumerable: false
});

Transform.prototype.getPropertiesInfo = function(v)
{
	if(v == "output")
	{
		return {
			position:"vec3",
			scaling:"vec3",
			rotation:"quat",
			matrix:"mat4",
			globalPosition:"vec3",
			globalMatrix:"mat4"
		};
	} 
	else //if(v == "input")
	{
		return {
			position:"vec3",
			scaling:"vec3",
			rotation:"quat",
			matrix:"mat4"
		};
	}
}


/**
* Copy the transform from another Transform
* @method copyFrom
* @param {Transform} src
*/
Transform.prototype.copyFrom = function(src)
{
	this.configure( src.serialize() );
}

/**
* Configure from a serialized object
* @method configure
* @param {Object} object with the serialized info
*/
Transform.prototype.configure = function(o)
{
	if(o.uid) this.uid = o.uid;
	if(o.position) this._position.set( o.position );
	if(o.scaling) this._scaling.set( o.scaling );

	if(o.rotation && o.rotation.length == 4)
		this._rotation.set( o.rotation );
	if(o.rotation && o.rotation.length == 3)
	{
		quat.identity( this._rotation );
		var R = quat.setAngleAxis( quat.create(), [1,0,0], o.rotation[0] * DEG2RAD);
		quat.multiply(this._rotation, this._rotation, R ); 
		quat.setAngleAxis( R, [0,1,0], o.rotation[1] * DEG2RAD );
		quat.multiply(this._rotation, this._rotation, R ); 
		quat.setAngleAxis( R, [0,0,1], o.rotation[2] * DEG2RAD );
		quat.multiply(this._rotation, this._rotation, R ); 
	}

	this._must_update = true;
	this.updateGlobalMatrix();
	this._on_change();
}

/**
* Serialize the object 
* @method serialize
* @return {Object} object with the serialized info
*/
Transform.prototype.serialize = function( simplified )
{
	
	var o = {
		object_class: "Transform",
		uid: this.uid,
		position: [ this._position[0],this._position[1],this._position[2] ],
		rotation: [ this._rotation[0],this._rotation[1],this._rotation[2],this._rotation[3] ],
		scaling: [ this._scaling[0],this._scaling[1],this._scaling[2] ]
	};

	if( !this.isIdentity() && !simplified )
		o.matrix = toArray( this._local_matrix );; //could be useful

	return o;
}

Transform.prototype.isIdentity = function()
{
	for(var i = 0; i < this._local_matrix.length; ++i)
		if( Math.abs( this._local_matrix[i] - LS.IDENTITY[i] ) > 0.001 )
			return false;
	return true;
}

/**
* Reset this transform
* @method identity
*/
Transform.prototype.identity = function()
{
	vec3.copy(this._position, LS.ZEROS );
	quat.identity( this._rotation );
	vec3.copy(this._scaling, LS.ONES );
	mat4.identity(this._local_matrix);
	mat4.identity(this._global_matrix);
	this._version += 1;
	this._must_update = false;
}

Transform.prototype.reset = Transform.prototype.identity;

/**
* Sets the rotation to identity
* @method resetRotation
*/
Transform.prototype.resetRotation = function()
{
	quat.identity( this._rotation );
	this._version += 1;
	this._must_update = true;
	this._on_change();
}

/**
* Sets the position to 0,0,0
* @method resetPosition
*/
Transform.prototype.resetPosition = function()
{
	vec3.copy( this._position, LS.ZEROS );
	this._version += 1;
	this._must_update = true;
	this._on_change(true);
}

/**
* Sets the scale to 1,1,1
* @method resetScale
*/
Transform.prototype.resetScale = function()
{
	vec3.copy( this._scaling, LS.ONES );
	this._version += 1;
	this._must_update = true;
	this._on_change(true);
}


/**
* Returns a copy of the local position
* @method getPosition
* @param {vec3} out [optional] where to store the result, otherwise one vec3 is created and returned
* @return {vec3} the position
*/
Transform.prototype.getPosition = function(out)
{
	out = out || vec3.create();
	out.set( this._position );
	return out;
}

/**
* Returns a copy of the global position
* @method getGlobalPosition
* @param {vec3} out [optional] where to store the result, otherwise one vec3 is created and returned
* @return {vec3} the position
*/
Transform.prototype.getGlobalPosition = function(out)
{
	out = out || vec3.create();
	if(this._parent) 
		return mat4.multiplyVec3( out, this.getGlobalMatrix(), Transform.ZERO ); //cannot reuse matrix in getGlobalMatrix, is recursive
	return vec3.copy(out, this._position );
}

/**
* Returns the rotation in quaternion array (a copy)
* @method getRotation
* @param {quat} out [optional] where to store the result, otherwise one quat is created and returned
* @return {quat} the rotation
*/
Transform.prototype.getRotation = function(out)
{
	out = out || quat.create();
	return vec3.copy(out,this._rotation);
}

/**
* Returns the global rotation in quaternion array (a copy)
* @method getRotation
* @param {quat} out [optional] where to store the result, otherwise one quat is created and returned
* @return {quat} the rotation
*/
Transform.prototype.getGlobalRotation = function(out)
{
	out = out || quat.create();
	if( !this._parent )
	{
		quat.copy(out, this._rotation);
		return out;
	}

	var aux = this._parent;
	quat.copy(out,this._rotation);
	while(aux)
	{
		quat.multiply(out, aux._rotation, out);
		aux = aux._parent;
	}
	return out;
}


/**
* Returns the scale (its a copy)
* @method getScale
* @param {vec3} out [optional] where to store the result, otherwise one vec3 is created and returned
* @return {vec3} the scale
*/
Transform.prototype.getScale = function(out)
{
	out = out || vec3.create();
	return vec3.copy(out,this._scaling);
}

/**
* Returns a copy of the global scale (this is not correct, there is no global_scale factor, because due to rotations the axis could change)
* @method getGlobalScale
* @param {vec3} out [optional] where to store the result, otherwise one vec3 is created and returned
* @return {vec3} the scale
*/
Transform.prototype.getGlobalScale = function(out)
{
	out = out || vec3.create();
	if( this._parent )
	{
		var aux = this;
		vec3.copy(out,this._scaling);
		while(aux._parent)
		{
			vec3.multiply(out, out, aux._scaling);
			aux = aux._parent;
		}
		return out;
	}
	return vec3.copy(out, this._scaling);
}

/**
* update the local Matrix to match the position,scale and rotation
* @method updateMatrix
*/
Transform.prototype.updateMatrix = function()
{
	mat4.fromRotationTranslation( this._local_matrix , this._rotation, this._position );
	mat4.scale(this._local_matrix, this._local_matrix, this._scaling);
	this._must_update = false;
	this._version += 1;
	this.updateDescendants();
}
Transform.prototype.updateLocalMatrix = Transform.prototype.updateMatrix;

/**
* updates the global matrix using the parents transformation
* @method updateGlobalMatrix
* @param {bool} fast it doesnt recompute parent matrices, just uses the stored one, is faster but could create errors if the parent doesnt have its global matrix update
*/
Transform.prototype.updateGlobalMatrix = function (fast)
{
	if(this._must_update)
		this.updateMatrix();
	if (this._parent)
		mat4.multiply( this._global_matrix, fast ? this._parent._global_matrix : this._parent.getGlobalMatrix( this._parent._global_matrix ), this._local_matrix );
	else
		this._global_matrix.set( this._local_matrix ); 
}

/**
* Returns a copy of the local matrix of this transform (it updates the matrix automatically)
* @method getMatrix
* @param {mat4} out [optional] where to store the result, otherwise one mat4 is created and returned
* @return {mat4} the matrix
*/
Transform.prototype.getMatrix = function (out)
{
	out = out || mat4.create();
	if(this._must_update)
		this.updateMatrix();
	return mat4.copy(out, this._local_matrix);
}
Transform.prototype.getLocalMatrix = Transform.prototype.getMatrix; //alias

/**
* Returns the original local matrix of this transform (it updates the matrix automatically)
* @method getLocalMatrixRef
* @return {mat4} the matrix in array format
*/
Transform.prototype.getLocalMatrixRef = function ()
{
	if(this._must_update)
		this.updateMatrix();
	return this._local_matrix;
}


/**
* Returns a copy of the global matrix of this transform (it updates the matrix automatically)
* @method getGlobalMatrix
* @param {mat4} out optional
* @param {boolean} fast this flags skips recomputing parents matrices
* @return {mat4} the matrix in array format
*/
Transform.prototype.getGlobalMatrix = function (out, fast)
{
	if(this._must_update)
		this.updateMatrix();
	out = out || mat4.create();
	if (this._parent)
		mat4.multiply( this._global_matrix, fast ? this._parent._global_matrix : this._parent.getGlobalMatrix( this._parent._global_matrix ), this._local_matrix );
	else
		mat4.copy( this._global_matrix, this._local_matrix ); 
	return mat4.copy(out, this._global_matrix);
}

/**
* Returns a copy of the global matrix of this transform (it updates the matrix automatically)
* @method getGlobalMatrix
* @return {mat4} the matrix in array format
*/
Transform.prototype.getGlobalMatrixRef = function (fast)
{
	this.updateGlobalMatrix(fast);
	return this._global_matrix;
}



/**
* Returns an array with all the ancestors
* @method getAncestors
* @return {Array} 
*/
Transform.prototype.getAncestors = function()
{
	var r = [ this ];
	var aux = this;
	while(aux = aux._parent)
		r.unshift(aux);	
	return r;
}

/**
* Returns a quaternion with all parents rotations
* @method getGlobalRotation
* @return {quat} Quaternion
*/
/*
Transform.prototype.getGlobalRotation = function (q)
{
	q = q || quat.create();
	q.set(this._rotation);

	//concatenate all parents rotations
	var aux = this._parent;
	while(aux)
	{
		quat.multiply(q,q,aux._rotation);
		aux = aux._parent;
	}
	return q;
}
*/
/**
* Returns a Matrix with all parents rotations
* @method getGlobalRotationMatrix
* @return {mat4} Matrix rotation
*/
/*
Transform.prototype.getGlobalRotationMatrix = function (m)
{
	var q = quat.clone(this._rotation);

	var aux = this._parent;
	while(aux)
	{
		quat.multiply(q, q, aux._rotation);
		aux = aux._parent;
	}

	m = m || mat4.create();
	return mat4.fromQuat(m,q);
}
*/


/**
* Returns the local matrix of this transform without the rotation or scale
* @method getGlobalTranslationMatrix
* @return {mat4} the matrix in array format
*/
Transform.prototype.getGlobalTranslationMatrix = function ()
{
	var pos = this.getGlobalPosition();
	return mat4.fromValues(1,0,0,0, 0,1,0,0, 0,0,1,0, pos[0], pos[1], pos[2], 1);
}

/**
* Returns the global rotation in quaternion array (a copy)
* @method getGlobalRotationMatrix
* @return {mat4} the rotation
*/
Transform.prototype.getGlobalRotationMatrix = function(out)
{
	var out = out || mat4.create();
	if( !this._parent )
		return mat4.fromQuat( out, this._rotation );
		
	var r = mat4.create();
	var aux = this;
	while( aux )
	{
		mat4.fromQuat(r, aux._rotation);
		mat4.multiply(out,out,r);
		aux = aux._parent;
	}
	return out;
}


/**
* Returns the local matrix of this transform without the scale
* @method getGlobalTranslationRotationMatrix
* @return {mat4} the matrix in array format
*/
Transform.prototype.getGlobalTranslationRotationMatrix = function ()
{
	var pos = this.getGlobalPosition();
	return mat4.fromRotationTranslation(mat4.create(), this.getGlobalRotation(), pos);
}
Transform.prototype.getGlobalMatrixWithoutScale = Transform.prototype.getGlobalTranslationRotationMatrix;



/**
* Returns the matrix for the normals in the shader
* @method getNormalMatrix
* @return {mat4} the matrix in array format
*/
Transform.prototype.getNormalMatrix = function (m)
{
	if(this._must_update)
		this.updateMatrix();

	m = m || mat4.create();
	if (this._parent)
		mat4.multiply( this._global_matrix, this._parent.getGlobalMatrix(), this._local_matrix );
	else
		m.set(this._local_matrix); //return local because it has no parent
	return mat4.transpose(m, mat4.invert(m,m) );
}

/**
* Configure the transform from a local Matrix (do not tested carefully)
* @method fromMatrix
* @param {mat4} matrix the matrix in array format
* @param {bool} is_global tells if the matrix is in global space [optional]
*/
Transform.prototype.fromMatrix = (function() { 

	var global_temp = mat4.create();
	var temp_mat4 = mat4.create();
	var temp_mat3 = mat3.create();
	var temp_vec3 = vec3.create();
	//var scale_temp = mat4.create();
	
	return function fromMatrix( m, is_global )
	{
		if(is_global && this._parent)
		{
			mat4.copy(this._global_matrix, m); //assign to global
			var M_parent = this._parent.getGlobalMatrix( global_temp ); //get parent transform
			var r = mat4.invert( M_parent, M_parent ); //invert
			if(!r)
				return;
			m = mat4.multiply( this._local_matrix, M_parent, m ); //transform from global to local
		}

		//pos
		var M = temp_mat4;
		M.set(m);
		mat4.multiplyVec3( this._position, M, LS.ZEROS );

		//compute scale
		this._scaling[0] = vec3.length( mat4.rotateVec3( temp_vec3, M, LS.RIGHT) );
		this._scaling[1] = vec3.length( mat4.rotateVec3( temp_vec3, M, LS.TOP) );
		this._scaling[2] = vec3.length( mat4.rotateVec3( temp_vec3, M, LS.BACK) );

		//apply scale, why the inverse? ??
		//mat4.scale( scale_temp, M, [1/this._scaling[0], 1/this._scaling[1], 1/this._scaling[2]] );

		//quat.fromMat4(this._rotation, M);
		//*
		//normalize system vectors
		vec3.normalize( M.subarray(0,3), M.subarray(0,3) );
		vec3.normalize( M.subarray(4,7), M.subarray(4,7) );
		vec3.normalize( M.subarray(8,11), M.subarray(8,11) );

		var M3 = mat3.fromMat4( temp_mat3, M );
		quat.fromMat3AndQuat( this._rotation, M3 );

		/* works with default fromMat3, not with fromMat3AndQuat
		var M3 = mat3.fromMat4( temp_mat3, M );
		mat3.transpose( M3, M3 ); //why transpose?!?!
		quat.fromMat3( this._rotation, M3 );
		quat.normalize( this._rotation, this._rotation );
		//*/

		if(m != this._local_matrix)
			mat4.copy(this._local_matrix, m);
		this._must_update = false;
		this._version += 1;
		this._on_change(true);
	}
})();

/**
* Configure the transform from a global Matrix (do not tested carefully)
* @method fromGlobalMatrix
* @param {mat4} matrix the matrix in array format
*/
Transform.prototype.fromGlobalMatrix = function(m)
{
	this.fromMatrix(m,true);	
}

Transform.fromMatrix4ToTransformData = (function() { 

	var global_temp = mat4.create();
	var temp_mat4 = mat4.create();
	var temp_mat3 = mat3.create();
	var temp_vec3 = vec3.create();
	
	return function fromMatrix4ToTransformData( m, out )
	{
		var data = out || new Float32Array( 3 + 4 + 3 ); //pos, rot, scale
		var position = data.subarray(0,3);
		var rotation = data.subarray(3,7);
		quat.identity(rotation);
		var scaling = data.subarray(7,10);

		//pos
		var M = temp_mat4;
		M.set(m);
		mat4.multiplyVec3( position, M, LS.ZEROS );

		//extract scaling by 
		scaling[0] = vec3.length( mat4.rotateVec3( temp_vec3, M, LS.RIGHT) );
		scaling[1] = vec3.length( mat4.rotateVec3( temp_vec3, M, LS.TOP) );
		scaling[2] = vec3.length( mat4.rotateVec3( temp_vec3, M, LS.BACK) );

		//quat.fromMat4( rotation, M ); //doesnt work

		//normalize axis vectors
		vec3.normalize( M.subarray(0,3), M.subarray(0,3) );
		vec3.normalize( M.subarray(4,7), M.subarray(4,7) );
		vec3.normalize( M.subarray(8,11), M.subarray(8,11) );

		var M3 = mat3.fromMat4( temp_mat3, M );
		mat3.transpose( M3, M3 );
		quat.fromMat3( rotation, M3 );
		quat.normalize( rotation, rotation );

		return data;
	}
})();


/**
* Configure the transform rotation from a vec3 Euler angles (heading,attitude,bank)
* @method setRotationFromEuler
* @param {mat4} src, the matrix in array format
*/
Transform.prototype.setRotationFromEuler = function(v)
{
	quat.fromEuler( this._rotation, v );
	this._must_update = true;
	this._on_change();
}

/**
* sets the position
* @method setPosition
* @param {number} x 
* @param {number} y
* @param {number} z 
*/
Transform.prototype.setPosition = function(x,y,z)
{
	if(arguments.length == 3)
		vec3.set(this._position, x,y,z);
	else
		vec3.copy(this._position, x);
	this._must_update = true;
	this._on_change(true);
}

/**
* sets the rotation from a quaternion or from an angle(rad) and axis
* @method setRotation
* @param {quat} rotation in quaterion format or angle
*/
Transform.prototype.setRotation = function(q_angle,axis)
{
	if(axis)
		quat.setAxisAngle( this._rotation, axis, q_angle );
	else
		quat.copy(this._rotation, q_angle );
	this._must_update = true;
	this._on_change(true);
}

/**
* sets the scale
* @method setScale
* @param {number} x 
* @param {number} y
* @param {number} z 
*/
Transform.prototype.setScale = function(x,y,z)
{
	if(arguments.length == 3)
		vec3.set(this._scaling, x,y,z);
	else
		vec3.set(this._scaling, x,x,x);
	this._must_update = true;
	this._on_change(true);
}

/**
* translates object in local coordinates (using the rotation and the scale)
* @method translate
* @param {number} x 
* @param {number} y
* @param {number} z 
*/
Transform.prototype.translate = (function(){
	var tmp = vec3.create();
	var tmp2 = vec3.create();
	
	return function(x,y,z)
	{
		if(arguments.length == 3)
		{
			tmp2[0] = x; tmp2[1] = y; tmp2[2] = z;
			vec3.add( this._position, this._position, this.transformVector(tmp2, tmp) );
		}
		else
			vec3.add( this._position, this._position, this.transformVector(x, tmp) );
		this._must_update = true;
		this._on_change(true);
	};
})();

/**
* translates object in local coordinates (adds to the position)
* @method translateGlobal
* @param {number} x 
* @param {number} y
* @param {number} z 
*/
Transform.prototype.translateGlobal = function(x,y,z)
{
	if(arguments.length == 3)
		vec3.add( this._position, this._position, [x,y,z] );
	else
		vec3.add( this._position, this._position, x );
	this._must_update = true;
	this._on_change(true);
}

/**
* rotate object in local space (axis is in local space)
* @method rotate
* @param {number} angle_in_deg 
* @param {vec3} axis
* @param {boolean} is_global tells if the axis is in global coordinates or local coordinates
*/
Transform.prototype.rotate = (function(){

	var temp = quat.create();
	var temp_axis = quat.create();

	return function(angle_in_deg, axis, is_global )
	{
		if( is_global ) //convert global vector to local
			axis = this.globalVectorToLocal( axis, temp_axis );
		quat.setAxisAngle( temp, axis, angle_in_deg * 0.0174532925 );
		quat.multiply( this._rotation, this._rotation, temp );
		this._must_update = true;
		this._on_change(true);
	}
})();

/**
* rotate object in local space in local X axis
* @method rotateX
* @param {number} angle_in_rad
*/
Transform.prototype.rotateX = function(angle_in_rad)
{
	quat.rotateX( this._rotation, this._rotation, angle_in_rad  );
	this._must_update = true;
	this._on_change(true);
}

/**
* rotate object in local space in local Y axis
* @method rotateY
* @param {number} angle_in_rad 
*/
Transform.prototype.rotateY = function(angle_in_rad)
{
	quat.rotateY( this._rotation, this._rotation, angle_in_rad );
	this._must_update = true;
	this._on_change();
}

/**
* rotate object in local space in local Z axis
* @method rotateZ
* @param {number} angle_in_rad 
*/
Transform.prototype.rotateZ = function(angle_in_rad)
{
	quat.rotateZ( this._rotation, this._rotation, angle_in_rad );
	this._must_update = true;
	this._on_change(true);
}


/**
* rotate object in global space (axis is in global space)
* @method rotateGlobal
* @param {number} angle_in_deg 
* @param {vec3} axis
*/
Transform.prototype.rotateGlobal = function(angle_in_deg, axis)
{
	var R = quat.setAxisAngle(quat.create(), axis, angle_in_deg * 0.0174532925);
	quat.multiply(this._rotation, R, this._rotation);
	this._must_update = true;
	this._on_change(true);
}

/**
* rotate object in local space using a quat
* @method rotateQuat
* @param {quat} quaternion
*/
Transform.prototype.rotateQuat = function(quaternion)
{
	quat.multiply(this._rotation, this._rotation, quaternion);
	this._must_update = true;
	this._on_change(true);
}

/**
* rotate object in global space using a quat
* @method rotateQuatGlobal
* @param {quat} quaternion
*/
Transform.prototype.rotateQuatGlobal = function(quaternion)
{
	quat.multiply(this._rotation, quaternion, this._rotation);
	this._must_update = true;
	this._on_change(true);
}

/**
* scale the object
* @method scale
* @param {number} x 
* @param {number} y
* @param {number} z 
*/
Transform.prototype.scale = function(x,y,z)
{
	if(arguments.length == 3)
		vec3.multiply(this._scaling, this._scaling, [x,y,z]);
	else
		vec3.multiply(this._scaling, this._scaling,x);
	this._must_update = true;
	this._on_change(true);
}

/**
* This method is static (call it from Transform.interpolate)
* interpolate the transform between two transforms and stores the result in another Transform
* @method interpolate
* @param {Transform} a 
* @param {Transform} b
* @param {number} factor from 0 to 1 
* @param {Transform} the destination
*/
Transform.interpolate = function( a, b, factor, result )
{
	vec3.lerp( result._scaling, a._scaling, b._scaling, factor); //scale
	vec3.lerp( result._position, a._position, b._position, factor); //position
	quat.slerp( result._rotation, a._rotation, b._rotation, factor); //rotation
	this._must_update = true;
	this._on_change();
}

/**
* Orbits around a point
* @method orbit
* @param {number} angle_in_deg
* @param {vec3} axis
* @param {vec3} center in local coordinates
*/
Transform.prototype.orbit = (function() { 
	var tmp_quat = quat.create();
	var tmp_vec3 = vec3.create();

	return function( angle_in_deg, axis, center )
	{
		if(!center)
			throw("Transform orbit requires a center");

		var R = quat.setAxisAngle( tmp_quat, axis, angle_in_deg * 0.0174532925 );
		tmp_vec3.set( this._position );
		vec3.sub(tmp_vec3, tmp_vec3, center );
		vec3.transformQuat( tmp_vec3, tmp_vec3, R );
		vec3.add(tmp_vec3, tmp_vec3, center );
		this._position.set( tmp_vec3 );
		this._must_update = true;
	};
})();


/**
* Orients the transform to look from one position to another
* @method lookAt
* @param {vec3} position
* @param {vec3} target
* @param {vec3} up
* @param {boolean} in_world tells if the values are in world coordinates (otherwise asume its in local coordinates)
*/
Transform.prototype.lookAt = (function() { 
	var temp = mat4.create();
	return function( pos, target, up, in_world )
	{
		//compute matrix in world space
		mat4.lookAt(temp, pos, target, up);
		mat4.invert(temp, temp);
		//pass it to fromMatrix
		this.fromMatrix(temp, true);
	}
})();


/**
* Orients the transform to look at a position
* @method orientTo
* @param {vec3} target the position where to look at
* @param {boolean} in_world tells if the target is in world coordinates (otherwise asume its in local coordinates)
* @param {vec3} top [optional] a helper top vector, otherwise [0,1,0] is assumed
* @param {bool} iterative_method [optional] uses an iterative method which smoothes a little bit the result over time but gives better results
*/
Transform.prototype.orientTo = (function() { 

	//avoid garbage
	var GM = mat4.create();
	var temp = mat3.create();
	var temp4 = mat4.create();
	var temp_front = vec3.create();
	var temp_right = vec3.create();
	var temp_top = vec3.create();
	var temp_pos = vec3.create();
	//function
	return function( pos, in_world, top, iterative_method )
	{
		top = top || LS.TOP;
		//convert to local space
		/*
		if(in_world && this._parent)
		{
			this._parent.globalToLocal( pos, temp_front );
		}
		else
			temp_front.set( pos );
		*/

		if(in_world)
		{
			this.getGlobalPosition( temp_pos );
			vec3.sub( temp_front, pos, temp_pos );
		}
		else
			temp_front.set( pos );

		vec3.scale( temp_front,temp_front,-1); //reverse?

		//vec3.sub( temp_front, temp_pos, temp_front );
		vec3.normalize( temp_front, temp_front );
		if(iterative_method)
		{
			mat3.setColumn( temp, LS.RIGHT, 0 );
			mat3.setColumn( temp, top, 1 );
			mat3.setColumn( temp, temp_front, 2 );
			quat.fromMat3AndQuat( this._rotation, temp );
		}
		else
		{
			/*
			vec3.cross( temp_right, temp_front, top );
			vec3.normalize( temp_right, temp_right );
			vec3.cross( temp_top, temp_right, temp_front );
			vec3.normalize( temp_top, temp_top );
			quat.lookRotation( this._rotation, temp_front, temp_top );
			*/
			quat.lookRotation( this._rotation, temp_front, top );

			/* using mat4 doesnt work
			temp4.set(temp_right);
			temp4.set(temp_top,4);
			temp4.set(temp_front,8);
			mat4.transpose(temp4,temp4);
			quat.fromMat4( this._rotation, temp4 );
			*/
			
			/* using mat3, doesnt work
			mat3.setColumn( temp, temp_right, 0 );
			mat3.setColumn( temp, temp_top, 1 );
			mat3.setColumn( temp, temp_front, 2 );
			mat3.transpose( temp, temp );
			quat.fromMat3( this._rotation, temp );
			*/
		}
		quat.normalize( this._rotation, this._rotation );
		this._must_update = true;
	}
})();

/**
* Orients the transform so the axis points in that direction
* @method orientAxis
* @param {vec3} vector the vector to use as axis
* @param {number} axis a enum that could be LS.POSX, LS.POSY, LS.POSZ, LS.NEGX, LS.NEGY, LS.NEGZ
*/
Transform.prototype.orientAxis = (function() { 
	//avoid garbage
	var GM = mat4.create();
	var temp = mat3.create();
	//function
	return function( vector, axis )
	{
		switch(axis)
		{
			case LS.POSX: 
				mat3.setColumn( temp, vector, 0 ); //x
				mat3.setColumn( temp, LS.TOP, 1 ); //y
				mat3.setColumn( temp, LS.FRONT, 2 ); //z
				break;
			case LS.POSY:
				mat3.setColumn( temp, LS.RIGHT, 0 ); //x
				mat3.setColumn( temp, vector, 1 ); //y
				mat3.setColumn( temp, LS.FRONT, 2 ); //z
				break;
			case LS.POSZ:
				mat3.setColumn( temp, LS.RIGHT, 0 ); //x
				mat3.setColumn( temp, LS.TOP, 1 ); //y
				mat3.setColumn( temp, vector, 2 ); //z
				break;
			case LS.NEGX: 
				mat3.setColumn( temp, vector, 0 ); //x
				mat3.setColumn( temp, LS.BOTTOM, 1 ); //y
				mat3.setColumn( temp, LS.BACK, 2 ); //z
				break;
			case LS.NEGY:
				mat3.setColumn( temp, LS.LEFT, 0 ); //x
				mat3.setColumn( temp, vector, 1 ); //y
				mat3.setColumn( temp, LS.BACK, 2 ); //z
				break;
			case LS.NEGZ:
				mat3.setColumn( temp, LS.LEFT, 0 ); //x
				mat3.setColumn( temp, LS.BOTTOM, 1 ); //y
				mat3.setColumn( temp, vector, 2 ); //z
				break;
			default:
				return;
		}
		quat.fromMat3( this._rotation, temp );
		this._must_update = true;
	}
})();

//Events
Transform.prototype._on_change = function(only_events)
{
	if(!only_events)
		this._must_update = true;
	/**
	 * Fired when the node has changed its transform
	 *
	 * @event changed
	 */
	LEvent.trigger(this, "changed", this);
	if(this._root)
		LEvent.trigger(this._root, "transformChanged", this);
}

//Transform
/**
* returns the [0,0,-1] vector in global space
* @method getFront
* @return {vec3}
*/
Transform.prototype.getFront = function(out) {
	return vec3.transformQuat(out || vec3.create(), Transform.FRONT, this.getGlobalRotation() );
}

/**
* returns the [0,1,0] vector in global space
* @method getTop
* @return {vec3}
*/
Transform.prototype.getTop = function(out) {
	return vec3.transformQuat(out || vec3.create(), Transform.UP, this.getGlobalRotation() );
}

/**
* returns the [1,0,0] vector in global space
* @method getRight
* @return {vec3}
*/
Transform.prototype.getRight = function(out) {
	return vec3.transformQuat(out || vec3.create(), Transform.RIGHT, this.getGlobalRotation() );
}

/**
* Multiplies a point by the local matrix (not global)
* If no destination is specified a new vector is created
* @method transformPoint
* @param {vec3} point
* @param {vec3} destination (optional)
*/
Transform.prototype.transformPoint = function(vec, dest) {
	dest = dest || vec3.create();
	if( this._must_update )
		this.updateMatrix();
	return mat4.multiplyVec3( dest, this._local_matrix, vec );
}


/**
* convert from local coordinates to global coordinates
* If no destination is specified a new vector is created
* @method localToGlobal
* @param {vec3} point
* @param {vec3} destination (optional)
*/
Transform.prototype.localToGlobal = function(vec, dest) {
	dest = dest || vec3.create();
	if(this._must_update)
		this.updateMatrix();
	return mat4.multiplyVec3( dest, this.getGlobalMatrixRef(), vec );
}

/**
* same as localToGlobal
* @method transformPointGlobal
* @param {vec3} point
* @param {vec3} destination (optional)
*/
Transform.prototype.transformPointGlobal = Transform.prototype.localToGlobal;

/**
* convert from global coordinates to local coordinates
* @method globalToLocal
* @param {vec3} point
* @param {vec3} destination (optional)
* @return {vec3} the global coordinate in local coordinates
*/
Transform.prototype.globalToLocal = (function(){ 
	var inv = mat4.create();
	return function(vec, dest) {
		dest = dest || vec3.create();
		if(this._must_update)
			this.updateMatrix();
		if( !mat4.invert( inv, this.getGlobalMatrixRef() ) )
			return dest;
		return mat4.multiplyVec3( dest, inv, vec );
	};
})();

/**
* Applies the transformation to a vector (rotate but not translate)
* @method transformVector
* @param {vec3} vector
* @param {vec3} destination (optional)
*/
Transform.prototype.transformVector = function( vec, dest ) {
	return vec3.transformQuat( dest || vec3.create(), vec, this._rotation );
}

/**
* Applies the transformation to a vector (rotate but not translate)
* @method localVectorToGlobal
* @param {vec3} vector
* @param {vec3} destination (optional)
*/
Transform.prototype.localVectorToGlobal = function(vec, dest) {
	return vec3.transformQuat( dest || vec3.create(), vec, this.getGlobalRotation() );
}

Transform.prototype.transformVectorGlobal = Transform.prototype.localVectorToGlobal;

Transform.prototype.globalVectorToLocal = function(vec, dest) {
	var Q = this.getGlobalRotation();
	quat.invert(Q,Q);
	return vec3.transformQuat(dest || vec3.create(), vec, Q );
}

/**
* Apply a transform to this transform
* @method applyTransform
*/
Transform.prototype.applyTransform = function( transform, center, is_global )
{
	//is local

	//apply translation
	vec3.add( this._position, this._position, transform._position );

	//apply rotation
	quat.multiply( this._rotation, this._rotation, transform._rotation );

	//apply scale
	vec3.multiply( this._scaling, this._scaling, transform._scaling );

	this._must_update = true; //matrix must be redone?
}



/**
* Applies the transformation using a matrix
* @method applyTransformMatrix
* @param {mat4} matrix with the transform
* @param {vec3} center different pivot [optional] if omited 0,0,0 will be used
* @param {bool} is_global (optional) tells if the transformation should be applied in global space or local space
*/
Transform.prototype.applyTransformMatrix = (function(){ 
	var T = mat4.create();
	var inv_center = vec3.create();
	var iT = mat4.create();
	var M = mat4.create();
	var temp = mat4.create();
	
	return function(matrix, center, is_global)
	{
		var M = matrix;

		if(center)
		{
			mat4.setTranslation( T, center);
			vec3.scale( inv_center, center, -1 );
			mat4.setTranslation( iT, inv_center);

			mat4.multiply( M, T, matrix );
			mat4.multiply( M, M, iT );
		}


		if(!this._parent)
		{
			if(is_global)
			{
				this.applyLocalTransformMatrix( M );
				return;
			}

			//is local
			this.applyLocalTransformMatrix( M );
			return;
		}

		/*
		//convert transform to local coordinates
		var GM = this.getGlobalMatrix();
		var temp_mat = mat4.multiply( mat4.create(), M, GM );

		var PGM = this._parent._global_matrix;
		var inv_pgm = mat4.invert( mat4.create(), PGM );

		mat4.multiply(temp_mat, inv_pgm, temp_mat );
		this.applyLocalTransformMatrix( temp_mat );
		//*/

		//*
		var GM = this.getGlobalMatrix();
		var PGM = this._parent._global_matrix;
		mat4.multiply( this._global_matrix, M, GM );

		if(!mat4.invert( temp, PGM ))
			return;
		
		mat4.multiply( this._local_matrix, temp, this._global_matrix );
		this.fromMatrix( this._local_matrix );
		//*/
	};
})();

//applies matrix to position, rotation and scale individually, doesnt take into account parents
Transform.prototype.applyLocalTransformMatrix = (function() {
	var temp = vec3.create();
	var temp_mat3 = mat3.create();
	var temp_mat4 = mat4.create();
	var temp_quat = quat.create();

	return (function( M )
	{
		//apply translation
		vec3.transformMat4( this._position, this._position, M );

		//apply scale
		mat4.rotateVec3( temp, M, [1,0,0] );
		this._scaling[0] *= vec3.length( temp );
		mat4.rotateVec3( temp, M, [0,1,0] );
		this._scaling[1] *= vec3.length( temp );
		mat4.rotateVec3( temp, M, [0,0,1] );
		this._scaling[2] *= vec3.length( temp );

		//apply rotation
		var m = mat4.invert( temp_mat4, M );
		if(!m)
			return;

		mat4.transpose(m, m);
		var m3 = mat3.fromMat4( temp_mat3, m);
		var q = quat.fromMat3( temp_quat, m3);
		quat.normalize(q, q);
		quat.multiply( this._rotation, q, this._rotation );

		this._must_update = true; //matrix must be redone?
		this._on_change();
	});
})();


/*
Transform.prototype.applyTransformMatrix = function(matrix, center, is_global)
{
	var M = matrix;

	if(center)
	{
		var T = mat4.setTranslation( mat4.create(), center);
		var inv_center = vec3.scale( vec3.create(), center, -1 );
		var iT = mat4.setTranslation( mat4.create(), inv_center);

		M = mat4.create();
		mat4.multiply( M, T, matrix );
		mat4.multiply( M, M, iT );
	}

	if(!this._parent)
	{
		if(is_global)
			mat4.multiply(this._local_matrix, M, this._local_matrix);
		else
			mat4.multiply(this._local_matrix, this._local_matrix, M);
		this.fromMatrix(this._local_matrix);
		mat4.copy(this._global_matrix, this._local_matrix); //no parent? then is the global too
		return;
	}

	var GM = this.getGlobalMatrix();
	var PGM = this._parent._global_matrix;
	var temp = mat4.create();
	mat4.multiply( this._global_matrix, M, GM );

	mat4.invert(temp,PGM);
	mat4.multiply(this._local_matrix, temp, this._global_matrix );
	this.fromMatrix(this._local_matrix);
}
*/

//marks descendants to be updated
Transform.prototype.updateDescendants = function()
{
	if(!this._root)
		return;
	var children = this._root._children;
	if(!children)
		return;

	for(var i = 0; i < children.length; ++i)
	{
		var node = children[i];
		if(!node.transform) //bug: what if the children doesnt have a transform but the grandchilden does?! TODO FIX THIS
			continue;
		node.transform._must_update = true;
		node.transform._version += 1;
		if(node._children && node._children.length)
			node.transform.updateDescendants();
	}
}


LS.registerComponent( Transform );
LS.Transform = Transform;

///@FILE:../src/components/camera.js
///@INFO: BASE
// ******* CAMERA **************************

/**
* Camera contains the info about a camera (matrices, near far planes, clear color, etc)
* @class Camera
* @namespace LS.Components
* @constructor
* @param {Object} object to configure from
*/

function Camera(o)
{
	this.enabled = true;
	this.layers = 3;

	this.clear_color = true;
	this.clear_depth = true;

	this._type = Camera.PERSPECTIVE;

	//contain the eye, center, up if local space
	this._eye = vec3.clone( Camera.DEFAULT_EYE ); //TODO: change to position
	this._center = vec3.clone( Camera.DEFAULT_CENTER );	//TODO: change to target
	this._up = vec3.clone( Camera.DEFAULT_UP );

	//in global coordinates
	this._global_eye = vec3.clone(this._eye);
	this._global_center = vec3.clone(this._center);
	this._global_up = vec3.clone(this._up);
	this._global_front = vec3.create();
	vec3.sub( this._global_front, this._global_center, this._global_eye );
	vec3.normalize( this._global_front, this._global_front );

	//clipping planes
	this._near = 0.1;
	this._far = 1000;

	//orthographics planes (near and far took from ._near and ._far)
	this._ortho = new Float32Array([-1,1,-1,1]);

	this._aspect = 1.0; //must be one, otherwise it gets deformed, the final one used is in final_aspect
	this._fov = 45; //persp
	this._frustum_size = 50; //ortho
	this._final_aspect = 1.0; //the one used when computing the projection matrix

	//viewport in normalized coordinates: left, bottom, width, height
	this._viewport = new Float32Array([0,0,1,1]);
	this._viewport_in_pixels = vec4.create(); //viewport in screen coordinates
	this._last_viewport_in_pixels = vec4.create(); //updated when the camera is enabled from Renderer.enableCamera

	this._background_color = vec4.fromValues(0,0,0,1);

	//in case we want to overwrite the view matrix manually
	this._use_custom_projection_matrix = false; 

	//in case we want to overwrite the shader of all visible objects
	this.overwrite_material = null;

	this._view_matrix = mat4.create();
	this._projection_matrix = mat4.create();
	this._viewprojection_matrix = mat4.create();
	this._model_matrix = mat4.create(); //inverse of viewmatrix (used for local vectors)
	this._previous_viewprojection_matrix = mat4.create(); //viewmatrix from previous frame, used in some algorithms

	//lazy upload
	this._must_update_view_matrix = true;
	this._must_update_projection_matrix = true;
	this._rendering_index = -1; //tells the number of this camera in the rendering process

	//used for render to texture
	this._frame = null;
	this.show_frame = true;

	if(o) 
		this.configure(o);
	//this.updateMatrices(); //done by configure

	this._uniforms = {
		u_view: this._view_matrix,
		u_viewprojection: this._viewprojection_matrix,
		u_camera_eye: this._global_eye,
		u_camera_front: this._global_front,
		u_camera_planes: vec2.fromValues( this.near, this.far ),
		u_camera_perspective: vec3.create(),
		u_background_color: this._background_color,
		u_previous_viewprojection: this._previous_viewprojection_matrix
	};

	this._frustum_planes = this.updateFrustumPlanes(); //to create
	this.updateMatrices();

	//LEvent.bind(this,"cameraEnabled", this.onCameraEnabled.bind(this));
}

Camera.icon = "mini-icon-camera.png";

Camera.main = null; //to store the main camera of the scene
Camera.current = null; //to store the current camera

Camera.PERSPECTIVE = 1;
Camera.ORTHOGRAPHIC = 2; //orthographic adapted to aspect ratio of viewport
Camera.ORTHO2D = 3; //orthographic with manually defined left,right,top,bottom

Camera.DEFAULT_EYE = [0,0,0];
Camera.DEFAULT_CENTER = [0,0,-100];
Camera.DEFAULT_UP = [0,1,0];

Camera["@type"] = { type: "enum", values: { "perspective": Camera.PERSPECTIVE, "orthographic": Camera.ORTHOGRAPHIC, "ortho2D": Camera.ORTHO2D } };
Camera["@eye"] = { type: "vec3", widget: "position" };
Camera["@center"] = { type: "vec3", widget: "position" };
Camera["@layers"] = { type: "layers" };

// used when rendering a cubemap to set the camera view direction (crossx and crossy are for when generating a CROSS cubemap image)

//OLD VERSION, it doesnt make sense but is the one that works perfectly
Camera.cubemap_camera_parameters = [
	{ name: "posx", dir: vec3.fromValues(1,0,0), up: vec3.fromValues(0,-1,0), crossx:2, crossy:1 },
	{ name: "negx", dir: vec3.fromValues(-1,0,0), up: vec3.fromValues(0,-1,0), crossx:0, crossy:1 },
	{ name: "posy", dir: vec3.fromValues(0,1,0), up: vec3.fromValues(0,0,1), crossx:1, crossy:0 },
	{ name: "negy", dir: vec3.fromValues(0,-1,0), up: vec3.fromValues(0,0,-1), crossx:1, crossy:2 },
	{ name: "posz", dir: vec3.fromValues(0,0,1), up: vec3.fromValues(0,-1,0), crossx:1, crossy:1 },
	{ name: "negz", dir: vec3.fromValues(0,0,-1), up: vec3.fromValues(0,-1,0), crossx:3, crossy:1 }
];
//*/
/*
Camera.cubemap_camera_parameters = [
	{ name: "posx", dir: vec3.fromValues(-1,0,0), up: vec3.fromValues(0,1,0), right: vec3.fromValues(0,0,-1), crossx:0, crossy:1 },
	{ name: "negx", dir: vec3.fromValues(1,0,0), up: vec3.fromValues(0,1,0), right: vec3.fromValues(0,0,1), crossx:2, crossy:1 },
	{ name: "posy", dir: vec3.fromValues(0,-1,0), up: vec3.fromValues(0,0,-1), right: vec3.fromValues(1,0,0), crossx:1, crossy:2 },
	{ name: "negy", dir: vec3.fromValues(0,1,0), up: vec3.fromValues(0,0,1), right: vec3.fromValues(-1,0,0), crossx:1, crossy:0 },
	{ name: "posz", dir: vec3.fromValues(0,0,-1), up: vec3.fromValues(0,1,0), right: vec3.fromValues(1,0,0), crossx:1, crossy:1 },
	{ name: "negz", dir: vec3.fromValues(0,0,1), up: vec3.fromValues(0,1,0), right: vec3.fromValues(-1,0,0), crossx:3,