# Component Variants

To easily mock a component in a variety of states **with type safety** you can create variants of a component simply by adding a sibling file using a component's filename and replacing the `svelte` extension with `variants.ts`. In actual usage, you won't need to create this file manually as Kitbook will generate it for you when you request, via a variety of options through the interface. See this Kitbook's [[SearchResult]] page for an example with several different variants.

Here's a simple Variant example, the exported object's name is the name of the Variant and that object's root-level properties will be passed into the component as props:

```ts title="Greeting.variants.ts"
import type { Variant } from 'kitbook'
import type Component from './Greeting.svelte'

export const Simple: Variant<Component> = {
  greeting: 'Hello World',
}
```

Variants have some configuration options which can be controlled via a reserved `_meta` property. Here's an example with some of them:

```ts title="Greeting.variants.ts"
import type { Variant } from 'kitbook'
import type Component from './Greeting.svelte'

export const Simple: Variant<Component> = {
  greeting: 'Hello World',
  _meta: {
    description: 'A simple greeting',
    viewports: [
      { width: 320, height: 480 },
      { width: 800, height: 600 },
    ],
    contexts: [
      {
        key: 'settings',
        context: { foo: 'baz' },
      }
    ],
    // read the intellisense for the rest (languages, csr, ssr, tests, etc.)
  }
}
```

When you want to share `_meta` configuration across variants, export a `shared_meta` object which will apply to all variants except for properties specifically overwritten in a variant's defintion:

```ts
import type { Variant, VariantMeta } from 'kitbook'

export const shared_meta: VariantMeta = {
  viewports: [
    { width: 320, height: 480 },
  ],
}
```

In practice, you will also find yourself wanting to share props across your variants. You can accomplish that like this:

```ts title="Header.variants.ts"
import type { Variant, VariantMeta } from 'kitbook'
import type Component from './Header.svelte'

// ...shared_meta here if you have any

const shared = {
  githubURL: 'https://github.com/jacob-8/kitbook/',
} satisfies Partial<Variant<Component>>

export const Home_Page: Variant<Component> = {
  ...shared,
  activeURL: '/',
  _meta: {
    description: 'Displays how the header will look from the home page',
  }
}

export const User_Dashboard_Page: Variant<Component> = {
  ...shared,
  activeURL: '/dashboard',
}
```

The use of Typescript's `satisfies` keyword in the `shared` object allows us to ensure that all required props are placed into either shared or the variant itself, but are not required in both. That's a bit much to type though, so when you create variants using Kitbook's `Add Variant` button or via the [[2-viewer]], Kitbook will handle getting all that stuff ready for you. You will just need to name your variants and adjust your props.

## Fine-grained control

To make sure you caught it, all `_meta` settings can be set for the entire variants file and overwritten for individual variants. This is a powerful feature that allows you to control how your component is displayed in different contexts, languages, viewports, etc. In summary, viewports and languages can be set globally by passing settings to the Kitbook Vite plugin; on a file basis in `shared_meta`; and on an individual basis in each export variant object.

## Page and Layout Variants

`+page.svelte` files are just plain Svelte components with a very important `data` prop. So Kitbook will help create a variants file for them as well but it will replace the `+` prefix with `_` (`_page.variants.ts`) because `+` is reserved for SvelteKit files. The same applies to layout files: `_layout.variants.ts`.

---

That will be enough to keep you busy for awhile, but you will come to a point where you want to mock something which uses slots or perhaps is a composition of components working together. You'll want to keep reading to learn about [[4-component-compositions|Compositions]].


[//begin]: # "Autogenerated link references for markdown compatibility"
[SearchResult]: ../lib/layout/sidebar/search/SearchResult.md "SearchResult"
[2-viewer]: 2-viewer.md "Viewer"
[4-component-compositions|Compositions]: 4-component-compositions.md "Component Compositions"
[//end]: # "Autogenerated link references"
