# Get Started

## How to Add Kitbook to Your SvelteKit App

Kitbook has a lot of powerful features, but let's start with the bare minimum to just get going:

- `npm install -D kitbook` or `pnpm install -D kitbook`

- Add the `kitbook()` Vite plugin *before* your `sveltekit()` plugin:
	```js twoslash title="vite.config.js" {3,7}
	import { defineConfig } from 'vite'
	import { sveltekit } from '@sveltejs/kit/vite'
	import { kitbook } from 'kitbook/plugins/vite'

	export default defineConfig({
	  plugins: [
	    kitbook(),
	    sveltekit(),
	  ],
	})
	```

	*You can pass optional configuration settings to the plugin. They include `viewports, languages, github url, viewer tool customization and more`. You will see them referenced throughout the docs. Read the intellisense types to better understand them.*

- Run your dev server as normal (`npm run dev`) and Kitbook will add a `/kitbook` route to your app (`src/routes/kitbook` folder). Click the Kitbook link in your terminal to navigate to the `/kitbook` route and see your Kitbook home. The menu bar will link to all of your Svelte components, *including `+page.svelte` and `+layout.svelte` files as they are just Svelte components with a very important `data` prop*, and any markdown files in your project.

- If you are working on a fresh project that doesn't have a styles reset, Kitbook will look a little odd. Add a styles reset of your choice.

## Suggested Folder Architecture

If you have app shell components or database initialization happening in your root layout files, then you'll notice that your Kitbook is also inheriting them. This won't work and you may need to adjust your folder structure to look like this:

```txt {2,6}
src/routes/
│ (app)/
│ ├ dashboard/
│ ├ item/
│ └ +layout... <-- app shell (headers, DB logic) 
│ kitbook/
└ +layout... <-- app + Kitbook shared items (i18n, CSS)

+layout means all types: .svelte, .ts, .server.ts
```

You may find it a bit jarring to have your component workbench included in your main app. Most component prototyping tools work as a companion app, but in Kitbook's early days this created a lot of friction:
- Starting two dev servers is a pain and you will find yourself only working in your workbench or only in your app, but not both which is the whole point of a tool. If you don't use it all the time, why have it?
- Setting up the component workbench with just the right scaffolding to match the main app is a pain. It's annoying to keep styles and i18n in sync. The layout structure above makes it easy to clearly specify what context is needed for components and what is app only. 
- Status checks for Playwright E2E tests running against Vercel deployments break down when your app deployment finishes before your Kitbook workbench deployment.

Furthermore the integrated approach is the only way to get the [[2-viewer]] tool. Let's move on to learn about Kitbook's [[2-viewer]] tool which bridges the gap between our app and our component workbench. 

---

If you don't need the workbench published you can also easily add a script to remove the `kitbook` routes folder before building for production. *You could also [[1-use-by-itself-(for-a-library)|create an entirely separate SvelteKit app]] by itself and update the `import.meta.glob` values in Kitbook's root `+layout.js` to point to the main app's folder, but then you lose the all the benefits of the [[2-viewer]] so I don't recommend that approach.*

[//begin]: # "Autogenerated link references for markdown compatibility"
[2-viewer]: 2-viewer.md "Viewer"
[1-use-by-itself-(for-a-library)|create an entirely separate SvelteKit app]: 3-customizations/1-use-by-itself-(for-a-library).md "Use Kitbook by Itself"
[//end]: # "Autogenerated link references"