from datetime import datetime

import pytest

from ..partner_batch_imports import (
    Biblio,
    is_low_quality_book,
    is_published_in_future_year,
)

csv_row = "USA01961304|0962561851||9780962561856|AC|I|TC||B||Sutra on Upasaka Precepts|The||||||||2006|20060531|Heng-ching, Shih|TR||||||||||||||226|ENG||0.545|22.860|15.240|||||||P|||||||74474||||||27181|USD|30.00||||||||||||||||||||||||||||SUTRAS|BUDDHISM_SACRED BOOKS|||||||||REL007030|REL032000|||||||||HRES|HRG|||||||||RB,BIP,MIR,SYN|1961304|00|9780962561856|67499962||PRN|75422798|||||||BDK America||1||||||||10.1604/9780962561856|91-060120||20060531|||||REL007030||||||"  # noqa: E501
csv_row_with_full_date = "USA01961304|0962561851||9780962561856|AC|I|TC||B||Sutra on Upasaka Precepts|The||||||||20060531|20060531|Heng-ching, Shih|TR||||||||||||||226|ENG||0.545|22.860|15.240|||||||P|||||||74474||||||27181|USD|30.00||||||||||||||||||||||||||||SUTRAS|BUDDHISM_SACRED BOOKS|||||||||REL007030|REL032000|||||||||HRES|HRG|||||||||RB,BIP,MIR,SYN|1961304|00|9780962561856|67499962||PRN|75422798|||||||BDK America||1||||||||10.1604/9780962561856|91-060120||20060531|||||REL007030||||||"  # noqa: E501


non_books = [
    "USA39372027|8866134201||9788866134206|AC|I|DI||||Moleskine Cahier Journal (Set of 3), Pocket, Ruled, Pebble Grey, Soft Cover (3. 5 X 5. 5)||||||||||20120808|Moleskine|AU|X|||||||||||||64|ENG||0.126|13.970|8.890|1.270|X|||||T|||Cahier Journals||||1161400||||||333510|USD|9.95||||||||||||||||||||||||||||||||||||||||||||||||||||||||||BIP,OTH|39372027|01|9788866134206|69270822|||29807389||||||2328606|Moleskine||1|||||||||||||||NO|NON000000|||WZS|||",  # noqa: E501
    "AUS52496256|1452145865||9781452145860|AC|I|ZZ||||People Pencils : 10 Graphite Pencils||||||||||20160501|Sukie|AU|X|||||||||||||10|ENG||0.170|19.685|8.890|2.235|X|||||T|||||||5882||||||1717043|AUD|24.99||||||||||||||||||||||||||||NON-CLASSIFIABLE||||||||||NON000000||||||||||||||||||||BIP,OTH|52496256|02|9781452145860|51743426|||48922851|||||||Chronicle Books LLC||80|||||||||||||||NO|ART048000|||AFH|WZS||",  # noqa: E501
    "AUS49413469|1423638298||9781423638292|AC|I|ZZ||O||Keep Calm and Hang on Mug|||1 vol.|||||||20141201|Gibbs Smith Publisher Staff|DE|X||||||||||||||ENG||0.350|7.620|9.322|9.449||||||T|||||||20748||||||326333|AUD|22.99||||||||||||||||||||||||||||||||||||||||||||||||||||||||||BIP,OTH|49413469||9781423638292|50573089||OTH|1192128|||||||Gibbs Smith, Publisher||7||||||||||||||||NON000000|||WZ|||",  # noqa: E501
    "USA52681473|0735346623||9780735346628|AC|I|TY||||Klimt Expectation 500 Piece Puzzle||||||||||20160119|Galison|AU|X|Klimt, Gustav|AT|||||||||||500|ENG||0.500|20.000|20.100|5.500|X|||||T|||||||10300||||||333510|USD|13.99||||||||||||||||||||||||||||||||||||||||||||||||||||||||||BIP,OTH|52681473|28|9780735346628|70053633|||32969171|773245||||||Galison||20|||||||||||20160119||||YES|NON000000|||WZS|||",  # noqa: E501
    "AUS49852633|1423639103||9781423639107|US|I|TS||||I Like Big Books T-Shirt X-Large|||1 vol.|||||||20141201|Gibbs Smith, Publisher|DE|X||||||||||||||ENG||0.280|27.940|22.860|2.540||||||T|||||||20748||||||326333|AUD|39.99||||||||||||||||||||||||||||||||||||||||||||||||||||||||||BIP,OTH|49852633|35|9781423639107|49099247|||19801468|||||||Gibbs Smith, Publisher||1||||||||||||||||NON000000|||WZ|||",  # noqa: E501
]


class TestBiblio:
    def test_sample_csv_row(self):
        b = Biblio(csv_row.strip().split('|'))
        data = {
            'title': 'Sutra on Upasaka Precepts',
            'isbn_13': ['9780962561856'],
            'publish_date': '2006',
            'publishers': ['BDK America'],
            'weight': '0.545',
            'authors': [{'name': 'Heng-ching, Shih'}],
            'number_of_pages': 226,
            'languages': ['eng'],
            'subjects': ['Sutras', 'Buddhism, sacred books'],
            'source_records': ['bwb:9780962561856'],
            'identifiers': {
                'doi': ['10.1604/9780962561856'],
            },
            'lccn': ['91-060120'],
        }
        assert b.json() == data

    def test_sample_csv_row_with_full_date(self):
        """Only import the year. The date in the input here is in YYYYMMDD format, but should come out YYYY."""
        b = Biblio(csv_row.strip().split('|'))
        data = {
            'title': 'Sutra on Upasaka Precepts',
            'isbn_13': ['9780962561856'],
            'publish_date': '2006',
            'publishers': ['BDK America'],
            'weight': '0.545',
            'authors': [{'name': 'Heng-ching, Shih'}],
            'number_of_pages': 226,
            'languages': ['eng'],
            'subjects': ['Sutras', 'Buddhism, sacred books'],
            'source_records': ['bwb:9780962561856'],
            'identifiers': {
                'doi': ['10.1604/9780962561856'],
            },
            'lccn': ['91-060120'],
        }
        assert b.json() == data

    @pytest.mark.parametrize('input_', non_books)
    def test_non_books_rejected(self, input_):
        data = input_.strip().split('|')
        code = data[6]
        pclass = data[121]
        with pytest.raises(AssertionError, match=f'{code}/{pclass} is NONBOOK'):
            _ = Biblio(data)


def test_is_low_quality_book():
    book = {"title": "A NoTeBoOk Z", "authors": [{"name": "Al"}, {"name": "Zach"}]}
    assert is_low_quality_book(book) is False, book
    book["authors"] = [{"name": "Al"}, {"name": "hOlO"}, {"name": "Zach"}]
    assert is_low_quality_book(book) is True, book
    book["title"] = "A NoTe-BoOk Z"
    assert is_low_quality_book(book) is True, book

    book = {"title": "NOTEBOOK", "authors": [{"name": "pickleball publishing"}]}
    assert is_low_quality_book(book) is True, book
    book["authors"] = [
        {"name": "hol"},
        {"name": "mad"},
        {"name": "mazz"},
        {"name": "mikemi"},
        {"name": "tobias publishers"},
    ]
    assert is_low_quality_book(book) is False, book
    book["authors"] = [
        {"name": "razal"},
        {"name": "tobias publishing"},
        {"name": "koraya"},
        {"name": "pickleball"},
        {"name": "d"},
    ]
    assert is_low_quality_book(book) is True, book

    book = {
        "title": "A aNNotaTEd Z",
        "publishers": ["Independently Published"],
        "publish_date": "2017-09-01T05:14:17",
    }
    assert is_low_quality_book(book) is False, book
    book["publish_date"] = "2018"
    assert is_low_quality_book(book) is True, book
    book["publishers"] = ["Independently Publish"]
    assert is_low_quality_book(book) is False, book
    book["publishers"] += ["Independently Published"]
    assert is_low_quality_book(book) is True, book
    book["title"] = "A aNNotaTE Z"
    assert is_low_quality_book(book) is False, book

    assert is_low_quality_book(
        {
            'title': 'A tale of two cities (annotated)',
            'publish_date': '2020',
            'publishers': ['Independently Published'],
        }
    )


def test_is_published_in_future_year() -> None:
    last_year = str(datetime.now().year - 1)
    last_year_book = {'publish_date': last_year}
    assert is_published_in_future_year(last_year_book) is False

    this_year = str(datetime.now().year)
    this_year_book = {'publish_date': this_year}
    assert is_published_in_future_year(this_year_book) is False

    next_year = str(datetime.now().year + 1)
    next_year_book = {'publish_date': next_year}
    assert is_published_in_future_year(next_year_book) is True

    # No publication year
    no_year_book = {'publish_date': '0'}
    assert is_published_in_future_year(no_year_book) is False
