import json
from unittest.mock import MagicMock

import httpx
from httpx import Response, ConnectError

from openlibrary.solr.utils import SolrUpdateRequest, solr_update


class TestSolrUpdate:
    def sample_response_200(self):
        return Response(
            200,
            request=MagicMock(),
            content=json.dumps(
                {
                    "responseHeader": {
                        "errors": [],
                        "maxErrors": -1,
                        "status": 0,
                        "QTime": 183,
                    }
                }
            ),
        )

    def sample_global_error(self):
        return Response(
            400,
            request=MagicMock(),
            content=json.dumps(
                {
                    'responseHeader': {
                        'errors': [],
                        'maxErrors': -1,
                        'status': 400,
                        'QTime': 76,
                    },
                    'error': {
                        'metadata': [
                            'error-class',
                            'org.apache.solr.common.SolrException',
                            'root-error-class',
                            'org.apache.solr.common.SolrException',
                        ],
                        'msg': "Unknown key 'key' at [14]",
                        'code': 400,
                    },
                }
            ),
        )

    def sample_individual_error(self):
        return Response(
            400,
            request=MagicMock(),
            content=json.dumps(
                {
                    'responseHeader': {
                        'errors': [
                            {
                                'type': 'ADD',
                                'id': '/books/OL1M',
                                'message': '[doc=/books/OL1M] missing required field: type',
                            }
                        ],
                        'maxErrors': -1,
                        'status': 0,
                        'QTime': 10,
                    }
                }
            ),
        )

    def sample_response_503(self):
        return Response(
            503,
            request=MagicMock(),
            content=b"<html><body><h1>503 Service Unavailable</h1>",
        )

    def test_successful_response(self, monkeypatch, monkeytime):
        mock_post = MagicMock(return_value=self.sample_response_200())
        monkeypatch.setattr(httpx, "post", mock_post)

        solr_update(
            SolrUpdateRequest(commit=True),
            solr_base_url="http://localhost:8983/solr/foobar",
        )

        assert mock_post.call_count == 1

    def test_non_json_solr_503(self, monkeypatch, monkeytime):
        mock_post = MagicMock(return_value=self.sample_response_503())
        monkeypatch.setattr(httpx, "post", mock_post)

        solr_update(
            SolrUpdateRequest(commit=True),
            solr_base_url="http://localhost:8983/solr/foobar",
        )

        assert mock_post.call_count > 1

    def test_solr_offline(self, monkeypatch, monkeytime):
        mock_post = MagicMock(side_effect=ConnectError('', request=None))
        monkeypatch.setattr(httpx, "post", mock_post)

        solr_update(
            SolrUpdateRequest(commit=True),
            solr_base_url="http://localhost:8983/solr/foobar",
        )

        assert mock_post.call_count > 1

    def test_invalid_solr_request(self, monkeypatch, monkeytime):
        mock_post = MagicMock(return_value=self.sample_global_error())
        monkeypatch.setattr(httpx, "post", mock_post)

        solr_update(
            SolrUpdateRequest(commit=True),
            solr_base_url="http://localhost:8983/solr/foobar",
        )

        assert mock_post.call_count == 1

    def test_bad_apple_in_solr_request(self, monkeypatch, monkeytime):
        mock_post = MagicMock(return_value=self.sample_individual_error())
        monkeypatch.setattr(httpx, "post", mock_post)

        solr_update(
            SolrUpdateRequest(commit=True),
            solr_base_url="http://localhost:8983/solr/foobar",
        )

        assert mock_post.call_count == 1

    def test_other_non_ok_status(self, monkeypatch, monkeytime):
        mock_post = MagicMock(
            return_value=Response(500, request=MagicMock(), content="{}")
        )
        monkeypatch.setattr(httpx, "post", mock_post)

        solr_update(
            SolrUpdateRequest(commit=True),
            solr_base_url="http://localhost:8983/solr/foobar",
        )

        assert mock_post.call_count > 1
