import datetime
import web


class TestMockSite:
    def test_new_key(self, mock_site):
        ekey = mock_site.new_key('/type/edition')
        assert ekey == '/books/OL1M'
        ekey = mock_site.new_key('/type/edition')
        assert ekey == '/books/OL2M'

        wkey = mock_site.new_key('/type/work')
        assert wkey == '/works/OL1W'
        wkey = mock_site.new_key('/type/work')
        assert wkey == '/works/OL2W'

        akey = mock_site.new_key('/type/author')
        assert akey == '/authors/OL1A'
        akey = mock_site.new_key('/type/author')
        assert akey == '/authors/OL2A'

    def test_get(self, mock_site):
        doc = {
            "key": "/books/OL1M",
            "type": {"key": "/type/edition"},
            "title": "The Test Book",
        }
        timestamp = datetime.datetime(2010, 1, 2, 3, 4, 5)

        mock_site.save(doc, timestamp=timestamp)

        assert mock_site.get("/books/OL1M").dict() == {
            "key": "/books/OL1M",
            "type": {"key": "/type/edition"},
            "title": "The Test Book",
            "revision": 1,
            "latest_revision": 1,
            "last_modified": {"type": "/type/datetime", "value": "2010-01-02T03:04:05"},
            "created": {"type": "/type/datetime", "value": "2010-01-02T03:04:05"},
        }
        assert mock_site.get("/books/OL1M").__class__.__name__ == "Edition"

    def test_query(self, mock_site):
        doc = {
            "key": "/books/OL1M",
            "type": {"key": "/type/edition"},
            "title": "The Test Book",
            "subjects": ["love", "san_francisco"],
            "isbn_10": ["0123456789"],
            "isbn_13": ["0123456789abc"],
        }
        timestamp = datetime.datetime(2010, 1, 2, 3, 4, 5)

        mock_site.reset()
        mock_site.save(doc, timestamp=timestamp)

        assert mock_site.things({"type": "/type/edition"}) == ["/books/OL1M"]
        assert mock_site.things({"type": "/type/work"}) == []

        assert mock_site.things({"type": "/type/edition", "subjects": "love"}) == [
            "/books/OL1M"
        ]
        assert mock_site.things({"type": "/type/edition", "subjects": "hate"}) == []

        assert mock_site.things({"key~": "/books/*"}) == ["/books/OL1M"]
        assert mock_site.things({"key~": "/works/*"}) == []

        assert mock_site.things({"last_modified>": "2010-01-01"}) == ["/books/OL1M"]
        assert mock_site.things({"last_modified>": "2010-01-03"}) == []

        assert mock_site.things({"isbn_10": ["nomatch", "0123456789"]}) == [
            "/books/OL1M"
        ]
        assert mock_site.things({"isbn_10": "0123456789"}) == ["/books/OL1M"]
        assert mock_site.things({"isbn_": "0123456789"}) == ["/books/OL1M"]
        assert mock_site.things({"isbn_": ["0123456789abc"]}) == ["/books/OL1M"]

    def test_work_authors(self, mock_site):
        a2 = mock_site.quicksave("/authors/OL2A", "/type/author", name="A2")
        work = mock_site.quicksave(
            "/works/OL1W",
            "/type/work",
            title="Foo",
            authors=[{"author": {"key": "/authors/OL2A"}}],
        )
        book = mock_site.quicksave(
            "/books/OL1M", "/type/edition", title="Foo", works=[{"key": "/works/OL1W"}]
        )

        w = book.works[0]

        assert w.dict() == work.dict()

        a = w.authors[0].author
        assert a.dict() == a2.dict()

        assert a.key == '/authors/OL2A'
        assert a.type.key == '/type/author'
        assert a.name == 'A2'

        assert [a.type.key for a in work.get_authors()] == ['/type/author']
        assert [a.type.key for a in work.get_authors()] == ['/type/author']

        # this is the query format used in openlibrary/openlibrary/catalog/works/find_works.py get_existing_works(akey)
        # and https://github.com/internetarchive/openlibrary/blob/dabd7b8c0c42e3ac2700779da9f303a6344073f6/openlibrary/plugins/openlibrary/api.py#L228
        author_works_q = {'type': '/type/work', 'authors': {'author': {'key': a.key}}}
        assert mock_site.things(author_works_q) == ['/works/OL1W']

    def test_ilike_compatibility(self, mock_site) -> None:
        name = "B. Baggins"
        mock_site.quicksave("/authors/OL1A", "/type/author", name="B. R. Mc Baggins")
        mock_site.quicksave("/authors/OL2A", "/type/author", name="B* Baggins")
        mock_site.quicksave("/authors/OL3A", "/type/author", name=".B. Baggins")
        mock_site.quicksave("/authors/OL4A", "/type/author", name="b_ BaGGinS")
        mock_site.quicksave("/authors/OL5A", "/type/author", name="b. BaGGinS")
        query = {"type": "/type/author", "name~": name}

        reply = web.ctx.site.things(query)
        assert reply == ["/authors/OL5A"]
