import json
from os import system
from os.path import abspath, dirname, join, pardir

import pytest
import web
import urllib

from openlibrary.coverstore import archive, code, config, coverlib, schema, utils

static_dir = abspath(join(dirname(__file__), pardir, pardir, pardir, 'static'))


@pytest.fixture(scope='module')
def setup_db():
    """These tests have to run as the openlibrary user."""
    system('dropdb coverstore_test')
    system('createdb coverstore_test')
    config.db_parameters = {
        'dbn': 'postgres',
        'db': 'coverstore_test',
        'user': 'openlibrary',
        'pw': '',
    }
    db_schema = schema.get_schema('postgres')
    db = web.database(**config.db_parameters)
    db.query(db_schema)
    db.insert('category', name='b')


@pytest.fixture
def image_dir(tmpdir):
    tmpdir.mkdir('localdisk')
    tmpdir.mkdir('items')
    config.data_root = str(tmpdir)


class Mock:
    def __init__(self):
        self.calls = []
        self.default = None

    def __call__(self, *a, **kw):
        for a2, kw2, _return in self.calls:
            if (a, kw) == (a2, kw2):
                return _return
        return self.default

    def setup_call(self, *a, **kw):
        _return = kw.pop("_return", None)
        call = a, kw, _return
        self.calls.append(call)


class WebTestCase:
    def setup_method(self, method):
        self.browser = code.app.browser()

    def jsonget(self, path):
        self.browser.open(path)
        return json.loads(self.browser.data)

    def upload(self, olid, path):
        """Uploads an image in static dir"""
        b = self.browser

        path = join(static_dir, path)
        content_type, data = utils.urlencode({'olid': olid, 'data': open(path).read()})
        b.open('/b/upload2', data, {'Content-Type': content_type})
        return json.loads(b.data)['id']

    def delete(self, id, redirect_url=None):
        b = self.browser

        params = {'id': id}
        if redirect_url:
            params['redirect_url'] = redirect_url
        b.open('/b/delete', urllib.parse.urlencode(params))
        return b.data

    def static_path(self, path):
        return join(static_dir, path)


@pytest.mark.skip(
    reason="Currently needs running db and openlibrary user. TODO: Make this more flexible."
)
class TestDB:
    def test_write(self, setup_db, image_dir):
        path = static_dir + '/logos/logo-en.png'
        data = open(path).read()
        d = coverlib.save_image(data, category='b', olid='OL1M')

        assert 'OL1M' in d.filename
        path = config.data_root + '/localdisk/' + d.filename
        assert open(path).read() == data


class TestWebapp(WebTestCase):
    def test_get(self):
        assert code.app.request('/').status == "200 OK"


@pytest.mark.skip(
    reason="Currently needs running db and openlibrary user. TODO: Make this more flexible."
)
class TestWebappWithDB(WebTestCase):
    def test_touch(self):
        pytest.skip('TODO: touch is no more used. Remove or fix this test later.')

        b = self.browser

        id1 = self.upload('OL1M', 'logos/logo-en.png')
        id2 = self.upload('OL1M', 'logos/logo-it.png')

        assert id1 < id2
        assert (
            b.open('/b/olid/OL1M.jpg').read()
            == open(static_dir + '/logos/logo-it.png').read()
        )

        b.open('/b/touch', urllib.parse.urlencode({'id': id1}))
        assert (
            b.open('/b/olid/OL1M.jpg').read()
            == open(static_dir + '/logos/logo-en.png').read()
        )

    def test_delete(self, setup_db):
        b = self.browser

        id1 = self.upload('OL1M', 'logos/logo-en.png')
        data = self.delete(id1)

        assert data == 'cover has been deleted successfully.'

    def test_upload(self):
        b = self.browser

        path = join(static_dir, 'logos/logo-en.png')
        filedata = open(path).read()
        content_type, data = utils.urlencode({'olid': 'OL1234M', 'data': filedata})
        b.open('/b/upload2', data, {'Content-Type': content_type})
        assert b.status == 200
        id = json.loads(b.data)['id']

        self.verify_upload(id, filedata, {'olid': 'OL1234M'})

    def test_upload_with_url(self, monkeypatch):
        b = self.browser
        filedata = open(join(static_dir, 'logos/logo-en.png')).read()
        source_url = 'http://example.com/bookcovers/1.jpg'

        mock = Mock()
        mock.setup_call(source_url, _return=filedata)
        monkeypatch.setattr(code, 'download', mock)

        content_type, data = utils.urlencode(
            {'olid': 'OL1234M', 'source_url': source_url}
        )
        b.open('/b/upload2', data, {'Content-Type': content_type})
        assert b.status == 200
        id = json.loads(b.data)['id']

        self.verify_upload(id, filedata, {'source_url': source_url, 'olid': 'OL1234M'})

    def verify_upload(self, id, data, expected_info=None):
        expected_info = expected_info or {}
        b = self.browser
        b.open('/b/id/%d.json' % id)
        info = json.loads(b.data)
        for k, v in expected_info.items():
            assert info[k] == v

        response = b.open('/b/id/%d.jpg' % id)
        assert b.status == 200
        assert response.info().getheader('Content-Type') == 'image/jpeg'
        assert b.data == data

        b.open('/b/id/%d-S.jpg' % id)
        assert b.status == 200

        b.open('/b/id/%d-M.jpg' % id)
        assert b.status == 200

        b.open('/b/id/%d-L.jpg' % id)
        assert b.status == 200

    def test_archive_status(self):
        id = self.upload('OL1M', 'logos/logo-en.png')
        d = self.jsonget('/b/id/%d.json' % id)
        assert d['archived'] is False
        assert d['deleted'] is False

    def test_archive(self):
        b = self.browser

        f1 = web.storage(olid='OL1M', filename='logos/logo-en.png')
        f2 = web.storage(olid='OL2M', filename='logos/logo-it.png')
        files = [f1, f2]

        for f in files:
            f.id = self.upload(f.olid, f.filename)
            f.path = join(static_dir, f.filename)
            assert b.open('/b/id/%d.jpg' % f.id).read() == open(f.path).read()

        archive.archive()

        for f in files:
            d = self.jsonget('/b/id/%d.json' % f.id)
            assert 'tar:' in d['filename']
            assert b.open('/b/id/%d.jpg' % f.id).read() == open(f.path).read()
