# Generated by Django 2.2.13 on 2020-06-10 08:56
from logging import getLogger
from django.utils import timezone

from django.db import migrations
from django.conf import settings
from camac.constants import kt_bern as constants
from caluma.caluma_user.models import BaseUser
from caluma.caluma_workflow.utils import get_jexl_groups
from caluma.caluma_workflow import models as workflow_models

log = getLogger()


INSTANCE_STATES = [
    constants.INSTANCE_STATE_NEW,
    # submit complete
    constants.INSTANCE_STATE_EBAU_NUMMER_VERGEBEN,
    constants.INSTANCE_STATE_DOSSIERPRUEFUNG,
    constants.INSTANCE_STATE_KOORDINATION,
    constants.INSTANCE_STATE_VERFAHRENSPROGRAMM_INIT,
    constants.INSTANCE_STATE_ZIRKULATION,
    constants.INSTANCE_STATE_CORRECTION_IN_PROGRESS,
    constants.INSTANCE_STATE_CORRECTED,
    constants.INSTANCE_STATE_FINISHED,
    # will be canceled later
    constants.INSTANCE_STATE_REJECTED,
    # nfd and ebau-number complete
    constants.INSTANCE_STATE_SB1,
    # sb1 complete
    constants.INSTANCE_STATE_SB2,
    # sb2 complete, case complete
    constants.INSTANCE_STATE_TO_BE_FINISHED,
    constants.INSTANCE_STATE_ARCHIVED,
    constants.INSTANCE_STATE_DONE,
]


def has_min_instance_state(current_instance_state, min_instance_state):
    return INSTANCE_STATES.index(current_instance_state) >= INSTANCE_STATES.index(
        min_instance_state
    )


def is_vorabklaerung(case):
    return case.workflow.slug == "preliminary-clarification"


def link_work_item_to_document(work_item, document):
    work_item.document = document
    work_item.save()

    document.meta = {}
    document.save()


def skip_work_item(work_item):
    work_item.status = workflow_models.WorkItem.STATUS_SKIPPED
    work_item.closed_at = timezone.now()
    work_item.save()


def skip_submit(case, documents, apps):
    Task = apps.get_model("caluma_workflow", "Task")
    WorkItem = apps.get_model("caluma_workflow", "WorkItem")

    skip_work_item(case.work_items.get(task_id="submit"))

    for task in Task.objects.filter(pk__in=["ebau-number", "nfd"]):
        WorkItem.objects.create(
            case=case,
            task=task,
            meta={"migrated": True},
            name=task.name,
            addressed_groups=get_jexl_groups(
                task.address_groups, task, case, BaseUser(), None
            ),
            controlling_groups=get_jexl_groups(
                task.control_groups, task, case, BaseUser(), None
            ),
            deadline=None,
            status=workflow_models.WorkItem.STATUS_READY,
        )

    link_work_item_to_document(
        case.work_items.get(task_id="nfd"), documents.get(form_id="nfd")
    )


def skip_ebau_nr_and_nfd(case, documents, apps):
    Task = apps.get_model("caluma_workflow", "Task")
    WorkItem = apps.get_model("caluma_workflow", "WorkItem")

    for work_item in case.work_items.filter(task_id__in=["ebau-number", "nfd"]):
        skip_work_item(work_item)

    if is_vorabklaerung(case):
        return

    task = Task.objects.get(pk="sb1")
    sb1_work_item = WorkItem.objects.create(
        case=case,
        task=task,
        meta={"migrated": True},
        name=task.name,
        addressed_groups=get_jexl_groups(
            task.address_groups, task, case, BaseUser(), None
        ),
        controlling_groups=get_jexl_groups(
            task.control_groups, task, case, BaseUser(), None
        ),
        deadline=None,
        status=workflow_models.WorkItem.STATUS_READY,
    )

    link_work_item_to_document(sb1_work_item, documents.get(form_id="sb1"))


def skip_sb1(case, documents, apps):
    if is_vorabklaerung(case):
        return

    Task = apps.get_model("caluma_workflow", "Task")
    WorkItem = apps.get_model("caluma_workflow", "WorkItem")

    skip_work_item(case.work_items.get(task_id="sb1"))

    task = Task.objects.get(pk="sb2")
    sb2_work_item = WorkItem.objects.create(
        case=case,
        task=task,
        meta={"migrated": True},
        name=task.name,
        addressed_groups=get_jexl_groups(
            task.address_groups, task, case, BaseUser(), None
        ),
        controlling_groups=get_jexl_groups(
            task.control_groups, task, case, BaseUser(), None
        ),
        deadline=None,
        status=workflow_models.WorkItem.STATUS_READY,
    )

    link_work_item_to_document(sb2_work_item, documents.get(form_id="sb2"))


def skip_sb2(case, documents, apps):
    if is_vorabklaerung(case):
        return

    skip_work_item(case.work_items.get(task_id="sb2"))


def document_to_case(apps, schema_editor):
    if settings.APPLICATION_NAME != "kt_bern":
        return

    Instance = apps.get_model("instance", "Instance")
    Document = apps.get_model("caluma_form", "Document")
    Case = apps.get_model("caluma_workflow", "Case")
    Workflow = apps.get_model("caluma_workflow", "Workflow")
    Task = apps.get_model("caluma_workflow", "Task")
    WorkItem = apps.get_model("caluma_workflow", "WorkItem")

    failed_instances = []

    for instance in Instance.objects.all():
        try:
            if Case.objects.filter(**{"meta__camac-instance-id": instance.pk}).exists():
                # case is already migrated, skip
                continue

            try:
                documents = Document.objects.filter(
                    **{"meta__camac-instance-id": instance.pk}
                )
                main_document = documents.get(**{"form__meta__is-main-form": True})
            except Document.DoesNotExist:
                failed_instances.append(instance.pk)

                log.error(f"Main document not found for instance {instance.pk}")
                continue

            workflow = Workflow.objects.get(
                pk=(
                    "preliminary-clarification"
                    if "vorabklaerung" in main_document.form.slug
                    else "building-permit"
                )
            )

            case = Case.objects.create(
                status=workflow_models.Case.STATUS_RUNNING,
                workflow=workflow,
                document=main_document,
                meta={**main_document.meta, **{"migrated": True}},
                created_at=main_document.created_at,
                modified_at=main_document.modified_at,
                created_by_user=main_document.created_by_user,
                created_by_group=main_document.created_by_group,
            )

            case.family = case
            case.save()

            submit_task = Task.objects.get(pk="submit")
            submit_work_item = WorkItem.objects.create(
                case=case,
                task=submit_task,
                addressed_groups=get_jexl_groups(
                    submit_task.address_groups, submit_task, case, BaseUser(), None
                ),
                controlling_groups=get_jexl_groups(
                    submit_task.control_groups, submit_task, case, BaseUser(), None
                ),
                deadline=None,
                meta={"migrated": True},
                name=submit_task.name,
                status=workflow_models.WorkItem.STATUS_READY,
                created_at=main_document.created_at,
                modified_at=main_document.modified_at,
                created_by_user=main_document.created_by_user,
                created_by_group=main_document.created_by_group,
            )

            if instance.instance_state.pk == constants.INSTANCE_STATE_ARCHIVED:
                submit_work_item.closed_at = timezone.now()
                submit_work_item.status = workflow_models.WorkItem.STATUS_CANCELED
                submit_work_item.save()

                case.status = workflow_models.Case.STATUS_CANCELED
                case.save()

                main_document.meta = {}
                main_document.save()

                continue

            if has_min_instance_state(
                instance.instance_state.pk,
                constants.INSTANCE_STATE_EBAU_NUMMER_VERGEBEN,
            ):
                skip_submit(case, documents, apps)

            if has_min_instance_state(
                instance.instance_state.pk, constants.INSTANCE_STATE_SB1
            ):
                skip_ebau_nr_and_nfd(case, documents, apps)

            if has_min_instance_state(
                instance.instance_state.pk, constants.INSTANCE_STATE_SB2
            ):
                skip_sb1(case, documents, apps)

            if has_min_instance_state(
                instance.instance_state.pk, constants.INSTANCE_STATE_TO_BE_FINISHED
            ):
                skip_sb2(case, documents, apps)

            if instance.instance_state.pk == constants.INSTANCE_STATE_DONE:
                # instance was rejected but a new instance was submitted which
                # sets the old instance to done. In this case the old case must
                # be cancelled. However, since we have no way to know at which
                # step the instance was rejected, so we asume it was right
                # after submitting
                if (
                    instance.previous_instance_state.pk
                    == constants.INSTANCE_STATE_REJECTED
                ):
                    # delete sb1 and sb2
                    case.work_items.exclude(task_id__in=["sb1", "sb2"]).delete()
                    # cancel nfd and ebau-number
                    case.work_items.filter(task_id__in=["nfd", "ebau-number"]).update(
                        closed_at=timezone.now(),
                        status=workflow_models.WorkItem.STATUS_CANCELED,
                    )

                    case.status = workflow_models.Case.STATUS_CANCELED
                else:
                    case.status = workflow_models.Case.STATUS_COMPLETED

                case.save()

            main_document.meta = {}
            main_document.save()

        except Exception as e:
            if case:
                case.delete()

            failed_instances.append(instance.pk)
            log.error(f"Error while migrating instance {instance.pk}: {e}")

    # cleanup all unused nfd, sb1 and sb2 documents
    obsolete_documents = Document.objects.filter(
        **{"meta__camac-instance-id__isnull": False}
    ).exclude(**{"meta__camac-instance-id__in": failed_instances})

    log.warning(f"Deleted {obsolete_documents.count()} obsolete documents")
    obsolete_documents.delete()

    if len(failed_instances):
        log.error(
            f"The following instances could not be migrated: {', '.join(list(map(str, failed_instances)))}."
        )


class Migration(migrations.Migration):

    dependencies = [
        ("instance", "0017_fix_submit_date"),
        ("caluma_workflow", "0027_add_modified_by_user_group"),
    ]

    operations = [
        migrations.RunSQL(
            sql="""DROP INDEX "document_instance_id";""",
            reverse_sql="""CREATE INDEX "document_instance_id" ON caluma_form_document((meta->'camac-instance-id'));""",
        ),
        migrations.RunSQL(
            sql="""CREATE INDEX "case_instance_id" ON caluma_workflow_case((meta->'camac-instance-id'));""",
            reverse_sql="""DROP INDEX "case_instance_id";""",
        ),
        migrations.RunPython(document_to_case, migrations.RunPython.noop),
    ]
