from collections import namedtuple

from caluma.caluma_user.models import AnonymousUser, OIDCUser
from caluma.caluma_user.views import AuthenticationGraphQLView, HttpResponseUnauthorized
from django.conf import settings
from django.http.response import HttpResponse
from graphene_django.views import HttpError
from rest_framework import status

from camac.caluma.utils import CamacRequest, extend_user
from camac.token_exchange.permissions import has_required_lot
from camac.user.models import User


class CamacAuthenticatedGraphQLView(AuthenticationGraphQLView):
    def get_user(self, request, *args, **kwargs):
        oidc_user = super().get_user(request, *args, **kwargs)

        if not isinstance(oidc_user, OIDCUser):
            if settings.APPLICATION.get("ENABLE_PUBLIC_CALUMA"):
                return AnonymousUser()
            else:
                # Raise a 401 error if the user is anything else than an OIDCUser
                # (e.g None, AnonymousUser)
                raise HttpError(HttpResponseUnauthorized())

        try:
            # Get the camac request containing the camac user and group
            request.user = oidc_user
            Info = namedtuple("Info", "context")
            request.camac_request = CamacRequest(Info(context=request)).request

            extend_user(oidc_user, request.camac_request)

            # Set the camac_user property on the caluma request
            request.camac_user = request.camac_request.user
        except User.DoesNotExist:
            # Raise a 401 error if the user was not found in the CAMAC database
            raise HttpError(HttpResponseUnauthorized())

        if not has_required_lot(request.camac_request):
            raise HttpError(HttpResponse(status=status.HTTP_403_FORBIDDEN))

        return oidc_user
