from functools import partial
from itertools import cycle
from typing import Dict, List, Optional, Union

from torch.utils.data import DataLoader
from transformers import PreTrainedModel, set_seed

from mutate_nlp.infer import TextGeneration
from mutate_nlp.parsers.text_classification import TextClassificationSynthesizeParser
from mutate_nlp.prompt_datasets.text_classification import (
    TextClassSynthesizePromptDataset,
)


class TextClassificationSynthesize:
    generate_kwargs = {
        "max_length": 300,
        "do_sample": True,
        "num_return_sequences": 3,
        "top_k": 40,
        "top_p": 0.80,
        "early_stopping": True,
        "no_repeat_ngram_size": 2,
    }

    def __init__(
        self,
        model: Union[str, PreTrainedModel],
        device: Optional[int] = -1,
        **generate_kwargs
    ):
        """
        Pipeline to synthesize Text classification examples from a given dataset.

        Pipelines are made of:
            - A `Prompt Dataset` - that converts a given dataset to suitable prompts
            - Model inference - to generate text from the prompt
            - A `Parser` to parse the generated texts for examples

        Args:
            model (Union[str, PreTrainedModel]): Path or 🤗 model hub identifier. Currently supports Causal LM models only.
            device (Optional[int], optional): GPU Device to run the inference. Defaults to -1, runs on CPU.
            generate_kwargs : Keyword arguments to override default generation params. This will be passed as generation params to model.generate -
            https://github.com/huggingface/transformers/blob/fcb4f11c9232cee2adce8140a3a7689578ea97de/src/transformers/generation_utils.py#L803

        Examples:
        --------

        Using the default generation params:

        >> pipe = TextClassificationSynthesize(
                model="EleutherAI/gpt-neo-2.7B",
                device=1,
            )

        Overriding the generation params:

        >> pipe = TextClassificationSynthesize(
                model="EleutherAI/gpt-neo-2.7B",
                device=1,
                max_length=300,
                do_sample=True,
                num_return_sequences=3,
                top_k=40,
                top_p=0.80,
                early_stopping=True,
                no_repeat_ngram_size=2,
            )

        """
        self.infer = TextGeneration(model_name=model, device=device)
        self._collate_fn = partial(
            TextClassSynthesizePromptDataset._collate_fn,
            self.infer.tokenizer,
            self.infer.device,
        )
        self.generate_kwargs = (
            self.generate_kwargs if not generate_kwargs else generate_kwargs
        )

    def __call__(
        self,
        dataset_path: str,
        text_column: str,
        label_column: str,
        task_desc: str,
        split: Optional[str] = None,
        data_files: Union[Dict, List] = None,
        data_dir: Optional[str] = None,
        text_column_alias: Optional[str] = None,
        label_column_alias: Optional[str] = None,
        class_names: Optional[List[str]] = [],
        dataset_args: Optional[List[str]] = [],
        dataset_kwargs: Optional[Dict[str, str]] = {},
        batch_size: Optional[int] = 2,
        shot_count: Optional[int] = 5,
        infinite_loop: Optional[bool] = False,
        **kwargs
    ):
        """
        Args:
            dataset_path (str): Path or name of the dataset.
                        Can be Huggingface dataset identifier like - `allenai/c4` or
                        For local datasets in CSV,JSON - this param can take `csv`,`json` as values
                        and corresponding path can be sepecified in the `data_files` param.
            task_desc (str): Description about the text classification task. A task description will improve
                             model generation quality. This is a good source about helpful task descriptions -
                             https://docs.cohere.ai/prompt-engineering-wiki
            text_column (str): Name of the column in the dataset which has examples to be used in the prompt and similar
                               examples are to be generated by the model
            label_column (str): Name of the column that contains the label / classes
            split (Optional[str]): Which Split of the dataset to load. Usage is same as in 🤗 Datasets. Defaults to `train` split.
            data_files (Union[Dict, List], optional): Paths to source data files .  Usage is same as in 🤗 Datasets.  Defaults to None.
            data_dir (Optional[str], optional): Directory of the dataset configuration.  Usage is same as in 🤗 Datasets.  Defaults to None.
            text_column_alias (Optional[str]):alias of `text_column`.If the `text_column` does not have a meaningful title. A meaningful title of the example can
                                          be specified here. Defaults to None.
            label_column_alias (Optional[str], optional): alias of `label_column`.If the `label_column` does not have a meaningful title. A meaningful title of the class can
                                          be specified here. Defaults to None.
            dataset_args (Optional[List[str]]): additional args to load the  🤗 Dataset. Defaults to [].
            dataset_kwargs (Optional[Dict[str,str]]): additional keyword arguments to load the 🤗 Dataset. Defaults to {}.
            class_names (Optional[List[str]], optional): Class names to be specified.
                If class names are not part of huggingface datasets features.if values in the label columns are label encoded.
                class name order is expected to follow the same order as the label encoding.Defaults to None.
            batch_size (Optional[Int]): Batch size for inference. Defaults to 2.
            shot_count (Optional[int], optional): Number of examples to be used in the few shot prompt. Defaults to 5.
            infinite_loop (Optional[bool]): if `True`, loops through the same dataset indefinetly. Can help generate more examples.
                                            defaults to `False`

        Yields:
            [Dict[str,str]]: yields the generated examples and corresponding label as dictionary

        Examples:
        --------

        Using 🤗 Datasets:

        >> pipe = TextClassificationSynthesize(
                model="EleutherAI/gpt-neo-2.7B",
                device=1,
            )
        >> task_dec = "Each item in the following contains customer service queries expressing the mentioned intent"
        >> synthesizerGen = pipe(
                        "banking77",
                        task_desc=task_dec,
                        text_column="text",
                        label_column="label",
                        text_column_alias="Queries", # since the `text_column` doesn't have a meaningful value
                        label_column_alias="Intent", # since the `label_column` doesn't have a meaningful value
                        dataset_args=["en"],
                        )

        >> for exp in synthesizerGen:
                print(exp)


        Using local csv files

        >> task_dec = "Each item in the following contains customer comments expressing the mentioned sentiment"
        >> pipe(
                "csv",
                data_files=["local/path/sentiment_classfication.csv"],
                task_desc=task_dec,
                text_column="text",
                label_column="label",
                text_column_alias="Comment",
                label_column_alias="sentiment",
                class_names=["positive","negative","neutral"],
                dataset_args=["en"],
            )
        """

        dataset = TextClassSynthesizePromptDataset(
            path=dataset_path,
            data_files=data_files,
            data_dir=data_dir,
            split=split,
            task_desc=task_desc,
            text_column=text_column,
            text_column_alias=text_column_alias,
            label_column=label_column,
            label_column_alias=label_column_alias,
            class_names=class_names,
            shot_count=shot_count,
            dataset_args=dataset_args,
            dataset_kwargs=dataset_kwargs,
        )

        parser = TextClassificationSynthesizeParser(
            label_title=dataset.label_title, example_title=dataset.example_title
        )

        prompt_dataloader = DataLoader(
            dataset, batch_size=batch_size, collate_fn=self._collate_fn
        )

        if infinite_loop:
            prompt_dataloader = cycle(prompt_dataloader)

        for batch in prompt_dataloader:
            batch_parsed_examples = []
            batch_class_names = []
            batch_generated_texts = self.infer.run_single_batch(
                batch, generate_args=self.generate_kwargs
            )

            num_return_sequences = len(batch_generated_texts) // len(batch["input_ids"])

            for idx, generated_text in enumerate(batch_generated_texts):
                class_idx = idx // num_return_sequences
                class_name = batch["class_names"][class_idx]

                parsed_examples = parser.parse(
                    generated_text=generated_text, label=class_name
                )

                batch_parsed_examples.extend(parsed_examples)
                batch_class_names.extend([class_name] * len(parsed_examples))

            yield {text_column: batch_parsed_examples, label_column: batch_class_names}
