import { auth } from "@clerk/nextjs";
import { NextResponse } from "next/server";
import { Configuration, OpenAIApi } from "openai";

import { checkApiLimit, incrementApiLimit } from "@/lib/api-limit";
import { checkSubscription } from "@/lib/subscription";

const configuration = new Configuration({
  apiKey: process.env.OPENAI_API_KEY,
});

const openai = new OpenAIApi(configuration);

export async function POST(req: Request) {
  try {
    const { userId } = auth();
    const body = await req.json();
    const { prompt, amount = 1, resolution = "512x512" } = body;
    const isPro = await checkSubscription();

    if (!userId) return new NextResponse("Unauthorized", { status: 401 });

    if (!configuration.apiKey)
      return new NextResponse("API key now configured", { status: 500 });

    if (!prompt)
      return new NextResponse("Prompt is required.", { status: 400 });
    if (!amount)
      return new NextResponse("Amount is required.", { status: 400 });
    if (!resolution)
      return new NextResponse("Resolution is required.", { status: 400 });

    const freeTrial = await checkApiLimit();
    if (!freeTrial && !isPro)
      return new NextResponse("Free trial limit reached", { status: 402 });

    const response = await openai.createImage({
      prompt,
      n: parseInt(amount, 10),
      size: resolution,
    });
    
    if (!isPro) await incrementApiLimit();

    return NextResponse.json(response.data.data);
  } catch (e) {
    console.log(`[IMAGE_ERROR] ${e}`);
    return new NextResponse("Internal error", { status: 500 });
  }
}
