# Nano Bots

![Nano Bot](https://user-images.githubusercontent.com/113217272/237534390-3f0e0dc7-1f92-4490-a12d-bfc9d145cddb.png)
_Image artificially created by Midjourney through a prompt generated by a Nano Bot specialized in Midjourney._

**Nano Bots** is an open specification that can be implemented in any programming language. It specifies a configuration file with human-readable instructions for creating small and specialized AI-powered bots that can be effortlessly shared as a single, lightweight file.

Here's what a Nano Bot _Cartridge_ looks like:

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  version: 1.0.0
  license: CC0-1.0
  description: A helpful assistant.

behaviors:
  interaction:
    directive: You are a helpful assistant.

provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
```

Here's what a fully-functional implementation of Nano Bots feels like:

```bash
nb - - eval "hello"
# => Hello! How may I assist you today?

nb to-en-us-translator.yml - eval "Salut, comment ça va?"
# => Hello, how are you doing?

nb midjourney.yml - eval "happy cyberpunk robot"
# => A cheerful and fun-loving robot is dancing wildly amidst a
#    futuristic and lively cityscape. Holographic advertisements
#    and vibrant neon colors can be seen in the background.

nb lisp.yml - eval "(+ 1 2)"
# => 3

cat article.txt |
  nb to-en-us-translator.yml - eval |
  nb summarizer.yml - eval
# -> LLM stands for Large Language Model, which refers to an
#    artificial intelligence algorithm capable of processing
#    and understanding vast amounts of natural language data,
#    allowing it to generate human-like responses and perform
#    a range of language-related tasks.
```

```bash
nb - - repl

nb assistant.yml - repl
```

```text
🤖> Hi, how are you doing?

As an AI language model, I do not experience emotions but I am functioning
well. How can I assist you?

🤖> |
```

## Rationale

While Artificial Intelligence and Large Language Models continue to evolve with increasingly larger and more complex projects emerging, Nano Bots takes a different approach. It focuses on creating smaller, specialized bots inspired by the [Unix philosophy](https://en.wikipedia.org/wiki/Unix_philosophy), which emphasizes simplicity, modularity, and composability in software design:

- Simplicity: Each Nano Bot focuses on a single task or domain, resulting in more efficient and manageable bots.

- Modularity: Nano Bots are designed as independent and self-contained modules, each responsible for a specific functionality. This allows for easier development, testing, and maintenance.

- Composability: Inspired by the [Unix pipeline](https://en.wikipedia.org/wiki/Pipeline_(Unix)), Nano Bots can be combined and composed to create more complex AI workflows, allowing users to tailor AI solutions to their needs without relying on monolithic projects.

## Definition

A **Nano Bot** is a cohesive combination of a `cartridge` [YAML](https://yaml.org) file, an `implementation`, and a `provider`.

The `cartridge` YAML file contains human-readable data that outlines the bot's goals, expected behaviors, and settings for authentication and provider utilization.

The `implementation` processes the YAML _cartridge_ file, comprehending its instructions, and interacts with the `provider`'s AI Service to enable the bot to perform the expected behavior.

Together, these components create a fully functional **Nano Bot**:

```text
Nano Bot = (cartridge.yml + implementation + provider)
```

## Implementations

Implementations of this specification:

- [Nano Bots CLI (Ruby)](https://github.com/icebaker/ruby-nano-bots)

### Projects

- [Nano Bots CLI (Ruby)](https://github.com/icebaker/ruby-nano-bots)
- [Nano Bots for Sublime Text](https://github.com/icebaker/sublime-nano-bots)
- [Nano Bots for Visual Studio Code](https://github.com/icebaker/vscode-nano-bots)
- [Nano Bots for Obsidian](https://github.com/icebaker/obsidian-nano-bots)
- [Nano Bots API](https://github.com/icebaker/nano-bots-api)
  - [Public API](https://api.nbots.io)
- [Nano Bots Clinic (Live Editor)](https://clinic.nbots.io)
- [Nano Bots Marketplace](https://nbots.io)

# Cartridges

![Cartridges](https://user-images.githubusercontent.com/113217272/237534411-61cd7610-5a8c-4294-8840-559a1c58c603.png)
_Image artificially created by Midjourney through a prompt generated by a Nano Bot specialized in Midjourney._

A cartridge (inspired by _[Game ROM Cartridges](https://en.wikipedia.org/wiki/ROM_cartridge)_) should contain all the information required for a Nano Bot to be functional in any implementation of this specification.

Here is a minimal sample of a cartridge [YAML](https://yaml.org) file:

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  version: 1.0.0
  license: CC0-1.0
  description: A helpful assistant.

behaviors:
  interaction:
    directive: You are a helpful assistant.

provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
```

## Meta

The `meta:` section provides basic information about the Cartridge.

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  version: 1.0.0
  license: CC0-1.0
  description: A helpful assistant.
```

`symbol:` is a single [Unicode emoji character](https://home.unicode.org/emoji/about-emoji/).

`name:` and `author:` are single-line text values encouraged to be short.

`version:` must follow [Semantic Versioning 2.0.0](https://semver.org).

`license:` must comply with the [SPDX Identifier](https://spdx.dev/ids/).

`description:` is a brief text that details what to expect from the Nano Bot, encouraged to be concise.

### License

The classification of _prompts_ for Large Language Models as code, art, or intellectual property, as well as their applicability to licensing, intellectual property rights, copyright, or patenting, remain subjects of ongoing and complex debate without a clear answer or definitive understanding at present.

When Nano Bots expand prompt writing into a more detailed scheme and add small pieces of code like [adapters](?id=adapters), the debate becomes even trickier and blurrier.

Due to the complexities of this debate, we recommend that authors publish their creations under licenses.
In doubt, in the spirit of openness, sharing, and open source, we recommend the license [CC0-1.0](https://creativecommons.org/publicdomain/zero/1.0/) wich enables commercial exploration of the licensed works and the creation of derivative works without copyleft. However, any [SPDX-known license](https://spdx.org/licenses/) may be chosen.

Please note that we are uncertain if this approach is practical or enforceable, so it should not be considered a guarantee of licensing but rather a statement of your intention.

## Behaviors

The `behaviors:` section provides information to influence the expected behavior of the bot. There are three types of concepts that can influence a behavior:


- `directive` communicates about how the bot should behave and aligns with the concept of a rule or guideline.
- `backdrop` sets the stage and provides contextual information to help the bot understand the situation and respond accordingly.
- `instruction` implies a direct command about what the bot should do next, it represents a specific action that the bot is expected to perform.

Sample of a behaviors section:

```yaml
---
behaviors:
  interaction:
    directive: You are a helpful assistant.
    backdrop: |
      The Moon is Earth's natural satellite, orbiting our planet.
      The user might use the term "Selene" when referring to the Moon.
  boot:
    directive: You are a helpful assistant.
    instruction: Provide a welcome message.
```

- `interaction:` is used when users directly interact with the Nano Bot through _REPL_ or _Eval_.
- `boot:` is used when the Nano Bot starts in interfaces that support an initial boot message, like _REPL_.

### Directive

```yaml
---
behaviors:
  interaction:
    directive: You are a helpful assistant.
```

A `directive` serves as a guideline for how the bot is expected to behave, also known as _system messages_. Some non-exhaustive examples of directives include:

```text
You are a helpful assistant.
```

```text
You are an AI assistant that responds strictly in JSON format for all answers.
```

```text
You are a helpful assistant that specializes in providing information about historical events.
```

```text
You are a programming tutor, helping users learn and troubleshoot code in various programming languages.
```

```text
You are an AI personality emulator that responds to all questions in the style of Shakespeare.
```

### Backdrop

```yaml
---
behaviors:
  interaction:
    backdrop: |
      The Moon is Earth's natural satellite, orbiting our planet.
      The user might use the term "Selene" when referring to the Moon.
```

Backdrop provides contextual information to help the bot understand the situation and respond appropriately. This may involve sharing examples of expected responses, supplying information the bot needs to learn before answering, among other strategies.

### Instruction

```yaml
---
behaviors:
  boot:
    instruction: Provide a welcome message.
```

An `instruction` is a clear and concise directive given to the bot, intended to guide it in performing a particular action or task.

## Safety

The `safety:` section provides safety configurations:

```yaml
---
safety:
  functions:
    sandboxed: true
  tools:
    confirmable: true
```

### Functions

The key `sandboxed:` within `functions:` defines that any function executed by the Nano Bot, whether it's an _adapter_ or a _tool_, should be _sandboxed_. This implies that it is expected to have limited functionality and operate within an isolated environment. Consequently, it should not be capable of performing actions such as accessing the internet, manipulating disk files, or accessing the user's computer operating system and environment. When set to `false`, functions may potentially have unrestricted access to all of these capabilities.

```yaml
---
safety:
  functions:
    sandboxed: true
```

### Tools

The key `confirmable:` within `tools:` specifies that any tool (function) execution request that the Nano Bot wants to run should not be executed until the user confirms that they allow the function to be executed. When set to `false`, the Nano Bot should freely run any available tool without confirmation.

```yaml
---
safety:
  tools:
    confirmable: true
```

## Interfaces

Implementations should support two possible interaction interfaces: REPL and Eval.

You can customize both the _input_ and _output_ with prefixes, suffixes, colors, and adapters for all interfaces. Additionally, the feedback loop involved in the execution of tools (functions) can be customized.

```yaml
---
interfaces:
  input:
    prefix: "\n"
    suffix: "\n"
    adapter:
      fennel: |
        (.. "```" content "```")
      lua: |
        return "```" .. content .. "```"
  output:
    stream: true
    prefix: "\n"
    suffix: "\n"
    color: cyan
    adapter:
      fennel: |
        (.. "```" content "```")
      lua: |
        return "```" .. content .. "```"
  tools:
    confirming:
      yeses: ['y', 'yes']
      default: 'n'
      prefix: "\n"
      suffix: " [yN] > "
      color: orangered
      adapter:
        fennel: |
          (.. name " | " parameters-as-json)
        lua: |
          return name .. " | " .. parameters_as_json
    executing:
      feedback: true
      prefix: "\n"
      suffix: "\n"
      color: olive
      adapter:
        fennel: |
          (.. name " | " parameters-as-json)
        lua: |
          return name .. " | " .. parameters_as_json
    responding:
      feedback: true
      color: aqua
      prefix: "\n"
      suffix: "\n\n"
      adapter:
        fennel: |
          (.. name " | " parameters-as-json "\n" output)
        lua: |
          return name .. " | " .. parameters_as_json .. "\n" .. output
```

### REPL

```yaml
---
interfaces:
  repl:
    input:
      prefix: "\n"
      suffix: "\n"
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    output:
      stream: true
      prefix: "\n"
      suffix: "\n"
      color: aqua
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    prompt:
      - text: '🤖'
      - text: '> '
    tools:

```

A Read-Eval-Print Loop (REPL) is a streamlined interactive programming environment that allows users to input individual commands, processes them, and returns the results to the user in real-time.

An implementation may opt to build its REPL from scratch or leverage existing technologies such as [nREPL](https://github.com/nrepl/nrepl), [Pry](https://github.com/pry/pry), [SLIME](https://slime.common-lisp.dev), [IPython](https://github.com/ipython/ipython), [LSP](https://microsoft.github.io/language-server-protocol/), [Jupyter Notebook](https://github.com/jupyter/notebook), etc.

An implementation would likely provide access to the REPL as follows:
```bash
nb assistant.yml - repl
```

This is an example of a functioning REPL based on the following YAML fragment:

```yaml
---
interfaces:
  repl:
    output:
      prefix: "\n"
      suffix: "\n"
    prompt:
      - text: '🤖'
      - text: '> '
```

```text
🤖> Hi, how are you doing?

As an AI language model, I do not experience emotions but I am functioning
well. How can I assist you?

🤖> I like the color pink.

That's great to hear! Is there anything related to the color pink that you
would like to know or need assistance with?

🤖> What color do I like?

Based on your previous statement, you mentioned that you like the color
pink. So, it can be assumed that you like the color pink. Is there
anything related to the color pink you need help with?

🤖> |
```

#### Boot

If specified in the cartridge YAML, a REPL should display a boot message:

```yaml
---
behaviors:
  boot:
    directive: You are a helpful assistant.
    instruction: Provide a welcome message.
```

```text
Hello and welcome! As your virtual assistant, I am here to assist you with
any questions or tasks you may have. Please don't hesitate to ask me anything.
I am here to make your life easier and more efficient.

🤖> |
```

#### Prompt

You can personalize your prompt using an array of texts, with optional color formatting available for each element:

```yaml
---
interfaces:
  repl:
    prompt:
      - text: 💀
      - text: '➜ '
        color: deeppink
```

You have the option to utilize [ANSI colors](https://github.com/sickill/rainbow#ansi-colors) such as `blue`, `magenta`, and so on, or [X11 colors](https://github.com/sickill/rainbow#x11-colors) such as `aquamarine`, `deeppink`, and so forth.

The previous fragment would result in the following prompt:


> <span>💀</span><span style="color: #ff00af;">➜</span> |

### Eval

```yaml
---
interfaces:
  eval:
    input:
      prefix: "\n"
      suffix: "\n"
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    output:
      stream: true
      prefix: "\n"
      suffix: "\n"
      color: olive
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    tools:

```

Eval (short for evaluation) refers to single-turn executions of the Nano Bot that, when given an input, produce an output.

An implementation would likely provide access to eval as follows:
```bash
nb assistant.yml - eval "What is the distance to the Moon?"
```

```text
The average distance between the Earth and the Moon
is about 238,855 miles (384,400 kilometers).
```

Evaluation executions do not provide boot messages.

Implementations should also be capable of receiving input from [standard streams](https://en.wikipedia.org/wiki/Standard_streams), allowing execution with [pipe operators](https://en.wikipedia.org/wiki/Pipeline_(Unix)):

```bash
echo "What is the distance to the Moon?" | nb assistant.yml - eval
```

```text
The average distance from the Earth to the Moon
is about 238,855 miles (384,400 kilometers).
```

### Tools (Functions)

This section allows customization of the feedback loop involved in executing tools (functions).

```yaml
---
interfaces:
  tools:
    confirming:
    executing:
    responding:

```

#### Confirming

When _tools (functions)_ are defined as _confirmable_ in the [safety](?id=safety) section, we can customize how the user is prompted to confirm that the Nano Bot is allowed to execute them.

The key `yeses` defines answers that are considered as affirmative confirmations by the user, thereby allowing the Nano Bot to execute the tool (function). Implementations should treat answers as case-insensitive. This means that if "yes" is included in `yeses`, then "Yes," "YES," and any other case variations of "yes" should also be recognized as confirmation. Anything that does not match the `yeses` values should be interpreted as the user not confirming/allowing the execution.

The `default` key defines what should be considered the default answer for when the user provides an empty answer, such as just pressing _Enter_ without typing anything.

The keys `prefix`, `suffix`, `color`, and `adapter` can be used to customize the output when prompting the user for an answer.

The `adapter:` environment and available values are described in the section [Adapters](?id=adapters).

```yaml
---
interfaces:
  tools:
    confirming:
      yeses: ['y', 'yes']
      default: 'n'
      prefix: "\n"
      suffix: " [yN] > "
      color: orangered
      adapter:
        fennel: |
          (.. name " | " parameters-as-json)
        lua: |
          return name .. " | " .. parameters_as_json
```

#### Executing

The key `feedback` determines whether interfaces should provide feedback when the Nano Bot **starts** the execution of a tool (function). When set to `false`, no feedback should be provided, and the other keys should be ignored.

The keys `prefix`, `suffix`, `color`, and `adapter` can be used to customize the output when informing the user about the start of execution.

The `adapter:` environment and available values are described in the section [Adapters](?id=adapters).

```yaml
---
interfaces:
  tools:
    executing:
      feedback: false
      prefix: "\n"
      suffix: "\n"
      color: olive
      adapter:
        fennel: |
          (.. name " | " parameters-as-json)
        lua: |
          return name .. " | " .. parameters_as_json
```

#### Responding

The key `feedback:` determines whether interfaces should provide feedback when the Nano Bot **completes** the execution of a tool (function). When set to `false`, no feedback should be provided, and the other keys should be ignored.

The keys `prefix`, `suffix`, `color`, and `adapter` can be used to customize the output when informing the user about a received response from a tool (function) execution completion.

The `adapter:` environment and available values are described in the section [Adapters](?id=adapters).

```yaml
---
interfaces:
  tools:
    responding:
      feedback: true
      color: aqua
      prefix: "\n"
      suffix: "\n\n"
      adapter:
        fennel: |
          (.. name " | " parameters-as-json "\n" output)
        lua: |
          return name .. " | " .. parameters_as_json .. "\n" .. output
```

### Adapters

Adapters are simple and small pieces of code that can manipulate inputs and outputs. Implementations should support two languages for adapters: [Lua](https://www.lua.org/about.html) and [Fennel](https://fennel-lang.org). 

These languages were chosen for their proven portability, widespread availability, lightweight nature, and ease of embedding. They are small, simple, fast, and support multiple platforms across various operating systems. Additionally, they can be seamlessly integrated into numerous other programming languages.

```yaml
---
adapter:
  fennel: |
    (.. "```" content "```")
```

```yaml
---
adapter:
  lua: |
    return "```" .. content .. "```"
```

An important note on the use of adapters: They should return a value that the implementation can use to properly print on appropriate interfaces. This means that actions such as calling _print_ functions should not occur inside adapters.

The values available within each adapter enviroment will depend on the context of the adapter.

#### REPL and Eval

REPL and Eval inside `interfaces:` have `input:` and `output:` keys with support for adapters. In this context, they will have a `content` value available that holds either the user's input or the Nano Bot's output:

```yaml
---
adapter:
  fennel: |
    (.. "```" content "```")
  lua: |
    return "```" .. content .. "```"
```

Output adapters are only activated when the stream functionality is not enabled.

Regarding input, the prefix, suffix, and any modifications made by the adapter are sent to the bot. If the interaction is not stateless, these elements are also preserved in the state history.

In contrast, for outputs, any changes made by the adapter, prefix, and suffix aren't saved or used in later messages to the bot. These changes are used only for displaying purposes or for pipeline operations.

#### Tools (Functions)

All adapters related to `tools:` will have access to the following values:

| Name                 | Value                                               |
|---------------------:|-----------------------------------------------------|
| `id`                 | Identification of the specific execution.           |
| `name`               | The name of the tool.                               |
| `parameters`         | The parameters values for the tool execution.       |
| `parameters-as-json` | The parameters values represented as a JSON string. |
| `output`             | The resulting response from the execution.          |

```yaml
---
adapter:
  fennel: |
    (.. id " | " name " | " parameters-as-json "\n" output)
  lua: |
    return id .. " | " name .. " | " .. parameters_as_json .. "\n" .. output
```

Note that, following language conventions and syntax, Lua uses `parameters_as_json` instead.

For simple inspection purposes, the `parameters-as-json` value is available, so there is no need to parse potentially complex nested structures just to inspect the parameters. However, the original parameters remain accessible:

```yaml
---
adapter:
  fennel: |
    (fenne.view parameters)
```

## Tools (Functions)

Tools (Functions) are powerful means to extend the capabilities of Nano Bots. Just as humans can amplify their capabilities with access to tools such as a wrench, a calculator, or internet access, a Nano Bot can enhance its capabilities when equipped with appropriate tools.

The definition of a tool includes a piece of code that embodies the function behind the tool and a specification that enables providers to become aware of the tool's availability and to reason about when and how to properly request its execution.

The tools' source code is written in [Lua](https://www.lua.org/about.html) or [Fennel](https://fennel-lang.org), chosen for their proven portability, widespread availability, lightweight nature, and ease of embedding. They are small, simple, fast, and support multiple platforms across various operating systems. Additionally, they can be seamlessly integrated into numerous other programming languages.

As an example, this tool provides a random number generator for the Nano Bot:

```yaml
---
tools:
- name: random-number
  description: Generates a random number between 1 and 100.
  fennel: |
    (math.random 1 100)
```

`name:` is the identifier name for the function behind the tool. The `description:` helps the Nano Bot understand the purpose of the tool so it can reason about when to properly use it. `fennel:` provides the source code for the function. Like adapters, you could use `lua:` instead:

```yaml
---
tools:
- name: random-number
  description: Generates a random number between 1 and 100.
  lua: |
    return math.random(1, 100)
```

This is what a REPL execution of a Nano Bot powered by this tool would look like:
```text
🤖> generate a random number

random-number {} [yN] y

random-number {}
39

The random number generated is 39.
```

### Parameters

A tool may expect parameters. Parameters are described following the [JSON Schema](https://json-schema.org) specification:

```yaml
---
tools:
- name: random-number
  description: Generates a random number within a given range.
  parameters:
    type: object
    properties:
      from:
        type: integer
        description: The minimum expected number for random generation.
      to:
        type: integer
        description: The maximum expected number for random generation.
    required:
      - from
      - to
  fennel: |
    (let [{ : from : to } parameters]
      (math.random from to))
```

Note that the function now has access to `parameters`. The same should be true for Lua source code:

```yaml
---
tools:
- name: random-number
  description: Generates a random number within a given range.
  parameters:
    type: object
    properties:
      from:
        type: integer
        description: The minimum expected number for random generation.
      to:
        type: integer
        description: The maximum expected number for random generation.
    required:
      - from
      - to
  lua: |
    return math.random(parameters['from'], parameters['to'])
```

This is what a REPL execution of a Nano Bot powered by this tool would look like:

```text
🤖> generate a random number between 15 and 25

random-number {"from":15,"to":25} [yN] y

random-number {"from":15,"to":25}
21

The random number generated between 15 and 25 is 21.
```

### Safety

By default, executions are [sandboxed](?id=functions) and [confirmable](?id=tools), meaning they will not have access to potentially dangerous operations such as accessing the user's computer operating system, executing commands with access to disk files, or performing requests to the internet, and they will not be executed without confirmation from the user.

You can change this behavior by [disabling the safety configurations](?id=safety). A non-sandboxed tool may have powers, such as the ability to access information from the users' computers.

```yaml
---
tools:
- name: what-time-is-it
    description: Returns the current date and time.
    fennel: |
      (os.date)
```

This level of access would allow aslo the execution of [system calls](https://en.wikipedia.org/wiki/System_call), meaning that the source could hipotetically does anything it wants in the operation system. The are two words of cautions here:

#### Predictability

As execution requests for the tools are generated by the Nano Bot through providers, their outcomes are unpredictable. This means that without being sandboxed and having a function that offers this possibility, they may perform unexpected [system calls](https://en.wikipedia.org/wiki/System_call), which could be dangerous, especially without [confirmability](?id=tools).

#### Portability

Without being sandboxed and having a function that offers the possibility of _system calls_, you may achieve unlimited powers regarding what a Nano Bot can do. However, you may start to go against the Nano Bot principle of portability, where:

> _A Cartridge YAML file should ultimately be the sole and only necessary information for the bot to operate as expected._

If you create a Nano Bot that performs a system call to locally installed software, which in turn depends on a local database, you may end up creating a non-portable Nano Bot. Such a Nano Bot could not be used by another user who only has the Cartridge YAML file without proper additional instructions and an extended setup process.

## Providers

Nano Bots should be **provider-agnostic**, which means that the same Nano Bot should be able to run on different providers.

Examples of popular providers include:

- [01.AI Yi](https://01.ai)
- [Anthropic Claude](https://www.anthropic.com)
- [Cohere Command](https://cohere.com)
- [Google Gemini](https://deepmind.google/technologies/gemini)
- [LMSYS Vicuna](https://github.com/lm-sys/FastChat)
- [Maritaca AI Sabiá](https://www.maritaca.ai)
- [Meta Llama](https://ai.meta.com/llama/)
- [Mistral AI](https://mistral.ai)
- [Ollama](https://ollama.ai)
- [Open AI ChatGPT](https://openai.com/chatgpt)
- [WizardLM](https://wizardlm.github.io)

The `provider:` section of the Cartridge must specify the `id` of the provider, followed by a `credentials` and a `settings` section that includes appropriate information for allowing the Nano Bot to communicate successfully with the provider while adhering to the provider's expected API schema. An extra `options` key may be provided if the provider requires options unrelated to its expected API schema.

Minimal sample of a provider section:

```yaml
---
provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
```

### Credentials

Although it is possible to set credentials directly in the cartridge YAML, it is important to consider that cartridges may be widely shared, and exposing your credentials poses a security risk. Therefore, it is recommended to use environment variables for credentials.

Implementations should apply the regular expression `^ENV.` and replace data with prefixes like `ENV/` or `ENV-` with the corresponding environment variable value. For example, `ENV/OPENAI_API_KEY` should load the environment variable `OPENAI_API_KEY`.

### Anthropic Claude

API Documentation: https://docs.anthropic.com/en/api/messages

```yaml
---
provider:
  id: anthropic
  credentials:
    api-key: ENV/ANTHROPIC_API_KEY
    anthropic-version: '2023-06-01'
  settings:
    model: claude-3-5-sonnet-20240620
    stream: true
    max_tokens: 4096
    temperature: 1.0
    top_k: null
    top_p: null
    tool_choice:
      type: auto
      name: null
    metadata:
      user_id: ENV/NANO_BOTS_END_USER
    stop_sequences:
      - .
```

### Cohere Command

API Documentation: https://docs.cohere.com/reference/chat

```yaml
---
provider:
  id: cohere
  credentials:
    address: ENV/COHERE_API_ADDRESS
    api-key: ENV/COHERE_API_KEY
  settings:
    model: command-r-plus
    stream: true
    prompt_truncation: 'OFF'
    connectors:
      - id: web-search
    search_queries_only: false
    documents:
      - title: Tall penguins
        text: Emperor penguins are the tallest.
    citation_quality: accurate
    temperature: 0.3
    max_tokens: null
    max_input_tokens: null
    k: 0
    p: 0.75
    seed: null
    stop_sequences:
      - .
    frequency_penalty: 0.0
    presence_penalty: 0.0
    force_single_step: false
```

### Google Gemini

API Documentation: https://cloud.google.com/vertex-ai/docs/generative-ai/model-reference/gemini

```yaml
---
provider:
  id: google
  credentials:
    service: generative-language-api
    api-key: ENV/GOOGLE_API_KEY
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
    file-path: ENV/GOOGLE_CREDENTIALS_FILE_PATH
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
    file_contents: ENV/GOOGLE_CREDENTIALS_FILE_CONTENTS
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
    file-path: ENV/GOOGLE_CREDENTIALS_FILE_PATH
    project-id: ENV/GOOGLE_PROJECT_ID
  options:
    model: gemini-1.5-pro
    stream: true
  settings:
    safetySettings:
      - category: HARM_CATEGORY_DANGEROUS_CONTENT
        threshold: BLOCK_NONE
    generationConfig:
      temperature: 0.9
      maxOutputTokens: null
      candidateCount: 1
      presencePenalty: null
      frequencyPenalty: null
      topK: null
      topP: 1.0
      stopSequences:
        - .
      responseMimeType: application/json
      responseSchema:
        type: object
        properties:
          name:
            type: string
```

### Maritaca AI Sabiá

API Documentation: https://chat.maritaca.ai/docs

```yaml
---
provider:
  id: maritaca
  credentials:
    address: ENV/MARITACA_API_ADDRESS
    api-key: ENV/MARITACA_API_KEY
  settings:
    stream: true
    model: sabia-2-medium
    max_tokens: null
    do_sample: true
    temperature: 0.7
    top_p: 0.95
    repetition_penalty: 1
    num_tokens_per_message: null
    stopping_tokens:
      - .

```

### Mistral AI

API Documentation: https://docs.mistral.ai/api/

```yaml
---
provider:
  id: mistral
  credentials:
    address: ENV/MISTRAL_API_ADDRESS
    api-key: ENV/MISTRAL_API_KEY
  settings:
    model: mistral-large
    temperature: 0.7
    top_p: 1
    max_tokens: null
    stream: true
    safe_prompt: false
    random_seed: null
```

### Ollama

API Documentation:
- https://github.com/jmorganca/ollama/blob/main/docs/api.md#generate-a-chat-completion
- https://github.com/jmorganca/ollama/blob/main/docs/modelfile.md#valid-parameters-and-values

```yaml
---
provider:
  id: ollama
  credentials:
    address: ENV/OLLAMA_API_ADDRESS
  settings:
    model: llama3
    format: json
    raw: false
    options:
      num_keep: 5
      seed: 42
      num_predict: 100
      top_k: 20
      top_p: 0.9
      tfs_z: 0.5
      typical_p: 0.7
      repeat_last_n: 33
      temperature: 0.8
      repeat_penalty: 1.2
      presence_penalty: 1.5
      frequency_penalty: 1.0
      mirostat: 1
      mirostat_tau: 0.8
      mirostat_eta: 0.6
      penalize_newline: true
      numa: false
      num_ctx: 1024
      num_batch: 2
      num_gpu: 1
      main_gpu: 0
      low_vram: false
      f16_kv: true
      vocab_only: false
      use_mmap: true
      use_mlock: false
      num_thread: 8
      stop: 
        - "\n"
        - "user:"
```

### OpenAI ChatGPT

API Documentation: https://platform.openai.com/docs/api-reference/chat

```yaml
---
provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
    stream: true
    stream_options:
      include_usage: false
    frequency_penalty: 0
    logit_bias: null
    logprobs: false
    top_logprobs: null
    max_tokens: null
    n: 1
    presence_penalty: 0
    response_format:
      type: json_object
    seed: null
    service_tier: null
    stop:
      - .
    temperature: 1
    top_p: 1
    tool_choice: auto
    parallel_tool_calls: true
```

## Miscellaneous

```yaml
---
miscellaneous:
  key: value
```

`miscellaneous:` is an optional key that can hold additional data for purposes other than making the Nano Bot functional. This data may include metadata for marketplaces, educational information, and so on. It's important to note that these keys must be not relevant for the Nano Bot's operation and should be deliberately ignored by implementations.

They can contain any information or structure, as shown in the following examples:

```yaml
---
miscellaneous:
  some-marketplace:
    background-color: blue
    tags:
      - creativity
      - programming
```

```yaml
---
miscellaneous:
  some-platform:
    technique: summarization
    details: |
      This Nano Bot exemplifies one of the
      common approaches to summarizing content.
```

# Implementations

Nano Bots can be implemented in any programming language. Typically, implementations provide executable binaries capable of interpreting the cartridge YAML file as specified here, correctly interacting with multiple providers, and offering the expected interaction interfaces (_REPL_ and _Eval_).

Here's what a fully-functional implementation of Nano Bots feels like:

```bash
nb - - eval "hello"
# => Hello! How may I assist you today?

nb to-en-us-translator.yml - eval "Salut, comment ça va?"
# => Hello, how are you doing?

nb midjourney.yml - eval "happy cyberpunk robot"
# => A cheerful and fun-loving robot is dancing wildly amidst a
#    futuristic and lively cityscape. Holographic advertisements
#    and vibrant neon colors can be seen in the background.

nb lisp.yml - eval "(+ 1 2)"
# => 3

cat article.txt |
  nb to-en-us-translator.yml - eval |
  nb summarizer.yml - eval
# -> LLM stands for Large Language Model, which refers to an
#    artificial intelligence algorithm capable of processing
#    and understanding vast amounts of natural language data,
#    allowing it to generate human-like responses and perform
#    a range of language-related tasks.
```

```bash
nb - - repl

nb assistant.yml - repl
```

```text
🤖> Hi, how are you doing?

As an AI language model, I do not experience emotions but I am functioning
well. How can I assist you?

🤖> |
```

You may name your binary as you wish, with `nb` being just an illustrative example.

## Cartridges

Cartridges are YML files and should be loaded according to the path specified by the user:

```bash
nb assistant.yml - repl
```

This command should attempt to load the `assistant.yml` file. The user may omit the file extension:

```bash
nb assistant - repl
```

In this case, the implementation should attempt to load either the `assistant.yml` or `assistant.yaml` file.

If the environment variable `NANO_BOTS_CARTRIDGES_PATH` is defined and the path was not found in the command's working path, the implementation should attempt to load the file from the path specified in the environment variable:

```bash
NANO_BOTS_CARTRIDGES_PATH=/home/aqua/cartridges:/home/lime/cartridges

nb assistant - repl
```

Paths that should be attempted to be loaded:

```text
/home/aqua/cartridges/assistant.yml
/home/aqua/cartridges/assistant.yaml

/home/lime/cartridges/assistant.yml
/home/lime/cartridges/assistant.yaml
```

If no file is found, the implementation should fallback to attempting to load from the default expected cartridges path, adhering to the [XDG specification](https://specifications.freedesktop.org/basedir-spec/basedir-spec-latest.html):


```text
/home/user/.local/share/nano-bots/cartridges/assistant.yml
/home/user/.local/share/nano-bots/cartridges/assistant.yaml
```

### Default

This is the minimum required Cartridge data:

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  version: 1.0.0
  license: CC0-1.0
  description: A helpful assistant.

provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
```

These are the default values when the following keys are not specified in the Cartridge file:

```yaml
---
safety:
  functions:
    sandboxed: true
  tools:
    confirmable: true

interfaces:
  repl:
    output:
      stream: true
      prefix: "\n"
      suffix: "\n"
    prompt:
      - text: '🤖'
      - text: '> '
  eval:
    output:
      stream: true
      suffix: "\n"
  tools:
    confirming:
      suffix: ' [yN] '
      default: 'n'
      yeses: ['y', 'yes']
    executing:
      feedback: false
    responding:
      suffix: "\n\n"
      feedback: true

provider:
  options:
    stream: true
  settings:
    stream: true
```

Users may choose not to provide a Cartridge file by using the `-` character:

```sh
nb - - repl
nb - - eval "Hi"
```

Under this circumstance, the implementation should apply the following default Cartridge:

```yaml
---
meta:
  symbol: 🤖
  name: Unknown
  author: None
  version: 0.0.0
  license: CC0-1.0
  description: Unknown

provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
```

### Full Specification

This example showcases all the possible keys present in a Nano Bot Cartridge YAML file:

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  version: 1.0.0
  license: CC0-1.0
  description: A helpful assistant.

behaviors:
  interaction:
    directive: You are a helpful assistant.
    backdrop: |
      The Moon is Earth's natural satellite, orbiting our planet.
      The user might use the term "Selene" when referring to the Moon.
    instruction: Answer the user's questions.
  boot:
    directive: You are a helpful assistant.
    backdrop: |
      This is a good example of a welcome message:
      "Welcome! How may I assist you?"
    instruction: Provide a welcome message.

interfaces:
  input:
    prefix: "\n"
    suffix: "\n"
    adapter:
      fennel: |
        (.. "```" content "```")
      lua: |
        return "```" .. content .. "```"
  output:
    stream: true
    prefix: "\n"
    suffix: "\n"
    color: aqua
    adapter:
      fennel: |
        (.. "```" content "```")
      lua: |
        return "```" .. content .. "```"
  tools:
    confirming:
      yeses: ['y', 'yes']
      default: 'n'
      prefix: "\n"
      suffix: " [yN] > "
      color: orangered
      adapter:
        fennel: |
          (.. name " | " parameters-as-json)
        lua: |
          return name .. " | " .. parameters_as_json
    executing:
      feedback: true
      prefix: "\n"
      suffix: "\n"
      color: olive
      adapter:
        fennel: |
          (.. name " | " parameters-as-json)
        lua: |
          return name .. " | " .. parameters_as_json
    responding:
      feedback: true
      color: aqua
      prefix: "\n"
      suffix: "\n\n"
      adapter:
        fennel: |
          (.. name " | " parameters-as-json "\n" output)
        lua: |
          return name .. " | " .. parameters_as_json .. "\n" .. output
  repl:
    input:
      prefix: "\n"
      suffix: "\n"
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    output:
      stream: true
      prefix: "\n"
      suffix: "\n"
      color: olive
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    prompt:
      - text: '🤖'
      - text: '> '
        color: blue
    tools:
      confirming:
        yeses: ['y', 'yes']
        default: 'n'
        prefix: "\n"
        suffix: " [yN] > "
        color: orangered
        adapter:
          fennel: |
            (.. name " | " parameters-as-json)
          lua: |
            return name .. " | " .. parameters_as_json
      executing:
        feedback: true
        prefix: "\n"
        suffix: "\n"
        color: olive
        adapter:
          fennel: |
            (.. name " | " parameters-as-json)
          lua: |
            return name .. " | " .. parameters_as_json
      responding:
        feedback: true
        color: aqua
        prefix: "\n"
        suffix: "\n\n"
        adapter:
          fennel: |
            (.. name " | " parameters-as-json "\n" output)
          lua: |
            return name .. " | " .. parameters_as_json .. "\n" .. output
  eval:
    input:
      prefix: "\n"
      suffix: "\n"
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    output:
      stream: true
      prefix: "\n"
      suffix: "\n"
      color: magenta
      adapter:
        fennel: |
          (.. "```" content "```")
        lua: |
          return "```" .. content .. "```"
    tools:
      confirming:
        yeses: ['y', 'yes']
        default: 'n'
        prefix: "\n"
        suffix: " [yN] > "
        color: orangered
        adapter:
          fennel: |
            (.. name " | " parameters-as-json)
          lua: |
            return name .. " | " .. parameters_as_json
      executing:
        feedback: true
        prefix: "\n"
        suffix: "\n"
        color: olive
        adapter:
          fennel: |
            (.. name " | " parameters-as-json)
          lua: |
            return name .. " | " .. parameters_as_json
      responding:
        feedback: true
        color: aqua
        prefix: "\n"
        suffix: "\n\n"
        adapter:
          fennel: |
            (.. name " | " parameters-as-json "\n" output)
          lua: |
            return name .. " | " .. parameters_as_json .. "\n" .. output

state:
  path: ENV/NANO_BOTS_STATE_PATH

safety:
  functions:
    sandboxed: true
  tools:
    confirmable: true

provider:
  id: openai
  credentials:
    address: ENV/OPENAI_API_ADDRESS
    access-token: ENV/OPENAI_API_KEY
  settings:
    user: ENV/NANO_BOTS_END_USER
    model: gpt-4o
    stream: true
    frequency_penalty: 0
    logit_bias: null
    logprobs: false
    top_logprobs: null
    max_tokens: null
    n: 1
    presence_penalty: 0
    response_format:
      type: json_object
    seed: null
    stop:
      - .
    temperature: 1
    top_p: 1
    tool_choice: auto

provider:
  id: google
  credentials:
    service: generative-language-api
    api-key: ENV/GOOGLE_API_KEY
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
    file-path: ENV/GOOGLE_CREDENTIALS_FILE_PATH
  credentials:
    service: vertex-ai-api
    region: ENV/GOOGLE_REGION
    file-path: ENV/GOOGLE_CREDENTIALS_FILE_PATH
    project-id: ENV/GOOGLE_PROJECT_ID
  options:
    model: gemini-pro
    stream: true
  settings:
    safetySettings:
      - category: HARM_CATEGORY_DANGEROUS_CONTENT
        threshold: BLOCK_NONE
    generationConfig:
      temperature: 0.9
      maxOutputTokens: 8192
      topK: null
      topP: 1.0
      stopSequences:
        - .

miscellaneous:
  key: value
```

## State

### Stateless

By default, Nano Bots are stateless.

An evaluation is a single-turn interaction.

A REPL should be capable of maintaining multi-turn interactions, which means it must retain the conversation history and utilize it during interactions throughout its lifetime. Once the user exits the REPL, all history are discarded.

The default stateless behavior is defined by the `-` character in interactions:

```bash
nb assistant.yml - repl
nb assistant.yml - eval "Hi"
```

### Stateful

Implementations should support storing state by identifying a state key different from `-`:

```bash
nb assistant.yml E15D repl
nb assistant.yml D9D6 eval "Hi"
```

In this example, both `E15D` and `D9D6` are distinct identifiers used to indicate which state key should be employed for storing and retrieving state information related to that interaction.

In this scenario, both Eval and REPL store their states (history) and should be capable of performing multi-turn interactions. Eval should remember its previous interactions, and a REPL should remember its previous interactions even if it is exited and started again.

By default, implementations should be [XDG compliant](https://specifications.freedesktop.org/basedir-spec/basedir-spec-latest.html), which means that the default storage path would be:

```text
/home/user/.local/state/nano-bots/your-implementation
```

If the `NANO_BOTS_STATE_PATH` environment variable exists, it should be used as the path to store the states.

A Cartridge may include a section that defines a custom path for storing the states. In this case, it should override both the default and the path specified by the environment variable:

```yaml
---
state:
  path: ENV/NANO_BOTS_STATE_PATH
```

The state should be stored in a manner that ensures isolation between multiple Nano Bots and Implementations.

In this example:

```bash
nb assistant.yml E15D repl
```

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Author Name
  version: 1.0.0
  license: CC0-1.0
  description: A helpful assistant.

state:
  path: /home/user/.local/state/nano-bots

provider:
  settings:
    user: end-user-name
```

The state should be stored in a path similar to this:

```text
/home/user/.local/state/nano-bots/your-implementation/author-name/nano-bot-name/0-0-1/end-user-name/E15D/state.json
```

If not specified in the provider settings, `end-user-name` should be loaded from the environment variable `NANO_BOTS_END_USER`. If the environment variable does not exist, use the default `unknown` value:

```text
/home/user/.local/state/nano-bots/your-implementation/author-name/nano-bot-name/0-0-1/unknown/E15D/state.json
```

JSON is merely an example; each implementation can choose the most suitable data format to work with.

> ⚠️ A Nano Bot should **never** depend on or rely on a state to function fully—**absolutely never**.

States serve as a convenience for users and should not be used to influence Nano Bot behaviors. Instead, such behaviors should be managed through the [Behaviors](?id=behaviors) section of the Cartridge YAML. This reinforces the notion that a Cartridge YAML file should ultimately be the sole and only necessary information for the bot to operate as expected.

## Stream

Unless otherwise specified in the Cartridge file, or if not supported by the provider, both the REPL and Eval Interfaces should be capable of streaming messages. This means they should be able to display content partially, whether character by character, token by token, or word by word.

# Breaking Changes

## 3.0.0

From version `2.3.0` to version `3.0.0`:

Cartridges must have a `.yml` or `.yaml` extension.

`NANO_BOTS_STATE_DIRECTORY` has been renamed to `NANO_BOTS_STATE_PATH`.
`NANO_BOTS_CARTRIDGES_DIRECTORY` has been renamed to `NANO_BOTS_CARTRIDGES_PATH`.

Although some implementations may offer backward compatibility for these changes, supporting the old names is not expected in version `3.0.0`.

## 2.0.0

From version `1.1.0` to version `2.0.0`:

Google Provider requires a new `service` key in `credentials`.

## 1.0.0

From version `0.2.0` to version `1.0.0`:

Lua functions should start to expect explicit `return` statements.

Before:

```yaml
---
lua: |
  "```" .. content .. "```"
```

After:
```yaml
---
lua: |
  return "```" .. content .. "```"
```

This decision was made to support complex multiline Lua functions. Without explicit `return` statements, implementations would need to infer where to inject the `return` statement to create valid Lua code, which could potentially compromise the functionality of the code through incorrect assumptions.

# Experimental

## Clojure Support

We are exploring the use of [Clojure](https://clojure.org) through [Babashka](https://babashka.org), powered by [GraalVM](https://www.graalvm.org).

Currently, [Lua](https://www.lua.org/about.html) and [Fennel](https://fennel-lang.org) are our primary supported languages, due to their proven portability, widespread availability, lightweight nature, and ease of embedding. If this specific flavor of Clojure continues to demonstrate the potential to offer the same attributes as observed in Lua and Fennel, it may become an additional language that is expected to be supported in our specification. Regardless of this potential move, we have absolutely no intention of removing or replacing Lua or Fennel as the primary supported languages.

The experimental support for Clojure would be similar to Lua and Fennel, using the `clojure:` key:

```yaml
---
clojure: |
  (-> (java.time.ZonedDateTime/now)
      (.format (java.time.format.DateTimeFormatter/ofPattern "yyyy-MM-dd HH:mm"))
      (clojure.string/trimr))
```

## Markdown Support

We are exploring the use of [Markdown](https://en.wikipedia.org/wiki/Markdown) as a cartridge source code format. The following file extensions become valid cartridges:

```text
.md .mkdn .mdown .markdown
```

This is an example of a Markdown cartridge:

````md
Start by defining a meta section:

```yaml
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  description: A helpful assistant.
```

You can also add version and license information:

```yaml
meta:
  version: 1.0.0
  license: CC0-1.0
```

Then, add a behavior section:

```yaml
behaviors:
  interaction:
    directive: You are a helpful assistant.
```
````

Implementations should extract each code block and merge them to end up with the complete cartridge's source code, ignoring everything else:

```yaml
---
meta:
  symbol: 🤖
  name: Nano Bot Name
  author: Your Name
  description: A helpful assistant.
  version: 1.0.0
  license: CC0-1.0
behaviors:
  interaction:
    directive: You are a helpful assistant.
```

Code blocks in supported languages that appear after a previous tool's definition should be incorporated as the function source code of the last defined tool:


````md
This is the specification of the tool:

```yaml
tools:
- name: date-and-time
  description: Returns the current date and time.
```

This is the function source code for the tool:

```fnl
(os.date)
```
````

The above Markdown cartridge should be extracted as:
```yaml
---
tools:
- name: date-and-time
  description: Returns the current date and time.
  fennel: |
    (os.date)
```

Root keys that are array values should be concatenated:

````md
This is the first tool:

```yaml
tools:
- name: date-and-time
  description: Returns the current date and time.
```

That has this source code:

```fnl
(os.date)
```

This is the second tool:

```yaml
tools:
- name: random-number
  description: Generates a random number.
```

With this source code:

```lua
return math.random()
```

````

The above Markdown cartridge should be extracted as:

```yaml
---
tools:
- name: date-and-time
  description: Returns the current date and time.
  fennel: |
    (os.date)
- name: random-number
  description: Generates a random number.
  lua: |
    return math.random()
```
