package main

import (
	_ "embed"
	"fmt"
	"io/ioutil"
	"net/http"

	"github.com/yuin/goldmark"
	"github.com/yuin/goldmark-meta"
	"github.com/yuin/goldmark/extension"
	"github.com/yuin/goldmark/parser"
	goldmarkHtml "github.com/yuin/goldmark/renderer/html"
)

//go:embed github.css
var gitHubCss string

const htmlHeader = `<html>
<head>
<style type="text/css"><!--
%s
	.markdown-body {
		box-sizing: border-box;
		min-width: 200px;
		max-width: 980px;
		margin: 0 auto;
		padding: 45px;
	}

	@media (max-width: 767px) {
		.markdown-body {
			padding: 15px;
		}
	}
// -->
</style>
</head><body class="markdown-body">`

const htmlFooter = `<hr />
Generated by <a href="https://github.com/hymkor/xnhttpd">xnhttpd</a>
Powered by <a href="https://github.com/yuin/goldmark">goldmark</a>
and <a href="https://github.com/sindresorhus/github-markdown-css">github-markdown-css</a>
</body></html>`

var markdownReader goldmark.Markdown

var markdownOptions = []goldmark.Option{
	goldmark.WithParserOptions(
		parser.WithAutoHeadingID()),
	goldmark.WithExtensions(
		extension.Table,
		extension.NewLinkify(
			extension.WithLinkifyAllowedProtocols([][]byte{
				[]byte("http:"),
				[]byte("https:"),
			})),
		extension.TaskList,
		extension.Footnote,
		meta.New(meta.WithTable())),
}

func setMarkdownOptions(enableHtml bool, hardwrap bool) {
	if enableHtml {
		markdownOptions = append(markdownOptions,
			goldmark.WithRendererOptions(goldmarkHtml.WithUnsafe()))
	}
	if hardwrap {
		markdownOptions = append(markdownOptions,
			goldmark.WithRendererOptions(goldmarkHtml.WithHardWraps()))
	}
	markdownReader = goldmark.New(markdownOptions...)
}

func catAsMarkdown(path string, w http.ResponseWriter) error {
	source, err := ioutil.ReadFile(path)
	if err != nil {
		return err
	}
	w.WriteHeader(http.StatusOK)
	fmt.Fprintf(w, htmlHeader, gitHubCss)
	if markdownReader == nil {
		setMarkdownOptions(false, false)
	}
	err = markdownReader.Convert(source, w)

	fmt.Fprintln(w, htmlFooter)
	return err
}
