import Vue from 'vue';
import axios from 'axios';
import VueAxios from 'vue-axios';
import TokenService from '@/common/token.service';

const ApiService = {
  init(baseUrl) {
    Vue.use(VueAxios, axios);
    Vue.axios.defaults.baseURL = baseUrl;
  },

  setHeader() {
    Vue.axios.defaults.headers.common[
      'Authorization'
    ] = `Token ${TokenService.getToken()}`;
  },

  query(resource, params) {
    return Vue.axios.get(resource, params).catch(error => {
      throw new Error(`ApiService.query() ${error}`);
    });
  },

  get(resource, slug = '') {
    return Vue.axios.get(`${resource}/${slug}`).catch(error => {
      throw new Error(`ApiService.get() ${error}`);
    });
  },

  post(resource, params) {
    return Vue.axios.post(`${resource}`, params);
  },

  update(resource, slug, params) {
    return Vue.axios.put(`${resource}/${slug}`, params);
  },

  put(resource, params) {
    return Vue.axios.put(`${resource}`, params);
  },

  delete(resource) {
    return Vue.axios.delete(resource).catch(error => {
      throw new Error(`ApiService.delete() ${error}`);
    });
  }
};

export default ApiService;

export const TagsService = {
  get() {
    return ApiService.get('tags');
  }
};

export const ArticlesService = {
  query(type, params) {
    return ApiService.query('articles' + (type === 'feed' ? '/feed' : ''), {
      params: params
    });
  },

  get(slug) {
    return ApiService.get('articles', slug);
  },

  create(params) {
    return ApiService.post('articles', { article: params });
  },

  update(slug, params) {
    return ApiService.update('articles', slug, { article: params });
  },

  destroy(slug) {
    return ApiService.delete(`articles/${slug}`);
  }
};

export const CommentsService = {
  get(slug) {
    if (typeof slug !== 'string') {
      throw new Error(
        'CommentsService.get() article slug required to fetch comments'
      );
    }
    return ApiService.get('articles', `${slug}/comments`);
  },

  post(slug, payload) {
    return ApiService.post(`articles/${slug}/comments`, {
      comment: { body: payload }
    });
  },

  destroy(slug, commentId) {
    return ApiService.delete(`articles/${slug}/comments/${commentId}`);
  }
};

export const FavoriteService = {
  add(slug) {
    return ApiService.post(`articles/${slug}/favorite`);
  },

  remove(slug) {
    return ApiService.delete(`articles/${slug}/favorite`);
  }
};
