// Copyright (c) HashiCorp, Inc.
// SPDX-License-Identifier: BUSL-1.1

package config_test

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"testing"
	"time"

	"github.com/hashicorp/boundary/internal/cmd/config"
	"github.com/hashicorp/boundary/internal/event"
	"github.com/hashicorp/boundary/internal/ratelimit"
	configutil "github.com/hashicorp/go-secure-stdlib/configutil/v2"
	"github.com/hashicorp/go-secure-stdlib/listenerutil"
	"github.com/stretchr/testify/require"
)

func TestLoad(t *testing.T) {
	apiHeaders := map[int]http.Header{
		0: {
			"Content-Security-Policy":   {"default-src 'none'"},
			"X-Content-Type-Options":    {"nosniff"},
			"Strict-Transport-Security": {"max-age=31536000; includeSubDomains"},
			"Cache-Control":             {"no-store"},
		},
	}
	uiHeaders := map[int]http.Header{
		0: {
			"Content-Security-Policy":   {"default-src 'none'; script-src 'self' 'wasm-unsafe-eval'; frame-src 'self'; font-src 'self'; connect-src 'self'; img-src 'self' data:; style-src 'self'; media-src 'self'; manifest-src 'self'; style-src-attr 'self'; frame-ancestors 'self'"},
			"X-Content-Type-Options":    {"nosniff"},
			"Strict-Transport-Security": {"max-age=31536000; includeSubDomains"},
			"Cache-Control":             {"no-store"},
		},
	}

	cases := []struct {
		name        string
		expected    *config.Config
		expectedErr error
	}{
		{
			"SingleFile",
			&config.Config{
				SharedConfig: &configutil.SharedConfig{
					EntSharedConfig: configutil.EntSharedConfig{},
					Listeners: []*listenerutil.ListenerConfig{
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "api",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"api"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              func() *bool { t := true; return &t }(),
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       []string{"*"},
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "cluster",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"cluster"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "proxy",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"proxy"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "ops",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"ops"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
					},
					Seals: []*configutil.KMS{
						{
							Type:             "aead",
							Purpose:          []string{"root"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "sP1fnF5Xz85RrXyELHFeZg9Ad2qt4Z4bgNHVGtD6ung=",
								"key_id":    "global_root",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"worker-auth"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_worker-auth",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"recovery"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_recovery",
							},
						},
					},
					Entropy:                      nil,
					DisableMlock:                 true,
					DisableMlockRaw:              nil,
					Telemetry:                    nil,
					DefaultMaxRequestDuration:    0,
					DefaultMaxRequestDurationRaw: nil,
					LogFormat:                    "",
					LogLevel:                     "",
					PidFile:                      "",
					ClusterName:                  "",
				},
				Worker: &config.Worker{
					Name:                                "worker0",
					Description:                         "A worker",
					PublicAddr:                          "",
					InitialUpstreams:                    []string{"boundary:9201"},
					InitialUpstreamsRaw:                 []any{"boundary:9201"},
					Tags:                                nil,
					TagsRaw:                             nil,
					SuccessfulStatusGracePeriodDuration: 0,
					StatusCallTimeoutDuration:           0,
					AuthStoragePath:                     "",
					ControllerGeneratedActivationToken:  "",
				},
				Controller: &config.Controller{
					Name:        "controller0",
					Description: "A controller",
					Database: &config.Database{
						Url:                       "env://LOAD_TEST_BOUNDARY_POSTGRES_URL",
						MigrationUrl:              "",
						MaxOpenConnections:        0,
						MaxOpenConnectionsRaw:     nil,
						MaxIdleConnections:        nil,
						MaxIdleConnectionsRaw:     nil,
						ConnMaxIdleTime:           nil,
						ConnMaxIdleTimeDuration:   nil,
						SkipSharedLockAcquisition: false,
					},
					PublicClusterAddr: "",
					Scheduler: config.Scheduler{
						JobRunInterval:  nil,
						MonitorInterval: nil,
					},
					AuthTokenTimeToLive:             nil,
					AuthTokenTimeToLiveDuration:     0,
					AuthTokenTimeToStale:            nil,
					AuthTokenTimeToStaleDuration:    0,
					GracefulShutdownWait:            nil,
					GracefulShutdownWaitDuration:    0,
					WorkerStatusGracePeriodDuration: 0,
					LivenessTimeToStaleDuration:     0,
					ApiRateLimits:                   make(ratelimit.Configs, 0),
					ApiRateLimiterMaxQuotas:         ratelimit.DefaultLimiterMaxQuotas(),
					MaxPageSizeRaw:                  nil,
					MaxPageSize:                     0,
				},
				DevController:           false,
				DevUiPassthroughDir:     "",
				DevControllerKey:        "",
				DevWorkerAuthKey:        "",
				DevWorkerAuthStorageKey: "",
				DevBsrKey:               "",
				DevRecoveryKey:          "",
				Eventing: &event.EventerConfig{
					AuditEnabled:        false,
					ObservationsEnabled: true,
					SysEventsEnabled:    true,
					Sinks: []*event.SinkConfig{
						{
							Name:           "default",
							Description:    "",
							EventTypes:     []event.Type{"*"},
							EventSourceUrl: "",
							AllowFilters:   nil,
							DenyFilters:    nil,
							Format:         event.JSONSinkFormat,
							Type:           event.StderrSink,
							StderrConfig:   nil,
							FileConfig:     nil,
							WriterConfig:   nil,
							AuditConfig: &event.AuditConfig{
								FilterOverrides: map[event.DataClassification]event.FilterOperation{
									event.SecretClassification:    event.RedactOperation,
									event.SensitiveClassification: event.RedactOperation,
								},
								FilterOverridesHCL: nil,
							},
						},
					},
				},
				Plugins: config.Plugins{
					ExecutionDir: "",
				},
				HcpbClusterId: "",
				Reporting: config.Reporting{
					License: config.License{
						Enabled: false,
					},
				},
			},
			nil,
		},
		{
			"MultiFile",
			&config.Config{
				SharedConfig: &configutil.SharedConfig{
					EntSharedConfig: configutil.EntSharedConfig{},
					Listeners: []*listenerutil.ListenerConfig{
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "api",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"api"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              func() *bool { t := true; return &t }(),
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       []string{"*"},
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "cluster",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"cluster"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "ops",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"ops"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "proxy",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"proxy"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
					},
					Seals: []*configutil.KMS{
						{
							Type:             "aead",
							Purpose:          []string{"root"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "sP1fnF5Xz85RrXyELHFeZg9Ad2qt4Z4bgNHVGtD6ung=",
								"key_id":    "global_root",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"worker-auth"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_worker-auth",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"recovery"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_recovery",
							},
						},
					},
					Entropy:                      nil,
					DisableMlock:                 true,
					DisableMlockRaw:              nil,
					Telemetry:                    nil,
					DefaultMaxRequestDuration:    0,
					DefaultMaxRequestDurationRaw: nil,
					LogFormat:                    "",
					LogLevel:                     "",
					PidFile:                      "",
					ClusterName:                  "",
				},
				Worker: &config.Worker{
					Name:                                "worker0",
					Description:                         "A worker",
					PublicAddr:                          "",
					InitialUpstreams:                    []string{"boundary:9201"},
					InitialUpstreamsRaw:                 []any{"boundary:9201"},
					Tags:                                nil,
					TagsRaw:                             nil,
					SuccessfulStatusGracePeriodDuration: 0,
					StatusCallTimeoutDuration:           0,
					AuthStoragePath:                     "",
					ControllerGeneratedActivationToken:  "",
				},
				Controller: &config.Controller{
					Name:        "controller0",
					Description: "A controller",
					Database: &config.Database{
						Url:                       "env://LOAD_TEST_BOUNDARY_POSTGRES_URL",
						MigrationUrl:              "",
						MaxOpenConnections:        0,
						MaxOpenConnectionsRaw:     nil,
						MaxIdleConnections:        nil,
						MaxIdleConnectionsRaw:     nil,
						ConnMaxIdleTime:           nil,
						ConnMaxIdleTimeDuration:   nil,
						SkipSharedLockAcquisition: false,
					},
					PublicClusterAddr: "",
					Scheduler: config.Scheduler{
						JobRunInterval:  nil,
						MonitorInterval: nil,
					},
					AuthTokenTimeToLive:             nil,
					AuthTokenTimeToLiveDuration:     0,
					AuthTokenTimeToStale:            nil,
					AuthTokenTimeToStaleDuration:    0,
					GracefulShutdownWait:            nil,
					GracefulShutdownWaitDuration:    0,
					WorkerStatusGracePeriodDuration: 0,
					LivenessTimeToStaleDuration:     0,
					ApiRateLimits:                   make(ratelimit.Configs, 0),
					ApiRateLimiterMaxQuotas:         ratelimit.DefaultLimiterMaxQuotas(),
					MaxPageSizeRaw:                  nil,
					MaxPageSize:                     0,
				},
				DevController:           false,
				DevUiPassthroughDir:     "",
				DevControllerKey:        "",
				DevWorkerAuthKey:        "",
				DevBsrKey:               "",
				DevWorkerAuthStorageKey: "",
				DevRecoveryKey:          "",
				Eventing: &event.EventerConfig{
					AuditEnabled:        false,
					ObservationsEnabled: true,
					SysEventsEnabled:    true,
					Sinks: []*event.SinkConfig{
						{
							Name:           "default",
							Description:    "",
							EventTypes:     []event.Type{"*"},
							EventSourceUrl: "",
							AllowFilters:   nil,
							DenyFilters:    nil,
							Format:         event.JSONSinkFormat,
							Type:           event.StderrSink,
							StderrConfig:   nil,
							FileConfig:     nil,
							WriterConfig:   nil,
							AuditConfig: &event.AuditConfig{
								FilterOverrides: map[event.DataClassification]event.FilterOperation{
									event.SecretClassification:    event.RedactOperation,
									event.SensitiveClassification: event.RedactOperation,
								},
								FilterOverridesHCL: nil,
							},
						},
					},
				},
				Plugins: config.Plugins{
					ExecutionDir: "",
				},
				HcpbClusterId: "",
			},
			nil,
		},
		{
			"MultiFileDuplicateStanza",
			&config.Config{
				SharedConfig: &configutil.SharedConfig{
					EntSharedConfig: configutil.EntSharedConfig{},
					Listeners: []*listenerutil.ListenerConfig{
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "api",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"api"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              func() *bool { t := true; return &t }(),
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       []string{"*"},
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "cluster",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"cluster"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "ops",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"ops"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "proxy",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"proxy"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
					},
					Seals: []*configutil.KMS{
						{
							Type:             "aead",
							Purpose:          []string{"root"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "sP1fnF5Xz85RrXyELHFeZg9Ad2qt4Z4bgNHVGtD6ung=",
								"key_id":    "global_root",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"worker-auth"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_worker-auth",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"recovery"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_recovery",
							},
						},
					},
					Entropy:                      nil,
					DisableMlock:                 true,
					DisableMlockRaw:              nil,
					Telemetry:                    nil,
					DefaultMaxRequestDuration:    0,
					DefaultMaxRequestDurationRaw: nil,
					LogFormat:                    "",
					LogLevel:                     "",
					PidFile:                      "",
					ClusterName:                  "",
				},
				Worker: &config.Worker{
					Name:                                "worker0",
					Description:                         "A worker",
					PublicAddr:                          "",
					InitialUpstreams:                    []string{"boundary:9201"},
					InitialUpstreamsRaw:                 []any{"boundary:9201"},
					Tags:                                nil,
					TagsRaw:                             nil,
					SuccessfulStatusGracePeriodDuration: 0,
					StatusCallTimeoutDuration:           0,
					AuthStoragePath:                     "",
					ControllerGeneratedActivationToken:  "",
				},
				Controller: &config.Controller{
					Name:        "controller0",
					Description: "A controller but duplicated",
					Database: &config.Database{
						Url:                       "env://LOAD_TEST_BOUNDARY_POSTGRES_URL",
						MigrationUrl:              "",
						MaxOpenConnections:        0,
						MaxOpenConnectionsRaw:     nil,
						MaxIdleConnections:        nil,
						MaxIdleConnectionsRaw:     nil,
						ConnMaxIdleTime:           nil,
						ConnMaxIdleTimeDuration:   nil,
						SkipSharedLockAcquisition: false,
					},
					PublicClusterAddr: "",
					Scheduler: config.Scheduler{
						JobRunInterval:  nil,
						MonitorInterval: nil,
					},
					AuthTokenTimeToLive:             nil,
					AuthTokenTimeToLiveDuration:     0,
					AuthTokenTimeToStale:            nil,
					AuthTokenTimeToStaleDuration:    0,
					GracefulShutdownWait:            nil,
					GracefulShutdownWaitDuration:    0,
					WorkerStatusGracePeriodDuration: 0,
					LivenessTimeToStaleDuration:     0,
					ApiRateLimits: ratelimit.Configs{
						{
							Resources: []string{"*"},
							Actions:   []string{"*"},
							Per:       "total",
							Limit:     50,
							PeriodHCL: "1m",
							Period:    time.Minute,
							Unlimited: false,
						},
						{
							Resources: []string{"*"},
							Actions:   []string{"list"},
							Per:       "total",
							Limit:     20,
							PeriodHCL: "1m",
							Period:    time.Minute,
							Unlimited: false,
						},
					},
					ApiRateLimiterMaxQuotas: ratelimit.DefaultLimiterMaxQuotas(),
					MaxPageSizeRaw:          nil,
					MaxPageSize:             0,
				},
				DevController:           false,
				DevUiPassthroughDir:     "",
				DevControllerKey:        "",
				DevWorkerAuthKey:        "",
				DevBsrKey:               "",
				DevWorkerAuthStorageKey: "",
				DevRecoveryKey:          "",
				Eventing: &event.EventerConfig{
					AuditEnabled:        false,
					ObservationsEnabled: true,
					SysEventsEnabled:    true,
					Sinks: []*event.SinkConfig{
						{
							Name:           "default",
							Description:    "",
							EventTypes:     []event.Type{"*"},
							EventSourceUrl: "",
							AllowFilters:   nil,
							DenyFilters:    nil,
							Format:         event.JSONSinkFormat,
							Type:           event.StderrSink,
							StderrConfig:   nil,
							FileConfig:     nil,
							WriterConfig:   nil,
							AuditConfig: &event.AuditConfig{
								FilterOverrides: map[event.DataClassification]event.FilterOperation{
									event.SecretClassification:    event.RedactOperation,
									event.SensitiveClassification: event.RedactOperation,
								},
								FilterOverridesHCL: nil,
							},
						},
					},
				},
				Plugins: config.Plugins{
					ExecutionDir: "",
				},
				HcpbClusterId: "",
			},
			nil,
		},
		{
			"SingleFileJSON",
			&config.Config{
				SharedConfig: &configutil.SharedConfig{
					EntSharedConfig: configutil.EntSharedConfig{},
					Listeners: []*listenerutil.ListenerConfig{
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "api",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"api"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              func() *bool { t := true; return &t }(),
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       []string{"*"},
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "cluster",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"cluster"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "proxy",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"proxy"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
						{
							RawConfig: map[string]any{
								"address":     "boundary",
								"purpose":     "ops",
								"tls_disable": true,
							},
							Type:                                "tcp",
							Purpose:                             []string{"ops"},
							PurposeRaw:                          nil,
							Address:                             "boundary",
							ClusterAddress:                      "",
							MaxRequestSize:                      0,
							MaxRequestSizeRaw:                   nil,
							MaxRequestDuration:                  0,
							MaxRequestDurationRaw:               nil,
							RequireRequestHeader:                false,
							RequireRequestHeaderRaw:             nil,
							TLSDisable:                          true,
							TLSDisableRaw:                       nil,
							TLSCertFile:                         "",
							TLSKeyFile:                          "",
							TLSMinVersion:                       "",
							TLSMaxVersion:                       "",
							TLSCipherSuites:                     nil,
							TLSCipherSuitesRaw:                  "",
							TLSPreferServerCipherSuites:         false,
							TLSPreferServerCipherSuitesRaw:      nil,
							TLSRequireAndVerifyClientCert:       false,
							TLSRequireAndVerifyClientCertRaw:    nil,
							TLSClientCAFile:                     "",
							TLSDisableClientCerts:               false,
							TLSDisableClientCertsRaw:            nil,
							HTTPReadTimeout:                     0,
							HTTPReadTimeoutRaw:                  nil,
							HTTPReadHeaderTimeout:               0,
							HTTPReadHeaderTimeoutRaw:            nil,
							HTTPWriteTimeout:                    0,
							HTTPWriteTimeoutRaw:                 nil,
							HTTPIdleTimeout:                     0,
							HTTPIdleTimeoutRaw:                  nil,
							ProxyProtocolBehavior:               "",
							ProxyProtocolAuthorizedAddrs:        nil,
							ProxyProtocolAuthorizedAddrsRaw:     nil,
							XForwardedForAuthorizedAddrs:        nil,
							XForwardedForAuthorizedAddrsRaw:     nil,
							XForwardedForHopSkips:               0,
							XForwardedForHopSkipsRaw:            nil,
							XForwardedForRejectNotPresent:       false,
							XForwardedForRejectNotPresentRaw:    nil,
							XForwardedForRejectNotAuthorized:    false,
							XForwardedForRejectNotAuthorizedRaw: nil,
							SocketMode:                          "",
							SocketUser:                          "",
							SocketGroup:                         "",
							Telemetry: listenerutil.ListenerTelemetry{
								UnauthenticatedMetricsAccess:    false,
								UnauthenticatedMetricsAccessRaw: nil,
							},
							RandomPort:                               false,
							CorsEnabledRaw:                           nil,
							CorsEnabled:                              nil,
							CorsDisableDefaultAllowedOriginValuesRaw: nil,
							CorsDisableDefaultAllowedOriginValues:    nil,
							CorsAllowedOrigins:                       nil,
							CorsAllowedHeaders:                       nil,
							CorsAllowedHeadersRaw:                    nil,
							CustomApiResponseHeaders:                 apiHeaders,
							CustomUiResponseHeaders:                  uiHeaders,
						},
					},
					Seals: []*configutil.KMS{
						{
							Type:             "aead",
							Purpose:          []string{"root"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "sP1fnF5Xz85RrXyELHFeZg9Ad2qt4Z4bgNHVGtD6ung=",
								"key_id":    "global_root",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"worker-auth"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_worker-auth",
							},
						},
						{
							Type:             "aead",
							Purpose:          []string{"recovery"},
							Disabled:         false,
							PluginPath:       "",
							PluginChecksum:   "",
							PluginHashMethod: "",
							Config: map[string]string{
								"aead_type": "aes-gcm",
								"key":       "8fZBjCUfN0TzjEGLQldGY4+iE9AkOvCfjh7+p0GtRBQ=",
								"key_id":    "global_recovery",
							},
						},
					},
					Entropy:                      nil,
					DisableMlock:                 true,
					DisableMlockRaw:              nil,
					Telemetry:                    nil,
					DefaultMaxRequestDuration:    0,
					DefaultMaxRequestDurationRaw: nil,
					LogFormat:                    "",
					LogLevel:                     "",
					PidFile:                      "",
					ClusterName:                  "",
				},
				Worker: &config.Worker{
					Name:                                "worker0",
					Description:                         "A worker",
					PublicAddr:                          "",
					InitialUpstreams:                    []string{"boundary:9201"},
					InitialUpstreamsRaw:                 []any{"boundary:9201"},
					Tags:                                nil,
					TagsRaw:                             nil,
					SuccessfulStatusGracePeriodDuration: 0,
					StatusCallTimeoutDuration:           0,
					AuthStoragePath:                     "",
					ControllerGeneratedActivationToken:  "",
				},
				Controller: &config.Controller{
					Name:        "controller0",
					Description: "A controller",
					Database: &config.Database{
						Url:                       "env://LOAD_TEST_BOUNDARY_POSTGRES_URL",
						MigrationUrl:              "",
						MaxOpenConnections:        0,
						MaxOpenConnectionsRaw:     nil,
						MaxIdleConnections:        nil,
						MaxIdleConnectionsRaw:     nil,
						ConnMaxIdleTime:           nil,
						ConnMaxIdleTimeDuration:   nil,
						SkipSharedLockAcquisition: false,
					},
					PublicClusterAddr: "",
					Scheduler: config.Scheduler{
						JobRunInterval:  nil,
						MonitorInterval: nil,
					},
					AuthTokenTimeToLive:             nil,
					AuthTokenTimeToLiveDuration:     0,
					AuthTokenTimeToStale:            nil,
					AuthTokenTimeToStaleDuration:    0,
					GracefulShutdownWait:            nil,
					GracefulShutdownWaitDuration:    0,
					WorkerStatusGracePeriodDuration: 0,
					LivenessTimeToStaleDuration:     0,
					ApiRateLimits:                   make(ratelimit.Configs, 0),
					ApiRateLimiterMaxQuotas:         ratelimit.DefaultLimiterMaxQuotas(),
					MaxPageSizeRaw:                  nil,
					MaxPageSize:                     0,
				},
				DevController:           false,
				DevUiPassthroughDir:     "",
				DevControllerKey:        "",
				DevWorkerAuthKey:        "",
				DevBsrKey:               "",
				DevWorkerAuthStorageKey: "",
				DevRecoveryKey:          "",
				Eventing: &event.EventerConfig{
					AuditEnabled:        false,
					ObservationsEnabled: true,
					SysEventsEnabled:    true,
					Sinks: []*event.SinkConfig{
						{
							Name:           "default",
							Description:    "",
							EventTypes:     []event.Type{"*"},
							EventSourceUrl: "",
							AllowFilters:   nil,
							DenyFilters:    nil,
							Format:         event.JSONSinkFormat,
							Type:           event.StderrSink,
							StderrConfig:   nil,
							FileConfig:     nil,
							WriterConfig:   nil,
							AuditConfig: &event.AuditConfig{
								FilterOverrides: map[event.DataClassification]event.FilterOperation{
									event.SecretClassification:    event.RedactOperation,
									event.SensitiveClassification: event.RedactOperation,
								},
								FilterOverridesHCL: nil,
							},
						},
					},
				},
				Plugins: config.Plugins{
					ExecutionDir: "",
				},
				HcpbClusterId: "",
			},
			nil,
		},
		{
			// Note that this does not error, but also does not work as expected/desired
			"MultiFileJSON",
			&config.Config{
				SharedConfig: &configutil.SharedConfig{
					EntSharedConfig:              configutil.EntSharedConfig{},
					Listeners:                    nil,
					Seals:                        nil,
					Entropy:                      nil,
					DisableMlock:                 true,
					DisableMlockRaw:              nil,
					Telemetry:                    nil,
					DefaultMaxRequestDuration:    0,
					DefaultMaxRequestDurationRaw: nil,
					LogFormat:                    "",
					LogLevel:                     "",
					PidFile:                      "",
					ClusterName:                  "",
				},
				Worker: nil,
				Controller: &config.Controller{
					Name:        "controller0",
					Description: "A controller",
					Database: &config.Database{
						Url:                       "env://LOAD_TEST_BOUNDARY_POSTGRES_URL",
						MigrationUrl:              "",
						MaxOpenConnections:        0,
						MaxOpenConnectionsRaw:     nil,
						MaxIdleConnections:        nil,
						MaxIdleConnectionsRaw:     nil,
						ConnMaxIdleTime:           nil,
						ConnMaxIdleTimeDuration:   nil,
						SkipSharedLockAcquisition: false,
					},
					PublicClusterAddr: "",
					Scheduler: config.Scheduler{
						JobRunInterval:  nil,
						MonitorInterval: nil,
					},
					AuthTokenTimeToLive:             nil,
					AuthTokenTimeToLiveDuration:     0,
					AuthTokenTimeToStale:            nil,
					AuthTokenTimeToStaleDuration:    0,
					GracefulShutdownWait:            nil,
					GracefulShutdownWaitDuration:    0,
					WorkerStatusGracePeriodDuration: 0,
					LivenessTimeToStaleDuration:     0,
					ApiRateLimits:                   make(ratelimit.Configs, 0),
					ApiRateLimiterMaxQuotas:         ratelimit.DefaultLimiterMaxQuotas(),
					MaxPageSizeRaw:                  nil,
					MaxPageSize:                     0,
				},
				DevController:           false,
				DevUiPassthroughDir:     "",
				DevControllerKey:        "",
				DevWorkerAuthKey:        "",
				DevBsrKey:               "",
				DevWorkerAuthStorageKey: "",
				DevRecoveryKey:          "",
				Eventing: &event.EventerConfig{
					AuditEnabled:        false,
					ObservationsEnabled: true,
					SysEventsEnabled:    true,
					Sinks: []*event.SinkConfig{
						{
							Name:           "default",
							Description:    "",
							EventTypes:     []event.Type{"*"},
							EventSourceUrl: "",
							AllowFilters:   nil,
							DenyFilters:    nil,
							Format:         event.JSONSinkFormat,
							Type:           event.StderrSink,
							StderrConfig:   nil,
							FileConfig:     nil,
							WriterConfig:   nil,
							AuditConfig: &event.AuditConfig{
								FilterOverrides: map[event.DataClassification]event.FilterOperation{
									event.SecretClassification:    event.RedactOperation,
									event.SensitiveClassification: event.RedactOperation,
								},
								FilterOverridesHCL: nil,
							},
						},
					},
				},
				Plugins: config.Plugins{
					ExecutionDir: "",
				},
				HcpbClusterId: "",
			},
			nil,
		},
		{
			"MultiFileMix",
			nil,
			fmt.Errorf("expected: IDENT | STRING got: LBRACE"),
		},
	}

	for _, tc := range cases {
		t.Run(tc.name, func(t *testing.T) {
			var paths []string
			files, err := os.ReadDir(fmt.Sprintf("testdata/%s", t.Name()))
			require.NoError(t, err)
			for _, file := range files {
				if !file.IsDir() {
					paths = append(paths, fmt.Sprintf("testdata/%s/%s", t.Name(), file.Name()))
				}
			}

			ctx := context.Background()
			cfg, err := config.Load(ctx, paths, "")
			if tc.expectedErr != nil {
				require.Contains(t, err.Error(), tc.expectedErr.Error())
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tc.expected, cfg)
		})
	}
}
