from contextlib import suppress
import json
from pathlib import Path
import re
import subprocess
import sys
import urllib.error
import urllib.request

REPO_DIR = Path(__file__).parent.parent
SCRIPTS_DIR = REPO_DIR / "scripts"
CONFIG_DIR = REPO_DIR / "config"
RESOURCES_DIR = REPO_DIR / "resources"
INCLUDE_PATHS = [REPO_DIR / "src"]


def cpp_expand(file, path, matched_files):
    out = ""
    for line in file:
        val = re.match(r'#include "(.*)"', line)
        if val:
            out += cpp_expand_path(val.group(1), matched_files)
        elif re.match(r"#pragma once", line):
            matched_files.add(path)
        else:
            out += line
    return out


def cpp_expand_path(path, matched_files):
    # i18n.hpp is auto-generated by converting i18n.tpl to shift-jis. This might
    # cause issues, so let's just take the original utf8-encoded template file.
    if path == "i18n.hpp":
        path = "i18n.tpl"

    if path in matched_files:
        return ""

    for include_path in INCLUDE_PATHS:
        with suppress(FileNotFoundError):
            with open(include_path / path) as f:
                return cpp_expand(f, path, matched_files)

    raise Exception("No include found for " + path + " in " + str(INCLUDE_PATHS))


def main():
    asm = subprocess.check_output(
        [
            str(SCRIPTS_DIR / "prefix" / "satsuki" / "satsuki"),
            "--mapping-file-csv",
            str(CONFIG_DIR / "mapping.csv"),
            "disassemble",
            "--att",
            "--force-address-zero",
            RESOURCES_DIR / "game.exe",
            sys.argv[1],
        ]
    ).decode("utf8")
    asm = ".att_syntax\n" + asm
    ctx = cpp_expand_path("th06.hpp", set())
    req = urllib.request.Request(
        "https://decomp.me/api/scratch",
        headers={
            "Content-Type": "application/json",
        },
        data=json.dumps(
            {
                "compiler": "msvc7.0",
                "compiler_flags": "/MT /EHsc /G5 /Gs /GS /Od /Oi /Ob1 /Op /TP",
                "context": ctx,
                "diff_flags": [],
                "diff_label": sys.argv[1],
                "libraries": [
                    {"name": "directx", "version": "8.0"},
                ],
                "platform": "win32",
                "preset": 111,
                "target_asm": asm,
            }
        ).encode("utf8"),
    )

    try:
        with urllib.request.urlopen(req) as res:
            out_data = json.load(res)
    except urllib.error.HTTPError as err:
        print(json.load(err.fp))
        raise

    print(
        "https://decomp.me/scratch/"
        + out_data["slug"]
        + "/claim?token="
        + out_data["claim_token"]
    )


if __name__ == "__main__":
    main()
