/**
******************************************************************************
* @file    rf_driver_hal_gpio.c
* @author  RF Application Team
* @brief   GPIO HAL module driver.
*          This file provides firmware functions to manage the following
*          functionalities of the General Purpose Input/Output (GPIO) peripheral:
*           + Initialization and de-initialization functions
*           + IO operation functions
*
@verbatim
==============================================================================
##### GPIO Peripheral features #####
==============================================================================
[..]
(+) Each port bit of the general-purpose I/O (GPIO) ports can be individually
configured by software in several modes:
(++) Input mode
(++) Analog mode
(++) Output mode
(++) Alternate function mode
(++) External interrupt lines

(+) During and just after reset, the alternate functions and external interrupt
lines are not active and the I/O ports are configured in input pull-up mode.

(+) All GPIO pins have weak internal pull-up and pull-down resistors, which can be
activated or not.

(+) In Output or Alternate mode, each IO can be configured on open-drain or push-pull
type and the IO speed can be selected depending on the VDD value.

(+) The microcontroller IO pins are connected to onboard peripherals/modules through a
multiplexer that allows only one peripheral alternate function (AF) connected
to an IO pin at a time. In this way, there can be no conflict between peripherals
sharing the same IO pin.

(+) All ports have external interrupt capability. To use external interrupt
lines, the port must be configured in input mode. All available GPIO pins are
connected to the 16 external interrupt lines.

(+) The external interrupt controller consists of 16 lines are connected to GPIO
for generating interrupt requests. Each input line can be independently configured
to select the interrupt and the corresponding trigger event 
(rising or falling or both or low level or high level).
Each line can also be masked independently.

##### How to use this driver #####
==============================================================================
[..]
(#) Enable the GPIO AHB clock using the following function: __HAL_RCC_GPIOx_CLK_ENABLE().

(#) Configure the GPIO pin(s) using HAL_GPIO_Init().
(++) Configure the IO mode using "Mode" member from GPIO_InitTypeDef structure
(++) Activate Pull-up, Pull-down resistor using "Pull" member from GPIO_InitTypeDef
structure.
(++) In case of Output or alternate function mode selection: the speed is
configured through "Speed" member from GPIO_InitTypeDef structure.
(++) In alternate mode is selection, the alternate function connected to the IO
is configured through "Alternate" member from GPIO_InitTypeDef structure.
(++) Analog mode is required when a pin is to be used as ADC channel.
(++) In case of external interrupt selection the "Mode" member from
GPIO_InitTypeDef structure select the type interrupt and
the corresponding trigger event (rising or falling or both or low level or high level).

(#) In case of external interrupt mode selection, configure NVIC IRQ priority
mapped to the EXTI line using HAL_NVIC_SetPriority() and enable it using
HAL_NVIC_EnableIRQ().

(#) To get the level of a pin configured in input mode use HAL_GPIO_ReadPin().

(#) To set/reset the level of a pin configured in output mode use
HAL_GPIO_WritePin()/HAL_GPIO_TogglePin().

(#) To lock pin configuration until next reset use HAL_GPIO_LockPin().

(#) During and just after reset, the alternate functions are not
active and the GPIO pins are configured in input pull-up mode (except JTAG
pins).

(#) The LSE oscillator pins OSC32_IN and OSC32_OUT can be used as general purpose
(PB12 and PB13, respectively) when the LSE oscillator is off. The LSE has
priority over the GPIO function.


@endverbatim
******************************************************************************
* @attention
*
* <h2><center>&copy; Copyright (c) 2018 STMicroelectronics. 
* All rights reserved.</center></h2>
*
* This software component is licensed by ST under BSD 3-Clause license,
* the "License"; You may not use this file except in compliance with the 
* License. You may obtain a copy of the License at:
*                        opensource.org/licenses/BSD-3-Clause
*
******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "rf_driver_hal.h"

/** @addtogroup RF_DRIVER_HAL_Driver
* @{
*/

/** @addtogroup GPIO
* @{
*/
/** MISRA C:2012 deviation rule has been granted for following rules:
* Rule-12.2 - Medium: RHS argument is in interval [0,INF] which is out of
* range of the shift operator in following API :
* HAL_GPIO_Init
* HAL_GPIO_DeInit
*/

#ifdef HAL_GPIO_MODULE_ENABLED

/* Private typedef -----------------------------------------------------------*/
/* Private defines ------------------------------------------------------------*/
/** @defgroup GPIO_Private_Constants GPIO Private Constants
* @{
*/
#define GPIO_MODE              (0x00000003u)
#define EXTI_MODE              (0x10000000u)
#define LEVEL_MODE             (0x00010000u)
#define RISING_EDGE_HIGH_LEVEL (0x00100000u)
#define FALLING_EDGE_LOW_LEVEL (0x00200000u)
#define BOTH_EDGE              (0x00300000u)
#define GPIO_OUTPUT_TYPE       (0x00000010u)

#define GPIO_NUMBER           (16u)
/**
* @}
*/

/* Private macros ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Exported functions --------------------------------------------------------*/

/** @addtogroup GPIO_Exported_Functions
* @{
*/

/** @addtogroup GPIO_Exported_Functions_Group1
*  @brief    Initialization and Configuration functions
*
@verbatim
===============================================================================
##### Initialization and de-initialization functions #####
===============================================================================

@endverbatim
* @{
*/

/**
* @brief  Initialize the GPIOx peripheral according to the specified parameters in the GPIO_Init.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Init pointer to a GPIO_InitTypeDef structure that contains
*         the configuration information for the specified GPIO peripheral.
* @retval None
*/
void HAL_GPIO_Init(GPIO_TypeDef  *GPIOx, GPIO_InitTypeDef *GPIO_Init)
{
  uint32_t position = 0x00u;
  uint32_t iocurrent;
  uint32_t temp;
  uint8_t  both_edge;
  
  /* Check the parameters */
  assert_param(IS_GPIO_ALL_INSTANCE(GPIOx));
  assert_param(IS_GPIO_PIN(GPIO_Init->Pin));
  assert_param(IS_GPIO_MODE(GPIO_Init->Mode));
  assert_param(IS_GPIO_PULL(GPIO_Init->Pull));
  
  /* Configure the port pins */
  while (((GPIO_Init->Pin) >> position) != 0x00u)
  {
    /* Get current io position */
    iocurrent = (GPIO_Init->Pin) & (1uL << position);
    
    if (iocurrent != 0x00u)
    {
      /*--------------------- GPIO Mode Configuration ------------------------*/
      /* In case of Alternate function mode selection */
      if((GPIO_Init->Mode == GPIO_MODE_AF_PP) || (GPIO_Init->Mode == GPIO_MODE_AF_OD))
      {
        /* Check the Alternate function parameters */
        assert_param(IS_GPIO_AF_INSTANCE(GPIOx));
        assert_param(IS_GPIO_AF(GPIO_Init->Alternate));
        
        /* Configure Alternate function mapped with the current IO */
        temp = GPIOx->AFR[position >> 3u];
        temp &= ~(0xFu << ((position & 0x07u) * 4u));
        temp |= ((GPIO_Init->Alternate) << ((position & 0x07u) * 4u));
        GPIOx->AFR[position >> 3u] = temp;
      }
      
      /* Configure IO Direction mode (Input, Output, Alternate or Analog) */
      temp = GPIOx->MODER;
      temp &= ~(GPIO_MODER_MODE0 << (position * 2u));
      temp |= ((GPIO_Init->Mode & GPIO_MODE) << (position * 2u));
      GPIOx->MODER = temp;
      
      /* In case of Output or Alternate function mode selection */
      if((GPIO_Init->Mode == GPIO_MODE_OUTPUT_PP) || (GPIO_Init->Mode == GPIO_MODE_AF_PP) ||
         (GPIO_Init->Mode == GPIO_MODE_OUTPUT_OD) || (GPIO_Init->Mode == GPIO_MODE_AF_OD))
      {
        /* Check the Speed parameter */
        assert_param(IS_GPIO_SPEED(GPIO_Init->Speed));
        /* Configure the IO Speed */
        temp = GPIOx->OSPEEDR;
        temp &= ~(GPIO_OSPEEDR_OSPEED0 << (position * 2u));
        temp |= (GPIO_Init->Speed << (position * 2u));
        GPIOx->OSPEEDR = temp;
        
        /* Configure the IO Output Type */
        temp = GPIOx->OTYPER;
        temp &= ~(GPIO_OTYPER_OT0 << position) ;
        temp |= (((GPIO_Init->Mode & GPIO_OUTPUT_TYPE) >> 4u) << position);
        GPIOx->OTYPER = temp;
      }
      
      /* To use the GPIO register configuration for PULL-UP PULL-DOWN and NO-PULL define USE_GPIO_PULL_REGISTER,
      * otherwise the IO PULL will be managed from the PWR peripheral block
      */
#ifdef USE_GPIO_PULL_REGISTER
      /* Activate the Pull-up or Pull down resistor for the current IO */
      temp = GPIOx->PUPDR;
      temp &= ~(GPIO_PUPDR_PUPD0 << (position * 2u));
      temp |= ((GPIO_Init->Pull) << (position * 2u));
      GPIOx->PUPDR = temp;
#else
  if (GPIOx == GPIOA)
  {
    switch (GPIO_Init->Pull)
    {
    case GPIO_NOPULL:
      LL_PWR_SetNoPullA(GPIO_Init->Pin);
      break;
    case GPIO_PULLUP:
      LL_PWR_EnablePUA(GPIO_Init->Pin);
      break;
    case GPIO_PULLDOWN:
      LL_PWR_EnablePDA(GPIO_Init->Pin);
      break;
    }
  } 
  else // GPIOB
  {
    switch (GPIO_Init->Pull)
    {
    case GPIO_NOPULL:
      LL_PWR_SetNoPullB(GPIO_Init->Pin);
      break;
    case GPIO_PULLUP:
      LL_PWR_EnablePUB(GPIO_Init->Pin);
      break;
    case GPIO_PULLDOWN:
      LL_PWR_EnablePDB(GPIO_Init->Pin);
      break;
    }
  }

#endif

      /*--------------------- EXTI Mode Configuration ------------------------*/
      /* Configure the External Interrupt for the current IO */
      if((GPIO_Init->Mode & EXTI_MODE) == EXTI_MODE)
      {
        both_edge = FALSE;
        
        /* Edge/Level line configuration */
        temp = SYSCFG->IO_DTR;
        if (GPIOx == GPIOA)
        {
          temp &= ~(1<<position);
          temp |= (((GPIO_Init->Mode & LEVEL_MODE)>>4) << position);
        }
        else
        {
          temp &= ~((1<<position) << 16);
          temp |= ((((GPIO_Init->Mode & LEVEL_MODE)>>4) << position) << 16);
        }
        SYSCFG->IO_DTR = temp;
        
        /* Both Edge line configuration */
        if (!(GPIO_Init->Mode & LEVEL_MODE))
        {
          temp = SYSCFG->IO_IBER;
          /* Clear Interrupt Edge register line configuration */ 
          if (GPIOx == GPIOA)
          {
            temp &= ~(1<<position);
          }
          else
          {
            temp &= ~((1<<position) << 16);
          }
          /* Enable both edges detection line configuration */
          if ((GPIO_Init->Mode & BOTH_EDGE) == BOTH_EDGE)
          {
            both_edge = TRUE;
            /* Clear Interrupt polarity event register */
            SYSCFG->IO_IEVR = temp;
            
            if (GPIOx == GPIOA)
            {
              temp |= (1<< position);
            }
            else
            {
              temp |= ((1<< position) << 16);
            }
          }
          SYSCFG->IO_IBER = temp;
        }
        
        /* Interrupt polarity line configuration */
        temp = SYSCFG->IO_IEVR;
        if (!both_edge) {
          /* Clear Interrupt polarity event register line configuration */ 
          if (GPIOx == GPIOA)
          {
            temp &= ~(1<<position);
          }
          else
          {
            temp &= ~((1<<position) << 16);
          }
          /* Enable rising edges detection line configuration */
          if ((GPIO_Init->Mode & RISING_EDGE_HIGH_LEVEL) == RISING_EDGE_HIGH_LEVEL)
          {
            if (GPIOx == GPIOA)
            {
              temp |= (1<< position);
            }
            else
            {
              temp |= ((1<< position) << 16);
            }
          }
        }
        SYSCFG->IO_IEVR = temp;
        
        /* Enable Interrupt line */
        __HAL_GPIO_EXTI_ENABLE_IT(GPIOx, position);
        __HAL_GPIO_EXTI_CLEAR_IT(GPIOx, position);        
      }
    }
    position++;
  }
}

/**
* @brief  De-initialize the GPIOx peripheral registers to their default reset values.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin specifies the port bit to be written.
*         This parameter can be one of GPIO_PIN_x where x can be (0..15).
* @retval None
*/
void HAL_GPIO_DeInit(GPIO_TypeDef  *GPIOx, uint32_t GPIO_Pin)
{
  uint32_t position = 0x00u;
  uint32_t iocurrent;
  uint32_t tmp;
  
  /* Check the parameters */
  assert_param(IS_GPIO_ALL_INSTANCE(GPIOx));
  assert_param(IS_GPIO_PIN(GPIO_Pin));
  
  /* Configure the port pins */
  while ((GPIO_Pin >> position) != 0x00u)
  {
    /* Get current io position */
    iocurrent = (GPIO_Pin) & (1uL << position);
    
    if (iocurrent != 0x00u)
    {
      /*------------------------- EXTI Mode Configuration --------------------*/
      /* Disable and Clear the External Interrupt for the current IO */
      __HAL_GPIO_EXTI_DISABLE_IT(GPIOx, position);
      __HAL_GPIO_EXTI_CLEAR_FLAG(GPIOx, position);
      
      if(GPIOx == GPIOA)
      {
        tmp = ~(1<<position);
      }
      else
      {
        tmp = ~((1<<position) << 16);
      }
      
      /* Clear EXTI line configuration */
      SYSCFG->IO_DTR &= tmp;
      SYSCFG->IO_IBER &= tmp;
      SYSCFG->IO_IEVR &= tmp;
      
      /*------------------------- GPIO Mode Configuration --------------------*/
      /* Configure IO in Analog Mode */
      GPIOx->MODER |= (GPIO_MODER_MODE0 << (position * 2u));
      
      /* Configure the default Alternate Function in current IO */
      GPIOx->AFR[position >> 3u] &= ~(0xFu << ((position & 0x07u) * 4u)) ;
      
      /* Configure the default value for IO Speed */
      GPIOx->OSPEEDR &= ~(GPIO_OSPEEDR_OSPEED0 << (position * 2u));
      
      /* Configure the default value IO Output Type */
      GPIOx->OTYPER  &= ~(GPIO_OTYPER_OT0 << position) ;
      
      /* Configure the default value Pull-up for the current IO */
      GPIOx->PUPDR &= ~(GPIO_PUPDR_PUPD0 << (position * 2u));
      GPIOx->PUPDR |= (GPIO_PULLUP << (position * 2u));
    }
    
    position++;
  }
}

/**
* @}
*/

/** @addtogroup GPIO_Exported_Functions_Group2
*  @brief GPIO Read, Write, Toggle, Lock and EXTI management functions.
*
@verbatim
===============================================================================
##### IO operation functions #####
===============================================================================

@endverbatim
* @{
*/

/**
* @brief  Read the specified input port pin.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin specifies the port bit to read.
*         This parameter can be GPIO_PIN_x where x can be (0..15).
* @retval The input port pin value.
*/
GPIO_PinState HAL_GPIO_ReadPin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin)
{
  GPIO_PinState bitstatus;
  
  /* Check the parameters */
  assert_param(IS_GPIO_PIN(GPIO_Pin));
  
  if ((GPIOx->IDR & GPIO_Pin) != 0x00u)
  {
    bitstatus = GPIO_PIN_SET;
  }
  else
  {
    bitstatus = GPIO_PIN_RESET;
  }
  return bitstatus;
}

/**
* @brief  Set or clear the selected data port bit.
*
* @note   This function uses GPIOx_BSRR and GPIOx_BRR registers to allow atomic read/modify
*         accesses. In this way, there is no risk of an IRQ occurring between
*         the read and the modify access.
*
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin specifies the port bit to be written.
*         This parameter can be one of GPIO_PIN_x where x can be (0..15).
* @param PinState specifies the value to be written to the selected bit.
*         This parameter can be one of the GPIO_PinState enum values:
*            @arg GPIO_PIN_RESET: to clear the port pin
*            @arg GPIO_PIN_SET: to set the port pin
* @retval None
*/
void HAL_GPIO_WritePin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin, GPIO_PinState PinState)
{
  /* Check the parameters */
  assert_param(IS_GPIO_PIN(GPIO_Pin));
  assert_param(IS_GPIO_PIN_ACTION(PinState));
  
  if(PinState != GPIO_PIN_RESET)
  {
    GPIOx->BSRR = (uint32_t)GPIO_Pin;
  }
  else
  {
    GPIOx->BRR = (uint32_t)GPIO_Pin;
  }
}

/**
* @brief  Toggle the specified GPIO pin.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin specifies the pin to be toggled.
*         This parameter can be one of GPIO_PIN_x where x can be (0..15).
* @retval None
*/
void HAL_GPIO_TogglePin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin)
{
  /* Check the parameters */
  assert_param(IS_GPIO_PIN(GPIO_Pin));
  
  if ((GPIOx->ODR & GPIO_Pin) == GPIO_Pin)
  {
    GPIOx->BSRR = (uint32_t)GPIO_Pin << GPIO_NUMBER;
  }
  else
  {
    GPIOx->BSRR = GPIO_Pin;
  }
}

/**
* @brief  Lock GPIO Pins configuration registers.
* @note   The locked registers are GPIOx_MODER, GPIOx_OTYPER, GPIOx_OSPEEDR,
*         GPIOx_PUPDR, GPIOx_AFRL and GPIOx_AFRH.
* @note   The configuration of the locked GPIO pins can no longer be modified
*         until the next reset.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin specifies the port bits to be locked.
*         This parameter can be any combination of GPIO_Pin_x where x can be (0..15).
* @retval None
*/
HAL_StatusTypeDef HAL_GPIO_LockPin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin)
{
  __IO uint32_t tmp = GPIO_LCKR_LCKK;
  
  /* Check the parameters */
  assert_param(IS_GPIO_LOCK_INSTANCE(GPIOx));
  assert_param(IS_GPIO_PIN(GPIO_Pin));
  
  /* Apply lock key write sequence */
  tmp |= GPIO_Pin;
  /* Set LCKx bit(s): LCKK='1' + LCK[15-0] */
  GPIOx->LCKR = tmp;
  /* Reset LCKx bit(s): LCKK='0' + LCK[15-0] */
  GPIOx->LCKR = GPIO_Pin;
  /* Set LCKx bit(s): LCKK='1' + LCK[15-0] */
  GPIOx->LCKR = tmp;
  /* Read LCKK bit*/
  tmp = GPIOx->LCKR;
  
  if ((GPIOx->LCKR & GPIO_LCKR_LCKK) != 0x00u)
  {
    return HAL_OK;
  }
  else
  {
    return HAL_ERROR;
  }
}

/**
* @brief  Handle EXTI interrupt request.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin Specifies the port pin connected to corresponding EXTI line.
* @retval None
*/
void HAL_GPIO_EXTI_IRQHandler(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin)
{
  /* EXTI line interrupt detected */
  if(__HAL_GPIO_EXTI_GET_IT(GPIOx, GPIO_Pin) != 0x00u)
  {
    __HAL_GPIO_EXTI_CLEAR_IT(GPIOx, GPIO_Pin);
    HAL_GPIO_EXTI_Callback(GPIOx, GPIO_Pin);
  }
}

/**
* @brief  EXTI line detection callback.
* @param GPIOx where x can be (A..B) to select the GPIO peripheral for BLUENRG_LP family
* @param GPIO_Pin Specifies the port pin connected to corresponding EXTI line.
* @retval None
*/
WEAK_FUNCTION(void HAL_GPIO_EXTI_Callback(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin))
{
  /* Prevent unused argument(s) compilation warning */
  UNUSED(GPIO_Pin);
  
  /* NOTE: This function should not be modified, when the callback is needed,
  the HAL_GPIO_EXTI_Callback could be implemented in the user file
  */
}

/**
* @}
*/


/**
* @}
*/

#endif /* HAL_GPIO_MODULE_ENABLED */
/**
* @}
*/

/**
* @}
*/

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
