/**
  ******************************************************************************
  * @file    rf_driver_hal_gpio.h
  * @author  RF Application Team
  * @brief   Header file of GPIO HAL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2018 STMicroelectronics. 
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the 
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef RF_DRIVER_HAL_GPIO_H
#define RF_DRIVER_HAL_GPIO_H

#ifdef __cplusplus
 extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "rf_driver_hal_def.h"

/** @addtogroup RF_DRIVER_HAL_Driver
  * @{
  */

/** @defgroup GPIO GPIO
  * @brief GPIO HAL module driver
  * @{
  */

/* Exported types ------------------------------------------------------------*/

/** @defgroup GPIO_Exported_Types GPIO Exported Types
  * @{
  */
/**
  * @brief   GPIO Init structure definition
  */
typedef struct
{
  uint32_t Pin;        /*!< Specifies the GPIO pins to be configured.
                           This parameter can be any value of @ref GPIO_pins */

  uint32_t Mode;       /*!< Specifies the operating mode for the selected pins.
                           This parameter can be a value of @ref GPIO_mode */

  uint32_t Pull;       /*!< Specifies the Pull-up or Pull-Down activation for the selected pins.
                           This parameter can be a value of @ref GPIO_pull */

  uint32_t Speed;      /*!< Specifies the speed for the selected pins.
                           This parameter can be a value of @ref GPIO_speed */

  uint32_t Alternate;  /*!< Peripheral to be connected to the selected pins 
                            This parameter can be a value of @ref GPIOEx_Alternate_function_selection */
}GPIO_InitTypeDef;

/**
  * @brief  GPIO Bit SET and Bit RESET enumeration
  */
typedef enum
{
  GPIO_PIN_RESET = 0U,
  GPIO_PIN_SET
}GPIO_PinState;
/**
  * @}
  */

/* Exported constants --------------------------------------------------------*/
/** @defgroup GPIO_Exported_Constants GPIO Exported Constants
  * @{
  */
/** @defgroup GPIO_pins GPIO pins
  * @{
  */
#define GPIO_PIN_0                 ((uint16_t)0x0001)  /* Pin 0 selected    */
#define GPIO_PIN_1                 ((uint16_t)0x0002)  /* Pin 1 selected    */
#define GPIO_PIN_2                 ((uint16_t)0x0004)  /* Pin 2 selected    */
#define GPIO_PIN_3                 ((uint16_t)0x0008)  /* Pin 3 selected    */
#define GPIO_PIN_4                 ((uint16_t)0x0010)  /* Pin 4 selected    */
#define GPIO_PIN_5                 ((uint16_t)0x0020)  /* Pin 5 selected    */
#define GPIO_PIN_6                 ((uint16_t)0x0040)  /* Pin 6 selected    */
#define GPIO_PIN_7                 ((uint16_t)0x0080)  /* Pin 7 selected    */
#define GPIO_PIN_8                 ((uint16_t)0x0100)  /* Pin 8 selected    */
#define GPIO_PIN_9                 ((uint16_t)0x0200)  /* Pin 9 selected    */
#define GPIO_PIN_10                ((uint16_t)0x0400)  /* Pin 10 selected   */
#define GPIO_PIN_11                ((uint16_t)0x0800)  /* Pin 11 selected   */
#define GPIO_PIN_12                ((uint16_t)0x1000)  /* Pin 12 selected   */
#define GPIO_PIN_13                ((uint16_t)0x2000)  /* Pin 13 selected   */
#define GPIO_PIN_14                ((uint16_t)0x4000)  /* Pin 14 selected   */
#define GPIO_PIN_15                ((uint16_t)0x8000)  /* Pin 15 selected   */
#define GPIO_PIN_All               ((uint16_t)0xFFFF)  /* All pins selected */

#define GPIO_PIN_MASK              ((uint32_t)0x0000FFFF) /* PIN mask for assert test */
/**
  * @}
  */

/** @defgroup GPIO_mode GPIO mode
  * @brief GPIO Configuration Mode
  *        Elements values convention: 0xX0yz00YZ
  *           - X  : GPIO mode or EXTI Mode
  *           - y  : External IT trigger detection
  *           - z  : IO configuration on External IT Edge/Level
  *           - Y  : Output type (Push Pull or Open Drain)
  *           - Z  : IO Direction mode (Input, Output, Alternate or Analog)
  * @{
  */
#define  GPIO_MODE_INPUT                        (0x00000000u)   /*!< Input Floating Mode                   */
#define  GPIO_MODE_OUTPUT_PP                    (0x00000001u)   /*!< Output Push Pull Mode                 */
#define  GPIO_MODE_OUTPUT_OD                    (0x00000011u)   /*!< Output Open Drain Mode                */
#define  GPIO_MODE_AF_PP                        (0x00000002u)   /*!< Alternate Function Push Pull Mode     */
#define  GPIO_MODE_AF_OD                        (0x00000012u)   /*!< Alternate Function Open Drain Mode    */
#define  GPIO_MODE_ANALOG                       (0x00000003u)   /*!< Analog Mode  */
#define  GPIO_MODE_IT_RISING                    (0x10100000u)   /*!< External Interrupt Mode with Rising edge trigger detection  */
#define  GPIO_MODE_IT_FALLING                   (0x10200000u)   /*!< External Interrupt Mode with Falling edge trigger detection */
#define  GPIO_MODE_IT_RISING_FALLING            (0x10300000u)   /*!< External Interrupt Mode with Rising/Falling edge trigger detection  */
#define  GPIO_MODE_IT_HIGH_LEVEL                (0x10110000u)   /*!< External Interrupt Mode with High level trigger detection           */
#define  GPIO_MODE_IT_LOW_LEVEL                 (0x10210000u)   /*!< External Interrupt Mode with Low level trigger detection            */
/**
  * @}
  */

/** @defgroup GPIO_speed GPIO speed
  * @brief GPIO Output Maximum frequency
  * @{
  */
#define  GPIO_SPEED_FREQ_LOW        (0x00000000u)  /*!< Low speed       */
#define  GPIO_SPEED_FREQ_MEDIUM     (0x00000001u)  /*!< Medium speed    */
#define  GPIO_SPEED_FREQ_HIGH       (0x00000002u)  /*!< High speed      */
#define  GPIO_SPEED_FREQ_VERY_HIGH  (0x00000003u)  /*!< Very high speed */
/**
  * @}
  */

 /** @defgroup GPIO_pull GPIO pull
   * @brief GPIO Pull-Up or Pull-Down Activation
   * @{
   */
#define  GPIO_NOPULL        (0x00000000u)   /*!< No Pull-up or Pull-down activation  */
#define  GPIO_PULLUP        (0x00000001u)   /*!< Pull-up activation                  */
#define  GPIO_PULLDOWN      (0x00000002u)   /*!< Pull-down activation                */
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/** @defgroup GPIO_Exported_Macros GPIO Exported Macros
  * @{
  */

/**
  * @brief  Enable the specified EXTI interrupt line.
  * @param __EXTI_PORT__ specifies the EXTI port.
            This parameter can be GPIOA or GPIOB.
  * @param __EXTI_LINE__ specifies the EXTI line to enable.
  *         This parameter can be GPIO_PIN_x where x can be(0..15)
  * @retval None.
  */
#define __HAL_GPIO_EXTI_ENABLE_IT(__EXTI_PORT__, __EXTI_LINE__)   ((((uint32_t)__EXTI_PORT__) == ((uint32_t)GPIOA)) ? (SYSCFG->IO_IER | (__EXTI_LINE__)) :  (SYSCFG->IO_IER | ((__EXTI_LINE__) << 16)))


  /**
  * @brief  Disable the specified EXTI interrupt line.
  * @param __EXTI_PORT__ specifies the EXTI port.
            This parameter can be GPIOA or GPIOB.
  * @param __EXTI_LINE__ specifies the EXTI line to disable.
  *         This parameter can be GPIO_PIN_x where x can be(0..15)
  * @retval None.
  */
#define __HAL_GPIO_EXTI_DISABLE_IT(__EXTI_PORT__, __EXTI_LINE__)   ((((uint32_t)__EXTI_PORT__) == ((uint32_t)GPIOA)) ? (SYSCFG->IO_IER & ~(__EXTI_LINE__)) :  (SYSCFG->IO_IER & (~((__EXTI_LINE__) << 16))))

/**
  * @brief  Check whether the specified EXTI line pending flag is set or not.
  * @param __EXTI_PORT__ specifies the EXTI port to check.
            This parameter can be GPIOA or GPIOB.
  * @param __EXTI_LINE__ specifies the EXTI line flag to check.
  *         This parameter can be GPIO_PIN_x where x can be(0..15)
  * @retval The new state of __EXTI_LINE__ (SET or RESET).
  */
#define __HAL_GPIO_EXTI_GET_FLAG(__EXTI_PORT__, __EXTI_LINE__)   ((((uint32_t)__EXTI_PORT__) == ((uint32_t)GPIOA)) ? (SYSCFG->IO_ISCR & (__EXTI_LINE__)) :  (SYSCFG->IO_ISCR & ((__EXTI_LINE__) << 16)))

/**
  * @brief  Clear the EXTI's line pending flags.
  * @param __EXTI_PORT__ specifies the EXTI port to check.
            This parameter can be GPIOA or GPIOB.
  * @param __EXTI_LINE__ specifies the EXTI lines flags to clear.
  *         This parameter can be any combination of GPIO_PIN_x where x can be (0..15)
  * @retval None
  */
#define __HAL_GPIO_EXTI_CLEAR_FLAG(__EXTI_PORT__, __EXTI_LINE__) ((((uint32_t)__EXTI_PORT__) == ((uint32_t)GPIOA)) ? (SYSCFG->IO_ISCR = (__EXTI_LINE__)) :  (SYSCFG->IO_ISCR = ((__EXTI_LINE__) << 16)))

/**
  * @brief  Check whether the specified EXTI line is asserted or not.
  * @param __EXTI_PORT__ specifies the EXTI port to check.
            This parameter can be GPIOA or GPIOB.
  * @param __EXTI_LINE__ specifies the EXTI line to check.
  *          This parameter can be GPIO_PIN_x where x can be(0..15)
  * @retval The new state of __EXTI_LINE__ (SET or RESET).
  */
#define __HAL_GPIO_EXTI_GET_IT(__EXTI_PORT__, __EXTI_LINE__)     ((((uint32_t)__EXTI_PORT__) == ((uint32_t)GPIOA)) ? (SYSCFG->IO_ISCR & (__EXTI_LINE__)) :  (SYSCFG->IO_ISCR & ((__EXTI_LINE__) << 16)))

/**
  * @brief  Clear the EXTI's line pending bits.
  * @param __EXTI_PORT__ specifies the EXTI port to check.
            This parameter can be GPIOA or GPIOB.
  * @param __EXTI_LINE__ specifies the EXTI lines to clear.
  *          This parameter can be any combination of GPIO_PIN_x where x can be (0..15)
  * @retval None
  */
#define __HAL_GPIO_EXTI_CLEAR_IT(__EXTI_PORT__, __EXTI_LINE__)   ((((uint32_t)__EXTI_PORT__) == ((uint32_t)GPIOA)) ? (SYSCFG->IO_ISCR = (__EXTI_LINE__)) :  (SYSCFG->IO_ISCR = ((__EXTI_LINE__) << 16)))

/**
  * @}
  */

/* Private macros ------------------------------------------------------------*/
/** @defgroup GPIO_Private_Macros GPIO Private Macros
  * @{
  */
#define IS_GPIO_PIN_ACTION(ACTION)  (((ACTION) == GPIO_PIN_RESET) || ((ACTION) == GPIO_PIN_SET))

#define IS_GPIO_PIN(__PIN__)        ((((__PIN__) & GPIO_PIN_MASK) != 0x00u) &&\
                                     (((__PIN__) & ~GPIO_PIN_MASK) == 0x00u))

#define IS_GPIO_MODE(__MODE__)      (((__MODE__) == GPIO_MODE_INPUT)              ||\
                                     ((__MODE__) == GPIO_MODE_OUTPUT_PP)          ||\
                                     ((__MODE__) == GPIO_MODE_OUTPUT_OD)          ||\
                                     ((__MODE__) == GPIO_MODE_AF_PP)              ||\
                                     ((__MODE__) == GPIO_MODE_AF_OD)              ||\
                                     ((__MODE__) == GPIO_MODE_IT_RISING)          ||\
                                     ((__MODE__) == GPIO_MODE_IT_FALLING)         ||\
                                     ((__MODE__) == GPIO_MODE_IT_RISING_FALLING)  ||\
                                     ((__MODE__) == GPIO_MODE_IT_HIGH_LEVEL)      ||\
                                     ((__MODE__) == GPIO_MODE_IT_LOW_LEVEL)       ||\
                                     ((__MODE__) == GPIO_MODE_ANALOG))

#define IS_GPIO_SPEED(__SPEED__)    (((__SPEED__) == GPIO_SPEED_FREQ_LOW)       ||\
                                     ((__SPEED__) == GPIO_SPEED_FREQ_MEDIUM)    ||\
                                     ((__SPEED__) == GPIO_SPEED_FREQ_HIGH)      ||\
                                     ((__SPEED__) == GPIO_SPEED_FREQ_VERY_HIGH))

#define IS_GPIO_PULL(__PULL__)      (((__PULL__) == GPIO_NOPULL)   ||\
                                     ((__PULL__) == GPIO_PULLUP)   || \
                                     ((__PULL__) == GPIO_PULLDOWN))
/**
  * @}
  */

/* Include GPIO HAL Extended module */
#include "rf_driver_hal_gpio_ex.h"

/* Exported functions --------------------------------------------------------*/ 
/** @defgroup GPIO_Exported_Functions GPIO Exported Functions
 *  @brief    GPIO Exported Functions
  * @{
  */

/** @defgroup GPIO_Exported_Functions_Group1 Initialization/de-initialization functions 
 *  @brief    Initialization and Configuration functions
 * @{
 */
     
/* Initialization and de-initialization functions *****************************/
void              HAL_GPIO_Init(GPIO_TypeDef  *GPIOx, GPIO_InitTypeDef *GPIO_Init);
void              HAL_GPIO_DeInit(GPIO_TypeDef  *GPIOx, uint32_t GPIO_Pin);

/**
  * @}
  */

/** @defgroup GPIO_Exported_Functions_Group2 IO operation functions 
 *  @brief    IO operation functions
 * @{
 */
   
/* IO operation functions *****************************************************/
GPIO_PinState     HAL_GPIO_ReadPin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin);
void              HAL_GPIO_WritePin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin, GPIO_PinState PinState);
void              HAL_GPIO_TogglePin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin);
HAL_StatusTypeDef HAL_GPIO_LockPin(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin);
void              HAL_GPIO_EXTI_IRQHandler(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin);
void              HAL_GPIO_EXTI_Callback(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */ 

/**
  * @}
  */  
  
#ifdef __cplusplus
}
#endif

#endif /* RF_DRIVER_HAL_GPIO_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
