# `TRY005` - Define `__reduce__` to make exception pickable

## Why is it bad

When using multiprocessing (or anything that serializes with Pickle), and you raise an exception [Pickle uses `__reduce__`](https://docs.python.org/2/library/pickle.html#object.__reduce__) to serialize it.

It breaks if your exception takes custom args (not string or not optional).

[Stack Overflow question](https://stackoverflow.com/questions/16244923/how-to-make-a-custom-exception-class-with-multiple-init-args-pickleable)

## How to enable it

Since not every project would care about it, this is an optional violation that can be enabled through `check_pickable`.

```toml
[tool.tryceratops]
check_pickable = true
```

## How it looks like

```py
class ManyArgsMissingReduce(Exception):
    def __init__(self, val1: str, val2: str) -> None:  # Requires pickable
        super().__init__(f"{val1} {val2}")


class CustomMissingReduce(Exception):
    def __init__(self, age: int) -> None:  # Requires pickable
        super().__init__(f"You're not old enough: {age}")
```

## How it should be

```py
# Generic implementation:
class GenericReduceException(Exception)
    def __init__(self, *args, **kwargs) -> None:
        self.args = tuple([*args, *kwargs.values()]) # Saves all args/kwargs
        super().__init__(*args)

    def __reduce__(self) -> str | tuple[Any, ...]:
        return (self.__class__, self.args) # Return them here

# You can also be a bit more verbose:
class ManyArgsWITHReduce(Exception):
    def __init__(self, val1: str, val2: str) -> None:
        self.val1, self.val2 = val1, val2
        super().__init__(f"{val1} {val2}")

    def __reduce__(self) -> str | tuple[Any, ...]:
        return (ManyArgsWITHReduce, (self.val1, self.val2))
```


## When this is fine

This is ok if you don't care about pickable exceptions 🤷 or either if you have an exception:

- Without a custom `__init__` defined; or
- With `__init__` that receives only one string argument
