/*
 * Copyright (c) 2017 Dmitry Osipenko
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <math.h>
#include <stdbool.h>
#include <unistd.h>

#include "grate.h"
#include "matrix.h"
#include "tgr_3d.xml.h"

static const float vertices[] = {
	-0.5f, -0.5f, 0.0f, 1.0f,
	 0.5f, -0.5f, 0.0f, 1.0f,
	 0.5f,  0.5f, 0.0f, 1.0f,
	-0.5f,  0.5f, 0.0f, 1.0f,
};

static const float uv[] = {
	0.0f, 0.0f,
	1.0f, 0.0f,
	1.0f, 1.0f,
	0.0f, 1.0f,
};

static const unsigned short indices[] = {
	0, 1, 2,
	0, 2, 3,
};

static const char *texfilter[] = {
	[GRATE_TEXTURE_NEAREST] = "NEAREST",
	[GRATE_TEXTURE_LINEAR] = "LINEAR",
	[GRATE_TEXTURE_NEAREST_MIPMAP_NEAREST] = "NEAREST_MIPMAP_NEAREST",
	[GRATE_TEXTURE_LINEAR_MIPMAP_NEAREST] = "LINEAR_MIPMAP_NEAREST",
	[GRATE_TEXTURE_NEAREST_MIPMAP_LINEAR] = "NEAREST_MIPMAP_LINEAR",
	[GRATE_TEXTURE_LINEAR_MIPMAP_LINEAR] = "LINEAR_MIPMAP_LINEAR",
};

static const char *mipdesc[] = {
	"Autogenerated",
	"Manually loaded",
	"Unavailable",
};

static struct grate_profile *profile;
static struct grate_framebuffer *fb;
static struct grate_options options;
static struct grate *grate;
static struct grate_3d_ctx *ctx;
static struct grate_font *font;
static struct grate_program *quad_program;
static struct grate_texture *tex[3];
static struct host1x_bo *idx_bo;
static struct mat4 projection;
static float font_scale;
static float elapsed;
static int active_tex;
static int location;
static int max_lod;

static void test1_lod_bias(enum grate_textute_filter min_filter,
			   enum grate_textute_filter mag_filter)
{
	float bias = powf(sinf(elapsed * 0.3f), 2.0f) * 13.0f - 1.0f;
	struct mat4 mvp, transform;

	mat4_translate(&transform, 0.0f, 0.0f, -1.0f);
	mat4_multiply(&mvp, &projection, &transform);

	location = grate_get_vertex_uniform_location(quad_program, "mvp");
	grate_3d_ctx_set_vertex_mat4_uniform(ctx, location, &mvp);

	grate_texture_set_min_filter(tex[active_tex], min_filter);
	grate_texture_set_mag_filter(tex[active_tex], mag_filter);
	grate_texture_set_max_lod(tex[active_tex], max_lod);
	grate_3d_ctx_bind_texture(ctx, 0, tex[active_tex]);

	location = grate_get_fragment_uniform_location(quad_program, "lod_bias");
	grate_3d_ctx_set_fragment_float_uniform(ctx, location, bias);

	grate_3d_draw_elements(ctx, TGR3D_PRIMITIVE_TYPE_TRIANGLES,
			       idx_bo, TGR3D_INDEX_MODE_UINT16,
			       ARRAY_SIZE(indices));
	grate_flush(grate);

	grate_3d_printf(grate, ctx, font, 1, -0.85f, 0.85f, font_scale,
			"LOD BIAS: %f\nMAX LOD: %d\nMIN: %s\nMAG: %s\nMIPMAP: %s\n"
			"Texture id: %d",
			bias, max_lod,
			texfilter[min_filter], texfilter[mag_filter],
			mipdesc[active_tex], active_tex);

	grate_3d_printf(grate, ctx, font, 1, 0.55f, 0.85f, font_scale, "Test #1");
}

static void test2_scale(enum grate_textute_filter min_filter,
			enum grate_textute_filter mag_filter)
{
	struct mat4 mvp, modelview, transform, scale;

	mat4_scale(&scale, 4.0f, 4.0f, 1.0f);
	mat4_translate(&transform, 0.0f, 0.0f,
		       -1.0f - powf(1.0f / cosf(elapsed * 0.3f), 4.0f));
	mat4_multiply(&modelview, &transform, &scale);
	mat4_multiply(&mvp, &projection, &modelview);

	location = grate_get_vertex_uniform_location(quad_program, "mvp");
	grate_3d_ctx_set_vertex_mat4_uniform(ctx, location, &mvp);

	grate_texture_set_min_filter(tex[active_tex], min_filter);
	grate_texture_set_mag_filter(tex[active_tex], mag_filter);
	grate_texture_set_max_lod(tex[active_tex], max_lod);
	grate_3d_ctx_bind_texture(ctx, 0, tex[active_tex]);

	location = grate_get_fragment_uniform_location(quad_program, "lod_bias");
	grate_3d_ctx_set_fragment_float_uniform(ctx, location, 0.0f);

	grate_3d_draw_elements(ctx, TGR3D_PRIMITIVE_TYPE_TRIANGLES,
			       idx_bo, TGR3D_INDEX_MODE_UINT16,
			       ARRAY_SIZE(indices));
	grate_flush(grate);

	grate_3d_printf(grate, ctx, font, 1, -0.85f, 0.85f, font_scale,
			"LOD BIAS: 0\nMAX LOD: %d\nMIN: %s\nMAG: %s\nMIPMAP: %s\n"
			"Texture id: %d",
			max_lod, texfilter[min_filter], texfilter[mag_filter],
			mipdesc[active_tex], active_tex);

	grate_3d_printf(grate, ctx, font, 1, 0.55f, 0.85f, font_scale,
			"Test #2");
}

static void test3_mag(enum grate_textute_filter min_filter,
		      enum grate_textute_filter mag_filter)
{
	struct mat4 mvp, modelview, transform, scale;

	mat4_scale(&scale, 10.0f, 10.0f, 1.0f);
	mat4_translate(&transform, 0.0f, 0.0f,
		       -1.0f - powf(cosf(elapsed), 2.0f) * 4.0f);
	mat4_multiply(&modelview, &transform, &scale);
	mat4_multiply(&mvp, &projection, &modelview);

	location = grate_get_vertex_uniform_location(quad_program, "mvp");
	grate_3d_ctx_set_vertex_mat4_uniform(ctx, location, &mvp);

	grate_texture_set_min_filter(tex[active_tex], min_filter);
	grate_texture_set_mag_filter(tex[active_tex], mag_filter);
	grate_texture_set_max_lod(tex[active_tex], max_lod);
	grate_3d_ctx_bind_texture(ctx, 0, tex[active_tex]);

	location = grate_get_fragment_uniform_location(quad_program, "lod_bias");
	grate_3d_ctx_set_fragment_float_uniform(ctx, location, 0.0f);

	grate_3d_draw_elements(ctx, TGR3D_PRIMITIVE_TYPE_TRIANGLES,
			       idx_bo, TGR3D_INDEX_MODE_UINT16,
			       ARRAY_SIZE(indices));
	grate_flush(grate);

	grate_3d_printf(grate, ctx, font, 1, -0.85f, 0.85f, font_scale,
			"LOD BIAS: 0\nMAX LOD: %d\nMIN: %s\nMAG: %s\nMIPMAP: %s\n"
			"Texture id: %d",
			max_lod, texfilter[min_filter], texfilter[mag_filter],
			mipdesc[active_tex], active_tex);

	grate_3d_printf(grate, ctx, font, 1, 0.55f, 0.85f, font_scale,
			"Test #3");
}

static void test4_max_lod(enum grate_textute_filter min_filter,
			  enum grate_textute_filter mag_filter)
{
	float bias = 12.0f;
	struct mat4 mvp, transform;

	mat4_translate(&transform, 0.0f, 0.0f, -1.0f);
	mat4_multiply(&mvp, &projection, &transform);

	location = grate_get_vertex_uniform_location(quad_program, "mvp");
	grate_3d_ctx_set_vertex_mat4_uniform(ctx, location, &mvp);

	grate_texture_set_min_filter(tex[active_tex], min_filter);
	grate_texture_set_mag_filter(tex[active_tex], mag_filter);
	grate_texture_set_max_lod(tex[active_tex], max_lod);
	grate_3d_ctx_bind_texture(ctx, 0, tex[active_tex]);

	location = grate_get_fragment_uniform_location(quad_program, "lod_bias");
	grate_3d_ctx_set_fragment_float_uniform(ctx, location, bias);

	grate_3d_draw_elements(ctx, TGR3D_PRIMITIVE_TYPE_TRIANGLES,
			       idx_bo, TGR3D_INDEX_MODE_UINT16,
			       ARRAY_SIZE(indices));
	grate_flush(grate);

	grate_3d_printf(grate, ctx, font, 1, -0.85f, 0.85f, font_scale,
			"LOD BIAS: %f\nMAX LOD: %d\nMIN: %s\nMAG: %s\nMIPMAP: %s\n"
			"Texture id: %d",
			bias, max_lod,
			texfilter[min_filter], texfilter[mag_filter],
			mipdesc[active_tex], active_tex);

	grate_3d_printf(grate, ctx, font, 1, 0.55f, 0.85f, font_scale,
			"Test #4");
}

int main(int argc, char *argv[])
{
	struct grate_shader *quad_vs, *quad_fs, *quad_linker;
	struct host1x_pixelbuffer *pixbuf;
	struct host1x_bo *bo;
	float aspect;

	grate_init_data_path(argv[0]);

	if (!grate_parse_command_line(&options, argc, argv))
		return 1;

	grate = grate_init(&options);
	if (!grate)
		return 1;

	fb = grate_framebuffer_create(grate, options.width, options.height,
				      PIX_BUF_FMT_RGBA8888,
				      PIX_BUF_LAYOUT_TILED_16x16,
				      GRATE_DOUBLE_BUFFERED);
	if (!fb) {
		fprintf(stderr, "grate_framebuffer_create() failed\n");
		return 1;
	}

	aspect = options.width / (float)options.height;
	mat4_perspective(&projection, 60.0f, aspect, 1.0f, 2048.0f);

	grate_clear_color(grate, 0.0, 0.0, 1.0, 1.0f);
	grate_bind_framebuffer(grate, fb);

	/* Prepare shaders */

	quad_vs = grate_shader_parse_vertex_asm_from_file(
				"tests/grate/asm/filter_quad_vs.txt");
	if (!quad_vs) {
		fprintf(stderr, "filter_quad_vs assembler parse failed\n");
		return 1;
	}

	quad_fs = grate_shader_parse_fragment_asm_from_file(
				"tests/grate/asm/filter_quad_fs.txt");
	if (!quad_fs) {
		fprintf(stderr, "filter_quad_fs assembler parse failed\n");
		return 1;
	}

	quad_linker = grate_shader_parse_linker_asm_from_file(
				"tests/grate/asm/filter_quad_linker.txt");
	if (!quad_linker) {
		fprintf(stderr, "filter_quad_linker assembler parse failed\n");
		return 1;
	}

	quad_program = grate_program_new(grate, quad_vs, quad_fs, quad_linker);
	if (!quad_program) {
		fprintf(stderr, "grate_program_new() failed\n");
		return 1;
	}

	grate_program_link(quad_program);

	/* Load font */

	font = grate_create_font(grate, "data/font.png", "data/font.fnt");
	if (!font) {
		fprintf(stderr, "failed to load font\n");
		return 1;
	}

	font_scale = options.width / 700.0f;

	/* Setup context */

	ctx = grate_3d_alloc_ctx(grate);

	grate_3d_ctx_bind_program(ctx, quad_program);
	grate_3d_ctx_set_depth_range(ctx, 0.0f, 1.0f);
	grate_3d_ctx_set_dither(ctx, 0x779);
	grate_3d_ctx_set_point_params(ctx, 0x1401);
	grate_3d_ctx_set_point_size(ctx, 1.0f);
	grate_3d_ctx_set_line_params(ctx, 0x2);
	grate_3d_ctx_set_line_width(ctx, 1.0f);
	grate_3d_ctx_set_viewport_bias(ctx, 0.0f, 0.0f, 0.5f);
	grate_3d_ctx_set_viewport_scale(ctx, options.width, options.height, 0.5f);
	grate_3d_ctx_use_guardband(ctx, true);
	grate_3d_ctx_set_front_direction_is_cw(ctx, false);
	grate_3d_ctx_set_cull_face(ctx, GRATE_3D_CTX_CULL_FACE_NONE);
	grate_3d_ctx_set_scissor(ctx, 0, options.width, 0, options.height);
	grate_3d_ctx_set_point_coord_range(ctx, 0.0f, 1.0f, 0.0f, 1.0f);
	grate_3d_ctx_set_polygon_offset(ctx, 0.0f, 0.0f);
	grate_3d_ctx_set_provoking_vtx_last(ctx, true);

	/* Setup quad attributes */

	location = grate_get_attribute_location(quad_program, "position");
	bo = grate_create_attrib_bo_from_data(grate, vertices);
	grate_3d_ctx_vertex_attrib_float_pointer(ctx, location, 4, bo);
	grate_3d_ctx_enable_vertex_attrib_array(ctx, location);

	location = grate_get_attribute_location(quad_program, "texcoord");
	bo = grate_create_attrib_bo_from_data(grate, uv);
	grate_3d_ctx_vertex_attrib_float_pointer(ctx, location, 2, bo);
	grate_3d_ctx_enable_vertex_attrib_array(ctx, location);

	/* Setup render target */

	grate_3d_ctx_enable_render_target(ctx, 1);

	/* Setup textures */
	tex[0] = grate_create_texture(grate, 2048, 700,
				      PIX_BUF_FMT_RGBA8888,
				      PIX_BUF_LAYOUT_LINEAR);
	grate_texture_load(grate, tex[0], "data/tegra.png");
	grate_texture_generate_mipmap(grate, tex[0]);

	tex[1] = grate_create_texture(grate, 1024, 1024,
				      PIX_BUF_FMT_RGBA8888,
				      PIX_BUF_LAYOUT_LINEAR);
	grate_texture_load(grate, tex[1], "data/miptest/nomipmap.png");
	grate_texture_load_miplevel(grate, tex[1], 0, "data/miptest/1024x1024.png");
	grate_texture_load_miplevel(grate, tex[1], 1, "data/miptest/512x512.png");
	grate_texture_load_miplevel(grate, tex[1], 2, "data/miptest/256x256.png");
	grate_texture_load_miplevel(grate, tex[1], 3, "data/miptest/128x128.png");
	grate_texture_load_miplevel(grate, tex[1], 4, "data/miptest/64x64.png");
	grate_texture_load_miplevel(grate, tex[1], 5, "data/miptest/32x32.png");
	grate_texture_load_miplevel(grate, tex[1], 6, "data/miptest/16x16.png");
	grate_texture_load_miplevel(grate, tex[1], 7, "data/miptest/8x8.png");
	grate_texture_load_miplevel(grate, tex[1], 8, "data/miptest/4x4.png");
	grate_texture_load_miplevel(grate, tex[1], 9, "data/miptest/2x2.png");
	grate_texture_load_miplevel(grate, tex[1], 10, "data/miptest/1x1.png");

	tex[2] = grate_create_texture(grate, 256, 256,
				      PIX_BUF_FMT_RGBA8888,
				      PIX_BUF_LAYOUT_LINEAR);
	grate_texture_load(grate, tex[2], "data/tegra.png");
	grate_texture_generate_mipmap(grate, tex[2]);

	/* Create indices BO */

	idx_bo = grate_create_attrib_bo_from_data(grate, indices);

	profile = grate_profile_start(grate);

	while (true) {
		grate_clear(grate);

		pixbuf = grate_get_draw_pixbuf(fb);
		grate_3d_ctx_bind_render_target(ctx, 1, pixbuf);

		active_tex = ((int)elapsed / 135) & 1;
		max_lod = 11;

		switch ((int)elapsed % 135) {
		case 0 ... 9:
			test1_lod_bias(GRATE_TEXTURE_NEAREST_MIPMAP_NEAREST,
				       GRATE_TEXTURE_LINEAR);
			break;
		case 10 ... 19:
			test1_lod_bias(GRATE_TEXTURE_LINEAR_MIPMAP_NEAREST,
				       GRATE_TEXTURE_LINEAR);
			break;
		case 20 ... 29:
			test1_lod_bias(GRATE_TEXTURE_NEAREST_MIPMAP_LINEAR,
				       GRATE_TEXTURE_LINEAR);
			break;
		case 30 ... 39:
			test1_lod_bias(GRATE_TEXTURE_LINEAR_MIPMAP_LINEAR,
				       GRATE_TEXTURE_LINEAR);
			break;
		case 40 ... 49:
			test2_scale(GRATE_TEXTURE_NEAREST,
				    GRATE_TEXTURE_LINEAR);
			break;
		case 50 ... 59:
			test2_scale(GRATE_TEXTURE_LINEAR,
				    GRATE_TEXTURE_LINEAR);
			break;
		case 60 ... 64:
			test2_scale(GRATE_TEXTURE_NEAREST_MIPMAP_NEAREST,
				    GRATE_TEXTURE_LINEAR);
			break;
		case 65 ... 69:
			test2_scale(GRATE_TEXTURE_NEAREST_MIPMAP_NEAREST,
				    GRATE_TEXTURE_NEAREST);
			break;
		case 70 ... 74:
			test2_scale(GRATE_TEXTURE_LINEAR_MIPMAP_NEAREST,
				    GRATE_TEXTURE_LINEAR);
			break;
		case 75 ... 79:
			test2_scale(GRATE_TEXTURE_LINEAR_MIPMAP_NEAREST,
				    GRATE_TEXTURE_NEAREST);
			break;
		case 80 ... 84:
			test2_scale(GRATE_TEXTURE_NEAREST_MIPMAP_LINEAR,
				    GRATE_TEXTURE_LINEAR);
			break;
		case 85 ... 89:
			test2_scale(GRATE_TEXTURE_NEAREST_MIPMAP_LINEAR,
				    GRATE_TEXTURE_NEAREST);
			break;
		case 90 ... 94:
			test2_scale(GRATE_TEXTURE_LINEAR_MIPMAP_LINEAR,
				    GRATE_TEXTURE_LINEAR);
			break;
		case 95 ... 99:
			test2_scale(GRATE_TEXTURE_LINEAR_MIPMAP_LINEAR,
				    GRATE_TEXTURE_NEAREST);
			break;
		case 100 ... 106:
			active_tex = 2;
			test3_mag(GRATE_TEXTURE_NEAREST, GRATE_TEXTURE_NEAREST);
			break;
		case 107 ... 113:
			active_tex = 2;
			test3_mag(GRATE_TEXTURE_NEAREST, GRATE_TEXTURE_LINEAR);
			break;
		case 114 ... 119:
			active_tex = 2;
			test3_mag(GRATE_TEXTURE_LINEAR, GRATE_TEXTURE_NEAREST);
			break;
		case 120 ... 134:
			max_lod = (powf(sinf(elapsed * 0.3f), 6.0f)) * 12;
			test4_max_lod(GRATE_TEXTURE_LINEAR_MIPMAP_LINEAR,
				      GRATE_TEXTURE_LINEAR);
			break;
		}

		grate_swap_buffers(grate);

		if (grate_key_pressed(grate))
			break;

		grate_profile_sample(profile);

		elapsed = grate_profile_time_elapsed(profile);
	}

	grate_profile_finish(profile);
	grate_profile_free(profile);

	grate_exit(grate);
	return 0;
}
