/* eslint-disable @typescript-eslint/no-non-null-assertion */

import { execSync } from "child_process";
import { join as pathJoin, relative as pathRelative } from "path";
import * as fs from "fs";

const tssReactDirPath = pathJoin(__dirname, "..", "..");

fs.writeFileSync(
    pathJoin(tssReactDirPath, "dist", "package.json"),
    Buffer.from(
        JSON.stringify(
            (() => {
                const packageJsonParsed = JSON.parse(
                    fs
                        .readFileSync(pathJoin(tssReactDirPath, "package.json"))
                        .toString("utf8")
                );

                return {
                    ...packageJsonParsed,
                    "main": packageJsonParsed["main"].replace(/^dist\//, ""),
                    "types": packageJsonParsed["types"].replace(/^dist\//, ""),
                    "module": packageJsonParsed["module"].replace(
                        /^dist\//,
                        ""
                    ),
                    "exports": Object.fromEntries(
                        Object.entries(packageJsonParsed["exports"]).map(
                            ([path, obj]) => [
                                path,
                                Object.fromEntries(
                                    Object.entries(
                                        obj as Record<string, string>
                                    ).map(([type, path]) => [
                                        type,
                                        path.replace(/^\.\/dist\//, "./")
                                    ])
                                )
                            ]
                        )
                    )
                };
            })(),
            null,
            2
        ),
        "utf8"
    )
);

const commonThirdPartyDeps = (() => {
    const namespaceModuleNames = ["@emotion"];
    const standaloneModuleNames = ["react", "@types/react"];

    return [
        ...namespaceModuleNames
            .map(namespaceModuleName =>
                fs
                    .readdirSync(
                        pathJoin(
                            tssReactDirPath,
                            "node_modules",
                            namespaceModuleName
                        )
                    )
                    .map(
                        submoduleName =>
                            `${namespaceModuleName}/${submoduleName}`
                    )
            )
            .reduce((prev, curr) => [...prev, ...curr], []),
        ...standaloneModuleNames
    ];
})();

const yarnHomeDirPath = pathJoin(tssReactDirPath, ".yarn_home");

fs.rmSync(yarnHomeDirPath, { "recursive": true, "force": true });

fs.mkdirSync(yarnHomeDirPath);

const execYarnLink = (params: { targetModuleName?: string; cwd: string }) => {
    const { targetModuleName, cwd } = params;

    const cmd = [
        "yarn",
        "link",
        ...(targetModuleName !== undefined ? [targetModuleName] : [])
    ].join(" ");

    console.log(`$ cd ${pathRelative(tssReactDirPath, cwd) || "."} && ${cmd}`);

    execSync(cmd, {
        cwd,
        "env": {
            ...process.env,
            "HOME": yarnHomeDirPath
        }
    });
};

const testAppNames = ["spa", "ssr", "next-appdir"] as const;

const getTestAppPath = (testAppName: typeof testAppNames[number]) =>
    pathJoin(tssReactDirPath, "src", "test", "apps", testAppName);

testAppNames.forEach(testAppName =>
    execSync("yarn install", { "cwd": getTestAppPath(testAppName) })
);

console.log("=== Linking common dependencies ===");

const total = commonThirdPartyDeps.length;
let current = 0;

commonThirdPartyDeps.forEach(commonThirdPartyDep => {
    current++;

    console.log(`${current}/${total} ${commonThirdPartyDep}`);

    const localInstallPath = pathJoin(
        ...[
            tssReactDirPath,
            "node_modules",
            ...(commonThirdPartyDep.startsWith("@")
                ? commonThirdPartyDep.split("/")
                : [commonThirdPartyDep])
        ]
    );

    execYarnLink({ "cwd": localInstallPath });

    testAppNames.forEach(testAppName =>
        execYarnLink({
            "cwd": getTestAppPath(testAppName),
            "targetModuleName": commonThirdPartyDep
        })
    );
});

console.log("=== Linking in house dependencies ===");

execYarnLink({ "cwd": pathJoin(tssReactDirPath, "dist") });

testAppNames.forEach(testAppName =>
    execYarnLink({
        "cwd": getTestAppPath(testAppName),
        "targetModuleName": "tss-react"
    })
);
