/*
 * This is the source code of Telegram for Android v. 5.x.x.
 * It is licensed under GNU GPL v. 2 or later.
 * You should have received a copy of the license in this archive (see LICENSE).
 *
 * Copyright Nikolai Kudashov, 2013-2018.
 */

package org.telegram.tgnet;

import android.graphics.Path;
import android.graphics.drawable.BitmapDrawable;
import android.os.Build;
import android.text.TextUtils;

import org.telegram.messenger.BuildVars;
import org.telegram.messenger.FileLoader;
import org.telegram.messenger.FileLog;
import org.telegram.messenger.ImageLoader;
import org.telegram.messenger.MessageObject;
import org.telegram.messenger.SvgHelper;
import org.telegram.messenger.Utilities;

import java.util.ArrayList;
import java.util.HashMap;

import androidx.annotation.Nullable;

//所有通信静态类
public class TLRPC {

    public static final int USER_FLAG_ACCESS_HASH           = 0x00000001;
    public static final int USER_FLAG_FIRST_NAME            = 0x00000002;
    public static final int USER_FLAG_LAST_NAME             = 0x00000004;
    public static final int USER_FLAG_USERNAME              = 0x00000008;
    public static final int USER_FLAG_PHONE                 = 0x00000010;
    public static final int USER_FLAG_PHOTO                 = 0x00000020;
    public static final int USER_FLAG_STATUS                = 0x00000040;
    public static final int USER_FLAG_UNUSED                = 0x00000080;
    public static final int USER_FLAG_UNUSED2               = 0x00000100;
    public static final int USER_FLAG_UNUSED3               = 0x00000200;
    //public static final int USER_FLAG_SELF                  = 0x00000400;
    //public static final int USER_FLAG_CONTACT               = 0x00000800;
    //public static final int USER_FLAG_MUTUAL_CONTACT        = 0x00001000;
    //public static final int USER_FLAG_DELETED               = 0x00002000;
    //public static final int USER_FLAG_BOT                   = 0x00004000;
    //public static final int USER_FLAG_BOT_READING_HISTORY   = 0x00008000;
    //public static final int USER_FLAG_BOT_CANT_JOIN_GROUP   = 0x00010000;
    //public static final int USER_FLAG_VERIFIED   			  = 0x00020000;

    //public static final int CHAT_FLAG_CREATOR               = 0x00000001;
    //public static final int CHAT_FLAG_USER_KICKED           = 0x00000002;
    //public static final int CHAT_FLAG_USER_LEFT             = 0x00000004;
    //public static final int CHAT_FLAG_USER_IS_EDITOR        = 0x00000008;
    //public static final int CHAT_FLAG_USER_IS_MODERATOR     = 0x00000010;
    //public static final int CHAT_FLAG_IS_BROADCAST          = 0x00000020;
    public static final int CHAT_FLAG_IS_PUBLIC             = 0x00000040;
    //public static final int CHAT_FLAG_IS_VERIFIED           = 0x00000080;

    //public static final int MESSAGE_FLAG_UNREAD             = 0x00000001;
    //public static final int MESSAGE_FLAG_OUT                = 0x00000002;
    public static final int MESSAGE_FLAG_FWD                = 0x00000004;
    public static final int MESSAGE_FLAG_REPLY              = 0x00000008;
    //public static final int MESSAGE_FLAG_MENTION            = 0x00000010;
    //public static final int MESSAGE_FLAG_CONTENT_UNREAD     = 0x00000020;
    public static final int MESSAGE_FLAG_HAS_MARKUP         = 0x00000040;
    public static final int MESSAGE_FLAG_HAS_ENTITIES       = 0x00000080;
    public static final int MESSAGE_FLAG_HAS_FROM_ID        = 0x00000100;
    public static final int MESSAGE_FLAG_HAS_MEDIA          = 0x00000200;
    public static final int MESSAGE_FLAG_HAS_VIEWS          = 0x00000400;
    public static final int MESSAGE_FLAG_HAS_BOT_ID         = 0x00000800;
    public static final int MESSAGE_FLAG_EDITED             = 0x00008000;

    public static final int LAYER = 155;

    public static class TL_stats_megagroupStats extends TLObject {
        public static int constructor = 0xef7ff916;
        public TL_statsDateRangeDays period;
        public TL_statsAbsValueAndPrev members;
        public TL_statsAbsValueAndPrev messages;
        public TL_statsAbsValueAndPrev viewers;
        public TL_statsAbsValueAndPrev posters;
        public StatsGraph growth_graph;
        public StatsGraph members_graph;
        public StatsGraph new_members_by_source_graph;
        public StatsGraph languages_graph;
        public StatsGraph messages_graph;
        public StatsGraph actions_graph;
        public StatsGraph top_hours_graph;
        public StatsGraph weekdays_graph;
        public ArrayList<TL_statsGroupTopPoster> top_posters = new ArrayList<>();
        public ArrayList<TL_statsGroupTopAdmin> top_admins = new ArrayList<>();
        public ArrayList<TL_statsGroupTopInviter> top_inviters = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_stats_megagroupStats TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_stats_megagroupStats.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_stats_megagroupStats", constructor));
                } else {
                    return null;
                }
            }
            TL_stats_megagroupStats result = new TL_stats_megagroupStats();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            period = TL_statsDateRangeDays.TLdeserialize(stream, stream.readInt32(exception), exception);
            members = TL_statsAbsValueAndPrev.TLdeserialize(stream, stream.readInt32(exception), exception);
            messages = TL_statsAbsValueAndPrev.TLdeserialize(stream, stream.readInt32(exception), exception);
            viewers = TL_statsAbsValueAndPrev.TLdeserialize(stream, stream.readInt32(exception), exception);
            posters = TL_statsAbsValueAndPrev.TLdeserialize(stream, stream.readInt32(exception), exception);
            growth_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            members_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            new_members_by_source_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            languages_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            messages_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            actions_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            top_hours_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            weekdays_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_statsGroupTopPoster object = TL_statsGroupTopPoster.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                top_posters.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_statsGroupTopAdmin object = TL_statsGroupTopAdmin.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                top_admins.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_statsGroupTopInviter object = TL_statsGroupTopInviter.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                top_inviters.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            period.serializeToStream(stream);
            members.serializeToStream(stream);
            messages.serializeToStream(stream);
            viewers.serializeToStream(stream);
            posters.serializeToStream(stream);
            growth_graph.serializeToStream(stream);
            members_graph.serializeToStream(stream);
            new_members_by_source_graph.serializeToStream(stream);
            languages_graph.serializeToStream(stream);
            messages_graph.serializeToStream(stream);
            actions_graph.serializeToStream(stream);
            top_hours_graph.serializeToStream(stream);
            weekdays_graph.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = top_posters.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                top_posters.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = top_admins.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                top_admins.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = top_inviters.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                top_inviters.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }


    public static abstract class EmailVerifyPurpose extends TLObject {

        public static EmailVerifyPurpose TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            EmailVerifyPurpose result = null;
            switch (constructor) {
                case 0x4345be73:
                    result = new TL_emailVerifyPurposeLoginSetup();
                    break;
                case 0x527d22eb:
                    result = new TL_emailVerifyPurposeLoginChange();
                    break;
                case 0xbbf51685:
                    result = new TL_emailVerifyPurposePassport();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in EmailVerifyPurpose", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_emailVerifyPurposeLoginSetup extends EmailVerifyPurpose {
        public static int constructor = 0x4345be73;

        public String phone_number;
        public String phone_code_hash;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            phone_code_hash = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(phone_code_hash);
        }
    }

    public static class TL_emailVerifyPurposeLoginChange extends EmailVerifyPurpose {
        public static int constructor = 0x527d22eb;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_emailVerifyPurposePassport extends EmailVerifyPurpose {
        public static int constructor = 0xbbf51685;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class EmailVerification extends TLObject {

        public static EmailVerification TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            EmailVerification result = null;
            switch (constructor) {
                case 0x922e55a9:
                    result = new TL_emailVerificationCode();
                    break;
                case 0xdb909ec2:
                    result = new TL_emailVerificationGoogle();
                    break;
                case 0x96d074fd:
                    result = new TL_emailVerificationApple();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in EmailVerification", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_emailVerificationCode extends EmailVerification {
        public static int constructor = 0x922e55a9;

        public String code;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            code = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(code);
        }
    }

    public static class TL_emailVerificationGoogle extends EmailVerification {
        public static int constructor = 0xdb909ec2;

        public String token;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            token = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(token);
        }
    }

    public static class TL_emailVerificationApple extends EmailVerification {
        public static int constructor = 0x96d074fd;

        public String token;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            token = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(token);
        }
    }

    public static abstract class account_EmailVerified extends TLObject {

        public static account_EmailVerified TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            account_EmailVerified result = null;
            switch (constructor) {
                case 0x2b96cd1b:
                    result = new TL_account_emailVerified();
                    break;
                case 0xe1bb0d61:
                    result = new TL_account_emailVerifiedLogin();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in account_EmailVerified", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_account_emailVerified extends account_EmailVerified {
        public static int constructor = 0x2b96cd1b;

        public String email;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            email = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(email);
        }
    }

    public static class TL_account_emailVerifiedLogin extends account_EmailVerified {
        public static int constructor = 0xe1bb0d61;

        public String email;
        public auth_SentCode sent_code;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            email = stream.readString(exception);
            sent_code = auth_SentCode.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(email);
            sent_code.serializeToStream(stream);
        }
    }

    public static class TL_chatBannedRights extends TLObject {
        public static int constructor = 0x9f120418;

        public int flags;
        public boolean view_messages;
        public boolean send_messages;
        public boolean send_media;
        public boolean send_stickers;
        public boolean send_gifs;
        public boolean send_games;
        public boolean send_inline;
        public boolean embed_links;
        public boolean send_polls;
        public boolean change_info;
        public boolean invite_users;
        public boolean pin_messages;
        public boolean manage_topics;
        public boolean send_photos;
        public boolean send_videos;
        public boolean send_roundvideos;
        public boolean send_audios;
        public boolean send_voices;
        public boolean send_docs;
        public boolean send_plain;
        public int until_date;

        public static TL_chatBannedRights TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_chatBannedRights.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_chatBannedRights", constructor));
                } else {
                    return null;
                }
            }
            TL_chatBannedRights result = new TL_chatBannedRights();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            view_messages = (flags & 1) != 0;
            send_messages = (flags & 2) != 0;
            send_media = (flags & 4) != 0;
            send_stickers = (flags & 8) != 0;
            send_gifs = (flags & 16) != 0;
            send_games = (flags & 32) != 0;
            send_inline = (flags & 64) != 0;
            embed_links = (flags & 128) != 0;
            send_polls = (flags & 256) != 0;
            change_info = (flags & 1024) != 0;
            invite_users = (flags & 32768) != 0;
            pin_messages = (flags & 131072) != 0;
            manage_topics = (flags & 262144) != 0;
            send_photos = (flags & 524288) != 0;
            send_videos = (flags & 1048576) != 0;
            send_roundvideos = (flags & 2097152) != 0;
            send_audios = (flags & 4194304) != 0;
            send_voices = (flags & 8388608) != 0;
            send_docs = (flags & 16777216) != 0;
            send_plain = (flags & 33554432) != 0;
            if (send_media) {
                send_photos = true;
                send_videos = true;
                send_roundvideos = true;
                send_audios = true;
                send_voices = true;
                send_docs = true;
            }
            until_date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            if (send_photos && send_videos && send_roundvideos && send_audios && send_voices && send_docs) {
                send_media = true;
            } else {
                send_media = false;
            }
            if (send_plain && send_media) {
                send_messages = true;
            } else {
                send_messages = false;
            }
            flags = view_messages ? (flags | 1) : (flags &~ 1);
            flags = send_messages ? (flags | 2) : (flags &~ 2);
            flags = send_media ? (flags | 4) : (flags &~ 4);
            flags = send_stickers ? (flags | 8) : (flags &~ 8);
            flags = send_gifs ? (flags | 16) : (flags &~ 16);
            flags = send_games ? (flags | 32) : (flags &~ 32);
            flags = send_inline ? (flags | 64) : (flags &~ 64);
            flags = embed_links ? (flags | 128) : (flags &~ 128);
            flags = send_polls ? (flags | 256) : (flags &~ 256);
            flags = change_info ? (flags | 1024) : (flags &~ 1024);
            flags = invite_users ? (flags | 32768) : (flags &~ 32768);
            flags = pin_messages ? (flags | 131072) : (flags &~ 131072);
            flags = manage_topics ? (flags | 262144) : (flags &~ 262144);
            flags = send_photos ? (flags | 524288) : (flags &~ 524288);
            flags = send_videos ? (flags | 1048576) : (flags &~ 1048576);
            flags = send_roundvideos ? (flags | 2097152) : (flags &~ 2097152);
            flags = send_audios ? (flags | 4194304) : (flags &~ 4194304);
            flags = send_voices ? (flags | 8388608) : (flags &~ 8388608);
            flags = send_docs ? (flags | 16777216) : (flags &~ 16777216);
            flags = send_plain ? (flags | 33554432) : (flags &~ 33554432);
            stream.writeInt32(flags);
            stream.writeInt32(until_date);
        }
    }

    public static class TL_stickers_suggestedShortName extends TLObject {
        public static int constructor = 0x85fea03f;

        public String short_name;

        public static TL_stickers_suggestedShortName TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_stickers_suggestedShortName.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_stickers_suggestedShortName", constructor));
                } else {
                    return null;
                }
            }
            TL_stickers_suggestedShortName result = new TL_stickers_suggestedShortName();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            short_name = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(short_name);
        }
    }

    public static abstract class DraftMessage extends TLObject {

        public int flags;
        public boolean no_webpage;
        public int reply_to_msg_id;
        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public int date;

        public static DraftMessage TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            DraftMessage result = null;
            switch (constructor) {
                case 0x1b0c841a:
                    result = new TL_draftMessageEmpty();
                    break;
                case 0xba4baec5:
                    result = new TL_draftMessageEmpty_layer81();
                    break;
                case 0xfd8e711f:
                    result = new TL_draftMessage();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in DraftMessage", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_draftMessageEmpty extends DraftMessage {
        public static int constructor = 0x1b0c841a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(date);
            }
        }
    }

    public static class TL_draftMessageEmpty_layer81 extends TL_draftMessageEmpty {
        public static int constructor = 0xba4baec5;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_draftMessage extends DraftMessage {
        public static int constructor = 0xfd8e711f;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 2) != 0;
            if ((flags & 1) != 0) {
                reply_to_msg_id = stream.readInt32(exception);
            }
            message = stream.readString(exception);
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    entities.add(object);
                }
            }
            date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(reply_to_msg_id);
            }
            stream.writeString(message);
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = entities.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    entities.get(a).serializeToStream(stream);
                }
            }
            stream.writeInt32(date);
        }
    }

    public static abstract class ChatPhoto extends TLObject {

        public int flags;
        public boolean has_video;
        public FileLocation photo_small;
        public FileLocation photo_big;
        public byte[] stripped_thumb;
        public int dc_id;
        public long photo_id;
        public BitmapDrawable strippedBitmap;

        public static ChatPhoto TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            return TLdeserialize(stream, constructor, exception, true);
        }

        public static ChatPhoto TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception,  boolean allowStripedThumb) {
            ChatPhoto result = null;
            switch (constructor) {
                case 0x1c6e1c11:
                    result = new TL_chatPhoto();
                    break;
                case 0x475cdbd5:
                    result = new TL_chatPhoto_layer115();
                    break;
                case 0x37c1011c:
                    result = new TL_chatPhotoEmpty();
                    break;
                case 0x6153276a:
                    result = new TL_chatPhoto_layer97();
                    break;
                case 0xd20b9f3c:
                    result = new TL_chatPhoto_layer126();
                    break;
                case 0x4790ee05:
                    result = new TL_chatPhoto_layer127();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ChatPhoto", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_chatPhoto extends ChatPhoto {
        public static int constructor = 0x1c6e1c11;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            readParams(stream, exception, true);
        }

        public void readParams(AbstractSerializedData stream, boolean exception, boolean allowStripedThumbs) {
            flags = stream.readInt32(exception);
            has_video = (flags & 1) != 0;
            photo_id = stream.readInt64(exception);
            if ((flags & 2) != 0) {
                stripped_thumb = stream.readByteArray(exception);
            }
            dc_id = stream.readInt32(exception);
            photo_small = new TL_fileLocationToBeDeprecated();
            photo_small.volume_id = -photo_id;
            photo_small.local_id = 'a';
            photo_big = new TL_fileLocationToBeDeprecated();
            photo_big.volume_id = -photo_id;
            photo_big.local_id = 'c';

            if (allowStripedThumbs && stripped_thumb != null && Build.VERSION.SDK_INT >= 21) {
                try {
                    strippedBitmap = new BitmapDrawable(ImageLoader.getStrippedPhotoBitmap(stripped_thumb, "b"));
                } catch (Throwable e) {
                    FileLog.e(e);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_video ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt64(photo_id);
            if ((flags & 2) != 0) {
                stream.writeByteArray(stripped_thumb);
            }
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_chatPhoto_layer115 extends TL_chatPhoto {
        public static int constructor = 0x475cdbd5;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_chatPhotoEmpty extends ChatPhoto {
        public static int constructor = 0x37c1011c;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_chatPhoto_layer97 extends TL_chatPhoto {
        public static int constructor = 0x6153276a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
        }
    }

    public static class TL_chatPhoto_layer126 extends TL_chatPhoto {
        public static int constructor = 0xd20b9f3c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_video = (flags & 1) != 0;
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_video ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_chatPhoto_layer127 extends TL_chatPhoto {
        public static int constructor = 0x4790ee05;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_video = (flags & 1) != 0;
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 2) != 0) {
                stripped_thumb = stream.readByteArray(exception);
                if (Build.VERSION.SDK_INT >= 21) {
                    try {
                        strippedBitmap = new BitmapDrawable(ImageLoader.getStrippedPhotoBitmap(stripped_thumb, "b"));
                    } catch (Throwable e) {
                        FileLog.e(e);
                    }
                }
            }
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_video ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
            if ((flags & 2) != 0) {
                stream.writeByteArray(stripped_thumb);
            }
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_help_termsOfService extends TLObject {
        public static int constructor = 0x780a0310;

        public int flags;
        public boolean popup;
        public TL_dataJSON id;
        public String text;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public int min_age_confirm;

        public static TL_help_termsOfService TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_help_termsOfService.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_help_termsOfService", constructor));
                } else {
                    return null;
                }
            }
            TL_help_termsOfService result = new TL_help_termsOfService();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            popup = (flags & 1) != 0;
            id = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            text = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                entities.add(object);
            }
            if ((flags & 2) != 0) {
                min_age_confirm = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = popup ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            id.serializeToStream(stream);
            stream.writeString(text);
            stream.writeInt32(0x1cb5c415);
            int count = entities.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                entities.get(a).serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(min_age_confirm);
            }
        }
    }

    public static class TL_payments_paymentReceipt extends TLObject {
        public static int constructor = 0x70c4fe03;

        public int flags;
        public int date;
        public long bot_id;
        public long provider_id;
        public String title;
        public String description;
        public WebDocument photo;
        public TL_invoice invoice;
        public TL_paymentRequestedInfo info;
        public TL_shippingOption shipping;
        public long tip_amount;
        public String currency;
        public long total_amount;
        public String credentials_title;
        public ArrayList<User> users = new ArrayList<>();

        public static TL_payments_paymentReceipt TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_payments_paymentReceipt.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_payments_paymentReceipt", constructor));
                } else {
                    return null;
                }
            }
            TL_payments_paymentReceipt result = new TL_payments_paymentReceipt();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            date = stream.readInt32(exception);
            bot_id = stream.readInt64(exception);
            provider_id = stream.readInt64(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 4) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                info = TL_paymentRequestedInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                shipping = TL_shippingOption.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                tip_amount = stream.readInt64(exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            credentials_title = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(date);
            stream.writeInt64(bot_id);
            stream.writeInt64(provider_id);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 4) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            if ((flags & 1) != 0) {
                info.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                shipping.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt64(tip_amount);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(credentials_title);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class NotifyPeer extends TLObject {

        public static NotifyPeer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            NotifyPeer result = null;
            switch (constructor) {
                case 0x226e6308:
                    result = new TL_notifyForumTopic();
                    break;
                case 0xd612e8ef:
                    result = new TL_notifyBroadcasts();
                    break;
                case 0xc007cec3:
                    result = new TL_notifyChats();
                    break;
                case 0xb4c83b4c:
                    result = new TL_notifyUsers();
                    break;
                case 0x9fd40bd8:
                    result = new TL_notifyPeer();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in NotifyPeer", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_notifyForumTopic extends NotifyPeer {
        public static int constructor = 0x226e6308;

        public Peer peer;
        public int top_msg_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            top_msg_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(top_msg_id);
        }
    }

    public static class TL_notifyBroadcasts extends NotifyPeer {
        public static int constructor = 0xd612e8ef;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notifyChats extends NotifyPeer {
        public static int constructor = 0xc007cec3;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notifyUsers extends NotifyPeer {
        public static int constructor = 0xb4c83b4c;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notifyPeer extends NotifyPeer {
        public static int constructor = 0x9fd40bd8;

        public Peer peer;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_phone_joinAsPeers extends TLObject {
        public static int constructor = 0xafe5623f;

        public ArrayList<Peer> peers = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_phone_joinAsPeers TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_phone_joinAsPeers.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_phone_joinAsPeers", constructor));
                } else {
                    return null;
                }
            }
            TL_phone_joinAsPeers result = new TL_phone_joinAsPeers();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Peer object = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                peers.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = peers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                peers.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_emojiKeywordsDifference extends TLObject {
        public static int constructor = 0x5cc761bd;

        public String lang_code;
        public int from_version;
        public int version;
        public ArrayList<EmojiKeyword> keywords = new ArrayList<>();

        public static TL_emojiKeywordsDifference TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_emojiKeywordsDifference.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_emojiKeywordsDifference", constructor));
                } else {
                    return null;
                }
            }
            TL_emojiKeywordsDifference result = new TL_emojiKeywordsDifference();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            lang_code = stream.readString(exception);
            from_version = stream.readInt32(exception);
            version = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                EmojiKeyword object = EmojiKeyword.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                keywords.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(lang_code);
            stream.writeInt32(from_version);
            stream.writeInt32(version);
            stream.writeInt32(0x1cb5c415);
            int count = keywords.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                keywords.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class messages_SentEncryptedMessage extends TLObject {
        public int date;
        public EncryptedFile file;

        public static messages_SentEncryptedMessage TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_SentEncryptedMessage result = null;
            switch (constructor) {
                case 0x560f8935:
                    result = new TL_messages_sentEncryptedMessage();
                    break;
                case 0x9493ff32:
                    result = new TL_messages_sentEncryptedFile();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_SentEncryptedMessage", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_sentEncryptedMessage extends messages_SentEncryptedMessage {
        public static int constructor = 0x560f8935;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(date);
        }
    }

    public static class TL_messages_sentEncryptedFile extends messages_SentEncryptedMessage {
        public static int constructor = 0x9493ff32;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            date = stream.readInt32(exception);
            file = EncryptedFile.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(date);
            file.serializeToStream(stream);
        }
    }

    public static class TL_premiumSubscriptionOption extends TLObject {
        public static int constructor = 0x5f2d1df2;

        public int flags;
        public boolean current;
        public String transaction;
        public boolean can_purchase_upgrade;
        public int months;
        public String currency;
        public long amount;
        public String bot_url;
        public String store_product;

        public static TL_premiumSubscriptionOption TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            TL_premiumSubscriptionOption result = null;
            switch (constructor) {
                case 0x5f2d1df2:
                    result = new TL_premiumSubscriptionOption();
                    break;
                case 0xb6f11ebe:
                    result = new TL_premiumSubscriptionOption_layer151();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in TL_premiumSubscriptionOption", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            current = (flags & 2) != 0;
            if ((flags & 8) != 0) {
                transaction = stream.readString(exception);
            }
            can_purchase_upgrade = (flags & 4) != 0;
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            bot_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = current ? (flags | 2) : (flags &~ 2);
            flags = can_purchase_upgrade ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            if ((flags & 8) != 0) {
                stream.writeString(transaction);
            }
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeString(bot_url);
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_premiumSubscriptionOption_layer151 extends TL_premiumSubscriptionOption {
        public static int constructor = 0xb6f11ebe;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            bot_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeString(bot_url);
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_premiumGiftOption extends TLObject {
        public static int constructor = 0x74c34319;

        public int flags;
        public int months;
        public String currency;
        public long amount;
        public String bot_url;
        public String store_product;

        public static TL_premiumGiftOption TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_premiumGiftOption.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_premiumGiftOption", constructor));
                } else {
                    return null;
                }
            }
            TL_premiumGiftOption result = new TL_premiumGiftOption();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            bot_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeString(bot_url);
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_error extends TLObject {
        public static int constructor = 0xc4b9f9bb;

        public int code;
        public String text;

        public static TL_error TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_error.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_error", constructor));
                } else {
                    return null;
                }
            }
            TL_error result = new TL_error();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            code = stream.readInt32(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(code);
            stream.writeString(text);
        }
    }

    public static abstract class UrlAuthResult extends TLObject {

        public static UrlAuthResult TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            UrlAuthResult result = null;
            switch (constructor) {
                case 0xa9d6db1f:
                    result = new TL_urlAuthResultDefault();
                    break;
                case 0x92d33a0e:
                    result = new TL_urlAuthResultRequest();
                    break;
                case 0x8f8c0e4e:
                    result = new TL_urlAuthResultAccepted();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in UrlAuthResult", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_urlAuthResultDefault extends UrlAuthResult {
        public static int constructor = 0xa9d6db1f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_urlAuthResultRequest extends UrlAuthResult {
        public static int constructor = 0x92d33a0e;

        public int flags;
        public boolean request_write_access;
        public User bot;
        public String domain;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            request_write_access = (flags & 1) != 0;
            bot = User.TLdeserialize(stream, stream.readInt32(exception), exception);
            domain = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = request_write_access ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            bot.serializeToStream(stream);
            stream.writeString(domain);
        }
    }

    public static class TL_urlAuthResultAccepted extends UrlAuthResult {
        public static int constructor = 0x8f8c0e4e;

        public String url;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_messages_chatFull extends TLObject {
        public static int constructor = 0xe5d7d19c;

        public ChatFull full_chat;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_chatFull TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_chatFull.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_chatFull", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_chatFull result = new TL_messages_chatFull();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            full_chat = ChatFull.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            full_chat.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_account_passwordSettings extends TLObject {
        public static int constructor = 0x9a5c33e5;

        public int flags;
        public String email;
        public TL_secureSecretSettings secure_settings;

        public static TL_account_passwordSettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_account_passwordSettings.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_account_passwordSettings", constructor));
                } else {
                    return null;
                }
            }
            TL_account_passwordSettings result = new TL_account_passwordSettings();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                email = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                secure_settings = TL_secureSecretSettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(email);
            }
            if ((flags & 2) != 0) {
                secure_settings.serializeToStream(stream);
            }
        }
    }

    public static abstract class DocumentAttribute extends TLObject {
        public String alt;
        public InputStickerSet stickerset;
        public int duration;
        public int flags;
        public TL_maskCoords mask_coords;
        public boolean round_message;
        public boolean supports_streaming;
        public String file_name;
        public int w;
        public int h;
        public boolean mask;
        public String title;
        public String performer;
        public boolean voice;
        public byte[] waveform;

        public static DocumentAttribute TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            DocumentAttribute result = null;
            switch (constructor) {
                case 0x3a556302:
                    result = new TL_documentAttributeSticker_layer55();
                    break;
                case 0x51448e5:
                    result = new TL_documentAttributeAudio_old();
                    break;
                case 0x6319d612:
                    result = new TL_documentAttributeSticker();
                    break;
                case 0x11b58939:
                    result = new TL_documentAttributeAnimated();
                    break;
                case 0x15590068:
                    result = new TL_documentAttributeFilename();
                    break;
                case 0xef02ce6:
                    result = new TL_documentAttributeVideo();
                    break;
                case 0x5910cccb:
                    result = new TL_documentAttributeVideo_layer65();
                    break;
                case 0xded218e0:
                    result = new TL_documentAttributeAudio_layer45();
                    break;
                case 0xfb0a5727:
                    result = new TL_documentAttributeSticker_old();
                    break;
                case 0x9801d2f7:
                    result = new TL_documentAttributeHasStickers();
                    break;
                case 0x994c9882:
                    result = new TL_documentAttributeSticker_old2();
                    break;
                case 0x6c37c15c:
                    result = new TL_documentAttributeImageSize();
                    break;
                case 0x9852f9c6:
                    result = new TL_documentAttributeAudio();
                    break;
                case 0xfd149899:
                    result = new TL_documentAttributeCustomEmoji();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in DocumentAttribute", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_documentAttributeSticker_layer55 extends TL_documentAttributeSticker {
        public static int constructor = 0x3a556302;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            alt = stream.readString(exception);
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(alt);
            stickerset.serializeToStream(stream);
        }
    }

    public static class TL_documentAttributeAudio_old extends TL_documentAttributeAudio {
        public static int constructor = 0x51448e5;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            duration = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(duration);
        }
    }

    public static class TL_documentAttributeSticker extends DocumentAttribute {
        public static int constructor = 0x6319d612;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            mask = (flags & 2) != 0;
            alt = stream.readString(exception);
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                mask_coords = TL_maskCoords.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = mask ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(alt);
            stickerset.serializeToStream(stream);
            if ((flags & 1) != 0) {
                mask_coords.serializeToStream(stream);
            }
        }
    }

    public static class TL_documentAttributeAnimated extends DocumentAttribute {
        public static int constructor = 0x11b58939;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_documentAttributeFilename extends DocumentAttribute {
        public static int constructor = 0x15590068;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            file_name = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(file_name);
        }
    }

    public static class TL_documentAttributeVideo extends DocumentAttribute {
        public static int constructor = 0xef02ce6;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            round_message = (flags & 1) != 0;
            supports_streaming = (flags & 2) != 0;
            duration = stream.readInt32(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = round_message ? (flags | 1) : (flags &~ 1);
            flags = supports_streaming ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_documentAttributeVideo_layer65 extends TL_documentAttributeVideo {
        public static int constructor = 0x5910cccb;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            duration = stream.readInt32(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_documentAttributeAudio_layer45 extends TL_documentAttributeAudio {
        public static int constructor = 0xded218e0;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            duration = stream.readInt32(exception);
            title = stream.readString(exception);
            performer = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(duration);
            stream.writeString(title);
            stream.writeString(performer);
        }
    }

    public static class TL_documentAttributeSticker_old extends TL_documentAttributeSticker {
        public static int constructor = 0xfb0a5727;


        public void readParams(AbstractSerializedData stream, boolean exception) {
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_documentAttributeHasStickers extends DocumentAttribute {
        public static int constructor = 0x9801d2f7;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_documentAttributeSticker_old2 extends TL_documentAttributeSticker {
        public static int constructor = 0x994c9882;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            alt = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(alt);
        }
    }

    public static class TL_documentAttributeImageSize extends DocumentAttribute {
        public static int constructor = 0x6c37c15c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_documentAttributeAudio extends DocumentAttribute {
        public static int constructor = 0x9852f9c6;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            voice = (flags & 1024) != 0;
            duration = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                performer = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                waveform = stream.readByteArray(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = voice ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            stream.writeInt32(duration);
            if ((flags & 1) != 0) {
                stream.writeString(title);
            }
            if ((flags & 2) != 0) {
                stream.writeString(performer);
            }
            if ((flags & 4) != 0) {
                stream.writeByteArray(waveform);
            }
        }
    }

    public static class TL_documentAttributeCustomEmoji extends DocumentAttribute {
        public static int constructor = 0xfd149899;

        public boolean free;
        public boolean text_color;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            free = (flags & 1) != 0;
            text_color = (flags & 2) != 0;
            alt = stream.readString(exception);
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = free ? (flags | 1) : (flags &~ 1);
            flags = text_color ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(alt);
            stickerset.serializeToStream(stream);
        }
    }

    public static class TL_textPhone extends RichText {
        public static int constructor = 0x1ccb966a;

        public RichText text;
        public String phone;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            phone = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(phone);
        }
    }

    public static class TL_textSuperscript extends RichText {
        public static int constructor = 0xc7fb5e01;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textImage extends RichText {
        public static int constructor = 0x81ccf4f;

        public long document_id;
        public int w;
        public int h;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            document_id = stream.readInt64(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(document_id);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_textEmpty extends RichText {
        public static int constructor = 0xdc3d824f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_textUrl extends RichText {
        public static int constructor = 0x3c2884c1;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            url = stream.readString(exception);
            webpage_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(url);
            stream.writeInt64(webpage_id);
        }
    }

    public static class TL_textAnchor extends RichText {
        public static int constructor = 0x35553762;

        public RichText text;
        public String name;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            name = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(name);
        }
    }

    public static class TL_textStrike extends RichText {
        public static int constructor = 0x9bf8bb95;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textMarked extends RichText {
        public static int constructor = 0x34b8621;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textFixed extends RichText {
        public static int constructor = 0x6c3f19b9;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textEmail extends RichText {
        public static int constructor = 0xde5a0dd6;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            email = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(email);
        }
    }

    public static class TL_textPlain extends RichText {
        public static int constructor = 0x744694e0;

        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_textConcat extends RichText {
        public static int constructor = 0x7e6260d7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                RichText object = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                texts.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = texts.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                texts.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_textBold extends RichText {
        public static int constructor = 0x6724abc4;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textItalic extends RichText {
        public static int constructor = 0xd912a59c;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textUnderline extends RichText {
        public static int constructor = 0xc12622c4;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textSubscript extends RichText {
        public static int constructor = 0xed6a8504;

        public RichText text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_statsURL extends TLObject {
        public static int constructor = 0x47a971e0;

        public String url;

        public static TL_statsURL TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_statsURL.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_statsURL", constructor));
                } else {
                    return null;
                }
            }
            TL_statsURL result = new TL_statsURL();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_popularContact extends TLObject {
        public static int constructor = 0x5ce14175;

        public long client_id;
        public int importers;

        public static TL_popularContact TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_popularContact.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_popularContact", constructor));
                } else {
                    return null;
                }
            }
            TL_popularContact result = new TL_popularContact();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            client_id = stream.readInt64(exception);
            importers = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(client_id);
            stream.writeInt32(importers);
        }
    }

    public static class TL_messages_botCallbackAnswer extends TLObject {
        public static int constructor = 0x36585ea4;

        public int flags;
        public boolean alert;
        public boolean has_url;
        public boolean native_ui;
        public String message;
        public String url;
        public int cache_time;

        public static TL_messages_botCallbackAnswer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_botCallbackAnswer.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_botCallbackAnswer", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_botCallbackAnswer result = new TL_messages_botCallbackAnswer();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            alert = (flags & 2) != 0;
            has_url = (flags & 8) != 0;
            native_ui = (flags & 16) != 0;
            if ((flags & 1) != 0) {
                message = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                url = stream.readString(exception);
            }
            cache_time = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = alert ? (flags | 2) : (flags &~ 2);
            flags = has_url ? (flags | 8) : (flags &~ 8);
            flags = native_ui ? (flags | 16) : (flags &~ 16);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(message);
            }
            if ((flags & 4) != 0) {
                stream.writeString(url);
            }
            stream.writeInt32(cache_time);
        }
    }

    public static class TL_dataJSON extends TLObject {
        public static int constructor = 0x7d748d04;

        public String data;

        public static TL_dataJSON TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_dataJSON.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_dataJSON", constructor));
                } else {
                    return null;
                }
            }
            TL_dataJSON result = new TL_dataJSON();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            data = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(data);
        }
    }

    public static class TL_contactStatus extends TLObject {
        public static int constructor = 0x16d9703b;

        public long user_id;
        public UserStatus status;

        public static TL_contactStatus TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_contactStatus.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_contactStatus", constructor));
                } else {
                    return null;
                }
            }
            TL_contactStatus result = new TL_contactStatus();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            status.serializeToStream(stream);
        }
    }

    public static abstract class GroupCall extends TLObject {

        public int flags;
        public boolean join_muted;
        public boolean can_change_join_muted;
        public boolean join_date_asc;
        public boolean schedule_start_subscribed;
        public boolean can_start_video;
        public boolean record_video_active;
        public long id;
        public long access_hash;
        public int participants_count;
        public String title;
        public int stream_dc_id;
        public int record_start_date;
        public int schedule_date;
        public int unmuted_video_count;
        public int unmuted_video_limit;
        public int version;
        public int duration;
        public boolean rtmp_stream;
        public boolean listeners_hidden;

        public static GroupCall TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            GroupCall result = null;
            switch (constructor) {
                case 0x7780bcb4:
                    result = new TL_groupCallDiscarded();
                    break;
                case 0xd597650c:
                    result = new TL_groupCall();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in GroupCall", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_groupCallDiscarded extends GroupCall {
        public static int constructor = 0x7780bcb4;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            duration = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(duration);
        }
    }

    public static class TL_groupCall extends GroupCall {
        public static int constructor = 0xd597650c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            join_muted = (flags & 2) != 0;
            can_change_join_muted = (flags & 4) != 0;
            join_date_asc = (flags & 64) != 0;
            schedule_start_subscribed = (flags & 256) != 0;
            can_start_video = (flags & 512) != 0;
            record_video_active = (flags & 2048) != 0;
            rtmp_stream = (flags & 4096) != 0;
            listeners_hidden = (flags & 8192) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                stream_dc_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                record_start_date = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                schedule_date = stream.readInt32(exception);
            }
            if ((flags & 1024) != 0) {
                unmuted_video_count = stream.readInt32(exception);
            }
            unmuted_video_limit = stream.readInt32(exception);
            version = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = join_muted ? (flags | 2) : (flags &~ 2);
            flags = can_change_join_muted ? (flags | 4) : (flags &~ 4);
            flags = join_date_asc ? (flags | 64) : (flags &~ 64);
            flags = schedule_start_subscribed ? (flags | 256) : (flags &~ 256);
            flags = can_start_video ? (flags | 512) : (flags &~ 512);
            flags = record_video_active ? (flags | 2048) : (flags &~ 2048);
            flags = rtmp_stream ? (flags | 4096) : (flags &~ 4096);
            flags = listeners_hidden ? (flags | 8192) : (flags &~ 8192);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(participants_count);
            if ((flags & 8) != 0) {
                stream.writeString(title);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(stream_dc_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(record_start_date);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(schedule_date);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(unmuted_video_count);
            }
            stream.writeInt32(unmuted_video_limit);
            stream.writeInt32(version);
        }
    }

    public static class TL_channelBannedRights_layer92 extends TLObject {
        public static int constructor = 0x58cf4249;

        public int flags;
        public boolean view_messages;
        public boolean send_messages;
        public boolean send_media;
        public boolean send_stickers;
        public boolean send_gifs;
        public boolean send_games;
        public boolean send_inline;
        public boolean embed_links;
        public int until_date;

        public static TL_channelBannedRights_layer92 TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_channelBannedRights_layer92.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_channelBannedRights_layer92", constructor));
                } else {
                    return null;
                }
            }
            TL_channelBannedRights_layer92 result = new TL_channelBannedRights_layer92();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            view_messages = (flags & 1) != 0;
            send_messages = (flags & 2) != 0;
            send_media = (flags & 4) != 0;
            send_stickers = (flags & 8) != 0;
            send_gifs = (flags & 16) != 0;
            send_games = (flags & 32) != 0;
            send_inline = (flags & 64) != 0;
            embed_links = (flags & 128) != 0;
            until_date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = view_messages ? (flags | 1) : (flags &~ 1);
            flags = send_messages ? (flags | 2) : (flags &~ 2);
            flags = send_media ? (flags | 4) : (flags &~ 4);
            flags = send_stickers ? (flags | 8) : (flags &~ 8);
            flags = send_gifs ? (flags | 16) : (flags &~ 16);
            flags = send_games ? (flags | 32) : (flags &~ 32);
            flags = send_inline ? (flags | 64) : (flags &~ 64);
            flags = embed_links ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            stream.writeInt32(until_date);
        }
    }

    public static abstract class DialogPeer extends TLObject {

        public static DialogPeer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            DialogPeer result = null;
            switch (constructor) {
                case 0xe56dbf05:
                    result = new TL_dialogPeer();
                    break;
                case 0x514519e2:
                    result = new TL_dialogPeerFolder();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in DialogPeer", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_dialogPeer extends DialogPeer {
        public static int constructor = 0xe56dbf05;

        public Peer peer;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_dialogPeerFolder extends DialogPeer {
        public static int constructor = 0x514519e2;

        public int folder_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            folder_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(folder_id);
        }
    }

    public static abstract class MessagePeerReaction extends TLObject {

        public int flags;
        public boolean big;
        public boolean unread;
        public Peer peer_id;
        public Reaction reaction;
        public int date;
        public boolean dateIsSeen; //custom

        public static MessagePeerReaction TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            MessagePeerReaction result = null;
            switch (constructor) {
                case 0x8c79b63c:
                    result = new TL_messagePeerReaction();
                    break;
                case 0xb156fe9c:
                    result = new TL_messagePeerReaction_layer154();
                    break;
                case 0x51b67eff:
                    result = new TL_messagePeerReaction_layer144();
                    break;
                case 0x932844fa:
                    result = new TL_messagePeerReaction_layer137();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in MessagePeerReaction", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messagePeerReaction extends MessagePeerReaction {
        public static int constructor = 0x8c79b63c;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            unread = (flags & 2) != 0;
            peer_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            date = stream.readInt32(exception);
            reaction = Reaction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            flags = unread ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            peer_id.serializeToStream(stream);
            stream.writeInt32(date);
            reaction.serializeToStream(stream);
        }
    }

    public static class TL_messagePeerReaction_layer154 extends MessagePeerReaction {
        public static int constructor = 0xb156fe9c;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            unread = (flags & 2) != 0;
            peer_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            reaction = Reaction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            flags = unread ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            peer_id.serializeToStream(stream);
            reaction.serializeToStream(stream);
        }
    }

    public static class TL_messagePeerReaction_layer144 extends MessagePeerReaction {
        public static int constructor = 0x51b67eff;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            unread = (flags & 2) != 0;
            peer_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            reaction = new TL_reactionEmoji();
            ((TL_reactionEmoji) reaction).emoticon = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            flags = unread ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            peer_id.serializeToStream(stream);
            if (reaction instanceof TL_reactionEmoji)
                stream.writeString(((TL_reactionEmoji) reaction).emoticon);
            else
                stream.writeString("");
        }
    }

    public static class TL_messagePeerReaction_layer137 extends MessagePeerReaction {
        public static int constructor = 0x932844fa;

        public long user_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            reaction = new TL_reactionEmoji();
            ((TL_reactionEmoji) reaction).emoticon = stream.readString(exception);
            peer_id = new TL_peerUser();
            peer_id.user_id = user_id;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            if (reaction instanceof TL_reactionEmoji)
                stream.writeString(((TL_reactionEmoji) reaction).emoticon);
            else
                stream.writeString("");
        }
    }

    public static abstract class auth_Authorization extends TLObject {

        public static auth_Authorization TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            auth_Authorization result = null;
            switch (constructor) {
                case 0x44747e9a:
                    result = new TL_auth_authorizationSignUpRequired();
                    break;
                case 0x33fb7bb8://TODO old constructor need remove
                    result = new TL_auth_authorization();
                    break;
                case 0x2ea2c0d4:
                    result = new TL_auth_authorization();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in auth_Authorization", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_auth_authorizationSignUpRequired extends auth_Authorization {
        public static int constructor = 0x44747e9a;

        public int flags;
        public TL_help_termsOfService terms_of_service;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                terms_of_service = TL_help_termsOfService.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                terms_of_service.serializeToStream(stream);
            }
        }
    }

    public static class TL_auth_authorization extends auth_Authorization {
        public static int constructor = 0x2ea2c0d4;

        public int flags;
        public boolean setup_password_required;
        public int otherwise_relogin_days;
        public int tmp_sessions;
        public byte[] future_auth_token;
        public User user;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            setup_password_required = (flags & 2) != 0;
            if ((flags & 2) != 0) {
                otherwise_relogin_days = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                tmp_sessions = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                future_auth_token = stream.readByteArray(exception);
            }
            user = User.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = setup_password_required ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                stream.writeInt32(otherwise_relogin_days);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(tmp_sessions);
            }
            if ((flags & 4) != 0) {
                stream.writeByteArray(future_auth_token);
            }
            user.serializeToStream(stream);
        }
    }

    public static class TL_pollAnswer extends TLObject {
        public static int constructor = 0x6ca9c2e9;

        public String text;
        public byte[] option;

        public static TL_pollAnswer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_pollAnswer.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_pollAnswer", constructor));
                } else {
                    return null;
                }
            }
            TL_pollAnswer result = new TL_pollAnswer();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            option = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeByteArray(option);
        }
    }

    public static abstract class JSONValue extends TLObject {

        public static JSONValue TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            JSONValue result = null;
            switch (constructor) {
                case 0xc7345e6a:
                    result = new TL_jsonBool();
                    break;
                case 0x3f6d7b68:
                    result = new TL_jsonNull();
                    break;
                case 0xb71e767a:
                    result = new TL_jsonString();
                    break;
                case 0xf7444763:
                    result = new TL_jsonArray();
                    break;
                case 0x99c1d49d:
                    result = new TL_jsonObject();
                    break;
                case 0x2be0dfa4:
                    result = new TL_jsonNumber();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in JSONValue", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_jsonBool extends JSONValue {
        public static int constructor = 0xc7345e6a;

        public boolean value;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            value = stream.readBool(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeBool(value);
        }
    }

    public static class TL_jsonNull extends JSONValue {
        public static int constructor = 0x3f6d7b68;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_jsonString extends JSONValue {
        public static int constructor = 0xb71e767a;

        public String value;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            value = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(value);
        }
    }

    public static class TL_jsonArray extends JSONValue {
        public static int constructor = 0xf7444763;

        public ArrayList<JSONValue> value = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                JSONValue object = JSONValue.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                value.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = value.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                value.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_jsonObject extends JSONValue {
        public static int constructor = 0x99c1d49d;

        public ArrayList<TL_jsonObjectValue> value = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_jsonObjectValue object = TL_jsonObjectValue.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                value.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = value.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                value.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_jsonNumber extends JSONValue {
        public static int constructor = 0x2be0dfa4;

        public double value;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            value = stream.readDouble(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(value);
        }
    }

    public static abstract class InputWallPaper extends TLObject {

        public static InputWallPaper TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputWallPaper result = null;
            switch (constructor) {
                case 0xe630b979:
                    result = new TL_inputWallPaper();
                    break;
                case 0x967a462e:
                    result = new TL_inputWallPaperNoFile();
                    break;
                case 0x72091c80:
                    result = new TL_inputWallPaperSlug();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputWallPaper", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputWallPaper extends InputWallPaper {
        public static int constructor = 0xe630b979;

        public long id;
        public long access_hash;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputWallPaperNoFile extends InputWallPaper {
        public static int constructor = 0x967a462e;

        public long id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_inputWallPaperSlug extends InputWallPaper {
        public static int constructor = 0x72091c80;

        public String slug;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            slug = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(slug);
        }
    }

    public static class TL_messages_historyImportParsed extends TLObject {
        public static int constructor = 0x5e0fb7b9;

        public int flags;
        public boolean pm;
        public boolean group;
        public String title;

        public static TL_messages_historyImportParsed TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_historyImportParsed.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_historyImportParsed", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_historyImportParsed result = new TL_messages_historyImportParsed();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            pm = (flags & 1) != 0;
            group = (flags & 2) != 0;
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = pm ? (flags | 1) : (flags &~ 1);
            flags = group ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
        }
    }

    public static class TL_folder extends TLObject {
        public static int constructor = 0xff544e65;

        public int flags;
        public boolean autofill_new_broadcasts;
        public boolean autofill_public_groups;
        public boolean autofill_new_correspondents;
        public int id;
        public String title;
        public ChatPhoto photo;

        public static TL_folder TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_folder.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_folder", constructor));
                } else {
                    return null;
                }
            }
            TL_folder result = new TL_folder();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            autofill_new_broadcasts = (flags & 1) != 0;
            autofill_public_groups = (flags & 2) != 0;
            autofill_new_correspondents = (flags & 4) != 0;
            id = stream.readInt32(exception);
            title = stream.readString(exception);
            if ((flags & 8) != 0) {
                photo = ChatPhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = autofill_new_broadcasts ? (flags | 1) : (flags &~ 1);
            flags = autofill_public_groups ? (flags | 2) : (flags &~ 2);
            flags = autofill_new_correspondents ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title);
            if ((flags & 8) != 0) {
                photo.serializeToStream(stream);
            }
        }
    }

    public static abstract class messages_Messages extends TLObject {
        public ArrayList<Message> messages = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public ArrayList<TL_forumTopic> topics = new ArrayList<>();
        public int flags;
        public boolean inexact;
        public int pts;
        public int count;
        public int next_rate;
        public int offset_id_offset;
        public ArrayList<Document> animatedEmoji;

        public static messages_Messages TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_Messages result = null;
            switch (constructor) {
                case 0x3a54685e:
                    result = new TL_messages_messagesSlice();
                    break;
                case 0x8c718e87:
                    result = new TL_messages_messages();
                    break;
                case 0xc776ba4e:
                    result = new TL_messages_channelMessages();
                    break;
                case 0x74535f21:
                    result = new TL_messages_messagesNotModified();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_Messages", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_messagesSlice extends messages_Messages {
        public static int constructor = 0x3a54685e;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            count = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                next_rate = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                offset_id_offset = stream.readInt32(exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Message object = Message.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(count);
            if ((flags & 1) != 0) {
                stream.writeInt32(next_rate);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(offset_id_offset);
            }
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_messages extends messages_Messages {
        public static int constructor = 0x8c718e87;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Message object = Message.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_channelMessages extends messages_Messages {
        public static int constructor = 0xc776ba4e;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            pts = stream.readInt32(exception);
            count = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                offset_id_offset = stream.readInt32(exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Message object = Message.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_forumTopic object = TL_forumTopic.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                topics.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(pts);
            stream.writeInt32(count);
            if ((flags & 4) != 0) {
                stream.writeInt32(offset_id_offset);
            }
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = topics.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                topics.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_messagesNotModified extends messages_Messages {
        public static int constructor = 0x74535f21;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
        }
    }

    public static class TL_statsGroupTopAdmin extends TLObject {
        public static int constructor = 0xd7584c87;

        public long user_id;
        public int deleted;
        public int kicked;
        public int banned;

        public static TL_statsGroupTopAdmin TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_statsGroupTopAdmin.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_statsGroupTopAdmin", constructor));
                } else {
                    return null;
                }
            }
            TL_statsGroupTopAdmin result = new TL_statsGroupTopAdmin();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            deleted = stream.readInt32(exception);
            kicked = stream.readInt32(exception);
            banned = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt32(deleted);
            stream.writeInt32(kicked);
            stream.writeInt32(banned);
        }
    }

    public static class TL_payments_paymentForm extends TLObject {
        public static int constructor = 0xa0058751;

        public int flags;
        public boolean can_save_credentials;
        public boolean password_missing;
        public long form_id;
        public long bot_id;
        public String title;
        public String description;
        public WebDocument photo;
        public TL_invoice invoice;
        public long provider_id;
        public String url;
        public String native_provider;
        public TL_dataJSON native_params;
        public ArrayList<TL_paymentFormMethod> additional_methods = new ArrayList<>();
        public TL_paymentRequestedInfo saved_info;
        public ArrayList<TL_paymentSavedCredentialsCard> saved_credentials = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_payments_paymentForm TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_payments_paymentForm.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_payments_paymentForm", constructor));
                } else {
                    return null;
                }
            }
            TL_payments_paymentForm result = new TL_payments_paymentForm();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_save_credentials = (flags & 4) != 0;
            password_missing = (flags & 8) != 0;
            form_id = stream.readInt64(exception);
            bot_id = stream.readInt64(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 32) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            provider_id = stream.readInt64(exception);
            url = stream.readString(exception);
            if ((flags & 16) != 0) {
                native_provider = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                native_params = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_paymentFormMethod object = TL_paymentFormMethod.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    additional_methods.add(object);
                }
            }
            if ((flags & 1) != 0) {
                saved_info = TL_paymentRequestedInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_paymentSavedCredentialsCard object = TL_paymentSavedCredentialsCard.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    saved_credentials.add(object);
                }
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_save_credentials ? (flags | 4) : (flags &~ 4);
            flags = password_missing ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt64(form_id);
            stream.writeInt64(bot_id);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            stream.writeInt64(provider_id);
            stream.writeString(url);
            if ((flags & 16) != 0) {
                stream.writeString(native_provider);
            }
            if ((flags & 16) != 0) {
                native_params.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = additional_methods.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    additional_methods.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 1) != 0) {
                saved_info.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = saved_credentials.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    saved_credentials.get(a).serializeToStream(stream);
                }
            }
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_paymentFormMethod extends TLObject {
        public static int constructor = 0x88f8f21b;

        public String url;
        public String title;

        public static TL_paymentFormMethod TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_paymentFormMethod.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_PaymentFormMethod", constructor));
                } else {
                    return null;
                }
            }
            TL_paymentFormMethod result = new TL_paymentFormMethod();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            title = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeString(title);
        }
    }

    public static abstract class ContactLink_layer101 extends TLObject {

        public static ContactLink_layer101 TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            ContactLink_layer101 result = null;
            switch (constructor) {
                case 0xfeedd3ad:
                    result = new TL_contactLinkNone();
                    break;
                case 0xd502c2d0:
                    result = new TL_contactLinkContact();
                    break;
                case 0x5f4f9247:
                    result = new TL_contactLinkUnknown();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ContactLink", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_contactLinkNone extends ContactLink_layer101 {
        public static int constructor = 0xfeedd3ad;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contactLinkContact extends ContactLink_layer101 {
        public static int constructor = 0xd502c2d0;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contactLinkUnknown extends ContactLink_layer101 {
        public static int constructor = 0x5f4f9247;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }


    public static class TL_contacts_link_layer101 extends TLObject {
        public static int constructor = 0x3ace484c;

        public ContactLink_layer101 my_link;
        public ContactLink_layer101 foreign_link;
        public User user;

        public static TL_contacts_link_layer101 TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_contacts_link_layer101.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_contacts_link", constructor));
                } else {
                    return null;
                }
            }
            TL_contacts_link_layer101 result = new TL_contacts_link_layer101();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            my_link = ContactLink_layer101.TLdeserialize(stream, stream.readInt32(exception), exception);
            foreign_link = ContactLink_layer101.TLdeserialize(stream, stream.readInt32(exception), exception);
            user = User.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            my_link.serializeToStream(stream);
            foreign_link.serializeToStream(stream);
            user.serializeToStream(stream);
        }
    }

    public static abstract class EncryptedFile extends TLObject {
        public long id;
        public long access_hash;
        public long size;
        public int dc_id;
        public int key_fingerprint;

        public static EncryptedFile TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            EncryptedFile result = null;
            switch (constructor) {
                case 0xa8008cd8:
                    result = new TL_encryptedFile();
                    break;
                case 0x4a70994c:
                    result = new TL_encryptedFile_layer142();
                    break;
                case 0xc21f497e:
                    result = new TL_encryptedFileEmpty();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in EncryptedFile", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_encryptedFile extends EncryptedFile {
        public static int constructor = 0xa8008cd8;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt64(exception);
            dc_id = stream.readInt32(exception);
            key_fingerprint = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt64(size);
            stream.writeInt32(dc_id);
            stream.writeInt32(key_fingerprint);
        }
    }

    public static class TL_encryptedFile_layer142 extends EncryptedFile {
        public static int constructor = 0x4a70994c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
            key_fingerprint = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) size);
            stream.writeInt32(dc_id);
            stream.writeInt32(key_fingerprint);
        }
    }

    public static class TL_encryptedFileEmpty extends EncryptedFile {
        public static int constructor = 0xc21f497e;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class Peer extends TLObject {

        public long user_id;
        public long chat_id;
        public long channel_id;

        public static Peer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            Peer result = null;
            switch (constructor) {
                case 0xa2a5371e:
                    result = new TL_peerChannel();
                    break;
                case 0xbddde532:
                    result = new TL_peerChannel_layer131();
                    break;
                case 0x59511722:
                    result = new TL_peerUser();
                    break;
                case 0x9db1bc6d:
                    result = new TL_peerUser_layer131();
                    break;
                case 0x36c6019a:
                    result = new TL_peerChat();
                    break;
                case 0xbad0e5bb:
                    result = new TL_peerChat_layer131();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in Peer", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_peerChannel_layer131 extends TL_peerChannel {
        public static int constructor = 0xbddde532;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            channel_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) channel_id);
        }
    }

    public static class TL_peerUser extends Peer {
        public static int constructor = 0x59511722;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_peerChannel extends Peer {
        public static int constructor = 0xa2a5371e;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            channel_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(channel_id);
        }
    }

    public static class TL_peerChat extends Peer {
        public static int constructor = 0x36c6019a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            chat_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(chat_id);
        }
    }

    public static class TL_peerUser_layer131 extends TL_peerUser {
        public static int constructor = 0x9db1bc6d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_peerChat_layer131 extends TL_peerChat {
        public static int constructor = 0xbad0e5bb;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            chat_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) chat_id);
        }
    }

    public static class TL_labeledPrice extends TLObject {
        public static int constructor = 0xcb296bf8;

        public String label;
        public long amount;

        public static TL_labeledPrice TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_labeledPrice.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_labeledPrice", constructor));
                } else {
                    return null;
                }
            }
            TL_labeledPrice result = new TL_labeledPrice();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            label = stream.readString(exception);
            amount = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(label);
            stream.writeInt64(amount);
        }
    }

    public static class TL_messages_exportedChatInvites extends TLObject {
        public static int constructor = 0xbdc62dcc;

        public int count;
        public ArrayList<ExportedChatInvite> invites = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_exportedChatInvites TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_exportedChatInvites.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_exportedChatInvites", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_exportedChatInvites result = new TL_messages_exportedChatInvites();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                ExportedChatInvite object = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                invites.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            stream.writeInt32(0x1cb5c415);
            int count = invites.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                invites.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_inputStickerSetItem extends TLObject {
        public static int constructor = 0xffa0a496;

        public int flags;
        public InputDocument document;
        public String emoji;
        public TL_maskCoords mask_coords;

        public static TL_inputStickerSetItem TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_inputStickerSetItem.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_inputStickerSetItem", constructor));
                } else {
                    return null;
                }
            }
            TL_inputStickerSetItem result = new TL_inputStickerSetItem();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            document = InputDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            emoji = stream.readString(exception);
            if ((flags & 1) != 0) {
                mask_coords = TL_maskCoords.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            document.serializeToStream(stream);
            stream.writeString(emoji);
            if ((flags & 1) != 0) {
                mask_coords.serializeToStream(stream);
            }
        }
    }

    public static class TL_langPackDifference extends TLObject {
        public static int constructor = 0xf385c1f6;

        public String lang_code;
        public int from_version;
        public int version;
        public ArrayList<LangPackString> strings = new ArrayList<>();

        public static TL_langPackDifference TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_langPackDifference.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_langPackDifference", constructor));
                } else {
                    return null;
                }
            }
            TL_langPackDifference result = new TL_langPackDifference();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            lang_code = stream.readString(exception);
            from_version = stream.readInt32(exception);
            version = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                LangPackString object = LangPackString.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                strings.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(lang_code);
            stream.writeInt32(from_version);
            stream.writeInt32(version);
            stream.writeInt32(0x1cb5c415);
            int count = strings.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                strings.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class help_DeepLinkInfo extends TLObject {

        public static help_DeepLinkInfo TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            help_DeepLinkInfo result = null;
            switch (constructor) {
                case 0x66afa166:
                    result = new TL_help_deepLinkInfoEmpty();
                    break;
                case 0x6a4ee832:
                    result = new TL_help_deepLinkInfo();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in help_DeepLinkInfo", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_help_deepLinkInfoEmpty extends help_DeepLinkInfo {
        public static int constructor = 0x66afa166;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_deepLinkInfo extends help_DeepLinkInfo {
        public static int constructor = 0x6a4ee832;

        public int flags;
        public boolean update_app;
        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            update_app = (flags & 1) != 0;
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    entities.add(object);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = update_app ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = entities.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    entities.get(a).serializeToStream(stream);
                }
            }
        }
    }

    public static class TL_chatAdminRights extends TLObject {
        public static int constructor = 0x5fb224d5;

        public int flags;
        public boolean change_info;
        public boolean post_messages;
        public boolean edit_messages;
        public boolean delete_messages;
        public boolean ban_users;
        public boolean invite_users;
        public boolean pin_messages;
        public boolean add_admins;
        public boolean anonymous;
        public boolean manage_call;
        public boolean other;
        public boolean manage_topics;

        public static TL_chatAdminRights TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_chatAdminRights.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_chatAdminRights", constructor));
                } else {
                    return null;
                }
            }
            TL_chatAdminRights result = new TL_chatAdminRights();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            change_info = (flags & 1) != 0;
            post_messages = (flags & 2) != 0;
            edit_messages = (flags & 4) != 0;
            delete_messages = (flags & 8) != 0;
            ban_users = (flags & 16) != 0;
            invite_users = (flags & 32) != 0;
            pin_messages = (flags & 128) != 0;
            add_admins = (flags & 512) != 0;
            anonymous = (flags & 1024) != 0;
            manage_call = (flags & 2048) != 0;
            other = (flags & 4096) != 0;
            manage_topics = (flags & 8192) != 0;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = change_info ? (flags | 1) : (flags &~ 1);
            flags = post_messages ? (flags | 2) : (flags &~ 2);
            flags = edit_messages ? (flags | 4) : (flags &~ 4);
            flags = delete_messages ? (flags | 8) : (flags &~ 8);
            flags = ban_users ? (flags | 16) : (flags &~ 16);
            flags = invite_users ? (flags | 32) : (flags &~ 32);
            flags = pin_messages ? (flags | 128) : (flags &~ 128);
            flags = add_admins ? (flags | 512) : (flags &~ 512);
            flags = anonymous ? (flags | 1024) : (flags &~ 1024);
            flags = manage_call ? (flags | 2048) : (flags &~ 2048);
            flags = other ? (flags | 4096) : (flags &~ 4096);
            flags = manage_topics ? (flags | 8192) : (flags &~ 8192);
            stream.writeInt32(flags);
        }
    }

    public static class TL_statsAbsValueAndPrev extends TLObject {
        public static int constructor = 0xcb43acde;

        public double current;
        public double previous;

        public static TL_statsAbsValueAndPrev TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_statsAbsValueAndPrev.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_statsAbsValueAndPrev", constructor));
                } else {
                    return null;
                }
            }
            TL_statsAbsValueAndPrev result = new TL_statsAbsValueAndPrev();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            current = stream.readDouble(exception);
            previous = stream.readDouble(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(current);
            stream.writeDouble(previous);
        }
    }

    public static abstract class PollResults extends TLObject {

        public int flags;
        public boolean min;
        public ArrayList<TL_pollAnswerVoters> results = new ArrayList<>();
        public int total_voters;
        public ArrayList<Long> recent_voters = new ArrayList<>();
        public String solution;
        public ArrayList<MessageEntity> solution_entities = new ArrayList<>();

        public static PollResults TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            PollResults result = null;
            switch (constructor) {
                case 0x5755785a:
                    result = new TL_pollResults_layer108();
                    break;
                case 0xc87024a2:
                    result = new TL_pollResults_layer111();
                    break;
                case 0xbadcc1a3:
                    result = new TL_pollResults_layer131();
                    break;
                case 0xdcb82ea3:
                    result = new TL_pollResults();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in PollResults", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_pollResults_layer108 extends TL_pollResults {
        public static int constructor = 0x5755785a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_pollAnswerVoters object = TL_pollAnswerVoters.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    results.add(object);
                }
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = results.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    results.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
        }
    }

    public static class TL_pollResults_layer111 extends PollResults {
        public static int constructor = 0xc87024a2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_pollAnswerVoters object = TL_pollAnswerVoters.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    results.add(object);
                }
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_voters.add((long) stream.readInt32(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = results.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    results.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_voters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt32((int) (long) recent_voters.get(a));
                }
            }
        }
    }

    public static class TL_pollResults_layer131 extends TL_pollResults {
        public static int constructor = 0xbadcc1a3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_pollAnswerVoters object = TL_pollAnswerVoters.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    results.add(object);
                }
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_voters.add((long) stream.readInt32(exception));
                }
            }
            if ((flags & 16) != 0) {
                solution = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    solution_entities.add(object);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = results.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    results.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_voters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt32((int) (long) recent_voters.get(a));
                }
            }
            if ((flags & 16) != 0) {
                stream.writeString(solution);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = solution_entities.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    solution_entities.get(a).serializeToStream(stream);
                }
            }
        }
    }

    public static class TL_pollResults extends PollResults {
        public static int constructor = 0xdcb82ea3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_pollAnswerVoters object = TL_pollAnswerVoters.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    results.add(object);
                }
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_voters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 16) != 0) {
                solution = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    solution_entities.add(object);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = results.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    results.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_voters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_voters.get(a));
                }
            }
            if ((flags & 16) != 0) {
                stream.writeString(solution);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = solution_entities.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    solution_entities.get(a).serializeToStream(stream);
                }
            }
        }
    }

    public static abstract class SecureFile extends TLObject {

        public static SecureFile TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            SecureFile result = null;
            switch (constructor) {
                case 0x64199744:
                    result = new TL_secureFileEmpty();
                    break;
                case 0x7d09c27e:
                    result = new TL_secureFile();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in SecureFile", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_secureFileEmpty extends SecureFile {
        public static int constructor = 0x64199744;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_secureFile extends SecureFile {
        public static int constructor = 0x7d09c27e;

        public long id;
        public long access_hash;
        public long size;
        public int dc_id;
        public int date;
        public byte[] file_hash;
        public byte[] secret;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt64(exception);
            dc_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            file_hash = stream.readByteArray(exception);
            secret = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt64(size);
            stream.writeInt32(dc_id);
            stream.writeInt32(date);
            stream.writeByteArray(file_hash);
            stream.writeByteArray(secret);
        }
    }

    public static class TL_secureFile_layer142 extends TL_secureFile {
        public static int constructor = 0xe0277a62;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            file_hash = stream.readByteArray(exception);
            secret = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) size);
            stream.writeInt32(dc_id);
            stream.writeInt32(date);
            stream.writeByteArray(file_hash);
            stream.writeByteArray(secret);
        }
    }

    public static class TL_messages_affectedMessages extends TLObject {
        public static int constructor = 0x84d19185;

        public int pts;
        public int pts_count;

        public static TL_messages_affectedMessages TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_affectedMessages.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_affectedMessages", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_affectedMessages result = new TL_messages_affectedMessages();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            pts = stream.readInt32(exception);
            pts_count = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(pts);
            stream.writeInt32(pts_count);
        }
    }

    public static class TL_messages_chatInviteImporters extends TLObject {
        public static int constructor = 0x81b6b00a;

        public int count;
        public ArrayList<TL_chatInviteImporter> importers = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_chatInviteImporters TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_chatInviteImporters.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_chatInviteImporters", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_chatInviteImporters result = new TL_messages_chatInviteImporters();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_chatInviteImporter object = TL_chatInviteImporter.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                importers.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            stream.writeInt32(0x1cb5c415);
            int count = importers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                importers.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_pollAnswerVoters extends TLObject {
        public static int constructor = 0x3b6ddad2;

        public int flags;
        public boolean chosen;
        public boolean correct;
        public byte[] option;
        public int voters;

        public static TL_pollAnswerVoters TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_pollAnswerVoters.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_pollAnswerVoters", constructor));
                } else {
                    return null;
                }
            }
            TL_pollAnswerVoters result = new TL_pollAnswerVoters();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            chosen = (flags & 1) != 0;
            correct = (flags & 2) != 0;
            option = stream.readByteArray(exception);
            voters = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = chosen ? (flags | 1) : (flags &~ 1);
            flags = correct ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeByteArray(option);
            stream.writeInt32(voters);
        }
    }

    public static class TL_channels_channelParticipant extends TLObject {
        public static int constructor = 0xdfb80317;

        public ChannelParticipant participant;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_channels_channelParticipant TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_channels_channelParticipant.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_channels_channelParticipant", constructor));
                } else {
                    return null;
                }
            }
            TL_channels_channelParticipant result = new TL_channels_channelParticipant();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            participant = ChannelParticipant.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            participant.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_authorization extends TLObject {
        public static int constructor = 0xad01d61d;

        public int flags;
        public boolean current;
        public boolean official_app;
        public boolean password_pending;
        public boolean encrypted_requests_disabled;
        public boolean call_requests_disabled;
        public long hash;
        public String device_model;
        public String platform;
        public String system_version;
        public int api_id;
        public String app_name;
        public String app_version;
        public int date_created;
        public int date_active;
        public String ip;
        public String country;
        public String region;

        public static TL_authorization TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_authorization.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_authorization", constructor));
                } else {
                    return null;
                }
            }
            TL_authorization result = new TL_authorization();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            current = (flags & 1) != 0;
            official_app = (flags & 2) != 0;
            password_pending = (flags & 4) != 0;
            encrypted_requests_disabled = (flags & 8) != 0;
            call_requests_disabled = (flags & 16) != 0;
            hash = stream.readInt64(exception);
            device_model = stream.readString(exception);
            platform = stream.readString(exception);
            system_version = stream.readString(exception);
            api_id = stream.readInt32(exception);
            app_name = stream.readString(exception);
            app_version = stream.readString(exception);
            date_created = stream.readInt32(exception);
            date_active = stream.readInt32(exception);
            ip = stream.readString(exception);
            country = stream.readString(exception);
            region = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = current ? (flags | 1) : (flags &~ 1);
            flags = official_app ? (flags | 2) : (flags &~ 2);
            flags = password_pending ? (flags | 4) : (flags &~ 4);
            flags = encrypted_requests_disabled ? (flags | 8) : (flags &~ 8);
            flags = call_requests_disabled ? (flags | 16) : (flags &~ 16);
            stream.writeInt32(flags);
            stream.writeInt64(hash);
            stream.writeString(device_model);
            stream.writeString(platform);
            stream.writeString(system_version);
            stream.writeInt32(api_id);
            stream.writeString(app_name);
            stream.writeString(app_version);
            stream.writeInt32(date_created);
            stream.writeInt32(date_active);
            stream.writeString(ip);
            stream.writeString(country);
            stream.writeString(region);
        }
    }

    public static abstract class StatsGraph extends TLObject {

        public static StatsGraph TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            StatsGraph result = null;
            switch (constructor) {
                case 0x8ea464b6:
                    result = new TL_statsGraph();
                    break;
                case 0x4a27eb2d:
                    result = new TL_statsGraphAsync();
                    break;
                case 0xbedc9822:
                    result = new TL_statsGraphError();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in StatsGraph", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_statsGraph extends StatsGraph {
        public static int constructor = 0x8ea464b6;

        public int flags;
        public TL_dataJSON json;
        public String zoom_token;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            json = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                zoom_token = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            json.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeString(zoom_token);
            }
        }
    }

    public static class TL_statsGraphAsync extends StatsGraph {
        public static int constructor = 0x4a27eb2d;

        public String token;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            token = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(token);
        }
    }

    public static class TL_statsGraphError extends StatsGraph {
        public static int constructor = 0xbedc9822;

        public String error;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            error = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(error);
        }
    }

    public static class TL_messageInteractionCounters extends TLObject {
        public static int constructor = 0xad4fc9bd;

        public int msg_id;
        public int views;
        public int forwards;

        public static TL_messageInteractionCounters TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messageInteractionCounters.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messageInteractionCounters", constructor));
                } else {
                    return null;
                }
            }
            TL_messageInteractionCounters result = new TL_messageInteractionCounters();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            msg_id = stream.readInt32(exception);
            views = stream.readInt32(exception);
            forwards = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(msg_id);
            stream.writeInt32(views);
            stream.writeInt32(forwards);
        }
    }

    public static abstract class updates_Difference extends TLObject {
        public ArrayList<Message> new_messages = new ArrayList<>();
        public ArrayList<EncryptedMessage> new_encrypted_messages = new ArrayList<>();
        public ArrayList<Update> other_updates = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public TL_updates_state state;
        public TL_updates_state intermediate_state;
        public int pts;
        public int date;
        public int seq;

        public static updates_Difference TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            updates_Difference result = null;
            switch (constructor) {
                case 0xf49ca0:
                    result = new TL_updates_difference();
                    break;
                case 0xa8fb1981:
                    result = new TL_updates_differenceSlice();
                    break;
                case 0x4afe8f6d:
                    result = new TL_updates_differenceTooLong();
                    break;
                case 0x5d75a138:
                    result = new TL_updates_differenceEmpty();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in updates_Difference", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_updates_difference extends updates_Difference {
        public static int constructor = 0xf49ca0;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Message object = Message.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                new_messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                EncryptedMessage object = EncryptedMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                new_encrypted_messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Update object = Update.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                other_updates.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
            state = TL_updates_state.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = new_messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                new_messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = new_encrypted_messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                new_encrypted_messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = other_updates.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                other_updates.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
            state.serializeToStream(stream);
        }
    }

    public static class TL_updates_differenceSlice extends updates_Difference {
        public static int constructor = 0xa8fb1981;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Message object = Message.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                new_messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                EncryptedMessage object = EncryptedMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                new_encrypted_messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Update object = Update.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                other_updates.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
            intermediate_state = TL_updates_state.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = new_messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                new_messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = new_encrypted_messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                new_encrypted_messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = other_updates.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                other_updates.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
            intermediate_state.serializeToStream(stream);
        }
    }

    public static class TL_updates_differenceTooLong extends updates_Difference {
        public static int constructor = 0x4afe8f6d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(pts);
        }
    }

    public static class TL_updates_differenceEmpty extends updates_Difference {
        public static int constructor = 0x5d75a138;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            date = stream.readInt32(exception);
            seq = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(date);
            stream.writeInt32(seq);
        }
    }

    public static abstract class PrivacyKey extends TLObject {

        public static PrivacyKey TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            PrivacyKey result = null;
            switch (constructor) {
                case 0xbc2eab30:
                    result = new TL_privacyKeyStatusTimestamp();
                    break;
                case 0x39491cc8:
                    result = new TL_privacyKeyPhoneP2P();
                    break;
                case 0x500e6dfa:
                    result = new TL_privacyKeyChatInvite();
                    break;
                case 0x42ffd42b:
                    result = new TL_privacyKeyAddedByPhone();
                    break;
                case 0x697f414:
                    result = new TL_privacyKeyVoiceMessages();
                    break;
                case 0x3d662b7b:
                    result = new TL_privacyKeyPhoneCall();
                    break;
                case 0x69ec56a3:
                    result = new TL_privacyKeyForwards();
                    break;
                case 0xd19ae46d:
                    result = new TL_privacyKeyPhoneNumber();
                    break;
                case 0x96151fed:
                    result = new TL_privacyKeyProfilePhoto();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in PrivacyKey", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_privacyKeyStatusTimestamp extends PrivacyKey {
        public static int constructor = 0xbc2eab30;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyPhoneP2P extends PrivacyKey {
        public static int constructor = 0x39491cc8;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyChatInvite extends PrivacyKey {
        public static int constructor = 0x500e6dfa;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyPhoneCall extends PrivacyKey {
        public static int constructor = 0x3d662b7b;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyAddedByPhone extends PrivacyKey {
        public static int constructor = 0x42ffd42b;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyVoiceMessages extends PrivacyKey {
        public static int constructor = 0x697f414;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyForwards extends PrivacyKey {
        public static int constructor = 0x69ec56a3;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyPhoneNumber extends PrivacyKey {
        public static int constructor = 0xd19ae46d;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyProfilePhoto extends PrivacyKey {
        public static int constructor = 0x96151fed;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class GeoPoint extends TLObject {
        public int flags;
        public double _long;
        public double lat;
        public int accuracy_radius;
        public long access_hash;

        public static GeoPoint TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            GeoPoint result = null;
            switch (constructor) {
                case 0x296f104:
                    result = new TL_geoPoint_layer119();
                    break;
                case 0x2049d70c:
                    result = new TL_geoPoint_layer81();
                    break;
                case 0x1117dd5f:
                    result = new TL_geoPointEmpty();
                    break;
                case 0xb2a2f663:
                    result = new TL_geoPoint();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in GeoPoint", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_geoPoint_layer119 extends TL_geoPoint {
        public static int constructor = 0x296f104;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            _long = stream.readDouble(exception);
            lat = stream.readDouble(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(_long);
            stream.writeDouble(lat);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_geoPoint_layer81 extends TL_geoPoint {
        public static int constructor = 0x2049d70c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            _long = stream.readDouble(exception);
            lat = stream.readDouble(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(_long);
            stream.writeDouble(lat);
        }
    }

    public static class TL_geoPointEmpty extends GeoPoint {
        public static int constructor = 0x1117dd5f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_geoPoint extends GeoPoint {
        public static int constructor = 0xb2a2f663;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            _long = stream.readDouble(exception);
            lat = stream.readDouble(exception);
            access_hash = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                accuracy_radius = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeDouble(_long);
            stream.writeDouble(lat);
            stream.writeInt64(access_hash);
            if ((flags & 1) != 0) {
                stream.writeInt32(accuracy_radius);
            }
        }
    }

    public static class TL_account_privacyRules extends TLObject {
        public static int constructor = 0x50a04e45;

        public ArrayList<PrivacyRule> rules = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_account_privacyRules TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_account_privacyRules.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_account_privacyRules", constructor));
                } else {
                    return null;
                }
            }
            TL_account_privacyRules result = new TL_account_privacyRules();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PrivacyRule object = PrivacyRule.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                rules.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = rules.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                rules.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class ChatInvite extends TLObject {

        public int flags;
        public boolean channel;
        public boolean broadcast;
        public boolean isPublic;
        public boolean megagroup;
        public boolean request_needed;
        public String title;
        public String about;
        public Photo photo;
        public int participants_count;
        public ArrayList<User> participants = new ArrayList<>();
        public Chat chat;
        public int expires;

        public static ChatInvite TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            ChatInvite result = null;
            switch (constructor) {
                case 0x300c44c1:
                    result = new TL_chatInvite();
                    break;
                case 0x61695cb0:
                    result = new TL_chatInvitePeek();
                    break;
                case 0x5a686d7c:
                    result = new TL_chatInviteAlready();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ChatInvite", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_chatInvite extends ChatInvite {
        public static int constructor = 0x300c44c1;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            channel = (flags & 1) != 0;
            broadcast = (flags & 2) != 0;
            isPublic = (flags & 4) != 0;
            megagroup = (flags & 8) != 0;
            request_needed = (flags & 64) != 0;
            title = stream.readString(exception);
            boolean hasAbout = (flags & 32) != 0;
            if (hasAbout) {
                about = stream.readString(exception);
            }
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 16) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    participants.add(object);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = channel ? (flags | 1) : (flags &~ 1);
            flags = broadcast ? (flags | 2) : (flags &~ 2);
            flags = isPublic ? (flags | 4) : (flags &~ 4);
            flags = megagroup ? (flags | 8) : (flags &~ 8);
            flags = about != null ? (flags | 32) : (flags &~ 32);
            flags = request_needed ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeString(title);
            if (about != null) {
                stream.writeString(about);
            }
            photo.serializeToStream(stream);
            stream.writeInt32(participants_count);
            if ((flags & 16) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = participants.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    participants.get(a).serializeToStream(stream);
                }
            }
        }
    }

    public static class TL_chatInvitePeek extends ChatInvite {
        public static int constructor = 0x61695cb0;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            chat = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
            expires = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            chat.serializeToStream(stream);
            stream.writeInt32(expires);
        }
    }

    public static class TL_chatInviteAlready extends ChatInvite {
        public static int constructor = 0x5a686d7c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            chat = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            chat.serializeToStream(stream);
        }
    }

    public static class TL_inputGroupCall extends TLObject {
        public static int constructor = 0xd8aa840f;

        public long id;
        public long access_hash;

        public static TL_inputGroupCall TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_inputGroupCall.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_inputGroupCall", constructor));
                } else {
                    return null;
                }
            }
            TL_inputGroupCall result = new TL_inputGroupCall();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static abstract class help_AppUpdate extends TLObject {

        public static help_AppUpdate TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            help_AppUpdate result = null;
            switch (constructor) {
                case 0xccbbce30:
                    result = new TL_help_appUpdate();
                    break;
                case 0xc45a6536:
                    result = new TL_help_noAppUpdate();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in help_AppUpdate", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_help_appUpdate extends help_AppUpdate {
        public static int constructor = 0xccbbce30;

        public int flags;
        public boolean can_not_skip;
        public int id;
        public String version;
        public String text;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public Document document;
        public String url;
        public Document sticker;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_not_skip = (flags & 1) != 0;
            id = stream.readInt32(exception);
            version = stream.readString(exception);
            text = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                entities.add(object);
            }
            if ((flags & 2) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                url = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                sticker = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_not_skip ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(version);
            stream.writeString(text);
            stream.writeInt32(0x1cb5c415);
            int count = entities.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                entities.get(a).serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeString(url);
            }
            if ((flags & 8) != 0) {
                sticker.serializeToStream(stream);
            }
        }
    }

    public static class TL_help_noAppUpdate extends help_AppUpdate {
        public static int constructor = 0xc45a6536;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_affectedFoundMessages extends TLObject {
        public static int constructor = 0xef8d3e6c;

        public int pts;
        public int pts_count;
        public int offset;
        public ArrayList<Integer> messages = new ArrayList<>();

        public static TL_messages_affectedFoundMessages TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_affectedFoundMessages.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_affectedFoundMessages", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_affectedFoundMessages result = new TL_messages_affectedFoundMessages();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            pts = stream.readInt32(exception);
            pts_count = stream.readInt32(exception);
            offset = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                messages.add(stream.readInt32(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(pts);
            stream.writeInt32(pts_count);
            stream.writeInt32(offset);
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32(messages.get(a));
            }
        }
    }

    public static class TL_channelAdminLogEvent extends TLObject {
        public static int constructor = 0x1fad68cd;

        public long id;
        public int date;
        public long user_id;
        public ChannelAdminLogEventAction action;

        public static TL_channelAdminLogEvent TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_channelAdminLogEvent.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_channelAdminLogEvent", constructor));
                } else {
                    return null;
                }
            }
            TL_channelAdminLogEvent result = new TL_channelAdminLogEvent();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            date = stream.readInt32(exception);
            user_id = stream.readInt64(exception);
            action = ChannelAdminLogEventAction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt32(date);
            stream.writeInt64(user_id);
            action.serializeToStream(stream);
        }
    }

    public static abstract class messages_FavedStickers extends TLObject {
        public long hash;
        public ArrayList<TL_stickerPack> packs = new ArrayList<>();
        public ArrayList<Document> stickers = new ArrayList<>();

        public static messages_FavedStickers TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_FavedStickers result = null;
            switch (constructor) {
                case 0x9e8fa6d3:
                    result = new TL_messages_favedStickersNotModified();
                    break;
                case 0x2cb51097:
                    result = new TL_messages_favedStickers();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_FavedStickers", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_favedStickersNotModified extends messages_FavedStickers {
        public static int constructor = 0x9e8fa6d3;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_favedStickers extends messages_FavedStickers {
        public static int constructor = 0x2cb51097;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            hash = stream.readInt64(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_stickerPack object = TL_stickerPack.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                packs.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                stickers.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(hash);
            stream.writeInt32(0x1cb5c415);
            int count = packs.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                packs.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = stickers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stickers.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_langPackLanguage extends TLObject {
        public static int constructor = 0xeeca5ce3;

        public int flags;
        public boolean official;
        public boolean rtl;
        public String name;
        public String native_name;
        public String lang_code;
        public String base_lang_code;
        public String plural_code;
        public int strings_count;
        public int translated_count;
        public String translations_url;

        public static TL_langPackLanguage TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_langPackLanguage.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_langPackLanguage", constructor));
                } else {
                    return null;
                }
            }
            TL_langPackLanguage result = new TL_langPackLanguage();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            official = (flags & 1) != 0;
            rtl = (flags & 4) != 0;
            name = stream.readString(exception);
            native_name = stream.readString(exception);
            lang_code = stream.readString(exception);
            if ((flags & 2) != 0) {
                base_lang_code = stream.readString(exception);
            }
            plural_code = stream.readString(exception);
            strings_count = stream.readInt32(exception);
            translated_count = stream.readInt32(exception);
            translations_url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = official ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeString(name);
            stream.writeString(native_name);
            stream.writeString(lang_code);
            if ((flags & 2) != 0) {
                stream.writeString(base_lang_code);
            }
            stream.writeString(plural_code);
            stream.writeInt32(strings_count);
            stream.writeInt32(translated_count);
            stream.writeString(translations_url);
        }
    }

    public static class TL_chatInviteImporter extends TLObject {
        public static int constructor = 0x8c5adfd9;

        public int flags;
        public boolean requested;
        public long user_id;
        public int date;
        public String about;
        public long approved_by;

        public static TL_chatInviteImporter TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_chatInviteImporter.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_chatInviteImporter", constructor));
                } else {
                    return null;
                }
            }
            TL_chatInviteImporter result = new TL_chatInviteImporter();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            requested = (flags & 1) != 0;
            user_id = stream.readInt64(exception);
            date = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                about = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                approved_by = stream.readInt64(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = requested ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt64(user_id);
            stream.writeInt32(date);
            if ((flags & 4) != 0) {
                stream.writeString(about);
            }
            if ((flags & 2) != 0) {
                stream.writeInt64(approved_by);
            }
        }
    }

    //聊天事件动作
    public static abstract class SendMessageAction extends TLObject {
        public int progress;

        public static SendMessageAction TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            SendMessageAction result = null;
            switch (constructor) {
                case 0xdd6a8f48:
                    result = new TL_sendMessageGamePlayAction();
                    break;
                case 0xd52f73f7:
                    result = new TL_sendMessageRecordAudioAction();
                    break;
                case 0x92042ff7:
                    result = new TL_sendMessageUploadVideoAction_old();
                    break;
                case 0xe6ac8a6f:
                    result = new TL_sendMessageUploadAudioAction_old();
                    break;
                case 0xf351d7ab:
                    result = new TL_sendMessageUploadAudioAction();
                    break;
                case 0xd1d34a26:
                    result = new TL_sendMessageUploadPhotoAction();
                    break;
                case 0x8faee98e:
                    result = new TL_sendMessageUploadDocumentAction_old();
                    break;
                case 0xe9763aec:
                    result = new TL_sendMessageUploadVideoAction();
                    break;
                case 0xfd5ec8f5:
                    result = new TL_sendMessageCancelAction();
                    break;
                case 0x176f8ba1:
                    result = new TL_sendMessageGeoLocationAction();
                    break;
                case 0x628cbc6f:
                    result = new TL_sendMessageChooseContactAction();
                    break;
                case 0xb05ac6b1:
                    result = new TL_sendMessageChooseStickerAction();
                    break;
                case 0x88f27fbc:
                    result = new TL_sendMessageRecordRoundAction();
                    break;
                case 0x243e1c66:
                    result = new TL_sendMessageUploadRoundAction();
                    break;
                case 0x16bf744e:
                    result = new TL_sendMessageTypingAction();
                    break;
                case 0xdbda9246:
                    result = new TL_sendMessageHistoryImportAction();
                    break;
                case 0x990a3c1a:
                    result = new TL_sendMessageUploadPhotoAction_old();
                    break;
                case 0xaa0cd9e4:
                    result = new TL_sendMessageUploadDocumentAction();
                    break;
                case 0xd92c2285:
                    result = new TL_speakingInGroupCallAction();
                    break;
                case 0xa187d66f:
                    result = new TL_sendMessageRecordVideoAction();
                    break;
                case 0x25972bcb:
                    result = new TL_sendMessageEmojiInteraction();
                    break;
                case 0xb665902e:
                    result = new TL_sendMessageEmojiInteractionSeen();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in SendMessageAction", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_sendMessageGamePlayAction extends SendMessageAction {
        public static int constructor = 0xdd6a8f48;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageRecordAudioAction extends SendMessageAction {
        public static int constructor = 0xd52f73f7;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadVideoAction_old extends TL_sendMessageUploadVideoAction {
        public static int constructor = 0x92042ff7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageEmojiInteraction extends SendMessageAction {
        public static int constructor = 0x25972bcb;

        public String emoticon;
        public int msg_id;
        public TL_dataJSON interaction;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            emoticon = stream.readString(exception);
            msg_id = stream.readInt32(exception);
            interaction = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(emoticon);
            stream.writeInt32(msg_id);
            interaction.serializeToStream(stream);
        }
    }

    public static class TL_sendMessageUploadAudioAction_old extends TL_sendMessageUploadAudioAction {
        public static int constructor = 0xe6ac8a6f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadAudioAction extends SendMessageAction {
        public static int constructor = 0xf351d7ab;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadPhotoAction extends SendMessageAction {
        public static int constructor = 0xd1d34a26;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadDocumentAction_old extends TL_sendMessageUploadDocumentAction {
        public static int constructor = 0x8faee98e;


        public void readParams(AbstractSerializedData stream, boolean exception) {
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadVideoAction extends SendMessageAction {
        public static int constructor = 0xe9763aec;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    //取消发送消息状态的消息
    public static class TL_sendMessageCancelAction extends SendMessageAction {
        public static int constructor = 0xfd5ec8f5;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageGeoLocationAction extends SendMessageAction {
        public static int constructor = 0x176f8ba1;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageChooseContactAction extends SendMessageAction {
        public static int constructor = 0x628cbc6f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageChooseStickerAction extends SendMessageAction {
        public static int constructor = 0xb05ac6b1;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageRecordRoundAction extends SendMessageAction {
        public static int constructor = 0x88f27fbc;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadRoundAction extends SendMessageAction {
        public static int constructor = 0x243e1c66;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageEmojiInteractionSeen extends SendMessageAction {
        public static int constructor = 0xb665902e;

        public String emoticon;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            emoticon = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(emoticon);
        }
    }

    public static class TL_sendMessageTypingAction extends SendMessageAction {
        public static int constructor = 0x16bf744e;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageHistoryImportAction extends SendMessageAction {
        public static int constructor = 0xdbda9246;

        public int progress;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadPhotoAction_old extends TL_sendMessageUploadPhotoAction {
        public static int constructor = 0x990a3c1a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadDocumentAction extends SendMessageAction {
        public static int constructor = 0xaa0cd9e4;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_speakingInGroupCallAction extends SendMessageAction {
        public static int constructor = 0xd92c2285;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageRecordVideoAction extends SendMessageAction {
        public static int constructor = 0xa187d66f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class auth_SentCodeType extends TLObject {
        public int flags;
        public String url;
        public int length;
        public String pattern;
        public String prefix;
        public boolean apple_signin_allowed;
        public boolean google_signin_allowed;
        public String email_pattern;
        public int next_phone_login_date;
        public byte[] nonce;
        public String receipt;
        public int push_timeout;
        public boolean verifiedFirebase; //custom

        public static auth_SentCodeType TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            auth_SentCodeType result = null;
            switch (constructor) {
                case 0x3dbb5986:
                    result = new TL_auth_sentCodeTypeApp();
                    break;
                case 0x5353e5a7:
                    result = new TL_auth_sentCodeTypeCall();
                    break;
                case 0x5a159841:
                    result = new TL_auth_sentCodeTypeEmailCode();
                    break;
                case 0xa5491dea:
                    result = new TL_auth_sentCodeTypeSetUpEmailRequired();
                    break;
                case 0xab03c6d9:
                    result = new TL_auth_sentCodeTypeFlashCall();
                    break;
                case 0x82006484:
                    result = new TL_auth_sentCodeTypeMissedCall();
                    break;
                case 0xc000bba2:
                    result = new TL_auth_sentCodeTypeSms();
                    break;
                case 0xd9565c39:
                    result = new TL_auth_sentCodeTypeFragmentSms();
                    break;
                case 0xe57b1432:
                    result = new TL_auth_sentCodeTypeFirebaseSms();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in auth_SentCodeType", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_auth_sentCodeTypeApp extends auth_SentCodeType {
        public static int constructor = 0x3dbb5986;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            length = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeCall extends auth_SentCodeType {
        public static int constructor = 0x5353e5a7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            length = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeEmailCode extends auth_SentCodeType {
        public static int constructor = 0x5a159841;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            apple_signin_allowed = (flags & 1) != 0;
            google_signin_allowed = (flags & 2) != 0;
            email_pattern = stream.readString(exception);
            length = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                next_phone_login_date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = apple_signin_allowed ? (flags | 1) : (flags &~ 1);
            flags = google_signin_allowed ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(email_pattern);
            stream.writeInt32(length);
            if ((flags & 4) != 0) {
                stream.writeInt32(next_phone_login_date);
            }
        }
    }

    public static class TL_auth_sentCodeTypeSetUpEmailRequired extends auth_SentCodeType {
        public static int constructor = 0xa5491dea;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            apple_signin_allowed = (flags & 1) != 0;
            google_signin_allowed = (flags & 2) != 0;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = apple_signin_allowed ? (flags | 1) : (flags &~ 1);
            flags = google_signin_allowed ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
        }
    }

    public static class TL_auth_sentCodeTypeFlashCall extends auth_SentCodeType {
        public static int constructor = 0xab03c6d9;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            pattern = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(pattern);
        }
    }

    public static class TL_auth_sentCodeTypeMissedCall extends auth_SentCodeType {
        public static int constructor = 0x82006484;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            prefix = stream.readString(exception);
            length = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(prefix);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeSms extends auth_SentCodeType {
        public static int constructor = 0xc000bba2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            length = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeFragmentSms extends auth_SentCodeType {
        public static int constructor = 0xd9565c39;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            length = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeFirebaseSms extends auth_SentCodeType {
        public static int constructor = 0xe57b1432;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                nonce = stream.readByteArray(exception);
            }
            if ((flags & 2) != 0) {
                receipt = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                push_timeout = stream.readInt32(exception);
            }
            length = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeByteArray(nonce);
            }
            if ((flags & 2) != 0) {
                stream.writeString(receipt);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(push_timeout);
            }
            stream.writeInt32(length);
        }
    }

    public static abstract class messages_StickerSetInstallResult extends TLObject {
        public ArrayList<StickerSetCovered> sets = new ArrayList<>();

        public static messages_StickerSetInstallResult TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_StickerSetInstallResult result = null;
            switch (constructor) {
                case 0x38641628:
                    result = new TL_messages_stickerSetInstallResultSuccess();
                    break;
                case 0x35e410a8:
                    result = new TL_messages_stickerSetInstallResultArchive();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_StickerSetInstallResult", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_stickerSetInstallResultSuccess extends messages_StickerSetInstallResult {
        public static int constructor = 0x38641628;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_stickerSetInstallResultArchive extends messages_StickerSetInstallResult {
        public static int constructor = 0x35e410a8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                StickerSetCovered object = StickerSetCovered.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                sets.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = sets.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                sets.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_peerSettings extends TLObject {
        public static int constructor = 0xa518110d;

        public int flags;
        public boolean report_spam;
        public boolean add_contact;
        public boolean block_contact;
        public boolean share_contact;
        public boolean need_contacts_exception;
        public boolean report_geo;
        public boolean autoarchived;
        public boolean invite_members;
        public boolean request_chat_broadcast;
        public int geo_distance;
        public String request_chat_title;
        public int request_chat_date;

        public static TL_peerSettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_peerSettings.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_peerSettings", constructor));
                } else {
                    return null;
                }
            }
            TL_peerSettings result = new TL_peerSettings();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            report_spam = (flags & 1) != 0;
            add_contact = (flags & 2) != 0;
            block_contact = (flags & 4) != 0;
            share_contact = (flags & 8) != 0;
            need_contacts_exception = (flags & 16) != 0;
            report_geo = (flags & 32) != 0;
            autoarchived = (flags & 128) != 0;
            invite_members = (flags & 256) != 0;
            request_chat_broadcast = (flags & 1024) != 0;
            if ((flags & 64) != 0) {
                geo_distance = stream.readInt32(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_title = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = report_spam ? (flags | 1) : (flags &~ 1);
            flags = add_contact ? (flags | 2) : (flags &~ 2);
            flags = block_contact ? (flags | 4) : (flags &~ 4);
            flags = share_contact ? (flags | 8) : (flags &~ 8);
            flags = need_contacts_exception ? (flags | 16) : (flags &~ 16);
            flags = report_geo ? (flags | 32) : (flags &~ 32);
            flags = autoarchived ? (flags | 128) : (flags &~ 128);
            flags = invite_members ? (flags | 256) : (flags &~ 256);
            flags = request_chat_broadcast ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(geo_distance);
            }
            if ((flags & 512) != 0) {
                stream.writeString(request_chat_title);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(request_chat_date);
            }
        }
    }

    public static class TL_readParticipantDate extends TLObject {
        public static int constructor = 0x4a4ff172;

        public long user_id;
        public int date;

        public static TL_readParticipantDate TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_readParticipantDate.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_readParticipantDate", constructor));
                } else {
                    return null;
                }
            }
            TL_readParticipantDate result = new TL_readParticipantDate();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt32(date);
        }
    }

    public static class TL_statsGroupTopInviter extends TLObject {
        public static int constructor = 0x535f779d;

        public long user_id;
        public int invitations;

        public static TL_statsGroupTopInviter TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_statsGroupTopInviter.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_statsGroupTopInviter", constructor));
                } else {
                    return null;
                }
            }
            TL_statsGroupTopInviter result = new TL_statsGroupTopInviter();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            invitations = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt32(invitations);
        }
    }

    public static abstract class InputDialogPeer extends TLObject {

        public static InputDialogPeer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputDialogPeer result = null;
            switch (constructor) {
                case 0xfcaafeb7:
                    result = new TL_inputDialogPeer();
                    break;
                case 0x64600527:
                    result = new TL_inputDialogPeerFolder();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputDialogPeer", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputDialogPeer extends InputDialogPeer {
        public static int constructor = 0xfcaafeb7;

        public InputPeer peer;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_inputDialogPeerFolder extends InputDialogPeer {
        public static int constructor = 0x64600527;

        public int folder_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            folder_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(folder_id);
        }
    }

    public static abstract class payments_PaymentResult extends TLObject {

        public static payments_PaymentResult TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            payments_PaymentResult result = null;
            switch (constructor) {
                case 0xd8411139:
                    result = new TL_payments_paymentVerificationNeeded();
                    break;
                case 0x4e5f810d:
                    result = new TL_payments_paymentResult();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in payments_PaymentResult", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_payments_paymentVerificationNeeded extends payments_PaymentResult {
        public static int constructor = 0xd8411139;

        public String url;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_payments_paymentResult extends payments_PaymentResult {
        public static int constructor = 0x4e5f810d;

        public Updates updates;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            updates = Updates.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            updates.serializeToStream(stream);
        }
    }

    public static class TL_channels_adminLogResults extends TLObject {
        public static int constructor = 0xed8af74d;

        public ArrayList<TL_channelAdminLogEvent> events = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_channels_adminLogResults TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_channels_adminLogResults.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_channels_adminLogResults", constructor));
                } else {
                    return null;
                }
            }
            TL_channels_adminLogResults result = new TL_channels_adminLogResults();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_channelAdminLogEvent object = TL_channelAdminLogEvent.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                events.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = events.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                events.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_inputPhoneContact extends TLObject {
        public static int constructor = 0xf392b7f4;

        public long client_id;
        public String phone;
        public String first_name;
        public String last_name;

        public static TL_inputPhoneContact TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_inputPhoneContact.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_inputPhoneContact", constructor));
                } else {
                    return null;
                }
            }
            TL_inputPhoneContact result = new TL_inputPhoneContact();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            client_id = stream.readInt64(exception);
            phone = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(client_id);
            stream.writeString(phone);
            stream.writeString(first_name);
            stream.writeString(last_name);
        }
    }

    public static abstract class ThemeSettings extends TLObject {

        public int flags;
        public boolean message_colors_animated;
        public BaseTheme base_theme;
        public int accent_color;
        public int outbox_accent_color;
        public ArrayList<Integer> message_colors = new ArrayList<>();
        public WallPaper wallpaper;

        public static ThemeSettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            ThemeSettings result = null;
            switch (constructor) {
                case 0xfa58b6d4:
                    result = new TL_themeSettings();
                    break;
                case 0x8db4e76c:
                    result = new TL_themeSettings_layer132();
                    break;
                case 0x9c14984a:
                    result = new TL_themeSettings_layer131();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ThemeSettings", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_themeSettings extends ThemeSettings {
        public static int constructor = 0xfa58b6d4;

        public static TL_themeSettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_themeSettings.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_themeSettings", constructor));
                } else {
                    return null;
                }
            }
            TL_themeSettings result = new TL_themeSettings();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message_colors_animated = (flags & 4) != 0;
            base_theme = BaseTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
            accent_color = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                outbox_accent_color = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    message_colors.add(stream.readInt32(exception));
                }
            }
            if ((flags & 2) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = message_colors_animated ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            base_theme.serializeToStream(stream);
            stream.writeInt32(accent_color);
            if ((flags & 8) != 0) {
                stream.writeInt32(outbox_accent_color);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = message_colors.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt32(message_colors.get(a));
                }
            }
            if ((flags & 2) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_themeSettings_layer132 extends ThemeSettings {
        public static int constructor = 0x8db4e76c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message_colors_animated = (flags & 4) != 0;
            base_theme = BaseTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
            accent_color = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    message_colors.add(stream.readInt32(exception));
                }
            }
            if ((flags & 2) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = message_colors_animated ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            base_theme.serializeToStream(stream);
            stream.writeInt32(accent_color);
            if ((flags & 1) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = message_colors.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt32(message_colors.get(a));
                }
            }
            if ((flags & 2) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_themeSettings_layer131 extends ThemeSettings {
        public static int constructor = 0x9c14984a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            base_theme = BaseTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
            accent_color = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                int color = stream.readInt32(exception);
                if (color != 0) {
                    message_colors.add(color);
                }
            }
            if ((flags & 1) != 0) {
                int color = stream.readInt32(exception);
                if (color != 0) {
                    message_colors.add(0, color);
                }
            }
            if ((flags & 2) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            base_theme.serializeToStream(stream);
            stream.writeInt32(accent_color);
            if ((flags & 1) != 0) {
                stream.writeInt32(message_colors.size() > 1 ? message_colors.get(1) : 0);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(message_colors.size() > 0 ? message_colors.get(0) : 0);
            }
            if ((flags & 2) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_pageCaption extends TLObject {
        public static int constructor = 0x6f747657;

        public RichText text;
        public RichText credit;

        public static TL_pageCaption TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_pageCaption.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_pageCaption", constructor));
                } else {
                    return null;
                }
            }
            TL_pageCaption result = new TL_pageCaption();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            credit = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            credit.serializeToStream(stream);
        }
    }

    public static abstract class PrivacyRule extends TLObject {

        public static PrivacyRule TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            PrivacyRule result = null;
            switch (constructor) {
                case 0xf888fa1a:
                    result = new TL_privacyValueDisallowContacts();
                    break;
                case 0xe4621141:
                    result = new TL_privacyValueDisallowUsers();
                    break;
                case 0x6b134e8e:
                    result = new TL_privacyValueAllowChatParticipants();
                    break;
                case 0x41c87565:
                    result = new TL_privacyValueDisallowChatParticipants();
                    break;
                case 0x65427b82:
                    result = new TL_privacyValueAllowAll();
                    break;
                case 0x8b73e763:
                    result = new TL_privacyValueDisallowAll();
                    break;
                case 0xb8905fb2:
                    result = new TL_privacyValueAllowUsers();
                    break;
                case 0xfffe1bac:
                    result = new TL_privacyValueAllowContacts();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in PrivacyRule", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_privacyValueDisallowContacts extends PrivacyRule {
        public static int constructor = 0xf888fa1a;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueDisallowUsers extends PrivacyRule {
        public static int constructor = 0xe4621141;

        public ArrayList<Long> users = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(users.get(a));
            }
        }
    }

    public static class TL_privacyValueAllowChatParticipants extends PrivacyRule {
        public static int constructor = 0x6b134e8e;

        public ArrayList<Long> chats = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                chats.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(chats.get(a));
            }
        }
    }

    public static class TL_privacyValueDisallowChatParticipants extends PrivacyRule {
        public static int constructor = 0x41c87565;

        public ArrayList<Long> chats = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                chats.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(chats.get(a));
            }
        }
    }

    public static class TL_privacyValueAllowAll extends PrivacyRule {
        public static int constructor = 0x65427b82;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueDisallowAll extends PrivacyRule {
        public static int constructor = 0x8b73e763;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueAllowUsers extends PrivacyRule {
        public static int constructor = 0xb8905fb2;

        public ArrayList<Long> users = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(users.get(a));
            }
        }
    }

    public static class TL_privacyValueAllowContacts extends PrivacyRule {
        public static int constructor = 0xfffe1bac;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_pageTableCell extends TLObject {
        public static int constructor = 0x34566b6a;

        public int flags;
        public boolean header;
        public boolean align_center;
        public boolean align_right;
        public boolean valign_middle;
        public boolean valign_bottom;
        public RichText text;
        public int colspan;
        public int rowspan;

        public static TL_pageTableCell TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_pageTableCell.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_pageTableCell", constructor));
                } else {
                    return null;
                }
            }
            TL_pageTableCell result = new TL_pageTableCell();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            header = (flags & 1) != 0;
            align_center = (flags & 8) != 0;
            align_right = (flags & 16) != 0;
            valign_middle = (flags & 32) != 0;
            valign_bottom = (flags & 64) != 0;
            if ((flags & 128) != 0) {
                text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                colspan = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                rowspan = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = header ? (flags | 1) : (flags &~ 1);
            flags = align_center ? (flags | 8) : (flags &~ 8);
            flags = align_right ? (flags | 16) : (flags &~ 16);
            flags = valign_middle ? (flags | 32) : (flags &~ 32);
            flags = valign_bottom ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            if ((flags & 128) != 0) {
                text.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(colspan);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(rowspan);
            }
        }
    }

    public static class TL_messageMediaUnsupported_old extends TL_messageMediaUnsupported {
        public static int constructor = 0x29632a36;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            bytes = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(bytes);
        }
    }

    public static class TL_messageMediaAudio_layer45 extends MessageMedia {
        public static int constructor = 0xc6b68300;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            audio_unused = Audio.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            audio_unused.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaPhoto_old extends TL_messageMediaPhoto {
        public static int constructor = 0xc8c45a2a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
        }
    }

    public static class MessageExtendedMedia extends TLObject {

        public static MessageExtendedMedia TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            MessageExtendedMedia result = null;
            switch (constructor) {
                case 0xad628cc8:
                    result = new TL_messageExtendedMediaPreview();
                    break;
                case 0xee479c64:
                    result = new TL_messageExtendedMedia();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in MessageExtendedMedia", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }

    }

    public static class TL_messageExtendedMediaPreview extends MessageExtendedMedia {
        public static int constructor = 0xad628cc8;

        public int flags;
        public int w;
        public int h;
        public PhotoSize thumb;
        public int video_duration;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                w = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                h = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                thumb = PhotoSize.TLdeserialize(0, 0, 0, stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                video_duration = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(w);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(h);
            }
            if ((flags & 2) != 0) {
                thumb.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(video_duration);
            }
        }
    }

    public static class TL_messageExtendedMedia extends MessageExtendedMedia {
        public static int constructor = 0xee479c64;

        public MessageMedia media;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            media = MessageMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            media.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaInvoice extends MessageMedia {
        public static int constructor = 0xf6a548d3;

        public WebDocument webPhoto;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            shipping_address_requested = (flags & 2) != 0;
            test = (flags & 8) != 0;
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                webPhoto = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                receipt_msg_id = stream.readInt32(exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            start_param = stream.readString(exception);
            if ((flags & 16) != 0) {
                extended_media = MessageExtendedMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = shipping_address_requested ? (flags | 2) : (flags &~ 2);
            flags = test ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                webPhoto.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(receipt_msg_id);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(start_param);
            if ((flags & 16) != 0) {
                extended_media.serializeToStream(stream);
            }
        }
    }

    public static class TL_messageMediaInvoice_layer145 extends TL_messageMediaInvoice {
        public static int constructor = 0x84551347;

        public WebDocument photo;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            shipping_address_requested = (flags & 2) != 0;
            test = (flags & 8) != 0;
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                receipt_msg_id = stream.readInt32(exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            start_param = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = shipping_address_requested ? (flags | 2) : (flags &~ 2);
            flags = test ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(receipt_msg_id);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(start_param);
        }
    }

    public static class TL_messageMediaUnsupported extends MessageMedia {
        public static int constructor = 0x9f84f49e;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageMediaEmpty extends MessageMedia {
        public static int constructor = 0x3ded6320;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageMediaVenue extends MessageMedia {
        public static int constructor = 0x2ec0533f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
            venue_type = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
            stream.writeString(venue_type);
        }
    }

    public static class TL_messageMediaVenue_layer71 extends MessageMedia {
        public static int constructor = 0x7912b71f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
        }
    }

    public static class TL_messageMediaVideo_old extends TL_messageMediaVideo_layer45 {
        public static int constructor = 0xa2d24290;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            video_unused = Video.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            video_unused.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaDocument extends MessageMedia {
        public static int constructor = 0x9cb070d7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            nopremium = (flags & 8) != 0;
            spoiler = (flags & 16) != 0;
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                document = new TL_documentEmpty();
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = nopremium ? (flags | 8) : (flags &~ 8);
            flags = spoiler ? (flags | 16) : (flags &~ 16);
            flags = document != null ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_layer74 extends TL_messageMediaDocument {
        public static int constructor = 0x7c4414d3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                document = new TL_documentEmpty();
            }
            if ((flags & 2) != 0) {
                captionLegacy = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeString(captionLegacy);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_old extends TL_messageMediaDocument {
        public static int constructor = 0x2fda2204;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            document.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaDocument_layer68 extends TL_messageMediaDocument {
        public static int constructor = 0xf3e02ea8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            captionLegacy = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            document.serializeToStream(stream);
            stream.writeString(captionLegacy);
        }
    }

    public static class TL_messageMediaPhoto extends MessageMedia {
        public static int constructor = 0x695150d7;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            spoiler = (flags & 8) != 0;
            if ((flags & 1) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                photo = new TL_photoEmpty();
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = spoiler ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaPoll extends MessageMedia {
        public static int constructor = 0x4bd6e798;

        public Poll poll;
        public PollResults results;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            poll = Poll.TLdeserialize(stream, stream.readInt32(exception), exception);
            results = PollResults.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            poll.serializeToStream(stream);
            results.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaPhoto_layer74 extends TL_messageMediaPhoto {
        public static int constructor = 0xb5223b0f;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                photo = new TL_photoEmpty();
            }
            if ((flags & 2) != 0) {
                captionLegacy = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeString(captionLegacy);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_inputMediaInvoice extends InputMedia {
        public static int constructor = 0x1f9338fe;

        public int flags;
        public String title;
        public String description;
        public TL_inputWebDocument photo;
        public TL_invoice invoice;
        public byte[] payload;
        public String provider;
        public TL_dataJSON provider_data;
        public String start_param;
        public InputMedia extend_media;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = TL_inputWebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            payload = stream.readByteArray(exception);
            provider = stream.readString(exception);
            provider_data = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 2) != 0) {
                start_param = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                extend_media = InputMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            stream.writeByteArray(payload);
            stream.writeString(provider);
            provider_data.serializeToStream(stream);
            if ((flags & 2) != 0) {
                stream.writeString(start_param);
            }
            if ((flags & 4) != 0) {
                extend_media.serializeToStream(stream);
            }
        }
    }

    public static class TL_messageMediaGeoLive extends MessageMedia {
        public static int constructor = 0xb940c666;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                heading = stream.readInt32(exception);
            }
            period = stream.readInt32(exception);
            if ((flags & 2) != 0) {
                proximity_notification_radius = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeInt32(heading);
            }
            stream.writeInt32(period);
            if ((flags & 2) != 0) {
                stream.writeInt32(proximity_notification_radius);
            }
        }
    }

    public static class TL_messageMediaGeoLive_layer119 extends TL_messageMediaGeoLive {
        public static int constructor = 0x7c3c2609;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            period = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
            stream.writeInt32(period);
        }
    }

    public static class TL_messageMediaGame extends MessageMedia {
        public static int constructor = 0xfdb19008;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            game = TL_game.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            game.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaContact_layer81 extends TL_messageMediaContact {
        public static int constructor = 0x5e7d2f39;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_messageMediaPhoto_layer68 extends TL_messageMediaPhoto {
        public static int constructor = 0x3d8ce53d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            captionLegacy = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
            stream.writeString(captionLegacy);
        }
    }

    public static class TL_messageMediaVideo_layer45 extends MessageMedia {
        public static int constructor = 0x5bcf1675;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            video_unused = Video.TLdeserialize(stream, stream.readInt32(exception), exception);
            captionLegacy = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            video_unused.serializeToStream(stream);
            stream.writeString(captionLegacy);
        }
    }

    public static class TL_messageMediaContact_layer131 extends TL_messageMediaContact {
        public static int constructor = 0xcbf24940;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            vcard = stream.readString(exception);
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(vcard);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_messageMediaContact extends MessageMedia {
        public static int constructor = 0x70322949;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            vcard = stream.readString(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(vcard);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_messageMediaDice extends MessageMedia {
        public static int constructor = 0x3f7ee58b;

        public int value;
        public String emoticon;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            value = stream.readInt32(exception);
            emoticon = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(value);
            stream.writeString(emoticon);
        }
    }

    public static class TL_messageMediaDice_layer111 extends TL_messageMediaDice {
        public static int constructor = 0x638fe46b;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            value = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(value);
        }
    }

    public static class TL_messageMediaGeo extends MessageMedia {
        public static int constructor = 0x56e0d474;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaWebPage extends MessageMedia {
        public static int constructor = 0xa32dd600;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            webpage = WebPage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            webpage.serializeToStream(stream);
        }
    }

    public static abstract class LangPackString extends TLObject {
        public int flags;
        public String key;
        public String zero_value;
        public String one_value;
        public String two_value;
        public String few_value;
        public String many_value;
        public String other_value;
        public String value;

        public static LangPackString TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            LangPackString result = null;
            switch (constructor) {
                case 0x6c47ac9f:
                    result = new TL_langPackStringPluralized();
                    break;
                case 0xcad181f6:
                    result = new TL_langPackString();
                    break;
                case 0x2979eeb2:
                    result = new TL_langPackStringDeleted();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in LangPackString", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_langPackStringPluralized extends LangPackString {
        public static int constructor = 0x6c47ac9f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            key = stream.readString(exception);
            if ((flags & 1) != 0) {
                zero_value = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                one_value = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                two_value = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                few_value = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                many_value = stream.readString(exception);
            }
            other_value = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(key);
            if ((flags & 1) != 0) {
                stream.writeString(zero_value);
            }
            if ((flags & 2) != 0) {
                stream.writeString(one_value);
            }
            if ((flags & 4) != 0) {
                stream.writeString(two_value);
            }
            if ((flags & 8) != 0) {
                stream.writeString(few_value);
            }
            if ((flags & 16) != 0) {
                stream.writeString(many_value);
            }
            stream.writeString(other_value);
        }
    }

    public static class TL_langPackString extends LangPackString {
        public static int constructor = 0xcad181f6;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            key = stream.readString(exception);
            value = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(key);
            stream.writeString(value);
        }
    }

    public static class TL_langPackStringDeleted extends LangPackString {
        public static int constructor = 0x2979eeb2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            key = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(key);
        }
    }

    public static abstract class auth_SentCode extends TLObject {

        public int flags;
        public auth_SentCodeType type;
        public String phone_code_hash;
        public auth_CodeType next_type;
        public int timeout;
        public auth_Authorization authorization;

        public static auth_SentCode TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            auth_SentCode result = null;
            switch (constructor) {
                case 0x2390fe44:
                    result = new TL_auth_sentCodeSuccess();
                    break;
                case 0x5e002502:
                    result = new TL_auth_sentCode();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in auth_SentCode", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_auth_sentCodeSuccess extends auth_SentCode {
        public static int constructor = 0x2390fe44;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            authorization = auth_Authorization.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            authorization.serializeToStream(stream);
        }
    }

    public static class TL_auth_sentCode extends auth_SentCode {
        public static int constructor = 0x5e002502;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = auth_SentCodeType.TLdeserialize(stream, stream.readInt32(exception), exception);
            phone_code_hash = stream.readString(exception);
            if ((flags & 2) != 0) {
                next_type = auth_CodeType.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                timeout = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
            stream.writeString(phone_code_hash);
            if ((flags & 2) != 0) {
                next_type.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(timeout);
            }
        }
    }

    public static abstract class BotInlineResult extends TLObject {

        public int flags;
        public String id;
        public String type;
        public Photo photo;
        public Document document;
        public String title;
        public String description;
        public String url;
        public WebDocument thumb;
        public WebDocument content;
        public BotInlineMessage send_message;
        public long query_id;

        public static BotInlineResult TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            BotInlineResult result = null;
            switch (constructor) {
                case 0x11965f3a:
                    result = new TL_botInlineResult();
                    break;
                case 0x17db940b:
                    result = new TL_botInlineMediaResult();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in BotInlineResult", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_botInlineResult extends BotInlineResult {
        public static int constructor = 0x11965f3a;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readString(exception);
            type = stream.readString(exception);
            if ((flags & 2) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                url = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                thumb = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                content = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            send_message = BotInlineMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(id);
            stream.writeString(type);
            if ((flags & 2) != 0) {
                stream.writeString(title);
            }
            if ((flags & 4) != 0) {
                stream.writeString(description);
            }
            if ((flags & 8) != 0) {
                stream.writeString(url);
            }
            if ((flags & 16) != 0) {
                thumb.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                content.serializeToStream(stream);
            }
            send_message.serializeToStream(stream);
        }
    }

    public static class TL_botInlineMediaResult extends BotInlineResult {
        public static int constructor = 0x17db940b;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readString(exception);
            type = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            send_message = BotInlineMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(id);
            stream.writeString(type);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            send_message.serializeToStream(stream);
        }
    }

    public static class TL_notificationSoundDefault extends NotificationSound {
        public static int constructor = 0x97e8bebe;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notificationSoundNone extends NotificationSound {
        public static int constructor = 0x6f0c34df;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notificationSoundRingtone extends NotificationSound {
        public static int constructor = 0xff6c8049;

        public long id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_notificationSoundLocal extends NotificationSound {
        public static int constructor = 0x830b9ae4;

        public String title;
        public String data;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            data = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeString(data);
        }
    }

    public static abstract class NotificationSound extends TLObject {

        public static NotificationSound TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            NotificationSound result = null;
            switch (constructor) {
                case 0x97e8bebe:
                    result = new TL_notificationSoundDefault();
                    break;
                case 0x6f0c34df:
                    result = new TL_notificationSoundNone();
                    break;
                case 0xff6c8049:
                    result = new TL_notificationSoundRingtone();
                    break;
                case 0x830b9ae4:
                    result = new TL_notificationSoundLocal();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in NotificationSound", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static abstract class PeerNotifySettings extends TLObject {
        public int flags;
        public int mute_until;
        public String sound;
        public boolean show_previews;
        public int events_mask;
        public boolean silent;
        public NotificationSound ios_sound;
        public NotificationSound android_sound;
        public NotificationSound other_sound;

        public static PeerNotifySettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            PeerNotifySettings result = null;
            switch (constructor) {
                case 0xa83b0426:
                    result = new TL_peerNotifySettings();
                    break;
                case 0x9acda4c0:
                    result = new TL_peerNotifySettings_layer77();
                    break;
                case 0xaf509d20:
                    result = new TL_peerNotifySettings_layer139();
                    break;
                case 0x8d5e11ee:
                    result = new TL_peerNotifySettings_layer47();
                    break;
                case 0x70a68512:
                    result = new TL_peerNotifySettingsEmpty_layer77();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in PeerNotifySettings", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_peerNotifySettings_layer77 extends TL_peerNotifySettings {
        public static int constructor = 0x9acda4c0;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            show_previews = (flags & 1) != 0;
            silent = (flags & 2) != 0;
            mute_until = stream.readInt32(exception);
            sound = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = show_previews ? (flags | 1) : (flags &~ 1);
            flags = silent ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(mute_until);
            stream.writeString(sound);
        }
    }

    public static class TL_peerNotifySettings extends PeerNotifySettings {
        public static int constructor = 0xa83b0426;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                ios_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16) != 0) {
                android_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                other_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                ios_sound.serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                android_sound.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                other_sound.serializeToStream(stream);
            }
        }
    }

    public static class TL_peerNotifySettings_layer139 extends TL_peerNotifySettings {
        public static int constructor = 0xaf509d20;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                sound = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                stream.writeString(sound);
            }
        }
    }

    public static class TL_peerNotifySettings_layer47 extends TL_peerNotifySettings {
        public static int constructor = 0x8d5e11ee;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            mute_until = stream.readInt32(exception);
            sound = stream.readString(exception);
            show_previews = stream.readBool(exception);
            events_mask = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(mute_until);
            stream.writeString(sound);
            stream.writeBool(show_previews);
            stream.writeInt32(events_mask);
        }
    }

    public static class TL_peerNotifySettingsEmpty_layer77 extends PeerNotifySettings {
        public static int constructor = 0x70a68512;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class contacts_Blocked extends TLObject {

        public ArrayList<TL_peerBlocked> blocked = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public int count;

        public static contacts_Blocked TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            contacts_Blocked result = null;
            switch (constructor) {
                case 0xade1591:
                    result = new TL_contacts_blocked();
                    break;
                case 0xe1664194:
                    result = new TL_contacts_blockedSlice();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in contacts_Blocked", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_contacts_blocked extends contacts_Blocked {
        public static int constructor = 0xade1591;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_peerBlocked object = TL_peerBlocked.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocked.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = blocked.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocked.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_contacts_blockedSlice extends contacts_Blocked {
        public static int constructor = 0xe1664194;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_peerBlocked object = TL_peerBlocked.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocked.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            stream.writeInt32(0x1cb5c415);
            int count = blocked.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocked.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_inputSecureValue extends TLObject {
        public static int constructor = 0xdb21d0a7;

        public int flags;
        public SecureValueType type;
        public TL_secureData data;
        public InputSecureFile front_side;
        public InputSecureFile reverse_side;
        public InputSecureFile selfie;
        public ArrayList<InputSecureFile> translation = new ArrayList<>();
        public ArrayList<InputSecureFile> files = new ArrayList<>();
        public SecurePlainData plain_data;

        public static TL_inputSecureValue TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_inputSecureValue.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_inputSecureValue", constructor));
                } else {
                    return null;
                }
            }
            TL_inputSecureValue result = new TL_inputSecureValue();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                data = TL_secureData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                front_side = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                reverse_side = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                selfie = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    InputSecureFile object = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    translation.add(object);
                }
            }
            if ((flags & 16) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    InputSecureFile object = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    files.add(object);
                }
            }
            if ((flags & 32) != 0) {
                plain_data = SecurePlainData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
            if ((flags & 1) != 0) {
                data.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                front_side.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                reverse_side.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                selfie.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = translation.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    translation.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = files.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    files.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 32) != 0) {
                plain_data.serializeToStream(stream);
            }
        }
    }

    public static abstract class help_AppConfig extends TLObject {

        public static help_AppConfig TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            help_AppConfig result = null;
            switch (constructor) {
                case 0xdd18782e:
                    result = new TL_help_appConfig();
                    break;
                case 0x7cde641d:
                    result = new TL_help_appConfigNotModified();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in help_AppConfig", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_help_appConfig extends help_AppConfig {
        public static int constructor = 0xdd18782e;

        public int hash;
        public JSONValue config;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            hash = stream.readInt32(exception);
            config = JSONValue.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(hash);
            config.serializeToStream(stream);
        }
    }

    public static class TL_help_appConfigNotModified extends help_AppConfig {
        public static int constructor = 0x7cde641d;

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class messages_DhConfig extends TLObject {
        public byte[] random;
        public int g;
        public byte[] p;
        public int version;

        public static messages_DhConfig TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_DhConfig result = null;
            switch (constructor) {
                case 0xc0e24635:
                    result = new TL_messages_dhConfigNotModified();
                    break;
                case 0x2c221edd:
                    result = new TL_messages_dhConfig();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_DhConfig", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_dhConfigNotModified extends messages_DhConfig {
        public static int constructor = 0xc0e24635;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            random = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(random);
        }
    }

    public static class TL_messages_dhConfig extends messages_DhConfig {
        public static int constructor = 0x2c221edd;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            g = stream.readInt32(exception);
            p = stream.readByteArray(exception);
            version = stream.readInt32(exception);
            random = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(g);
            stream.writeByteArray(p);
            stream.writeInt32(version);
            stream.writeByteArray(random);
        }
    }

    public static class TL_globalPrivacySettings extends TLObject {
        public static int constructor = 0xbea2f424;

        public int flags;
        public boolean archive_and_mute_new_noncontact_peers;

        public static TL_globalPrivacySettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_globalPrivacySettings.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_globalPrivacySettings", constructor));
                } else {
                    return null;
                }
            }
            TL_globalPrivacySettings result = new TL_globalPrivacySettings();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                archive_and_mute_new_noncontact_peers = stream.readBool(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(archive_and_mute_new_noncontact_peers);
            }
        }
    }

    public static class TL_help_premiumPromo_layer144 extends TL_help_premiumPromo {
        public static int constructor = 0x8a4f3c29;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            status_text = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                status_entities.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                video_sections.add(stream.readString(exception));
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                videos.add(object);
            }
            currency = stream.readString(exception);
            monthly_amount = stream.readInt64(exception);
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(status_text);
            stream.writeInt32(0x1cb5c415);
            int count = status_entities.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                status_entities.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = video_sections.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeString(video_sections.get(a));
            }
            stream.writeInt32(0x1cb5c415);
            count = videos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                videos.get(a).serializeToStream(stream);
            }
            stream.writeString(currency);
            stream.writeInt64(monthly_amount);
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_help_premiumPromo_layer140 extends TL_help_premiumPromo {
        public static int constructor = 0xe0360f1b;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            status_text = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                status_entities.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                video_sections.add(stream.readString(exception));
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                videos.add(object);
            }
            currency = stream.readString(exception);
            monthly_amount = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(status_text);
            stream.writeInt32(0x1cb5c415);
            int count = status_entities.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                status_entities.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = video_sections.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeString(video_sections.get(a));
            }
            stream.writeInt32(0x1cb5c415);
            count = videos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                videos.get(a).serializeToStream(stream);
            }
            stream.writeString(currency);
            stream.writeInt64(monthly_amount);
        }
    }

    public static class TL_help_premiumPromo extends TLObject {
        public static int constructor = 0x5334759c;

        public String status_text;
        public ArrayList<MessageEntity> status_entities = new ArrayList<>();
        public ArrayList<String> video_sections = new ArrayList<>();
        public ArrayList<Document> videos = new ArrayList<>();
        public ArrayList<TL_premiumSubscriptionOption> period_options = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public String currency;
        public long monthly_amount;

        public static TL_help_premiumPromo TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            TL_help_premiumPromo result = null;
            switch (constructor) {
                case 0x5334759c:
                    result = new TL_help_premiumPromo();
                    break;
                case 0x8a4f3c29:
                    result = new TL_help_premiumPromo_layer144();
                    break;
                case 0xe0360f1b:
                    result = new TL_help_premiumPromo_layer140();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in TL_help_premiumPromo", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);

                if (result.currency != null) {
                    TL_help_premiumPromo finalResult = result;
                    result.period_options.add(new TL_premiumSubscriptionOption() {{
                        months = 1;
                        currency = finalResult.currency;
                        amount = finalResult.monthly_amount;
                        store_product = "telegram_premium";
                    }});
                }
            }
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            status_text = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                status_entities.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                video_sections.add(stream.readString(exception));
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                videos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_premiumSubscriptionOption object = TL_premiumSubscriptionOption.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                period_options.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(status_text);
            stream.writeInt32(0x1cb5c415);
            int count = status_entities.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                status_entities.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = video_sections.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeString(video_sections.get(a));
            }
            stream.writeInt32(0x1cb5c415);
            count = videos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                videos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = period_options.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                period_options.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class help_UserInfo extends TLObject {

        public static help_UserInfo TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            help_UserInfo result = null;
            switch (constructor) {
                case 0xf3ae2eed:
                    result = new TL_help_userInfoEmpty();
                    break;
                case 0x1eb3758:
                    result = new TL_help_userInfo();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in help_UserInfo", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_help_userInfoEmpty extends help_UserInfo {
        public static int constructor = 0xf3ae2eed;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_userInfo extends help_UserInfo {
        public static int constructor = 0x1eb3758;

        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public String author;
        public int date;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            message = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                entities.add(object);
            }
            author = stream.readString(exception);
            date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(message);
            stream.writeInt32(0x1cb5c415);
            int count = entities.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                entities.get(a).serializeToStream(stream);
            }
            stream.writeString(author);
            stream.writeInt32(date);
        }
    }

    public static class TL_secureValueHash extends TLObject {
        public static int constructor = 0xed1ecdb0;

        public SecureValueType type;
        public byte[] hash;

        public static TL_secureValueHash TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_secureValueHash.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_secureValueHash", constructor));
                } else {
                    return null;
                }
            }
            TL_secureValueHash result = new TL_secureValueHash();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            hash = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(hash);
        }
    }

    public static abstract class messages_StickerSet extends TLObject {

        public StickerSet set;
        public ArrayList<TL_stickerPack> packs = new ArrayList<>();
        public ArrayList<TL_stickerKeyword> keywords = new ArrayList<>();
        public ArrayList<Document> documents = new ArrayList<>();

        public static TL_messages_stickerSet TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            TL_messages_stickerSet result = null;
            switch (constructor) {
                case 0x6e153f16:
                    result = new TL_messages_stickerSet();
                    break;
                case 0xb60a24a6:
                    result = new TL_messages_stickerSet_layer146();
                    break;
                case 0xd3f924eb:
                    result = new TL_messages_stickerSetNotModified();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_StickerSet", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_stickerSet_layer146 extends TL_messages_stickerSet {
        public static int constructor = 0xb60a24a6;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            set = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_stickerPack object = TL_stickerPack.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                packs.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            set.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = packs.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                packs.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_stickerSet extends messages_StickerSet {
        public static int constructor = 0x6e153f16;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            set = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_stickerPack object = TL_stickerPack.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                packs.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_stickerKeyword object = TL_stickerKeyword.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                keywords.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            set.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = packs.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                packs.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = keywords.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                keywords.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_stickerSetNotModified extends TL_messages_stickerSet {
        public static int constructor = 0xd3f924eb;

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class InputGeoPoint extends TLObject {

        public int flags;
        public double lat;
        public double _long;
        public int accuracy_radius;

        public static InputGeoPoint TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputGeoPoint result = null;
            switch (constructor) {
                case 0x48222faf:
                    result = new TL_inputGeoPoint();
                    break;
                case 0xe4c123d6:
                    result = new TL_inputGeoPointEmpty();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputGeoPoint", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputGeoPoint extends InputGeoPoint {
        public static int constructor = 0x48222faf;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            lat = stream.readDouble(exception);
            _long = stream.readDouble(exception);
            if ((flags & 1) != 0) {
                accuracy_radius = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeDouble(lat);
            stream.writeDouble(_long);
            if ((flags & 1) != 0) {
                stream.writeInt32(accuracy_radius);
            }
        }
    }

    public static class TL_inputGeoPointEmpty extends InputGeoPoint {
        public static int constructor = 0xe4c123d6;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_inviteText extends TLObject {
        public static int constructor = 0x18cb9f78;

        public String message;

        public static TL_help_inviteText TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_help_inviteText.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_help_inviteText", constructor));
                } else {
                    return null;
                }
            }
            TL_help_inviteText result = new TL_help_inviteText();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            message = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(message);
        }
    }

    public static abstract class Audio extends TLObject {
        public long id;
        public long access_hash;
        public int date;
        public int duration;
        public String mime_type;
        public int size;
        public int dc_id;
        public long user_id;
        public byte[] key;
        public byte[] iv;

        public static Audio TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            Audio result = null;
            switch (constructor) {
                case 0x586988d8:
                    result = new TL_audioEmpty_layer45();
                    break;
                case 0xf9e35055:
                    result = new TL_audio_layer45();
                    break;
                case 0x427425e7:
                    result = new TL_audio_old();
                    break;
                case 0x555555F6:
                    result = new TL_audioEncrypted();
                    break;
                case 0xc7ac6496:
                    result = new TL_audio_old2();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in Audio", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_audioEmpty_layer45 extends Audio {
        public static int constructor = 0x586988d8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_audio_layer45 extends Audio {
        public static int constructor = 0xf9e35055;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            mime_type = stream.readString(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeString(mime_type);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_audio_old extends TL_audio_layer45 {
        public static int constructor = 0x427425e7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            user_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) user_id);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_audioEncrypted extends TL_audio_layer45 {
        public static int constructor = 0x555555F6;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            user_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
            key = stream.readByteArray(exception);
            iv = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) user_id);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
            stream.writeByteArray(key);
            stream.writeByteArray(iv);
        }
    }

    public static class TL_audio_old2 extends TL_audio_layer45 {
        public static int constructor = 0xc7ac6496;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            user_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            mime_type = stream.readString(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) user_id);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeString(mime_type);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_help_country extends TLObject {
        public static int constructor = 0xc3878e23;

        public int flags;
        public boolean hidden;
        public String iso2;
        public String default_name;
        public String name;
        public ArrayList<TL_help_countryCode> country_codes = new ArrayList<>();

        public static TL_help_country TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_help_country.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_help_country", constructor));
                } else {
                    return null;
                }
            }
            TL_help_country result = new TL_help_country();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            hidden = (flags & 1) != 0;
            iso2 = stream.readString(exception);
            default_name = stream.readString(exception);
            if ((flags & 2) != 0) {
                name = stream.readString(exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_help_countryCode object = TL_help_countryCode.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                country_codes.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = hidden ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(iso2);
            stream.writeString(default_name);
            if ((flags & 2) != 0) {
                stream.writeString(name);
            }
            stream.writeInt32(0x1cb5c415);
            int count = country_codes.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                country_codes.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class SecurePasswordKdfAlgo extends TLObject {

        public static SecurePasswordKdfAlgo TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            SecurePasswordKdfAlgo result = null;
            switch (constructor) {
                case 0xbbf2dda0:
                    result = new TL_securePasswordKdfAlgoPBKDF2HMACSHA512iter100000();
                    break;
                case 0x86471d92:
                    result = new TL_securePasswordKdfAlgoSHA512();
                    break;
                case 0x4a8537:
                    result = new TL_securePasswordKdfAlgoUnknown();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in SecurePasswordKdfAlgo", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_securePasswordKdfAlgoPBKDF2HMACSHA512iter100000 extends SecurePasswordKdfAlgo {
        public static int constructor = 0xbbf2dda0;

        public byte[] salt;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            salt = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(salt);
        }
    }

    public static class TL_securePasswordKdfAlgoSHA512 extends SecurePasswordKdfAlgo {
        public static int constructor = 0x86471d92;

        public byte[] salt;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            salt = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(salt);
        }
    }

    public static class TL_securePasswordKdfAlgoUnknown extends SecurePasswordKdfAlgo {
        public static int constructor = 0x4a8537;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_historyImport extends TLObject {
        public static int constructor = 0x1662af0b;

        public long id;

        public static TL_messages_historyImport TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_historyImport.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_historyImport", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_historyImport result = new TL_messages_historyImport();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static abstract class BotInfo extends TLObject {
        public long user_id;
        public String description;
        public ArrayList<TL_botCommand> commands = new ArrayList<>();
        public int version;
        public BotMenuButton menu_button;
        public int flags;
        public Photo description_photo;
        public Document description_document;

        public static BotInfo TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            BotInfo result = null;
            switch (constructor) {
                case 0xe4169b5d:
                    result = new TL_botInfo_layer140();
                    break;
                case 0xbb2e37ce:
                    result = new TL_botInfoEmpty_layer48();
                    break;
                case 0x98e81d3a:
                    result = new TL_botInfo_layer131();
                    break;
                case 0x9cf585d:
                    result = new TL_botInfo_layer48();
                    break;
                case 0x1b74b335:
                    result = new TL_botInfo_layer139();
                    break;
                case 0x8f300b57:
                    result = new TL_botInfo();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in BotInfo", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_botInfoEmpty_layer48 extends TL_botInfo {
        public static int constructor = 0xbb2e37ce;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_botInfo_layer131 extends TL_botInfo {
        public static int constructor = 0x98e81d3a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
            description = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_botCommand object = TL_botCommand.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                commands.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
            stream.writeString(description);
            stream.writeInt32(0x1cb5c415);
            int count = commands.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                commands.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_botInfo_layer48 extends TL_botInfo {
        public static int constructor = 0x9cf585d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
            version = stream.readInt32(exception);
            stream.readString(exception);
            description = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_botCommand object = TL_botCommand.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                commands.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
            stream.writeInt32(version);
            stream.writeString("");
            stream.writeString(description);
            stream.writeInt32(0x1cb5c415);
            int count = commands.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                commands.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_botInfo_layer139 extends BotInfo {
        public static int constructor = 0x1b74b335;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            description = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_botCommand object = TL_botCommand.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                commands.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeString(description);
            stream.writeInt32(0x1cb5c415);
            int count = commands.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                commands.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_botInfo extends BotInfo {
        public static int constructor = 0x8f300b57;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                user_id = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                description_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                description_document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_botCommand object = TL_botCommand.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    commands.add(object);
                }
            }
            if ((flags & 8) != 0) {
                menu_button = BotMenuButton.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt64(user_id);
            }
            if ((flags & 2) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                description_photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                description_document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = commands.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    commands.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 8) != 0) {
                menu_button.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInfo_layer140 extends TL_botInfo {
        public static int constructor = 0xe4169b5d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            description = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_botCommand object = TL_botCommand.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                commands.add(object);
            }
            menu_button = BotMenuButton.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeString(description);
            stream.writeInt32(0x1cb5c415);
            int count = commands.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                commands.get(a).serializeToStream(stream);
            }
            menu_button.serializeToStream(stream);
        }
    }

    public static abstract class InputGame extends TLObject {
        public InputUser bot_id;
        public String short_name;
        public long id;
        public long access_hash;

        public static InputGame TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputGame result = null;
            switch (constructor) {
                case 0xc331e80a:
                    result = new TL_inputGameShortName();
                    break;
                case 0x32c3e77:
                    result = new TL_inputGameID();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputGame", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputGameShortName extends InputGame {
        public static int constructor = 0xc331e80a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            bot_id = InputUser.TLdeserialize(stream, stream.readInt32(exception), exception);
            short_name = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            bot_id.serializeToStream(stream);
            stream.writeString(short_name);
        }
    }

    public static class TL_inputGameID extends InputGame {
        public static int constructor = 0x32c3e77;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static abstract class MessageReplies extends TLObject {

        public int flags;
        public boolean comments;
        public int replies;
        public int replies_pts;
        public ArrayList<Peer> recent_repliers = new ArrayList<>();
        public long channel_id;
        public int max_id;
        public int read_max_id;

        public static MessageReplies TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            MessageReplies result = null;
            switch (constructor) {
                case 0x4128faac:
                    result = new TL_messageReplies_layer131();
                    break;
                case 0x83d60fc2:
                    result = new TL_messageReplies();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in MessageReplies", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messageReplies_layer131 extends TL_messageReplies {
        public static int constructor = 0x4128faac;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            comments = (flags & 1) != 0;
            replies = stream.readInt32(exception);
            replies_pts = stream.readInt32(exception);
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    Peer object = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    recent_repliers.add(object);
                }
            }
            if ((flags & 1) != 0) {
                channel_id = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                max_id = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                read_max_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = comments ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt32(replies);
            stream.writeInt32(replies_pts);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_repliers.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    recent_repliers.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 1) != 0) {
                stream.writeInt32((int) channel_id);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(max_id);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(read_max_id);
            }
        }
    }

    public static class TL_messageReplies extends MessageReplies {
        public static int constructor = 0x83d60fc2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            comments = (flags & 1) != 0;
            replies = stream.readInt32(exception);
            replies_pts = stream.readInt32(exception);
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    Peer object = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    recent_repliers.add(object);
                }
            }
            if ((flags & 1) != 0) {
                channel_id = stream.readInt64(exception);
            }
            if ((flags & 4) != 0) {
                max_id = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                read_max_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = comments ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt32(replies);
            stream.writeInt32(replies_pts);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_repliers.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    recent_repliers.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 1) != 0) {
                stream.writeInt64(channel_id);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(max_id);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(read_max_id);
            }
        }
    }

    public static abstract class messages_SponsoredMessages extends TLObject {
        public int flags;
        public int posts_between;
        public ArrayList<TL_sponsoredMessage> messages = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static messages_SponsoredMessages TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_SponsoredMessages result = null;
            switch (constructor) {
                case 0x1839490f:
                    result = new TL_messages_sponsoredMessagesEmpty();
                    break;
                case 0xc9ee1d87:
                    result = new TL_messages_sponsoredMessages();
                    break;
                case 0x65a4c7d5:
                    result = new TL_messages_sponsoredMessagesLayer147();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_SponsoredMessages", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_sponsoredMessagesEmpty extends messages_SponsoredMessages {
        public static int constructor = 0x1839490f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_sponsoredMessagesLayer147 extends messages_SponsoredMessages {
        public static int constructor = 0xc9ee1d87;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_sponsoredMessage object = TL_sponsoredMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_sponsoredMessages extends messages_SponsoredMessages {
        public static int constructor = 0xc9ee1d87;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                posts_between = stream.readInt32(exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_sponsoredMessage object = TL_sponsoredMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                messages.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(posts_between);
            }
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                messages.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messageViews extends TLObject {
        public static int constructor = 0x455b853d;

        public int flags;
        public int views;
        public int forwards;
        public MessageReplies replies;

        public static TL_messageViews TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messageViews.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messageViews", constructor));
                } else {
                    return null;
                }
            }
            TL_messageViews result = new TL_messageViews();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                views = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                forwards = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                replies = MessageReplies.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(views);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(forwards);
            }
            if ((flags & 4) != 0) {
                replies.serializeToStream(stream);
            }
        }
    }

    public static abstract class ReplyMarkup extends TLObject {
        public int flags;
        public boolean resize;
        public boolean single_use;
        public boolean selective;
        public String placeholder;
        public ArrayList<TL_keyboardButtonRow> rows = new ArrayList<>();


        public static ReplyMarkup TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            ReplyMarkup result = null;
            switch (constructor) {
                case 0x85dd99d1:
                    result = new TL_replyKeyboardMarkup();
                    break;
                case 0xa03e5b85:
                    result = new TL_replyKeyboardHide();
                    break;
                case 0x86b40b08:
                    result = new TL_replyKeyboardForceReply();
                    break;
                case 0x3502758c:
                    result = new TL_replyKeyboardMarkup_layer129();
                    break;
                case 0xf4108aa0:
                    result = new TL_replyKeyboardForceReply_layer129();
                    break;
                case 0x48a30254:
                    result = new TL_replyInlineMarkup();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ReplyMarkup", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_replyKeyboardMarkup extends ReplyMarkup {
        public static int constructor = 0x85dd99d1;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            resize = (flags & 1) != 0;
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_keyboardButtonRow object = TL_keyboardButtonRow.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                rows.add(object);
            }
            if ((flags & 8) != 0) {
                placeholder = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = resize ? (flags | 1) : (flags &~ 1);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeInt32(0x1cb5c415);
            int count = rows.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                rows.get(a).serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeString(placeholder);
            }
        }
    }

    public static class TL_replyKeyboardHide extends ReplyMarkup {
        public static int constructor = 0xa03e5b85;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            selective = (flags & 4) != 0;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
        }
    }

    public static class TL_replyKeyboardForceReply extends ReplyMarkup {
        public static int constructor = 0x86b40b08;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
            if ((flags & 8) != 0) {
                placeholder = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            if ((flags & 8) != 0) {
                stream.writeString(placeholder);
            }
        }
    }

    public static class TL_replyKeyboardMarkup_layer129 extends TL_replyKeyboardMarkup {
        public static int constructor = 0x3502758c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            resize = (flags & 1) != 0;
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_keyboardButtonRow object = TL_keyboardButtonRow.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                rows.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = resize ? (flags | 1) : (flags &~ 1);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeInt32(0x1cb5c415);
            int count = rows.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                rows.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_replyKeyboardForceReply_layer129 extends TL_replyKeyboardForceReply {
        public static int constructor = 0xf4108aa0;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
        }
    }

    public static class TL_replyInlineMarkup extends ReplyMarkup {
        public static int constructor = 0x48a30254;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_keyboardButtonRow object = TL_keyboardButtonRow.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                rows.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = rows.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                rows.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_webPageAttributeTheme extends TLObject {
        public static int constructor = 0x54b56617;

        public int flags;
        public ArrayList<Document> documents = new ArrayList<>();
        public ThemeSettings settings;

        public static TL_webPageAttributeTheme TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_webPageAttributeTheme.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_webPageAttributeTheme", constructor));
                } else {
                    return null;
                }
            }
            TL_webPageAttributeTheme result = new TL_webPageAttributeTheme();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    documents.add(object);
                }
            }
            if ((flags & 2) != 0) {
                settings = ThemeSettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = documents.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    documents.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 2) != 0) {
                settings.serializeToStream(stream);
            }
        }
    }

    public static abstract class contacts_Contacts extends TLObject {
        public ArrayList<TL_contact> contacts = new ArrayList<>();
        public int saved_count;
        public ArrayList<User> users = new ArrayList<>();

        public static contacts_Contacts TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            contacts_Contacts result = null;
            switch (constructor) {
                case 0xb74ba9d2:
                    result = new TL_contacts_contactsNotModified();
                    break;
                case 0xeae87e42:
                    result = new TL_contacts_contacts();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in contacts_Contacts", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_contacts_contactsNotModified extends contacts_Contacts {
        public static int constructor = 0xb74ba9d2;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contacts_contacts extends contacts_Contacts {
        public static int constructor = 0xeae87e42;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_contact object = TL_contact.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                contacts.add(object);
            }
            saved_count = stream.readInt32(exception);
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = contacts.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                contacts.get(a).serializeToStream(stream);
            }
            stream.writeInt32(saved_count);
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class SecureRequiredType extends TLObject {

        public static SecureRequiredType TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            SecureRequiredType result = null;
            switch (constructor) {
                case 0x829d99da:
                    result = new TL_secureRequiredType();
                    break;
                case 0x27477b4:
                    result = new TL_secureRequiredTypeOneOf();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in SecureRequiredType", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_secureRequiredType extends SecureRequiredType {
        public static int constructor = 0x829d99da;

        public int flags;
        public boolean native_names;
        public boolean selfie_required;
        public boolean translation_required;
        public SecureValueType type;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            native_names = (flags & 1) != 0;
            selfie_required = (flags & 2) != 0;
            translation_required = (flags & 4) != 0;
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = native_names ? (flags | 1) : (flags &~ 1);
            flags = selfie_required ? (flags | 2) : (flags &~ 2);
            flags = translation_required ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
        }
    }

    public static class TL_secureRequiredTypeOneOf extends SecureRequiredType {
        public static int constructor = 0x27477b4;

        public ArrayList<SecureRequiredType> types = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                SecureRequiredType object = SecureRequiredType.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                types.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = types.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                types.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class InputPrivacyKey extends TLObject {

        public static InputPrivacyKey TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputPrivacyKey result = null;
            switch (constructor) {
                case 0x4f96cb18:
                    result = new TL_inputPrivacyKeyStatusTimestamp();
                    break;
                case 0xbdfb0426:
                    result = new TL_inputPrivacyKeyChatInvite();
                    break;
                case 0xfabadc5f:
                    result = new TL_inputPrivacyKeyPhoneCall();
                    break;
                case 0xa4dd4c08:
                    result = new TL_inputPrivacyKeyForwards();
                    break;
                case 0x5719bacc:
                    result = new TL_inputPrivacyKeyProfilePhoto();
                    break;
                case 0x352dafa:
                    result = new TL_inputPrivacyKeyPhoneNumber();
                    break;
                case 0xd1219bdd:
                    result = new TL_inputPrivacyKeyAddedByPhone();
                    break;
                case 0xaee69d68:
                    result = new TL_inputPrivacyKeyVoiceMessages();
                    break;
                case 0xdb9e70d2:
                    result = new TL_inputPrivacyKeyPhoneP2P();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputPrivacyKey", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputPrivacyKeyStatusTimestamp extends InputPrivacyKey {
        public static int constructor = 0x4f96cb18;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyChatInvite extends InputPrivacyKey {
        public static int constructor = 0xbdfb0426;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyPhoneCall extends InputPrivacyKey {
        public static int constructor = 0xfabadc5f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyForwards extends InputPrivacyKey {
        public static int constructor = 0xa4dd4c08;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyPhoneNumber extends InputPrivacyKey {
        public static int constructor = 0x352dafa;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyAddedByPhone extends InputPrivacyKey {
        public static int constructor = 0xd1219bdd;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyVoiceMessages extends InputPrivacyKey {
        public static int constructor = 0xaee69d68;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyProfilePhoto extends InputPrivacyKey {
        public static int constructor = 0x5719bacc;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyPhoneP2P extends InputPrivacyKey {
        public static int constructor = 0xdb9e70d2;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class messages_ExportedChatInvite extends TLObject {

        public ExportedChatInvite invite;
        public ArrayList<User> users = new ArrayList<>();

        public static messages_ExportedChatInvite TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_ExportedChatInvite result = null;
            switch (constructor) {
                case 0x222600ef:
                    result = new TL_messages_exportedChatInviteReplaced();
                    break;
                case 0x1871be50:
                    result = new TL_messages_exportedChatInvite();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_ExportedChatInvite", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_exportedChatInviteReplaced extends messages_ExportedChatInvite {
        public static int constructor = 0x222600ef;

        public ExportedChatInvite new_invite;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            new_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            invite.serializeToStream(stream);
            new_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_exportedChatInvite extends messages_ExportedChatInvite {
        public static int constructor = 0x1871be50;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class InputTheme extends TLObject {

        public static InputTheme TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputTheme result = null;
            switch (constructor) {
                case 0xf5890df1:
                    result = new TL_inputThemeSlug();
                    break;
                case 0x3c5693e9:
                    result = new TL_inputTheme();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputTheme", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputThemeSlug extends InputTheme {
        public static int constructor = 0xf5890df1;

        public String slug;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            slug = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(slug);
        }
    }

    public static class TL_inputTheme extends InputTheme {
        public static int constructor = 0x3c5693e9;

        public long id;
        public long access_hash;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static abstract class photos_Photos extends TLObject {
        public ArrayList<Photo> photos = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public int count;

        public static photos_Photos TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            photos_Photos result = null;
            switch (constructor) {
                case 0x8dca6aa5:
                    result = new TL_photos_photos();
                    break;
                case 0x15051f54:
                    result = new TL_photos_photosSlice();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in photos_Photos", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_photos_photos extends photos_Photos {
        public static int constructor = 0x8dca6aa5;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_photos_photosSlice extends photos_Photos {
        public static int constructor = 0x15051f54;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            stream.writeInt32(0x1cb5c415);
            int count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class ChatFull extends TLObject {
        public long id;
        public ChatParticipants participants;
        public Photo chat_photo;
        public PeerNotifySettings notify_settings;
        public TL_chatInviteExported exported_invite;
        public ArrayList<BotInfo> bot_info = new ArrayList<>();
        public int flags;
        public boolean can_view_participants;
        public boolean can_set_username;
        public boolean has_scheduled;
        public String about;
        public int participants_count;
        public int admins_count;
        public int read_inbox_max_id;
        public int read_outbox_max_id;
        public int unread_count;
        public long migrated_from_chat_id;
        public int migrated_from_max_id;
        public int pinned_msg_id;
        public int kicked_count;
        public int unread_important_count;
        public int folder_id;
        public boolean can_set_stickers;
        public boolean hidden_prehistory;
        public boolean can_view_stats;
        public boolean can_set_location;
        public boolean blocked;
        public int banned_count;
        public int online_count;
        public StickerSet stickerset;
        public int available_min_id;
        public int call_msg_id;
        public long linked_chat_id;
        public ChannelLocation location;
        public int slowmode_seconds;
        public int slowmode_next_send_date;
        public int stats_dc;
        public int pts;
        public TL_inputGroupCall call;
        public int ttl_period;
        public ArrayList<String> pending_suggestions = new ArrayList<>();
        public Peer groupcall_default_join_as;
        public ArrayList<Long> recent_requesters = new ArrayList<>();
        public String theme_emoticon;
        public int requests_pending;
        public Peer default_send_as;
        public ArrayList<String> available_reactions_legacy = new ArrayList<>();
        public int flags2;
        public boolean can_delete_channel;
        public boolean antispam;
        public boolean participants_hidden;
        public boolean translations_disabled;
        public ChatReactions available_reactions;

        public long inviterId; //custom
        public int invitesCount; //custom

        public static ChatFull TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            ChatFull result = null;
            switch (constructor) {
                case 0xc9d31138:
                    result = new TL_chatFull();
                    break;
                case 0xf2355507:
                    result = new TL_channelFull();
                    break;
                case 0xd18ee226:
                    result = new TL_chatFull_layer144();
                    break;
                case 0xea68a619:
                    result = new TL_channelFull_layer144();
                    break;
                case 0xe13c3d20:
                    result = new TL_channelFull_layer139();
                    break;
                case 0x56662e2e:
                    result = new TL_channelFull_layer135();
                    break;
                case 0x59cff963:
                    result = new TL_channelFull_layer134();
                    break;
                case 0x1c87a71a:
                    result = new TL_channelFull_layer98();
                    break;
                case 0x3648977:
                    result = new TL_channelFull_layer99();
                    break;
                case 0x2e02a614:
                    result = new TL_chatFull_layer87();
                    break;
                case 0xef3a6acd:
                    result = new TL_channelFull_layer122();
                    break;
                case 0xf0e6672a:
                    result = new TL_channelFull_layer121();
                    break;
                case 0x2d895c74:
                    result = new TL_channelFull_layer110();
                    break;
                case 0x10916653:
                    result = new TL_channelFull_layer103();
                    break;
                case 0x9882e516:
                    result = new TL_channelFull_layer101();
                    break;
                case 0x17f45fcf:
                    result = new TL_channelFull_layer71();
                    break;
                case 0x76af5481:
                    result = new TL_channelFull_layer72();
                    break;
                case 0x97bee562:
                    result = new TL_channelFull_layer52();
                    break;
                case 0xc3d5512f:
                    result = new TL_channelFull_layer67();
                    break;
                case 0x9e341ddf:
                    result = new TL_channelFull_layer48();
                    break;
                case 0xdc8c181:
                    result = new TL_chatFull_layer122();
                    break;
                case 0xf3474af6:
                    result = new TL_chatFull_layer123();
                    break;
                case 0x7a7de4f7:
                    result = new TL_channelFull_layer123();
                    break;
                case 0xf06c4018:
                    result = new TL_chatFull_layer124();
                    break;
                case 0x2548c037:
                    result = new TL_channelFull_layer124();
                    break;
                case 0x8a1e2983:
                    result = new TL_chatFull_layer131();
                    break;
                case 0x548c3f93:
                    result = new TL_channelFull_layer131();
                    break;
                case 0x49a0a5d9:
                    result = new TL_chatFull_layer132();
                    break;
                case 0x2f532f3c:
                    result = new TL_channelFull_layer132();
                    break;
                case 0x4dbdc099:
                    result = new TL_chatFull_layer133();
                    break;
                case 0xe9b27a17:
                    result = new TL_channelFull_layer133();
                    break;
                case 0x1b7c9db3:
                    result = new TL_chatFull_layer121();
                    break;
                case 0x22a235da:
                    result = new TL_chatFull_layer98();
                    break;
                case 0xedd2a791:
                    result = new TL_chatFull_layer92();
                    break;
                case 0x46a6ffb4:
                    result = new TL_chatFull_layer135();
                    break;
                case 0xfab31aa3:
                    result = new TL_channelFull_old();
                    break;
                case 0x95cb5f57:
                    result = new TL_channelFull_layer70();
                    break;
                case 0xcbb62890:
                    result = new TL_channelFull_layer89();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ChatFull", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
                if (result.available_reactions == null) {
                    if (!result.available_reactions_legacy.isEmpty()) {
                        TL_chatReactionsSome someReactions = new TL_chatReactionsSome();
                        for (int i = 0; i < result.available_reactions_legacy.size(); i++) {
                            TL_reactionEmoji reaction = new TL_reactionEmoji();
                            reaction.emoticon = result.available_reactions_legacy.get(i);
                            someReactions.reactions.add(reaction);
                        }
                        result.available_reactions = someReactions;
                    } else {
                        result.available_reactions = new TL_chatReactionsNone();
                    }
                }
            }
            return result;
        }
    }

    public static class TL_channelFull_layer110 extends ChatFull {
        public static int constructor = 0x2d895c74;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_chatFull_layer124 extends TL_chatFull {
        public static int constructor = 0xf06c4018;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
        }
    }

    public static class TL_channelFull_layer124 extends TL_channelFull {
        public static int constructor = 0x2548c037;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
        }
    }

    public static class TL_chatFull_layer123 extends TL_chatFull {
        public static int constructor = 0xf3474af6;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer123 extends TL_channelFull {
        public static int constructor = 0x7a7de4f7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer131 extends TL_chatFull {
        public static int constructor = 0x8a1e2983;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer132 extends TL_chatFull {
        public static int constructor = 0x49a0a5d9;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_channelFull_layer132 extends TL_channelFull {
        public static int constructor = 0x2f532f3c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_chatFull_layer133 extends ChatFull {
        public static int constructor = 0x4dbdc099;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_chatFull_layer135 extends ChatFull {
        public static int constructor = 0x46a6ffb4;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
        }
    }

    public static class TL_channelFull_layer135 extends ChatFull {
        public static int constructor = 0x56662e2e;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer134 extends ChatFull {
        public static int constructor = 0x59cff963;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
        }
    }

    public static class TL_channelFull_layer133 extends ChatFull {
        public static int constructor = 0xe9b27a17;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_chatFull extends ChatFull {
        public static int constructor = 0xc9d31138;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 262144) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
            if ((flags & 262144) != 0) {
                available_reactions.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer144 extends ChatFull {
        public static int constructor = 0xd18ee226;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 262144) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    available_reactions_legacy.add(stream.readString(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = available_reactions_legacy.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(available_reactions_legacy.get(a));
                }
            }
        }
    }

    public static class TL_channelFull extends ChatFull {
        public static int constructor = 0xf2355507;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer144 extends ChatFull {
        public static int constructor = 0xea68a619;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    available_reactions_legacy.add(stream.readString(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = available_reactions_legacy.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(available_reactions_legacy.get(a));
                }
            }
        }
    }

    public static class TL_channelFull_layer139 extends ChatFull {
        public static int constructor = 0xe13c3d20;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    recent_requesters.add(stream.readInt64(exception));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    available_reactions_legacy.add(stream.readString(exception));
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = recent_requesters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_requesters.get(a));
                }
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = available_reactions_legacy.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(available_reactions_legacy.get(a));
                }
            }
        }
    }

    public static class TL_channelFull_layer131 extends TL_channelFull {
        public static int constructor = 0x548c3f93;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    pending_suggestions.add(stream.readString(exception));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                stream.writeInt32(0x1cb5c415);
                count = pending_suggestions.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeString(pending_suggestions.get(a));
                }
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer122 extends TL_channelFull {
        public static int constructor = 0xef3a6acd;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if (exported_invite != null) {
                exported_invite.serializeToStream(stream);
            } else {
                new TLRPC.TL_chatInviteEmpty_layer122().serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer121 extends TL_channelFull {
        public static int constructor = 0xf0e6672a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer103 extends TL_channelFull {
        public static int constructor = 0x10916653;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            can_set_location = (flags & 65536) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer101 extends TL_channelFull {
        public static int constructor = 0x9882e516;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer99 extends TL_channelFull {
        public static int constructor = 0x3648977;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer98 extends TL_channelFull {
        public static int constructor = 0x1c87a71a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
        }
    }

    public static class TL_chatFull_layer87 extends TL_chatFull {
        public static int constructor = 0x2e02a614;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            participants.serializeToStream(stream);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer89 extends TL_channelFull {
        public static int constructor = 0xcbb62890;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                call_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(call_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer72 extends TL_channelFull {
        public static int constructor = 0x76af5481;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
        }
    }

    public static class TL_channelFull_layer71 extends TL_channelFull {
        public static int constructor = 0x17f45fcf;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer122 extends TL_chatFull {
        public static int constructor = 0xdc8c181;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer121 extends TL_chatFull {
        public static int constructor = 0x1b7c9db3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
        }
    }

    public static class TL_chatFull_layer98 extends TL_chatFull {
        public static int constructor = 0x22a235da;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_chatFull_layer92 extends TL_chatFull {
        public static int constructor = 0xedd2a791;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt32(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    bot_info.add(object);
                }
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = bot_info.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    bot_info.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer70 extends TL_channelFull {
        public static int constructor = 0x95cb5f57;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer52 extends TL_channelFull {
        public static int constructor = 0x97bee562;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            unread_important_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(unread_count);
            stream.writeInt32(unread_important_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer67 extends TL_channelFull {
        public static int constructor = 0xc3d5512f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer48 extends TL_channelFull {
        public static int constructor = 0x9e341ddf;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            unread_important_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                BotInfo object = BotInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                bot_info.add(object);
            }
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(unread_count);
            stream.writeInt32(unread_important_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = bot_info.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                bot_info.get(a).serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
        }
    }

    public static class TL_channelFull_old extends TL_channelFull {
        public static int constructor = 0xfab31aa3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            unread_important_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(unread_count);
            stream.writeInt32(unread_important_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
        }
    }

    public static class TL_inputPeerNotifySettings extends TLObject {
        public static int constructor = 0xdf1f002b;

        public int flags;
        public boolean show_previews;
        public boolean silent;
        public int mute_until;
        public NotificationSound sound;

        public static TL_inputPeerNotifySettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_inputPeerNotifySettings.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_inputPeerNotifySettings", constructor));
                } else {
                    return null;
                }
            }
            TL_inputPeerNotifySettings result = new TL_inputPeerNotifySettings();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                sound.serializeToStream(stream);
            }
        }
    }

    public static abstract class auth_LoginToken extends TLObject {

        public static auth_LoginToken TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            auth_LoginToken result = null;
            switch (constructor) {
                case 0x629f1980:
                    result = new TL_auth_loginToken();
                    break;
                case 0x68e9916:
                    result = new TL_auth_loginTokenMigrateTo();
                    break;
                case 0x390d5c5e:
                    result = new TL_auth_loginTokenSuccess();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in auth_LoginToken", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_auth_loginToken extends auth_LoginToken {
        public static int constructor = 0x629f1980;

        public int expires;
        public byte[] token;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            expires = stream.readInt32(exception);
            token = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(expires);
            stream.writeByteArray(token);
        }
    }

    public static class TL_auth_loginTokenMigrateTo extends auth_LoginToken {
        public static int constructor = 0x68e9916;

        public int dc_id;
        public byte[] token;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            dc_id = stream.readInt32(exception);
            token = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(dc_id);
            stream.writeByteArray(token);
        }
    }

    public static class TL_auth_loginTokenSuccess extends auth_LoginToken {
        public static int constructor = 0x390d5c5e;

        public auth_Authorization authorization;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            authorization = auth_Authorization.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            authorization.serializeToStream(stream);
        }
    }

    public static class TL_codeSettings extends TLObject {
        public static int constructor = 0xad253d78;

        public int flags;
        public boolean allow_flashcall;
        public boolean current_number;
        public boolean allow_app_hash;
        public boolean allow_missed_call;
        public boolean allow_firebase;
        public ArrayList<byte[]> logout_tokens = new ArrayList<>();
        public String token;
        public boolean app_sandbox;

        public static TL_codeSettings TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_codeSettings.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_codeSettings", constructor));
                } else {
                    return null;
                }
            }
            TL_codeSettings result = new TL_codeSettings();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            allow_flashcall = (flags & 1) != 0;
            current_number = (flags & 2) != 0;
            allow_app_hash = (flags & 16) != 0;
            allow_missed_call = (flags & 32) != 0;
            allow_firebase = (flags & 128) != 0;
            if ((flags & 64) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    byte[] object = stream.readByteArray(exception);
                    if (object == null) {
                        return;
                    }
                    logout_tokens.add(object);
                }
            }
            if ((flags & 256) != 0) {
                token = stream.readString(exception);
            }
            app_sandbox = (flags & 256) != 0;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = allow_flashcall ? (flags | 1) : (flags &~ 1);
            flags = current_number ? (flags | 2) : (flags &~ 2);
            flags = allow_app_hash ? (flags | 16) : (flags &~ 16);
            flags = allow_missed_call ? (flags | 32) : (flags &~ 32);
            flags = allow_firebase ? (flags | 128) : (flags &~ 128);
            flags = app_sandbox ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = logout_tokens.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeByteArray(logout_tokens.get(a));
                }
            }
            if ((flags & 256) != 0) {
                stream.writeString(token);
            }
        }
    }

    public static abstract class account_EmojiStatuses extends TLObject {

        public long hash;
        public ArrayList<EmojiStatus> statuses = new ArrayList<>();

        public static account_EmojiStatuses TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            account_EmojiStatuses result = null;
            switch (constructor) {
                case 0xd08ce645:
                    result = new TL_account_emojiStatusesNotModified();
                    break;
                case 0x90c467d1:
                    result = new TL_account_emojiStatuses();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in account_EmojiStatuses", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_account_emojiStatusesNotModified extends account_EmojiStatuses {
        public static int constructor = 0xd08ce645;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_account_emojiStatuses extends account_EmojiStatuses {
        public static int constructor = 0x90c467d1;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            hash = stream.readInt64(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                EmojiStatus object = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                statuses.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(hash);
            stream.writeInt32(0x1cb5c415);
            int count = statuses.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                statuses.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_stats_messageStats extends TLObject {
        public static int constructor = 0x8999f295;

        public StatsGraph views_graph;

        public static TL_stats_messageStats TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_stats_messageStats.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_stats_messageStats", constructor));
                } else {
                    return null;
                }
            }
            TL_stats_messageStats result = new TL_stats_messageStats();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            views_graph = StatsGraph.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            views_graph.serializeToStream(stream);
        }
    }

    public static abstract class BaseTheme extends TLObject {

        public static BaseTheme TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            BaseTheme result = null;
            switch (constructor) {
                case 0x5b11125a:
                    result = new TL_baseThemeArctic();
                    break;
                case 0xb7b31ea8:
                    result = new TL_baseThemeNight();
                    break;
                case 0xc3a12462:
                    result = new TL_baseThemeClassic();
                    break;
                case 0x6d5f77ee:
                    result = new TL_baseThemeTinted();
                    break;
                case 0xfbd81688:
                    result = new TL_baseThemeDay();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in BaseTheme", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_baseThemeArctic extends BaseTheme {
        public static int constructor = 0x5b11125a;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeNight extends BaseTheme {
        public static int constructor = 0xb7b31ea8;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeClassic extends BaseTheme {
        public static int constructor = 0xc3a12462;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeTinted extends BaseTheme {
        public static int constructor = 0x6d5f77ee;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeDay extends BaseTheme {
        public static int constructor = 0xfbd81688;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_null extends TLObject {
        public static int constructor = 0x56730bcc;


        public static TL_null TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_null.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_null", constructor));
                } else {
                    return null;
                }
            }
            TL_null result = new TL_null();
            result.readParams(stream, exception);
            return result;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class Page extends TLObject {

        public int flags;
        public boolean part;
        public boolean rtl;
        public String url;
        public ArrayList<PageBlock> blocks = new ArrayList<>();
        public ArrayList<Photo> photos = new ArrayList<>();
        public ArrayList<Document> documents = new ArrayList<>();
        public boolean v2;
        public int views;

        public static Page TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            Page result = null;
            switch (constructor) {
                case 0x8dee6c44:
                    result = new TL_pagePart_layer67();
                    break;
                case 0x8e3f9ebe:
                    result = new TL_pagePart_layer82();
                    break;
                case 0xd7a19d69:
                    result = new TL_pageFull_layer67();
                    break;
                case 0x556ec7aa:
                    result = new TL_pageFull_layer82();
                    break;
                case 0xae891bec:
                    result = new TL_page_layer110();
                    break;
                case 0x98657f0d:
                    result = new TL_page();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in Page", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_pagePart_layer67 extends TL_pagePart_layer82 {
        public static int constructor = 0x8dee6c44;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PageBlock object = PageBlock.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocks.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = blocks.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocks.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_pagePart_layer82 extends Page {
        public static int constructor = 0x8e3f9ebe;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PageBlock object = PageBlock.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocks.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = blocks.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocks.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_pageFull_layer67 extends TL_page {
        public static int constructor = 0xd7a19d69;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PageBlock object = PageBlock.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocks.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = blocks.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocks.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_pageFull_layer82 extends TL_page {
        public static int constructor = 0x556ec7aa;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PageBlock object = PageBlock.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocks.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = blocks.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocks.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_page_layer110 extends TL_page {
        public static int constructor = 0xae891bec;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            part = (flags & 1) != 0;
            rtl = (flags & 2) != 0;
            url = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PageBlock object = PageBlock.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocks.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = part ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(url);
            stream.writeInt32(0x1cb5c415);
            int count = blocks.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocks.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_page extends Page {
        public static int constructor = 0x98657f0d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            part = (flags & 1) != 0;
            rtl = (flags & 2) != 0;
            v2 = (flags & 4) != 0;
            url = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PageBlock object = PageBlock.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                blocks.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Photo object = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                photos.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                documents.add(object);
            }
            if ((flags & 8) != 0) {
                views = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = part ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 2) : (flags &~ 2);
            flags = v2 ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeString(url);
            stream.writeInt32(0x1cb5c415);
            int count = blocks.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                blocks.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = photos.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                photos.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = documents.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                documents.get(a).serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(views);
            }
        }
    }

    public static class TL_topPeerCategoryPeers extends TLObject {
        public static int constructor = 0xfb834291;

        public TopPeerCategory category;
        public int count;
        public ArrayList<TL_topPeer> peers = new ArrayList<>();

        public static TL_topPeerCategoryPeers TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_topPeerCategoryPeers.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_topPeerCategoryPeers", constructor));
                } else {
                    return null;
                }
            }
            TL_topPeerCategoryPeers result = new TL_topPeerCategoryPeers();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            category = TopPeerCategory.TLdeserialize(stream, stream.readInt32(exception), exception);
            count = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_topPeer object = TL_topPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                peers.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            category.serializeToStream(stream);
            stream.writeInt32(count);
            stream.writeInt32(0x1cb5c415);
            int count = peers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                peers.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class InputUser extends TLObject {

        public long user_id;
        public long access_hash;

        public static InputUser TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputUser result = null;
            switch (constructor) {
                case 0xf7c1b13f:
                    result = new TL_inputUserSelf();
                    break;
                case 0xf21158c6:
                    result = new TL_inputUser();
                    break;
                case 0xb98886cf:
                    result = new TL_inputUserEmpty();
                    break;
                case 0x1da448e2:
                    result = new TL_inputUserFromMessage();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputUser", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputUserSelf extends InputUser {
        public static int constructor = 0xf7c1b13f;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputUser extends InputUser {
        public static int constructor = 0xf21158c6;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputUserEmpty extends InputUser {
        public static int constructor = 0xb98886cf;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputUserFromMessage extends InputUser {
        public static int constructor = 0x1da448e2;

        public InputPeer peer;
        public int msg_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            msg_id = stream.readInt32(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(msg_id);
            stream.writeInt64(user_id);
        }
    }

    public static abstract class KeyboardButton extends TLObject {
        public String text;
        public String url;
        public int flags;
        public boolean same_peer;
        public String query;
        public byte[] data;
        public int button_id;
        public boolean request_write_access;
        public InputUser bot;
        public String fwd_text;
        public boolean quiz;
        public boolean requires_password;
        public long user_id;
        public InputUser inputUser;

        public static KeyboardButton TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            KeyboardButton result = null;
            switch (constructor) {
                case 0xa0c0505c:
                    result = new TL_keyboardButtonSimpleWebView();
                    break;
                case 0xb16a6c29:
                    result = new TL_keyboardButtonRequestPhone();
                    break;
                case 0x50f41ccf:
                    result = new TL_keyboardButtonGame();
                    break;
                case 0x258aff05:
                    result = new TL_keyboardButtonUrl();
                    break;
                case 0x568a748:
                    result = new TL_keyboardButtonSwitchInline();
                    break;
                case 0xfc796b3f:
                    result = new TL_keyboardButtonRequestGeoLocation();
                    break;
                case 0x10b78d29:
                    result = new TL_keyboardButtonUrlAuth();
                    break;
                case 0xd02e7fd4:
                    result = new TL_inputKeyboardButtonUrlAuth();
                    break;
                case 0xbbc7515d:
                    result = new TL_keyboardButtonRequestPoll();
                    break;
                case 0xafd93fbb:
                    result = new TL_keyboardButtonBuy();
                    break;
                case 0x683a5e46:
                    result = new TL_keyboardButtonCallback_layer117();
                    break;
                case 0xa2fa4880:
                    result = new TL_keyboardButton();
                    break;
                case 0x35bbdb6b:
                    result = new TL_keyboardButtonCallback();
                    break;
                case 0xe988037b:
                    result = new TL_inputKeyboardButtonUserProfile();
                    break;
                case 0x308660c1:
                    result = new TL_keyboardButtonUserProfile();
                    break;
                case 0x13767230:
                    result = new TL_keyboardButtonWebView();
                    break;
                case 0xd0b468c:
                    result = new TL_keyboardButtonRequestPeer();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in KeyboardButton", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_keyboardButtonRequestPhone extends KeyboardButton {
        public static int constructor = 0xb16a6c29;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonGame extends KeyboardButton {
        public static int constructor = 0x50f41ccf;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonUrl extends KeyboardButton {
        public static int constructor = 0x258aff05;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeString(url);
        }
    }

    public static class TL_keyboardButtonSwitchInline extends KeyboardButton {
        public static int constructor = 0x568a748;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            same_peer = (flags & 1) != 0;
            text = stream.readString(exception);
            query = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = same_peer ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            stream.writeString(query);
        }
    }

    public static class TL_keyboardButtonRequestGeoLocation extends KeyboardButton {
        public static int constructor = 0xfc796b3f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonUrlAuth extends KeyboardButton {
        public static int constructor = 0x10b78d29;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            text = stream.readString(exception);
            if ((flags & 1) != 0) {
                fwd_text = stream.readString(exception);
            }
            url = stream.readString(exception);
            button_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(text);
            if ((flags & 1) != 0) {
                stream.writeString(fwd_text);
            }
            stream.writeString(url);
            stream.writeInt32(button_id);
        }
    }

    public static class TL_inputKeyboardButtonUrlAuth extends KeyboardButton {
        public static int constructor = 0xd02e7fd4;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            request_write_access = (flags & 1) != 0;
            text = stream.readString(exception);
            if ((flags & 2) != 0) {
                fwd_text = stream.readString(exception);
            }
            url = stream.readString(exception);
            bot = InputUser.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = request_write_access ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            if ((flags & 2) != 0) {
                stream.writeString(fwd_text);
            }
            stream.writeString(url);
            bot.serializeToStream(stream);
        }
    }

    public static class TL_keyboardButtonRequestPoll extends KeyboardButton {
        public static int constructor = 0xbbc7515d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                quiz = stream.readBool(exception);
            }
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(quiz);
            }
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonBuy extends KeyboardButton {
        public static int constructor = 0xafd93fbb;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_inputKeyboardButtonUserProfile extends KeyboardButton {
        public static int constructor = 0xe988037b;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            inputUser = InputUser.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            inputUser.serializeToStream(stream);
        }
    }

    public static class TL_keyboardButtonUserProfile extends KeyboardButton {
        public static int constructor = 0x308660c1;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_keyboardButtonCallback extends KeyboardButton {
        public static int constructor = 0x35bbdb6b;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            requires_password = (flags & 1) != 0;
            text = stream.readString(exception);
            data = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = requires_password ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            stream.writeByteArray(data);
        }
    }

    public static class TL_keyboardButtonCallback_layer117 extends TL_keyboardButtonCallback {
        public static int constructor = 0x683a5e46;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            data = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeByteArray(data);
        }
    }

    public static class TL_keyboardButton extends KeyboardButton {
        public static int constructor = 0xa2fa4880;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_videoSizeEmojiMarkup extends VideoSize {
        public static int constructor = 0xf85c413c;

        public long emoji_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            emoji_id = stream.readInt64(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                background_colors.add(stream.readInt32(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(emoji_id);
            stream.writeInt32(0x1cb5c415);
            int count = background_colors.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32(background_colors.get(a));
            }
        }
    }

    public static class TL_videoSizeStickerMarkup extends VideoSize {
        public static int constructor = 0xda082fe;

        public InputStickerSet stickerset;
        public long sticker_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            sticker_id = stream.readInt64(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                background_colors.add(stream.readInt32(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stickerset.serializeToStream(stream);
            stream.writeInt64(sticker_id);
            stream.writeInt32(0x1cb5c415);
            int count = background_colors.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32(background_colors.get(a));
            }
        }
    }

    public static abstract class VideoSize extends TLObject {

        public int flags;
        public String type;
        public FileLocation location;
        public int w;
        public int h;
        public int size;
        public double video_start_ts;
        public ArrayList<Integer> background_colors = new ArrayList<>();

        public static VideoSize TLdeserialize(long photo_id, long document_id, AbstractSerializedData stream, int constructor, boolean exception) {
            VideoSize result = null;
            switch (constructor) {
                case 0x435bb987:
                    result = new TL_videoSize_layer115();
                    break;
                case 0xe831c556:
                    result = new TL_videoSize_layer127();
                    break;
                case 0xde33b094:
                    result = new TL_videoSize();
                    break;
                case 0xda082fe:
                    result = new TL_videoSizeStickerMarkup();
                    break;
                case 0xf85c413c:
                    result = new TL_videoSizeEmojiMarkup();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in VideoSize", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
                if (result.location == null) {
                    if (!TextUtils.isEmpty(result.type) && (photo_id != 0 || document_id != 0)) {
                        result.location = new TL_fileLocationToBeDeprecated();
                        if (photo_id != 0) {
                            result.location.volume_id = -photo_id;
                            result.location.local_id = result.type.charAt(0);
                        } else {
                            result.location.volume_id = -document_id;
                            result.location.local_id = 1000 + result.type.charAt(0);
                        }
                    } else {
                        result.location = new TL_fileLocationUnavailable();
                    }
                }
            }
            return result;
        }
    }

    public static class TL_videoSize_layer115 extends TL_videoSize {
        public static int constructor = 0x435bb987;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = stream.readString(exception);
            location = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            size = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(type);
            location.serializeToStream(stream);
            stream.writeInt32(w);
            stream.writeInt32(h);
            stream.writeInt32(size);
        }
    }

    public static class TL_videoSize_layer127 extends TL_videoSize {
        public static int constructor = 0xe831c556;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = stream.readString(exception);
            location = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            size = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                video_start_ts = stream.readDouble(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(type);
            location.serializeToStream(stream);
            stream.writeInt32(w);
            stream.writeInt32(h);
            stream.writeInt32(size);
            if ((flags & 1) != 0) {
                stream.writeDouble(video_start_ts);
            }
        }
    }

    public static class TL_videoSize extends VideoSize {
        public static int constructor = 0xde33b094;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = stream.readString(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            size = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                video_start_ts = stream.readDouble(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(type);
            stream.writeInt32(w);
            stream.writeInt32(h);
            stream.writeInt32(size);
            if ((flags & 1) != 0) {
                stream.writeDouble(video_start_ts);
            }
        }
    }

    public static abstract class BotInlineMessage extends TLObject {
        public int flags;
        public GeoPoint geo;
        public String title;
        public String address;
        public String provider;
        public String venue_id;
        public String venue_type;
        public ReplyMarkup reply_markup;
        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public String phone_number;
        public String first_name;
        public String last_name;
        public String vcard;
        public boolean no_webpage;
        public int period;
        public int heading;
        public int proximity_notification_radius;

        public static BotInlineMessage TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            BotInlineMessage result = null;
            switch (constructor) {
                case 0x4366232e:
                    result = new TL_botInlineMessageMediaVenue_layer77();
                    break;
                case 0x8a86659c:
                    result = new TL_botInlineMessageMediaVenue();
                    break;
                case 0x3a8fd8b8:
                    result = new TL_botInlineMessageMediaGeo_layer71();
                    break;
                case 0x764cf810:
                    result = new TL_botInlineMessageMediaAuto();
                    break;
                case 0xa74b15b:
                    result = new TL_botInlineMessageMediaAuto_layer74();
                    break;
                case 0x35edb4d4:
                    result = new TL_botInlineMessageMediaContact_layer81();
                    break;
                case 0x18d1cdc2:
                    result = new TL_botInlineMessageMediaContact();
                    break;
                case 0x8c7f65e2:
                    result = new TL_botInlineMessageText();
                    break;
                case 0xb722de65:
                    result = new TL_botInlineMessageMediaGeo_layer119();
                    break;
                case 0x51846fd:
                    result = new TL_botInlineMessageMediaGeo();
                    break;
                case 0x354a9b09:
                    result = new TL_botInlineMessageMediaInvoice();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in BotInlineMessage", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_botInlineMessageMediaVenue_layer77 extends TL_botInlineMessageMediaVenue {
        public static int constructor = 0x4366232e;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaVenue extends BotInlineMessage {
        public static int constructor = 0x8a86659c;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
            venue_type = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
            stream.writeString(venue_type);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaGeo_layer71 extends TL_botInlineMessageMediaGeo {
        public static int constructor = 0x3a8fd8b8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaAuto extends BotInlineMessage {
        public static int constructor = 0x764cf810;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    entities.add(object);
                }
            }
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = entities.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    entities.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaAuto_layer74 extends TL_botInlineMessageMediaAuto {
        public static int constructor = 0xa74b15b;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaContact_layer81 extends TL_botInlineMessageMediaContact {
        public static int constructor = 0x35edb4d4;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaContact extends BotInlineMessage {
        public static int constructor = 0x18d1cdc2;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            vcard = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(vcard);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageText extends BotInlineMessage {
        public static int constructor = 0x8c7f65e2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 1) != 0;
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    MessageEntity object = MessageEntity.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    entities.add(object);
                }
            }
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = entities.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    entities.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaGeo extends BotInlineMessage {
        public static int constructor = 0x51846fd;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                heading = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                period = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                proximity_notification_radius = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeInt32(heading);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(period);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(proximity_notification_radius);
            }
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaGeo_layer119 extends TL_botInlineMessageMediaGeo {
        public static int constructor = 0xb722de65;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            period = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            stream.writeInt32(period);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaInvoice extends BotInlineMessage {
        public static int constructor = 0x354a9b09;

        public boolean shipping_address_requested;
        public boolean test;
        public String description;
        public WebDocument photo;
        public String currency;
        public long total_amount;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            shipping_address_requested = (flags & 2) != 0;
            test = (flags & 8) != 0;
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = shipping_address_requested ? (flags | 2) : (flags &~ 2);
            flags = test ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_keyboardButtonRow extends TLObject {
        public static int constructor = 0x77608b83;

        public ArrayList<KeyboardButton> buttons = new ArrayList<>();

        public static TL_keyboardButtonRow TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_keyboardButtonRow.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_keyboardButtonRow", constructor));
                } else {
                    return null;
                }
            }
            TL_keyboardButtonRow result = new TL_keyboardButtonRow();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                KeyboardButton object = KeyboardButton.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                buttons.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = buttons.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                buttons.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class Bool extends TLObject {

        public static Bool TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            Bool result = null;
            switch (constructor) {
                case 0x997275b5:
                    result = new TL_boolTrue();
                    break;
                case 0xbc799737:
                    result = new TL_boolFalse();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in Bool", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_boolTrue extends Bool {
        public static int constructor = 0x997275b5;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_boolFalse extends Bool {
        public static int constructor = 0xbc799737;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_discussionMessage extends TLObject {
        public static int constructor = 0xa6341782;

        public int flags;
        public ArrayList<Message> messages = new ArrayList<>();
        public int max_id;
        public int read_inbox_max_id;
        public int read_outbox_max_id;
        public int unread_count;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_discussionMessage TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_discussionMessage.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_discussionMessage", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_discussionMessage result = new TL_messages_discussionMessage();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Message object = Message.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                messages.add(object);
            }
            if ((flags & 1) != 0) {
                max_id = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                read_inbox_max_id = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                read_outbox_max_id = stream.readInt32(exception);
            }
            unread_count = stream.readInt32(exception);
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(0x1cb5c415);
            int count = messages.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                messages.get(a).serializeToStream(stream);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(max_id);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(read_inbox_max_id);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(read_outbox_max_id);
            }
            stream.writeInt32(unread_count);
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_searchCounter extends TLObject {
        public static int constructor = 0xe844ebff;

        public int flags;
        public boolean inexact;
        public MessagesFilter filter;
        public int count;

        public static TL_messages_searchCounter TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_searchCounter.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_searchCounter", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_searchCounter result = new TL_messages_searchCounter();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            filter = MessagesFilter.TLdeserialize(stream, stream.readInt32(exception), exception);
            count = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            filter.serializeToStream(stream);
            stream.writeInt32(count);
        }
    }

    public static class TL_inputAppEvent extends TLObject {
        public static int constructor = 0x1d1b1245;

        public double time;
        public String type;
        public long peer;
        public JSONValue data;

        public static TL_inputAppEvent TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_inputAppEvent.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_inputAppEvent", constructor));
                } else {
                    return null;
                }
            }
            TL_inputAppEvent result = new TL_inputAppEvent();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            time = stream.readDouble(exception);
            type = stream.readString(exception);
            peer = stream.readInt64(exception);
            data = JSONValue.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(time);
            stream.writeString(type);
            stream.writeInt64(peer);
            data.serializeToStream(stream);
        }
    }

    public static class TL_auth_exportedAuthorization extends TLObject {
        public static int constructor = 0xb434e2b8;

        public long id;
        public byte[] bytes;

        public static TL_auth_exportedAuthorization TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_auth_exportedAuthorization.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_auth_exportedAuthorization", constructor));
                } else {
                    return null;
                }
            }
            TL_auth_exportedAuthorization result = new TL_auth_exportedAuthorization();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            bytes = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeByteArray(bytes);
        }
    }

    public static abstract class WebPage extends TLObject {
        public int flags;
        public long id;
        public String url;
        public String display_url;
        public int hash;
        public String type;
        public String site_name;
        public String title;
        public String description;
        public Photo photo;
        public String embed_url;
        public String embed_type;
        public int embed_width;
        public int embed_height;
        public int duration;
        public String author;
        public Document document;
        public Page cached_page;
        public int date;
        public ArrayList<TL_webPageAttributeTheme> attributes = new ArrayList<>();

        public static WebPage TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            WebPage result = null;
            switch (constructor) {
                case 0xe89c45b2:
                    result = new TL_webPage();
                    break;
                case 0xfa64e172:
                    result = new TL_webPage_layer107();
                    break;
                case 0x5f07b4bc:
                    result = new TL_webPage_layer104();
                    break;
                case 0xa31ea0b5:
                    result = new TL_webPage_old();
                    break;
                case 0xeb1477e8:
                    result = new TL_webPageEmpty();
                    break;
                case 0xd41a5167:
                    result = new TL_webPageUrlPending();
                    break;
                case 0x7311ca11:
                    result = new TL_webPageNotModified();
                    break;
                case 0xc586da1c:
                    result = new TL_webPagePending();
                    break;
                case 0x85849473:
                    result = new TL_webPageNotModified_layer110();
                    break;
                case 0xca820ed7:
                    result = new TL_webPage_layer58();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in WebPage", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_webPage extends WebPage {
        public static int constructor = 0xe89c45b2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            hash = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1024) != 0) {
                cached_page = Page.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4096) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_webPageAttributeTheme object = TL_webPageAttributeTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    attributes.add(object);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            stream.writeInt32(hash);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 1024) != 0) {
                cached_page.serializeToStream(stream);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = attributes.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    attributes.get(a).serializeToStream(stream);
                }
            }
        }
    }

    public static class TL_webPage_layer107 extends TL_webPage {
        public static int constructor = 0xfa64e172;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            hash = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2048) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                TL_webPageAttributeTheme attribute = new TL_webPageAttributeTheme();
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    attribute.documents.add(object);
                }
                attributes.add(attribute);
            }
            if ((flags & 1024) != 0) {
                cached_page = Page.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            stream.writeInt32(hash);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(0x1cb5c415);
                stream.writeInt32(0);
            }
            if ((flags & 1024) != 0) {
                cached_page.serializeToStream(stream);
            }
        }
    }

    public static class TL_webPage_layer104 extends TL_webPage {
        public static int constructor = 0x5f07b4bc;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            hash = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1024) != 0) {
                cached_page = Page.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            stream.writeInt32(hash);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 1024) != 0) {
                cached_page.serializeToStream(stream);
            }
        }
    }

    public static class TL_webPage_old extends TL_webPage {
        public static int constructor = 0xa31ea0b5;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
        }
    }

    public static class TL_webPageEmpty extends WebPage {
        public static int constructor = 0xeb1477e8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_webPageUrlPending extends WebPage {
        public static int constructor = 0xd41a5167;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_webPagePending extends WebPage {
        public static int constructor = 0xc586da1c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt32(date);
        }
    }

    public static class TL_webPageNotModified_layer110 extends TL_webPageNotModified {
        public static int constructor = 0x85849473;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_webPageNotModified extends WebPage {
        public static int constructor = 0x7311ca11;

        public int cached_page_views;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                cached_page_views = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(cached_page_views);
            }
        }
    }

    public static class TL_webPage_layer58 extends TL_webPage {
        public static int constructor = 0xca820ed7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
        }
    }

    public static abstract class messages_FeaturedStickers extends TLObject {

        public static messages_FeaturedStickers TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            messages_FeaturedStickers result = null;
            switch (constructor) {
                case 0xbe382906:
                    result = new TL_messages_featuredStickers();
                    break;
                case 0xc6dc0c66:
                    result = new TL_messages_featuredStickersNotModified();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in messages_FeaturedStickers", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messages_featuredStickers extends messages_FeaturedStickers {
        public static int constructor = 0xbe382906;

        public int flags;
        public boolean premium;
        public long hash;
        public int count;
        public ArrayList<StickerSetCovered> sets = new ArrayList<>();
        public ArrayList<Long> unread = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            premium = (flags & 1) != 0;
            hash = stream.readInt64(exception);
            count = stream.readInt32(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                StickerSetCovered object = StickerSetCovered.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                sets.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                unread.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = premium ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt64(hash);
            stream.writeInt32(count);
            stream.writeInt32(0x1cb5c415);
            int count = sets.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                sets.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = unread.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(unread.get(a));
            }
        }
    }

    public static class TL_messages_featuredStickersNotModified extends messages_FeaturedStickers {
        public static int constructor = 0xc6dc0c66;

        public int count;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
        }
    }

    public static abstract class SecureValueError extends TLObject {

        public static SecureValueError TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            SecureValueError result = null;
            switch (constructor) {
                case 0x7a700873:
                    result = new TL_secureValueErrorFile();
                    break;
                case 0xbe3dfa:
                    result = new TL_secureValueErrorFrontSide();
                    break;
                case 0x666220e9:
                    result = new TL_secureValueErrorFiles();
                    break;
                case 0x868a2aa5:
                    result = new TL_secureValueErrorReverseSide();
                    break;
                case 0xa1144770:
                    result = new TL_secureValueErrorTranslationFile();
                    break;
                case 0x869d758f:
                    result = new TL_secureValueError();
                    break;
                case 0xe8a40bd9:
                    result = new TL_secureValueErrorData();
                    break;
                case 0x34636dd8:
                    result = new TL_secureValueErrorTranslationFiles();
                    break;
                case 0xe537ced6:
                    result = new TL_secureValueErrorSelfie();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in SecureValueError", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_secureValueErrorFile extends SecureValueError {
        public static int constructor = 0x7a700873;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorFrontSide extends SecureValueError {
        public static int constructor = 0xbe3dfa;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorFiles extends SecureValueError {
        public static int constructor = 0x666220e9;

        public SecureValueType type;
        public ArrayList<byte[]> file_hash = new ArrayList<>();
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                file_hash.add(stream.readByteArray(exception));
            }
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = file_hash.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeByteArray(file_hash.get(a));
            }
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorReverseSide extends SecureValueError {
        public static int constructor = 0x868a2aa5;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorTranslationFile extends SecureValueError {
        public static int constructor = 0xa1144770;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueError extends SecureValueError {
        public static int constructor = 0x869d758f;

        public SecureValueType type;
        public byte[] hash;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorData extends SecureValueError {
        public static int constructor = 0xe8a40bd9;

        public SecureValueType type;
        public byte[] data_hash;
        public String field;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            data_hash = stream.readByteArray(exception);
            field = stream.readString(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(data_hash);
            stream.writeString(field);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorTranslationFiles extends SecureValueError {
        public static int constructor = 0x34636dd8;

        public SecureValueType type;
        public ArrayList<byte[]> file_hash = new ArrayList<>();
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                file_hash.add(stream.readByteArray(exception));
            }
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = file_hash.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeByteArray(file_hash.get(a));
            }
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorSelfie extends SecureValueError {
        public static int constructor = 0xe537ced6;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValue extends TLObject {
        public static int constructor = 0x187fa0ca;

        public int flags;
        public SecureValueType type;
        public TL_secureData data;
        public SecureFile front_side;
        public SecureFile reverse_side;
        public SecureFile selfie;
        public ArrayList<SecureFile> translation = new ArrayList<>();
        public ArrayList<SecureFile> files = new ArrayList<>();
        public SecurePlainData plain_data;
        public byte[] hash;

        public static TL_secureValue TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_secureValue.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_secureValue", constructor));
                } else {
                    return null;
                }
            }
            TL_secureValue result = new TL_secureValue();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                data = TL_secureData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                front_side = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                reverse_side = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                selfie = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    SecureFile object = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    translation.add(object);
                }
            }
            if ((flags & 16) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    SecureFile object = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    files.add(object);
                }
            }
            if ((flags & 32) != 0) {
                plain_data = SecurePlainData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            hash = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
            if ((flags & 1) != 0) {
                data.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                front_side.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                reverse_side.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                selfie.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = translation.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    translation.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = files.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    files.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 32) != 0) {
                plain_data.serializeToStream(stream);
            }
            stream.writeByteArray(hash);
        }
    }

    public static abstract class PhoneCallDiscardReason extends TLObject {
        public byte[] encrypted_key;

        public static PhoneCallDiscardReason TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            PhoneCallDiscardReason result = null;
            switch (constructor) {
                case 0x57adc690:
                    result = new TL_phoneCallDiscardReasonHangup();
                    break;
                case 0xfaf7e8c9:
                    result = new TL_phoneCallDiscardReasonBusy();
                    break;
                case 0x85e42301:
                    result = new TL_phoneCallDiscardReasonMissed();
                    break;
                case 0xe095c1a0:
                    result = new TL_phoneCallDiscardReasonDisconnect();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in PhoneCallDiscardReason", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_phoneCallDiscardReasonHangup extends PhoneCallDiscardReason {
        public static int constructor = 0x57adc690;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonBusy extends PhoneCallDiscardReason {
        public static int constructor = 0xfaf7e8c9;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonMissed extends PhoneCallDiscardReason {
        public static int constructor = 0x85e42301;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonDisconnect extends PhoneCallDiscardReason {
        public static int constructor = 0xe095c1a0;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class DialogFilter extends TLObject {

        public int flags;
        public boolean contacts;
        public boolean non_contacts;
        public boolean groups;
        public boolean broadcasts;
        public boolean bots;
        public boolean exclude_muted;
        public boolean exclude_read;
        public boolean exclude_archived;
        public int id;
        public String title;
        public String emoticon;
        public ArrayList<InputPeer> pinned_peers = new ArrayList<>();
        public ArrayList<InputPeer> include_peers = new ArrayList<>();
        public ArrayList<InputPeer> exclude_peers = new ArrayList<>();

        public static DialogFilter TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            DialogFilter result = null;
            switch (constructor) {
                case 0x363293ae:
                    result = new TL_dialogFilterDefault();
                    break;
                case 0x7438f7e8:
                    result = new TL_dialogFilter();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in DialogFilter", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_dialogFilterDefault extends DialogFilter {
        public static int constructor = 0x363293ae;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_dialogFilter extends DialogFilter {
        public static int constructor = 0x7438f7e8;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            contacts = (flags & 1) != 0;
            non_contacts = (flags & 2) != 0;
            groups = (flags & 4) != 0;
            broadcasts = (flags & 8) != 0;
            bots = (flags & 16) != 0;
            exclude_muted = (flags & 2048) != 0;
            exclude_read = (flags & 4096) != 0;
            exclude_archived = (flags & 8192) != 0;
            id = stream.readInt32(exception);
            title = stream.readString(exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                InputPeer object = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                pinned_peers.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                InputPeer object = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                include_peers.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                InputPeer object = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                exclude_peers.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = contacts ? (flags | 1) : (flags &~ 1);
            flags = non_contacts ? (flags | 2) : (flags &~ 2);
            flags = groups ? (flags | 4) : (flags &~ 4);
            flags = broadcasts ? (flags | 8) : (flags &~ 8);
            flags = bots ? (flags | 16) : (flags &~ 16);
            flags = exclude_muted ? (flags | 2048) : (flags &~ 2048);
            flags = exclude_read ? (flags | 4096) : (flags &~ 4096);
            flags = exclude_archived ? (flags | 8192) : (flags &~ 8192);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            stream.writeInt32(0x1cb5c415);
            int count = pinned_peers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                pinned_peers.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = include_peers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                include_peers.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = exclude_peers.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                exclude_peers.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_auth_passwordRecovery extends TLObject {
        public static int constructor = 0x137948a5;

        public String email_pattern;

        public static TL_auth_passwordRecovery TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_auth_passwordRecovery.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_auth_passwordRecovery", constructor));
                } else {
                    return null;
                }
            }
            TL_auth_passwordRecovery result = new TL_auth_passwordRecovery();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            email_pattern = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(email_pattern);
        }
    }

    public static class TL_exportedContactToken extends TLObject {
        public static int constructor = 0x41bf109b;

        public String url;
        public int expires;

        public static TL_exportedContactToken TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_exportedContactToken.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_exportedContactToken", constructor));
                } else {
                    return null;
                }
            }
            TL_exportedContactToken result = new TL_exportedContactToken();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            expires = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeInt32(expires);
        }
    }

    public static class TL_botCommand extends TLObject {
        public static int constructor = 0xc27ac8c7;

        public String command;
        public String description;

        public static TL_botCommand TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_botCommand.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_botCommand", constructor));
                } else {
                    return null;
                }
            }
            TL_botCommand result = new TL_botCommand();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            command = stream.readString(exception);
            description = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(command);
            stream.writeString(description);
        }
    }

    public static abstract class InputNotifyPeer extends TLObject {

        public static InputNotifyPeer TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputNotifyPeer result = null;
            switch (constructor) {
                case 0x4a95e84e:
                    result = new TL_inputNotifyChats();
                    break;
                case 0xb8bc5b0c:
                    result = new TL_inputNotifyPeer();
                    break;
                case 0x193b4417:
                    result = new TL_inputNotifyUsers();
                    break;
                case 0xb1db7c7e:
                    result = new TL_inputNotifyBroadcasts();
                    break;
                case 0x5c467992:
                    result = new TL_inputNotifyForumTopic();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputNotifyPeer", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputNotifyChats extends InputNotifyPeer {
        public static int constructor = 0x4a95e84e;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputNotifyPeer extends InputNotifyPeer {
        public static int constructor = 0xb8bc5b0c;

        public InputPeer peer;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_inputNotifyUsers extends InputNotifyPeer {
        public static int constructor = 0x193b4417;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputNotifyBroadcasts extends InputNotifyPeer {
        public static int constructor = 0xb1db7c7e;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputNotifyForumTopic extends InputNotifyPeer {
        public static int constructor = 0x5c467992;

        public InputPeer peer;
        public int top_msg_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            top_msg_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(top_msg_id);
        }
    }

    public static abstract class InputFileLocation extends TLObject {

        public long id;
        public long access_hash;
        public byte[] file_reference;
        public String thumb_size;
        public int flags;
        public long volume_id;
        public int local_id;
        public long secret;

        public static InputFileLocation TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            InputFileLocation result = null;
            switch (constructor) {
                case 0x40181ffe:
                    result = new TL_inputPhotoFileLocation();
                    break;
                case 0xbad07584:
                    result = new TL_inputDocumentFileLocation();
                    break;
                case 0x37257e99:
                    result = new TL_inputPeerPhotoFileLocation();
                    break;
                case 0xf5235d55:
                    result = new TL_inputEncryptedFileLocation();
                    break;
                case 0x9d84f3db:
                    result = new TL_inputStickerSetThumb();
                    break;
                case 0x598a92a:
                    result = new TL_inputGroupCallStream();
                    break;
                case 0xdfdaabe1:
                    result = new TL_inputFileLocation();
                    break;
                case 0xcbc7ee28:
                    result = new TL_inputSecureFileLocation();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in InputFileLocation", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_inputPhotoFileLocation extends InputFileLocation {
        public static int constructor = 0x40181ffe;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            file_reference = stream.readByteArray(exception);
            thumb_size = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeByteArray(file_reference);
            stream.writeString(thumb_size);
        }
    }

    public static class TL_inputDocumentFileLocation extends InputFileLocation {
        public static int constructor = 0xbad07584;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            file_reference = stream.readByteArray(exception);
            thumb_size = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeByteArray(file_reference);
            stream.writeString(thumb_size);
        }
    }

    public static class TL_inputPeerPhotoFileLocation extends InputFileLocation {
        public static int constructor = 0x37257e99;

        public boolean big;
        public InputPeer peer;
        public long photo_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            peer.serializeToStream(stream);
            stream.writeInt64(photo_id);
        }
    }

    public static class TL_inputEncryptedFileLocation extends InputFileLocation {
        public static int constructor = 0xf5235d55;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputStickerSetThumb extends InputFileLocation {
        public static int constructor = 0x9d84f3db;

        public InputStickerSet stickerset;
        public int thumb_version;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            thumb_version = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stickerset.serializeToStream(stream);
            stream.writeInt32(thumb_version);
        }
    }

    public static class TL_inputGroupCallStream extends InputFileLocation {
        public static int constructor = 0x598a92a;

        public TL_inputGroupCall call;
        public long time_ms;
        public int scale;
        public int video_channel;
        public int video_quality;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            time_ms = stream.readInt64(exception);
            scale = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                video_channel = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                video_quality = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            call.serializeToStream(stream);
            stream.writeInt64(time_ms);
            stream.writeInt32(scale);
            if ((flags & 1) != 0) {
                stream.writeInt32(video_channel);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(video_quality);
            }
        }
    }

    public static class TL_inputFileLocation extends InputFileLocation {
        public static int constructor = 0xdfdaabe1;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            volume_id = stream.readInt64(exception);
            local_id = stream.readInt32(exception);
            secret = stream.readInt64(exception);
            file_reference = stream.readByteArray(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(volume_id);
            stream.writeInt32(local_id);
            stream.writeInt64(secret);
            stream.writeByteArray(file_reference);
        }
    }

    public static class TL_inputSecureFileLocation extends InputFileLocation {
        public static int constructor = 0xcbc7ee28;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_photos_photo extends TLObject {
        public static int constructor = 0x20212ca8;

        public Photo photo;
        public ArrayList<User> users = new ArrayList<>();

        public static TL_photos_photo TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_photos_photo.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_photos_photo", constructor));
                } else {
                    return null;
                }
            }
            TL_photos_photo result = new TL_photos_photo();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class help_TermsOfServiceUpdate extends TLObject {

        public static help_TermsOfServiceUpdate TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            help_TermsOfServiceUpdate result = null;
            switch (constructor) {
                case 0x28ecf961:
                    result = new TL_help_termsOfServiceUpdate();
                    break;
                case 0xe3309f7f:
                    result = new TL_help_termsOfServiceUpdateEmpty();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in help_TermsOfServiceUpdate", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_help_termsOfServiceUpdate extends help_TermsOfServiceUpdate {
        public static int constructor = 0x28ecf961;

        public int expires;
        public TL_help_termsOfService terms_of_service;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            expires = stream.readInt32(exception);
            terms_of_service = TL_help_termsOfService.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(expires);
            terms_of_service.serializeToStream(stream);
        }
    }

    public static class
    TL_help_termsOfServiceUpdateEmpty extends help_TermsOfServiceUpdate {
        public static int constructor = 0xe3309f7f;

        public int expires;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            expires = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(expires);
        }
    }

    public static class TL_messages_chatAdminsWithInvites extends TLObject {
        public static int constructor = 0xb69b72d7;

        public ArrayList<TL_chatAdminWithInvites> admins = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_chatAdminsWithInvites TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_chatAdminsWithInvites.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_chatAdminsWithInvites", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_chatAdminsWithInvites result = new TL_messages_chatAdminsWithInvites();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_chatAdminWithInvites object = TL_chatAdminWithInvites.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                admins.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = admins.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                admins.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_phone_groupCall extends TLObject {
        public static int constructor = 0x9e727aad;

        public GroupCall call;
        public ArrayList<TL_groupCallParticipant> participants = new ArrayList<>();
        public String participants_next_offset;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_phone_groupCall TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_phone_groupCall.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_phone_groupCall", constructor));
                } else {
                    return null;
                }
            }
            TL_phone_groupCall result = new TL_phone_groupCall();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            call = GroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_groupCallParticipant object = TL_groupCallParticipant.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                participants.add(object);
            }
            participants_next_offset = stream.readString(exception);
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                Chat object = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                chats.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            call.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = participants.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                participants.get(a).serializeToStream(stream);
            }
            stream.writeString(participants_next_offset);
            stream.writeInt32(0x1cb5c415);
            count = chats.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                chats.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class PhoneCall extends TLObject {

        public int flags;
        public boolean p2p_allowed;
        public long id;
        public long access_hash;
        public int date;
        public long admin_id;
        public long participant_id;
        public byte[] g_a_or_b;
        public long key_fingerprint;
        public PhoneCallProtocol protocol;
        public ArrayList<PhoneConnection> connections = new ArrayList<>();
        public int start_date;
        public boolean need_rating;
        public boolean need_debug;
        public boolean video;
        public PhoneCallDiscardReason reason;
        public int duration;
        public byte[] g_a_hash;
        public byte[] g_b;
        public int receive_date;

        public static PhoneCall TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            PhoneCall result = null;
            switch (constructor) {
                case 0x14b0ed0c:
                    result = new TL_phoneCallRequested();
                    break;
                case 0x967f7c67:
                    result = new TL_phoneCall();
                    break;
                case 0x5366c915:
                    result = new TL_phoneCallEmpty();
                    break;
                case 0x3660c311:
                    result = new TL_phoneCallAccepted();
                    break;
                case 0xc5226f17:
                    result = new TL_phoneCallWaiting();
                    break;
                case 0x50ca4de1:
                    result = new TL_phoneCallDiscarded();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in PhoneCall", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_phoneCallRequested extends PhoneCall {
        public static int constructor = 0x14b0ed0c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            video = (flags & 64) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            date = stream.readInt32(exception);
            admin_id = stream.readInt64(exception);
            participant_id = stream.readInt64(exception);
            g_a_hash = stream.readByteArray(exception);
            protocol = PhoneCallProtocol.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = video ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(date);
            stream.writeInt64(admin_id);
            stream.writeInt64(participant_id);
            stream.writeByteArray(g_a_hash);
            protocol.serializeToStream(stream);
        }
    }

    public static class TL_phoneCall extends PhoneCall {
        public static int constructor = 0x967f7c67;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            p2p_allowed = (flags & 32) != 0;
            video = (flags & 64) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            date = stream.readInt32(exception);
            admin_id = stream.readInt64(exception);
            participant_id = stream.readInt64(exception);
            g_a_or_b = stream.readByteArray(exception);
            key_fingerprint = stream.readInt64(exception);
            protocol = PhoneCallProtocol.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                PhoneConnection object = PhoneConnection.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                connections.add(object);
            }
            start_date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = p2p_allowed ? (flags | 32) : (flags &~ 32);
            flags = video ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(date);
            stream.writeInt64(admin_id);
            stream.writeInt64(participant_id);
            stream.writeByteArray(g_a_or_b);
            stream.writeInt64(key_fingerprint);
            protocol.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = connections.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                connections.get(a).serializeToStream(stream);
            }
            stream.writeInt32(start_date);
        }
    }

    public static class TL_phoneCallEmpty extends PhoneCall {
        public static int constructor = 0x5366c915;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_phoneCallAccepted extends PhoneCall {
        public static int constructor = 0x3660c311;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            video = (flags & 64) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            date = stream.readInt32(exception);
            admin_id = stream.readInt64(exception);
            participant_id = stream.readInt64(exception);
            g_b = stream.readByteArray(exception);
            protocol = PhoneCallProtocol.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = video ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(date);
            stream.writeInt64(admin_id);
            stream.writeInt64(participant_id);
            stream.writeByteArray(g_b);
            protocol.serializeToStream(stream);
        }
    }

    public static class TL_phoneCallWaiting extends PhoneCall {
        public static int constructor = 0xc5226f17;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            video = (flags & 64) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            date = stream.readInt32(exception);
            admin_id = stream.readInt64(exception);
            participant_id = stream.readInt64(exception);
            protocol = PhoneCallProtocol.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                receive_date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = video ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(date);
            stream.writeInt64(admin_id);
            stream.writeInt64(participant_id);
            protocol.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeInt32(receive_date);
            }
        }
    }

    public static class TL_phoneCallDiscarded extends PhoneCall {
        public static int constructor = 0x50ca4de1;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            need_rating = (flags & 4) != 0;
            need_debug = (flags & 8) != 0;
            video = (flags & 64) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                reason = PhoneCallDiscardReason.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                duration = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = need_rating ? (flags | 4) : (flags &~ 4);
            flags = need_debug ? (flags | 8) : (flags &~ 8);
            flags = video ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                reason.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(duration);
            }
        }
    }

    public static abstract class help_CountriesList extends TLObject {

        public static help_CountriesList TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            help_CountriesList result = null;
            switch (constructor) {
                case 0x93cc1f32:
                    result = new TL_help_countriesListNotModified();
                    break;
                case 0x87d0759e:
                    result = new TL_help_countriesList();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in help_CountriesList", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_help_countriesListNotModified extends help_CountriesList {
        public static int constructor = 0x93cc1f32;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_countriesList extends help_CountriesList {
        public static int constructor = 0x87d0759e;

        public ArrayList<TL_help_country> countries = new ArrayList<>();
        public int hash;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_help_country object = TL_help_country.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                countries.add(object);
            }
            hash = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = countries.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                countries.get(a).serializeToStream(stream);
            }
            stream.writeInt32(hash);
        }
    }

    public static abstract class User extends TLObject {
        public long id;
        public String first_name;
        public String last_name;
        public String username;
        public long access_hash;
        public String phone;
        public UserProfilePhoto photo;
        public UserStatus status;
        public int flags;
        public int flags2;
        public boolean self;
        public boolean contact;
        public boolean mutual_contact;
        public boolean deleted;
        public boolean bot;
        public boolean bot_chat_history;
        public boolean bot_nochats;
        public boolean verified;
        public boolean restricted;
        public boolean min;
        public boolean bot_inline_geo;
        public boolean support;
        public boolean scam;
        public boolean apply_min_photo;
        public boolean fake;
        public boolean premium;
        public int bot_info_version;
        public String bot_inline_placeholder;
        public String lang_code;
        public boolean inactive;
        public boolean explicit_content;
        public ArrayList<TL_restrictionReason> restriction_reason = new ArrayList<>();
        public boolean bot_attach_menu;
        public boolean bot_menu_webview;
        public boolean attach_menu_enabled;
        public EmojiStatus emoji_status;
        public ArrayList<TL_username> usernames = new ArrayList<>();
        public String prompt;
        public int aiModel;
        public double temperature;
        public int contextLimit;
        public int tokenLimit;
        public String customModel;

        public static User TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            User result = null;
            switch (constructor) {
                case 0xcab35e18:
                    result = new TL_userContact_old2();
                    break;
                case 0xf2fb8319:
                    result = new TL_userContact_old();
                    break;
                case 0xd3bc4b7a:
                    result = new TL_userEmpty();
                    break;
                case 0x8f97c628:
                    result = new TL_user();
                    break;
                case 0x5d99adee:
                    result = new TL_user_layer147();
                    break;
                case 0x3ff6ecb0:
                    result = new TL_user_layer144();
                    break;
                case 0x938458c1:
                    result = new TL_user_layer131();
                    break;
                case 0x2e13f4c3:
                    result = new TL_user_layer104();
                    break;
                case 0x720535ec:
                    result = new TL_userSelf_old();
                    break;
                case 0x1c60e608:
                    result = new TL_userSelf_old3();
                    break;
                case 0xd6016d7a:
                    result = new TL_userDeleted_old2();
                    break;
                case 0x200250ba:
                    result = new TL_userEmpty_layer131();
                    break;
                case 0x22e8ceb0:
                    result = new TL_userRequest_old();
                    break;
                case 0x5214c89d:
                    result = new TL_userForeign_old();
                    break;
                case 0x75cf7a8:
                    result = new TL_userForeign_old2();
                    break;
                case 0xd9ccc4ef:
                    result = new TL_userRequest_old2();
                    break;
                case 0xb29ad7cc:
                    result = new TL_userDeleted_old();
                    break;
                case 0xd10d979a:
                    result = new TL_user_layer65();
                    break;
                case 0x22e49072:
                    result = new TL_user_old();
                    break;
                case 0x7007b451:
                    result = new TL_userSelf_old2();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in User", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_userContact_old2 extends User {
        public static int constructor = 0xcab35e18;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
            stream.writeInt64(access_hash);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userContact_old extends TL_userContact_old2 {
        public static int constructor = 0xf2fb8319;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeInt64(access_hash);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userEmpty extends User {
        public static int constructor = 0xd3bc4b7a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_user extends User {
        public static int constructor = 0x8f97c628;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_restrictionReason object = TL_restrictionReason.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    restriction_reason.add(object);
                }
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_username object = TL_username.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    usernames.add(object);
                }
            }
            if (BuildVars.IS_CHAT_AIR) {
                if ((flags2 & 1024) != 0) {
                    prompt = stream.readString(exception);
                }
                if ((flags2 & 2048) != 0) {
                    aiModel = stream.readInt32(exception);
                }
                if ((flags2 & 4096) != 0) {
                    temperature = stream.readDouble(exception);
                }
                if ((flags2 & 8192) != 0) {
                    contextLimit = stream.readInt32(exception);
                }
                if ((flags2 & 16384) != 0) {
                    tokenLimit = stream.readInt32(exception);
                }
                if ((flags2 & 32768) != 0) {
                    customModel = stream.readString(exception);
                }
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = restriction_reason.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    restriction_reason.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = usernames.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    usernames.get(a).serializeToStream(stream);
                }
            }

            if (BuildVars.IS_CHAT_AIR) {
                //需要自己手动添加，就不另单独做bool值
                if ((flags2 & 1024) != 0) {
                    stream.writeString(prompt);
                }
                if ((flags2 & 2048) != 0) {
                    stream.writeInt32(aiModel);
                }
                if ((flags2 & 4096) != 0) {
                    stream.writeDouble(temperature);
                }
                if ((flags2 & 8192) != 0) {
                    stream.writeInt32(contextLimit);
                }
                if ((flags2 & 16384) != 0) {
                    stream.writeInt32(tokenLimit);
                }
                if ((flags2 & 32768) != 0) {
                    stream.writeString(customModel);
                }
            }
        }
    }

    public static class TL_user_layer147 extends User {
        public static int constructor = 0x5d99adee;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_restrictionReason object = TL_restrictionReason.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    restriction_reason.add(object);
                }
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = restriction_reason.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    restriction_reason.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
        }
    }

    public static class TL_user_layer144 extends User {
        public static int constructor = 0x3ff6ecb0;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_restrictionReason object = TL_restrictionReason.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    restriction_reason.add(object);
                }
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = restriction_reason.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    restriction_reason.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
        }
    }

    public static class TL_user_layer131 extends TL_user {
        public static int constructor = 0x938458c1;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_restrictionReason object = TL_restrictionReason.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    restriction_reason.add(object);
                }
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = restriction_reason.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    restriction_reason.get(a).serializeToStream(stream);
                }
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
        }
    }

    public static class TL_user_layer104 extends TL_user {
        public static int constructor = 0x2e13f4c3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                stream.readString(exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                stream.writeString("");
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
        }
    }

    public static class TL_userSelf_old extends TL_userSelf_old3 {
        public static int constructor = 0x720535ec;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            inactive = stream.readBool(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
            stream.writeBool(inactive);
        }
    }

    public static class TL_userSelf_old3 extends User {
        public static int constructor = 0x1c60e608;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userDeleted_old2 extends User {
        public static int constructor = 0xd6016d7a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
        }
    }

    public static class TL_userEmpty_layer131 extends TL_userEmpty {
        public static int constructor = 0x200250ba;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
        }
    }

    public static class TL_userRequest_old extends TL_userRequest_old2 {
        public static int constructor = 0x22e8ceb0;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeInt64(access_hash);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userForeign_old extends TL_userForeign_old2 {
        public static int constructor = 0x5214c89d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeInt64(access_hash);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userForeign_old2 extends User {
        public static int constructor = 0x75cf7a8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
            stream.writeInt64(access_hash);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userRequest_old2 extends User {
        public static int constructor = 0xd9ccc4ef;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
            stream.writeInt64(access_hash);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userDeleted_old extends TL_userDeleted_old2 {
        public static int constructor = 0xb29ad7cc;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
        }
    }

    public static class TL_user_layer65 extends TL_user {
        public static int constructor = 0xd10d979a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                stream.readString(exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                stream.writeString("");
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
        }
    }

    public static class TL_user_old extends TL_user {
        public static int constructor = 0x22e49072;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            explicit_content = (flags & 262144) != 0;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = explicit_content ? (flags | 262144) : (flags &~ 262144);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
        }
    }

    public static class TL_userSelf_old2 extends TL_userSelf_old3 {
        public static int constructor = 0x7007b451;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            inactive = stream.readBool(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
            stream.writeBool(inactive);
        }
    }

    public static class TL_phone_groupCallStreamRtmpUrl extends TLObject {
        public static int constructor = 0x2dbf3432;

        public String url;
        public String key;

        public static TL_phone_groupCallStreamRtmpUrl TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_phone_groupCallStreamRtmpUrl.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_phone_groupCallStreamRtmpUrl", constructor));
                } else {
                    return null;
                }
            }
            TL_phone_groupCallStreamRtmpUrl result = new TL_phone_groupCallStreamRtmpUrl();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            key = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeString(key);
        }
    }

    public static abstract class RecentMeUrl extends TLObject {

        public long chat_id;
        public String url;
        public ChatInvite chat_invite;
        public long user_id;
        public StickerSetCovered set;

        public static RecentMeUrl TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            RecentMeUrl result = null;
            switch (constructor) {
                case 0xb2da71d2:
                    result = new TL_recentMeUrlChat();
                    break;
                case 0xbc0a57dc:
                    result = new TL_recentMeUrlStickerSet();
                    break;
                case 0x46e1d13d:
                    result = new TL_recentMeUrlUnknown();
                    break;
                case 0xeb49081d:
                    result = new TL_recentMeUrlChatInvite();
                    break;
                case 0xb92c09e2:
                    result = new TL_recentMeUrlUser();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in RecentMeUrl", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_recentMeUrlChat extends RecentMeUrl {
        public static int constructor = 0xb2da71d2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            chat_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeInt64(chat_id);
        }
    }

    public static class TL_recentMeUrlStickerSet extends RecentMeUrl {
        public static int constructor = 0xbc0a57dc;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            set = StickerSetCovered.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            set.serializeToStream(stream);
        }
    }

    public static class TL_recentMeUrlUnknown extends RecentMeUrl {
        public static int constructor = 0x46e1d13d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_recentMeUrlChatInvite extends RecentMeUrl {
        public static int constructor = 0xeb49081d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            chat_invite = ChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            chat_invite.serializeToStream(stream);
        }
    }

    public static class TL_recentMeUrlUser extends RecentMeUrl {
        public static int constructor = 0xb92c09e2;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_statsGroupTopPoster extends TLObject {
        public static int constructor = 0x9d04af9b;

        public long user_id;
        public int messages;
        public int avg_chars;

        public static TL_statsGroupTopPoster TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_statsGroupTopPoster.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_statsGroupTopPoster", constructor));
                } else {
                    return null;
                }
            }
            TL_statsGroupTopPoster result = new TL_statsGroupTopPoster();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            messages = stream.readInt32(exception);
            avg_chars = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt32(messages);
            stream.writeInt32(avg_chars);
        }
    }

    public static class TL_statsDateRangeDays extends TLObject {
        public static int constructor = 0xb637edaf;

        public int min_date;
        public int max_date;

        public static TL_statsDateRangeDays TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_statsDateRangeDays.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_statsDateRangeDays", constructor));
                } else {
                    return null;
                }
            }
            TL_statsDateRangeDays result = new TL_statsDateRangeDays();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            min_date = stream.readInt32(exception);
            max_date = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(min_date);
            stream.writeInt32(max_date);
        }
    }

    public static class TL_payments_bankCardData extends TLObject {
        public static int constructor = 0x3e24e573;

        public String title;
        public ArrayList<TL_bankCardOpenUrl> open_urls = new ArrayList<>();

        public static TL_payments_bankCardData TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_payments_bankCardData.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_payments_bankCardData", constructor));
                } else {
                    return null;
                }
            }
            TL_payments_bankCardData result = new TL_payments_bankCardData();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_bankCardOpenUrl object = TL_bankCardOpenUrl.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                open_urls.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeInt32(0x1cb5c415);
            int count = open_urls.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                open_urls.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messages_highScores extends TLObject {
        public static int constructor = 0x9a3bfd99;

        public ArrayList<TL_highScore> scores = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_highScores TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_messages_highScores.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_messages_highScores", constructor));
                } else {
                    return null;
                }
            }
            TL_messages_highScores result = new TL_messages_highScores();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                TL_highScore object = TL_highScore.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                scores.add(object);
            }
            magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = scores.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                scores.get(a).serializeToStream(stream);
            }
            stream.writeInt32(0x1cb5c415);
            count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_phone_phoneCall extends TLObject {
        public static int constructor = 0xec82e140;

        public PhoneCall phone_call;
        public ArrayList<User> users = new ArrayList<>();

        public static TL_phone_phoneCall TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            if (TL_phone_phoneCall.constructor != constructor) {
                if (exception) {
                    throw new RuntimeException(String.format("can't parse magic %x in TL_phone_phoneCall", constructor));
                } else {
                    return null;
                }
            }
            TL_phone_phoneCall result = new TL_phone_phoneCall();
            result.readParams(stream, exception);
            return result;
        }

        public void readParams(AbstractSerializedData stream, boolean exception) {
            phone_call = PhoneCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                User object = User.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                users.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            phone_call.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                users.get(a).serializeToStream(stream);
            }
        }
    }

    public static abstract class ChannelParticipantsFilter extends TLObject {

        public String q;

        public static ChannelParticipantsFilter TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            ChannelParticipantsFilter result = null;
            switch (constructor) {
                case 0xe04b5ceb:
                    result = new TL_channelParticipantsMentions();
                    break;
                case 0xbb6ae88d:
                    result = new TL_channelParticipantsContacts();
                    break;
                case 0xb4608969:
                    result = new TL_channelParticipantsAdmins();
                    break;
                case 0xa3b54985:
                    result = new TL_channelParticipantsKicked();
                    break;
                case 0x656ac4b:
                    result = new TL_channelParticipantsSearch();
                    break;
                case 0xb0d1865b:
                    result = new TL_channelParticipantsBots();
                    break;
                case 0x1427a5e1:
                    result = new TL_channelParticipantsBanned();
                    break;
                case 0xde3f3c79:
                    result = new TL_channelParticipantsRecent();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in ChannelParticipantsFilter", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_channelParticipantsMentions extends ChannelParticipantsFilter {
        public static int constructor = 0xe04b5ceb;

        public int flags;
        public int top_msg_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                q = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                top_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(q);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(top_msg_id);
            }
        }
    }

    public static class TL_channelParticipantsContacts extends ChannelParticipantsFilter {
        public static int constructor = 0xbb6ae88d;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            q = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(q);
        }
    }

    public static class TL_channelParticipantsAdmins extends ChannelParticipantsFilter {
        public static int constructor = 0xb4608969;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_channelParticipantsKicked extends ChannelParticipantsFilter {
        public static int constructor = 0xa3b54985;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            q = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(q);
        }
    }

    public static class TL_channelParticipantsSearch extends ChannelParticipantsFilter {
        public static int constructor = 0x656ac4b;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            q = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(q);
        }
    }

    public static class TL_channelParticipantsBots extends ChannelParticipantsFilter {
        public static int constructor = 0xb0d1865b;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_channelParticipantsBanned extends ChannelParticipantsFilter {
        public static int constructor = 0x1427a5e1;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            q = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(q);
        }
    }

    public static class TL_channelParticipantsRecent extends ChannelParticipantsFilter {
        public static int constructor = 0xde3f3c79;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class MessageAction extends TLObject {
        public String title;
        public String address;
        public DecryptedMessageAction encryptedAction;
        public String message;
        public ArrayList<Long> users = new ArrayList<>();
        public TL_inputGroupCall call;
        public long channel_id;
        public Photo photo;
        public long chat_id;
        public long user_id;
        public UserProfilePhoto newUserPhoto;
        public long inviter_id;
        public int ttl;
        public int flags;
        public long call_id;
        public PhoneCallDiscardReason reason;
        public int duration;
        public boolean recurring_init;
        public boolean recurring_used;
        public String currency;
        public long amount;
        public long total_amount;
        public String invoice_slug;
        public long game_id;
        public int score;
        public boolean video;
        public int months;

        public static MessageAction TLdeserialize(AbstractSerializedData stream, int constructor, boolean exception) {
            MessageAction result = null;
            switch (constructor) {
                case 0x555555F5:
                    result = new TL_messageActionLoginUnknownLocation();
                    break;
                case 0x555555F7:
                    result = new TL_messageEncryptedAction();
                    break;
                case 0xfae69f56:
                    result = new TL_messageActionCustomAction();
                    break;
                case 0xa6638b9a:
                    result = new TL_messageActionChatCreate_layer131();
                    break;
                case 0xbd47cbad:
                    result = new TL_messageActionChatCreate();
                    break;
                case 0x502f92f7:
                    result = new TL_messageActionInviteToGroupCall();
                    break;
                case 0x15cefd00:
                    result = new TL_messageActionChatAddUser();
                    break;
                case 0xa43f30cc:
                    result = new TL_messageActionChatDeleteUser();
                    break;
                case 0x51bdb021:
                    result = new TL_messageActionChatMigrateTo_layer131();
                    break;
                case 0xe1037f92:
                    result = new TL_messageActionChatMigrateTo();
                    break;
                case 0xea3948e9:
                    result = new TL_messageActionChannelMigrateFrom();
                    break;
                case 0x7a0d7f42:
                    result = new TL_messageActionGroupCall();
                    break;
                case 0xaa786345:
                    result = new TL_messageActionSetChatTheme();
                    break;
                case 0x4792929b:
                    result = new TL_messageActionScreenshotTaken();
                    break;
                case 0x1baa035:
                    result = new TL_messageActionPhoneNumberRequest();
                    break;
                case 0xebbca3cb:
                    result = new TL_messageActionChatJoinedByRequest();
                    break;
                case 0x9fbab604:
                    result = new TL_messageActionHistoryClear();
                    break;
                case 0x7fcb13a8:
                    result = new TL_messageActionChatEditPhoto();
                    break;
                case 0x31224c3:
                    result = new TL_messageActionChatJoinedByLink();
                    break;
                case 0xb055eaee:
                    result = new TL_messageActionChannelMigrateFrom_layer131();
                    break;
                case 0x488a7337:
                    result = new TL_messageActionChatAddUser_layer131();
                    break;
                case 0xb2ae9b0c:
                    result = new TL_messageActionChatDeleteUser_layer131();
                    break;
                case 0x55555557:
                    result = new TL_messageActionCreatedBroadcastList();
                    break;
                case 0x55555550:
                    result = new TL_messageActionUserJoined();
                    break;
                case 0xb18a431c:
                    result = new TL_messageActionTopicEdit_layer149();
                    break;
                case 0xc0944820:
                    result = new TL_messageActionTopicEdit();
                    break;
                case 0x55555551:
                    result = new TL_messageActionUserUpdatedPhoto();
                    break;
                case 0x5e3cfc4b:
                    result = new TL_messageActionChatAddUser_old();
                    break;
                case 0x55555552:
                    result = new TL_messageActionTTLChange();
                    break;
                case 0x3c134d7b:
                    result = new TL_messageActionSetMessagesTTL();
                    break;
                case 0xaa1afbfd:
                    result = new TL_messageActionSetMessagesTTL_layer149();
                    break;
                case 0xd95c6154:
                    result = new TL_messageActionSecureValuesSent();
                    break;
                case 0x47dd8079:
                    result = new TL_messageActionWebViewDataSentMe();
                    break;
                case 0xb4c38cb5:
                    result = new TL_messageActionWebViewDataSent();
                    break;
                case 0xf89cf5e8:
                    result = new TL_messageActionChatJoinedByLink_layer131();
                    break;
                case 0xf3f25f76:
                    result = new TL_messageActionContactSignUp();
                    break;
                case 0x95d2ac92:
                    result = new TL_messageActionChannelCreate();
                    break;
                case 0x94bd38ed:
                    result = new TL_messageActionPinMessage();
                    break;
                case 0x98e0d697:
                    result = new TL_messageActionGeoProximityReached();
                    break;
                case 0x57de635e:
                    result = new TL_messageActionSuggestProfilePhoto();
                    break;
                case 0xe7e75f97:
                    result = new TL_messageActionAttachMenuBotAllowed();
                    break;
                case 0x95e3fbef:
                    result = new TL_messageActionChatDeletePhoto();
                    break;
                case 0x76b9f11a:
                    result = new TL_messageActionInviteToGroupCall_layer131();
                    break;
                case 0x80e11a7f:
                    result = new TL_messageActionPhoneCall();
                    break;
                case 0xd999256:
                    result = new TL_messageActionTopicCreate();
                    break;
                case 0xb5a1ce5a:
                    result = new TL_messageActionChatEditTitle();
                    break;
                case 0xabe9affe:
                    result = new TL_messageActionBotAllowed_layer153();
                    break;
                case 0xc516d679:
                    result = new TL_messageActionBotAllowed();
                    break;
                case 0x96163f56:
                    result = new TL_messageActionPaymentSent();
                    break;
                case 0x40699cd0:
                    result = new TL_messageActionPaymentSent_layer140();
                    break;
                case 0xb6aef7b0:
                    result = new TL_messageActionEmpty();
                    break;
                case 0x92a72876:
                    result = new TL_messageActionGameScore();
                    break;
                case 0xb3a07661:
                    result = new TL_messageActionGroupCallScheduled();
                    break;
                case 0x8f31b327:
                    result = new TL_messageActionPaymentSentMe();
                    break;
                case 0xaba0f5c6:
                    result = new TL_messageActionGiftPremium();
                    break;
                case 0xfe77345d:
                    result = new TL_messageActionRequestedPeer();
                case 0xe7e75f96:
                    result = new TL_messageActionClearContext();
                    break;
            }
            if (result == null && exception) {
                throw new RuntimeException(String.format("can't parse magic %x in MessageAction", constructor));
            }
            if (result != null) {
                result.readParams(stream, exception);
            }
            return result;
        }
    }

    public static class TL_messageActionLoginUnknownLocation extends MessageAction {
        public static int constructor = 0x555555F5;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            address = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeString(address);
        }
    }

    public static class TL_messageEncryptedAction extends MessageAction {
        public static int constructor = 0x555555F7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            encryptedAction = DecryptedMessageAction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            encryptedAction.serializeToStream(stream);
        }
    }

    public static class TL_messageActionCustomAction extends MessageAction {
        public static int constructor = 0xfae69f56;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            message = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(message);
        }
    }

    public static class TL_messageActionChatCreate extends MessageAction {
        public static int constructor = 0xbd47cbad;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(users.get(a));
            }
        }
    }

    public static class TL_messageActionChatCreate_layer131 extends TL_messageActionChatCreate {
        public static int constructor = 0xa6638b9a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add((long) stream.readInt32(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32((int) (long) users.get(a));
            }
        }
    }

    public static class TL_messageActionInviteToGroupCall extends MessageAction {
        public static int constructor = 0x502f92f7;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            call.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(users.get(a));
            }
        }
    }

    public static class TL_messageActionChannelMigrateFrom extends MessageAction {
        public static int constructor = 0xea3948e9;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            chat_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeInt64(chat_id);
        }
    }

    public static class TL_messageActionGroupCall extends MessageAction {
        public static int constructor = 0x7a0d7f42;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                duration = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            call.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeInt32(duration);
            }
        }
    }

    public static class TL_messageActionSetChatTheme extends MessageAction {
        public static int constructor = 0xaa786345;

        public String emoticon;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            emoticon = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(emoticon);
        }
    }

    public static class TL_messages_getExtendedMedia extends TLObject {
        public static int constructor = 0x84f80814;

        public InputPeer peer;
        public ArrayList<Integer> id = new ArrayList<>();

        public TLObject deserializeResponse(AbstractSerializedData stream, int constructor, boolean exception) {
            return Updates.TLdeserialize(stream, constructor, exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = id.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32(id.get(a));
            }
        }
    }

    public static class TL_messages_getMessageReadParticipants extends TLObject {
        public static int constructor = 0x31c1c44f;

        public InputPeer peer;
        public int msg_id;

        public TLObject deserializeResponse(AbstractSerializedData stream, int constructor, boolean exception) {
            Vector vector = new Vector();
            int size = stream.readInt32(exception);
            for (int a = 0; a < size; a++) {
                TL_readParticipantDate object = TL_readParticipantDate.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return vector;
                }
                vector.objects.add(object);
            }
            return vector;
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(msg_id);
        }
    }

    public static class TL_messageActionChatDeleteUser extends MessageAction {
        public static int constructor = 0xa43f30cc;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_messageActionChatAddUser extends MessageAction {
        public static int constructor = 0x15cefd00;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add(stream.readInt64(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt64(users.get(a));
            }
        }
    }

    public static class TL_messageActionChatMigrateTo_layer131 extends TL_messageActionChatMigrateTo {
        public static int constructor = 0x51bdb021;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            channel_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) channel_id);
        }
    }

    public static class TL_messageActionHistoryClear extends MessageAction {
        public static int constructor = 0x9fbab604;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionChatEditPhoto extends MessageAction {
        public static int constructor = 0x7fcb13a8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
        }
    }

    public static class TL_messageActionScreenshotTaken extends MessageAction {
        public static int constructor = 0x4792929b;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionPhoneNumberRequest extends MessageAction {
        public static int constructor = 0x1baa035;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionTopicEdit extends MessageAction {
        public static int constructor = 0xc0944820;

        public long icon_emoji_id;
        public boolean closed;
        public boolean hidden;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                icon_emoji_id = stream.readInt64(exception);
            }
            if ((flags & 4) != 0) {
                closed = stream.readBool(exception);
            }
            if ((flags & 8) != 0) {
                hidden = stream.readBool(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(title);
            }
            if ((flags & 2) != 0) {
                stream.writeInt64(icon_emoji_id);
            }
            if ((flags & 4) != 0) {
                stream.writeBool(closed);
            }
            if ((flags & 8) != 0) {
                stream.writeBool(hidden);
            }
        }
    }

    public static class TL_messageActionTopicEdit_layer149 extends TL_messageActionTopicEdit {
        public static int constructor = 0xb18a431c;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                icon_emoji_id = stream.readInt64(exception);
            }
            if ((flags & 4) != 0) {
                closed = stream.readBool(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(title);
            }
            if ((flags & 2) != 0) {
                stream.writeInt64(icon_emoji_id);
            }
            if ((flags & 4) != 0) {
                stream.writeBool(closed);
            }
        }
    }

    public static class TL_messageActionChatJoinedByRequest extends MessageAction {
        public static int constructor = 0xebbca3cb;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionChatMigrateTo extends MessageAction {
        public static int constructor = 0xe1037f92;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            channel_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(channel_id);
        }
    }

    public static class TL_messageActionChatJoinedByLink extends MessageAction {
        public static int constructor = 0x31224c3;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            inviter_id = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(inviter_id);
        }
    }

    public static class TL_messageActionChannelMigrateFrom_layer131 extends TL_messageActionChannelMigrateFrom {
        public static int constructor = 0xb055eaee;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            chat_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeInt32((int) chat_id);
        }
    }

    public static class TL_messageActionChatAddUser_layer131 extends TL_messageActionChatAddUser {
        public static int constructor = 0x488a7337;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add((long) stream.readInt32(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32((int) (long) users.get(a));
            }
        }
    }

    public static class TL_messageActionChatDeleteUser_layer131 extends TL_messageActionChatDeleteUser {
        public static int constructor = 0xb2ae9b0c;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_messageActionCreatedBroadcastList extends MessageAction {
        public static int constructor = 0x55555557;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionInviteToGroupCall_layer131 extends TL_messageActionInviteToGroupCall {
        public static int constructor = 0x76b9f11a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            call = TL_inputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                users.add((long) stream.readInt32(exception));
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            call.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = users.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeInt32((int) (long) users.get(a));
            }
        }
    }

    public static class TL_messageActionWebViewDataSentMe extends MessageAction {
        public static int constructor = 0x47dd8079;

        public String text;
        public String data;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            data = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeString(data);
        }
    }

    public static class TL_messageActionWebViewDataSent extends MessageAction {
        public static int constructor = 0xb4c38cb5;

        public String text;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_messageActionUserJoined extends MessageAction {
        public static int constructor = 0x55555550;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionUserUpdatedPhoto extends MessageAction {
        public static int constructor = 0x55555551;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            newUserPhoto = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            newUserPhoto.serializeToStream(stream);
        }
    }

    public static class TL_messageActionContactSignUp extends MessageAction {
        public static int constructor = 0xf3f25f76;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionChatAddUser_old extends TL_messageActionChatAddUser {
        public static int constructor = 0x5e3cfc4b;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_messageActionTTLChange extends MessageAction {
        public static int constructor = 0x55555552;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            ttl = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(ttl);
        }
    }

    public static class TL_messageActionSetMessagesTTL extends MessageAction {
        public static int constructor = 0x3c134d7b;

        public int period;
        public long auto_setting_from;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            period = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                auto_setting_from = stream.readInt64(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(period);
            if ((flags & 1) != 0) {
                stream.writeInt64(auto_setting_from);
            }
        }
    }

    public static class TL_messageActionSetMessagesTTL_layer149 extends TL_messageActionSetMessagesTTL {
        public static int constructor = 0xaa1afbfd;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            period = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(period);
        }
    }

    public static class TL_messageActionChatJoinedByLink_layer131 extends TL_messageActionChatJoinedByLink {
        public static int constructor = 0xf89cf5e8;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            inviter_id = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) inviter_id);
        }
    }

    public static class TL_messageActionChannelCreate extends MessageAction {
        public static int constructor = 0x95d2ac92;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
        }
    }

    public static class TL_messageActionSecureValuesSent extends MessageAction {
        public static int constructor = 0xd95c6154;

        public ArrayList<SecureValueType> types = new ArrayList<>();

        public void readParams(AbstractSerializedData stream, boolean exception) {
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                SecureValueType object = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
                if (object == null) {
                    return;
                }
                types.add(object);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(0x1cb5c415);
            int count = types.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                types.get(a).serializeToStream(stream);
            }
        }
    }

    public static class TL_messageActionPinMessage extends MessageAction {
        public static int constructor = 0x94bd38ed;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionGeoProximityReached extends MessageAction {
        public static int constructor = 0x98e0d697;

        public Peer from_id;
        public Peer to_id;
        public int distance;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            from_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            to_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            distance = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            from_id.serializeToStream(stream);
            to_id.serializeToStream(stream);
            stream.writeInt32(distance);
        }
    }

    public static class TL_messageActionChatDeletePhoto extends MessageAction {
        public static int constructor = 0x95e3fbef;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionPhoneCall extends MessageAction {
        public static int constructor = 0x80e11a7f;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            video = (flags & 4) != 0;
            call_id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                reason = PhoneCallDiscardReason.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                duration = stream.readInt32(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = video ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeInt64(call_id);
            if ((flags & 1) != 0) {
                reason.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(duration);
            }
        }
    }

    public static class TL_messageActionTopicCreate extends MessageAction {
        public static int constructor = 0xd999256;

        public int icon_color;
        public long icon_emoji_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            title = stream.readString(exception);
            icon_color = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                icon_emoji_id = stream.readInt64(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeInt32(icon_color);
            if ((flags & 1) != 0) {
                stream.writeInt64(icon_emoji_id);
            }
        }
    }

    public static class TL_messageActionChatEditTitle extends MessageAction {
        public static int constructor = 0xb5a1ce5a;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            title = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
        }
    }

    public static class TL_messageActionPaymentSent extends MessageAction {
        public static int constructor = 0x96163f56;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            recurring_init = (flags & 4) != 0;
            recurring_used = (flags & 8) != 0;
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                invoice_slug = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = recurring_init ? (flags | 4) : (flags &~ 4);
            flags = recurring_used ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            if ((flags & 1) != 0) {
                stream.writeString(invoice_slug);
            }
        }
    }

    public static class TL_messageActionPaymentSent_layer140 extends TL_messageActionPaymentSent {
        public static int constructor = 0x40699cd0;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(currency);
            stream.writeInt64(total_amount);
        }
    }

    public static class TL_messageActionBotAllowed extends MessageAction {
        public static int constructor = 0xc516d679;

        public boolean attach_menu;
        public String domain;
        public BotApp app;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            attach_menu = (flags & 2) != 0;
            if ((flags & 1) != 0) {
                domain = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                app = BotApp.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = attach_menu ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(domain);
            }
            if ((flags & 4) != 0) {
                app.serializeToStream(stream);
            }
        }
    }

    public static class TL_messageActionBotAllowed_layer153 extends MessageAction {
        public static int constructor = 0xabe9affe;

        public String domain;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            domain = stream.readString(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(domain);
        }
    }

    public static class TL_messageActionEmpty extends MessageAction {
        public static int constructor = 0xb6aef7b0;


        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageActionGameScore extends MessageAction {
        public static int constructor = 0x92a72876;


        public void readParams(AbstractSerializedData stream, boolean exception) {
            game_id = stream.readInt64(exception);
            score = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(game_id);
            stream.writeInt32(score);
        }
    }

    public static class TL_messageActionGiftPremium extends MessageAction {
        public static int constructor = 0xaba0f5c6;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            months = stream.readInt32(exception);
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeInt32(months);
        }
    }

    public static class TL_messageActionPaymentSentMe extends MessageAction {
        public static int constructor = 0x8f31b327;

        public int flags;
        public byte[] payload;
        public TL_paymentRequestedInfo info;
        public String shipping_option_id;

        public void readParams(AbstractSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            recurring_init = (flags & 4) != 0;
            recurring_used = (flags & 8) != 0;
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            payload = stream.readByteArray(exception);
            if ((flags & 1) != 0) {
                info = TL_paymentRequestedInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                shipping_option_id = stream.readString(exception);
            }
        }

        public void serializeToStream(AbstractSerializedData stream) {
            stream.writeInt32(constructor);
            flags = recurring_init ? (flags | 4) : (flags &~ 4);
            flags = recurring_used ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeByteArray(payload);
            if ((flags & 1) != 0) {
                info.serializeToStream(stream);
            }
       