import { Tooltip } from 'antd'
import classNames from 'classnames'
import { observer } from 'mobx-react-lite'
import { forwardRef, useState } from 'react'

import Icon, {
	BarsOutlined,
	CloudDownloadOutlined,
	CloudSyncOutlined,
	FileProtectOutlined,
	FileTextOutlined,
	LogoutOutlined,
	PartitionOutlined,
	SaveOutlined,
	SearchOutlined,
	SettingOutlined,
	TagsOutlined,
	UndoOutlined,
	UnlockOutlined,
} from '@ant-design/icons'

import { showNebula } from '@/components/Nebula'
import { Settings } from '@/components/Settings'
import {
	TAB_CONTENT_TAGS,
	TAB_FILE_TREE,
	TAB_LINK_TAGS,
	TAB_OUTLINE,
	TAB_SAVE,
	TAB_SEARCH,
	TYPE_MD,
	TYPE_NONE,
	TYPE_SOURCE_CODE,
} from '@/constants'
import i18n from '@/i18n/index'
import LinkSvg from '@/icons/link.svg?react'
import starsSvg from '@/icons/stars.svg?react'
import { invoker } from '@/invoker'
import globalStore from '@/stores/globalStore'
import settingStore from '@/stores/settingStore'
import { showOutline } from '@/stores_utils'
import { syncAdapter, syncIsEnabled } from '@/synchronizer'
import { Func_Any_Void, Func_Empty_Void, TabId } from '@/types'
import { isMobileScreen } from '@/utils/media_query'

import styles from './styles.module.scss'

const t = i18n.t
type Props = {
	saveEditorContent: Func_Any_Void
	onDecryptContent: Func_Any_Void
	onSaveEncrypt: Func_Any_Void
	onSaveUnencrypt: Func_Any_Void
	onRestoreContent: Func_Any_Void
	onSyncFinished: Func_Empty_Void
	onOpenFile: (filePath: string, callback?: Func_Empty_Void) => void
	toggleTab: (tab: TabId) => void
	className?: string
}

const iconStyle = {
	padding: `var(--fvm-desktop-activity-bar-padding)`,
	fontSize: `var(--fvm-desktop-activity-bar-icon-size)`,
}

const ActivityBar = forwardRef<HTMLDivElement, Props>(
	(
		{
			saveEditorContent,
			onDecryptContent,
			onSaveEncrypt,
			onSaveUnencrypt,
			onRestoreContent,
			onSyncFinished,
			onOpenFile,
			toggleTab,
			className,
		},
		ref,
	) => {
		const GD = globalStore.getData()
		const SD = settingStore.getData()

		const syncer = syncAdapter()
		const tabPosition = isMobileScreen() ? 'top' : 'left'

		const [settingVisible, setSettingVisible] = useState(false)
		const closeSetting = () => {
			setSettingVisible(false)
		}

		const initAsRemote = async () => {
			if (
				await invoker.confirm(
					t('Initializing data from a remote end will result in clearing local data. Are you sure?'),
					'Warning',
				)
			) {
				if (syncIsEnabled()) invoker.alert(t('Syncing'))
				const rrr = await syncer.initFromRemoteZip()
				if (rrr.success) {
					invoker.success(t('Sync successful'))
					onSyncFinished()
				} else {
					console.error(t('Sync failed'), rrr.errMsg)
					invoker.warning(t('Sync failed'))
				}
			}
		}

		const syncData = async () => {
			if (syncIsEnabled()) invoker.alert(t('Syncing'))
			const rrr = await syncer.sync()
			if (rrr.success) {
				invoker.success(t('Sync successful'))
				onSyncFinished()
			} else {
				console.error(t('Sync failed'), rrr.errMsg)
				invoker.warning(t('Sync failed') + '\n' + rrr.errMsg)
			}
		}

		const refreshPage = () => {
			invoker.reoladWindow()
		}

		const fileOpened = () => {
			return GD.currentFileName !== ''
		}

		return (
			<>
				<div className={classNames(styles.ActivityBar, className)} data-tauri-drag-region>
					<Tooltip placement="right" title={t('File tree')}>
						<div
							className={classNames(styles.Item, GD.tabId === TAB_FILE_TREE ? styles.Active : '')}
							onClick={() => {
								toggleTab(TAB_FILE_TREE)
							}}
						>
							<PartitionOutlined style={iconStyle} />
						</div>
					</Tooltip>

					{fileOpened() && showOutline() && (
						<Tooltip placement="right" title={t('Outline')}>
							<div
								className={classNames(styles.Item, GD.tabId === TAB_OUTLINE ? styles.Active : '')}
								onClick={() => {
									toggleTab(TAB_OUTLINE)
								}}
							>
								<BarsOutlined style={iconStyle} />
							</div>
						</Tooltip>
					)}

					<Tooltip placement="right" title={t('Search')}>
						<div
							className={classNames(styles.Item, GD.tabId === TAB_SEARCH ? styles.Active : '')}
							onClick={() => {
								toggleTab(TAB_SEARCH)
							}}
						>
							<SearchOutlined style={iconStyle} />
						</div>
					</Tooltip>

					<Tooltip placement="right" title={t('Content tags')}>
						<div
							className={classNames(styles.Item, GD.tabId === TAB_CONTENT_TAGS ? styles.Active : '')}
							onClick={() => {
								toggleTab(TAB_CONTENT_TAGS)
							}}
						>
							<TagsOutlined style={iconStyle} />
						</div>
					</Tooltip>

					{fileOpened() && (
						<>
							<Tooltip placement="right" title={t('Link tags')}>
								<div
									className={classNames(styles.Item, GD.tabId === TAB_LINK_TAGS ? styles.Active : '')}
									onClick={() => {
										toggleTab(TAB_LINK_TAGS)
									}}
								>
									<Icon component={LinkSvg} style={iconStyle} />
								</div>
							</Tooltip>
						</>
					)}

					<>
						<Tooltip placement="right" title={t('Nebula')}>
							<div
								className={classNames(styles.Item)}
								onClick={() => {
									showNebula(onOpenFile)
								}}
							>
								<Icon component={starsSvg} style={iconStyle} />
							</div>
						</Tooltip>
					</>

					{/* 
					{(GD.editorType === TYPE_MD || GD.editorType === TYPE_SOURCE_CODE) &&
						!GD.currentFileName.endsWith(SD.encryptedFileExt) && (
							<>
								<Tooltip placement="right" title={t('Try to decrypt the content')}>
									<div className={classNames(styles.Item, styles.TextAction)} onClick={onDecryptContent}>
										<UnlockOutlined style={iconStyle} />
									</div>
								</Tooltip>
								<Tooltip placement="right" title={t('Encryot on save')}>
									<div className={classNames(styles.Item, styles.TextAction)} onClick={onSaveEncrypt}>
										<FileProtectOutlined style={iconStyle} />
									</div>
								</Tooltip>
								<Tooltip placement="right" title={t('Not encryot on save')}>
									<div className={classNames(styles.Item, styles.TextAction)} onClick={onSaveUnencrypt}>
										<FileTextOutlined style={iconStyle} />
									</div>
								</Tooltip>
								<Tooltip placement="right" title={t('Revert')}>
									<div className={classNames(styles.Item, styles.TextAction)} onClick={onRestoreContent}>
										<UndoOutlined style={iconStyle} />
									</div>
								</Tooltip>
							</>
						)} 
						 */}

					{SD.sync.mode !== TYPE_NONE && (
						<>
							<Tooltip placement="right" title={t('Sync')}>
								<div className={classNames(styles.Item)} onClick={syncData}>
									<CloudSyncOutlined style={iconStyle} />
								</div>
							</Tooltip>
							<Tooltip placement="right" title={t('Initialize from remote end')}>
								<div className={classNames(styles.Item)} onClick={initAsRemote}>
									<CloudDownloadOutlined style={iconStyle} />
								</div>
							</Tooltip>
						</>
					)}

					<div className={classNames(styles.Bottom)}>
						{fileOpened() && (
							<Tooltip placement="right" title={t('Save')}>
								<div
									className={classNames(styles.Item, GD.tabId === TAB_SAVE ? styles.Active : '')}
									onClick={saveEditorContent}
								>
									<SaveOutlined style={iconStyle} />
								</div>
							</Tooltip>
						)}

						<Tooltip placement="right" title={t('Logout')}>
							<div className={classNames(styles.Item)} onClick={refreshPage}>
								<LogoutOutlined style={iconStyle} />
							</div>
						</Tooltip>

						<Tooltip placement="right" title={t('Setting')}>
							<div
								className={classNames(styles.Item)}
								onClick={() => {
									setSettingVisible(true)
								}}
							>
								<SettingOutlined style={iconStyle} />
							</div>
						</Tooltip>
					</div>
				</div>

				<Settings
					showModal={settingVisible}
					onModalOk={closeSetting}
					onModalCancel={closeSetting}
					tabPosition={tabPosition}
				/>
			</>
		)
	},
)

export default observer(ActivityBar)
