// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// Copyright 2024 FeatureForm Inc.
//

package provider

import (
	"bytes"
	"encoding/json"
	"fmt"
	"os"
	"os/exec"
	"strconv"
	"strings"
	"time"

	"github.com/featureform/metadata"
	"github.com/parquet-go/parquet-go"
	"github.com/pkg/errors"

	dp "github.com/novln/docker-parser"
	"go.uber.org/zap"
	_ "gocloud.dev/blob/fileblob"
	_ "gocloud.dev/blob/memblob"
	"golang.org/x/exp/slices"

	cfg "github.com/featureform/config"
	"github.com/featureform/fferr"
	filestore "github.com/featureform/filestore"
	"github.com/featureform/helpers"
	"github.com/featureform/kubernetes"
	"github.com/featureform/logging"
	pl "github.com/featureform/provider/location"
	pc "github.com/featureform/provider/provider_config"
	ps "github.com/featureform/provider/provider_schema"
)

type pandasOfflineQueries struct {
	defaultPythonOfflineQueries
}

func (q pandasOfflineQueries) trainingSetCreate(def TrainingSetDef, featureSchemas []ResourceSchema, labelSchema ResourceSchema) string {
	columns := make([]string, 0)
	joinQueries := make([]string, 0)
	featureTimestamps := make([]string, 0)
	for i, feature := range def.Features {
		featureColumnName := createQuotedIdentifier(feature)
		columns = append(columns, featureColumnName)
		var featureWindowQuery string
		// if no timestamp column, set to default generated by resource registration
		if featureSchemas[i].TS == "" {
			featureWindowQuery = fmt.Sprintf("SELECT * FROM (SELECT %s as t%d_entity, %s as %s, 0 as t%d_ts FROM source_%d) ORDER BY t%d_ts ASC", featureSchemas[i].Entity, i+1, featureSchemas[i].Value, featureColumnName, i+1, i+1, i+1)
		} else {
			featureWindowQuery = fmt.Sprintf("SELECT * FROM (SELECT %s as t%d_entity, %s as %s, %s as t%d_ts FROM source_%d) ORDER BY t%d_ts ASC", featureSchemas[i].Entity, i+1, featureSchemas[i].Value, featureColumnName, featureSchemas[i].TS, i+1, i+1, i+1)
		}
		featureJoinQuery := fmt.Sprintf("LEFT OUTER JOIN (%s) t%d ON (t%d_entity = entity AND t%d_ts <= label_ts)", featureWindowQuery, i+1, i+1, i+1)
		joinQueries = append(joinQueries, featureJoinQuery)
		featureTimestamps = append(featureTimestamps, fmt.Sprintf("t%d_ts", i+1))
	}
	for i, lagFeature := range def.LagFeatures {
		lagFeaturesOffset := len(def.Features)
		idx := slices.IndexFunc(def.Features, func(id ResourceID) bool {
			return id.Name == lagFeature.FeatureName && id.Variant == lagFeature.FeatureVariant
		})
		lagSource := fmt.Sprintf("source_%d", idx)
		lagColumnName := sanitize(lagFeature.LagName)
		if lagFeature.LagName == "" {
			lagColumnName = sanitize(fmt.Sprintf("%s_%s_lag_%s", lagFeature.FeatureName, lagFeature.FeatureVariant, lagFeature.LagDelta))
		}
		columns = append(columns, lagColumnName)
		timeDeltaSeconds := lagFeature.LagDelta.Seconds() //parquet stores time as microseconds
		curIdx := lagFeaturesOffset + i + 1
		var lagWindowQuery string
		if featureSchemas[idx].TS == "" {
			lagWindowQuery = fmt.Sprintf("SELECT * FROM (SELECT %s as t%d_entity, %s as %s, 0 as t%d_ts FROM %s) ORDER BY t%d_ts ASC", featureSchemas[idx].Entity, curIdx, featureSchemas[idx].Value, lagColumnName, curIdx, lagSource, curIdx)
		} else {
			lagWindowQuery = fmt.Sprintf("SELECT * FROM (SELECT %s as t%d_entity, %s as %s, %s as t%d_ts FROM %s) ORDER BY t%d_ts ASC", featureSchemas[idx].Entity, curIdx, featureSchemas[idx].Value, lagColumnName, featureSchemas[idx].TS, curIdx, lagSource, curIdx)
		}
		lagJoinQuery := fmt.Sprintf("LEFT OUTER JOIN (%s) t%d ON (t%d_entity = entity AND DATETIME(t%d_ts, '+%f seconds') <= label_ts)", lagWindowQuery, curIdx, curIdx, curIdx, timeDeltaSeconds)
		joinQueries = append(joinQueries, lagJoinQuery)
		featureTimestamps = append(featureTimestamps, fmt.Sprintf("t%d_ts", curIdx))
	}
	columnStr := strings.Join(columns, ", ")
	joinQueryString := strings.Join(joinQueries, " ")
	var labelWindowQuery string
	if labelSchema.TS == "" {
		labelWindowQuery = fmt.Sprintf("SELECT %s AS entity, %s AS value, 0 AS label_ts FROM source_0", labelSchema.Entity, labelSchema.Value)
	} else {
		labelWindowQuery = fmt.Sprintf("SELECT %s AS entity, %s AS value, %s AS label_ts FROM source_0", labelSchema.Entity, labelSchema.Value, labelSchema.TS)
	}
	labelPartitionQuery := fmt.Sprintf("(SELECT * FROM (SELECT entity, value, label_ts FROM (%s) t ) t0)", labelWindowQuery)
	labelJoinQuery := fmt.Sprintf("%s %s", labelPartitionQuery, joinQueryString)

	timeStamps := strings.Join(featureTimestamps, ", ")
	timeStampsDesc := strings.Join(featureTimestamps, " DESC,")
	fullQuery := fmt.Sprintf("SELECT %s, value AS %s, entity, label_ts, %s, ROW_NUMBER() over (PARTITION BY entity, value, label_ts ORDER BY label_ts DESC, %s DESC) as row_number FROM (%s) tt", columnStr, createQuotedIdentifier(def.Label), timeStamps, timeStampsDesc, labelJoinQuery)
	finalQuery := fmt.Sprintf("SELECT %s, %s FROM (SELECT * FROM (SELECT *, row_number FROM (%s) WHERE row_number=1 ))  ORDER BY label_ts", columnStr, createQuotedIdentifier(def.Label), fullQuery)
	return finalQuery
}

type K8sOfflineStore struct {
	executor Executor
	store    FileStore
	logger   *zap.SugaredLogger
	query    *pandasOfflineQueries
	BaseProvider
}

func (k8s *K8sOfflineStore) AsOfflineStore() (OfflineStore, error) {
	return k8s, nil
}

func (k8s *K8sOfflineStore) Close() error {
	return k8s.store.Close()
}

func (k8s K8sOfflineStore) GetBatchFeatures(tables []ResourceID) (BatchFeatureIterator, error) {
	return nil, fferr.NewInternalError(fmt.Errorf("batch features not implemented for this provider"))
}

type Config []byte

type ExecutorFactory func(config Config, logger *zap.SugaredLogger) (Executor, error)

var executorFactoryMap = make(map[string]ExecutorFactory)

func RegisterExecutorFactory(name string, executorFactory ExecutorFactory) error {
	if _, exists := executorFactoryMap[name]; exists {
		return fferr.NewInternalError(fmt.Errorf("factory already registered: %s", name))
	}
	executorFactoryMap[name] = executorFactory
	return nil
}

func CreateExecutor(name string, config Config, logger *zap.SugaredLogger) (Executor, error) {
	factory, exists := executorFactoryMap[name]
	if !exists {
		return nil, fferr.NewInternalError(fmt.Errorf("factory does not exist: %s", name))
	}
	executor, err := factory(config, logger)
	if err != nil {
		return nil, err
	}
	return executor, nil
}

type FileStoreFactory func(config Config) (FileStore, error)

var fileStoreFactoryMap = make(map[string]FileStoreFactory)

func RegisterFileStoreFactory(name string, FileStoreFactory FileStoreFactory) error {
	if _, exists := fileStoreFactoryMap[name]; exists {
		return fferr.NewInternalError(fmt.Errorf("factory already registered: %s", name))
	}
	fileStoreFactoryMap[name] = FileStoreFactory
	return nil
}

func CreateFileStore(name string, config Config) (FileStore, error) {
	factory, exists := fileStoreFactoryMap[name]
	if !exists {
		return nil, fferr.NewInternalError(fmt.Errorf("factory does not exist: %s", name))
	}
	FileStore, err := factory(config)
	if err != nil {
		return nil, fferr.NewInternalError(fmt.Errorf("failed to create FileStore: %v", err))
	}
	return FileStore, nil
}

func init() {
	FileStoreFactoryMap := map[filestore.FileStoreType]FileStoreFactory{
		filestore.FileSystem: NewLocalFileStore,
		filestore.Azure:      NewAzureFileStore,
		filestore.S3:         NewS3FileStore,
		filestore.GCS:        NewGCSFileStore,
		filestore.HDFS:       NewHDFSFileStore,
	}
	executorFactoryMap := map[pc.ExecutorType]ExecutorFactory{
		pc.GoProc: NewLocalExecutor,
		pc.K8s:    NewKubernetesExecutor,
	}
	for storeType, factory := range FileStoreFactoryMap {
		err := RegisterFileStoreFactory(string(storeType), factory)
		if err != nil {
			panic(err)
		}
	}
	for executorType, factory := range executorFactoryMap {
		err := RegisterExecutorFactory(string(executorType), factory)
		if err != nil {
			panic(err)
		}
	}
}

func k8sOfflineStoreFactory(config pc.SerializedConfig) (Provider, error) {
	k8 := pc.K8sConfig{}
	logger := logging.NewLogger("kubernetes")
	if err := k8.Deserialize(config); err != nil {
		logger.Errorw("Invalid config to initialize k8s offline store", "error", err)
		return nil, err
	}
	logger.Info("Creating executor with type:", k8.ExecutorType)
	execConfig := k8.ExecutorConfig.(pc.ExecutorConfig)
	serializedExecutor, err := execConfig.Serialize()
	if err != nil {
		logger.Errorw("Failure serializing executor", "executor_type", k8.ExecutorType, "error", err)
		return nil, err
	}
	executor, err := CreateExecutor(string(k8.ExecutorType), serializedExecutor, logger.SugaredLogger)
	if err != nil {
		logger.Errorw("Failure initializing executor", "executor_type", k8.ExecutorType, "error", err)
		return nil, err
	}

	serializedBlob, err := k8.StoreConfig.Serialize()
	if err != nil {
		return nil, err
	}

	logger.Info("Creating blob store with type:", k8.StoreType)
	store, err := CreateFileStore(string(k8.StoreType), serializedBlob)
	if err != nil {
		logger.Errorw("Failure initializing blob store with type", "type", k8.StoreType, "error", err)
		return nil, err
	}
	logger.Debugf("Store type: %s", k8.StoreType)
	queries := pandasOfflineQueries{}
	k8sOfflineStore := K8sOfflineStore{
		executor: executor,
		store:    store,
		logger:   logger.SugaredLogger,
		query:    &queries,
		BaseProvider: BaseProvider{
			ProviderType:   "K8S_OFFLINE",
			ProviderConfig: config,
		},
	}
	return &k8sOfflineStore, nil
}

type Executor interface {
	ExecuteScript(envVars map[string]string, args *metadata.KubernetesArgs) error
}

type LocalExecutor struct {
	scriptPath string
}

func (local LocalExecutor) ExecuteScript(envVars map[string]string, args *metadata.KubernetesArgs) error {
	envVars["MODE"] = "local"
	for key, value := range envVars {
		if err := os.Setenv(key, value); err != nil {
			return fferr.NewInternalError(fmt.Errorf("could not set env variable: %s: %w", key, err))
		}
	}
	cmd := exec.Command("python3", local.scriptPath)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	if err := cmd.Run(); err != nil {
		return fferr.NewInternalError(fmt.Errorf("could not execute python function: %v", err))
	}
	return nil
}

type LocalExecutorConfig struct {
	ScriptPath string
}

func (config *LocalExecutorConfig) Serialize() ([]byte, error) {
	data, err := json.Marshal(config)
	if err != nil {
		return nil, err
	}
	return data, nil
}

func (config *LocalExecutorConfig) Deserialize(data []byte) error {
	err := json.Unmarshal(data, config)
	if err != nil {
		return fferr.NewInternalError(err)
	}
	return nil
}

func NewLocalExecutor(config Config, logger *zap.SugaredLogger) (Executor, error) {
	localConfig := LocalExecutorConfig{}
	if err := localConfig.Deserialize(config); err != nil {
		return nil, err
	}
	_, err := os.Open(localConfig.ScriptPath)
	if err != nil {
		return nil, fferr.NewInternalError(err)
	}
	return LocalExecutor{
		scriptPath: localConfig.ScriptPath,
	}, nil
}

type KubernetesExecutor struct {
	logger *zap.SugaredLogger
	image  string
}

// isDefaultImage checks that the current image name (excluding the tag) is the same as the default image
// name config.PandasBaseImage. It also validates that the name is a valid docker image name
func (kube *KubernetesExecutor) isDefaultImage() (bool, error) {
	parse, err := dp.Parse(kube.image)
	if err != nil {
		return false, fferr.NewInvalidArgumentError(fmt.Errorf("invalid Docker image name (%s): %w", kube.image, err))
	}
	return parse.ShortName() == cfg.PandasBaseImage, nil
}

func (kube *KubernetesExecutor) setCustomImage(image string) {
	if image != "" {
		kube.image = image
	}
}

func (kube *KubernetesExecutor) ExecuteScript(envVars map[string]string, args *metadata.KubernetesArgs) error {
	kube.logger.Debugw("Executing k8s script", "args", args)
	var specs metadata.KubernetesResourceSpecs
	if args != nil {
		kube.setCustomImage(args.DockerImage)
		specs = args.Specs
	}
	if isDefault, err := kube.isDefaultImage(); err != nil {
		return err
	} else if !isDefault {
		kube.logger.Warnf("You are using a custom Docker Image (%s) for a Kubernetes job. This may have unintended behavior.", kube.image)
	}
	envVars["MODE"] = "k8s"
	resourceType, err := strconv.Atoi(envVars["RESOURCE_TYPE"])
	if err != nil {
		resourceType = 0
	}
	config := kubernetes.KubernetesRunnerConfig{
		JobPrefix: "kcf",
		EnvVars:   envVars,
		Image:     kube.image,
		NumTasks:  1,
		Resource: metadata.ResourceID{
			Name:    envVars["RESOURCE_NAME"],
			Variant: envVars["RESOURCE_VARIANT"],
			Type:    ProviderToMetadataResourceType[OfflineResourceType(resourceType)],
		},
		Specs: specs,
	}
	jobRunner, err := kubernetes.NewKubernetesRunner(config)
	if err != nil {
		return err
	}
	completionWatcher, err := jobRunner.Run()
	if err != nil {
		return err
	}
	if err := completionWatcher.Wait(); err != nil {
		return err
	}
	return nil
}

func NewKubernetesExecutor(config Config, logger *zap.SugaredLogger) (Executor, error) {
	var c pc.ExecutorConfig
	err := c.Deserialize(config)
	if err != nil {
		return nil, err
	}
	return &KubernetesExecutor{
		image:  c.GetImage(),
		logger: logger,
	}, nil
}

type BlobOfflineTable struct {
	store  FileStore
	schema ResourceSchema
}

// TODO: implement
func (tbl *BlobOfflineTable) Write(ResourceRecord) error {
	return fferr.NewInternalError(fmt.Errorf("not yet implemented"))
}

func (tbl *BlobOfflineTable) WriteBatch(records []ResourceRecord) error {
	destination, isFilestoreLocation := tbl.schema.SourceTable.(*pl.FileStoreLocation)
	if !isFilestoreLocation {
		return fferr.NewInternalErrorf("source table is not a FileStoreLocation")
	}
	exists, err := tbl.store.Exists(destination)
	if err != nil {
		return err
	}
	// It's very possible the resource table's data source already exists, in which case we'll
	// need to read the existing data and append the new records to it.
	if exists {
		// **NOTE:**Consider a better way of handling records written to resource tables such that
		// we don't rely on the order of rows in the file for materializations; currently, we're
		// counting on the implicit ordering of rows once these files are read into Spark and
		// queried for materializations.
		iter, err := tbl.store.Serve([]filestore.Filepath{destination.Filepath()})
		if err != nil {
			return err
		}
		records, err = tbl.append(iter, records)
		if err != nil {
			return err
		}
	}
	data, err := tbl.writeRecordsToParquetBytes(records)
	if err != nil {
		return err
	}
	return tbl.store.Write(destination.Filepath(), data)
}

func (table *BlobOfflineTable) Location() pl.Location {
	return nil
}

// TODO: Add unit tests for this method
func (tbl *BlobOfflineTable) append(iter Iterator, newRecords []ResourceRecord) ([]ResourceRecord, error) {
	records := make([]ResourceRecord, 0)
	for {
		val, err := iter.Next()
		if err != nil {
			return nil, err
		}
		if val == nil {
			break
		}
		record := ResourceRecord{
			Entity: val["Entity"].(string),
			Value:  val["Value"],
			TS:     val["TS"].(time.Time),
		}
		records = append(records, record)
	}
	records = append(records, newRecords...)
	return records, nil
}

// TODO: Add unit tests for this method
func (tbl *BlobOfflineTable) convertToGenericResourceRecord(record *ResourceRecord) (interface{}, error) {
	switch v := record.Value.(type) {
	case int:
		// **NOTE:** github.com/parquet-go/parquet-go does not support int, so this value was being cast to int64
		// at the time of writing to the filestore. This remains an issue for us given int is a valid ScalarType,
		// so we'll need to determine how to handle this.
		return &GenericResourceRecord[int32]{Entity: record.Entity, Value: int32(v), TS: record.TS}, nil
	case int32:
		return &GenericResourceRecord[int32]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	case int64:
		return &GenericResourceRecord[int64]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	case float32:
		return &GenericResourceRecord[float32]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	case float64:
		return &GenericResourceRecord[float64]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	case string:
		return &GenericResourceRecord[string]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	case bool:
		return &GenericResourceRecord[bool]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	case time.Time:
		return &GenericResourceRecord[time.Time]{Entity: record.Entity, Value: v, TS: record.TS}, nil
	default:
		return nil, fferr.NewDataTypeNotFoundErrorf(v, "unable to convert to generic resource record")
	}
}

// TODO: Add unit tests for this method
func (tbl *BlobOfflineTable) writeRecordsToParquetBytes(records []ResourceRecord) ([]byte, error) {
	parquetRecords := []any{}
	for _, record := range records {
		r, err := tbl.convertToGenericResourceRecord(&record)
		if err != nil {
			return nil, err
		}
		parquetRecords = append(parquetRecords, r)
	}
	buf := new(bytes.Buffer)
	var schemaRecord interface{}
	// If there are no records, we still need to write the schema to the parquet file
	// to avoid Spark errors (e.g. "... failed to read footer ...") when reading the file.
	if len(parquetRecords) > 0 {
		schemaRecord = parquetRecords[0]
	} else {
		schemaRecord = &GenericResourceRecord[int16]{}
	}
	schema := parquet.SchemaOf(schemaRecord)
	err := parquet.Write[any](buf, parquetRecords, schema)
	if err != nil {
		return nil, fferr.NewInternalError(err)
	}
	return buf.Bytes(), nil
}

func (k8s *K8sOfflineStore) RegisterResourceFromSourceTable(id ResourceID, schema ResourceSchema, opts ...ResourceOption) (OfflineTable, error) {
	if len(opts) > 0 {
		return nil, fferr.NewInternalError(fmt.Errorf("K8s does not support resource options"))
	}
	return blobRegisterResourceFromSourceTable(id, schema, k8s.logger, k8s.store)
}

func blobRegisterResourceFromSourceTable(id ResourceID, sourceSchema ResourceSchema, logger *zap.SugaredLogger, store FileStore) (OfflineTable, error) {
	if err := id.check(Feature, Label); err != nil {
		logger.Errorw("Failure checking ID", "error", err)
		return nil, err
	}
	destination, err := store.CreateFilePath(id.ToFilestorePath(), false)
	if err != nil {
		return nil, err
	}
	resourceExists, err := store.Exists(pl.NewFileLocation(destination))
	if err != nil {
		logger.Errorw("Error checking if resource exists", "error", err)
		return nil, err
	}
	if resourceExists {
		logger.Errorw("Resource already exists in blob store", "id", id, "ResourceKey", destination.Key())
		return nil, fferr.NewDatasetAlreadyExistsError(id.Name, id.Variant, fmt.Errorf("resource already exists in blob store: %s", destination.Key()))
	}
	serializedSchema, err := sourceSchema.Serialize()
	if err != nil {
		return nil, err
	}
	if err := store.Write(destination, serializedSchema); err != nil {
		return nil, err
	}
	logger.Debugw("Registered resource table", "resourceID", id, "for source", sourceSchema.SourceTable)
	return &BlobOfflineTable{schema: sourceSchema, store: store}, nil
}

func (k8s *K8sOfflineStore) RegisterPrimaryFromSourceTable(id ResourceID, tableLocation pl.Location) (PrimaryTable, error) {
	fileStoreLocation, isFileStoreLocation := tableLocation.(*pl.FileStoreLocation)
	if !isFileStoreLocation {
		return nil, fferr.NewInternalError(fmt.Errorf("location is not a FileStoreLocation"))
	}
	return blobRegisterPrimary(id, *fileStoreLocation, k8s.logger, k8s.store)
}

func blobRegisterPrimary(id ResourceID, location pl.FileStoreLocation, logger *zap.SugaredLogger, store FileStore) (PrimaryTable, error) {
	storeExists, err := store.Exists(&location)
	if err != nil {
		return nil, err
	}
	if !storeExists {
		return nil, fferr.NewDatasetNotFoundError(id.Name, id.Variant, fmt.Errorf(location.Location()))
	}

	filepath, err := store.CreateFilePath(id.ToFilestorePath(), false)
	if err != nil {
		return nil, err
	}
	logger.Infow("Checking if resource key exists", "key", filepath.Key())
	primaryFilePathLocation := pl.NewFileLocation(filepath)
	primaryExists, err := store.Exists(primaryFilePathLocation)
	if err != nil {
		logger.Errorw("Error checking if primary exists", "error", err)
		return nil, err
	}
	if primaryExists {
		logger.Errorw("File already registered", "source", location.Location())
		return nil, fferr.NewDatasetAlreadyExistsError(id.Name, id.Variant, fmt.Errorf(location.Location()))
	}
	logger.Debugw("Registering primary table", "id", id, "source", location.Location())
	// TODO: determine how to handle the schema of the primary table; if it's a parquet file, we _could_
	// read the file and infer a schema based; however, this wouldn't be possible for CSV files.
	schema := TableSchema{
		SourceTable: location.Location(),
	}
	data, err := schema.Serialize()
	if err != nil {
		return nil, err
	}
	// **NOTE:** The data we're writing to the blob store is the path to the primary source data file.
	// This blob will be read by other processes (e.g. transformation jobs) to fetch where the primary
	// data is stored prior to acting on it. You can verify this by accessing the object stored at
	// /featureform/Primary/<NAME>/<VARIANT>
	if err := store.Write(filepath, data); err != nil {
		logger.Errorw("Could not write primary table", "error", err)
		return nil, err
	}

	logger.Debugw("Successfully registered primary table", "id", id, "source", location.Location())
	return &FileStorePrimaryTable{store, location.Filepath(), schema, false, id}, nil
}

func (store *K8sOfflineStore) SupportsTransformationOption(opt TransformationOptionType) (bool, error) {
	return false, nil
}

func (k8s *K8sOfflineStore) CreateTransformation(config TransformationConfig, opts ...TransformationOption) error {
	if len(opts) > 0 {
		return fferr.NewInternalErrorf("K8s does not support transformation options")
	}
	return k8s.transformation(config, false)
}

func (k8s *K8sOfflineStore) transformation(config TransformationConfig, isUpdate bool) error {
	if config.Type == SQLTransformation {
		return k8s.sqlTransformation(config, isUpdate)
	} else if config.Type == DFTransformation {
		return k8s.dfTransformation(config, isUpdate)
	} else {
		k8s.logger.Errorw("the transformation type is not supported", "type", config.Type)
		return fferr.NewInternalError(fmt.Errorf("the transformation type '%v' is not supported", config.Type))
	}
}

func addETCDVars(envVars map[string]string) map[string]string {
	etcdHost := helpers.GetEnv("ETCD_HOST", "localhost")
	etcdPort := helpers.GetEnv("ETCD_PORT", "2379")
	etcdPassword := helpers.GetEnv("ETCD_PASSWORD", "secretpassword")
	etcdUsername := helpers.GetEnv("ETCD_USERNAME", "root")
	envVars["ETCD_HOST"] = etcdHost
	envVars["ETCD_PASSWORD"] = etcdPassword
	envVars["ETCD_PORT"] = etcdPort
	envVars["ETCD_USERNAME"] = etcdUsername
	return envVars
}

func (k8s *K8sOfflineStore) pandasRunnerArgs(outputURI string, updatedQuery string, sources []string, jobType JobType) map[string]string {
	sourceList := strings.Join(sources, ",")
	envVars := map[string]string{
		"OUTPUT_URI":          outputURI,
		"SOURCES":             sourceList,
		"TRANSFORMATION_TYPE": "sql",
		"TRANSFORMATION":      updatedQuery,
	}
	envVars = k8s.store.AddEnvVars(envVars)
	return envVars
}

func (k8s K8sOfflineStore) getDFArgs(outputURI string, code string, mapping []SourceMapping, sources []string) map[string]string {
	sourceList := strings.Join(sources, ",")
	envVars := map[string]string{
		"OUTPUT_URI":          outputURI,
		"SOURCES":             sourceList,
		"TRANSFORMATION_TYPE": "df",
		"TRANSFORMATION":      code,
	}
	envVars = k8s.store.AddEnvVars(envVars)
	return envVars
}

func addResourceID(envVars map[string]string, id ResourceID) map[string]string {
	envVars["RESOURCE_NAME"] = id.Name
	envVars["RESOURCE_VARIANT"] = id.Variant
	envVars["RESOURCE_TYPE"] = fmt.Sprintf("%d", id.Type)
	return envVars
}

func (k8s *K8sOfflineStore) sqlTransformation(config TransformationConfig, isUpdate bool) error {
	updatedQuery, sources, err := k8s.updateQuery(config.Query, config.SourceMapping)
	if err != nil {
		k8s.logger.Errorw("Could not generate updated query for k8s transformation", err)
		return err
	}

	filepath, err := k8s.store.CreateFilePath(ps.ResourceToDirectoryPath(config.TargetTableID.Type.String(), config.TargetTableID.Name, config.TargetTableID.Variant), false)
	if err != nil {
		return err
	}
	transformationDestinationExactPath, err := k8s.store.NewestFileOfType(filepath, filestore.Parquet)
	if err != nil {
		k8s.logger.Errorw("Could not get newest blob", "location", filepath.Key(), "error", err)
		return err
	}
	exists := transformationDestinationExactPath != nil
	if !isUpdate && exists {
		k8s.logger.Errorw("Creation when transformation already exists", "target_table", config.TargetTableID, "destination", filepath.ToURI())
		return fferr.NewDatasetAlreadyExistsError(config.TargetTableID.Name, config.TargetTableID.Variant, fmt.Errorf(filepath.ToURI()))
	} else if isUpdate && !exists {
		k8s.logger.Errorw("Update job attempted when transformation does not exist", "target_table", config.TargetTableID, "destination", filepath.ToURI())
		return fferr.NewDatasetNotFoundError(config.TargetTableID.Name, config.TargetTableID.Variant, fmt.Errorf(filepath.ToURI()))
	}
	k8s.logger.Debugw("Running SQL transformation", "target_table", config.TargetTableID, "query", config.Query)
	runnerArgs := k8s.pandasRunnerArgs(filepath.ToURI(), updatedQuery, sources, Transform)
	runnerArgs = addResourceID(runnerArgs, config.TargetTableID)

	args, err := k8s.checkArgs(config.Args)
	if err != nil {
		return err
	}
	if err := k8s.executor.ExecuteScript(runnerArgs, &args); err != nil {
		k8s.logger.Errorw("job for transformation failed to run", "target_table", config.TargetTableID, "error", err)
		return err
	}

	k8s.logger.Debugw("Successfully ran SQL transformation", "target_table", config.TargetTableID, "query", config.Query)
	return nil
}

func (k8s *K8sOfflineStore) checkArgs(args metadata.TransformationArgs) (metadata.KubernetesArgs, error) {
	k8sArgs, ok := args.(metadata.KubernetesArgs)
	if !ok {
		return metadata.KubernetesArgs{}, fferr.NewInternalError(fmt.Errorf("args is not of type KubernetesArgs"))
	}
	return k8sArgs, nil
}

func (k8s *K8sOfflineStore) dfTransformation(config TransformationConfig, isUpdate bool) error {
	_, sources, err := k8s.updateQuery(config.Query, config.SourceMapping)
	if err != nil {
		return err
	}
	filepath, err := k8s.store.CreateFilePath(ps.ResourceToDirectoryPath(config.TargetTableID.Type.String(), config.TargetTableID.Name, config.TargetTableID.Variant), false)
	if err != nil {
		return err
	}
	exists, err := k8s.store.Exists(pl.NewFileLocation(filepath))
	if err != nil {
		k8s.logger.Errorw("Error checking if resource exists", "error", err)
		return err
	}

	if !isUpdate && exists {
		k8s.logger.Errorw("Transformation already exists", "target_table", config.TargetTableID, "destination", filepath.ToURI())
		return fferr.NewDatasetAlreadyExistsError(config.TargetTableID.Name, config.TargetTableID.Variant, fmt.Errorf(filepath.ToURI()))
	} else if isUpdate && !exists {
		k8s.logger.Errorw("Transformation doesn't exists at destination and you are trying to update", "target_table", config.TargetTableID, "destination", filepath.ToURI())
		return fferr.NewDatasetNotFoundError(config.TargetTableID.Name, config.TargetTableID.Variant, fmt.Errorf(filepath.ToURI()))
	}

	transformationFilepath, err := k8s.store.CreateFilePath(ps.ResourceToPicklePath(config.TargetTableID.Name, config.TargetTableID.Variant), false)
	if err != nil {
		return err
	}
	err = k8s.store.Write(transformationFilepath, config.Code)
	if err != nil {
		return err
	}

	dfArgs := k8s.getDFArgs(filepath.ToURI(), transformationFilepath.Key(), config.SourceMapping, sources)
	dfArgs = addResourceID(dfArgs, config.TargetTableID)
	k8s.logger.Debugw("Running DF transformation", "target_table", config.TargetTableID)
	args, err := k8s.checkArgs(config.Args)
	if err != nil {
		return err
	}
	if err := k8s.executor.ExecuteScript(dfArgs, &args); err != nil {
		k8s.logger.Errorw("Error running dataframe job", "error", err)
		return err
	}

	k8s.logger.Debugw("Successfully ran DF transformation", "target_table", config.TargetTableID)
	return nil
}

func (k8s *K8sOfflineStore) updateQuery(query string, mapping []SourceMapping) (string, []string, error) {
	sources := make([]string, len(mapping))
	replacements := make([]string, len(mapping)*2) // It's times 2 because each replacement will be a pair; (original, replacedValue)

	for i, m := range mapping {
		replacements = append(replacements, m.Template)
		replacements = append(replacements, fmt.Sprintf("source_%v", i))

		sourcePath := ""
		sourcePath, err := k8s.getSourcePath(m.Source)
		k8s.logger.Debugw("Fetched Source Path", "source", m.Source, "sourcePath", sourcePath)
		if err != nil {
			k8s.logger.Errorw("Error getting source path of source", "source", m.Source, "error", err)
			return "", nil, err
		}

		sources[i] = sourcePath
	}

	replacer := strings.NewReplacer(replacements...)
	updatedQuery := replacer.Replace(query)

	if strings.Contains(updatedQuery, "{{") {
		k8s.logger.Errorw("Template replace failed", "query", updatedQuery)
		err := fferr.NewInternalError(fmt.Errorf("template replacement error"))
		err.AddDetail("Mapping", strings.Join(replacements, ","))
		err.AddDetail("Query", updatedQuery)
		return "", nil, err
	}
	return updatedQuery, sources, nil
}

// TODO: Determine if this should return filestore.Filepath or string
func (k8s *K8sOfflineStore) getSourcePath(path string) (string, error) {
	fileType, fileName, fileVariant := k8s.getResourceInformationFromFilePath(path)
	k8s.logger.Debugw("Retrieved source path", "fileType", fileType, "fileName", fileName, "fileVariant", fileVariant)

	var filePath string
	if fileType == "primary" {
		fileResourceId := ResourceID{Name: fileName, Variant: fileVariant, Type: Primary}
		fileTable, err := k8s.GetPrimaryTable(fileResourceId, metadata.SourceVariant{}) // At the moment, we don't need the source variant
		if err != nil {
			k8s.logger.Errorw("Issue getting primary table", "id", fileResourceId, "error", err)
			return "", err
		}
		filePath = fileTable.GetName()
		return filePath, nil
	} else if fileType == "transformation" {
		fileResourceId := ResourceID{Name: fileName, Variant: fileVariant, Type: Transformation}
		resourcePath := ps.ResourceToDirectoryPath(Transformation.String(), fileName, fileVariant)
		fileResourcePath, err := k8s.store.CreateFilePath(resourcePath, false)
		if err != nil {
			k8s.logger.Errorw("Could not create file path", "error", err, "fileStoreResourcePath", resourcePath)
			return "", err
		}
		k8s.logger.Debugw("Retrieved transformation source", "ResourceId", fileResourceId, "fileResourcePath", fileResourcePath)
		// get file type of source
		exactFileResourcePath, err := k8s.store.NewestFileOfType(fileResourcePath, fileResourcePath.Ext())
		k8s.logger.Debugw("Retrieved latest file path", "exactFileResourcePath", exactFileResourcePath)
		if err != nil {
			k8s.logger.Errorw("Could not get newest blob", "location", fileResourcePath, "error", err)
			return "", err
		}
		exists, err := k8s.store.Exists(pl.NewFileLocation(exactFileResourcePath))
		if err != nil {
			k8s.logger.Errorw("Issue getting transformation table", "id", fileResourceId)
			return "", err
		}
		if !exists {
			k8s.logger.Errorw("Transformation table does not exist", "id", fileResourceId)
			return "", fferr.NewDatasetNotFoundError(fileResourceId.Name, fileResourceId.Variant, fmt.Errorf(fileResourcePath.ToURI()))
		}
		return exactFileResourcePath.ToURI(), nil
	} else {
		return filePath, fferr.NewDatasetNotFoundError(fileName, fileVariant, fmt.Errorf(path))
	}
}

func (k8s *K8sOfflineStore) getResourceInformationFromFilePath(path string) (string, string, string) {
	var fileType string
	var fileName string
	var fileVariant string
	if path[:5] == "s3://" {
		filePaths := strings.Split(path[len("s3://"):], "/")
		if len(filePaths) <= 4 {
			return "", "", ""
		}
		fileType, fileName, fileVariant = strings.ToLower(filePaths[2]), filePaths[3], filePaths[4]
	} else if path[:5] == filestore.HDFSPrefix {
		filePaths := strings.Split(path[len(filestore.HDFSPrefix):], "/")
		if len(filePaths) <= 4 {
			return "", "", ""
		}
		fileType, fileName, fileVariant = strings.ToLower(filePaths[2]), filePaths[3], filePaths[4]
	} else {
		filePaths := strings.Split(path[len("featureform_"):], "__")
		if len(filePaths) <= 2 {
			return "", "", ""
		}
		fileType, fileName, fileVariant = filePaths[0], filePaths[1], filePaths[2]
	}
	return fileType, fileName, fileVariant
}

func (k8s *K8sOfflineStore) GetTransformationTable(id ResourceID) (TransformationTable, error) {
	resourceKey := ps.ResourceToDirectoryPath(id.Type.String(), id.Name, id.Variant)
	transformationFilepath, err := k8s.store.CreateFilePath(resourceKey, false)
	if err != nil {
		k8s.logger.Errorw("Could not create file path", "error", err, "fileStoreResourcePath", resourceKey)
		return nil, err
	}
	k8s.logger.Debugw("Retrieved transformation source", "ResourceId", id, "transformationPath", transformationFilepath.ToURI())
	if err != nil {
		k8s.logger.Errorw("Could not parse full path", "error", err, "transformationPath", transformationFilepath.ToURI())
		return nil, err
	}
	if err != nil {
		k8s.logger.Errorw("Could not create empty filepath", "error", err, "storeType", k8s.store.FilestoreType(), "transformationPath", transformationFilepath.ToURI())
		return nil, err
	}
	// TODO: populate schema
	return &FileStorePrimaryTable{k8s.store, transformationFilepath, TableSchema{}, true, id}, nil
}

func (k8s *K8sOfflineStore) UpdateTransformation(config TransformationConfig, opts ...TransformationOption) error {
	if len(opts) > 0 {
		return fferr.NewInternalErrorf("K8s does not support transformation options")
	}
	return k8s.transformation(config, true)
}
func (k8s *K8sOfflineStore) CreatePrimaryTable(id ResourceID, schema TableSchema) (PrimaryTable, error) {
	return nil, fferr.NewInternalError(fmt.Errorf("not implemented"))
}

func (k8s *K8sOfflineStore) GetPrimaryTable(id ResourceID, source metadata.SourceVariant) (PrimaryTable, error) {
	return fileStoreGetPrimary(id, k8s.store, k8s.logger)
}

func fileStoreGetPrimary(id ResourceID, store FileStore, logger *zap.SugaredLogger) (PrimaryTable, error) {
	filepath, err := store.CreateFilePath(id.ToFilestorePath(), false)
	if err != nil {
		return nil, err
	}
	logger.Debugw("Getting primary table", "id", id, "resourceKey", filepath.Key())
	table, err := store.Read(filepath)
	logger.Debugw("Read primary table", "table", string(table), "error", err)
	if err != nil {
		return nil, err
	}
	logger.Debugw("Successfully retrieved primary table", "id", id)
	schema := TableSchema{}
	if err := schema.Deserialize(table); err != nil {
		return nil, err
	}
	sourcePath, err := store.ParseFilePath(schema.SourceTable)
	if err != nil {
		return nil, err
	}
	return &FileStorePrimaryTable{store, sourcePath, schema, false, id}, nil
}

func (k8s *K8sOfflineStore) CreateResourceTable(id ResourceID, schema TableSchema) (OfflineTable, error) {
	return nil, fferr.NewInternalError(fmt.Errorf("not implemented"))
}

func (k8s *K8sOfflineStore) GetResourceTable(id ResourceID) (OfflineTable, error) {
	return fileStoreGetResourceTable(id, k8s.store, k8s.logger)
}

func fileStoreGetResourceTable(id ResourceID, store FileStore, logger *zap.SugaredLogger) (OfflineTable, error) {
	filepath, err := store.CreateFilePath(id.ToFilestorePath(), false)
	if err != nil {
		return nil, err
	}
	if exists, err := store.Exists(pl.NewFileLocation(filepath)); err != nil {
		return nil, err
	} else if !exists {
		return nil, fferr.NewDatasetNotFoundError(id.Name, id.Variant, fmt.Errorf(filepath.ToURI()))
	}
	logger.Debugw("Getting resource table", "id", id, "resourceKey", filepath.Key())
	serializedSchema, err := store.Read(filepath)
	if err != nil {
		return nil, err
	}
	resourceSchema := ResourceSchema{}
	if err := resourceSchema.Deserialize(serializedSchema); err != nil {
		return nil, err
	}
	if store.FilestoreType() == filestore.GCS {
		src, err := completePrimarySourceTablePathForGCS(resourceSchema.SourceTable.Location(), store)
		if err != nil {
			return nil, err
		}
		if src != nil {
			resourceSchema.SourceTable = pl.NewFileLocation(src)
		}
	}
	logger.Debugw("Successfully fetched resource table", "id", id)
	return &BlobOfflineTable{schema: resourceSchema, store: store}, nil
}

// **NOTE:** This is a temporary fix for GCS. The primary source table path is stored in the resource schema,
// which will contain only a directory path for resources other than a primary source. For example, or a Label that was
// created from a primary source, the value for `SourceTable` on the label schema will be:
// `featureform/Primary/<NAME>/<VARIANT>`
// This has been the pattern for all file stores given that Spark can recursively discover files "under" this
// directory path; however, in the case of GCS, Spark was interpreting this source path _as_ a parquet file,
// which results in a failure to read the file. To fix this, we need to get the remainder of the primary source's
// path to the parquet file and append it to the source path stored in the resource schema.
func completePrimarySourceTablePathForGCS(sourceTable string, store FileStore) (filestore.Filepath, error) {
	srcID := ResourceID{}
	if err := srcID.FromFilestorePath(sourceTable); err != nil {
		return nil, err
	}
	if srcID.Type == Primary {
		srcPath, err := store.CreateFilePath(srcID.ToFilestorePath(), true)
		if err != nil {
			return nil, err
		}
		src, err := store.NewestFileOfType(srcPath, filestore.Parquet)
		if err != nil {
			return nil, err
		}
		return src, nil
	}
	return nil, nil
}

func (k8s *K8sOfflineStore) CreateMaterialization(id ResourceID, opts MaterializationOptions) (Materialization, error) {
	return k8s.materialization(id, false)
}

func (store *K8sOfflineStore) SupportsMaterializationOption(opt MaterializationOptionType) (bool, error) {
	return false, nil
}

func (k8s *K8sOfflineStore) GetMaterialization(id MaterializationID) (Materialization, error) {
	return fileStoreGetMaterialization(id, k8s.store, k8s.logger)
}

func (k8s *K8sOfflineStore) ResourceLocation(id ResourceID, resource any) (pl.Location, error) {
	path, err := k8s.store.CreateFilePath(id.ToFilestorePath(), true)
	if err != nil {
		return nil, errors.Wrap(err, "could not create dir path")
	}

	newestFile, err := k8s.store.NewestFileOfType(path, filestore.Parquet)
	if err != nil {
		return nil, errors.Wrap(err, "could not get newest file")
	}

	newestDirPathDateTime, err := k8s.store.CreateFilePath(newestFile.KeyPrefix(), true)
	if err != nil {
		return nil, fmt.Errorf("could not create directory path for k8s newest file: %v", err)
	}
	return pl.NewFileLocation(newestDirPathDateTime), nil
}

func fileStoreGetMaterialization(id MaterializationID, store FileStore, logger *zap.SugaredLogger) (Materialization, error) {
	s := strings.Split(string(id), "/")
	if len(s) != 3 {
		logger.Errorw("Invalid materialization", "id", id)
		return nil, fferr.NewInternalError(fmt.Errorf("invalid materialization id: %v", id))
	}
	materializationID := ResourceID{s[1], s[2], FeatureMaterialization}
	logger.Debugw("Getting materialization", "id", id)
	logger.Debugw("Successfully retrieved materialization", "id", id)
	return &FileStoreMaterialization{materializationID, store}, nil
}

type FileStoreMaterialization struct {
	id    ResourceID
	store FileStore
}

func (mat FileStoreMaterialization) ID() MaterializationID {
	return MaterializationID(fmt.Sprintf("%s/%s/%s", FeatureMaterialization, mat.id.Name, mat.id.Variant))
}

func (mat FileStoreMaterialization) NumRows() (int64, error) {
	resourceKey := ps.ResourceToDirectoryPath(mat.id.Type.String(), mat.id.Name, mat.id.Variant)
	materializationFilepath, err := mat.store.CreateFilePath(resourceKey, false)
	if err != nil {
		return 0, err
	}
	// TODO(simba) THIS IS INCORRECT!
	latestMaterializationPath, err := mat.store.NewestFileOfType(materializationFilepath, filestore.Parquet)
	if err != nil {
		return 0, err
	}
	// TODO: convert NumRows to accept a list of files to get a total count
	return mat.store.NumRows(latestMaterializationPath)
}

func (mat FileStoreMaterialization) IterateSegment(begin, end int64) (FeatureIterator, error) {
	resourceKey := ps.ResourceToDirectoryPath(mat.id.Type.String(), mat.id.Name, mat.id.Variant)
	searchPath, err := mat.store.CreateFilePath(resourceKey, false)
	if err != nil {
		return nil, err
	}
	files, err := mat.store.List(searchPath, filestore.Parquet)
	if err != nil {
		return nil, err
	}
	groups, err := filestore.NewFilePathGroup(files, filestore.DateTimeDirectoryGrouping)
	if err != nil {
		return nil, err
	}
	newestFiles, err := groups.GetFirst()
	if err != nil {
		return nil, err
	}
	iter, err := mat.store.Serve(newestFiles)
	if err != nil {
		return nil, err
	}
	i := int64(0)
	for i = 0; i < begin; i++ {
		// TODO(simba) Can't actually skip the error here
		_, _ = iter.Next()
	}
	return &FileStoreFeatureIterator{
		iter:   iter,
		curIdx: i,
		maxIdx: end,
	}, nil
}

func (mat FileStoreMaterialization) NumChunks() (int, error) {
	searchPath, err := mat.store.CreateFilePath(ps.ResourceToDirectoryPath(mat.id.Type.String(), mat.id.Name, mat.id.Variant), false)
	if err != nil {
		return -1, err
	}
	files, err := mat.store.List(searchPath, filestore.Parquet)
	if err != nil {
		return -1, err
	}
	groups, err := filestore.NewFilePathGroup(files, filestore.DateTimeDirectoryGrouping)
	if err != nil {
		return -1, err
	}
	newestFiles, err := groups.GetFirst()
	if err != nil {
		return -1, err
	}
	return len(newestFiles), nil
}

func (mat FileStoreMaterialization) IterateChunk(idx int) (FeatureIterator, error) {
	searchPath, err := mat.store.CreateFilePath(ps.ResourceToDirectoryPath(mat.id.Type.String(), mat.id.Name, mat.id.Variant), false)
	if err != nil {
		return nil, err
	}
	files, err := mat.store.List(searchPath, filestore.Parquet)
	if err != nil {
		return nil, err
	}
	groups, err := filestore.NewFilePathGroup(files, filestore.DateTimeDirectoryGrouping)
	if err != nil {
		return nil, err
	}
	newestFiles, err := groups.GetFirst()
	if err != nil {
		return nil, err
	}
	numChunks := len(newestFiles)
	if idx >= numChunks {
		return nil, fferr.NewInternalErrorf("Chunk out of range\nIdx: %d\nChunks: %d\n", idx, numChunks)
	}
	iter, err := mat.store.Serve([]filestore.Filepath{newestFiles[idx]})
	if err != nil {
		return nil, err
	}
	return &FileStoreFeatureIterator{
		iter:   iter,
		maxIdx: -1,
	}, nil
}

func (mat FileStoreMaterialization) Location() pl.Location {
	return nil
}

type FileStoreFeatureIterator struct {
	iter   Iterator
	err    error
	cur    ResourceRecord
	curIdx int64
	// set to -1 to ignore
	maxIdx int64
}

func (iter *FileStoreFeatureIterator) Next() bool {
	iter.curIdx += 1
	if iter.maxIdx != -1 && iter.curIdx > iter.maxIdx {
		return false
	}
	nextVal, err := iter.iter.Next()
	if err != nil {
		iter.err = err
		return false
	}
	if nextVal == nil {
		return false
	}
	value := nextVal["value"]
	if err != nil {
		iter.err = err
		return false
	}
	timestamp := time.UnixMilli(0).UTC()
	ts, hasTimestamp := nextVal["ts"]
	if hasTimestamp {
		if castedTS, err := castToTimestamp(ts); err != nil {
			iter.err = err
			return false
		} else {
			timestamp = castedTS
		}
	}
	rec := ResourceRecord{
		Value: value,
		TS:    timestamp,
	}
	if err := rec.SetEntity(nextVal["entity"]); err != nil {
		iter.err = err
		return false
	}
	iter.cur = rec
	return true
}

func castToTimestamp(timestamp interface{}) (time.Time, error) {
	if ts, ok := timestamp.(time.Time); !ok {
		return time.UnixMilli(0).UTC(), fferr.NewDataTypeNotFoundErrorf(timestamp, "expected timestamp to be of type time.Time")
	} else {
		return ts, nil
	}
}

func (iter *FileStoreFeatureIterator) Value() ResourceRecord {
	return iter.cur
}

func (iter *FileStoreFeatureIterator) Err() error {
	return iter.err
}

func (iter *FileStoreFeatureIterator) Close() error {
	return nil
}

func (k8s *K8sOfflineStore) UpdateMaterialization(id ResourceID, opts MaterializationOptions) (Materialization, error) {
	return k8s.materialization(id, true)
}

func (k8s *K8sOfflineStore) materialization(id ResourceID, isUpdate bool) (Materialization, error) {
	if id.Type != Feature {
		k8s.logger.Errorw("Attempted to create a materialization of a non feature resource", "type", id.Type)
		return nil, fferr.NewInternalError(fmt.Errorf("only features can be materialized"))
	}
	resourceTable, err := k8s.GetResourceTable(id)
	if err != nil {
		k8s.logger.Errorw("Attempted to fetch resource table of non registered resource", "error", err)
		return nil, err
	}
	k8sResourceTable, ok := resourceTable.(*BlobOfflineTable)
	if !ok {
		k8s.logger.Errorw("Could not convert resource table to blob offline table", "id", id)
		return nil, fferr.NewInternalError(fmt.Errorf("could not convert offline table with id %v to k8sResourceTable", id))
	}
	featureResourceKey := ps.ResourceToDirectoryPath(FeatureMaterialization.String(), id.Name, id.Variant)
	destinationPath, err := k8s.store.CreateFilePath(featureResourceKey, false)
	if err != nil {
		k8s.logger.Errorw("Could not create file path", "error", err, "featureResourcePath", featureResourceKey)
		return nil, err
	}
	materializationNewestFile, err := k8s.store.NewestFileOfType(destinationPath, filestore.Parquet)
	if err != nil {
		k8s.logger.Errorw("Could not get newest blob", "location", destinationPath, "error", err)
		return nil, err
	}
	k8s.logger.Debugw("Running Materialization", "id", id, "destinationPath", destinationPath, "materializationNewestFile", materializationNewestFile)
	materializationExists, err := k8s.store.Exists(pl.NewFileLocation(materializationNewestFile))
	if err != nil {
		k8s.logger.Errorw("Could not determine whether materialization exists", err)
		return nil, err
	}
	if !isUpdate && materializationExists {
		k8s.logger.Errorw("Attempted to materialize a materialization that already exists", "id", id)
		return nil, fferr.NewDatasetAlreadyExistsError(id.Name, id.Variant, fmt.Errorf(destinationPath.ToURI()))
	} else if isUpdate && !materializationExists {
		k8s.logger.Errorw("Attempted to update a materialization that does not exist", "id", id)
		return nil, fferr.NewDatasetNotFoundError(id.Name, id.Variant, fmt.Errorf(destinationPath.ToURI()))
	}
	materializationQuery, err := k8s.query.materializationCreate(k8sResourceTable.schema)
	if err != nil {
		return nil, err
	}
	sourcePath, isFilestoreLocation := k8sResourceTable.schema.SourceTable.(*pl.FileStoreLocation)
	if !isFilestoreLocation {
		k8s.logger.Errorw("Source table is not a filestore location", "sourceTable", k8sResourceTable.schema.SourceTable)
		return nil, fferr.NewInternalErrorf("source table is not a filestore location")
	}
	// get source path file type; note, it's possible it doesn't have it
	newestSourcePath, err := k8s.store.NewestFileOfType(sourcePath.Filepath(), sourcePath.Filepath().Ext())
	k8s.logger.Debugw("Retrieved newest source path", "sourcePath", sourcePath, "newestSourcePath", newestSourcePath)
	if err != nil {
		k8s.logger.Errorw("Could not determine newest source file for materialization", "sourcePath", sourcePath, "error", err)
		return nil, err
	}
	k8sArgs := k8s.pandasRunnerArgs(destinationPath.ToURI(), materializationQuery, []string{newestSourcePath.ToURI()}, Materialize)
	k8sArgs = addResourceID(k8sArgs, id)
	if err := k8s.executor.ExecuteScript(k8sArgs, nil); err != nil {
		k8s.logger.Errorw("Job failed to run", "error", err)
		return nil, err
	}

	k8s.logger.Debugw("Successfully created materialization", "id", id)
	return &FileStoreMaterialization{ResourceID{Name: id.Name, Variant: id.Variant, Type: FeatureMaterialization}, k8s.store}, nil
}

func (k8s *K8sOfflineStore) DeleteMaterialization(id MaterializationID) error {
	return fileStoreDeleteMaterialization(id, k8s.store, k8s.logger)
}

func fileStoreDeleteMaterialization(id MaterializationID, store FileStore, logger *zap.SugaredLogger) error {
	s := strings.Split(string(id), "/")
	if len(s) != 3 {
		logger.Errorw("Invalid materialization id", id)
		return fferr.NewDatasetNotFoundError(string(id), "", fmt.Errorf("invalid materialization id"))
	}
	featureResourceKey := ps.ResourceToDirectoryPath(FeatureMaterialization.String(), s[1], s[2])
	materializationPath, err := store.CreateFilePath(featureResourceKey, false)
	if err != nil {
		return err
	}
	exits, err := store.Exists(pl.NewFileLocation(materializationPath))
	if err != nil {
		return err
	}
	if !exits {
		return fferr.NewDatasetNotFoundError(string(id), "", nil)
	}
	return store.DeleteAll(materializationPath)
}

func (k8s *K8sOfflineStore) CreateTrainingSet(def TrainingSetDef) error {
	return k8s.trainingSet(def, false)
}

func (k8s *K8sOfflineStore) UpdateTrainingSet(def TrainingSetDef) error {
	return k8s.trainingSet(def, true)
}

func (k8s *K8sOfflineStore) registeredResourceSchema(id ResourceID) (ResourceSchema, error) {
	k8s.logger.Debugw("Getting resource schema", "id", id)
	table, err := k8s.GetResourceTable(id)
	if err != nil {
		k8s.logger.Errorw("Resource not registered in blob store", "id", id, "error", err)
		return ResourceSchema{}, err
	}
	blobResourceTable, ok := table.(*BlobOfflineTable)
	if !ok {
		k8s.logger.Errorw("could not convert offline table to blobResourceTable", "id", id)
		return ResourceSchema{}, fferr.NewInternalError(fmt.Errorf("could not convert offline table with id %v to blobResourceTable", id))
	}
	k8s.logger.Debugw("Successfully retrieved resource schema", "id", id, "schema", blobResourceTable.schema)
	return blobResourceTable.schema, nil
}

func (k8s *K8sOfflineStore) trainingSet(def TrainingSetDef, isUpdate bool) error {
	if err := def.check(); err != nil {
		k8s.logger.Errorw("Training set definition not valid", def, err)
		return err
	}
	sourcePaths := make([]string, 0)
	featureSchemas := make([]ResourceSchema, 0)
	resourceKey := ps.ResourceToDirectoryPath(def.ID.Type.String(), def.ID.Name, def.ID.Variant)
	destinationPath, err := k8s.store.CreateFilePath(resourceKey, false)
	if err != nil {
		return err
	}
	trainingSetExactPath, err := k8s.store.NewestFileOfType(destinationPath, filestore.Parquet)

	k8s.logger.Debugw("Running Training Set", "id", def.ID, "destinationPath", destinationPath, "trainingSetExactPath", trainingSetExactPath)

	if err != nil {
		return err
	}
	trainingSetExists, err := k8s.store.Exists(pl.NewFileLocation(trainingSetExactPath))
	if err != nil {
		k8s.logger.Errorw("Could not determine whether training set exists", err)
		return err
	}
	if !isUpdate && trainingSetExists {
		k8s.logger.Errorw("Training set already exists", "id", def.ID)
		return fferr.NewDatasetAlreadyExistsError(def.ID.Name, def.ID.Variant, fmt.Errorf(destinationPath.ToURI()))
	} else if isUpdate && !trainingSetExists {
		k8s.logger.Errorw("Training set doesn't exist for update job", def.ID)
		return fferr.NewDatasetNotFoundError(def.ID.Name, def.ID.Variant, fmt.Errorf(destinationPath.ToURI()))
	}
	labelSchema, err := k8s.registeredResourceSchema(def.Label)
	if err != nil {
		k8s.logger.Errorw("Could not get schema of label in store", "id", def.Label, "error", err)
		return err
	}
	labelPath, isFilestoreLocation := labelSchema.SourceTable.(*pl.FileStoreLocation)
	if !isFilestoreLocation {
		k8s.logger.Errorw("Label source table is not a filestore location", "sourceTable", labelSchema.SourceTable)
		return fferr.NewInternalErrorf("label source table is not a filestore location")
	}
	latestLabelFile, err := k8s.store.NewestFileOfType(labelPath.Filepath(), labelPath.Filepath().Ext())
	k8s.logger.Debugw("Latest label file", "labelPath", labelPath, "latestLabelFile", latestLabelFile)
	if err != nil {
		k8s.logger.Errorw("Could not get latest label file", "error", err)
		return err
	}
	sourcePaths = append(sourcePaths, labelPath.Filepath().ToURI())
	for _, feature := range def.Features {
		featureSchema, err := k8s.registeredResourceSchema(feature)
		if err != nil {
			k8s.logger.Errorw("Could not get schema of feature in store", "feature", feature, "error", err)
			return err
		}
		featurePath, isFilestoreLocation := featureSchema.SourceTable.(*pl.FileStoreLocation)
		if !isFilestoreLocation {
			k8s.logger.Errorw("Feature source table is not a filestore location", "sourceTable", featureSchema.SourceTable)
			return fferr.NewInternalErrorf("feature source table is not a filestore location")
		}
		latestFeatureFile, err := k8s.store.NewestFileOfType(featurePath.Filepath(), featurePath.Filepath().Ext())
		k8s.logger.Debugw("Latest feature file", "featurePath", featurePath, "latestFeatureFile", latestFeatureFile)
		if err != nil {
			k8s.logger.Errorw("Could not get latest feature file", "error", err)
			return err
		}
		sourcePaths = append(sourcePaths, featurePath.Filepath().ToURI())
		featureSchemas = append(featureSchemas, featureSchema)
	}
	trainingSetQuery := k8s.query.trainingSetCreate(def, featureSchemas, labelSchema)
	k8s.logger.Debugw("Source List", "SourceFiles", sourcePaths)
	k8s.logger.Debugw("Training Set Query", "list", trainingSetQuery)
	pandasArgs := k8s.pandasRunnerArgs(destinationPath.ToURI(), trainingSetQuery, sourcePaths, CreateTrainingSet)
	pandasArgs = addResourceID(pandasArgs, def.ID)
	k8s.logger.Debugw("Creating training set", "definition", def)

	if err := k8s.executor.ExecuteScript(pandasArgs, nil); err != nil {
		k8s.logger.Errorw("training set job failed to run", "definition", def.ID, "error", err)
		return err
	}
	k8s.logger.Debugw("Successfully created training set:", "definition", def)
	return nil
}

func (k8s *K8sOfflineStore) GetTrainingSet(id ResourceID) (TrainingSetIterator, error) {
	return fileStoreGetTrainingSet(id, k8s.store, k8s.logger)
}

func (k8s *K8sOfflineStore) CreateTrainTestSplit(def TrainTestSplitDef) (func() error, error) {
	return nil, fmt.Errorf("not Implemented")
}

func (k8s *K8sOfflineStore) GetTrainTestSplit(def TrainTestSplitDef) (TrainingSetIterator, TrainingSetIterator, error) {
	return nil, nil, fmt.Errorf("not Implemented")
}

func (k8s *K8sOfflineStore) CheckHealth() (bool, error) {
	return false, fferr.NewInternalError(fmt.Errorf("provider health check not implemented"))
}

func fileStoreGetTrainingSet(id ResourceID, store FileStore, logger *zap.SugaredLogger) (TrainingSetIterator, error) {
	if err := id.check(TrainingSet); err != nil {
		logger.Errorw("Resource is not of type training set", "error", err)
		return nil, fmt.Errorf("resource is not training set: %w", err)
	}
	resourceKey := ps.ResourceToDirectoryPath(id.Type.String(), id.Name, id.Variant)
	filepath, err := store.CreateFilePath(resourceKey, false)
	if err != nil {
		logger.Errorw("Could not create file path", "error", err)
		return nil, err
	}
	trainingSetExists, err := store.Exists(pl.NewFileLocation(filepath))
	if !trainingSetExists {
		return nil, fferr.NewTrainingSetNotFoundError(id.Name, id.Variant, fmt.Errorf(filepath.ToURI()))
	}
	if err != nil {
		return nil, err
	}
	files, err := store.List(filepath, filestore.Parquet)
	if err != nil {
		return nil, err
	}
	groups, err := filestore.NewFilePathGroup(files, filestore.DateTimeDirectoryGrouping)
	if err != nil {
		return nil, err
	}
	newestFiles, err := groups.GetFirst()
	if err != nil {
		return nil, err
	}
	iterator, err := store.Serve(newestFiles)
	if err != nil {
		return nil, err
	}
	return &FileStoreTrainingSet{id: id, store: store, iter: iterator}, nil
}

type FileStoreTrainingSet struct {
	id       ResourceID
	store    FileStore
	iter     Iterator
	Error    error
	features []interface{}
	label    interface{}
}

func (ts *FileStoreTrainingSet) Next() bool {
	row, err := ts.iter.Next()
	if err != nil {
		ts.Error = err
		return false
	}
	if row == nil {
		return false
	}
	featureValues := make([]interface{}, len(ts.iter.FeatureColumns()))
	for i, key := range ts.iter.FeatureColumns() {
		featureValues[i] = row[key]
	}
	ts.features = featureValues
	ts.label = row[ts.iter.LabelColumn()]
	return true
}

func (ts *FileStoreTrainingSet) Features() []interface{} {
	return ts.features
}

func (ts *FileStoreTrainingSet) Label() interface{} {
	return ts.label
}

func (ts *FileStoreTrainingSet) Err() error {
	return ts.Error
}

type FileStoreBatchServing struct {
	store       FileStore
	iter        Iterator
	numFeatures int
	Error       error
	entity      interface{}
	features    []interface{}
}

func (ts *FileStoreBatchServing) Next() bool {
	row, err := ts.iter.Next()
	if err != nil {
		ts.Error = err
		return false
	}
	if row == nil {
		return false
	}
	featureValues := make([]interface{}, ts.numFeatures)
	for i := 1; i <= ts.numFeatures; i++ {
		featureValues[i-1] = row[fmt.Sprintf("feature%d", i)]
	}
	ts.features = featureValues
	ts.entity = row["entity"]
	return true
}

func (ts *FileStoreBatchServing) Features() GenericRecord {
	return ts.features
}

func (ts *FileStoreBatchServing) Entity() interface{} {
	return ts.entity
}

func (ts *FileStoreBatchServing) Err() error {
	return ts.Error
}

func (ts *FileStoreBatchServing) Close() error {
	return ts.store.Close()
}

func (ts *FileStoreBatchServing) Columns() []string {
	return nil
}
