// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//
// Copyright 2024 FeatureForm Inc.
//

package metadata

import (
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"log"
	"reflect"
	"testing"
	"time"

	"github.com/featureform/fferr"
	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	clientv3 "go.etcd.io/etcd/client/v3"

	pb "github.com/featureform/metadata/proto"
	"github.com/featureform/provider/types"
	"google.golang.org/protobuf/encoding/protojson"
	"google.golang.org/protobuf/proto"

	tspb "google.golang.org/protobuf/types/known/timestamppb"
)

type Etcd struct {
	client *clientv3.Client
}

type etcdHelpers interface {
	init() error
	clearDatabase()
}

func (etcd *Etcd) init() {
	client, err := clientv3.New(clientv3.Config{
		Endpoints:   []string{"localhost:2379"},
		DialTimeout: time.Second * 1,
	})
	if err != nil {
		log.Fatalf("Could not connect to etcd client: %v", err)
	}
	etcd.client = client
}

func (etcd *Etcd) clearDatabase() {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
	_, err := etcd.client.Delete(ctx, "", clientv3.WithPrefix())
	cancel()
	if err != nil {
		log.Fatalf("Could not clear database: %v", err)
	}
}

func Test_EtcdResourceLookup_Set(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		Etcd EtcdConfig
	}
	type args struct {
		id  ResourceID
		res Resource
	}

	args1 := args{
		ResourceID{Name: "test", Variant: FEATURE_VARIANT.String(), Type: FEATURE},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "featureVariantResource",
			Type:    types.Float32.ToProto(),
			Created: tspb.Now(),
		}},
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{"Successful Set", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, args1, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			client, err := tt.fields.Etcd.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("Set() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}
			if err := lookup.Set(context.TODO(), tt.args.id, tt.args.res); (err != nil) != tt.wantErr {
				t.Fatalf("Set() error = %v, wantErr %v", err, tt.wantErr)
			}

			newclient, err := clientv3.New(clientv3.Config{
				Endpoints:   []string{"localhost:2379"},
				DialTimeout: time.Second * 1,
			})
			if err != nil {
				fmt.Println(err)
			}

			ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
			resp, err := newclient.Get(ctx, createKey(tt.args.id))
			if err != nil {
				t.Fatalf("Could not get from client: %v", err)
			}
			cancel()
			value := resp.Kvs[0].Value
			resource := featureVariantResource{
				&pb.FeatureVariant{},
			}

			var msg EtcdRowTemp
			if err := json.Unmarshal(value, &msg); err != nil {
				log.Fatalln("Failed To Parse Resource", err)
			}
			if err := protojson.Unmarshal([]byte(msg.Message), resource.Proto()); err != nil {
				log.Fatalln("Failed to parse:", err)
			}
			if !proto.Equal(args1.res.Proto(), resource.Proto()) {
				t.Fatalf("Set() Expected: %v, Received: %v", args1.res.Proto(), resource.Proto())
			}
		})
	}
	connect := Etcd{}
	connect.init()
	t.Cleanup(connect.clearDatabase)
}

func Test_EtcdResourceLookup_Lookup(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	doWant := &featureVariantResource{&pb.FeatureVariant{
		Name:    "featureVariant",
		Type:    types.Float32.ToProto(),
		Created: tspb.Now(),
	}}

	args1 := ResourceID{Name: "featureVariant", Type: FEATURE_VARIANT}

	type fields struct {
		Etcd EtcdConfig
	}
	type args struct {
		id ResourceID
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Resource
		wantErr bool
	}{
		{"Successful Lookup", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, args{args1}, doWant, false},
	}
	for _, tt := range tests {
		newclient, err := clientv3.New(clientv3.Config{
			Endpoints:   []string{"localhost:2379"},
			DialTimeout: time.Second * 1,
		})
		if err != nil {
			t.Fatalf("Could not create new etcd client: %v", err)
		}
		p, _ := ToJsonString(doWant.Proto())
		msg := EtcdRow{
			ResourceType:      args1.Type,
			Message:           p,
			StorageType:       RESOURCE,
			SerializedVersion: 1,
		}

		strmsg, err := json.Marshal(msg)
		if err != nil {
			t.Fatalf("Could not marshal string message: %v", err)
		}
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
		_, err = newclient.Put(ctx, createKey(args1), string(strmsg))
		if err != nil {
			t.Fatalf("Could not put key: %v", err)
		}
		cancel()

		t.Run(tt.name, func(t *testing.T) {
			client, err := tt.fields.Etcd.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("Lookup() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}
			got, err := lookup.Lookup(ctx, tt.args.id)
			if (err != nil) != tt.wantErr {
				t.Fatalf("Lookup() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !(proto.Equal(got.Proto(), tt.want.Proto()) || tt.wantErr) {
				t.Fatalf("Lookup() got = %v, want %v", got.Proto(), tt.want.Proto())
			}
		})
	}
	connect := Etcd{}
	connect.init()
	t.Cleanup(connect.clearDatabase)
}

func Test_EtcdResourceLookup_Has(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		Etcd EtcdConfig
	}
	type args struct {
		id ResourceID
	}
	doWant := &featureVariantResource{&pb.FeatureVariant{
		Name:    "resource1",
		Type:    types.Float32.ToProto(),
		Created: tspb.Now(),
	}}
	args1 := args{
		ResourceID{
			Name: "resource1",
			Type: FEATURE,
		},
	}
	args2 := args{
		ResourceID{
			Name: "resource1",
			Type: FEATURE_VARIANT,
		},
	}

	tests := []struct {
		name    string
		fields  fields
		args    args
		want    bool
		wantErr bool
	}{
		{"Does not have", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, args1, false, false},
		{"Successful Has", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, args2, true, false},
	}
	for _, tt := range tests {
		if tt.want {
			newclient, err := clientv3.New(clientv3.Config{
				Endpoints:   []string{"localhost:2379"},
				DialTimeout: time.Second * 1,
			})
			if err != nil {
				t.Fatalf("Could not connect to client: %v", err)
			}
			p, _ := ToJsonString(doWant.Proto())
			msg := EtcdRow{
				ResourceType:      tt.args.id.Type,
				Message:           p,
				StorageType:       RESOURCE,
				SerializedVersion: 1,
			}

			strmsg, err := json.Marshal(msg)
			if err != nil {
				t.Fatalf("Could not marshal string message: %v", err)
			}
			ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
			_, err = newclient.Put(ctx, createKey(tt.args.id), string(strmsg))
			if err != nil {
				t.Fatalf("Could not put key: %v", err)
			}
			cancel()
		}
		t.Run(tt.name, func(t *testing.T) {
			client, err := tt.fields.Etcd.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("Has() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}
			got, err := lookup.Has(context.TODO(), tt.args.id)
			if (err != nil) != tt.wantErr {
				t.Fatalf("Has() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Fatalf("Has() got = %v, want %v", got, tt.want)
			}
		})
	}
	connect := Etcd{}
	connect.init()
	t.Cleanup(connect.clearDatabase)
}

func Test_EtcdResourceLookup_ListForType(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		Etcd EtcdConfig
	}
	type args struct {
		t ResourceType
	}

	featureResources := []Resource{
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature1",
			Created: tspb.Now(),
		}},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature2",
			Created: tspb.Now(),
		}},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature3",
			Created: tspb.Now(),
		}},
	}

	tests := []struct {
		name    string
		fields  fields
		args    args
		want    []Resource
		wantErr bool
	}{
		{"Successful ListForType", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, args{FEATURE_VARIANT}, featureResources, false},
	}
	newclient, err := clientv3.New(clientv3.Config{
		Endpoints:   []string{"localhost:2379"},
		DialTimeout: time.Second * 1,
	})
	if err != nil {
		t.Fatalf("Could not create new client: %v", err)
	}
	for _, res := range featureResources {
		p, _ := ToJsonString(res.Proto())
		msg := EtcdRow{
			ResourceType:      res.ID().Type,
			Message:           p,
			StorageType:       RESOURCE,
			SerializedVersion: 1,
		}
		strmsg, err := json.Marshal(msg)
		if err != nil {
			t.Fatalf("Could not marshal string message: %v", err)
		}
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
		_, err = newclient.Put(ctx, createKey(res.ID()), string(strmsg))
		if err != nil {
			t.Fatalf("Could not put key: %v", err)
		}
		cancel()
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			client, err := tt.fields.Etcd.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("ListForType() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}
			got, err := lookup.ListForType(context.TODO(), tt.args.t)
			if (err != nil) != tt.wantErr {
				t.Fatalf("ListForType() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			for i, r := range got {
				if !proto.Equal(r.Proto(), tt.want[i].Proto()) {
					t.Fatalf("ListForType() got = %v, want %v", r.Proto(), tt.want[i].Proto())
				}
			}
		})
	}
	connect := Etcd{}
	connect.init()
	t.Cleanup(connect.clearDatabase)
}

func Test_EtcdResourceLookup_List(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	newclient, err := clientv3.New(clientv3.Config{
		Endpoints:   []string{"localhost:2379"},
		DialTimeout: time.Second * 1,
	})
	if err != nil {
		t.Fatalf("Could not create new client: %v", err)
	}
	type fields struct {
		Etcd EtcdConfig
	}

	featureResources := []Resource{
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature1",
			Created: tspb.Now(),
		}},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature2",
			Created: tspb.Now(),
		}},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature3",
			Created: tspb.Now(),
		}},
	}

	tests := []struct {
		name    string
		fields  fields
		want    []Resource
		wantErr bool
	}{
		{"Successful List", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, featureResources, false},
	}
	for _, res := range featureResources {
		p, _ := ToJsonString(res.Proto())
		msg := EtcdRow{
			ResourceType:      res.ID().Type,
			Message:           p,
			StorageType:       RESOURCE,
			SerializedVersion: 1,
		}
		strmsg, err := json.Marshal(msg)
		if err != nil {
			t.Fatalf("Could not marshal string message: %v", err)
		}
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
		_, err = newclient.Put(ctx, createKey(res.ID()), string(strmsg))
		if err != nil {
			t.Fatalf("Could not put key: %v", err)
		}
		cancel()
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			client, err := tt.fields.Etcd.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("List() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}
			got, err := lookup.List(context.TODO())
			if (err != nil) != tt.wantErr {
				t.Fatalf("List() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			for i, r := range got {
				if !proto.Equal(r.Proto(), tt.want[i].Proto()) {
					t.Fatalf("List() got = %v, want %v", r.Proto(), tt.want[i].Proto())
				}
			}
		})
	}
	connect := Etcd{}
	connect.init()
	t.Cleanup(connect.clearDatabase)
}

func Test_EtcdResourceLookup_Submap(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	client, err := clientv3.New(clientv3.Config{
		Endpoints:   []string{"localhost:2379"},
		DialTimeout: time.Second * 1,
	})
	if err != nil {
		t.Fatalf("Could not connect to client: %v", err)
	}
	type fields struct {
		Etcd EtcdConfig
	}
	type args struct {
		ids []ResourceID
	}

	ids := []ResourceID{
		{Name: "feature1", Type: FEATURE_VARIANT},
		{Name: "feature2", Type: FEATURE_VARIANT},
		{Name: "feature3", Type: FEATURE_VARIANT},
	}

	featureResources := []Resource{
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature1",
			Created: tspb.Now(),
		}},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature2",
			Created: tspb.Now(),
		}},
		&featureVariantResource{&pb.FeatureVariant{
			Name:    "feature3",
			Created: tspb.Now(),
		}}}

	resources := LocalResourceLookup{
		ids[0]: featureResources[0],
		ids[1]: featureResources[1],
		ids[2]: featureResources[2],
	}

	tests := []struct {
		name    string
		fields  fields
		args    args
		want    ResourceLookup
		wantErr bool
	}{
		{"Successful Submap", fields{EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}}, args{ids: ids}, resources, false},
	}
	for _, res := range featureResources {
		p, _ := ToJsonString(res.Proto())
		msg := EtcdRow{
			ResourceType:      res.ID().Type,
			Message:           p,
			StorageType:       RESOURCE,
			SerializedVersion: 1,
		}
		strmsg, err := json.Marshal(msg)
		if err != nil {
			t.Fatalf("Could not marshal string message %v", err)
		}
		ctx, cancel := context.WithTimeout(context.Background(), time.Second*1)
		_, err = client.Put(ctx, createKey(res.ID()), string(strmsg))
		cancel()
		if err != nil {
			t.Fatalf("Could not put key: %v", err)
		}
	}
	for _, tt := range tests {

		t.Run(tt.name, func(t *testing.T) {

			client, err := tt.fields.Etcd.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("Submap() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}

			got, err := lookup.Submap(context.TODO(), tt.args.ids)
			if (err != nil) != tt.wantErr {
				t.Fatalf("Submap() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			elem, err := got.List(context.TODO())
			if err != nil {
				t.Fatalf("Could not get list from submap: %v", err)
			}

			for _, res := range elem {
				if err != nil {
					t.Fatalf("%s\n", err)
					t.Fatalf("Submap(): Error with lookup:  %v\n", res.Proto())
				}
				has := false
				for _, expected := range featureResources {
					if proto.Equal(res.Proto(), expected.Proto()) {
						has = true
						break
					}
				}
				if !has {
					t.Fatalf("Submap() item not in expected list:\ngot:  %v\n ", res.Proto())
				}

			}
		})
	}
	connect := Etcd{}
	connect.init()
	t.Cleanup(connect.clearDatabase)
}

func Test_EtcdResourceLookup_findResourceType(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		connection EtcdConfig
	}
	type args struct {
		t ResourceType
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Resource
		wantErr bool
	}{
		{"Test Feature", fields{}, args{FEATURE}, &featureResource{&pb.Feature{}}, false},
		{"Test Feature Variant", fields{}, args{FEATURE_VARIANT}, &featureVariantResource{&pb.FeatureVariant{}}, false},
		{"Test Label", fields{}, args{LABEL}, &labelResource{&pb.Label{}}, false},
		{"Test Label Variant", fields{}, args{LABEL_VARIANT}, &labelVariantResource{&pb.LabelVariant{}}, false},
		{"Test User", fields{}, args{USER}, &userResource{&pb.User{}}, false},
		{"Test Entity", fields{}, args{ENTITY}, &entityResource{&pb.Entity{}}, false},
		{"Test Provider", fields{}, args{PROVIDER}, &providerResource{&pb.Provider{}}, false},
		{"Test Source", fields{}, args{SOURCE}, &sourceResource{&pb.Source{}}, false},
		{"Test Source Variant", fields{}, args{SOURCE_VARIANT}, &sourceVariantResource{&pb.SourceVariant{}}, false},
		{"Test Training Set", fields{}, args{TRAINING_SET}, &trainingSetResource{&pb.TrainingSet{}}, false},
		{"Test Training Set Variant", fields{}, args{TRAINING_SET_VARIANT}, &trainingSetVariantResource{&pb.TrainingSetVariant{}}, false},
		{"Test Model", fields{}, args{MODEL}, &modelResource{&pb.Model{}}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			config := EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}
			client, err := config.InitClient()
			store := EtcdStorage{
				Client: client,
			}
			if err != nil {
				t.Fatalf("createEmptyResource() could not initialize client: %s", err)
			}
			lookup := EtcdResourceLookup{
				Connection: store,
			}
			got, err := lookup.createEmptyResource(tt.args.t)
			if (err != nil) != tt.wantErr {
				t.Fatalf("createEmptyResource() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Fatalf("createEmptyResource() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestEtcdConfig_Put(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		Host string
		Port string
	}
	type args struct {
		key   string
		value string
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{"Test Put Success", fields{"localhost", "2379"}, args{key: "key", value: "value"}, false},
		{"Test Put Error", fields{"localhost", ""}, args{key: "", value: ""}, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			config := EtcdConfig{[]EtcdNode{{Host: tt.fields.Host, Port: tt.fields.Port}}}
			c, err := config.InitClient()
			if err != nil && !tt.wantErr {
				t.Errorf("Put() could not initialize client: %v", err)
			} else if err != nil && tt.wantErr {
				return
			}
			client := EtcdStorage{c}
			if err := client.Put(tt.args.key, tt.args.value); (err != nil) != tt.wantErr {
				t.Fatalf("Put() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestEtcdConfig_InvalidServer(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	config := EtcdConfig{[]EtcdNode{{Host: "localhost", Port: ""}}}
	_, err := config.InitClient()
	if err == nil {
		t.Errorf("InitClient() should have failed with invalid server")
	}
}

func TestEtcdConfig_Get(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	config := EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}
	c, err := config.InitClient()
	if err != nil {
		t.Errorf("InitClient() could not initialize client: %v", err)
	}
	client := EtcdStorage{c}

	if err = client.Put("key", "value"); err != nil {
		t.Errorf("Put() could not put key: %v", err)
	}

	tests := []struct {
		name  string
		key   string
		error error
	}{
		{"Test Get Empty Key", "", &fferr.InvalidArgumentError{}},
		{"Test Get Non Existent Key", "non_existent", &fferr.KeyNotFoundError{}},
		{"Test Get Valid Key", "key", nil},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			_, err := client.Get(tt.key)
			if err != nil && tt.error == nil {
				t.Errorf("Get() should not have failed with error: %v", err)
			} else if err != nil && tt.error != nil {
				expectedErrorType := reflect.TypeOf(tt.error)
				if reflect.TypeOf(err) != expectedErrorType {
					t.Errorf("Expected error of type %v but got type %v", expectedErrorType, reflect.TypeOf(err))
				}
			}
		})
	}
}

func TestEtcdConfig_GetWithPrefix(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		Host string
		Port string
	}
	type args struct {
		key string
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    [][]byte
		wantErr bool
	}{
		// TODO: Add test cases.
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			config := EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}
			client, err := config.InitClient()
			if err != nil && !tt.wantErr {
				t.Errorf("GetWithPrefix() could not initialize client: %v", err)
			} else if err != nil && tt.wantErr {
				return
			}
			store := EtcdStorage{
				Client: client,
			}
			got, err := store.GetWithPrefix(tt.args.key)
			if (err != nil) != tt.wantErr {
				t.Fatalf("GetWithPrefix() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Fatalf("GetWithPrefix() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestEtcdConfig_GetCountWithPrefix(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type fields struct {
		Host string
		Port string
	}
	type args struct {
		key string
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    int64
		wantErr bool
	}{
		// TODO: Add test cases.
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			config := EtcdConfig{[]EtcdNode{{Host: "localhost", Port: "2379"}}}
			client, err := config.InitClient()
			if err != nil {
				t.Fatalf("GetCountWithPrefix() could not initialize client: %s", err)
			}
			store := EtcdStorage{
				Client: client,
			}
			if err != nil && !tt.wantErr {
				t.Errorf("GetCountWithPrefix() could not initialize client: %v", err)
			} else if err != nil && tt.wantErr {
				return
			}
			got, err := store.GetCountWithPrefix(tt.args.key)
			if (err != nil) != tt.wantErr {
				t.Fatalf("GetCountWithPrefix() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Fatalf("GetCountWithPrefix() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestEtcdConfig_ParseResource(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}
	type args struct {
		res     EtcdRow
		resType Resource
	}
	doWant := &featureVariantResource{&pb.FeatureVariant{
		Name:    "featureVariant",
		Type:    types.Float32.ToProto(),
		Created: tspb.Now(),
	}}
	doWantProto, _ := proto.Marshal(doWant.Proto())
	doWantString := base64.StdEncoding.EncodeToString(doWantProto)
	doWantJson, _ := ToJsonString(doWant.Proto())
	tests := []struct {
		name    string
		args    args
		want    Resource
		wantErr bool
	}{
		{"Test Invalid Type", args{EtcdRow{StorageType: JOB, SerializedVersion: 1}, &featureResource{&pb.Feature{}}}, nil, true},
		{"Test Nil Message", args{EtcdRow{StorageType: RESOURCE, Message: "", SerializedVersion: 1}, &featureResource{&pb.Feature{}}}, nil, true},
		{"Test Failed Message", args{EtcdRow{StorageType: RESOURCE, SerializedVersion: 1}, &featureResource{&pb.Feature{}}}, nil, true},
		{"Test Failed Resource", args{EtcdRow{StorageType: RESOURCE, Message: "{name: test}", SerializedVersion: 1}, &featureResource{&pb.Feature{}}}, nil, true},
		{"Test Proto Serialize", args{EtcdRow{StorageType: RESOURCE, Message: doWantString, SerializedVersion: 0}, doWant}, doWant, false},
		{"Test Json Serialize", args{EtcdRow{StorageType: RESOURCE, Message: doWantJson, SerializedVersion: 1}, doWant}, doWant, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := ParseResource(tt.args.res, tt.args.resType)
			if (err != nil) != tt.wantErr {
				t.Fatalf("ParseResource() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Fatalf("ParseResource() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestResource_ParseResource(t *testing.T) {
	if testing.Short() {
		t.Skip()
	}

	recordV0 := `{"ResourceType":8,"StorageType":"Resource","Message":"ChNxdWlja3N0YXJ0LXBvc3RncmVzGhBQT1NUR1JFU19PRkZMSU5FIghwb3N0Z3JlczKNAXs` +
		`iSG9zdCI6ICJxdWlja3N0YXJ0LXBvc3RncmVzIiwgIlBvcnQiOiAiNTQzMiIsICJVc2VybmFtZSI6ICJwb3N0Z3JlcyIsICJQYXNzd29yZCI6ICJwYXNzd29yZCIsICJEYXRh` +
		`YmFzZSI6ICJwb3N0Z3JlcyIsICJTU0xNb2RlIjogImRpc2FibGUifToCCANCIwoMdHJhbnNhY3Rpb25zEhMyMDI0LTA4LTAxdDE2LTA3LTI1YgQKAnYwagA="}`
	recordV1 := `{"ResourceType":8,"StorageType":"Resource","Message":"{\"name\":\"v1-quickstart-postgres\",\"type\":\"POSTGRES_OFFLINE\",` +
		`\"software\":\"postgres\",\"serializedConfig\":\"eyJIb3N0IjogInF1aWNrc3RhcnQtcG9zdGdyZXMiLCAiUG9ydCI6ICI1NDMyIiwgIlVzZXJuYW1lIjog` +
		`InBvc3RncmVzIiwgIlBhc3N3b3JkIjogInBhc3N3b3JkIiwgIkRhdGFiYXNlIjogInBvc3RncmVzIiwgIlNTTE1vZGUiOiAiZGlzYWJsZSJ9\",` +
		`\"status\":{\"status\":\"READY\"},\"sources\":[{\"name\":\"transactions\",\"variant\":\"2024-08-02t17-50-06\"}],\` +
		`"tags\":{\"tag\":[\"v1\"]},\"properties\":{}}","SerializedVersion":1}`

	testCases := []struct {
		name              string
		serializedVersion int
		record            string
		providerProto     *pb.Provider
		expectedResource  Resource
	}{
		{
			name:              "Version 0 (encoded proto)",
			serializedVersion: 0,
			record:            recordV0,
			providerProto:     &pb.Provider{},
			expectedResource: &providerResource{&pb.Provider{
				Name:     "quickstart-postgres",
				Type:     "POSTGRES_OFFLINE",
				Software: "postgres",
				SerializedConfig: []uint8(
					`{"Host": "quickstart-postgres", "Port": "5432", ` +
						`"Username": "postgres", "Password": "password", ` +
						`"Database": "postgres", "SSLMode": "disable"}`,
				),
				Tags:       &pb.Tags{Tag: []string{"v0"}},
				Properties: &pb.Properties{},
				Status:     &pb.ResourceStatus{Status: pb.ResourceStatus_READY},
				Sources:    []*pb.NameVariant{{Name: "transactions", Variant: "2024-08-01t16-07-25"}},
			}}},
		{
			name:              "Version 1 (JSON)",
			serializedVersion: 1,
			record:            recordV1,
			providerProto:     &pb.Provider{},
			expectedResource: &providerResource{&pb.Provider{
				Name:     "v1-quickstart-postgres",
				Type:     "POSTGRES_OFFLINE",
				Software: "postgres",
				SerializedConfig: []uint8(
					`{"Host": "quickstart-postgres", "Port": "5432", ` +
						`"Username": "postgres", "Password": "password", ` +
						`"Database": "postgres", "SSLMode": "disable"}`,
				),
				Tags:       &pb.Tags{Tag: []string{"v1"}},
				Properties: &pb.Properties{},
				Status:     &pb.ResourceStatus{Status: pb.ResourceStatus_READY},
				Sources:    []*pb.NameVariant{{Name: "transactions", Variant: "2024-08-02t17-50-06"}},
			}}},
	}

	for _, currTest := range testCases {
		t.Run(currTest.name, func(t *testing.T) {
			var etcdRow EtcdRow
			if err := json.Unmarshal([]byte(currTest.record), &etcdRow); err != nil {
				t.Fatalf("The test setup unmarshal failed: %v", err)
			}

			result, err := ParseResource(etcdRow, &providerResource{currTest.providerProto})
			if err != nil {
				t.Errorf("Failed ParseResource with the following error: %v", err)
			}

			if diff := cmp.Diff(currTest.expectedResource.Proto(), result.Proto(), cmpopts.IgnoreUnexported(
				pb.Provider{}, pb.ResourceStatus{}, pb.NameVariant{}, pb.Tags{}, pb.Properties{})); diff != "" {
				t.Errorf("Parse resource mismatch! Version: (%d) (-want +got):\n%v", currTest.serializedVersion, diff)
			}
		})
	}
}

func TestCoordinatorScheduleJobSerialize(t *testing.T) {
	resID := ResourceID{Name: "test", Variant: "foo", Type: FEATURE}
	scheduleJob := &CoordinatorScheduleJob{
		Attempts: 0,
		Resource: resID,
		Schedule: "* * * * *",
	}
	serialized, err := scheduleJob.Serialize()
	if err != nil {
		t.Fatalf("Could not serialize schedule job")
	}
	copyScheduleJob := &CoordinatorScheduleJob{}
	if err := copyScheduleJob.Deserialize(serialized); err != nil {
		t.Fatalf("Could not deserialize schedule job")
	}
	if !reflect.DeepEqual(copyScheduleJob, scheduleJob) {
		t.Fatalf("Information changed on serialization and deserialization for schedule job")
	}
}

func TestGetJobKeys(t *testing.T) {
	resID := ResourceID{Name: "test", Variant: "foo", Type: FEATURE}
	expectedJobKey := "JOB__FEATURE__test__foo"
	expectedScheduleJobKey := "SCHEDULEJOB__FEATURE__test__foo"
	jobKey := GetJobKey(resID)
	scheduleJobKey := GetScheduleJobKey(resID)
	if jobKey != expectedJobKey {
		t.Fatalf("Could not generate correct job key")
	}
	if scheduleJobKey != expectedScheduleJobKey {
		t.Fatalf("Could not generate correct schedule job key")
	}
}
