---
layout: page
title: Написание кода промежуточных обработчиков для использования в приложениях Express
menu: guide
lang: ru
description: Learn how to write custom middleware functions for Express.js applications,
  including examples and best practices for enhancing request and response handling.
---

# Написание кода промежуточных обработчиков для использования в приложениях Express

<h2>Обзор</h2>

Функции *промежуточной обработки* (middleware) - это функции, имеющие доступ к [объекту запроса](/{{ page.lang }}/4x/api.html#req)  (`req`), [объекту ответа](/{{ page.lang }}/4x/api.html#res) (`res`) и к следующей функции промежуточной обработки в цикле "запрос-ответ" приложения. Следующая функция промежуточной обработки, как правило, обозначается переменной `next`.

Функции промежуточной обработки могут выполнять следующие задачи:

* Выполнение любого кода.
* Внесение изменений в объекты запросов и ответов.
* Завершение цикла "запрос-ответ".
* Вызов следующего промежуточного обработчика из стека.

Если текущая функция промежуточной обработки не завершает цикл "запрос-ответ", она должна вызвать `next()` для передачи управления следующей функции промежуточной обработки. В противном случае запрос зависнет.

Ниже представлены элементы вызова функции промежуточного обработчика:

<table id="mw-fig">
<tr><td id="mw-fig-imgcell">
<img src="/images/express-mw.png" id="mw-fig-img" />
</td>
<td class="mw-fig-callouts">
<div class="callout" id="callout1">Метод HTTP, к которому применяется данный промежуточный обработчик.</div>

<div class="callout" id="callout2">Путь (маршрут), к которому применяется данный промежуточный обработчик.</div>

<div class="callout" id="callout3">Функция промежуточного обработчика.</div>

<div class="callout" id="callout4">Аргумент обратного вызова для функции промежуточного обработчика, именуемый "next" согласно стандарту.</div>

<div class="callout" id="callout5">Аргумент <a href="../4x/api.html#res">ответа</a> HTTP, именуемый "res" согласно стандарту.</div>

<div class="callout" id="callout6">Аргумент <a href="../4x/api.html#req">запроса</a> HTTP, именуемый "req" согласно стандарту.</div>
</td></tr>
</table>

Далее приводится пример простого приложения Ниже Express "Hello World", для которого будут определены две функции промежуточных обработчиков:

<pre>
<code class="language-javascript" translate="no">
var express = require('express');
var app = express();

app.get('/', function (req, res) {
  res.send('Hello World!');
});

app.listen(3000);
</code>
</pre>

<h2>Разработка</h2>

Ниже приводится простой пример промежуточного обработчика "myLogger". Эта функция печатает слово "LOGGED" при прохождении запроса, адресованного приложению, через приложение. Данная функция промежуточного обработчика присвоена переменной с именем `myLogger`.

<pre>
<code class="language-javascript" translate="no">
var myLogger = function (req, res, next) {
  console.log('LOGGED');
  next();
};
</code>
</pre>

<div class="doc-box doc-notice" markdown="1">
Обратите внимание на вызов `next()` выше.  Вызов этой функции активирует следующую функцию промежуточной обработки в приложении.
Функция `next()` не является частью Node.js или Express API, но представляет собой третий аргумент, передаваемый в функцию промежуточного обработчика.  Функция `next()` могла бы иметь любое имя, но, согласно стандарту, она всегда называется "next". Во избежание путаницы, рекомендуется всегда придерживаться данного стандарта.
</div>

Для того чтобы загрузить функцию промежуточного обработчика вызовите `app.use()` с указанием соответствующей функции.
Например, приведенный ниже код загружает функцию промежуточного обработчика `myLogger` перед маршрутом к корневому расположению (/).

<pre>
<code class="language-javascript" translate="no">
var express = require('express');
var app = express();

var myLogger = function (req, res, next) {
  console.log('LOGGED');
  next();
};

app.use(myLogger);

app.get('/', function (req, res) {
  res.send('Hello World!');
});

app.listen(3000);
</code>
</pre>

Каждый раз при получении запроса приложение выводит на терминал сообщение "LOGGED".

Порядок загрузки промежуточных обработчиков очень важен: функции промежуточных обработчиков, загруженные первыми, выполняются в первую очередь.

Если `myLogger` загружается после маршрута к корневому расположению, запрос никогда не достигает его, и приложением не выводится сообщение "LOGGED", поскольку обработчик маршрута корневого пути завершает цикл "запрос-ответ".

Промежуточный обработчик `myLogger` всего лишь выводит сообщение, затем передает запрос далее, следующему промежуточному обработчику в стеке, путем вызова функции `next()`.

В следующем примере выполняется добавление свойства `requestTime` в объект запроса. Назовем эту функцию промежуточного обработчика "requestTime".

<pre>
<code class="language-javascript" translate="no">
var requestTime = function (req, res, next) {
  req.requestTime = Date.now();
  next();
};
</code>
</pre>

Теперь приложением используется функция промежуточного обработчика `requestTime`. Кроме того, функция обратного вызова маршрута корневого расположения (пути) использует свойство, добавленную функций промежуточного обработчика в `req` (объект запроса).

<pre>
<code class="language-javascript" translate="no">
var express = require('express');
var app = express();

var requestTime = function (req, res, next) {
  req.requestTime = Date.now();
  next();
};

app.use(requestTime);

app.get('/', function (req, res) {
  var responseText = 'Hello World!<br>';
  responseText += '<small>Requested at: ' + req.requestTime + '</small>';
  res.send(responseText);
});

app.listen(3000);
</code>
</pre>

Если запрос адресован корневому каталогу приложения, приложение выводит на экран системное время запроса в браузере.

Благодаря наличию доступа к объекту запроса, объекту ответа, следующей функции промежуточного обработчика в стеке и к API Node.js в целом, возможности, связанные с промежуточными обработчиками, являются бесконечными.

Дополнительная информация о промежуточных обработчиках Express содержится в разделе  [Использование промежуточных обработчиков Express](/{{ page.lang }}/guide/using-middleware.html).
