# 驱动开发

在YiYiYa中，对设备的操作就会涉及到驱动开发，常见的开发有LCD屏幕、串口驱动等。这就涉及到YiYiYa的模块组件，模块可以是一个驱动设备，也可以是一个软件设备，或者特定内核功能。

## 模块

由于YiYiYa使用模块化的的实现，所以大部分要做的事情是定义模块。基本的过程是定义模块，然后注册模块。


1、定义模块

引入内核头kernel.h，定义一个hello_module模块，为moudle_t的类型。

```c
#include "kernel/kernel.h"

int hello_init(void) {
  kprintf("Hello World\n");
  //驱动设备初始化
  serial_init();
  ...
  return 0;
}

void hello_exit(void) { kprintf("hello exit\n"); }

module_t hello_module = {
    .name ="hello",
    .init=hello_init,
    .exit=hello_exit
};
```

2、注册模块

在内核中需要在内核启动的线程中注册，这样YiYiYa操作系统就会识别到模块，然后在内核线程启动的时候，完成了模块的加载，就可以运行在内核中了。

```c
//引入模块
extern module_t hello_module;

void modules_init(void) {
    ...
    //注册模块
    module_regist(&hello_module);
}

```


## 驱动开发相关函数

在使用内核过程中，需要使用到相关的内核功能。下面就介绍基础的函数和使用方式。

### 设备类型device_t

device_t是驱动中重要的概念，是设备特有的结构，里面存设备的信息，设备的类型字符char，block块设备，显示器设备等，也包含定义了读写操作等。当你初始化一个设备后，通过device_add添加到设备管理列表中，在其他地方就可以使用了，通过device_find_name查找。

```c
//设备device_t定义
typedef struct device {
  char* name;
  u32 id;
  u32 type;  // char block vga
  dev_read_fn read;
  dev_write_fn write;
  dev_ioctl_fn ioctl;
  dev_stat_fn stat;
  void* data;  // data
} device_t;

//创建一个设备
device_t*  serial=device_create("serial",DEVICE_SERIAL,DEVICE_TYPE_CHAR);

//添加到设备列表中
device_add(serial);

//根据设备名字查找
device_t* serial= device_find_name("serial");

```




### 内存分配kmalloc、kfree

在内核驱动开发中，初始化一般在module init 方式，通过kmalloc分配一个设备类型的结构，后续可以对这个设备类型进行初始化，并添加到设备列表中。

```c
int serial_init(void) {
    //申请一个设备类型
    device_t* dev = kmalloc(sizeof(device_t),DEFAULT_TYPE);//等同 device_create
    //初始化
    dev->name = "serial";
    dev->read = read;
    dev->write = write;
    dev->id = DEVICE_SERIAL;
    dev->type = DEVICE_TYPE_CHAR;
    //添加到设备列表中
    device_add(dev);
    ...
}
```


### 文件系统相关vfs_create_node、vfs_mount


vfs_create_node创建一个文件节点，这个文件系统节点就可以进行挂载操作，通过vfs_mount挂载到特定目录下。这样，在文件系统访问路径/dev/serial就可以对设备进行读写操作了。


```c
int serial_init(void) {
  device_t* dev = kmalloc(sizeof(device_t),DEFAULT_TYPE);
  dev->name = "serial";
  dev->read = read;
  dev->write = write;
  dev->id = DEVICE_SERIAL;
  dev->type = DEVICE_TYPE_CHAR;
  device_add(dev);

  //挂载设备文件系统到dev目录下，名字为series
  vnode_t* series = vfs_create_node("series", V_FILE);
  vfs_mount(NULL, "/dev", series);
  series->device = dev;
  series->op = &device_operator;

  return 0;
}
```

### 内存操作map_page

当有些设备，需要映射到特定地址，让程序访问，这时候就需要使用到map_page操作，以页为单位。如映射显存地址。如果不映射，会存在操作系统会报错page fault。这样你就访问不到设备了。

```c
for (int i = 0; i < size * vga->framebuffer_count / PAGE_SIZE/8; i++) {
    map_page(addr, addr, PAGE_P | PAGE_USU | PAGE_RWW);
    addr += PAGE_SIZE;
}
```