<?php
/************************************************************************
 * This file is part of EspoCRM.
 *
 * EspoCRM – Open Source CRM application.
 * Copyright (C) 2014-2024 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
 * Website: https://www.espocrm.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
 ************************************************************************/

namespace Espo\Tools\User\Api;

use Espo\Core\Acl;
use Espo\Core\Api\Action;
use Espo\Core\Api\Request;
use Espo\Core\Api\Response;
use Espo\Core\Api\ResponseComposer;
use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Record\EntityProvider;
use Espo\Entities\Team;
use Espo\Entities\User;
use Espo\ORM\EntityManager;

/**
 * @noinspection PhpUnused
 */
class PutTeamUserPosition implements Action
{
    public function __construct(
        private Acl $acl,
        private User $user,
        private EntityProvider $entityProvider,
        private EntityManager $entityManager,
    ) {}

    public function process(Request $request): Response
    {
        if (!$this->user->isAdmin()) {
            throw new Forbidden();
        }

        $teamId = $request->getRouteParam('id') ?? throw new BadRequest();
        $userId = $request->getParsedBody()->id;
        $position = $request->getParsedBody()->position;

        if (!is_string($userId) || !$userId) {
            throw new BadRequest("Bad userId.");
        }

        if (!is_string($position) && $position !== null) {
            throw new BadRequest("Bad position.");
        }

        $user = $this->entityProvider->getByClass(User::class, $userId);
        $team = $this->entityProvider->getByClass(Team::class, $teamId);

        if (!$this->acl->checkEntityEdit($user)) {
            throw new Forbidden();
        }

        if ($position !== null && !in_array($position, $team->getPositionList())) {
            throw new BadRequest("Not allowed position.");
        }

        $this->entityManager
            ->getRelation($team, 'users')
            ->updateColumns($user, ['role' => $position]);

        return ResponseComposer::json(true);
    }
}
