//
// This file is used by the JetBrains ReSharper product to enable better
// and more accurate code analysis. 
//
// For more information ReSharper code annotations, see:
// http://www.google.com/search?q=resharper+%22code+annotations%22+site%3Ajetbrains.com
//

namespace JetBrains.Annotations
{
    using System;

    /// <summary>
    /// Indicates that the marked method is assertion method, i.e. it halts control flow if one of the conditions is satisfied. 
    /// To set the condition, mark one of the parameters with <see cref="AssertionConditionAttribute"/> attribute.
    /// </summary>
    /// <seealso cref="AssertionConditionAttribute"/>
    
    [AttributeUsage(AttributeTargets.Method, AllowMultiple = false, Inherited = true)]
    internal sealed class AssertionMethodAttribute : Attribute
    {
    }

    /// <summary>
    /// Indicates the condition parameter of the assertion method. 
    /// The method itself should be marked by <see cref="AssertionMethodAttribute"/> attribute.
    /// The mandatory argument of the attribute is the assertion type.
    /// </summary>
    /// <seealso cref="AssertionConditionType"/>
    
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = false, Inherited = true)]
    internal sealed class AssertionConditionAttribute : Attribute
    {
        private readonly AssertionConditionType myConditionType;

        /// <summary>
        /// Initializes new instance of AssertionConditionAttribute.
        /// </summary>
        /// <param name="conditionType">Specifies condition type.</param>
        
        public AssertionConditionAttribute(AssertionConditionType conditionType)
        {
            myConditionType = conditionType;
        }

        /// <summary>
        /// Gets condition type.
        /// </summary>
        
        public AssertionConditionType ConditionType
        {
            get { return myConditionType; }
        }
    }

    /// <summary>
    /// Specifies assertion type. If the assertion method argument satisifes the condition, then the execution continues. 
    /// Otherwise, execution is assumed to be halted.
    /// </summary>

    internal enum AssertionConditionType
    {
        /// <summary>
        /// Indicates that the marked parameter should be evaluated to true.
        /// </summary>
    
        IS_TRUE = 0,

        /// <summary>
        /// Indicates that the marked parameter should be evaluated to false.
        /// </summary>
        
        IS_FALSE = 1,

        /// <summary>
        /// Indicates that the marked parameter should be evaluated to null value.
        /// </summary>
        
        IS_NULL = 2,

        /// <summary>
        /// Indicates that the marked parameter should be evaluated to not null value.
        /// </summary>
        
        IS_NOT_NULL = 3,
    }
}