<?php

declare(strict_types=1);

namespace Symplify\EasyCodingStandard\FixerRunner\Application;

use Nette\Utils\FileSystem;
use PhpCsFixer\Differ\DifferInterface;
use PhpCsFixer\Fixer\FixerInterface;
use PhpCsFixer\Tokenizer\Token;
use PhpCsFixer\Tokenizer\Tokens;
use SplFileInfo;
use Symplify\EasyCodingStandard\Console\Style\EasyCodingStandardStyle;
use Symplify\EasyCodingStandard\Contract\Application\FileProcessorInterface;
use Symplify\EasyCodingStandard\Error\FileDiffFactory;
use Symplify\EasyCodingStandard\Exception\ShouldNotHappenException;
use Symplify\EasyCodingStandard\FixerRunner\Parser\FileToTokensParser;
use Symplify\EasyCodingStandard\Parallel\ValueObject\Bridge;
use Symplify\EasyCodingStandard\Skipper\Skipper\Skipper;
use Symplify\EasyCodingStandard\ValueObject\Configuration;
use Symplify\EasyCodingStandard\ValueObject\Error\FileDiff;
use Throwable;

/**
 * @see \Symplify\EasyCodingStandard\Tests\Error\ErrorCollector\FixerFileProcessorTest
 */
final class FixerFileProcessor implements FileProcessorInterface
{
    /**
     * @var FixerInterface[]
     */
    private array $fixers = [];

    private readonly bool $isDebug;

    /**
     * @param FixerInterface[] $fixers
     */
    public function __construct(
        private readonly FileToTokensParser $fileToTokensParser,
        private readonly Skipper $skipper,
        private readonly DifferInterface $differ,
        private readonly EasyCodingStandardStyle $easyCodingStandardStyle,
        private readonly FileDiffFactory $fileDiffFactory,
        array $fixers
    ) {
        $this->fixers = $this->sortFixers($fixers);
        $this->isDebug = $easyCodingStandardStyle->isDebug();
    }

    /**
     * @return FixerInterface[]
     */
    public function getCheckers(): array
    {
        return $this->fixers;
    }

    /**
     * @return array{file_diffs?: FileDiff[]}
     */
    public function processFile(string $filePath, Configuration $configuration): array
    {
        $tokens = $this->fileToTokensParser->parseFromFilePath($filePath);

        $appliedFixers = [];

        foreach ($this->fixers as $fixer) {
            if ($this->processTokensByFixer($filePath, $tokens, $fixer)) {
                $appliedFixers[] = $fixer::class;
            }
        }

        if ($appliedFixers === []) {
            return [];
        }

        $fileContents = FileSystem::read($filePath);
        $diff = $this->differ->diff($fileContents, $tokens->generateCode());

        // some fixer with feature overlap can null each other
        if ($diff === '') {
            return [];
        }

        $fileDiffs = [];

        // file has changed
        $fileDiffs[] = $this->fileDiffFactory->createFromDiffAndAppliedCheckers($filePath, $diff, $appliedFixers);

        $tokenGeneratedCode = $tokens->generateCode();
        if ($configuration->isFixer()) {
            FileSystem::write($filePath, $tokenGeneratedCode, null);
        }

        Tokens::clearCache();

        return [
            Bridge::FILE_DIFFS => $fileDiffs,
        ];
    }

    public function processFileToString(string $filePath): string
    {
        $tokens = $this->fileToTokensParser->parseFromFilePath($filePath);

        $appliedFixers = [];
        foreach ($this->fixers as $fixer) {
            if ($this->processTokensByFixer($filePath, $tokens, $fixer)) {
                $appliedFixers[] = $fixer::class;
            }
        }

        $contents = FileSystem::read($filePath);
        if ($appliedFixers === []) {
            return $contents;
        }

        $diff = $this->differ->diff($contents, $tokens->generateCode());
        // some fixer with feature overlap can null each other
        if ($diff === '') {
            return $contents;
        }

        return $tokens->generateCode();
    }

    /**
     * @param FixerInterface[] $fixers
     * @return FixerInterface[]
     */
    private function sortFixers(array $fixers): array
    {
        usort(
            $fixers,
            static fn (FixerInterface $firstFixer, FixerInterface $secondFixer): int => $secondFixer->getPriority() <=> $firstFixer->getPriority()
        );

        return $fixers;
    }

    /**
     * @param Tokens<Token> $tokens
     * @return bool If fixer applied
     */
    private function processTokensByFixer(string $filePath, Tokens $tokens, FixerInterface $fixer): bool
    {
        if ($this->shouldSkip($filePath, $fixer, $tokens)) {
            return false;
        }

        // show current fixer in --debug / -vvv
        if ($this->isDebug) {
            $this->easyCodingStandardStyle->writeln('     [fixer] ' . $fixer::class);
        }

        try {
            $fixer->fix(new SplFileInfo($filePath), $tokens);
        } catch (Throwable $throwable) {
            throw new ShouldNotHappenException(sprintf(
                'Fixing of "%s" file by "%s" failed: %s in file %s on line %d',
                $filePath,
                $fixer::class,
                $throwable->getMessage(),
                $throwable->getFile(),
                $throwable->getLine()
            ), $throwable->getCode(), $throwable);
        }

        if (! $tokens->isChanged()) {
            return false;
        }

        $tokens->clearEmptyTokens();
        $tokens->clearChanged();

        return true;
    }

    /**
     * @param Tokens<Token> $tokens
     */
    private function shouldSkip(string $filePath, FixerInterface $fixer, Tokens $tokens): bool
    {
        if ($this->skipper->shouldSkipElementAndFilePath($fixer, $filePath)) {
            return true;
        }

        if (! $fixer->supports(new SplFileInfo($filePath))) {
            return true;
        }

        return ! $fixer->isCandidate($tokens);
    }
}
