/* eslint-disable no-console */
const { readFileSync, writeFile } = require('fs')
const { parseSync, stringify } = require('svgson')

const fileHeader = `/* This file is autogenerated, do not edit - it will be overwritten anyway */`

// functions

function parseSVG(skinIconsFile) {
    const icons = readFileSync(skinIconsFile).toString()
    return (icons && parseSync(icons, { camelcase: true }) || {}).children || []
}

function parseSVGSymbols(skinIconsFile) {
    const icons = parseSVG(skinIconsFile);
    return icons.filter(({ name }) => name === 'symbol');
}

function getIconKeys(icons = []) {
    const symbolIds = icons.map(symbol => symbol.attributes.id)
    const skipped = symbolIds.filter(id => !id.startsWith('icon-'));

    if (skipped.length) {
        console.log(`Skipping ${skipped.length} icons: ${skipped.join(', ')}.`)
    }

    return {
        iconKeys: symbolIds
            .filter(symbolId => symbolId.startsWith('icon-'))
            .map(symbolId => camelCased(symbolId.replace(/^icon-/, '')))
            .sort()
    }
}

function saveIconType(keys, typesFile) {
    const typesText = `${fileHeader}\n
export type Icon =
    ${keys.map(key => `'${key}'`).join(' |\n    ')}
`

    writeFile(typesFile, typesText, err => {
        if (err) console.error(err)
        else console.log(`${keys.length} enums written to ${typesFile}.`)
    })
}

function saveIconConstants(keys, constFile) {
    const constText = `${fileHeader}\n
import { Icon } from '../types'

export const icons: Icon[] = [
    ${keys.map(key => `'${key}'`).join(',\n    ')}
]
`

    writeFile(constFile, constText, err => {
        if (err) console.error(err)
        else console.log(`${keys.length} enums written to ${constFile}.`)
    })
}

function saveSvgIcons(collection, ebaySvgFile) {
    const svgText = generateEbaySVG(collection)
    writeFile(ebaySvgFile, svgText, err => {
        if (err) console.error(err)
        else console.log(`${collection.length} SVG symbols written to ${ebaySvgFile}.`)
    })
}

function generateEbaySVG(svgSymbols) {
    const symbolStrings = svgSymbols.map(symbol => stringify(symbol, {
        transformAttr: transformSVGtoReact
    }));

    return `/* eslint-disable */
// @ts-nocheck
${fileHeader}\n
import React from 'react'

export const symbols = [
    ${symbolStrings.join(',\n    ')}
]
`
}

function transformSVGtoReact(key, value, escape) {
    const updatedKeys = {
        class: 'className',
        viewbox: 'viewBox'
    }

    const updatedValue = key === 'style' ? `{{${styleToReact(value)}}}`: `"${escape(value)}"`

    return `${updatedKeys[key] || key}=${updatedValue}`
}

function styleToReact(styleStr) {
    return styleStr
        .split(';')
        .filter(Boolean)
        .map(style => {
            [key, value] = style.split(':')
            return `${camelCased(key.trim())}: '${value.trim()}'`
        })
        .join(', ')
}

function camelCased(str) {
    return str.replace(/-([^-])/g, (s, c) => c.toUpperCase())
}

module.exports = {
    getIconKeys,
    parseSVG,
    parseSVGSymbols,
    saveIconType,
    saveIconConstants,
    saveSvgIcons,
    generateEbaySVG
}
