/*
 outline_presenter.h     MindForger thinking notebook

 Copyright (C) 2016-2024 Martin Dvorak <martin.dvorak@mindforger.com>

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef M8RUI_ORLOJ_PRESENTER_H
#define M8RUI_ORLOJ_PRESENTER_H

#include <iostream>

#include "../../lib/src/debug.h"
#include "../../lib/src/mind/mind.h"

#include <QtWidgets>

#include "orloj_view.h"
#include "organizers_table_presenter.h"
#include "organizer_presenter.h"
#include "kanban_presenter.h"
#include "tags_table_presenter.h"
#include "recent_notes_table_presenter.h"
#include "main_window_presenter.h"
#include "outlines_table_presenter.h"
#include "outlines_map_presenter.h"
#include "notes_table_presenter.h"
#include "outline_view_presenter.h"
#include "outline_header_view_presenter.h"
#include "outline_header_edit_presenter.h"
#include "note_view_presenter.h"
#include "note_edit_presenter.h"
#include "navigator_presenter.h"

namespace m8r {

class OrganizersTablePresenter;
class OrganizerPresenter;
class KanbanPresenter;
class TagCloudPresenter;
class NotePresenter;
class NoteViewPresenter;
class NoteEditPresenter;
class OutlineHeaderViewPresenter;
class OutlineHeaderEditPresenter;
class OutlineViewPresenter;
class OutlinesMapPresenter;
class OrlojView;

enum OrlojPresenterFacets {
    FACET_NONE,                   // 0
    FACET_LIST_OUTLINES,          // 1
    FACET_VIEW_OUTLINE,           // 2
    FACET_VIEW_OUTLINE_HEADER,    // 3
    FACET_EDIT_OUTLINE_HEADER,    // 4
    FACET_VIEW_NOTE,              // 5
    FACET_EDIT_NOTE,              // 6
    FACET_FTS_VIEW_NOTE,          // 7
    FACET_ORGANIZER,              // 8
    FACET_TAG_CLOUD,              // 9
    FACET_RECENT_NOTES,           // 10
    FACET_NAVIGATOR,              // 11
    FACET_LIST_ORGANIZERS,        // 12
    FACET_KANBAN,                 // 13
    FACET_MAP_OUTLINES            // 14
};

// aspect modifies facet
enum OrlojPresenterFacetAspect {
    ASPECT_NONE,
    ASPECT_LIVE_PREVIEW
};

enum OrlojButtonRoles {
    AUTOSAVE_ROLE,
    DISCARD_ROLE,
    EDIT_ROLE,
    SAVE_ROLE,
    INVALID_ROLE = -1
};

/**
 * @brief Orloj presenter handles signals from around MindForger to show desired views.
 */
class OrlojPresenter : public QObject
{
    Q_OBJECT

private:
    MainWindowPresenter* mainPresenter;

    OrlojPresenterFacets activeFacet;
    OrlojPresenterFacetAspect activeAspect;

    OrlojView* view;
    Configuration& config;
    Mind* mind;

    OrganizersTablePresenter* organizersTablePresenter;
    OrganizerPresenter* organizerPresenter;
    KanbanPresenter* kanbanPresenter;
    TagsTablePresenter* tagCloudPresenter;
    OutlinesTablePresenter* outlinesTablePresenter;
    OutlinesMapPresenter* outlinesMapPresenter;
    RecentNotesTablePresenter* recentNotesTablePresenter;
    OutlineViewPresenter* outlineViewPresenter;
    OutlineHeaderViewPresenter* outlineHeaderViewPresenter;
    OutlineHeaderEditPresenter* outlineHeaderEditPresenter;
    NoteViewPresenter* noteViewPresenter;
    NoteEditPresenter* noteEditPresenter;
    NavigatorPresenter* navigatorPresenter;

    bool skipEditNoteCheck;

public:
    explicit OrlojPresenter(MainWindowPresenter* mainPresenter,
        OrlojView* view,
        Mind* mind
    );

    Mind* getMind() { return mind; }

    OrlojView* getView() const { return view; }
    OrganizerPresenter* getOrganizer() const { return organizerPresenter; }
    KanbanPresenter* getKanban() const { return kanbanPresenter; }
    NavigatorPresenter* getNavigator() const { return navigatorPresenter; }
    MainWindowPresenter* getMainPresenter() const { return mainPresenter; }
    OutlinesTablePresenter* getOutlinesTable() const { return outlinesTablePresenter; }
    OutlinesMapPresenter* getOutlinesMap() const { return outlinesMapPresenter; }
    RecentNotesTablePresenter* getRecentNotesTable() const { return recentNotesTablePresenter; }
    OutlineViewPresenter* getOutlineView() const { return outlineViewPresenter; }
    OutlineHeaderViewPresenter* getOutlineHeaderView() const { return outlineHeaderViewPresenter; }
    OutlineHeaderEditPresenter* getOutlineHeaderEdit() const { return outlineHeaderEditPresenter; }
    NoteViewPresenter* getNoteView() const { return noteViewPresenter; }
    NoteEditPresenter* getNoteEdit() const { return noteEditPresenter; }

    bool isFacetActive(const OrlojPresenterFacets facet) const { return activeFacet==facet;}
    bool isFacetActiveOutlineOrNoteView() {
        if(isFacetActive(OrlojPresenterFacets::FACET_VIEW_OUTLINE)
             ||
           isFacetActive(OrlojPresenterFacets::FACET_VIEW_NOTE)
             ||
           isFacetActive(OrlojPresenterFacets::FACET_VIEW_OUTLINE_HEADER))
        {
            return true;
        } else {
            return false;
        }
    }
    bool isFacetActiveOutlineOrNoteEdit() {
        if(isFacetActive(OrlojPresenterFacets::FACET_EDIT_NOTE)
             ||
           isFacetActive(OrlojPresenterFacets::FACET_EDIT_OUTLINE_HEADER))
        {
            return true;
        } else {
            return false;
        }
    }

    OrlojPresenterFacets getFacet() const { return activeFacet; }
    void setFacet(OrlojPresenterFacets facet) {
        onFacetChange(facet);
        activeFacet = facet;
    }

    OrlojPresenterFacetAspect getAspect() const { return activeAspect; }
    void setAspect(OrlojPresenterFacetAspect aspect) { activeAspect = aspect; }
    bool isAspectActive(OrlojPresenterFacetAspect aspect) { return activeAspect == aspect; }

    /**
     * @brief This method is invoked whenever a facet is changed i.e. it allows to perform desired actions.
     */
    void onFacetChange(const OrlojPresenterFacets targetFacet) const;

    void showFacetOrganizerList(const std::vector<Organizer*>& organizers);
    void showFacetEisenhowerMatrix(
            Organizer* organizer,
            const std::vector<Note*>& outlinesAndNotes,
            const std::vector<Outline*>& outlines,
            const std::vector<Note*>& notes
    );
    void showFacetKanban(
            Kanban* kanban,
            const std::vector<Note*>& outlinesAndNotes,
            const std::vector<Outline*>& outlines,
            const std::vector<Note*>& notes
    );
    void showFacetTagCloud();
    void showFacetOutlineList(const std::vector<Outline*>& outlines);
    void showFacetOutlinesMap(Outline* outlinesMap);
    void showFacetRecentNotes(const std::vector<Note*>& notes);
    void showFacetKnowledgeGraphNavigator();
    void showFacetFtsResult(std::vector<Note*>* result);
    void showFacetOutline(Outline* outline);
    void showFacetNoteView();
    void showFacetNoteView(Note* note);
    void showFacetNoteEdit(Note* note);
    void showFacetOutlineHeaderEdit(Outline* outline);

    bool applyFacetHoisting();

    void refreshLiveNotePreview();

    void fromOutlineHeaderEditBackToView(Outline* outline);
    void fromNoteEditBackToView(Note* note);

    bool avoidDataLossOnLinkClick();

public slots:
    void slotShowSelectedOrganizer();
    void slotShowOutlines();
    void slotShowSelectedOutline();
    void slotMapShowSelectedOutline();
    void slotShowOutline(const QItemSelection& selected, const QItemSelection& deselected);
    void slotShowOutlineHeader();
    void slotShowNote(const QItemSelection& selected, const QItemSelection& deselected);
    void slotShowSelectedRecentNote();
    void slotShowRecentNote(const QItemSelection& selected, const QItemSelection& deselected);
    void slotShowSelectedTagRecallDialog();
    void slotShowTagRecallDialog(const QItemSelection& selected, const QItemSelection& deselected);
    void slotShowNavigator();
    void slotShowNoteNavigator(Note* note);
    void slotShowOutlineNavigator(Outline* outline);
    void slotGetLinksForPattern(const QString& pattern);
    void slotRefreshCurrentNotePreview();
    void slotOutlinesTableSorted(int column);
    void slotToggleFullOutlinePreview();
    void slotEditStartLinkCompletion();

signals:
    void signalLinksForPattern(const QString& completionPrefix, std::vector<std::string>* links);
    void signalLinksForHeaderPattern(const QString& completionPrefix, std::vector<std::string>* links);

private:
    bool avoidDataLossOnNoteEdit();
};

} // m8r namespace

#endif // M8RUI_ORLOJ_PRESENTER_H
