import { RuntimeDirEntry } from "../runtimes";

/**
 * Represents a file system that can be interacted with.
 */
export interface FileSystemHost {
  /** Gets if this file system is case sensitive. */
  isCaseSensitive(): boolean;
  /** Asynchronously deletes the specified file or directory. */
  delete(path: string): Promise<void>;
  /** Synchronously deletes the specified file or directory */
  deleteSync(path: string): void;
  /**
   * Reads all the child directories and files.
   * @remarks Implementers should have this return the full file path.
   */
  readDirSync(dirPath: string): RuntimeDirEntry[];
  /** Asynchronously reads a file at the specified path. */
  readFile(filePath: string, encoding?: string): Promise<string>;
  /** Synchronously reads a file at the specified path. */
  readFileSync(filePath: string, encoding?: string): string;
  /** Asynchronously writes a file to the file system. */
  writeFile(filePath: string, fileText: string): Promise<void>;
  /** Synchronously writes a file to the file system. */
  writeFileSync(filePath: string, fileText: string): void;
  /** Asynchronously creates a directory at the specified path. */
  mkdir(dirPath: string): Promise<void>;
  /** Synchronously creates a directory at the specified path. */
  mkdirSync(dirPath: string): void;
  /** Asynchronously moves a file or directory. */
  move(srcPath: string, destPath: string): Promise<void>;
  /** Synchronously moves a file or directory. */
  moveSync(srcPath: string, destPath: string): void;
  /** Asynchronously copies a file or directory. */
  copy(srcPath: string, destPath: string): Promise<void>;
  /** Synchronously copies a file or directory. */
  copySync(srcPath: string, destPath: string): void;
  /** Asynchronously checks if a file exists.
   * @remarks Implementers should throw an `errors.FileNotFoundError` when it does not exist.
   */
  fileExists(filePath: string): Promise<boolean>;
  /** Synchronously checks if a file exists.
   * @remarks Implementers should throw an `errors.FileNotFoundError` when it does not exist.
   */
  fileExistsSync(filePath: string): boolean;
  /** Asynchronously checks if a directory exists. */
  directoryExists(dirPath: string): Promise<boolean>;
  /** Synchronously checks if a directory exists. */
  directoryExistsSync(dirPath: string): boolean;
  /** See https://nodejs.org/api/fs.html#fs_fs_realpathsync_path_options */
  realpathSync(path: string): string;
  /** Gets the current directory of the environment. */
  getCurrentDirectory(): string;
  /** Uses pattern matching to find files or directories. */
  glob(patterns: ReadonlyArray<string>): Promise<string[]>;
  /** Synchronously uses pattern matching to find files or directories. */
  globSync(patterns: ReadonlyArray<string>): string[];
}
