Ddoc

$(DERS_BOLUMU $(IX mixin) Mixins)

$(P
Mixins are for $(I mixing in) generated code into the source code. The mixed-in code may be generated as a template instance or a $(C string).
)

$(P
Code can be inserted into the program as a $(I string import) as well.
)

$(H5 $(IX template mixin) Template mixins)

$(P
We have seen in the $(LINK2 templates.html, Templates) and $(LINK2 templates_more.html, More Templates) chapters that templates define code as a pattern, for the compiler to generate actual instances from that pattern. Templates can generate functions, structs, unions, classes, interfaces, and any other legal D code.
)

$(P
Template mixins insert instantiations of templates into the code by the $(C mixin) keyword:
)

---
    mixin $(I a_template)!($(I template_parameters))
---

$(P
As we will see in the example below, the $(C mixin) keyword is used in the definitions of template mixins as well.
)

$(P
The instantiation of the template for the specific set of template parameters is inserted into the source code right where the $(C mixin) keyword appears.
)

$(P
For example, let's have a template that defines both an array of edges and a pair of functions that operate on those edges:
)

---
$(CODE_NAME EdgeArrayFeature)$(HILITE mixin) template EdgeArrayFeature(T, size_t count) {
    T[count] edges;

    void setEdge(size_t index, T edge) {
        edges[index] = edge;
    }

    void printEdges() {
        writeln("The edges:");

        foreach (i, edge; edges) {
            writef("%s:%s ", i, edge);
        }

        writeln();
    }
}
---

$(P
That template leaves the type and number of array elements flexible. The instantiation of that template for $(C int) and $(C 2) would be mixed in by the following syntax:
)

---
    $(HILITE mixin) EdgeArrayFeature!(int, 2);
---

$(P
For example, the $(C mixin) above can insert the two-element $(C int) array and the two functions that are generated by the template right inside a $(C struct) definition:
)

---
$(CODE_NAME Line)$(CODE_XREF EdgeArrayFeature)struct Line {
     mixin EdgeArrayFeature!(int, 2);
}
---

$(P
As a result, $(C Line) ends up defining a member array and two member functions:
)

---
$(CODE_XREF Line)import std.stdio;

void main() {
    auto line = Line();
    line.setEdge(0, 100);
    line.setEdge(1, 200);
    line.printEdges();
}
---

$(P
The output:
)

$(SHELL
The edges:
0:100 1:200 
)

$(P
Another instantiation of the same template can be used e.g. inside a function:
)

---
$(CODE_XREF EdgeArrayFeature)struct Point {
    int x;
    int y;
}

void main() {
    $(HILITE mixin) EdgeArrayFeature!($(HILITE Point), 5);

    setEdge(3, Point(3, 3));
    printEdges();
}
---

$(P
That $(C mixin) inserts an array and two local functions inside $(C main()). The output:
)

$(SHELL
The edges:
0:Point(0, 0) 1:Point(0, 0) 2:Point(0, 0) 3:Point(3, 3) 4:Point(0, 0) 
)

$(H6 $(IX local import) $(IX import, local) Template mixins must use local imports)

$(P
Mixing in template instantiations $(I as is) can cause problems about the modules that the template itself is making use of: Those modules may not be available at the $(C mixin) site.
)

$(P
Let's consider the following module named $(C a). Naturally, it would have to import the $(C std.string) module that it is making use of:
)

---
module a;

$(HILITE import std.string;)    $(CODE_NOTE wrong place)

mixin template A(T) {
    string a() {
        T[] array;
        // ...
        return format("%(%s, %)", array);
    }
}
---

$(P
However, if $(C std.string) is not imported at the actual $(C mixin) site, then the compiler would not be able to find the definition of $(C format()) at that point. Let's consider the following program that imports $(C a) and tries to mix in $(C A!int) from that module:
)

---
import a;

void main() {
    mixin A!int;    $(DERLEME_HATASI)
}
---

$(SHELL
Error: $(HILITE undefined identifier format)
Error: mixin deneme.main.A!int error instantiating
)

$(P
For that reason, the modules that template mixins use must be imported in local scopes:
)

---
module a;

mixin template A(T) {
    string a() {
        $(HILITE import std.string;)    $(CODE_NOTE right place)

        T[] array;
        // ...
        return format("%(%s, %)", array);
    }
}
---

$(P
As long as it is inside the template definition, the $(C import) directive above can be outside of the $(C a()) function as well.
)

$(H6 $(IX this, template parameter) Identifying the type that is mixing in)

$(P
Sometimes a mixin may need to identify the actual type that is mixing it in. That information is available through $(I $(C this) template parameters) as we have seen in $(LINK2 templates_more.html, the More Templates chapter):
)

---
mixin template MyMixin(T) {
    void foo$(HILITE (this MixingType))() {
        import std.stdio;
        writefln("The actual type that is mixing in: %s",
                 $(HILITE MixingType).stringof);
    }
}

struct MyStruct {
    mixin MyMixin!(int);
}

void main() {
    auto a = MyStruct();
    a.foo();
}
---

$(P
The output of the program shows that the actual type is available inside the template as $(C MyStruct):
)

$(SHELL
The actual type that is mixing in: MyStruct
)

$(H5 $(IX string mixin) String mixins)

$(P
Another powerful feature of D is being able to insert code as $(C string) as long as that string is known at compile time. The syntax of string mixins requires the use of parentheses:
)

---
    mixin $(HILITE $(PARANTEZ_AC))$(I compile_time_generated_string)$(HILITE $(PARANTEZ_KAPA))
---

$(P
For example, the $(I hello world) program can be written with a $(C mixin) as well:
)

---
import std.stdio;

void main() {
    mixin (`writeln("Hello, World!");`);
}
---

$(P
The string gets inserted as code and the program produces the following output:
)

$(SHELL
Hello, World!
)

$(P
We can go further and insert all of the program as a string mixin:
)

---
mixin (
`import std.stdio; void main() { writeln("Hello, World!"); }`
);
---

$(P
Obviously, there is no need for mixins in these examples, as the strings could have been written as code as well.
)

$(P
The power of string mixins comes from the fact that the code can be generated at compile time. The following example takes advantage of CTFE to generate statements at compile time:
)

---
import std.stdio;

string printStatement(string message) {
    return `writeln("` ~ message ~ `");`;
}

void main() {
    mixin (printStatement("Hello, World!"));
    mixin (printStatement("Hi, World!"));
}
---

$(P
The output:
)

$(SHELL
Hello, World!
Hi, World!
)

$(P
Note that the $(STRING "writeln") expressions are not executed inside $(C printStatement()). Rather, $(C printStatement()) generates code that includes $(C writeln()) expressions that are executed inside $(C main()). The generated code is the equivalent of the following:
)

---
import std.stdio;

void main() {
    writeln("Hello, World!");
    writeln("Hi, World!");
}
---

$(H6 Multiple $(C mixin) arguments)

$(P
As long as they are all known at compile time, $(C mixin) can take multiple arguments and automatically concatenates their string representations:
)

---
  mixin ("const a = ", int.sizeof, ";");
---

$(P
This can be more convenient compared to using e.g. a $(C format) expression:
)

---
  mixin (format!"const a = %s;"(int.sizeof));  // Same as above
---

$(H6 Debugging string mixins)

$(P
$(IX -mixin, compiler switch) Because generated code is not readily visible as a whole in source code, it can be difficult to identify causes of compilation errors with $(C mixin) expressions. To help with debugging string mixins, there is the $(C dmd) compiler switch $(C -mixin), which writes all mixed-in code to a specified file.
)

$(P
Let's consider the following program that has a syntax error in code that is being mixed in. It is not obvious from the compiler error that the syntax error is the missing semicolon at the end of the definition of the $(C struct) member:
)

---
string makeStruct(string name, string member) {
  import std.format;
  return format!"struct %s {\n  int %s\n}"(name, member);
}

mixin (makeStruct("S", "m"));    $(DERLEME_HATASI)

void main() {
}
---

$(P
When compiled with the $(C -mixin) switch, the compilation error would point at a line inside the specified file ($(C mixed_in_code) in the example below):
)

$(SHELL
$(SHELL_OBSERVED $) dmd $(HILITE -mixin=mixed_in_code) deneme.d
mixed_in_code($(HILITE 154)): Error: semicolon expected, not `}`
)

$(P
Along with all other code that are mixed-in by the standard library, there would be the following code at the specified line inside file $(C mixed_in_code):
)

$(SHELL
[...]
// expansion at deneme.d(6)
struct S {
  int m
}        $(SHELL_NOTE Line 154)
)


$(P
Another option for debugging string mixins is $(LINK2 pragma.html, $(C pragma(msg))), which would print the generated code during compilation. This option is less practical because it requires replacing the $(C mixin) keyword with $(C pragma(msg)) temporarily for debugging:
)

---
pragma(msg, makeStruct("S", "m"));
---

$(H5 $(IX name space, mixin) Mixin name spaces)

$(P
It is possible to avoid and resolve name ambiguities in template mixins.
)

$(P
For example, there are two $(C i) variables defined inside $(C main()) in the following program: one is defined explicitly in $(C main) and the other is mixed in. When a mixed-in name is the same as a name that is in the surrounding scope, then the name that is in the surrounding scope gets used:
)

---
import std.stdio;

template Templ() {
    $(HILITE int i;)

    void print() {
        writeln(i);  // Always the 'i' that is defined in Templ
    }
}

void main() {
    $(HILITE int i;)
    mixin Templ;

    i = 42;      // Sets the 'i' that is defined explicitly in main
    writeln(i);  // Prints the 'i' that is defined explicitly in main
    print();     // Prints the 'i' that is mixed in
}
---

$(P
As implied in the comments above, template mixins define a name space for their contents and the names that appear in the template code are first looked up in that name space. We can see this in the behavior of $(C print()):
)

$(SHELL
42
0     $(SHELL_NOTE printed by print())
)

$(P
The compiler cannot resolve name conflicts if the same name is defined by more than one template mixin. Let's see this in a short program that mixes in the same template instance twice:
)

---
template Templ() {
    int i;
}

void main() {
    mixin Templ;
    mixin Templ;

    i = 42;        $(DERLEME_HATASI)
}
---

$(SHELL
Error: deneme.main.Templ!().i at ... $(HILITE conflicts with)
deneme.main.Templ!().i at ...
)

$(P
To prevent this, it is possible to assign name space identifiers for template mixins and refer to contained names by those identifiers:
)

---
    mixin Templ $(HILITE A);    // Defines A.i
    mixin Templ $(HILITE B);    // Defines B.i

    $(HILITE A.)i = 42;         // ← not ambiguous anymore
---

$(P
String mixins do not have these name space features. However, it is trivial to use a string as a template mixin simply by passing it through a simple wrapper template.
)

$(P
Let's first see a similar name conflict with string mixins:
)

---
void main() {
    mixin ("int i;");
    mixin ("int i;");    $(DERLEME_HATASI)

    i = 42;
}
---

$(SHELL
Error: declaration deneme.main.i is $(HILITE already defined)
)

$(P
One way of resolving this issue is to pass the $(C string) through the following trivial template that effectively converts a string mixin to a template mixin:
)

---
template Templatize(string str) {
    mixin (str);
}

void main() {
    mixin Templatize!("int i;") A;    // Defines A.i
    mixin Templatize!("int i;") B;    // Defines B.i

    A.i = 42;                         // ← not ambiguous anymore
}
---

$(H5 $(IX operator overloading, mixin) String mixins in operator overloading)

$(P
We have seen in $(LINK2 operator_overloading.html, the Operator Overloading chapter) how $(C mixin) expressions helped with the definitions of some of the operators.
)

$(P
In fact, the reason why most operator member functions are defined as templates is to make the operators available as $(C string) values so that they can be used for code generation. We have seen examples of this both in that chapter and its exercise solutions.
)

$(H5 $(IX destructor, mixin) Mixed in destructors)

$(P
It is possible to mix in multiple destructors to a user defined type. Those destructors are called in the reverse order of the $(C mixin) statements that added them. This feature allows mixing in different resources to a type, each introducing its own cleanup code.
)

---
import std.stdio;

mixin template Foo() {
    ~this() {
        writeln("Destructor mixed-in by Foo");
    }
}

mixin template Bar() {
    ~this() {
        writeln("Destructor mixed-in by Bar");
    }
}

struct S {
    ~this() {
        writeln("Actual destructor");
    }
    mixin Foo;
    mixin Bar;
}

void main() {
    auto s = S();
}
---

$(SHELL
Destructor mixed-in by Bar
Destructor mixed-in by Foo
Actual destructor
)

$(P
Due to a bug as of this writing, the same behavior does not apply to other special functions like constructors. Additionally, a destructor mixed in by a string mixin does conflict with the existing destructor of the type.
)

$(H5 $(IX text file import) $(IX file import) $(IX import, file) $(IX string import) $(IX import, string) Importing text files)

$(P
It is possible to insert contents of text files into code at compile time. The contents are treated as $(C string) literals and can be used anywhere strings can be used. For example, they can be mixed in as code.
)

$(P
For example, let's assume there are two text files on the file system named $(C file_one) and $(C file_two) having the following contents.
)

$(UL
$(LI
$(C file_one):

$(SHELL
Hello
)
)

$(LI
$(C file_two):

$(SHELL
s ~= ", World!";
import std.stdio;
writeln(s);
)
)

)

$(P
The two $(C import) directives in the following program would correspond to the contents of those files converted to $(C string) literals at compile time:
)

---
void main() {
    string s = import ("file_one");
    mixin (import ("file_two"));
}
---

$(P
$(IX -J, compiler switch) Text file imports (a.k.a. string imports) require the $(C -J) compiler switch which tells the compiler where to find the text files. For example, if the two files are in the $(I current directory) (specified with $(C .) in Linux environments), the program can be compiled with the following command:
)

$(SHELL
$ dmd $(HILITE -J.) deneme.d
)

$(P
The output:
)

$(SHELL
Hello, World!
)

$(P
Considering the file contents as $(C string) literals, the program is the equivalent of the following one:
)

---
void main() {
    string s = `Hello`;         $(CODE_NOTE Content of file_one as string)
    mixin (`s ~= ", World!";
import std.stdio;
writeln(s);`);                  $(CODE_NOTE Content of file_two as string)
}
---

$(P
Further, considering the mixed-in string as well, the program is the equivalent of the following one:
)

---
void main() {
    string s = `Hello`;
    s ~= ", World!";
    import std.stdio;
    writeln(s);
}
---

$(H5 Example)

$(P
($(I $(B Note:) Specifying predicates as strings was used more commonly before the lambda syntax was added to D. Although string predicates as in this example are still used in Phobos, the $(C =>) lambda syntax may be more suitable in most cases.))
)

$(P
Let's consider the following function template that takes an array of numbers and returns another array that consists of the elements that satisfy a specific condition:
)

---
int[] filter($(HILITE string predicate))(int[] numbers) {
    int[] result;

    foreach (number; numbers) {
        if ($(HILITE mixin (predicate))) {
            result ~= number;
        }
    }

    return result;
}
---

$(P
That function template takes the filtering condition as its template parameter and inserts that condition directly into an $(C if) statement as is.
)

$(P
For that condition to choose numbers that are e.g. less than 7, the $(C if) condition should look like the following code:
)

---
        if (number < 7) {
---

$(P
The users of $(C filter()) template can provide the condition as a $(C string):
)

---
    int[] numbers = [ 1, 8, 6, -2, 10 ];
    int[] chosen = filter!$(HILITE "number < 7")(numbers);
---

$(P
Importantly, the name used in the template parameter must match the name of the variable used in the implementation of $(C filter()). So, the template must document what that name should be and the users must use that name.
)

$(P
Phobos uses names consisting of single letters like a, b, n, etc.
)

macros:
        TITLE=Mixins

        DESCRIPTION=Template mixins and string mixins.

        KEYWORDS=d programming language tutorial book mixin

SOZLER=
$(katma)
$(sablon)
