using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Collections.ObjectModel;
using System.Linq;

namespace Discord
{
    /// <summary>
    ///     A Unicode emoji.
    /// </summary>
    public class Emoji : IEmote
    {
        /// <inheritdoc />
        public string Name { get; }

        /// <summary>
        ///     Gets the Unicode representation of this emoji.
        /// </summary>
        /// <returns>
        ///     A string that resolves to <see cref="Emoji.Name"/>.
        /// </returns>
        public override string ToString() => Name;

        /// <summary>
        ///     Initializes a new <see cref="Emoji"/> class with the provided Unicode.
        /// </summary>
        /// <param name="unicode">The pure UTF-8 encoding of an emoji.</param>
        public Emoji(string unicode)
        {
            Name = unicode;
        }

        /// <summary>
        ///     Determines whether the specified emoji is equal to the current one.
        /// </summary>
        /// <param name="other">The object to compare with the current object.</param>
        public override bool Equals(object other)
        {
            if (other == null)
                return false;

            if (other == this)
                return true;

            return other is Emoji otherEmoji && string.Equals(Name, otherEmoji.Name);
        }

        /// <summary> Tries to parse an <see cref="Emoji"/> from its raw format. </summary>
        /// <param name="text">The raw encoding of an emoji. For example: <code>:heart: or ❤</code></param>
        /// <param name="result">An emoji.</param>
        public static bool TryParse(string text, out Emoji result)
        {
            result = null;
            if (string.IsNullOrWhiteSpace(text))
                return false;

            if (NamesAndUnicodes.ContainsKey(text))
                result = new Emoji(NamesAndUnicodes[text]);

            if (Unicodes.Contains(text))
                result = new Emoji(text);

            return result != null;
        }

        /// <summary> Parse an <see cref="Emoji"/> from its raw format.</summary>
        /// <param name="emojiStr">The raw encoding of an emoji. For example: <c>:heart: or ❤</c></param>
        /// <exception cref="FormatException">String is not emoji or unicode!</exception>
        public static Emoji Parse(string emojiStr)
        {
            if (!TryParse(emojiStr, out var emoji))
                throw new FormatException("String is not emoji name or unicode!");

            return emoji;
        }

        /// <inheritdoc />
        public override int GetHashCode() => Name.GetHashCode();

        private static IReadOnlyDictionary<string, string> NamesAndUnicodes { get; } = new Dictionary<string, string>
        {
            [",:("] = "\uD83D\uDE13",
            [",:)"] = "\uD83D\uDE05",
            [",:-("] = "\uD83D\uDE13",
            [",:-)"] = "\uD83D\uDE05",
            [",=("] = "\uD83D\uDE13",
            [",=)"] = "\uD83D\uDE05",
            [",=-("] = "\uD83D\uDE13",
            [",=-)"] = "\uD83D\uDE05",
            ["0:)"] = "\uD83D\uDE07",
            ["0:-)"] = "\uD83D\uDE07",
            ["0=)"] = "\uD83D\uDE07",
            ["0=-)"] = "\uD83D\uDE07",
            ["8-)"] = "\uD83D\uDE0E",
            [":$"] = "\uD83D\uDE12",
            [":'("] = "\uD83D\uDE22",
            [":')"] = "\uD83D\uDE02",
            [":'-("] = "\uD83D\uDE22",
            [":'-)"] = "\uD83D\uDE02",
            [":'-D"] = "\uD83D\uDE02",
            [":'D"] = "\uD83D\uDE02",
            [":("] = "\uD83D\uDE26",
            [":)"] = "\uD83D\uDE42",
            [":*"] = "\uD83D\uDE17",
            [":+1:"] = "\uD83D\uDC4D",
            [":+1::skin-tone-1:"] = "\uD83D\uDC4D\uD83C\uDFFB",
            [":+1::skin-tone-2:"] = "\uD83D\uDC4D\uD83C\uDFFC",
            [":+1::skin-tone-3:"] = "\uD83D\uDC4D\uD83C\uDFFD",
            [":+1::skin-tone-4:"] = "\uD83D\uDC4D\uD83C\uDFFE",
            [":+1::skin-tone-5:"] = "\uD83D\uDC4D\uD83C\uDFFF",
            [":+1_tone1:"] = "\uD83D\uDC4D\uD83C\uDFFB",
            [":+1_tone2:"] = "\uD83D\uDC4D\uD83C\uDFFC",
            [":+1_tone3:"] = "\uD83D\uDC4D\uD83C\uDFFD",
            [":+1_tone4:"] = "\uD83D\uDC4D\uD83C\uDFFE",
            [":+1_tone5:"] = "\uD83D\uDC4D\uD83C\uDFFF",
            [":,'("] = "\uD83D\uDE2D",
            [":,'-("] = "\uD83D\uDE2D",
            [":,("] = "\uD83D\uDE22",
            [":,)"] = "\uD83D\uDE02",
            [":,-("] = "\uD83D\uDE22",
            [":,-)"] = "\uD83D\uDE02",
            [":,-D"] = "\uD83D\uDE02",
            [":,D"] = "\uD83D\uDE02",
            [":-$"] = "\uD83D\uDE12",
            [":-("] = "\uD83D\uDE26",
            [":-)"] = "\uD83D\uDE42",
            [":-*"] = "\uD83D\uDE17",
            [":-/"] = "\uD83D\uDE15",
            [":-1:"] = "\uD83D\uDC4E",
            [":-1::skin-tone-1:"] = "\uD83D\uDC4E\uD83C\uDFFB",
            [":-1::skin-tone-2:"] = "\uD83D\uDC4E\uD83C\uDFFC",
            [":-1::skin-tone-3:"] = "\uD83D\uDC4E\uD83C\uDFFD",
            [":-1::skin-tone-4:"] = "\uD83D\uDC4E\uD83C\uDFFE",
            [":-1::skin-tone-5:"] = "\uD83D\uDC4E\uD83C\uDFFF",
            [":-@"] = "\uD83D\uDE21",
            [":-D"] = "\uD83D\uDE04",
            [":-O"] = "\uD83D\uDE2E",
            [":-P"] = "\uD83D\uDE1B",
            [":-S"] = "\uD83D\uDE12",
            [":-Z"] = "\uD83D\uDE12",
            [":-\")"] = "\uD83D\uDE0A",
            [":-\\"] = "\uD83D\uDE15",
            [":-o"] = "\uD83D\uDE2E",
            [":-|"] = "\uD83D\uDE10",
            [":100:"] = "\uD83D\uDCAF",
            [":1234:"] = "\uD83D\uDD22",
            [":8ball:"] = "\uD83C\uDFB1",
            [":@"] = "\uD83D\uDE21",
            [":D"] = "\uD83D\uDE04",
            [":O"] = "\uD83D\uDE2E",
            [":P"] = "\uD83D\uDE1B",
            [":\")"] = "\uD83D\uDE0A",
            [":_1_tone1:"] = "\uD83D\uDC4E\uD83C\uDFFB",
            [":_1_tone2:"] = "\uD83D\uDC4E\uD83C\uDFFC",
            [":_1_tone3:"] = "\uD83D\uDC4E\uD83C\uDFFD",
            [":_1_tone4:"] = "\uD83D\uDC4E\uD83C\uDFFE",
            [":_1_tone5:"] = "\uD83D\uDC4E\uD83C\uDFFF",
            [":a:"] = "\uD83C\uDD70️",
            [":ab:"] = "\uD83C\uDD8E",
            [":abacus:"] = "\uD83E\uDDEE",
            [":abc:"] = "\uD83D\uDD24",
            [":abcd:"] = "\uD83D\uDD21",
            [":accept:"] = "\uD83C\uDE51",
            [":accordion:"] = "\uD83E\uDE97",
            [":adhesive_bandage:"] = "\uD83E\uDE79",
            [":admission_tickets:"] = "\uD83C\uDF9F️",
            [":adult:"] = "\uD83E\uDDD1",
            [":adult::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB",
            [":adult::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC",
            [":adult::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFD",
            [":adult::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE",
            [":adult::skin-tone-5:"] = "\uD83E\uDDD1\uD83C\uDFFF",
            [":adult_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF",
            [":adult_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB",
            [":adult_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE",
            [":adult_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC",
            [":adult_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD",
            [":adult_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB",
            [":adult_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC",
            [":adult_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFD",
            [":adult_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE",
            [":adult_tone5:"] = "\uD83E\uDDD1\uD83C\uDFFF",
            [":aerial_tramway:"] = "\uD83D\uDEA1",
            [":airplane:"] = "✈️",
            [":airplane_arriving:"] = "\uD83D\uDEEC",
            [":airplane_departure:"] = "\uD83D\uDEEB",
            [":airplane_small:"] = "\uD83D\uDEE9️",
            [":alarm_clock:"] = "⏰",
            [":alembic:"] = "⚗️",
            [":alien:"] = "\uD83D\uDC7D",
            [":ambulance:"] = "\uD83D\uDE91",
            [":amphora:"] = "\uD83C\uDFFA",
            [":anatomical_heart:"] = "\uD83E\uDEC0",
            [":anchor:"] = "⚓",
            [":angel:"] = "\uD83D\uDC7C",
            [":angel::skin-tone-1:"] = "\uD83D\uDC7C\uD83C\uDFFB",
            [":angel::skin-tone-2:"] = "\uD83D\uDC7C\uD83C\uDFFC",
            [":angel::skin-tone-3:"] = "\uD83D\uDC7C\uD83C\uDFFD",
            [":angel::skin-tone-4:"] = "\uD83D\uDC7C\uD83C\uDFFE",
            [":angel::skin-tone-5:"] = "\uD83D\uDC7C\uD83C\uDFFF",
            [":angel_tone1:"] = "\uD83D\uDC7C\uD83C\uDFFB",
            [":angel_tone2:"] = "\uD83D\uDC7C\uD83C\uDFFC",
            [":angel_tone3:"] = "\uD83D\uDC7C\uD83C\uDFFD",
            [":angel_tone4:"] = "\uD83D\uDC7C\uD83C\uDFFE",
            [":angel_tone5:"] = "\uD83D\uDC7C\uD83C\uDFFF",
            [":anger:"] = "\uD83D\uDCA2",
            [":anger_right:"] = "\uD83D\uDDEF️",
            [":angry:"] = "\uD83D\uDE20",
            [":anguished:"] = "\uD83D\uDE27",
            [":ant:"] = "\uD83D\uDC1C",
            [":apple:"] = "\uD83C\uDF4E",
            [":aquarius:"] = "♒",
            [":archery:"] = "\uD83C\uDFF9",
            [":aries:"] = "♈",
            [":arrow_backward:"] = "◀️",
            [":arrow_double_down:"] = "⏬",
            [":arrow_double_up:"] = "⏫",
            [":arrow_down:"] = "⬇️",
            [":arrow_down_small:"] = "\uD83D\uDD3D",
            [":arrow_forward:"] = "▶️",
            [":arrow_heading_down:"] = "⤵️",
            [":arrow_heading_up:"] = "⤴️",
            [":arrow_left:"] = "⬅️",
            [":arrow_lower_left:"] = "↙️",
            [":arrow_lower_right:"] = "↘️",
            [":arrow_right:"] = "➡️",
            [":arrow_right_hook:"] = "↪️",
            [":arrow_up:"] = "⬆️",
            [":arrow_up_down:"] = "↕️",
            [":arrow_up_small:"] = "\uD83D\uDD3C",
            [":arrow_upper_left:"] = "↖️",
            [":arrow_upper_right:"] = "↗️",
            [":arrows_clockwise:"] = "\uD83D\uDD03",
            [":arrows_counterclockwise:"] = "\uD83D\uDD04",
            [":art:"] = "\uD83C\uDFA8",
            [":articulated_lorry:"] = "\uD83D\uDE9B",
            [":asterisk:"] = "*️⃣",
            [":astonished:"] = "\uD83D\uDE32",
            [":athletic_shoe:"] = "\uD83D\uDC5F",
            [":atm:"] = "\uD83C\uDFE7",
            [":atom:"] = "⚛️",
            [":atom_symbol:"] = "⚛️",
            [":auto_rickshaw:"] = "\uD83D\uDEFA",
            [":avocado:"] = "\uD83E\uDD51",
            [":axe:"] = "\uD83E\uDE93",
            [":b:"] = "\uD83C\uDD71️",
            [":baby:"] = "\uD83D\uDC76",
            [":baby::skin-tone-1:"] = "\uD83D\uDC76\uD83C\uDFFB",
            [":baby::skin-tone-2:"] = "\uD83D\uDC76\uD83C\uDFFC",
            [":baby::skin-tone-3:"] = "\uD83D\uDC76\uD83C\uDFFD",
            [":baby::skin-tone-4:"] = "\uD83D\uDC76\uD83C\uDFFE",
            [":baby::skin-tone-5:"] = "\uD83D\uDC76\uD83C\uDFFF",
            [":baby_bottle:"] = "\uD83C\uDF7C",
            [":baby_chick:"] = "\uD83D\uDC24",
            [":baby_symbol:"] = "\uD83D\uDEBC",
            [":baby_tone1:"] = "\uD83D\uDC76\uD83C\uDFFB",
            [":baby_tone2:"] = "\uD83D\uDC76\uD83C\uDFFC",
            [":baby_tone3:"] = "\uD83D\uDC76\uD83C\uDFFD",
            [":baby_tone4:"] = "\uD83D\uDC76\uD83C\uDFFE",
            [":baby_tone5:"] = "\uD83D\uDC76\uD83C\uDFFF",
            [":back:"] = "\uD83D\uDD19",
            [":back_of_hand:"] = "\uD83E\uDD1A",
            [":back_of_hand::skin-tone-1:"] = "\uD83E\uDD1A\uD83C\uDFFB",
            [":back_of_hand::skin-tone-2:"] = "\uD83E\uDD1A\uD83C\uDFFC",
            [":back_of_hand::skin-tone-3:"] = "\uD83E\uDD1A\uD83C\uDFFD",
            [":back_of_hand::skin-tone-4:"] = "\uD83E\uDD1A\uD83C\uDFFE",
            [":back_of_hand::skin-tone-5:"] = "\uD83E\uDD1A\uD83C\uDFFF",
            [":back_of_hand_tone1:"] = "\uD83E\uDD1A\uD83C\uDFFB",
            [":back_of_hand_tone2:"] = "\uD83E\uDD1A\uD83C\uDFFC",
            [":back_of_hand_tone3:"] = "\uD83E\uDD1A\uD83C\uDFFD",
            [":back_of_hand_tone4:"] = "\uD83E\uDD1A\uD83C\uDFFE",
            [":back_of_hand_tone5:"] = "\uD83E\uDD1A\uD83C\uDFFF",
            [":bacon:"] = "\uD83E\uDD53",
            [":badger:"] = "\uD83E\uDDA1",
            [":badminton:"] = "\uD83C\uDFF8",
            [":bagel:"] = "\uD83E\uDD6F",
            [":baggage_claim:"] = "\uD83D\uDEC4",
            [":baguette_bread:"] = "\uD83E\uDD56",
            [":ballet_shoes:"] = "\uD83E\uDE70",
            [":balloon:"] = "\uD83C\uDF88",
            [":ballot_box:"] = "\uD83D\uDDF3️",
            [":ballot_box_with_ballot:"] = "\uD83D\uDDF3️",
            [":ballot_box_with_check:"] = "☑️",
            [":bamboo:"] = "\uD83C\uDF8D",
            [":banana:"] = "\uD83C\uDF4C",
            [":bangbang:"] = "‼️",
            [":banjo:"] = "\uD83E\uDE95",
            [":bank:"] = "\uD83C\uDFE6",
            [":bar_chart:"] = "\uD83D\uDCCA",
            [":barber:"] = "\uD83D\uDC88",
            [":baseball:"] = "⚾",
            [":basket:"] = "\uD83E\uDDFA",
            [":basketball:"] = "\uD83C\uDFC0",
            [":basketball_player:"] = "⛹️",
            [":basketball_player::skin-tone-1:"] = "⛹\uD83C\uDFFB",
            [":basketball_player::skin-tone-2:"] = "⛹\uD83C\uDFFC",
            [":basketball_player::skin-tone-3:"] = "⛹\uD83C\uDFFD",
            [":basketball_player::skin-tone-4:"] = "⛹\uD83C\uDFFE",
            [":basketball_player::skin-tone-5:"] = "⛹\uD83C\uDFFF",
            [":basketball_player_tone1:"] = "⛹\uD83C\uDFFB",
            [":basketball_player_tone2:"] = "⛹\uD83C\uDFFC",
            [":basketball_player_tone3:"] = "⛹\uD83C\uDFFD",
            [":basketball_player_tone4:"] = "⛹\uD83C\uDFFE",
            [":basketball_player_tone5:"] = "⛹\uD83C\uDFFF",
            [":bat:"] = "\uD83E\uDD87",
            [":bath:"] = "\uD83D\uDEC0",
            [":bath::skin-tone-1:"] = "\uD83D\uDEC0\uD83C\uDFFB",
            [":bath::skin-tone-2:"] = "\uD83D\uDEC0\uD83C\uDFFC",
            [":bath::skin-tone-3:"] = "\uD83D\uDEC0\uD83C\uDFFD",
            [":bath::skin-tone-4:"] = "\uD83D\uDEC0\uD83C\uDFFE",
            [":bath::skin-tone-5:"] = "\uD83D\uDEC0\uD83C\uDFFF",
            [":bath_tone1:"] = "\uD83D\uDEC0\uD83C\uDFFB",
            [":bath_tone2:"] = "\uD83D\uDEC0\uD83C\uDFFC",
            [":bath_tone3:"] = "\uD83D\uDEC0\uD83C\uDFFD",
            [":bath_tone4:"] = "\uD83D\uDEC0\uD83C\uDFFE",
            [":bath_tone5:"] = "\uD83D\uDEC0\uD83C\uDFFF",
            [":bathtub:"] = "\uD83D\uDEC1",
            [":battery:"] = "\uD83D\uDD0B",
            [":beach:"] = "\uD83C\uDFD6️",
            [":beach_umbrella:"] = "⛱️",
            [":beach_with_umbrella:"] = "\uD83C\uDFD6️",
            [":beans:"] = "\uD83E\uDED8",
            [":bear:"] = "\uD83D\uDC3B",
            [":bearded_person:"] = "\uD83E\uDDD4",
            [":bearded_person::skin-tone-1:"] = "\uD83E\uDDD4\uD83C\uDFFB",
            [":bearded_person::skin-tone-2:"] = "\uD83E\uDDD4\uD83C\uDFFC",
            [":bearded_person::skin-tone-3:"] = "\uD83E\uDDD4\uD83C\uDFFD",
            [":bearded_person::skin-tone-4:"] = "\uD83E\uDDD4\uD83C\uDFFE",
            [":bearded_person::skin-tone-5:"] = "\uD83E\uDDD4\uD83C\uDFFF",
            [":bearded_person_dark_skin_tone:"] = "\uD83E\uDDD4\uD83C\uDFFF",
            [":bearded_person_light_skin_tone:"] = "\uD83E\uDDD4\uD83C\uDFFB",
            [":bearded_person_medium_dark_skin_tone:"] = "\uD83E\uDDD4\uD83C\uDFFE",
            [":bearded_person_medium_light_skin_tone:"] = "\uD83E\uDDD4\uD83C\uDFFC",
            [":bearded_person_medium_skin_tone:"] = "\uD83E\uDDD4\uD83C\uDFFD",
            [":bearded_person_tone1:"] = "\uD83E\uDDD4\uD83C\uDFFB",
            [":bearded_person_tone2:"] = "\uD83E\uDDD4\uD83C\uDFFC",
            [":bearded_person_tone3:"] = "\uD83E\uDDD4\uD83C\uDFFD",
            [":bearded_person_tone4:"] = "\uD83E\uDDD4\uD83C\uDFFE",
            [":bearded_person_tone5:"] = "\uD83E\uDDD4\uD83C\uDFFF",
            [":beaver:"] = "\uD83E\uDDAB",
            [":bed:"] = "\uD83D\uDECF️",
            [":bee:"] = "\uD83D\uDC1D",
            [":beer:"] = "\uD83C\uDF7A",
            [":beers:"] = "\uD83C\uDF7B",
            [":beetle:"] = "\uD83D\uDC1E",
            [":beetle:"] = "\uD83E\uDEB2",
            [":beginner:"] = "\uD83D\uDD30",
            [":bell:"] = "\uD83D\uDD14",
            [":bell_pepper:"] = "\uD83E\uDED1",
            [":bellhop:"] = "\uD83D\uDECE️",
            [":bellhop_bell:"] = "\uD83D\uDECE️",
            [":bento:"] = "\uD83C\uDF71",
            [":beverage_box:"] = "\uD83E\uDDC3",
            [":bicyclist:"] = "\uD83D\uDEB4",
            [":bicyclist::skin-tone-1:"] = "\uD83D\uDEB4\uD83C\uDFFB",
            [":bicyclist::skin-tone-2:"] = "\uD83D\uDEB4\uD83C\uDFFC",
            [":bicyclist::skin-tone-3:"] = "\uD83D\uDEB4\uD83C\uDFFD",
            [":bicyclist::skin-tone-4:"] = "\uD83D\uDEB4\uD83C\uDFFE",
            [":bicyclist::skin-tone-5:"] = "\uD83D\uDEB4\uD83C\uDFFF",
            [":bicyclist_tone1:"] = "\uD83D\uDEB4\uD83C\uDFFB",
            [":bicyclist_tone2:"] = "\uD83D\uDEB4\uD83C\uDFFC",
            [":bicyclist_tone3:"] = "\uD83D\uDEB4\uD83C\uDFFD",
            [":bicyclist_tone4:"] = "\uD83D\uDEB4\uD83C\uDFFE",
            [":bicyclist_tone5:"] = "\uD83D\uDEB4\uD83C\uDFFF",
            [":bike:"] = "\uD83D\uDEB2",
            [":bikini:"] = "\uD83D\uDC59",
            [":billed_cap:"] = "\uD83E\uDDE2",
            [":biohazard:"] = "☣️",
            [":biohazard_sign:"] = "☣️",
            [":bird:"] = "\uD83D\uDC26",
            [":birthday:"] = "\uD83C\uDF82",
            [":bison:"] = "\uD83E\uDDAC",
            [":biting_lip:"] = "\uD83E\uDEE6",
            [":black_cat:"] = "\uD83D\uDC08\u200D\u2B1B",
            [":black_circle:"] = "⚫",
            [":black_heart:"] = "\uD83D\uDDA4",
            [":black_joker:"] = "\uD83C\uDCCF",
            [":black_large_square:"] = "⬛",
            [":black_medium_small_square:"] = "◾",
            [":black_medium_square:"] = "◼️",
            [":black_nib:"] = "✒️",
            [":black_small_square:"] = "▪️",
            [":black_square_button:"] = "\uD83D\uDD32",
            [":blond_haired_man:"] = "\uD83D\uDC71\u200D♂️",
            [":blond_haired_man::skin-tone-1:"] = "\uD83D\uDC71\uD83C\uDFFB\u200D♂️",
            [":blond_haired_man::skin-tone-2:"] = "\uD83D\uDC71\uD83C\uDFFC\u200D♂️",
            [":blond_haired_man::skin-tone-3:"] = "\uD83D\uDC71\uD83C\uDFFD\u200D♂️",
            [":blond_haired_man::skin-tone-4:"] = "\uD83D\uDC71\uD83C\uDFFE\u200D♂️",
            [":blond_haired_man::skin-tone-5:"] = "\uD83D\uDC71\uD83C\uDFFF\u200D♂️",
            [":blond_haired_man_dark_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFF\u200D♂️",
            [":blond_haired_man_light_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFB\u200D♂️",
            [":blond_haired_man_medium_dark_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFE\u200D♂️",
            [":blond_haired_man_medium_light_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFC\u200D♂️",
            [":blond_haired_man_medium_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFD\u200D♂️",
            [":blond_haired_man_tone1:"] = "\uD83D\uDC71\uD83C\uDFFB\u200D♂️",
            [":blond_haired_man_tone2:"] = "\uD83D\uDC71\uD83C\uDFFC\u200D♂️",
            [":blond_haired_man_tone3:"] = "\uD83D\uDC71\uD83C\uDFFD\u200D♂️",
            [":blond_haired_man_tone4:"] = "\uD83D\uDC71\uD83C\uDFFE\u200D♂️",
            [":blond_haired_man_tone5:"] = "\uD83D\uDC71\uD83C\uDFFF\u200D♂️",
            [":blond_haired_person:"] = "\uD83D\uDC71",
            [":blond_haired_person::skin-tone-1:"] = "\uD83D\uDC71\uD83C\uDFFB",
            [":blond_haired_person::skin-tone-2:"] = "\uD83D\uDC71\uD83C\uDFFC",
            [":blond_haired_person::skin-tone-3:"] = "\uD83D\uDC71\uD83C\uDFFD",
            [":blond_haired_person::skin-tone-4:"] = "\uD83D\uDC71\uD83C\uDFFE",
            [":blond_haired_person::skin-tone-5:"] = "\uD83D\uDC71\uD83C\uDFFF",
            [":blond_haired_person_tone1:"] = "\uD83D\uDC71\uD83C\uDFFB",
            [":blond_haired_person_tone2:"] = "\uD83D\uDC71\uD83C\uDFFC",
            [":blond_haired_person_tone3:"] = "\uD83D\uDC71\uD83C\uDFFD",
            [":blond_haired_person_tone4:"] = "\uD83D\uDC71\uD83C\uDFFE",
            [":blond_haired_person_tone5:"] = "\uD83D\uDC71\uD83C\uDFFF",
            [":blond_haired_woman:"] = "\uD83D\uDC71\u200D♀️",
            [":blond_haired_woman::skin-tone-1:"] = "\uD83D\uDC71\uD83C\uDFFB\u200D♀️",
            [":blond_haired_woman::skin-tone-2:"] = "\uD83D\uDC71\uD83C\uDFFC\u200D♀️",
            [":blond_haired_woman::skin-tone-3:"] = "\uD83D\uDC71\uD83C\uDFFD\u200D♀️",
            [":blond_haired_woman::skin-tone-4:"] = "\uD83D\uDC71\uD83C\uDFFE\u200D♀️",
            [":blond_haired_woman::skin-tone-5:"] = "\uD83D\uDC71\uD83C\uDFFF\u200D♀️",
            [":blond_haired_woman_dark_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFF\u200D♀️",
            [":blond_haired_woman_light_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFB\u200D♀️",
            [":blond_haired_woman_medium_dark_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFE\u200D♀️",
            [":blond_haired_woman_medium_light_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFC\u200D♀️",
            [":blond_haired_woman_medium_skin_tone:"] = "\uD83D\uDC71\uD83C\uDFFD\u200D♀️",
            [":blond_haired_woman_tone1:"] = "\uD83D\uDC71\uD83C\uDFFB\u200D♀️",
            [":blond_haired_woman_tone2:"] = "\uD83D\uDC71\uD83C\uDFFC\u200D♀️",
            [":blond_haired_woman_tone3:"] = "\uD83D\uDC71\uD83C\uDFFD\u200D♀️",
            [":blond_haired_woman_tone4:"] = "\uD83D\uDC71\uD83C\uDFFE\u200D♀️",
            [":blond_haired_woman_tone5:"] = "\uD83D\uDC71\uD83C\uDFFF\u200D♀️",
            [":blossom:"] = "\uD83C\uDF3C",
            [":blowfish:"] = "\uD83D\uDC21",
            [":blue_book:"] = "\uD83D\uDCD8",
            [":blue_car:"] = "\uD83D\uDE99",
            [":blue_circle:"] = "\uD83D\uDD35",
            [":blue_heart:"] = "\uD83D\uDC99",
            [":blue_square:"] = "\uD83D\uDFE6",
            [":blueberries:"] = "\uD83E\uDED0",
            [":blush:"] = "\uD83D\uDE0A",
            [":boar:"] = "\uD83D\uDC17",
            [":bomb:"] = "\uD83D\uDCA3",
            [":bone:"] = "\uD83E\uDDB4",
            [":book:"] = "\uD83D\uDCD6",
            [":bookmark:"] = "\uD83D\uDD16",
            [":bookmark_tabs:"] = "\uD83D\uDCD1",
            [":books:"] = "\uD83D\uDCDA",
            [":boom:"] = "\uD83D\uDCA5",
            [":boomerang:"] = "\uD83E\uDE83",
            [":boot:"] = "\uD83D\uDC62",
            [":bottle_with_popping_cork:"] = "\uD83C\uDF7E",
            [":bouquet:"] = "\uD83D\uDC90",
            [":bow:"] = "\uD83D\uDE47",
            [":bow::skin-tone-1:"] = "\uD83D\uDE47\uD83C\uDFFB",
            [":bow::skin-tone-2:"] = "\uD83D\uDE47\uD83C\uDFFC",
            [":bow::skin-tone-3:"] = "\uD83D\uDE47\uD83C\uDFFD",
            [":bow::skin-tone-4:"] = "\uD83D\uDE47\uD83C\uDFFE",
            [":bow::skin-tone-5:"] = "\uD83D\uDE47\uD83C\uDFFF",
            [":bow_and_arrow:"] = "\uD83C\uDFF9",
            [":bow_tone1:"] = "\uD83D\uDE47\uD83C\uDFFB",
            [":bow_tone2:"] = "\uD83D\uDE47\uD83C\uDFFC",
            [":bow_tone3:"] = "\uD83D\uDE47\uD83C\uDFFD",
            [":bow_tone4:"] = "\uD83D\uDE47\uD83C\uDFFE",
            [":bow_tone5:"] = "\uD83D\uDE47\uD83C\uDFFF",
            [":bowl_with_spoon:"] = "\uD83E\uDD63",
            [":bowling:"] = "\uD83C\uDFB3",
            [":boxing_glove:"] = "\uD83E\uDD4A",
            [":boxing_gloves:"] = "\uD83E\uDD4A",
            [":boy:"] = "\uD83D\uDC66",
            [":boy::skin-tone-1:"] = "\uD83D\uDC66\uD83C\uDFFB",
            [":boy::skin-tone-2:"] = "\uD83D\uDC66\uD83C\uDFFC",
            [":boy::skin-tone-3:"] = "\uD83D\uDC66\uD83C\uDFFD",
            [":boy::skin-tone-4:"] = "\uD83D\uDC66\uD83C\uDFFE",
            [":boy::skin-tone-5:"] = "\uD83D\uDC66\uD83C\uDFFF",
            [":boy_tone1:"] = "\uD83D\uDC66\uD83C\uDFFB",
            [":boy_tone2:"] = "\uD83D\uDC66\uD83C\uDFFC",
            [":boy_tone3:"] = "\uD83D\uDC66\uD83C\uDFFD",
            [":boy_tone4:"] = "\uD83D\uDC66\uD83C\uDFFE",
            [":boy_tone5:"] = "\uD83D\uDC66\uD83C\uDFFF",
            [":brain:"] = "\uD83E\uDDE0",
            [":bread:"] = "\uD83C\uDF5E",
            [":breast_feeding:"] = "\uD83E\uDD31",
            [":breast_feeding::skin-tone-1:"] = "\uD83E\uDD31\uD83C\uDFFB",
            [":breast_feeding::skin-tone-2:"] = "\uD83E\uDD31\uD83C\uDFFC",
            [":breast_feeding::skin-tone-3:"] = "\uD83E\uDD31\uD83C\uDFFD",
            [":breast_feeding::skin-tone-4:"] = "\uD83E\uDD31\uD83C\uDFFE",
            [":breast_feeding::skin-tone-5:"] = "\uD83E\uDD31\uD83C\uDFFF",
            [":breast_feeding_dark_skin_tone:"] = "\uD83E\uDD31\uD83C\uDFFF",
            [":breast_feeding_light_skin_tone:"] = "\uD83E\uDD31\uD83C\uDFFB",
            [":breast_feeding_medium_dark_skin_tone:"] = "\uD83E\uDD31\uD83C\uDFFE",
            [":breast_feeding_medium_light_skin_tone:"] = "\uD83E\uDD31\uD83C\uDFFC",
            [":breast_feeding_medium_skin_tone:"] = "\uD83E\uDD31\uD83C\uDFFD",
            [":breast_feeding_tone1:"] = "\uD83E\uDD31\uD83C\uDFFB",
            [":breast_feeding_tone2:"] = "\uD83E\uDD31\uD83C\uDFFC",
            [":breast_feeding_tone3:"] = "\uD83E\uDD31\uD83C\uDFFD",
            [":breast_feeding_tone4:"] = "\uD83E\uDD31\uD83C\uDFFE",
            [":breast_feeding_tone5:"] = "\uD83E\uDD31\uD83C\uDFFF",
            [":bricks:"] = "\uD83E\uDDF1",
            [":bride_with_veil:"] = "\uD83D\uDC70",
            [":bride_with_veil::skin-tone-1:"] = "\uD83D\uDC70\uD83C\uDFFB",
            [":bride_with_veil::skin-tone-2:"] = "\uD83D\uDC70\uD83C\uDFFC",
            [":bride_with_veil::skin-tone-3:"] = "\uD83D\uDC70\uD83C\uDFFD",
            [":bride_with_veil::skin-tone-4:"] = "\uD83D\uDC70\uD83C\uDFFE",
            [":bride_with_veil::skin-tone-5:"] = "\uD83D\uDC70\uD83C\uDFFF",
            [":bride_with_veil_tone1:"] = "\uD83D\uDC70\uD83C\uDFFB",
            [":bride_with_veil_tone2:"] = "\uD83D\uDC70\uD83C\uDFFC",
            [":bride_with_veil_tone3:"] = "\uD83D\uDC70\uD83C\uDFFD",
            [":bride_with_veil_tone4:"] = "\uD83D\uDC70\uD83C\uDFFE",
            [":bride_with_veil_tone5:"] = "\uD83D\uDC70\uD83C\uDFFF",
            [":bridge_at_night:"] = "\uD83C\uDF09",
            [":briefcase:"] = "\uD83D\uDCBC",
            [":briefs:"] = "\uD83E\uDE72",
            [":broccoli:"] = "\uD83E\uDD66",
            [":broken_heart:"] = "\uD83D\uDC94",
            [":broom:"] = "\uD83E\uDDF9",
            [":brown_circle:"] = "\uD83D\uDFE4",
            [":brown_heart:"] = "\uD83E\uDD0E",
            [":brown_square:"] = "\uD83D\uDFEB",
            [":bubble_tea:"] = "\uD83E\uDDCB",
            [":bubbles:"] = "\uD83E\uDEE7",
            [":bucket:"] = "\uD83E\uDEA3",
            [":bug:"] = "\uD83D\uDC1B",
            [":building_construction:"] = "\uD83C\uDFD7️",
            [":bulb:"] = "\uD83D\uDCA1",
            [":bullettrain_front:"] = "\uD83D\uDE85",
            [":bullettrain_side:"] = "\uD83D\uDE84",
            [":burrito:"] = "\uD83C\uDF2F",
            [":bus:"] = "\uD83D\uDE8C",
            [":busstop:"] = "\uD83D\uDE8F",
            [":bust_in_silhouette:"] = "\uD83D\uDC64",
            [":busts_in_silhouette:"] = "\uD83D\uDC65",
            [":butter:"] = "\uD83E\uDDC8",
            [":butterfly:"] = "\uD83E\uDD8B",
            [":cactus:"] = "\uD83C\uDF35",
            [":cake:"] = "\uD83C\uDF70",
            [":calendar:"] = "\uD83D\uDCC6",
            [":calendar_spiral:"] = "\uD83D\uDDD3️",
            [":call_me:"] = "\uD83E\uDD19",
            [":call_me::skin-tone-1:"] = "\uD83E\uDD19\uD83C\uDFFB",
            [":call_me::skin-tone-2:"] = "\uD83E\uDD19\uD83C\uDFFC",
            [":call_me::skin-tone-3:"] = "\uD83E\uDD19\uD83C\uDFFD",
            [":call_me::skin-tone-4:"] = "\uD83E\uDD19\uD83C\uDFFE",
            [":call_me::skin-tone-5:"] = "\uD83E\uDD19\uD83C\uDFFF",
            [":call_me_hand:"] = "\uD83E\uDD19",
            [":call_me_hand::skin-tone-1:"] = "\uD83E\uDD19\uD83C\uDFFB",
            [":call_me_hand::skin-tone-2:"] = "\uD83E\uDD19\uD83C\uDFFC",
            [":call_me_hand::skin-tone-3:"] = "\uD83E\uDD19\uD83C\uDFFD",
            [":call_me_hand::skin-tone-4:"] = "\uD83E\uDD19\uD83C\uDFFE",
            [":call_me_hand::skin-tone-5:"] = "\uD83E\uDD19\uD83C\uDFFF",
            [":call_me_hand_tone1:"] = "\uD83E\uDD19\uD83C\uDFFB",
            [":call_me_hand_tone2:"] = "\uD83E\uDD19\uD83C\uDFFC",
            [":call_me_hand_tone3:"] = "\uD83E\uDD19\uD83C\uDFFD",
            [":call_me_hand_tone4:"] = "\uD83E\uDD19\uD83C\uDFFE",
            [":call_me_hand_tone5:"] = "\uD83E\uDD19\uD83C\uDFFF",
            [":call_me_tone1:"] = "\uD83E\uDD19\uD83C\uDFFB",
            [":call_me_tone2:"] = "\uD83E\uDD19\uD83C\uDFFC",
            [":call_me_tone3:"] = "\uD83E\uDD19\uD83C\uDFFD",
            [":call_me_tone4:"] = "\uD83E\uDD19\uD83C\uDFFE",
            [":call_me_tone5:"] = "\uD83E\uDD19\uD83C\uDFFF",
            [":calling:"] = "\uD83D\uDCF2",
            [":camel:"] = "\uD83D\uDC2B",
            [":camera:"] = "\uD83D\uDCF7",
            [":camera_with_flash:"] = "\uD83D\uDCF8",
            [":camping:"] = "\uD83C\uDFD5️",
            [":cancer:"] = "♋",
            [":candle:"] = "\uD83D\uDD6F️",
            [":candy:"] = "\uD83C\uDF6C",
            [":canned_food:"] = "\uD83E\uDD6B",
            [":canoe:"] = "\uD83D\uDEF6",
            [":capital_abcd:"] = "\uD83D\uDD20",
            [":capricorn:"] = "♑",
            [":card_box:"] = "\uD83D\uDDC3️",
            [":card_file_box:"] = "\uD83D\uDDC3️",
            [":card_index:"] = "\uD83D\uDCC7",
            [":card_index_dividers:"] = "\uD83D\uDDC2️",
            [":carousel_horse:"] = "\uD83C\uDFA0",
            [":carpentry_saw:"] = "\uD83E\uDE9A",
            [":carrot:"] = "\uD83E\uDD55",
            [":cartwheel:"] = "\uD83E\uDD38",
            [":cartwheel::skin-tone-1:"] = "\uD83E\uDD38\uD83C\uDFFB",
            [":cartwheel::skin-tone-2:"] = "\uD83E\uDD38\uD83C\uDFFC",
            [":cartwheel::skin-tone-3:"] = "\uD83E\uDD38\uD83C\uDFFD",
            [":cartwheel::skin-tone-4:"] = "\uD83E\uDD38\uD83C\uDFFE",
            [":cartwheel::skin-tone-5:"] = "\uD83E\uDD38\uD83C\uDFFF",
            [":cartwheel_tone1:"] = "\uD83E\uDD38\uD83C\uDFFB",
            [":cartwheel_tone2:"] = "\uD83E\uDD38\uD83C\uDFFC",
            [":cartwheel_tone3:"] = "\uD83E\uDD38\uD83C\uDFFD",
            [":cartwheel_tone4:"] = "\uD83E\uDD38\uD83C\uDFFE",
            [":cartwheel_tone5:"] = "\uD83E\uDD38\uD83C\uDFFF",
            [":cat2:"] = "\uD83D\uDC08",
            [":cat:"] = "\uD83D\uDC31",
            [":cd:"] = "\uD83D\uDCBF",
            [":chains:"] = "⛓️",
            [":chair:"] = "\uD83E\uDE91",
            [":champagne:"] = "\uD83C\uDF7E",
            [":champagne_glass:"] = "\uD83E\uDD42",
            [":chart:"] = "\uD83D\uDCB9",
            [":chart_with_downwards_trend:"] = "\uD83D\uDCC9",
            [":chart_with_upwards_trend:"] = "\uD83D\uDCC8",
            [":checkered_flag:"] = "\uD83C\uDFC1",
            [":cheese:"] = "\uD83E\uDDC0",
            [":cheese_wedge:"] = "\uD83E\uDDC0",
            [":cherries:"] = "\uD83C\uDF52",
            [":cherry_blossom:"] = "\uD83C\uDF38",
            [":chess_pawn:"] = "♟️",
            [":chestnut:"] = "\uD83C\uDF30",
            [":chicken:"] = "\uD83D\uDC14",
            [":child:"] = "\uD83E\uDDD2",
            [":child::skin-tone-1:"] = "\uD83E\uDDD2\uD83C\uDFFB",
            [":child::skin-tone-2:"] = "\uD83E\uDDD2\uD83C\uDFFC",
            [":child::skin-tone-3:"] = "\uD83E\uDDD2\uD83C\uDFFD",
            [":child::skin-tone-4:"] = "\uD83E\uDDD2\uD83C\uDFFE",
            [":child::skin-tone-5:"] = "\uD83E\uDDD2\uD83C\uDFFF",
            [":child_dark_skin_tone:"] = "\uD83E\uDDD2\uD83C\uDFFF",
            [":child_light_skin_tone:"] = "\uD83E\uDDD2\uD83C\uDFFB",
            [":child_medium_dark_skin_tone:"] = "\uD83E\uDDD2\uD83C\uDFFE",
            [":child_medium_light_skin_tone:"] = "\uD83E\uDDD2\uD83C\uDFFC",
            [":child_medium_skin_tone:"] = "\uD83E\uDDD2\uD83C\uDFFD",
            [":child_tone1:"] = "\uD83E\uDDD2\uD83C\uDFFB",
            [":child_tone2:"] = "\uD83E\uDDD2\uD83C\uDFFC",
            [":child_tone3:"] = "\uD83E\uDDD2\uD83C\uDFFD",
            [":child_tone4:"] = "\uD83E\uDDD2\uD83C\uDFFE",
            [":child_tone5:"] = "\uD83E\uDDD2\uD83C\uDFFF",
            [":children_crossing:"] = "\uD83D\uDEB8",
            [":chipmunk:"] = "\uD83D\uDC3F️",
            [":chocolate_bar:"] = "\uD83C\uDF6B",
            [":chopsticks:"] = "\uD83E\uDD62",
            [":christmas_tree:"] = "\uD83C\uDF84",
            [":church:"] = "⛪",
            [":cinema:"] = "\uD83C\uDFA6",
            [":circus_tent:"] = "\uD83C\uDFAA",
            [":city_dusk:"] = "\uD83C\uDF06",
            [":city_sunrise:"] = "\uD83C\uDF07",
            [":city_sunset:"] = "\uD83C\uDF07",
            [":cityscape:"] = "\uD83C\uDFD9️",
            [":cl:"] = "\uD83C\uDD91",
            [":clap:"] = "\uD83D\uDC4F",
            [":clap::skin-tone-1:"] = "\uD83D\uDC4F\uD83C\uDFFB",
            [":clap::skin-tone-2:"] = "\uD83D\uDC4F\uD83C\uDFFC",
            [":clap::skin-tone-3:"] = "\uD83D\uDC4F\uD83C\uDFFD",
            [":clap::skin-tone-4:"] = "\uD83D\uDC4F\uD83C\uDFFE",
            [":clap::skin-tone-5:"] = "\uD83D\uDC4F\uD83C\uDFFF",
            [":clap_tone1:"] = "\uD83D\uDC4F\uD83C\uDFFB",
            [":clap_tone2:"] = "\uD83D\uDC4F\uD83C\uDFFC",
            [":clap_tone3:"] = "\uD83D\uDC4F\uD83C\uDFFD",
            [":clap_tone4:"] = "\uD83D\uDC4F\uD83C\uDFFE",
            [":clap_tone5:"] = "\uD83D\uDC4F\uD83C\uDFFF",
            [":clapper:"] = "\uD83C\uDFAC",
            [":classical_building:"] = "\uD83C\uDFDB️",
            [":clinking_glass:"] = "\uD83E\uDD42",
            [":clipboard:"] = "\uD83D\uDCCB",
            [":clock1030:"] = "\uD83D\uDD65",
            [":clock10:"] = "\uD83D\uDD59",
            [":clock1130:"] = "\uD83D\uDD66",
            [":clock11:"] = "\uD83D\uDD5A",
            [":clock1230:"] = "\uD83D\uDD67",
            [":clock12:"] = "\uD83D\uDD5B",
            [":clock130:"] = "\uD83D\uDD5C",
            [":clock1:"] = "\uD83D\uDD50",
            [":clock230:"] = "\uD83D\uDD5D",
            [":clock2:"] = "\uD83D\uDD51",
            [":clock330:"] = "\uD83D\uDD5E",
            [":clock3:"] = "\uD83D\uDD52",
            [":clock430:"] = "\uD83D\uDD5F",
            [":clock4:"] = "\uD83D\uDD53",
            [":clock530:"] = "\uD83D\uDD60",
            [":clock5:"] = "\uD83D\uDD54",
            [":clock630:"] = "\uD83D\uDD61",
            [":clock6:"] = "\uD83D\uDD55",
            [":clock730:"] = "\uD83D\uDD62",
            [":clock7:"] = "\uD83D\uDD56",
            [":clock830:"] = "\uD83D\uDD63",
            [":clock8:"] = "\uD83D\uDD57",
            [":clock930:"] = "\uD83D\uDD64",
            [":clock9:"] = "\uD83D\uDD58",
            [":clock:"] = "\uD83D\uDD70️",
            [":closed_book:"] = "\uD83D\uDCD5",
            [":closed_lock_with_key:"] = "\uD83D\uDD10",
            [":closed_umbrella:"] = "\uD83C\uDF02",
            [":cloud:"] = "☁️",
            [":cloud_lightning:"] = "\uD83C\uDF29️",
            [":cloud_rain:"] = "\uD83C\uDF27️",
            [":cloud_snow:"] = "\uD83C\uDF28️",
            [":cloud_tornado:"] = "\uD83C\uDF2A️",
            [":cloud_with_lightning:"] = "\uD83C\uDF29️",
            [":cloud_with_rain:"] = "\uD83C\uDF27️",
            [":cloud_with_snow:"] = "\uD83C\uDF28️",
            [":cloud_with_tornado:"] = "\uD83C\uDF2A️",
            [":clown:"] = "\uD83E\uDD21",
            [":clown_face:"] = "\uD83E\uDD21",
            [":clubs:"] = "♣️",
            [":coat:"] = "\uD83E\uDDE5",
            [":cockroach:"] = "\uD83E\uDEB3",
            [":cocktail:"] = "\uD83C\uDF78",
            [":coconut:"] = "\uD83E\uDD65",
            [":coffee:"] = "☕",
            [":coffin:"] = "⚰️",
            [":coin:"] = "\uD83E\uDE99",
            [":coin:"] = "\uD83E\uDE99",
            [":cold_face:"] = "\uD83E\uDD76",
            [":cold_sweat:"] = "\uD83D\uDE30",
            [":comet:"] = "☄️",
            [":compass:"] = "\uD83E\uDDED",
            [":compression:"] = "\uD83D\uDDDC️",
            [":computer:"] = "\uD83D\uDCBB",
            [":confetti_ball:"] = "\uD83C\uDF8A",
            [":confounded:"] = "\uD83D\uDE16",
            [":confused:"] = "\uD83D\uDE15",
            [":congratulations:"] = "㊗️",
            [":construction:"] = "\uD83D\uDEA7",
            [":construction_site:"] = "\uD83C\uDFD7️",
            [":construction_worker:"] = "\uD83D\uDC77",
            [":construction_worker::skin-tone-1:"] = "\uD83D\uDC77\uD83C\uDFFB",
            [":construction_worker::skin-tone-2:"] = "\uD83D\uDC77\uD83C\uDFFC",
            [":construction_worker::skin-tone-3:"] = "\uD83D\uDC77\uD83C\uDFFD",
            [":construction_worker::skin-tone-4:"] = "\uD83D\uDC77\uD83C\uDFFE",
            [":construction_worker::skin-tone-5:"] = "\uD83D\uDC77\uD83C\uDFFF",
            [":construction_worker_tone1:"] = "\uD83D\uDC77\uD83C\uDFFB",
            [":construction_worker_tone2:"] = "\uD83D\uDC77\uD83C\uDFFC",
            [":construction_worker_tone3:"] = "\uD83D\uDC77\uD83C\uDFFD",
            [":construction_worker_tone4:"] = "\uD83D\uDC77\uD83C\uDFFE",
            [":construction_worker_tone5:"] = "\uD83D\uDC77\uD83C\uDFFF",
            [":control_knobs:"] = "\uD83C\uDF9B️",
            [":convenience_store:"] = "\uD83C\uDFEA",
            [":cookie:"] = "\uD83C\uDF6A",
            [":cooking:"] = "\uD83C\uDF73",
            [":cool:"] = "\uD83C\uDD92",
            [":cop:"] = "\uD83D\uDC6E",
            [":cop::skin-tone-1:"] = "\uD83D\uDC6E\uD83C\uDFFB",
            [":cop::skin-tone-2:"] = "\uD83D\uDC6E\uD83C\uDFFC",
            [":cop::skin-tone-3:"] = "\uD83D\uDC6E\uD83C\uDFFD",
            [":cop::skin-tone-4:"] = "\uD83D\uDC6E\uD83C\uDFFE",
            [":cop::skin-tone-5:"] = "\uD83D\uDC6E\uD83C\uDFFF",
            [":cop_tone1:"] = "\uD83D\uDC6E\uD83C\uDFFB",
            [":cop_tone2:"] = "\uD83D\uDC6E\uD83C\uDFFC",
            [":cop_tone3:"] = "\uD83D\uDC6E\uD83C\uDFFD",
            [":cop_tone4:"] = "\uD83D\uDC6E\uD83C\uDFFE",
            [":cop_tone5:"] = "\uD83D\uDC6E\uD83C\uDFFF",
            [":copyright:"] = "©️",
            [":coral:"] = "\uD83E\uDEB8",
            [":corn:"] = "\uD83C\uDF3D",
            [":couch:"] = "\uD83D\uDECB️",
            [":couch_and_lamp:"] = "\uD83D\uDECB️",
            [":couple:"] = "\uD83D\uDC6B",
            [":couple_mm:"] = "\uD83D\uDC68\u200D❤️\u200D\uD83D\uDC68",
            [":couple_with_heart:"] = "\uD83D\uDC91",
            [":couple_with_heart::skin-tone-1:"] = "\uD83D\uDC91\uD83C\uDFFB",
            [":couple_with_heart::skin-tone-2:"] = "\uD83D\uDC91\uD83C\uDFFC",
            [":couple_with_heart::skin-tone-3:"] = "\uD83D\uDC91\uD83C\uDFFD",
            [":couple_with_heart::skin-tone-4:"] = "\uD83D\uDC91\uD83C\uDFFE",
            [":couple_with_heart::skin-tone-5:"] = "\uD83D\uDC91\uD83C\uDFFF",
            [":couple_with_heart_dark_skin_tone:"] = "\uD83D\uDC91\uD83C\uDFFF",
            [":couple_with_heart_light_skin_tone:"] = "\uD83D\uDC91\uD83C\uDFFB",
            [":couple_with_heart_man_man::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_dark_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_dark_skin_tone_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone_tone3:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_dark_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_dark_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_light_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_light_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_light_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_light_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_light_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_medium_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_medium_light_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_man_man_medium_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_medium_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_medium_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_medium_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_medium_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_man_man_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_man_man_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_man_man_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_man_man_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_medium_dark_skin_tone:"] = "\uD83D\uDC91\uD83C\uDFFE",
            [":couple_with_heart_medium_light_skin_tone:"] = "\uD83D\uDC91\uD83C\uDFFC",
            [":couple_with_heart_medium_skin_tone:"] = "\uD83D\uDC91\uD83C\uDFFD",
            [":couple_with_heart_mm:"] = "\uD83D\uDC68\u200D❤️\u200D\uD83D\uDC68",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_dark_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_dark_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_dark_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_dark_skin_tone_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_dark_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_dark_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_light_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_light_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_light_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_light_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_light_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_light_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_light_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_light_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_light_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_light_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_light_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_light_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_medium_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_medium_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_medium_dark_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_medium_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_medium_light_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_medium_light_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_medium_light_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_medium_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_medium_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_medium_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":couple_with_heart_person_person_medium_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_medium_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_medium_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":couple_with_heart_person_person_medium_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_person_person_medium_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":couple_with_heart_person_person_medium_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":couple_with_heart_person_person_medium_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":couple_with_heart_tone1:"] = "\uD83D\uDC91\uD83C\uDFFB",
            [":couple_with_heart_tone2:"] = "\uD83D\uDC91\uD83C\uDFFC",
            [":couple_with_heart_tone3:"] = "\uD83D\uDC91\uD83C\uDFFD",
            [":couple_with_heart_tone4:"] = "\uD83D\uDC91\uD83C\uDFFE",
            [":couple_with_heart_tone5:"] = "\uD83D\uDC91\uD83C\uDFFF",
            [":couple_with_heart_woman_man:"] = "\uD83D\uDC69\u200D❤️\u200D\uD83D\uDC68",
            [":couple_with_heart_woman_man::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_light_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_medium_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_medium_light_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":couple_with_heart_woman_man_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":couple_with_heart_woman_man_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":couple_with_heart_woman_man_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":couple_with_heart_woman_man_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":couple_with_heart_woman_woman::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_light_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_medium_light_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":couple_with_heart_woman_woman_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":couple_with_heart_woman_woman_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":couple_with_heart_woman_woman_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":couple_with_heart_woman_woman_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":couple_with_heart_ww:"] = "\uD83D\uDC69\u200D❤️\u200D\uD83D\uDC69",
            [":couple_ww:"] = "\uD83D\uDC69\u200D❤️\u200D\uD83D\uDC69",
            [":couplekiss:"] = "\uD83D\uDC8F",
            [":couplekiss_mm:"] = "\uD83D\uDC68\u200D❤️\u200D\uD83D\uDC8B\u200D\uD83D\uDC68",
            [":couplekiss_ww:"] = "\uD83D\uDC69\u200D❤️\u200D\uD83D\uDC8B\u200D\uD83D\uDC69",
            [":cow2:"] = "\uD83D\uDC04",
            [":cow:"] = "\uD83D\uDC2E",
            [":cowboy:"] = "\uD83E\uDD20",
            [":crab:"] = "\uD83E\uDD80",
            [":crayon:"] = "\uD83D\uDD8D️",
            [":credit_card:"] = "\uD83D\uDCB3",
            [":crescent_moon:"] = "\uD83C\uDF19",
            [":cricket:"] = "\uD83E\uDD97",
            [":cricket_bat_ball:"] = "\uD83C\uDFCF",
            [":cricket_game:"] = "\uD83C\uDFCF",
            [":crocodile:"] = "\uD83D\uDC0A",
            [":croissant:"] = "\uD83E\uDD50",
            [":cross:"] = "✝️",
            [":crossed_flags:"] = "\uD83C\uDF8C",
            [":crossed_swords:"] = "⚔️",
            [":crown:"] = "\uD83D\uDC51",
            [":cruise_ship:"] = "\uD83D\uDEF3️",
            [":crutch:"] = "\uD83E\uDE7C",
            [":cry:"] = "\uD83D\uDE22",
            [":crying_cat_face:"] = "\uD83D\uDE3F",
            [":crystal_ball:"] = "\uD83D\uDD2E",
            [":cucumber:"] = "\uD83E\uDD52",
            [":cup_with_straw:"] = "\uD83E\uDD64",
            [":cupcake:"] = "\uD83E\uDDC1",
            [":cupid:"] = "\uD83D\uDC98",
            [":curling_stone:"] = "\uD83E\uDD4C",
            [":curly_loop:"] = "➰",
            [":currency_exchange:"] = "\uD83D\uDCB1",
            [":curry:"] = "\uD83C\uDF5B",
            [":custard:"] = "\uD83C\uDF6E",
            [":customs:"] = "\uD83D\uDEC3",
            [":cut_of_meat:"] = "\uD83E\uDD69",
            [":cyclone:"] = "\uD83C\uDF00",
            [":dagger:"] = "\uD83D\uDDE1️",
            [":dagger_knife:"] = "\uD83D\uDDE1️",
            [":dancer:"] = "\uD83D\uDC83",
            [":dancer::skin-tone-1:"] = "\uD83D\uDC83\uD83C\uDFFB",
            [":dancer::skin-tone-2:"] = "\uD83D\uDC83\uD83C\uDFFC",
            [":dancer::skin-tone-3:"] = "\uD83D\uDC83\uD83C\uDFFD",
            [":dancer::skin-tone-4:"] = "\uD83D\uDC83\uD83C\uDFFE",
            [":dancer::skin-tone-5:"] = "\uD83D\uDC83\uD83C\uDFFF",
            [":dancer_tone1:"] = "\uD83D\uDC83\uD83C\uDFFB",
            [":dancer_tone2:"] = "\uD83D\uDC83\uD83C\uDFFC",
            [":dancer_tone3:"] = "\uD83D\uDC83\uD83C\uDFFD",
            [":dancer_tone4:"] = "\uD83D\uDC83\uD83C\uDFFE",
            [":dancer_tone5:"] = "\uD83D\uDC83\uD83C\uDFFF",
            [":dancers:"] = "\uD83D\uDC6F",
            [":dango:"] = "\uD83C\uDF61",
            [":dark_sunglasses:"] = "\uD83D\uDD76️",
            [":dart:"] = "\uD83C\uDFAF",
            [":dash:"] = "\uD83D\uDCA8",
            [":date:"] = "\uD83D\uDCC5",
            [":deaf_man:"] = "\uD83E\uDDCF\u200D♂️",
            [":deaf_man::skin-tone-1:"] = "\uD83E\uDDCF\uD83C\uDFFB\u200D♂️",
            [":deaf_man::skin-tone-2:"] = "\uD83E\uDDCF\uD83C\uDFFC\u200D♂️",
            [":deaf_man::skin-tone-3:"] = "\uD83E\uDDCF\uD83C\uDFFD\u200D♂️",
            [":deaf_man::skin-tone-4:"] = "\uD83E\uDDCF\uD83C\uDFFE\u200D♂️",
            [":deaf_man::skin-tone-5:"] = "\uD83E\uDDCF\uD83C\uDFFF\u200D♂️",
            [":deaf_man_dark_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFF\u200D♂️",
            [":deaf_man_light_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFB\u200D♂️",
            [":deaf_man_medium_dark_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFE\u200D♂️",
            [":deaf_man_medium_light_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFC\u200D♂️",
            [":deaf_man_medium_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFD\u200D♂️",
            [":deaf_man_tone1:"] = "\uD83E\uDDCF\uD83C\uDFFB\u200D♂️",
            [":deaf_man_tone2:"] = "\uD83E\uDDCF\uD83C\uDFFC\u200D♂️",
            [":deaf_man_tone3:"] = "\uD83E\uDDCF\uD83C\uDFFD\u200D♂️",
            [":deaf_man_tone4:"] = "\uD83E\uDDCF\uD83C\uDFFE\u200D♂️",
            [":deaf_man_tone5:"] = "\uD83E\uDDCF\uD83C\uDFFF\u200D♂️",
            [":deaf_person:"] = "\uD83E\uDDCF",
            [":deaf_person::skin-tone-1:"] = "\uD83E\uDDCF\uD83C\uDFFB",
            [":deaf_person::skin-tone-2:"] = "\uD83E\uDDCF\uD83C\uDFFC",
            [":deaf_person::skin-tone-3:"] = "\uD83E\uDDCF\uD83C\uDFFD",
            [":deaf_person::skin-tone-4:"] = "\uD83E\uDDCF\uD83C\uDFFE",
            [":deaf_person::skin-tone-5:"] = "\uD83E\uDDCF\uD83C\uDFFF",
            [":deaf_person_dark_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFF",
            [":deaf_person_light_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFB",
            [":deaf_person_medium_dark_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFE",
            [":deaf_person_medium_light_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFC",
            [":deaf_person_medium_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFD",
            [":deaf_person_tone1:"] = "\uD83E\uDDCF\uD83C\uDFFB",
            [":deaf_person_tone2:"] = "\uD83E\uDDCF\uD83C\uDFFC",
            [":deaf_person_tone3:"] = "\uD83E\uDDCF\uD83C\uDFFD",
            [":deaf_person_tone4:"] = "\uD83E\uDDCF\uD83C\uDFFE",
            [":deaf_person_tone5:"] = "\uD83E\uDDCF\uD83C\uDFFF",
            [":deaf_woman:"] = "\uD83E\uDDCF\u200D♀️",
            [":deaf_woman::skin-tone-1:"] = "\uD83E\uDDCF\uD83C\uDFFB\u200D♀️",
            [":deaf_woman::skin-tone-2:"] = "\uD83E\uDDCF\uD83C\uDFFC\u200D♀️",
            [":deaf_woman::skin-tone-3:"] = "\uD83E\uDDCF\uD83C\uDFFD\u200D♀️",
            [":deaf_woman::skin-tone-4:"] = "\uD83E\uDDCF\uD83C\uDFFE\u200D♀️",
            [":deaf_woman::skin-tone-5:"] = "\uD83E\uDDCF\uD83C\uDFFF\u200D♀️",
            [":deaf_woman_dark_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFF\u200D♀️",
            [":deaf_woman_light_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFB\u200D♀️",
            [":deaf_woman_medium_dark_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFE\u200D♀️",
            [":deaf_woman_medium_light_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFC\u200D♀️",
            [":deaf_woman_medium_skin_tone:"] = "\uD83E\uDDCF\uD83C\uDFFD\u200D♀️",
            [":deaf_woman_tone1:"] = "\uD83E\uDDCF\uD83C\uDFFB\u200D♀️",
            [":deaf_woman_tone2:"] = "\uD83E\uDDCF\uD83C\uDFFC\u200D♀️",
            [":deaf_woman_tone3:"] = "\uD83E\uDDCF\uD83C\uDFFD\u200D♀️",
            [":deaf_woman_tone4:"] = "\uD83E\uDDCF\uD83C\uDFFE\u200D♀️",
            [":deaf_woman_tone5:"] = "\uD83E\uDDCF\uD83C\uDFFF\u200D♀️",
            [":deciduous_tree:"] = "\uD83C\uDF33",
            [":deer:"] = "\uD83E\uDD8C",
            [":department_store:"] = "\uD83C\uDFEC",
            [":derelict_house_building:"] = "\uD83C\uDFDA️",
            [":desert:"] = "\uD83C\uDFDC️",
            [":desert_island:"] = "\uD83C\uDFDD️",
            [":desktop:"] = "\uD83D\uDDA5️",
            [":desktop_computer:"] = "\uD83D\uDDA5️",
            [":detective:"] = "\uD83D\uDD75️",
            [":detective::skin-tone-1:"] = "\uD83D\uDD75\uD83C\uDFFB",
            [":detective::skin-tone-2:"] = "\uD83D\uDD75\uD83C\uDFFC",
            [":detective::skin-tone-3:"] = "\uD83D\uDD75\uD83C\uDFFD",
            [":detective::skin-tone-4:"] = "\uD83D\uDD75\uD83C\uDFFE",
            [":detective::skin-tone-5:"] = "\uD83D\uDD75\uD83C\uDFFF",
            [":detective_tone1:"] = "\uD83D\uDD75\uD83C\uDFFB",
            [":detective_tone2:"] = "\uD83D\uDD75\uD83C\uDFFC",
            [":detective_tone3:"] = "\uD83D\uDD75\uD83C\uDFFD",
            [":detective_tone4:"] = "\uD83D\uDD75\uD83C\uDFFE",
            [":detective_tone5:"] = "\uD83D\uDD75\uD83C\uDFFF",
            [":diamond_shape_with_a_dot_inside:"] = "\uD83D\uDCA0",
            [":diamonds:"] = "♦️",
            [":disappointed:"] = "\uD83D\uDE1E",
            [":disappointed_relieved:"] = "\uD83D\uDE25",
            [":disguised_face:"] = "\uD83E\uDD78",
            [":disguised_face:"] = "\uD83E\uDD78",
            [":dividers:"] = "\uD83D\uDDC2️",
            [":diving_mask:"] = "\uD83E\uDD3F",
            [":diya_lamp:"] = "\uD83E\uDE94",
            [":dizzy:"] = "\uD83D\uDCAB",
            [":dizzy_face:"] = "\uD83D\uDE35",
            [":dna:"] = "\uD83E\uDDEC",
            [":do_not_litter:"] = "\uD83D\uDEAF",
            [":dodo:"] = "\uD83E\uDDA4",
            [":dog2:"] = "\uD83D\uDC15",
            [":dog:"] = "\uD83D\uDC36",
            [":dollar:"] = "\uD83D\uDCB5",
            [":dolls:"] = "\uD83C\uDF8E",
            [":dolphin:"] = "\uD83D\uDC2C",
            [":door:"] = "\uD83D\uDEAA",
            [":dotted_line_face:"] = "\uD83E\uDEE5",
            [":double_vertical_bar:"] = "⏸️",
            [":doughnut:"] = "\uD83C\uDF69",
            [":dove:"] = "\uD83D\uDD4A️",
            [":dove_of_peace:"] = "\uD83D\uDD4A️",
            [":dragon:"] = "\uD83D\uDC09",
            [":dragon_face:"] = "\uD83D\uDC32",
            [":dress:"] = "\uD83D\uDC57",
            [":dromedary_camel:"] = "\uD83D\uDC2A",
            [":drool:"] = "\uD83E\uDD24",
            [":drooling_face:"] = "\uD83E\uDD24",
            [":drop_of_blood:"] = "\uD83E\uDE78",
            [":droplet:"] = "\uD83D\uDCA7",
            [":drum:"] = "\uD83E\uDD41",
            [":drum_with_drumsticks:"] = "\uD83E\uDD41",
            [":duck:"] = "\uD83E\uDD86",
            [":dumpling:"] = "\uD83E\uDD5F",
            [":dvd:"] = "\uD83D\uDCC0",
            [":e_mail:"] = "\uD83D\uDCE7",
            [":eagle:"] = "\uD83E\uDD85",
            [":ear:"] = "\uD83D\uDC42",
            [":ear::skin-tone-1:"] = "\uD83D\uDC42\uD83C\uDFFB",
            [":ear::skin-tone-2:"] = "\uD83D\uDC42\uD83C\uDFFC",
            [":ear::skin-tone-3:"] = "\uD83D\uDC42\uD83C\uDFFD",
            [":ear::skin-tone-4:"] = "\uD83D\uDC42\uD83C\uDFFE",
            [":ear::skin-tone-5:"] = "\uD83D\uDC42\uD83C\uDFFF",
            [":ear_of_rice:"] = "\uD83C\uDF3E",
            [":ear_tone1:"] = "\uD83D\uDC42\uD83C\uDFFB",
            [":ear_tone2:"] = "\uD83D\uDC42\uD83C\uDFFC",
            [":ear_tone3:"] = "\uD83D\uDC42\uD83C\uDFFD",
            [":ear_tone4:"] = "\uD83D\uDC42\uD83C\uDFFE",
            [":ear_tone5:"] = "\uD83D\uDC42\uD83C\uDFFF",
            [":ear_with_hearing_aid:"] = "\uD83E\uDDBB",
            [":ear_with_hearing_aid::skin-tone-1:"] = "\uD83E\uDDBB\uD83C\uDFFB",
            [":ear_with_hearing_aid::skin-tone-2:"] = "\uD83E\uDDBB\uD83C\uDFFC",
            [":ear_with_hearing_aid::skin-tone-3:"] = "\uD83E\uDDBB\uD83C\uDFFD",
            [":ear_with_hearing_aid::skin-tone-4:"] = "\uD83E\uDDBB\uD83C\uDFFE",
            [":ear_with_hearing_aid::skin-tone-5:"] = "\uD83E\uDDBB\uD83C\uDFFF",
            [":ear_with_hearing_aid_dark_skin_tone:"] = "\uD83E\uDDBB\uD83C\uDFFF",
            [":ear_with_hearing_aid_light_skin_tone:"] = "\uD83E\uDDBB\uD83C\uDFFB",
            [":ear_with_hearing_aid_medium_dark_skin_tone:"] = "\uD83E\uDDBB\uD83C\uDFFE",
            [":ear_with_hearing_aid_medium_light_skin_tone:"] = "\uD83E\uDDBB\uD83C\uDFFC",
            [":ear_with_hearing_aid_medium_skin_tone:"] = "\uD83E\uDDBB\uD83C\uDFFD",
            [":ear_with_hearing_aid_tone1:"] = "\uD83E\uDDBB\uD83C\uDFFB",
            [":ear_with_hearing_aid_tone2:"] = "\uD83E\uDDBB\uD83C\uDFFC",
            [":ear_with_hearing_aid_tone3:"] = "\uD83E\uDDBB\uD83C\uDFFD",
            [":ear_with_hearing_aid_tone4:"] = "\uD83E\uDDBB\uD83C\uDFFE",
            [":ear_with_hearing_aid_tone5:"] = "\uD83E\uDDBB\uD83C\uDFFF",
            [":earth_africa:"] = "\uD83C\uDF0D",
            [":earth_americas:"] = "\uD83C\uDF0E",
            [":earth_asia:"] = "\uD83C\uDF0F",
            [":egg:"] = "\uD83E\uDD5A",
            [":eggplant:"] = "\uD83C\uDF46",
            [":eight:"] = "8️⃣",
            [":eight_pointed_black_star:"] = "✴️",
            [":eight_spoked_asterisk:"] = "✳️",
            [":eject:"] = "⏏️",
            [":eject_symbol:"] = "⏏️",
            [":electric_plug:"] = "\uD83D\uDD0C",
            [":elephant:"] = "\uD83D\uDC18",
            [":elevator:"] = "\uD83D\uDED7",
            [":elf:"] = "\uD83E\uDDDD",
            [":elf::skin-tone-1:"] = "\uD83E\uDDDD\uD83C\uDFFB",
            [":elf::skin-tone-2:"] = "\uD83E\uDDDD\uD83C\uDFFC",
            [":elf::skin-tone-3:"] = "\uD83E\uDDDD\uD83C\uDFFD",
            [":elf::skin-tone-4:"] = "\uD83E\uDDDD\uD83C\uDFFE",
            [":elf::skin-tone-5:"] = "\uD83E\uDDDD\uD83C\uDFFF",
            [":elf_dark_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFF",
            [":elf_light_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFB",
            [":elf_medium_dark_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFE",
            [":elf_medium_light_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFC",
            [":elf_medium_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFD",
            [":elf_tone1:"] = "\uD83E\uDDDD\uD83C\uDFFB",
            [":elf_tone2:"] = "\uD83E\uDDDD\uD83C\uDFFC",
            [":elf_tone3:"] = "\uD83E\uDDDD\uD83C\uDFFD",
            [":elf_tone4:"] = "\uD83E\uDDDD\uD83C\uDFFE",
            [":elf_tone5:"] = "\uD83E\uDDDD\uD83C\uDFFF",
            [":email:"] = "\uD83D\uDCE7",
            [":empty_nest:"] = "\uD83E\uDEB9",
            [":end:"] = "\uD83D\uDD1A",
            [":england:"] = "\uD83C\uDFF4\uDB40\uDC67\uDB40\uDC62\uDB40\uDC65\uDB40\uDC6E\uDB40\uDC67\uDB40\uDC7F",
            [":envelope:"] = "✉️",
            [":envelope_with_arrow:"] = "\uD83D\uDCE9",
            [":euro:"] = "\uD83D\uDCB6",
            [":european_castle:"] = "\uD83C\uDFF0",
            [":european_post_office:"] = "\uD83C\uDFE4",
            [":evergreen_tree:"] = "\uD83C\uDF32",
            [":exclamation:"] = "❗",
            [":expecting_woman:"] = "\uD83E\uDD30",
            [":expecting_woman::skin-tone-1:"] = "\uD83E\uDD30\uD83C\uDFFB",
            [":expecting_woman::skin-tone-2:"] = "\uD83E\uDD30\uD83C\uDFFC",
            [":expecting_woman::skin-tone-3:"] = "\uD83E\uDD30\uD83C\uDFFD",
            [":expecting_woman::skin-tone-4:"] = "\uD83E\uDD30\uD83C\uDFFE",
            [":expecting_woman::skin-tone-5:"] = "\uD83E\uDD30\uD83C\uDFFF",
            [":expecting_woman_tone1:"] = "\uD83E\uDD30\uD83C\uDFFB",
            [":expecting_woman_tone2:"] = "\uD83E\uDD30\uD83C\uDFFC",
            [":expecting_woman_tone3:"] = "\uD83E\uDD30\uD83C\uDFFD",
            [":expecting_woman_tone4:"] = "\uD83E\uDD30\uD83C\uDFFE",
            [":expecting_woman_tone5:"] = "\uD83E\uDD30\uD83C\uDFFF",
            [":exploding_head:"] = "\uD83E\uDD2F",
            [":expressionless:"] = "\uD83D\uDE11",
            [":eye:"] = "\uD83D\uDC41️",
            [":eye_in_speech_bubble:"] = "\uD83D\uDC41\u200D\uD83D\uDDE8",
            [":eyeglasses:"] = "\uD83D\uDC53",
            [":eyes:"] = "\uD83D\uDC40",
            [":face_exhaling:"] = "\uD83D\uDE2E\u200D\uD83D\uDCA8",
            [":face_holding_back_tears:"] = "\uD83E\uDD79",
            [":face_in_clouds:"] = "\uD83D\uDE36\u200D\uD83C\uDF2B\uFE0F",
            [":face_palm:"] = "\uD83E\uDD26",
            [":face_palm::skin-tone-1:"] = "\uD83E\uDD26\uD83C\uDFFB",
            [":face_palm::skin-tone-2:"] = "\uD83E\uDD26\uD83C\uDFFC",
            [":face_palm::skin-tone-3:"] = "\uD83E\uDD26\uD83C\uDFFD",
            [":face_palm::skin-tone-4:"] = "\uD83E\uDD26\uD83C\uDFFE",
            [":face_palm::skin-tone-5:"] = "\uD83E\uDD26\uD83C\uDFFF",
            [":face_palm_tone1:"] = "\uD83E\uDD26\uD83C\uDFFB",
            [":face_palm_tone2:"] = "\uD83E\uDD26\uD83C\uDFFC",
            [":face_palm_tone3:"] = "\uD83E\uDD26\uD83C\uDFFD",
            [":face_palm_tone4:"] = "\uD83E\uDD26\uD83C\uDFFE",
            [":face_palm_tone5:"] = "\uD83E\uDD26\uD83C\uDFFF",
            [":face_vomiting:"] = "\uD83E\uDD2E",
            [":face_with_cowboy_hat:"] = "\uD83E\uDD20",
            [":face_with_diagonal_mouth:"] = "\uD83E\uDEE4",
            [":face_with_hand_over_mouth:"] = "\uD83E\uDD2D",
            [":face_with_head_bandage:"] = "\uD83E\uDD15",
            [":face_with_monocle:"] = "\uD83E\uDDD0",
            [":face_with_open_eyes_and_hand_over_mouth:"] = "\uD83E\uDEE0",
            [":face_with_peeking_eye:"] = "\uD83E\uDEE3",
            [":face_with_raised_eyebrow:"] = "\uD83E\uDD28",
            [":face_with_rolling_eyes:"] = "\uD83D\uDE44",
            [":face_with_spiral_eyes:"] = "\uD83D\uDE35\u200D\uD83D\uDCAB",
            [":face_with_symbols_over_mouth:"] = "\uD83E\uDD2C",
            [":face_with_thermometer:"] = "\uD83E\uDD12",
            [":facepalm:"] = "\uD83E\uDD26",
            [":facepalm::skin-tone-1:"] = "\uD83E\uDD26\uD83C\uDFFB",
            [":facepalm::skin-tone-2:"] = "\uD83E\uDD26\uD83C\uDFFC",
            [":facepalm::skin-tone-3:"] = "\uD83E\uDD26\uD83C\uDFFD",
            [":facepalm::skin-tone-4:"] = "\uD83E\uDD26\uD83C\uDFFE",
            [":facepalm::skin-tone-5:"] = "\uD83E\uDD26\uD83C\uDFFF",
            [":facepalm_tone1:"] = "\uD83E\uDD26\uD83C\uDFFB",
            [":facepalm_tone2:"] = "\uD83E\uDD26\uD83C\uDFFC",
            [":facepalm_tone3:"] = "\uD83E\uDD26\uD83C\uDFFD",
            [":facepalm_tone4:"] = "\uD83E\uDD26\uD83C\uDFFE",
            [":facepalm_tone5:"] = "\uD83E\uDD26\uD83C\uDFFF",
            [":factory:"] = "\uD83C\uDFED",
            [":fairy:"] = "\uD83E\uDDDA",
            [":fairy::skin-tone-1:"] = "\uD83E\uDDDA\uD83C\uDFFB",
            [":fairy::skin-tone-2:"] = "\uD83E\uDDDA\uD83C\uDFFC",
            [":fairy::skin-tone-3:"] = "\uD83E\uDDDA\uD83C\uDFFD",
            [":fairy::skin-tone-4:"] = "\uD83E\uDDDA\uD83C\uDFFE",
            [":fairy::skin-tone-5:"] = "\uD83E\uDDDA\uD83C\uDFFF",
            [":fairy_dark_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFF",
            [":fairy_light_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFB",
            [":fairy_medium_dark_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFE",
            [":fairy_medium_light_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFC",
            [":fairy_medium_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFD",
            [":fairy_tone1:"] = "\uD83E\uDDDA\uD83C\uDFFB",
            [":fairy_tone2:"] = "\uD83E\uDDDA\uD83C\uDFFC",
            [":fairy_tone3:"] = "\uD83E\uDDDA\uD83C\uDFFD",
            [":fairy_tone4:"] = "\uD83E\uDDDA\uD83C\uDFFE",
            [":fairy_tone5:"] = "\uD83E\uDDDA\uD83C\uDFFF",
            [":falafel:"] = "\uD83E\uDDC6",
            [":fallen_leaf:"] = "\uD83C\uDF42",
            [":family:"] = "\uD83D\uDC6A",
            [":family_man_boy:"] = "\uD83D\uDC68\u200D\uD83D\uDC66",
            [":family_man_boy_boy:"] = "\uD83D\uDC68\u200D\uD83D\uDC66\u200D\uD83D\uDC66",
            [":family_man_girl:"] = "\uD83D\uDC68\u200D\uD83D\uDC67",
            [":family_man_girl_boy:"] = "\uD83D\uDC68\u200D\uD83D\uDC67\u200D\uD83D\uDC66",
            [":family_man_girl_girl:"] = "\uD83D\uDC68\u200D\uD83D\uDC67\u200D\uD83D\uDC67",
            [":family_man_woman_boy:"] = "\uD83D\uDC68\u200D\uD83D\uDC69\u200D\uD83D\uDC66",
            [":family_mmb:"] = "\uD83D\uDC68\u200D\uD83D\uDC68\u200D\uD83D\uDC66",
            [":family_mmbb:"] = "\uD83D\uDC68\u200D\uD83D\uDC68\u200D\uD83D\uDC66\u200D\uD83D\uDC66",
            [":family_mmg:"] = "\uD83D\uDC68\u200D\uD83D\uDC68\u200D\uD83D\uDC67",
            [":family_mmgb:"] = "\uD83D\uDC68\u200D\uD83D\uDC68\u200D\uD83D\uDC67\u200D\uD83D\uDC66",
            [":family_mmgg:"] = "\uD83D\uDC68\u200D\uD83D\uDC68\u200D\uD83D\uDC67\u200D\uD83D\uDC67",
            [":family_mwbb:"] = "\uD83D\uDC68\u200D\uD83D\uDC69\u200D\uD83D\uDC66\u200D\uD83D\uDC66",
            [":family_mwg:"] = "\uD83D\uDC68\u200D\uD83D\uDC69\u200D\uD83D\uDC67",
            [":family_mwgb:"] = "\uD83D\uDC68\u200D\uD83D\uDC69\u200D\uD83D\uDC67\u200D\uD83D\uDC66",
            [":family_mwgg:"] = "\uD83D\uDC68\u200D\uD83D\uDC69\u200D\uD83D\uDC67\u200D\uD83D\uDC67",
            [":family_woman_boy:"] = "\uD83D\uDC69\u200D\uD83D\uDC66",
            [":family_woman_boy_boy:"] = "\uD83D\uDC69\u200D\uD83D\uDC66\u200D\uD83D\uDC66",
            [":family_woman_girl:"] = "\uD83D\uDC69\u200D\uD83D\uDC67",
            [":family_woman_girl_boy:"] = "\uD83D\uDC69\u200D\uD83D\uDC67\u200D\uD83D\uDC66",
            [":family_woman_girl_girl:"] = "\uD83D\uDC69\u200D\uD83D\uDC67\u200D\uD83D\uDC67",
            [":family_wwb:"] = "\uD83D\uDC69\u200D\uD83D\uDC69\u200D\uD83D\uDC66",
            [":family_wwbb:"] = "\uD83D\uDC69\u200D\uD83D\uDC69\u200D\uD83D\uDC66\u200D\uD83D\uDC66",
            [":family_wwg:"] = "\uD83D\uDC69\u200D\uD83D\uDC69\u200D\uD83D\uDC67",
            [":family_wwgb:"] = "\uD83D\uDC69\u200D\uD83D\uDC69\u200D\uD83D\uDC67\u200D\uD83D\uDC66",
            [":family_wwgg:"] = "\uD83D\uDC69\u200D\uD83D\uDC69\u200D\uD83D\uDC67\u200D\uD83D\uDC67",
            [":fast_forward:"] = "⏩",
            [":fax:"] = "\uD83D\uDCE0",
            [":fearful:"] = "\uD83D\uDE28",
            [":feather:"] = "\uD83E\uDEB6",
            [":feet:"] = "\uD83D\uDC3E",
            [":female_sign:"] = "♀️",
            [":fencer:"] = "\uD83E\uDD3A",
            [":fencing:"] = "\uD83E\uDD3A",
            [":ferris_wheel:"] = "\uD83C\uDFA1",
            [":ferry:"] = "⛴️",
            [":field_hockey:"] = "\uD83C\uDFD1",
            [":file_cabinet:"] = "\uD83D\uDDC4️",
            [":file_folder:"] = "\uD83D\uDCC1",
            [":film_frames:"] = "\uD83C\uDF9E️",
            [":film_projector:"] = "\uD83D\uDCFD️",
            [":fingers_crossed:"] = "\uD83E\uDD1E",
            [":fingers_crossed::skin-tone-1:"] = "\uD83E\uDD1E\uD83C\uDFFB",
            [":fingers_crossed::skin-tone-2:"] = "\uD83E\uDD1E\uD83C\uDFFC",
            [":fingers_crossed::skin-tone-3:"] = "\uD83E\uDD1E\uD83C\uDFFD",
            [":fingers_crossed::skin-tone-4:"] = "\uD83E\uDD1E\uD83C\uDFFE",
            [":fingers_crossed::skin-tone-5:"] = "\uD83E\uDD1E\uD83C\uDFFF",
            [":fingers_crossed_tone1:"] = "\uD83E\uDD1E\uD83C\uDFFB",
            [":fingers_crossed_tone2:"] = "\uD83E\uDD1E\uD83C\uDFFC",
            [":fingers_crossed_tone3:"] = "\uD83E\uDD1E\uD83C\uDFFD",
            [":fingers_crossed_tone4:"] = "\uD83E\uDD1E\uD83C\uDFFE",
            [":fingers_crossed_tone5:"] = "\uD83E\uDD1E\uD83C\uDFFF",
            [":fire:"] = "\uD83D\uDD25",
            [":fire_engine:"] = "\uD83D\uDE92",
            [":fire_extinguisher:"] = "\uD83E\uDDEF",
            [":firecracker:"] = "\uD83E\uDDE8",
            [":fireworks:"] = "\uD83C\uDF86",
            [":first_place:"] = "\uD83E\uDD47",
            [":first_place_medal:"] = "\uD83E\uDD47",
            [":first_quarter_moon:"] = "\uD83C\uDF13",
            [":first_quarter_moon_with_face:"] = "\uD83C\uDF1B",
            [":fish:"] = "\uD83D\uDC1F",
            [":fish_cake:"] = "\uD83C\uDF65",
            [":fishing_pole_and_fish:"] = "\uD83C\uDFA3",
            [":fist:"] = "✊",
            [":fist::skin-tone-1:"] = "✊\uD83C\uDFFB",
            [":fist::skin-tone-2:"] = "✊\uD83C\uDFFC",
            [":fist::skin-tone-3:"] = "✊\uD83C\uDFFD",
            [":fist::skin-tone-4:"] = "✊\uD83C\uDFFE",
            [":fist::skin-tone-5:"] = "✊\uD83C\uDFFF",
            [":fist_tone1:"] = "✊\uD83C\uDFFB",
            [":fist_tone2:"] = "✊\uD83C\uDFFC",
            [":fist_tone3:"] = "✊\uD83C\uDFFD",
            [":fist_tone4:"] = "✊\uD83C\uDFFE",
            [":fist_tone5:"] = "✊\uD83C\uDFFF",
            [":five:"] = "5️⃣",
            [":flag_ac:"] = "\uD83C\uDDE6\uD83C\uDDE8",
            [":flag_ad:"] = "\uD83C\uDDE6\uD83C\uDDE9",
            [":flag_ae:"] = "\uD83C\uDDE6\uD83C\uDDEA",
            [":flag_af:"] = "\uD83C\uDDE6\uD83C\uDDEB",
            [":flag_ag:"] = "\uD83C\uDDE6\uD83C\uDDEC",
            [":flag_ai:"] = "\uD83C\uDDE6\uD83C\uDDEE",
            [":flag_al:"] = "\uD83C\uDDE6\uD83C\uDDF1",
            [":flag_am:"] = "\uD83C\uDDE6\uD83C\uDDF2",
            [":flag_ao:"] = "\uD83C\uDDE6\uD83C\uDDF4",
            [":flag_aq:"] = "\uD83C\uDDE6\uD83C\uDDF6",
            [":flag_ar:"] = "\uD83C\uDDE6\uD83C\uDDF7",
            [":flag_as:"] = "\uD83C\uDDE6\uD83C\uDDF8",
            [":flag_at:"] = "\uD83C\uDDE6\uD83C\uDDF9",
            [":flag_au:"] = "\uD83C\uDDE6\uD83C\uDDFA",
            [":flag_aw:"] = "\uD83C\uDDE6\uD83C\uDDFC",
            [":flag_ax:"] = "\uD83C\uDDE6\uD83C\uDDFD",
            [":flag_az:"] = "\uD83C\uDDE6\uD83C\uDDFF",
            [":flag_ba:"] = "\uD83C\uDDE7\uD83C\uDDE6",
            [":flag_bb:"] = "\uD83C\uDDE7\uD83C\uDDE7",
            [":flag_bd:"] = "\uD83C\uDDE7\uD83C\uDDE9",
            [":flag_be:"] = "\uD83C\uDDE7\uD83C\uDDEA",
            [":flag_bf:"] = "\uD83C\uDDE7\uD83C\uDDEB",
            [":flag_bg:"] = "\uD83C\uDDE7\uD83C\uDDEC",
            [":flag_bh:"] = "\uD83C\uDDE7\uD83C\uDDED",
            [":flag_bi:"] = "\uD83C\uDDE7\uD83C\uDDEE",
            [":flag_bj:"] = "\uD83C\uDDE7\uD83C\uDDEF",
            [":flag_bl:"] = "\uD83C\uDDE7\uD83C\uDDF1",
            [":flag_black:"] = "\uD83C\uDFF4",
            [":flag_bm:"] = "\uD83C\uDDE7\uD83C\uDDF2",
            [":flag_bn:"] = "\uD83C\uDDE7\uD83C\uDDF3",
            [":flag_bo:"] = "\uD83C\uDDE7\uD83C\uDDF4",
            [":flag_bq:"] = "\uD83C\uDDE7\uD83C\uDDF6",
            [":flag_br:"] = "\uD83C\uDDE7\uD83C\uDDF7",
            [":flag_bs:"] = "\uD83C\uDDE7\uD83C\uDDF8",
            [":flag_bt:"] = "\uD83C\uDDE7\uD83C\uDDF9",
            [":flag_bv:"] = "\uD83C\uDDE7\uD83C\uDDFB",
            [":flag_bw:"] = "\uD83C\uDDE7\uD83C\uDDFC",
            [":flag_by:"] = "\uD83C\uDDE7\uD83C\uDDFE",
            [":flag_bz:"] = "\uD83C\uDDE7\uD83C\uDDFF",
            [":flag_ca:"] = "\uD83C\uDDE8\uD83C\uDDE6",
            [":flag_cc:"] = "\uD83C\uDDE8\uD83C\uDDE8",
            [":flag_cd:"] = "\uD83C\uDDE8\uD83C\uDDE9",
            [":flag_cf:"] = "\uD83C\uDDE8\uD83C\uDDEB",
            [":flag_cg:"] = "\uD83C\uDDE8\uD83C\uDDEC",
            [":flag_ch:"] = "\uD83C\uDDE8\uD83C\uDDED",
            [":flag_ci:"] = "\uD83C\uDDE8\uD83C\uDDEE",
            [":flag_ck:"] = "\uD83C\uDDE8\uD83C\uDDF0",
            [":flag_cl:"] = "\uD83C\uDDE8\uD83C\uDDF1",
            [":flag_cm:"] = "\uD83C\uDDE8\uD83C\uDDF2",
            [":flag_cn:"] = "\uD83C\uDDE8\uD83C\uDDF3",
            [":flag_co:"] = "\uD83C\uDDE8\uD83C\uDDF4",
            [":flag_cp:"] = "\uD83C\uDDE8\uD83C\uDDF5",
            [":flag_cr:"] = "\uD83C\uDDE8\uD83C\uDDF7",
            [":flag_cu:"] = "\uD83C\uDDE8\uD83C\uDDFA",
            [":flag_cv:"] = "\uD83C\uDDE8\uD83C\uDDFB",
            [":flag_cw:"] = "\uD83C\uDDE8\uD83C\uDDFC",
            [":flag_cx:"] = "\uD83C\uDDE8\uD83C\uDDFD",
            [":flag_cy:"] = "\uD83C\uDDE8\uD83C\uDDFE",
            [":flag_cz:"] = "\uD83C\uDDE8\uD83C\uDDFF",
            [":flag_de:"] = "\uD83C\uDDE9\uD83C\uDDEA",
            [":flag_dg:"] = "\uD83C\uDDE9\uD83C\uDDEC",
            [":flag_dj:"] = "\uD83C\uDDE9\uD83C\uDDEF",
            [":flag_dk:"] = "\uD83C\uDDE9\uD83C\uDDF0",
            [":flag_dm:"] = "\uD83C\uDDE9\uD83C\uDDF2",
            [":flag_do:"] = "\uD83C\uDDE9\uD83C\uDDF4",
            [":flag_dz:"] = "\uD83C\uDDE9\uD83C\uDDFF",
            [":flag_ea:"] = "\uD83C\uDDEA\uD83C\uDDE6",
            [":flag_ec:"] = "\uD83C\uDDEA\uD83C\uDDE8",
            [":flag_ee:"] = "\uD83C\uDDEA\uD83C\uDDEA",
            [":flag_eg:"] = "\uD83C\uDDEA\uD83C\uDDEC",
            [":flag_eh:"] = "\uD83C\uDDEA\uD83C\uDDED",
            [":flag_er:"] = "\uD83C\uDDEA\uD83C\uDDF7",
            [":flag_es:"] = "\uD83C\uDDEA\uD83C\uDDF8",
            [":flag_et:"] = "\uD83C\uDDEA\uD83C\uDDF9",
            [":flag_eu:"] = "\uD83C\uDDEA\uD83C\uDDFA",
            [":flag_fi:"] = "\uD83C\uDDEB\uD83C\uDDEE",
            [":flag_fj:"] = "\uD83C\uDDEB\uD83C\uDDEF",
            [":flag_fk:"] = "\uD83C\uDDEB\uD83C\uDDF0",
            [":flag_fm:"] = "\uD83C\uDDEB\uD83C\uDDF2",
            [":flag_fo:"] = "\uD83C\uDDEB\uD83C\uDDF4",
            [":flag_fr:"] = "\uD83C\uDDEB\uD83C\uDDF7",
            [":flag_ga:"] = "\uD83C\uDDEC\uD83C\uDDE6",
            [":flag_gb:"] = "\uD83C\uDDEC\uD83C\uDDE7",
            [":flag_gd:"] = "\uD83C\uDDEC\uD83C\uDDE9",
            [":flag_ge:"] = "\uD83C\uDDEC\uD83C\uDDEA",
            [":flag_gf:"] = "\uD83C\uDDEC\uD83C\uDDEB",
            [":flag_gg:"] = "\uD83C\uDDEC\uD83C\uDDEC",
            [":flag_gh:"] = "\uD83C\uDDEC\uD83C\uDDED",
            [":flag_gi:"] = "\uD83C\uDDEC\uD83C\uDDEE",
            [":flag_gl:"] = "\uD83C\uDDEC\uD83C\uDDF1",
            [":flag_gm:"] = "\uD83C\uDDEC\uD83C\uDDF2",
            [":flag_gn:"] = "\uD83C\uDDEC\uD83C\uDDF3",
            [":flag_gp:"] = "\uD83C\uDDEC\uD83C\uDDF5",
            [":flag_gq:"] = "\uD83C\uDDEC\uD83C\uDDF6",
            [":flag_gr:"] = "\uD83C\uDDEC\uD83C\uDDF7",
            [":flag_gs:"] = "\uD83C\uDDEC\uD83C\uDDF8",
            [":flag_gt:"] = "\uD83C\uDDEC\uD83C\uDDF9",
            [":flag_gu:"] = "\uD83C\uDDEC\uD83C\uDDFA",
            [":flag_gw:"] = "\uD83C\uDDEC\uD83C\uDDFC",
            [":flag_gy:"] = "\uD83C\uDDEC\uD83C\uDDFE",
            [":flag_hk:"] = "\uD83C\uDDED\uD83C\uDDF0",
            [":flag_hm:"] = "\uD83C\uDDED\uD83C\uDDF2",
            [":flag_hn:"] = "\uD83C\uDDED\uD83C\uDDF3",
            [":flag_hr:"] = "\uD83C\uDDED\uD83C\uDDF7",
            [":flag_ht:"] = "\uD83C\uDDED\uD83C\uDDF9",
            [":flag_hu:"] = "\uD83C\uDDED\uD83C\uDDFA",
            [":flag_ic:"] = "\uD83C\uDDEE\uD83C\uDDE8",
            [":flag_id:"] = "\uD83C\uDDEE\uD83C\uDDE9",
            [":flag_ie:"] = "\uD83C\uDDEE\uD83C\uDDEA",
            [":flag_il:"] = "\uD83C\uDDEE\uD83C\uDDF1",
            [":flag_im:"] = "\uD83C\uDDEE\uD83C\uDDF2",
            [":flag_in:"] = "\uD83C\uDDEE\uD83C\uDDF3",
            [":flag_io:"] = "\uD83C\uDDEE\uD83C\uDDF4",
            [":flag_iq:"] = "\uD83C\uDDEE\uD83C\uDDF6",
            [":flag_ir:"] = "\uD83C\uDDEE\uD83C\uDDF7",
            [":flag_is:"] = "\uD83C\uDDEE\uD83C\uDDF8",
            [":flag_it:"] = "\uD83C\uDDEE\uD83C\uDDF9",
            [":flag_je:"] = "\uD83C\uDDEF\uD83C\uDDEA",
            [":flag_jm:"] = "\uD83C\uDDEF\uD83C\uDDF2",
            [":flag_jo:"] = "\uD83C\uDDEF\uD83C\uDDF4",
            [":flag_jp:"] = "\uD83C\uDDEF\uD83C\uDDF5",
            [":flag_ke:"] = "\uD83C\uDDF0\uD83C\uDDEA",
            [":flag_kg:"] = "\uD83C\uDDF0\uD83C\uDDEC",
            [":flag_kh:"] = "\uD83C\uDDF0\uD83C\uDDED",
            [":flag_ki:"] = "\uD83C\uDDF0\uD83C\uDDEE",
            [":flag_km:"] = "\uD83C\uDDF0\uD83C\uDDF2",
            [":flag_kn:"] = "\uD83C\uDDF0\uD83C\uDDF3",
            [":flag_kp:"] = "\uD83C\uDDF0\uD83C\uDDF5",
            [":flag_kr:"] = "\uD83C\uDDF0\uD83C\uDDF7",
            [":flag_kw:"] = "\uD83C\uDDF0\uD83C\uDDFC",
            [":flag_ky:"] = "\uD83C\uDDF0\uD83C\uDDFE",
            [":flag_kz:"] = "\uD83C\uDDF0\uD83C\uDDFF",
            [":flag_la:"] = "\uD83C\uDDF1\uD83C\uDDE6",
            [":flag_lb:"] = "\uD83C\uDDF1\uD83C\uDDE7",
            [":flag_lc:"] = "\uD83C\uDDF1\uD83C\uDDE8",
            [":flag_li:"] = "\uD83C\uDDF1\uD83C\uDDEE",
            [":flag_lk:"] = "\uD83C\uDDF1\uD83C\uDDF0",
            [":flag_lr:"] = "\uD83C\uDDF1\uD83C\uDDF7",
            [":flag_ls:"] = "\uD83C\uDDF1\uD83C\uDDF8",
            [":flag_lt:"] = "\uD83C\uDDF1\uD83C\uDDF9",
            [":flag_lu:"] = "\uD83C\uDDF1\uD83C\uDDFA",
            [":flag_lv:"] = "\uD83C\uDDF1\uD83C\uDDFB",
            [":flag_ly:"] = "\uD83C\uDDF1\uD83C\uDDFE",
            [":flag_ma:"] = "\uD83C\uDDF2\uD83C\uDDE6",
            [":flag_mc:"] = "\uD83C\uDDF2\uD83C\uDDE8",
            [":flag_md:"] = "\uD83C\uDDF2\uD83C\uDDE9",
            [":flag_me:"] = "\uD83C\uDDF2\uD83C\uDDEA",
            [":flag_mf:"] = "\uD83C\uDDF2\uD83C\uDDEB",
            [":flag_mg:"] = "\uD83C\uDDF2\uD83C\uDDEC",
            [":flag_mh:"] = "\uD83C\uDDF2\uD83C\uDDED",
            [":flag_mk:"] = "\uD83C\uDDF2\uD83C\uDDF0",
            [":flag_ml:"] = "\uD83C\uDDF2\uD83C\uDDF1",
            [":flag_mm:"] = "\uD83C\uDDF2\uD83C\uDDF2",
            [":flag_mn:"] = "\uD83C\uDDF2\uD83C\uDDF3",
            [":flag_mo:"] = "\uD83C\uDDF2\uD83C\uDDF4",
            [":flag_mp:"] = "\uD83C\uDDF2\uD83C\uDDF5",
            [":flag_mq:"] = "\uD83C\uDDF2\uD83C\uDDF6",
            [":flag_mr:"] = "\uD83C\uDDF2\uD83C\uDDF7",
            [":flag_ms:"] = "\uD83C\uDDF2\uD83C\uDDF8",
            [":flag_mt:"] = "\uD83C\uDDF2\uD83C\uDDF9",
            [":flag_mu:"] = "\uD83C\uDDF2\uD83C\uDDFA",
            [":flag_mv:"] = "\uD83C\uDDF2\uD83C\uDDFB",
            [":flag_mw:"] = "\uD83C\uDDF2\uD83C\uDDFC",
            [":flag_mx:"] = "\uD83C\uDDF2\uD83C\uDDFD",
            [":flag_my:"] = "\uD83C\uDDF2\uD83C\uDDFE",
            [":flag_mz:"] = "\uD83C\uDDF2\uD83C\uDDFF",
            [":flag_na:"] = "\uD83C\uDDF3\uD83C\uDDE6",
            [":flag_nc:"] = "\uD83C\uDDF3\uD83C\uDDE8",
            [":flag_ne:"] = "\uD83C\uDDF3\uD83C\uDDEA",
            [":flag_nf:"] = "\uD83C\uDDF3\uD83C\uDDEB",
            [":flag_ng:"] = "\uD83C\uDDF3\uD83C\uDDEC",
            [":flag_ni:"] = "\uD83C\uDDF3\uD83C\uDDEE",
            [":flag_nl:"] = "\uD83C\uDDF3\uD83C\uDDF1",
            [":flag_no:"] = "\uD83C\uDDF3\uD83C\uDDF4",
            [":flag_np:"] = "\uD83C\uDDF3\uD83C\uDDF5",
            [":flag_nr:"] = "\uD83C\uDDF3\uD83C\uDDF7",
            [":flag_nu:"] = "\uD83C\uDDF3\uD83C\uDDFA",
            [":flag_nz:"] = "\uD83C\uDDF3\uD83C\uDDFF",
            [":flag_om:"] = "\uD83C\uDDF4\uD83C\uDDF2",
            [":flag_pa:"] = "\uD83C\uDDF5\uD83C\uDDE6",
            [":flag_pe:"] = "\uD83C\uDDF5\uD83C\uDDEA",
            [":flag_pf:"] = "\uD83C\uDDF5\uD83C\uDDEB",
            [":flag_pg:"] = "\uD83C\uDDF5\uD83C\uDDEC",
            [":flag_ph:"] = "\uD83C\uDDF5\uD83C\uDDED",
            [":flag_pk:"] = "\uD83C\uDDF5\uD83C\uDDF0",
            [":flag_pl:"] = "\uD83C\uDDF5\uD83C\uDDF1",
            [":flag_pm:"] = "\uD83C\uDDF5\uD83C\uDDF2",
            [":flag_pn:"] = "\uD83C\uDDF5\uD83C\uDDF3",
            [":flag_pr:"] = "\uD83C\uDDF5\uD83C\uDDF7",
            [":flag_ps:"] = "\uD83C\uDDF5\uD83C\uDDF8",
            [":flag_pt:"] = "\uD83C\uDDF5\uD83C\uDDF9",
            [":flag_pw:"] = "\uD83C\uDDF5\uD83C\uDDFC",
            [":flag_py:"] = "\uD83C\uDDF5\uD83C\uDDFE",
            [":flag_qa:"] = "\uD83C\uDDF6\uD83C\uDDE6",
            [":flag_re:"] = "\uD83C\uDDF7\uD83C\uDDEA",
            [":flag_ro:"] = "\uD83C\uDDF7\uD83C\uDDF4",
            [":flag_rs:"] = "\uD83C\uDDF7\uD83C\uDDF8",
            [":flag_ru:"] = "\uD83C\uDDF7\uD83C\uDDFA",
            [":flag_rw:"] = "\uD83C\uDDF7\uD83C\uDDFC",
            [":flag_sa:"] = "\uD83C\uDDF8\uD83C\uDDE6",
            [":flag_sb:"] = "\uD83C\uDDF8\uD83C\uDDE7",
            [":flag_sc:"] = "\uD83C\uDDF8\uD83C\uDDE8",
            [":flag_sd:"] = "\uD83C\uDDF8\uD83C\uDDE9",
            [":flag_se:"] = "\uD83C\uDDF8\uD83C\uDDEA",
            [":flag_sg:"] = "\uD83C\uDDF8\uD83C\uDDEC",
            [":flag_sh:"] = "\uD83C\uDDF8\uD83C\uDDED",
            [":flag_si:"] = "\uD83C\uDDF8\uD83C\uDDEE",
            [":flag_sj:"] = "\uD83C\uDDF8\uD83C\uDDEF",
            [":flag_sk:"] = "\uD83C\uDDF8\uD83C\uDDF0",
            [":flag_sl:"] = "\uD83C\uDDF8\uD83C\uDDF1",
            [":flag_sm:"] = "\uD83C\uDDF8\uD83C\uDDF2",
            [":flag_sn:"] = "\uD83C\uDDF8\uD83C\uDDF3",
            [":flag_so:"] = "\uD83C\uDDF8\uD83C\uDDF4",
            [":flag_sr:"] = "\uD83C\uDDF8\uD83C\uDDF7",
            [":flag_ss:"] = "\uD83C\uDDF8\uD83C\uDDF8",
            [":flag_st:"] = "\uD83C\uDDF8\uD83C\uDDF9",
            [":flag_sv:"] = "\uD83C\uDDF8\uD83C\uDDFB",
            [":flag_sx:"] = "\uD83C\uDDF8\uD83C\uDDFD",
            [":flag_sy:"] = "\uD83C\uDDF8\uD83C\uDDFE",
            [":flag_sz:"] = "\uD83C\uDDF8\uD83C\uDDFF",
            [":flag_ta:"] = "\uD83C\uDDF9\uD83C\uDDE6",
            [":flag_tc:"] = "\uD83C\uDDF9\uD83C\uDDE8",
            [":flag_td:"] = "\uD83C\uDDF9\uD83C\uDDE9",
            [":flag_tf:"] = "\uD83C\uDDF9\uD83C\uDDEB",
            [":flag_tg:"] = "\uD83C\uDDF9\uD83C\uDDEC",
            [":flag_th:"] = "\uD83C\uDDF9\uD83C\uDDED",
            [":flag_tj:"] = "\uD83C\uDDF9\uD83C\uDDEF",
            [":flag_tk:"] = "\uD83C\uDDF9\uD83C\uDDF0",
            [":flag_tl:"] = "\uD83C\uDDF9\uD83C\uDDF1",
            [":flag_tm:"] = "\uD83C\uDDF9\uD83C\uDDF2",
            [":flag_tn:"] = "\uD83C\uDDF9\uD83C\uDDF3",
            [":flag_to:"] = "\uD83C\uDDF9\uD83C\uDDF4",
            [":flag_tr:"] = "\uD83C\uDDF9\uD83C\uDDF7",
            [":flag_tt:"] = "\uD83C\uDDF9\uD83C\uDDF9",
            [":flag_tv:"] = "\uD83C\uDDF9\uD83C\uDDFB",
            [":flag_tw:"] = "\uD83C\uDDF9\uD83C\uDDFC",
            [":flag_tz:"] = "\uD83C\uDDF9\uD83C\uDDFF",
            [":flag_ua:"] = "\uD83C\uDDFA\uD83C\uDDE6",
            [":flag_ug:"] = "\uD83C\uDDFA\uD83C\uDDEC",
            [":flag_um:"] = "\uD83C\uDDFA\uD83C\uDDF2",
            [":flag_us:"] = "\uD83C\uDDFA\uD83C\uDDF8",
            [":flag_uy:"] = "\uD83C\uDDFA\uD83C\uDDFE",
            [":flag_uz:"] = "\uD83C\uDDFA\uD83C\uDDFF",
            [":flag_va:"] = "\uD83C\uDDFB\uD83C\uDDE6",
            [":flag_vc:"] = "\uD83C\uDDFB\uD83C\uDDE8",
            [":flag_ve:"] = "\uD83C\uDDFB\uD83C\uDDEA",
            [":flag_vg:"] = "\uD83C\uDDFB\uD83C\uDDEC",
            [":flag_vi:"] = "\uD83C\uDDFB\uD83C\uDDEE",
            [":flag_vn:"] = "\uD83C\uDDFB\uD83C\uDDF3",
            [":flag_vu:"] = "\uD83C\uDDFB\uD83C\uDDFA",
            [":flag_wf:"] = "\uD83C\uDDFC\uD83C\uDDEB",
            [":flag_white:"] = "\uD83C\uDFF3️",
            [":flag_ws:"] = "\uD83C\uDDFC\uD83C\uDDF8",
            [":flag_xk:"] = "\uD83C\uDDFD\uD83C\uDDF0",
            [":flag_ye:"] = "\uD83C\uDDFE\uD83C\uDDEA",
            [":flag_yt:"] = "\uD83C\uDDFE\uD83C\uDDF9",
            [":flag_za:"] = "\uD83C\uDDFF\uD83C\uDDE6",
            [":flag_zm:"] = "\uD83C\uDDFF\uD83C\uDDF2",
            [":flag_zw:"] = "\uD83C\uDDFF\uD83C\uDDFC",
            [":flags:"] = "\uD83C\uDF8F",
            [":flame:"] = "\uD83D\uDD25",
            [":flamingo:"] = "\uD83E\uDDA9",
            [":flan:"] = "\uD83C\uDF6E",
            [":flashlight:"] = "\uD83D\uDD26",
            [":flatbread:"] = "\uD83E\uDED3",
            [":fleur_de_lis:"] = "⚜️",
            [":floppy_disk:"] = "\uD83D\uDCBE",
            [":flower_playing_cards:"] = "\uD83C\uDFB4",
            [":flushed:"] = "\uD83D\uDE33",
            [":fly:"] = "\uD83E\uDEB0",
            [":flying_disc:"] = "\uD83E\uDD4F",
            [":flying_saucer:"] = "\uD83D\uDEF8",
            [":fog:"] = "\uD83C\uDF2B️",
            [":foggy:"] = "\uD83C\uDF01",
            [":fondue:"] = "\uD83E\uDED5",
            [":foot:"] = "\uD83E\uDDB6",
            [":foot::skin-tone-1:"] = "\uD83E\uDDB6\uD83C\uDFFB",
            [":foot::skin-tone-2:"] = "\uD83E\uDDB6\uD83C\uDFFC",
            [":foot::skin-tone-3:"] = "\uD83E\uDDB6\uD83C\uDFFD",
            [":foot::skin-tone-4:"] = "\uD83E\uDDB6\uD83C\uDFFE",
            [":foot::skin-tone-5:"] = "\uD83E\uDDB6\uD83C\uDFFF",
            [":foot_dark_skin_tone:"] = "\uD83E\uDDB6\uD83C\uDFFF",
            [":foot_light_skin_tone:"] = "\uD83E\uDDB6\uD83C\uDFFB",
            [":foot_medium_dark_skin_tone:"] = "\uD83E\uDDB6\uD83C\uDFFE",
            [":foot_medium_light_skin_tone:"] = "\uD83E\uDDB6\uD83C\uDFFC",
            [":foot_medium_skin_tone:"] = "\uD83E\uDDB6\uD83C\uDFFD",
            [":foot_tone1:"] = "\uD83E\uDDB6\uD83C\uDFFB",
            [":foot_tone2:"] = "\uD83E\uDDB6\uD83C\uDFFC",
            [":foot_tone3:"] = "\uD83E\uDDB6\uD83C\uDFFD",
            [":foot_tone4:"] = "\uD83E\uDDB6\uD83C\uDFFE",
            [":foot_tone5:"] = "\uD83E\uDDB6\uD83C\uDFFF",
            [":football:"] = "\uD83C\uDFC8",
            [":footprints:"] = "\uD83D\uDC63",
            [":fork_and_knife:"] = "\uD83C\uDF74",
            [":fork_and_knife_with_plate:"] = "\uD83C\uDF7D️",
            [":fork_knife_plate:"] = "\uD83C\uDF7D️",
            [":fortune_cookie:"] = "\uD83E\uDD60",
            [":fountain:"] = "⛲",
            [":four:"] = "4️⃣",
            [":four_leaf_clover:"] = "\uD83C\uDF40",
            [":fox:"] = "\uD83E\uDD8A",
            [":fox_face:"] = "\uD83E\uDD8A",
            [":frame_photo:"] = "\uD83D\uDDBC️",
            [":frame_with_picture:"] = "\uD83D\uDDBC️",
            [":free:"] = "\uD83C\uDD93",
            [":french_bread:"] = "\uD83E\uDD56",
            [":fried_shrimp:"] = "\uD83C\uDF64",
            [":fries:"] = "\uD83C\uDF5F",
            [":frog:"] = "\uD83D\uDC38",
            [":frowning2:"] = "☹️",
            [":frowning:"] = "\uD83D\uDE26",
            [":fuelpump:"] = "⛽",
            [":full_moon:"] = "\uD83C\uDF15",
            [":full_moon_with_face:"] = "\uD83C\uDF1D",
            [":funeral_urn:"] = "⚱️",
            [":game_die:"] = "\uD83C\uDFB2",
            [":garlic:"] = "\uD83E\uDDC4",
            [":gay_pride_flag:"] = "\uD83C\uDFF3️\u200D\uD83C\uDF08",
            [":gear:"] = "⚙️",
            [":gem:"] = "\uD83D\uDC8E",
            [":gemini:"] = "♊",
            [":genie:"] = "\uD83E\uDDDE",
            [":ghost:"] = "\uD83D\uDC7B",
            [":gift:"] = "\uD83C\uDF81",
            [":gift_heart:"] = "\uD83D\uDC9D",
            [":giraffe:"] = "\uD83E\uDD92",
            [":girl:"] = "\uD83D\uDC67",
            [":girl::skin-tone-1:"] = "\uD83D\uDC67\uD83C\uDFFB",
            [":girl::skin-tone-2:"] = "\uD83D\uDC67\uD83C\uDFFC",
            [":girl::skin-tone-3:"] = "\uD83D\uDC67\uD83C\uDFFD",
            [":girl::skin-tone-4:"] = "\uD83D\uDC67\uD83C\uDFFE",
            [":girl::skin-tone-5:"] = "\uD83D\uDC67\uD83C\uDFFF",
            [":girl_tone1:"] = "\uD83D\uDC67\uD83C\uDFFB",
            [":girl_tone2:"] = "\uD83D\uDC67\uD83C\uDFFC",
            [":girl_tone3:"] = "\uD83D\uDC67\uD83C\uDFFD",
            [":girl_tone4:"] = "\uD83D\uDC67\uD83C\uDFFE",
            [":girl_tone5:"] = "\uD83D\uDC67\uD83C\uDFFF",
            [":glass_of_milk:"] = "\uD83E\uDD5B",
            [":globe_with_meridians:"] = "\uD83C\uDF10",
            [":gloves:"] = "\uD83E\uDDE4",
            [":goal:"] = "\uD83E\uDD45",
            [":goal_net:"] = "\uD83E\uDD45",
            [":goat:"] = "\uD83D\uDC10",
            [":goggles:"] = "\uD83E\uDD7D",
            [":golf:"] = "⛳",
            [":golfer:"] = "\uD83C\uDFCC️",
            [":golfer::skin-tone-1:"] = "\uD83C\uDFCC\uD83C\uDFFB",
            [":golfer::skin-tone-2:"] = "\uD83C\uDFCC\uD83C\uDFFC",
            [":golfer::skin-tone-3:"] = "\uD83C\uDFCC\uD83C\uDFFD",
            [":golfer::skin-tone-4:"] = "\uD83C\uDFCC\uD83C\uDFFE",
            [":golfer::skin-tone-5:"] = "\uD83C\uDFCC\uD83C\uDFFF",
            [":gorilla:"] = "\uD83E\uDD8D",
            [":grandma:"] = "\uD83D\uDC75",
            [":grandma::skin-tone-1:"] = "\uD83D\uDC75\uD83C\uDFFB",
            [":grandma::skin-tone-2:"] = "\uD83D\uDC75\uD83C\uDFFC",
            [":grandma::skin-tone-3:"] = "\uD83D\uDC75\uD83C\uDFFD",
            [":grandma::skin-tone-4:"] = "\uD83D\uDC75\uD83C\uDFFE",
            [":grandma::skin-tone-5:"] = "\uD83D\uDC75\uD83C\uDFFF",
            [":grandma_tone1:"] = "\uD83D\uDC75\uD83C\uDFFB",
            [":grandma_tone2:"] = "\uD83D\uDC75\uD83C\uDFFC",
            [":grandma_tone3:"] = "\uD83D\uDC75\uD83C\uDFFD",
            [":grandma_tone4:"] = "\uD83D\uDC75\uD83C\uDFFE",
            [":grandma_tone5:"] = "\uD83D\uDC75\uD83C\uDFFF",
            [":grapes:"] = "\uD83C\uDF47",
            [":green_apple:"] = "\uD83C\uDF4F",
            [":green_book:"] = "\uD83D\uDCD7",
            [":green_circle:"] = "\uD83D\uDFE2",
            [":green_heart:"] = "\uD83D\uDC9A",
            [":green_salad:"] = "\uD83E\uDD57",
            [":green_square:"] = "\uD83D\uDFE9",
            [":grey_exclamation:"] = "❕",
            [":grey_question:"] = "❔",
            [":grimacing:"] = "\uD83D\uDE2C",
            [":grin:"] = "\uD83D\uDE01",
            [":grinning:"] = "\uD83D\uDE00",
            [":guard:"] = "\uD83D\uDC82",
            [":guard::skin-tone-1:"] = "\uD83D\uDC82\uD83C\uDFFB",
            [":guard::skin-tone-2:"] = "\uD83D\uDC82\uD83C\uDFFC",
            [":guard::skin-tone-3:"] = "\uD83D\uDC82\uD83C\uDFFD",
            [":guard::skin-tone-4:"] = "\uD83D\uDC82\uD83C\uDFFE",
            [":guard::skin-tone-5:"] = "\uD83D\uDC82\uD83C\uDFFF",
            [":guard_tone1:"] = "\uD83D\uDC82\uD83C\uDFFB",
            [":guard_tone2:"] = "\uD83D\uDC82\uD83C\uDFFC",
            [":guard_tone3:"] = "\uD83D\uDC82\uD83C\uDFFD",
            [":guard_tone4:"] = "\uD83D\uDC82\uD83C\uDFFE",
            [":guard_tone5:"] = "\uD83D\uDC82\uD83C\uDFFF",
            [":guardsman:"] = "\uD83D\uDC82",
            [":guardsman::skin-tone-1:"] = "\uD83D\uDC82\uD83C\uDFFB",
            [":guardsman::skin-tone-2:"] = "\uD83D\uDC82\uD83C\uDFFC",
            [":guardsman::skin-tone-3:"] = "\uD83D\uDC82\uD83C\uDFFD",
            [":guardsman::skin-tone-4:"] = "\uD83D\uDC82\uD83C\uDFFE",
            [":guardsman::skin-tone-5:"] = "\uD83D\uDC82\uD83C\uDFFF",
            [":guardsman_tone1:"] = "\uD83D\uDC82\uD83C\uDFFB",
            [":guardsman_tone2:"] = "\uD83D\uDC82\uD83C\uDFFC",
            [":guardsman_tone3:"] = "\uD83D\uDC82\uD83C\uDFFD",
            [":guardsman_tone4:"] = "\uD83D\uDC82\uD83C\uDFFE",
            [":guardsman_tone5:"] = "\uD83D\uDC82\uD83C\uDFFF",
            [":guide_dog:"] = "\uD83E\uDDAE",
            [":guitar:"] = "\uD83C\uDFB8",
            [":gun:"] = "\uD83D\uDD2B",
            [":haircut:"] = "\uD83D\uDC87",
            [":haircut::skin-tone-1:"] = "\uD83D\uDC87\uD83C\uDFFB",
            [":haircut::skin-tone-2:"] = "\uD83D\uDC87\uD83C\uDFFC",
            [":haircut::skin-tone-3:"] = "\uD83D\uDC87\uD83C\uDFFD",
            [":haircut::skin-tone-4:"] = "\uD83D\uDC87\uD83C\uDFFE",
            [":haircut::skin-tone-5:"] = "\uD83D\uDC87\uD83C\uDFFF",
            [":haircut_tone1:"] = "\uD83D\uDC87\uD83C\uDFFB",
            [":haircut_tone2:"] = "\uD83D\uDC87\uD83C\uDFFC",
            [":haircut_tone3:"] = "\uD83D\uDC87\uD83C\uDFFD",
            [":haircut_tone4:"] = "\uD83D\uDC87\uD83C\uDFFE",
            [":haircut_tone5:"] = "\uD83D\uDC87\uD83C\uDFFF",
            [":hamburger:"] = "\uD83C\uDF54",
            [":hammer:"] = "\uD83D\uDD28",
            [":hammer_and_pick:"] = "⚒️",
            [":hammer_and_wrench:"] = "\uD83D\uDEE0️",
            [":hammer_pick:"] = "⚒️",
            [":hamsa:"] = "\uD83E\uDEAC",
            [":hamster:"] = "\uD83D\uDC39",
            [":hand_splayed:"] = "\uD83D\uDD90️",
            [":hand_splayed::skin-tone-1:"] = "\uD83D\uDD90\uD83C\uDFFB",
            [":hand_splayed::skin-tone-2:"] = "\uD83D\uDD90\uD83C\uDFFC",
            [":hand_splayed::skin-tone-3:"] = "\uD83D\uDD90\uD83C\uDFFD",
            [":hand_splayed::skin-tone-4:"] = "\uD83D\uDD90\uD83C\uDFFE",
            [":hand_splayed::skin-tone-5:"] = "\uD83D\uDD90\uD83C\uDFFF",
            [":hand_splayed_tone1:"] = "\uD83D\uDD90\uD83C\uDFFB",
            [":hand_splayed_tone2:"] = "\uD83D\uDD90\uD83C\uDFFC",
            [":hand_splayed_tone3:"] = "\uD83D\uDD90\uD83C\uDFFD",
            [":hand_splayed_tone4:"] = "\uD83D\uDD90\uD83C\uDFFE",
            [":hand_splayed_tone5:"] = "\uD83D\uDD90\uD83C\uDFFF",
            [":hand_with_index_and_middle_finger_crossed:"] = "\uD83E\uDD1E",
            [":hand_with_index_and_middle_finger_crossed::skin-tone-1:"] = "\uD83E\uDD1E\uD83C\uDFFB",
            [":hand_with_index_and_middle_finger_crossed::skin-tone-2:"] = "\uD83E\uDD1E\uD83C\uDFFC",
            [":hand_with_index_and_middle_finger_crossed::skin-tone-3:"] = "\uD83E\uDD1E\uD83C\uDFFD",
            [":hand_with_index_and_middle_finger_crossed::skin-tone-4:"] = "\uD83E\uDD1E\uD83C\uDFFE",
            [":hand_with_index_and_middle_finger_crossed::skin-tone-5:"] = "\uD83E\uDD1E\uD83C\uDFFF",
            [":hand_with_index_and_middle_fingers_crossed_tone1:"] = "\uD83E\uDD1E\uD83C\uDFFB",
            [":hand_with_index_and_middle_fingers_crossed_tone2:"] = "\uD83E\uDD1E\uD83C\uDFFC",
            [":hand_with_index_and_middle_fingers_crossed_tone3:"] = "\uD83E\uDD1E\uD83C\uDFFD",
            [":hand_with_index_and_middle_fingers_crossed_tone4:"] = "\uD83E\uDD1E\uD83C\uDFFE",
            [":hand_with_index_and_middle_fingers_crossed_tone5:"] = "\uD83E\uDD1E\uD83C\uDFFF",
            [":hand_with_index_finger_and_thumb_crossed:"] = "\uD83E\uDEF4",
            [":hand_with_index_finger_and_thumb_crossed::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hand_with_index_finger_and_thumb_crossed_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handbag:"] = "\uD83D\uDC5C",
            [":handball:"] = "\uD83E\uDD3E",
            [":handball::skin-tone-1:"] = "\uD83E\uDD3E\uD83C\uDFFB",
            [":handball::skin-tone-2:"] = "\uD83E\uDD3E\uD83C\uDFFC",
            [":handball::skin-tone-3:"] = "\uD83E\uDD3E\uD83C\uDFFD",
            [":handball::skin-tone-4:"] = "\uD83E\uDD3E\uD83C\uDFFE",
            [":handball::skin-tone-5:"] = "\uD83E\uDD3E\uD83C\uDFFF",
            [":handball_tone1:"] = "\uD83E\uDD3E\uD83C\uDFFB",
            [":handball_tone2:"] = "\uD83E\uDD3E\uD83C\uDFFC",
            [":handball_tone3:"] = "\uD83E\uDD3E\uD83C\uDFFD",
            [":handball_tone4:"] = "\uD83E\uDD3E\uD83C\uDFFE",
            [":handball_tone5:"] = "\uD83E\uDD3E\uD83C\uDFFF",
            [":handshake:"] = "\uD83E\uDD1D",
            [":handshake::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_dark_skin_tone_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_light_skin_tone_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_dark_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_light_skin_tone_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_light_skin_tone_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_light_skin_tone_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_medium_skin_tone_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":handshake_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":hankey:"] = "\uD83D\uDCA9",
            [":hash:"] = "#️⃣",
            [":hatched_chick:"] = "\uD83D\uDC25",
            [":hatching_chick:"] = "\uD83D\uDC23",
            [":head_bandage:"] = "\uD83E\uDD15",
            [":headphones:"] = "\uD83C\uDFA7",
            [":headstone:"] = "\uD83E\uDEA6",
            [":hear_no_evil:"] = "\uD83D\uDE49",
            [":heart:"] = "❤️",
            [":heart_decoration:"] = "\uD83D\uDC9F",
            [":heart_exclamation:"] = "❣️",
            [":heart_eyes:"] = "\uD83D\uDE0D",
            [":heart_eyes_cat:"] = "\uD83D\uDE3B",
            [":heart_hands:"] = "\uD83E\uDEF6",
            [":heart_hands::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_hands_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":heart_on_fire:"] = "\u2764\uFE0F\u200D\uD83D\uDD25",
            [":heartbeat:"] = "\uD83D\uDC93",
            [":heartpulse:"] = "\uD83D\uDC97",
            [":hearts:"] = "♥️",
            [":heavy_check_mark:"] = "✔️",
            [":heavy_division_sign:"] = "➗",
            [":heavy_dollar_sign:"] = "\uD83D\uDCB2",
            [":heavy_equals_sign:"] = "\uD83D\uDFF0",
            [":heavy_heart_exclamation_mark_ornament:"] = "❣️",
            [":heavy_minus_sign:"] = "➖",
            [":heavy_multiplication_x:"] = "✖️",
            [":heavy_plus_sign:"] = "➕",
            [":hedgehog:"] = "\uD83E\uDD94",
            [":helicopter:"] = "\uD83D\uDE81",
            [":helmet_with_cross:"] = "⛑️",
            [":helmet_with_white_cross:"] = "⛑️",
            [":herb:"] = "\uD83C\uDF3F",
            [":hibiscus:"] = "\uD83C\uDF3A",
            [":high_brightness:"] = "\uD83D\uDD06",
            [":high_heel:"] = "\uD83D\uDC60",
            [":hiking_boot:"] = "\uD83E\uDD7E",
            [":hindu_temple:"] = "\uD83D\uDED5",
            [":hippopotamus:"] = "\uD83E\uDD9B",
            [":hockey:"] = "\uD83C\uDFD2",
            [":hole:"] = "\uD83D\uDD73️",
            [":homes:"] = "\uD83C\uDFD8️",
            [":honey_pot:"] = "\uD83C\uDF6F",
            [":hook:"] = "\uD83E\uDE9D",
            [":horse:"] = "\uD83D\uDC34",
            [":horse_racing:"] = "\uD83C\uDFC7",
            [":horse_racing::skin-tone-1:"] = "\uD83C\uDFC7\uD83C\uDFFB",
            [":horse_racing::skin-tone-2:"] = "\uD83C\uDFC7\uD83C\uDFFC",
            [":horse_racing::skin-tone-3:"] = "\uD83C\uDFC7\uD83C\uDFFD",
            [":horse_racing::skin-tone-4:"] = "\uD83C\uDFC7\uD83C\uDFFE",
            [":horse_racing::skin-tone-5:"] = "\uD83C\uDFC7\uD83C\uDFFF",
            [":horse_racing_tone1:"] = "\uD83C\uDFC7\uD83C\uDFFB",
            [":horse_racing_tone2:"] = "\uD83C\uDFC7\uD83C\uDFFC",
            [":horse_racing_tone3:"] = "\uD83C\uDFC7\uD83C\uDFFD",
            [":horse_racing_tone4:"] = "\uD83C\uDFC7\uD83C\uDFFE",
            [":horse_racing_tone5:"] = "\uD83C\uDFC7\uD83C\uDFFF",
            [":hospital:"] = "\uD83C\uDFE5",
            [":hot_dog:"] = "\uD83C\uDF2D",
            [":hot_face:"] = "\uD83E\uDD75",
            [":hot_pepper:"] = "\uD83C\uDF36️",
            [":hotdog:"] = "\uD83C\uDF2D",
            [":hotel:"] = "\uD83C\uDFE8",
            [":hotsprings:"] = "♨️",
            [":hourglass:"] = "⌛",
            [":hourglass_flowing_sand:"] = "⏳",
            [":house:"] = "\uD83C\uDFE0",
            [":house_abandoned:"] = "\uD83C\uDFDA️",
            [":house_buildings:"] = "\uD83C\uDFD8️",
            [":house_with_garden:"] = "\uD83C\uDFE1",
            [":hugging:"] = "\uD83E\uDD17",
            [":hugging_face:"] = "\uD83E\uDD17",
            [":hushed:"] = "\uD83D\uDE2F",
            [":hut:"] = "\uD83D\uDED6",
            [":ice_cream:"] = "\uD83C\uDF68",
            [":ice_cube:"] = "\uD83E\uDDCA",
            [":ice_skate:"] = "⛸️",
            [":icecream:"] = "\uD83C\uDF66",
            [":id:"] = "\uD83C\uDD94",
            [":identification_card:"] = "\uD83E\uDEAA",
            [":ideograph_advantage:"] = "\uD83C\uDE50",
            [":imp:"] = "\uD83D\uDC7F",
            [":inbox_tray:"] = "\uD83D\uDCE5",
            [":incoming_envelope:"] = "\uD83D\uDCE8",
            [":index_pointing_at_the_viewer:"] = "\uD83E\uDEF4",
            [":index_pointing_at_the_viewer::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":index_pointing_at_the_viewer_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":infinity:"] = "♾️",
            [":information_desk_person:"] = "\uD83D\uDC81",
            [":information_desk_person::skin-tone-1:"] = "\uD83D\uDC81\uD83C\uDFFB",
            [":information_desk_person::skin-tone-2:"] = "\uD83D\uDC81\uD83C\uDFFC",
            [":information_desk_person::skin-tone-3:"] = "\uD83D\uDC81\uD83C\uDFFD",
            [":information_desk_person::skin-tone-4:"] = "\uD83D\uDC81\uD83C\uDFFE",
            [":information_desk_person::skin-tone-5:"] = "\uD83D\uDC81\uD83C\uDFFF",
            [":information_desk_person_tone1:"] = "\uD83D\uDC81\uD83C\uDFFB",
            [":information_desk_person_tone2:"] = "\uD83D\uDC81\uD83C\uDFFC",
            [":information_desk_person_tone3:"] = "\uD83D\uDC81\uD83C\uDFFD",
            [":information_desk_person_tone4:"] = "\uD83D\uDC81\uD83C\uDFFE",
            [":information_desk_person_tone5:"] = "\uD83D\uDC81\uD83C\uDFFF",
            [":information_source:"] = "ℹ️",
            [":innocent:"] = "\uD83D\uDE07",
            [":interrobang:"] = "⁉️",
            [":iphone:"] = "\uD83D\uDCF1",
            [":island:"] = "\uD83C\uDFDD️",
            [":izakaya_lantern:"] = "\uD83C\uDFEE",
            [":jack_o_lantern:"] = "\uD83C\uDF83",
            [":japan:"] = "\uD83D\uDDFE",
            [":japanese_castle:"] = "\uD83C\uDFEF",
            [":japanese_goblin:"] = "\uD83D\uDC7A",
            [":japanese_ogre:"] = "\uD83D\uDC79",
            [":jar:"] = "\uD83E\uDED9",
            [":jeans:"] = "\uD83D\uDC56",
            [":jigsaw:"] = "\uD83E\uDDE9",
            [":joy:"] = "\uD83D\uDE02",
            [":joy_cat:"] = "\uD83D\uDE39",
            [":joystick:"] = "\uD83D\uDD79️",
            [":juggler:"] = "\uD83E\uDD39",
            [":juggler::skin-tone-1:"] = "\uD83E\uDD39\uD83C\uDFFB",
            [":juggler::skin-tone-2:"] = "\uD83E\uDD39\uD83C\uDFFC",
            [":juggler::skin-tone-3:"] = "\uD83E\uDD39\uD83C\uDFFD",
            [":juggler::skin-tone-4:"] = "\uD83E\uDD39\uD83C\uDFFE",
            [":juggler::skin-tone-5:"] = "\uD83E\uDD39\uD83C\uDFFF",
            [":juggler_tone1:"] = "\uD83E\uDD39\uD83C\uDFFB",
            [":juggler_tone2:"] = "\uD83E\uDD39\uD83C\uDFFC",
            [":juggler_tone3:"] = "\uD83E\uDD39\uD83C\uDFFD",
            [":juggler_tone4:"] = "\uD83E\uDD39\uD83C\uDFFE",
            [":juggler_tone5:"] = "\uD83E\uDD39\uD83C\uDFFF",
            [":juggling:"] = "\uD83E\uDD39",
            [":juggling::skin-tone-1:"] = "\uD83E\uDD39\uD83C\uDFFB",
            [":juggling::skin-tone-2:"] = "\uD83E\uDD39\uD83C\uDFFC",
            [":juggling::skin-tone-3:"] = "\uD83E\uDD39\uD83C\uDFFD",
            [":juggling::skin-tone-4:"] = "\uD83E\uDD39\uD83C\uDFFE",
            [":juggling::skin-tone-5:"] = "\uD83E\uDD39\uD83C\uDFFF",
            [":juggling_tone1:"] = "\uD83E\uDD39\uD83C\uDFFB",
            [":juggling_tone2:"] = "\uD83E\uDD39\uD83C\uDFFC",
            [":juggling_tone3:"] = "\uD83E\uDD39\uD83C\uDFFD",
            [":juggling_tone4:"] = "\uD83E\uDD39\uD83C\uDFFE",
            [":juggling_tone5:"] = "\uD83E\uDD39\uD83C\uDFFF",
            [":kaaba:"] = "\uD83D\uDD4B",
            [":kangaroo:"] = "\uD83E\uDD98",
            [":karate_uniform:"] = "\uD83E\uDD4B",
            [":kayak:"] = "\uD83D\uDEF6",
            [":key2:"] = "\uD83D\uDDDD️",
            [":key:"] = "\uD83D\uDD11",
            [":keyboard:"] = "⌨️",
            [":keycap_asterisk:"] = "*️⃣",
            [":keycap_ten:"] = "\uD83D\uDD1F",
            [":kimono:"] = "\uD83D\uDC58",
            [":kiss:"] = "\uD83D\uDC8B",
            [":kiss::skin-tone-1:"] = "\uD83D\uDC8F\uD83C\uDFFB",
            [":kiss::skin-tone-2:"] = "\uD83D\uDC8F\uD83C\uDFFC",
            [":kiss::skin-tone-3:"] = "\uD83D\uDC8F\uD83C\uDFFD",
            [":kiss::skin-tone-4:"] = "\uD83D\uDC8F\uD83C\uDFFE",
            [":kiss::skin-tone-5:"] = "\uD83D\uDC8F\uD83C\uDFFF",
            [":kiss_dark_skin_tone:"] = "\uD83D\uDC8F\uD83C\uDFFF",
            [":kiss_light_skin_tone:"] = "\uD83D\uDC8F\uD83C\uDFFB",
            [":kiss_man_man::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_dark_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_dark_skin_tone_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone_tone3:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_dark_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_dark_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_light_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_light_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_light_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_light_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_light_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_medium_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_medium_light_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_man_man_medium_skin_tone_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone_tone1:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_medium_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_medium_skin_tone_tone2:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_medium_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_medium_skin_tone_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_man_man_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_man_man_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_man_man_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_man_man_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_medium_dark_skin_tone:"] = "\uD83D\uDC8F\uD83C\uDFFE",
            [":kiss_medium_light_skin_tone:"] = "\uD83D\uDC8F\uD83C\uDFFC",
            [":kiss_medium_skin_tone:"] = "\uD83D\uDC8F\uD83C\uDFFD",
            [":kiss_mm:"] = "\uD83D\uDC68\u200D❤️\u200D\uD83D\uDC8B\u200D\uD83D\uDC68",
            [":kiss_person_person_dark_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_dark_skin_tone::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_dark_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_dark_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_dark_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_dark_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_dark_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_dark_skin_tone_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_dark_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_dark_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_light_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_light_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_light_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_light_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_light_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_light_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_light_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_light_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_light_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_light_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_light_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_light_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_medium_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_medium_dark_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_medium_dark_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_medium_dark_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_medium_light_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_medium_light_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_medium_light_skin_tone_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_skin_tone::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_medium_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_skin_tone::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_medium_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_medium_skin_tone::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_skin_tone_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFF",
            [":kiss_person_person_medium_skin_tone_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_medium_skin_tone_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_medium_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_skin_tone_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFB",
            [":kiss_person_person_medium_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_person_person_medium_skin_tone_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFC",
            [":kiss_person_person_medium_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFE",
            [":kiss_person_person_medium_skin_tone_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83E\uDDD1\uD83C\uDFFD",
            [":kiss_tone1:"] = "\uD83D\uDC8F\uD83C\uDFFB",
            [":kiss_tone2:"] = "\uD83D\uDC8F\uD83C\uDFFC",
            [":kiss_tone3:"] = "\uD83D\uDC8F\uD83C\uDFFD",
            [":kiss_tone4:"] = "\uD83D\uDC8F\uD83C\uDFFE",
            [":kiss_tone5:"] = "\uD83D\uDC8F\uD83C\uDFFF",
            [":kiss_woman_man:"] = "\uD83D\uDC69\u200D❤️\u200D\uD83D\uDC8B\u200D\uD83D\uDC68",
            [":kiss_woman_man::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_light_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_medium_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_medium_light_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_man_medium_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFB",
            [":kiss_woman_man_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFC",
            [":kiss_woman_man_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFD",
            [":kiss_woman_man_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFE",
            [":kiss_woman_man_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC68\uD83C\uDFFF",
            [":kiss_woman_woman::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_dark_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone_tone3:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_dark_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_light_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_light_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_light_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_light_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_light_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_dark_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_dark_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_medium_dark_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_medium_dark_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_dark_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_dark_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_light_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_medium_light_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_medium_light_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_light_skin_tone_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_skin_tone::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_woman_woman_medium_skin_tone_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_medium_skin_tone_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_skin_tone_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone_tone1:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_medium_skin_tone_tone2:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_medium_skin_tone_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFB",
            [":kiss_woman_woman_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFC",
            [":kiss_woman_woman_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFD",
            [":kiss_woman_woman_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFE",
            [":kiss_woman_woman_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\u2764\uFE0F\u200D\uD83D\uDC8B\u200D\uD83D\uDC69\uD83C\uDFFF",
            [":kiss_ww:"] = "\uD83D\uDC69\u200D❤️\u200D\uD83D\uDC8B\u200D\uD83D\uDC69",
            [":kissing:"] = "\uD83D\uDE17",
            [":kissing_cat:"] = "\uD83D\uDE3D",
            [":kissing_closed_eyes:"] = "\uD83D\uDE1A",
            [":kissing_heart:"] = "\uD83D\uDE18",
            [":kissing_smiling_eyes:"] = "\uD83D\uDE19",
            [":kite:"] = "\uD83E\uDE81",
            [":kiwi:"] = "\uD83E\uDD5D",
            [":kiwifruit:"] = "\uD83E\uDD5D",
            [":knife:"] = "\uD83D\uDD2A",
            [":knot:"] = "\uD83E\uDEA2",
            [":koala:"] = "\uD83D\uDC28",
            [":koko:"] = "\uD83C\uDE01",
            [":lab_coat:"] = "\uD83E\uDD7C",
            [":label:"] = "\uD83C\uDFF7️",
            [":lacrosse:"] = "\uD83E\uDD4D",
            [":ladder:"] = "\uD83E\uDE9C",
            [":large_blue_diamond:"] = "\uD83D\uDD37",
            [":large_orange_diamond:"] = "\uD83D\uDD36",
            [":last_quarter_moon:"] = "\uD83C\uDF17",
            [":last_quarter_moon_with_face:"] = "\uD83C\uDF1C",
            [":latin_cross:"] = "✝️",
            [":laughing:"] = "\uD83D\uDE06",
            [":leafy_green:"] = "\uD83E\uDD6C",
            [":leaves:"] = "\uD83C\uDF43",
            [":ledger:"] = "\uD83D\uDCD2",
            [":left_facing_fist:"] = "\uD83E\uDD1B",
            [":left_facing_fist::skin-tone-1:"] = "\uD83E\uDD1B\uD83C\uDFFB",
            [":left_facing_fist::skin-tone-2:"] = "\uD83E\uDD1B\uD83C\uDFFC",
            [":left_facing_fist::skin-tone-3:"] = "\uD83E\uDD1B\uD83C\uDFFD",
            [":left_facing_fist::skin-tone-4:"] = "\uD83E\uDD1B\uD83C\uDFFE",
            [":left_facing_fist::skin-tone-5:"] = "\uD83E\uDD1B\uD83C\uDFFF",
            [":left_facing_fist_tone1:"] = "\uD83E\uDD1B\uD83C\uDFFB",
            [":left_facing_fist_tone2:"] = "\uD83E\uDD1B\uD83C\uDFFC",
            [":left_facing_fist_tone3:"] = "\uD83E\uDD1B\uD83C\uDFFD",
            [":left_facing_fist_tone4:"] = "\uD83E\uDD1B\uD83C\uDFFE",
            [":left_facing_fist_tone5:"] = "\uD83E\uDD1B\uD83C\uDFFF",
            [":left_fist:"] = "\uD83E\uDD1B",
            [":left_fist::skin-tone-1:"] = "\uD83E\uDD1B\uD83C\uDFFB",
            [":left_fist::skin-tone-2:"] = "\uD83E\uDD1B\uD83C\uDFFC",
            [":left_fist::skin-tone-3:"] = "\uD83E\uDD1B\uD83C\uDFFD",
            [":left_fist::skin-tone-4:"] = "\uD83E\uDD1B\uD83C\uDFFE",
            [":left_fist::skin-tone-5:"] = "\uD83E\uDD1B\uD83C\uDFFF",
            [":left_fist_tone1:"] = "\uD83E\uDD1B\uD83C\uDFFB",
            [":left_fist_tone2:"] = "\uD83E\uDD1B\uD83C\uDFFC",
            [":left_fist_tone3:"] = "\uD83E\uDD1B\uD83C\uDFFD",
            [":left_fist_tone4:"] = "\uD83E\uDD1B\uD83C\uDFFE",
            [":left_fist_tone5:"] = "\uD83E\uDD1B\uD83C\uDFFF",
            [":left_luggage:"] = "\uD83D\uDEC5",
            [":left_right_arrow:"] = "↔️",
            [":left_speech_bubble:"] = "\uD83D\uDDE8️",
            [":leftwards_arrow_with_hook:"] = "↩️",
            [":leftwards_hand:"] = "\uD83E\uDEF2",
            [":leftwards_hand::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leftwards_hand_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":leg:"] = "\uD83E\uDDB5",
            [":leg::skin-tone-1:"] = "\uD83E\uDDB5\uD83C\uDFFB",
            [":leg::skin-tone-2:"] = "\uD83E\uDDB5\uD83C\uDFFC",
            [":leg::skin-tone-3:"] = "\uD83E\uDDB5\uD83C\uDFFD",
            [":leg::skin-tone-4:"] = "\uD83E\uDDB5\uD83C\uDFFE",
            [":leg::skin-tone-5:"] = "\uD83E\uDDB5\uD83C\uDFFF",
            [":leg_dark_skin_tone:"] = "\uD83E\uDDB5\uD83C\uDFFF",
            [":leg_light_skin_tone:"] = "\uD83E\uDDB5\uD83C\uDFFB",
            [":leg_medium_dark_skin_tone:"] = "\uD83E\uDDB5\uD83C\uDFFE",
            [":leg_medium_light_skin_tone:"] = "\uD83E\uDDB5\uD83C\uDFFC",
            [":leg_medium_skin_tone:"] = "\uD83E\uDDB5\uD83C\uDFFD",
            [":leg_tone1:"] = "\uD83E\uDDB5\uD83C\uDFFB",
            [":leg_tone2:"] = "\uD83E\uDDB5\uD83C\uDFFC",
            [":leg_tone3:"] = "\uD83E\uDDB5\uD83C\uDFFD",
            [":leg_tone4:"] = "\uD83E\uDDB5\uD83C\uDFFE",
            [":leg_tone5:"] = "\uD83E\uDDB5\uD83C\uDFFF",
            [":lemon:"] = "\uD83C\uDF4B",
            [":leo:"] = "♌",
            [":leopard:"] = "\uD83D\uDC06",
            [":level_slider:"] = "\uD83C\uDF9A️",
            [":levitate:"] = "\uD83D\uDD74️",
            [":levitate::skin-tone-1:"] = "\uD83D\uDD74\uD83C\uDFFB",
            [":levitate::skin-tone-2:"] = "\uD83D\uDD74\uD83C\uDFFC",
            [":levitate::skin-tone-3:"] = "\uD83D\uDD74\uD83C\uDFFD",
            [":levitate::skin-tone-4:"] = "\uD83D\uDD74\uD83C\uDFFE",
            [":levitate::skin-tone-5:"] = "\uD83D\uDD74\uD83C\uDFFF",
            [":levitate_tone1:"] = "\uD83D\uDD74\uD83C\uDFFB",
            [":levitate_tone2:"] = "\uD83D\uDD74\uD83C\uDFFC",
            [":levitate_tone3:"] = "\uD83D\uDD74\uD83C\uDFFD",
            [":levitate_tone4:"] = "\uD83D\uDD74\uD83C\uDFFE",
            [":levitate_tone5:"] = "\uD83D\uDD74\uD83C\uDFFF",
            [":liar:"] = "\uD83E\uDD25",
            [":libra:"] = "♎",
            [":lifter:"] = "\uD83C\uDFCB️",
            [":lifter::skin-tone-1:"] = "\uD83C\uDFCB\uD83C\uDFFB",
            [":lifter::skin-tone-2:"] = "\uD83C\uDFCB\uD83C\uDFFC",
            [":lifter::skin-tone-3:"] = "\uD83C\uDFCB\uD83C\uDFFD",
            [":lifter::skin-tone-4:"] = "\uD83C\uDFCB\uD83C\uDFFE",
            [":lifter::skin-tone-5:"] = "\uD83C\uDFCB\uD83C\uDFFF",
            [":lifter_tone1:"] = "\uD83C\uDFCB\uD83C\uDFFB",
            [":lifter_tone2:"] = "\uD83C\uDFCB\uD83C\uDFFC",
            [":lifter_tone3:"] = "\uD83C\uDFCB\uD83C\uDFFD",
            [":lifter_tone4:"] = "\uD83C\uDFCB\uD83C\uDFFE",
            [":lifter_tone5:"] = "\uD83C\uDFCB\uD83C\uDFFF",
            [":light_rail:"] = "\uD83D\uDE88",
            [":link:"] = "\uD83D\uDD17",
            [":linked_paperclips:"] = "\uD83D\uDD87️",
            [":lion:"] = "\uD83E\uDD81",
            [":lion_face:"] = "\uD83E\uDD81",
            [":lips:"] = "\uD83D\uDC44",
            [":lipstick:"] = "\uD83D\uDC84",
            [":lizard:"] = "\uD83E\uDD8E",
            [":llama:"] = "\uD83E\uDD99",
            [":lobster:"] = "\uD83E\uDD9E",
            [":lock:"] = "\uD83D\uDD12",
            [":lock_with_ink_pen:"] = "\uD83D\uDD0F",
            [":lollipop:"] = "\uD83C\uDF6D",
            [":long_drum:"] = "\uD83E\uDE98",
            [":loop:"] = "➿",
            [":lotus:"] = "\uD83E\uDEB7",
            [":loud_sound:"] = "\uD83D\uDD0A",
            [":loudspeaker:"] = "\uD83D\uDCE2",
            [":love_hotel:"] = "\uD83C\uDFE9",
            [":love_letter:"] = "\uD83D\uDC8C",
            [":love_you_gesture:"] = "\uD83E\uDD1F",
            [":love_you_gesture::skin-tone-1:"] = "\uD83E\uDD1F\uD83C\uDFFB",
            [":love_you_gesture::skin-tone-2:"] = "\uD83E\uDD1F\uD83C\uDFFC",
            [":love_you_gesture::skin-tone-3:"] = "\uD83E\uDD1F\uD83C\uDFFD",
            [":love_you_gesture::skin-tone-4:"] = "\uD83E\uDD1F\uD83C\uDFFE",
            [":love_you_gesture::skin-tone-5:"] = "\uD83E\uDD1F\uD83C\uDFFF",
            [":love_you_gesture_dark_skin_tone:"] = "\uD83E\uDD1F\uD83C\uDFFF",
            [":love_you_gesture_light_skin_tone:"] = "\uD83E\uDD1F\uD83C\uDFFB",
            [":love_you_gesture_medium_dark_skin_tone:"] = "\uD83E\uDD1F\uD83C\uDFFE",
            [":love_you_gesture_medium_light_skin_tone:"] = "\uD83E\uDD1F\uD83C\uDFFC",
            [":love_you_gesture_medium_skin_tone:"] = "\uD83E\uDD1F\uD83C\uDFFD",
            [":love_you_gesture_tone1:"] = "\uD83E\uDD1F\uD83C\uDFFB",
            [":love_you_gesture_tone2:"] = "\uD83E\uDD1F\uD83C\uDFFC",
            [":love_you_gesture_tone3:"] = "\uD83E\uDD1F\uD83C\uDFFD",
            [":love_you_gesture_tone4:"] = "\uD83E\uDD1F\uD83C\uDFFE",
            [":love_you_gesture_tone5:"] = "\uD83E\uDD1F\uD83C\uDFFF",
            [":low_battery:"] = "\uD83E\uDEAB",
            [":low_brightness:"] = "\uD83D\uDD05",
            [":lower_left_ballpoint_pen:"] = "\uD83D\uDD8A️",
            [":lower_left_crayon:"] = "\uD83D\uDD8D️",
            [":lower_left_fountain_pen:"] = "\uD83D\uDD8B️",
            [":lower_left_paintbrush:"] = "\uD83D\uDD8C️",
            [":luggage:"] = "\uD83E\uDDF3",
            [":lungs:"] = "\uD83E\uDEC1",
            [":lying_face:"] = "\uD83E\uDD25",
            [":m:"] = "Ⓜ️",
            [":mag:"] = "\uD83D\uDD0D",
            [":mag_right:"] = "\uD83D\uDD0E",
            [":mage:"] = "\uD83E\uDDD9",
            [":mage::skin-tone-1:"] = "\uD83E\uDDD9\uD83C\uDFFB",
            [":mage::skin-tone-2:"] = "\uD83E\uDDD9\uD83C\uDFFC",
            [":mage::skin-tone-3:"] = "\uD83E\uDDD9\uD83C\uDFFD",
            [":mage::skin-tone-4:"] = "\uD83E\uDDD9\uD83C\uDFFE",
            [":mage::skin-tone-5:"] = "\uD83E\uDDD9\uD83C\uDFFF",
            [":mage_dark_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFF",
            [":mage_light_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFB",
            [":mage_medium_dark_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFE",
            [":mage_medium_light_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFC",
            [":mage_medium_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFD",
            [":mage_tone1:"] = "\uD83E\uDDD9\uD83C\uDFFB",
            [":mage_tone2:"] = "\uD83E\uDDD9\uD83C\uDFFC",
            [":mage_tone3:"] = "\uD83E\uDDD9\uD83C\uDFFD",
            [":mage_tone4:"] = "\uD83E\uDDD9\uD83C\uDFFE",
            [":mage_tone5:"] = "\uD83E\uDDD9\uD83C\uDFFF",
            [":magic_wand:"] = "\uD83E\uDE84",
            [":magnet:"] = "\uD83E\uDDF2",
            [":mahjong:"] = "\uD83C\uDC04",
            [":mailbox:"] = "\uD83D\uDCEB",
            [":mailbox_closed:"] = "\uD83D\uDCEA",
            [":mailbox_with_mail:"] = "\uD83D\uDCEC",
            [":mailbox_with_no_mail:"] = "\uD83D\uDCED",
            [":male_dancer:"] = "\uD83D\uDD7A",
            [":male_dancer::skin-tone-1:"] = "\uD83D\uDD7A\uD83C\uDFFB",
            [":male_dancer::skin-tone-2:"] = "\uD83D\uDD7A\uD83C\uDFFC",
            [":male_dancer::skin-tone-3:"] = "\uD83D\uDD7A\uD83C\uDFFD",
            [":male_dancer::skin-tone-4:"] = "\uD83D\uDD7A\uD83C\uDFFE",
            [":male_dancer::skin-tone-5:"] = "\uD83D\uDD7A\uD83C\uDFFF",
            [":male_dancer_tone1:"] = "\uD83D\uDD7A\uD83C\uDFFB",
            [":male_dancer_tone2:"] = "\uD83D\uDD7A\uD83C\uDFFC",
            [":male_dancer_tone3:"] = "\uD83D\uDD7A\uD83C\uDFFD",
            [":male_dancer_tone4:"] = "\uD83D\uDD7A\uD83C\uDFFE",
            [":male_dancer_tone5:"] = "\uD83D\uDD7A\uD83C\uDFFF",
            [":male_sign:"] = "♂️",
            [":mammoth:"] = "\uD83E\uDDA3",
            [":man:"] = "\uD83D\uDC68",
            [":man::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB",
            [":man::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC",
            [":man::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD",
            [":man::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE",
            [":man::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF",
            [":man_artist:"] = "\uD83D\uDC68\u200D\uD83C\uDFA8",
            [":man_artist::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFA8",
            [":man_artist::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFA8",
            [":man_artist::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFA8",
            [":man_artist::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFA8",
            [":man_artist::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFA8",
            [":man_artist_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFA8",
            [":man_artist_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFA8",
            [":man_artist_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFA8",
            [":man_artist_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFA8",
            [":man_artist_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFA8",
            [":man_artist_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFA8",
            [":man_artist_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFA8",
            [":man_artist_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFA8",
            [":man_artist_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFA8",
            [":man_artist_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFA8",
            [":man_astronaut:"] = "\uD83D\uDC68\u200D\uD83D\uDE80",
            [":man_astronaut::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDE80",
            [":man_astronaut::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDE80",
            [":man_astronaut::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDE80",
            [":man_astronaut::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDE80",
            [":man_astronaut::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDE80",
            [":man_astronaut_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDE80",
            [":man_astronaut_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDE80",
            [":man_astronaut_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDE80",
            [":man_astronaut_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDE80",
            [":man_astronaut_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDE80",
            [":man_astronaut_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDE80",
            [":man_astronaut_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDE80",
            [":man_astronaut_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDE80",
            [":man_astronaut_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDE80",
            [":man_astronaut_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDE80",
            [":man_bald:"] = "\uD83D\uDC68\u200D\uD83E\uDDB2",
            [":man_bald::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB2",
            [":man_bald::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB2",
            [":man_bald::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB2",
            [":man_bald::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB2",
            [":man_bald::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB2",
            [":man_bald_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB2",
            [":man_bald_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB2",
            [":man_bald_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB2",
            [":man_bald_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB2",
            [":man_bald_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB2",
            [":man_bald_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB2",
            [":man_bald_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB2",
            [":man_bald_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB2",
            [":man_bald_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB2",
            [":man_bald_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB2",
            [":man_beard:"] = "\uD83E\uDDD4\u200D\u2642\uFE0F",
            [":man_beard::skin-tone-1:"] = "\uD83E\uDDD4\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_beard::skin-tone-2:"] = "\uD83E\uDDD4\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_beard::skin-tone-3:"] = "\uD83E\uDDD4\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_beard::skin-tone-4:"] = "\uD83E\uDDD4\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_beard::skin-tone-5:"] = "\uD83E\uDDD4\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_beard_tone1:"] = "\uD83E\uDDD4\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_beard_tone2:"] = "\uD83E\uDDD4\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_beard_tone3:"] = "\uD83E\uDDD4\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_beard_tone4:"] = "\uD83E\uDDD4\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_beard_tone5:"] = "\uD83E\uDDD4\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_biking:"] = "\uD83D\uDEB4\u200D♂️",
            [":man_biking::skin-tone-1:"] = "\uD83D\uDEB4\uD83C\uDFFB\u200D♂️",
            [":man_biking::skin-tone-2:"] = "\uD83D\uDEB4\uD83C\uDFFC\u200D♂️",
            [":man_biking::skin-tone-3:"] = "\uD83D\uDEB4\uD83C\uDFFD\u200D♂️",
            [":man_biking::skin-tone-4:"] = "\uD83D\uDEB4\uD83C\uDFFE\u200D♂️",
            [":man_biking::skin-tone-5:"] = "\uD83D\uDEB4\uD83C\uDFFF\u200D♂️",
            [":man_biking_dark_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFF\u200D♂️",
            [":man_biking_light_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFB\u200D♂️",
            [":man_biking_medium_dark_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFE\u200D♂️",
            [":man_biking_medium_light_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFC\u200D♂️",
            [":man_biking_medium_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFD\u200D♂️",
            [":man_biking_tone1:"] = "\uD83D\uDEB4\uD83C\uDFFB\u200D♂️",
            [":man_biking_tone2:"] = "\uD83D\uDEB4\uD83C\uDFFC\u200D♂️",
            [":man_biking_tone3:"] = "\uD83D\uDEB4\uD83C\uDFFD\u200D♂️",
            [":man_biking_tone4:"] = "\uD83D\uDEB4\uD83C\uDFFE\u200D♂️",
            [":man_biking_tone5:"] = "\uD83D\uDEB4\uD83C\uDFFF\u200D♂️",
            [":man_bouncing_ball:"] = "⛹️\u200D♂️",
            [":man_bouncing_ball::skin-tone-1:"] = "⛹\uD83C\uDFFB\u200D♂️",
            [":man_bouncing_ball::skin-tone-2:"] = "⛹\uD83C\uDFFC\u200D♂️",
            [":man_bouncing_ball::skin-tone-3:"] = "⛹\uD83C\uDFFD\u200D♂️",
            [":man_bouncing_ball::skin-tone-4:"] = "⛹\uD83C\uDFFE\u200D♂️",
            [":man_bouncing_ball::skin-tone-5:"] = "⛹\uD83C\uDFFF\u200D♂️",
            [":man_bouncing_ball_dark_skin_tone:"] = "⛹\uD83C\uDFFF\u200D♂️",
            [":man_bouncing_ball_light_skin_tone:"] = "⛹\uD83C\uDFFB\u200D♂️",
            [":man_bouncing_ball_medium_dark_skin_tone:"] = "⛹\uD83C\uDFFE\u200D♂️",
            [":man_bouncing_ball_medium_light_skin_tone:"] = "⛹\uD83C\uDFFC\u200D♂️",
            [":man_bouncing_ball_medium_skin_tone:"] = "⛹\uD83C\uDFFD\u200D♂️",
            [":man_bouncing_ball_tone1:"] = "⛹\uD83C\uDFFB\u200D♂️",
            [":man_bouncing_ball_tone2:"] = "⛹\uD83C\uDFFC\u200D♂️",
            [":man_bouncing_ball_tone3:"] = "⛹\uD83C\uDFFD\u200D♂️",
            [":man_bouncing_ball_tone4:"] = "⛹\uD83C\uDFFE\u200D♂️",
            [":man_bouncing_ball_tone5:"] = "⛹\uD83C\uDFFF\u200D♂️",
            [":man_bowing:"] = "\uD83D\uDE47\u200D♂️",
            [":man_bowing::skin-tone-1:"] = "\uD83D\uDE47\uD83C\uDFFB\u200D♂️",
            [":man_bowing::skin-tone-2:"] = "\uD83D\uDE47\uD83C\uDFFC\u200D♂️",
            [":man_bowing::skin-tone-3:"] = "\uD83D\uDE47\uD83C\uDFFD\u200D♂️",
            [":man_bowing::skin-tone-4:"] = "\uD83D\uDE47\uD83C\uDFFE\u200D♂️",
            [":man_bowing::skin-tone-5:"] = "\uD83D\uDE47\uD83C\uDFFF\u200D♂️",
            [":man_bowing_dark_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFF\u200D♂️",
            [":man_bowing_light_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFB\u200D♂️",
            [":man_bowing_medium_dark_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFE\u200D♂️",
            [":man_bowing_medium_light_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFC\u200D♂️",
            [":man_bowing_medium_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFD\u200D♂️",
            [":man_bowing_tone1:"] = "\uD83D\uDE47\uD83C\uDFFB\u200D♂️",
            [":man_bowing_tone2:"] = "\uD83D\uDE47\uD83C\uDFFC\u200D♂️",
            [":man_bowing_tone3:"] = "\uD83D\uDE47\uD83C\uDFFD\u200D♂️",
            [":man_bowing_tone4:"] = "\uD83D\uDE47\uD83C\uDFFE\u200D♂️",
            [":man_bowing_tone5:"] = "\uD83D\uDE47\uD83C\uDFFF\u200D♂️",
            [":man_cartwheeling:"] = "\uD83E\uDD38\u200D♂️",
            [":man_cartwheeling::skin-tone-1:"] = "\uD83E\uDD38\uD83C\uDFFB\u200D♂️",
            [":man_cartwheeling::skin-tone-2:"] = "\uD83E\uDD38\uD83C\uDFFC\u200D♂️",
            [":man_cartwheeling::skin-tone-3:"] = "\uD83E\uDD38\uD83C\uDFFD\u200D♂️",
            [":man_cartwheeling::skin-tone-4:"] = "\uD83E\uDD38\uD83C\uDFFE\u200D♂️",
            [":man_cartwheeling::skin-tone-5:"] = "\uD83E\uDD38\uD83C\uDFFF\u200D♂️",
            [":man_cartwheeling_dark_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFF\u200D♂️",
            [":man_cartwheeling_light_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFB\u200D♂️",
            [":man_cartwheeling_medium_dark_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFE\u200D♂️",
            [":man_cartwheeling_medium_light_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFC\u200D♂️",
            [":man_cartwheeling_medium_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFD\u200D♂️",
            [":man_cartwheeling_tone1:"] = "\uD83E\uDD38\uD83C\uDFFB\u200D♂️",
            [":man_cartwheeling_tone2:"] = "\uD83E\uDD38\uD83C\uDFFC\u200D♂️",
            [":man_cartwheeling_tone3:"] = "\uD83E\uDD38\uD83C\uDFFD\u200D♂️",
            [":man_cartwheeling_tone4:"] = "\uD83E\uDD38\uD83C\uDFFE\u200D♂️",
            [":man_cartwheeling_tone5:"] = "\uD83E\uDD38\uD83C\uDFFF\u200D♂️",
            [":man_climbing:"] = "\uD83E\uDDD7\u200D♂️",
            [":man_climbing::skin-tone-1:"] = "\uD83E\uDDD7\uD83C\uDFFB\u200D♂️",
            [":man_climbing::skin-tone-2:"] = "\uD83E\uDDD7\uD83C\uDFFC\u200D♂️",
            [":man_climbing::skin-tone-3:"] = "\uD83E\uDDD7\uD83C\uDFFD\u200D♂️",
            [":man_climbing::skin-tone-4:"] = "\uD83E\uDDD7\uD83C\uDFFE\u200D♂️",
            [":man_climbing::skin-tone-5:"] = "\uD83E\uDDD7\uD83C\uDFFF\u200D♂️",
            [":man_climbing_dark_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFF\u200D♂️",
            [":man_climbing_light_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFB\u200D♂️",
            [":man_climbing_medium_dark_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFE\u200D♂️",
            [":man_climbing_medium_light_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFC\u200D♂️",
            [":man_climbing_medium_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFD\u200D♂️",
            [":man_climbing_tone1:"] = "\uD83E\uDDD7\uD83C\uDFFB\u200D♂️",
            [":man_climbing_tone2:"] = "\uD83E\uDDD7\uD83C\uDFFC\u200D♂️",
            [":man_climbing_tone3:"] = "\uD83E\uDDD7\uD83C\uDFFD\u200D♂️",
            [":man_climbing_tone4:"] = "\uD83E\uDDD7\uD83C\uDFFE\u200D♂️",
            [":man_climbing_tone5:"] = "\uD83E\uDDD7\uD83C\uDFFF\u200D♂️",
            [":man_construction_worker:"] = "\uD83D\uDC77\u200D♂️",
            [":man_construction_worker::skin-tone-1:"] = "\uD83D\uDC77\uD83C\uDFFB\u200D♂️",
            [":man_construction_worker::skin-tone-2:"] = "\uD83D\uDC77\uD83C\uDFFC\u200D♂️",
            [":man_construction_worker::skin-tone-3:"] = "\uD83D\uDC77\uD83C\uDFFD\u200D♂️",
            [":man_construction_worker::skin-tone-4:"] = "\uD83D\uDC77\uD83C\uDFFE\u200D♂️",
            [":man_construction_worker::skin-tone-5:"] = "\uD83D\uDC77\uD83C\uDFFF\u200D♂️",
            [":man_construction_worker_dark_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFF\u200D♂️",
            [":man_construction_worker_light_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFB\u200D♂️",
            [":man_construction_worker_medium_dark_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFE\u200D♂️",
            [":man_construction_worker_medium_light_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFC\u200D♂️",
            [":man_construction_worker_medium_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFD\u200D♂️",
            [":man_construction_worker_tone1:"] = "\uD83D\uDC77\uD83C\uDFFB\u200D♂️",
            [":man_construction_worker_tone2:"] = "\uD83D\uDC77\uD83C\uDFFC\u200D♂️",
            [":man_construction_worker_tone3:"] = "\uD83D\uDC77\uD83C\uDFFD\u200D♂️",
            [":man_construction_worker_tone4:"] = "\uD83D\uDC77\uD83C\uDFFE\u200D♂️",
            [":man_construction_worker_tone5:"] = "\uD83D\uDC77\uD83C\uDFFF\u200D♂️",
            [":man_cook:"] = "\uD83D\uDC68\u200D\uD83C\uDF73",
            [":man_cook::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF73",
            [":man_cook::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF73",
            [":man_cook::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF73",
            [":man_cook::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF73",
            [":man_cook::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF73",
            [":man_cook_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF73",
            [":man_cook_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF73",
            [":man_cook_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF73",
            [":man_cook_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF73",
            [":man_cook_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF73",
            [":man_cook_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF73",
            [":man_cook_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF73",
            [":man_cook_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF73",
            [":man_cook_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF73",
            [":man_cook_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF73",
            [":man_curly_haired:"] = "\uD83D\uDC68\u200D\uD83E\uDDB1",
            [":man_curly_haired::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB1",
            [":man_curly_haired::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB1",
            [":man_curly_haired::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB1",
            [":man_curly_haired::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB1",
            [":man_curly_haired::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB1",
            [":man_curly_haired_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB1",
            [":man_curly_haired_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB1",
            [":man_curly_haired_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB1",
            [":man_curly_haired_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB1",
            [":man_curly_haired_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB1",
            [":man_curly_haired_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB1",
            [":man_curly_haired_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB1",
            [":man_curly_haired_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB1",
            [":man_curly_haired_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB1",
            [":man_curly_haired_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB1",
            [":man_dancing:"] = "\uD83D\uDD7A",
            [":man_dancing::skin-tone-1:"] = "\uD83D\uDD7A\uD83C\uDFFB",
            [":man_dancing::skin-tone-2:"] = "\uD83D\uDD7A\uD83C\uDFFC",
            [":man_dancing::skin-tone-3:"] = "\uD83D\uDD7A\uD83C\uDFFD",
            [":man_dancing::skin-tone-4:"] = "\uD83D\uDD7A\uD83C\uDFFE",
            [":man_dancing::skin-tone-5:"] = "\uD83D\uDD7A\uD83C\uDFFF",
            [":man_dancing_tone1:"] = "\uD83D\uDD7A\uD83C\uDFFB",
            [":man_dancing_tone2:"] = "\uD83D\uDD7A\uD83C\uDFFC",
            [":man_dancing_tone3:"] = "\uD83D\uDD7A\uD83C\uDFFD",
            [":man_dancing_tone4:"] = "\uD83D\uDD7A\uD83C\uDFFE",
            [":man_dancing_tone5:"] = "\uD83D\uDD7A\uD83C\uDFFF",
            [":man_dark_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_detective:"] = "\uD83D\uDD75️\u200D♂️",
            [":man_detective::skin-tone-1:"] = "\uD83D\uDD75\uD83C\uDFFB\u200D♂️",
            [":man_detective::skin-tone-2:"] = "\uD83D\uDD75\uD83C\uDFFC\u200D♂️",
            [":man_detective::skin-tone-3:"] = "\uD83D\uDD75\uD83C\uDFFD\u200D♂️",
            [":man_detective::skin-tone-4:"] = "\uD83D\uDD75\uD83C\uDFFE\u200D♂️",
            [":man_detective::skin-tone-5:"] = "\uD83D\uDD75\uD83C\uDFFF\u200D♂️",
            [":man_detective_dark_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFF\u200D♂️",
            [":man_detective_light_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFB\u200D♂️",
            [":man_detective_medium_dark_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFE\u200D♂️",
            [":man_detective_medium_light_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFC\u200D♂️",
            [":man_detective_medium_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFD\u200D♂️",
            [":man_detective_tone1:"] = "\uD83D\uDD75\uD83C\uDFFB\u200D♂️",
            [":man_detective_tone2:"] = "\uD83D\uDD75\uD83C\uDFFC\u200D♂️",
            [":man_detective_tone3:"] = "\uD83D\uDD75\uD83C\uDFFD\u200D♂️",
            [":man_detective_tone4:"] = "\uD83D\uDD75\uD83C\uDFFE\u200D♂️",
            [":man_detective_tone5:"] = "\uD83D\uDD75\uD83C\uDFFF\u200D♂️",
            [":man_elf:"] = "\uD83E\uDDDD\u200D♂️",
            [":man_elf::skin-tone-1:"] = "\uD83E\uDDDD\uD83C\uDFFB\u200D♂️",
            [":man_elf::skin-tone-2:"] = "\uD83E\uDDDD\uD83C\uDFFC\u200D♂️",
            [":man_elf::skin-tone-3:"] = "\uD83E\uDDDD\uD83C\uDFFD\u200D♂️",
            [":man_elf::skin-tone-4:"] = "\uD83E\uDDDD\uD83C\uDFFE\u200D♂️",
            [":man_elf::skin-tone-5:"] = "\uD83E\uDDDD\uD83C\uDFFF\u200D♂️",
            [":man_elf_dark_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFF\u200D♂️",
            [":man_elf_light_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFB\u200D♂️",
            [":man_elf_medium_dark_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFE\u200D♂️",
            [":man_elf_medium_light_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFC\u200D♂️",
            [":man_elf_medium_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFD\u200D♂️",
            [":man_elf_tone1:"] = "\uD83E\uDDDD\uD83C\uDFFB\u200D♂️",
            [":man_elf_tone2:"] = "\uD83E\uDDDD\uD83C\uDFFC\u200D♂️",
            [":man_elf_tone3:"] = "\uD83E\uDDDD\uD83C\uDFFD\u200D♂️",
            [":man_elf_tone4:"] = "\uD83E\uDDDD\uD83C\uDFFE\u200D♂️",
            [":man_elf_tone5:"] = "\uD83E\uDDDD\uD83C\uDFFF\u200D♂️",
            [":man_facepalming:"] = "\uD83E\uDD26\u200D♂️",
            [":man_facepalming::skin-tone-1:"] = "\uD83E\uDD26\uD83C\uDFFB\u200D♂️",
            [":man_facepalming::skin-tone-2:"] = "\uD83E\uDD26\uD83C\uDFFC\u200D♂️",
            [":man_facepalming::skin-tone-3:"] = "\uD83E\uDD26\uD83C\uDFFD\u200D♂️",
            [":man_facepalming::skin-tone-4:"] = "\uD83E\uDD26\uD83C\uDFFE\u200D♂️",
            [":man_facepalming::skin-tone-5:"] = "\uD83E\uDD26\uD83C\uDFFF\u200D♂️",
            [":man_facepalming_dark_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFF\u200D♂️",
            [":man_facepalming_light_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFB\u200D♂️",
            [":man_facepalming_medium_dark_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFE\u200D♂️",
            [":man_facepalming_medium_light_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFC\u200D♂️",
            [":man_facepalming_medium_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFD\u200D♂️",
            [":man_facepalming_tone1:"] = "\uD83E\uDD26\uD83C\uDFFB\u200D♂️",
            [":man_facepalming_tone2:"] = "\uD83E\uDD26\uD83C\uDFFC\u200D♂️",
            [":man_facepalming_tone3:"] = "\uD83E\uDD26\uD83C\uDFFD\u200D♂️",
            [":man_facepalming_tone4:"] = "\uD83E\uDD26\uD83C\uDFFE\u200D♂️",
            [":man_facepalming_tone5:"] = "\uD83E\uDD26\uD83C\uDFFF\u200D♂️",
            [":man_factory_worker:"] = "\uD83D\uDC68\u200D\uD83C\uDFED",
            [":man_factory_worker::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFED",
            [":man_factory_worker::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFED",
            [":man_factory_worker::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFED",
            [":man_factory_worker::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFED",
            [":man_factory_worker::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFED",
            [":man_factory_worker_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFED",
            [":man_factory_worker_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFED",
            [":man_factory_worker_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFED",
            [":man_factory_worker_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFED",
            [":man_factory_worker_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFED",
            [":man_factory_worker_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFED",
            [":man_factory_worker_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFED",
            [":man_factory_worker_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFED",
            [":man_factory_worker_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFED",
            [":man_factory_worker_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFED",
            [":man_fairy:"] = "\uD83E\uDDDA\u200D♂️",
            [":man_fairy::skin-tone-1:"] = "\uD83E\uDDDA\uD83C\uDFFB\u200D♂️",
            [":man_fairy::skin-tone-2:"] = "\uD83E\uDDDA\uD83C\uDFFC\u200D♂️",
            [":man_fairy::skin-tone-3:"] = "\uD83E\uDDDA\uD83C\uDFFD\u200D♂️",
            [":man_fairy::skin-tone-4:"] = "\uD83E\uDDDA\uD83C\uDFFE\u200D♂️",
            [":man_fairy::skin-tone-5:"] = "\uD83E\uDDDA\uD83C\uDFFF\u200D♂️",
            [":man_fairy_dark_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFF\u200D♂️",
            [":man_fairy_light_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFB\u200D♂️",
            [":man_fairy_medium_dark_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFE\u200D♂️",
            [":man_fairy_medium_light_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFC\u200D♂️",
            [":man_fairy_medium_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFD\u200D♂️",
            [":man_fairy_tone1:"] = "\uD83E\uDDDA\uD83C\uDFFB\u200D♂️",
            [":man_fairy_tone2:"] = "\uD83E\uDDDA\uD83C\uDFFC\u200D♂️",
            [":man_fairy_tone3:"] = "\uD83E\uDDDA\uD83C\uDFFD\u200D♂️",
            [":man_fairy_tone4:"] = "\uD83E\uDDDA\uD83C\uDFFE\u200D♂️",
            [":man_fairy_tone5:"] = "\uD83E\uDDDA\uD83C\uDFFF\u200D♂️",
            [":man_farmer:"] = "\uD83D\uDC68\u200D\uD83C\uDF3E",
            [":man_farmer::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF3E",
            [":man_farmer::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF3E",
            [":man_farmer::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF3E",
            [":man_farmer::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF3E",
            [":man_farmer::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF3E",
            [":man_farmer_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF3E",
            [":man_farmer_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF3E",
            [":man_farmer_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF3E",
            [":man_farmer_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF3E",
            [":man_farmer_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF3E",
            [":man_farmer_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF3E",
            [":man_farmer_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF3E",
            [":man_farmer_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF3E",
            [":man_farmer_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF3E",
            [":man_farmer_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF3E",
            [":man_feeding_baby:"] = "\uD83D\uDC68\u200D\uD83C\uDF7C",
            [":man_feeding_baby::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":man_feeding_baby::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":man_feeding_baby::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":man_feeding_baby::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":man_feeding_baby::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":man_feeding_baby_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":man_feeding_baby_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":man_feeding_baby_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":man_feeding_baby_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":man_feeding_baby_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":man_feeding_baby_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":man_feeding_baby_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":man_feeding_baby_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":man_feeding_baby_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":man_feeding_baby_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":man_firefighter:"] = "\uD83D\uDC68\u200D\uD83D\uDE92",
            [":man_firefighter::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDE92",
            [":man_firefighter::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDE92",
            [":man_firefighter::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDE92",
            [":man_firefighter::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDE92",
            [":man_firefighter::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDE92",
            [":man_firefighter_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDE92",
            [":man_firefighter_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDE92",
            [":man_firefighter_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDE92",
            [":man_firefighter_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDE92",
            [":man_firefighter_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDE92",
            [":man_firefighter_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDE92",
            [":man_firefighter_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDE92",
            [":man_firefighter_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDE92",
            [":man_firefighter_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDE92",
            [":man_firefighter_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDE92",
            [":man_frowning:"] = "\uD83D\uDE4D\u200D♂️",
            [":man_frowning::skin-tone-1:"] = "\uD83D\uDE4D\uD83C\uDFFB\u200D♂️",
            [":man_frowning::skin-tone-2:"] = "\uD83D\uDE4D\uD83C\uDFFC\u200D♂️",
            [":man_frowning::skin-tone-3:"] = "\uD83D\uDE4D\uD83C\uDFFD\u200D♂️",
            [":man_frowning::skin-tone-4:"] = "\uD83D\uDE4D\uD83C\uDFFE\u200D♂️",
            [":man_frowning::skin-tone-5:"] = "\uD83D\uDE4D\uD83C\uDFFF\u200D♂️",
            [":man_frowning_dark_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFF\u200D♂️",
            [":man_frowning_light_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFB\u200D♂️",
            [":man_frowning_medium_dark_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFE\u200D♂️",
            [":man_frowning_medium_light_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFC\u200D♂️",
            [":man_frowning_medium_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFD\u200D♂️",
            [":man_frowning_tone1:"] = "\uD83D\uDE4D\uD83C\uDFFB\u200D♂️",
            [":man_frowning_tone2:"] = "\uD83D\uDE4D\uD83C\uDFFC\u200D♂️",
            [":man_frowning_tone3:"] = "\uD83D\uDE4D\uD83C\uDFFD\u200D♂️",
            [":man_frowning_tone4:"] = "\uD83D\uDE4D\uD83C\uDFFE\u200D♂️",
            [":man_frowning_tone5:"] = "\uD83D\uDE4D\uD83C\uDFFF\u200D♂️",
            [":man_genie:"] = "\uD83E\uDDDE\u200D♂️",
            [":man_gesturing_no:"] = "\uD83D\uDE45\u200D♂️",
            [":man_gesturing_no::skin-tone-1:"] = "\uD83D\uDE45\uD83C\uDFFB\u200D♂️",
            [":man_gesturing_no::skin-tone-2:"] = "\uD83D\uDE45\uD83C\uDFFC\u200D♂️",
            [":man_gesturing_no::skin-tone-3:"] = "\uD83D\uDE45\uD83C\uDFFD\u200D♂️",
            [":man_gesturing_no::skin-tone-4:"] = "\uD83D\uDE45\uD83C\uDFFE\u200D♂️",
            [":man_gesturing_no::skin-tone-5:"] = "\uD83D\uDE45\uD83C\uDFFF\u200D♂️",
            [":man_gesturing_no_dark_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFF\u200D♂️",
            [":man_gesturing_no_light_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFB\u200D♂️",
            [":man_gesturing_no_medium_dark_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFE\u200D♂️",
            [":man_gesturing_no_medium_light_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFC\u200D♂️",
            [":man_gesturing_no_medium_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFD\u200D♂️",
            [":man_gesturing_no_tone1:"] = "\uD83D\uDE45\uD83C\uDFFB\u200D♂️",
            [":man_gesturing_no_tone2:"] = "\uD83D\uDE45\uD83C\uDFFC\u200D♂️",
            [":man_gesturing_no_tone3:"] = "\uD83D\uDE45\uD83C\uDFFD\u200D♂️",
            [":man_gesturing_no_tone4:"] = "\uD83D\uDE45\uD83C\uDFFE\u200D♂️",
            [":man_gesturing_no_tone5:"] = "\uD83D\uDE45\uD83C\uDFFF\u200D♂️",
            [":man_gesturing_ok:"] = "\uD83D\uDE46\u200D♂️",
            [":man_gesturing_ok::skin-tone-1:"] = "\uD83D\uDE46\uD83C\uDFFB\u200D♂️",
            [":man_gesturing_ok::skin-tone-2:"] = "\uD83D\uDE46\uD83C\uDFFC\u200D♂️",
            [":man_gesturing_ok::skin-tone-3:"] = "\uD83D\uDE46\uD83C\uDFFD\u200D♂️",
            [":man_gesturing_ok::skin-tone-4:"] = "\uD83D\uDE46\uD83C\uDFFE\u200D♂️",
            [":man_gesturing_ok::skin-tone-5:"] = "\uD83D\uDE46\uD83C\uDFFF\u200D♂️",
            [":man_gesturing_ok_dark_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFF\u200D♂️",
            [":man_gesturing_ok_light_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFB\u200D♂️",
            [":man_gesturing_ok_medium_dark_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFE\u200D♂️",
            [":man_gesturing_ok_medium_light_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFC\u200D♂️",
            [":man_gesturing_ok_medium_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFD\u200D♂️",
            [":man_gesturing_ok_tone1:"] = "\uD83D\uDE46\uD83C\uDFFB\u200D♂️",
            [":man_gesturing_ok_tone2:"] = "\uD83D\uDE46\uD83C\uDFFC\u200D♂️",
            [":man_gesturing_ok_tone3:"] = "\uD83D\uDE46\uD83C\uDFFD\u200D♂️",
            [":man_gesturing_ok_tone4:"] = "\uD83D\uDE46\uD83C\uDFFE\u200D♂️",
            [":man_gesturing_ok_tone5:"] = "\uD83D\uDE46\uD83C\uDFFF\u200D♂️",
            [":man_getting_face_massage:"] = "\uD83D\uDC86\u200D♂️",
            [":man_getting_face_massage::skin-tone-1:"] = "\uD83D\uDC86\uD83C\uDFFB\u200D♂️",
            [":man_getting_face_massage::skin-tone-2:"] = "\uD83D\uDC86\uD83C\uDFFC\u200D♂️",
            [":man_getting_face_massage::skin-tone-3:"] = "\uD83D\uDC86\uD83C\uDFFD\u200D♂️",
            [":man_getting_face_massage::skin-tone-4:"] = "\uD83D\uDC86\uD83C\uDFFE\u200D♂️",
            [":man_getting_face_massage::skin-tone-5:"] = "\uD83D\uDC86\uD83C\uDFFF\u200D♂️",
            [":man_getting_face_massage_dark_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFF\u200D♂️",
            [":man_getting_face_massage_light_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFB\u200D♂️",
            [":man_getting_face_massage_medium_dark_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFE\u200D♂️",
            [":man_getting_face_massage_medium_light_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFC\u200D♂️",
            [":man_getting_face_massage_medium_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFD\u200D♂️",
            [":man_getting_face_massage_tone1:"] = "\uD83D\uDC86\uD83C\uDFFB\u200D♂️",
            [":man_getting_face_massage_tone2:"] = "\uD83D\uDC86\uD83C\uDFFC\u200D♂️",
            [":man_getting_face_massage_tone3:"] = "\uD83D\uDC86\uD83C\uDFFD\u200D♂️",
            [":man_getting_face_massage_tone4:"] = "\uD83D\uDC86\uD83C\uDFFE\u200D♂️",
            [":man_getting_face_massage_tone5:"] = "\uD83D\uDC86\uD83C\uDFFF\u200D♂️",
            [":man_getting_haircut:"] = "\uD83D\uDC87\u200D♂️",
            [":man_getting_haircut::skin-tone-1:"] = "\uD83D\uDC87\uD83C\uDFFB\u200D♂️",
            [":man_getting_haircut::skin-tone-2:"] = "\uD83D\uDC87\uD83C\uDFFC\u200D♂️",
            [":man_getting_haircut::skin-tone-3:"] = "\uD83D\uDC87\uD83C\uDFFD\u200D♂️",
            [":man_getting_haircut::skin-tone-4:"] = "\uD83D\uDC87\uD83C\uDFFE\u200D♂️",
            [":man_getting_haircut::skin-tone-5:"] = "\uD83D\uDC87\uD83C\uDFFF\u200D♂️",
            [":man_getting_haircut_dark_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFF\u200D♂️",
            [":man_getting_haircut_light_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFB\u200D♂️",
            [":man_getting_haircut_medium_dark_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFE\u200D♂️",
            [":man_getting_haircut_medium_light_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFC\u200D♂️",
            [":man_getting_haircut_medium_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFD\u200D♂️",
            [":man_getting_haircut_tone1:"] = "\uD83D\uDC87\uD83C\uDFFB\u200D♂️",
            [":man_getting_haircut_tone2:"] = "\uD83D\uDC87\uD83C\uDFFC\u200D♂️",
            [":man_getting_haircut_tone3:"] = "\uD83D\uDC87\uD83C\uDFFD\u200D♂️",
            [":man_getting_haircut_tone4:"] = "\uD83D\uDC87\uD83C\uDFFE\u200D♂️",
            [":man_getting_haircut_tone5:"] = "\uD83D\uDC87\uD83C\uDFFF\u200D♂️",
            [":man_golfing:"] = "\uD83C\uDFCC️\u200D♂️",
            [":man_golfing::skin-tone-1:"] = "\uD83C\uDFCC\uD83C\uDFFB\u200D♂️",
            [":man_golfing::skin-tone-2:"] = "\uD83C\uDFCC\uD83C\uDFFC\u200D♂️",
            [":man_golfing::skin-tone-3:"] = "\uD83C\uDFCC\uD83C\uDFFD\u200D♂️",
            [":man_golfing::skin-tone-4:"] = "\uD83C\uDFCC\uD83C\uDFFE\u200D♂️",
            [":man_golfing::skin-tone-5:"] = "\uD83C\uDFCC\uD83C\uDFFF\u200D♂️",
            [":man_golfing_dark_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFF\u200D♂️",
            [":man_golfing_light_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFB\u200D♂️",
            [":man_golfing_medium_dark_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFE\u200D♂️",
            [":man_golfing_medium_light_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFC\u200D♂️",
            [":man_golfing_medium_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFD\u200D♂️",
            [":man_golfing_tone1:"] = "\uD83C\uDFCC\uD83C\uDFFB\u200D♂️",
            [":man_golfing_tone2:"] = "\uD83C\uDFCC\uD83C\uDFFC\u200D♂️",
            [":man_golfing_tone3:"] = "\uD83C\uDFCC\uD83C\uDFFD\u200D♂️",
            [":man_golfing_tone4:"] = "\uD83C\uDFCC\uD83C\uDFFE\u200D♂️",
            [":man_golfing_tone5:"] = "\uD83C\uDFCC\uD83C\uDFFF\u200D♂️",
            [":man_guard:"] = "\uD83D\uDC82\u200D♂️",
            [":man_guard::skin-tone-1:"] = "\uD83D\uDC82\uD83C\uDFFB\u200D♂️",
            [":man_guard::skin-tone-2:"] = "\uD83D\uDC82\uD83C\uDFFC\u200D♂️",
            [":man_guard::skin-tone-3:"] = "\uD83D\uDC82\uD83C\uDFFD\u200D♂️",
            [":man_guard::skin-tone-4:"] = "\uD83D\uDC82\uD83C\uDFFE\u200D♂️",
            [":man_guard::skin-tone-5:"] = "\uD83D\uDC82\uD83C\uDFFF\u200D♂️",
            [":man_guard_dark_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFF\u200D♂️",
            [":man_guard_light_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFB\u200D♂️",
            [":man_guard_medium_dark_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFE\u200D♂️",
            [":man_guard_medium_light_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFC\u200D♂️",
            [":man_guard_medium_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFD\u200D♂️",
            [":man_guard_tone1:"] = "\uD83D\uDC82\uD83C\uDFFB\u200D♂️",
            [":man_guard_tone2:"] = "\uD83D\uDC82\uD83C\uDFFC\u200D♂️",
            [":man_guard_tone3:"] = "\uD83D\uDC82\uD83C\uDFFD\u200D♂️",
            [":man_guard_tone4:"] = "\uD83D\uDC82\uD83C\uDFFE\u200D♂️",
            [":man_guard_tone5:"] = "\uD83D\uDC82\uD83C\uDFFF\u200D♂️",
            [":man_health_worker:"] = "\uD83D\uDC68\u200D⚕️",
            [":man_health_worker::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D⚕️",
            [":man_health_worker::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D⚕️",
            [":man_health_worker::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D⚕️",
            [":man_health_worker::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D⚕️",
            [":man_health_worker::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D⚕️",
            [":man_health_worker_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D⚕️",
            [":man_health_worker_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D⚕️",
            [":man_health_worker_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D⚕️",
            [":man_health_worker_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D⚕️",
            [":man_health_worker_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D⚕️",
            [":man_health_worker_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D⚕️",
            [":man_health_worker_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D⚕️",
            [":man_health_worker_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D⚕️",
            [":man_health_worker_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D⚕️",
            [":man_health_worker_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D⚕️",
            [":man_in_business_suit_levitating:"] = "\uD83D\uDD74️",
            [":man_in_business_suit_levitating::skin-tone-1:"] = "\uD83D\uDD74\uD83C\uDFFB",
            [":man_in_business_suit_levitating::skin-tone-2:"] = "\uD83D\uDD74\uD83C\uDFFC",
            [":man_in_business_suit_levitating::skin-tone-3:"] = "\uD83D\uDD74\uD83C\uDFFD",
            [":man_in_business_suit_levitating::skin-tone-4:"] = "\uD83D\uDD74\uD83C\uDFFE",
            [":man_in_business_suit_levitating::skin-tone-5:"] = "\uD83D\uDD74\uD83C\uDFFF",
            [":man_in_business_suit_levitating_dark_skin_tone:"] = "\uD83D\uDD74\uD83C\uDFFF",
            [":man_in_business_suit_levitating_light_skin_tone:"] = "\uD83D\uDD74\uD83C\uDFFB",
            [":man_in_business_suit_levitating_medium_dark_skin_tone:"] = "\uD83D\uDD74\uD83C\uDFFE",
            [":man_in_business_suit_levitating_medium_light_skin_tone:"] = "\uD83D\uDD74\uD83C\uDFFC",
            [":man_in_business_suit_levitating_medium_skin_tone:"] = "\uD83D\uDD74\uD83C\uDFFD",
            [":man_in_business_suit_levitating_tone1:"] = "\uD83D\uDD74\uD83C\uDFFB",
            [":man_in_business_suit_levitating_tone2:"] = "\uD83D\uDD74\uD83C\uDFFC",
            [":man_in_business_suit_levitating_tone3:"] = "\uD83D\uDD74\uD83C\uDFFD",
            [":man_in_business_suit_levitating_tone4:"] = "\uD83D\uDD74\uD83C\uDFFE",
            [":man_in_business_suit_levitating_tone5:"] = "\uD83D\uDD74\uD83C\uDFFF",
            [":man_in_lotus_position:"] = "\uD83E\uDDD8\u200D♂️",
            [":man_in_lotus_position::skin-tone-1:"] = "\uD83E\uDDD8\uD83C\uDFFB\u200D♂️",
            [":man_in_lotus_position::skin-tone-2:"] = "\uD83E\uDDD8\uD83C\uDFFC\u200D♂️",
            [":man_in_lotus_position::skin-tone-3:"] = "\uD83E\uDDD8\uD83C\uDFFD\u200D♂️",
            [":man_in_lotus_position::skin-tone-4:"] = "\uD83E\uDDD8\uD83C\uDFFE\u200D♂️",
            [":man_in_lotus_position::skin-tone-5:"] = "\uD83E\uDDD8\uD83C\uDFFF\u200D♂️",
            [":man_in_lotus_position_dark_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFF\u200D♂️",
            [":man_in_lotus_position_light_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFB\u200D♂️",
            [":man_in_lotus_position_medium_dark_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFE\u200D♂️",
            [":man_in_lotus_position_medium_light_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFC\u200D♂️",
            [":man_in_lotus_position_medium_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFD\u200D♂️",
            [":man_in_lotus_position_tone1:"] = "\uD83E\uDDD8\uD83C\uDFFB\u200D♂️",
            [":man_in_lotus_position_tone2:"] = "\uD83E\uDDD8\uD83C\uDFFC\u200D♂️",
            [":man_in_lotus_position_tone3:"] = "\uD83E\uDDD8\uD83C\uDFFD\u200D♂️",
            [":man_in_lotus_position_tone4:"] = "\uD83E\uDDD8\uD83C\uDFFE\u200D♂️",
            [":man_in_lotus_position_tone5:"] = "\uD83E\uDDD8\uD83C\uDFFF\u200D♂️",
            [":man_in_manual_wheelchair:"] = "\uD83D\uDC68\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDBD",
            [":man_in_manual_wheelchair_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDBD",
            [":man_in_motorized_wheelchair:"] = "\uD83D\uDC68\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDBC",
            [":man_in_motorized_wheelchair_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDBC",
            [":man_in_steamy_room:"] = "\uD83E\uDDD6\u200D♂️",
            [":man_in_steamy_room::skin-tone-1:"] = "\uD83E\uDDD6\uD83C\uDFFB\u200D♂️",
            [":man_in_steamy_room::skin-tone-2:"] = "\uD83E\uDDD6\uD83C\uDFFC\u200D♂️",
            [":man_in_steamy_room::skin-tone-3:"] = "\uD83E\uDDD6\uD83C\uDFFD\u200D♂️",
            [":man_in_steamy_room::skin-tone-4:"] = "\uD83E\uDDD6\uD83C\uDFFE\u200D♂️",
            [":man_in_steamy_room::skin-tone-5:"] = "\uD83E\uDDD6\uD83C\uDFFF\u200D♂️",
            [":man_in_steamy_room_dark_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFF\u200D♂️",
            [":man_in_steamy_room_light_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFB\u200D♂️",
            [":man_in_steamy_room_medium_dark_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFE\u200D♂️",
            [":man_in_steamy_room_medium_light_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFC\u200D♂️",
            [":man_in_steamy_room_medium_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFD\u200D♂️",
            [":man_in_steamy_room_tone1:"] = "\uD83E\uDDD6\uD83C\uDFFB\u200D♂️",
            [":man_in_steamy_room_tone2:"] = "\uD83E\uDDD6\uD83C\uDFFC\u200D♂️",
            [":man_in_steamy_room_tone3:"] = "\uD83E\uDDD6\uD83C\uDFFD\u200D♂️",
            [":man_in_steamy_room_tone4:"] = "\uD83E\uDDD6\uD83C\uDFFE\u200D♂️",
            [":man_in_steamy_room_tone5:"] = "\uD83E\uDDD6\uD83C\uDFFF\u200D♂️",
            [":man_in_tuxedo:"] = "\uD83E\uDD35",
            [":man_in_tuxedo:"] = "\uD83E\uDD35\u200D\u2642\uFE0F",
            [":man_in_tuxedo::skin-tone-1:"] = "\uD83E\uDD35\uD83C\uDFFB",
            [":man_in_tuxedo::skin-tone-1:"] = "\uD83E\uDD35\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_in_tuxedo::skin-tone-2:"] = "\uD83E\uDD35\uD83C\uDFFC",
            [":man_in_tuxedo::skin-tone-2:"] = "\uD83E\uDD35\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_in_tuxedo::skin-tone-3:"] = "\uD83E\uDD35\uD83C\uDFFD",
            [":man_in_tuxedo::skin-tone-3:"] = "\uD83E\uDD35\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_in_tuxedo::skin-tone-4:"] = "\uD83E\uDD35\uD83C\uDFFE",
            [":man_in_tuxedo::skin-tone-4:"] = "\uD83E\uDD35\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_in_tuxedo::skin-tone-5:"] = "\uD83E\uDD35\uD83C\uDFFF",
            [":man_in_tuxedo::skin-tone-5:"] = "\uD83E\uDD35\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_in_tuxedo_dark_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_in_tuxedo_light_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_in_tuxedo_medium_dark_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_in_tuxedo_medium_light_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_in_tuxedo_medium_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_in_tuxedo_tone1:"] = "\uD83E\uDD35\uD83C\uDFFB",
            [":man_in_tuxedo_tone1:"] = "\uD83E\uDD35\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_in_tuxedo_tone2:"] = "\uD83E\uDD35\uD83C\uDFFC",
            [":man_in_tuxedo_tone2:"] = "\uD83E\uDD35\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_in_tuxedo_tone3:"] = "\uD83E\uDD35\uD83C\uDFFD",
            [":man_in_tuxedo_tone3:"] = "\uD83E\uDD35\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_in_tuxedo_tone4:"] = "\uD83E\uDD35\uD83C\uDFFE",
            [":man_in_tuxedo_tone4:"] = "\uD83E\uDD35\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_in_tuxedo_tone5:"] = "\uD83E\uDD35\uD83C\uDFFF",
            [":man_in_tuxedo_tone5:"] = "\uD83E\uDD35\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_judge:"] = "\uD83D\uDC68\u200D⚖️",
            [":man_judge::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D⚖️",
            [":man_judge::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D⚖️",
            [":man_judge::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D⚖️",
            [":man_judge::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D⚖️",
            [":man_judge::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D⚖️",
            [":man_judge_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D⚖️",
            [":man_judge_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D⚖️",
            [":man_judge_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D⚖️",
            [":man_judge_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D⚖️",
            [":man_judge_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D⚖️",
            [":man_judge_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D⚖️",
            [":man_judge_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D⚖️",
            [":man_judge_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D⚖️",
            [":man_judge_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D⚖️",
            [":man_judge_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D⚖️",
            [":man_juggling:"] = "\uD83E\uDD39\u200D♂️",
            [":man_juggling::skin-tone-1:"] = "\uD83E\uDD39\uD83C\uDFFB\u200D♂️",
            [":man_juggling::skin-tone-2:"] = "\uD83E\uDD39\uD83C\uDFFC\u200D♂️",
            [":man_juggling::skin-tone-3:"] = "\uD83E\uDD39\uD83C\uDFFD\u200D♂️",
            [":man_juggling::skin-tone-4:"] = "\uD83E\uDD39\uD83C\uDFFE\u200D♂️",
            [":man_juggling::skin-tone-5:"] = "\uD83E\uDD39\uD83C\uDFFF\u200D♂️",
            [":man_juggling_dark_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFF\u200D♂️",
            [":man_juggling_light_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFB\u200D♂️",
            [":man_juggling_medium_dark_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFE\u200D♂️",
            [":man_juggling_medium_light_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFC\u200D♂️",
            [":man_juggling_medium_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFD\u200D♂️",
            [":man_juggling_tone1:"] = "\uD83E\uDD39\uD83C\uDFFB\u200D♂️",
            [":man_juggling_tone2:"] = "\uD83E\uDD39\uD83C\uDFFC\u200D♂️",
            [":man_juggling_tone3:"] = "\uD83E\uDD39\uD83C\uDFFD\u200D♂️",
            [":man_juggling_tone4:"] = "\uD83E\uDD39\uD83C\uDFFE\u200D♂️",
            [":man_juggling_tone5:"] = "\uD83E\uDD39\uD83C\uDFFF\u200D♂️",
            [":man_kneeling:"] = "\uD83E\uDDCE\u200D♂️",
            [":man_kneeling::skin-tone-1:"] = "\uD83E\uDDCE\uD83C\uDFFB\u200D♂️",
            [":man_kneeling::skin-tone-2:"] = "\uD83E\uDDCE\uD83C\uDFFC\u200D♂️",
            [":man_kneeling::skin-tone-3:"] = "\uD83E\uDDCE\uD83C\uDFFD\u200D♂️",
            [":man_kneeling::skin-tone-4:"] = "\uD83E\uDDCE\uD83C\uDFFE\u200D♂️",
            [":man_kneeling::skin-tone-5:"] = "\uD83E\uDDCE\uD83C\uDFFF\u200D♂️",
            [":man_kneeling_dark_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFF\u200D♂️",
            [":man_kneeling_light_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFB\u200D♂️",
            [":man_kneeling_medium_dark_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFE\u200D♂️",
            [":man_kneeling_medium_light_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFC\u200D♂️",
            [":man_kneeling_medium_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFD\u200D♂️",
            [":man_kneeling_tone1:"] = "\uD83E\uDDCE\uD83C\uDFFB\u200D♂️",
            [":man_kneeling_tone2:"] = "\uD83E\uDDCE\uD83C\uDFFC\u200D♂️",
            [":man_kneeling_tone3:"] = "\uD83E\uDDCE\uD83C\uDFFD\u200D♂️",
            [":man_kneeling_tone4:"] = "\uD83E\uDDCE\uD83C\uDFFE\u200D♂️",
            [":man_kneeling_tone5:"] = "\uD83E\uDDCE\uD83C\uDFFF\u200D♂️",
            [":man_lifting_weights:"] = "\uD83C\uDFCB️\u200D♂️",
            [":man_lifting_weights::skin-tone-1:"] = "\uD83C\uDFCB\uD83C\uDFFB\u200D♂️",
            [":man_lifting_weights::skin-tone-2:"] = "\uD83C\uDFCB\uD83C\uDFFC\u200D♂️",
            [":man_lifting_weights::skin-tone-3:"] = "\uD83C\uDFCB\uD83C\uDFFD\u200D♂️",
            [":man_lifting_weights::skin-tone-4:"] = "\uD83C\uDFCB\uD83C\uDFFE\u200D♂️",
            [":man_lifting_weights::skin-tone-5:"] = "\uD83C\uDFCB\uD83C\uDFFF\u200D♂️",
            [":man_lifting_weights_dark_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFF\u200D♂️",
            [":man_lifting_weights_light_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFB\u200D♂️",
            [":man_lifting_weights_medium_dark_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFE\u200D♂️",
            [":man_lifting_weights_medium_light_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFC\u200D♂️",
            [":man_lifting_weights_medium_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFD\u200D♂️",
            [":man_lifting_weights_tone1:"] = "\uD83C\uDFCB\uD83C\uDFFB\u200D♂️",
            [":man_lifting_weights_tone2:"] = "\uD83C\uDFCB\uD83C\uDFFC\u200D♂️",
            [":man_lifting_weights_tone3:"] = "\uD83C\uDFCB\uD83C\uDFFD\u200D♂️",
            [":man_lifting_weights_tone4:"] = "\uD83C\uDFCB\uD83C\uDFFE\u200D♂️",
            [":man_lifting_weights_tone5:"] = "\uD83C\uDFCB\uD83C\uDFFF\u200D♂️",
            [":man_light_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_mage:"] = "\uD83E\uDDD9\u200D♂️",
            [":man_mage::skin-tone-1:"] = "\uD83E\uDDD9\uD83C\uDFFB\u200D♂️",
            [":man_mage::skin-tone-2:"] = "\uD83E\uDDD9\uD83C\uDFFC\u200D♂️",
            [":man_mage::skin-tone-3:"] = "\uD83E\uDDD9\uD83C\uDFFD\u200D♂️",
            [":man_mage::skin-tone-4:"] = "\uD83E\uDDD9\uD83C\uDFFE\u200D♂️",
            [":man_mage::skin-tone-5:"] = "\uD83E\uDDD9\uD83C\uDFFF\u200D♂️",
            [":man_mage_dark_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFF\u200D♂️",
            [":man_mage_light_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFB\u200D♂️",
            [":man_mage_medium_dark_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFE\u200D♂️",
            [":man_mage_medium_light_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFC\u200D♂️",
            [":man_mage_medium_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFD\u200D♂️",
            [":man_mage_tone1:"] = "\uD83E\uDDD9\uD83C\uDFFB\u200D♂️",
            [":man_mage_tone2:"] = "\uD83E\uDDD9\uD83C\uDFFC\u200D♂️",
            [":man_mage_tone3:"] = "\uD83E\uDDD9\uD83C\uDFFD\u200D♂️",
            [":man_mage_tone4:"] = "\uD83E\uDDD9\uD83C\uDFFE\u200D♂️",
            [":man_mage_tone5:"] = "\uD83E\uDDD9\uD83C\uDFFF\u200D♂️",
            [":man_mechanic:"] = "\uD83D\uDC68\u200D\uD83D\uDD27",
            [":man_mechanic::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDD27",
            [":man_mechanic::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDD27",
            [":man_mechanic::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDD27",
            [":man_mechanic::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDD27",
            [":man_mechanic::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDD27",
            [":man_mechanic_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDD27",
            [":man_mechanic_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDD27",
            [":man_mechanic_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDD27",
            [":man_mechanic_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDD27",
            [":man_mechanic_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDD27",
            [":man_mechanic_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDD27",
            [":man_mechanic_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDD27",
            [":man_mechanic_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDD27",
            [":man_mechanic_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDD27",
            [":man_mechanic_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDD27",
            [":man_medium_dark_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_medium_light_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_medium_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_mountain_biking:"] = "\uD83D\uDEB5\u200D♂️",
            [":man_mountain_biking::skin-tone-1:"] = "\uD83D\uDEB5\uD83C\uDFFB\u200D♂️",
            [":man_mountain_biking::skin-tone-2:"] = "\uD83D\uDEB5\uD83C\uDFFC\u200D♂️",
            [":man_mountain_biking::skin-tone-3:"] = "\uD83D\uDEB5\uD83C\uDFFD\u200D♂️",
            [":man_mountain_biking::skin-tone-4:"] = "\uD83D\uDEB5\uD83C\uDFFE\u200D♂️",
            [":man_mountain_biking::skin-tone-5:"] = "\uD83D\uDEB5\uD83C\uDFFF\u200D♂️",
            [":man_mountain_biking_dark_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFF\u200D♂️",
            [":man_mountain_biking_light_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFB\u200D♂️",
            [":man_mountain_biking_medium_dark_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFE\u200D♂️",
            [":man_mountain_biking_medium_light_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFC\u200D♂️",
            [":man_mountain_biking_medium_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFD\u200D♂️",
            [":man_mountain_biking_tone1:"] = "\uD83D\uDEB5\uD83C\uDFFB\u200D♂️",
            [":man_mountain_biking_tone2:"] = "\uD83D\uDEB5\uD83C\uDFFC\u200D♂️",
            [":man_mountain_biking_tone3:"] = "\uD83D\uDEB5\uD83C\uDFFD\u200D♂️",
            [":man_mountain_biking_tone4:"] = "\uD83D\uDEB5\uD83C\uDFFE\u200D♂️",
            [":man_mountain_biking_tone5:"] = "\uD83D\uDEB5\uD83C\uDFFF\u200D♂️",
            [":man_office_worker:"] = "\uD83D\uDC68\u200D\uD83D\uDCBC",
            [":man_office_worker::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDCBC",
            [":man_office_worker::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDCBC",
            [":man_office_worker::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDCBC",
            [":man_office_worker::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDCBC",
            [":man_office_worker::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDCBC",
            [":man_office_worker_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDCBC",
            [":man_office_worker_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDCBC",
            [":man_office_worker_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDCBC",
            [":man_office_worker_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDCBC",
            [":man_office_worker_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDCBC",
            [":man_office_worker_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDCBC",
            [":man_office_worker_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDCBC",
            [":man_office_worker_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDCBC",
            [":man_office_worker_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDCBC",
            [":man_office_worker_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDCBC",
            [":man_pilot:"] = "\uD83D\uDC68\u200D✈️",
            [":man_pilot::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D✈️",
            [":man_pilot::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D✈️",
            [":man_pilot::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D✈️",
            [":man_pilot::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D✈️",
            [":man_pilot::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D✈️",
            [":man_pilot_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D✈️",
            [":man_pilot_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D✈️",
            [":man_pilot_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D✈️",
            [":man_pilot_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D✈️",
            [":man_pilot_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D✈️",
            [":man_pilot_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D✈️",
            [":man_pilot_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D✈️",
            [":man_pilot_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D✈️",
            [":man_pilot_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D✈️",
            [":man_pilot_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D✈️",
            [":man_playing_handball:"] = "\uD83E\uDD3E\u200D♂️",
            [":man_playing_handball::skin-tone-1:"] = "\uD83E\uDD3E\uD83C\uDFFB\u200D♂️",
            [":man_playing_handball::skin-tone-2:"] = "\uD83E\uDD3E\uD83C\uDFFC\u200D♂️",
            [":man_playing_handball::skin-tone-3:"] = "\uD83E\uDD3E\uD83C\uDFFD\u200D♂️",
            [":man_playing_handball::skin-tone-4:"] = "\uD83E\uDD3E\uD83C\uDFFE\u200D♂️",
            [":man_playing_handball::skin-tone-5:"] = "\uD83E\uDD3E\uD83C\uDFFF\u200D♂️",
            [":man_playing_handball_dark_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFF\u200D♂️",
            [":man_playing_handball_light_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFB\u200D♂️",
            [":man_playing_handball_medium_dark_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFE\u200D♂️",
            [":man_playing_handball_medium_light_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFC\u200D♂️",
            [":man_playing_handball_medium_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFD\u200D♂️",
            [":man_playing_handball_tone1:"] = "\uD83E\uDD3E\uD83C\uDFFB\u200D♂️",
            [":man_playing_handball_tone2:"] = "\uD83E\uDD3E\uD83C\uDFFC\u200D♂️",
            [":man_playing_handball_tone3:"] = "\uD83E\uDD3E\uD83C\uDFFD\u200D♂️",
            [":man_playing_handball_tone4:"] = "\uD83E\uDD3E\uD83C\uDFFE\u200D♂️",
            [":man_playing_handball_tone5:"] = "\uD83E\uDD3E\uD83C\uDFFF\u200D♂️",
            [":man_playing_water_polo:"] = "\uD83E\uDD3D\u200D♂️",
            [":man_playing_water_polo::skin-tone-1:"] = "\uD83E\uDD3D\uD83C\uDFFB\u200D♂️",
            [":man_playing_water_polo::skin-tone-2:"] = "\uD83E\uDD3D\uD83C\uDFFC\u200D♂️",
            [":man_playing_water_polo::skin-tone-3:"] = "\uD83E\uDD3D\uD83C\uDFFD\u200D♂️",
            [":man_playing_water_polo::skin-tone-4:"] = "\uD83E\uDD3D\uD83C\uDFFE\u200D♂️",
            [":man_playing_water_polo::skin-tone-5:"] = "\uD83E\uDD3D\uD83C\uDFFF\u200D♂️",
            [":man_playing_water_polo_dark_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFF\u200D♂️",
            [":man_playing_water_polo_light_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFB\u200D♂️",
            [":man_playing_water_polo_medium_dark_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFE\u200D♂️",
            [":man_playing_water_polo_medium_light_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFC\u200D♂️",
            [":man_playing_water_polo_medium_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFD\u200D♂️",
            [":man_playing_water_polo_tone1:"] = "\uD83E\uDD3D\uD83C\uDFFB\u200D♂️",
            [":man_playing_water_polo_tone2:"] = "\uD83E\uDD3D\uD83C\uDFFC\u200D♂️",
            [":man_playing_water_polo_tone3:"] = "\uD83E\uDD3D\uD83C\uDFFD\u200D♂️",
            [":man_playing_water_polo_tone4:"] = "\uD83E\uDD3D\uD83C\uDFFE\u200D♂️",
            [":man_playing_water_polo_tone5:"] = "\uD83E\uDD3D\uD83C\uDFFF\u200D♂️",
            [":man_police_officer:"] = "\uD83D\uDC6E\u200D♂️",
            [":man_police_officer::skin-tone-1:"] = "\uD83D\uDC6E\uD83C\uDFFB\u200D♂️",
            [":man_police_officer::skin-tone-2:"] = "\uD83D\uDC6E\uD83C\uDFFC\u200D♂️",
            [":man_police_officer::skin-tone-3:"] = "\uD83D\uDC6E\uD83C\uDFFD\u200D♂️",
            [":man_police_officer::skin-tone-4:"] = "\uD83D\uDC6E\uD83C\uDFFE\u200D♂️",
            [":man_police_officer::skin-tone-5:"] = "\uD83D\uDC6E\uD83C\uDFFF\u200D♂️",
            [":man_police_officer_dark_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFF\u200D♂️",
            [":man_police_officer_light_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFB\u200D♂️",
            [":man_police_officer_medium_dark_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFE\u200D♂️",
            [":man_police_officer_medium_light_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFC\u200D♂️",
            [":man_police_officer_medium_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFD\u200D♂️",
            [":man_police_officer_tone1:"] = "\uD83D\uDC6E\uD83C\uDFFB\u200D♂️",
            [":man_police_officer_tone2:"] = "\uD83D\uDC6E\uD83C\uDFFC\u200D♂️",
            [":man_police_officer_tone3:"] = "\uD83D\uDC6E\uD83C\uDFFD\u200D♂️",
            [":man_police_officer_tone4:"] = "\uD83D\uDC6E\uD83C\uDFFE\u200D♂️",
            [":man_police_officer_tone5:"] = "\uD83D\uDC6E\uD83C\uDFFF\u200D♂️",
            [":man_pouting:"] = "\uD83D\uDE4E\u200D♂️",
            [":man_pouting::skin-tone-1:"] = "\uD83D\uDE4E\uD83C\uDFFB\u200D♂️",
            [":man_pouting::skin-tone-2:"] = "\uD83D\uDE4E\uD83C\uDFFC\u200D♂️",
            [":man_pouting::skin-tone-3:"] = "\uD83D\uDE4E\uD83C\uDFFD\u200D♂️",
            [":man_pouting::skin-tone-4:"] = "\uD83D\uDE4E\uD83C\uDFFE\u200D♂️",
            [":man_pouting::skin-tone-5:"] = "\uD83D\uDE4E\uD83C\uDFFF\u200D♂️",
            [":man_pouting_dark_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFF\u200D♂️",
            [":man_pouting_light_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFB\u200D♂️",
            [":man_pouting_medium_dark_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFE\u200D♂️",
            [":man_pouting_medium_light_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFC\u200D♂️",
            [":man_pouting_medium_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFD\u200D♂️",
            [":man_pouting_tone1:"] = "\uD83D\uDE4E\uD83C\uDFFB\u200D♂️",
            [":man_pouting_tone2:"] = "\uD83D\uDE4E\uD83C\uDFFC\u200D♂️",
            [":man_pouting_tone3:"] = "\uD83D\uDE4E\uD83C\uDFFD\u200D♂️",
            [":man_pouting_tone4:"] = "\uD83D\uDE4E\uD83C\uDFFE\u200D♂️",
            [":man_pouting_tone5:"] = "\uD83D\uDE4E\uD83C\uDFFF\u200D♂️",
            [":man_raising_hand:"] = "\uD83D\uDE4B\u200D♂️",
            [":man_raising_hand::skin-tone-1:"] = "\uD83D\uDE4B\uD83C\uDFFB\u200D♂️",
            [":man_raising_hand::skin-tone-2:"] = "\uD83D\uDE4B\uD83C\uDFFC\u200D♂️",
            [":man_raising_hand::skin-tone-3:"] = "\uD83D\uDE4B\uD83C\uDFFD\u200D♂️",
            [":man_raising_hand::skin-tone-4:"] = "\uD83D\uDE4B\uD83C\uDFFE\u200D♂️",
            [":man_raising_hand::skin-tone-5:"] = "\uD83D\uDE4B\uD83C\uDFFF\u200D♂️",
            [":man_raising_hand_dark_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFF\u200D♂️",
            [":man_raising_hand_light_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFB\u200D♂️",
            [":man_raising_hand_medium_dark_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFE\u200D♂️",
            [":man_raising_hand_medium_light_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFC\u200D♂️",
            [":man_raising_hand_medium_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFD\u200D♂️",
            [":man_raising_hand_tone1:"] = "\uD83D\uDE4B\uD83C\uDFFB\u200D♂️",
            [":man_raising_hand_tone2:"] = "\uD83D\uDE4B\uD83C\uDFFC\u200D♂️",
            [":man_raising_hand_tone3:"] = "\uD83D\uDE4B\uD83C\uDFFD\u200D♂️",
            [":man_raising_hand_tone4:"] = "\uD83D\uDE4B\uD83C\uDFFE\u200D♂️",
            [":man_raising_hand_tone5:"] = "\uD83D\uDE4B\uD83C\uDFFF\u200D♂️",
            [":man_red_haired:"] = "\uD83D\uDC68\u200D\uD83E\uDDB0",
            [":man_red_haired::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB0",
            [":man_red_haired::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB0",
            [":man_red_haired::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB0",
            [":man_red_haired::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB0",
            [":man_red_haired::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB0",
            [":man_red_haired_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB0",
            [":man_red_haired_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB0",
            [":man_red_haired_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB0",
            [":man_red_haired_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB0",
            [":man_red_haired_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB0",
            [":man_red_haired_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB0",
            [":man_red_haired_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB0",
            [":man_red_haired_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB0",
            [":man_red_haired_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB0",
            [":man_red_haired_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB0",
            [":man_rowing_boat:"] = "\uD83D\uDEA3\u200D♂️",
            [":man_rowing_boat::skin-tone-1:"] = "\uD83D\uDEA3\uD83C\uDFFB\u200D♂️",
            [":man_rowing_boat::skin-tone-2:"] = "\uD83D\uDEA3\uD83C\uDFFC\u200D♂️",
            [":man_rowing_boat::skin-tone-3:"] = "\uD83D\uDEA3\uD83C\uDFFD\u200D♂️",
            [":man_rowing_boat::skin-tone-4:"] = "\uD83D\uDEA3\uD83C\uDFFE\u200D♂️",
            [":man_rowing_boat::skin-tone-5:"] = "\uD83D\uDEA3\uD83C\uDFFF\u200D♂️",
            [":man_rowing_boat_dark_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFF\u200D♂️",
            [":man_rowing_boat_light_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFB\u200D♂️",
            [":man_rowing_boat_medium_dark_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFE\u200D♂️",
            [":man_rowing_boat_medium_light_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFC\u200D♂️",
            [":man_rowing_boat_medium_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFD\u200D♂️",
            [":man_rowing_boat_tone1:"] = "\uD83D\uDEA3\uD83C\uDFFB\u200D♂️",
            [":man_rowing_boat_tone2:"] = "\uD83D\uDEA3\uD83C\uDFFC\u200D♂️",
            [":man_rowing_boat_tone3:"] = "\uD83D\uDEA3\uD83C\uDFFD\u200D♂️",
            [":man_rowing_boat_tone4:"] = "\uD83D\uDEA3\uD83C\uDFFE\u200D♂️",
            [":man_rowing_boat_tone5:"] = "\uD83D\uDEA3\uD83C\uDFFF\u200D♂️",
            [":man_running:"] = "\uD83C\uDFC3\u200D♂️",
            [":man_running::skin-tone-1:"] = "\uD83C\uDFC3\uD83C\uDFFB\u200D♂️",
            [":man_running::skin-tone-2:"] = "\uD83C\uDFC3\uD83C\uDFFC\u200D♂️",
            [":man_running::skin-tone-3:"] = "\uD83C\uDFC3\uD83C\uDFFD\u200D♂️",
            [":man_running::skin-tone-4:"] = "\uD83C\uDFC3\uD83C\uDFFE\u200D♂️",
            [":man_running::skin-tone-5:"] = "\uD83C\uDFC3\uD83C\uDFFF\u200D♂️",
            [":man_running_dark_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFF\u200D♂️",
            [":man_running_light_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFB\u200D♂️",
            [":man_running_medium_dark_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFE\u200D♂️",
            [":man_running_medium_light_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFC\u200D♂️",
            [":man_running_medium_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFD\u200D♂️",
            [":man_running_tone1:"] = "\uD83C\uDFC3\uD83C\uDFFB\u200D♂️",
            [":man_running_tone2:"] = "\uD83C\uDFC3\uD83C\uDFFC\u200D♂️",
            [":man_running_tone3:"] = "\uD83C\uDFC3\uD83C\uDFFD\u200D♂️",
            [":man_running_tone4:"] = "\uD83C\uDFC3\uD83C\uDFFE\u200D♂️",
            [":man_running_tone5:"] = "\uD83C\uDFC3\uD83C\uDFFF\u200D♂️",
            [":man_scientist:"] = "\uD83D\uDC68\u200D\uD83D\uDD2C",
            [":man_scientist::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDD2C",
            [":man_scientist::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDD2C",
            [":man_scientist::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDD2C",
            [":man_scientist::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDD2C",
            [":man_scientist::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDD2C",
            [":man_scientist_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDD2C",
            [":man_scientist_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDD2C",
            [":man_scientist_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDD2C",
            [":man_scientist_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDD2C",
            [":man_scientist_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDD2C",
            [":man_scientist_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDD2C",
            [":man_scientist_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDD2C",
            [":man_scientist_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDD2C",
            [":man_scientist_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDD2C",
            [":man_scientist_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDD2C",
            [":man_shrugging:"] = "\uD83E\uDD37\u200D♂️",
            [":man_shrugging::skin-tone-1:"] = "\uD83E\uDD37\uD83C\uDFFB\u200D♂️",
            [":man_shrugging::skin-tone-2:"] = "\uD83E\uDD37\uD83C\uDFFC\u200D♂️",
            [":man_shrugging::skin-tone-3:"] = "\uD83E\uDD37\uD83C\uDFFD\u200D♂️",
            [":man_shrugging::skin-tone-4:"] = "\uD83E\uDD37\uD83C\uDFFE\u200D♂️",
            [":man_shrugging::skin-tone-5:"] = "\uD83E\uDD37\uD83C\uDFFF\u200D♂️",
            [":man_shrugging_dark_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFF\u200D♂️",
            [":man_shrugging_light_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFB\u200D♂️",
            [":man_shrugging_medium_dark_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFE\u200D♂️",
            [":man_shrugging_medium_light_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFC\u200D♂️",
            [":man_shrugging_medium_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFD\u200D♂️",
            [":man_shrugging_tone1:"] = "\uD83E\uDD37\uD83C\uDFFB\u200D♂️",
            [":man_shrugging_tone2:"] = "\uD83E\uDD37\uD83C\uDFFC\u200D♂️",
            [":man_shrugging_tone3:"] = "\uD83E\uDD37\uD83C\uDFFD\u200D♂️",
            [":man_shrugging_tone4:"] = "\uD83E\uDD37\uD83C\uDFFE\u200D♂️",
            [":man_shrugging_tone5:"] = "\uD83E\uDD37\uD83C\uDFFF\u200D♂️",
            [":man_singer:"] = "\uD83D\uDC68\u200D\uD83C\uDFA4",
            [":man_singer::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFA4",
            [":man_singer::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFA4",
            [":man_singer::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFA4",
            [":man_singer::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFA4",
            [":man_singer::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFA4",
            [":man_singer_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFA4",
            [":man_singer_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFA4",
            [":man_singer_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFA4",
            [":man_singer_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFA4",
            [":man_singer_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFA4",
            [":man_singer_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFA4",
            [":man_singer_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFA4",
            [":man_singer_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFA4",
            [":man_singer_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFA4",
            [":man_singer_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFA4",
            [":man_standing:"] = "\uD83E\uDDCD\u200D♂️",
            [":man_standing::skin-tone-1:"] = "\uD83E\uDDCD\uD83C\uDFFB\u200D♂️",
            [":man_standing::skin-tone-2:"] = "\uD83E\uDDCD\uD83C\uDFFC\u200D♂️",
            [":man_standing::skin-tone-3:"] = "\uD83E\uDDCD\uD83C\uDFFD\u200D♂️",
            [":man_standing::skin-tone-4:"] = "\uD83E\uDDCD\uD83C\uDFFE\u200D♂️",
            [":man_standing::skin-tone-5:"] = "\uD83E\uDDCD\uD83C\uDFFF\u200D♂️",
            [":man_standing_dark_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFF\u200D♂️",
            [":man_standing_light_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFB\u200D♂️",
            [":man_standing_medium_dark_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFE\u200D♂️",
            [":man_standing_medium_light_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFC\u200D♂️",
            [":man_standing_medium_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFD\u200D♂️",
            [":man_standing_tone1:"] = "\uD83E\uDDCD\uD83C\uDFFB\u200D♂️",
            [":man_standing_tone2:"] = "\uD83E\uDDCD\uD83C\uDFFC\u200D♂️",
            [":man_standing_tone3:"] = "\uD83E\uDDCD\uD83C\uDFFD\u200D♂️",
            [":man_standing_tone4:"] = "\uD83E\uDDCD\uD83C\uDFFE\u200D♂️",
            [":man_standing_tone5:"] = "\uD83E\uDDCD\uD83C\uDFFF\u200D♂️",
            [":man_student:"] = "\uD83D\uDC68\u200D\uD83C\uDF93",
            [":man_student::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF93",
            [":man_student::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF93",
            [":man_student::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF93",
            [":man_student::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF93",
            [":man_student::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF93",
            [":man_student_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF93",
            [":man_student_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF93",
            [":man_student_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF93",
            [":man_student_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF93",
            [":man_student_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF93",
            [":man_student_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDF93",
            [":man_student_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDF93",
            [":man_student_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDF93",
            [":man_student_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDF93",
            [":man_student_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDF93",
            [":man_superhero:"] = "\uD83E\uDDB8\u200D♂️",
            [":man_superhero::skin-tone-1:"] = "\uD83E\uDDB8\uD83C\uDFFB\u200D♂️",
            [":man_superhero::skin-tone-2:"] = "\uD83E\uDDB8\uD83C\uDFFC\u200D♂️",
            [":man_superhero::skin-tone-3:"] = "\uD83E\uDDB8\uD83C\uDFFD\u200D♂️",
            [":man_superhero::skin-tone-4:"] = "\uD83E\uDDB8\uD83C\uDFFE\u200D♂️",
            [":man_superhero::skin-tone-5:"] = "\uD83E\uDDB8\uD83C\uDFFF\u200D♂️",
            [":man_superhero_dark_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFF\u200D♂️",
            [":man_superhero_light_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFB\u200D♂️",
            [":man_superhero_medium_dark_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFE\u200D♂️",
            [":man_superhero_medium_light_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFC\u200D♂️",
            [":man_superhero_medium_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFD\u200D♂️",
            [":man_superhero_tone1:"] = "\uD83E\uDDB8\uD83C\uDFFB\u200D♂️",
            [":man_superhero_tone2:"] = "\uD83E\uDDB8\uD83C\uDFFC\u200D♂️",
            [":man_superhero_tone3:"] = "\uD83E\uDDB8\uD83C\uDFFD\u200D♂️",
            [":man_superhero_tone4:"] = "\uD83E\uDDB8\uD83C\uDFFE\u200D♂️",
            [":man_superhero_tone5:"] = "\uD83E\uDDB8\uD83C\uDFFF\u200D♂️",
            [":man_supervillain:"] = "\uD83E\uDDB9\u200D♂️",
            [":man_supervillain::skin-tone-1:"] = "\uD83E\uDDB9\uD83C\uDFFB\u200D♂️",
            [":man_supervillain::skin-tone-2:"] = "\uD83E\uDDB9\uD83C\uDFFC\u200D♂️",
            [":man_supervillain::skin-tone-3:"] = "\uD83E\uDDB9\uD83C\uDFFD\u200D♂️",
            [":man_supervillain::skin-tone-4:"] = "\uD83E\uDDB9\uD83C\uDFFE\u200D♂️",
            [":man_supervillain::skin-tone-5:"] = "\uD83E\uDDB9\uD83C\uDFFF\u200D♂️",
            [":man_supervillain_dark_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFF\u200D♂️",
            [":man_supervillain_light_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFB\u200D♂️",
            [":man_supervillain_medium_dark_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFE\u200D♂️",
            [":man_supervillain_medium_light_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFC\u200D♂️",
            [":man_supervillain_medium_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFD\u200D♂️",
            [":man_supervillain_tone1:"] = "\uD83E\uDDB9\uD83C\uDFFB\u200D♂️",
            [":man_supervillain_tone2:"] = "\uD83E\uDDB9\uD83C\uDFFC\u200D♂️",
            [":man_supervillain_tone3:"] = "\uD83E\uDDB9\uD83C\uDFFD\u200D♂️",
            [":man_supervillain_tone4:"] = "\uD83E\uDDB9\uD83C\uDFFE\u200D♂️",
            [":man_supervillain_tone5:"] = "\uD83E\uDDB9\uD83C\uDFFF\u200D♂️",
            [":man_surfing:"] = "\uD83C\uDFC4\u200D♂️",
            [":man_surfing::skin-tone-1:"] = "\uD83C\uDFC4\uD83C\uDFFB\u200D♂️",
            [":man_surfing::skin-tone-2:"] = "\uD83C\uDFC4\uD83C\uDFFC\u200D♂️",
            [":man_surfing::skin-tone-3:"] = "\uD83C\uDFC4\uD83C\uDFFD\u200D♂️",
            [":man_surfing::skin-tone-4:"] = "\uD83C\uDFC4\uD83C\uDFFE\u200D♂️",
            [":man_surfing::skin-tone-5:"] = "\uD83C\uDFC4\uD83C\uDFFF\u200D♂️",
            [":man_surfing_dark_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFF\u200D♂️",
            [":man_surfing_light_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFB\u200D♂️",
            [":man_surfing_medium_dark_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFE\u200D♂️",
            [":man_surfing_medium_light_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFC\u200D♂️",
            [":man_surfing_medium_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFD\u200D♂️",
            [":man_surfing_tone1:"] = "\uD83C\uDFC4\uD83C\uDFFB\u200D♂️",
            [":man_surfing_tone2:"] = "\uD83C\uDFC4\uD83C\uDFFC\u200D♂️",
            [":man_surfing_tone3:"] = "\uD83C\uDFC4\uD83C\uDFFD\u200D♂️",
            [":man_surfing_tone4:"] = "\uD83C\uDFC4\uD83C\uDFFE\u200D♂️",
            [":man_surfing_tone5:"] = "\uD83C\uDFC4\uD83C\uDFFF\u200D♂️",
            [":man_swimming:"] = "\uD83C\uDFCA\u200D♂️",
            [":man_swimming::skin-tone-1:"] = "\uD83C\uDFCA\uD83C\uDFFB\u200D♂️",
            [":man_swimming::skin-tone-2:"] = "\uD83C\uDFCA\uD83C\uDFFC\u200D♂️",
            [":man_swimming::skin-tone-3:"] = "\uD83C\uDFCA\uD83C\uDFFD\u200D♂️",
            [":man_swimming::skin-tone-4:"] = "\uD83C\uDFCA\uD83C\uDFFE\u200D♂️",
            [":man_swimming::skin-tone-5:"] = "\uD83C\uDFCA\uD83C\uDFFF\u200D♂️",
            [":man_swimming_dark_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFF\u200D♂️",
            [":man_swimming_light_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFB\u200D♂️",
            [":man_swimming_medium_dark_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFE\u200D♂️",
            [":man_swimming_medium_light_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFC\u200D♂️",
            [":man_swimming_medium_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFD\u200D♂️",
            [":man_swimming_tone1:"] = "\uD83C\uDFCA\uD83C\uDFFB\u200D♂️",
            [":man_swimming_tone2:"] = "\uD83C\uDFCA\uD83C\uDFFC\u200D♂️",
            [":man_swimming_tone3:"] = "\uD83C\uDFCA\uD83C\uDFFD\u200D♂️",
            [":man_swimming_tone4:"] = "\uD83C\uDFCA\uD83C\uDFFE\u200D♂️",
            [":man_swimming_tone5:"] = "\uD83C\uDFCA\uD83C\uDFFF\u200D♂️",
            [":man_teacher:"] = "\uD83D\uDC68\u200D\uD83C\uDFEB",
            [":man_teacher::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFEB",
            [":man_teacher::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFEB",
            [":man_teacher::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFEB",
            [":man_teacher::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFEB",
            [":man_teacher::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFEB",
            [":man_teacher_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFEB",
            [":man_teacher_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFEB",
            [":man_teacher_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFEB",
            [":man_teacher_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFEB",
            [":man_teacher_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFEB",
            [":man_teacher_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83C\uDFEB",
            [":man_teacher_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83C\uDFEB",
            [":man_teacher_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83C\uDFEB",
            [":man_teacher_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83C\uDFEB",
            [":man_teacher_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83C\uDFEB",
            [":man_technologist:"] = "\uD83D\uDC68\u200D\uD83D\uDCBB",
            [":man_technologist::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDCBB",
            [":man_technologist::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDCBB",
            [":man_technologist::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDCBB",
            [":man_technologist::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDCBB",
            [":man_technologist::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDCBB",
            [":man_technologist_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDCBB",
            [":man_technologist_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDCBB",
            [":man_technologist_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDCBB",
            [":man_technologist_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDCBB",
            [":man_technologist_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDCBB",
            [":man_technologist_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83D\uDCBB",
            [":man_technologist_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83D\uDCBB",
            [":man_technologist_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83D\uDCBB",
            [":man_technologist_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83D\uDCBB",
            [":man_technologist_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83D\uDCBB",
            [":man_tipping_hand:"] = "\uD83D\uDC81\u200D♂️",
            [":man_tipping_hand::skin-tone-1:"] = "\uD83D\uDC81\uD83C\uDFFB\u200D♂️",
            [":man_tipping_hand::skin-tone-2:"] = "\uD83D\uDC81\uD83C\uDFFC\u200D♂️",
            [":man_tipping_hand::skin-tone-3:"] = "\uD83D\uDC81\uD83C\uDFFD\u200D♂️",
            [":man_tipping_hand::skin-tone-4:"] = "\uD83D\uDC81\uD83C\uDFFE\u200D♂️",
            [":man_tipping_hand::skin-tone-5:"] = "\uD83D\uDC81\uD83C\uDFFF\u200D♂️",
            [":man_tipping_hand_dark_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFF\u200D♂️",
            [":man_tipping_hand_light_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFB\u200D♂️",
            [":man_tipping_hand_medium_dark_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFE\u200D♂️",
            [":man_tipping_hand_medium_light_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFC\u200D♂️",
            [":man_tipping_hand_medium_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFD\u200D♂️",
            [":man_tipping_hand_tone1:"] = "\uD83D\uDC81\uD83C\uDFFB\u200D♂️",
            [":man_tipping_hand_tone2:"] = "\uD83D\uDC81\uD83C\uDFFC\u200D♂️",
            [":man_tipping_hand_tone3:"] = "\uD83D\uDC81\uD83C\uDFFD\u200D♂️",
            [":man_tipping_hand_tone4:"] = "\uD83D\uDC81\uD83C\uDFFE\u200D♂️",
            [":man_tipping_hand_tone5:"] = "\uD83D\uDC81\uD83C\uDFFF\u200D♂️",
            [":man_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB",
            [":man_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC",
            [":man_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD",
            [":man_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE",
            [":man_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF",
            [":man_vampire:"] = "\uD83E\uDDDB\u200D♂️",
            [":man_vampire::skin-tone-1:"] = "\uD83E\uDDDB\uD83C\uDFFB\u200D♂️",
            [":man_vampire::skin-tone-2:"] = "\uD83E\uDDDB\uD83C\uDFFC\u200D♂️",
            [":man_vampire::skin-tone-3:"] = "\uD83E\uDDDB\uD83C\uDFFD\u200D♂️",
            [":man_vampire::skin-tone-4:"] = "\uD83E\uDDDB\uD83C\uDFFE\u200D♂️",
            [":man_vampire::skin-tone-5:"] = "\uD83E\uDDDB\uD83C\uDFFF\u200D♂️",
            [":man_vampire_dark_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFF\u200D♂️",
            [":man_vampire_light_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFB\u200D♂️",
            [":man_vampire_medium_dark_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFE\u200D♂️",
            [":man_vampire_medium_light_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFC\u200D♂️",
            [":man_vampire_medium_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFD\u200D♂️",
            [":man_vampire_tone1:"] = "\uD83E\uDDDB\uD83C\uDFFB\u200D♂️",
            [":man_vampire_tone2:"] = "\uD83E\uDDDB\uD83C\uDFFC\u200D♂️",
            [":man_vampire_tone3:"] = "\uD83E\uDDDB\uD83C\uDFFD\u200D♂️",
            [":man_vampire_tone4:"] = "\uD83E\uDDDB\uD83C\uDFFE\u200D♂️",
            [":man_vampire_tone5:"] = "\uD83E\uDDDB\uD83C\uDFFF\u200D♂️",
            [":man_walking:"] = "\uD83D\uDEB6\u200D♂️",
            [":man_walking::skin-tone-1:"] = "\uD83D\uDEB6\uD83C\uDFFB\u200D♂️",
            [":man_walking::skin-tone-2:"] = "\uD83D\uDEB6\uD83C\uDFFC\u200D♂️",
            [":man_walking::skin-tone-3:"] = "\uD83D\uDEB6\uD83C\uDFFD\u200D♂️",
            [":man_walking::skin-tone-4:"] = "\uD83D\uDEB6\uD83C\uDFFE\u200D♂️",
            [":man_walking::skin-tone-5:"] = "\uD83D\uDEB6\uD83C\uDFFF\u200D♂️",
            [":man_walking_dark_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFF\u200D♂️",
            [":man_walking_light_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFB\u200D♂️",
            [":man_walking_medium_dark_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFE\u200D♂️",
            [":man_walking_medium_light_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFC\u200D♂️",
            [":man_walking_medium_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFD\u200D♂️",
            [":man_walking_tone1:"] = "\uD83D\uDEB6\uD83C\uDFFB\u200D♂️",
            [":man_walking_tone2:"] = "\uD83D\uDEB6\uD83C\uDFFC\u200D♂️",
            [":man_walking_tone3:"] = "\uD83D\uDEB6\uD83C\uDFFD\u200D♂️",
            [":man_walking_tone4:"] = "\uD83D\uDEB6\uD83C\uDFFE\u200D♂️",
            [":man_walking_tone5:"] = "\uD83D\uDEB6\uD83C\uDFFF\u200D♂️",
            [":man_wearing_turban:"] = "\uD83D\uDC73\u200D♂️",
            [":man_wearing_turban::skin-tone-1:"] = "\uD83D\uDC73\uD83C\uDFFB\u200D♂️",
            [":man_wearing_turban::skin-tone-2:"] = "\uD83D\uDC73\uD83C\uDFFC\u200D♂️",
            [":man_wearing_turban::skin-tone-3:"] = "\uD83D\uDC73\uD83C\uDFFD\u200D♂️",
            [":man_wearing_turban::skin-tone-4:"] = "\uD83D\uDC73\uD83C\uDFFE\u200D♂️",
            [":man_wearing_turban::skin-tone-5:"] = "\uD83D\uDC73\uD83C\uDFFF\u200D♂️",
            [":man_wearing_turban_dark_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFF\u200D♂️",
            [":man_wearing_turban_light_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFB\u200D♂️",
            [":man_wearing_turban_medium_dark_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFE\u200D♂️",
            [":man_wearing_turban_medium_light_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFC\u200D♂️",
            [":man_wearing_turban_medium_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFD\u200D♂️",
            [":man_wearing_turban_tone1:"] = "\uD83D\uDC73\uD83C\uDFFB\u200D♂️",
            [":man_wearing_turban_tone2:"] = "\uD83D\uDC73\uD83C\uDFFC\u200D♂️",
            [":man_wearing_turban_tone3:"] = "\uD83D\uDC73\uD83C\uDFFD\u200D♂️",
            [":man_wearing_turban_tone4:"] = "\uD83D\uDC73\uD83C\uDFFE\u200D♂️",
            [":man_wearing_turban_tone5:"] = "\uD83D\uDC73\uD83C\uDFFF\u200D♂️",
            [":man_white_haired:"] = "\uD83D\uDC68\u200D\uD83E\uDDB3",
            [":man_white_haired::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB3",
            [":man_white_haired::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB3",
            [":man_white_haired::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB3",
            [":man_white_haired::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB3",
            [":man_white_haired::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB3",
            [":man_white_haired_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB3",
            [":man_white_haired_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB3",
            [":man_white_haired_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB3",
            [":man_white_haired_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB3",
            [":man_white_haired_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB3",
            [":man_white_haired_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDB3",
            [":man_white_haired_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDB3",
            [":man_white_haired_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDB3",
            [":man_white_haired_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDB3",
            [":man_white_haired_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDB3",
            [":man_with_chinese_cap:"] = "\uD83D\uDC72",
            [":man_with_chinese_cap::skin-tone-1:"] = "\uD83D\uDC72\uD83C\uDFFB",
            [":man_with_chinese_cap::skin-tone-2:"] = "\uD83D\uDC72\uD83C\uDFFC",
            [":man_with_chinese_cap::skin-tone-3:"] = "\uD83D\uDC72\uD83C\uDFFD",
            [":man_with_chinese_cap::skin-tone-4:"] = "\uD83D\uDC72\uD83C\uDFFE",
            [":man_with_chinese_cap::skin-tone-5:"] = "\uD83D\uDC72\uD83C\uDFFF",
            [":man_with_chinese_cap_tone1:"] = "\uD83D\uDC72\uD83C\uDFFB",
            [":man_with_chinese_cap_tone2:"] = "\uD83D\uDC72\uD83C\uDFFC",
            [":man_with_chinese_cap_tone3:"] = "\uD83D\uDC72\uD83C\uDFFD",
            [":man_with_chinese_cap_tone4:"] = "\uD83D\uDC72\uD83C\uDFFE",
            [":man_with_chinese_cap_tone5:"] = "\uD83D\uDC72\uD83C\uDFFF",
            [":man_with_gua_pi_mao:"] = "\uD83D\uDC72",
            [":man_with_gua_pi_mao::skin-tone-1:"] = "\uD83D\uDC72\uD83C\uDFFB",
            [":man_with_gua_pi_mao::skin-tone-2:"] = "\uD83D\uDC72\uD83C\uDFFC",
            [":man_with_gua_pi_mao::skin-tone-3:"] = "\uD83D\uDC72\uD83C\uDFFD",
            [":man_with_gua_pi_mao::skin-tone-4:"] = "\uD83D\uDC72\uD83C\uDFFE",
            [":man_with_gua_pi_mao::skin-tone-5:"] = "\uD83D\uDC72\uD83C\uDFFF",
            [":man_with_gua_pi_mao_tone1:"] = "\uD83D\uDC72\uD83C\uDFFB",
            [":man_with_gua_pi_mao_tone2:"] = "\uD83D\uDC72\uD83C\uDFFC",
            [":man_with_gua_pi_mao_tone3:"] = "\uD83D\uDC72\uD83C\uDFFD",
            [":man_with_gua_pi_mao_tone4:"] = "\uD83D\uDC72\uD83C\uDFFE",
            [":man_with_gua_pi_mao_tone5:"] = "\uD83D\uDC72\uD83C\uDFFF",
            [":man_with_probing_cane:"] = "\uD83D\uDC68\u200D\uD83E\uDDAF",
            [":man_with_probing_cane::skin-tone-1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDAF",
            [":man_with_probing_cane::skin-tone-2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDAF",
            [":man_with_probing_cane::skin-tone-3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDAF",
            [":man_with_probing_cane::skin-tone-4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDAF",
            [":man_with_probing_cane::skin-tone-5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_medium_dark_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_medium_light_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_medium_skin_tone:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_tone1:"] = "\uD83D\uDC68\uD83C\uDFFB\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_tone2:"] = "\uD83D\uDC68\uD83C\uDFFC\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_tone3:"] = "\uD83D\uDC68\uD83C\uDFFD\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_tone4:"] = "\uD83D\uDC68\uD83C\uDFFE\u200D\uD83E\uDDAF",
            [":man_with_probing_cane_tone5:"] = "\uD83D\uDC68\uD83C\uDFFF\u200D\uD83E\uDDAF",
            [":man_with_turban:"] = "\uD83D\uDC73",
            [":man_with_turban::skin-tone-1:"] = "\uD83D\uDC73\uD83C\uDFFB",
            [":man_with_turban::skin-tone-2:"] = "\uD83D\uDC73\uD83C\uDFFC",
            [":man_with_turban::skin-tone-3:"] = "\uD83D\uDC73\uD83C\uDFFD",
            [":man_with_turban::skin-tone-4:"] = "\uD83D\uDC73\uD83C\uDFFE",
            [":man_with_turban::skin-tone-5:"] = "\uD83D\uDC73\uD83C\uDFFF",
            [":man_with_turban_tone1:"] = "\uD83D\uDC73\uD83C\uDFFB",
            [":man_with_turban_tone2:"] = "\uD83D\uDC73\uD83C\uDFFC",
            [":man_with_turban_tone3:"] = "\uD83D\uDC73\uD83C\uDFFD",
            [":man_with_turban_tone4:"] = "\uD83D\uDC73\uD83C\uDFFE",
            [":man_with_turban_tone5:"] = "\uD83D\uDC73\uD83C\uDFFF",
            [":man_with_veil:"] = "\uD83D\uDC70\u200D\u2642\uFE0F",
            [":man_with_veil::skin-tone-1:"] = "\uD83D\uDC70\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_with_veil::skin-tone-2:"] = "\uD83D\uDC70\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_with_veil::skin-tone-3:"] = "\uD83D\uDC70\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_with_veil::skin-tone-4:"] = "\uD83D\uDC70\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_with_veil::skin-tone-5:"] = "\uD83D\uDC70\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_with_veil_dark_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_with_veil_light_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_with_veil_medium_dark_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_with_veil_medium_light_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_with_veil_medium_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_with_veil_tone1:"] = "\uD83D\uDC70\uD83C\uDFFB\u200D\u2642\uFE0F",
            [":man_with_veil_tone2:"] = "\uD83D\uDC70\uD83C\uDFFC\u200D\u2642\uFE0F",
            [":man_with_veil_tone3:"] = "\uD83D\uDC70\uD83C\uDFFD\u200D\u2642\uFE0F",
            [":man_with_veil_tone4:"] = "\uD83D\uDC70\uD83C\uDFFE\u200D\u2642\uFE0F",
            [":man_with_veil_tone5:"] = "\uD83D\uDC70\uD83C\uDFFF\u200D\u2642\uFE0F",
            [":man_zombie:"] = "\uD83E\uDDDF\u200D♂️",
            [":mango:"] = "\uD83E\uDD6D",
            [":mans_shoe:"] = "\uD83D\uDC5E",
            [":mantlepiece_clock:"] = "\uD83D\uDD70️",
            [":manual_wheelchair:"] = "\uD83E\uDDBD",
            [":map:"] = "\uD83D\uDDFA️",
            [":maple_leaf:"] = "\uD83C\uDF41",
            [":martial_arts_uniform:"] = "\uD83E\uDD4B",
            [":mask:"] = "\uD83D\uDE37",
            [":massage:"] = "\uD83D\uDC86",
            [":massage::skin-tone-1:"] = "\uD83D\uDC86\uD83C\uDFFB",
            [":massage::skin-tone-2:"] = "\uD83D\uDC86\uD83C\uDFFC",
            [":massage::skin-tone-3:"] = "\uD83D\uDC86\uD83C\uDFFD",
            [":massage::skin-tone-4:"] = "\uD83D\uDC86\uD83C\uDFFE",
            [":massage::skin-tone-5:"] = "\uD83D\uDC86\uD83C\uDFFF",
            [":massage_tone1:"] = "\uD83D\uDC86\uD83C\uDFFB",
            [":massage_tone2:"] = "\uD83D\uDC86\uD83C\uDFFC",
            [":massage_tone3:"] = "\uD83D\uDC86\uD83C\uDFFD",
            [":massage_tone4:"] = "\uD83D\uDC86\uD83C\uDFFE",
            [":massage_tone5:"] = "\uD83D\uDC86\uD83C\uDFFF",
            [":mate:"] = "\uD83E\uDDC9",
            [":meat_on_bone:"] = "\uD83C\uDF56",
            [":mechanical_arm:"] = "\uD83E\uDDBE",
            [":mechanical_leg:"] = "\uD83E\uDDBF",
            [":medal:"] = "\uD83C\uDFC5",
            [":medical_symbol:"] = "⚕️",
            [":mega:"] = "\uD83D\uDCE3",
            [":melon:"] = "\uD83C\uDF48",
            [":melting_face:"] = "\uD83E\uDEE0",
            [":memo:"] = "\uD83D\uDCDD",
            [":men_with_bunny_ears_partying:"] = "\uD83D\uDC6F\u200D♂️",
            [":men_wrestling:"] = "\uD83E\uDD3C\u200D♂️",
            [":mending_heart:"] = "\u2764\uFE0F\u200D\uD83E\uDE79",
            [":menorah:"] = "\uD83D\uDD4E",
            [":mens:"] = "\uD83D\uDEB9",
            [":mermaid:"] = "\uD83E\uDDDC\u200D♀️",
            [":mermaid::skin-tone-1:"] = "\uD83E\uDDDC\uD83C\uDFFB\u200D♀️",
            [":mermaid::skin-tone-2:"] = "\uD83E\uDDDC\uD83C\uDFFC\u200D♀️",
            [":mermaid::skin-tone-3:"] = "\uD83E\uDDDC\uD83C\uDFFD\u200D♀️",
            [":mermaid::skin-tone-4:"] = "\uD83E\uDDDC\uD83C\uDFFE\u200D♀️",
            [":mermaid::skin-tone-5:"] = "\uD83E\uDDDC\uD83C\uDFFF\u200D♀️",
            [":mermaid_dark_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFF\u200D♀️",
            [":mermaid_light_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFB\u200D♀️",
            [":mermaid_medium_dark_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFE\u200D♀️",
            [":mermaid_medium_light_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFC\u200D♀️",
            [":mermaid_medium_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFD\u200D♀️",
            [":mermaid_tone1:"] = "\uD83E\uDDDC\uD83C\uDFFB\u200D♀️",
            [":mermaid_tone2:"] = "\uD83E\uDDDC\uD83C\uDFFC\u200D♀️",
            [":mermaid_tone3:"] = "\uD83E\uDDDC\uD83C\uDFFD\u200D♀️",
            [":mermaid_tone4:"] = "\uD83E\uDDDC\uD83C\uDFFE\u200D♀️",
            [":mermaid_tone5:"] = "\uD83E\uDDDC\uD83C\uDFFF\u200D♀️",
            [":merman:"] = "\uD83E\uDDDC\u200D♂️",
            [":merman::skin-tone-1:"] = "\uD83E\uDDDC\uD83C\uDFFB\u200D♂️",
            [":merman::skin-tone-2:"] = "\uD83E\uDDDC\uD83C\uDFFC\u200D♂️",
            [":merman::skin-tone-3:"] = "\uD83E\uDDDC\uD83C\uDFFD\u200D♂️",
            [":merman::skin-tone-4:"] = "\uD83E\uDDDC\uD83C\uDFFE\u200D♂️",
            [":merman::skin-tone-5:"] = "\uD83E\uDDDC\uD83C\uDFFF\u200D♂️",
            [":merman_dark_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFF\u200D♂️",
            [":merman_light_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFB\u200D♂️",
            [":merman_medium_dark_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFE\u200D♂️",
            [":merman_medium_light_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFC\u200D♂️",
            [":merman_medium_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFD\u200D♂️",
            [":merman_tone1:"] = "\uD83E\uDDDC\uD83C\uDFFB\u200D♂️",
            [":merman_tone2:"] = "\uD83E\uDDDC\uD83C\uDFFC\u200D♂️",
            [":merman_tone3:"] = "\uD83E\uDDDC\uD83C\uDFFD\u200D♂️",
            [":merman_tone4:"] = "\uD83E\uDDDC\uD83C\uDFFE\u200D♂️",
            [":merman_tone5:"] = "\uD83E\uDDDC\uD83C\uDFFF\u200D♂️",
            [":merperson:"] = "\uD83E\uDDDC",
            [":merperson::skin-tone-1:"] = "\uD83E\uDDDC\uD83C\uDFFB",
            [":merperson::skin-tone-2:"] = "\uD83E\uDDDC\uD83C\uDFFC",
            [":merperson::skin-tone-3:"] = "\uD83E\uDDDC\uD83C\uDFFD",
            [":merperson::skin-tone-4:"] = "\uD83E\uDDDC\uD83C\uDFFE",
            [":merperson::skin-tone-5:"] = "\uD83E\uDDDC\uD83C\uDFFF",
            [":merperson_dark_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFF",
            [":merperson_light_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFB",
            [":merperson_medium_dark_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFE",
            [":merperson_medium_light_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFC",
            [":merperson_medium_skin_tone:"] = "\uD83E\uDDDC\uD83C\uDFFD",
            [":merperson_tone1:"] = "\uD83E\uDDDC\uD83C\uDFFB",
            [":merperson_tone2:"] = "\uD83E\uDDDC\uD83C\uDFFC",
            [":merperson_tone3:"] = "\uD83E\uDDDC\uD83C\uDFFD",
            [":merperson_tone4:"] = "\uD83E\uDDDC\uD83C\uDFFE",
            [":merperson_tone5:"] = "\uD83E\uDDDC\uD83C\uDFFF",
            [":metal:"] = "\uD83E\uDD18",
            [":metal::skin-tone-1:"] = "\uD83E\uDD18\uD83C\uDFFB",
            [":metal::skin-tone-2:"] = "\uD83E\uDD18\uD83C\uDFFC",
            [":metal::skin-tone-3:"] = "\uD83E\uDD18\uD83C\uDFFD",
            [":metal::skin-tone-4:"] = "\uD83E\uDD18\uD83C\uDFFE",
            [":metal::skin-tone-5:"] = "\uD83E\uDD18\uD83C\uDFFF",
            [":metal_tone1:"] = "\uD83E\uDD18\uD83C\uDFFB",
            [":metal_tone2:"] = "\uD83E\uDD18\uD83C\uDFFC",
            [":metal_tone3:"] = "\uD83E\uDD18\uD83C\uDFFD",
            [":metal_tone4:"] = "\uD83E\uDD18\uD83C\uDFFE",
            [":metal_tone5:"] = "\uD83E\uDD18\uD83C\uDFFF",
            [":metro:"] = "\uD83D\uDE87",
            [":microbe:"] = "\uD83E\uDDA0",
            [":microphone2:"] = "\uD83C\uDF99️",
            [":microphone:"] = "\uD83C\uDFA4",
            [":microscope:"] = "\uD83D\uDD2C",
            [":middle_finger:"] = "\uD83D\uDD95",
            [":middle_finger::skin-tone-1:"] = "\uD83D\uDD95\uD83C\uDFFB",
            [":middle_finger::skin-tone-2:"] = "\uD83D\uDD95\uD83C\uDFFC",
            [":middle_finger::skin-tone-3:"] = "\uD83D\uDD95\uD83C\uDFFD",
            [":middle_finger::skin-tone-4:"] = "\uD83D\uDD95\uD83C\uDFFE",
            [":middle_finger::skin-tone-5:"] = "\uD83D\uDD95\uD83C\uDFFF",
            [":middle_finger_tone1:"] = "\uD83D\uDD95\uD83C\uDFFB",
            [":middle_finger_tone2:"] = "\uD83D\uDD95\uD83C\uDFFC",
            [":middle_finger_tone3:"] = "\uD83D\uDD95\uD83C\uDFFD",
            [":middle_finger_tone4:"] = "\uD83D\uDD95\uD83C\uDFFE",
            [":middle_finger_tone5:"] = "\uD83D\uDD95\uD83C\uDFFF",
            [":military_helmet:"] = "\uD83E\uDE96",
            [":military_medal:"] = "\uD83C\uDF96️",
            [":milk:"] = "\uD83E\uDD5B",
            [":milky_way:"] = "\uD83C\uDF0C",
            [":minibus:"] = "\uD83D\uDE90",
            [":minidisc:"] = "\uD83D\uDCBD",
            [":mirror:"] = "\uD83E\uDE9E",
            [":mirror_ball:"] = "\uD83E\uDEA9",
            [":mobile_phone_off:"] = "\uD83D\uDCF4",
            [":money_mouth:"] = "\uD83E\uDD11",
            [":money_mouth_face:"] = "\uD83E\uDD11",
            [":money_with_wings:"] = "\uD83D\uDCB8",
            [":moneybag:"] = "\uD83D\uDCB0",
            [":monkey:"] = "\uD83D\uDC12",
            [":monkey_face:"] = "\uD83D\uDC35",
            [":monorail:"] = "\uD83D\uDE9D",
            [":moon_cake:"] = "\uD83E\uDD6E",
            [":mortar_board:"] = "\uD83C\uDF93",
            [":mosque:"] = "\uD83D\uDD4C",
            [":mosquito:"] = "\uD83E\uDD9F",
            [":mother_christmas:"] = "\uD83E\uDD36",
            [":mother_christmas::skin-tone-1:"] = "\uD83E\uDD36\uD83C\uDFFB",
            [":mother_christmas::skin-tone-2:"] = "\uD83E\uDD36\uD83C\uDFFC",
            [":mother_christmas::skin-tone-3:"] = "\uD83E\uDD36\uD83C\uDFFD",
            [":mother_christmas::skin-tone-4:"] = "\uD83E\uDD36\uD83C\uDFFE",
            [":mother_christmas::skin-tone-5:"] = "\uD83E\uDD36\uD83C\uDFFF",
            [":mother_christmas_tone1:"] = "\uD83E\uDD36\uD83C\uDFFB",
            [":mother_christmas_tone2:"] = "\uD83E\uDD36\uD83C\uDFFC",
            [":mother_christmas_tone3:"] = "\uD83E\uDD36\uD83C\uDFFD",
            [":mother_christmas_tone4:"] = "\uD83E\uDD36\uD83C\uDFFE",
            [":mother_christmas_tone5:"] = "\uD83E\uDD36\uD83C\uDFFF",
            [":motor_scooter:"] = "\uD83D\uDEF5",
            [":motorbike:"] = "\uD83D\uDEF5",
            [":motorboat:"] = "\uD83D\uDEE5️",
            [":motorcycle:"] = "\uD83C\uDFCD️",
            [":motorized_wheelchair:"] = "\uD83E\uDDBC",
            [":motorway:"] = "\uD83D\uDEE3️",
            [":mount_fuji:"] = "\uD83D\uDDFB",
            [":mountain:"] = "⛰️",
            [":mountain_bicyclist:"] = "\uD83D\uDEB5",
            [":mountain_bicyclist::skin-tone-1:"] = "\uD83D\uDEB5\uD83C\uDFFB",
            [":mountain_bicyclist::skin-tone-2:"] = "\uD83D\uDEB5\uD83C\uDFFC",
            [":mountain_bicyclist::skin-tone-3:"] = "\uD83D\uDEB5\uD83C\uDFFD",
            [":mountain_bicyclist::skin-tone-4:"] = "\uD83D\uDEB5\uD83C\uDFFE",
            [":mountain_bicyclist::skin-tone-5:"] = "\uD83D\uDEB5\uD83C\uDFFF",
            [":mountain_bicyclist_tone1:"] = "\uD83D\uDEB5\uD83C\uDFFB",
            [":mountain_bicyclist_tone2:"] = "\uD83D\uDEB5\uD83C\uDFFC",
            [":mountain_bicyclist_tone3:"] = "\uD83D\uDEB5\uD83C\uDFFD",
            [":mountain_bicyclist_tone4:"] = "\uD83D\uDEB5\uD83C\uDFFE",
            [":mountain_bicyclist_tone5:"] = "\uD83D\uDEB5\uD83C\uDFFF",
            [":mountain_cableway:"] = "\uD83D\uDEA0",
            [":mountain_railway:"] = "\uD83D\uDE9E",
            [":mountain_snow:"] = "\uD83C\uDFD4️",
            [":mouse2:"] = "\uD83D\uDC01",
            [":mouse:"] = "\uD83D\uDC2D",
            [":mouse_three_button:"] = "\uD83D\uDDB1️",
            [":mouse_trap:"] = "\uD83E\uDEA4",
            [":movie_camera:"] = "\uD83C\uDFA5",
            [":moyai:"] = "\uD83D\uDDFF",
            [":mrs_claus:"] = "\uD83E\uDD36",
            [":mrs_claus::skin-tone-1:"] = "\uD83E\uDD36\uD83C\uDFFB",
            [":mrs_claus::skin-tone-2:"] = "\uD83E\uDD36\uD83C\uDFFC",
            [":mrs_claus::skin-tone-3:"] = "\uD83E\uDD36\uD83C\uDFFD",
            [":mrs_claus::skin-tone-4:"] = "\uD83E\uDD36\uD83C\uDFFE",
            [":mrs_claus::skin-tone-5:"] = "\uD83E\uDD36\uD83C\uDFFF",
            [":mrs_claus_tone1:"] = "\uD83E\uDD36\uD83C\uDFFB",
            [":mrs_claus_tone2:"] = "\uD83E\uDD36\uD83C\uDFFC",
            [":mrs_claus_tone3:"] = "\uD83E\uDD36\uD83C\uDFFD",
            [":mrs_claus_tone4:"] = "\uD83E\uDD36\uD83C\uDFFE",
            [":mrs_claus_tone5:"] = "\uD83E\uDD36\uD83C\uDFFF",
            [":muscle:"] = "\uD83D\uDCAA",
            [":muscle::skin-tone-1:"] = "\uD83D\uDCAA\uD83C\uDFFB",
            [":muscle::skin-tone-2:"] = "\uD83D\uDCAA\uD83C\uDFFC",
            [":muscle::skin-tone-3:"] = "\uD83D\uDCAA\uD83C\uDFFD",
            [":muscle::skin-tone-4:"] = "\uD83D\uDCAA\uD83C\uDFFE",
            [":muscle::skin-tone-5:"] = "\uD83D\uDCAA\uD83C\uDFFF",
            [":muscle_tone1:"] = "\uD83D\uDCAA\uD83C\uDFFB",
            [":muscle_tone2:"] = "\uD83D\uDCAA\uD83C\uDFFC",
            [":muscle_tone3:"] = "\uD83D\uDCAA\uD83C\uDFFD",
            [":muscle_tone4:"] = "\uD83D\uDCAA\uD83C\uDFFE",
            [":muscle_tone5:"] = "\uD83D\uDCAA\uD83C\uDFFF",
            [":mushroom:"] = "\uD83C\uDF44",
            [":musical_keyboard:"] = "\uD83C\uDFB9",
            [":musical_note:"] = "\uD83C\uDFB5",
            [":musical_score:"] = "\uD83C\uDFBC",
            [":mute:"] = "\uD83D\uDD07",
            [":mx_claus:"] = "\uD83E\uDDD1\u200D\uD83C\uDF84",
            [":mx_claus::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\uD83C\uDF84",
            [":mx_claus::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\uD83C\uDF84",
            [":mx_claus::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\uD83C\uDF84",
            [":mx_claus::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\uD83C\uDF84",
            [":mx_claus::skin-tone-5:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\uD83C\uDF84",
            [":mx_claus_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\uD83C\uDF84",
            [":mx_claus_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\uD83C\uDF84",
            [":mx_claus_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\uD83C\uDF84",
            [":mx_claus_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\uD83C\uDF84",
            [":mx_claus_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\uD83C\uDF84",
            [":mx_claus_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\uD83C\uDF84",
            [":mx_claus_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\uD83C\uDF84",
            [":mx_claus_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\uD83C\uDF84",
            [":mx_claus_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\uD83C\uDF84",
            [":mx_claus_tone5:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\uD83C\uDF84",
            [":nail_care:"] = "\uD83D\uDC85",
            [":nail_care::skin-tone-1:"] = "\uD83D\uDC85\uD83C\uDFFB",
            [":nail_care::skin-tone-2:"] = "\uD83D\uDC85\uD83C\uDFFC",
            [":nail_care::skin-tone-3:"] = "\uD83D\uDC85\uD83C\uDFFD",
            [":nail_care::skin-tone-4:"] = "\uD83D\uDC85\uD83C\uDFFE",
            [":nail_care::skin-tone-5:"] = "\uD83D\uDC85\uD83C\uDFFF",
            [":nail_care_tone1:"] = "\uD83D\uDC85\uD83C\uDFFB",
            [":nail_care_tone2:"] = "\uD83D\uDC85\uD83C\uDFFC",
            [":nail_care_tone3:"] = "\uD83D\uDC85\uD83C\uDFFD",
            [":nail_care_tone4:"] = "\uD83D\uDC85\uD83C\uDFFE",
            [":nail_care_tone5:"] = "\uD83D\uDC85\uD83C\uDFFF",
            [":name_badge:"] = "\uD83D\uDCDB",
            [":national_park:"] = "\uD83C\uDFDE️",
            [":nauseated_face:"] = "\uD83E\uDD22",
            [":nazar_amulet:"] = "\uD83E\uDDFF",
            [":necktie:"] = "\uD83D\uDC54",
            [":negative_squared_cross_mark:"] = "❎",
            [":nerd:"] = "\uD83E\uDD13",
            [":nerd_face:"] = "\uD83E\uDD13",
            [":nest_with_eggs:"] = "\uD83E\uDEBA",
            [":nesting_dolls:"] = "\uD83E\uDE86",
            [":neutral_face:"] = "\uD83D\uDE10",
            [":new:"] = "\uD83C\uDD95",
            [":new_moon:"] = "\uD83C\uDF11",
            [":new_moon_with_face:"] = "\uD83C\uDF1A",
            [":newspaper2:"] = "\uD83D\uDDDE️",
            [":newspaper:"] = "\uD83D\uDCF0",
            [":next_track:"] = "⏭️",
            [":ng:"] = "\uD83C\uDD96",
            [":night_with_stars:"] = "\uD83C\uDF03",
            [":nine:"] = "9️⃣",
            [":ninja:"] = "\uD83E\uDD77",
            [":ninja::skin-tone-1:"] = "\uD83E\uDD77\uD83C\uDFFB",
            [":ninja::skin-tone-2:"] = "\uD83E\uDD77\uD83C\uDFFC",
            [":ninja::skin-tone-3:"] = "\uD83E\uDD77\uD83C\uDFFD",
            [":ninja::skin-tone-4:"] = "\uD83E\uDD77\uD83C\uDFFE",
            [":ninja::skin-tone-5:"] = "\uD83E\uDD77\uD83C\uDFFF",
            [":ninja_dark_skin_tone:"] = "\uD83E\uDD77\uD83C\uDFFF",
            [":ninja_light_skin_tone:"] = "\uD83E\uDD77\uD83C\uDFFB",
            [":ninja_medium_dark_skin_tone:"] = "\uD83E\uDD77\uD83C\uDFFE",
            [":ninja_medium_light_skin_tone:"] = "\uD83E\uDD77\uD83C\uDFFC",
            [":ninja_medium_skin_tone:"] = "\uD83E\uDD77\uD83C\uDFFD",
            [":ninja_tone1:"] = "\uD83E\uDD77\uD83C\uDFFB",
            [":ninja_tone2:"] = "\uD83E\uDD77\uD83C\uDFFC",
            [":ninja_tone3:"] = "\uD83E\uDD77\uD83C\uDFFD",
            [":ninja_tone4:"] = "\uD83E\uDD77\uD83C\uDFFE",
            [":ninja_tone5:"] = "\uD83E\uDD77\uD83C\uDFFF",
            [":no_bell:"] = "\uD83D\uDD15",
            [":no_bicycles:"] = "\uD83D\uDEB3",
            [":no_entry:"] = "⛔",
            [":no_entry_sign:"] = "\uD83D\uDEAB",
            [":no_good:"] = "\uD83D\uDE45",
            [":no_good::skin-tone-1:"] = "\uD83D\uDE45\uD83C\uDFFB",
            [":no_good::skin-tone-2:"] = "\uD83D\uDE45\uD83C\uDFFC",
            [":no_good::skin-tone-3:"] = "\uD83D\uDE45\uD83C\uDFFD",
            [":no_good::skin-tone-4:"] = "\uD83D\uDE45\uD83C\uDFFE",
            [":no_good::skin-tone-5:"] = "\uD83D\uDE45\uD83C\uDFFF",
            [":no_good_tone1:"] = "\uD83D\uDE45\uD83C\uDFFB",
            [":no_good_tone2:"] = "\uD83D\uDE45\uD83C\uDFFC",
            [":no_good_tone3:"] = "\uD83D\uDE45\uD83C\uDFFD",
            [":no_good_tone4:"] = "\uD83D\uDE45\uD83C\uDFFE",
            [":no_good_tone5:"] = "\uD83D\uDE45\uD83C\uDFFF",
            [":no_mobile_phones:"] = "\uD83D\uDCF5",
            [":no_mouth:"] = "\uD83D\uDE36",
            [":no_pedestrians:"] = "\uD83D\uDEB7",
            [":no_smoking:"] = "\uD83D\uDEAD",
            [":non_potable_water:"] = "\uD83D\uDEB1",
            [":nose:"] = "\uD83D\uDC43",
            [":nose::skin-tone-1:"] = "\uD83D\uDC43\uD83C\uDFFB",
            [":nose::skin-tone-2:"] = "\uD83D\uDC43\uD83C\uDFFC",
            [":nose::skin-tone-3:"] = "\uD83D\uDC43\uD83C\uDFFD",
            [":nose::skin-tone-4:"] = "\uD83D\uDC43\uD83C\uDFFE",
            [":nose::skin-tone-5:"] = "\uD83D\uDC43\uD83C\uDFFF",
            [":nose_tone1:"] = "\uD83D\uDC43\uD83C\uDFFB",
            [":nose_tone2:"] = "\uD83D\uDC43\uD83C\uDFFC",
            [":nose_tone3:"] = "\uD83D\uDC43\uD83C\uDFFD",
            [":nose_tone4:"] = "\uD83D\uDC43\uD83C\uDFFE",
            [":nose_tone5:"] = "\uD83D\uDC43\uD83C\uDFFF",
            [":notebook:"] = "\uD83D\uDCD3",
            [":notebook_with_decorative_cover:"] = "\uD83D\uDCD4",
            [":notepad_spiral:"] = "\uD83D\uDDD2️",
            [":notes:"] = "\uD83C\uDFB6",
            [":nut_and_bolt:"] = "\uD83D\uDD29",
            [":o"] = "\uD83D\uDE2E",
            [":o2:"] = "\uD83C\uDD7E️",
            [":o:"] = "⭕",
            [":ocean:"] = "\uD83C\uDF0A",
            [":octagonal_sign:"] = "\uD83D\uDED1",
            [":octopus:"] = "\uD83D\uDC19",
            [":oden:"] = "\uD83C\uDF62",
            [":office:"] = "\uD83C\uDFE2",
            [":oil:"] = "\uD83D\uDEE2️",
            [":oil_drum:"] = "\uD83D\uDEE2️",
            [":ok:"] = "\uD83C\uDD97",
            [":ok_hand:"] = "\uD83D\uDC4C",
            [":ok_hand::skin-tone-1:"] = "\uD83D\uDC4C\uD83C\uDFFB",
            [":ok_hand::skin-tone-2:"] = "\uD83D\uDC4C\uD83C\uDFFC",
            [":ok_hand::skin-tone-3:"] = "\uD83D\uDC4C\uD83C\uDFFD",
            [":ok_hand::skin-tone-4:"] = "\uD83D\uDC4C\uD83C\uDFFE",
            [":ok_hand::skin-tone-5:"] = "\uD83D\uDC4C\uD83C\uDFFF",
            [":ok_hand_tone1:"] = "\uD83D\uDC4C\uD83C\uDFFB",
            [":ok_hand_tone2:"] = "\uD83D\uDC4C\uD83C\uDFFC",
            [":ok_hand_tone3:"] = "\uD83D\uDC4C\uD83C\uDFFD",
            [":ok_hand_tone4:"] = "\uD83D\uDC4C\uD83C\uDFFE",
            [":ok_hand_tone5:"] = "\uD83D\uDC4C\uD83C\uDFFF",
            [":ok_woman:"] = "\uD83D\uDE46",
            [":ok_woman::skin-tone-1:"] = "\uD83D\uDE46\uD83C\uDFFB",
            [":ok_woman::skin-tone-2:"] = "\uD83D\uDE46\uD83C\uDFFC",
            [":ok_woman::skin-tone-3:"] = "\uD83D\uDE46\uD83C\uDFFD",
            [":ok_woman::skin-tone-4:"] = "\uD83D\uDE46\uD83C\uDFFE",
            [":ok_woman::skin-tone-5:"] = "\uD83D\uDE46\uD83C\uDFFF",
            [":ok_woman_tone1:"] = "\uD83D\uDE46\uD83C\uDFFB",
            [":ok_woman_tone2:"] = "\uD83D\uDE46\uD83C\uDFFC",
            [":ok_woman_tone3:"] = "\uD83D\uDE46\uD83C\uDFFD",
            [":ok_woman_tone4:"] = "\uD83D\uDE46\uD83C\uDFFE",
            [":ok_woman_tone5:"] = "\uD83D\uDE46\uD83C\uDFFF",
            [":old_key:"] = "\uD83D\uDDDD️",
            [":older_adult:"] = "\uD83E\uDDD3",
            [":older_adult::skin-tone-1:"] = "\uD83E\uDDD3\uD83C\uDFFB",
            [":older_adult::skin-tone-2:"] = "\uD83E\uDDD3\uD83C\uDFFC",
            [":older_adult::skin-tone-3:"] = "\uD83E\uDDD3\uD83C\uDFFD",
            [":older_adult::skin-tone-4:"] = "\uD83E\uDDD3\uD83C\uDFFE",
            [":older_adult::skin-tone-5:"] = "\uD83E\uDDD3\uD83C\uDFFF",
            [":older_adult_dark_skin_tone:"] = "\uD83E\uDDD3\uD83C\uDFFF",
            [":older_adult_light_skin_tone:"] = "\uD83E\uDDD3\uD83C\uDFFB",
            [":older_adult_medium_dark_skin_tone:"] = "\uD83E\uDDD3\uD83C\uDFFE",
            [":older_adult_medium_light_skin_tone:"] = "\uD83E\uDDD3\uD83C\uDFFC",
            [":older_adult_medium_skin_tone:"] = "\uD83E\uDDD3\uD83C\uDFFD",
            [":older_adult_tone1:"] = "\uD83E\uDDD3\uD83C\uDFFB",
            [":older_adult_tone2:"] = "\uD83E\uDDD3\uD83C\uDFFC",
            [":older_adult_tone3:"] = "\uD83E\uDDD3\uD83C\uDFFD",
            [":older_adult_tone4:"] = "\uD83E\uDDD3\uD83C\uDFFE",
            [":older_adult_tone5:"] = "\uD83E\uDDD3\uD83C\uDFFF",
            [":older_man:"] = "\uD83D\uDC74",
            [":older_man::skin-tone-1:"] = "\uD83D\uDC74\uD83C\uDFFB",
            [":older_man::skin-tone-2:"] = "\uD83D\uDC74\uD83C\uDFFC",
            [":older_man::skin-tone-3:"] = "\uD83D\uDC74\uD83C\uDFFD",
            [":older_man::skin-tone-4:"] = "\uD83D\uDC74\uD83C\uDFFE",
            [":older_man::skin-tone-5:"] = "\uD83D\uDC74\uD83C\uDFFF",
            [":older_man_tone1:"] = "\uD83D\uDC74\uD83C\uDFFB",
            [":older_man_tone2:"] = "\uD83D\uDC74\uD83C\uDFFC",
            [":older_man_tone3:"] = "\uD83D\uDC74\uD83C\uDFFD",
            [":older_man_tone4:"] = "\uD83D\uDC74\uD83C\uDFFE",
            [":older_man_tone5:"] = "\uD83D\uDC74\uD83C\uDFFF",
            [":older_woman:"] = "\uD83D\uDC75",
            [":older_woman::skin-tone-1:"] = "\uD83D\uDC75\uD83C\uDFFB",
            [":older_woman::skin-tone-2:"] = "\uD83D\uDC75\uD83C\uDFFC",
            [":older_woman::skin-tone-3:"] = "\uD83D\uDC75\uD83C\uDFFD",
            [":older_woman::skin-tone-4:"] = "\uD83D\uDC75\uD83C\uDFFE",
            [":older_woman::skin-tone-5:"] = "\uD83D\uDC75\uD83C\uDFFF",
            [":older_woman_tone1:"] = "\uD83D\uDC75\uD83C\uDFFB",
            [":older_woman_tone2:"] = "\uD83D\uDC75\uD83C\uDFFC",
            [":older_woman_tone3:"] = "\uD83D\uDC75\uD83C\uDFFD",
            [":older_woman_tone4:"] = "\uD83D\uDC75\uD83C\uDFFE",
            [":older_woman_tone5:"] = "\uD83D\uDC75\uD83C\uDFFF",
            [":olive:"] = "\uD83E\uDED2",
            [":om_symbol:"] = "\uD83D\uDD49️",
            [":on:"] = "\uD83D\uDD1B",
            [":oncoming_automobile:"] = "\uD83D\uDE98",
            [":oncoming_bus:"] = "\uD83D\uDE8D",
            [":oncoming_police_car:"] = "\uD83D\uDE94",
            [":oncoming_taxi:"] = "\uD83D\uDE96",
            [":one:"] = "1️⃣",
            [":one_piece_swimsuit:"] = "\uD83E\uDE71",
            [":onion:"] = "\uD83E\uDDC5",
            [":open_file_folder:"] = "\uD83D\uDCC2",
            [":open_hands:"] = "\uD83D\uDC50",
            [":open_hands::skin-tone-1:"] = "\uD83D\uDC50\uD83C\uDFFB",
            [":open_hands::skin-tone-2:"] = "\uD83D\uDC50\uD83C\uDFFC",
            [":open_hands::skin-tone-3:"] = "\uD83D\uDC50\uD83C\uDFFD",
            [":open_hands::skin-tone-4:"] = "\uD83D\uDC50\uD83C\uDFFE",
            [":open_hands::skin-tone-5:"] = "\uD83D\uDC50\uD83C\uDFFF",
            [":open_hands_tone1:"] = "\uD83D\uDC50\uD83C\uDFFB",
            [":open_hands_tone2:"] = "\uD83D\uDC50\uD83C\uDFFC",
            [":open_hands_tone3:"] = "\uD83D\uDC50\uD83C\uDFFD",
            [":open_hands_tone4:"] = "\uD83D\uDC50\uD83C\uDFFE",
            [":open_hands_tone5:"] = "\uD83D\uDC50\uD83C\uDFFF",
            [":open_mouth:"] = "\uD83D\uDE2E",
            [":ophiuchus:"] = "⛎",
            [":orange_book:"] = "\uD83D\uDCD9",
            [":orange_circle:"] = "\uD83D\uDFE0",
            [":orange_heart:"] = "\uD83E\uDDE1",
            [":orange_square:"] = "\uD83D\uDFE7",
            [":orangutan:"] = "\uD83E\uDDA7",
            [":orthodox_cross:"] = "☦️",
            [":otter:"] = "\uD83E\uDDA6",
            [":outbox_tray:"] = "\uD83D\uDCE4",
            [":owl:"] = "\uD83E\uDD89",
            [":ox:"] = "\uD83D\uDC02",
            [":oyster:"] = "\uD83E\uDDAA",
            [":package:"] = "\uD83D\uDCE6",
            [":paella:"] = "\uD83E\uDD58",
            [":page_facing_up:"] = "\uD83D\uDCC4",
            [":page_with_curl:"] = "\uD83D\uDCC3",
            [":pager:"] = "\uD83D\uDCDF",
            [":paintbrush:"] = "\uD83D\uDD8C️",
            [":palm_down_hand:"] = "\uD83E\uDEF3",
            [":palm_down_hand::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_down_hand_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_tree:"] = "\uD83C\uDF34",
            [":palm_up_hand:"] = "\uD83E\uDEF4",
            [":palm_up_hand::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palm_up_hand_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":palms_up_together:"] = "\uD83E\uDD32",
            [":palms_up_together::skin-tone-1:"] = "\uD83E\uDD32\uD83C\uDFFB",
            [":palms_up_together::skin-tone-2:"] = "\uD83E\uDD32\uD83C\uDFFC",
            [":palms_up_together::skin-tone-3:"] = "\uD83E\uDD32\uD83C\uDFFD",
            [":palms_up_together::skin-tone-4:"] = "\uD83E\uDD32\uD83C\uDFFE",
            [":palms_up_together::skin-tone-5:"] = "\uD83E\uDD32\uD83C\uDFFF",
            [":palms_up_together_dark_skin_tone:"] = "\uD83E\uDD32\uD83C\uDFFF",
            [":palms_up_together_light_skin_tone:"] = "\uD83E\uDD32\uD83C\uDFFB",
            [":palms_up_together_medium_dark_skin_tone:"] = "\uD83E\uDD32\uD83C\uDFFE",
            [":palms_up_together_medium_light_skin_tone:"] = "\uD83E\uDD32\uD83C\uDFFC",
            [":palms_up_together_medium_skin_tone:"] = "\uD83E\uDD32\uD83C\uDFFD",
            [":palms_up_together_tone1:"] = "\uD83E\uDD32\uD83C\uDFFB",
            [":palms_up_together_tone2:"] = "\uD83E\uDD32\uD83C\uDFFC",
            [":palms_up_together_tone3:"] = "\uD83E\uDD32\uD83C\uDFFD",
            [":palms_up_together_tone4:"] = "\uD83E\uDD32\uD83C\uDFFE",
            [":palms_up_together_tone5:"] = "\uD83E\uDD32\uD83C\uDFFF",
            [":pancakes:"] = "\uD83E\uDD5E",
            [":panda_face:"] = "\uD83D\uDC3C",
            [":paperclip:"] = "\uD83D\uDCCE",
            [":paperclips:"] = "\uD83D\uDD87️",
            [":parachute:"] = "\uD83E\uDE82",
            [":park:"] = "\uD83C\uDFDE️",
            [":parking:"] = "\uD83C\uDD7F️",
            [":parrot:"] = "\uD83E\uDD9C",
            [":part_alternation_mark:"] = "〽️",
            [":partly_sunny:"] = "⛅",
            [":partying_face:"] = "\uD83E\uDD73",
            [":passenger_ship:"] = "\uD83D\uDEF3️",
            [":passport_control:"] = "\uD83D\uDEC2",
            [":pause_button:"] = "⏸️",
            [":paw_prints:"] = "\uD83D\uDC3E",
            [":peace:"] = "☮️",
            [":peace_symbol:"] = "☮️",
            [":peach:"] = "\uD83C\uDF51",
            [":peacock:"] = "\uD83E\uDD9A",
            [":peanuts:"] = "\uD83E\uDD5C",
            [":pear:"] = "\uD83C\uDF50",
            [":pen_ballpoint:"] = "\uD83D\uDD8A️",
            [":pen_fountain:"] = "\uD83D\uDD8B️",
            [":pencil2:"] = "✏️",
            [":pencil:"] = "\uD83D\uDCDD",
            [":penguin:"] = "\uD83D\uDC27",
            [":pensive:"] = "\uD83D\uDE14",
            [":people_holding_hands:"] = "\uD83E\uDDD1\u200D\uD83E\uDD1D\u200D\uD83E\uDDD1",
            [":people_hugging:"] = "\uD83E\uDEC2",
            [":people_with_bunny_ears_partying:"] = "\uD83D\uDC6F",
            [":people_wrestling:"] = "\uD83E\uDD3C",
            [":performing_arts:"] = "\uD83C\uDFAD",
            [":persevere:"] = "\uD83D\uDE23",
            [":person_biking:"] = "\uD83D\uDEB4",
            [":person_biking::skin-tone-1:"] = "\uD83D\uDEB4\uD83C\uDFFB",
            [":person_biking::skin-tone-2:"] = "\uD83D\uDEB4\uD83C\uDFFC",
            [":person_biking::skin-tone-3:"] = "\uD83D\uDEB4\uD83C\uDFFD",
            [":person_biking::skin-tone-4:"] = "\uD83D\uDEB4\uD83C\uDFFE",
            [":person_biking::skin-tone-5:"] = "\uD83D\uDEB4\uD83C\uDFFF",
            [":person_biking_tone1:"] = "\uD83D\uDEB4\uD83C\uDFFB",
            [":person_biking_tone2:"] = "\uD83D\uDEB4\uD83C\uDFFC",
            [":person_biking_tone3:"] = "\uD83D\uDEB4\uD83C\uDFFD",
            [":person_biking_tone4:"] = "\uD83D\uDEB4\uD83C\uDFFE",
            [":person_biking_tone5:"] = "\uD83D\uDEB4\uD83C\uDFFF",
            [":person_bouncing_ball:"] = "⛹️",
            [":person_bouncing_ball::skin-tone-1:"] = "⛹\uD83C\uDFFB",
            [":person_bouncing_ball::skin-tone-2:"] = "⛹\uD83C\uDFFC",
            [":person_bouncing_ball::skin-tone-3:"] = "⛹\uD83C\uDFFD",
            [":person_bouncing_ball::skin-tone-4:"] = "⛹\uD83C\uDFFE",
            [":person_bouncing_ball::skin-tone-5:"] = "⛹\uD83C\uDFFF",
            [":person_bouncing_ball_tone1:"] = "⛹\uD83C\uDFFB",
            [":person_bouncing_ball_tone2:"] = "⛹\uD83C\uDFFC",
            [":person_bouncing_ball_tone3:"] = "⛹\uD83C\uDFFD",
            [":person_bouncing_ball_tone4:"] = "⛹\uD83C\uDFFE",
            [":person_bouncing_ball_tone5:"] = "⛹\uD83C\uDFFF",
            [":person_bowing:"] = "\uD83D\uDE47",
            [":person_bowing::skin-tone-1:"] = "\uD83D\uDE47\uD83C\uDFFB",
            [":person_bowing::skin-tone-2:"] = "\uD83D\uDE47\uD83C\uDFFC",
            [":person_bowing::skin-tone-3:"] = "\uD83D\uDE47\uD83C\uDFFD",
            [":person_bowing::skin-tone-4:"] = "\uD83D\uDE47\uD83C\uDFFE",
            [":person_bowing::skin-tone-5:"] = "\uD83D\uDE47\uD83C\uDFFF",
            [":person_bowing_tone1:"] = "\uD83D\uDE47\uD83C\uDFFB",
            [":person_bowing_tone2:"] = "\uD83D\uDE47\uD83C\uDFFC",
            [":person_bowing_tone3:"] = "\uD83D\uDE47\uD83C\uDFFD",
            [":person_bowing_tone4:"] = "\uD83D\uDE47\uD83C\uDFFE",
            [":person_bowing_tone5:"] = "\uD83D\uDE47\uD83C\uDFFF",
            [":person_climbing:"] = "\uD83E\uDDD7",
            [":person_climbing::skin-tone-1:"] = "\uD83E\uDDD7\uD83C\uDFFB",
            [":person_climbing::skin-tone-2:"] = "\uD83E\uDDD7\uD83C\uDFFC",
            [":person_climbing::skin-tone-3:"] = "\uD83E\uDDD7\uD83C\uDFFD",
            [":person_climbing::skin-tone-4:"] = "\uD83E\uDDD7\uD83C\uDFFE",
            [":person_climbing::skin-tone-5:"] = "\uD83E\uDDD7\uD83C\uDFFF",
            [":person_climbing_dark_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFF",
            [":person_climbing_light_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFB",
            [":person_climbing_medium_dark_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFE",
            [":person_climbing_medium_light_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFC",
            [":person_climbing_medium_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFD",
            [":person_climbing_tone1:"] = "\uD83E\uDDD7\uD83C\uDFFB",
            [":person_climbing_tone2:"] = "\uD83E\uDDD7\uD83C\uDFFC",
            [":person_climbing_tone3:"] = "\uD83E\uDDD7\uD83C\uDFFD",
            [":person_climbing_tone4:"] = "\uD83E\uDDD7\uD83C\uDFFE",
            [":person_climbing_tone5:"] = "\uD83E\uDDD7\uD83C\uDFFF",
            [":person_doing_cartwheel:"] = "\uD83E\uDD38",
            [":person_doing_cartwheel::skin-tone-1:"] = "\uD83E\uDD38\uD83C\uDFFB",
            [":person_doing_cartwheel::skin-tone-2:"] = "\uD83E\uDD38\uD83C\uDFFC",
            [":person_doing_cartwheel::skin-tone-3:"] = "\uD83E\uDD38\uD83C\uDFFD",
            [":person_doing_cartwheel::skin-tone-4:"] = "\uD83E\uDD38\uD83C\uDFFE",
            [":person_doing_cartwheel::skin-tone-5:"] = "\uD83E\uDD38\uD83C\uDFFF",
            [":person_doing_cartwheel_tone1:"] = "\uD83E\uDD38\uD83C\uDFFB",
            [":person_doing_cartwheel_tone2:"] = "\uD83E\uDD38\uD83C\uDFFC",
            [":person_doing_cartwheel_tone3:"] = "\uD83E\uDD38\uD83C\uDFFD",
            [":person_doing_cartwheel_tone4:"] = "\uD83E\uDD38\uD83C\uDFFE",
            [":person_doing_cartwheel_tone5:"] = "\uD83E\uDD38\uD83C\uDFFF",
            [":person_facepalming:"] = "\uD83E\uDD26",
            [":person_facepalming::skin-tone-1:"] = "\uD83E\uDD26\uD83C\uDFFB",
            [":person_facepalming::skin-tone-2:"] = "\uD83E\uDD26\uD83C\uDFFC",
            [":person_facepalming::skin-tone-3:"] = "\uD83E\uDD26\uD83C\uDFFD",
            [":person_facepalming::skin-tone-4:"] = "\uD83E\uDD26\uD83C\uDFFE",
            [":person_facepalming::skin-tone-5:"] = "\uD83E\uDD26\uD83C\uDFFF",
            [":person_facepalming_tone1:"] = "\uD83E\uDD26\uD83C\uDFFB",
            [":person_facepalming_tone2:"] = "\uD83E\uDD26\uD83C\uDFFC",
            [":person_facepalming_tone3:"] = "\uD83E\uDD26\uD83C\uDFFD",
            [":person_facepalming_tone4:"] = "\uD83E\uDD26\uD83C\uDFFE",
            [":person_facepalming_tone5:"] = "\uD83E\uDD26\uD83C\uDFFF",
            [":person_feeding_baby:"] = "\uD83E\uDDD1\u200D\uD83C\uDF7C",
            [":person_feeding_baby::skin-tone-1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":person_feeding_baby::skin-tone-2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":person_feeding_baby::skin-tone-3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":person_feeding_baby::skin-tone-4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":person_feeding_baby::skin-tone-5:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":person_feeding_baby_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":person_feeding_baby_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":person_feeding_baby_medium_dark_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":person_feeding_baby_medium_light_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":person_feeding_baby_medium_skin_tone:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":person_feeding_baby_tone1:"] = "\uD83E\uDDD1\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":person_feeding_baby_tone2:"] = "\uD83E\uDDD1\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":person_feeding_baby_tone3:"] = "\uD83E\uDDD1\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":person_feeding_baby_tone4:"] = "\uD83E\uDDD1\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":person_feeding_baby_tone5:"] = "\uD83E\uDDD1\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":person_fencing:"] = "\uD83E\uDD3A",
            [":person_frowning:"] = "\uD83D\uDE4D",
            [":person_frowning::skin-tone-1:"] = "\uD83D\uDE4D\uD83C\uDFFB",
            [":person_frowning::skin-tone-2:"] = "\uD83D\uDE4D\uD83C\uDFFC",
            [":person_frowning::skin-tone-3:"] = "\uD83D\uDE4D\uD83C\uDFFD",
            [":person_frowning::skin-tone-4:"] = "\uD83D\uDE4D\uD83C\uDFFE",
            [":person_frowning::skin-tone-5:"] = "\uD83D\uDE4D\uD83C\uDFFF",
            [":person_frowning_tone1:"] = "\uD83D\uDE4D\uD83C\uDFFB",
            [":person_frowning_tone2:"] = "\uD83D\uDE4D\uD83C\uDFFC",
            [":person_frowning_tone3:"] = "\uD83D\uDE4D\uD83C\uDFFD",
            [":person_frowning_tone4:"] = "\uD83D\uDE4D\uD83C\uDFFE",
            [":person_frowning_tone5:"] = "\uD83D\uDE4D\uD83C\uDFFF",
            [":person_gesturing_no:"] = "\uD83D\uDE45",
            [":person_gesturing_no::skin-tone-1:"] = "\uD83D\uDE45\uD83C\uDFFB",
            [":person_gesturing_no::skin-tone-2:"] = "\uD83D\uDE45\uD83C\uDFFC",
            [":person_gesturing_no::skin-tone-3:"] = "\uD83D\uDE45\uD83C\uDFFD",
            [":person_gesturing_no::skin-tone-4:"] = "\uD83D\uDE45\uD83C\uDFFE",
            [":person_gesturing_no::skin-tone-5:"] = "\uD83D\uDE45\uD83C\uDFFF",
            [":person_gesturing_no_tone1:"] = "\uD83D\uDE45\uD83C\uDFFB",
            [":person_gesturing_no_tone2:"] = "\uD83D\uDE45\uD83C\uDFFC",
            [":person_gesturing_no_tone3:"] = "\uD83D\uDE45\uD83C\uDFFD",
            [":person_gesturing_no_tone4:"] = "\uD83D\uDE45\uD83C\uDFFE",
            [":person_gesturing_no_tone5:"] = "\uD83D\uDE45\uD83C\uDFFF",
            [":person_gesturing_ok:"] = "\uD83D\uDE46",
            [":person_gesturing_ok::skin-tone-1:"] = "\uD83D\uDE46\uD83C\uDFFB",
            [":person_gesturing_ok::skin-tone-2:"] = "\uD83D\uDE46\uD83C\uDFFC",
            [":person_gesturing_ok::skin-tone-3:"] = "\uD83D\uDE46\uD83C\uDFFD",
            [":person_gesturing_ok::skin-tone-4:"] = "\uD83D\uDE46\uD83C\uDFFE",
            [":person_gesturing_ok::skin-tone-5:"] = "\uD83D\uDE46\uD83C\uDFFF",
            [":person_gesturing_ok_tone1:"] = "\uD83D\uDE46\uD83C\uDFFB",
            [":person_gesturing_ok_tone2:"] = "\uD83D\uDE46\uD83C\uDFFC",
            [":person_gesturing_ok_tone3:"] = "\uD83D\uDE46\uD83C\uDFFD",
            [":person_gesturing_ok_tone4:"] = "\uD83D\uDE46\uD83C\uDFFE",
            [":person_gesturing_ok_tone5:"] = "\uD83D\uDE46\uD83C\uDFFF",
            [":person_getting_haircut:"] = "\uD83D\uDC87",
            [":person_getting_haircut::skin-tone-1:"] = "\uD83D\uDC87\uD83C\uDFFB",
            [":person_getting_haircut::skin-tone-2:"] = "\uD83D\uDC87\uD83C\uDFFC",
            [":person_getting_haircut::skin-tone-3:"] = "\uD83D\uDC87\uD83C\uDFFD",
            [":person_getting_haircut::skin-tone-4:"] = "\uD83D\uDC87\uD83C\uDFFE",
            [":person_getting_haircut::skin-tone-5:"] = "\uD83D\uDC87\uD83C\uDFFF",
            [":person_getting_haircut_tone1:"] = "\uD83D\uDC87\uD83C\uDFFB",
            [":person_getting_haircut_tone2:"] = "\uD83D\uDC87\uD83C\uDFFC",
            [":person_getting_haircut_tone3:"] = "\uD83D\uDC87\uD83C\uDFFD",
            [":person_getting_haircut_tone4:"] = "\uD83D\uDC87\uD83C\uDFFE",
            [":person_getting_haircut_tone5:"] = "\uD83D\uDC87\uD83C\uDFFF",
            [":person_getting_massage:"] = "\uD83D\uDC86",
            [":person_getting_massage::skin-tone-1:"] = "\uD83D\uDC86\uD83C\uDFFB",
            [":person_getting_massage::skin-tone-2:"] = "\uD83D\uDC86\uD83C\uDFFC",
            [":person_getting_massage::skin-tone-3:"] = "\uD83D\uDC86\uD83C\uDFFD",
            [":person_getting_massage::skin-tone-4:"] = "\uD83D\uDC86\uD83C\uDFFE",
            [":person_getting_massage::skin-tone-5:"] = "\uD83D\uDC86\uD83C\uDFFF",
            [":person_getting_massage_tone1:"] = "\uD83D\uDC86\uD83C\uDFFB",
            [":person_getting_massage_tone2:"] = "\uD83D\uDC86\uD83C\uDFFC",
            [":person_getting_massage_tone3:"] = "\uD83D\uDC86\uD83C\uDFFD",
            [":person_getting_massage_tone4:"] = "\uD83D\uDC86\uD83C\uDFFE",
            [":person_getting_massage_tone5:"] = "\uD83D\uDC86\uD83C\uDFFF",
            [":person_golfing:"] = "\uD83C\uDFCC️",
            [":person_golfing::skin-tone-1:"] = "\uD83C\uDFCC\uD83C\uDFFB",
            [":person_golfing::skin-tone-2:"] = "\uD83C\uDFCC\uD83C\uDFFC",
            [":person_golfing::skin-tone-3:"] = "\uD83C\uDFCC\uD83C\uDFFD",
            [":person_golfing::skin-tone-4:"] = "\uD83C\uDFCC\uD83C\uDFFE",
            [":person_golfing::skin-tone-5:"] = "\uD83C\uDFCC\uD83C\uDFFF",
            [":person_golfing_dark_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFF",
            [":person_golfing_light_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFB",
            [":person_golfing_medium_dark_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFE",
            [":person_golfing_medium_light_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFC",
            [":person_golfing_medium_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFD",
            [":person_golfing_tone1:"] = "\uD83C\uDFCC\uD83C\uDFFB",
            [":person_golfing_tone2:"] = "\uD83C\uDFCC\uD83C\uDFFC",
            [":person_golfing_tone3:"] = "\uD83C\uDFCC\uD83C\uDFFD",
            [":person_golfing_tone4:"] = "\uD83C\uDFCC\uD83C\uDFFE",
            [":person_golfing_tone5:"] = "\uD83C\uDFCC\uD83C\uDFFF",
            [":person_in_bed_dark_skin_tone:"] = "\uD83D\uDECC\uD83C\uDFFF",
            [":person_in_bed_light_skin_tone:"] = "\uD83D\uDECC\uD83C\uDFFB",
            [":person_in_bed_medium_dark_skin_tone:"] = "\uD83D\uDECC\uD83C\uDFFE",
            [":person_in_bed_medium_light_skin_tone:"] = "\uD83D\uDECC\uD83C\uDFFC",
            [":person_in_bed_medium_skin_tone:"] = "\uD83D\uDECC\uD83C\uDFFD",
            [":person_in_bed_tone1:"] = "\uD83D\uDECC\uD83C\uDFFB",
            [":person_in_bed_tone2:"] = "\uD83D\uDECC\uD83C\uDFFC",
            [":person_in_bed_tone3:"] = "\uD83D\uDECC\uD83C\uDFFD",
            [":person_in_bed_tone4:"] = "\uD83D\uDECC\uD83C\uDFFE",
            [":person_in_bed_tone5:"] = "\uD83D\uDECC\uD83C\uDFFF",
            [":person_in_lotus_position:"] = "\uD83E\uDDD8",
            [":person_in_lotus_position::skin-tone-1:"] = "\uD83E\uDDD8\uD83C\uDFFB",
            [":person_in_lotus_position::skin-tone-2:"] = "\uD83E\uDDD8\uD83C\uDFFC",
            [":person_in_lotus_position::skin-tone-3:"] = "\uD83E\uDDD8\uD83C\uDFFD",
            [":person_in_lotus_position::skin-tone-4:"] = "\uD83E\uDDD8\uD83C\uDFFE",
            [":person_in_lotus_position::skin-tone-5:"] = "\uD83E\uDDD8\uD83C\uDFFF",
            [":person_in_lotus_position_dark_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFF",
            [":person_in_lotus_position_light_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFB",
            [":person_in_lotus_position_medium_dark_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFE",
            [":person_in_lotus_position_medium_light_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFC",
            [":person_in_lotus_position_medium_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFD",
            [":person_in_lotus_position_tone1:"] = "\uD83E\uDDD8\uD83C\uDFFB",
            [":person_in_lotus_position_tone2:"] = "\uD83E\uDDD8\uD83C\uDFFC",
            [":person_in_lotus_position_tone3:"] = "\uD83E\uDDD8\uD83C\uDFFD",
            [":person_in_lotus_position_tone4:"] = "\uD83E\uDDD8\uD83C\uDFFE",
            [":person_in_lotus_position_tone5:"] = "\uD83E\uDDD8\uD83C\uDFFF",
            [":person_in_steamy_room:"] = "\uD83E\uDDD6",
            [":person_in_steamy_room::skin-tone-1:"] = "\uD83E\uDDD6\uD83C\uDFFB",
            [":person_in_steamy_room::skin-tone-2:"] = "\uD83E\uDDD6\uD83C\uDFFC",
            [":person_in_steamy_room::skin-tone-3:"] = "\uD83E\uDDD6\uD83C\uDFFD",
            [":person_in_steamy_room::skin-tone-4:"] = "\uD83E\uDDD6\uD83C\uDFFE",
            [":person_in_steamy_room::skin-tone-5:"] = "\uD83E\uDDD6\uD83C\uDFFF",
            [":person_in_steamy_room_dark_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFF",
            [":person_in_steamy_room_light_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFB",
            [":person_in_steamy_room_medium_dark_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFE",
            [":person_in_steamy_room_medium_light_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFC",
            [":person_in_steamy_room_medium_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFD",
            [":person_in_steamy_room_tone1:"] = "\uD83E\uDDD6\uD83C\uDFFB",
            [":person_in_steamy_room_tone2:"] = "\uD83E\uDDD6\uD83C\uDFFC",
            [":person_in_steamy_room_tone3:"] = "\uD83E\uDDD6\uD83C\uDFFD",
            [":person_in_steamy_room_tone4:"] = "\uD83E\uDDD6\uD83C\uDFFE",
            [":person_in_steamy_room_tone5:"] = "\uD83E\uDDD6\uD83C\uDFFF",
            [":person_juggling:"] = "\uD83E\uDD39",
            [":person_juggling::skin-tone-1:"] = "\uD83E\uDD39\uD83C\uDFFB",
            [":person_juggling::skin-tone-2:"] = "\uD83E\uDD39\uD83C\uDFFC",
            [":person_juggling::skin-tone-3:"] = "\uD83E\uDD39\uD83C\uDFFD",
            [":person_juggling::skin-tone-4:"] = "\uD83E\uDD39\uD83C\uDFFE",
            [":person_juggling::skin-tone-5:"] = "\uD83E\uDD39\uD83C\uDFFF",
            [":person_juggling_tone1:"] = "\uD83E\uDD39\uD83C\uDFFB",
            [":person_juggling_tone2:"] = "\uD83E\uDD39\uD83C\uDFFC",
            [":person_juggling_tone3:"] = "\uD83E\uDD39\uD83C\uDFFD",
            [":person_juggling_tone4:"] = "\uD83E\uDD39\uD83C\uDFFE",
            [":person_juggling_tone5:"] = "\uD83E\uDD39\uD83C\uDFFF",
            [":person_kneeling:"] = "\uD83E\uDDCE",
            [":person_kneeling::skin-tone-1:"] = "\uD83E\uDDCE\uD83C\uDFFB",
            [":person_kneeling::skin-tone-2:"] = "\uD83E\uDDCE\uD83C\uDFFC",
            [":person_kneeling::skin-tone-3:"] = "\uD83E\uDDCE\uD83C\uDFFD",
            [":person_kneeling::skin-tone-4:"] = "\uD83E\uDDCE\uD83C\uDFFE",
            [":person_kneeling::skin-tone-5:"] = "\uD83E\uDDCE\uD83C\uDFFF",
            [":person_kneeling_dark_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFF",
            [":person_kneeling_light_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFB",
            [":person_kneeling_medium_dark_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFE",
            [":person_kneeling_medium_light_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFC",
            [":person_kneeling_medium_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFD",
            [":person_kneeling_tone1:"] = "\uD83E\uDDCE\uD83C\uDFFB",
            [":person_kneeling_tone2:"] = "\uD83E\uDDCE\uD83C\uDFFC",
            [":person_kneeling_tone3:"] = "\uD83E\uDDCE\uD83C\uDFFD",
            [":person_kneeling_tone4:"] = "\uD83E\uDDCE\uD83C\uDFFE",
            [":person_kneeling_tone5:"] = "\uD83E\uDDCE\uD83C\uDFFF",
            [":person_lifting_weights:"] = "\uD83C\uDFCB️",
            [":person_lifting_weights::skin-tone-1:"] = "\uD83C\uDFCB\uD83C\uDFFB",
            [":person_lifting_weights::skin-tone-2:"] = "\uD83C\uDFCB\uD83C\uDFFC",
            [":person_lifting_weights::skin-tone-3:"] = "\uD83C\uDFCB\uD83C\uDFFD",
            [":person_lifting_weights::skin-tone-4:"] = "\uD83C\uDFCB\uD83C\uDFFE",
            [":person_lifting_weights::skin-tone-5:"] = "\uD83C\uDFCB\uD83C\uDFFF",
            [":person_lifting_weights_tone1:"] = "\uD83C\uDFCB\uD83C\uDFFB",
            [":person_lifting_weights_tone2:"] = "\uD83C\uDFCB\uD83C\uDFFC",
            [":person_lifting_weights_tone3:"] = "\uD83C\uDFCB\uD83C\uDFFD",
            [":person_lifting_weights_tone4:"] = "\uD83C\uDFCB\uD83C\uDFFE",
            [":person_lifting_weights_tone5:"] = "\uD83C\uDFCB\uD83C\uDFFF",
            [":person_mountain_biking:"] = "\uD83D\uDEB5",
            [":person_mountain_biking::skin-tone-1:"] = "\uD83D\uDEB5\uD83C\uDFFB",
            [":person_mountain_biking::skin-tone-2:"] = "\uD83D\uDEB5\uD83C\uDFFC",
            [":person_mountain_biking::skin-tone-3:"] = "\uD83D\uDEB5\uD83C\uDFFD",
            [":person_mountain_biking::skin-tone-4:"] = "\uD83D\uDEB5\uD83C\uDFFE",
            [":person_mountain_biking::skin-tone-5:"] = "\uD83D\uDEB5\uD83C\uDFFF",
            [":person_mountain_biking_tone1:"] = "\uD83D\uDEB5\uD83C\uDFFB",
            [":person_mountain_biking_tone2:"] = "\uD83D\uDEB5\uD83C\uDFFC",
            [":person_mountain_biking_tone3:"] = "\uD83D\uDEB5\uD83C\uDFFD",
            [":person_mountain_biking_tone4:"] = "\uD83D\uDEB5\uD83C\uDFFE",
            [":person_mountain_biking_tone5:"] = "\uD83D\uDEB5\uD83C\uDFFF",
            [":person_playing_handball:"] = "\uD83E\uDD3E",
            [":person_playing_handball::skin-tone-1:"] = "\uD83E\uDD3E\uD83C\uDFFB",
            [":person_playing_handball::skin-tone-2:"] = "\uD83E\uDD3E\uD83C\uDFFC",
            [":person_playing_handball::skin-tone-3:"] = "\uD83E\uDD3E\uD83C\uDFFD",
            [":person_playing_handball::skin-tone-4:"] = "\uD83E\uDD3E\uD83C\uDFFE",
            [":person_playing_handball::skin-tone-5:"] = "\uD83E\uDD3E\uD83C\uDFFF",
            [":person_playing_handball_tone1:"] = "\uD83E\uDD3E\uD83C\uDFFB",
            [":person_playing_handball_tone2:"] = "\uD83E\uDD3E\uD83C\uDFFC",
            [":person_playing_handball_tone3:"] = "\uD83E\uDD3E\uD83C\uDFFD",
            [":person_playing_handball_tone4:"] = "\uD83E\uDD3E\uD83C\uDFFE",
            [":person_playing_handball_tone5:"] = "\uD83E\uDD3E\uD83C\uDFFF",
            [":person_playing_water_polo:"] = "\uD83E\uDD3D",
            [":person_playing_water_polo::skin-tone-1:"] = "\uD83E\uDD3D\uD83C\uDFFB",
            [":person_playing_water_polo::skin-tone-2:"] = "\uD83E\uDD3D\uD83C\uDFFC",
            [":person_playing_water_polo::skin-tone-3:"] = "\uD83E\uDD3D\uD83C\uDFFD",
            [":person_playing_water_polo::skin-tone-4:"] = "\uD83E\uDD3D\uD83C\uDFFE",
            [":person_playing_water_polo::skin-tone-5:"] = "\uD83E\uDD3D\uD83C\uDFFF",
            [":person_playing_water_polo_tone1:"] = "\uD83E\uDD3D\uD83C\uDFFB",
            [":person_playing_water_polo_tone2:"] = "\uD83E\uDD3D\uD83C\uDFFC",
            [":person_playing_water_polo_tone3:"] = "\uD83E\uDD3D\uD83C\uDFFD",
            [":person_playing_water_polo_tone4:"] = "\uD83E\uDD3D\uD83C\uDFFE",
            [":person_playing_water_polo_tone5:"] = "\uD83E\uDD3D\uD83C\uDFFF",
            [":person_pouting:"] = "\uD83D\uDE4E",
            [":person_pouting::skin-tone-1:"] = "\uD83D\uDE4E\uD83C\uDFFB",
            [":person_pouting::skin-tone-2:"] = "\uD83D\uDE4E\uD83C\uDFFC",
            [":person_pouting::skin-tone-3:"] = "\uD83D\uDE4E\uD83C\uDFFD",
            [":person_pouting::skin-tone-4:"] = "\uD83D\uDE4E\uD83C\uDFFE",
            [":person_pouting::skin-tone-5:"] = "\uD83D\uDE4E\uD83C\uDFFF",
            [":person_pouting_tone1:"] = "\uD83D\uDE4E\uD83C\uDFFB",
            [":person_pouting_tone2:"] = "\uD83D\uDE4E\uD83C\uDFFC",
            [":person_pouting_tone3:"] = "\uD83D\uDE4E\uD83C\uDFFD",
            [":person_pouting_tone4:"] = "\uD83D\uDE4E\uD83C\uDFFE",
            [":person_pouting_tone5:"] = "\uD83D\uDE4E\uD83C\uDFFF",
            [":person_raising_hand:"] = "\uD83D\uDE4B",
            [":person_raising_hand::skin-tone-1:"] = "\uD83D\uDE4B\uD83C\uDFFB",
            [":person_raising_hand::skin-tone-2:"] = "\uD83D\uDE4B\uD83C\uDFFC",
            [":person_raising_hand::skin-tone-3:"] = "\uD83D\uDE4B\uD83C\uDFFD",
            [":person_raising_hand::skin-tone-4:"] = "\uD83D\uDE4B\uD83C\uDFFE",
            [":person_raising_hand::skin-tone-5:"] = "\uD83D\uDE4B\uD83C\uDFFF",
            [":person_raising_hand_tone1:"] = "\uD83D\uDE4B\uD83C\uDFFB",
            [":person_raising_hand_tone2:"] = "\uD83D\uDE4B\uD83C\uDFFC",
            [":person_raising_hand_tone3:"] = "\uD83D\uDE4B\uD83C\uDFFD",
            [":person_raising_hand_tone4:"] = "\uD83D\uDE4B\uD83C\uDFFE",
            [":person_raising_hand_tone5:"] = "\uD83D\uDE4B\uD83C\uDFFF",
            [":person_rowing_boat:"] = "\uD83D\uDEA3",
            [":person_rowing_boat::skin-tone-1:"] = "\uD83D\uDEA3\uD83C\uDFFB",
            [":person_rowing_boat::skin-tone-2:"] = "\uD83D\uDEA3\uD83C\uDFFC",
            [":person_rowing_boat::skin-tone-3:"] = "\uD83D\uDEA3\uD83C\uDFFD",
            [":person_rowing_boat::skin-tone-4:"] = "\uD83D\uDEA3\uD83C\uDFFE",
            [":person_rowing_boat::skin-tone-5:"] = "\uD83D\uDEA3\uD83C\uDFFF",
            [":person_rowing_boat_tone1:"] = "\uD83D\uDEA3\uD83C\uDFFB",
            [":person_rowing_boat_tone2:"] = "\uD83D\uDEA3\uD83C\uDFFC",
            [":person_rowing_boat_tone3:"] = "\uD83D\uDEA3\uD83C\uDFFD",
            [":person_rowing_boat_tone4:"] = "\uD83D\uDEA3\uD83C\uDFFE",
            [":person_rowing_boat_tone5:"] = "\uD83D\uDEA3\uD83C\uDFFF",
            [":person_running:"] = "\uD83C\uDFC3",
            [":person_running::skin-tone-1:"] = "\uD83C\uDFC3\uD83C\uDFFB",
            [":person_running::skin-tone-2:"] = "\uD83C\uDFC3\uD83C\uDFFC",
            [":person_running::skin-tone-3:"] = "\uD83C\uDFC3\uD83C\uDFFD",
            [":person_running::skin-tone-4:"] = "\uD83C\uDFC3\uD83C\uDFFE",
            [":person_running::skin-tone-5:"] = "\uD83C\uDFC3\uD83C\uDFFF",
            [":person_running_tone1:"] = "\uD83C\uDFC3\uD83C\uDFFB",
            [":person_running_tone2:"] = "\uD83C\uDFC3\uD83C\uDFFC",
            [":person_running_tone3:"] = "\uD83C\uDFC3\uD83C\uDFFD",
            [":person_running_tone4:"] = "\uD83C\uDFC3\uD83C\uDFFE",
            [":person_running_tone5:"] = "\uD83C\uDFC3\uD83C\uDFFF",
            [":person_shrugging:"] = "\uD83E\uDD37",
            [":person_shrugging::skin-tone-1:"] = "\uD83E\uDD37\uD83C\uDFFB",
            [":person_shrugging::skin-tone-2:"] = "\uD83E\uDD37\uD83C\uDFFC",
            [":person_shrugging::skin-tone-3:"] = "\uD83E\uDD37\uD83C\uDFFD",
            [":person_shrugging::skin-tone-4:"] = "\uD83E\uDD37\uD83C\uDFFE",
            [":person_shrugging::skin-tone-5:"] = "\uD83E\uDD37\uD83C\uDFFF",
            [":person_shrugging_tone1:"] = "\uD83E\uDD37\uD83C\uDFFB",
            [":person_shrugging_tone2:"] = "\uD83E\uDD37\uD83C\uDFFC",
            [":person_shrugging_tone3:"] = "\uD83E\uDD37\uD83C\uDFFD",
            [":person_shrugging_tone4:"] = "\uD83E\uDD37\uD83C\uDFFE",
            [":person_shrugging_tone5:"] = "\uD83E\uDD37\uD83C\uDFFF",
            [":person_standing:"] = "\uD83E\uDDCD",
            [":person_standing::skin-tone-1:"] = "\uD83E\uDDCD\uD83C\uDFFB",
            [":person_standing::skin-tone-2:"] = "\uD83E\uDDCD\uD83C\uDFFC",
            [":person_standing::skin-tone-3:"] = "\uD83E\uDDCD\uD83C\uDFFD",
            [":person_standing::skin-tone-4:"] = "\uD83E\uDDCD\uD83C\uDFFE",
            [":person_standing::skin-tone-5:"] = "\uD83E\uDDCD\uD83C\uDFFF",
            [":person_standing_dark_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFF",
            [":person_standing_light_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFB",
            [":person_standing_medium_dark_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFE",
            [":person_standing_medium_light_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFC",
            [":person_standing_medium_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFD",
            [":person_standing_tone1:"] = "\uD83E\uDDCD\uD83C\uDFFB",
            [":person_standing_tone2:"] = "\uD83E\uDDCD\uD83C\uDFFC",
            [":person_standing_tone3:"] = "\uD83E\uDDCD\uD83C\uDFFD",
            [":person_standing_tone4:"] = "\uD83E\uDDCD\uD83C\uDFFE",
            [":person_standing_tone5:"] = "\uD83E\uDDCD\uD83C\uDFFF",
            [":person_surfing:"] = "\uD83C\uDFC4",
            [":person_surfing::skin-tone-1:"] = "\uD83C\uDFC4\uD83C\uDFFB",
            [":person_surfing::skin-tone-2:"] = "\uD83C\uDFC4\uD83C\uDFFC",
            [":person_surfing::skin-tone-3:"] = "\uD83C\uDFC4\uD83C\uDFFD",
            [":person_surfing::skin-tone-4:"] = "\uD83C\uDFC4\uD83C\uDFFE",
            [":person_surfing::skin-tone-5:"] = "\uD83C\uDFC4\uD83C\uDFFF",
            [":person_surfing_tone1:"] = "\uD83C\uDFC4\uD83C\uDFFB",
            [":person_surfing_tone2:"] = "\uD83C\uDFC4\uD83C\uDFFC",
            [":person_surfing_tone3:"] = "\uD83C\uDFC4\uD83C\uDFFD",
            [":person_surfing_tone4:"] = "\uD83C\uDFC4\uD83C\uDFFE",
            [":person_surfing_tone5:"] = "\uD83C\uDFC4\uD83C\uDFFF",
            [":person_swimming:"] = "\uD83C\uDFCA",
            [":person_swimming::skin-tone-1:"] = "\uD83C\uDFCA\uD83C\uDFFB",
            [":person_swimming::skin-tone-2:"] = "\uD83C\uDFCA\uD83C\uDFFC",
            [":person_swimming::skin-tone-3:"] = "\uD83C\uDFCA\uD83C\uDFFD",
            [":person_swimming::skin-tone-4:"] = "\uD83C\uDFCA\uD83C\uDFFE",
            [":person_swimming::skin-tone-5:"] = "\uD83C\uDFCA\uD83C\uDFFF",
            [":person_swimming_tone1:"] = "\uD83C\uDFCA\uD83C\uDFFB",
            [":person_swimming_tone2:"] = "\uD83C\uDFCA\uD83C\uDFFC",
            [":person_swimming_tone3:"] = "\uD83C\uDFCA\uD83C\uDFFD",
            [":person_swimming_tone4:"] = "\uD83C\uDFCA\uD83C\uDFFE",
            [":person_swimming_tone5:"] = "\uD83C\uDFCA\uD83C\uDFFF",
            [":person_tipping_hand:"] = "\uD83D\uDC81",
            [":person_tipping_hand::skin-tone-1:"] = "\uD83D\uDC81\uD83C\uDFFB",
            [":person_tipping_hand::skin-tone-2:"] = "\uD83D\uDC81\uD83C\uDFFC",
            [":person_tipping_hand::skin-tone-3:"] = "\uD83D\uDC81\uD83C\uDFFD",
            [":person_tipping_hand::skin-tone-4:"] = "\uD83D\uDC81\uD83C\uDFFE",
            [":person_tipping_hand::skin-tone-5:"] = "\uD83D\uDC81\uD83C\uDFFF",
            [":person_tipping_hand_tone1:"] = "\uD83D\uDC81\uD83C\uDFFB",
            [":person_tipping_hand_tone2:"] = "\uD83D\uDC81\uD83C\uDFFC",
            [":person_tipping_hand_tone3:"] = "\uD83D\uDC81\uD83C\uDFFD",
            [":person_tipping_hand_tone4:"] = "\uD83D\uDC81\uD83C\uDFFE",
            [":person_tipping_hand_tone5:"] = "\uD83D\uDC81\uD83C\uDFFF",
            [":person_walking:"] = "\uD83D\uDEB6",
            [":person_walking::skin-tone-1:"] = "\uD83D\uDEB6\uD83C\uDFFB",
            [":person_walking::skin-tone-2:"] = "\uD83D\uDEB6\uD83C\uDFFC",
            [":person_walking::skin-tone-3:"] = "\uD83D\uDEB6\uD83C\uDFFD",
            [":person_walking::skin-tone-4:"] = "\uD83D\uDEB6\uD83C\uDFFE",
            [":person_walking::skin-tone-5:"] = "\uD83D\uDEB6\uD83C\uDFFF",
            [":person_walking_tone1:"] = "\uD83D\uDEB6\uD83C\uDFFB",
            [":person_walking_tone2:"] = "\uD83D\uDEB6\uD83C\uDFFC",
            [":person_walking_tone3:"] = "\uD83D\uDEB6\uD83C\uDFFD",
            [":person_walking_tone4:"] = "\uD83D\uDEB6\uD83C\uDFFE",
            [":person_walking_tone5:"] = "\uD83D\uDEB6\uD83C\uDFFF",
            [":person_wearing_turban:"] = "\uD83D\uDC73",
            [":person_wearing_turban::skin-tone-1:"] = "\uD83D\uDC73\uD83C\uDFFB",
            [":person_wearing_turban::skin-tone-2:"] = "\uD83D\uDC73\uD83C\uDFFC",
            [":person_wearing_turban::skin-tone-3:"] = "\uD83D\uDC73\uD83C\uDFFD",
            [":person_wearing_turban::skin-tone-4:"] = "\uD83D\uDC73\uD83C\uDFFE",
            [":person_wearing_turban::skin-tone-5:"] = "\uD83D\uDC73\uD83C\uDFFF",
            [":person_wearing_turban_tone1:"] = "\uD83D\uDC73\uD83C\uDFFB",
            [":person_wearing_turban_tone2:"] = "\uD83D\uDC73\uD83C\uDFFC",
            [":person_wearing_turban_tone3:"] = "\uD83D\uDC73\uD83C\uDFFD",
            [":person_wearing_turban_tone4:"] = "\uD83D\uDC73\uD83C\uDFFE",
            [":person_wearing_turban_tone5:"] = "\uD83D\uDC73\uD83C\uDFFF",
            [":person_with_ball:"] = "⛹️",
            [":person_with_ball::skin-tone-1:"] = "⛹\uD83C\uDFFB",
            [":person_with_ball::skin-tone-2:"] = "⛹\uD83C\uDFFC",
            [":person_with_ball::skin-tone-3:"] = "⛹\uD83C\uDFFD",
            [":person_with_ball::skin-tone-4:"] = "⛹\uD83C\uDFFE",
            [":person_with_ball::skin-tone-5:"] = "⛹\uD83C\uDFFF",
            [":person_with_ball_tone1:"] = "⛹\uD83C\uDFFB",
            [":person_with_ball_tone2:"] = "⛹\uD83C\uDFFC",
            [":person_with_ball_tone3:"] = "⛹\uD83C\uDFFD",
            [":person_with_ball_tone4:"] = "⛹\uD83C\uDFFE",
            [":person_with_ball_tone5:"] = "⛹\uD83C\uDFFF",
            [":person_with_blond_hair:"] = "\uD83D\uDC71",
            [":person_with_blond_hair::skin-tone-1:"] = "\uD83D\uDC71\uD83C\uDFFB",
            [":person_with_blond_hair::skin-tone-2:"] = "\uD83D\uDC71\uD83C\uDFFC",
            [":person_with_blond_hair::skin-tone-3:"] = "\uD83D\uDC71\uD83C\uDFFD",
            [":person_with_blond_hair::skin-tone-4:"] = "\uD83D\uDC71\uD83C\uDFFE",
            [":person_with_blond_hair::skin-tone-5:"] = "\uD83D\uDC71\uD83C\uDFFF",
            [":person_with_blond_hair_tone1:"] = "\uD83D\uDC71\uD83C\uDFFB",
            [":person_with_blond_hair_tone2:"] = "\uD83D\uDC71\uD83C\uDFFC",
            [":person_with_blond_hair_tone3:"] = "\uD83D\uDC71\uD83C\uDFFD",
            [":person_with_blond_hair_tone4:"] = "\uD83D\uDC71\uD83C\uDFFE",
            [":person_with_blond_hair_tone5:"] = "\uD83D\uDC71\uD83C\uDFFF",
            [":person_with_crown:"] = "\uD83E\uDEC5",
            [":person_with_crown::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown::skin-tone-5:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":person_with_crown_dark_skin_tone:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":person_with_crown_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":person_with_crown_tone5:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":person_with_pouting_face:"] = "\uD83D\uDE4E",
            [":person_with_pouting_face::skin-tone-1:"] = "\uD83D\uDE4E\uD83C\uDFFB",
            [":person_with_pouting_face::skin-tone-2:"] = "\uD83D\uDE4E\uD83C\uDFFC",
            [":person_with_pouting_face::skin-tone-3:"] = "\uD83D\uDE4E\uD83C\uDFFD",
            [":person_with_pouting_face::skin-tone-4:"] = "\uD83D\uDE4E\uD83C\uDFFE",
            [":person_with_pouting_face::skin-tone-5:"] = "\uD83D\uDE4E\uD83C\uDFFF",
            [":person_with_pouting_face_tone1:"] = "\uD83D\uDE4E\uD83C\uDFFB",
            [":person_with_pouting_face_tone2:"] = "\uD83D\uDE4E\uD83C\uDFFC",
            [":person_with_pouting_face_tone3:"] = "\uD83D\uDE4E\uD83C\uDFFD",
            [":person_with_pouting_face_tone4:"] = "\uD83D\uDE4E\uD83C\uDFFE",
            [":person_with_pouting_face_tone5:"] = "\uD83D\uDE4E\uD83C\uDFFF",
            [":petri_dish:"] = "\uD83E\uDDEB",
            [":pick:"] = "⛏️",
            [":pickup_truck:"] = "\uD83D\uDEFB",
            [":pie:"] = "\uD83E\uDD67",
            [":pig2:"] = "\uD83D\uDC16",
            [":pig:"] = "\uD83D\uDC37",
            [":pig_nose:"] = "\uD83D\uDC3D",
            [":pill:"] = "\uD83D\uDC8A",
            [":pinata:"] = "\uD83E\uDE85",
            [":pinched_fingers:"] = "\uD83E\uDD0C",
            [":pinched_fingers::skin-tone-1:"] = "\uD83E\uDD0C\uD83C\uDFFB",
            [":pinched_fingers::skin-tone-2:"] = "\uD83E\uDD0C\uD83C\uDFFC",
            [":pinched_fingers::skin-tone-3:"] = "\uD83E\uDD0C\uD83C\uDFFD",
            [":pinched_fingers::skin-tone-4:"] = "\uD83E\uDD0C\uD83C\uDFFE",
            [":pinched_fingers::skin-tone-5:"] = "\uD83E\uDD0C\uD83C\uDFFF",
            [":pinched_fingers_dark_skin_tone:"] = "\uD83E\uDD0C\uD83C\uDFFF",
            [":pinched_fingers_light_skin_tone:"] = "\uD83E\uDD0C\uD83C\uDFFB",
            [":pinched_fingers_medium_dark_skin_tone:"] = "\uD83E\uDD0C\uD83C\uDFFE",
            [":pinched_fingers_medium_light_skin_tone:"] = "\uD83E\uDD0C\uD83C\uDFFC",
            [":pinched_fingers_medium_skin_tone:"] = "\uD83E\uDD0C\uD83C\uDFFD",
            [":pinched_fingers_tone1:"] = "\uD83E\uDD0C\uD83C\uDFFB",
            [":pinched_fingers_tone2:"] = "\uD83E\uDD0C\uD83C\uDFFC",
            [":pinched_fingers_tone3:"] = "\uD83E\uDD0C\uD83C\uDFFD",
            [":pinched_fingers_tone4:"] = "\uD83E\uDD0C\uD83C\uDFFE",
            [":pinched_fingers_tone5:"] = "\uD83E\uDD0C\uD83C\uDFFF",
            [":pinching_hand:"] = "\uD83E\uDD0F",
            [":pinching_hand::skin-tone-1:"] = "\uD83E\uDD0F\uD83C\uDFFB",
            [":pinching_hand::skin-tone-2:"] = "\uD83E\uDD0F\uD83C\uDFFC",
            [":pinching_hand::skin-tone-3:"] = "\uD83E\uDD0F\uD83C\uDFFD",
            [":pinching_hand::skin-tone-4:"] = "\uD83E\uDD0F\uD83C\uDFFE",
            [":pinching_hand::skin-tone-5:"] = "\uD83E\uDD0F\uD83C\uDFFF",
            [":pinching_hand_dark_skin_tone:"] = "\uD83E\uDD0F\uD83C\uDFFF",
            [":pinching_hand_light_skin_tone:"] = "\uD83E\uDD0F\uD83C\uDFFB",
            [":pinching_hand_medium_dark_skin_tone:"] = "\uD83E\uDD0F\uD83C\uDFFE",
            [":pinching_hand_medium_light_skin_tone:"] = "\uD83E\uDD0F\uD83C\uDFFC",
            [":pinching_hand_medium_skin_tone:"] = "\uD83E\uDD0F\uD83C\uDFFD",
            [":pinching_hand_tone1:"] = "\uD83E\uDD0F\uD83C\uDFFB",
            [":pinching_hand_tone2:"] = "\uD83E\uDD0F\uD83C\uDFFC",
            [":pinching_hand_tone3:"] = "\uD83E\uDD0F\uD83C\uDFFD",
            [":pinching_hand_tone4:"] = "\uD83E\uDD0F\uD83C\uDFFE",
            [":pinching_hand_tone5:"] = "\uD83E\uDD0F\uD83C\uDFFF",
            [":pineapple:"] = "\uD83C\uDF4D",
            [":ping_pong:"] = "\uD83C\uDFD3",
            [":pirate_flag:"] = "\uD83C\uDFF4\u200D☠️",
            [":pisces:"] = "♓",
            [":pizza:"] = "\uD83C\uDF55",
            [":placard:"] = "\uD83E\uDEA7",
            [":place_of_worship:"] = "\uD83D\uDED0",
            [":play_pause:"] = "⏯️",
            [":playground_slide:"] = "\uD83D\uDEDD",
            [":pleading_face:"] = "\uD83E\uDD7A",
            [":plunger:"] = "\uD83E\uDEA0",
            [":point_down:"] = "\uD83D\uDC47",
            [":point_down::skin-tone-1:"] = "\uD83D\uDC47\uD83C\uDFFB",
            [":point_down::skin-tone-2:"] = "\uD83D\uDC47\uD83C\uDFFC",
            [":point_down::skin-tone-3:"] = "\uD83D\uDC47\uD83C\uDFFD",
            [":point_down::skin-tone-4:"] = "\uD83D\uDC47\uD83C\uDFFE",
            [":point_down::skin-tone-5:"] = "\uD83D\uDC47\uD83C\uDFFF",
            [":point_down_tone1:"] = "\uD83D\uDC47\uD83C\uDFFB",
            [":point_down_tone2:"] = "\uD83D\uDC47\uD83C\uDFFC",
            [":point_down_tone3:"] = "\uD83D\uDC47\uD83C\uDFFD",
            [":point_down_tone4:"] = "\uD83D\uDC47\uD83C\uDFFE",
            [":point_down_tone5:"] = "\uD83D\uDC47\uD83C\uDFFF",
            [":point_left:"] = "\uD83D\uDC48",
            [":point_left::skin-tone-1:"] = "\uD83D\uDC48\uD83C\uDFFB",
            [":point_left::skin-tone-2:"] = "\uD83D\uDC48\uD83C\uDFFC",
            [":point_left::skin-tone-3:"] = "\uD83D\uDC48\uD83C\uDFFD",
            [":point_left::skin-tone-4:"] = "\uD83D\uDC48\uD83C\uDFFE",
            [":point_left::skin-tone-5:"] = "\uD83D\uDC48\uD83C\uDFFF",
            [":point_left_tone1:"] = "\uD83D\uDC48\uD83C\uDFFB",
            [":point_left_tone2:"] = "\uD83D\uDC48\uD83C\uDFFC",
            [":point_left_tone3:"] = "\uD83D\uDC48\uD83C\uDFFD",
            [":point_left_tone4:"] = "\uD83D\uDC48\uD83C\uDFFE",
            [":point_left_tone5:"] = "\uD83D\uDC48\uD83C\uDFFF",
            [":point_right:"] = "\uD83D\uDC49",
            [":point_right::skin-tone-1:"] = "\uD83D\uDC49\uD83C\uDFFB",
            [":point_right::skin-tone-2:"] = "\uD83D\uDC49\uD83C\uDFFC",
            [":point_right::skin-tone-3:"] = "\uD83D\uDC49\uD83C\uDFFD",
            [":point_right::skin-tone-4:"] = "\uD83D\uDC49\uD83C\uDFFE",
            [":point_right::skin-tone-5:"] = "\uD83D\uDC49\uD83C\uDFFF",
            [":point_right_tone1:"] = "\uD83D\uDC49\uD83C\uDFFB",
            [":point_right_tone2:"] = "\uD83D\uDC49\uD83C\uDFFC",
            [":point_right_tone3:"] = "\uD83D\uDC49\uD83C\uDFFD",
            [":point_right_tone4:"] = "\uD83D\uDC49\uD83C\uDFFE",
            [":point_right_tone5:"] = "\uD83D\uDC49\uD83C\uDFFF",
            [":point_up:"] = "☝️",
            [":point_up::skin-tone-1:"] = "☝\uD83C\uDFFB",
            [":point_up::skin-tone-2:"] = "☝\uD83C\uDFFC",
            [":point_up::skin-tone-3:"] = "☝\uD83C\uDFFD",
            [":point_up::skin-tone-4:"] = "☝\uD83C\uDFFE",
            [":point_up::skin-tone-5:"] = "☝\uD83C\uDFFF",
            [":point_up_2:"] = "\uD83D\uDC46",
            [":point_up_2::skin-tone-1:"] = "\uD83D\uDC46\uD83C\uDFFB",
            [":point_up_2::skin-tone-2:"] = "\uD83D\uDC46\uD83C\uDFFC",
            [":point_up_2::skin-tone-3:"] = "\uD83D\uDC46\uD83C\uDFFD",
            [":point_up_2::skin-tone-4:"] = "\uD83D\uDC46\uD83C\uDFFE",
            [":point_up_2::skin-tone-5:"] = "\uD83D\uDC46\uD83C\uDFFF",
            [":point_up_2_tone1:"] = "\uD83D\uDC46\uD83C\uDFFB",
            [":point_up_2_tone2:"] = "\uD83D\uDC46\uD83C\uDFFC",
            [":point_up_2_tone3:"] = "\uD83D\uDC46\uD83C\uDFFD",
            [":point_up_2_tone4:"] = "\uD83D\uDC46\uD83C\uDFFE",
            [":point_up_2_tone5:"] = "\uD83D\uDC46\uD83C\uDFFF",
            [":point_up_tone1:"] = "☝\uD83C\uDFFB",
            [":point_up_tone2:"] = "☝\uD83C\uDFFC",
            [":point_up_tone3:"] = "☝\uD83C\uDFFD",
            [":point_up_tone4:"] = "☝\uD83C\uDFFE",
            [":point_up_tone5:"] = "☝\uD83C\uDFFF",
            [":polar_bear:"] = "\uD83D\uDC3B\u200D\u2744\uFE0F",
            [":police_car:"] = "\uD83D\uDE93",
            [":police_officer:"] = "\uD83D\uDC6E",
            [":police_officer::skin-tone-1:"] = "\uD83D\uDC6E\uD83C\uDFFB",
            [":police_officer::skin-tone-2:"] = "\uD83D\uDC6E\uD83C\uDFFC",
            [":police_officer::skin-tone-3:"] = "\uD83D\uDC6E\uD83C\uDFFD",
            [":police_officer::skin-tone-4:"] = "\uD83D\uDC6E\uD83C\uDFFE",
            [":police_officer::skin-tone-5:"] = "\uD83D\uDC6E\uD83C\uDFFF",
            [":police_officer_tone1:"] = "\uD83D\uDC6E\uD83C\uDFFB",
            [":police_officer_tone2:"] = "\uD83D\uDC6E\uD83C\uDFFC",
            [":police_officer_tone3:"] = "\uD83D\uDC6E\uD83C\uDFFD",
            [":police_officer_tone4:"] = "\uD83D\uDC6E\uD83C\uDFFE",
            [":police_officer_tone5:"] = "\uD83D\uDC6E\uD83C\uDFFF",
            [":poo:"] = "\uD83D\uDCA9",
            [":poodle:"] = "\uD83D\uDC29",
            [":poop:"] = "\uD83D\uDCA9",
            [":popcorn:"] = "\uD83C\uDF7F",
            [":post_office:"] = "\uD83C\uDFE3",
            [":postal_horn:"] = "\uD83D\uDCEF",
            [":postbox:"] = "\uD83D\uDCEE",
            [":potable_water:"] = "\uD83D\uDEB0",
            [":potato:"] = "\uD83E\uDD54",
            [":potted_plant:"] = "\uD83E\uDEB4",
            [":pouch:"] = "\uD83D\uDC5D",
            [":poultry_leg:"] = "\uD83C\uDF57",
            [":pound:"] = "\uD83D\uDCB7",
            [":pouring_liquid:"] = "\uD83E\uDED7",
            [":pouting_cat:"] = "\uD83D\uDE3E",
            [":pray:"] = "\uD83D\uDE4F",
            [":pray::skin-tone-1:"] = "\uD83D\uDE4F\uD83C\uDFFB",
            [":pray::skin-tone-2:"] = "\uD83D\uDE4F\uD83C\uDFFC",
            [":pray::skin-tone-3:"] = "\uD83D\uDE4F\uD83C\uDFFD",
            [":pray::skin-tone-4:"] = "\uD83D\uDE4F\uD83C\uDFFE",
            [":pray::skin-tone-5:"] = "\uD83D\uDE4F\uD83C\uDFFF",
            [":pray_tone1:"] = "\uD83D\uDE4F\uD83C\uDFFB",
            [":pray_tone2:"] = "\uD83D\uDE4F\uD83C\uDFFC",
            [":pray_tone3:"] = "\uD83D\uDE4F\uD83C\uDFFD",
            [":pray_tone4:"] = "\uD83D\uDE4F\uD83C\uDFFE",
            [":pray_tone5:"] = "\uD83D\uDE4F\uD83C\uDFFF",
            [":prayer_beads:"] = "\uD83D\uDCFF",
            [":pregnant_man:"] = "\uD83E\uDEC3",
            [":pregnant_man::skin-tone-1:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":pregnant_man::skin-tone-2:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":pregnant_man::skin-tone-3:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man::skin-tone-4:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man::skin-tone-5:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man_dark_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man_light_skin_tone:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":pregnant_man_medium_dark_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man_medium_light_skin_tone:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":pregnant_man_medium_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man_tone1:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":pregnant_man_tone2:"] = "\uD83E\uDEC5\uD83C\uDFFF",
            [":pregnant_man_tone3:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man_tone4:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_man_tone5:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person:"] = "\uD83E\uDEC4",
            [":pregnant_person::skin-tone-1:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person::skin-tone-2:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person::skin-tone-3:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person::skin-tone-4:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person::skin-tone-5:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_dark_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_light_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_medium_dark_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_medium_light_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_medium_skin_tone:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_tone1:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_tone2:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_tone3:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_tone4:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_person_tone5:"] = "\uD83E\uDEC3\uD83C\uDFFD",
            [":pregnant_woman:"] = "\uD83E\uDD30",
            [":pregnant_woman::skin-tone-1:"] = "\uD83E\uDD30\uD83C\uDFFB",
            [":pregnant_woman::skin-tone-2:"] = "\uD83E\uDD30\uD83C\uDFFC",
            [":pregnant_woman::skin-tone-3:"] = "\uD83E\uDD30\uD83C\uDFFD",
            [":pregnant_woman::skin-tone-4:"] = "\uD83E\uDD30\uD83C\uDFFE",
            [":pregnant_woman::skin-tone-5:"] = "\uD83E\uDD30\uD83C\uDFFF",
            [":pregnant_woman_tone1:"] = "\uD83E\uDD30\uD83C\uDFFB",
            [":pregnant_woman_tone2:"] = "\uD83E\uDD30\uD83C\uDFFC",
            [":pregnant_woman_tone3:"] = "\uD83E\uDD30\uD83C\uDFFD",
            [":pregnant_woman_tone4:"] = "\uD83E\uDD30\uD83C\uDFFE",
            [":pregnant_woman_tone5:"] = "\uD83E\uDD30\uD83C\uDFFF",
            [":pretzel:"] = "\uD83E\uDD68",
            [":previous_track:"] = "⏮️",
            [":prince:"] = "\uD83E\uDD34",
            [":prince::skin-tone-1:"] = "\uD83E\uDD34\uD83C\uDFFB",
            [":prince::skin-tone-2:"] = "\uD83E\uDD34\uD83C\uDFFC",
            [":prince::skin-tone-3:"] = "\uD83E\uDD34\uD83C\uDFFD",
            [":prince::skin-tone-4:"] = "\uD83E\uDD34\uD83C\uDFFE",
            [":prince::skin-tone-5:"] = "\uD83E\uDD34\uD83C\uDFFF",
            [":prince_tone1:"] = "\uD83E\uDD34\uD83C\uDFFB",
            [":prince_tone2:"] = "\uD83E\uDD34\uD83C\uDFFC",
            [":prince_tone3:"] = "\uD83E\uDD34\uD83C\uDFFD",
            [":prince_tone4:"] = "\uD83E\uDD34\uD83C\uDFFE",
            [":prince_tone5:"] = "\uD83E\uDD34\uD83C\uDFFF",
            [":princess:"] = "\uD83D\uDC78",
            [":princess::skin-tone-1:"] = "\uD83D\uDC78\uD83C\uDFFB",
            [":princess::skin-tone-2:"] = "\uD83D\uDC78\uD83C\uDFFC",
            [":princess::skin-tone-3:"] = "\uD83D\uDC78\uD83C\uDFFD",
            [":princess::skin-tone-4:"] = "\uD83D\uDC78\uD83C\uDFFE",
            [":princess::skin-tone-5:"] = "\uD83D\uDC78\uD83C\uDFFF",
            [":princess_tone1:"] = "\uD83D\uDC78\uD83C\uDFFB",
            [":princess_tone2:"] = "\uD83D\uDC78\uD83C\uDFFC",
            [":princess_tone3:"] = "\uD83D\uDC78\uD83C\uDFFD",
            [":princess_tone4:"] = "\uD83D\uDC78\uD83C\uDFFE",
            [":princess_tone5:"] = "\uD83D\uDC78\uD83C\uDFFF",
            [":printer:"] = "\uD83D\uDDA8️",
            [":probing_cane:"] = "\uD83E\uDDAF",
            [":projector:"] = "\uD83D\uDCFD️",
            [":pudding:"] = "\uD83C\uDF6E",
            [":punch:"] = "\uD83D\uDC4A",
            [":punch::skin-tone-1:"] = "\uD83D\uDC4A\uD83C\uDFFB",
            [":punch::skin-tone-2:"] = "\uD83D\uDC4A\uD83C\uDFFC",
            [":punch::skin-tone-3:"] = "\uD83D\uDC4A\uD83C\uDFFD",
            [":punch::skin-tone-4:"] = "\uD83D\uDC4A\uD83C\uDFFE",
            [":punch::skin-tone-5:"] = "\uD83D\uDC4A\uD83C\uDFFF",
            [":punch_tone1:"] = "\uD83D\uDC4A\uD83C\uDFFB",
            [":punch_tone2:"] = "\uD83D\uDC4A\uD83C\uDFFC",
            [":punch_tone3:"] = "\uD83D\uDC4A\uD83C\uDFFD",
            [":punch_tone4:"] = "\uD83D\uDC4A\uD83C\uDFFE",
            [":punch_tone5:"] = "\uD83D\uDC4A\uD83C\uDFFF",
            [":purple_circle:"] = "\uD83D\uDFE3",
            [":purple_heart:"] = "\uD83D\uDC9C",
            [":purple_square:"] = "\uD83D\uDFEA",
            [":purse:"] = "\uD83D\uDC5B",
            [":pushpin:"] = "\uD83D\uDCCC",
            [":put_litter_in_its_place:"] = "\uD83D\uDEAE",
            [":question:"] = "❓",
            [":rabbit2:"] = "\uD83D\uDC07",
            [":rabbit:"] = "\uD83D\uDC30",
            [":raccoon:"] = "\uD83E\uDD9D",
            [":race_car:"] = "\uD83C\uDFCE️",
            [":racehorse:"] = "\uD83D\uDC0E",
            [":racing_car:"] = "\uD83C\uDFCE️",
            [":racing_motorcycle:"] = "\uD83C\uDFCD️",
            [":radio:"] = "\uD83D\uDCFB",
            [":radio_button:"] = "\uD83D\uDD18",
            [":radioactive:"] = "☢️",
            [":radioactive_sign:"] = "☢️",
            [":rage:"] = "\uD83D\uDE21",
            [":railroad_track:"] = "\uD83D\uDEE4️",
            [":railway_car:"] = "\uD83D\uDE83",
            [":railway_track:"] = "\uD83D\uDEE4️",
            [":rainbow:"] = "\uD83C\uDF08",
            [":rainbow_flag:"] = "\uD83C\uDFF3️\u200D\uD83C\uDF08",
            [":raised_back_of_hand:"] = "\uD83E\uDD1A",
            [":raised_back_of_hand::skin-tone-1:"] = "\uD83E\uDD1A\uD83C\uDFFB",
            [":raised_back_of_hand::skin-tone-2:"] = "\uD83E\uDD1A\uD83C\uDFFC",
            [":raised_back_of_hand::skin-tone-3:"] = "\uD83E\uDD1A\uD83C\uDFFD",
            [":raised_back_of_hand::skin-tone-4:"] = "\uD83E\uDD1A\uD83C\uDFFE",
            [":raised_back_of_hand::skin-tone-5:"] = "\uD83E\uDD1A\uD83C\uDFFF",
            [":raised_back_of_hand_tone1:"] = "\uD83E\uDD1A\uD83C\uDFFB",
            [":raised_back_of_hand_tone2:"] = "\uD83E\uDD1A\uD83C\uDFFC",
            [":raised_back_of_hand_tone3:"] = "\uD83E\uDD1A\uD83C\uDFFD",
            [":raised_back_of_hand_tone4:"] = "\uD83E\uDD1A\uD83C\uDFFE",
            [":raised_back_of_hand_tone5:"] = "\uD83E\uDD1A\uD83C\uDFFF",
            [":raised_hand:"] = "✋",
            [":raised_hand::skin-tone-1:"] = "✋\uD83C\uDFFB",
            [":raised_hand::skin-tone-2:"] = "✋\uD83C\uDFFC",
            [":raised_hand::skin-tone-3:"] = "✋\uD83C\uDFFD",
            [":raised_hand::skin-tone-4:"] = "✋\uD83C\uDFFE",
            [":raised_hand::skin-tone-5:"] = "✋\uD83C\uDFFF",
            [":raised_hand_tone1:"] = "✋\uD83C\uDFFB",
            [":raised_hand_tone2:"] = "✋\uD83C\uDFFC",
            [":raised_hand_tone3:"] = "✋\uD83C\uDFFD",
            [":raised_hand_tone4:"] = "✋\uD83C\uDFFE",
            [":raised_hand_tone5:"] = "✋\uD83C\uDFFF",
            [":raised_hand_with_fingers_splayed:"] = "\uD83D\uDD90️",
            [":raised_hand_with_fingers_splayed::skin-tone-1:"] = "\uD83D\uDD90\uD83C\uDFFB",
            [":raised_hand_with_fingers_splayed::skin-tone-2:"] = "\uD83D\uDD90\uD83C\uDFFC",
            [":raised_hand_with_fingers_splayed::skin-tone-3:"] = "\uD83D\uDD90\uD83C\uDFFD",
            [":raised_hand_with_fingers_splayed::skin-tone-4:"] = "\uD83D\uDD90\uD83C\uDFFE",
            [":raised_hand_with_fingers_splayed::skin-tone-5:"] = "\uD83D\uDD90\uD83C\uDFFF",
            [":raised_hand_with_fingers_splayed_tone1:"] = "\uD83D\uDD90\uD83C\uDFFB",
            [":raised_hand_with_fingers_splayed_tone2:"] = "\uD83D\uDD90\uD83C\uDFFC",
            [":raised_hand_with_fingers_splayed_tone3:"] = "\uD83D\uDD90\uD83C\uDFFD",
            [":raised_hand_with_fingers_splayed_tone4:"] = "\uD83D\uDD90\uD83C\uDFFE",
            [":raised_hand_with_fingers_splayed_tone5:"] = "\uD83D\uDD90\uD83C\uDFFF",
            [":raised_hand_with_part_between_middle_and_ring_fingers:"] = "\uD83D\uDD96",
            [":raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-1:"] = "\uD83D\uDD96\uD83C\uDFFB",
            [":raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-2:"] = "\uD83D\uDD96\uD83C\uDFFC",
            [":raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-3:"] = "\uD83D\uDD96\uD83C\uDFFD",
            [":raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-4:"] = "\uD83D\uDD96\uD83C\uDFFE",
            [":raised_hand_with_part_between_middle_and_ring_fingers::skin-tone-5:"] = "\uD83D\uDD96\uD83C\uDFFF",
            [":raised_hand_with_part_between_middle_and_ring_fingers_tone1:"] = "\uD83D\uDD96\uD83C\uDFFB",
            [":raised_hand_with_part_between_middle_and_ring_fingers_tone2:"] = "\uD83D\uDD96\uD83C\uDFFC",
            [":raised_hand_with_part_between_middle_and_ring_fingers_tone3:"] = "\uD83D\uDD96\uD83C\uDFFD",
            [":raised_hand_with_part_between_middle_and_ring_fingers_tone4:"] = "\uD83D\uDD96\uD83C\uDFFE",
            [":raised_hand_with_part_between_middle_and_ring_fingers_tone5:"] = "\uD83D\uDD96\uD83C\uDFFF",
            [":raised_hands:"] = "\uD83D\uDE4C",
            [":raised_hands::skin-tone-1:"] = "\uD83D\uDE4C\uD83C\uDFFB",
            [":raised_hands::skin-tone-2:"] = "\uD83D\uDE4C\uD83C\uDFFC",
            [":raised_hands::skin-tone-3:"] = "\uD83D\uDE4C\uD83C\uDFFD",
            [":raised_hands::skin-tone-4:"] = "\uD83D\uDE4C\uD83C\uDFFE",
            [":raised_hands::skin-tone-5:"] = "\uD83D\uDE4C\uD83C\uDFFF",
            [":raised_hands_tone1:"] = "\uD83D\uDE4C\uD83C\uDFFB",
            [":raised_hands_tone2:"] = "\uD83D\uDE4C\uD83C\uDFFC",
            [":raised_hands_tone3:"] = "\uD83D\uDE4C\uD83C\uDFFD",
            [":raised_hands_tone4:"] = "\uD83D\uDE4C\uD83C\uDFFE",
            [":raised_hands_tone5:"] = "\uD83D\uDE4C\uD83C\uDFFF",
            [":raising_hand:"] = "\uD83D\uDE4B",
            [":raising_hand::skin-tone-1:"] = "\uD83D\uDE4B\uD83C\uDFFB",
            [":raising_hand::skin-tone-2:"] = "\uD83D\uDE4B\uD83C\uDFFC",
            [":raising_hand::skin-tone-3:"] = "\uD83D\uDE4B\uD83C\uDFFD",
            [":raising_hand::skin-tone-4:"] = "\uD83D\uDE4B\uD83C\uDFFE",
            [":raising_hand::skin-tone-5:"] = "\uD83D\uDE4B\uD83C\uDFFF",
            [":raising_hand_tone1:"] = "\uD83D\uDE4B\uD83C\uDFFB",
            [":raising_hand_tone2:"] = "\uD83D\uDE4B\uD83C\uDFFC",
            [":raising_hand_tone3:"] = "\uD83D\uDE4B\uD83C\uDFFD",
            [":raising_hand_tone4:"] = "\uD83D\uDE4B\uD83C\uDFFE",
            [":raising_hand_tone5:"] = "\uD83D\uDE4B\uD83C\uDFFF",
            [":ram:"] = "\uD83D\uDC0F",
            [":ramen:"] = "\uD83C\uDF5C",
            [":rat:"] = "\uD83D\uDC00",
            [":razor:"] = "\uD83E\uDE92",
            [":receipt:"] = "\uD83E\uDDFE",
            [":record_button:"] = "⏺️",
            [":recycle:"] = "♻️",
            [":red_car:"] = "\uD83D\uDE97",
            [":red_circle:"] = "\uD83D\uDD34",
            [":red_envelope:"] = "\uD83E\uDDE7",
            [":red_square:"] = "\uD83D\uDFE5",
            [":regional_indicator_a:"] = "\uD83C\uDDE6",
            [":regional_indicator_b:"] = "\uD83C\uDDE7",
            [":regional_indicator_c:"] = "\uD83C\uDDE8",
            [":regional_indicator_d:"] = "\uD83C\uDDE9",
            [":regional_indicator_e:"] = "\uD83C\uDDEA",
            [":regional_indicator_f:"] = "\uD83C\uDDEB",
            [":regional_indicator_g:"] = "\uD83C\uDDEC",
            [":regional_indicator_h:"] = "\uD83C\uDDED",
            [":regional_indicator_i:"] = "\uD83C\uDDEE",
            [":regional_indicator_j:"] = "\uD83C\uDDEF",
            [":regional_indicator_k:"] = "\uD83C\uDDF0",
            [":regional_indicator_l:"] = "\uD83C\uDDF1",
            [":regional_indicator_m:"] = "\uD83C\uDDF2",
            [":regional_indicator_n:"] = "\uD83C\uDDF3",
            [":regional_indicator_o:"] = "\uD83C\uDDF4",
            [":regional_indicator_p:"] = "\uD83C\uDDF5",
            [":regional_indicator_q:"] = "\uD83C\uDDF6",
            [":regional_indicator_r:"] = "\uD83C\uDDF7",
            [":regional_indicator_s:"] = "\uD83C\uDDF8",
            [":regional_indicator_t:"] = "\uD83C\uDDF9",
            [":regional_indicator_u:"] = "\uD83C\uDDFA",
            [":regional_indicator_v:"] = "\uD83C\uDDFB",
            [":regional_indicator_w:"] = "\uD83C\uDDFC",
            [":regional_indicator_x:"] = "\uD83C\uDDFD",
            [":regional_indicator_y:"] = "\uD83C\uDDFE",
            [":regional_indicator_z:"] = "\uD83C\uDDFF",
            [":registered:"] = "®️",
            [":relaxed:"] = "☺️",
            [":relieved:"] = "\uD83D\uDE0C",
            [":reminder_ribbon:"] = "\uD83C\uDF97️",
            [":repeat:"] = "\uD83D\uDD01",
            [":repeat_one:"] = "\uD83D\uDD02",
            [":restroom:"] = "\uD83D\uDEBB",
            [":reversed_hand_with_middle_finger_extended:"] = "\uD83D\uDD95",
            [":reversed_hand_with_middle_finger_extended::skin-tone-1:"] = "\uD83D\uDD95\uD83C\uDFFB",
            [":reversed_hand_with_middle_finger_extended::skin-tone-2:"] = "\uD83D\uDD95\uD83C\uDFFC",
            [":reversed_hand_with_middle_finger_extended::skin-tone-3:"] = "\uD83D\uDD95\uD83C\uDFFD",
            [":reversed_hand_with_middle_finger_extended::skin-tone-4:"] = "\uD83D\uDD95\uD83C\uDFFE",
            [":reversed_hand_with_middle_finger_extended::skin-tone-5:"] = "\uD83D\uDD95\uD83C\uDFFF",
            [":reversed_hand_with_middle_finger_extended_tone1:"] = "\uD83D\uDD95\uD83C\uDFFB",
            [":reversed_hand_with_middle_finger_extended_tone2:"] = "\uD83D\uDD95\uD83C\uDFFC",
            [":reversed_hand_with_middle_finger_extended_tone3:"] = "\uD83D\uDD95\uD83C\uDFFD",
            [":reversed_hand_with_middle_finger_extended_tone4:"] = "\uD83D\uDD95\uD83C\uDFFE",
            [":reversed_hand_with_middle_finger_extended_tone5:"] = "\uD83D\uDD95\uD83C\uDFFF",
            [":revolving_hearts:"] = "\uD83D\uDC9E",
            [":rewind:"] = "⏪",
            [":rhino:"] = "\uD83E\uDD8F",
            [":rhinoceros:"] = "\uD83E\uDD8F",
            [":ribbon:"] = "\uD83C\uDF80",
            [":rice:"] = "\uD83C\uDF5A",
            [":rice_ball:"] = "\uD83C\uDF59",
            [":rice_cracker:"] = "\uD83C\uDF58",
            [":rice_scene:"] = "\uD83C\uDF91",
            [":right_anger_bubble:"] = "\uD83D\uDDEF️",
            [":right_facing_fist:"] = "\uD83E\uDD1C",
            [":right_facing_fist::skin-tone-1:"] = "\uD83E\uDD1C\uD83C\uDFFB",
            [":right_facing_fist::skin-tone-2:"] = "\uD83E\uDD1C\uD83C\uDFFC",
            [":right_facing_fist::skin-tone-3:"] = "\uD83E\uDD1C\uD83C\uDFFD",
            [":right_facing_fist::skin-tone-4:"] = "\uD83E\uDD1C\uD83C\uDFFE",
            [":right_facing_fist::skin-tone-5:"] = "\uD83E\uDD1C\uD83C\uDFFF",
            [":right_facing_fist_tone1:"] = "\uD83E\uDD1C\uD83C\uDFFB",
            [":right_facing_fist_tone2:"] = "\uD83E\uDD1C\uD83C\uDFFC",
            [":right_facing_fist_tone3:"] = "\uD83E\uDD1C\uD83C\uDFFD",
            [":right_facing_fist_tone4:"] = "\uD83E\uDD1C\uD83C\uDFFE",
            [":right_facing_fist_tone5:"] = "\uD83E\uDD1C\uD83C\uDFFF",
            [":right_fist:"] = "\uD83E\uDD1C",
            [":right_fist::skin-tone-1:"] = "\uD83E\uDD1C\uD83C\uDFFB",
            [":right_fist::skin-tone-2:"] = "\uD83E\uDD1C\uD83C\uDFFC",
            [":right_fist::skin-tone-3:"] = "\uD83E\uDD1C\uD83C\uDFFD",
            [":right_fist::skin-tone-4:"] = "\uD83E\uDD1C\uD83C\uDFFE",
            [":right_fist::skin-tone-5:"] = "\uD83E\uDD1C\uD83C\uDFFF",
            [":right_fist_tone1:"] = "\uD83E\uDD1C\uD83C\uDFFB",
            [":right_fist_tone2:"] = "\uD83E\uDD1C\uD83C\uDFFC",
            [":right_fist_tone3:"] = "\uD83E\uDD1C\uD83C\uDFFD",
            [":right_fist_tone4:"] = "\uD83E\uDD1C\uD83C\uDFFE",
            [":right_fist_tone5:"] = "\uD83E\uDD1C\uD83C\uDFFF",
            [":rightwards_hand:"] = "\uD83E\uDEF1",
            [":rightwards_hand::skin-tone-1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand::skin-tone-2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand::skin-tone-3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand::skin-tone-4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand::skin-tone-5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_medium_dark_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_medium_light_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_medium_skin_tone:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_tone1:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_tone2:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_tone3:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_tone4:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":rightwards_hand_tone5:"] = "\uD83E\uDEF1\uD83C\uDFFB",
            [":ring:"] = "\uD83D\uDC8D",
            [":ring_buoy:"] = "\uD83D\uDEDF",
            [":ringed_planet:"] = "\uD83E\uDE90",
            [":robot:"] = "\uD83E\uDD16",
            [":robot_face:"] = "\uD83E\uDD16",
            [":rock:"] = "\uD83E\uDEA8",
            [":rocket:"] = "\uD83D\uDE80",
            [":rofl:"] = "\uD83E\uDD23",
            [":roll_of_paper:"] = "\uD83E\uDDFB",
            [":rolled_up_newspaper:"] = "\uD83D\uDDDE️",
            [":roller_coaster:"] = "\uD83C\uDFA2",
            [":roller_skate:"] = "\uD83D\uDEFC",
            [":rolling_eyes:"] = "\uD83D\uDE44",
            [":rolling_on_the_floor_laughing:"] = "\uD83E\uDD23",
            [":rooster:"] = "\uD83D\uDC13",
            [":rose:"] = "\uD83C\uDF39",
            [":rosette:"] = "\uD83C\uDFF5️",
            [":rotating_light:"] = "\uD83D\uDEA8",
            [":round_pushpin:"] = "\uD83D\uDCCD",
            [":rowboat:"] = "\uD83D\uDEA3",
            [":rowboat::skin-tone-1:"] = "\uD83D\uDEA3\uD83C\uDFFB",
            [":rowboat::skin-tone-2:"] = "\uD83D\uDEA3\uD83C\uDFFC",
            [":rowboat::skin-tone-3:"] = "\uD83D\uDEA3\uD83C\uDFFD",
            [":rowboat::skin-tone-4:"] = "\uD83D\uDEA3\uD83C\uDFFE",
            [":rowboat::skin-tone-5:"] = "\uD83D\uDEA3\uD83C\uDFFF",
            [":rowboat_tone1:"] = "\uD83D\uDEA3\uD83C\uDFFB",
            [":rowboat_tone2:"] = "\uD83D\uDEA3\uD83C\uDFFC",
            [":rowboat_tone3:"] = "\uD83D\uDEA3\uD83C\uDFFD",
            [":rowboat_tone4:"] = "\uD83D\uDEA3\uD83C\uDFFE",
            [":rowboat_tone5:"] = "\uD83D\uDEA3\uD83C\uDFFF",
            [":rugby_football:"] = "\uD83C\uDFC9",
            [":runner:"] = "\uD83C\uDFC3",
            [":runner::skin-tone-1:"] = "\uD83C\uDFC3\uD83C\uDFFB",
            [":runner::skin-tone-2:"] = "\uD83C\uDFC3\uD83C\uDFFC",
            [":runner::skin-tone-3:"] = "\uD83C\uDFC3\uD83C\uDFFD",
            [":runner::skin-tone-4:"] = "\uD83C\uDFC3\uD83C\uDFFE",
            [":runner::skin-tone-5:"] = "\uD83C\uDFC3\uD83C\uDFFF",
            [":runner_tone1:"] = "\uD83C\uDFC3\uD83C\uDFFB",
            [":runner_tone2:"] = "\uD83C\uDFC3\uD83C\uDFFC",
            [":runner_tone3:"] = "\uD83C\uDFC3\uD83C\uDFFD",
            [":runner_tone4:"] = "\uD83C\uDFC3\uD83C\uDFFE",
            [":runner_tone5:"] = "\uD83C\uDFC3\uD83C\uDFFF",
            [":running_shirt_with_sash:"] = "\uD83C\uDFBD",
            [":s"] = "\uD83D\uDE12",
            [":sa:"] = "\uD83C\uDE02️",
            [":safety_pin:"] = "\uD83E\uDDF7",
            [":safety_vest:"] = "\uD83E\uDDBA",
            [":sagittarius:"] = "♐",
            [":sailboat:"] = "⛵",
            [":sake:"] = "\uD83C\uDF76",
            [":salad:"] = "\uD83E\uDD57",
            [":salt:"] = "\uD83E\uDDC2",
            [":saluting_face:"] = "\uD83E\uDEE1",
            [":sandal:"] = "\uD83D\uDC61",
            [":sandwich:"] = "\uD83E\uDD6A",
            [":santa:"] = "\uD83C\uDF85",
            [":santa::skin-tone-1:"] = "\uD83C\uDF85\uD83C\uDFFB",
            [":santa::skin-tone-2:"] = "\uD83C\uDF85\uD83C\uDFFC",
            [":santa::skin-tone-3:"] = "\uD83C\uDF85\uD83C\uDFFD",
            [":santa::skin-tone-4:"] = "\uD83C\uDF85\uD83C\uDFFE",
            [":santa::skin-tone-5:"] = "\uD83C\uDF85\uD83C\uDFFF",
            [":santa_tone1:"] = "\uD83C\uDF85\uD83C\uDFFB",
            [":santa_tone2:"] = "\uD83C\uDF85\uD83C\uDFFC",
            [":santa_tone3:"] = "\uD83C\uDF85\uD83C\uDFFD",
            [":santa_tone4:"] = "\uD83C\uDF85\uD83C\uDFFE",
            [":santa_tone5:"] = "\uD83C\uDF85\uD83C\uDFFF",
            [":sari:"] = "\uD83E\uDD7B",
            [":satellite:"] = "\uD83D\uDCE1",
            [":satellite_orbital:"] = "\uD83D\uDEF0️",
            [":satisfied:"] = "\uD83D\uDE06",
            [":sauropod:"] = "\uD83E\uDD95",
            [":saxophone:"] = "\uD83C\uDFB7",
            [":scales:"] = "⚖️",
            [":scarf:"] = "\uD83E\uDDE3",
            [":school:"] = "\uD83C\uDFEB",
            [":school_satchel:"] = "\uD83C\uDF92",
            [":scissors:"] = "✂️",
            [":scooter:"] = "\uD83D\uDEF4",
            [":scorpion:"] = "\uD83E\uDD82",
            [":scorpius:"] = "♏",
            [":scotland:"] = "\uD83C\uDFF4\uDB40\uDC67\uDB40\uDC62\uDB40\uDC73\uDB40\uDC63\uDB40\uDC74\uDB40\uDC7F",
            [":scream:"] = "\uD83D\uDE31",
            [":scream_cat:"] = "\uD83D\uDE40",
            [":screwdriver:"] = "\uD83E\uDE9B",
            [":scroll:"] = "\uD83D\uDCDC",
            [":seal:"] = "\uD83E\uDDAD",
            [":seat:"] = "\uD83D\uDCBA",
            [":second_place:"] = "\uD83E\uDD48",
            [":second_place_medal:"] = "\uD83E\uDD48",
            [":secret:"] = "㊙️",
            [":see_no_evil:"] = "\uD83D\uDE48",
            [":seedling:"] = "\uD83C\uDF31",
            [":selfie:"] = "\uD83E\uDD33",
            [":selfie::skin-tone-1:"] = "\uD83E\uDD33\uD83C\uDFFB",
            [":selfie::skin-tone-2:"] = "\uD83E\uDD33\uD83C\uDFFC",
            [":selfie::skin-tone-3:"] = "\uD83E\uDD33\uD83C\uDFFD",
            [":selfie::skin-tone-4:"] = "\uD83E\uDD33\uD83C\uDFFE",
            [":selfie::skin-tone-5:"] = "\uD83E\uDD33\uD83C\uDFFF",
            [":selfie_tone1:"] = "\uD83E\uDD33\uD83C\uDFFB",
            [":selfie_tone2:"] = "\uD83E\uDD33\uD83C\uDFFC",
            [":selfie_tone3:"] = "\uD83E\uDD33\uD83C\uDFFD",
            [":selfie_tone4:"] = "\uD83E\uDD33\uD83C\uDFFE",
            [":selfie_tone5:"] = "\uD83E\uDD33\uD83C\uDFFF",
            [":service_dog:"] = "\uD83D\uDC15\u200D\uD83E\uDDBA",
            [":seven:"] = "7️⃣",
            [":sewing_needle:"] = "\uD83E\uDEA1",
            [":shaking_hands:"] = "\uD83E\uDD1D",
            [":shallow_pan_of_food:"] = "\uD83E\uDD58",
            [":shamrock:"] = "☘️",
            [":shark:"] = "\uD83E\uDD88",
            [":shaved_ice:"] = "\uD83C\uDF67",
            [":sheep:"] = "\uD83D\uDC11",
            [":shell:"] = "\uD83D\uDC1A",
            [":shelled_peanut:"] = "\uD83E\uDD5C",
            [":shield:"] = "\uD83D\uDEE1️",
            [":shinto_shrine:"] = "⛩️",
            [":ship:"] = "\uD83D\uDEA2",
            [":shirt:"] = "\uD83D\uDC55",
            [":shit:"] = "\uD83D\uDCA9",
            [":shopping_bags:"] = "\uD83D\uDECD️",
            [":shopping_cart:"] = "\uD83D\uDED2",
            [":shopping_trolley:"] = "\uD83D\uDED2",
            [":shorts:"] = "\uD83E\uDE73",
            [":shower:"] = "\uD83D\uDEBF",
            [":shrimp:"] = "\uD83E\uDD90",
            [":shrug:"] = "\uD83E\uDD37",
            [":shrug::skin-tone-1:"] = "\uD83E\uDD37\uD83C\uDFFB",
            [":shrug::skin-tone-2:"] = "\uD83E\uDD37\uD83C\uDFFC",
            [":shrug::skin-tone-3:"] = "\uD83E\uDD37\uD83C\uDFFD",
            [":shrug::skin-tone-4:"] = "\uD83E\uDD37\uD83C\uDFFE",
            [":shrug::skin-tone-5:"] = "\uD83E\uDD37\uD83C\uDFFF",
            [":shrug_tone1:"] = "\uD83E\uDD37\uD83C\uDFFB",
            [":shrug_tone2:"] = "\uD83E\uDD37\uD83C\uDFFC",
            [":shrug_tone3:"] = "\uD83E\uDD37\uD83C\uDFFD",
            [":shrug_tone4:"] = "\uD83E\uDD37\uD83C\uDFFE",
            [":shrug_tone5:"] = "\uD83E\uDD37\uD83C\uDFFF",
            [":shushing_face:"] = "\uD83E\uDD2B",
            [":sick:"] = "\uD83E\uDD22",
            [":sign_of_the_horns:"] = "\uD83E\uDD18",
            [":sign_of_the_horns::skin-tone-1:"] = "\uD83E\uDD18\uD83C\uDFFB",
            [":sign_of_the_horns::skin-tone-2:"] = "\uD83E\uDD18\uD83C\uDFFC",
            [":sign_of_the_horns::skin-tone-3:"] = "\uD83E\uDD18\uD83C\uDFFD",
            [":sign_of_the_horns::skin-tone-4:"] = "\uD83E\uDD18\uD83C\uDFFE",
            [":sign_of_the_horns::skin-tone-5:"] = "\uD83E\uDD18\uD83C\uDFFF",
            [":sign_of_the_horns_tone1:"] = "\uD83E\uDD18\uD83C\uDFFB",
            [":sign_of_the_horns_tone2:"] = "\uD83E\uDD18\uD83C\uDFFC",
            [":sign_of_the_horns_tone3:"] = "\uD83E\uDD18\uD83C\uDFFD",
            [":sign_of_the_horns_tone4:"] = "\uD83E\uDD18\uD83C\uDFFE",
            [":sign_of_the_horns_tone5:"] = "\uD83E\uDD18\uD83C\uDFFF",
            [":signal_strength:"] = "\uD83D\uDCF6",
            [":six:"] = "6️⃣",
            [":six_pointed_star:"] = "\uD83D\uDD2F",
            [":skateboard:"] = "\uD83D\uDEF9",
            [":skeleton:"] = "\uD83D\uDC80",
            [":ski:"] = "\uD83C\uDFBF",
            [":skier:"] = "⛷️",
            [":skull:"] = "\uD83D\uDC80",
            [":skull_and_crossbones:"] = "☠️",
            [":skull_crossbones:"] = "☠️",
            [":skunk:"] = "\uD83E\uDDA8",
            [":sled:"] = "\uD83D\uDEF7",
            [":sleeping:"] = "\uD83D\uDE34",
            [":sleeping_accommodation:"] = "\uD83D\uDECC",
            [":sleeping_accommodation::skin-tone-1:"] = "\uD83D\uDECC\uD83C\uDFFB",
            [":sleeping_accommodation::skin-tone-2:"] = "\uD83D\uDECC\uD83C\uDFFC",
            [":sleeping_accommodation::skin-tone-3:"] = "\uD83D\uDECC\uD83C\uDFFD",
            [":sleeping_accommodation::skin-tone-4:"] = "\uD83D\uDECC\uD83C\uDFFE",
            [":sleeping_accommodation::skin-tone-5:"] = "\uD83D\uDECC\uD83C\uDFFF",
            [":sleepy:"] = "\uD83D\uDE2A",
            [":sleuth_or_spy:"] = "\uD83D\uDD75️",
            [":sleuth_or_spy::skin-tone-1:"] = "\uD83D\uDD75\uD83C\uDFFB",
            [":sleuth_or_spy::skin-tone-2:"] = "\uD83D\uDD75\uD83C\uDFFC",
            [":sleuth_or_spy::skin-tone-3:"] = "\uD83D\uDD75\uD83C\uDFFD",
            [":sleuth_or_spy::skin-tone-4:"] = "\uD83D\uDD75\uD83C\uDFFE",
            [":sleuth_or_spy::skin-tone-5:"] = "\uD83D\uDD75\uD83C\uDFFF",
            [":sleuth_or_spy_tone1:"] = "\uD83D\uDD75\uD83C\uDFFB",
            [":sleuth_or_spy_tone2:"] = "\uD83D\uDD75\uD83C\uDFFC",
            [":sleuth_or_spy_tone3:"] = "\uD83D\uDD75\uD83C\uDFFD",
            [":sleuth_or_spy_tone4:"] = "\uD83D\uDD75\uD83C\uDFFE",
            [":sleuth_or_spy_tone5:"] = "\uD83D\uDD75\uD83C\uDFFF",
            [":slight_frown:"] = "\uD83D\uDE41",
            [":slight_smile:"] = "\uD83D\uDE42",
            [":slightly_frowning_face:"] = "\uD83D\uDE41",
            [":slightly_smiling_face:"] = "\uD83D\uDE42",
            [":slot_machine:"] = "\uD83C\uDFB0",
            [":sloth:"] = "\uD83E\uDDA5",
            [":small_airplane:"] = "\uD83D\uDEE9️",
            [":small_blue_diamond:"] = "\uD83D\uDD39",
            [":small_orange_diamond:"] = "\uD83D\uDD38",
            [":small_red_triangle:"] = "\uD83D\uDD3A",
            [":small_red_triangle_down:"] = "\uD83D\uDD3B",
            [":smile:"] = "\uD83D\uDE04",
            [":smile_cat:"] = "\uD83D\uDE38",
            [":smiley:"] = "\uD83D\uDE03",
            [":smiley_cat:"] = "\uD83D\uDE3A",
            [":smiling_face_with_3_hearts:"] = "\uD83E\uDD70",
            [":smiling_face_with_tear:"] = "\uD83E\uDD72",
            [":smiling_face_with_tear:"] = "\uD83E\uDD72",
            [":smiling_imp:"] = "\uD83D\uDE08",
            [":smirk:"] = "\uD83D\uDE0F",
            [":smirk_cat:"] = "\uD83D\uDE3C",
            [":smoking:"] = "\uD83D\uDEAC",
            [":snail:"] = "\uD83D\uDC0C",
            [":snake:"] = "\uD83D\uDC0D",
            [":sneeze:"] = "\uD83E\uDD27",
            [":sneezing_face:"] = "\uD83E\uDD27",
            [":snow_capped_mountain:"] = "\uD83C\uDFD4️",
            [":snowboarder:"] = "\uD83C\uDFC2",
            [":snowboarder::skin-tone-1:"] = "\uD83C\uDFC2\uD83C\uDFFB",
            [":snowboarder::skin-tone-2:"] = "\uD83C\uDFC2\uD83C\uDFFC",
            [":snowboarder::skin-tone-3:"] = "\uD83C\uDFC2\uD83C\uDFFD",
            [":snowboarder::skin-tone-4:"] = "\uD83C\uDFC2\uD83C\uDFFE",
            [":snowboarder::skin-tone-5:"] = "\uD83C\uDFC2\uD83C\uDFFF",
            [":snowboarder_dark_skin_tone:"] = "\uD83C\uDFC2\uD83C\uDFFF",
            [":snowboarder_light_skin_tone:"] = "\uD83C\uDFC2\uD83C\uDFFB",
            [":snowboarder_medium_dark_skin_tone:"] = "\uD83C\uDFC2\uD83C\uDFFE",
            [":snowboarder_medium_light_skin_tone:"] = "\uD83C\uDFC2\uD83C\uDFFC",
            [":snowboarder_medium_skin_tone:"] = "\uD83C\uDFC2\uD83C\uDFFD",
            [":snowboarder_tone1:"] = "\uD83C\uDFC2\uD83C\uDFFB",
            [":snowboarder_tone2:"] = "\uD83C\uDFC2\uD83C\uDFFC",
            [":snowboarder_tone3:"] = "\uD83C\uDFC2\uD83C\uDFFD",
            [":snowboarder_tone4:"] = "\uD83C\uDFC2\uD83C\uDFFE",
            [":snowboarder_tone5:"] = "\uD83C\uDFC2\uD83C\uDFFF",
            [":snowflake:"] = "❄️",
            [":snowman2:"] = "☃️",
            [":snowman:"] = "⛄",
            [":soap:"] = "\uD83E\uDDFC",
            [":sob:"] = "\uD83D\uDE2D",
            [":soccer:"] = "⚽",
            [":socks:"] = "\uD83E\uDDE6",
            [":softball:"] = "\uD83E\uDD4E",
            [":soon:"] = "\uD83D\uDD1C",
            [":sos:"] = "\uD83C\uDD98",
            [":sound:"] = "\uD83D\uDD09",
            [":space_invader:"] = "\uD83D\uDC7E",
            [":spades:"] = "♠️",
            [":spaghetti:"] = "\uD83C\uDF5D",
            [":sparkle:"] = "❇️",
            [":sparkler:"] = "\uD83C\uDF87",
            [":sparkles:"] = "✨",
            [":sparkling_heart:"] = "\uD83D\uDC96",
            [":speak_no_evil:"] = "\uD83D\uDE4A",
            [":speaker:"] = "\uD83D\uDD08",
            [":speaking_head:"] = "\uD83D\uDDE3️",
            [":speaking_head_in_silhouette:"] = "\uD83D\uDDE3️",
            [":speech_balloon:"] = "\uD83D\uDCAC",
            [":speech_left:"] = "\uD83D\uDDE8️",
            [":speedboat:"] = "\uD83D\uDEA4",
            [":spider:"] = "\uD83D\uDD77️",
            [":spider_web:"] = "\uD83D\uDD78️",
            [":spiral_calendar_pad:"] = "\uD83D\uDDD3️",
            [":spiral_note_pad:"] = "\uD83D\uDDD2️",
            [":sponge:"] = "\uD83E\uDDFD",
            [":spoon:"] = "\uD83E\uDD44",
            [":sports_medal:"] = "\uD83C\uDFC5",
            [":spy:"] = "\uD83D\uDD75️",
            [":spy::skin-tone-1:"] = "\uD83D\uDD75\uD83C\uDFFB",
            [":spy::skin-tone-2:"] = "\uD83D\uDD75\uD83C\uDFFC",
            [":spy::skin-tone-3:"] = "\uD83D\uDD75\uD83C\uDFFD",
            [":spy::skin-tone-4:"] = "\uD83D\uDD75\uD83C\uDFFE",
            [":spy::skin-tone-5:"] = "\uD83D\uDD75\uD83C\uDFFF",
            [":spy_tone1:"] = "\uD83D\uDD75\uD83C\uDFFB",
            [":spy_tone2:"] = "\uD83D\uDD75\uD83C\uDFFC",
            [":spy_tone3:"] = "\uD83D\uDD75\uD83C\uDFFD",
            [":spy_tone4:"] = "\uD83D\uDD75\uD83C\uDFFE",
            [":spy_tone5:"] = "\uD83D\uDD75\uD83C\uDFFF",
            [":squeeze_bottle:"] = "\uD83E\uDDF4",
            [":squid:"] = "\uD83E\uDD91",
            [":stadium:"] = "\uD83C\uDFDF️",
            [":star2:"] = "\uD83C\uDF1F",
            [":star:"] = "⭐",
            [":star_and_crescent:"] = "☪️",
            [":star_of_david:"] = "✡️",
            [":star_struck:"] = "\uD83E\uDD29",
            [":stars:"] = "\uD83C\uDF20",
            [":station:"] = "\uD83D\uDE89",
            [":statue_of_liberty:"] = "\uD83D\uDDFD",
            [":steam_locomotive:"] = "\uD83D\uDE82",
            [":stethoscope:"] = "\uD83E\uDE7A",
            [":stew:"] = "\uD83C\uDF72",
            [":stop_button:"] = "⏹️",
            [":stop_sign:"] = "\uD83D\uDED1",
            [":stopwatch:"] = "⏱️",
            [":straight_ruler:"] = "\uD83D\uDCCF",
            [":strawberry:"] = "\uD83C\uDF53",
            [":stuck_out_tongue:"] = "\uD83D\uDE1B",
            [":stuck_out_tongue_closed_eyes:"] = "\uD83D\uDE1D",
            [":stuck_out_tongue_winking_eye:"] = "\uD83D\uDE1C",
            [":studio_microphone:"] = "\uD83C\uDF99️",
            [":stuffed_flatbread:"] = "\uD83E\uDD59",
            [":stuffed_pita:"] = "\uD83E\uDD59",
            [":sun_with_face:"] = "\uD83C\uDF1E",
            [":sunflower:"] = "\uD83C\uDF3B",
            [":sunglasses:"] = "\uD83D\uDE0E",
            [":sunny:"] = "☀️",
            [":sunrise:"] = "\uD83C\uDF05",
            [":sunrise_over_mountains:"] = "\uD83C\uDF04",
            [":superhero:"] = "\uD83E\uDDB8",
            [":superhero::skin-tone-1:"] = "\uD83E\uDDB8\uD83C\uDFFB",
            [":superhero::skin-tone-2:"] = "\uD83E\uDDB8\uD83C\uDFFC",
            [":superhero::skin-tone-3:"] = "\uD83E\uDDB8\uD83C\uDFFD",
            [":superhero::skin-tone-4:"] = "\uD83E\uDDB8\uD83C\uDFFE",
            [":superhero::skin-tone-5:"] = "\uD83E\uDDB8\uD83C\uDFFF",
            [":superhero_dark_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFF",
            [":superhero_light_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFB",
            [":superhero_medium_dark_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFE",
            [":superhero_medium_light_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFC",
            [":superhero_medium_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFD",
            [":superhero_tone1:"] = "\uD83E\uDDB8\uD83C\uDFFB",
            [":superhero_tone2:"] = "\uD83E\uDDB8\uD83C\uDFFC",
            [":superhero_tone3:"] = "\uD83E\uDDB8\uD83C\uDFFD",
            [":superhero_tone4:"] = "\uD83E\uDDB8\uD83C\uDFFE",
            [":superhero_tone5:"] = "\uD83E\uDDB8\uD83C\uDFFF",
            [":supervillain:"] = "\uD83E\uDDB9",
            [":supervillain::skin-tone-1:"] = "\uD83E\uDDB9\uD83C\uDFFB",
            [":supervillain::skin-tone-2:"] = "\uD83E\uDDB9\uD83C\uDFFC",
            [":supervillain::skin-tone-3:"] = "\uD83E\uDDB9\uD83C\uDFFD",
            [":supervillain::skin-tone-4:"] = "\uD83E\uDDB9\uD83C\uDFFE",
            [":supervillain::skin-tone-5:"] = "\uD83E\uDDB9\uD83C\uDFFF",
            [":supervillain_dark_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFF",
            [":supervillain_light_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFB",
            [":supervillain_medium_dark_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFE",
            [":supervillain_medium_light_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFC",
            [":supervillain_medium_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFD",
            [":supervillain_tone1:"] = "\uD83E\uDDB9\uD83C\uDFFB",
            [":supervillain_tone2:"] = "\uD83E\uDDB9\uD83C\uDFFC",
            [":supervillain_tone3:"] = "\uD83E\uDDB9\uD83C\uDFFD",
            [":supervillain_tone4:"] = "\uD83E\uDDB9\uD83C\uDFFE",
            [":supervillain_tone5:"] = "\uD83E\uDDB9\uD83C\uDFFF",
            [":surfer:"] = "\uD83C\uDFC4",
            [":surfer::skin-tone-1:"] = "\uD83C\uDFC4\uD83C\uDFFB",
            [":surfer::skin-tone-2:"] = "\uD83C\uDFC4\uD83C\uDFFC",
            [":surfer::skin-tone-3:"] = "\uD83C\uDFC4\uD83C\uDFFD",
            [":surfer::skin-tone-4:"] = "\uD83C\uDFC4\uD83C\uDFFE",
            [":surfer::skin-tone-5:"] = "\uD83C\uDFC4\uD83C\uDFFF",
            [":surfer_tone1:"] = "\uD83C\uDFC4\uD83C\uDFFB",
            [":surfer_tone2:"] = "\uD83C\uDFC4\uD83C\uDFFC",
            [":surfer_tone3:"] = "\uD83C\uDFC4\uD83C\uDFFD",
            [":surfer_tone4:"] = "\uD83C\uDFC4\uD83C\uDFFE",
            [":surfer_tone5:"] = "\uD83C\uDFC4\uD83C\uDFFF",
            [":sushi:"] = "\uD83C\uDF63",
            [":suspension_railway:"] = "\uD83D\uDE9F",
            [":swan:"] = "\uD83E\uDDA2",
            [":sweat:"] = "\uD83D\uDE13",
            [":sweat_drops:"] = "\uD83D\uDCA6",
            [":sweat_smile:"] = "\uD83D\uDE05",
            [":sweet_potato:"] = "\uD83C\uDF60",
            [":swimmer:"] = "\uD83C\uDFCA",
            [":swimmer::skin-tone-1:"] = "\uD83C\uDFCA\uD83C\uDFFB",
            [":swimmer::skin-tone-2:"] = "\uD83C\uDFCA\uD83C\uDFFC",
            [":swimmer::skin-tone-3:"] = "\uD83C\uDFCA\uD83C\uDFFD",
            [":swimmer::skin-tone-4:"] = "\uD83C\uDFCA\uD83C\uDFFE",
            [":swimmer::skin-tone-5:"] = "\uD83C\uDFCA\uD83C\uDFFF",
            [":swimmer_tone1:"] = "\uD83C\uDFCA\uD83C\uDFFB",
            [":swimmer_tone2:"] = "\uD83C\uDFCA\uD83C\uDFFC",
            [":swimmer_tone3:"] = "\uD83C\uDFCA\uD83C\uDFFD",
            [":swimmer_tone4:"] = "\uD83C\uDFCA\uD83C\uDFFE",
            [":swimmer_tone5:"] = "\uD83C\uDFCA\uD83C\uDFFF",
            [":symbols:"] = "\uD83D\uDD23",
            [":synagogue:"] = "\uD83D\uDD4D",
            [":syringe:"] = "\uD83D\uDC89",
            [":t_rex:"] = "\uD83E\uDD96",
            [":table_tennis:"] = "\uD83C\uDFD3",
            [":taco:"] = "\uD83C\uDF2E",
            [":tada:"] = "\uD83C\uDF89",
            [":takeout_box:"] = "\uD83E\uDD61",
            [":tamale:"] = "\uD83E\uDED4",
            [":tanabata_tree:"] = "\uD83C\uDF8B",
            [":tangerine:"] = "\uD83C\uDF4A",
            [":taurus:"] = "♉",
            [":taxi:"] = "\uD83D\uDE95",
            [":tea:"] = "\uD83C\uDF75",
            [":teapot:"] = "\uD83E\uDED6",
            [":teddy_bear:"] = "\uD83E\uDDF8",
            [":telephone:"] = "☎️",
            [":telephone_receiver:"] = "\uD83D\uDCDE",
            [":telescope:"] = "\uD83D\uDD2D",
            [":tennis:"] = "\uD83C\uDFBE",
            [":tent:"] = "⛺",
            [":test_tube:"] = "\uD83E\uDDEA",
            [":thermometer:"] = "\uD83C\uDF21️",
            [":thermometer_face:"] = "\uD83E\uDD12",
            [":thinking:"] = "\uD83E\uDD14",
            [":thinking_face:"] = "\uD83E\uDD14",
            [":third_place:"] = "\uD83E\uDD49",
            [":third_place_medal:"] = "\uD83E\uDD49",
            [":thong_sandal:"] = "\uD83E\uDE74",
            [":thought_balloon:"] = "\uD83D\uDCAD",
            [":thread:"] = "\uD83E\uDDF5",
            [":three:"] = "3️⃣",
            [":three_button_mouse:"] = "\uD83D\uDDB1️",
            [":thumbdown:"] = "\uD83D\uDC4E",
            [":thumbdown::skin-tone-1:"] = "\uD83D\uDC4E\uD83C\uDFFB",
            [":thumbdown::skin-tone-2:"] = "\uD83D\uDC4E\uD83C\uDFFC",
            [":thumbdown::skin-tone-3:"] = "\uD83D\uDC4E\uD83C\uDFFD",
            [":thumbdown::skin-tone-4:"] = "\uD83D\uDC4E\uD83C\uDFFE",
            [":thumbdown::skin-tone-5:"] = "\uD83D\uDC4E\uD83C\uDFFF",
            [":thumbdown_tone1:"] = "\uD83D\uDC4E\uD83C\uDFFB",
            [":thumbdown_tone2:"] = "\uD83D\uDC4E\uD83C\uDFFC",
            [":thumbdown_tone3:"] = "\uD83D\uDC4E\uD83C\uDFFD",
            [":thumbdown_tone4:"] = "\uD83D\uDC4E\uD83C\uDFFE",
            [":thumbdown_tone5:"] = "\uD83D\uDC4E\uD83C\uDFFF",
            [":thumbsdown:"] = "\uD83D\uDC4E",
            [":thumbsdown::skin-tone-1:"] = "\uD83D\uDC4E\uD83C\uDFFB",
            [":thumbsdown::skin-tone-2:"] = "\uD83D\uDC4E\uD83C\uDFFC",
            [":thumbsdown::skin-tone-3:"] = "\uD83D\uDC4E\uD83C\uDFFD",
            [":thumbsdown::skin-tone-4:"] = "\uD83D\uDC4E\uD83C\uDFFE",
            [":thumbsdown::skin-tone-5:"] = "\uD83D\uDC4E\uD83C\uDFFF",
            [":thumbsdown_tone1:"] = "\uD83D\uDC4E\uD83C\uDFFB",
            [":thumbsdown_tone2:"] = "\uD83D\uDC4E\uD83C\uDFFC",
            [":thumbsdown_tone3:"] = "\uD83D\uDC4E\uD83C\uDFFD",
            [":thumbsdown_tone4:"] = "\uD83D\uDC4E\uD83C\uDFFE",
            [":thumbsdown_tone5:"] = "\uD83D\uDC4E\uD83C\uDFFF",
            [":thumbsup:"] = "\uD83D\uDC4D",
            [":thumbsup::skin-tone-1:"] = "\uD83D\uDC4D\uD83C\uDFFB",
            [":thumbsup::skin-tone-2:"] = "\uD83D\uDC4D\uD83C\uDFFC",
            [":thumbsup::skin-tone-3:"] = "\uD83D\uDC4D\uD83C\uDFFD",
            [":thumbsup::skin-tone-4:"] = "\uD83D\uDC4D\uD83C\uDFFE",
            [":thumbsup::skin-tone-5:"] = "\uD83D\uDC4D\uD83C\uDFFF",
            [":thumbsup_tone1:"] = "\uD83D\uDC4D\uD83C\uDFFB",
            [":thumbsup_tone2:"] = "\uD83D\uDC4D\uD83C\uDFFC",
            [":thumbsup_tone3:"] = "\uD83D\uDC4D\uD83C\uDFFD",
            [":thumbsup_tone4:"] = "\uD83D\uDC4D\uD83C\uDFFE",
            [":thumbsup_tone5:"] = "\uD83D\uDC4D\uD83C\uDFFF",
            [":thumbup:"] = "\uD83D\uDC4D",
            [":thumbup::skin-tone-1:"] = "\uD83D\uDC4D\uD83C\uDFFB",
            [":thumbup::skin-tone-2:"] = "\uD83D\uDC4D\uD83C\uDFFC",
            [":thumbup::skin-tone-3:"] = "\uD83D\uDC4D\uD83C\uDFFD",
            [":thumbup::skin-tone-4:"] = "\uD83D\uDC4D\uD83C\uDFFE",
            [":thumbup::skin-tone-5:"] = "\uD83D\uDC4D\uD83C\uDFFF",
            [":thumbup_tone1:"] = "\uD83D\uDC4D\uD83C\uDFFB",
            [":thumbup_tone2:"] = "\uD83D\uDC4D\uD83C\uDFFC",
            [":thumbup_tone3:"] = "\uD83D\uDC4D\uD83C\uDFFD",
            [":thumbup_tone4:"] = "\uD83D\uDC4D\uD83C\uDFFE",
            [":thumbup_tone5:"] = "\uD83D\uDC4D\uD83C\uDFFF",
            [":thunder_cloud_and_rain:"] = "⛈️",
            [":thunder_cloud_rain:"] = "⛈️",
            [":ticket:"] = "\uD83C\uDFAB",
            [":tickets:"] = "\uD83C\uDF9F️",
            [":tiger2:"] = "\uD83D\uDC05",
            [":tiger:"] = "\uD83D\uDC2F",
            [":timer:"] = "⏲️",
            [":timer_clock:"] = "⏲️",
            [":tired_face:"] = "\uD83D\uDE2B",
            [":tm:"] = "™️",
            [":toilet:"] = "\uD83D\uDEBD",
            [":tokyo_tower:"] = "\uD83D\uDDFC",
            [":tomato:"] = "\uD83C\uDF45",
            [":tongue:"] = "\uD83D\uDC45",
            [":toolbox:"] = "\uD83E\uDDF0",
            [":tools:"] = "\uD83D\uDEE0️",
            [":tooth:"] = "\uD83E\uDDB7",
            [":toothbrush:"] = "\uD83E\uDEA5",
            [":top:"] = "\uD83D\uDD1D",
            [":tophat:"] = "\uD83C\uDFA9",
            [":track_next:"] = "⏭️",
            [":track_previous:"] = "⏮️",
            [":trackball:"] = "\uD83D\uDDB2️",
            [":tractor:"] = "\uD83D\uDE9C",
            [":traffic_light:"] = "\uD83D\uDEA5",
            [":train2:"] = "\uD83D\uDE86",
            [":train:"] = "\uD83D\uDE8B",
            [":tram:"] = "\uD83D\uDE8A",
            [":transgender_flag:"] = "\uD83C\uDFF3\uFE0F\u200D\u26A7\uFE0F",
            [":transgender_symbol:"] = "\u26A7\uFE0F",
            [":triangular_flag_on_post:"] = "\uD83D\uDEA9",
            [":triangular_ruler:"] = "\uD83D\uDCD0",
            [":trident:"] = "\uD83D\uDD31",
            [":triumph:"] = "\uD83D\uDE24",
            [":troll:"] = "\uD83E\uDDCC",
            [":trolleybus:"] = "\uD83D\uDE8E",
            [":trophy:"] = "\uD83C\uDFC6",
            [":tropical_drink:"] = "\uD83C\uDF79",
            [":tropical_fish:"] = "\uD83D\uDC20",
            [":truck:"] = "\uD83D\uDE9A",
            [":trumpet:"] = "\uD83C\uDFBA",
            [":tulip:"] = "\uD83C\uDF37",
            [":tumbler_glass:"] = "\uD83E\uDD43",
            [":turkey:"] = "\uD83E\uDD83",
            [":turtle:"] = "\uD83D\uDC22",
            [":tuxedo_tone1:"] = "\uD83E\uDD35\uD83C\uDFFB",
            [":tuxedo_tone2:"] = "\uD83E\uDD35\uD83C\uDFFC",
            [":tuxedo_tone3:"] = "\uD83E\uDD35\uD83C\uDFFD",
            [":tuxedo_tone4:"] = "\uD83E\uDD35\uD83C\uDFFE",
            [":tuxedo_tone5:"] = "\uD83E\uDD35\uD83C\uDFFF",
            [":tv:"] = "\uD83D\uDCFA",
            [":twisted_rightwards_arrows:"] = "\uD83D\uDD00",
            [":two:"] = "2️⃣",
            [":two_hearts:"] = "\uD83D\uDC95",
            [":two_men_holding_hands:"] = "\uD83D\uDC6C",
            [":two_women_holding_hands:"] = "\uD83D\uDC6D",
            [":u5272:"] = "\uD83C\uDE39",
            [":u5408:"] = "\uD83C\uDE34",
            [":u55b6:"] = "\uD83C\uDE3A",
            [":u6307:"] = "\uD83C\uDE2F",
            [":u6708:"] = "\uD83C\uDE37️",
            [":u6709:"] = "\uD83C\uDE36",
            [":u6e80:"] = "\uD83C\uDE35",
            [":u7121:"] = "\uD83C\uDE1A",
            [":u7533:"] = "\uD83C\uDE38",
            [":u7981:"] = "\uD83C\uDE32",
            [":u7a7a:"] = "\uD83C\uDE33",
            [":umbrella2:"] = "☂️",
            [":umbrella:"] = "☔",
            [":umbrella_on_ground:"] = "⛱️",
            [":unamused:"] = "\uD83D\uDE12",
            [":underage:"] = "\uD83D\uDD1E",
            [":unicorn:"] = "\uD83E\uDD84",
            [":unicorn_face:"] = "\uD83E\uDD84",
            [":united_nations:"] = "\uD83C\uDDFA\uD83C\uDDF3",
            [":unlock:"] = "\uD83D\uDD13",
            [":up:"] = "\uD83C\uDD99",
            [":upside_down:"] = "\uD83D\uDE43",
            [":upside_down_face:"] = "\uD83D\uDE43",
            [":urn:"] = "⚱️",
            [":v:"] = "✌️",
            [":v::skin-tone-1:"] = "✌\uD83C\uDFFB",
            [":v::skin-tone-2:"] = "✌\uD83C\uDFFC",
            [":v::skin-tone-3:"] = "✌\uD83C\uDFFD",
            [":v::skin-tone-4:"] = "✌\uD83C\uDFFE",
            [":v::skin-tone-5:"] = "✌\uD83C\uDFFF",
            [":v_tone1:"] = "✌\uD83C\uDFFB",
            [":v_tone2:"] = "✌\uD83C\uDFFC",
            [":v_tone3:"] = "✌\uD83C\uDFFD",
            [":v_tone4:"] = "✌\uD83C\uDFFE",
            [":v_tone5:"] = "✌\uD83C\uDFFF",
            [":vampire:"] = "\uD83E\uDDDB",
            [":vampire::skin-tone-1:"] = "\uD83E\uDDDB\uD83C\uDFFB",
            [":vampire::skin-tone-2:"] = "\uD83E\uDDDB\uD83C\uDFFC",
            [":vampire::skin-tone-3:"] = "\uD83E\uDDDB\uD83C\uDFFD",
            [":vampire::skin-tone-4:"] = "\uD83E\uDDDB\uD83C\uDFFE",
            [":vampire::skin-tone-5:"] = "\uD83E\uDDDB\uD83C\uDFFF",
            [":vampire_dark_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFF",
            [":vampire_light_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFB",
            [":vampire_medium_dark_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFE",
            [":vampire_medium_light_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFC",
            [":vampire_medium_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFD",
            [":vampire_tone1:"] = "\uD83E\uDDDB\uD83C\uDFFB",
            [":vampire_tone2:"] = "\uD83E\uDDDB\uD83C\uDFFC",
            [":vampire_tone3:"] = "\uD83E\uDDDB\uD83C\uDFFD",
            [":vampire_tone4:"] = "\uD83E\uDDDB\uD83C\uDFFE",
            [":vampire_tone5:"] = "\uD83E\uDDDB\uD83C\uDFFF",
            [":vertical_traffic_light:"] = "\uD83D\uDEA6",
            [":vhs:"] = "\uD83D\uDCFC",
            [":vibration_mode:"] = "\uD83D\uDCF3",
            [":video_camera:"] = "\uD83D\uDCF9",
            [":video_game:"] = "\uD83C\uDFAE",
            [":violin:"] = "\uD83C\uDFBB",
            [":virgo:"] = "♍",
            [":volcano:"] = "\uD83C\uDF0B",
            [":volleyball:"] = "\uD83C\uDFD0",
            [":vs:"] = "\uD83C\uDD9A",
            [":vulcan:"] = "\uD83D\uDD96",
            [":vulcan::skin-tone-1:"] = "\uD83D\uDD96\uD83C\uDFFB",
            [":vulcan::skin-tone-2:"] = "\uD83D\uDD96\uD83C\uDFFC",
            [":vulcan::skin-tone-3:"] = "\uD83D\uDD96\uD83C\uDFFD",
            [":vulcan::skin-tone-4:"] = "\uD83D\uDD96\uD83C\uDFFE",
            [":vulcan::skin-tone-5:"] = "\uD83D\uDD96\uD83C\uDFFF",
            [":vulcan_tone1:"] = "\uD83D\uDD96\uD83C\uDFFB",
            [":vulcan_tone2:"] = "\uD83D\uDD96\uD83C\uDFFC",
            [":vulcan_tone3:"] = "\uD83D\uDD96\uD83C\uDFFD",
            [":vulcan_tone4:"] = "\uD83D\uDD96\uD83C\uDFFE",
            [":vulcan_tone5:"] = "\uD83D\uDD96\uD83C\uDFFF",
            [":waffle:"] = "\uD83E\uDDC7",
            [":wales:"] = "\uD83C\uDFF4\uDB40\uDC67\uDB40\uDC62\uDB40\uDC77\uDB40\uDC6C\uDB40\uDC73\uDB40\uDC7F",
            [":walking:"] = "\uD83D\uDEB6",
            [":walking::skin-tone-1:"] = "\uD83D\uDEB6\uD83C\uDFFB",
            [":walking::skin-tone-2:"] = "\uD83D\uDEB6\uD83C\uDFFC",
            [":walking::skin-tone-3:"] = "\uD83D\uDEB6\uD83C\uDFFD",
            [":walking::skin-tone-4:"] = "\uD83D\uDEB6\uD83C\uDFFE",
            [":walking::skin-tone-5:"] = "\uD83D\uDEB6\uD83C\uDFFF",
            [":walking_tone1:"] = "\uD83D\uDEB6\uD83C\uDFFB",
            [":walking_tone2:"] = "\uD83D\uDEB6\uD83C\uDFFC",
            [":walking_tone3:"] = "\uD83D\uDEB6\uD83C\uDFFD",
            [":walking_tone4:"] = "\uD83D\uDEB6\uD83C\uDFFE",
            [":walking_tone5:"] = "\uD83D\uDEB6\uD83C\uDFFF",
            [":waning_crescent_moon:"] = "\uD83C\uDF18",
            [":waning_gibbous_moon:"] = "\uD83C\uDF16",
            [":warning:"] = "⚠️",
            [":wastebasket:"] = "\uD83D\uDDD1️",
            [":watch:"] = "⌚",
            [":water_buffalo:"] = "\uD83D\uDC03",
            [":water_polo:"] = "\uD83E\uDD3D",
            [":water_polo::skin-tone-1:"] = "\uD83E\uDD3D\uD83C\uDFFB",
            [":water_polo::skin-tone-2:"] = "\uD83E\uDD3D\uD83C\uDFFC",
            [":water_polo::skin-tone-3:"] = "\uD83E\uDD3D\uD83C\uDFFD",
            [":water_polo::skin-tone-4:"] = "\uD83E\uDD3D\uD83C\uDFFE",
            [":water_polo::skin-tone-5:"] = "\uD83E\uDD3D\uD83C\uDFFF",
            [":water_polo_tone1:"] = "\uD83E\uDD3D\uD83C\uDFFB",
            [":water_polo_tone2:"] = "\uD83E\uDD3D\uD83C\uDFFC",
            [":water_polo_tone3:"] = "\uD83E\uDD3D\uD83C\uDFFD",
            [":water_polo_tone4:"] = "\uD83E\uDD3D\uD83C\uDFFE",
            [":water_polo_tone5:"] = "\uD83E\uDD3D\uD83C\uDFFF",
            [":watermelon:"] = "\uD83C\uDF49",
            [":wave:"] = "\uD83D\uDC4B",
            [":wave::skin-tone-1:"] = "\uD83D\uDC4B\uD83C\uDFFB",
            [":wave::skin-tone-2:"] = "\uD83D\uDC4B\uD83C\uDFFC",
            [":wave::skin-tone-3:"] = "\uD83D\uDC4B\uD83C\uDFFD",
            [":wave::skin-tone-4:"] = "\uD83D\uDC4B\uD83C\uDFFE",
            [":wave::skin-tone-5:"] = "\uD83D\uDC4B\uD83C\uDFFF",
            [":wave_tone1:"] = "\uD83D\uDC4B\uD83C\uDFFB",
            [":wave_tone2:"] = "\uD83D\uDC4B\uD83C\uDFFC",
            [":wave_tone3:"] = "\uD83D\uDC4B\uD83C\uDFFD",
            [":wave_tone4:"] = "\uD83D\uDC4B\uD83C\uDFFE",
            [":wave_tone5:"] = "\uD83D\uDC4B\uD83C\uDFFF",
            [":wavy_dash:"] = "〰️",
            [":waxing_crescent_moon:"] = "\uD83C\uDF12",
            [":waxing_gibbous_moon:"] = "\uD83C\uDF14",
            [":wc:"] = "\uD83D\uDEBE",
            [":weary:"] = "\uD83D\uDE29",
            [":wedding:"] = "\uD83D\uDC92",
            [":weight_lifter:"] = "\uD83C\uDFCB️",
            [":weight_lifter::skin-tone-1:"] = "\uD83C\uDFCB\uD83C\uDFFB",
            [":weight_lifter::skin-tone-2:"] = "\uD83C\uDFCB\uD83C\uDFFC",
            [":weight_lifter::skin-tone-3:"] = "\uD83C\uDFCB\uD83C\uDFFD",
            [":weight_lifter::skin-tone-4:"] = "\uD83C\uDFCB\uD83C\uDFFE",
            [":weight_lifter::skin-tone-5:"] = "\uD83C\uDFCB\uD83C\uDFFF",
            [":weight_lifter_tone1:"] = "\uD83C\uDFCB\uD83C\uDFFB",
            [":weight_lifter_tone2:"] = "\uD83C\uDFCB\uD83C\uDFFC",
            [":weight_lifter_tone3:"] = "\uD83C\uDFCB\uD83C\uDFFD",
            [":weight_lifter_tone4:"] = "\uD83C\uDFCB\uD83C\uDFFE",
            [":weight_lifter_tone5:"] = "\uD83C\uDFCB\uD83C\uDFFF",
            [":whale2:"] = "\uD83D\uDC0B",
            [":whale:"] = "\uD83D\uDC33",
            [":wheel:"] = "\uD83D\uDEDE",
            [":wheel_of_dharma:"] = "☸️",
            [":wheelchair:"] = "♿",
            [":whisky:"] = "\uD83E\uDD43",
            [":white_check_mark:"] = "✅",
            [":white_circle:"] = "⚪",
            [":white_flower:"] = "\uD83D\uDCAE",
            [":white_frowning_face:"] = "☹️",
            [":white_heart:"] = "\uD83E\uDD0D",
            [":white_large_square:"] = "⬜",
            [":white_medium_small_square:"] = "◽",
            [":white_medium_square:"] = "◻️",
            [":white_small_square:"] = "▫️",
            [":white_square_button:"] = "\uD83D\uDD33",
            [":white_sun_behind_cloud:"] = "\uD83C\uDF25️",
            [":white_sun_behind_cloud_with_rain:"] = "\uD83C\uDF26️",
            [":white_sun_cloud:"] = "\uD83C\uDF25️",
            [":white_sun_rain_cloud:"] = "\uD83C\uDF26️",
            [":white_sun_small_cloud:"] = "\uD83C\uDF24️",
            [":white_sun_with_small_cloud:"] = "\uD83C\uDF24️",
            [":wilted_flower:"] = "\uD83E\uDD40",
            [":wilted_rose:"] = "\uD83E\uDD40",
            [":wind_blowing_face:"] = "\uD83C\uDF2C️",
            [":wind_chime:"] = "\uD83C\uDF90",
            [":window:"] = "\uD83E\uDE9F",
            [":wine_glass:"] = "\uD83C\uDF77",
            [":wink:"] = "\uD83D\uDE09",
            [":wolf:"] = "\uD83D\uDC3A",
            [":woman:"] = "\uD83D\uDC69",
            [":woman::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB",
            [":woman::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC",
            [":woman::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD",
            [":woman::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE",
            [":woman::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF",
            [":woman_artist:"] = "\uD83D\uDC69\u200D\uD83C\uDFA8",
            [":woman_artist::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFA8",
            [":woman_artist::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFA8",
            [":woman_artist::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFA8",
            [":woman_artist::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFA8",
            [":woman_artist::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFA8",
            [":woman_artist_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFA8",
            [":woman_artist_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFA8",
            [":woman_artist_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFA8",
            [":woman_artist_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFA8",
            [":woman_artist_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFA8",
            [":woman_artist_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFA8",
            [":woman_artist_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFA8",
            [":woman_artist_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFA8",
            [":woman_artist_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFA8",
            [":woman_artist_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFA8",
            [":woman_astronaut:"] = "\uD83D\uDC69\u200D\uD83D\uDE80",
            [":woman_astronaut::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDE80",
            [":woman_astronaut::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDE80",
            [":woman_astronaut::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDE80",
            [":woman_astronaut::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDE80",
            [":woman_astronaut::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDE80",
            [":woman_astronaut_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDE80",
            [":woman_astronaut_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDE80",
            [":woman_astronaut_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDE80",
            [":woman_astronaut_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDE80",
            [":woman_astronaut_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDE80",
            [":woman_astronaut_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDE80",
            [":woman_astronaut_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDE80",
            [":woman_astronaut_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDE80",
            [":woman_astronaut_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDE80",
            [":woman_astronaut_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDE80",
            [":woman_bald:"] = "\uD83D\uDC69\u200D\uD83E\uDDB2",
            [":woman_bald::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB2",
            [":woman_bald::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB2",
            [":woman_bald::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB2",
            [":woman_bald::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB2",
            [":woman_bald::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB2",
            [":woman_bald_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB2",
            [":woman_bald_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB2",
            [":woman_bald_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB2",
            [":woman_bald_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB2",
            [":woman_bald_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB2",
            [":woman_bald_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB2",
            [":woman_bald_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB2",
            [":woman_bald_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB2",
            [":woman_bald_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB2",
            [":woman_bald_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB2",
            [":woman_beard:"] = "\uD83E\uDDD4\u200D\u2640\uFE0F",
            [":woman_beard::skin-tone-1:"] = "\uD83E\uDDD4\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_beard::skin-tone-2:"] = "\uD83E\uDDD4\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_beard::skin-tone-3:"] = "\uD83E\uDDD4\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_beard::skin-tone-4:"] = "\uD83E\uDDD4\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_beard::skin-tone-5:"] = "\uD83E\uDDD4\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_beard_tone1:"] = "\uD83E\uDDD4\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_beard_tone2:"] = "\uD83E\uDDD4\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_beard_tone3:"] = "\uD83E\uDDD4\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_beard_tone4:"] = "\uD83E\uDDD4\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_beard_tone5:"] = "\uD83E\uDDD4\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_biking:"] = "\uD83D\uDEB4\u200D♀️",
            [":woman_biking::skin-tone-1:"] = "\uD83D\uDEB4\uD83C\uDFFB\u200D♀️",
            [":woman_biking::skin-tone-2:"] = "\uD83D\uDEB4\uD83C\uDFFC\u200D♀️",
            [":woman_biking::skin-tone-3:"] = "\uD83D\uDEB4\uD83C\uDFFD\u200D♀️",
            [":woman_biking::skin-tone-4:"] = "\uD83D\uDEB4\uD83C\uDFFE\u200D♀️",
            [":woman_biking::skin-tone-5:"] = "\uD83D\uDEB4\uD83C\uDFFF\u200D♀️",
            [":woman_biking_dark_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFF\u200D♀️",
            [":woman_biking_light_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFB\u200D♀️",
            [":woman_biking_medium_dark_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFE\u200D♀️",
            [":woman_biking_medium_light_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFC\u200D♀️",
            [":woman_biking_medium_skin_tone:"] = "\uD83D\uDEB4\uD83C\uDFFD\u200D♀️",
            [":woman_biking_tone1:"] = "\uD83D\uDEB4\uD83C\uDFFB\u200D♀️",
            [":woman_biking_tone2:"] = "\uD83D\uDEB4\uD83C\uDFFC\u200D♀️",
            [":woman_biking_tone3:"] = "\uD83D\uDEB4\uD83C\uDFFD\u200D♀️",
            [":woman_biking_tone4:"] = "\uD83D\uDEB4\uD83C\uDFFE\u200D♀️",
            [":woman_biking_tone5:"] = "\uD83D\uDEB4\uD83C\uDFFF\u200D♀️",
            [":woman_bouncing_ball:"] = "⛹️\u200D♀️",
            [":woman_bouncing_ball::skin-tone-1:"] = "⛹\uD83C\uDFFB\u200D♀️",
            [":woman_bouncing_ball::skin-tone-2:"] = "⛹\uD83C\uDFFC\u200D♀️",
            [":woman_bouncing_ball::skin-tone-3:"] = "⛹\uD83C\uDFFD\u200D♀️",
            [":woman_bouncing_ball::skin-tone-4:"] = "⛹\uD83C\uDFFE\u200D♀️",
            [":woman_bouncing_ball::skin-tone-5:"] = "⛹\uD83C\uDFFF\u200D♀️",
            [":woman_bouncing_ball_dark_skin_tone:"] = "⛹\uD83C\uDFFF\u200D♀️",
            [":woman_bouncing_ball_light_skin_tone:"] = "⛹\uD83C\uDFFB\u200D♀️",
            [":woman_bouncing_ball_medium_dark_skin_tone:"] = "⛹\uD83C\uDFFE\u200D♀️",
            [":woman_bouncing_ball_medium_light_skin_tone:"] = "⛹\uD83C\uDFFC\u200D♀️",
            [":woman_bouncing_ball_medium_skin_tone:"] = "⛹\uD83C\uDFFD\u200D♀️",
            [":woman_bouncing_ball_tone1:"] = "⛹\uD83C\uDFFB\u200D♀️",
            [":woman_bouncing_ball_tone2:"] = "⛹\uD83C\uDFFC\u200D♀️",
            [":woman_bouncing_ball_tone3:"] = "⛹\uD83C\uDFFD\u200D♀️",
            [":woman_bouncing_ball_tone4:"] = "⛹\uD83C\uDFFE\u200D♀️",
            [":woman_bouncing_ball_tone5:"] = "⛹\uD83C\uDFFF\u200D♀️",
            [":woman_bowing:"] = "\uD83D\uDE47\u200D♀️",
            [":woman_bowing::skin-tone-1:"] = "\uD83D\uDE47\uD83C\uDFFB\u200D♀️",
            [":woman_bowing::skin-tone-2:"] = "\uD83D\uDE47\uD83C\uDFFC\u200D♀️",
            [":woman_bowing::skin-tone-3:"] = "\uD83D\uDE47\uD83C\uDFFD\u200D♀️",
            [":woman_bowing::skin-tone-4:"] = "\uD83D\uDE47\uD83C\uDFFE\u200D♀️",
            [":woman_bowing::skin-tone-5:"] = "\uD83D\uDE47\uD83C\uDFFF\u200D♀️",
            [":woman_bowing_dark_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFF\u200D♀️",
            [":woman_bowing_light_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFB\u200D♀️",
            [":woman_bowing_medium_dark_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFE\u200D♀️",
            [":woman_bowing_medium_light_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFC\u200D♀️",
            [":woman_bowing_medium_skin_tone:"] = "\uD83D\uDE47\uD83C\uDFFD\u200D♀️",
            [":woman_bowing_tone1:"] = "\uD83D\uDE47\uD83C\uDFFB\u200D♀️",
            [":woman_bowing_tone2:"] = "\uD83D\uDE47\uD83C\uDFFC\u200D♀️",
            [":woman_bowing_tone3:"] = "\uD83D\uDE47\uD83C\uDFFD\u200D♀️",
            [":woman_bowing_tone4:"] = "\uD83D\uDE47\uD83C\uDFFE\u200D♀️",
            [":woman_bowing_tone5:"] = "\uD83D\uDE47\uD83C\uDFFF\u200D♀️",
            [":woman_cartwheeling:"] = "\uD83E\uDD38\u200D♀️",
            [":woman_cartwheeling::skin-tone-1:"] = "\uD83E\uDD38\uD83C\uDFFB\u200D♀️",
            [":woman_cartwheeling::skin-tone-2:"] = "\uD83E\uDD38\uD83C\uDFFC\u200D♀️",
            [":woman_cartwheeling::skin-tone-3:"] = "\uD83E\uDD38\uD83C\uDFFD\u200D♀️",
            [":woman_cartwheeling::skin-tone-4:"] = "\uD83E\uDD38\uD83C\uDFFE\u200D♀️",
            [":woman_cartwheeling::skin-tone-5:"] = "\uD83E\uDD38\uD83C\uDFFF\u200D♀️",
            [":woman_cartwheeling_dark_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFF\u200D♀️",
            [":woman_cartwheeling_light_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFB\u200D♀️",
            [":woman_cartwheeling_medium_dark_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFE\u200D♀️",
            [":woman_cartwheeling_medium_light_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFC\u200D♀️",
            [":woman_cartwheeling_medium_skin_tone:"] = "\uD83E\uDD38\uD83C\uDFFD\u200D♀️",
            [":woman_cartwheeling_tone1:"] = "\uD83E\uDD38\uD83C\uDFFB\u200D♀️",
            [":woman_cartwheeling_tone2:"] = "\uD83E\uDD38\uD83C\uDFFC\u200D♀️",
            [":woman_cartwheeling_tone3:"] = "\uD83E\uDD38\uD83C\uDFFD\u200D♀️",
            [":woman_cartwheeling_tone4:"] = "\uD83E\uDD38\uD83C\uDFFE\u200D♀️",
            [":woman_cartwheeling_tone5:"] = "\uD83E\uDD38\uD83C\uDFFF\u200D♀️",
            [":woman_climbing:"] = "\uD83E\uDDD7\u200D♀️",
            [":woman_climbing::skin-tone-1:"] = "\uD83E\uDDD7\uD83C\uDFFB\u200D♀️",
            [":woman_climbing::skin-tone-2:"] = "\uD83E\uDDD7\uD83C\uDFFC\u200D♀️",
            [":woman_climbing::skin-tone-3:"] = "\uD83E\uDDD7\uD83C\uDFFD\u200D♀️",
            [":woman_climbing::skin-tone-4:"] = "\uD83E\uDDD7\uD83C\uDFFE\u200D♀️",
            [":woman_climbing::skin-tone-5:"] = "\uD83E\uDDD7\uD83C\uDFFF\u200D♀️",
            [":woman_climbing_dark_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFF\u200D♀️",
            [":woman_climbing_light_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFB\u200D♀️",
            [":woman_climbing_medium_dark_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFE\u200D♀️",
            [":woman_climbing_medium_light_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFC\u200D♀️",
            [":woman_climbing_medium_skin_tone:"] = "\uD83E\uDDD7\uD83C\uDFFD\u200D♀️",
            [":woman_climbing_tone1:"] = "\uD83E\uDDD7\uD83C\uDFFB\u200D♀️",
            [":woman_climbing_tone2:"] = "\uD83E\uDDD7\uD83C\uDFFC\u200D♀️",
            [":woman_climbing_tone3:"] = "\uD83E\uDDD7\uD83C\uDFFD\u200D♀️",
            [":woman_climbing_tone4:"] = "\uD83E\uDDD7\uD83C\uDFFE\u200D♀️",
            [":woman_climbing_tone5:"] = "\uD83E\uDDD7\uD83C\uDFFF\u200D♀️",
            [":woman_construction_worker:"] = "\uD83D\uDC77\u200D♀️",
            [":woman_construction_worker::skin-tone-1:"] = "\uD83D\uDC77\uD83C\uDFFB\u200D♀️",
            [":woman_construction_worker::skin-tone-2:"] = "\uD83D\uDC77\uD83C\uDFFC\u200D♀️",
            [":woman_construction_worker::skin-tone-3:"] = "\uD83D\uDC77\uD83C\uDFFD\u200D♀️",
            [":woman_construction_worker::skin-tone-4:"] = "\uD83D\uDC77\uD83C\uDFFE\u200D♀️",
            [":woman_construction_worker::skin-tone-5:"] = "\uD83D\uDC77\uD83C\uDFFF\u200D♀️",
            [":woman_construction_worker_dark_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFF\u200D♀️",
            [":woman_construction_worker_light_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFB\u200D♀️",
            [":woman_construction_worker_medium_dark_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFE\u200D♀️",
            [":woman_construction_worker_medium_light_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFC\u200D♀️",
            [":woman_construction_worker_medium_skin_tone:"] = "\uD83D\uDC77\uD83C\uDFFD\u200D♀️",
            [":woman_construction_worker_tone1:"] = "\uD83D\uDC77\uD83C\uDFFB\u200D♀️",
            [":woman_construction_worker_tone2:"] = "\uD83D\uDC77\uD83C\uDFFC\u200D♀️",
            [":woman_construction_worker_tone3:"] = "\uD83D\uDC77\uD83C\uDFFD\u200D♀️",
            [":woman_construction_worker_tone4:"] = "\uD83D\uDC77\uD83C\uDFFE\u200D♀️",
            [":woman_construction_worker_tone5:"] = "\uD83D\uDC77\uD83C\uDFFF\u200D♀️",
            [":woman_cook:"] = "\uD83D\uDC69\u200D\uD83C\uDF73",
            [":woman_cook::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF73",
            [":woman_cook::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF73",
            [":woman_cook::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF73",
            [":woman_cook::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF73",
            [":woman_cook::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF73",
            [":woman_cook_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF73",
            [":woman_cook_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF73",
            [":woman_cook_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF73",
            [":woman_cook_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF73",
            [":woman_cook_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF73",
            [":woman_cook_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF73",
            [":woman_cook_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF73",
            [":woman_cook_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF73",
            [":woman_cook_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF73",
            [":woman_cook_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF73",
            [":woman_curly_haired:"] = "\uD83D\uDC69\u200D\uD83E\uDDB1",
            [":woman_curly_haired::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB1",
            [":woman_curly_haired::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB1",
            [":woman_curly_haired::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB1",
            [":woman_curly_haired::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB1",
            [":woman_curly_haired::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB1",
            [":woman_curly_haired_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB1",
            [":woman_curly_haired_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB1",
            [":woman_curly_haired_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB1",
            [":woman_curly_haired_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB1",
            [":woman_curly_haired_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB1",
            [":woman_curly_haired_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB1",
            [":woman_curly_haired_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB1",
            [":woman_curly_haired_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB1",
            [":woman_curly_haired_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB1",
            [":woman_curly_haired_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB1",
            [":woman_dark_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_detective:"] = "\uD83D\uDD75️\u200D♀️",
            [":woman_detective::skin-tone-1:"] = "\uD83D\uDD75\uD83C\uDFFB\u200D♀️",
            [":woman_detective::skin-tone-2:"] = "\uD83D\uDD75\uD83C\uDFFC\u200D♀️",
            [":woman_detective::skin-tone-3:"] = "\uD83D\uDD75\uD83C\uDFFD\u200D♀️",
            [":woman_detective::skin-tone-4:"] = "\uD83D\uDD75\uD83C\uDFFE\u200D♀️",
            [":woman_detective::skin-tone-5:"] = "\uD83D\uDD75\uD83C\uDFFF\u200D♀️",
            [":woman_detective_dark_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFF\u200D♀️",
            [":woman_detective_light_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFB\u200D♀️",
            [":woman_detective_medium_dark_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFE\u200D♀️",
            [":woman_detective_medium_light_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFC\u200D♀️",
            [":woman_detective_medium_skin_tone:"] = "\uD83D\uDD75\uD83C\uDFFD\u200D♀️",
            [":woman_detective_tone1:"] = "\uD83D\uDD75\uD83C\uDFFB\u200D♀️",
            [":woman_detective_tone2:"] = "\uD83D\uDD75\uD83C\uDFFC\u200D♀️",
            [":woman_detective_tone3:"] = "\uD83D\uDD75\uD83C\uDFFD\u200D♀️",
            [":woman_detective_tone4:"] = "\uD83D\uDD75\uD83C\uDFFE\u200D♀️",
            [":woman_detective_tone5:"] = "\uD83D\uDD75\uD83C\uDFFF\u200D♀️",
            [":woman_elf:"] = "\uD83E\uDDDD\u200D♀️",
            [":woman_elf::skin-tone-1:"] = "\uD83E\uDDDD\uD83C\uDFFB\u200D♀️",
            [":woman_elf::skin-tone-2:"] = "\uD83E\uDDDD\uD83C\uDFFC\u200D♀️",
            [":woman_elf::skin-tone-3:"] = "\uD83E\uDDDD\uD83C\uDFFD\u200D♀️",
            [":woman_elf::skin-tone-4:"] = "\uD83E\uDDDD\uD83C\uDFFE\u200D♀️",
            [":woman_elf::skin-tone-5:"] = "\uD83E\uDDDD\uD83C\uDFFF\u200D♀️",
            [":woman_elf_dark_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFF\u200D♀️",
            [":woman_elf_light_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFB\u200D♀️",
            [":woman_elf_medium_dark_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFE\u200D♀️",
            [":woman_elf_medium_light_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFC\u200D♀️",
            [":woman_elf_medium_skin_tone:"] = "\uD83E\uDDDD\uD83C\uDFFD\u200D♀️",
            [":woman_elf_tone1:"] = "\uD83E\uDDDD\uD83C\uDFFB\u200D♀️",
            [":woman_elf_tone2:"] = "\uD83E\uDDDD\uD83C\uDFFC\u200D♀️",
            [":woman_elf_tone3:"] = "\uD83E\uDDDD\uD83C\uDFFD\u200D♀️",
            [":woman_elf_tone4:"] = "\uD83E\uDDDD\uD83C\uDFFE\u200D♀️",
            [":woman_elf_tone5:"] = "\uD83E\uDDDD\uD83C\uDFFF\u200D♀️",
            [":woman_facepalming:"] = "\uD83E\uDD26\u200D♀️",
            [":woman_facepalming::skin-tone-1:"] = "\uD83E\uDD26\uD83C\uDFFB\u200D♀️",
            [":woman_facepalming::skin-tone-2:"] = "\uD83E\uDD26\uD83C\uDFFC\u200D♀️",
            [":woman_facepalming::skin-tone-3:"] = "\uD83E\uDD26\uD83C\uDFFD\u200D♀️",
            [":woman_facepalming::skin-tone-4:"] = "\uD83E\uDD26\uD83C\uDFFE\u200D♀️",
            [":woman_facepalming::skin-tone-5:"] = "\uD83E\uDD26\uD83C\uDFFF\u200D♀️",
            [":woman_facepalming_dark_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFF\u200D♀️",
            [":woman_facepalming_light_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFB\u200D♀️",
            [":woman_facepalming_medium_dark_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFE\u200D♀️",
            [":woman_facepalming_medium_light_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFC\u200D♀️",
            [":woman_facepalming_medium_skin_tone:"] = "\uD83E\uDD26\uD83C\uDFFD\u200D♀️",
            [":woman_facepalming_tone1:"] = "\uD83E\uDD26\uD83C\uDFFB\u200D♀️",
            [":woman_facepalming_tone2:"] = "\uD83E\uDD26\uD83C\uDFFC\u200D♀️",
            [":woman_facepalming_tone3:"] = "\uD83E\uDD26\uD83C\uDFFD\u200D♀️",
            [":woman_facepalming_tone4:"] = "\uD83E\uDD26\uD83C\uDFFE\u200D♀️",
            [":woman_facepalming_tone5:"] = "\uD83E\uDD26\uD83C\uDFFF\u200D♀️",
            [":woman_factory_worker:"] = "\uD83D\uDC69\u200D\uD83C\uDFED",
            [":woman_factory_worker::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFED",
            [":woman_factory_worker::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFED",
            [":woman_factory_worker::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFED",
            [":woman_factory_worker::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFED",
            [":woman_factory_worker::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFED",
            [":woman_factory_worker_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFED",
            [":woman_factory_worker_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFED",
            [":woman_factory_worker_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFED",
            [":woman_factory_worker_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFED",
            [":woman_factory_worker_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFED",
            [":woman_factory_worker_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFED",
            [":woman_factory_worker_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFED",
            [":woman_factory_worker_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFED",
            [":woman_factory_worker_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFED",
            [":woman_factory_worker_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFED",
            [":woman_fairy:"] = "\uD83E\uDDDA\u200D♀️",
            [":woman_fairy::skin-tone-1:"] = "\uD83E\uDDDA\uD83C\uDFFB\u200D♀️",
            [":woman_fairy::skin-tone-2:"] = "\uD83E\uDDDA\uD83C\uDFFC\u200D♀️",
            [":woman_fairy::skin-tone-3:"] = "\uD83E\uDDDA\uD83C\uDFFD\u200D♀️",
            [":woman_fairy::skin-tone-4:"] = "\uD83E\uDDDA\uD83C\uDFFE\u200D♀️",
            [":woman_fairy::skin-tone-5:"] = "\uD83E\uDDDA\uD83C\uDFFF\u200D♀️",
            [":woman_fairy_dark_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFF\u200D♀️",
            [":woman_fairy_light_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFB\u200D♀️",
            [":woman_fairy_medium_dark_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFE\u200D♀️",
            [":woman_fairy_medium_light_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFC\u200D♀️",
            [":woman_fairy_medium_skin_tone:"] = "\uD83E\uDDDA\uD83C\uDFFD\u200D♀️",
            [":woman_fairy_tone1:"] = "\uD83E\uDDDA\uD83C\uDFFB\u200D♀️",
            [":woman_fairy_tone2:"] = "\uD83E\uDDDA\uD83C\uDFFC\u200D♀️",
            [":woman_fairy_tone3:"] = "\uD83E\uDDDA\uD83C\uDFFD\u200D♀️",
            [":woman_fairy_tone4:"] = "\uD83E\uDDDA\uD83C\uDFFE\u200D♀️",
            [":woman_fairy_tone5:"] = "\uD83E\uDDDA\uD83C\uDFFF\u200D♀️",
            [":woman_farmer:"] = "\uD83D\uDC69\u200D\uD83C\uDF3E",
            [":woman_farmer::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF3E",
            [":woman_farmer::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF3E",
            [":woman_farmer::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF3E",
            [":woman_farmer::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF3E",
            [":woman_farmer::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF3E",
            [":woman_farmer_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF3E",
            [":woman_farmer_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF3E",
            [":woman_farmer_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF3E",
            [":woman_farmer_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF3E",
            [":woman_farmer_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF3E",
            [":woman_farmer_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF3E",
            [":woman_farmer_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF3E",
            [":woman_farmer_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF3E",
            [":woman_farmer_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF3E",
            [":woman_farmer_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF3E",
            [":woman_feeding_baby:"] = "\uD83D\uDC69\u200D\uD83C\uDF7C",
            [":woman_feeding_baby::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":woman_feeding_baby::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":woman_feeding_baby::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":woman_feeding_baby::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":woman_feeding_baby::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF7C",
            [":woman_feeding_baby_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF7C",
            [":woman_firefighter:"] = "\uD83D\uDC69\u200D\uD83D\uDE92",
            [":woman_firefighter::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDE92",
            [":woman_firefighter::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDE92",
            [":woman_firefighter::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDE92",
            [":woman_firefighter::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDE92",
            [":woman_firefighter::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDE92",
            [":woman_firefighter_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDE92",
            [":woman_firefighter_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDE92",
            [":woman_firefighter_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDE92",
            [":woman_firefighter_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDE92",
            [":woman_firefighter_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDE92",
            [":woman_firefighter_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDE92",
            [":woman_firefighter_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDE92",
            [":woman_firefighter_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDE92",
            [":woman_firefighter_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDE92",
            [":woman_firefighter_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDE92",
            [":woman_frowning:"] = "\uD83D\uDE4D\u200D♀️",
            [":woman_frowning::skin-tone-1:"] = "\uD83D\uDE4D\uD83C\uDFFB\u200D♀️",
            [":woman_frowning::skin-tone-2:"] = "\uD83D\uDE4D\uD83C\uDFFC\u200D♀️",
            [":woman_frowning::skin-tone-3:"] = "\uD83D\uDE4D\uD83C\uDFFD\u200D♀️",
            [":woman_frowning::skin-tone-4:"] = "\uD83D\uDE4D\uD83C\uDFFE\u200D♀️",
            [":woman_frowning::skin-tone-5:"] = "\uD83D\uDE4D\uD83C\uDFFF\u200D♀️",
            [":woman_frowning_dark_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFF\u200D♀️",
            [":woman_frowning_light_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFB\u200D♀️",
            [":woman_frowning_medium_dark_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFE\u200D♀️",
            [":woman_frowning_medium_light_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFC\u200D♀️",
            [":woman_frowning_medium_skin_tone:"] = "\uD83D\uDE4D\uD83C\uDFFD\u200D♀️",
            [":woman_frowning_tone1:"] = "\uD83D\uDE4D\uD83C\uDFFB\u200D♀️",
            [":woman_frowning_tone2:"] = "\uD83D\uDE4D\uD83C\uDFFC\u200D♀️",
            [":woman_frowning_tone3:"] = "\uD83D\uDE4D\uD83C\uDFFD\u200D♀️",
            [":woman_frowning_tone4:"] = "\uD83D\uDE4D\uD83C\uDFFE\u200D♀️",
            [":woman_frowning_tone5:"] = "\uD83D\uDE4D\uD83C\uDFFF\u200D♀️",
            [":woman_genie:"] = "\uD83E\uDDDE\u200D♀️",
            [":woman_gesturing_no:"] = "\uD83D\uDE45\u200D♀️",
            [":woman_gesturing_no::skin-tone-1:"] = "\uD83D\uDE45\uD83C\uDFFB\u200D♀️",
            [":woman_gesturing_no::skin-tone-2:"] = "\uD83D\uDE45\uD83C\uDFFC\u200D♀️",
            [":woman_gesturing_no::skin-tone-3:"] = "\uD83D\uDE45\uD83C\uDFFD\u200D♀️",
            [":woman_gesturing_no::skin-tone-4:"] = "\uD83D\uDE45\uD83C\uDFFE\u200D♀️",
            [":woman_gesturing_no::skin-tone-5:"] = "\uD83D\uDE45\uD83C\uDFFF\u200D♀️",
            [":woman_gesturing_no_dark_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFF\u200D♀️",
            [":woman_gesturing_no_light_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFB\u200D♀️",
            [":woman_gesturing_no_medium_dark_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFE\u200D♀️",
            [":woman_gesturing_no_medium_light_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFC\u200D♀️",
            [":woman_gesturing_no_medium_skin_tone:"] = "\uD83D\uDE45\uD83C\uDFFD\u200D♀️",
            [":woman_gesturing_no_tone1:"] = "\uD83D\uDE45\uD83C\uDFFB\u200D♀️",
            [":woman_gesturing_no_tone2:"] = "\uD83D\uDE45\uD83C\uDFFC\u200D♀️",
            [":woman_gesturing_no_tone3:"] = "\uD83D\uDE45\uD83C\uDFFD\u200D♀️",
            [":woman_gesturing_no_tone4:"] = "\uD83D\uDE45\uD83C\uDFFE\u200D♀️",
            [":woman_gesturing_no_tone5:"] = "\uD83D\uDE45\uD83C\uDFFF\u200D♀️",
            [":woman_gesturing_ok:"] = "\uD83D\uDE46\u200D♀️",
            [":woman_gesturing_ok::skin-tone-1:"] = "\uD83D\uDE46\uD83C\uDFFB\u200D♀️",
            [":woman_gesturing_ok::skin-tone-2:"] = "\uD83D\uDE46\uD83C\uDFFC\u200D♀️",
            [":woman_gesturing_ok::skin-tone-3:"] = "\uD83D\uDE46\uD83C\uDFFD\u200D♀️",
            [":woman_gesturing_ok::skin-tone-4:"] = "\uD83D\uDE46\uD83C\uDFFE\u200D♀️",
            [":woman_gesturing_ok::skin-tone-5:"] = "\uD83D\uDE46\uD83C\uDFFF\u200D♀️",
            [":woman_gesturing_ok_dark_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFF\u200D♀️",
            [":woman_gesturing_ok_light_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFB\u200D♀️",
            [":woman_gesturing_ok_medium_dark_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFE\u200D♀️",
            [":woman_gesturing_ok_medium_light_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFC\u200D♀️",
            [":woman_gesturing_ok_medium_skin_tone:"] = "\uD83D\uDE46\uD83C\uDFFD\u200D♀️",
            [":woman_gesturing_ok_tone1:"] = "\uD83D\uDE46\uD83C\uDFFB\u200D♀️",
            [":woman_gesturing_ok_tone2:"] = "\uD83D\uDE46\uD83C\uDFFC\u200D♀️",
            [":woman_gesturing_ok_tone3:"] = "\uD83D\uDE46\uD83C\uDFFD\u200D♀️",
            [":woman_gesturing_ok_tone4:"] = "\uD83D\uDE46\uD83C\uDFFE\u200D♀️",
            [":woman_gesturing_ok_tone5:"] = "\uD83D\uDE46\uD83C\uDFFF\u200D♀️",
            [":woman_getting_face_massage:"] = "\uD83D\uDC86\u200D♀️",
            [":woman_getting_face_massage::skin-tone-1:"] = "\uD83D\uDC86\uD83C\uDFFB\u200D♀️",
            [":woman_getting_face_massage::skin-tone-2:"] = "\uD83D\uDC86\uD83C\uDFFC\u200D♀️",
            [":woman_getting_face_massage::skin-tone-3:"] = "\uD83D\uDC86\uD83C\uDFFD\u200D♀️",
            [":woman_getting_face_massage::skin-tone-4:"] = "\uD83D\uDC86\uD83C\uDFFE\u200D♀️",
            [":woman_getting_face_massage::skin-tone-5:"] = "\uD83D\uDC86\uD83C\uDFFF\u200D♀️",
            [":woman_getting_face_massage_dark_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFF\u200D♀️",
            [":woman_getting_face_massage_light_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFB\u200D♀️",
            [":woman_getting_face_massage_medium_dark_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFE\u200D♀️",
            [":woman_getting_face_massage_medium_light_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFC\u200D♀️",
            [":woman_getting_face_massage_medium_skin_tone:"] = "\uD83D\uDC86\uD83C\uDFFD\u200D♀️",
            [":woman_getting_face_massage_tone1:"] = "\uD83D\uDC86\uD83C\uDFFB\u200D♀️",
            [":woman_getting_face_massage_tone2:"] = "\uD83D\uDC86\uD83C\uDFFC\u200D♀️",
            [":woman_getting_face_massage_tone3:"] = "\uD83D\uDC86\uD83C\uDFFD\u200D♀️",
            [":woman_getting_face_massage_tone4:"] = "\uD83D\uDC86\uD83C\uDFFE\u200D♀️",
            [":woman_getting_face_massage_tone5:"] = "\uD83D\uDC86\uD83C\uDFFF\u200D♀️",
            [":woman_getting_haircut:"] = "\uD83D\uDC87\u200D♀️",
            [":woman_getting_haircut::skin-tone-1:"] = "\uD83D\uDC87\uD83C\uDFFB\u200D♀️",
            [":woman_getting_haircut::skin-tone-2:"] = "\uD83D\uDC87\uD83C\uDFFC\u200D♀️",
            [":woman_getting_haircut::skin-tone-3:"] = "\uD83D\uDC87\uD83C\uDFFD\u200D♀️",
            [":woman_getting_haircut::skin-tone-4:"] = "\uD83D\uDC87\uD83C\uDFFE\u200D♀️",
            [":woman_getting_haircut::skin-tone-5:"] = "\uD83D\uDC87\uD83C\uDFFF\u200D♀️",
            [":woman_getting_haircut_dark_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFF\u200D♀️",
            [":woman_getting_haircut_light_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFB\u200D♀️",
            [":woman_getting_haircut_medium_dark_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFE\u200D♀️",
            [":woman_getting_haircut_medium_light_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFC\u200D♀️",
            [":woman_getting_haircut_medium_skin_tone:"] = "\uD83D\uDC87\uD83C\uDFFD\u200D♀️",
            [":woman_getting_haircut_tone1:"] = "\uD83D\uDC87\uD83C\uDFFB\u200D♀️",
            [":woman_getting_haircut_tone2:"] = "\uD83D\uDC87\uD83C\uDFFC\u200D♀️",
            [":woman_getting_haircut_tone3:"] = "\uD83D\uDC87\uD83C\uDFFD\u200D♀️",
            [":woman_getting_haircut_tone4:"] = "\uD83D\uDC87\uD83C\uDFFE\u200D♀️",
            [":woman_getting_haircut_tone5:"] = "\uD83D\uDC87\uD83C\uDFFF\u200D♀️",
            [":woman_golfing:"] = "\uD83C\uDFCC️\u200D♀️",
            [":woman_golfing::skin-tone-1:"] = "\uD83C\uDFCC\uD83C\uDFFB\u200D♀️",
            [":woman_golfing::skin-tone-2:"] = "\uD83C\uDFCC\uD83C\uDFFC\u200D♀️",
            [":woman_golfing::skin-tone-3:"] = "\uD83C\uDFCC\uD83C\uDFFD\u200D♀️",
            [":woman_golfing::skin-tone-4:"] = "\uD83C\uDFCC\uD83C\uDFFE\u200D♀️",
            [":woman_golfing::skin-tone-5:"] = "\uD83C\uDFCC\uD83C\uDFFF\u200D♀️",
            [":woman_golfing_dark_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFF\u200D♀️",
            [":woman_golfing_light_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFB\u200D♀️",
            [":woman_golfing_medium_dark_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFE\u200D♀️",
            [":woman_golfing_medium_light_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFC\u200D♀️",
            [":woman_golfing_medium_skin_tone:"] = "\uD83C\uDFCC\uD83C\uDFFD\u200D♀️",
            [":woman_golfing_tone1:"] = "\uD83C\uDFCC\uD83C\uDFFB\u200D♀️",
            [":woman_golfing_tone2:"] = "\uD83C\uDFCC\uD83C\uDFFC\u200D♀️",
            [":woman_golfing_tone3:"] = "\uD83C\uDFCC\uD83C\uDFFD\u200D♀️",
            [":woman_golfing_tone4:"] = "\uD83C\uDFCC\uD83C\uDFFE\u200D♀️",
            [":woman_golfing_tone5:"] = "\uD83C\uDFCC\uD83C\uDFFF\u200D♀️",
            [":woman_guard:"] = "\uD83D\uDC82\u200D♀️",
            [":woman_guard::skin-tone-1:"] = "\uD83D\uDC82\uD83C\uDFFB\u200D♀️",
            [":woman_guard::skin-tone-2:"] = "\uD83D\uDC82\uD83C\uDFFC\u200D♀️",
            [":woman_guard::skin-tone-3:"] = "\uD83D\uDC82\uD83C\uDFFD\u200D♀️",
            [":woman_guard::skin-tone-4:"] = "\uD83D\uDC82\uD83C\uDFFE\u200D♀️",
            [":woman_guard::skin-tone-5:"] = "\uD83D\uDC82\uD83C\uDFFF\u200D♀️",
            [":woman_guard_dark_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFF\u200D♀️",
            [":woman_guard_light_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFB\u200D♀️",
            [":woman_guard_medium_dark_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFE\u200D♀️",
            [":woman_guard_medium_light_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFC\u200D♀️",
            [":woman_guard_medium_skin_tone:"] = "\uD83D\uDC82\uD83C\uDFFD\u200D♀️",
            [":woman_guard_tone1:"] = "\uD83D\uDC82\uD83C\uDFFB\u200D♀️",
            [":woman_guard_tone2:"] = "\uD83D\uDC82\uD83C\uDFFC\u200D♀️",
            [":woman_guard_tone3:"] = "\uD83D\uDC82\uD83C\uDFFD\u200D♀️",
            [":woman_guard_tone4:"] = "\uD83D\uDC82\uD83C\uDFFE\u200D♀️",
            [":woman_guard_tone5:"] = "\uD83D\uDC82\uD83C\uDFFF\u200D♀️",
            [":woman_health_worker:"] = "\uD83D\uDC69\u200D⚕️",
            [":woman_health_worker::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D⚕️",
            [":woman_health_worker::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D⚕️",
            [":woman_health_worker::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D⚕️",
            [":woman_health_worker::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D⚕️",
            [":woman_health_worker::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D⚕️",
            [":woman_health_worker_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D⚕️",
            [":woman_health_worker_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D⚕️",
            [":woman_health_worker_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D⚕️",
            [":woman_health_worker_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D⚕️",
            [":woman_health_worker_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D⚕️",
            [":woman_health_worker_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D⚕️",
            [":woman_health_worker_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D⚕️",
            [":woman_health_worker_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D⚕️",
            [":woman_health_worker_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D⚕️",
            [":woman_health_worker_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D⚕️",
            [":woman_in_lotus_position:"] = "\uD83E\uDDD8\u200D♀️",
            [":woman_in_lotus_position::skin-tone-1:"] = "\uD83E\uDDD8\uD83C\uDFFB\u200D♀️",
            [":woman_in_lotus_position::skin-tone-2:"] = "\uD83E\uDDD8\uD83C\uDFFC\u200D♀️",
            [":woman_in_lotus_position::skin-tone-3:"] = "\uD83E\uDDD8\uD83C\uDFFD\u200D♀️",
            [":woman_in_lotus_position::skin-tone-4:"] = "\uD83E\uDDD8\uD83C\uDFFE\u200D♀️",
            [":woman_in_lotus_position::skin-tone-5:"] = "\uD83E\uDDD8\uD83C\uDFFF\u200D♀️",
            [":woman_in_lotus_position_dark_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFF\u200D♀️",
            [":woman_in_lotus_position_light_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFB\u200D♀️",
            [":woman_in_lotus_position_medium_dark_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFE\u200D♀️",
            [":woman_in_lotus_position_medium_light_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFC\u200D♀️",
            [":woman_in_lotus_position_medium_skin_tone:"] = "\uD83E\uDDD8\uD83C\uDFFD\u200D♀️",
            [":woman_in_lotus_position_tone1:"] = "\uD83E\uDDD8\uD83C\uDFFB\u200D♀️",
            [":woman_in_lotus_position_tone2:"] = "\uD83E\uDDD8\uD83C\uDFFC\u200D♀️",
            [":woman_in_lotus_position_tone3:"] = "\uD83E\uDDD8\uD83C\uDFFD\u200D♀️",
            [":woman_in_lotus_position_tone4:"] = "\uD83E\uDDD8\uD83C\uDFFE\u200D♀️",
            [":woman_in_lotus_position_tone5:"] = "\uD83E\uDDD8\uD83C\uDFFF\u200D♀️",
            [":woman_in_manual_wheelchair:"] = "\uD83D\uDC69\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDBD",
            [":woman_in_manual_wheelchair_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDBD",
            [":woman_in_motorized_wheelchair:"] = "\uD83D\uDC69\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDBC",
            [":woman_in_motorized_wheelchair_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDBC",
            [":woman_in_steamy_room:"] = "\uD83E\uDDD6\u200D♀️",
            [":woman_in_steamy_room::skin-tone-1:"] = "\uD83E\uDDD6\uD83C\uDFFB\u200D♀️",
            [":woman_in_steamy_room::skin-tone-2:"] = "\uD83E\uDDD6\uD83C\uDFFC\u200D♀️",
            [":woman_in_steamy_room::skin-tone-3:"] = "\uD83E\uDDD6\uD83C\uDFFD\u200D♀️",
            [":woman_in_steamy_room::skin-tone-4:"] = "\uD83E\uDDD6\uD83C\uDFFE\u200D♀️",
            [":woman_in_steamy_room::skin-tone-5:"] = "\uD83E\uDDD6\uD83C\uDFFF\u200D♀️",
            [":woman_in_steamy_room_dark_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFF\u200D♀️",
            [":woman_in_steamy_room_light_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFB\u200D♀️",
            [":woman_in_steamy_room_medium_dark_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFE\u200D♀️",
            [":woman_in_steamy_room_medium_light_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFC\u200D♀️",
            [":woman_in_steamy_room_medium_skin_tone:"] = "\uD83E\uDDD6\uD83C\uDFFD\u200D♀️",
            [":woman_in_steamy_room_tone1:"] = "\uD83E\uDDD6\uD83C\uDFFB\u200D♀️",
            [":woman_in_steamy_room_tone2:"] = "\uD83E\uDDD6\uD83C\uDFFC\u200D♀️",
            [":woman_in_steamy_room_tone3:"] = "\uD83E\uDDD6\uD83C\uDFFD\u200D♀️",
            [":woman_in_steamy_room_tone4:"] = "\uD83E\uDDD6\uD83C\uDFFE\u200D♀️",
            [":woman_in_steamy_room_tone5:"] = "\uD83E\uDDD6\uD83C\uDFFF\u200D♀️",
            [":woman_in_tuxedo:"] = "\uD83E\uDD35\u200D\u2640\uFE0F",
            [":woman_in_tuxedo::skin-tone-1:"] = "\uD83E\uDD35\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_in_tuxedo::skin-tone-2:"] = "\uD83E\uDD35\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_in_tuxedo::skin-tone-3:"] = "\uD83E\uDD35\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_in_tuxedo::skin-tone-4:"] = "\uD83E\uDD35\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_in_tuxedo::skin-tone-5:"] = "\uD83E\uDD35\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_dark_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_light_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_medium_dark_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_medium_light_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_medium_skin_tone:"] = "\uD83E\uDD35\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_tone1:"] = "\uD83E\uDD35\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_tone2:"] = "\uD83E\uDD35\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_tone3:"] = "\uD83E\uDD35\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_tone4:"] = "\uD83E\uDD35\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_in_tuxedo_tone5:"] = "\uD83E\uDD35\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_judge:"] = "\uD83D\uDC69\u200D⚖️",
            [":woman_judge::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D⚖️",
            [":woman_judge::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D⚖️",
            [":woman_judge::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D⚖️",
            [":woman_judge::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D⚖️",
            [":woman_judge::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D⚖️",
            [":woman_judge_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D⚖️",
            [":woman_judge_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D⚖️",
            [":woman_judge_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D⚖️",
            [":woman_judge_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D⚖️",
            [":woman_judge_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D⚖️",
            [":woman_judge_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D⚖️",
            [":woman_judge_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D⚖️",
            [":woman_judge_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D⚖️",
            [":woman_judge_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D⚖️",
            [":woman_judge_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D⚖️",
            [":woman_juggling:"] = "\uD83E\uDD39\u200D♀️",
            [":woman_juggling::skin-tone-1:"] = "\uD83E\uDD39\uD83C\uDFFB\u200D♀️",
            [":woman_juggling::skin-tone-2:"] = "\uD83E\uDD39\uD83C\uDFFC\u200D♀️",
            [":woman_juggling::skin-tone-3:"] = "\uD83E\uDD39\uD83C\uDFFD\u200D♀️",
            [":woman_juggling::skin-tone-4:"] = "\uD83E\uDD39\uD83C\uDFFE\u200D♀️",
            [":woman_juggling::skin-tone-5:"] = "\uD83E\uDD39\uD83C\uDFFF\u200D♀️",
            [":woman_juggling_dark_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFF\u200D♀️",
            [":woman_juggling_light_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFB\u200D♀️",
            [":woman_juggling_medium_dark_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFE\u200D♀️",
            [":woman_juggling_medium_light_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFC\u200D♀️",
            [":woman_juggling_medium_skin_tone:"] = "\uD83E\uDD39\uD83C\uDFFD\u200D♀️",
            [":woman_juggling_tone1:"] = "\uD83E\uDD39\uD83C\uDFFB\u200D♀️",
            [":woman_juggling_tone2:"] = "\uD83E\uDD39\uD83C\uDFFC\u200D♀️",
            [":woman_juggling_tone3:"] = "\uD83E\uDD39\uD83C\uDFFD\u200D♀️",
            [":woman_juggling_tone4:"] = "\uD83E\uDD39\uD83C\uDFFE\u200D♀️",
            [":woman_juggling_tone5:"] = "\uD83E\uDD39\uD83C\uDFFF\u200D♀️",
            [":woman_kneeling:"] = "\uD83E\uDDCE\u200D♀️",
            [":woman_kneeling::skin-tone-1:"] = "\uD83E\uDDCE\uD83C\uDFFB\u200D♀️",
            [":woman_kneeling::skin-tone-2:"] = "\uD83E\uDDCE\uD83C\uDFFC\u200D♀️",
            [":woman_kneeling::skin-tone-3:"] = "\uD83E\uDDCE\uD83C\uDFFD\u200D♀️",
            [":woman_kneeling::skin-tone-4:"] = "\uD83E\uDDCE\uD83C\uDFFE\u200D♀️",
            [":woman_kneeling::skin-tone-5:"] = "\uD83E\uDDCE\uD83C\uDFFF\u200D♀️",
            [":woman_kneeling_dark_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFF\u200D♀️",
            [":woman_kneeling_light_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFB\u200D♀️",
            [":woman_kneeling_medium_dark_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFE\u200D♀️",
            [":woman_kneeling_medium_light_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFC\u200D♀️",
            [":woman_kneeling_medium_skin_tone:"] = "\uD83E\uDDCE\uD83C\uDFFD\u200D♀️",
            [":woman_kneeling_tone1:"] = "\uD83E\uDDCE\uD83C\uDFFB\u200D♀️",
            [":woman_kneeling_tone2:"] = "\uD83E\uDDCE\uD83C\uDFFC\u200D♀️",
            [":woman_kneeling_tone3:"] = "\uD83E\uDDCE\uD83C\uDFFD\u200D♀️",
            [":woman_kneeling_tone4:"] = "\uD83E\uDDCE\uD83C\uDFFE\u200D♀️",
            [":woman_kneeling_tone5:"] = "\uD83E\uDDCE\uD83C\uDFFF\u200D♀️",
            [":woman_lifting_weights:"] = "\uD83C\uDFCB️\u200D♀️",
            [":woman_lifting_weights::skin-tone-1:"] = "\uD83C\uDFCB\uD83C\uDFFB\u200D♀️",
            [":woman_lifting_weights::skin-tone-2:"] = "\uD83C\uDFCB\uD83C\uDFFC\u200D♀️",
            [":woman_lifting_weights::skin-tone-3:"] = "\uD83C\uDFCB\uD83C\uDFFD\u200D♀️",
            [":woman_lifting_weights::skin-tone-4:"] = "\uD83C\uDFCB\uD83C\uDFFE\u200D♀️",
            [":woman_lifting_weights::skin-tone-5:"] = "\uD83C\uDFCB\uD83C\uDFFF\u200D♀️",
            [":woman_lifting_weights_dark_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFF\u200D♀️",
            [":woman_lifting_weights_light_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFB\u200D♀️",
            [":woman_lifting_weights_medium_dark_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFE\u200D♀️",
            [":woman_lifting_weights_medium_light_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFC\u200D♀️",
            [":woman_lifting_weights_medium_skin_tone:"] = "\uD83C\uDFCB\uD83C\uDFFD\u200D♀️",
            [":woman_lifting_weights_tone1:"] = "\uD83C\uDFCB\uD83C\uDFFB\u200D♀️",
            [":woman_lifting_weights_tone2:"] = "\uD83C\uDFCB\uD83C\uDFFC\u200D♀️",
            [":woman_lifting_weights_tone3:"] = "\uD83C\uDFCB\uD83C\uDFFD\u200D♀️",
            [":woman_lifting_weights_tone4:"] = "\uD83C\uDFCB\uD83C\uDFFE\u200D♀️",
            [":woman_lifting_weights_tone5:"] = "\uD83C\uDFCB\uD83C\uDFFF\u200D♀️",
            [":woman_light_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_mage:"] = "\uD83E\uDDD9\u200D♀️",
            [":woman_mage::skin-tone-1:"] = "\uD83E\uDDD9\uD83C\uDFFB\u200D♀️",
            [":woman_mage::skin-tone-2:"] = "\uD83E\uDDD9\uD83C\uDFFC\u200D♀️",
            [":woman_mage::skin-tone-3:"] = "\uD83E\uDDD9\uD83C\uDFFD\u200D♀️",
            [":woman_mage::skin-tone-4:"] = "\uD83E\uDDD9\uD83C\uDFFE\u200D♀️",
            [":woman_mage::skin-tone-5:"] = "\uD83E\uDDD9\uD83C\uDFFF\u200D♀️",
            [":woman_mage_dark_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFF\u200D♀️",
            [":woman_mage_light_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFB\u200D♀️",
            [":woman_mage_medium_dark_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFE\u200D♀️",
            [":woman_mage_medium_light_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFC\u200D♀️",
            [":woman_mage_medium_skin_tone:"] = "\uD83E\uDDD9\uD83C\uDFFD\u200D♀️",
            [":woman_mage_tone1:"] = "\uD83E\uDDD9\uD83C\uDFFB\u200D♀️",
            [":woman_mage_tone2:"] = "\uD83E\uDDD9\uD83C\uDFFC\u200D♀️",
            [":woman_mage_tone3:"] = "\uD83E\uDDD9\uD83C\uDFFD\u200D♀️",
            [":woman_mage_tone4:"] = "\uD83E\uDDD9\uD83C\uDFFE\u200D♀️",
            [":woman_mage_tone5:"] = "\uD83E\uDDD9\uD83C\uDFFF\u200D♀️",
            [":woman_mechanic:"] = "\uD83D\uDC69\u200D\uD83D\uDD27",
            [":woman_mechanic::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDD27",
            [":woman_mechanic::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDD27",
            [":woman_mechanic::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDD27",
            [":woman_mechanic::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDD27",
            [":woman_mechanic::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDD27",
            [":woman_mechanic_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDD27",
            [":woman_mechanic_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDD27",
            [":woman_mechanic_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDD27",
            [":woman_mechanic_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDD27",
            [":woman_mechanic_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDD27",
            [":woman_mechanic_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDD27",
            [":woman_mechanic_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDD27",
            [":woman_mechanic_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDD27",
            [":woman_mechanic_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDD27",
            [":woman_mechanic_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDD27",
            [":woman_medium_dark_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_medium_light_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_medium_skin_tone_beard:"] = "\uD83E\uDDD4\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_mountain_biking:"] = "\uD83D\uDEB5\u200D♀️",
            [":woman_mountain_biking::skin-tone-1:"] = "\uD83D\uDEB5\uD83C\uDFFB\u200D♀️",
            [":woman_mountain_biking::skin-tone-2:"] = "\uD83D\uDEB5\uD83C\uDFFC\u200D♀️",
            [":woman_mountain_biking::skin-tone-3:"] = "\uD83D\uDEB5\uD83C\uDFFD\u200D♀️",
            [":woman_mountain_biking::skin-tone-4:"] = "\uD83D\uDEB5\uD83C\uDFFE\u200D♀️",
            [":woman_mountain_biking::skin-tone-5:"] = "\uD83D\uDEB5\uD83C\uDFFF\u200D♀️",
            [":woman_mountain_biking_dark_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFF\u200D♀️",
            [":woman_mountain_biking_light_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFB\u200D♀️",
            [":woman_mountain_biking_medium_dark_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFE\u200D♀️",
            [":woman_mountain_biking_medium_light_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFC\u200D♀️",
            [":woman_mountain_biking_medium_skin_tone:"] = "\uD83D\uDEB5\uD83C\uDFFD\u200D♀️",
            [":woman_mountain_biking_tone1:"] = "\uD83D\uDEB5\uD83C\uDFFB\u200D♀️",
            [":woman_mountain_biking_tone2:"] = "\uD83D\uDEB5\uD83C\uDFFC\u200D♀️",
            [":woman_mountain_biking_tone3:"] = "\uD83D\uDEB5\uD83C\uDFFD\u200D♀️",
            [":woman_mountain_biking_tone4:"] = "\uD83D\uDEB5\uD83C\uDFFE\u200D♀️",
            [":woman_mountain_biking_tone5:"] = "\uD83D\uDEB5\uD83C\uDFFF\u200D♀️",
            [":woman_office_worker:"] = "\uD83D\uDC69\u200D\uD83D\uDCBC",
            [":woman_office_worker::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDCBC",
            [":woman_office_worker::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDCBC",
            [":woman_office_worker::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDCBC",
            [":woman_office_worker::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDCBC",
            [":woman_office_worker::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDCBC",
            [":woman_office_worker_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDCBC",
            [":woman_office_worker_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDCBC",
            [":woman_office_worker_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDCBC",
            [":woman_office_worker_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDCBC",
            [":woman_office_worker_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDCBC",
            [":woman_office_worker_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDCBC",
            [":woman_office_worker_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDCBC",
            [":woman_office_worker_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDCBC",
            [":woman_office_worker_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDCBC",
            [":woman_office_worker_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDCBC",
            [":woman_pilot:"] = "\uD83D\uDC69\u200D✈️",
            [":woman_pilot::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D✈️",
            [":woman_pilot::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D✈️",
            [":woman_pilot::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D✈️",
            [":woman_pilot::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D✈️",
            [":woman_pilot::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D✈️",
            [":woman_pilot_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D✈️",
            [":woman_pilot_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D✈️",
            [":woman_pilot_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D✈️",
            [":woman_pilot_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D✈️",
            [":woman_pilot_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D✈️",
            [":woman_pilot_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D✈️",
            [":woman_pilot_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D✈️",
            [":woman_pilot_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D✈️",
            [":woman_pilot_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D✈️",
            [":woman_pilot_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D✈️",
            [":woman_playing_handball:"] = "\uD83E\uDD3E\u200D♀️",
            [":woman_playing_handball::skin-tone-1:"] = "\uD83E\uDD3E\uD83C\uDFFB\u200D♀️",
            [":woman_playing_handball::skin-tone-2:"] = "\uD83E\uDD3E\uD83C\uDFFC\u200D♀️",
            [":woman_playing_handball::skin-tone-3:"] = "\uD83E\uDD3E\uD83C\uDFFD\u200D♀️",
            [":woman_playing_handball::skin-tone-4:"] = "\uD83E\uDD3E\uD83C\uDFFE\u200D♀️",
            [":woman_playing_handball::skin-tone-5:"] = "\uD83E\uDD3E\uD83C\uDFFF\u200D♀️",
            [":woman_playing_handball_dark_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFF\u200D♀️",
            [":woman_playing_handball_light_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFB\u200D♀️",
            [":woman_playing_handball_medium_dark_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFE\u200D♀️",
            [":woman_playing_handball_medium_light_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFC\u200D♀️",
            [":woman_playing_handball_medium_skin_tone:"] = "\uD83E\uDD3E\uD83C\uDFFD\u200D♀️",
            [":woman_playing_handball_tone1:"] = "\uD83E\uDD3E\uD83C\uDFFB\u200D♀️",
            [":woman_playing_handball_tone2:"] = "\uD83E\uDD3E\uD83C\uDFFC\u200D♀️",
            [":woman_playing_handball_tone3:"] = "\uD83E\uDD3E\uD83C\uDFFD\u200D♀️",
            [":woman_playing_handball_tone4:"] = "\uD83E\uDD3E\uD83C\uDFFE\u200D♀️",
            [":woman_playing_handball_tone5:"] = "\uD83E\uDD3E\uD83C\uDFFF\u200D♀️",
            [":woman_playing_water_polo:"] = "\uD83E\uDD3D\u200D♀️",
            [":woman_playing_water_polo::skin-tone-1:"] = "\uD83E\uDD3D\uD83C\uDFFB\u200D♀️",
            [":woman_playing_water_polo::skin-tone-2:"] = "\uD83E\uDD3D\uD83C\uDFFC\u200D♀️",
            [":woman_playing_water_polo::skin-tone-3:"] = "\uD83E\uDD3D\uD83C\uDFFD\u200D♀️",
            [":woman_playing_water_polo::skin-tone-4:"] = "\uD83E\uDD3D\uD83C\uDFFE\u200D♀️",
            [":woman_playing_water_polo::skin-tone-5:"] = "\uD83E\uDD3D\uD83C\uDFFF\u200D♀️",
            [":woman_playing_water_polo_dark_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFF\u200D♀️",
            [":woman_playing_water_polo_light_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFB\u200D♀️",
            [":woman_playing_water_polo_medium_dark_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFE\u200D♀️",
            [":woman_playing_water_polo_medium_light_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFC\u200D♀️",
            [":woman_playing_water_polo_medium_skin_tone:"] = "\uD83E\uDD3D\uD83C\uDFFD\u200D♀️",
            [":woman_playing_water_polo_tone1:"] = "\uD83E\uDD3D\uD83C\uDFFB\u200D♀️",
            [":woman_playing_water_polo_tone2:"] = "\uD83E\uDD3D\uD83C\uDFFC\u200D♀️",
            [":woman_playing_water_polo_tone3:"] = "\uD83E\uDD3D\uD83C\uDFFD\u200D♀️",
            [":woman_playing_water_polo_tone4:"] = "\uD83E\uDD3D\uD83C\uDFFE\u200D♀️",
            [":woman_playing_water_polo_tone5:"] = "\uD83E\uDD3D\uD83C\uDFFF\u200D♀️",
            [":woman_police_officer:"] = "\uD83D\uDC6E\u200D♀️",
            [":woman_police_officer::skin-tone-1:"] = "\uD83D\uDC6E\uD83C\uDFFB\u200D♀️",
            [":woman_police_officer::skin-tone-2:"] = "\uD83D\uDC6E\uD83C\uDFFC\u200D♀️",
            [":woman_police_officer::skin-tone-3:"] = "\uD83D\uDC6E\uD83C\uDFFD\u200D♀️",
            [":woman_police_officer::skin-tone-4:"] = "\uD83D\uDC6E\uD83C\uDFFE\u200D♀️",
            [":woman_police_officer::skin-tone-5:"] = "\uD83D\uDC6E\uD83C\uDFFF\u200D♀️",
            [":woman_police_officer_dark_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFF\u200D♀️",
            [":woman_police_officer_light_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFB\u200D♀️",
            [":woman_police_officer_medium_dark_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFE\u200D♀️",
            [":woman_police_officer_medium_light_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFC\u200D♀️",
            [":woman_police_officer_medium_skin_tone:"] = "\uD83D\uDC6E\uD83C\uDFFD\u200D♀️",
            [":woman_police_officer_tone1:"] = "\uD83D\uDC6E\uD83C\uDFFB\u200D♀️",
            [":woman_police_officer_tone2:"] = "\uD83D\uDC6E\uD83C\uDFFC\u200D♀️",
            [":woman_police_officer_tone3:"] = "\uD83D\uDC6E\uD83C\uDFFD\u200D♀️",
            [":woman_police_officer_tone4:"] = "\uD83D\uDC6E\uD83C\uDFFE\u200D♀️",
            [":woman_police_officer_tone5:"] = "\uD83D\uDC6E\uD83C\uDFFF\u200D♀️",
            [":woman_pouting:"] = "\uD83D\uDE4E\u200D♀️",
            [":woman_pouting::skin-tone-1:"] = "\uD83D\uDE4E\uD83C\uDFFB\u200D♀️",
            [":woman_pouting::skin-tone-2:"] = "\uD83D\uDE4E\uD83C\uDFFC\u200D♀️",
            [":woman_pouting::skin-tone-3:"] = "\uD83D\uDE4E\uD83C\uDFFD\u200D♀️",
            [":woman_pouting::skin-tone-4:"] = "\uD83D\uDE4E\uD83C\uDFFE\u200D♀️",
            [":woman_pouting::skin-tone-5:"] = "\uD83D\uDE4E\uD83C\uDFFF\u200D♀️",
            [":woman_pouting_dark_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFF\u200D♀️",
            [":woman_pouting_light_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFB\u200D♀️",
            [":woman_pouting_medium_dark_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFE\u200D♀️",
            [":woman_pouting_medium_light_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFC\u200D♀️",
            [":woman_pouting_medium_skin_tone:"] = "\uD83D\uDE4E\uD83C\uDFFD\u200D♀️",
            [":woman_pouting_tone1:"] = "\uD83D\uDE4E\uD83C\uDFFB\u200D♀️",
            [":woman_pouting_tone2:"] = "\uD83D\uDE4E\uD83C\uDFFC\u200D♀️",
            [":woman_pouting_tone3:"] = "\uD83D\uDE4E\uD83C\uDFFD\u200D♀️",
            [":woman_pouting_tone4:"] = "\uD83D\uDE4E\uD83C\uDFFE\u200D♀️",
            [":woman_pouting_tone5:"] = "\uD83D\uDE4E\uD83C\uDFFF\u200D♀️",
            [":woman_raising_hand:"] = "\uD83D\uDE4B\u200D♀️",
            [":woman_raising_hand::skin-tone-1:"] = "\uD83D\uDE4B\uD83C\uDFFB\u200D♀️",
            [":woman_raising_hand::skin-tone-2:"] = "\uD83D\uDE4B\uD83C\uDFFC\u200D♀️",
            [":woman_raising_hand::skin-tone-3:"] = "\uD83D\uDE4B\uD83C\uDFFD\u200D♀️",
            [":woman_raising_hand::skin-tone-4:"] = "\uD83D\uDE4B\uD83C\uDFFE\u200D♀️",
            [":woman_raising_hand::skin-tone-5:"] = "\uD83D\uDE4B\uD83C\uDFFF\u200D♀️",
            [":woman_raising_hand_dark_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFF\u200D♀️",
            [":woman_raising_hand_light_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFB\u200D♀️",
            [":woman_raising_hand_medium_dark_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFE\u200D♀️",
            [":woman_raising_hand_medium_light_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFC\u200D♀️",
            [":woman_raising_hand_medium_skin_tone:"] = "\uD83D\uDE4B\uD83C\uDFFD\u200D♀️",
            [":woman_raising_hand_tone1:"] = "\uD83D\uDE4B\uD83C\uDFFB\u200D♀️",
            [":woman_raising_hand_tone2:"] = "\uD83D\uDE4B\uD83C\uDFFC\u200D♀️",
            [":woman_raising_hand_tone3:"] = "\uD83D\uDE4B\uD83C\uDFFD\u200D♀️",
            [":woman_raising_hand_tone4:"] = "\uD83D\uDE4B\uD83C\uDFFE\u200D♀️",
            [":woman_raising_hand_tone5:"] = "\uD83D\uDE4B\uD83C\uDFFF\u200D♀️",
            [":woman_red_haired:"] = "\uD83D\uDC69\u200D\uD83E\uDDB0",
            [":woman_red_haired::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB0",
            [":woman_red_haired::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB0",
            [":woman_red_haired::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB0",
            [":woman_red_haired::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB0",
            [":woman_red_haired::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB0",
            [":woman_red_haired_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB0",
            [":woman_red_haired_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB0",
            [":woman_red_haired_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB0",
            [":woman_red_haired_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB0",
            [":woman_red_haired_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB0",
            [":woman_red_haired_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB0",
            [":woman_red_haired_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB0",
            [":woman_red_haired_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB0",
            [":woman_red_haired_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB0",
            [":woman_red_haired_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB0",
            [":woman_rowing_boat:"] = "\uD83D\uDEA3\u200D♀️",
            [":woman_rowing_boat::skin-tone-1:"] = "\uD83D\uDEA3\uD83C\uDFFB\u200D♀️",
            [":woman_rowing_boat::skin-tone-2:"] = "\uD83D\uDEA3\uD83C\uDFFC\u200D♀️",
            [":woman_rowing_boat::skin-tone-3:"] = "\uD83D\uDEA3\uD83C\uDFFD\u200D♀️",
            [":woman_rowing_boat::skin-tone-4:"] = "\uD83D\uDEA3\uD83C\uDFFE\u200D♀️",
            [":woman_rowing_boat::skin-tone-5:"] = "\uD83D\uDEA3\uD83C\uDFFF\u200D♀️",
            [":woman_rowing_boat_dark_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFF\u200D♀️",
            [":woman_rowing_boat_light_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFB\u200D♀️",
            [":woman_rowing_boat_medium_dark_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFE\u200D♀️",
            [":woman_rowing_boat_medium_light_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFC\u200D♀️",
            [":woman_rowing_boat_medium_skin_tone:"] = "\uD83D\uDEA3\uD83C\uDFFD\u200D♀️",
            [":woman_rowing_boat_tone1:"] = "\uD83D\uDEA3\uD83C\uDFFB\u200D♀️",
            [":woman_rowing_boat_tone2:"] = "\uD83D\uDEA3\uD83C\uDFFC\u200D♀️",
            [":woman_rowing_boat_tone3:"] = "\uD83D\uDEA3\uD83C\uDFFD\u200D♀️",
            [":woman_rowing_boat_tone4:"] = "\uD83D\uDEA3\uD83C\uDFFE\u200D♀️",
            [":woman_rowing_boat_tone5:"] = "\uD83D\uDEA3\uD83C\uDFFF\u200D♀️",
            [":woman_running:"] = "\uD83C\uDFC3\u200D♀️",
            [":woman_running::skin-tone-1:"] = "\uD83C\uDFC3\uD83C\uDFFB\u200D♀️",
            [":woman_running::skin-tone-2:"] = "\uD83C\uDFC3\uD83C\uDFFC\u200D♀️",
            [":woman_running::skin-tone-3:"] = "\uD83C\uDFC3\uD83C\uDFFD\u200D♀️",
            [":woman_running::skin-tone-4:"] = "\uD83C\uDFC3\uD83C\uDFFE\u200D♀️",
            [":woman_running::skin-tone-5:"] = "\uD83C\uDFC3\uD83C\uDFFF\u200D♀️",
            [":woman_running_dark_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFF\u200D♀️",
            [":woman_running_light_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFB\u200D♀️",
            [":woman_running_medium_dark_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFE\u200D♀️",
            [":woman_running_medium_light_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFC\u200D♀️",
            [":woman_running_medium_skin_tone:"] = "\uD83C\uDFC3\uD83C\uDFFD\u200D♀️",
            [":woman_running_tone1:"] = "\uD83C\uDFC3\uD83C\uDFFB\u200D♀️",
            [":woman_running_tone2:"] = "\uD83C\uDFC3\uD83C\uDFFC\u200D♀️",
            [":woman_running_tone3:"] = "\uD83C\uDFC3\uD83C\uDFFD\u200D♀️",
            [":woman_running_tone4:"] = "\uD83C\uDFC3\uD83C\uDFFE\u200D♀️",
            [":woman_running_tone5:"] = "\uD83C\uDFC3\uD83C\uDFFF\u200D♀️",
            [":woman_scientist:"] = "\uD83D\uDC69\u200D\uD83D\uDD2C",
            [":woman_scientist::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDD2C",
            [":woman_scientist::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDD2C",
            [":woman_scientist::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDD2C",
            [":woman_scientist::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDD2C",
            [":woman_scientist::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDD2C",
            [":woman_scientist_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDD2C",
            [":woman_scientist_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDD2C",
            [":woman_scientist_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDD2C",
            [":woman_scientist_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDD2C",
            [":woman_scientist_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDD2C",
            [":woman_scientist_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDD2C",
            [":woman_scientist_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDD2C",
            [":woman_scientist_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDD2C",
            [":woman_scientist_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDD2C",
            [":woman_scientist_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDD2C",
            [":woman_shrugging:"] = "\uD83E\uDD37\u200D♀️",
            [":woman_shrugging::skin-tone-1:"] = "\uD83E\uDD37\uD83C\uDFFB\u200D♀️",
            [":woman_shrugging::skin-tone-2:"] = "\uD83E\uDD37\uD83C\uDFFC\u200D♀️",
            [":woman_shrugging::skin-tone-3:"] = "\uD83E\uDD37\uD83C\uDFFD\u200D♀️",
            [":woman_shrugging::skin-tone-4:"] = "\uD83E\uDD37\uD83C\uDFFE\u200D♀️",
            [":woman_shrugging::skin-tone-5:"] = "\uD83E\uDD37\uD83C\uDFFF\u200D♀️",
            [":woman_shrugging_dark_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFF\u200D♀️",
            [":woman_shrugging_light_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFB\u200D♀️",
            [":woman_shrugging_medium_dark_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFE\u200D♀️",
            [":woman_shrugging_medium_light_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFC\u200D♀️",
            [":woman_shrugging_medium_skin_tone:"] = "\uD83E\uDD37\uD83C\uDFFD\u200D♀️",
            [":woman_shrugging_tone1:"] = "\uD83E\uDD37\uD83C\uDFFB\u200D♀️",
            [":woman_shrugging_tone2:"] = "\uD83E\uDD37\uD83C\uDFFC\u200D♀️",
            [":woman_shrugging_tone3:"] = "\uD83E\uDD37\uD83C\uDFFD\u200D♀️",
            [":woman_shrugging_tone4:"] = "\uD83E\uDD37\uD83C\uDFFE\u200D♀️",
            [":woman_shrugging_tone5:"] = "\uD83E\uDD37\uD83C\uDFFF\u200D♀️",
            [":woman_singer:"] = "\uD83D\uDC69\u200D\uD83C\uDFA4",
            [":woman_singer::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFA4",
            [":woman_singer::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFA4",
            [":woman_singer::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFA4",
            [":woman_singer::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFA4",
            [":woman_singer::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFA4",
            [":woman_singer_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFA4",
            [":woman_singer_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFA4",
            [":woman_singer_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFA4",
            [":woman_singer_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFA4",
            [":woman_singer_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFA4",
            [":woman_singer_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFA4",
            [":woman_singer_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFA4",
            [":woman_singer_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFA4",
            [":woman_singer_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFA4",
            [":woman_singer_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFA4",
            [":woman_standing:"] = "\uD83E\uDDCD\u200D♀️",
            [":woman_standing::skin-tone-1:"] = "\uD83E\uDDCD\uD83C\uDFFB\u200D♀️",
            [":woman_standing::skin-tone-2:"] = "\uD83E\uDDCD\uD83C\uDFFC\u200D♀️",
            [":woman_standing::skin-tone-3:"] = "\uD83E\uDDCD\uD83C\uDFFD\u200D♀️",
            [":woman_standing::skin-tone-4:"] = "\uD83E\uDDCD\uD83C\uDFFE\u200D♀️",
            [":woman_standing::skin-tone-5:"] = "\uD83E\uDDCD\uD83C\uDFFF\u200D♀️",
            [":woman_standing_dark_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFF\u200D♀️",
            [":woman_standing_light_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFB\u200D♀️",
            [":woman_standing_medium_dark_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFE\u200D♀️",
            [":woman_standing_medium_light_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFC\u200D♀️",
            [":woman_standing_medium_skin_tone:"] = "\uD83E\uDDCD\uD83C\uDFFD\u200D♀️",
            [":woman_standing_tone1:"] = "\uD83E\uDDCD\uD83C\uDFFB\u200D♀️",
            [":woman_standing_tone2:"] = "\uD83E\uDDCD\uD83C\uDFFC\u200D♀️",
            [":woman_standing_tone3:"] = "\uD83E\uDDCD\uD83C\uDFFD\u200D♀️",
            [":woman_standing_tone4:"] = "\uD83E\uDDCD\uD83C\uDFFE\u200D♀️",
            [":woman_standing_tone5:"] = "\uD83E\uDDCD\uD83C\uDFFF\u200D♀️",
            [":woman_student:"] = "\uD83D\uDC69\u200D\uD83C\uDF93",
            [":woman_student::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF93",
            [":woman_student::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF93",
            [":woman_student::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF93",
            [":woman_student::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF93",
            [":woman_student::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF93",
            [":woman_student_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF93",
            [":woman_student_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF93",
            [":woman_student_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF93",
            [":woman_student_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF93",
            [":woman_student_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF93",
            [":woman_student_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDF93",
            [":woman_student_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDF93",
            [":woman_student_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDF93",
            [":woman_student_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDF93",
            [":woman_student_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDF93",
            [":woman_superhero:"] = "\uD83E\uDDB8\u200D♀️",
            [":woman_superhero::skin-tone-1:"] = "\uD83E\uDDB8\uD83C\uDFFB\u200D♀️",
            [":woman_superhero::skin-tone-2:"] = "\uD83E\uDDB8\uD83C\uDFFC\u200D♀️",
            [":woman_superhero::skin-tone-3:"] = "\uD83E\uDDB8\uD83C\uDFFD\u200D♀️",
            [":woman_superhero::skin-tone-4:"] = "\uD83E\uDDB8\uD83C\uDFFE\u200D♀️",
            [":woman_superhero::skin-tone-5:"] = "\uD83E\uDDB8\uD83C\uDFFF\u200D♀️",
            [":woman_superhero_dark_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFF\u200D♀️",
            [":woman_superhero_light_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFB\u200D♀️",
            [":woman_superhero_medium_dark_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFE\u200D♀️",
            [":woman_superhero_medium_light_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFC\u200D♀️",
            [":woman_superhero_medium_skin_tone:"] = "\uD83E\uDDB8\uD83C\uDFFD\u200D♀️",
            [":woman_superhero_tone1:"] = "\uD83E\uDDB8\uD83C\uDFFB\u200D♀️",
            [":woman_superhero_tone2:"] = "\uD83E\uDDB8\uD83C\uDFFC\u200D♀️",
            [":woman_superhero_tone3:"] = "\uD83E\uDDB8\uD83C\uDFFD\u200D♀️",
            [":woman_superhero_tone4:"] = "\uD83E\uDDB8\uD83C\uDFFE\u200D♀️",
            [":woman_superhero_tone5:"] = "\uD83E\uDDB8\uD83C\uDFFF\u200D♀️",
            [":woman_supervillain:"] = "\uD83E\uDDB9\u200D♀️",
            [":woman_supervillain::skin-tone-1:"] = "\uD83E\uDDB9\uD83C\uDFFB\u200D♀️",
            [":woman_supervillain::skin-tone-2:"] = "\uD83E\uDDB9\uD83C\uDFFC\u200D♀️",
            [":woman_supervillain::skin-tone-3:"] = "\uD83E\uDDB9\uD83C\uDFFD\u200D♀️",
            [":woman_supervillain::skin-tone-4:"] = "\uD83E\uDDB9\uD83C\uDFFE\u200D♀️",
            [":woman_supervillain::skin-tone-5:"] = "\uD83E\uDDB9\uD83C\uDFFF\u200D♀️",
            [":woman_supervillain_dark_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFF\u200D♀️",
            [":woman_supervillain_light_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFB\u200D♀️",
            [":woman_supervillain_medium_dark_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFE\u200D♀️",
            [":woman_supervillain_medium_light_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFC\u200D♀️",
            [":woman_supervillain_medium_skin_tone:"] = "\uD83E\uDDB9\uD83C\uDFFD\u200D♀️",
            [":woman_supervillain_tone1:"] = "\uD83E\uDDB9\uD83C\uDFFB\u200D♀️",
            [":woman_supervillain_tone2:"] = "\uD83E\uDDB9\uD83C\uDFFC\u200D♀️",
            [":woman_supervillain_tone3:"] = "\uD83E\uDDB9\uD83C\uDFFD\u200D♀️",
            [":woman_supervillain_tone4:"] = "\uD83E\uDDB9\uD83C\uDFFE\u200D♀️",
            [":woman_supervillain_tone5:"] = "\uD83E\uDDB9\uD83C\uDFFF\u200D♀️",
            [":woman_surfing:"] = "\uD83C\uDFC4\u200D♀️",
            [":woman_surfing::skin-tone-1:"] = "\uD83C\uDFC4\uD83C\uDFFB\u200D♀️",
            [":woman_surfing::skin-tone-2:"] = "\uD83C\uDFC4\uD83C\uDFFC\u200D♀️",
            [":woman_surfing::skin-tone-3:"] = "\uD83C\uDFC4\uD83C\uDFFD\u200D♀️",
            [":woman_surfing::skin-tone-4:"] = "\uD83C\uDFC4\uD83C\uDFFE\u200D♀️",
            [":woman_surfing::skin-tone-5:"] = "\uD83C\uDFC4\uD83C\uDFFF\u200D♀️",
            [":woman_surfing_dark_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFF\u200D♀️",
            [":woman_surfing_light_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFB\u200D♀️",
            [":woman_surfing_medium_dark_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFE\u200D♀️",
            [":woman_surfing_medium_light_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFC\u200D♀️",
            [":woman_surfing_medium_skin_tone:"] = "\uD83C\uDFC4\uD83C\uDFFD\u200D♀️",
            [":woman_surfing_tone1:"] = "\uD83C\uDFC4\uD83C\uDFFB\u200D♀️",
            [":woman_surfing_tone2:"] = "\uD83C\uDFC4\uD83C\uDFFC\u200D♀️",
            [":woman_surfing_tone3:"] = "\uD83C\uDFC4\uD83C\uDFFD\u200D♀️",
            [":woman_surfing_tone4:"] = "\uD83C\uDFC4\uD83C\uDFFE\u200D♀️",
            [":woman_surfing_tone5:"] = "\uD83C\uDFC4\uD83C\uDFFF\u200D♀️",
            [":woman_swimming:"] = "\uD83C\uDFCA\u200D♀️",
            [":woman_swimming::skin-tone-1:"] = "\uD83C\uDFCA\uD83C\uDFFB\u200D♀️",
            [":woman_swimming::skin-tone-2:"] = "\uD83C\uDFCA\uD83C\uDFFC\u200D♀️",
            [":woman_swimming::skin-tone-3:"] = "\uD83C\uDFCA\uD83C\uDFFD\u200D♀️",
            [":woman_swimming::skin-tone-4:"] = "\uD83C\uDFCA\uD83C\uDFFE\u200D♀️",
            [":woman_swimming::skin-tone-5:"] = "\uD83C\uDFCA\uD83C\uDFFF\u200D♀️",
            [":woman_swimming_dark_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFF\u200D♀️",
            [":woman_swimming_light_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFB\u200D♀️",
            [":woman_swimming_medium_dark_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFE\u200D♀️",
            [":woman_swimming_medium_light_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFC\u200D♀️",
            [":woman_swimming_medium_skin_tone:"] = "\uD83C\uDFCA\uD83C\uDFFD\u200D♀️",
            [":woman_swimming_tone1:"] = "\uD83C\uDFCA\uD83C\uDFFB\u200D♀️",
            [":woman_swimming_tone2:"] = "\uD83C\uDFCA\uD83C\uDFFC\u200D♀️",
            [":woman_swimming_tone3:"] = "\uD83C\uDFCA\uD83C\uDFFD\u200D♀️",
            [":woman_swimming_tone4:"] = "\uD83C\uDFCA\uD83C\uDFFE\u200D♀️",
            [":woman_swimming_tone5:"] = "\uD83C\uDFCA\uD83C\uDFFF\u200D♀️",
            [":woman_teacher:"] = "\uD83D\uDC69\u200D\uD83C\uDFEB",
            [":woman_teacher::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFEB",
            [":woman_teacher::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFEB",
            [":woman_teacher::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFEB",
            [":woman_teacher::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFEB",
            [":woman_teacher::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFEB",
            [":woman_teacher_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFEB",
            [":woman_teacher_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFEB",
            [":woman_teacher_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFEB",
            [":woman_teacher_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFEB",
            [":woman_teacher_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFEB",
            [":woman_teacher_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83C\uDFEB",
            [":woman_teacher_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83C\uDFEB",
            [":woman_teacher_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83C\uDFEB",
            [":woman_teacher_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83C\uDFEB",
            [":woman_teacher_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83C\uDFEB",
            [":woman_technologist:"] = "\uD83D\uDC69\u200D\uD83D\uDCBB",
            [":woman_technologist::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDCBB",
            [":woman_technologist::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDCBB",
            [":woman_technologist::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDCBB",
            [":woman_technologist::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDCBB",
            [":woman_technologist::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDCBB",
            [":woman_technologist_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDCBB",
            [":woman_technologist_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDCBB",
            [":woman_technologist_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDCBB",
            [":woman_technologist_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDCBB",
            [":woman_technologist_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDCBB",
            [":woman_technologist_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83D\uDCBB",
            [":woman_technologist_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83D\uDCBB",
            [":woman_technologist_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83D\uDCBB",
            [":woman_technologist_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83D\uDCBB",
            [":woman_technologist_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83D\uDCBB",
            [":woman_tipping_hand:"] = "\uD83D\uDC81\u200D♀️",
            [":woman_tipping_hand::skin-tone-1:"] = "\uD83D\uDC81\uD83C\uDFFB\u200D♀️",
            [":woman_tipping_hand::skin-tone-2:"] = "\uD83D\uDC81\uD83C\uDFFC\u200D♀️",
            [":woman_tipping_hand::skin-tone-3:"] = "\uD83D\uDC81\uD83C\uDFFD\u200D♀️",
            [":woman_tipping_hand::skin-tone-4:"] = "\uD83D\uDC81\uD83C\uDFFE\u200D♀️",
            [":woman_tipping_hand::skin-tone-5:"] = "\uD83D\uDC81\uD83C\uDFFF\u200D♀️",
            [":woman_tipping_hand_dark_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFF\u200D♀️",
            [":woman_tipping_hand_light_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFB\u200D♀️",
            [":woman_tipping_hand_medium_dark_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFE\u200D♀️",
            [":woman_tipping_hand_medium_light_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFC\u200D♀️",
            [":woman_tipping_hand_medium_skin_tone:"] = "\uD83D\uDC81\uD83C\uDFFD\u200D♀️",
            [":woman_tipping_hand_tone1:"] = "\uD83D\uDC81\uD83C\uDFFB\u200D♀️",
            [":woman_tipping_hand_tone2:"] = "\uD83D\uDC81\uD83C\uDFFC\u200D♀️",
            [":woman_tipping_hand_tone3:"] = "\uD83D\uDC81\uD83C\uDFFD\u200D♀️",
            [":woman_tipping_hand_tone4:"] = "\uD83D\uDC81\uD83C\uDFFE\u200D♀️",
            [":woman_tipping_hand_tone5:"] = "\uD83D\uDC81\uD83C\uDFFF\u200D♀️",
            [":woman_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB",
            [":woman_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC",
            [":woman_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD",
            [":woman_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE",
            [":woman_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF",
            [":woman_vampire:"] = "\uD83E\uDDDB\u200D♀️",
            [":woman_vampire::skin-tone-1:"] = "\uD83E\uDDDB\uD83C\uDFFB\u200D♀️",
            [":woman_vampire::skin-tone-2:"] = "\uD83E\uDDDB\uD83C\uDFFC\u200D♀️",
            [":woman_vampire::skin-tone-3:"] = "\uD83E\uDDDB\uD83C\uDFFD\u200D♀️",
            [":woman_vampire::skin-tone-4:"] = "\uD83E\uDDDB\uD83C\uDFFE\u200D♀️",
            [":woman_vampire::skin-tone-5:"] = "\uD83E\uDDDB\uD83C\uDFFF\u200D♀️",
            [":woman_vampire_dark_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFF\u200D♀️",
            [":woman_vampire_light_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFB\u200D♀️",
            [":woman_vampire_medium_dark_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFE\u200D♀️",
            [":woman_vampire_medium_light_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFC\u200D♀️",
            [":woman_vampire_medium_skin_tone:"] = "\uD83E\uDDDB\uD83C\uDFFD\u200D♀️",
            [":woman_vampire_tone1:"] = "\uD83E\uDDDB\uD83C\uDFFB\u200D♀️",
            [":woman_vampire_tone2:"] = "\uD83E\uDDDB\uD83C\uDFFC\u200D♀️",
            [":woman_vampire_tone3:"] = "\uD83E\uDDDB\uD83C\uDFFD\u200D♀️",
            [":woman_vampire_tone4:"] = "\uD83E\uDDDB\uD83C\uDFFE\u200D♀️",
            [":woman_vampire_tone5:"] = "\uD83E\uDDDB\uD83C\uDFFF\u200D♀️",
            [":woman_walking:"] = "\uD83D\uDEB6\u200D♀️",
            [":woman_walking::skin-tone-1:"] = "\uD83D\uDEB6\uD83C\uDFFB\u200D♀️",
            [":woman_walking::skin-tone-2:"] = "\uD83D\uDEB6\uD83C\uDFFC\u200D♀️",
            [":woman_walking::skin-tone-3:"] = "\uD83D\uDEB6\uD83C\uDFFD\u200D♀️",
            [":woman_walking::skin-tone-4:"] = "\uD83D\uDEB6\uD83C\uDFFE\u200D♀️",
            [":woman_walking::skin-tone-5:"] = "\uD83D\uDEB6\uD83C\uDFFF\u200D♀️",
            [":woman_walking_dark_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFF\u200D♀️",
            [":woman_walking_light_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFB\u200D♀️",
            [":woman_walking_medium_dark_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFE\u200D♀️",
            [":woman_walking_medium_light_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFC\u200D♀️",
            [":woman_walking_medium_skin_tone:"] = "\uD83D\uDEB6\uD83C\uDFFD\u200D♀️",
            [":woman_walking_tone1:"] = "\uD83D\uDEB6\uD83C\uDFFB\u200D♀️",
            [":woman_walking_tone2:"] = "\uD83D\uDEB6\uD83C\uDFFC\u200D♀️",
            [":woman_walking_tone3:"] = "\uD83D\uDEB6\uD83C\uDFFD\u200D♀️",
            [":woman_walking_tone4:"] = "\uD83D\uDEB6\uD83C\uDFFE\u200D♀️",
            [":woman_walking_tone5:"] = "\uD83D\uDEB6\uD83C\uDFFF\u200D♀️",
            [":woman_wearing_turban:"] = "\uD83D\uDC73\u200D♀️",
            [":woman_wearing_turban::skin-tone-1:"] = "\uD83D\uDC73\uD83C\uDFFB\u200D♀️",
            [":woman_wearing_turban::skin-tone-2:"] = "\uD83D\uDC73\uD83C\uDFFC\u200D♀️",
            [":woman_wearing_turban::skin-tone-3:"] = "\uD83D\uDC73\uD83C\uDFFD\u200D♀️",
            [":woman_wearing_turban::skin-tone-4:"] = "\uD83D\uDC73\uD83C\uDFFE\u200D♀️",
            [":woman_wearing_turban::skin-tone-5:"] = "\uD83D\uDC73\uD83C\uDFFF\u200D♀️",
            [":woman_wearing_turban_dark_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFF\u200D♀️",
            [":woman_wearing_turban_light_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFB\u200D♀️",
            [":woman_wearing_turban_medium_dark_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFE\u200D♀️",
            [":woman_wearing_turban_medium_light_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFC\u200D♀️",
            [":woman_wearing_turban_medium_skin_tone:"] = "\uD83D\uDC73\uD83C\uDFFD\u200D♀️",
            [":woman_wearing_turban_tone1:"] = "\uD83D\uDC73\uD83C\uDFFB\u200D♀️",
            [":woman_wearing_turban_tone2:"] = "\uD83D\uDC73\uD83C\uDFFC\u200D♀️",
            [":woman_wearing_turban_tone3:"] = "\uD83D\uDC73\uD83C\uDFFD\u200D♀️",
            [":woman_wearing_turban_tone4:"] = "\uD83D\uDC73\uD83C\uDFFE\u200D♀️",
            [":woman_wearing_turban_tone5:"] = "\uD83D\uDC73\uD83C\uDFFF\u200D♀️",
            [":woman_white_haired:"] = "\uD83D\uDC69\u200D\uD83E\uDDB3",
            [":woman_white_haired::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB3",
            [":woman_white_haired::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB3",
            [":woman_white_haired::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB3",
            [":woman_white_haired::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB3",
            [":woman_white_haired::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB3",
            [":woman_white_haired_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB3",
            [":woman_white_haired_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB3",
            [":woman_white_haired_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB3",
            [":woman_white_haired_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB3",
            [":woman_white_haired_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB3",
            [":woman_white_haired_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDB3",
            [":woman_white_haired_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDB3",
            [":woman_white_haired_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDB3",
            [":woman_white_haired_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDB3",
            [":woman_white_haired_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDB3",
            [":woman_with_headscarf:"] = "\uD83E\uDDD5",
            [":woman_with_headscarf::skin-tone-1:"] = "\uD83E\uDDD5\uD83C\uDFFB",
            [":woman_with_headscarf::skin-tone-2:"] = "\uD83E\uDDD5\uD83C\uDFFC",
            [":woman_with_headscarf::skin-tone-3:"] = "\uD83E\uDDD5\uD83C\uDFFD",
            [":woman_with_headscarf::skin-tone-4:"] = "\uD83E\uDDD5\uD83C\uDFFE",
            [":woman_with_headscarf::skin-tone-5:"] = "\uD83E\uDDD5\uD83C\uDFFF",
            [":woman_with_headscarf_dark_skin_tone:"] = "\uD83E\uDDD5\uD83C\uDFFF",
            [":woman_with_headscarf_light_skin_tone:"] = "\uD83E\uDDD5\uD83C\uDFFB",
            [":woman_with_headscarf_medium_dark_skin_tone:"] = "\uD83E\uDDD5\uD83C\uDFFE",
            [":woman_with_headscarf_medium_light_skin_tone:"] = "\uD83E\uDDD5\uD83C\uDFFC",
            [":woman_with_headscarf_medium_skin_tone:"] = "\uD83E\uDDD5\uD83C\uDFFD",
            [":woman_with_headscarf_tone1:"] = "\uD83E\uDDD5\uD83C\uDFFB",
            [":woman_with_headscarf_tone2:"] = "\uD83E\uDDD5\uD83C\uDFFC",
            [":woman_with_headscarf_tone3:"] = "\uD83E\uDDD5\uD83C\uDFFD",
            [":woman_with_headscarf_tone4:"] = "\uD83E\uDDD5\uD83C\uDFFE",
            [":woman_with_headscarf_tone5:"] = "\uD83E\uDDD5\uD83C\uDFFF",
            [":woman_with_probing_cane:"] = "\uD83D\uDC69\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane::skin-tone-1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane::skin-tone-2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane::skin-tone-3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane::skin-tone-4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane::skin-tone-5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_medium_dark_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_medium_light_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_medium_skin_tone:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_tone1:"] = "\uD83D\uDC69\uD83C\uDFFB\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_tone2:"] = "\uD83D\uDC69\uD83C\uDFFC\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_tone3:"] = "\uD83D\uDC69\uD83C\uDFFD\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_tone4:"] = "\uD83D\uDC69\uD83C\uDFFE\u200D\uD83E\uDDAF",
            [":woman_with_probing_cane_tone5:"] = "\uD83D\uDC69\uD83C\uDFFF\u200D\uD83E\uDDAF",
            [":woman_with_veil:"] = "\uD83D\uDC70\u200D\u2640\uFE0F",
            [":woman_with_veil::skin-tone-1:"] = "\uD83D\uDC70\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_with_veil::skin-tone-2:"] = "\uD83D\uDC70\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_with_veil::skin-tone-3:"] = "\uD83D\uDC70\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_with_veil::skin-tone-4:"] = "\uD83D\uDC70\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_with_veil::skin-tone-5:"] = "\uD83D\uDC70\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_with_veil_dark_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_with_veil_light_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_with_veil_medium_dark_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_with_veil_medium_light_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_with_veil_medium_skin_tone:"] = "\uD83D\uDC70\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_with_veil_tone1:"] = "\uD83D\uDC70\uD83C\uDFFB\u200D\u2640\uFE0F",
            [":woman_with_veil_tone2:"] = "\uD83D\uDC70\uD83C\uDFFC\u200D\u2640\uFE0F",
            [":woman_with_veil_tone3:"] = "\uD83D\uDC70\uD83C\uDFFD\u200D\u2640\uFE0F",
            [":woman_with_veil_tone4:"] = "\uD83D\uDC70\uD83C\uDFFE\u200D\u2640\uFE0F",
            [":woman_with_veil_tone5:"] = "\uD83D\uDC70\uD83C\uDFFF\u200D\u2640\uFE0F",
            [":woman_zombie:"] = "\uD83E\uDDDF\u200D♀️",
            [":womans_clothes:"] = "\uD83D\uDC5A",
            [":womans_flat_shoe:"] = "\uD83E\uDD7F",
            [":womans_hat:"] = "\uD83D\uDC52",
            [":women_with_bunny_ears_partying:"] = "\uD83D\uDC6F\u200D♀️",
            [":women_wrestling:"] = "\uD83E\uDD3C\u200D♀️",
            [":womens:"] = "\uD83D\uDEBA",
            [":wood:"] = "\uD83E\uDEB5",
            [":woozy_face:"] = "\uD83E\uDD74",
            [":world_map:"] = "\uD83D\uDDFA️",
            [":worm:"] = "\uD83E\uDEB1",
            [":worried:"] = "\uD83D\uDE1F",
            [":worship_symbol:"] = "\uD83D\uDED0",
            [":wrench:"] = "\uD83D\uDD27",
            [":wrestlers:"] = "\uD83E\uDD3C",
            [":wrestling:"] = "\uD83E\uDD3C",
            [":writing_hand:"] = "✍️",
            [":writing_hand::skin-tone-1:"] = "✍\uD83C\uDFFB",
            [":writing_hand::skin-tone-2:"] = "✍\uD83C\uDFFC",
            [":writing_hand::skin-tone-3:"] = "✍\uD83C\uDFFD",
            [":writing_hand::skin-tone-4:"] = "✍\uD83C\uDFFE",
            [":writing_hand::skin-tone-5:"] = "✍\uD83C\uDFFF",
            [":writing_hand_tone1:"] = "✍\uD83C\uDFFB",
            [":writing_hand_tone2:"] = "✍\uD83C\uDFFC",
            [":writing_hand_tone3:"] = "✍\uD83C\uDFFD",
            [":writing_hand_tone4:"] = "✍\uD83C\uDFFE",
            [":writing_hand_tone5:"] = "✍\uD83C\uDFFF",
            [":x:"] = "❌",
            [":x_ray:"] = "\uD83E\uDE7B",
            [":yarn:"] = "\uD83E\uDDF6",
            [":yawning_face:"] = "\uD83E\uDD71",
            [":yellow_circle:"] = "\uD83D\uDFE1",
            [":yellow_heart:"] = "\uD83D\uDC9B",
            [":yellow_square:"] = "\uD83D\uDFE8",
            [":yen:"] = "\uD83D\uDCB4",
            [":yin_yang:"] = "☯️",
            [":yo_yo:"] = "\uD83E\uDE80",
            [":yum:"] = "\uD83D\uDE0B",
            [":z"] = "\uD83D\uDE12",
            [":zany_face:"] = "\uD83E\uDD2A",
            [":zap:"] = "⚡",
            [":zebra:"] = "\uD83E\uDD93",
            [":zero:"] = "0️⃣",
            [":zipper_mouth:"] = "\uD83E\uDD10",
            [":zipper_mouth_face:"] = "\uD83E\uDD10",
            [":zombie:"] = "\uD83E\uDDDF",
            [":zzz:"] = "\uD83D\uDCA4",
            [":|"] = "\uD83D\uDE10",
            [";("] = "\uD83D\uDE2D",
            [";)"] = "\uD83D\uDE09",
            [";-("] = "\uD83D\uDE2D",
            [";-)"] = "\uD83D\uDE09",
            ["</3"] = "\uD83D\uDC94",
            ["<3"] = "❤️",
            ["<\\3"] = "\uD83D\uDC94",
            ["=$"] = "\uD83D\uDE12",
            ["='("] = "\uD83D\uDE22",
            ["=')"] = "\uD83D\uDE02",
            ["='-("] = "\uD83D\uDE22",
            ["='-)"] = "\uD83D\uDE02",
            ["='-D"] = "\uD83D\uDE02",
            ["='D"] = "\uD83D\uDE02",
            ["=("] = "\uD83D\uDE26",
            ["=)"] = "\uD83D\uDE42",
            ["=*"] = "\uD83D\uDE17",
            ["=,'("] = "\uD83D\uDE2D",
            ["=,'-("] = "\uD83D\uDE2D",
            ["=,("] = "\uD83D\uDE22",
            ["=,)"] = "\uD83D\uDE02",
            ["=,-("] = "\uD83D\uDE22",
            ["=,-)"] = "\uD83D\uDE02",
            ["=,-D"] = "\uD83D\uDE02",
            ["=,D"] = "\uD83D\uDE02",
            ["=-$"] = "\uD83D\uDE12",
            ["=-("] = "\uD83D\uDE26",
            ["=-)"] = "\uD83D\uDE42",
            ["=-*"] = "\uD83D\uDE17",
            ["=-/"] = "\uD83D\uDE15",
            ["=-@"] = "\uD83D\uDE21",
            ["=-D"] = "\uD83D\uDE04",
            ["=-O"] = "\uD83D\uDE2E",
            ["=-P"] = "\uD83D\uDE1B",
            ["=-S"] = "\uD83D\uDE12",
            ["=-Z"] = "\uD83D\uDE12",
            ["=-\")"] = "\uD83D\uDE0A",
            ["=-\\"] = "\uD83D\uDE15",
            ["=-o"] = "\uD83D\uDE2E",
            ["=-|"] = "\uD83D\uDE10",
            ["=@"] = "\uD83D\uDE21",
            ["=D"] = "\uD83D\uDE04",
            ["=O"] = "\uD83D\uDE2E",
            ["=P"] = "\uD83D\uDE1B",
            ["=\")"] = "\uD83D\uDE0A",
            ["=o"] = "\uD83D\uDE2E",
            ["=s"] = "\uD83D\uDE12",
            ["=z"] = "\uD83D\uDE12",
            ["=|"] = "\uD83D\uDE10",
            [">:("] = "\uD83D\uDE20",
            [">:-("] = "\uD83D\uDE20",
            [">=("] = "\uD83D\uDE20",
            [">=-("] = "\uD83D\uDE20",
            ["B-)"] = "\uD83D\uDE0E",
            ["O:)"] = "\uD83D\uDE07",
            ["O:-)"] = "\uD83D\uDE07",
            ["O=)"] = "\uD83D\uDE07",
            ["O=-)"] = "\uD83D\uDE07",
            ["X-)"] = "\uD83D\uDE06",
            ["]:("] = "\uD83D\uDC7F",
            ["]:)"] = "\uD83D\uDE08",
            ["]:-("] = "\uD83D\uDC7F",
            ["]:-)"] = "\uD83D\uDE08",
            ["]=("] = "\uD83D\uDC7F",
            ["]=)"] = "\uD83D\uDE08",
            ["]=-("] = "\uD83D\uDC7F",
            ["]=-)"] = "\uD83D\uDE08",
            ["o:)"] = "\uD83D\uDE07",
            ["o:-)"] = "\uD83D\uDE07",
            ["o=)"] = "\uD83D\uDE07",
            ["o=-)"] = "\uD83D\uDE07",
            ["x-)"] = "\uD83D\uDE06",
            ["♡"] = "❤️"
        };

        private static IReadOnlyCollection<string> _unicodes;
        private static IReadOnlyCollection<string> Unicodes
        {
            get
            {
                _unicodes ??= NamesAndUnicodes.Select(kvp => kvp.Value).ToImmutableHashSet();
                return _unicodes;
            }
        }

        private static IReadOnlyDictionary<string, ReadOnlyCollection<string>> _unicodesAndNames;
        private static IReadOnlyDictionary<string, ReadOnlyCollection<string>> UnicodesAndNames
        {
            get
            {
                _unicodesAndNames ??=
                    NamesAndUnicodes
                        .GroupBy(kvp => kvp.Value)
                        .ToImmutableDictionary(
                            grouping => grouping.Key,
                            grouping => grouping.Select(kvp => kvp.Key)
                                .ToList()
                                .AsReadOnly()
                        );
                return _unicodesAndNames;
            }
        }

        public static implicit operator Emoji(string s) => Parse(s);
    }
}
